
/**
 * @file /magma/core/checksum/crc.c
 *
 * @brief	An x86 implementation of the 32-bit and 64-bit CRC algorithms.
 */

#include "magma.h"

#define A1 A
#define A(x) ((x) & 0xFF)
#define B(x) (((x) >> 8) & 0xFF)
#define C(x) (((x) >> 16) & 0xFF)
#define D(x) ((x) >> 24)

#define S8(x) ((x) >> 8)
#define S32(x) ((x) >> 32)

extern const uint32_t crc24_table[1024];
extern const uint32_t crc32_table[8][256];
extern const uint64_t crc64_table[4][256];

#define CRC24_INIT 0xB704CEL
#define CRC24_POLY 0x1864CFBL

 uint32_t crc24_init(void) {
	 return CRC24_INIT;
 }

 uint32_t crc24_update(void *buffer, size_t length, uint32_t crc) {

	 if (!buffer || !length) {
		 return crc;
	 }

	 for (size_t i = 0; i < length; i++) {
		crc ^= (*(unsigned char *)buffer++) << 16;

		for (int i = 0; i < 8; i++) {
			crc <<= 1;
			if (crc & 0x1000000) crc ^= CRC24_POLY;
		}
	 }

	 return crc;
 }

uint32_t crc24_final(uint32_t crc) {
  return crc & 0xFFFFFFL;
}

/**
 * @brief	Generate a 24-bit CRC value for the Privacy Enhanced Message format.
 * @note	This function uses the predefined initial value and polynomial defined in RFC 4880.
 * @param	buffer	a pointer to the data to be checked.
 * @param	length	the length, in bytes, of the input buffer.
 * @return	the updated 24-bit CRC value in a 32-bit unsigned integer.
 */
 uint32_t crc24_checksum(void *buffer, size_t length) {

	uint32_t crc = crc24_init();

	while (length--) {

		crc ^= (*(unsigned char *)buffer++) << 16;

		for (int i = 0; i < 8; i++) {
			crc <<= 1;
			if (crc & 0x1000000) crc ^= CRC24_POLY;
		}
	}

	return crc24_final(crc);
}

/**
 * @brief	Update a 32-bit CRC value with a check of additional data.
 * @param	buffer	a pointer to the data to be checked.
 * @param	length	the length, in bytes, of the input buffer.
 * @param	crc		the previously computed CRC value, or 0 if this is the initial pass.
 * @return	the updated 32-bit CRC value of the specified data.
 */
uint32_t crc32_update(void *buffer, size_t length, uint32_t crc) {
	uint8_t *limit;
	uint32_t holder;
	crc = ~crc;
	if (length > 8) {
		while ((uintptr_t)(buffer) & 7) {
			crc = crc32_table[0][*((uint8_t *)buffer++) ^ A(crc)] ^ S8(crc);
			--length;
		}
		limit = buffer + (length & ~(size_t)(7));
		length &= (size_t)(7);
		while (((uint8_t *)buffer) < limit) {
			crc ^= *(uint32_t *)(buffer);
			buffer += 4;
			crc = crc32_table[7][A(crc)] ^ crc32_table[6][B(crc)] ^ crc32_table[5][C(crc)] ^ crc32_table[4][D(crc)];
			holder = *(uint32_t *)(buffer);
			buffer += 4;
			crc = crc32_table[3][A(holder)] ^ crc32_table[2][B(holder)] ^ crc ^ crc32_table[1][C(holder)] ^ crc32_table[0][D(holder)];
		}
	}
	while (length-- != 0) {
		crc = crc32_table[0][*((uint8_t *)buffer++) ^ A(crc)] ^ S8(crc);
	}
	return ~crc;
}

/**
 * @brief	Get 32-bit CRC value for a specified block of data.
 * @param	buffer	a pointer to the data to be checked.
 * @param	length	the length, in bytes, of the input buffer.
 * @return	the 32-bit CRC value of the specified data.
 */
uint32_t crc32_checksum(void *buffer, size_t length) {
	return crc32_update(buffer, length, 0);
}

/**
 * @brief	Update a 64-bit CRC value with a check of additional data.
 * @param	buffer	a pointer to the data to be checked.
 * @param	length	the length, in bytes, of the input buffer.
 * @param	crc		the previously computed CRC value, or 0 if this is the initial pass.
 * @return	the updated 64-bit CRC value of the specified data.
 */
uint64_t crc64_update(void *buffer, size_t length, uint64_t crc) {

	uint8_t *limit;
	uint32_t holder;
	crc = ~crc;
	if (length > 4) {
		while ((uintptr_t)(buffer) & 3) {
			crc = crc64_table[0][*((uint8_t *)buffer++) ^ A1(crc)] ^ S8(crc);
			--length;
		}
		limit = buffer + (length & ~(size_t)(3));
		length &= (size_t)(3);
		while (((uint8_t *)buffer) < limit) {
			holder = crc ^ *(uint32_t *)(buffer);
			buffer += 4;
			crc = crc64_table[3][A(holder)] ^ crc64_table[2][B(holder)] ^ S32(crc) ^ crc64_table[1][C(holder)] ^ crc64_table[0][D(holder)];
		}
	}
	while (length-- != 0) {
		crc = crc64_table[0][*((uint8_t *)buffer++) ^ A1(crc)] ^ S8(crc);
	}
	return ~crc;
}

/**
 * @brief	Get 64-bit CRC value for a specified block of data.
 * @param	buffer	a pointer to the data to be checked.
 * @param	length	the length, in bytes, of the input buffer.
 * @return	the 64-bit CRC value of the specified data.
 */
uint64_t crc64_checksum(void *buffer, size_t length) {
	return crc64_update(buffer, length, 0);
}

const uint32_t crc24_table[1024] = {
  0x00000000, 0x00fb4c86, 0x000dd58a, 0x00f6990c,
  0x00e1e693, 0x001aaa15, 0x00ec3319, 0x00177f9f,
  0x003981a1, 0x00c2cd27, 0x0034542b, 0x00cf18ad,
  0x00d86732, 0x00232bb4, 0x00d5b2b8, 0x002efe3e,
  0x00894ec5, 0x00720243, 0x00849b4f, 0x007fd7c9,
  0x0068a856, 0x0093e4d0, 0x00657ddc, 0x009e315a,
  0x00b0cf64, 0x004b83e2, 0x00bd1aee, 0x00465668,
  0x005129f7, 0x00aa6571, 0x005cfc7d, 0x00a7b0fb,
  0x00e9d10c, 0x00129d8a, 0x00e40486, 0x001f4800,
  0x0008379f, 0x00f37b19, 0x0005e215, 0x00feae93,
  0x00d050ad, 0x002b1c2b, 0x00dd8527, 0x0026c9a1,
  0x0031b63e, 0x00cafab8, 0x003c63b4, 0x00c72f32,
  0x00609fc9, 0x009bd34f, 0x006d4a43, 0x009606c5,
  0x0081795a, 0x007a35dc, 0x008cacd0, 0x0077e056,
  0x00591e68, 0x00a252ee, 0x0054cbe2, 0x00af8764,
  0x00b8f8fb, 0x0043b47d, 0x00b52d71, 0x004e61f7,
  0x00d2a319, 0x0029ef9f, 0x00df7693, 0x00243a15,
  0x0033458a, 0x00c8090c, 0x003e9000, 0x00c5dc86,
  0x00eb22b8, 0x00106e3e, 0x00e6f732, 0x001dbbb4,
  0x000ac42b, 0x00f188ad, 0x000711a1, 0x00fc5d27,
  0x005beddc, 0x00a0a15a, 0x00563856, 0x00ad74d0,
  0x00ba0b4f, 0x004147c9, 0x00b7dec5, 0x004c9243,
  0x00626c7d, 0x009920fb, 0x006fb9f7, 0x0094f571,
  0x00838aee, 0x0078c668, 0x008e5f64, 0x007513e2,
  0x003b7215, 0x00c03e93, 0x0036a79f, 0x00cdeb19,
  0x00da9486, 0x0021d800, 0x00d7410c, 0x002c0d8a,
  0x0002f3b4, 0x00f9bf32, 0x000f263e, 0x00f46ab8,
  0x00e31527, 0x001859a1, 0x00eec0ad, 0x00158c2b,
  0x00b23cd0, 0x00497056, 0x00bfe95a, 0x0044a5dc,
  0x0053da43, 0x00a896c5, 0x005e0fc9, 0x00a5434f,
  0x008bbd71, 0x0070f1f7, 0x008668fb, 0x007d247d,
  0x006a5be2, 0x00911764, 0x00678e68, 0x009cc2ee,
  0x00a44733, 0x005f0bb5, 0x00a992b9, 0x0052de3f,
  0x0045a1a0, 0x00beed26, 0x0048742a, 0x00b338ac,
  0x009dc692, 0x00668a14, 0x00901318, 0x006b5f9e,
  0x007c2001, 0x00876c87, 0x0071f58b, 0x008ab90d,
  0x002d09f6, 0x00d64570, 0x0020dc7c, 0x00db90fa,
  0x00ccef65, 0x0037a3e3, 0x00c13aef, 0x003a7669,
  0x00148857, 0x00efc4d1, 0x00195ddd, 0x00e2115b,
  0x00f56ec4, 0x000e2242, 0x00f8bb4e, 0x0003f7c8,
  0x004d963f, 0x00b6dab9, 0x004043b5, 0x00bb0f33,
  0x00ac70ac, 0x00573c2a, 0x00a1a526, 0x005ae9a0,
  0x0074179e, 0x008f5b18, 0x0079c214, 0x00828e92,
  0x0095f10d, 0x006ebd8b, 0x00982487, 0x00636801,
  0x00c4d8fa, 0x003f947c, 0x00c90d70, 0x003241f6,
  0x00253e69, 0x00de72ef, 0x0028ebe3, 0x00d3a765,
  0x00fd595b, 0x000615dd, 0x00f08cd1, 0x000bc057,
  0x001cbfc8, 0x00e7f34e, 0x00116a42, 0x00ea26c4,
  0x0076e42a, 0x008da8ac, 0x007b31a0, 0x00807d26,
  0x009702b9, 0x006c4e3f, 0x009ad733, 0x00619bb5,
  0x004f658b, 0x00b4290d, 0x0042b001, 0x00b9fc87,
  0x00ae8318, 0x0055cf9e, 0x00a35692, 0x00581a14,
  0x00ffaaef, 0x0004e669, 0x00f27f65, 0x000933e3,
  0x001e4c7c, 0x00e500fa, 0x001399f6, 0x00e8d570,
  0x00c62b4e, 0x003d67c8, 0x00cbfec4, 0x0030b242,
  0x0027cddd, 0x00dc815b, 0x002a1857, 0x00d154d1,
  0x009f3526, 0x006479a0, 0x0092e0ac, 0x0069ac2a,
  0x007ed3b5, 0x00859f33, 0x0073063f, 0x00884ab9,
  0x00a6b487, 0x005df801, 0x00ab610d, 0x00502d8b,
  0x00475214, 0x00bc1e92, 0x004a879e, 0x00b1cb18,
  0x00167be3, 0x00ed3765, 0x001bae69, 0x00e0e2ef,
  0x00f79d70, 0x000cd1f6, 0x00fa48fa, 0x0001047c,
  0x002ffa42, 0x00d4b6c4, 0x00222fc8, 0x00d9634e,
  0x00ce1cd1, 0x00355057, 0x00c3c95b, 0x003885dd,
  0x00000000, 0x00488f66, 0x00901ecd, 0x00d891ab,
  0x00db711c, 0x0093fe7a, 0x004b6fd1, 0x0003e0b7,
  0x00b6e338, 0x00fe6c5e, 0x0026fdf5, 0x006e7293,
  0x006d9224, 0x00251d42, 0x00fd8ce9, 0x00b5038f,
  0x006cc771, 0x00244817, 0x00fcd9bc, 0x00b456da,
  0x00b7b66d, 0x00ff390b, 0x0027a8a0, 0x006f27c6,
  0x00da2449, 0x0092ab2f, 0x004a3a84, 0x0002b5e2,
  0x00015555, 0x0049da33, 0x00914b98, 0x00d9c4fe,
  0x00d88ee3, 0x00900185, 0x0048902e, 0x00001f48,
  0x0003ffff, 0x004b7099, 0x0093e132, 0x00db6e54,
  0x006e6ddb, 0x0026e2bd, 0x00fe7316, 0x00b6fc70,
  0x00b51cc7, 0x00fd93a1, 0x0025020a, 0x006d8d6c,
  0x00b44992, 0x00fcc6f4, 0x0024575f, 0x006cd839,
  0x006f388e, 0x0027b7e8, 0x00ff2643, 0x00b7a925,
  0x0002aaaa, 0x004a25cc, 0x0092b467, 0x00da3b01,
  0x00d9dbb6, 0x009154d0, 0x0049c57b, 0x00014a1d,
  0x004b5141, 0x0003de27, 0x00db4f8c, 0x0093c0ea,
  0x0090205d, 0x00d8af3b, 0x00003e90, 0x0048b1f6,
  0x00fdb279, 0x00b53d1f, 0x006dacb4, 0x002523d2,
  0x0026c365, 0x006e4c03, 0x00b6dda8, 0x00fe52ce,
  0x00279630, 0x006f1956, 0x00b788fd, 0x00ff079b,
  0x00fce72c, 0x00b4684a, 0x006cf9e1, 0x00247687,
  0x00917508, 0x00d9fa6e, 0x00016bc5, 0x0049e4a3,
  0x004a0414, 0x00028b72, 0x00da1ad9, 0x009295bf,
  0x0093dfa2, 0x00db50c4, 0x0003c16f, 0x004b4e09,
  0x0048aebe, 0x000021d8, 0x00d8b073, 0x00903f15,
  0x00253c9a, 0x006db3fc, 0x00b52257, 0x00fdad31,
  0x00fe4d86, 0x00b6c2e0, 0x006e534b, 0x0026dc2d,
  0x00ff18d3, 0x00b797b5, 0x006f061e, 0x00278978,
  0x002469cf, 0x006ce6a9, 0x00b47702, 0x00fcf864,
  0x0049fbeb, 0x0001748d, 0x00d9e526, 0x00916a40,
  0x00928af7, 0x00da0591, 0x0002943a, 0x004a1b5c,
  0x0096a282, 0x00de2de4, 0x0006bc4f, 0x004e3329,
  0x004dd39e, 0x00055cf8, 0x00ddcd53, 0x00954235,
  0x002041ba, 0x0068cedc, 0x00b05f77, 0x00f8d011,
  0x00fb30a6, 0x00b3bfc0, 0x006b2e6b, 0x0023a10d,
  0x00fa65f3, 0x00b2ea95, 0x006a7b3e, 0x0022f458,
  0x002114ef, 0x00699b89, 0x00b10a22, 0x00f98544,
  0x004c86cb, 0x000409ad, 0x00dc9806, 0x00941760,
  0x0097f7d7, 0x00df78b1, 0x0007e91a, 0x004f667c,
  0x004e2c61, 0x0006a307, 0x00de32ac, 0x0096bdca,
  0x00955d7d, 0x00ddd21b, 0x000543b0, 0x004dccd6,
  0x00f8cf59, 0x00b0403f, 0x0068d194, 0x00205ef2,
  0x0023be45, 0x006b3123, 0x00b3a088, 0x00fb2fee,
  0x0022eb10, 0x006a6476, 0x00b2f5dd, 0x00fa7abb,
  0x00f99a0c, 0x00b1156a, 0x006984c1, 0x00210ba7,
  0x00940828, 0x00dc874e, 0x000416e5, 0x004c9983,
  0x004f7934, 0x0007f652, 0x00df67f9, 0x0097e89f,
  0x00ddf3c3, 0x00957ca5, 0x004ded0e, 0x00056268,
  0x000682df, 0x004e0db9, 0x00969c12, 0x00de1374,
  0x006b10fb, 0x00239f9d, 0x00fb0e36, 0x00b38150,
  0x00b061e7, 0x00f8ee81, 0x00207f2a, 0x0068f04c,
  0x00b134b2, 0x00f9bbd4, 0x00212a7f, 0x0069a519,
  0x006a45ae, 0x0022cac8, 0x00fa5b63, 0x00b2d405,
  0x0007d78a, 0x004f58ec, 0x0097c947, 0x00df4621,
  0x00dca696, 0x009429f0, 0x004cb85b, 0x0004373d,
  0x00057d20, 0x004df246, 0x009563ed, 0x00ddec8b,
  0x00de0c3c, 0x0096835a, 0x004e12f1, 0x00069d97,
  0x00b39e18, 0x00fb117e, 0x002380d5, 0x006b0fb3,
  0x0068ef04, 0x00206062, 0x00f8f1c9, 0x00b07eaf,
  0x0069ba51, 0x00213537, 0x00f9a49c, 0x00b12bfa,
  0x00b2cb4d, 0x00fa442b, 0x0022d580, 0x006a5ae6,
  0x00df5969, 0x0097d60f, 0x004f47a4, 0x0007c8c2,
  0x00042875, 0x004ca713, 0x009436b8, 0x00dcb9de,
  0x00000000, 0x00d70983, 0x00555f80, 0x00825603,
  0x0051f286, 0x0086fb05, 0x0004ad06, 0x00d3a485,
  0x0059a88b, 0x008ea108, 0x000cf70b, 0x00dbfe88,
  0x00085a0d, 0x00df538e, 0x005d058d, 0x008a0c0e,
  0x00491c91, 0x009e1512, 0x001c4311, 0x00cb4a92,
  0x0018ee17, 0x00cfe794, 0x004db197, 0x009ab814,
  0x0010b41a, 0x00c7bd99, 0x0045eb9a, 0x0092e219,
  0x0041469c, 0x00964f1f, 0x0014191c, 0x00c3109f,
  0x006974a4, 0x00be7d27, 0x003c2b24, 0x00eb22a7,
  0x00388622, 0x00ef8fa1, 0x006dd9a2, 0x00bad021,
  0x0030dc2f, 0x00e7d5ac, 0x006583af, 0x00b28a2c,
  0x00612ea9, 0x00b6272a, 0x00347129, 0x00e378aa,
  0x00206835, 0x00f761b6, 0x007537b5, 0x00a23e36,
  0x00719ab3, 0x00a69330, 0x0024c533, 0x00f3ccb0,
  0x0079c0be, 0x00aec93d, 0x002c9f3e, 0x00fb96bd,
  0x00283238, 0x00ff3bbb, 0x007d6db8, 0x00aa643b,
  0x0029a4ce, 0x00fead4d, 0x007cfb4e, 0x00abf2cd,
  0x00785648, 0x00af5fcb, 0x002d09c8, 0x00fa004b,
  0x00700c45, 0x00a705c6, 0x002553c5, 0x00f25a46,
  0x0021fec3, 0x00f6f740, 0x0074a143, 0x00a3a8c0,
  0x0060b85f, 0x00b7b1dc, 0x0035e7df, 0x00e2ee5c,
  0x00314ad9, 0x00e6435a, 0x00641559, 0x00b31cda,
  0x003910d4, 0x00ee1957, 0x006c4f54, 0x00bb46d7,
  0x0068e252, 0x00bfebd1, 0x003dbdd2, 0x00eab451,
  0x0040d06a, 0x0097d9e9, 0x00158fea, 0x00c28669,
  0x001122ec, 0x00c62b6f, 0x00447d6c, 0x009374ef,
  0x001978e1, 0x00ce7162, 0x004c2761, 0x009b2ee2,
  0x00488a67, 0x009f83e4, 0x001dd5e7, 0x00cadc64,
  0x0009ccfb, 0x00dec578, 0x005c937b, 0x008b9af8,
  0x00583e7d, 0x008f37fe, 0x000d61fd, 0x00da687e,
  0x00506470, 0x00876df3, 0x00053bf0, 0x00d23273,
  0x000196f6, 0x00d69f75, 0x0054c976, 0x0083c0f5,
  0x00a9041b, 0x007e0d98, 0x00fc5b9b, 0x002b5218,
  0x00f8f69d, 0x002fff1e, 0x00ada91d, 0x007aa09e,
  0x00f0ac90, 0x0027a513, 0x00a5f310, 0x0072fa93,
  0x00a15e16, 0x00765795, 0x00f40196, 0x00230815,
  0x00e0188a, 0x00371109, 0x00b5470a, 0x00624e89,
  0x00b1ea0c, 0x0066e38f, 0x00e4b58c, 0x0033bc0f,
  0x00b9b001, 0x006eb982, 0x00ecef81, 0x003be602,
  0x00e84287, 0x003f4b04, 0x00bd1d07, 0x006a1484,
  0x00c070bf, 0x0017793c, 0x00952f3f, 0x004226bc,
  0x00918239, 0x00468bba, 0x00c4ddb9, 0x0013d43a,
  0x0099d834, 0x004ed1b7, 0x00cc87b4, 0x001b8e37,
  0x00c82ab2, 0x001f2331, 0x009d7532, 0x004a7cb1,
  0x00896c2e, 0x005e65ad, 0x00dc33ae, 0x000b3a2d,
  0x00d89ea8, 0x000f972b, 0x008dc128, 0x005ac8ab,
  0x00d0c4a5, 0x0007cd26, 0x00859b25, 0x005292a6,
  0x00813623, 0x00563fa0, 0x00d469a3, 0x00036020,
  0x0080a0d5, 0x0057a956, 0x00d5ff55, 0x0002f6d6,
  0x00d15253, 0x00065bd0, 0x00840dd3, 0x00530450,
  0x00d9085e, 0x000e01dd, 0x008c57de, 0x005b5e5d,
  0x0088fad8, 0x005ff35b, 0x00dda558, 0x000aacdb,
  0x00c9bc44, 0x001eb5c7, 0x009ce3c4, 0x004bea47,
  0x00984ec2, 0x004f4741, 0x00cd1142, 0x001a18c1,
  0x009014cf, 0x00471d4c, 0x00c54b4f, 0x001242cc,
  0x00c1e649, 0x0016efca, 0x0094b9c9, 0x0043b04a,
  0x00e9d471, 0x003eddf2, 0x00bc8bf1, 0x006b8272,
  0x00b826f7, 0x006f2f74, 0x00ed7977, 0x003a70f4,
  0x00b07cfa, 0x00677579, 0x00e5237a, 0x00322af9,
  0x00e18e7c, 0x003687ff, 0x00b4d1fc, 0x0063d87f,
  0x00a0c8e0, 0x0077c163, 0x00f59760, 0x00229ee3,
  0x00f13a66, 0x002633e5, 0x00a465e6, 0x00736c65,
  0x00f9606b, 0x002e69e8, 0x00ac3feb, 0x007b3668,
  0x00a892ed, 0x007f9b6e, 0x00fdcd6d, 0x002ac4ee,
  0x00000000, 0x00520936, 0x00a4126c, 0x00f61b5a,
  0x004825d8, 0x001a2cee, 0x00ec37b4, 0x00be3e82,
  0x006b0636, 0x00390f00, 0x00cf145a, 0x009d1d6c,
  0x002323ee, 0x00712ad8, 0x00873182, 0x00d538b4,
  0x00d60c6c, 0x0084055a, 0x00721e00, 0x00201736,
  0x009e29b4, 0x00cc2082, 0x003a3bd8, 0x006832ee,
  0x00bd0a5a, 0x00ef036c, 0x00191836, 0x004b1100,
  0x00f52f82, 0x00a726b4, 0x00513dee, 0x000334d8,
  0x00ac19d8, 0x00fe10ee, 0x00080bb4, 0x005a0282,
  0x00e43c00, 0x00b63536, 0x00402e6c, 0x0012275a,
  0x00c71fee, 0x009516d8, 0x00630d82, 0x003104b4,
  0x008f3a36, 0x00dd3300, 0x002b285a, 0x0079216c,
  0x007a15b4, 0x00281c82, 0x00de07d8, 0x008c0eee,
  0x0032306c, 0x0060395a, 0x00962200, 0x00c42b36,
  0x00111382, 0x00431ab4, 0x00b501ee, 0x00e708d8,
  0x0059365a, 0x000b3f6c, 0x00fd2436, 0x00af2d00,
  0x00a37f36, 0x00f17600, 0x00076d5a, 0x0055646c,
  0x00eb5aee, 0x00b953d8, 0x004f4882, 0x001d41b4,
  0x00c87900, 0x009a7036, 0x006c6b6c, 0x003e625a,
  0x00805cd8, 0x00d255ee, 0x00244eb4, 0x00764782,
  0x0075735a, 0x00277a6c, 0x00d16136, 0x00836800,
  0x003d5682, 0x006f5fb4, 0x009944ee, 0x00cb4dd8,
  0x001e756c, 0x004c7c5a, 0x00ba6700, 0x00e86e36,
  0x005650b4, 0x00045982, 0x00f242d8, 0x00a04bee,
  0x000f66ee, 0x005d6fd8, 0x00ab7482, 0x00f97db4,
  0x00474336, 0x00154a00, 0x00e3515a, 0x00b1586c,
  0x006460d8, 0x003669ee, 0x00c072b4, 0x00927b82,
  0x002c4500, 0x007e4c36, 0x0088576c, 0x00da5e5a,
  0x00d96a82, 0x008b63b4, 0x007d78ee, 0x002f71d8,
  0x00914f5a, 0x00c3466c, 0x00355d36, 0x00675400,
  0x00b26cb4, 0x00e06582, 0x00167ed8, 0x004477ee,
  0x00fa496c, 0x00a8405a, 0x005e5b00, 0x000c5236,
  0x0046ff6c, 0x0014f65a, 0x00e2ed00, 0x00b0e436,
  0x000edab4, 0x005cd382, 0x00aac8d8, 0x00f8c1ee,
  0x002df95a, 0x007ff06c, 0x0089eb36, 0x00dbe200,
  0x0065dc82, 0x0037d5b4, 0x00c1ceee, 0x0093c7d8,
  0x0090f300, 0x00c2fa36, 0x0034e16c, 0x0066e85a,
  0x00d8d6d8, 0x008adfee, 0x007cc4b4, 0x002ecd82,
  0x00fbf536, 0x00a9fc00, 0x005fe75a, 0x000dee6c,
  0x00b3d0ee, 0x00e1d9d8, 0x0017c282, 0x0045cbb4,
  0x00eae6b4, 0x00b8ef82, 0x004ef4d8, 0x001cfdee,
  0x00a2c36c, 0x00f0ca5a, 0x0006d100, 0x0054d836,
  0x0081e082, 0x00d3e9b4, 0x0025f2ee, 0x0077fbd8,
  0x00c9c55a, 0x009bcc6c, 0x006dd736, 0x003fde00,
  0x003cead8, 0x006ee3ee, 0x0098f8b4, 0x00caf182,
  0x0074cf00, 0x0026c636, 0x00d0dd6c, 0x0082d45a,
  0x0057ecee, 0x0005e5d8, 0x00f3fe82, 0x00a1f7b4,
  0x001fc936, 0x004dc000, 0x00bbdb5a, 0x00e9d26c,
  0x00e5805a, 0x00b7896c, 0x00419236, 0x00139b00,
  0x00ada582, 0x00ffacb4, 0x0009b7ee, 0x005bbed8,
  0x008e866c, 0x00dc8f5a, 0x002a9400, 0x00789d36,
  0x00c6a3b4, 0x0094aa82, 0x0062b1d8, 0x0030b8ee,
  0x00338c36, 0x00618500, 0x00979e5a, 0x00c5976c,
  0x007ba9ee, 0x0029a0d8, 0x00dfbb82, 0x008db2b4,
  0x00588a00, 0x000a8336, 0x00fc986c, 0x00ae915a,
  0x0010afd8, 0x0042a6ee, 0x00b4bdb4, 0x00e6b482,
  0x00499982, 0x001b90b4, 0x00ed8bee, 0x00bf82d8,
  0x0001bc5a, 0x0053b56c, 0x00a5ae36, 0x00f7a700,
  0x00229fb4, 0x00709682, 0x00868dd8, 0x00d484ee,
  0x006aba6c, 0x0038b35a, 0x00cea800, 0x009ca136,
  0x009f95ee, 0x00cd9cd8, 0x003b8782, 0x00698eb4,
  0x00d7b036, 0x0085b900, 0x0073a25a, 0x0021ab6c,
  0x00f493d8, 0x00a69aee, 0x005081b4, 0x00028882,
  0x00bcb600, 0x00eebf36, 0x0018a46c, 0x004aad5a
};

const uint32_t crc32_table[8][256] = {
	{
		0x00000000, 0x77073096, 0xEE0E612C, 0x990951BA,
		0x076DC419, 0x706AF48F, 0xE963A535, 0x9E6495A3,
		0x0EDB8832, 0x79DCB8A4, 0xE0D5E91E, 0x97D2D988,
		0x09B64C2B, 0x7EB17CBD, 0xE7B82D07, 0x90BF1D91,
		0x1DB71064, 0x6AB020F2, 0xF3B97148, 0x84BE41DE,
		0x1ADAD47D, 0x6DDDE4EB, 0xF4D4B551, 0x83D385C7,
		0x136C9856, 0x646BA8C0, 0xFD62F97A, 0x8A65C9EC,
		0x14015C4F, 0x63066CD9, 0xFA0F3D63, 0x8D080DF5,
		0x3B6E20C8, 0x4C69105E, 0xD56041E4, 0xA2677172,
		0x3C03E4D1, 0x4B04D447, 0xD20D85FD, 0xA50AB56B,
		0x35B5A8FA, 0x42B2986C, 0xDBBBC9D6, 0xACBCF940,
		0x32D86CE3, 0x45DF5C75, 0xDCD60DCF, 0xABD13D59,
		0x26D930AC, 0x51DE003A, 0xC8D75180, 0xBFD06116,
		0x21B4F4B5, 0x56B3C423, 0xCFBA9599, 0xB8BDA50F,
		0x2802B89E, 0x5F058808, 0xC60CD9B2, 0xB10BE924,
		0x2F6F7C87, 0x58684C11, 0xC1611DAB, 0xB6662D3D,
		0x76DC4190, 0x01DB7106, 0x98D220BC, 0xEFD5102A,
		0x71B18589, 0x06B6B51F, 0x9FBFE4A5, 0xE8B8D433,
		0x7807C9A2, 0x0F00F934, 0x9609A88E, 0xE10E9818,
		0x7F6A0DBB, 0x086D3D2D, 0x91646C97, 0xE6635C01,
		0x6B6B51F4, 0x1C6C6162, 0x856530D8, 0xF262004E,
		0x6C0695ED, 0x1B01A57B, 0x8208F4C1, 0xF50FC457,
		0x65B0D9C6, 0x12B7E950, 0x8BBEB8EA, 0xFCB9887C,
		0x62DD1DDF, 0x15DA2D49, 0x8CD37CF3, 0xFBD44C65,
		0x4DB26158, 0x3AB551CE, 0xA3BC0074, 0xD4BB30E2,
		0x4ADFA541, 0x3DD895D7, 0xA4D1C46D, 0xD3D6F4FB,
		0x4369E96A, 0x346ED9FC, 0xAD678846, 0xDA60B8D0,
		0x44042D73, 0x33031DE5, 0xAA0A4C5F, 0xDD0D7CC9,
		0x5005713C, 0x270241AA, 0xBE0B1010, 0xC90C2086,
		0x5768B525, 0x206F85B3, 0xB966D409, 0xCE61E49F,
		0x5EDEF90E, 0x29D9C998, 0xB0D09822, 0xC7D7A8B4,
		0x59B33D17, 0x2EB40D81, 0xB7BD5C3B, 0xC0BA6CAD,
		0xEDB88320, 0x9ABFB3B6, 0x03B6E20C, 0x74B1D29A,
		0xEAD54739, 0x9DD277AF, 0x04DB2615, 0x73DC1683,
		0xE3630B12, 0x94643B84, 0x0D6D6A3E, 0x7A6A5AA8,
		0xE40ECF0B, 0x9309FF9D, 0x0A00AE27, 0x7D079EB1,
		0xF00F9344, 0x8708A3D2, 0x1E01F268, 0x6906C2FE,
		0xF762575D, 0x806567CB, 0x196C3671, 0x6E6B06E7,
		0xFED41B76, 0x89D32BE0, 0x10DA7A5A, 0x67DD4ACC,
		0xF9B9DF6F, 0x8EBEEFF9, 0x17B7BE43, 0x60B08ED5,
		0xD6D6A3E8, 0xA1D1937E, 0x38D8C2C4, 0x4FDFF252,
		0xD1BB67F1, 0xA6BC5767, 0x3FB506DD, 0x48B2364B,
		0xD80D2BDA, 0xAF0A1B4C, 0x36034AF6, 0x41047A60,
		0xDF60EFC3, 0xA867DF55, 0x316E8EEF, 0x4669BE79,
		0xCB61B38C, 0xBC66831A, 0x256FD2A0, 0x5268E236,
		0xCC0C7795, 0xBB0B4703, 0x220216B9, 0x5505262F,
		0xC5BA3BBE, 0xB2BD0B28, 0x2BB45A92, 0x5CB36A04,
		0xC2D7FFA7, 0xB5D0CF31, 0x2CD99E8B, 0x5BDEAE1D,
		0x9B64C2B0, 0xEC63F226, 0x756AA39C, 0x026D930A,
		0x9C0906A9, 0xEB0E363F, 0x72076785, 0x05005713,
		0x95BF4A82, 0xE2B87A14, 0x7BB12BAE, 0x0CB61B38,
		0x92D28E9B, 0xE5D5BE0D, 0x7CDCEFB7, 0x0BDBDF21,
		0x86D3D2D4, 0xF1D4E242, 0x68DDB3F8, 0x1FDA836E,
		0x81BE16CD, 0xF6B9265B, 0x6FB077E1, 0x18B74777,
		0x88085AE6, 0xFF0F6A70, 0x66063BCA, 0x11010B5C,
		0x8F659EFF, 0xF862AE69, 0x616BFFD3, 0x166CCF45,
		0xA00AE278, 0xD70DD2EE, 0x4E048354, 0x3903B3C2,
		0xA7672661, 0xD06016F7, 0x4969474D, 0x3E6E77DB,
		0xAED16A4A, 0xD9D65ADC, 0x40DF0B66, 0x37D83BF0,
		0xA9BCAE53, 0xDEBB9EC5, 0x47B2CF7F, 0x30B5FFE9,
		0xBDBDF21C, 0xCABAC28A, 0x53B39330, 0x24B4A3A6,
		0xBAD03605, 0xCDD70693, 0x54DE5729, 0x23D967BF,
		0xB3667A2E, 0xC4614AB8, 0x5D681B02, 0x2A6F2B94,
		0xB40BBE37, 0xC30C8EA1, 0x5A05DF1B, 0x2D02EF8D
	}, {
		0x00000000, 0x191B3141, 0x32366282, 0x2B2D53C3,
		0x646CC504, 0x7D77F445, 0x565AA786, 0x4F4196C7,
		0xC8D98A08, 0xD1C2BB49, 0xFAEFE88A, 0xE3F4D9CB,
		0xACB54F0C, 0xB5AE7E4D, 0x9E832D8E, 0x87981CCF,
		0x4AC21251, 0x53D92310, 0x78F470D3, 0x61EF4192,
		0x2EAED755, 0x37B5E614, 0x1C98B5D7, 0x05838496,
		0x821B9859, 0x9B00A918, 0xB02DFADB, 0xA936CB9A,
		0xE6775D5D, 0xFF6C6C1C, 0xD4413FDF, 0xCD5A0E9E,
		0x958424A2, 0x8C9F15E3, 0xA7B24620, 0xBEA97761,
		0xF1E8E1A6, 0xE8F3D0E7, 0xC3DE8324, 0xDAC5B265,
		0x5D5DAEAA, 0x44469FEB, 0x6F6BCC28, 0x7670FD69,
		0x39316BAE, 0x202A5AEF, 0x0B07092C, 0x121C386D,
		0xDF4636F3, 0xC65D07B2, 0xED705471, 0xF46B6530,
		0xBB2AF3F7, 0xA231C2B6, 0x891C9175, 0x9007A034,
		0x179FBCFB, 0x0E848DBA, 0x25A9DE79, 0x3CB2EF38,
		0x73F379FF, 0x6AE848BE, 0x41C51B7D, 0x58DE2A3C,
		0xF0794F05, 0xE9627E44, 0xC24F2D87, 0xDB541CC6,
		0x94158A01, 0x8D0EBB40, 0xA623E883, 0xBF38D9C2,
		0x38A0C50D, 0x21BBF44C, 0x0A96A78F, 0x138D96CE,
		0x5CCC0009, 0x45D73148, 0x6EFA628B, 0x77E153CA,
		0xBABB5D54, 0xA3A06C15, 0x888D3FD6, 0x91960E97,
		0xDED79850, 0xC7CCA911, 0xECE1FAD2, 0xF5FACB93,
		0x7262D75C, 0x6B79E61D, 0x4054B5DE, 0x594F849F,
		0x160E1258, 0x0F152319, 0x243870DA, 0x3D23419B,
		0x65FD6BA7, 0x7CE65AE6, 0x57CB0925, 0x4ED03864,
		0x0191AEA3, 0x188A9FE2, 0x33A7CC21, 0x2ABCFD60,
		0xAD24E1AF, 0xB43FD0EE, 0x9F12832D, 0x8609B26C,
		0xC94824AB, 0xD05315EA, 0xFB7E4629, 0xE2657768,
		0x2F3F79F6, 0x362448B7, 0x1D091B74, 0x04122A35,
		0x4B53BCF2, 0x52488DB3, 0x7965DE70, 0x607EEF31,
		0xE7E6F3FE, 0xFEFDC2BF, 0xD5D0917C, 0xCCCBA03D,
		0x838A36FA, 0x9A9107BB, 0xB1BC5478, 0xA8A76539,
		0x3B83984B, 0x2298A90A, 0x09B5FAC9, 0x10AECB88,
		0x5FEF5D4F, 0x46F46C0E, 0x6DD93FCD, 0x74C20E8C,
		0xF35A1243, 0xEA412302, 0xC16C70C1, 0xD8774180,
		0x9736D747, 0x8E2DE606, 0xA500B5C5, 0xBC1B8484,
		0x71418A1A, 0x685ABB5B, 0x4377E898, 0x5A6CD9D9,
		0x152D4F1E, 0x0C367E5F, 0x271B2D9C, 0x3E001CDD,
		0xB9980012, 0xA0833153, 0x8BAE6290, 0x92B553D1,
		0xDDF4C516, 0xC4EFF457, 0xEFC2A794, 0xF6D996D5,
		0xAE07BCE9, 0xB71C8DA8, 0x9C31DE6B, 0x852AEF2A,
		0xCA6B79ED, 0xD37048AC, 0xF85D1B6F, 0xE1462A2E,
		0x66DE36E1, 0x7FC507A0, 0x54E85463, 0x4DF36522,
		0x02B2F3E5, 0x1BA9C2A4, 0x30849167, 0x299FA026,
		0xE4C5AEB8, 0xFDDE9FF9, 0xD6F3CC3A, 0xCFE8FD7B,
		0x80A96BBC, 0x99B25AFD, 0xB29F093E, 0xAB84387F,
		0x2C1C24B0, 0x350715F1, 0x1E2A4632, 0x07317773,
		0x4870E1B4, 0x516BD0F5, 0x7A468336, 0x635DB277,
		0xCBFAD74E, 0xD2E1E60F, 0xF9CCB5CC, 0xE0D7848D,
		0xAF96124A, 0xB68D230B, 0x9DA070C8, 0x84BB4189,
		0x03235D46, 0x1A386C07, 0x31153FC4, 0x280E0E85,
		0x674F9842, 0x7E54A903, 0x5579FAC0, 0x4C62CB81,
		0x8138C51F, 0x9823F45E, 0xB30EA79D, 0xAA1596DC,
		0xE554001B, 0xFC4F315A, 0xD7626299, 0xCE7953D8,
		0x49E14F17, 0x50FA7E56, 0x7BD72D95, 0x62CC1CD4,
		0x2D8D8A13, 0x3496BB52, 0x1FBBE891, 0x06A0D9D0,
		0x5E7EF3EC, 0x4765C2AD, 0x6C48916E, 0x7553A02F,
		0x3A1236E8, 0x230907A9, 0x0824546A, 0x113F652B,
		0x96A779E4, 0x8FBC48A5, 0xA4911B66, 0xBD8A2A27,
		0xF2CBBCE0, 0xEBD08DA1, 0xC0FDDE62, 0xD9E6EF23,
		0x14BCE1BD, 0x0DA7D0FC, 0x268A833F, 0x3F91B27E,
		0x70D024B9, 0x69CB15F8, 0x42E6463B, 0x5BFD777A,
		0xDC656BB5, 0xC57E5AF4, 0xEE530937, 0xF7483876,
		0xB809AEB1, 0xA1129FF0, 0x8A3FCC33, 0x9324FD72
	}, {
		0x00000000, 0x01C26A37, 0x0384D46E, 0x0246BE59,
		0x0709A8DC, 0x06CBC2EB, 0x048D7CB2, 0x054F1685,
		0x0E1351B8, 0x0FD13B8F, 0x0D9785D6, 0x0C55EFE1,
		0x091AF964, 0x08D89353, 0x0A9E2D0A, 0x0B5C473D,
		0x1C26A370, 0x1DE4C947, 0x1FA2771E, 0x1E601D29,
		0x1B2F0BAC, 0x1AED619B, 0x18ABDFC2, 0x1969B5F5,
		0x1235F2C8, 0x13F798FF, 0x11B126A6, 0x10734C91,
		0x153C5A14, 0x14FE3023, 0x16B88E7A, 0x177AE44D,
		0x384D46E0, 0x398F2CD7, 0x3BC9928E, 0x3A0BF8B9,
		0x3F44EE3C, 0x3E86840B, 0x3CC03A52, 0x3D025065,
		0x365E1758, 0x379C7D6F, 0x35DAC336, 0x3418A901,
		0x3157BF84, 0x3095D5B3, 0x32D36BEA, 0x331101DD,
		0x246BE590, 0x25A98FA7, 0x27EF31FE, 0x262D5BC9,
		0x23624D4C, 0x22A0277B, 0x20E69922, 0x2124F315,
		0x2A78B428, 0x2BBADE1F, 0x29FC6046, 0x283E0A71,
		0x2D711CF4, 0x2CB376C3, 0x2EF5C89A, 0x2F37A2AD,
		0x709A8DC0, 0x7158E7F7, 0x731E59AE, 0x72DC3399,
		0x7793251C, 0x76514F2B, 0x7417F172, 0x75D59B45,
		0x7E89DC78, 0x7F4BB64F, 0x7D0D0816, 0x7CCF6221,
		0x798074A4, 0x78421E93, 0x7A04A0CA, 0x7BC6CAFD,
		0x6CBC2EB0, 0x6D7E4487, 0x6F38FADE, 0x6EFA90E9,
		0x6BB5866C, 0x6A77EC5B, 0x68315202, 0x69F33835,
		0x62AF7F08, 0x636D153F, 0x612BAB66, 0x60E9C151,
		0x65A6D7D4, 0x6464BDE3, 0x662203BA, 0x67E0698D,
		0x48D7CB20, 0x4915A117, 0x4B531F4E, 0x4A917579,
		0x4FDE63FC, 0x4E1C09CB, 0x4C5AB792, 0x4D98DDA5,
		0x46C49A98, 0x4706F0AF, 0x45404EF6, 0x448224C1,
		0x41CD3244, 0x400F5873, 0x4249E62A, 0x438B8C1D,
		0x54F16850, 0x55330267, 0x5775BC3E, 0x56B7D609,
		0x53F8C08C, 0x523AAABB, 0x507C14E2, 0x51BE7ED5,
		0x5AE239E8, 0x5B2053DF, 0x5966ED86, 0x58A487B1,
		0x5DEB9134, 0x5C29FB03, 0x5E6F455A, 0x5FAD2F6D,
		0xE1351B80, 0xE0F771B7, 0xE2B1CFEE, 0xE373A5D9,
		0xE63CB35C, 0xE7FED96B, 0xE5B86732, 0xE47A0D05,
		0xEF264A38, 0xEEE4200F, 0xECA29E56, 0xED60F461,
		0xE82FE2E4, 0xE9ED88D3, 0xEBAB368A, 0xEA695CBD,
		0xFD13B8F0, 0xFCD1D2C7, 0xFE976C9E, 0xFF5506A9,
		0xFA1A102C, 0xFBD87A1B, 0xF99EC442, 0xF85CAE75,
		0xF300E948, 0xF2C2837F, 0xF0843D26, 0xF1465711,
		0xF4094194, 0xF5CB2BA3, 0xF78D95FA, 0xF64FFFCD,
		0xD9785D60, 0xD8BA3757, 0xDAFC890E, 0xDB3EE339,
		0xDE71F5BC, 0xDFB39F8B, 0xDDF521D2, 0xDC374BE5,
		0xD76B0CD8, 0xD6A966EF, 0xD4EFD8B6, 0xD52DB281,
		0xD062A404, 0xD1A0CE33, 0xD3E6706A, 0xD2241A5D,
		0xC55EFE10, 0xC49C9427, 0xC6DA2A7E, 0xC7184049,
		0xC25756CC, 0xC3953CFB, 0xC1D382A2, 0xC011E895,
		0xCB4DAFA8, 0xCA8FC59F, 0xC8C97BC6, 0xC90B11F1,
		0xCC440774, 0xCD866D43, 0xCFC0D31A, 0xCE02B92D,
		0x91AF9640, 0x906DFC77, 0x922B422E, 0x93E92819,
		0x96A63E9C, 0x976454AB, 0x9522EAF2, 0x94E080C5,
		0x9FBCC7F8, 0x9E7EADCF, 0x9C381396, 0x9DFA79A1,
		0x98B56F24, 0x99770513, 0x9B31BB4A, 0x9AF3D17D,
		0x8D893530, 0x8C4B5F07, 0x8E0DE15E, 0x8FCF8B69,
		0x8A809DEC, 0x8B42F7DB, 0x89044982, 0x88C623B5,
		0x839A6488, 0x82580EBF, 0x801EB0E6, 0x81DCDAD1,
		0x8493CC54, 0x8551A663, 0x8717183A, 0x86D5720D,
		0xA9E2D0A0, 0xA820BA97, 0xAA6604CE, 0xABA46EF9,
		0xAEEB787C, 0xAF29124B, 0xAD6FAC12, 0xACADC625,
		0xA7F18118, 0xA633EB2F, 0xA4755576, 0xA5B73F41,
		0xA0F829C4, 0xA13A43F3, 0xA37CFDAA, 0xA2BE979D,
		0xB5C473D0, 0xB40619E7, 0xB640A7BE, 0xB782CD89,
		0xB2CDDB0C, 0xB30FB13B, 0xB1490F62, 0xB08B6555,
		0xBBD72268, 0xBA15485F, 0xB853F606, 0xB9919C31,
		0xBCDE8AB4, 0xBD1CE083, 0xBF5A5EDA, 0xBE9834ED
	}, {
		0x00000000, 0xB8BC6765, 0xAA09C88B, 0x12B5AFEE,
		0x8F629757, 0x37DEF032, 0x256B5FDC, 0x9DD738B9,
		0xC5B428EF, 0x7D084F8A, 0x6FBDE064, 0xD7018701,
		0x4AD6BFB8, 0xF26AD8DD, 0xE0DF7733, 0x58631056,
		0x5019579F, 0xE8A530FA, 0xFA109F14, 0x42ACF871,
		0xDF7BC0C8, 0x67C7A7AD, 0x75720843, 0xCDCE6F26,
		0x95AD7F70, 0x2D111815, 0x3FA4B7FB, 0x8718D09E,
		0x1ACFE827, 0xA2738F42, 0xB0C620AC, 0x087A47C9,
		0xA032AF3E, 0x188EC85B, 0x0A3B67B5, 0xB28700D0,
		0x2F503869, 0x97EC5F0C, 0x8559F0E2, 0x3DE59787,
		0x658687D1, 0xDD3AE0B4, 0xCF8F4F5A, 0x7733283F,
		0xEAE41086, 0x525877E3, 0x40EDD80D, 0xF851BF68,
		0xF02BF8A1, 0x48979FC4, 0x5A22302A, 0xE29E574F,
		0x7F496FF6, 0xC7F50893, 0xD540A77D, 0x6DFCC018,
		0x359FD04E, 0x8D23B72B, 0x9F9618C5, 0x272A7FA0,
		0xBAFD4719, 0x0241207C, 0x10F48F92, 0xA848E8F7,
		0x9B14583D, 0x23A83F58, 0x311D90B6, 0x89A1F7D3,
		0x1476CF6A, 0xACCAA80F, 0xBE7F07E1, 0x06C36084,
		0x5EA070D2, 0xE61C17B7, 0xF4A9B859, 0x4C15DF3C,
		0xD1C2E785, 0x697E80E0, 0x7BCB2F0E, 0xC377486B,
		0xCB0D0FA2, 0x73B168C7, 0x6104C729, 0xD9B8A04C,
		0x446F98F5, 0xFCD3FF90, 0xEE66507E, 0x56DA371B,
		0x0EB9274D, 0xB6054028, 0xA4B0EFC6, 0x1C0C88A3,
		0x81DBB01A, 0x3967D77F, 0x2BD27891, 0x936E1FF4,
		0x3B26F703, 0x839A9066, 0x912F3F88, 0x299358ED,
		0xB4446054, 0x0CF80731, 0x1E4DA8DF, 0xA6F1CFBA,
		0xFE92DFEC, 0x462EB889, 0x549B1767, 0xEC277002,
		0x71F048BB, 0xC94C2FDE, 0xDBF98030, 0x6345E755,
		0x6B3FA09C, 0xD383C7F9, 0xC1366817, 0x798A0F72,
		0xE45D37CB, 0x5CE150AE, 0x4E54FF40, 0xF6E89825,
		0xAE8B8873, 0x1637EF16, 0x048240F8, 0xBC3E279D,
		0x21E91F24, 0x99557841, 0x8BE0D7AF, 0x335CB0CA,
		0xED59B63B, 0x55E5D15E, 0x47507EB0, 0xFFEC19D5,
		0x623B216C, 0xDA874609, 0xC832E9E7, 0x708E8E82,
		0x28ED9ED4, 0x9051F9B1, 0x82E4565F, 0x3A58313A,
		0xA78F0983, 0x1F336EE6, 0x0D86C108, 0xB53AA66D,
		0xBD40E1A4, 0x05FC86C1, 0x1749292F, 0xAFF54E4A,
		0x322276F3, 0x8A9E1196, 0x982BBE78, 0x2097D91D,
		0x78F4C94B, 0xC048AE2E, 0xD2FD01C0, 0x6A4166A5,
		0xF7965E1C, 0x4F2A3979, 0x5D9F9697, 0xE523F1F2,
		0x4D6B1905, 0xF5D77E60, 0xE762D18E, 0x5FDEB6EB,
		0xC2098E52, 0x7AB5E937, 0x680046D9, 0xD0BC21BC,
		0x88DF31EA, 0x3063568F, 0x22D6F961, 0x9A6A9E04,
		0x07BDA6BD, 0xBF01C1D8, 0xADB46E36, 0x15080953,
		0x1D724E9A, 0xA5CE29FF, 0xB77B8611, 0x0FC7E174,
		0x9210D9CD, 0x2AACBEA8, 0x38191146, 0x80A57623,
		0xD8C66675, 0x607A0110, 0x72CFAEFE, 0xCA73C99B,
		0x57A4F122, 0xEF189647, 0xFDAD39A9, 0x45115ECC,
		0x764DEE06, 0xCEF18963, 0xDC44268D, 0x64F841E8,
		0xF92F7951, 0x41931E34, 0x5326B1DA, 0xEB9AD6BF,
		0xB3F9C6E9, 0x0B45A18C, 0x19F00E62, 0xA14C6907,
		0x3C9B51BE, 0x842736DB, 0x96929935, 0x2E2EFE50,
		0x2654B999, 0x9EE8DEFC, 0x8C5D7112, 0x34E11677,
		0xA9362ECE, 0x118A49AB, 0x033FE645, 0xBB838120,
		0xE3E09176, 0x5B5CF613, 0x49E959FD, 0xF1553E98,
		0x6C820621, 0xD43E6144, 0xC68BCEAA, 0x7E37A9CF,
		0xD67F4138, 0x6EC3265D, 0x7C7689B3, 0xC4CAEED6,
		0x591DD66F, 0xE1A1B10A, 0xF3141EE4, 0x4BA87981,
		0x13CB69D7, 0xAB770EB2, 0xB9C2A15C, 0x017EC639,
		0x9CA9FE80, 0x241599E5, 0x36A0360B, 0x8E1C516E,
		0x866616A7, 0x3EDA71C2, 0x2C6FDE2C, 0x94D3B949,
		0x090481F0, 0xB1B8E695, 0xA30D497B, 0x1BB12E1E,
		0x43D23E48, 0xFB6E592D, 0xE9DBF6C3, 0x516791A6,
		0xCCB0A91F, 0x740CCE7A, 0x66B96194, 0xDE0506F1
	}, {
		0x00000000, 0x3D6029B0, 0x7AC05360, 0x47A07AD0,
		0xF580A6C0, 0xC8E08F70, 0x8F40F5A0, 0xB220DC10,
		0x30704BC1, 0x0D106271, 0x4AB018A1, 0x77D03111,
		0xC5F0ED01, 0xF890C4B1, 0xBF30BE61, 0x825097D1,
		0x60E09782, 0x5D80BE32, 0x1A20C4E2, 0x2740ED52,
		0x95603142, 0xA80018F2, 0xEFA06222, 0xD2C04B92,
		0x5090DC43, 0x6DF0F5F3, 0x2A508F23, 0x1730A693,
		0xA5107A83, 0x98705333, 0xDFD029E3, 0xE2B00053,
		0xC1C12F04, 0xFCA106B4, 0xBB017C64, 0x866155D4,
		0x344189C4, 0x0921A074, 0x4E81DAA4, 0x73E1F314,
		0xF1B164C5, 0xCCD14D75, 0x8B7137A5, 0xB6111E15,
		0x0431C205, 0x3951EBB5, 0x7EF19165, 0x4391B8D5,
		0xA121B886, 0x9C419136, 0xDBE1EBE6, 0xE681C256,
		0x54A11E46, 0x69C137F6, 0x2E614D26, 0x13016496,
		0x9151F347, 0xAC31DAF7, 0xEB91A027, 0xD6F18997,
		0x64D15587, 0x59B17C37, 0x1E1106E7, 0x23712F57,
		0x58F35849, 0x659371F9, 0x22330B29, 0x1F532299,
		0xAD73FE89, 0x9013D739, 0xD7B3ADE9, 0xEAD38459,
		0x68831388, 0x55E33A38, 0x124340E8, 0x2F236958,
		0x9D03B548, 0xA0639CF8, 0xE7C3E628, 0xDAA3CF98,
		0x3813CFCB, 0x0573E67B, 0x42D39CAB, 0x7FB3B51B,
		0xCD93690B, 0xF0F340BB, 0xB7533A6B, 0x8A3313DB,
		0x0863840A, 0x3503ADBA, 0x72A3D76A, 0x4FC3FEDA,
		0xFDE322CA, 0xC0830B7A, 0x872371AA, 0xBA43581A,
		0x9932774D, 0xA4525EFD, 0xE3F2242D, 0xDE920D9D,
		0x6CB2D18D, 0x51D2F83D, 0x167282ED, 0x2B12AB5D,
		0xA9423C8C, 0x9422153C, 0xD3826FEC, 0xEEE2465C,
		0x5CC29A4C, 0x61A2B3FC, 0x2602C92C, 0x1B62E09C,
		0xF9D2E0CF, 0xC4B2C97F, 0x8312B3AF, 0xBE729A1F,
		0x0C52460F, 0x31326FBF, 0x7692156F, 0x4BF23CDF,
		0xC9A2AB0E, 0xF4C282BE, 0xB362F86E, 0x8E02D1DE,
		0x3C220DCE, 0x0142247E, 0x46E25EAE, 0x7B82771E,
		0xB1E6B092, 0x8C869922, 0xCB26E3F2, 0xF646CA42,
		0x44661652, 0x79063FE2, 0x3EA64532, 0x03C66C82,
		0x8196FB53, 0xBCF6D2E3, 0xFB56A833, 0xC6368183,
		0x74165D93, 0x49767423, 0x0ED60EF3, 0x33B62743,
		0xD1062710, 0xEC660EA0, 0xABC67470, 0x96A65DC0,
		0x248681D0, 0x19E6A860, 0x5E46D2B0, 0x6326FB00,
		0xE1766CD1, 0xDC164561, 0x9BB63FB1, 0xA6D61601,
		0x14F6CA11, 0x2996E3A1, 0x6E369971, 0x5356B0C1,
		0x70279F96, 0x4D47B626, 0x0AE7CCF6, 0x3787E546,
		0x85A73956, 0xB8C710E6, 0xFF676A36, 0xC2074386,
		0x4057D457, 0x7D37FDE7, 0x3A978737, 0x07F7AE87,
		0xB5D77297, 0x88B75B27, 0xCF1721F7, 0xF2770847,
		0x10C70814, 0x2DA721A4, 0x6A075B74, 0x576772C4,
		0xE547AED4, 0xD8278764, 0x9F87FDB4, 0xA2E7D404,
		0x20B743D5, 0x1DD76A65, 0x5A7710B5, 0x67173905,
		0xD537E515, 0xE857CCA5, 0xAFF7B675, 0x92979FC5,
		0xE915E8DB, 0xD475C16B, 0x93D5BBBB, 0xAEB5920B,
		0x1C954E1B, 0x21F567AB, 0x66551D7B, 0x5B3534CB,
		0xD965A31A, 0xE4058AAA, 0xA3A5F07A, 0x9EC5D9CA,
		0x2CE505DA, 0x11852C6A, 0x562556BA, 0x6B457F0A,
		0x89F57F59, 0xB49556E9, 0xF3352C39, 0xCE550589,
		0x7C75D999, 0x4115F029, 0x06B58AF9, 0x3BD5A349,
		0xB9853498, 0x84E51D28, 0xC34567F8, 0xFE254E48,
		0x4C059258, 0x7165BBE8, 0x36C5C138, 0x0BA5E888,
		0x28D4C7DF, 0x15B4EE6F, 0x521494BF, 0x6F74BD0F,
		0xDD54611F, 0xE03448AF, 0xA794327F, 0x9AF41BCF,
		0x18A48C1E, 0x25C4A5AE, 0x6264DF7E, 0x5F04F6CE,
		0xED242ADE, 0xD044036E, 0x97E479BE, 0xAA84500E,
		0x4834505D, 0x755479ED, 0x32F4033D, 0x0F942A8D,
		0xBDB4F69D, 0x80D4DF2D, 0xC774A5FD, 0xFA148C4D,
		0x78441B9C, 0x4524322C, 0x028448FC, 0x3FE4614C,
		0x8DC4BD5C, 0xB0A494EC, 0xF704EE3C, 0xCA64C78C
	}, {
		0x00000000, 0xCB5CD3A5, 0x4DC8A10B, 0x869472AE,
		0x9B914216, 0x50CD91B3, 0xD659E31D, 0x1D0530B8,
		0xEC53826D, 0x270F51C8, 0xA19B2366, 0x6AC7F0C3,
		0x77C2C07B, 0xBC9E13DE, 0x3A0A6170, 0xF156B2D5,
		0x03D6029B, 0xC88AD13E, 0x4E1EA390, 0x85427035,
		0x9847408D, 0x531B9328, 0xD58FE186, 0x1ED33223,
		0xEF8580F6, 0x24D95353, 0xA24D21FD, 0x6911F258,
		0x7414C2E0, 0xBF481145, 0x39DC63EB, 0xF280B04E,
		0x07AC0536, 0xCCF0D693, 0x4A64A43D, 0x81387798,
		0x9C3D4720, 0x57619485, 0xD1F5E62B, 0x1AA9358E,
		0xEBFF875B, 0x20A354FE, 0xA6372650, 0x6D6BF5F5,
		0x706EC54D, 0xBB3216E8, 0x3DA66446, 0xF6FAB7E3,
		0x047A07AD, 0xCF26D408, 0x49B2A6A6, 0x82EE7503,
		0x9FEB45BB, 0x54B7961E, 0xD223E4B0, 0x197F3715,
		0xE82985C0, 0x23755665, 0xA5E124CB, 0x6EBDF76E,
		0x73B8C7D6, 0xB8E41473, 0x3E7066DD, 0xF52CB578,
		0x0F580A6C, 0xC404D9C9, 0x4290AB67, 0x89CC78C2,
		0x94C9487A, 0x5F959BDF, 0xD901E971, 0x125D3AD4,
		0xE30B8801, 0x28575BA4, 0xAEC3290A, 0x659FFAAF,
		0x789ACA17, 0xB3C619B2, 0x35526B1C, 0xFE0EB8B9,
		0x0C8E08F7, 0xC7D2DB52, 0x4146A9FC, 0x8A1A7A59,
		0x971F4AE1, 0x5C439944, 0xDAD7EBEA, 0x118B384F,
		0xE0DD8A9A, 0x2B81593F, 0xAD152B91, 0x6649F834,
		0x7B4CC88C, 0xB0101B29, 0x36846987, 0xFDD8BA22,
		0x08F40F5A, 0xC3A8DCFF, 0x453CAE51, 0x8E607DF4,
		0x93654D4C, 0x58399EE9, 0xDEADEC47, 0x15F13FE2,
		0xE4A78D37, 0x2FFB5E92, 0xA96F2C3C, 0x6233FF99,
		0x7F36CF21, 0xB46A1C84, 0x32FE6E2A, 0xF9A2BD8F,
		0x0B220DC1, 0xC07EDE64, 0x46EAACCA, 0x8DB67F6F,
		0x90B34FD7, 0x5BEF9C72, 0xDD7BEEDC, 0x16273D79,
		0xE7718FAC, 0x2C2D5C09, 0xAAB92EA7, 0x61E5FD02,
		0x7CE0CDBA, 0xB7BC1E1F, 0x31286CB1, 0xFA74BF14,
		0x1EB014D8, 0xD5ECC77D, 0x5378B5D3, 0x98246676,
		0x852156CE, 0x4E7D856B, 0xC8E9F7C5, 0x03B52460,
		0xF2E396B5, 0x39BF4510, 0xBF2B37BE, 0x7477E41B,
		0x6972D4A3, 0xA22E0706, 0x24BA75A8, 0xEFE6A60D,
		0x1D661643, 0xD63AC5E6, 0x50AEB748, 0x9BF264ED,
		0x86F75455, 0x4DAB87F0, 0xCB3FF55E, 0x006326FB,
		0xF135942E, 0x3A69478B, 0xBCFD3525, 0x77A1E680,
		0x6AA4D638, 0xA1F8059D, 0x276C7733, 0xEC30A496,
		0x191C11EE, 0xD240C24B, 0x54D4B0E5, 0x9F886340,
		0x828D53F8, 0x49D1805D, 0xCF45F2F3, 0x04192156,
		0xF54F9383, 0x3E134026, 0xB8873288, 0x73DBE12D,
		0x6EDED195, 0xA5820230, 0x2316709E, 0xE84AA33B,
		0x1ACA1375, 0xD196C0D0, 0x5702B27E, 0x9C5E61DB,
		0x815B5163, 0x4A0782C6, 0xCC93F068, 0x07CF23CD,
		0xF6999118, 0x3DC542BD, 0xBB513013, 0x700DE3B6,
		0x6D08D30E, 0xA65400AB, 0x20C07205, 0xEB9CA1A0,
		0x11E81EB4, 0xDAB4CD11, 0x5C20BFBF, 0x977C6C1A,
		0x8A795CA2, 0x41258F07, 0xC7B1FDA9, 0x0CED2E0C,
		0xFDBB9CD9, 0x36E74F7C, 0xB0733DD2, 0x7B2FEE77,
		0x662ADECF, 0xAD760D6A, 0x2BE27FC4, 0xE0BEAC61,
		0x123E1C2F, 0xD962CF8A, 0x5FF6BD24, 0x94AA6E81,
		0x89AF5E39, 0x42F38D9C, 0xC467FF32, 0x0F3B2C97,
		0xFE6D9E42, 0x35314DE7, 0xB3A53F49, 0x78F9ECEC,
		0x65FCDC54, 0xAEA00FF1, 0x28347D5F, 0xE368AEFA,
		0x16441B82, 0xDD18C827, 0x5B8CBA89, 0x90D0692C,
		0x8DD55994, 0x46898A31, 0xC01DF89F, 0x0B412B3A,
		0xFA1799EF, 0x314B4A4A, 0xB7DF38E4, 0x7C83EB41,
		0x6186DBF9, 0xAADA085C, 0x2C4E7AF2, 0xE712A957,
		0x15921919, 0xDECECABC, 0x585AB812, 0x93066BB7,
		0x8E035B0F, 0x455F88AA, 0xC3CBFA04, 0x089729A1,
		0xF9C19B74, 0x329D48D1, 0xB4093A7F, 0x7F55E9DA,
		0x6250D962, 0xA90C0AC7, 0x2F987869, 0xE4C4ABCC
	}, {
		0x00000000, 0xA6770BB4, 0x979F1129, 0x31E81A9D,
		0xF44F2413, 0x52382FA7, 0x63D0353A, 0xC5A73E8E,
		0x33EF4E67, 0x959845D3, 0xA4705F4E, 0x020754FA,
		0xC7A06A74, 0x61D761C0, 0x503F7B5D, 0xF64870E9,
		0x67DE9CCE, 0xC1A9977A, 0xF0418DE7, 0x56368653,
		0x9391B8DD, 0x35E6B369, 0x040EA9F4, 0xA279A240,
		0x5431D2A9, 0xF246D91D, 0xC3AEC380, 0x65D9C834,
		0xA07EF6BA, 0x0609FD0E, 0x37E1E793, 0x9196EC27,
		0xCFBD399C, 0x69CA3228, 0x582228B5, 0xFE552301,
		0x3BF21D8F, 0x9D85163B, 0xAC6D0CA6, 0x0A1A0712,
		0xFC5277FB, 0x5A257C4F, 0x6BCD66D2, 0xCDBA6D66,
		0x081D53E8, 0xAE6A585C, 0x9F8242C1, 0x39F54975,
		0xA863A552, 0x0E14AEE6, 0x3FFCB47B, 0x998BBFCF,
		0x5C2C8141, 0xFA5B8AF5, 0xCBB39068, 0x6DC49BDC,
		0x9B8CEB35, 0x3DFBE081, 0x0C13FA1C, 0xAA64F1A8,
		0x6FC3CF26, 0xC9B4C492, 0xF85CDE0F, 0x5E2BD5BB,
		0x440B7579, 0xE27C7ECD, 0xD3946450, 0x75E36FE4,
		0xB044516A, 0x16335ADE, 0x27DB4043, 0x81AC4BF7,
		0x77E43B1E, 0xD19330AA, 0xE07B2A37, 0x460C2183,
		0x83AB1F0D, 0x25DC14B9, 0x14340E24, 0xB2430590,
		0x23D5E9B7, 0x85A2E203, 0xB44AF89E, 0x123DF32A,
		0xD79ACDA4, 0x71EDC610, 0x4005DC8D, 0xE672D739,
		0x103AA7D0, 0xB64DAC64, 0x87A5B6F9, 0x21D2BD4D,
		0xE47583C3, 0x42028877, 0x73EA92EA, 0xD59D995E,
		0x8BB64CE5, 0x2DC14751, 0x1C295DCC, 0xBA5E5678,
		0x7FF968F6, 0xD98E6342, 0xE86679DF, 0x4E11726B,
		0xB8590282, 0x1E2E0936, 0x2FC613AB, 0x89B1181F,
		0x4C162691, 0xEA612D25, 0xDB8937B8, 0x7DFE3C0C,
		0xEC68D02B, 0x4A1FDB9F, 0x7BF7C102, 0xDD80CAB6,
		0x1827F438, 0xBE50FF8C, 0x8FB8E511, 0x29CFEEA5,
		0xDF879E4C, 0x79F095F8, 0x48188F65, 0xEE6F84D1,
		0x2BC8BA5F, 0x8DBFB1EB, 0xBC57AB76, 0x1A20A0C2,
		0x8816EAF2, 0x2E61E146, 0x1F89FBDB, 0xB9FEF06F,
		0x7C59CEE1, 0xDA2EC555, 0xEBC6DFC8, 0x4DB1D47C,
		0xBBF9A495, 0x1D8EAF21, 0x2C66B5BC, 0x8A11BE08,
		0x4FB68086, 0xE9C18B32, 0xD82991AF, 0x7E5E9A1B,
		0xEFC8763C, 0x49BF7D88, 0x78576715, 0xDE206CA1,
		0x1B87522F, 0xBDF0599B, 0x8C184306, 0x2A6F48B2,
		0xDC27385B, 0x7A5033EF, 0x4BB82972, 0xEDCF22C6,
		0x28681C48, 0x8E1F17FC, 0xBFF70D61, 0x198006D5,
		0x47ABD36E, 0xE1DCD8DA, 0xD034C247, 0x7643C9F3,
		0xB3E4F77D, 0x1593FCC9, 0x247BE654, 0x820CEDE0,
		0x74449D09, 0xD23396BD, 0xE3DB8C20, 0x45AC8794,
		0x800BB91A, 0x267CB2AE, 0x1794A833, 0xB1E3A387,
		0x20754FA0, 0x86024414, 0xB7EA5E89, 0x119D553D,
		0xD43A6BB3, 0x724D6007, 0x43A57A9A, 0xE5D2712E,
		0x139A01C7, 0xB5ED0A73, 0x840510EE, 0x22721B5A,
		0xE7D525D4, 0x41A22E60, 0x704A34FD, 0xD63D3F49,
		0xCC1D9F8B, 0x6A6A943F, 0x5B828EA2, 0xFDF58516,
		0x3852BB98, 0x9E25B02C, 0xAFCDAAB1, 0x09BAA105,
		0xFFF2D1EC, 0x5985DA58, 0x686DC0C5, 0xCE1ACB71,
		0x0BBDF5FF, 0xADCAFE4B, 0x9C22E4D6, 0x3A55EF62,
		0xABC30345, 0x0DB408F1, 0x3C5C126C, 0x9A2B19D8,
		0x5F8C2756, 0xF9FB2CE2, 0xC813367F, 0x6E643DCB,
		0x982C4D22, 0x3E5B4696, 0x0FB35C0B, 0xA9C457BF,
		0x6C636931, 0xCA146285, 0xFBFC7818, 0x5D8B73AC,
		0x03A0A617, 0xA5D7ADA3, 0x943FB73E, 0x3248BC8A,
		0xF7EF8204, 0x519889B0, 0x6070932D, 0xC6079899,
		0x304FE870, 0x9638E3C4, 0xA7D0F959, 0x01A7F2ED,
		0xC400CC63, 0x6277C7D7, 0x539FDD4A, 0xF5E8D6FE,
		0x647E3AD9, 0xC209316D, 0xF3E12BF0, 0x55962044,
		0x90311ECA, 0x3646157E, 0x07AE0FE3, 0xA1D90457,
		0x579174BE, 0xF1E67F0A, 0xC00E6597, 0x66796E23,
		0xA3DE50AD, 0x05A95B19, 0x34414184, 0x92364A30
	}, {
		0x00000000, 0xCCAA009E, 0x4225077D, 0x8E8F07E3,
		0x844A0EFA, 0x48E00E64, 0xC66F0987, 0x0AC50919,
		0xD3E51BB5, 0x1F4F1B2B, 0x91C01CC8, 0x5D6A1C56,
		0x57AF154F, 0x9B0515D1, 0x158A1232, 0xD92012AC,
		0x7CBB312B, 0xB01131B5, 0x3E9E3656, 0xF23436C8,
		0xF8F13FD1, 0x345B3F4F, 0xBAD438AC, 0x767E3832,
		0xAF5E2A9E, 0x63F42A00, 0xED7B2DE3, 0x21D12D7D,
		0x2B142464, 0xE7BE24FA, 0x69312319, 0xA59B2387,
		0xF9766256, 0x35DC62C8, 0xBB53652B, 0x77F965B5,
		0x7D3C6CAC, 0xB1966C32, 0x3F196BD1, 0xF3B36B4F,
		0x2A9379E3, 0xE639797D, 0x68B67E9E, 0xA41C7E00,
		0xAED97719, 0x62737787, 0xECFC7064, 0x205670FA,
		0x85CD537D, 0x496753E3, 0xC7E85400, 0x0B42549E,
		0x01875D87, 0xCD2D5D19, 0x43A25AFA, 0x8F085A64,
		0x562848C8, 0x9A824856, 0x140D4FB5, 0xD8A74F2B,
		0xD2624632, 0x1EC846AC, 0x9047414F, 0x5CED41D1,
		0x299DC2ED, 0xE537C273, 0x6BB8C590, 0xA712C50E,
		0xADD7CC17, 0x617DCC89, 0xEFF2CB6A, 0x2358CBF4,
		0xFA78D958, 0x36D2D9C6, 0xB85DDE25, 0x74F7DEBB,
		0x7E32D7A2, 0xB298D73C, 0x3C17D0DF, 0xF0BDD041,
		0x5526F3C6, 0x998CF358, 0x1703F4BB, 0xDBA9F425,
		0xD16CFD3C, 0x1DC6FDA2, 0x9349FA41, 0x5FE3FADF,
		0x86C3E873, 0x4A69E8ED, 0xC4E6EF0E, 0x084CEF90,
		0x0289E689, 0xCE23E617, 0x40ACE1F4, 0x8C06E16A,
		0xD0EBA0BB, 0x1C41A025, 0x92CEA7C6, 0x5E64A758,
		0x54A1AE41, 0x980BAEDF, 0x1684A93C, 0xDA2EA9A2,
		0x030EBB0E, 0xCFA4BB90, 0x412BBC73, 0x8D81BCED,
		0x8744B5F4, 0x4BEEB56A, 0xC561B289, 0x09CBB217,
		0xAC509190, 0x60FA910E, 0xEE7596ED, 0x22DF9673,
		0x281A9F6A, 0xE4B09FF4, 0x6A3F9817, 0xA6959889,
		0x7FB58A25, 0xB31F8ABB, 0x3D908D58, 0xF13A8DC6,
		0xFBFF84DF, 0x37558441, 0xB9DA83A2, 0x7570833C,
		0x533B85DA, 0x9F918544, 0x111E82A7, 0xDDB48239,
		0xD7718B20, 0x1BDB8BBE, 0x95548C5D, 0x59FE8CC3,
		0x80DE9E6F, 0x4C749EF1, 0xC2FB9912, 0x0E51998C,
		0x04949095, 0xC83E900B, 0x46B197E8, 0x8A1B9776,
		0x2F80B4F1, 0xE32AB46F, 0x6DA5B38C, 0xA10FB312,
		0xABCABA0B, 0x6760BA95, 0xE9EFBD76, 0x2545BDE8,
		0xFC65AF44, 0x30CFAFDA, 0xBE40A839, 0x72EAA8A7,
		0x782FA1BE, 0xB485A120, 0x3A0AA6C3, 0xF6A0A65D,
		0xAA4DE78C, 0x66E7E712, 0xE868E0F1, 0x24C2E06F,
		0x2E07E976, 0xE2ADE9E8, 0x6C22EE0B, 0xA088EE95,
		0x79A8FC39, 0xB502FCA7, 0x3B8DFB44, 0xF727FBDA,
		0xFDE2F2C3, 0x3148F25D, 0xBFC7F5BE, 0x736DF520,
		0xD6F6D6A7, 0x1A5CD639, 0x94D3D1DA, 0x5879D144,
		0x52BCD85D, 0x9E16D8C3, 0x1099DF20, 0xDC33DFBE,
		0x0513CD12, 0xC9B9CD8C, 0x4736CA6F, 0x8B9CCAF1,
		0x8159C3E8, 0x4DF3C376, 0xC37CC495, 0x0FD6C40B,
		0x7AA64737, 0xB60C47A9, 0x3883404A, 0xF42940D4,
		0xFEEC49CD, 0x32464953, 0xBCC94EB0, 0x70634E2E,
		0xA9435C82, 0x65E95C1C, 0xEB665BFF, 0x27CC5B61,
		0x2D095278, 0xE1A352E6, 0x6F2C5505, 0xA386559B,
		0x061D761C, 0xCAB77682, 0x44387161, 0x889271FF,
		0x825778E6, 0x4EFD7878, 0xC0727F9B, 0x0CD87F05,
		0xD5F86DA9, 0x19526D37, 0x97DD6AD4, 0x5B776A4A,
		0x51B26353, 0x9D1863CD, 0x1397642E, 0xDF3D64B0,
		0x83D02561, 0x4F7A25FF, 0xC1F5221C, 0x0D5F2282,
		0x079A2B9B, 0xCB302B05, 0x45BF2CE6, 0x89152C78,
		0x50353ED4, 0x9C9F3E4A, 0x121039A9, 0xDEBA3937,
		0xD47F302E, 0x18D530B0, 0x965A3753, 0x5AF037CD,
		0xFF6B144A, 0x33C114D4, 0xBD4E1337, 0x71E413A9,
		0x7B211AB0, 0xB78B1A2E, 0x39041DCD, 0xF5AE1D53,
		0x2C8E0FFF, 0xE0240F61, 0x6EAB0882, 0xA201081C,
		0xA8C40105, 0x646E019B, 0xEAE10678, 0x264B06E6
	}
};

const uint64_t crc64_table[4][256] = {
	{
		0x0000000000000000UL, 0xB32E4CBE03A75F6FUL,
		0xF4843657A840A05BUL, 0x47AA7AE9ABE7FF34UL,
		0x7BD0C384FF8F5E33UL, 0xC8FE8F3AFC28015CUL,
		0x8F54F5D357CFFE68UL, 0x3C7AB96D5468A107UL,
		0xF7A18709FF1EBC66UL, 0x448FCBB7FCB9E309UL,
		0x0325B15E575E1C3DUL, 0xB00BFDE054F94352UL,
		0x8C71448D0091E255UL, 0x3F5F08330336BD3AUL,
		0x78F572DAA8D1420EUL, 0xCBDB3E64AB761D61UL,
		0x7D9BA13851336649UL, 0xCEB5ED8652943926UL,
		0x891F976FF973C612UL, 0x3A31DBD1FAD4997DUL,
		0x064B62BCAEBC387AUL, 0xB5652E02AD1B6715UL,
		0xF2CF54EB06FC9821UL, 0x41E11855055BC74EUL,
		0x8A3A2631AE2DDA2FUL, 0x39146A8FAD8A8540UL,
		0x7EBE1066066D7A74UL, 0xCD905CD805CA251BUL,
		0xF1EAE5B551A2841CUL, 0x42C4A90B5205DB73UL,
		0x056ED3E2F9E22447UL, 0xB6409F5CFA457B28UL,
		0xFB374270A266CC92UL, 0x48190ECEA1C193FDUL,
		0x0FB374270A266CC9UL, 0xBC9D3899098133A6UL,
		0x80E781F45DE992A1UL, 0x33C9CD4A5E4ECDCEUL,
		0x7463B7A3F5A932FAUL, 0xC74DFB1DF60E6D95UL,
		0x0C96C5795D7870F4UL, 0xBFB889C75EDF2F9BUL,
		0xF812F32EF538D0AFUL, 0x4B3CBF90F69F8FC0UL,
		0x774606FDA2F72EC7UL, 0xC4684A43A15071A8UL,
		0x83C230AA0AB78E9CUL, 0x30EC7C140910D1F3UL,
		0x86ACE348F355AADBUL, 0x3582AFF6F0F2F5B4UL,
		0x7228D51F5B150A80UL, 0xC10699A158B255EFUL,
		0xFD7C20CC0CDAF4E8UL, 0x4E526C720F7DAB87UL,
		0x09F8169BA49A54B3UL, 0xBAD65A25A73D0BDCUL,
		0x710D64410C4B16BDUL, 0xC22328FF0FEC49D2UL,
		0x85895216A40BB6E6UL, 0x36A71EA8A7ACE989UL,
		0x0ADDA7C5F3C4488EUL, 0xB9F3EB7BF06317E1UL,
		0xFE5991925B84E8D5UL, 0x4D77DD2C5823B7BAUL,
		0x64B62BCAEBC387A1UL, 0xD7986774E864D8CEUL,
		0x90321D9D438327FAUL, 0x231C512340247895UL,
		0x1F66E84E144CD992UL, 0xAC48A4F017EB86FDUL,
		0xEBE2DE19BC0C79C9UL, 0x58CC92A7BFAB26A6UL,
		0x9317ACC314DD3BC7UL, 0x2039E07D177A64A8UL,
		0x67939A94BC9D9B9CUL, 0xD4BDD62ABF3AC4F3UL,
		0xE8C76F47EB5265F4UL, 0x5BE923F9E8F53A9BUL,
		0x1C4359104312C5AFUL, 0xAF6D15AE40B59AC0UL,
		0x192D8AF2BAF0E1E8UL, 0xAA03C64CB957BE87UL,
		0xEDA9BCA512B041B3UL, 0x5E87F01B11171EDCUL,
		0x62FD4976457FBFDBUL, 0xD1D305C846D8E0B4UL,
		0x96797F21ED3F1F80UL, 0x2557339FEE9840EFUL,
		0xEE8C0DFB45EE5D8EUL, 0x5DA24145464902E1UL,
		0x1A083BACEDAEFDD5UL, 0xA9267712EE09A2BAUL,
		0x955CCE7FBA6103BDUL, 0x267282C1B9C65CD2UL,
		0x61D8F8281221A3E6UL, 0xD2F6B4961186FC89UL,
		0x9F8169BA49A54B33UL, 0x2CAF25044A02145CUL,
		0x6B055FEDE1E5EB68UL, 0xD82B1353E242B407UL,
		0xE451AA3EB62A1500UL, 0x577FE680B58D4A6FUL,
		0x10D59C691E6AB55BUL, 0xA3FBD0D71DCDEA34UL,
		0x6820EEB3B6BBF755UL, 0xDB0EA20DB51CA83AUL,
		0x9CA4D8E41EFB570EUL, 0x2F8A945A1D5C0861UL,
		0x13F02D374934A966UL, 0xA0DE61894A93F609UL,
		0xE7741B60E174093DUL, 0x545A57DEE2D35652UL,
		0xE21AC88218962D7AUL, 0x5134843C1B317215UL,
		0x169EFED5B0D68D21UL, 0xA5B0B26BB371D24EUL,
		0x99CA0B06E7197349UL, 0x2AE447B8E4BE2C26UL,
		0x6D4E3D514F59D312UL, 0xDE6071EF4CFE8C7DUL,
		0x15BB4F8BE788911CUL, 0xA6950335E42FCE73UL,
		0xE13F79DC4FC83147UL, 0x521135624C6F6E28UL,
		0x6E6B8C0F1807CF2FUL, 0xDD45C0B11BA09040UL,
		0x9AEFBA58B0476F74UL, 0x29C1F6E6B3E0301BUL,
		0xC96C5795D7870F42UL, 0x7A421B2BD420502DUL,
		0x3DE861C27FC7AF19UL, 0x8EC62D7C7C60F076UL,
		0xB2BC941128085171UL, 0x0192D8AF2BAF0E1EUL,
		0x4638A2468048F12AUL, 0xF516EEF883EFAE45UL,
		0x3ECDD09C2899B324UL, 0x8DE39C222B3EEC4BUL,
		0xCA49E6CB80D9137FUL, 0x7967AA75837E4C10UL,
		0x451D1318D716ED17UL, 0xF6335FA6D4B1B278UL,
		0xB199254F7F564D4CUL, 0x02B769F17CF11223UL,
		0xB4F7F6AD86B4690BUL, 0x07D9BA1385133664UL,
		0x4073C0FA2EF4C950UL, 0xF35D8C442D53963FUL,
		0xCF273529793B3738UL, 0x7C0979977A9C6857UL,
		0x3BA3037ED17B9763UL, 0x888D4FC0D2DCC80CUL,
		0x435671A479AAD56DUL, 0xF0783D1A7A0D8A02UL,
		0xB7D247F3D1EA7536UL, 0x04FC0B4DD24D2A59UL,
		0x3886B22086258B5EUL, 0x8BA8FE9E8582D431UL,
		0xCC0284772E652B05UL, 0x7F2CC8C92DC2746AUL,
		0x325B15E575E1C3D0UL, 0x8175595B76469CBFUL,
		0xC6DF23B2DDA1638BUL, 0x75F16F0CDE063CE4UL,
		0x498BD6618A6E9DE3UL, 0xFAA59ADF89C9C28CUL,
		0xBD0FE036222E3DB8UL, 0x0E21AC88218962D7UL,
		0xC5FA92EC8AFF7FB6UL, 0x76D4DE52895820D9UL,
		0x317EA4BB22BFDFEDUL, 0x8250E80521188082UL,
		0xBE2A516875702185UL, 0x0D041DD676D77EEAUL,
		0x4AAE673FDD3081DEUL, 0xF9802B81DE97DEB1UL,
		0x4FC0B4DD24D2A599UL, 0xFCEEF8632775FAF6UL,
		0xBB44828A8C9205C2UL, 0x086ACE348F355AADUL,
		0x34107759DB5DFBAAUL, 0x873E3BE7D8FAA4C5UL,
		0xC094410E731D5BF1UL, 0x73BA0DB070BA049EUL,
		0xB86133D4DBCC19FFUL, 0x0B4F7F6AD86B4690UL,
		0x4CE50583738CB9A4UL, 0xFFCB493D702BE6CBUL,
		0xC3B1F050244347CCUL, 0x709FBCEE27E418A3UL,
		0x3735C6078C03E797UL, 0x841B8AB98FA4B8F8UL,
		0xADDA7C5F3C4488E3UL, 0x1EF430E13FE3D78CUL,
		0x595E4A08940428B8UL, 0xEA7006B697A377D7UL,
		0xD60ABFDBC3CBD6D0UL, 0x6524F365C06C89BFUL,
		0x228E898C6B8B768BUL, 0x91A0C532682C29E4UL,
		0x5A7BFB56C35A3485UL, 0xE955B7E8C0FD6BEAUL,
		0xAEFFCD016B1A94DEUL, 0x1DD181BF68BDCBB1UL,
		0x21AB38D23CD56AB6UL, 0x9285746C3F7235D9UL,
		0xD52F0E859495CAEDUL, 0x6601423B97329582UL,
		0xD041DD676D77EEAAUL, 0x636F91D96ED0B1C5UL,
		0x24C5EB30C5374EF1UL, 0x97EBA78EC690119EUL,
		0xAB911EE392F8B099UL, 0x18BF525D915FEFF6UL,
		0x5F1528B43AB810C2UL, 0xEC3B640A391F4FADUL,
		0x27E05A6E926952CCUL, 0x94CE16D091CE0DA3UL,
		0xD3646C393A29F297UL, 0x604A2087398EADF8UL,
		0x5C3099EA6DE60CFFUL, 0xEF1ED5546E415390UL,
		0xA8B4AFBDC5A6ACA4UL, 0x1B9AE303C601F3CBUL,
		0x56ED3E2F9E224471UL, 0xE5C372919D851B1EUL,
		0xA26908783662E42AUL, 0x114744C635C5BB45UL,
		0x2D3DFDAB61AD1A42UL, 0x9E13B115620A452DUL,
		0xD9B9CBFCC9EDBA19UL, 0x6A978742CA4AE576UL,
		0xA14CB926613CF817UL, 0x1262F598629BA778UL,
		0x55C88F71C97C584CUL, 0xE6E6C3CFCADB0723UL,
		0xDA9C7AA29EB3A624UL, 0x69B2361C9D14F94BUL,
		0x2E184CF536F3067FUL, 0x9D36004B35545910UL,
		0x2B769F17CF112238UL, 0x9858D3A9CCB67D57UL,
		0xDFF2A94067518263UL, 0x6CDCE5FE64F6DD0CUL,
		0x50A65C93309E7C0BUL, 0xE388102D33392364UL,
		0xA4226AC498DEDC50UL, 0x170C267A9B79833FUL,
		0xDCD7181E300F9E5EUL, 0x6FF954A033A8C131UL,
		0x28532E49984F3E05UL, 0x9B7D62F79BE8616AUL,
		0xA707DB9ACF80C06DUL, 0x14299724CC279F02UL,
		0x5383EDCD67C06036UL, 0xE0ADA17364673F59UL
	}, {
		0x0000000000000000UL, 0x54E979925CD0F10DUL,
		0xA9D2F324B9A1E21AUL, 0xFD3B8AB6E5711317UL,
		0xC17D4962DC4DDAB1UL, 0x959430F0809D2BBCUL,
		0x68AFBA4665EC38ABUL, 0x3C46C3D4393CC9A6UL,
		0x10223DEE1795ABE7UL, 0x44CB447C4B455AEAUL,
		0xB9F0CECAAE3449FDUL, 0xED19B758F2E4B8F0UL,
		0xD15F748CCBD87156UL, 0x85B60D1E9708805BUL,
		0x788D87A87279934CUL, 0x2C64FE3A2EA96241UL,
		0x20447BDC2F2B57CEUL, 0x74AD024E73FBA6C3UL,
		0x899688F8968AB5D4UL, 0xDD7FF16ACA5A44D9UL,
		0xE13932BEF3668D7FUL, 0xB5D04B2CAFB67C72UL,
		0x48EBC19A4AC76F65UL, 0x1C02B80816179E68UL,
		0x3066463238BEFC29UL, 0x648F3FA0646E0D24UL,
		0x99B4B516811F1E33UL, 0xCD5DCC84DDCFEF3EUL,
		0xF11B0F50E4F32698UL, 0xA5F276C2B823D795UL,
		0x58C9FC745D52C482UL, 0x0C2085E60182358FUL,
		0x4088F7B85E56AF9CUL, 0x14618E2A02865E91UL,
		0xE95A049CE7F74D86UL, 0xBDB37D0EBB27BC8BUL,
		0x81F5BEDA821B752DUL, 0xD51CC748DECB8420UL,
		0x28274DFE3BBA9737UL, 0x7CCE346C676A663AUL,
		0x50AACA5649C3047BUL, 0x0443B3C41513F576UL,
		0xF9783972F062E661UL, 0xAD9140E0ACB2176CUL,
		0x91D78334958EDECAUL, 0xC53EFAA6C95E2FC7UL,
		0x380570102C2F3CD0UL, 0x6CEC098270FFCDDDUL,
		0x60CC8C64717DF852UL, 0x3425F5F62DAD095FUL,
		0xC91E7F40C8DC1A48UL, 0x9DF706D2940CEB45UL,
		0xA1B1C506AD3022E3UL, 0xF558BC94F1E0D3EEUL,
		0x086336221491C0F9UL, 0x5C8A4FB0484131F4UL,
		0x70EEB18A66E853B5UL, 0x2407C8183A38A2B8UL,
		0xD93C42AEDF49B1AFUL, 0x8DD53B3C839940A2UL,
		0xB193F8E8BAA58904UL, 0xE57A817AE6757809UL,
		0x18410BCC03046B1EUL, 0x4CA8725E5FD49A13UL,
		0x8111EF70BCAD5F38UL, 0xD5F896E2E07DAE35UL,
		0x28C31C54050CBD22UL, 0x7C2A65C659DC4C2FUL,
		0x406CA61260E08589UL, 0x1485DF803C307484UL,
		0xE9BE5536D9416793UL, 0xBD572CA48591969EUL,
		0x9133D29EAB38F4DFUL, 0xC5DAAB0CF7E805D2UL,
		0x38E121BA129916C5UL, 0x6C0858284E49E7C8UL,
		0x504E9BFC77752E6EUL, 0x04A7E26E2BA5DF63UL,
		0xF99C68D8CED4CC74UL, 0xAD75114A92043D79UL,
		0xA15594AC938608F6UL, 0xF5BCED3ECF56F9FBUL,
		0x088767882A27EAECUL, 0x5C6E1E1A76F71BE1UL,
		0x6028DDCE4FCBD247UL, 0x34C1A45C131B234AUL,
		0xC9FA2EEAF66A305DUL, 0x9D135778AABAC150UL,
		0xB177A9428413A311UL, 0xE59ED0D0D8C3521CUL,
		0x18A55A663DB2410BUL, 0x4C4C23F46162B006UL,
		0x700AE020585E79A0UL, 0x24E399B2048E88ADUL,
		0xD9D81304E1FF9BBAUL, 0x8D316A96BD2F6AB7UL,
		0xC19918C8E2FBF0A4UL, 0x9570615ABE2B01A9UL,
		0x684BEBEC5B5A12BEUL, 0x3CA2927E078AE3B3UL,
		0x00E451AA3EB62A15UL, 0x540D28386266DB18UL,
		0xA936A28E8717C80FUL, 0xFDDFDB1CDBC73902UL,
		0xD1BB2526F56E5B43UL, 0x85525CB4A9BEAA4EUL,
		0x7869D6024CCFB959UL, 0x2C80AF90101F4854UL,
		0x10C66C44292381F2UL, 0x442F15D675F370FFUL,
		0xB9149F60908263E8UL, 0xEDFDE6F2CC5292E5UL,
		0xE1DD6314CDD0A76AUL, 0xB5341A8691005667UL,
		0x480F903074714570UL, 0x1CE6E9A228A1B47DUL,
		0x20A02A76119D7DDBUL, 0x744953E44D4D8CD6UL,
		0x8972D952A83C9FC1UL, 0xDD9BA0C0F4EC6ECCUL,
		0xF1FF5EFADA450C8DUL, 0xA51627688695FD80UL,
		0x582DADDE63E4EE97UL, 0x0CC4D44C3F341F9AUL,
		0x308217980608D63CUL, 0x646B6E0A5AD82731UL,
		0x9950E4BCBFA93426UL, 0xCDB99D2EE379C52BUL,
		0x90FB71CAD654A0F5UL, 0xC41208588A8451F8UL,
		0x392982EE6FF542EFUL, 0x6DC0FB7C3325B3E2UL,
		0x518638A80A197A44UL, 0x056F413A56C98B49UL,
		0xF854CB8CB3B8985EUL, 0xACBDB21EEF686953UL,
		0x80D94C24C1C10B12UL, 0xD43035B69D11FA1FUL,
		0x290BBF007860E908UL, 0x7DE2C69224B01805UL,
		0x41A405461D8CD1A3UL, 0x154D7CD4415C20AEUL,
		0xE876F662A42D33B9UL, 0xBC9F8FF0F8FDC2B4UL,
		0xB0BF0A16F97FF73BUL, 0xE4567384A5AF0636UL,
		0x196DF93240DE1521UL, 0x4D8480A01C0EE42CUL,
		0x71C2437425322D8AUL, 0x252B3AE679E2DC87UL,
		0xD810B0509C93CF90UL, 0x8CF9C9C2C0433E9DUL,
		0xA09D37F8EEEA5CDCUL, 0xF4744E6AB23AADD1UL,
		0x094FC4DC574BBEC6UL, 0x5DA6BD4E0B9B4FCBUL,
		0x61E07E9A32A7866DUL, 0x350907086E777760UL,
		0xC8328DBE8B066477UL, 0x9CDBF42CD7D6957AUL,
		0xD073867288020F69UL, 0x849AFFE0D4D2FE64UL,
		0x79A1755631A3ED73UL, 0x2D480CC46D731C7EUL,
		0x110ECF10544FD5D8UL, 0x45E7B682089F24D5UL,
		0xB8DC3C34EDEE37C2UL, 0xEC3545A6B13EC6CFUL,
		0xC051BB9C9F97A48EUL, 0x94B8C20EC3475583UL,
		0x698348B826364694UL, 0x3D6A312A7AE6B799UL,
		0x012CF2FE43DA7E3FUL, 0x55C58B6C1F0A8F32UL,
		0xA8FE01DAFA7B9C25UL, 0xFC177848A6AB6D28UL,
		0xF037FDAEA72958A7UL, 0xA4DE843CFBF9A9AAUL,
		0x59E50E8A1E88BABDUL, 0x0D0C771842584BB0UL,
		0x314AB4CC7B648216UL, 0x65A3CD5E27B4731BUL,
		0x989847E8C2C5600CUL, 0xCC713E7A9E159101UL,
		0xE015C040B0BCF340UL, 0xB4FCB9D2EC6C024DUL,
		0x49C73364091D115AUL, 0x1D2E4AF655CDE057UL,
		0x216889226CF129F1UL, 0x7581F0B03021D8FCUL,
		0x88BA7A06D550CBEBUL, 0xDC53039489803AE6UL,
		0x11EA9EBA6AF9FFCDUL, 0x4503E72836290EC0UL,
		0xB8386D9ED3581DD7UL, 0xECD1140C8F88ECDAUL,
		0xD097D7D8B6B4257CUL, 0x847EAE4AEA64D471UL,
		0x794524FC0F15C766UL, 0x2DAC5D6E53C5366BUL,
		0x01C8A3547D6C542AUL, 0x5521DAC621BCA527UL,
		0xA81A5070C4CDB630UL, 0xFCF329E2981D473DUL,
		0xC0B5EA36A1218E9BUL, 0x945C93A4FDF17F96UL,
		0x6967191218806C81UL, 0x3D8E608044509D8CUL,
		0x31AEE56645D2A803UL, 0x65479CF41902590EUL,
		0x987C1642FC734A19UL, 0xCC956FD0A0A3BB14UL,
		0xF0D3AC04999F72B2UL, 0xA43AD596C54F83BFUL,
		0x59015F20203E90A8UL, 0x0DE826B27CEE61A5UL,
		0x218CD888524703E4UL, 0x7565A11A0E97F2E9UL,
		0x885E2BACEBE6E1FEUL, 0xDCB7523EB73610F3UL,
		0xE0F191EA8E0AD955UL, 0xB418E878D2DA2858UL,
		0x492362CE37AB3B4FUL, 0x1DCA1B5C6B7BCA42UL,
		0x5162690234AF5051UL, 0x058B1090687FA15CUL,
		0xF8B09A268D0EB24BUL, 0xAC59E3B4D1DE4346UL,
		0x901F2060E8E28AE0UL, 0xC4F659F2B4327BEDUL,
		0x39CDD344514368FAUL, 0x6D24AAD60D9399F7UL,
		0x414054EC233AFBB6UL, 0x15A92D7E7FEA0ABBUL,
		0xE892A7C89A9B19ACUL, 0xBC7BDE5AC64BE8A1UL,
		0x803D1D8EFF772107UL, 0xD4D4641CA3A7D00AUL,
		0x29EFEEAA46D6C31DUL, 0x7D0697381A063210UL,
		0x712612DE1B84079FUL, 0x25CF6B4C4754F692UL,
		0xD8F4E1FAA225E585UL, 0x8C1D9868FEF51488UL,
		0xB05B5BBCC7C9DD2EUL, 0xE4B2222E9B192C23UL,
		0x1989A8987E683F34UL, 0x4D60D10A22B8CE39UL,
		0x61042F300C11AC78UL, 0x35ED56A250C15D75UL,
		0xC8D6DC14B5B04E62UL, 0x9C3FA586E960BF6FUL,
		0xA0796652D05C76C9UL, 0xF4901FC08C8C87C4UL,
		0x09AB957669FD94D3UL, 0x5D42ECE4352D65DEUL
	}, {
		0x0000000000000000UL, 0x3F0BE14A916A6DCBUL,
		0x7E17C29522D4DB96UL, 0x411C23DFB3BEB65DUL,
		0xFC2F852A45A9B72CUL, 0xC3246460D4C3DAE7UL,
		0x823847BF677D6CBAUL, 0xBD33A6F5F6170171UL,
		0x6A87A57F245D70DDUL, 0x558C4435B5371D16UL,
		0x149067EA0689AB4BUL, 0x2B9B86A097E3C680UL,
		0x96A8205561F4C7F1UL, 0xA9A3C11FF09EAA3AUL,
		0xE8BFE2C043201C67UL, 0xD7B4038AD24A71ACUL,
		0xD50F4AFE48BAE1BAUL, 0xEA04ABB4D9D08C71UL,
		0xAB18886B6A6E3A2CUL, 0x94136921FB0457E7UL,
		0x2920CFD40D135696UL, 0x162B2E9E9C793B5DUL,
		0x57370D412FC78D00UL, 0x683CEC0BBEADE0CBUL,
		0xBF88EF816CE79167UL, 0x80830ECBFD8DFCACUL,
		0xC19F2D144E334AF1UL, 0xFE94CC5EDF59273AUL,
		0x43A76AAB294E264BUL, 0x7CAC8BE1B8244B80UL,
		0x3DB0A83E0B9AFDDDUL, 0x02BB49749AF09016UL,
		0x38C63AD73E7BDDF1UL, 0x07CDDB9DAF11B03AUL,
		0x46D1F8421CAF0667UL, 0x79DA19088DC56BACUL,
		0xC4E9BFFD7BD26ADDUL, 0xFBE25EB7EAB80716UL,
		0xBAFE7D685906B14BUL, 0x85F59C22C86CDC80UL,
		0x52419FA81A26AD2CUL, 0x6D4A7EE28B4CC0E7UL,
		0x2C565D3D38F276BAUL, 0x135DBC77A9981B71UL,
		0xAE6E1A825F8F1A00UL, 0x9165FBC8CEE577CBUL,
		0xD079D8177D5BC196UL, 0xEF72395DEC31AC5DUL,
		0xEDC9702976C13C4BUL, 0xD2C29163E7AB5180UL,
		0x93DEB2BC5415E7DDUL, 0xACD553F6C57F8A16UL,
		0x11E6F50333688B67UL, 0x2EED1449A202E6ACUL,
		0x6FF1379611BC50F1UL, 0x50FAD6DC80D63D3AUL,
		0x874ED556529C4C96UL, 0xB845341CC3F6215DUL,
		0xF95917C370489700UL, 0xC652F689E122FACBUL,
		0x7B61507C1735FBBAUL, 0x446AB136865F9671UL,
		0x057692E935E1202CUL, 0x3A7D73A3A48B4DE7UL,
		0x718C75AE7CF7BBE2UL, 0x4E8794E4ED9DD629UL,
		0x0F9BB73B5E236074UL, 0x30905671CF490DBFUL,
		0x8DA3F084395E0CCEUL, 0xB2A811CEA8346105UL,
		0xF3B432111B8AD758UL, 0xCCBFD35B8AE0BA93UL,
		0x1B0BD0D158AACB3FUL, 0x2400319BC9C0A6F4UL,
		0x651C12447A7E10A9UL, 0x5A17F30EEB147D62UL,
		0xE72455FB1D037C13UL, 0xD82FB4B18C6911D8UL,
		0x9933976E3FD7A785UL, 0xA6387624AEBDCA4EUL,
		0xA4833F50344D5A58UL, 0x9B88DE1AA5273793UL,
		0xDA94FDC5169981CEUL, 0xE59F1C8F87F3EC05UL,
		0x58ACBA7A71E4ED74UL, 0x67A75B30E08E80BFUL,
		0x26BB78EF533036E2UL, 0x19B099A5C25A5B29UL,
		0xCE049A2F10102A85UL, 0xF10F7B65817A474EUL,
		0xB01358BA32C4F113UL, 0x8F18B9F0A3AE9CD8UL,
		0x322B1F0555B99DA9UL, 0x0D20FE4FC4D3F062UL,
		0x4C3CDD90776D463FUL, 0x73373CDAE6072BF4UL,
		0x494A4F79428C6613UL, 0x7641AE33D3E60BD8UL,
		0x375D8DEC6058BD85UL, 0x08566CA6F132D04EUL,
		0xB565CA530725D13FUL, 0x8A6E2B19964FBCF4UL,
		0xCB7208C625F10AA9UL, 0xF479E98CB49B6762UL,
		0x23CDEA0666D116CEUL, 0x1CC60B4CF7BB7B05UL,
		0x5DDA28934405CD58UL, 0x62D1C9D9D56FA093UL,
		0xDFE26F2C2378A1E2UL, 0xE0E98E66B212CC29UL,
		0xA1F5ADB901AC7A74UL, 0x9EFE4CF390C617BFUL,
		0x9C4505870A3687A9UL, 0xA34EE4CD9B5CEA62UL,
		0xE252C71228E25C3FUL, 0xDD592658B98831F4UL,
		0x606A80AD4F9F3085UL, 0x5F6161E7DEF55D4EUL,
		0x1E7D42386D4BEB13UL, 0x2176A372FC2186D8UL,
		0xF6C2A0F82E6BF774UL, 0xC9C941B2BF019ABFUL,
		0x88D5626D0CBF2CE2UL, 0xB7DE83279DD54129UL,
		0x0AED25D26BC24058UL, 0x35E6C498FAA82D93UL,
		0x74FAE74749169BCEUL, 0x4BF1060DD87CF605UL,
		0xE318EB5CF9EF77C4UL, 0xDC130A1668851A0FUL,
		0x9D0F29C9DB3BAC52UL, 0xA204C8834A51C199UL,
		0x1F376E76BC46C0E8UL, 0x203C8F3C2D2CAD23UL,
		0x6120ACE39E921B7EUL, 0x5E2B4DA90FF876B5UL,
		0x899F4E23DDB20719UL, 0xB694AF694CD86AD2UL,
		0xF7888CB6FF66DC8FUL, 0xC8836DFC6E0CB144UL,
		0x75B0CB09981BB035UL, 0x4ABB2A430971DDFEUL,
		0x0BA7099CBACF6BA3UL, 0x34ACE8D62BA50668UL,
		0x3617A1A2B155967EUL, 0x091C40E8203FFBB5UL,
		0x4800633793814DE8UL, 0x770B827D02EB2023UL,
		0xCA382488F4FC2152UL, 0xF533C5C265964C99UL,
		0xB42FE61DD628FAC4UL, 0x8B2407574742970FUL,
		0x5C9004DD9508E6A3UL, 0x639BE59704628B68UL,
		0x2287C648B7DC3D35UL, 0x1D8C270226B650FEUL,
		0xA0BF81F7D0A1518FUL, 0x9FB460BD41CB3C44UL,
		0xDEA84362F2758A19UL, 0xE1A3A228631FE7D2UL,
		0xDBDED18BC794AA35UL, 0xE4D530C156FEC7FEUL,
		0xA5C9131EE54071A3UL, 0x9AC2F254742A1C68UL,
		0x27F154A1823D1D19UL, 0x18FAB5EB135770D2UL,
		0x59E69634A0E9C68FUL, 0x66ED777E3183AB44UL,
		0xB15974F4E3C9DAE8UL, 0x8E5295BE72A3B723UL,
		0xCF4EB661C11D017EUL, 0xF045572B50776CB5UL,
		0x4D76F1DEA6606DC4UL, 0x727D1094370A000FUL,
		0x3361334B84B4B652UL, 0x0C6AD20115DEDB99UL,
		0x0ED19B758F2E4B8FUL, 0x31DA7A3F1E442644UL,
		0x70C659E0ADFA9019UL, 0x4FCDB8AA3C90FDD2UL,
		0xF2FE1E5FCA87FCA3UL, 0xCDF5FF155BED9168UL,
		0x8CE9DCCAE8532735UL, 0xB3E23D8079394AFEUL,
		0x64563E0AAB733B52UL, 0x5B5DDF403A195699UL,
		0x1A41FC9F89A7E0C4UL, 0x254A1DD518CD8D0FUL,
		0x9879BB20EEDA8C7EUL, 0xA7725A6A7FB0E1B5UL,
		0xE66E79B5CC0E57E8UL, 0xD96598FF5D643A23UL,
		0x92949EF28518CC26UL, 0xAD9F7FB81472A1EDUL,
		0xEC835C67A7CC17B0UL, 0xD388BD2D36A67A7BUL,
		0x6EBB1BD8C0B17B0AUL, 0x51B0FA9251DB16C1UL,
		0x10ACD94DE265A09CUL, 0x2FA73807730FCD57UL,
		0xF8133B8DA145BCFBUL, 0xC718DAC7302FD130UL,
		0x8604F9188391676DUL, 0xB90F185212FB0AA6UL,
		0x043CBEA7E4EC0BD7UL, 0x3B375FED7586661CUL,
		0x7A2B7C32C638D041UL, 0x45209D785752BD8AUL,
		0x479BD40CCDA22D9CUL, 0x789035465CC84057UL,
		0x398C1699EF76F60AUL, 0x0687F7D37E1C9BC1UL,
		0xBBB45126880B9AB0UL, 0x84BFB06C1961F77BUL,
		0xC5A393B3AADF4126UL, 0xFAA872F93BB52CEDUL,
		0x2D1C7173E9FF5D41UL, 0x121790397895308AUL,
		0x530BB3E6CB2B86D7UL, 0x6C0052AC5A41EB1CUL,
		0xD133F459AC56EA6DUL, 0xEE3815133D3C87A6UL,
		0xAF2436CC8E8231FBUL, 0x902FD7861FE85C30UL,
		0xAA52A425BB6311D7UL, 0x9559456F2A097C1CUL,
		0xD44566B099B7CA41UL, 0xEB4E87FA08DDA78AUL,
		0x567D210FFECAA6FBUL, 0x6976C0456FA0CB30UL,
		0x286AE39ADC1E7D6DUL, 0x176102D04D7410A6UL,
		0xC0D5015A9F3E610AUL, 0xFFDEE0100E540CC1UL,
		0xBEC2C3CFBDEABA9CUL, 0x81C922852C80D757UL,
		0x3CFA8470DA97D626UL, 0x03F1653A4BFDBBEDUL,
		0x42ED46E5F8430DB0UL, 0x7DE6A7AF6929607BUL,
		0x7F5DEEDBF3D9F06DUL, 0x40560F9162B39DA6UL,
		0x014A2C4ED10D2BFBUL, 0x3E41CD0440674630UL,
		0x83726BF1B6704741UL, 0xBC798ABB271A2A8AUL,
		0xFD65A96494A49CD7UL, 0xC26E482E05CEF11CUL,
		0x15DA4BA4D78480B0UL, 0x2AD1AAEE46EEED7BUL,
		0x6BCD8931F5505B26UL, 0x54C6687B643A36EDUL,
		0xE9F5CE8E922D379CUL, 0xD6FE2FC403475A57UL,
		0x97E20C1BB0F9EC0AUL, 0xA8E9ED51219381C1UL
	}, {
		0x0000000000000000UL, 0x1DEE8A5E222CA1DCUL,
		0x3BDD14BC445943B8UL, 0x26339EE26675E264UL,
		0x77BA297888B28770UL, 0x6A54A326AA9E26ACUL,
		0x4C673DC4CCEBC4C8UL, 0x5189B79AEEC76514UL,
		0xEF7452F111650EE0UL, 0xF29AD8AF3349AF3CUL,
		0xD4A9464D553C4D58UL, 0xC947CC137710EC84UL,
		0x98CE7B8999D78990UL, 0x8520F1D7BBFB284CUL,
		0xA3136F35DD8ECA28UL, 0xBEFDE56BFFA26BF4UL,
		0x4C300AC98DC40345UL, 0x51DE8097AFE8A299UL,
		0x77ED1E75C99D40FDUL, 0x6A03942BEBB1E121UL,
		0x3B8A23B105768435UL, 0x2664A9EF275A25E9UL,
		0x0057370D412FC78DUL, 0x1DB9BD5363036651UL,
		0xA34458389CA10DA5UL, 0xBEAAD266BE8DAC79UL,
		0x98994C84D8F84E1DUL, 0x8577C6DAFAD4EFC1UL,
		0xD4FE714014138AD5UL, 0xC910FB1E363F2B09UL,
		0xEF2365FC504AC96DUL, 0xF2CDEFA2726668B1UL,
		0x986015931B88068AUL, 0x858E9FCD39A4A756UL,
		0xA3BD012F5FD14532UL, 0xBE538B717DFDE4EEUL,
		0xEFDA3CEB933A81FAUL, 0xF234B6B5B1162026UL,
		0xD4072857D763C242UL, 0xC9E9A209F54F639EUL,
		0x771447620AED086AUL, 0x6AFACD3C28C1A9B6UL,
		0x4CC953DE4EB44BD2UL, 0x5127D9806C98EA0EUL,
		0x00AE6E1A825F8F1AUL, 0x1D40E444A0732EC6UL,
		0x3B737AA6C606CCA2UL, 0x269DF0F8E42A6D7EUL,
		0xD4501F5A964C05CFUL, 0xC9BE9504B460A413UL,
		0xEF8D0BE6D2154677UL, 0xF26381B8F039E7ABUL,
		0xA3EA36221EFE82BFUL, 0xBE04BC7C3CD22363UL,
		0x9837229E5AA7C107UL, 0x85D9A8C0788B60DBUL,
		0x3B244DAB87290B2FUL, 0x26CAC7F5A505AAF3UL,
		0x00F95917C3704897UL, 0x1D17D349E15CE94BUL,
		0x4C9E64D30F9B8C5FUL, 0x5170EE8D2DB72D83UL,
		0x7743706F4BC2CFE7UL, 0x6AADFA3169EE6E3BUL,
		0xA218840D981E1391UL, 0xBFF60E53BA32B24DUL,
		0x99C590B1DC475029UL, 0x842B1AEFFE6BF1F5UL,
		0xD5A2AD7510AC94E1UL, 0xC84C272B3280353DUL,
		0xEE7FB9C954F5D759UL, 0xF391339776D97685UL,
		0x4D6CD6FC897B1D71UL, 0x50825CA2AB57BCADUL,
		0x76B1C240CD225EC9UL, 0x6B5F481EEF0EFF15UL,
		0x3AD6FF8401C99A01UL, 0x273875DA23E53BDDUL,
		0x010BEB384590D9B9UL, 0x1CE5616667BC7865UL,
		0xEE288EC415DA10D4UL, 0xF3C6049A37F6B108UL,
		0xD5F59A785183536CUL, 0xC81B102673AFF2B0UL,
		0x9992A7BC9D6897A4UL, 0x847C2DE2BF443678UL,
		0xA24FB300D931D41CUL, 0xBFA1395EFB1D75C0UL,
		0x015CDC3504BF1E34UL, 0x1CB2566B2693BFE8UL,
		0x3A81C88940E65D8CUL, 0x276F42D762CAFC50UL,
		0x76E6F54D8C0D9944UL, 0x6B087F13AE213898UL,
		0x4D3BE1F1C854DAFCUL, 0x50D56BAFEA787B20UL,
		0x3A78919E8396151BUL, 0x27961BC0A1BAB4C7UL,
		0x01A58522C7CF56A3UL, 0x1C4B0F7CE5E3F77FUL,
		0x4DC2B8E60B24926BUL, 0x502C32B8290833B7UL,
		0x761FAC5A4F7DD1D3UL, 0x6BF126046D51700FUL,
		0xD50CC36F92F31BFBUL, 0xC8E24931B0DFBA27UL,
		0xEED1D7D3D6AA5843UL, 0xF33F5D8DF486F99FUL,
		0xA2B6EA171A419C8BUL, 0xBF586049386D3D57UL,
		0x996BFEAB5E18DF33UL, 0x848574F57C347EEFUL,
		0x76489B570E52165EUL, 0x6BA611092C7EB782UL,
		0x4D958FEB4A0B55E6UL, 0x507B05B56827F43AUL,
		0x01F2B22F86E0912EUL, 0x1C1C3871A4CC30F2UL,
		0x3A2FA693C2B9D296UL, 0x27C12CCDE095734AUL,
		0x993CC9A61F3718BEUL, 0x84D243F83D1BB962UL,
		0xA2E1DD1A5B6E5B06UL, 0xBF0F57447942FADAUL,
		0xEE86E0DE97859FCEUL, 0xF3686A80B5A93E12UL,
		0xD55BF462D3DCDC76UL, 0xC8B57E3CF1F07DAAUL,
		0xD6E9A7309F3239A7UL, 0xCB072D6EBD1E987BUL,
		0xED34B38CDB6B7A1FUL, 0xF0DA39D2F947DBC3UL,
		0xA1538E481780BED7UL, 0xBCBD041635AC1F0BUL,
		0x9A8E9AF453D9FD6FUL, 0x876010AA71F55CB3UL,
		0x399DF5C18E573747UL, 0x24737F9FAC7B969BUL,
		0x0240E17DCA0E74FFUL, 0x1FAE6B23E822D523UL,
		0x4E27DCB906E5B037UL, 0x53C956E724C911EBUL,
		0x75FAC80542BCF38FUL, 0x6814425B60905253UL,
		0x9AD9ADF912F63AE2UL, 0x873727A730DA9B3EUL,
		0xA104B94556AF795AUL, 0xBCEA331B7483D886UL,
		0xED6384819A44BD92UL, 0xF08D0EDFB8681C4EUL,
		0xD6BE903DDE1DFE2AUL, 0xCB501A63FC315FF6UL,
		0x75ADFF0803933402UL, 0x6843755621BF95DEUL,
		0x4E70EBB447CA77BAUL, 0x539E61EA65E6D666UL,
		0x0217D6708B21B372UL, 0x1FF95C2EA90D12AEUL,
		0x39CAC2CCCF78F0CAUL, 0x24244892ED545116UL,
		0x4E89B2A384BA3F2DUL, 0x536738FDA6969EF1UL,
		0x7554A61FC0E37C95UL, 0x68BA2C41E2CFDD49UL,
		0x39339BDB0C08B85DUL, 0x24DD11852E241981UL,
		0x02EE8F674851FBE5UL, 0x1F0005396A7D5A39UL,
		0xA1FDE05295DF31CDUL, 0xBC136A0CB7F39011UL,
		0x9A20F4EED1867275UL, 0x87CE7EB0F3AAD3A9UL,
		0xD647C92A1D6DB6BDUL, 0xCBA943743F411761UL,
		0xED9ADD965934F505UL, 0xF07457C87B1854D9UL,
		0x02B9B86A097E3C68UL, 0x1F5732342B529DB4UL,
		0x3964ACD64D277FD0UL, 0x248A26886F0BDE0CUL,
		0x7503911281CCBB18UL, 0x68ED1B4CA3E01AC4UL,
		0x4EDE85AEC595F8A0UL, 0x53300FF0E7B9597CUL,
		0xEDCDEA9B181B3288UL, 0xF02360C53A379354UL,
		0xD610FE275C427130UL, 0xCBFE74797E6ED0ECUL,
		0x9A77C3E390A9B5F8UL, 0x879949BDB2851424UL,
		0xA1AAD75FD4F0F640UL, 0xBC445D01F6DC579CUL,
		0x74F1233D072C2A36UL, 0x691FA96325008BEAUL,
		0x4F2C37814375698EUL, 0x52C2BDDF6159C852UL,
		0x034B0A458F9EAD46UL, 0x1EA5801BADB20C9AUL,
		0x38961EF9CBC7EEFEUL, 0x257894A7E9EB4F22UL,
		0x9B8571CC164924D6UL, 0x866BFB923465850AUL,
		0xA05865705210676EUL, 0xBDB6EF2E703CC6B2UL,
		0xEC3F58B49EFBA3A6UL, 0xF1D1D2EABCD7027AUL,
		0xD7E24C08DAA2E01EUL, 0xCA0CC656F88E41C2UL,
		0x38C129F48AE82973UL, 0x252FA3AAA8C488AFUL,
		0x031C3D48CEB16ACBUL, 0x1EF2B716EC9DCB17UL,
		0x4F7B008C025AAE03UL, 0x52958AD220760FDFUL,
		0x74A614304603EDBBUL, 0x69489E6E642F4C67UL,
		0xD7B57B059B8D2793UL, 0xCA5BF15BB9A1864FUL,
		0xEC686FB9DFD4642BUL, 0xF186E5E7FDF8C5F7UL,
		0xA00F527D133FA0E3UL, 0xBDE1D8233113013FUL,
		0x9BD246C15766E35BUL, 0x863CCC9F754A4287UL,
		0xEC9136AE1CA42CBCUL, 0xF17FBCF03E888D60UL,
		0xD74C221258FD6F04UL, 0xCAA2A84C7AD1CED8UL,
		0x9B2B1FD69416ABCCUL, 0x86C59588B63A0A10UL,
		0xA0F60B6AD04FE874UL, 0xBD188134F26349A8UL,
		0x03E5645F0DC1225CUL, 0x1E0BEE012FED8380UL,
		0x383870E3499861E4UL, 0x25D6FABD6BB4C038UL,
		0x745F4D278573A52CUL, 0x69B1C779A75F04F0UL,
		0x4F82599BC12AE694UL, 0x526CD3C5E3064748UL,
		0xA0A13C6791602FF9UL, 0xBD4FB639B34C8E25UL,
		0x9B7C28DBD5396C41UL, 0x8692A285F715CD9DUL,
		0xD71B151F19D2A889UL, 0xCAF59F413BFE0955UL,
		0xECC601A35D8BEB31UL, 0xF1288BFD7FA74AEDUL,
		0x4FD56E9680052119UL, 0x523BE4C8A22980C5UL,
		0x74087A2AC45C62A1UL, 0x69E6F074E670C37DUL,
		0x386F47EE08B7A669UL, 0x2581CDB02A9B07B5UL,
		0x03B253524CEEE5D1UL, 0x1E5CD90C6EC2440DUL
	}
};
