
/**
 * @file /check/magma/magma_check.c
 *
 * @brief	The unit test executable entry point.
 */

#include "magma_check.h"

extern FILE *log_descriptor;
chr_t *barrister_unit_test = NULL;
int_t case_timeout = RUN_TEST_CASE_TIMEOUT;
bool_t do_virus_check = true, do_tank_check = true, do_spf_check = true, do_dkim_check = true;

/**
 * @brief Enable the log so we can print status information. We're only concerned with whether the
		test passed or failed. The actual error message will be recorded by the check library, and
		then printed en masse when all the test cases have concluded.
 * @param test	The NULL terminated string showing the suite, test, and threading information.
 * @param error	The error string, which should be NULL if the test was skipped, or if the test passed.
 */
void log_test(chr_t *test, stringer_t *error) {

	log_enable();

	if (!status() || (st_populated(error) && !st_cmp_ci_eq(NULLER("SKIPPED"), error))) {
		log_unit("%-64.64s%s%10.10s%s\n", test, color_yellow_bold(), "SKIPPED", color_reset());
	}
	else if (st_populated(error)) {
		log_unit("%-64.64s%s%10.10s%s\n", test, color_red_bold(), "FAILED", color_reset());
	}
	else {
		log_unit("%-64.64s%s%10.10s%s\n", test, color_green(), "PASSED", color_reset());
	}

	return;
}

Suite * suite_check_magma(void) {
  Suite *s = suite_create("\n\tMagma");
  return s;
}

/**
 * @brief This function will construct a special suite with only a single unit test that is extracted from the running executable based on it's name.
 * @note	This function will allow us to create a special test suite with any single test case available in the global symbols table. This same functionality
 * 		is available using the CK_RUN_CASE environment variable in the current version of libcheck, but wasn't available in the version currently available on our
 * 		target platform, which is why this method of achieving the same end was created.
 * @param testname The name of the unit test we want our barrister to run.
 * @return The initialized suite, assuming we found the requested unit test, otherwise NULL is returned.
 */
Suite * suite_check_barrister(chr_t *testname) {

	void *handle;
	void *unittest;
	Suite *s = suite_create("\tBarrister");;

	// By calling dlopen() with NULL for the filename, we are attempting to establish a handle for the currently running executable image.
	if (!(handle = dlopen(NULL, RTLD_LOCAL | RTLD_NOW))) {
		log_error("Unable to open a dynamic symbol handle for the currently running program.");
		return NULL;
	}

	// Check the currently running program for the requested function name.
	if (!(*(void **)(&unittest) = dlsym(handle, testname))) {
		log_info("%s", dlerror());
	}
	else {
		suite_check_testcase(s, "", testname, unittest);
	}

	dlclose(handle);
	return s;
}

void suite_check_testcase(Suite *s, const char *tags, const char *name, TFun func) {
	suite_check_testcase_timeout(s, tags, name, func, case_timeout);
}

void suite_check_testcase_timeout(Suite *s, const char *tags, const char *name, TFun func, double timeout) {

	TCase *tc = NULL;

	tcase_add_test((tc = tcase_create(name)), func);
	tcase_set_timeout(tc, timeout);
	tcase_set_tags(tc, tags);
	suite_add_tcase(s, tc);
	return;
}

/***
 * @return Will return -1 if the code is unable to determine whether tracing is active, 0 if tracing is disabled and
 *		1 if tracing has been detected. Tracing indicates the process is running atop a debugger.
 */
int_t running_on_debugger(void) {

	int_t status, ret;
	pid_t pid, parent;

	if ((pid = fork()) == -1) {
		return -1;
	}

	// If were the child, we'll try to start tracing the parent process. If our trace request fails, we assume that means
	// its already being traced by a debugger.
	else if (pid == 0) {

		parent = getppid();

		if (ptrace(PTRACE_ATTACH, parent, NULL, NULL) == 0) {
			waitpid(parent, NULL, 0);
			ptrace(PTRACE_CONT, NULL, NULL);
			ptrace(PTRACE_DETACH, getppid(), NULL, NULL);
			exit(0);
		}

		exit(1);
	}
	else if ((ret = waitpid(pid, &status, 0)) == pid && WIFEXITED(status) == true) {

		// Use a ternary to guarantee only two possible return values.
		return WEXITSTATUS(status) ? 1 : 0;
	}

	return -1;
}

/***
 * @return Will return -1 if the code is unable to determine whether profiling is active, 0 if profiling is disabled and
 *		1 if profiling has been detected.
 */
int_t running_on_profiler(void) {

	void *mcount = NULL;

	// This function is still causing problems, so we'll disable it for now.
	return 0;

	if ((mcount = dlsym(NULL, "_mcount"))) {
		return 1;
	}

	return 0;
}

/**
 * @brief	Show the check program usage information.
 * @see		display_usage()
 */
void check_display_help (chr_t *invalid_option) {

	log_info("%s%s%s" \
			"\tmagmad.check [options] [config_file]\n\n" \
			"\t%-25.25s\t\t%s\n\n\t%-25.25s\t\t%s\n\t%-25.25s\t\t%s\n\t%-25.25s\t\t%s\n\n\t%-25.25s\t\t%s\n\t%-25.25s\t\t%s\n\n",
			(invalid_option ? "The command line option \"" : ""), (invalid_option ? invalid_option : ""),
			(invalid_option ? "\" is invalid. Please consult the text below and try again.\n\n" : "\n"),
			"-c, --check NAME", "run a single unit test",
			"    --tank-path PATH",  "set the tank checker path, or disable the check if none is specified.",
			"    --virus-path PATH", "set the virus checker path, or disable the check if none is specified.",
			"    --disable-spf", "disable the SPF checker.",
			"-h, --help", "display the magma unit tester command line options and exit",
			"-v, --version", "display the magma unit tester version information and exit");

	return;
}

/**
 * @brief	Check to see if the specified command line option specified an optional parameter, and adjust the index accordingly.
 * @note	This function will automatically generate an error through display_usage() if the parameters are incorrect.
 * @param	xargv	a pointer to the main function's argv array.
 * @param	iptr	a pointer to the xargv option index to be checked and updated.
 * @param	xargc	the number of items in the xargv array.
 * @return	a pointer to the value of the current index's optional parameter if specified, or NULL if one wasn't.
 */
chr_t * check_next_opt(char *xargv[], int *iptr, int xargc) {

	chr_t *result;

	// If this is an optional parameter then there still must be a config file specified after it.
	if (*iptr == (xargc-1)) {
		return NULL;
	}
	// If the next argument begins with '-' then our option has a null parameter.
	else if (!mm_cmp_cs_eq(xargv[*iptr + 1], "-", 1)) {
		(*iptr)++;
		return NULL;
	}

	(*iptr) += 2;

	if (!(result = ns_dupe(xargv[*iptr - 1]))) {
		log_error("Memory allocation failure encountered while parsing the command line options.");
	}

	return result;
}

/**
 * @brief	Process any command line arguments supplied to magma unit tester.
 * @note	A few command line options are supported: -c (--check), -h (--help), -v (--version), and along with options for overriding
 * 			the hard coded file system paths, and the disabling the spf check. If the final option doesn't start with "-" then it's assumed
 * 			to be the config file path.
 * @return	Returns -1 if the program should exit with a failure code, returns 0 if the program should simply exit and returns 1 if the program should continue.
 */
int_t check_args_parse(int argc, char *argv[]) {

	int_t i = 1;

	while (i < argc) {

		if (!st_cmp_cs_eq(NULLER(argv[i]), PLACER("-c", 2)) || !st_cmp_cs_eq(NULLER(argv[i]), PLACER("--check", 7))) {

			if (barrister_unit_test) {
				log_error("The \"--check\" parameter may only be used once. Exiting.\n");
				return -1;
			}
			else if (!(barrister_unit_test = check_next_opt(argv, &i, argc))) {
				log_error("The individual unit test name is missing. Exiting.\n");
				return -1;
			}

		}
		else if (!st_cmp_cs_eq(NULLER(argv[i]), PLACER("--disable-spf", 13))) {
			do_spf_check = false;
			i++;
		}
		else if (!st_cmp_cs_eq(NULLER(argv[i]), PLACER("-v", 2)) || !st_cmp_cs_eq(NULLER(argv[i]), PLACER("--version", 9))) {
			log_info("\n\t%s\n\n\t%-20.20s %14.14s\n\t%-20.20s %14.14s\n\t%-20.20s %14.14s\n\n", "magmad.check",
				"version", build_version(), "commit", build_commit(), "build", build_stamp());
			return 0;
		}
		// Display the help if it's explicitly requested, or if we encounter an option we don't recognize.
		else if (!st_cmp_cs_eq(NULLER(argv[i]), PLACER("-h", 2)) || !st_cmp_cs_eq(NULLER(argv[i]), PLACER("--help", 6))) {
			check_display_help(NULL);
			return 0;
		}
		// Otherwise it's the config file
		else if (i == (argc - 1) && mm_cmp_cs_eq(argv[i], "-", 1)) {
			snprintf(magma.config.file, MAGMA_FILEPATH_MAX, "%s", argv[i]);
			i++;
		}
		else {
			check_display_help(argv[i]);
			return -1;
		}

	}

	return 1;
}

int main(int argc, char *argv[]) {

	SRunner *sr;
	int_t failed = 0, debugger = 0, profiler = 0, result;
	pthread_t *net_listen_thread = NULL;
	time_t prog_start, test_start, test_end;

	if (process_kill(PLACER("magmad", 6), SIGTERM, 10) < 0 || process_kill(PLACER("magmad.check", 12), SIGTERM, 10) < 0) {
		log_error("Another instance of the Magma Daemon is already running and refuses to die.");
		exit(EXIT_FAILURE);
	}

	// Setup
	prog_start = time(NULL);

	// Updates the location of the config file if it was specified on the command line.
	if ((result = check_args_parse(argc, argv)) != 1) {
		exit(result ? EXIT_FAILURE : EXIT_SUCCESS);
	}

	if (!process_start()) {
		log_error("Initialization error. Exiting.\n");
		status_set(-1);
		process_stop();
		exit(EXIT_FAILURE);
	}

	// Run the network listener using a background thread.
	net_listen_thread = thread_alloc(net_listen, NULL);

	// During development, we flush the cache system before running the unit tests.
	cache_flush();

	// A number of the test accounts, and values used involve using short passwords, so we set the minimum valid password
	// length to 4.
	magma.secure.minimum_password_length = 4;

	// Unit Test Config
	sr = srunner_create(suite_check_magma());

	// If the command line told us to run a specific test only add that individual test using the special barrister suite.
	if (barrister_unit_test) {
		srunner_add_suite(sr, suite_check_barrister(barrister_unit_test));
	}
	// Otherwise add all of the unit tests to the suite runner.
	else {
		srunner_add_suite(sr, suite_check_sample());
		srunner_add_suite(sr, suite_check_core());
		srunner_add_suite(sr, suite_check_provide());
		srunner_add_suite(sr, suite_check_prime());
		srunner_add_suite(sr, suite_check_config());
		srunner_add_suite(sr, suite_check_engine());
		srunner_add_suite(sr, suite_check_network());
		srunner_add_suite(sr, suite_check_objects());
		srunner_add_suite(sr, suite_check_users());
		srunner_add_suite(sr, suite_check_mail());
		srunner_add_suite(sr, suite_check_smtp());
		srunner_add_suite(sr, suite_check_pop());
		srunner_add_suite(sr, suite_check_imap());
		srunner_add_suite(sr, suite_check_http());
		srunner_add_suite(sr, suite_check_camel());
		srunner_add_suite(sr, suite_check_regression());
	}


	// Check whether valgrind, gdb, or gprof are active. Otherwise execute normally. The profiler check should be done
	// before the debugger check, as the latter involves forking, which will result in a spurious gprof output file if
	// if profiling is indeed active.
	if (RUNNING_ON_VALGRIND == 0 && (profiler = running_on_profiler()) == 0 && (debugger = running_on_debugger()) == 0) {
		log_unit("Not being traced or profiled...\n");
		srunner_set_fork_status (sr, CK_FORK);
		case_timeout = RUN_TEST_CASE_TIMEOUT;
	}

	// A profiler was detected.
	else if (profiler) {
		log_unit("A profiler has been detected...\n");
		srunner_set_fork_status (sr, CK_NOFORK);
		case_timeout = RUN_TEST_CASE_TIMEOUT;
	}

	// If were being run under Valgrind, or atop a debugger like gdb, we disable forking and increase the default timeout.
	else {
		if (debugger == -1) log_unit("Trace detection was thwarted.\n");
		else log_unit("Tracing or debugging is active...\n");
		srunner_set_fork_status (sr, CK_NOFORK);
		case_timeout = PROFILE_TEST_CASE_TIMEOUT;
	}

	// Execute
	log_unit("--------------------------------------------------------------------------\n");

	log_disable();
	test_start = time(NULL);
	srunner_run_all(sr, CK_SILENT);
	test_end = time(NULL);
	log_enable();

	// Output timing.
	log_unit("--------------------------------------------------------------------------\n");
	log_unit("%-63.63s %9lus\n", "TEST DURATION:", test_end - test_start);
	log_unit("%-63.63s %9lus\n", "TOTAL DURATION:", test_end - prog_start);

	// Summary
	log_unit("--------------------------------------------------------------------------\n");
	failed = srunner_ntests_failed(sr);
	srunner_print(sr, CK_NORMAL);

	// The Check Output Ending
	log_unit("--------------------------------------------------------------------------\n");

	// Cleanup and free the resources allocated by the check code.
	status_set(-1);
	srunner_free(sr);

	// Cleanup the background listening thread.
	net_trigger(false);

	thread_join(*net_listen_thread);
	mm_free(net_listen_thread);

	// Cleanup and free the resources allocated by the magma code.
	process_stop();

	ns_cleanup(barrister_unit_test);

	// We used a log file handle, we need to close it.
	if (log_descriptor) fclose(log_descriptor);

	// Close the console descriptors, if they are still valid.
	if ((errno = 0) || (fcntl(STDIN_FILENO, F_GETFL) != -1 && errno != EBADF)) close(STDIN_FILENO);
	if ((errno = 0) || (fcntl(STDOUT_FILENO, F_GETFL) != -1 && errno != EBADF)) close(STDOUT_FILENO);
	if ((errno = 0) || (fcntl(STDERR_FILENO, F_GETFL) != -1 && errno != EBADF)) close(STDERR_FILENO);

	exit((failed == 0) ? EXIT_SUCCESS : EXIT_FAILURE);

}
