export class Client {
    constructor() {
        this.endpoint = 'https://appwrite.io/v1';
        this.headers = {
            'content-type': '',
            'x-sdk-version': 'appwrite:deno:0.0.2',
        };
    }
    setProject(value) {
        this.addHeader('X-Appwrite-Project', value);
        return this;
    }
    setKey(value) {
        this.addHeader('X-Appwrite-Key', value);
        return this;
    }
    setLocale(value) {
        this.addHeader('X-Appwrite-Locale', value);
        return this;
    }
    setEndpoint(endpoint) {
        this.endpoint = endpoint;
        return this;
    }
    addHeader(key, value) {
        this.headers[key.toLowerCase()] = value.toLowerCase();
        return this;
    }
    withoutHeader(key, headers) {
        return Object.keys(headers).reduce((acc, cv) => {
            if (cv == 'content-type')
                return acc;
            acc[cv] = headers[cv];
            return acc;
        }, {});
    }
    async call(method, path = '', headers = {}, params = {}) {
        headers = Object.assign(this.headers, headers);
        let body;
        const url = new URL(this.endpoint + path);
        if (method.toUpperCase() === 'GET') {
            url.search = new URLSearchParams(this.flatten(params)).toString();
            body = null;
        }
        else if (headers['content-type'].toLowerCase().startsWith('multipart/form-data')) {
            headers = this.withoutHeader('content-type', headers);
            const formData = new FormData();
            const flatParams = this.flatten(params);
            for (const key in flatParams) {
                formData.append(key, flatParams[key]);
            }
            body = formData;
        }
        else {
            body = JSON.stringify(params);
        }
        const options = {
            method: method.toUpperCase(),
            headers: headers,
            body: body,
        };
        let response = await fetch(url.toString(), options);
        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
            return response.json();
        }
        return response;
    }
    flatten(data, prefix = '') {
        let output = {};
        for (const key in data) {
            let value = data[key];
            let finalKey = prefix ? prefix + '[' + key + ']' : key;
            if (Array.isArray(value)) {
                output = Object.assign(output, this.flatten(value, finalKey));
            }
            else {
                output[finalKey] = value;
            }
        }
        return output;
    }
}
//# sourceMappingURL=data:application/json;base64,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