package keys

import (
	"context"
	"encoding/json"
	"strings"

	"github.com/keys-pub/keys/docs"
	"github.com/keys-pub/keys/tsutil"
	"github.com/pkg/errors"
)

// Sigchains stores sigchains.
type Sigchains struct {
	ds    docs.Documents
	clock tsutil.Clock
}

// NewSigchains creates a Sigchains from Documents.
func NewSigchains(ds docs.Documents) *Sigchains {
	return &Sigchains{
		ds:    ds,
		clock: tsutil.NewClock(),
	}
}

// SetClock to use a custom time.Now.
func (s *Sigchains) SetClock(clock tsutil.Clock) {
	s.clock = clock
}

// KIDs returns all keys.
func (s *Sigchains) KIDs() ([]ID, error) {
	iter, err := s.ds.DocumentIterator(context.TODO(), "sigchain", docs.NoData())
	if err != nil {
		return nil, err
	}
	ids := NewIDSet()
	for {
		doc, err := iter.Next()
		if err != nil {
			return nil, err
		}
		if doc == nil {
			break
		}
		pc := docs.PathLast(doc.Path)
		str := strings.Split(pc, "-")[0]
		id, err := ParseID(str)
		if err != nil {
			return nil, errors.Wrapf(err, "invalid path %q", doc.Path)
		}
		ids.Add(id)
	}
	iter.Release()
	return ids.IDs(), nil
}

// Save sigchain.
func (s *Sigchains) Save(sc *Sigchain) error {
	if len(sc.Statements()) == 0 {
		return errors.Errorf("failed to save sigchain: no statements")
	}
	for _, st := range sc.Statements() {
		b, err := st.Bytes()
		if err != nil {
			return err
		}
		if err := s.ds.Set(context.TODO(), docs.Path("sigchain", st.Key()), b); err != nil {
			return err
		}
	}
	if err := s.Index(sc.KID()); err != nil {
		return err
	}
	return nil
}

func statementFromDocument(doc *docs.Document) (*Statement, error) {
	var st Statement
	if err := json.Unmarshal(doc.Data, &st); err != nil {
		return nil, err
	}
	return &st, nil
}

// Sigchain returns sigchain for key.
func (s *Sigchains) Sigchain(kid ID) (*Sigchain, error) {
	logger.Debugf("Loading sigchain %s", kid)
	iter, err := s.ds.DocumentIterator(context.TODO(), "sigchain", docs.Prefix(kid.String()))
	if err != nil {
		return nil, err
	}

	sc := NewSigchain(kid)
	for {
		doc, err := iter.Next()
		if err != nil {
			return nil, err
		}
		if doc == nil {
			break
		}
		st, err := statementFromDocument(doc)
		if err != nil {
			return nil, err
		}
		if err := sc.Add(st); err != nil {
			return nil, err
		}
	}

	iter.Release()
	return sc, nil
}

func (s *Sigchains) sigchainPaths(kid ID) ([]string, error) {
	iter, err := s.ds.DocumentIterator(context.TODO(), "sigchain", docs.Prefix(kid.String()), docs.NoData())
	if err != nil {
		return nil, err
	}
	defer iter.Release()
	paths := make([]string, 0, 100)
	for {
		doc, err := iter.Next()
		if err != nil {
			return nil, err
		}
		if doc == nil {
			break
		}
		paths = append(paths, doc.Path)
	}
	return paths, nil
}

// Delete sigchain.
func (s *Sigchains) Delete(kid ID) (bool, error) {
	paths, err := s.sigchainPaths(kid)
	if err != nil {
		return false, err
	}

	if len(paths) == 0 {
		return false, nil
	}

	for _, path := range paths {
		if _, err := s.ds.Delete(context.TODO(), path); err != nil {
			return false, err
		}
	}

	// TODO: Delete reverse key lookup?
	return true, nil
}

// Exists returns true if sigchain exists.
func (s *Sigchains) Exists(kid ID) (bool, error) {
	return s.ds.Exists(context.TODO(), docs.Path("sigchain", StatementKey(kid, 1)))
}

// indexRKL is collection for reverse key lookups.
const indexRKL = "rkl"

// Lookup key identifier.
func (s *Sigchains) Lookup(kid ID) (ID, error) {
	path := docs.Path(indexRKL, kid.String())
	doc, err := s.ds.Get(context.TODO(), path)
	if err != nil {
		return "", err
	}
	if doc == nil {
		return "", nil
	}
	rkid, err := ParseID(string(doc.Data))
	if err != nil {
		return "", err
	}
	return rkid, nil
}

// Index key identifier.
func (s *Sigchains) Index(key Key) error {
	if key.Type() == EdX25519Public {
		rk, err := Convert(key, X25519Public)
		if err != nil {
			return err
		}
		rklPath := docs.Path(indexRKL, rk.ID())
		if err := s.ds.Set(context.TODO(), rklPath, []byte(key.ID().String())); err != nil {
			return err
		}
	}
	return nil
}
