package loncha

import (
	"golang.org/x/exp/constraints"
)

type curryParam[T any] struct {
	Default T
}

// OptCurry ... functional option
type OptCurry[T any] func(*curryParam[T]) OptCurry[T]

func Default[T any](t T) OptCurry[T] {

	return func(p *curryParam[T]) OptCurry[T] {
		prev := p.Default
		p.Default = t
		return Default(prev)
	}
}

func (p *curryParam[T]) Options(opts ...OptCurry[T]) (prevs []OptCurry[T]) {

	for _, opt := range opts {
		prevs = append(prevs, opt(p))
	}
	return
}

func NewOpt[T any](opts ...OptCurry[T]) *curryParam[T] {

	p := curryParam[T]{}
	p.Options(opts...)
	return &p
}

// FilterFunc ... function  generated by Filterlize()
type FilterFunc[T any] func([]T) []T

type GetFunc[T any] func([]T) T

// Filterable ... generate filter function for slice
func Filterable[T any](fns ...CondFunc2[T]) FilterFunc[T] {
	return innerfilterlable(true, fns...)
}

// Deletable ... generate deleting function by fns Condition for slice
func Deletable[T comparable](fns ...CondFunc2[T]) FilterFunc[T] {
	return innerfilterlable(false, fns...)
}

// Selectable ... generate deleting function by fns Condition for slice
func Selectable[T any](fns ...CondFunc2[T]) FilterFunc[T] {
	return Filterable(fns...)
}

// Gettable	 ... generate deleting function by fns Condition for slice
func Gettable[T any](fns ...CondFunc2[T]) GetFunc[T] {
	return func(datas []T) T {
		result := Filterable(fns...)(datas)
		if len(result) > 0 {
			return result[0]
		}
		return *new(T)
	}
}

func innerfilterlable[T any](keep bool, fns ...CondFunc2[T]) FilterFunc[T] {
	return func(srcs []T) (dsts []T) {
		dsts = srcs
		innerFilter2(&dsts, keep, fns...)
		return
	}
}

// InjecterFunc ... condition function for Injectable
type InjecterFunc[T any, R any] func([]T) R

// Injectable ... generate Inject functions
func Injectable[T any, V any](injectFn InjectFn[T, V], opts ...OptCurry[V]) InjecterFunc[T, V] {

	return func(src []T) (result V) {
		return Inject(src, injectFn, opts...)
	}
}

// Reducable ... alias of Injectable
func Reducable[T any, V any](injectFn InjectFn[T, V], opts ...OptCurry[V]) InjecterFunc[T, V] {
	return Injectable(injectFn, opts...)
}

// Containable ... generate function of slice contain.
func Containable[T comparable](fn CondFunc2[T]) func([]T) bool {

	return func(srcs []T) bool {
		for _, src := range srcs {
			if fn(&src) {
				return true
			}
		}
		return false
	}
}

// Every ... Determines whether all the members of an array satisfy the specified test.
func Every[T any](fn CondFunc2[T]) func(...T) bool {

	return func(srcs ...T) bool {
		result := true
		for _, src := range srcs {
			if result && !fn(&src) {
				result = false
			}
		}
		return result
	}
}

// EveryWithIndex ... Determines whether all the members of an array satisfy the specified test.
func EveryWithIndex[T comparable](fn CondFuncWithIndex[T]) func(...T) bool {

	return func(srcs ...T) bool {
		result := true
		for i, src := range srcs {
			if result && !fn(i, &src) {
				result = false
			}
		}
		return result
	}
}

// Convertable ...  generate function of slice conversion.
func Convertable[S, D any](fn ConvFunc[S, D]) func([]S) []D {
	return func(srcs []S) (dsts []D) {
		dsts = []D{}

		for _, src := range srcs {
			if d, removed := fn(src); !removed {
				dsts = append(dsts, d)
			}
		}
		return
	}
}

// Number ... Number constraints
type Number interface {
	constraints.Integer | constraints.Float
}

func max[T Number](s T, d T) T {

	if s < d {
		return d
	}
	return s
}

func min[T Number](s T, d T) T {
	return -max(-s, -d)
}

// Zipper ...  return tuple operation result using fn  with double slice,
func Zipper[T, S, R any](fn func(R, T, S) R, start R) func([]T, []S) R {

	return func(tlist []T, slist []S) (result R) {

		result = start
		for i := range tlist[:min(len(tlist), len(slist))] {
			result = fn(result, tlist[i], slist[i])
		}
		return
	}

}

// ToMap ... function for Zipper . return map from double array.
func ToMap[K, V constraints.Ordered](r map[K]V, k K, v V) map[K]V {

	r[k] = v
	return r

}
