package tool

import (
	"bytes"
	"html/template"

	"github.com/pkg/errors"
	"github.com/spf13/afero"
)

// Writer creates a tool file to manage tool dependencies.
type Writer interface {
	Write(path string, m *Manifest) error
}

// NewWriter creates a new Writer instance.
func NewWriter(fs afero.Fs) Writer {
	return &writerImpl{
		fs: fs,
	}
}

type writerImpl struct {
	fs afero.Fs
}

func (w *writerImpl) Write(path string, m *Manifest) error {
	buf := new(bytes.Buffer)
	err := toolsGoTemplate.Execute(buf, m)
	if err != nil {
		return errors.Wrap(err, "failed to create a manifest file")
	}
	err = afero.WriteFile(w.fs, path, buf.Bytes(), 0644)
	if err != nil {
		return errors.Wrap(err, "failed to write a manifest file")
	}
	return nil
}

var (
	toolsGoTemplate = template.Must(template.New("tools.go").Parse(`// Code generated by github.com/izumin5210/gex. DO NOT EDIT.

// +build tools

package tools

// tool dependencies
import (
{{- range $t := .Tools}}
	_ "{{$t}}"
{{- end}}
)

// If you want to use tools, please run the following command:
//  go generate ./tools.go
//
{{- range $t := .Tools}}
//go:generate go build -v -o=./bin/{{$t.Name}} {{if $.ManagerType.Vendor}}./vendor/{{end}}{{$t}}
{{- end}}
`))
)
