/*
 * The MIT License
 *
 * Copyright (c) 2013-2014 Karol Bucek LTD.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.jruby.rack.embed;

import java.io.OutputStream;
import java.io.PrintStream;
import java.util.Map;

import org.jruby.CompatVersion;
import org.jruby.Ruby;
import org.jruby.rack.DefaultRackConfig;
import org.jruby.rack.RackConfig;
import org.jruby.rack.RackLogger;

/**
 * A rack config for an embedded environment.
 *
 * @author kares
 */
public class Config implements RackConfig {

    private final DefaultRackConfig delegate;

    private RackLogger logger;
    private Map<String, String> rubyENV;
    private CompatVersion compatVersion;

    public Config() {
        delegate = new DefaultRackConfig() {

            @Override
            public String getProperty(String key, String defaultValue) {
                String value = Config.this.resolveProperty(key);
                return value != null ? value : super.getProperty(key, defaultValue);
            }

            @Override
            public RackLogger defaultLogger() { return null; }

        };
    }

    @SuppressWarnings("unchecked")
    public void doInitialize(final Ruby runtime) {
        setOut( runtime.getOut() );
        setErr( runtime.getErr() );
        rubyENV = runtime.getENV();
        compatVersion = runtime.getInstanceConfig().getCompatVersion();
    }


    protected String resolveProperty(String key) {
        String value = null;
        if ( rubyENV != null ) value = rubyENV.get(key);
        return value;
    }

    @Override
    public final String getProperty(String key) {
        return delegate.getProperty(key);
    }

    @Override
    public final String getProperty(String key, String defaultValue) {
        return delegate.getProperty(key, defaultValue);
    }

    @Override
    public final Boolean getBooleanProperty(String key) {
        return delegate.getBooleanProperty(key);
    }

    @Override
    public final Boolean getBooleanProperty(String key, Boolean defaultValue) {
        return delegate.getBooleanProperty(key, defaultValue);
    }

    @Override
    public final Number getNumberProperty(String key) {
        return delegate.getNumberProperty(key);
    }

    @Override
    public final Number getNumberProperty(String key, Number defaultValue) {
        return delegate.getNumberProperty(key, defaultValue);
    }

    @Override
    public CompatVersion getCompatVersion() {
        return compatVersion;
    }

    @Override
    public RackLogger getLogger() {
        if (logger == null) {
            logger = delegate.getLogger();
        }
        return logger;
    }

    public void setLogger(RackLogger logger) {
        this.logger = logger;
    }

    @Override
    public PrintStream getOut() {
        return delegate.getOut();
    }

    public void setOut(OutputStream out) {
        delegate.setOut(out);
    }

    @Override
    public PrintStream getErr() {
        return delegate.getErr();
    }

    public void setErr(OutputStream err) {
        delegate.setErr(err);
    }

    @Override
    public boolean isRewindable() {
        return delegate.isRewindable();
    }

    @Override
    public Integer getInitialMemoryBufferSize() {
        return delegate.getInitialMemoryBufferSize();
    }

    @Override
    public Integer getMaximumMemoryBufferSize() {
        return delegate.getMaximumMemoryBufferSize();
    }

    @Override
    public String getRackup() {
        return delegate.getRackup();
    }

    @Override
    public String getRackupPath() {
        return delegate.getRackupPath();
    }

    // runtime pooling in embedded ENVs not implemented :

    @Override
    public Integer getRuntimeTimeoutSeconds() {
        throw new UnsupportedOperationException("getRuntimeTimeoutSeconds()");
    }

    @Override
    public Integer getInitialRuntimes() {
        throw new UnsupportedOperationException("getInitialRuntimes()");
    }

    @Override
    public Integer getMaximumRuntimes() {
        throw new UnsupportedOperationException("getMaximumRuntimes()");
    }

    @Override
    public String[] getRuntimeArguments() {
        throw new UnsupportedOperationException("getRuntimeArguments()");
    }

    @Override
    public Integer getNumInitializerThreads() {
        throw new UnsupportedOperationException("getNumInitializerThreads()");
    }

    @Override
    public boolean isSerialInitialization() {
        throw new UnsupportedOperationException("isSerialInitialization()");
    }

    @Override
    public boolean isIgnoreEnvironment() {
        throw new UnsupportedOperationException("isIgnoreEnvironment()");
    }

    @Override
    public Map<String, String> getRuntimeEnvironment() {
        throw new UnsupportedOperationException("getRuntimeEnvironment()");
    }

    // RackFilter aint's used with embed :

    @Override
    public boolean isFilterAddsHtml() {
        throw new UnsupportedOperationException("isFilterAddsHtml()");
    }

    @Override
    public boolean isFilterVerifiesResource() {
        throw new UnsupportedOperationException("isFilterVerifiesResource()");
    }

    // JMS configuration not used with embed :

    @Override
    public String getJmsConnectionFactory() {
        throw new UnsupportedOperationException("getJmsConnectionFactory()");
    }

    @Override
    public String getJmsJndiProperties() {
        throw new UnsupportedOperationException("getJmsJndiProperties()");
    }

}
