package filesys_test

import (
	"path/filepath"
	"testing"

	. "sigs.k8s.io/kustomize/api/filesys"
)

// Confirm behavior of filepath.Match
func TestFilePathMatch(t *testing.T) {
	cases := []struct {
		pattern  string
		path     string
		expected bool
	}{
		{
			pattern:  "*e*",
			path:     "hey",
			expected: true,
		},
		{
			pattern:  "*e*",
			path:     "hay",
			expected: false,
		},
		{
			pattern:  "*e*",
			path:     filepath.Join("h", "e", "y"),
			expected: false,
		},
		{
			pattern:  "*/e/*",
			path:     filepath.Join("h", "e", "y"),
			expected: true,
		},
		{
			pattern:  "h/e/*",
			path:     filepath.Join("h", "e", "y"),
			expected: true,
		},
		{
			pattern:  "*/e/y",
			path:     filepath.Join("h", "e", "y"),
			expected: true,
		},
		{
			pattern:  "*/*/*",
			path:     filepath.Join("h", "e", "y"),
			expected: true,
		},
		{
			pattern:  "*/*/*",
			path:     filepath.Join("h", "e", "y", "there"),
			expected: false,
		},
		{
			pattern:  "*/*/*/t*e",
			path:     filepath.Join("h", "e", "y", "there"),
			expected: true,
		},
	}
	for _, item := range cases {
		match, err := filepath.Match(item.pattern, item.path)
		if err != nil {
			t.Fatalf("unexpected err: %v", err)
		}
		if match != item.expected {
			t.Fatalf("'%s' '%s' %v\n", item.pattern, item.path, match)
		}
	}
}

// Confirm behavior of filepath.Split
func TestFilePathSplit(t *testing.T) {
	cases := []struct {
		full string
		dir  string
		file string
	}{
		{
			full: "",
			dir:  "",
			file: "",
		},
		{
			full: SelfDir,
			dir:  "",
			file: SelfDir,
		},
		{
			full: "rabbit.jpg",
			dir:  "",
			file: "rabbit.jpg",
		},
		{
			full: "/beans",
			dir:  "/",
			file: "beans",
		},
		{
			full: "/home/foo/bar",
			dir:  "/home/foo/",
			file: "bar",
		},
		{
			full: "/usr/local/",
			dir:  "/usr/local/",
			file: "",
		},
		{
			full: "/usr//local//go",
			dir:  "/usr//local//",
			file: "go",
		},
	}
	for _, p := range cases {
		dir, file := filepath.Split(p.full)
		if dir != p.dir || file != p.file {
			t.Fatalf(
				"in '%s',\ngot dir='%s' (expected '%s'),\n got file='%s' (expected %s).",
				p.full, dir, p.dir, file, p.file)
		}
	}
}

func TestStripTrailingSeps(t *testing.T) {
	cases := []struct {
		full string
		rem  string
	}{
		{
			full: "foo",
			rem:  "foo",
		},
		{
			full: "",
			rem:  "",
		},
		{
			full: "foo/",
			rem:  "foo",
		},
		{
			full: "foo///bar///",
			rem:  "foo///bar",
		},
		{
			full: "/////",
			rem:  "",
		},
		{
			full: "/",
			rem:  "",
		},
	}
	for _, p := range cases {
		dir := StripTrailingSeps(p.full)
		if dir != p.rem {
			t.Fatalf(
				"in '%s', got dir='%s' (expected '%s')",
				p.full, dir, p.rem)
		}
	}
}

func TestStripLeadingSeps(t *testing.T) {
	cases := []struct {
		full string
		rem  string
	}{
		{
			full: "foo",
			rem:  "foo",
		},
		{
			full: "",
			rem:  "",
		},
		{
			full: "/foo",
			rem:  "foo",
		},
		{
			full: "///foo///bar///",
			rem:  "foo///bar///",
		},
		{
			full: "/////",
			rem:  "",
		},
		{
			full: "/",
			rem:  "",
		},
	}
	for _, p := range cases {
		dir := StripLeadingSeps(p.full)
		if dir != p.rem {
			t.Fatalf(
				"in '%s', got dir='%s' (expected '%s')",
				p.full, dir, p.rem)
		}
	}
}
