require 'digest'

module Colore
  # This is the unique identifier for a document, a composite of
  # application name (app) and document identifier (doc_id). Doc_id
  # is generated by the application, so we can only guarantee
  # uniqueness by combining with app name.
  class DocKey
    STORAGE_SUBDIRECTORY_LENGTH = 2

    attr_accessor :app
    attr_accessor :doc_id

    # Constructs [DocKey] from the input string, which should be the output from [#to_s].
    def self.parse doc_key_str
      self.new *(doc_key_str.split '/')
    end

    # Constructor. The app and doc_id may only be comprised of alphanumeric letters and
    # the special characters "_" and "-".
    # @param app [String] the app name
    # @param doc_id [String] the document identifier
    def initialize app, doc_id
      validate(app)
      validate(doc_id)
      @app = app
      @doc_id = doc_id
    end

    # Doc key as path
    def path
      Pathname.new(app) + subdirectory + doc_id
    end

    # Doc key as string
    def to_s
      "#{@app}/#{@doc_id}"
    end

    # Returns the first 2 characters of a SHA256 hash for this doc, suitable for
    # subdirectory names into which the docs can be distributed, rather than clumping
    # them into a single base directory (which will not scale well)
    def subdirectory
      Digest::MD5.hexdigest(@doc_id)[0..(STORAGE_SUBDIRECTORY_LENGTH-1)]
    end

    private

    # Validates the parameter (app or doc_id)
    def validate val
      raise InvalidParameter.new unless val =~ /^[A-Za-z0-9_-]+$/
    end
  end
end
