<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */

namespace Ibexa\Tests\Core\MVC\Symfony\View;

use Ibexa\Contracts\Core\Repository\ContentService as APIContentService;
use Ibexa\Contracts\Core\Repository\Repository;
use Ibexa\Contracts\Core\Repository\Values\Content\ContentInfo;
use Ibexa\Contracts\Core\SiteAccess\ConfigResolverInterface;
use Ibexa\Core\MVC\Symfony\View\Configurator;
use Ibexa\Core\MVC\Symfony\View\Manager;
use Ibexa\Core\MVC\Symfony\View\View;
use Ibexa\Core\Repository\ContentService;
use Ibexa\Core\Repository\Values\Content\Content;
use Ibexa\Core\Repository\Values\Content\Location;
use Ibexa\Core\Repository\Values\Content\VersionInfo;
use PHPUnit\Framework\TestCase;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Twig\Environment;

/**
 * @group mvc
 */
class ViewManagerTest extends TestCase
{
    /** @var \Ibexa\Core\MVC\Symfony\View\Manager */
    private $viewManager;

    /** @var \PHPUnit\Framework\MockObject\MockObject|\Twig\Environment */
    private $templateEngineMock;

    /** @var \PHPUnit\Framework\MockObject\MockObject|\Symfony\Component\EventDispatcher\EventDispatcherInterface */
    private $eventDispatcherMock;

    /** @var \PHPUnit\Framework\MockObject\MockObject|\Ibexa\Contracts\Core\Repository\Repository */
    private $repositoryMock;

    /** @var \PHPUnit\Framework\MockObject\MockObject|\Ibexa\Contracts\Core\SiteAccess\ConfigResolverInterface */
    private $configResolverMock;

    /** @var \Ibexa\Core\MVC\Symfony\View\Configurator|\PHPUnit\Framework\MockObject\MockObject */
    private $viewConfigurator;

    private $viewBaseLayout = 'IbexaCoreBundle::viewbase.html.twig';

    protected function setUp(): void
    {
        parent::setUp();
        $this->templateEngineMock = $this->createMock(Environment::class);
        $this->eventDispatcherMock = $this->createMock(EventDispatcherInterface::class);
        $this->repositoryMock = $this->createMock(Repository::class);
        $this->configResolverMock = $this->createMock(ConfigResolverInterface::class);
        $this->viewConfigurator = $this->createMock(Configurator::class);
        $this->viewManager = new Manager(
            $this->templateEngineMock,
            $this->eventDispatcherMock,
            $this->repositoryMock,
            $this->configResolverMock,
            $this->viewBaseLayout,
            $this->viewConfigurator
        );
    }

    public function testRenderContent()
    {
        $content = new Content(
            ['versionInfo' => new VersionInfo(['contentInfo' => new ContentInfo()])]
        );

        $params = ['foo' => 'bar'];
        $templateIdentifier = 'foo:bar:baz';
        $this->viewConfigurator
            ->expects(self::once())
            ->method('configure')
            ->will(
                self::returnCallback(
                    static function (View $view) use ($templateIdentifier) {
                        $view->setTemplateIdentifier($templateIdentifier);
                    }
                )
            );

        // Configuring template engine behaviour
        $expectedTemplateResult = 'This is content rendering';
        $this->templateEngineMock
            ->expects(self::once())
            ->method('render')
            ->with(
                $templateIdentifier,
                $params + ['content' => $content, 'view_base_layout' => $this->viewBaseLayout]
            )
            ->will(self::returnValue($expectedTemplateResult));

        self::assertSame($expectedTemplateResult, $this->viewManager->renderContent($content, 'customViewType', $params));
    }

    public function testRenderContentWithClosure()
    {
        $content = new Content(
            ['versionInfo' => new VersionInfo(['contentInfo' => new ContentInfo()])]
        );

        // Configuring view provider behaviour
        $closure = static function ($params): string {
            return serialize(array_keys($params));
        };
        $params = ['foo' => 'bar'];
        $this->viewConfigurator
            ->expects(self::once())
            ->method('configure')
            ->will(
                self::returnCallback(
                    static function (View $view) use ($closure) {
                        $view->setTemplateIdentifier($closure);
                    }
                )
            );

        // Configuring template engine behaviour
        $params += ['content' => $content, 'view_base_layout' => $this->viewBaseLayout];
        $expectedTemplateResult = array_keys($params);
        $this->templateEngineMock
            ->expects(self::never())
            ->method('render');

        $templateResult = unserialize($this->viewManager->renderContent($content, 'full', $params));

        self::assertEqualsCanonicalizing($expectedTemplateResult, $templateResult);
    }

    public function testRenderLocation()
    {
        $content = new Content(['versionInfo' => new VersionInfo(['contentInfo' => new ContentInfo()])]);
        $location = new Location(['contentInfo' => new ContentInfo()]);

        // Configuring view provider behaviour
        $templateIdentifier = 'foo:bar:baz';
        $params = ['foo' => 'bar'];
        $this->viewConfigurator
            ->expects(self::once())
            ->method('configure')
            ->will(
                self::returnCallback(
                    static function (View $view) use ($templateIdentifier) {
                        $view->setTemplateIdentifier($templateIdentifier);
                    }
                )
            );

        $languages = ['eng-GB'];
        $this->configResolverMock
            ->expects(self::any())
            ->method('getParameter')
            ->with('languages')
            ->will(self::returnValue($languages));

        $contentService = $this->createMock(APIContentService::class);

        $contentService->expects(self::any())
            ->method('loadContentByContentInfo')
            ->with($location->contentInfo, $languages)
            ->will(self::returnValue($content));

        $this->repositoryMock
            ->expects(self::any())
            ->method('getContentService')
            ->will(self::returnValue($contentService));

        // Configuring template engine behaviour
        $expectedTemplateResult = 'This is location rendering';
        $this->templateEngineMock
            ->expects(self::once())
            ->method('render')
            ->with($templateIdentifier, $params + ['location' => $location, 'content' => $content, 'view_base_layout' => $this->viewBaseLayout])
            ->will(self::returnValue($expectedTemplateResult));

        self::assertSame($expectedTemplateResult, $this->viewManager->renderLocation($location, 'customViewType', $params));
    }

    public function testRenderLocationWithContentPassed()
    {
        $content = new Content(['versionInfo' => new VersionInfo(['contentInfo' => new ContentInfo()])]);
        $location = new Location(['contentInfo' => new ContentInfo()]);

        // Configuring view provider behaviour
        $templateIdentifier = 'foo:bar:baz';
        $params = ['foo' => 'bar', 'content' => $content];
        $this->viewConfigurator
            ->expects(self::once())
            ->method('configure')
            ->will(
                self::returnCallback(
                    static function (View $view) use ($templateIdentifier) {
                        $view->setTemplateIdentifier($templateIdentifier);
                    }
                )
            );

        $contentService = $this->createMock(ContentService::class);

        $contentService->expects(self::any())
            ->method('loadContentByContentInfo')
            ->with($content->contentInfo)
            ->will(
                self::returnValue($content)
            );

        $this->repositoryMock
            ->expects(self::any())
            ->method('getContentService')
            ->will(
                self::returnValue(
                    $contentService
                )
            );

        // Configuring template engine behaviour
        $expectedTemplateResult = 'This is location rendering';
        $this->templateEngineMock
            ->expects(self::once())
            ->method('render')
            ->with(
                $templateIdentifier,
                $params + ['location' => $location, 'content' => $content, 'view_base_layout' => $this->viewBaseLayout]
            )
            ->will(self::returnValue($expectedTemplateResult));

        self::assertSame($expectedTemplateResult, $this->viewManager->renderLocation($location, 'customViewType', $params));
    }

    public function testRenderLocationWithClosure()
    {
        $content = new Content(['versionInfo' => new VersionInfo(['contentInfo' => new ContentInfo()])]);
        $location = new Location(['contentInfo' => new ContentInfo()]);

        // Configuring view provider behaviour
        $closure = static function ($params): string {
            return serialize(array_keys($params));
        };
        $params = ['foo' => 'bar'];
        $this->viewConfigurator
            ->expects(self::once())
            ->method('configure')
            ->will(
                self::returnCallback(
                    static function (View $view) use ($closure) {
                        $view->setTemplateIdentifier($closure);
                    }
                )
            );

        $contentService = $this->createMock(ContentService::class);

        $contentService->expects(self::any())
            ->method('loadContentByContentInfo')
            ->with($content->contentInfo)
            ->will(
                self::returnValue($content)
            );

        $this->repositoryMock
            ->expects(self::any())
            ->method('getContentService')
            ->will(
                self::returnValue(
                    $contentService
                )
            );

        // Configuring template engine behaviour
        $params += ['location' => $location, 'content' => $content, 'view_base_layout' => $this->viewBaseLayout];
        $this->templateEngineMock
            ->expects(self::never())
            ->method('render');

        $expectedTemplateResult = array_keys($params);
        $templateResult = unserialize($this->viewManager->renderLocation($location, 'full', $params));
        sort($expectedTemplateResult);
        sort($templateResult);

        self::assertSame($expectedTemplateResult, $templateResult);
    }
}
