<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */
declare(strict_types=1);

namespace Ibexa\Tests\Core\MVC\Symfony\Templating\Twig;

use Ibexa\Contracts\Core\SiteAccess\ConfigResolverInterface;
use Ibexa\Core\MVC\Symfony\Templating\Twig\ResourceProvider;
use PHPUnit\Framework\TestCase;

class ResourceProviderTest extends TestCase
{
    /** @var \Ibexa\Contracts\Core\SiteAccess\ConfigResolverInterface */
    protected $configResolver;

    /** @var \Ibexa\Core\MVC\Symfony\Templating\Twig\ResourceProvider */
    protected $resourceProvider;

    protected function setUp(): void
    {
        parent::setUp();

        $this->configResolver = $this->getConfigResolverMock();
        $this->resourceProvider = new ResourceProvider($this->configResolver);
    }

    public function testGetFieldViewResources(): void
    {
        $resources = $this->resourceProvider->getFieldViewResources();
        $templates = array_column($resources, 'template');
        $priorities = array_column($resources, 'priority');

        self::assertEquals('templates/fields_override1.html.twig', array_shift($templates));
        self::assertEquals('templates/fields_override2.html.twig', array_shift($templates));
        self::assertEquals('templates/fields_default.html.twig', array_shift($templates));

        self::assertEquals(10, array_shift($priorities));
        self::assertEquals(20, array_shift($priorities));
        self::assertEquals(0, array_shift($priorities));
    }

    public function testGetFieldEditResources(): void
    {
        $resources = $this->resourceProvider->getFieldEditResources();
        $templates = array_column($resources, 'template');
        $priorities = array_column($resources, 'priority');

        self::assertEquals('templates/fields_override1.html.twig', array_shift($templates));
        self::assertEquals('templates/fields_override2.html.twig', array_shift($templates));
        self::assertEquals('templates/fields_default.html.twig', array_shift($templates));

        self::assertEquals(10, array_shift($priorities));
        self::assertEquals(20, array_shift($priorities));
        self::assertEquals(0, array_shift($priorities));
    }

    public function testGetFieldDefinitionViewResources(): void
    {
        $resources = $this->resourceProvider->getFieldDefinitionViewResources();
        $templates = array_column($resources, 'template');
        $priorities = array_column($resources, 'priority');

        self::assertEquals('templates/settings_override1.html.twig', array_shift($templates));
        self::assertEquals('templates/settings_override2.html.twig', array_shift($templates));
        self::assertEquals('templates/settings_default.html.twig', array_shift($templates));

        self::assertEquals(10, array_shift($priorities));
        self::assertEquals(20, array_shift($priorities));
        self::assertEquals(0, array_shift($priorities));
    }

    public function testGetFieldDefinitionEditResources(): void
    {
        $resources = $this->resourceProvider->getFieldDefinitionEditResources();
        $templates = array_column($resources, 'template');
        $priorities = array_column($resources, 'priority');

        self::assertEquals('templates/settings_override1.html.twig', array_shift($templates));
        self::assertEquals('templates/settings_override2.html.twig', array_shift($templates));
        self::assertEquals('templates/settings_default.html.twig', array_shift($templates));

        self::assertEquals(10, array_shift($priorities));
        self::assertEquals(20, array_shift($priorities));
        self::assertEquals(0, array_shift($priorities));
    }

    /**
     * Returns mocked ConfigResolver.
     *
     * Make sure returned resource lists are not sorted as ResourceProvider is sorting them
     *
     * @return \Ibexa\Contracts\Core\SiteAccess\ConfigResolverInterface|\PHPUnit\Framework\MockObject\MockObject
     */
    protected function getConfigResolverMock(): ConfigResolverInterface
    {
        $mock = $this->createMock(ConfigResolverInterface::class);
        $mock
            ->method('getParameter')
            ->willReturnMap([
                [
                    'field_templates',
                    null,
                    null,
                    [
                        [
                            'template' => 'templates/fields_override1.html.twig',
                            'priority' => 10,
                        ],
                        [
                            'template' => 'templates/fields_override2.html.twig',
                            'priority' => 20,
                        ],
                        [
                            'template' => 'templates/fields_default.html.twig',
                            'priority' => 0,
                        ],
                    ],
                ],
                [
                    'field_edit_templates',
                    null,
                    null,
                    [
                        [
                            'template' => 'templates/fields_override1.html.twig',
                            'priority' => 10,
                        ],
                        [
                            'template' => 'templates/fields_override2.html.twig',
                            'priority' => 20,
                        ],
                        [
                            'template' => 'templates/fields_default.html.twig',
                            'priority' => 0,
                        ],
                    ],
                ],
                [
                    'fielddefinition_settings_templates',
                    null,
                    null,
                    [
                        [
                            'template' => 'templates/settings_override1.html.twig',
                            'priority' => 10,
                        ],
                        [
                            'template' => 'templates/settings_override2.html.twig',
                            'priority' => 20,
                        ],
                        [
                            'template' => 'templates/settings_default.html.twig',
                            'priority' => 0,
                        ],
                    ],
                ],
                [
                    'fielddefinition_edit_templates',
                    null,
                    null,
                    [
                        [
                            'template' => 'templates/settings_override1.html.twig',
                            'priority' => 10,
                        ],
                        [
                            'template' => 'templates/settings_override2.html.twig',
                            'priority' => 20,
                        ],
                        [
                            'template' => 'templates/settings_default.html.twig',
                            'priority' => 0,
                        ],
                    ],
                ],
            ])
        ;

        return $mock;
    }
}
