<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */

namespace Ibexa\Tests\Integration\Core\Repository\FieldType;

use Ibexa\Contracts\Core\Repository\Values\Content\Field;
use Ibexa\Core\Base\Exceptions\InvalidArgumentType;
use Ibexa\Core\FieldType\MapLocation\Value as MapLocationValue;

/**
 * Integration test for use field type.
 *
 * @group integration
 * @group field-type
 */
class MapLocationIntegrationTest extends BaseIntegrationTestCase
{
    /**
     * Get name of tested field type.
     *
     * @return string
     */
    public function getTypeName(): string
    {
        return 'ibexa_gmap_location';
    }

    /**
     * Get expected settings schema.
     *
     * @return array
     */
    public function getSettingsSchema()
    {
        return [];
    }

    /**
     * Get a valid $fieldSettings value.
     *
     * @return mixed
     */
    public function getValidFieldSettings()
    {
        return [];
    }

    /**
     * Get $fieldSettings value not accepted by the field type.
     *
     * @return mixed
     */
    public function getInvalidFieldSettings()
    {
        return [
            'somethingUnknown' => 0,
        ];
    }

    /**
     * Get expected validator schema.
     *
     * @return array
     */
    public function getValidatorSchema()
    {
        return [];
    }

    /**
     * Get a valid $validatorConfiguration.
     *
     * @return mixed
     */
    public function getValidValidatorConfiguration()
    {
        return [];
    }

    /**
     * Get $validatorConfiguration not accepted by the field type.
     *
     * @return mixed
     */
    public function getInvalidValidatorConfiguration()
    {
        return [
            'unknown' => ['value' => 23],
        ];
    }

    /**
     * Get initial field data for valid object creation.
     *
     * @return mixed
     */
    public function getValidCreationFieldData()
    {
        return new MapLocationValue(
            [
                'latitude' => 51.559997,
                'longitude' => 6.767921,
                'address' => 'Bielefeld',
            ]
        );
    }

    /**
     * Get name generated by the given field type (via fieldType->getName()).
     *
     * @return string
     */
    public function getFieldName(): string
    {
        return 'Bielefeld';
    }

    /**
     * Asserts that the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was stored and loaded correctly.
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Field $field
     */
    public function assertFieldDataLoadedCorrect(Field $field)
    {
        self::assertEquals(
            $this->getValidCreationFieldData(),
            $field->value
        );
    }

    public function provideInvalidCreationFieldData()
    {
        return [
            [
                new \stdClass(),
                InvalidArgumentType::class,
            ],
            [
                23,
                InvalidArgumentType::class,
            ],
            [
                new MapLocationValue(
                    [
                        'latitude' => 'string',
                    ]
                ),
                InvalidArgumentType::class,
            ],
            [
                new MapLocationValue(
                    [
                        'latitude' => 23.42,
                        'longitude' => 'invalid',
                    ]
                ),
                InvalidArgumentType::class,
            ],
            [
                new MapLocationValue(
                    [
                        'latitude' => 23.42,
                        'longitude' => 42.23,
                        'address' => true,
                    ]
                ),
                InvalidArgumentType::class,
            ],
        ];
    }

    /**
     * Get update field externals data.
     *
     * @return array
     */
    public function getValidUpdateFieldData()
    {
        // https://maps.google.de/maps?qll=,&spn=0.139491,0.209942&sll=51.983611,8.574829&sspn=0.36242,0.839767&oq=Punta+Cana&t=h&hnear=Punta+Cana,+La+Altagracia,+Dominikanische+Republik&z=13
        return new MapLocationValue(
            [
                'latitude' => 18.524701,
                'longitude' => -68.363113,
                'address' => 'Punta Cana',
            ]
        );
    }

    /**
     * Get externals updated field data values.
     *
     * This is a PHPUnit data provider
     *
     * @return array
     */
    public function assertUpdatedFieldDataLoadedCorrect(Field $field)
    {
        self::assertEquals(
            $this->getValidUpdateFieldData(),
            $field->value
        );
    }

    public function provideInvalidUpdateFieldData()
    {
        return $this->provideInvalidCreationFieldData();
    }

    /**
     * Asserts the the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was copied and loaded correctly.
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Field $field
     */
    public function assertCopiedFieldDataLoadedCorrectly(Field $field)
    {
        self::assertEquals(
            $this->getValidCreationFieldData(),
            $field->value
        );
    }

    /**
     * Get data to test to hash method.
     *
     * This is a PHPUnit data provider
     *
     * The returned records must have the the original value assigned to the
     * first index and the expected hash result to the second. For example:
     *
     * <code>
     * array(
     *      array(
     *          new MyValue( true ),
     *          array( 'myValue' => true ),
     *      ),
     *      // ...
     * );
     * </code>
     *
     * @return array
     */
    public function provideToHashData()
    {
        return [
            [
                new MapLocationValue(
                    [
                        'latitude' => 51.559997,
                        'longitude' => 6.767921,
                        'address' => 'Bielefeld',
                    ]
                ),
                [
                    'latitude' => 51.559997,
                    'longitude' => 6.767921,
                    'address' => 'Bielefeld',
                ],
            ],
        ];
    }

    /**
     * Get expectations for the fromHash call on our field value.
     *
     * This is a PHPUnit data provider
     *
     * @return array
     */
    public function provideFromHashData()
    {
        return [
            [
                [
                    'latitude' => 51.559997,
                    'longitude' => 6.767921,
                    'address' => 'Bielefeld',
                ],
                new MapLocationValue(
                    [
                        'latitude' => 51.559997,
                        'longitude' => 6.767921,
                        'address' => 'Bielefeld',
                    ]
                ),
            ],
        ];
    }

    public function providerForTestIsEmptyValue()
    {
        return [
            [new MapLocationValue()],
            [
                new MapLocationValue(
                    [
                        'latitude' => null,
                        'longitude' => null,
                    ]
                ),
            ],
        ];
    }

    public function providerForTestIsNotEmptyValue()
    {
        return [
            [
                $this->getValidCreationFieldData(),
            ],
            [
                new MapLocationValue(
                    [
                        'latitude' => 0,
                        'longitude' => 0,
                    ]
                ),
            ],
        ];
    }
}
