<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */
declare(strict_types=1);

namespace Ibexa\Tests\Core\QueryType\BuiltIn\SortSpec\SortClauseParser;

use Ibexa\Contracts\Core\Repository\Values\Content\Query;
use Ibexa\Contracts\Core\Repository\Values\Content\Query\SortClause\Location;
use Ibexa\Core\QueryType\BuiltIn\SortSpec\Exception\UnsupportedSortClauseException;
use Ibexa\Core\QueryType\BuiltIn\SortSpec\SortClauseParser\DefaultSortClauseParser;
use Ibexa\Core\QueryType\BuiltIn\SortSpec\SortClauseParserInterface;
use Ibexa\Core\QueryType\BuiltIn\SortSpec\SortSpecParserInterface;
use PHPUnit\Framework\TestCase;

final class DefaultSortClauseParserTest extends TestCase
{
    /** @var \Ibexa\Core\QueryType\BuiltIn\SortSpec\SortClauseParser\DefaultSortClauseParser */
    private $defaultSortClauseParser;

    protected function setUp(): void
    {
        $this->defaultSortClauseParser = new DefaultSortClauseParser([
            'depth' => Location\Depth::class,
            'priority' => Location\Priority::class,
            'id' => Location\Id::class,
        ]);
    }

    public function testParse(): void
    {
        $parser = $this->createMock(SortSpecParserInterface::class);
        $parser->method('parseSortDirection')->willReturn(Query::SORT_ASC);

        self::assertEquals(
            new Location\Depth(Query::SORT_ASC),
            $this->defaultSortClauseParser->parse($parser, 'depth')
        );

        self::assertEquals(
            new Location\Priority(Query::SORT_ASC),
            $this->defaultSortClauseParser->parse($parser, 'priority')
        );
    }

    public function testParseThrowsUnsupportedSortClauseException(): void
    {
        $this->expectException(UnsupportedSortClauseException::class);
        $this->expectExceptionMessage(sprintf(
            'Could not find %s for unsupported sort clause',
            SortClauseParserInterface::class
        ));

        $this->defaultSortClauseParser->parse(
            $this->createMock(SortSpecParserInterface::class),
            'unsupported'
        );
    }

    public function testSupports(): void
    {
        self::assertTrue($this->defaultSortClauseParser->supports('depth'));
        self::assertTrue($this->defaultSortClauseParser->supports('priority'));
        self::assertTrue($this->defaultSortClauseParser->supports('id'));

        self::assertFalse($this->defaultSortClauseParser->supports('unsupported'));
    }
}
