<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */

namespace Ibexa\Tests\Integration\Core\Repository\FieldType;

use Ibexa\Contracts\Core\Repository\Exceptions\ContentFieldValidationException;
use Ibexa\Contracts\Core\Repository\Values\Content\Field;
use Ibexa\Core\Base\Exceptions\InvalidArgumentType;
use Ibexa\Core\FieldType\Selection\Value as SelectionValue;

/**
 * Integration test for use field type.
 *
 * @group integration
 * @group field-type
 */
class SelectionIntegrationTest extends SearchMultivaluedBaseIntegrationTestCase
{
    /**
     * Get name of tested field type.
     *
     * @return string
     */
    public function getTypeName()
    {
        return 'ibexa_selection';
    }

    /**
     * {@inheritdoc}
     *
     * If Selection is improved to be able to index + search for string also with LegacySearch, then adapt this too.
     */
    protected function supportsLikeWildcard($value)
    {
        parent::supportsLikeWildcard($value);

        return false;
    }

    /**
     * Get expected settings schema.
     *
     * @return array
     */
    public function getSettingsSchema()
    {
        return [
            'isMultiple' => [
                'type' => 'bool',
                'default' => false,
            ],
            'options' => [
                'type' => 'hash',
                'default' => [],
            ],
            'multilingualOptions' => [
                'type' => 'hash',
                'default' => [],
            ],
        ];
    }

    /**
     * Get a valid $fieldSettings value.
     *
     * @return mixed
     */
    public function getValidFieldSettings()
    {
        return [
            'isMultiple' => true,
            'options' => [
                0 => 'A first',
                1 => 'Bielefeld',
                2 => 'Sindelfingen',
                3 => 'Turtles',
                4 => 'Zombies',
            ],
            'multilingualOptions' => [
                'eng-GB' => [
                    0 => 'A first',
                    1 => 'Bielefeld',
                    2 => 'Sindelfingen',
                    3 => 'Turtles',
                    4 => 'Zombies',
                ],
            ],
        ];
    }

    /**
     * Get $fieldSettings value not accepted by the field type.
     *
     * @return mixed
     */
    public function getInvalidFieldSettings()
    {
        return [
            'somethingUnknown' => 0,
            'isMultiple' => [],
            'options' => new \stdClass(),
        ];
    }

    /**
     * Get expected validator schema.
     *
     * @return array
     */
    public function getValidatorSchema()
    {
        return [];
    }

    /**
     * Get a valid $validatorConfiguration.
     *
     * @return mixed
     */
    public function getValidValidatorConfiguration()
    {
        return [];
    }

    /**
     * Get $validatorConfiguration not accepted by the field type.
     *
     * @return mixed
     */
    public function getInvalidValidatorConfiguration()
    {
        return [
            'unknown' => ['value' => 23],
        ];
    }

    /**
     * Get initial field data for valid object creation.
     *
     * @return mixed
     */
    public function getValidCreationFieldData()
    {
        return new SelectionValue([0, 2]);
    }

    /**
     * Get name generated by the given field type (via fieldType->getName()).
     *
     * @return string
     */
    public function getFieldName()
    {
        return 'A first' . ' ' . 'Sindelfingen';
    }

    /**
     * Asserts that the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was stored and loaded correctly.
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Field $field
     */
    public function assertFieldDataLoadedCorrect(Field $field)
    {
        self::assertInstanceOf(
            SelectionValue::class,
            $field->value
        );

        $expectedData = [
            'selection' => [0, 2],
        ];
        $this->assertPropertiesCorrect(
            $expectedData,
            $field->value
        );
    }

    public function provideInvalidCreationFieldData()
    {
        return [
            [
                new \stdClass(),
                InvalidArgumentType::class,
            ],
            [
                new SelectionValue([7]),
                ContentFieldValidationException::class,
            ],
        ];
    }

    /**
     * Get update field externals data.
     *
     * @return array
     */
    public function getValidUpdateFieldData()
    {
        return new SelectionValue([1]);
    }

    /**
     * Get externals updated field data values.
     *
     * This is a PHPUnit data provider
     *
     * @return array
     */
    public function assertUpdatedFieldDataLoadedCorrect(Field $field)
    {
        self::assertInstanceOf(
            SelectionValue::class,
            $field->value
        );

        $expectedData = [
            'selection' => [1],
        ];
        $this->assertPropertiesCorrect(
            $expectedData,
            $field->value
        );
    }

    public function provideInvalidUpdateFieldData()
    {
        return $this->provideInvalidCreationFieldData();
    }

    /**
     * Asserts the the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was copied and loaded correctly.
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Field $field
     */
    public function assertCopiedFieldDataLoadedCorrectly(Field $field)
    {
        self::assertInstanceOf(
            SelectionValue::class,
            $field->value
        );

        $expectedData = [
            'selection' => [0, 2],
        ];
        $this->assertPropertiesCorrect(
            $expectedData,
            $field->value
        );
    }

    /**
     * Get data to test to hash method.
     *
     * This is a PHPUnit data provider
     *
     * The returned records must have the the original value assigned to the
     * first index and the expected hash result to the second. For example:
     *
     * <code>
     * array(
     *      array(
     *          new MyValue( true ),
     *          array( 'myValue' => true ),
     *      ),
     *      // ...
     * );
     * </code>
     *
     * @return array
     */
    public function provideToHashData()
    {
        return [
            [
                new SelectionValue([0, 2]),
                [0, 2],
            ],
        ];
    }

    /**
     * Get expectations for the fromHash call on our field value.
     *
     * This is a PHPUnit data provider
     *
     * @return array
     */
    public function provideFromHashData()
    {
        return [
            [
                [0, 2],
                new SelectionValue([0, 2]),
            ],
        ];
    }

    public function providerForTestIsEmptyValue()
    {
        return [
            [new SelectionValue()],
            [new SelectionValue([])],
        ];
    }

    public function providerForTestIsNotEmptyValue()
    {
        return [
            [
                $this->getValidCreationFieldData(),
            ],
            [
                new SelectionValue([0]),
            ],
        ];
    }

    protected function getValidSearchValueOne()
    {
        return [1];
    }

    protected function getValidSearchValueTwo()
    {
        return [2];
    }

    protected function getSearchTargetValueOne()
    {
        return 1;
    }

    protected function getSearchTargetValueTwo()
    {
        return 2;
    }

    protected function getAdditionallyIndexedFieldData()
    {
        return [
            [
                'selected_option_value',
                'Bielefeld',
                'Sindelfingen',
            ],
            [
                'sort_value',
                '1',
                '2',
            ],
        ];
    }

    protected function getValidMultivaluedSearchValuesOne()
    {
        return [0, 1];
    }

    protected function getValidMultivaluedSearchValuesTwo()
    {
        return [2, 3, 4];
    }

    protected function getAdditionallyIndexedMultivaluedFieldData()
    {
        return [
            [
                'selected_option_value',
                ['A first', 'Bielefeld'],
                ['Sindelfingen', 'Turtles', 'Zombies'],
            ],
        ];
    }

    protected function getFullTextIndexedFieldData()
    {
        return [
            ['Bielefeld', 'Sindelfingen'],
        ];
    }
}
