<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */

namespace Ibexa\Contracts\Core\Persistence\Content\UrlAlias;

/**
 * The UrlAlias Handler interface provides nice urls management.
 *
 * Its methods operate on a representation of the url alias data structure held
 * inside a storage engine.
 */
interface Handler
{
    /**
     * This method creates or updates an urlalias from a new or changed content name in a language
     * (if published). It also can be used to create an alias for a new location of content.
     * On update the old alias is linked to the new one (i.e. a history alias is generated).
     *
     * $alwaysAvailable controls whether the url alias is accessible in all
     * languages.
     *
     * @param mixed $locationId
     * @param mixed $parentLocationId
     * @param string $name the new name computed by the name schema or url alias schema
     * @param string $languageCode
     * @param bool $alwaysAvailable
     */
    public function publishUrlAliasForLocation(
        $locationId,
        $parentLocationId,
        $name,
        $languageCode,
        $alwaysAvailable = false
    ): string;

    /**
     * Create a user chosen $alias pointing to $locationId in $languageCode.
     *
     * If $languageCode is null the $alias is created in the system's default
     * language. $alwaysAvailable makes the alias available in all languages.
     *
     * @param mixed $locationId
     * @param string $path
     * @param bool $forwarding
     * @param string|null $languageCode
     * @param bool $alwaysAvailable
     *
     * @return \Ibexa\Contracts\Core\Persistence\Content\UrlAlias
     */
    public function createCustomUrlAlias($locationId, $path, $forwarding = false, $languageCode = null, $alwaysAvailable = false);

    /**
     * Create a user chosen $alias pointing to a resource in $languageCode.
     * This method does not handle location resources - if a user enters a location target
     * the createCustomUrlAlias method has to be used.
     *
     * If $languageCode is null the $alias is created in the system's default
     * language. $alwaysAvailable makes the alias available in all languages.
     *
     * @param string $resource
     * @param string $path
     * @param bool $forwarding
     * @param string|null $languageCode
     * @param bool $alwaysAvailable
     *
     * @return \Ibexa\Contracts\Core\Persistence\Content\UrlAlias
     */
    public function createGlobalUrlAlias($resource, $path, $forwarding = false, $languageCode = null, $alwaysAvailable = false);

    /**
     * List global aliases.
     *
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\BadStateException if path for any of the global URL aliases is broken
     *
     * @param string|null $languageCode
     * @param int $offset
     * @param int $limit
     *
     * @return \Ibexa\Contracts\Core\Persistence\Content\UrlAlias[]
     */
    public function listGlobalURLAliases($languageCode = null, $offset = 0, $limit = -1);

    /**
     * List of url entries of $urlType, pointing to $locationId.
     *
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\BadStateException if any path for the Location is broken
     *
     * @param mixed $locationId
     * @param bool $custom if true the user generated aliases are listed otherwise the autogenerated
     *
     * @return \Ibexa\Contracts\Core\Persistence\Content\UrlAlias[]
     */
    public function listURLAliasesForLocation($locationId, $custom = false);

    /**
     * Removes url aliases.
     *
     * Autogenerated aliases are not removed by this method.
     *
     * @param \Ibexa\Contracts\Core\Persistence\Content\UrlAlias[] $urlAliases
     *
     * @return bool
     */
    public function removeURLAliases(array $urlAliases);

    /**
     * Looks up a url alias for the given url.
     *
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\NotFoundException
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\InvalidArgumentException
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\BadStateException if the stored path for the given URL is broken
     *
     * @param string $url
     *
     * @return \Ibexa\Contracts\Core\Persistence\Content\UrlAlias
     */
    public function lookup($url);

    /**
     * Loads URL alias by given $id.
     *
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\BadStateException if path for the given URL alias is broken
     *
     * @param string $id unique identifier in the form of "<parentId>-<text_md5>"
     *
     * @return \Ibexa\Contracts\Core\Persistence\Content\UrlAlias
     */
    public function loadUrlAlias($id);

    /**
     * Notifies the underlying engine that a location has moved.
     *
     * This method triggers the change of the autogenerated aliases
     *
     * @param mixed $locationId
     * @param mixed $oldParentId
     * @param mixed $newParentId
     */
    public function locationMoved($locationId, $oldParentId, $newParentId);

    /**
     * Notifies the underlying engine that a location was copied.
     *
     * This method triggers the creation of the autogenerated aliases for the copied locations
     *
     * @param mixed $locationId
     * @param mixed $newLocationId
     * @param mixed $newParentId
     */
    public function locationCopied($locationId, $newLocationId, $newParentId);

    /**
     * Notifies the underlying engine that a location was swapped.
     *
     * This method triggers the change of the autogenerated aliases.
     *
     * @param string|int $location1Id
     * @param string|int $location1ParentId
     * @param string|int $location2Id
     * @param string|int $location2ParentId
     */
    public function locationSwapped($location1Id, $location1ParentId, $location2Id, $location2ParentId);

    /**
     * Notifies the underlying engine that a location was deleted or moved to trash.
     *
     * @param mixed $locationId
     *
     * @return array
     */
    public function locationDeleted($locationId): array;

    /**
     * Notifies the underlying engine that Locations Content Translation was removed.
     *
     * @param int[] $locationIds all Locations of the Content that got Translation removed
     * @param string $languageCode language code of the removed Translation
     */
    public function translationRemoved(array $locationIds, $languageCode);

    /**
     * Archive UrlAliases for Translations that were removed from the underlying published content.
     *
     * @param int $locationId Location of underlying published Content Object
     * @param int $parentLocationId
     * @param array $languageCodes Language codes of currently published Content Object Translations
     */
    public function archiveUrlAliasesForDeletedTranslations($locationId, $parentLocationId, array $languageCodes);

    /**
     * Delete corrupted URL aliases (global, custom and system).
     *
     * @return int Number of deleted URL aliases
     */
    public function deleteCorruptedUrlAliases();

    /**
     * Attempt repairing auto-generated URL aliases for the given Location (including history).
     *
     * Note: it is assumed that at this point original, working, URL Alias for Location is published.
     *
     * @param int $locationId
     *
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\BadStateException
     */
    public function repairBrokenUrlAliasesForLocation(int $locationId);
}
