<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */

namespace Ibexa\Bundle\Core\Command;

use Exception;
use Ibexa\Contracts\Core\Repository\Repository;
use Ibexa\Contracts\Core\Repository\Values\Content\Language;
use Ibexa\Contracts\Core\Repository\Values\Content\Location;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Question\ConfirmationQuestion;

/**
 * The ibexa:urls:regenerate-aliases Symfony command implementation.
 * Recreates system URL aliases for all existing Locations and cleanups corrupted URL alias nodes.
 */
#[AsCommand(
    name: 'ibexa:urls:regenerate-aliases',
    description: 'Regenerates Location URL aliases (autogenerated) and cleans up custom Location and global URL aliases stored in the Legacy Storage Engine'
)]
class RegenerateUrlAliasesCommand extends Command
{
    public const DEFAULT_ITERATION_COUNT = 1000;
    public const BEFORE_RUNNING_HINTS = <<<EOT
<error>Before you continue:</error>
- Make sure to back up your database.
- If you are regenerating URL aliases for all Locations, take the installation offline. The database should not be modified while the script is being executed.
- Run this command without memory limit, because processing large numbers of Locations (e.g. 300k) can take up to 1 GB of RAM.
- Run this command in production environment using <info>--env=prod</info>
- Manually clear HTTP cache after running this command.
EOT;

    /** @var \Ibexa\Contracts\Core\Repository\Repository */
    private $repository;

    /** @var \Psr\Log\LoggerInterface */
    private $logger;

    /**
     * @param \Ibexa\Contracts\Core\Repository\Repository $repository
     * @param \Psr\Log\LoggerInterface $logger
     */
    public function __construct(Repository $repository, ?LoggerInterface $logger = null)
    {
        parent::__construct();

        $this->repository = $repository;
        $this->logger = null !== $logger ? $logger : new NullLogger();
    }

    /**
     * {@inheritdoc}
     */
    protected function configure(): void
    {
        $beforeRunningHints = self::BEFORE_RUNNING_HINTS;
        $this
            ->addOption(
                'iteration-count',
                'c',
                InputOption::VALUE_OPTIONAL,
                'Number of Locations fetched into memory and processed at once',
                self::DEFAULT_ITERATION_COUNT
            )->addOption(
                'location-id',
                null,
                InputOption::VALUE_OPTIONAL | InputOption::VALUE_IS_ARRAY,
                'Only Locations with provided ID\'s will have URL aliases regenerated',
                []
            )->addOption(
                'force',
                'f',
                InputOption::VALUE_NONE,
                'Prevents confirmation dialog when used with --no-interaction. Please use it carefully.'
            )->setHelp(
                <<<EOT
{$beforeRunningHints}

The command <info>%command.name%</info> regenerates URL aliases for Locations and cleans up
corrupted URL aliases (pointing to non-existent Locations).
Existing aliases are archived (will redirect to the new ones).

Note: This script can potentially run for a very long time.

Due to performance issues the command does not send any Events.

<comment>You need to clear HTTP cache manually after executing this command.</comment>

EOT
            );
    }

    /**
     * Regenerate URL aliases.
     *
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $iterationCount = (int)$input->getOption('iteration-count');
        $locationIds = $input->getOption('location-id');

        if (!empty($locationIds)) {
            $locationIds = $this->getFilteredLocationList($locationIds);
            $locationsCount = count($locationIds);
        } else {
            $locationsCount = $this->repository->sudo(
                static function (Repository $repository): int {
                    return $repository->getLocationService()->getAllLocationsCount();
                }
            );
        }

        if ($locationsCount === 0) {
            $output->writeln('<info>No location was found. Exiting.</info>');

            return self::SUCCESS;
        }

        if (!$input->getOption('no-interaction')) {
            $helper = $this->getHelper('question');
            $question = new ConfirmationQuestion(
                sprintf(
                    "<info>Found %d Locations.</info>\n%s\n<info>Do you want to proceed? [y/N] </info>",
                    $locationsCount,
                    self::BEFORE_RUNNING_HINTS
                ),
                false
            );
            if (!$helper->ask($input, $output, $question)) {
                return self::SUCCESS;
            }
        } elseif (!$input->getOption('force')) {
            return self::FAILURE;
        }

        $this->regenerateSystemUrlAliases($output, $locationsCount, $locationIds, $iterationCount);

        $output->writeln('<info>Cleaning up corrupted URL aliases...</info>');
        $corruptedAliasesCount = $this->repository->sudo(
            static function (Repository $repository): int {
                return $repository->getURLAliasService()->deleteCorruptedUrlAliases();
            }
        );
        $output->writeln("<info>Done. Deleted {$corruptedAliasesCount} entries.</info>");
        $output->writeln('<comment>Make sure to clear HTTP cache.</comment>');

        return self::SUCCESS;
    }

    /**
     * Return configured progress bar helper.
     *
     * @param int $maxSteps
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     *
     * @return \Symfony\Component\Console\Helper\ProgressBar
     */
    protected function getProgressBar($maxSteps, OutputInterface $output)
    {
        $progressBar = new ProgressBar($output, $maxSteps);
        $progressBar->setFormat(
            ' %current%/%max% [%bar%] %percent:3s%% %elapsed:6s%/%estimated:-6s% %memory:6s%'
        );

        return $progressBar;
    }

    /**
     * Process single results page of fetched Locations.
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Location[] $locations
     * @param \Symfony\Component\Console\Helper\ProgressBar $progressBar
     */
    private function processLocations(array $locations, ProgressBar $progressBar): void
    {
        $contentList = $this->repository->sudo(
            static function (Repository $repository) use ($locations) {
                $contentInfoList = array_map(
                    static function (Location $location) {
                        return $location->contentInfo;
                    },
                    $locations
                );

                // load Content list in all languages
                return $repository->getContentService()->loadContentListByContentInfo(
                    $contentInfoList,
                    Language::ALL,
                    false
                );
            }
        );
        $contentList = iterator_to_array($contentList);
        foreach ($locations as $location) {
            try {
                // ignore missing Content items
                if (!isset($contentList[$location->contentId])) {
                    continue;
                }

                $this->repository->sudo(
                    static function (Repository $repository) use ($location) {
                        $repository->getURLAliasService()->refreshSystemUrlAliasesForLocation(
                            $location
                        );
                    }
                );
            } catch (Exception $e) {
                $contentInfo = $location->getContentInfo();
                $msg = sprintf(
                    'Failed processing location %d - [%d] %s (%s: %s)',
                    $location->id,
                    $contentInfo->id,
                    $contentInfo->name,
                    get_class($e),
                    $e->getMessage()
                );
                $this->logger->warning($msg);
                // in debug mode log full exception with a trace
                $this->logger->debug($e);
            } finally {
                $progressBar->advance(1);
            }
        }
    }

    /**
     * @param int $offset
     * @param int $iterationCount
     *
     * @return \Ibexa\Contracts\Core\Repository\Values\Content\Location[]
     *
     * @throws \Exception
     */
    private function loadAllLocations(int $offset, int $iterationCount): array
    {
        return $this->repository->sudo(
            static function (Repository $repository) use ($offset, $iterationCount) {
                return $repository->getLocationService()->loadAllLocations($offset, $iterationCount);
            }
        );
    }

    /**
     * @param int[] $locationIds
     * @param int $offset
     * @param int $iterationCount
     *
     * @return \Ibexa\Contracts\Core\Repository\Values\Content\Location[]
     *
     * @throws \Exception
     */
    private function loadSpecificLocations(array $locationIds, int $offset, int $iterationCount): array
    {
        $locationIds = array_slice($locationIds, $offset, $iterationCount);

        return $this->repository->sudo(
            static function (Repository $repository) use ($locationIds) {
                return $repository->getLocationService()->loadLocationList($locationIds);
            }
        );
    }

    /**
     * @param int[] $locationIds
     *
     * @return int[]
     *
     * @throws \Exception
     */
    private function getFilteredLocationList(array $locationIds): array
    {
        $locations = $this->repository->sudo(
            static function (Repository $repository) use ($locationIds) {
                $locationService = $repository->getLocationService();

                return $locationService->loadLocationList($locationIds);
            }
        );

        return array_map(
            static function (Location $location) {
                return $location->id;
            },
            $locations
        );
    }

    /**
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     * @param int $locationsCount
     * @param int[] $locationIds
     * @param int $iterationCount
     */
    private function regenerateSystemUrlAliases(
        OutputInterface $output,
        int $locationsCount,
        array $locationIds,
        int $iterationCount
    ): void {
        $output->writeln('Regenerating System URL aliases...');

        $progressBar = $this->getProgressBar($locationsCount, $output);
        $progressBar->start();

        for ($offset = 0; $offset <= $locationsCount; $offset += $iterationCount) {
            gc_disable();
            if (!empty($locationIds)) {
                $locations = $this->loadSpecificLocations($locationIds, $offset, $iterationCount);
            } else {
                $locations = $this->loadAllLocations($offset, $iterationCount);
            }
            $this->processLocations($locations, $progressBar);
            gc_enable();
        }
        $progressBar->finish();
        $output->writeln('');
        $output->writeln('<info>Done.</info>');
    }
}
