<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */
namespace Ibexa\Tests\Core\FieldType;

use Ibexa\Core\Base\Exceptions\InvalidArgumentException;
use Ibexa\Core\FieldType\Keyword\Type as KeywordType;
use Ibexa\Core\FieldType\Keyword\Value as KeywordValue;
use Ibexa\Core\FieldType\ValidationError;

/**
 * @group fieldType
 * @group ezinteger
 */
class KeywordTest extends FieldTypeTest
{
    /**
     * Returns the field type under test.
     *
     * This method is used by all test cases to retrieve the field type under
     * test. Just create the FieldType instance using mocks from the provided
     * get*Mock() methods and/or custom get*Mock() implementations. You MUST
     * NOT take care for test case wide caching of the field type, just return
     * a new instance from this method!
     *
     * @return \Ibexa\Core\FieldType\FieldType
     */
    protected function createFieldTypeUnderTest()
    {
        $fieldType = new KeywordType();
        $fieldType->setTransformationProcessor($this->getTransformationProcessorMock());

        return $fieldType;
    }

    /**
     * Returns the validator configuration schema expected from the field type.
     *
     * @return array
     */
    protected function getValidatorConfigurationSchemaExpectation()
    {
        return [];
    }

    /**
     * Returns the settings schema expected from the field type.
     *
     * @return array
     */
    protected function getSettingsSchemaExpectation()
    {
        return [];
    }

    /**
     * Returns the empty value expected from the field type.
     */
    protected function getEmptyValueExpectation()
    {
        return new KeywordValue([]);
    }

    public function provideInvalidInputForAcceptValue()
    {
        return [
            [
                23,
                InvalidArgumentException::class,
            ],
        ];
    }

    /**
     * Data provider for valid input to acceptValue().
     *
     * Returns an array of data provider sets with 2 arguments: 1. The valid
     * input to acceptValue(), 2. The expected return value from acceptValue().
     * For example:
     *
     * <code>
     *  return array(
     *      array(
     *          null,
     *          null
     *      ),
     *      array(
     *          __FILE__,
     *          new BinaryFileValue( array(
     *              'path' => __FILE__,
     *              'fileName' => basename( __FILE__ ),
     *              'fileSize' => filesize( __FILE__ ),
     *              'downloadCount' => 0,
     *              'mimeType' => 'text/plain',
     *          ) )
     *      ),
     *      // ...
     *  );
     * </code>
     *
     * @return array
     */
    public function provideValidInputForAcceptValue()
    {
        return [
            [
                null,
                new KeywordValue([]),
            ],
            [
                [],
                new KeywordValue([]),
            ],
            [
                'foo',
                new KeywordValue(['foo']),
            ],
            [
                ['foo'],
                new KeywordValue(['foo']),
            ],
            [
                new KeywordValue(['foo']),
                new KeywordValue(['foo']),
            ],
        ];
    }

    /**
     * Provide input for the toHash() method.
     *
     * Returns an array of data provider sets with 2 arguments: 1. The valid
     * input to toHash(), 2. The expected return value from toHash().
     * For example:
     *
     * <code>
     *  return array(
     *      array(
     *          null,
     *          null
     *      ),
     *      array(
     *          new BinaryFileValue( array(
     *              'path' => 'some/file/here',
     *              'fileName' => 'sindelfingen.jpg',
     *              'fileSize' => 2342,
     *              'downloadCount' => 0,
     *              'mimeType' => 'image/jpeg',
     *          ) ),
     *          array(
     *              'path' => 'some/file/here',
     *              'fileName' => 'sindelfingen.jpg',
     *              'fileSize' => 2342,
     *              'downloadCount' => 0,
     *              'mimeType' => 'image/jpeg',
     *          )
     *      ),
     *      // ...
     *  );
     * </code>
     *
     * @return array
     */
    public function provideInputForToHash()
    {
        return [
            [
                new KeywordValue([]),
                [],
            ],
            [
                new KeywordValue(['foo', 'bar']),
                ['foo', 'bar'],
            ],
        ];
    }

    /**
     * Provide input to fromHash() method.
     *
     * Returns an array of data provider sets with 2 arguments: 1. The valid
     * input to fromHash(), 2. The expected return value from fromHash().
     * For example:
     *
     * <code>
     *  return array(
     *      array(
     *          null,
     *          null
     *      ),
     *      array(
     *          array(
     *              'path' => 'some/file/here',
     *              'fileName' => 'sindelfingen.jpg',
     *              'fileSize' => 2342,
     *              'downloadCount' => 0,
     *              'mimeType' => 'image/jpeg',
     *          ),
     *          new BinaryFileValue( array(
     *              'path' => 'some/file/here',
     *              'fileName' => 'sindelfingen.jpg',
     *              'fileSize' => 2342,
     *              'downloadCount' => 0,
     *              'mimeType' => 'image/jpeg',
     *          ) )
     *      ),
     *      // ...
     *  );
     * </code>
     *
     * @return array
     */
    public function provideInputForFromHash()
    {
        return [
            [
                [],
                new KeywordValue([]),
            ],
            [
                ['foo', 'bar'],
                new KeywordValue(['foo', 'bar']),
            ],
        ];
    }

    protected function provideFieldTypeIdentifier()
    {
        return 'ezkeyword';
    }

    public function provideDataForGetName(): array
    {
        return [
            [$this->getEmptyValueExpectation(), '', [], 'en_GB'],
            [new KeywordValue(['foo', 'bar']), 'foo, bar', [], 'en_GB'],
        ];
    }

    /**
     * @return iterable<string, array{0: array<string, mixed>, 1: \Ibexa\Core\FieldType\Keyword\Value}>
     */
    public function provideValidDataForValidate(): iterable
    {
        yield 'multiple keywords' => [
            [],
            new KeywordValue(['foo', 'bar']),
        ];

        yield 'empty string keyword' => [
            [],
            new KeywordValue(['']),
        ];

        yield 'empty keyword list' => [
            [],
            new KeywordValue([]),
        ];
    }

    /**
     * @return iterable<string, array{
     *     0: array<string, mixed>,
     *     1: \Ibexa\Core\FieldType\Keyword\Value,
     *     2: array<\Ibexa\Contracts\Core\FieldType\ValidationError>
     * }>
     */
    public function provideInvalidDataForValidate(): iterable
    {
        $maxLen = KeywordType::MAX_KEYWORD_LENGTH;

        yield 'non-string keyword (int)' => [
            [],
            // @phpstan-ignore-next-line
            new KeywordValue(['valid', 123]),
            [
                new ValidationError(
                    'Each keyword must be a string.',
                    null,
                    [],
                    'values'
                ),
            ],
        ];

        yield 'non-string keyword (null)' => [
            [],
            // @phpstan-ignore-next-line
            new KeywordValue(['valid', null]),
            [
                new ValidationError(
                    'Each keyword must be a string.',
                    null,
                    [],
                    'values'
                ),
            ],
        ];

        yield 'too long keyword' => [
            [],
            new KeywordValue(['valid', str_repeat('a', $maxLen + 1)]),
            [
                new ValidationError(
                    'Keyword value must be less than or equal to ' . $maxLen . ' characters.',
                    null,
                    [],
                    'values'
                ),
            ],
        ];
    }
}

class_alias(KeywordTest::class, 'eZ\Publish\Core\FieldType\Tests\KeywordTest');
