<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */
namespace Ibexa\Tests\Integration\Core\Repository\FieldType;

use Ibexa\Contracts\Core\Repository\Values\Content\Field;
use Ibexa\Core\Base\Exceptions\ContentFieldValidationException;
use Ibexa\Core\Base\Exceptions\InvalidArgumentException;
use Ibexa\Core\FieldType\EmailAddress\Value as EmailAddressValue;

/**
 * Integration test for use field type.
 *
 * @group integration
 * @group field-type
 */
class EmailAddressIntegrationTest extends SearchBaseIntegrationTest
{
    /**
     * Get name of tested field type.
     *
     * @return string
     */
    public function getTypeName()
    {
        return 'ezemail';
    }

    /**
     * Get expected settings schema.
     *
     * @return array
     */
    public function getSettingsSchema()
    {
        return [];
    }

    /**
     * Get a valid $fieldSettings value.
     *
     * @return mixed
     */
    public function getValidFieldSettings()
    {
        return [];
    }

    /**
     * Get $fieldSettings value not accepted by the field type.
     *
     * @return mixed
     */
    public function getInvalidFieldSettings()
    {
        return [
            'somethingUnknown' => 0,
        ];
    }

    /**
     * Get expected validator schema.
     *
     * @return array
     */
    public function getValidatorSchema()
    {
        return [
            'EmailAddressValidator' => [],
        ];
    }

    /**
     * Get a valid $validatorConfiguration.
     *
     * @return mixed
     */
    public function getValidValidatorConfiguration()
    {
        return [
            'EmailAddressValidator' => [],
        ];
    }

    /**
     * Get $validatorConfiguration not accepted by the field type.
     *
     * @return mixed
     */
    public function getInvalidValidatorConfiguration()
    {
        return [
            'StringLengthValidator' => [
                'minStringLength' => new \stdClass(),
            ],
        ];
    }

    /**
     * Get initial field data for valid object creation.
     *
     * @return mixed
     */
    public function getValidCreationFieldData()
    {
        return new EmailAddressValue('spam@ibexa.co');
    }

    /**
     * Get name generated by the given field type (via fieldType->getName()).
     *
     * @return string
     */
    public function getFieldName()
    {
        return 'spam@ibexa.co';
    }

    /**
     * Asserts that the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was stored and loaded correctly.
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Field $field
     */
    public function assertFieldDataLoadedCorrect(Field $field)
    {
        $this->assertInstanceOf(
            EmailAddressValue::class,
            $field->value
        );

        $expectedData = [
            'email' => 'spam@ibexa.co',
        ];
        $this->assertPropertiesCorrect(
            $expectedData,
            $field->value
        );
    }

    public function provideInvalidCreationFieldData()
    {
        return [
            [
                new \stdClass(),
                InvalidArgumentException::class,
            ],
            [
                42,
                InvalidArgumentException::class,
            ],
            [
                new EmailAddressValue(str_repeat('.', 64)),
                ContentFieldValidationException::class,
            ],
            [
                new EmailAddressValue('spam@'),
                ContentFieldValidationException::class,
            ],
            [
                new EmailAddressValue('@ibexa.co'),
                ContentFieldValidationException::class,
            ],
            [
                new EmailAddressValue('spam@ibexa'),
                ContentFieldValidationException::class,
            ],
        ];
    }

    /**
     * Get update field externals data.
     *
     * @return array
     */
    public function getValidUpdateFieldData()
    {
        return new EmailAddressValue('spam_name@ibexa-some-thing.co');
    }

    /**
     * Get externals updated field data values.
     *
     * This is a PHPUnit data provider
     *
     * @return array
     */
    public function assertUpdatedFieldDataLoadedCorrect(Field $field)
    {
        $this->assertInstanceOf(
            EmailAddressValue::class,
            $field->value
        );

        $expectedData = [
            'email' => 'spam_name@ibexa-some-thing.co',
        ];
        $this->assertPropertiesCorrect(
            $expectedData,
            $field->value
        );
    }

    public function provideInvalidUpdateFieldData()
    {
        return $this->provideInvalidCreationFieldData();
    }

    /**
     * Asserts the the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was copied and loaded correctly.
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Field $field
     */
    public function assertCopiedFieldDataLoadedCorrectly(Field $field)
    {
        $this->assertInstanceOf(
            EmailAddressValue::class,
            $field->value
        );

        $expectedData = [
            'email' => 'spam@ibexa.co',
        ];
        $this->assertPropertiesCorrect(
            $expectedData,
            $field->value
        );
    }

    /**
     * Get data to test to hash method.
     *
     * This is a PHPUnit data provider
     *
     * The returned records must have the the original value assigned to the
     * first index and the expected hash result to the second. For example:
     *
     * <code>
     * array(
     *      array(
     *          new MyValue( true ),
     *          array( 'myValue' => true ),
     *      ),
     *      // ...
     * );
     * </code>
     *
     * @return array
     */
    public function provideToHashData()
    {
        return [
            [
                new EmailAddressValue('spam@example.no'),
                'spam@example.no',
            ],
        ];
    }

    /**
     * Get expectations for the fromHash call on our field value.
     *
     * This is a PHPUnit data provider
     *
     * @return array
     */
    public function provideFromHashData()
    {
        return [
            [
                'spam@example.no',
                new EmailAddressValue('spam@example.no'),
            ],
        ];
    }

    public function providerForTestIsEmptyValue()
    {
        return [
            [new EmailAddressValue()],
            [new EmailAddressValue(null)],
            [new EmailAddressValue('')],
        ];
    }

    public function providerForTestIsNotEmptyValue()
    {
        return [
            [
                $this->getValidCreationFieldData(),
            ],
        ];
    }

    protected function getValidSearchValueOne()
    {
        return 'holmes4@ibexa.co';
    }

    protected function getSearchTargetValueOne()
    {
        // ensure case-insensitivity
        return strtoupper($this->getValidSearchValueOne());
    }

    protected function getValidSearchValueTwo()
    {
        return 'wyoming.knott@o2.ru';
    }

    protected function getSearchTargetValueTwo()
    {
        // ensure case-insensitivity
        return strtoupper($this->getValidSearchValueTwo());
    }

    protected function getFullTextIndexedFieldData()
    {
        return [
            ['holmes4@ibexa.co', 'wyoming.knott@o2.ru'],
        ];
    }
}

class_alias(EmailAddressIntegrationTest::class, 'eZ\Publish\API\Repository\Tests\FieldType\EmailAddressIntegrationTest');
