<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */
namespace Ibexa\Tests\Integration\Core\Repository\FieldType;

use Ibexa\Contracts\Core\Repository\Exceptions\InvalidArgumentException;
use Ibexa\Contracts\Core\Repository\Values\Content\Field;
use Ibexa\Core\FieldType\Author\Author;
use Ibexa\Core\FieldType\Author\AuthorCollection;
use Ibexa\Core\FieldType\Author\Type;
use Ibexa\Core\FieldType\Author\Value as AuthorValue;

/**
 * Integration test for use field type.
 *
 * @group integration
 * @group field-type
 */
class AuthorIntegrationTest extends SearchMultivaluedBaseIntegrationTest
{
    /**
     * Get name of tested field type.
     *
     * @return string
     */
    public function getTypeName()
    {
        return 'ezauthor';
    }

    /**
     * Get expected settings schema.
     *
     * @return array
     */
    public function getSettingsSchema()
    {
        return [
            'defaultAuthor' => [
                'type' => 'choice',
                'default' => Type::DEFAULT_VALUE_EMPTY,
            ],
        ];
    }

    /**
     * Get a valid $fieldSettings value.
     *
     * @return mixed
     */
    public function getValidFieldSettings()
    {
        return [
            'defaultAuthor' => Type::DEFAULT_VALUE_EMPTY,
        ];
    }

    /**
     * Get $fieldSettings value not accepted by the field type.
     *
     * @return mixed
     */
    public function getInvalidFieldSettings()
    {
        return [
            'somethingUnknown' => 0,
        ];
    }

    /**
     * Get expected validator schema.
     *
     * @return array
     */
    public function getValidatorSchema()
    {
        return [];
    }

    /**
     * Get a valid $validatorConfiguration.
     *
     * @return mixed
     */
    public function getValidValidatorConfiguration()
    {
        return [];
    }

    /**
     * Get $validatorConfiguration not accepted by the field type.
     *
     * @return mixed
     */
    public function getInvalidValidatorConfiguration()
    {
        return [
            'unknown' => ['value' => 42],
        ];
    }

    /**
     * Get initial field data for valid object creation.
     *
     * @return \Ibexa\Core\FieldType\Author\Value
     */
    public function getValidCreationFieldData()
    {
        return new AuthorValue(
            [
                new Author(
                    [
                        'id' => 23,
                        'name' => 'Hans Mueller',
                        'email' => 'hans@example.com',
                    ]
                ),
            ]
        );
    }

    /**
     * Get name generated by the given field type (via fieldType->getName()).
     *
     * @return string
     */
    public function getFieldName()
    {
        return 'Hans Mueller';
    }

    /**
     * Asserts that the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was stored and loaded correctly.
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Field $field
     */
    public function assertFieldDataLoadedCorrect(Field $field)
    {
        $this->assertInstanceOf(
            AuthorValue::class,
            $field->value
        );

        $expectedData = [
            'authors' => new AuthorCollection(
                [
                    new Author(
                        [
                            'id' => 23,
                            'name' => 'Hans Mueller',
                            'email' => 'hans@example.com',
                        ]
                    ),
                ]
            ),
        ];
        $this->assertPropertiesCorrect(
            $expectedData,
            $field->value
        );
    }

    /**
     * Get field data which will result in errors during creation.
     *
     * This is a PHPUnit data provider.
     *
     * The returned records must contain of an error producing data value and
     * the expected exception class (from the API or SPI, not implementation
     * specific!) as the second element. For example:
     *
     * <code>
     * [
     *      [
     *          new DoomedValue( true ),
     *          ContentValidationException::class
     *      ],
     *      // ...
     * ];
     * </code>
     *
     * @return array[]
     */
    public function provideInvalidCreationFieldData()
    {
        return [
            ['Sindelfingen', InvalidArgumentException::class],
        ];
    }

    /**
     * Get update field externals data.
     *
     * @return \Ibexa\Core\FieldType\Author\Value
     */
    public function getValidUpdateFieldData()
    {
        return new AuthorValue(
            [
                new Author(
                    [
                        'id' => 42,
                        'name' => 'Lieschen Mueller',
                        'email' => 'lieschen@example.com',
                    ]
                ),
            ]
        );
    }

    /**
     * Get externals updated field data values.
     *
     * This is a PHPUnit data provider
     *
     * @return array
     */
    public function assertUpdatedFieldDataLoadedCorrect(Field $field)
    {
        $this->assertInstanceOf(
            AuthorValue::class,
            $field->value
        );

        $expectedData = [
            'authors' => new AuthorCollection(
                [
                    new Author(
                        [
                            'id' => 42,
                            'name' => 'Lieschen Mueller',
                            'email' => 'lieschen@example.com',
                        ]
                    ),
                ]
            ),
        ];
        $this->assertPropertiesCorrect(
            $expectedData,
            $field->value
        );
    }

    public function provideInvalidUpdateFieldData()
    {
        return $this->provideInvalidCreationFieldData();
    }

    /**
     * Asserts the the field data was loaded correctly.
     *
     * Asserts that the data provided by {@link getValidCreationFieldData()}
     * was copied and loaded correctly.
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Field $field
     */
    public function assertCopiedFieldDataLoadedCorrectly(Field $field)
    {
        $this->assertInstanceOf(
            AuthorValue::class,
            $field->value
        );

        $expectedData = [
            'authors' => new AuthorCollection(
                [
                    new Author(
                        [
                            'id' => 23,
                            'name' => 'Hans Mueller',
                            'email' => 'hans@example.com',
                        ]
                    ),
                ]
            ),
        ];
        $this->assertPropertiesCorrect(
            $expectedData,
            $field->value
        );
    }

    /**
     * Get data to test to hash method.
     *
     * This is a PHPUnit data provider
     *
     * The returned records must have the the original value assigned to the
     * first index and the expected hash result to the second. For example:
     *
     * <code>
     * array(
     *      array(
     *          new MyValue( true ),
     *          array( 'myValue' => true ),
     *      ),
     *      // ...
     * );
     * </code>
     *
     * @return array
     */
    public function provideToHashData()
    {
        return [
            [
                new AuthorValue(
                    [
                        new Author(
                            [
                                'id' => 23,
                                'name' => 'Hans Mueller',
                                'email' => 'hans@example.com',
                            ]
                        ),
                    ]
                ),
                [
                    [
                        'id' => 23,
                        'name' => 'Hans Mueller',
                        'email' => 'hans@example.com',
                    ],
                ],
            ],
        ];
    }

    /**
     * Get expectations for the fromHash call on our field value.
     *
     * This is a PHPUnit data provider
     *
     * @return array
     */
    public function provideFromHashData()
    {
        return [
            [
                [
                    [
                        'id' => 23,
                        'name' => 'Hans Mueller',
                        'email' => 'hans@example.com',
                    ],
                ],
                new AuthorValue(
                    [
                        new Author(
                            [
                                'id' => 23,
                                'name' => 'Hans Mueller',
                                'email' => 'hans@example.com',
                            ]
                        ),
                    ]
                ),
            ],
        ];
    }

    public function providerForTestIsEmptyValue()
    {
        return [
            [new AuthorValue()],
            [new AuthorValue([])],
        ];
    }

    public function providerForTestIsNotEmptyValue()
    {
        return [
            [
                $this->getValidCreationFieldData(),
            ],
            [
                new AuthorValue(
                    [
                        new Author(
                            [
                                'id' => 23,
                                'name' => 'Hans Mueller',
                                'email' => 'hans@example.com',
                            ]
                        ),
                    ]
                ),
            ],
        ];
    }

    protected function getValidSearchValueOne()
    {
        return [
            new Author(
                [
                    'id' => 2,
                    'name' => 'Ferdinand',
                    'email' => 'ferdinand@example.com',
                ]
            ),
        ];
    }

    protected function getValidSearchValueTwo()
    {
        return [
            new Author(
                [
                    'id' => 3,
                    'name' => 'Greta',
                    'email' => 'greta@example.com',
                ]
            ),
        ];
    }

    protected function getSearchTargetValueOne()
    {
        return 'Ferdinand';
    }

    protected function getSearchTargetValueTwo()
    {
        return 'Greta';
    }

    protected function getAdditionallyIndexedFieldData()
    {
        return [
            [
                'id',
                2,
                3,
            ],
            [
                'email',
                'ferdinand@example.com',
                'greta@example.com',
            ],
            [
                'sort_value',
                'Ferdinand',
                'Greta',
            ],
        ];
    }

    protected function getValidMultivaluedSearchValuesOne()
    {
        return [
            new Author(
                [
                    'id' => 1,
                    'name' => 'Antoinette',
                    'email' => 'antoinette@example.com',
                ]
            ),
            new Author(
                [
                    'id' => 2,
                    'name' => 'Ferdinand',
                    'email' => 'ferdinand@example.com',
                ]
            ),
        ];
    }

    protected function getValidMultivaluedSearchValuesTwo()
    {
        return [
            new Author(
                [
                    'id' => 3,
                    'name' => 'Greta',
                    'email' => 'greta@example.com',
                ]
            ),
            new Author(
                [
                    'id' => 4,
                    'name' => 'Leopold',
                    'email' => 'leopold@example.com',
                ]
            ),
            new Author(
                [
                    'id' => 5,
                    'name' => 'Maximilian',
                    'email' => 'maximilian@example.com',
                ]
            ),
        ];
    }

    protected function getMultivaluedSearchTargetValuesOne()
    {
        return ['Antoinette', 'Ferdinand'];
    }

    protected function getMultivaluedSearchTargetValuesTwo()
    {
        return ['Greta', 'Leopold', 'Maximilian'];
    }

    protected function getAdditionallyIndexedMultivaluedFieldData()
    {
        return [
            [
                'id',
                [1, 2],
                [3, 4, 5],
            ],
            [
                'email',
                ['antoinette@example.com', 'ferdinand@example.com'],
                ['greta@example.com', 'leopold@example.com', 'maximilian@example.com'],
            ],
        ];
    }

    protected function getFullTextIndexedFieldData()
    {
        return [
            ['Ferdinand', 'Greta'],
        ];
    }
}

class_alias(AuthorIntegrationTest::class, 'eZ\Publish\API\Repository\Tests\FieldType\AuthorIntegrationTest');
