<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */
declare(strict_types=1);

namespace Ibexa\Core\MVC\Symfony\Templating\Twig\Extension;

use Twig\Extension\AbstractExtension;
use Twig\TwigFilter;

/**
 * Twig common extension for general use helpers.
 */
class DataAttributesExtension extends AbstractExtension
{
    use DeprecationOptionsTrait;

    /**
     * Returns a list of functions to add to the existing list.
     */
    public function getFilters(): array
    {
        return [
            new TwigFilter(
                'ez_data_attributes_serialize',
                [$this, 'serializeDataAttributes'],
                array_merge(
                    [
                        'is_safe' => ['html'],
                    ],
                    $this->getDeprecationOptions('ibexa_data_attributes_serialize'),
                ),
            ),
            new TwigFilter(
                'ibexa_data_attributes_serialize',
                [$this, 'serializeDataAttributes'],
                ['is_safe' => ['html']]
            ),
        ];
    }

    /**
     * Processes an associative list of data attributes and returns them as HTML attributes list
     * in the form of <code>data-<attribute_name>="<attribute_value>"</code>.
     *
     * @param array<string, mixed> $dataAttributes
     *
     * @return string
     */
    public function serializeDataAttributes(array $dataAttributes): string
    {
        $result = '';
        foreach ($dataAttributes as $attributeName => $attributeValue) {
            if (!\is_string($attributeValue)) {
                $attributeValue = json_encode($attributeValue);
            }

            $result .= sprintf('data-%s="%s" ', $attributeName, htmlspecialchars($attributeValue));
        }

        return rtrim($result);
    }
}

class_alias(DataAttributesExtension::class, 'eZ\Publish\Core\MVC\Symfony\Templating\Twig\Extension\DataAttributesExtension');
