<?php

/**
 * @copyright Copyright (C) Ibexa AS. All rights reserved.
 * @license For full copyright and license information view LICENSE file distributed with this source code.
 */
declare(strict_types=1);

namespace Ibexa\Contracts\Core\Repository;

use Ibexa\Contracts\Core\Repository\Values\Content\Location;
use Ibexa\Contracts\Core\Repository\Values\Content\URLAlias;

/**
 * URLAlias service.
 */
interface URLAliasService
{
    /**
     * Create a user chosen $alias pointing to $location in $languageCode.
     *
     * This method runs URL filters and transformers before storing them.
     * Hence the path returned in the URLAlias Value may differ from the given.
     * $alwaysAvailable makes the alias available in all languages.
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Location $location
     * @param string $path
     * @param string $languageCode the languageCode for which this alias is valid
     * @param bool $forwarding if true a redirect is performed
     * @param bool $alwaysAvailable
     *
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\UnauthorizedException if the user is not allowed to create url alias
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\InvalidArgumentException if the path already exists for the given context
     *
     * @return \Ibexa\Contracts\Core\Repository\Values\Content\URLAlias
     */
    public function createUrlAlias(
        Location $location,
        string $path,
        string $languageCode,
        bool $forwarding = false,
        bool $alwaysAvailable = false
    ): URLAlias;

    /**
     * Create a user chosen $alias pointing to a resource in $languageCode.
     *
     * This method does not handle location resources - if a user enters a location target
     * the createCustomUrlAlias method has to be used.
     * This method runs URL filters and and transformers before storing them.
     * Hence the path returned in the URLAlias Value may differ from the given.
     *
     * $alwaysAvailable makes the alias available in all languages.
     *
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\UnauthorizedException if the user is not allowed to create global
     *          url alias
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\InvalidArgumentException if the path already exists for the given
     *         language or if resource is not valid
     *
     * @param string $resource
     * @param string $path
     * @param string $languageCode
     * @param bool $forwarding
     * @param bool $alwaysAvailable
     *
     * @return \Ibexa\Contracts\Core\Repository\Values\Content\URLAlias
     */
    public function createGlobalUrlAlias(
        string $resource,
        string $path,
        string $languageCode,
        bool $forwarding = false,
        bool $alwaysAvailable = false
    ): URLAlias;

    /**
     * List of url aliases pointing to $location, sorted by language priority.
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Location $location
     * @param bool $custom if true the user generated aliases are listed otherwise the autogenerated
     * @param string|null $languageCode filters those which are valid for the given language
     * @param bool|null $showAllTranslations If enabled will include all alias as if they where always available.
     * @param string[]|null $prioritizedLanguages If set used as prioritized language codes, returning first match.
     *
     * @return \Ibexa\Contracts\Core\Repository\Values\Content\URLAlias[]
     */
    public function listLocationAliases(
        Location $location,
        bool $custom = true,
        ?string $languageCode = null,
        ?bool $showAllTranslations = null,
        ?array $prioritizedLanguages = null
    ): iterable;

    /**
     * List global aliases.
     *
     * @param string|null $languageCode filters those which are valid for the given language
     * @param int $offset
     * @param int $limit
     *
     * @return \Ibexa\Contracts\Core\Repository\Values\Content\URLAlias[]
     */
    public function listGlobalAliases(?string $languageCode = null, int $offset = 0, int $limit = -1): iterable;

    /**
     * Removes urls aliases.
     *
     * This method does not remove autogenerated aliases for locations.
     *
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\UnauthorizedException if the user is not allowed to remove url alias
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\InvalidArgumentException if alias list contains
     *         autogenerated alias
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\URLAlias[] $aliasList
     */
    public function removeAliases(array $aliasList): void;

    /**
     * looks up the URLAlias for the given url.
     *
     * @param string $url
     * @param string|null $languageCode
     *
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\NotFoundException if the path does not exist or is not valid for the given language
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\InvalidArgumentException if the path exceeded maximum depth level
     *
     * @return \Ibexa\Contracts\Core\Repository\Values\Content\URLAlias
     */
    public function lookup(string $url, ?string $languageCode = null): URLAlias;

    /**
     * Returns the URL alias for the given location in the given language.
     *
     * If $languageCode is null the method returns the url alias in the most prioritized language.
     *
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\NotFoundException if no url alias exist for the given language
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Location $location
     * @param string|null $languageCode filters those which are valid for the given language
     * @param bool|null $showAllTranslations If enabled will include all alias as if they where always available.
     * @param string[]|null $prioritizedLanguages If set used as prioritized language codes, returning first match.
     *
     * @return \Ibexa\Contracts\Core\Repository\Values\Content\URLAlias
     */
    public function reverseLookup(
        Location $location,
        ?string $languageCode = null,
        ?bool $showAllTranslations = null,
        ?array $prioritizedLanguages = null
    ): URLAlias;

    /**
     * Loads URL alias by given $id.
     *
     * @throws \Ibexa\Contracts\Core\Repository\Exceptions\NotFoundException
     *
     * @param string $id
     *
     * @return \Ibexa\Contracts\Core\Repository\Values\Content\URLAlias
     */
    public function load(string $id): URLAlias;

    /**
     * Refresh all system URL aliases for the given Location (and historize outdated if needed).
     *
     * @param \Ibexa\Contracts\Core\Repository\Values\Content\Location $location
     */
    public function refreshSystemUrlAliasesForLocation(Location $location): void;

    /**
     * Delete global, system or custom URL alias pointing to non-existent Locations.
     *
     * @return int Number of deleted URL aliases
     */
    public function deleteCorruptedUrlAliases(): int;
}

class_alias(URLAliasService::class, 'eZ\Publish\API\Repository\URLAliasService');
