package metrics

import (
	"context"
	"net/url"
	"os"
	"runtime"
	"sync"
	"time"

	"github.com/honeycombio/refinery/config"
	"github.com/honeycombio/refinery/logger"
	"go.opentelemetry.io/otel/attribute"
	"go.opentelemetry.io/otel/exporters/otlp/otlpmetric/otlpmetrichttp"
	"go.opentelemetry.io/otel/metric"
	sdkmetric "go.opentelemetry.io/otel/sdk/metric"
	"go.opentelemetry.io/otel/sdk/metric/metricdata"
	"go.opentelemetry.io/otel/sdk/resource"
)

var _ MetricsBackend = (*OTelMetrics)(nil)

// OTelMetrics sends metrics to Honeycomb using the OpenTelemetry protocol. One
// particular thing to note is that OTel metrics treats histograms very
// differently than Honeycomb's Legacy metrics. In particular, Legacy metrics
// pre-aggregates histograms and sends columns corresponding to the histogram
// aggregates (e.g. avg, p50, p95, etc.). OTel, on the other hand, sends the raw
// histogram values and lets Honeycomb do the aggregation on ingest. The columns
// in the resulting datasets will not be the same.
type OTelMetrics struct {
	Config  config.Config `inject:""`
	Logger  logger.Logger `inject:""`
	Version string        `inject:"version"`

	meter        metric.Meter
	shutdownFunc func(ctx context.Context) error
	testReader   sdkmetric.Reader

	lock             sync.RWMutex
	counters         map[string]metric.Int64Counter
	gauges           map[string]metric.Float64Gauge
	histograms       map[string]metric.Float64Histogram
	updowns          map[string]metric.Int64UpDownCounter
	observableGauges map[string]metric.Float64ObservableGauge
}

// Start initializes all metrics or resets all metrics to zero
func (o *OTelMetrics) Start() error {
	cfg := o.Config.GetOTelMetricsConfig()

	o.lock.Lock()
	defer o.lock.Unlock()

	o.counters = make(map[string]metric.Int64Counter)
	o.gauges = make(map[string]metric.Float64Gauge)
	o.histograms = make(map[string]metric.Float64Histogram)
	o.updowns = make(map[string]metric.Int64UpDownCounter)
	o.observableGauges = make(map[string]metric.Float64ObservableGauge)

	ctx := context.Background()

	// otel can't handle the default endpoint so we have to parse it
	host, err := url.Parse(cfg.APIHost)
	if err != nil {
		o.Logger.Error().WithString("msg", "failed to parse metrics apihost").WithString("apihost", cfg.APIHost)
		return err
	}

	compression := otlpmetrichttp.GzipCompression
	if cfg.Compression == "none" {
		compression = otlpmetrichttp.NoCompression
	}

	options := []otlpmetrichttp.Option{
		otlpmetrichttp.WithEndpoint(host.Host),
		otlpmetrichttp.WithCompression(compression),
		// this is how we tell otel to reset metrics every time they're sent -- for some kinds of metrics.
		// Updown counters should not be reset, nor should gauges.
		// Histograms should definitely be reset.
		// Counters are a bit of a tossup, but we'll reset them because Legacy metrics did.
		otlpmetrichttp.WithTemporalitySelector(func(ik sdkmetric.InstrumentKind) metricdata.Temporality {
			switch ik {
			// These are the ones we care about today. If we add more, we'll need to add them here.
			case sdkmetric.InstrumentKindCounter, sdkmetric.InstrumentKindHistogram:
				return metricdata.DeltaTemporality
			default:
				return metricdata.CumulativeTemporality
			}
		}),
	}
	// if we ever need to add user-specified headers, that would go here
	hdrs := map[string]string{}
	if cfg.APIKey != "" {
		hdrs["x-honeycomb-team"] = cfg.APIKey
	}
	if cfg.Dataset != "" {
		hdrs["x-honeycomb-dataset"] = cfg.Dataset
	}
	if len(hdrs) > 0 {
		options = append(options, otlpmetrichttp.WithHeaders(hdrs))
	}

	if host.Scheme == "http" {
		options = append(options, otlpmetrichttp.WithInsecure())
	}

	exporter, err := otlpmetrichttp.New(ctx, options...)

	if err != nil {
		return err
	}

	// Fetch the hostname once and add it as an attribute to all metrics
	var hostname string
	if hn, err := os.Hostname(); err != nil {
		hostname = "unknown: " + err.Error()
	} else {
		hostname = hn
	}

	res, err := resource.New(ctx,
		resource.WithAttributes(resource.Default().Attributes()...),
		resource.WithAttributes(attribute.KeyValue{Key: "service.name", Value: attribute.StringValue("refinery")}),
		resource.WithAttributes(attribute.KeyValue{Key: "service.version", Value: attribute.StringValue(o.Version)}),
		resource.WithAttributes(attribute.KeyValue{Key: "host.name", Value: attribute.StringValue(hostname)}),
		resource.WithAttributes(attribute.KeyValue{Key: "hostname", Value: attribute.StringValue(hostname)}),
	)

	if err != nil {
		return err
	}

	var reader sdkmetric.Reader
	if o.testReader != nil {
		reader = o.testReader
	} else {
		reader = sdkmetric.NewPeriodicReader(exporter,
			sdkmetric.WithInterval(time.Duration(cfg.ReportingInterval)),
		)
	}

	provider := sdkmetric.NewMeterProvider(
		sdkmetric.WithReader(reader),
		sdkmetric.WithResource(res),
	)
	o.meter = provider.Meter("otelmetrics")
	o.shutdownFunc = provider.Shutdown

	// These metrics are dynamic fields that should always be collected
	name := "num_goroutines"
	var fgo metric.Float64Callback = func(_ context.Context, result metric.Float64Observer) error {
		result.Observe(float64(runtime.NumGoroutine()))
		return nil
	}
	g, err := o.meter.Float64ObservableGauge(name, metric.WithFloat64Callback(fgo))
	if err != nil {
		return err
	}

	o.observableGauges[name] = g

	name = "memory_inuse"
	// This is just reporting the gauge we already track under a different name.
	var fmem metric.Float64Callback = func(_ context.Context, result metric.Float64Observer) error {
		stats := &runtime.MemStats{}
		runtime.ReadMemStats(stats)
		result.Observe(float64(stats.HeapAlloc))
		return nil
	}
	g, err = o.meter.Float64ObservableGauge(name, metric.WithFloat64Callback(fmem))
	if err != nil {
		return err
	}
	o.observableGauges[name] = g

	startTime := time.Now()
	name = "process_uptime_seconds"
	var fup metric.Float64Callback = func(_ context.Context, result metric.Float64Observer) error {
		result.Observe(float64(time.Since(startTime) / time.Second))
		return nil
	}
	g, err = o.meter.Float64ObservableGauge(name, metric.WithFloat64Callback(fup))
	if err != nil {
		return err
	}
	o.observableGauges[name] = g

	return nil
}

func (o *OTelMetrics) Stop() {
	if o.shutdownFunc != nil {
		o.shutdownFunc(context.Background())
	}
}

// Register creates a new metric with the given metadata
// and initialize it with zero value.
func (o *OTelMetrics) Register(metadata Metadata) {
	switch metadata.Type {
	case Counter:
		_, err := o.getOrInitCounter(metadata)
		if err != nil {
			o.Logger.Error().WithString("name", metadata.Name).Logf("failed to create counter. %s", err.Error())
			return
		}
	case Gauge:
		_, err := o.getOrInitGauge(metadata)
		if err != nil {
			o.Logger.Error().WithString("name", metadata.Name).Logf("failed to create gauge. %s", err.Error())
			return
		}
	case Histogram:
		_, err := o.getOrInitHistogram(metadata)
		if err != nil {
			o.Logger.Error().WithString("name", metadata.Name).Logf("failed to create histogram. %s", err.Error())
			return
		}
	case UpDown:
		_, err := o.getOrInitUpDown(metadata)
		if err != nil {
			o.Logger.Error().WithString("name", metadata.Name).Logf("failed to create updown counter. %s", err.Error())
			return
		}
	default:
		o.Logger.Error().WithString("type", metadata.Type.String()).Logf("unknown metric type")
		return
	}
}

func (o *OTelMetrics) Increment(name string) {
	ctr, err := o.getOrInitCounter(Metadata{
		Name: name,
	})

	if err != nil {
		return
	}

	ctr.Add(context.Background(), 1)
}

func (o *OTelMetrics) Gauge(name string, val float64) {
	g, err := o.getOrInitGauge(Metadata{
		Name: name,
	})
	if err != nil {
		return
	}
	g.Record(context.Background(), val)
}

func (o *OTelMetrics) Count(name string, val int64) {
	ctr, err := o.getOrInitCounter(Metadata{Name: name})
	if err != nil {
		return
	}
	ctr.Add(context.Background(), val)
}

func (o *OTelMetrics) Histogram(name string, val float64) {
	h, err := o.getOrInitHistogram(Metadata{Name: name})
	if err != nil {
		return
	}
	h.Record(context.Background(), val)
}

func (o *OTelMetrics) Up(name string) {
	ud, err := o.getOrInitUpDown(Metadata{Name: name})
	if err != nil {
		return
	}
	ud.Add(context.Background(), 1)
}

func (o *OTelMetrics) Down(name string) {
	ud, err := o.getOrInitUpDown(Metadata{Name: name})
	if err != nil {
		return
	}
	ud.Add(context.Background(), -1)
}

// getOrInitCounter returns a counter metric with the given metadata.
// It manages the locks it needs; do not call inside a lock on o.lock.
func (o *OTelMetrics) getOrInitCounter(metadata Metadata) (ctr metric.Int64Counter, err error) {
	o.lock.RLock()
	ctr, ok := o.counters[metadata.Name]
	o.lock.RUnlock()

	if ok { // yey, the counter exists; return it
		return ctr, nil
	}

	// oh, so sad; gotta make the counter
	o.lock.Lock()
	defer o.lock.Unlock()

	ctr, ok = o.counters[metadata.Name] // confirm it wasn't created since we checked earlier
	if !ok {
		ctr, err = o.meter.Int64Counter(metadata.Name,
			metric.WithUnit(string(metadata.Unit)),
			metric.WithDescription(metadata.Description),
		)
		if err != nil {
			return nil, err
		}

		// Give the counter an initial value of 0 so that OTel will send it
		ctr.Add(context.Background(), 0)
		o.counters[metadata.Name] = ctr
	}
	return ctr, nil
}

// getOrInitGauge returns a gauge metric with the given metadata.
// It manages the locks it needs; do not call inside a lock on o.lock.
func (o *OTelMetrics) getOrInitGauge(metadata Metadata) (g metric.Float64Gauge, err error) {
	o.lock.RLock()
	g, ok := o.gauges[metadata.Name]
	o.lock.RUnlock()

	if ok { // yey, the guage exists; return it
		return g, nil
	}

	// oh, so sad; gotta make the gauge
	o.lock.Lock()
	defer o.lock.Unlock()

	g, ok = o.gauges[metadata.Name] // confirm it wasn't created since we checked earlier
	if !ok {
		g, err = o.meter.Float64Gauge(metadata.Name,
			metric.WithUnit(string(metadata.Unit)),
			metric.WithDescription(metadata.Description),
		)
		if err != nil {
			return nil, err
		}

		o.gauges[metadata.Name] = g
	}
	return g, nil
}

// getOrInitHistogram initializes a new histogram metric with the given metadata
// It manages the locks it needs; do not call inside a lock on o.lock.
func (o *OTelMetrics) getOrInitHistogram(metadata Metadata) (h metric.Float64Histogram, err error) {
	o.lock.RLock()
	h, ok := o.histograms[metadata.Name]
	o.lock.RUnlock()

	if ok { // yey, the histogram exists; return it
		return h, nil
	}

	// oh, so sad; gotta make the histogram
	o.lock.Lock()
	defer o.lock.Unlock()

	h, ok = o.histograms[metadata.Name] // confirm it wasn't created since we checked earlier
	if !ok {
		unit := string(metadata.Unit)
		h, err = o.meter.Float64Histogram(metadata.Name,
			metric.WithUnit(unit),
			metric.WithDescription(metadata.Description),
		)
		if err != nil {
			return nil, err
		}
		h.Record(context.Background(), 0)
		o.histograms[metadata.Name] = h
	}
	return h, nil

}

// getOrInitUpDown initializes a new updown counter metric with the given metadata
// It manages the locks it needs; do not call inside a lock on o.lock.
func (o *OTelMetrics) getOrInitUpDown(metadata Metadata) (ud metric.Int64UpDownCounter, err error) {
	o.lock.RLock()
	ud, ok := o.updowns[metadata.Name]
	o.lock.RUnlock()

	if ok { // yey, the updown counter exists; return it
		return ud, nil
	}

	// oh, so sad; gotta make the updown counter
	o.lock.Lock()
	defer o.lock.Unlock()

	ud, ok = o.updowns[metadata.Name] // confirm it wasn't created since we checked earlier
	if !ok {
		unit := string(metadata.Unit)
		ud, err = o.meter.Int64UpDownCounter(metadata.Name,
			metric.WithUnit(unit),
			metric.WithDescription(metadata.Description),
		)
		if err != nil {
			return nil, err
		}
		ud.Add(context.Background(), 0)
		o.updowns[metadata.Name] = ud
	}
	return ud, nil
}
