package cache

import (
	"math/rand"

	"github.com/dgryski/go-wyhash"
	"github.com/honeycombio/refinery/metrics"
)

// SentReasonsCache is a cache of reasons a trace was sent.
// It acts as a mapping between the string representation of send reason
// and a uint.
// This is used to reduce the memory footprint of the trace cache.
// It is not concurrency-safe.

type KeptReasonsCache struct {
	Metrics metrics.Metrics

	data []string
	keys map[uint64]uint32

	hashSeed uint64
}

var keptReasonCacheMetrics = []metrics.Metadata{
	{Name: "collect_sent_reasons_cache_entries", Type: metrics.Histogram, Unit: metrics.Dimensionless, Description: "Number of entries in the sent reasons cache"},
}

// NewKeptReasonsCache returns a new SentReasonsCache.
func NewKeptReasonsCache(met metrics.Metrics) *KeptReasonsCache {
	for _, metric := range keptReasonCacheMetrics {
		met.Register(metric)
	}

	return &KeptReasonsCache{
		Metrics:  met,
		keys:     make(map[uint64]uint32),
		hashSeed: rand.Uint64(),
	}
}

// Set adds a new reason to the cache, returning the key.
// The key is generated by incrementing a counter.
func (c *KeptReasonsCache) Set(key string) uint {
	// generate a hash
	hash := wyhash.Hash([]byte(key), c.hashSeed)

	val, ok := c.keys[hash]
	if !ok {
		c.data = append(c.data, key)
		val = uint32(len(c.data))
		c.keys[hash] = val
		c.Metrics.Increment("collect_sent_reasons_cache_entries")
	}
	return uint(val)
}

// Get returns a reason from the cache, if it exists.
func (c *KeptReasonsCache) Get(key uint) (string, bool) {
	if key == 0 {
		return "", false
	}
	return c.data[key-1], true
}
