# 🌺 무궁화 (Mugunghwa)

[![Wiki](https://img.shields.io/badge/📖_Wiki-deepwiki-blue)](https://deepwiki.com/hmmhmmhm/node-packages)

> 한글 기반 72진법 인코딩 라이브러리

무궁화는 10진수 숫자를 한글로 인코딩/디코딩하는 TypeScript 라이브러리입니다. 한글의 특성을 활용하여 숫자를 간결하고 읽기 쉬운 한글 문자열로 변환합니다.

## ✨ 특징

- **72진법 기반**: 한글 조합을 활용한 효율적인 숫자 인코딩
- **비속어 필터링**: 부적절한 단어 조합을 자동으로 배제
- **양방향 변환**: 숫자 ↔ 한글 문자열 상호 변환 지원
- **오타 보정**: 유사한 글자의 오타를 자동으로 수정
- **TypeScript 지원**: 완전한 타입 정의 제공

## 📦 설치

```bash
npm install mugunghwa
```

또는

```bash
pnpm add mugunghwa
```

## 🚀 사용법

### 기본 사용

```typescript
import { encode, decode } from 'mugunghwa';

// 숫자를 한글로 인코딩
const encoded = encode(31252352334);
console.log(encoded); // 예: "가나-다라-마바"

// 한글을 숫자로 디코딩
const decoded = decode(encoded);
console.log(decoded); // 31252352334
```

### 커스텀 구분자 사용

```typescript
import { encode, decode } from 'mugunghwa';

// 언더스코어를 구분자로 사용
const encoded = encode(12345, '_');
console.log(encoded); // 예: "가나_다라"

const decoded = decode(encoded, '_');
console.log(decoded); // 12345
```

### 디버그 모드

```typescript
import { decode, setDebug } from 'mugunghwa';

// 디버그 모드 활성화 (에러 발생 시 상세 정보 출력)
setDebug(true);

try {
  const result = decode('잘못된코드');
  console.log(result);
} catch (error) {
  console.error(error); // 상세한 에러 메시지 출력
}

// 디버그 모드 비활성화 (기본값: false)
setDebug(false);
const result = decode('잘못된코드'); // null 반환
```

### CLI 스크립트 사용

```bash
# 숫자 인코딩
pnpm run encode 12345

# 문자열 디코딩
pnpm run decode "가나-다라"

# 랜덤 테스트
pnpm run random

# 유효성 검증
pnpm run validate
```

## 🔧 API

### `encode(index: number, separator?: string): string | null`

10진수 숫자를 한글 문자열로 인코딩합니다.

**매개변수:**
- `index`: 인코딩할 양의 정수 (0 이상)
- `separator`: 구분자 문자열 (기본값: `"-"`)

**반환값:**
- 성공 시: 인코딩된 한글 문자열
- 실패 시: `null` (음수, 소수, NaN 등)

### `decode(code: string, separator?: string): number | null`

한글 문자열을 10진수 숫자로 디코딩합니다.

**매개변수:**
- `code`: 디코딩할 한글 문자열
- `separator`: 구분자 문자열 (기본값: `"-"`)

**반환값:**
- 성공 시: 디코딩된 숫자
- 실패 시: `null` (잘못된 형식)

### 유틸리티 함수

```typescript
// 복합 다진수를 10진수로 변환
decimalBase(maxMatrix: number[], indexMatrix: number[]): number

// 10진수를 복합 다진수로 변환
multipleBase(maxMatrix: number[], indexDecimal: number): number[]

// 예상 인코딩 길이 계산
expectLength(index: number): number

// 디버그 모드 설정 (에러 발생 시 상세 정보 출력)
setDebug(value: boolean): void
```

## 🧮 무궁화72 구현 원리

### 왜 72진법인가?

무궁화는 한글의 자음과 모음 조합을 활용하여 72진법을 구현합니다. 하지만 단순히 72진법만 사용하면 무작위 글자 조합에서 비속어가 발생할 수 있습니다.

### 비속어 필터링 전략

#### 짝수 단위: 84진법 활용

짝수 위치의 글자들은 84진법으로 표기한 후, 부적절한 조합을 배제합니다.

```
72² = 5,184
84² = 7,056
7,056 - 5,184 = 1,872

→ 5,184개의 조합 중 1,872개(36.1%)의 부정 조합을 배제
```

#### 홀수 단위: 98진법 활용

홀수 위치의 글자들은 98진법으로 표기하여 더 많은 부정 조합을 배제합니다.

```
84² = 7,056
98² = 9,604
9,604 - 7,056 = 2,548

기존 1,872개 부정 조합 제외:
2,548 - 1,872 = 676

→ 2,548개의 조합 중 676개(26.5%)의 부정 조합을 배제
```

### 구현 세부사항

- **짝수 단위**: 72진법 + 12개 추가 글자 = 84진법
- **홀수 단위**: 84진법 + 14개 추가 글자 = 98진법
- **오타 보정**: 유사한 글자 자동 수정 기능 내장
- **구분자 처리**: 가독성을 위한 구분자 자동 삽입
- **데이터 압축**: lz-string을 활용한 인코딩 테이블 압축 저장

## 📊 예제

```typescript
import { encode, decode } from 'mugunghwa';

// 다양한 숫자 테스트
const numbers = [0, 1, 100, 1000, 1000000];

numbers.forEach(num => {
  const encoded = encode(num);
  const decoded = decode(encoded!);
  console.log(`${num} → ${encoded} → ${decoded}`);
});

// 결과 예시:
// 0 → 가 → 0
// 1 → 나 → 1
// 100 → 다라 → 100
// 1000 → 마바-사 → 1000
// 1000000 → 아자-차카-타 → 1000000
```

## 🧪 테스트

```bash
# 테스트 실행
pnpm test

# 빌드
pnpm build

# 개발 모드 (watch)
pnpm watch
```

## 📝 라이선스

MIT License

Copyright (c) hmmhmmhm

## 🔗 링크

- [GitHub Repository](https://github.com/hmmhmmhm/node-packages)
- [Issues](https://github.com/hmmhmmhm/node-packages/issues)

## 📜 히스토리

- TypeScript로 완전히 재작성
- 모던 빌드 시스템 적용 (microbundle)
- 포괄적인 테스트 커버리지 추가