## OAuth 2.0
Hitobito is an OAuth 2.0 provider, meaning that an external application can authenticate users via hitobito (usually in the form of a "Login via hitobito" feature, similar to Google and Facebook etc.). The external application can then query information about the user or use the [API](../common/api/README.md) in the name of the user, if the user has granted this permission. This works without giving the user's password or token to the external application.

### Setup
You need to create an RSA private key for JWT signing:

`openssl genpkey -algorithm RSA -out key -pkeyopt rsa_keygen_bits:2048`

Provide its content by environment variable `JWT_SIGNING_KEY` in rails server environment.

### Managing OAuth applications in hitobito
Users with admin privilege on the application root group (e.g. "Bundesebene" or "Dachverband") are allowed to manage the OAuth applications in the global hitobito settings.

Applications must be registered with allowed callback URLs and allowed scopes (see below). The external application is identified by a UId (unique identifier) and an application secret, both of which are generated by hitobito.

#### Scopes
Different external applications need different types of data about their users. Scopes are used to define what information an external application can request and access.

| Scope           | Available user data / functionality       |
| ---             | ---                                       |
| `email`         | id, email                                 |
| `name`          | id, email, firstname, lastname, nickname  |
| `with_roles`    | All the profile data, roles within groups |
| `oidc`          | OIDC identity token (userinfo endpoint)   |
| `api`           | All REST APIs                             |
| `events`        | event API endpoints                       |
| `groups`        | group API endpoints                       |
| `people`        | people API endpoints                      |
| `invoices`      | invoice API endpoints                     |
| `mailing_lists` | mailing list API endpoints                |
| `nextcloud`     | nextcloud groups based on user's roles    |

The scope `with_roles` contains most of the profile information, including the fields of the name scope. Multiple scopes can be combined in a single authorization request.

### OAuth flow

#### Make hitobito ask the user for permission
Before an external application can read the user's data, it must first get the approval of the user. To do this, it redirects the user to a hitobito page displaying a confirmation dialogue to the user. The url to this dialogue page has the following parameters:

* `response_type` For the authorization request we always use the value `code`
* `client_id` The UId of the external application, this will be provided to you by your hitobito root group admin
* `redirect_uri` A URL where your application can receive the authorization code once the user grants the permission request. The hitobito root group admin needs to know this URL (or all possible URLs if the external application has multiple) in advance in order to create your `client_id`.
  * The URLs must have the `https://` protocol. Exception: On integration environments only, `localhost` URLs can also be used with `http://` (for testing and development purposes).
  * If you want to only test the OAuth functionality, you can use the special value `urn:ietf:wg:oauth:2.0:oob`, which will display the authorization token instead of redirecting the browser
* `scope` One or more scopes (see above) needed by your application, separated by spaces or `%20`. The hitobito root group admin needs to know this list in advance in order to create your `client_id`

An example of the full URL, where your application could redirect the user for asking permission:
```
https://demo.hitobito.com/oauth/authorize?response_type=code&client_id=df57d88580cd81d2f7ca9974ff5a45dece279a36d4f55fe741502dd3ebb60ba8&redirect_uri=https://my-external-application.ch/callback&scope=email%20name
```

If the user accepts the request, the user's browser will be redirected to the specified redirect URL, with the authorization code in the `code` query parameter. Make sure this is a valid URL in your application. In our example this could be `https://my-external-application.ch/callback?code=2dd87982736ffbd2b68820a2dbcc1f3bd59c5653c5d166339ce68c320f8d83c5`

#### Acquire an access token
The code returned by the authorization request can be used in your application to acquire an access token.

```bash
curl -X POST \
     -H "Accept: application/json" \
     -d "grant_type=authorization_code" \
     -d "client_id=df57d88580cd81d2f7ca9974ff5a45dece279a36d4f55fe741502dd3ebb60ba8" \
     -d "client_secret=2e1a8ef6676e3b9a13a75a7e6ae55879a02b8a5194af010dd8af0eb1a2ca0957" \
     -d "redirect_uri=https://my-external-application.ch/callback" \
     -d "code=2dd87982736ffbd2b68820a2dbcc1f3bd59c5653c5d166339ce68c320f8d83c5" \
     https://demo.hitobito.com/oauth/token
```

* `grant_type` For the token request we use the value `authorization_code`
* `client_id` The UId for the application, ask your hitobito root group admin
* `client_secret` The application secret, ask your hitobito root group admin
* `redirect_uri` Must be the same value as in the authorization request
* `code` The code received from the authorization request

The response will contain the access token (formatted here for readability):
```json
{
  "access_token": "A5t3yMUvKCZLxasqJbQBhu0bRKMvWdwcBSw2DJbUiaQ",
  "token_type": "Bearer",
  "expires_in": 7200,
  "scope": "email name",
  "created_at": 1560862298
}
```

#### Use your access token to get information about the user

The freshly acquired token can be used in the `Authorization` HTTP header when accessing the OAuth profile endpoint. You also have to set the `X-Scope` HTTP header to one of the scopes you requested when asking the user for permission.
```bash
curl -H "Authorization: Bearer A5t3yMUvKCZLxasqJbQBhu0bRKMvWdwcBSw2DJbUiaQ" \
     -H "X-Scope: name" \
     https://demo.hitobito.com/oauth/profile
```

An example response could be (formatted here for readability):
```json
{
  "id": 34,
  "email": "julia@example.com",
  "first_name": "Julia",
  "last_name": "Keller",
  "nickname": "Polka",
  "address": null,
  "zip_code": "",
  "town": null,
  "country": null
}
```

#### `with_roles` scope

When using the `with_roles` scope, the /oauth/profile endpoint yields all public attributes of the logged in person, as well as a list of their roles:

```json
{
  "id": 34,
  "email": "julia@example.com",
  "first_name": "Julia",
  "last_name": "Keller",
  "nickname": "Polka",
  "company_name": null,
  "company": false,
  "address": null,
  "zip_code": "",
  "town": null,
  "country": null,
  "gender": null,
  "birthday": "1999-09-09",
  "primary_group_id": 1,
  "language": "de",
  "roles": [
    {
      "group_id": 1,
      "group_name": "hitobito",
      "role_name": "Mitarbeiter*in GS",
      "role_class": "Group::Bund::MitarbeiterGs",
      "permissions": [
        "layer_and_below_full",
        "contact_data",
        "admin"
      ]
    }
  ]
}
```

### OpenID Connect (OIDC)

When using the `oidc` scope, the flow initially works exactly the same as for OAuth 2.0. The response to the /oauth/token endpoint will contain an additional `id_token` field:
```json
{
  "access_token": "A5t3yMUvKCZLxasqJbQBhu0bRKMvWdwcBSw2DJbUiaQ",
  "token_type": "Bearer",
  "expires_in": 7200,
  "scope": "openid name",
  "created_at": 1709940937,
  "id_token": "eyJraWQiOiJha2NwUWpvYjVaSGdKUmpsZWdidlZuUjVyYjRZYWxyVDlJNnpjN3NTOERJIiwiYWxnIjoiUlMyNTYifQ.eyJpc3MiOiJodHRwOi8vbG9jYWxob3N0OjMwMDAiLCJzdWIiOiIxIiwiYXVkIjoiWEM5LUhXQndBbGdmaUFvVFh6bXVPbnlUX0FraHhNaUZ3WlZRdGpCcXdCNCIsImV4cCI6MTcwOTk0MTA1NywiaWF0IjoxNzA5OTQwOTM3LCJub25jZSI6ImRvN243ZzV4NXVpIn0.OCGY2-cNx7FeQwqGVAI3_puPt17yagwZGcAvRJjOifhQDIMBKqulcWP3OwdeZjhD62BmMwNPBtUsRF-jsoaF6iDqrFGWXPkjuPxSypF6g8r0eGk5vkL9KlAbhr5pBxrAgMhryDPh-o48YdV7iFe5pGu7Amc5ufAIWpa0eE8hhMYJIE-M37b8ZqjCYiB7UfV1gz24aEYbWk-dC78-tn3A1OLGc8hPQZqXI0KqMQ7Dub7FHi6d7IAGitzftLLkmJl-ObSuDemjX43-xFQQQ91RGR6LJ2Y-RzmkNPfJrSS1-Gf-kSluHSKGQ0aZbq4E2e-xP1Gf-rKzCmB1l62--ZnmBw"
}
```

This ID token is a JWT token which, when decoded, contains standardized OIDC fields:
```json
{
  "iss": "https://demo.hitobito.com",
  "sub": "34",
  "aud": "XC9-HWBwAlgfiAoTXzmuOnyT_AkhxMiFwZVQtjBqwB4",
  "exp": 1709941057,
  "iat": 1709940937,
  "nonce": "do7n7g5x5ui"
}
```

The token is signed using the hitobito server's private key. `iss` is the issuer, which is the URL of the identity provider. `sub` is the id of the user who just logged in. When using the `email` scope, the JWT token payload additionally contains an `email` field containing the main email address of the authenticated person.

#### Userinfo endpoint

Using the access token, the standardized OIDC Userinfo endpoint can also be queried. This endpoint is similar to the profile endpoint of the OAuth flow, and will yield information depending on the used scopes. Here is an example response from the /oauth/userinfo endpoint, when using the `email` and `with_roles` scopes (additionally to the mandatory `openid` scope):

```bash
curl -H "Authorization: Bearer A5t3yMUvKCZLxasqJbQBhu0bRKMvWdwcBSw2DJbUiaQ" \
     https://demo.hitobito.com/oauth/userinfo
```

```json
{
  "sub": "34",
  "email": "julia@example.com",
  "first_name": "Julia",
  "last_name": "Keller",
  "nickname": "Polka",
  "company_name": null,
  "company": false,
  "address": null,
  "zip_code": "",
  "town": null,
  "country": null,
  "gender": null,
  "birthday": "1999-09-09",
  "primary_group_id": 1,
  "language": "de",
  "roles": [
    {
      "group_id": 1,
      "group_name": "hitobito",
      "role": "Group::Bund::MitarbeiterGs",
      "role_class": "Group::Bund::MitarbeiterGs",
      "role_name": "Mitarbeiter*in GS",
      "permissions": [
        "layer_and_below_full"
      ]
    }
  ]
}
```

#### Well-known openid configuration

All endpoints related to OIDC, along with some machine-readable info for automated OIDC client generators, are documented at https://demo.hitobito.com/.well-known/openid-configuration

### Accessing the JSON-API

All endpoints from the [JSON API](../common/api/json_api.md) can be used with a personal OAuth access token, if the token has the `api` scope. Individual endpoints can also be accessed if the token has the matching scope: `events`, `groups`, `people`, `invoices`, `mailing_lists`. There are two possibilities to use the API:

* **Query parameter**: Send `access_token` as query parameter in the URL, and append `.json` to the URL path
```bash
curl "https://demo.hitobito.com/de/groups/1.json?access_token=A5t3yMUvKCZLxasqJbQBhu0bRKMvWdwcBSw2DJbUiaQ"
```

* **Request headers**: Set the following headers on the HTTP request: `Authorization: Bearer <access_token>` and `Accept: application/json`
```bash
curl -H "Authorization: Bearer A5t3yMUvKCZLxasqJbQBhu0bRKMvWdwcBSw2DJbUiaQ" \
     -H "Accept: application/json" \
     https://demo.hitobito.com/de/groups/1
```
