#  Copyright (c) 2012-2013, Jungwacht Blauring Schweiz. This file is part of
#  hitobito and licensed under the Affero General Public License version 3
#  or later. See the COPYING file at the top-level directory or at
#  https://github.com/hitobito/hitobito.

require "spec_helper"

describe Devise::Hitobito::PasswordsController do
  let(:bottom_group) { groups(:bottom_group_one_one) }

  before do
    request.env["devise.mapping"] = Devise.mappings[:person]
    ActionMailer::Base.deliveries = []
  end

  describe "#create" do
    it "#create with invalid email invalid password" do
      post :create, params: {person: {email: "asdf"}}
      expect(last_email).not_to be_present
      expect(controller.send(:resource).errors[:email]).to eq ["nicht gefunden"]
    end

    context "with login permission" do
      let(:person) { Fabricate("Group::BottomGroup::Leader", group: bottom_group).person.reload }

      it "#create shows invalid password" do
        post :create, params: {person: {email: person.email}}
        # rubocop:todo Layout/LineLength
        expect(flash[:notice]).to eq "Wenn uns die angegebene E-Mail-Adresse bekannt ist, erhältst du in wenigen Minuten eine E-Mail mit der Anleitung, wie Du Dein Passwort zurücksetzen kannst."
        # rubocop:enable Layout/LineLength
        expect(last_email).to be_present
      end

      it "#create sends localized email" do
        @cached_locales = I18n.available_locales
        @cached_languages = Settings.application.languages
        Settings.application.languages = {de: "Deutsch", fr: "Français"}
        I18n.available_locales = Settings.application.languages.keys

        person.update!(language: "fr")
        expect(I18n.locale).to eq(:de)
        expect(I18n).to receive(:"locale=").with(:de).ordered
        expect(I18n).to receive(:"locale=").with("fr").ordered
        expect(Devise.mailer).to receive(:reset_password_instructions).and_call_original.ordered
        expect(I18n).to receive(:"locale=").with(:de).ordered
        expect(I18n).to receive(:"locale=").with(:de).ordered
        expect(I18n).to receive(:"locale=").with(:de).ordered
        post :create, params: {person: {email: person.email}}
        # rubocop:todo Layout/LineLength
        expect(flash[:notice]).to eq "Wenn uns die angegebene E-Mail-Adresse bekannt ist, erhältst du in wenigen Minuten eine E-Mail mit der Anleitung, wie Du Dein Passwort zurücksetzen kannst."
        # rubocop:enable Layout/LineLength

        I18n.available_locales = @cached_locales
        Settings.application.languages = @cached_languages
        I18n.locale = I18n.default_locale
      end

      # person language can be a language, that does not exist as a locale, for better description
      # of the issue: https://github.com/hitobito/hitobito_sac_cas/issues/1392
      it "should use previous_locale if person language is not a registered language in application" do
        @original_locales = I18n.available_locales.dup
        I18n.available_locales = Settings.application.languages.keys
        person.update!(language: "en")
        expect(I18n.locale).to eq(:de)
        expect(I18n).not_to receive(:"locale=").with("en")
        post :create, params: {person: {email: person.email}}
      ensure
        I18n.available_locales = @original_locales
      end
    end

    context "without login permission" do
      it "#create shows flash messagge" do
        post :create, params: {person: {email: "not-existing@example.com"}}
        expect(last_email).not_to be_present
        # rubocop:todo Layout/LineLength
        expect(flash[:notice]).to eq "Wenn uns die angegebene E-Mail-Adresse bekannt ist, erhältst du in wenigen Minuten eine E-Mail mit der Anleitung, wie Du Dein Passwort zurücksetzen kannst."
        # rubocop:enable Layout/LineLength
      end
    end

    def last_email
      ActionMailer::Base.deliveries.last
    end
  end
end
