# frozen_string_literal: true

#  Copyright (c) 2023-2024, Pfadibewegung Schweiz. This file is part of
#  hitobito and licensed under the Affero General Public License version 3
#  or later. See the COPYING file at the top-level directory or at
#  https://github.com/hitobito/hitobito.

class People::CleanupJob < RecurringJob
  run_every 1.day

  private

  def perform_internal
    return unless FeatureGate.enabled?("people.cleanup_job")

    People::CleanupFinder.new.run.find_in_batches do |batch|
      batch.each do |person|
        if event_participation_in_cutoff?(person)
          minimize_person!(person)
        else
          destroy_person!(person)
        end
      end
    end
  end

  def cutoff
    Settings.people.cleanup_cutoff_duration.regarding_event_participations.months.ago
  end

  def event_participation_in_cutoff?(person)
    Event.joins(:participations, :dates)
      .where(participations: {participant_id: person.id, participant_type: person.class.sti_name})
      .where("event_dates.start_at > :cutoff OR event_dates.finish_at > :cutoff", cutoff: cutoff)
      .any?
  end

  def destroy_person!(person)
    People::Destroyer.new(person).run
  rescue => e
    error(self, e, person_id: person.id)
  end

  def minimize_person!(person)
    People::Minimizer.new(person).run
  rescue => e
    error(self, e, person_id: person.id)
  end
end
