/*
 * Copyright 2001-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.logging.impl;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogConfigurationException;
import org.apache.commons.logging.LogFactory;

/**
 * <p>
 * Concrete subclass of {@link LogFactory} that implements the following algorithm to dynamically select a logging implementation class to instantiate a wrapper for.
 * </p>
 * <ul>
 * <li>Use a factory configuration attribute named <code>org.apache.commons.logging.Log</code> to identify the requested implementation class.</li>
 * <li>Use the <code>org.apache.commons.logging.Log</code> system property to identify the requested implementation class.</li>
 * <li>If <em>Log4J</em> is available, return an instance of <code>org.apache.commons.logging.impl.Log4JLogger</code>.</li>
 * <li>If <em>JDK 1.4 or later</em> is available, return an instance of <code>org.apache.commons.logging.impl.Jdk14Logger</code>.</li>
 * <li>Otherwise, return an instance of <code>org.apache.commons.logging.impl.SimpleLog</code>.</li>
 * </ul>
 * 
 * <p>
 * If the selected {@link Log} implementation class has a <code>setLogFactory()</code> method that accepts a {@link LogFactory} parameter, this method will be called on each newly created instance
 * to identify the associated factory. This makes factory configuration attributes available to the Log instance, if it so desires.
 * </p>
 * 
 * <p>
 * This factory will remember previously created <code>Log</code> instances for the same name, and will return them on repeated requests to the <code>getInstance()</code> method. This
 * implementation ignores any configured attributes.
 * </p>
 * 
 * @author Rod Waldhoff
 * @author Craig R. McClanahan
 * @author Richard A. Sitze
 * @version $Revision: 1.33 $ $Date: 2004/03/06 21:52:59 $
 */

public class LogFactoryImpl extends LogFactory {
  public LogFactoryImpl() {
    super();
  }

  public static final String LOG_PROPERTY = "org.apache.commons.logging.Log";

  protected static final String LOG_PROPERTY_OLD = "org.apache.commons.logging.log";

  protected Hashtable attributes = new Hashtable();

  protected Hashtable instances = new Hashtable();

  private String logClassName;

  protected Constructor logConstructor = null;

  protected Class logConstructorSignature[] = { java.lang.String.class };

  protected Method logMethod = null;

  protected Class logMethodSignature[] = { LogFactory.class };

  public Object getAttribute(String name) {
    return (attributes.get(name));
  }

  public String[] getAttributeNames() {

    Vector names = new Vector();
    Enumeration keys = attributes.keys();
    while (keys.hasMoreElements()) {
      names.addElement((String) keys.nextElement());
    }
    String results[] = new String[names.size()];
    for (int i = 0; i < results.length; i++) {
      results[i] = (String) names.elementAt(i);
    }
    return (results);

  }

  public Log getInstance(Class clazz) throws LogConfigurationException {

    return (getInstance(clazz.getName()));

  }

  public Log getInstance(String name) throws LogConfigurationException {

    return new NoOpLog();

  }

  public void release() {
    instances.clear();
  }

  public void removeAttribute(String name) {
  }

  public void setAttribute(String name, Object value) {
  }

  protected final String getLogClassName() {
    logClassName = "org.apache.commons.logging.impl.NoOpLog";
    return (logClassName);
  }

  protected Constructor getLogConstructor() throws LogConfigurationException {

    Class logClass = NoOpLog.class;

    // Identify the corresponding constructor to be used
    try {
      logConstructor = logClass.getConstructor(logConstructorSignature);
      return (logConstructor);
    } catch (Throwable t) {
      throw new LogConfigurationException("No suitable Log constructor "
          + logConstructorSignature + " for " + logClassName, t);
    }
  }

  protected boolean isJdk13LumberjackAvailable() {
    return false;
  }

  protected boolean isJdk14Available() {
    return false;
  }

  protected boolean isLog4JAvailable() {
    return false;
  }

  protected Log newInstance(String name) throws LogConfigurationException {
    return new NoOpLog();
  }

}
