<?php

use Appwrite\Auth\Auth;
use Utopia\Config\Config;
use Utopia\Database\Database;
use Utopia\Database\Helpers\ID;

$providers = Config::getParam('oAuthProviders', []);
$auth = Config::getParam('auth', []);

/**
 * $collection => id of the parent collection where this will be inserted
 * $id => id of this collection
 * name => name of this collection
 * project => whether this collection should be created per project
 * attributes => list of attributes
 * indexes => list of indexes
 */

$commonCollections = [
    'cache' => [
        '$collection' => Database::METADATA,
        '$id' => 'cache',
        'name' => 'Cache',
        'attributes' => [
            [
                '$id' => 'resource',
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 255,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => 'resourceType',
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 255,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('mimeType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 255, // https://tools.ietf.org/html/rfc4288#section-4.2
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => 'accessedAt',
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => 'signature',
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 255,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => '_key_accessedAt',
                'type' => Database::INDEX_KEY,
                'attributes' => ['accessedAt'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => '_key_resource',
                'type' => Database::INDEX_KEY,
                'attributes' => ['resource'],
                'lengths' => [],
                'orders' => [],
            ],
        ],
    ],

    'users' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('users'),
        'name' => 'Users',
        'attributes' => [
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('email'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 320,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('phone'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16, // leading '+' and 15 digitts maximum by E.164 format
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('status'),
                'type' => Database::VAR_BOOLEAN,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('labels'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('passwordHistory'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('password'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ],
            [
                '$id' => 'hash', // Hashing algorithm used to hash the password
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => Auth::DEFAULT_ALGO,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('hashOptions'), // Configuration of hashing algorithm
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 65535,
                'signed' => true,
                'required' => false,
                'default' => Auth::DEFAULT_ALGO_OPTIONS,
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('passwordUpdate'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('prefs'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 65535,
                'signed' => true,
                'required' => false,
                'default' => new \stdClass(),
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('registration'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('emailVerification'),
                'type' => Database::VAR_BOOLEAN,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('phoneVerification'),
                'type' => Database::VAR_BOOLEAN,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('reset'),
                'type' => Database::VAR_BOOLEAN,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('mfa'),
                'type' => Database::VAR_BOOLEAN,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('mfaRecoveryCodes'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => true,
                'filters' => ['encrypt'],
            ],
            [
                '$id' => ID::custom('authenticators'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQueryAuthenticators'],
            ],
            [
                '$id' => ID::custom('sessions'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQuerySessions'],
            ],
            [
                '$id' => ID::custom('tokens'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQueryTokens'],
            ],
            [
                '$id' => ID::custom('challenges'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQueryChallenges'],
            ],
            [
                '$id' => ID::custom('memberships'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQueryMemberships'],
            ],
            [
                '$id' => ID::custom('targets'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQueryTargets'],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['userSearch'],
            ],
            [
                '$id' => ID::custom('accessedAt'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_name'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['name'],
                'lengths' => [256],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_email'),
                'type' => Database::INDEX_UNIQUE,
                'attributes' => ['email'],
                'lengths' => [256],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_phone'),
                'type' => Database::INDEX_UNIQUE,
                'attributes' => ['phone'],
                'lengths' => [16],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_status'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['status'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_passwordUpdate'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['passwordUpdate'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_registration'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['registration'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_emailVerification'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['emailVerification'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_phoneVerification'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['phoneVerification'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => '_key_accessedAt',
                'type' => Database::INDEX_KEY,
                'attributes' => ['accessedAt'],
                'lengths' => [],
                'orders' => [],
            ],
        ],
    ],

    'tokens' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('tokens'),
        'name' => 'Tokens',
        'attributes' => [
            [
                '$id' => ID::custom('userInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('userId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('type'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('secret'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 512, // https://www.tutorialspoint.com/how-long-is-the-sha256-hash-in-mysql (512 for encryption)
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ],
            [
                '$id' => ID::custom('expire'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('userAgent'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('ip'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 45, // https://stackoverflow.com/a/166157/2299554
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ]
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_user'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'authenticators' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('authenticators'),
        'name' => 'Authenticators',
        'attributes' => [
            [
                '$id' => ID::custom('userInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('userId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('type'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('verified'),
                'type' => Database::VAR_BOOLEAN,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => false,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('data'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 65535,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => false,
                'filters' => ['json', 'encrypt'],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_userInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ]
        ],
    ],

    'challenges' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('challenges'),
        'name' => 'Challenges',
        'attributes' => [
            [
                '$id' => ID::custom('userInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('userId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('type'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('token'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 512, // https://www.tutorialspoint.com/how-long-is-the-sha256-hash-in-mysql (512 for encryption)
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ], [
                '$id' => ID::custom('code'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 512,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ], [
                '$id' => ID::custom('expire'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ]
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_user'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ]
        ],
    ],

    'sessions' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('sessions'),
        'name' => 'Sessions',
        'attributes' => [
            [
                '$id' => ID::custom('userInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('userId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('provider'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerUid'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerAccessToken'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ],
            [
                '$id' => ID::custom('providerAccessTokenExpiry'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('providerRefreshToken'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ],
            [
                '$id' => ID::custom('secret'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 512, // https://www.tutorialspoint.com/how-long-is-the-sha256-hash-in-mysql (512 for encryption)
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ],
            [
                '$id' => ID::custom('userAgent'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('ip'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 45, // https://stackoverflow.com/a/166157/2299554
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('countryCode'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('osCode'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('osName'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('osVersion'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('clientType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('clientCode'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('clientName'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('clientVersion'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('clientEngine'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('clientEngineVersion'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('deviceName'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('deviceBrand'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('deviceModel'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('factors'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('expire'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('mfaUpdatedAt'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_provider_providerUid'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['provider', 'providerUid'],
                'lengths' => [128, 128],
                'orders' => [Database::ORDER_ASC, Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_user'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'identities' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('identities'),
        'name' => 'Identities',
        'attributes' => [
            [
                '$id' => ID::custom('userInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('userId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('provider'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerUid'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerEmail'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 320,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerAccessToken'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ],
            [
                '$id' => ID::custom('providerAccessTokenExpiry'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('providerRefreshToken'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ],
            [
                // Used to store data from provider that may or may not be sensitive
                '$id' => ID::custom('secrets'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => false,
                'filters' => ['json', 'encrypt'],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_userInternalId_provider_providerUid'),
                'type' => Database::INDEX_UNIQUE,
                'attributes' => ['userInternalId', 'provider', 'providerUid'],
                'lengths' => [11, 128, 128],
                'orders' => [Database::ORDER_ASC, Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_provider_providerUid'),
                'type' => Database::INDEX_UNIQUE,
                'attributes' => ['provider', 'providerUid'],
                'lengths' => [128, 128],
                'orders' => [Database::ORDER_ASC, Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_userId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_userInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_provider'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['provider'],
                'lengths' => [128],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_providerUid'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['providerUid'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_providerEmail'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['providerEmail'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_providerAccessTokenExpiry'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['providerAccessTokenExpiry'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'teams' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('teams'),
        'name' => 'Teams',
        'attributes' => [
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('total'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('prefs'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 65535,
                'signed' => true,
                'required' => false,
                'default' => new \stdClass(),
                'array' => false,
                'filters' => ['json'],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_name'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['name'],
                'lengths' => [128],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_total'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['total'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'memberships' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('memberships'),
        'name' => 'Memberships',
        'attributes' => [
            [
                '$id' => ID::custom('userInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('userId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('teamInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('teamId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('roles'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('invited'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('joined'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('confirm'),
                'type' => Database::VAR_BOOLEAN,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('secret'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_unique'),
                'type' => Database::INDEX_UNIQUE,
                'attributes' => ['teamInternalId', 'userInternalId'],
                'lengths' => [Database::LENGTH_KEY, Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC, Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_user'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_team'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['teamInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_userId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_teamId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['teamId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_invited'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['invited'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_joined'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['joined'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_confirm'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['confirm'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'buckets' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('buckets'),
        'name' => 'Buckets',
        'attributes' => [
            [
                '$id' => ID::custom('enabled'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => 128,
                'format' => '',
                'filters' => [],
                'required' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('fileSecurity'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 1,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('maximumFileSize'),
                'type' => Database::VAR_INTEGER,
                'signed' => false,
                'size' => 8,
                'format' => '',
                'filters' => [],
                'required' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('allowedFileExtensions'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => 64,
                'format' => '',
                'filters' => [],
                'required' => true,
                'array' => true,
            ],
            [
                '$id' => 'compression',
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => 10,
                'format' => '',
                'filters' => [],
                'required' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('encryption'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('antivirus'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_fulltext_name'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['name'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_enabled'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['enabled'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_name'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['name'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_fileSecurity'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['fileSecurity'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_maximumFileSize'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['maximumFileSize'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_encryption'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['encryption'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_antivirus'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['antivirus'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
        ]
    ],

    'stats' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('stats'),
        'name' => 'Stats',
        'attributes' => [
            [
                '$id' => ID::custom('metric'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 255,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('region'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 255,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('value'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 8,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('time'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('period'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 4,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_time'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['time'],
                'lengths' => [],
                'orders' => [Database::ORDER_DESC],
            ],
            [
                '$id' => ID::custom('_key_period_time'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['period', 'time'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_metric_period_time'),
                'type' => Database::INDEX_UNIQUE,
                'attributes' => ['metric', 'period', 'time'],
                'lengths' => [],
                'orders' => [Database::ORDER_DESC],
            ],
        ],
    ],

    'providers' => [
        '$collection' => ID::custom(DATABASE::METADATA),
        '$id' => ID::custom('providers'),
        'name' => 'Providers',
        'attributes' => [
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('provider'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('type'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('enabled'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => true,
                'default' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('credentials'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => ['json', 'encrypt'],
            ],
            [
                '$id' => ID::custom('options'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 65535,
                'signed' => true,
                'required' => false,
                'default' => '',
                'array' => false,
                'filters' => ['providerSearch'],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_provider'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['provider'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_name'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['name'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_type'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['type'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_enabled_type'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['enabled', 'type'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ]
        ],
    ],

    'messages' => [
        '$collection' => ID::custom(DATABASE::METADATA),
        '$id' => ID::custom('messages'),
        'name' => 'Messages',
        'attributes' => [
            [
                '$id' => ID::custom('providerType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('status'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => 'processing',
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('data'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 65535,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('topics'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 21845,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('users'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 21845,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('targets'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 21845,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('scheduledAt'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('scheduleInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('scheduleId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('deliveredAt'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('deliveryErrors'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 65535,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('deliveredTotal'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => 0,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => '',
                'array' => false,
                'filters' => ['messageSearch'],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
        ],
    ],

    'topics' => [
        '$collection' => ID::custom(DATABASE::METADATA),
        '$id' => ID::custom('topics'),
        'name' => 'Topics',
        'attributes' => [
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('subscribe'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('emailTotal'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => 0,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('smsTotal'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => 0,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('pushTotal'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => 0,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('targets'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQueryTopicTargets'],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => '',
                'array' => false,
                'filters' => ['topicSearch'],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_name'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['name'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ]
        ],
    ],

    'subscribers' => [
        '$collection' => ID::custom(DATABASE::METADATA),
        '$id' => ID::custom('subscribers'),
        'name' => 'Subscribers',
        'attributes' => [
            [
                '$id' => ID::custom('targetId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('targetInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('userId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('userInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('topicId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('topicInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_targetId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['targetId'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_targetInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['targetInternalId'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_userId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userId'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_userInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userInternalId'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_topicId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['topicId'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_topicInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['topicInternalId'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_unique_target_topic'),
                'type' => Database::INDEX_UNIQUE,
                'attributes' => ['targetInternalId', 'topicInternalId'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_fulltext_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
        ],
    ],

    'targets' => [
        '$collection' => ID::custom(DATABASE::METADATA),
        '$id' => ID::custom('targets'),
        'name' => 'Targets',
        'attributes' => [
            [
                '$id' => ID::custom('userId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('userInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('sessionId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('sessionInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('identifier'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('expired'),
                'type' => Database::VAR_BOOLEAN,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => false,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_userId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userId'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_userInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['userInternalId'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_providerId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['providerId'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_providerInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['providerInternalId'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_identifier'),
                'type' => Database::INDEX_UNIQUE,
                'attributes' => ['identifier'],
                'lengths' => [],
                'orders' => [],
            ],
        ],
    ],
];

$projectCollections = array_merge([
    'databases' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('databases'),
        'name' => 'Databases',
        'attributes' => [
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'size' => 256,
                'required' => true,
                'signed' => true,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('enabled'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => false,
                'default' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('originalId'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'default' => null,
                'array' => false,
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_fulltext_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_name'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['name'],
                'lengths' => [256],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'attributes' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('attributes'),
        'name' => 'Attributes',
        'attributes' => [
            [
                '$id' => ID::custom('databaseInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('databaseId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => false,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('collectionInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('collectionId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('key'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('type'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('status'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('error'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('size'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('required'),
                'type' => Database::VAR_BOOLEAN,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('default'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['casting'],
            ],
            [
                '$id' => ID::custom('signed'),
                'type' => Database::VAR_BOOLEAN,
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('array'),
                'type' => Database::VAR_BOOLEAN,
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('format'),
                'type' => Database::VAR_STRING,
                'size' => 64,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('formatOptions'),
                'type' => Database::VAR_STRING,
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => new stdClass(),
                'array' => false,
                'filters' => ['json', 'range', 'enum'],
            ],
            [
                '$id' => ID::custom('filters'),
                'type' => Database::VAR_STRING,
                'size' => 64,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('options'),
                'type' => Database::VAR_STRING,
                'size' => 16384,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['json'],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_db_collection'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['databaseInternalId', 'collectionInternalId'],
                'lengths' => [Database::LENGTH_KEY, Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC, Database::ORDER_ASC],
            ],
        ],
    ],

    'indexes' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('indexes'),
        'name' => 'Indexes',
        'attributes' => [
            [
                '$id' => ID::custom('databaseInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('databaseId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => false,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('collectionInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('collectionId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('key'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('type'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('status'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('error'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('attributes'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('lengths'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('orders'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 4,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_db_collection'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['databaseInternalId', 'collectionInternalId'],
                'lengths' => [Database::LENGTH_KEY, Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC, Database::ORDER_ASC],
            ],
        ],
    ],

    'functions' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('functions'),
        'name' => 'Functions',
        'attributes' => [
            [
                '$id' => ID::custom('execute'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('enabled'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('live'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('installationId'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('installationInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerRepositoryId'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('repositoryId'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('repositoryInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerBranch'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerRootDirectory'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerSilentMode'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => false,
                'default' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('logging'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('runtime'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('deploymentInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('deployment'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('vars'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQueryVariables'],
            ],
            [
                '$id' => ID::custom('varsProject'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQueryProjectVariables'],
            ],
            [
                '$id' => ID::custom('events'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('scheduleInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('scheduleId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('schedule'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('timeout'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('version'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 8,
                'signed' => true,
                'required' => false,
                'default' => 'v4',
                'array' => false,
                'filters' => [],
            ],
            [
                'array' => false,
                '$id' => ID::custom('entrypoint'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'filters' => [],
            ],
            [
                'array' => false,
                '$id' => ID::custom('commands'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'filters' => [],
            ],
            [
                'array' => false,
                '$id' => ID::custom('specification'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => false,
                'required' => false,
                'default' => APP_FUNCTION_SPECIFICATION_DEFAULT,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('scopes'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => true,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_name'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['name'],
                'lengths' => [256],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_enabled'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['enabled'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_installationId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['installationId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_installationInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['installationInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_providerRepositoryId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['providerRepositoryId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_repositoryId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['repositoryId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_repositoryInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['repositoryInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_runtime'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['runtime'],
                'lengths' => [64],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_deployment'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['deployment'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ]
        ],
    ],

    'deployments' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('deployments'),
        'name' => 'Deployments',
        'attributes' => [
            [
                '$id' => ID::custom('resourceInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('buildInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('buildId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                'array' => false,
                '$id' => ID::custom('entrypoint'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'filters' => [],
            ],
            [
                'array' => false,
                '$id' => ID::custom('commands'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('path'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('type'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('installationId'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('installationInternalId'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerRepositoryId'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('repositoryId'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('repositoryInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerRepositoryName'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerRepositoryOwner'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerRepositoryUrl'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerCommitHash'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerCommitAuthorUrl'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerCommitAuthor'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerCommitMessage'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerCommitUrl'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerBranch'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerBranchUrl'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerRootDirectory'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('providerCommentId'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => 2048,
                'format' => '',
                'filters' => [],
                'required' => false,
                'array' => false,
            ],
            [
                '$id' => ID::custom('size'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('metadata'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384, // https://tools.ietf.org/html/rfc4288#section-4.2
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('chunksTotal'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('chunksUploaded'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('activate'),
                'type' => Database::VAR_BOOLEAN,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => false,
                'array' => false,
                'filters' => [],
            ]
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_resource'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['resourceId'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_resource_type'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['resourceType'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_size'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['size'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_buildId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['buildId'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_activate'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['activate'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'builds' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('builds'),
        'name' => 'Builds',
        'attributes' => [
            [
                '$id' => ID::custom('startTime'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('endTime'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('duration'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('size'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('deploymentInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('deploymentId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('runtime'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => true,
                'default' => '',
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('status'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => true,
                'default' => 'processing',
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('path'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => '',
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('logs'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 1000000,
                'signed' => true,
                'required' => false,
                'default' => '',
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('sourceType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => true,
                'default' => 'local',
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('source'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => true,
                'default' => '',
                'array' => false,
                'filters' => [],
            ]
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_deployment'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['deploymentId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ]
        ],
    ],

    'executions' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('executions'),
        'name' => 'Executions',
        'attributes' => [
            [
                '$id' => ID::custom('functionInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('functionId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('deploymentInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('deploymentId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                'array' => false,
                '$id' => ID::custom('trigger'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('status'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('duration'),
                'type' => Database::VAR_FLOAT,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('errors'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 1000000,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('logs'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 1000000,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                'array' => false,
                '$id' => ID::custom('requestMethod'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'filters' => [],
            ],
            [
                'array' => false,
                '$id' => ID::custom('requestPath'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('requestHeaders'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('responseStatusCode'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('responseHeaders'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('scheduledAt'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('scheduleInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('scheduleId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_function'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['functionId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_fulltext_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_trigger'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['trigger'],
                'lengths' => [32],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_status'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['status'],
                'lengths' => [32],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_requestMethod'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['requestMethod'],
                'lengths' => [128],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_requestPath'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['requestPath'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_deployment'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['deploymentId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_responseStatusCode'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['responseStatusCode'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_duration'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['duration'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'variables' => [
        '$collection' => Database::METADATA,
        '$id' => 'variables',
        'name' => 'variables',
        'attributes' => [
            [
                '$id' => ID::custom('resourceType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 100,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => 'key',
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => 'value',
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 8192,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt']
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => '_key_resourceInternalId',
                'type' => Database::INDEX_KEY,
                'attributes' => ['resourceInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [],
            ],
            [
                '$id' => '_key_resourceType',
                'type' => Database::INDEX_KEY,
                'attributes' => ['resourceType'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => '_key_resourceId_resourceType',
                'type' => Database::INDEX_KEY,
                'attributes' => ['resourceId', 'resourceType'],
                'lengths' => [Database::LENGTH_KEY, 100],
                'orders' => [Database::ORDER_ASC, Database::ORDER_ASC],
            ],
            [
                '$id' => '_key_uniqueKey',
                'type' => Database::INDEX_UNIQUE,
                'attributes' => ['resourceId', 'key', 'resourceType'],
                'lengths' => [Database::LENGTH_KEY, Database::LENGTH_KEY, 100],
                'orders' => [Database::ORDER_ASC, Database::ORDER_ASC, Database::ORDER_ASC],
            ],
            [
                '$id' => '_key_key',
                'type' => Database::INDEX_KEY,
                'attributes' => ['key'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_fulltext_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
        ],
    ],

    'migrations' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('migrations'),
        'name' => 'Migrations',
        'attributes' => [
            [
                '$id' => ID::custom('status'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('stage'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('source'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 8192, // reduce size
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('destination'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false, // make true after patch script
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('credentials'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 65536,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => false,
                'filters' => ['json', 'encrypt'],
            ],
            [
                '$id' => ID::custom('resources'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => [],
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('statusCounters'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 3000,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('resourceData'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 131070,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('errors'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 65535,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ]
        ],
        'indexes' => [
            [
                '$id' => '_key_status',
                'type' => Database::INDEX_KEY,
                'attributes' => ['status'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => '_key_stage',
                'type' => Database::INDEX_KEY,
                'attributes' => ['stage'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => '_key_source',
                'type' => Database::INDEX_KEY,
                'attributes' => ['source'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_fulltext_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ]
        ],
    ],
], $commonCollections);

$consoleCollections = array_merge([
    'projects' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('projects'),
        'name' => 'Projects',
        'attributes' => [
            [
                '$id' => ID::custom('teamInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('teamId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('region'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('description'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('database'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('logo'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('url'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('version'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('legalName'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('legalCountry'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('legalState'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('legalCity'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('legalAddress'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('legalTaxId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => 'accessedAt',
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('services'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('apis'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('smtp'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => false,
                'filters' => ['json', 'encrypt'],
            ],
            [
                '$id' => ID::custom('templates'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 1000000, // TODO make sure size fits
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('auths'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('oAuthProviders'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => [],
                'array' => false,
                'filters' => ['json', 'encrypt'],
            ],
            [
                '$id' => ID::custom('platforms'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQueryPlatforms'],
            ],
            [
                '$id' => ID::custom('webhooks'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQueryWebhooks'],
            ],
            [
                '$id' => ID::custom('keys'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['subQueryKeys'],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('pingCount'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => 0,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('pingedAt'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ]
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_name'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['name'],
                'lengths' => [128],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_team'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['teamId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_pingCount'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['pingCount'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_pingedAt'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['pingedAt'],
                'lengths' => [],
                'orders' => [],
            ]
        ],
    ],

    'schedules' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('schedules'),
        'name' => 'schedules',
        'attributes' => [
            [
                '$id' => ID::custom('resourceType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 100,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceUpdatedAt'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('projectId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('schedule'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 100,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('data'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 65535,
                'signed' => true,
                'required' => false,
                'default' => new \stdClass(),
                'array' => false,
                'filters' => ['json', 'encrypt'],
            ],
            [
                '$id' => ID::custom('active'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => false,
                'default' => null,
                'array' => false,
            ],
            [
                '$id' => ID::custom('region'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 10,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_region_resourceType_resourceUpdatedAt'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['region', 'resourceType', 'resourceUpdatedAt'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_region_resourceType_projectId_resourceId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['region', 'resourceType', 'projectId', 'resourceId'],
                'lengths' => [],
                'orders' => [],
            ],
        ],
    ],

    'platforms' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('platforms'),
        'name' => 'platforms',
        'attributes' => [
            [
                '$id' => ID::custom('projectInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('projectId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('type'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('key'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('store'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('hostname'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 256,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ]
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_project'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['projectInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'keys' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('keys'),
        'name' => 'keys',
        'attributes' => [
            [
                '$id' => ID::custom('projectInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('projectId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => 0,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('scopes'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('secret'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 512, // var_dump of \bin2hex(\random_bytes(128)) => string(256) doubling for encryption
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ],
            [
                '$id' => ID::custom('expire'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('accessedAt'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('sdks'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_project'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['projectInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => '_key_accessedAt',
                'type' => Database::INDEX_KEY,
                'attributes' => ['accessedAt'],
                'lengths' => [],
                'orders' => [],
            ],
        ],
    ],

    'webhooks' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('webhooks'),
        'name' => 'webhooks',
        'attributes' => [
            [
                '$id' => ID::custom('projectInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('projectId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('url'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('httpUser'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('httpPass'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY, // TODO will the length suffice after encryption?
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['encrypt'],
            ],
            [
                '$id' => ID::custom('security'),
                'type' => Database::VAR_BOOLEAN,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('events'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('signatureKey'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('enabled'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => false,
                'default' => true,
                'array' => false,
            ],
            [
                '$id' => ID::custom('logs'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 1000000,
                'signed' => true,
                'required' => false,
                'default' => '',
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('attempts'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => 0,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_project'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['projectInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ]
        ],
    ],

    'certificates' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('certificates'),
        'name' => 'Certificates',
        'attributes' => [
            [
                '$id' => ID::custom('domain'),
                'type' => Database::VAR_STRING,
                'format' => '',
                // The maximum total length of a domain name or number is 255 characters.
                // https://datatracker.ietf.org/doc/html/rfc2821#section-4.5.3.1
                // https://datatracker.ietf.org/doc/html/rfc5321#section-4.5.3.1.2
                'size' => 255,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('issueDate'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('renewDate'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('attempts'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('logs'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 1000000,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('updated'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_domain'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['domain'],
                'lengths' => [255],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'realtime' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('realtime'),
        'name' => 'Realtime Connections',
        'attributes' => [
            [
                '$id' => ID::custom('container'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('timestamp'),
                'type' => Database::VAR_DATETIME,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['datetime'],
            ],
            [
                '$id' => ID::custom('value'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [], //TODO: use json filter
            ]
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_timestamp'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['timestamp'],
                'lengths' => [],
                'orders' => [Database::ORDER_DESC],
            ],
        ]
    ],

    'rules' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('rules'),
        'name' => 'Rules',
        'attributes' => [
            [
                '$id' => ID::custom('projectId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('projectInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('domain'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 100,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('status'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('certificateId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ]
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_domain'),
                'type' => Database::INDEX_UNIQUE,
                'attributes' => ['domain'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_projectInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['projectInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_projectId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['projectId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => '_key_resourceInternalId',
                'type' => Database::INDEX_KEY,
                'attributes' => ['resourceInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => '_key_resourceId',
                'type' => Database::INDEX_KEY,
                'attributes' => ['resourceId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => '_key_resourceType',
                'type' => Database::INDEX_KEY,
                'attributes' => ['resourceType'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'installations' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('installations'),
        'name' => 'installations',
        'attributes' => [
            [
                '$id' => ID::custom('projectId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('projectInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerInstallationId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('organization'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('provider'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('personal'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => false,
                'default' => false,
                'array' => false,
            ],
        ],
        'indexes' => [

            [
                '$id' => ID::custom('_key_projectInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['projectInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_projectId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['projectId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_providerInstallationId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['providerInstallationId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'repositories' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('repositories'),
        'name' => 'repositories',
        'attributes' => [
            [
                '$id' => ID::custom('installationId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => []
            ],
            [
                '$id' => ID::custom('installationInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('projectId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => []
            ],
            [
                '$id' => ID::custom('projectInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerRepositoryId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => []
            ],
            [
                '$id' => ID::custom('resourceId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('resourceType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => []
            ],
            [
                '$id' => ID::custom('providerPullRequestIds'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 128,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => true,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_installationId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['installationId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_installationInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['installationInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_projectInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['projectInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_projectId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['projectId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_providerRepositoryId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['providerRepositoryId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_resourceId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['resourceId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => '_key_resourceInternalId',
                'type' => Database::INDEX_KEY,
                'attributes' => ['resourceInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_resourceType'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['resourceType'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ]
        ],
    ],

    'vcsComments' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('vcsComments'),
        'name' => 'vcsComments',
        'attributes' => [
            [
                '$id' => ID::custom('installationId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => []
            ],
            [
                '$id' => ID::custom('installationInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('projectId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => []
            ],
            [
                '$id' => ID::custom('projectInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('providerRepositoryId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => []
            ],
            [
                '$id' => ID::custom('providerCommentId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => []
            ],
            [
                '$id' => ID::custom('providerPullRequestId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => []
            ],
            [
                '$id' => ID::custom('providerBranch'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => []
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_installationId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['installationId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_installationInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['installationInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_projectInternalId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['projectInternalId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_projectId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['projectId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_providerRepositoryId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['providerRepositoryId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_providerPullRequestId'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['providerPullRequestId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_providerBranch'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['providerBranch'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ],

    'vcsCommentLocks' => [
        '$collection' => ID::custom(Database::METADATA),
        '$id' => ID::custom('vcsCommentLocks'),
        'name' => 'vcsCommentLocks',
        'attributes' => [],
        'indexes' => []
    ],
], $commonCollections);

$bucketCollections = [
    'files' => [
        '$collection' => ID::custom('buckets'),
        '$id' => ID::custom('files'),
        '$name' => 'Files',
        'attributes' => [
            [
                'array' => false,
                '$id' => ID::custom('bucketId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => false,
                'default' => null,
                'filters' => [],
            ],
            [
                'array' => false,
                '$id' => ID::custom('bucketInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('path'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('signature'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('mimeType'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 255, // https://tools.ietf.org/html/rfc4288#section-4.2
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('metadata'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 75000, // https://tools.ietf.org/html/rfc4288#section-4.2
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => ['json'],
            ],
            [
                '$id' => ID::custom('sizeOriginal'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 8,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('sizeActual'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 8,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('algorithm'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 255,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('comment'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('openSSLVersion'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 64,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('openSSLCipher'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 64,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('openSSLTag'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('openSSLIV'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 2048,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('chunksTotal'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('chunksUploaded'),
                'type' => Database::VAR_INTEGER,
                'format' => '',
                'size' => 0,
                'signed' => false,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_key_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_bucket'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['bucketId'],
                'lengths' => [Database::LENGTH_KEY],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_name'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['name'],
                'lengths' => [256],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_signature'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['signature'],
                'lengths' => [256],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_mimeType'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['mimeType'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_sizeOriginal'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['sizeOriginal'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_chunksTotal'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['chunksTotal'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_chunksUploaded'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['chunksUploaded'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
        ]
    ],
];

$dbCollections = [
    'collections' => [
        '$collection' => ID::custom('databases'),
        '$id' => ID::custom('collections'),
        'name' => 'Collections',
        'attributes' => [
            [
                '$id' => ID::custom('databaseInternalId'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => Database::LENGTH_KEY,
                'signed' => true,
                'required' => true,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('databaseId'),
                'type' => Database::VAR_STRING,
                'signed' => true,
                'size' => Database::LENGTH_KEY,
                'format' => '',
                'filters' => [],
                'required' => true,
                'default' => null,
                'array' => false,
            ],
            [
                '$id' => ID::custom('name'),
                'type' => Database::VAR_STRING,
                'size' => 256,
                'required' => true,
                'signed' => true,
                'array' => false,
                'filters' => [],
            ],
            [
                '$id' => ID::custom('enabled'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => true,
                'default' => null,
                'array' => false,
            ],
            [
                '$id' => ID::custom('documentSecurity'),
                'type' => Database::VAR_BOOLEAN,
                'signed' => true,
                'size' => 0,
                'format' => '',
                'filters' => [],
                'required' => true,
                'default' => null,
                'array' => false,
            ],
            [
                '$id' => ID::custom('attributes'),
                'type' => Database::VAR_STRING,
                'size' => 1000000,
                'required' => false,
                'signed' => true,
                'array' => false,
                'filters' => ['subQueryAttributes'],
            ],
            [
                '$id' => ID::custom('indexes'),
                'type' => Database::VAR_STRING,
                'size' => 1000000,
                'required' => false,
                'signed' => true,
                'array' => false,
                'filters' => ['subQueryIndexes'],
            ],
            [
                '$id' => ID::custom('search'),
                'type' => Database::VAR_STRING,
                'format' => '',
                'size' => 16384,
                'signed' => true,
                'required' => false,
                'default' => null,
                'array' => false,
                'filters' => [],
            ],
        ],
        'indexes' => [
            [
                '$id' => ID::custom('_fulltext_search'),
                'type' => Database::INDEX_FULLTEXT,
                'attributes' => ['search'],
                'lengths' => [],
                'orders' => [],
            ],
            [
                '$id' => ID::custom('_key_name'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['name'],
                'lengths' => [256],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_enabled'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['enabled'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
            [
                '$id' => ID::custom('_key_documentSecurity'),
                'type' => Database::INDEX_KEY,
                'attributes' => ['documentSecurity'],
                'lengths' => [],
                'orders' => [Database::ORDER_ASC],
            ],
        ],
    ]
];


$collections = [
    'projects' => $projectCollections,
    'console'  => $consoleCollections,
    'buckets' => $bucketCollections,
    'databases' => $dbCollections
];

return $collections;
