/////////////////////////////////////////////////////////////////////////////
// Name:        event.h
// Purpose:     interface of wxEvtHandler, wxEventBlocker and many
//              wxEvent-derived classes
// Author:      wxWidgets team
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

#if wxUSE_BASE

/**
    The predefined constants for the number of times we propagate event
    upwards window child-parent chain.
*/
enum wxEventPropagation
{
    /// don't propagate it at all
    wxEVENT_PROPAGATE_NONE = 0,

    /// propagate it until it is processed
    wxEVENT_PROPAGATE_MAX = INT_MAX
};

/**
    The different categories for a wxEvent; see wxEvent::GetEventCategory.

    @note They are used as OR-combinable flags by wxEventLoopBase::YieldFor.
*/
enum wxEventCategory
{
    /**
        This is the category for those events which are generated to update
        the appearance of the GUI but which (usually) do not comport data
        processing, i.e. which do not provide input or output data
        (e.g. size events, scroll events, etc).
        They are events NOT directly generated by the user's input devices.
    */
    wxEVT_CATEGORY_UI = 1,

    /**
        This category groups those events which are generated directly from the
        user through input devices like mouse and keyboard and usually result in
        data to be processed from the application
        (e.g. mouse clicks, key presses, etc).
    */
    wxEVT_CATEGORY_USER_INPUT = 2,

    /// This category is for wxSocketEvent
    wxEVT_CATEGORY_SOCKET = 4,

    /// This category is for wxTimerEvent
    wxEVT_CATEGORY_TIMER = 8,

    /**
        This category is for any event used to send notifications from the
        secondary threads to the main one or in general for notifications among
        different threads (which may or may not be user-generated).
        See e.g. wxThreadEvent.
    */
    wxEVT_CATEGORY_THREAD = 16,

    /**
        This mask is used in wxEventLoopBase::YieldFor to specify that all event
        categories should be processed.
    */
    wxEVT_CATEGORY_ALL =
        wxEVT_CATEGORY_UI|wxEVT_CATEGORY_USER_INPUT|wxEVT_CATEGORY_SOCKET| \
        wxEVT_CATEGORY_TIMER|wxEVT_CATEGORY_THREAD
};

/**
    @class wxEvent

    An event is a structure holding information about an event passed to a
    callback or member function.

    wxEvent used to be a multipurpose event object, and is an abstract base class
    for other event classes (see below).

    For more information about events, see the @ref overview_events overview.

    @beginWxPerlOnly
    In wxPerl custom event classes should be derived from
    @c Wx::PlEvent and @c Wx::PlCommandEvent.
    @endWxPerlOnly

    @library{wxbase}
    @category{events}

    @see wxCommandEvent, wxMouseEvent
*/
class wxEvent : public wxObject
{
public:
    /**
        Constructor.

        Notice that events are usually created by wxWidgets itself and creating
        e.g. a wxPaintEvent in your code and sending it to e.g. a wxTextCtrl
        will not usually affect it at all as native controls have no specific
        knowledge about wxWidgets events. However you may construct objects of
        specific types and pass them to wxEvtHandler::ProcessEvent() if you
        want to create your own custom control and want to process its events
        in the same manner as the standard ones.

        Also please notice that the order of parameters in this constructor is
        different from almost all the derived classes which specify the event
        type as the first argument.

        @param id
            The identifier of the object (window, timer, ...) which generated
            this event.
        @param eventType
            The unique type of event, e.g. @c wxEVT_PAINT, @c wxEVT_SIZE or
            @c wxEVT_BUTTON.
    */
    wxEvent(int id = 0, wxEventType eventType = wxEVT_NULL);

    /**
        Returns a copy of the event.

        Any event that is posted to the wxWidgets event system for later action
        (via wxEvtHandler::AddPendingEvent, wxEvtHandler::QueueEvent or wxPostEvent())
        must implement this method.

        All wxWidgets events fully implement this method, but any derived events
        implemented by the user should also implement this method just in case they
        (or some event derived from them) are ever posted.

        All wxWidgets events implement a copy constructor, so the easiest way of
        implementing the Clone function is to implement a copy constructor for
        a new event (call it MyEvent) and then define the Clone function like this:

        @code
        wxEvent *Clone() const { return new MyEvent(*this); }
        @endcode
    */
    virtual wxEvent* Clone() const = 0;

    /**
        Returns the object (usually a window) associated with the event, if any.
    */
    wxObject* GetEventObject() const;

    /**
        Returns the identifier of the given event type, such as @c wxEVT_BUTTON.
    */
    wxEventType GetEventType() const;

    /**
        Returns a generic category for this event.
        wxEvent implementation returns @c wxEVT_CATEGORY_UI by default.

        This function is used to selectively process events in wxEventLoopBase::YieldFor.
    */
    virtual wxEventCategory GetEventCategory() const;

    /**
        Returns the identifier associated with this event, such as a button command id.
    */
    int GetId() const;

    /**
        Return the user data associated with a dynamically connected event handler.

        wxEvtHandler::Connect() and wxEvtHandler::Bind() allow associating
        optional @c userData pointer with the handler and this method returns
        the value of this pointer.

        The returned pointer is owned by wxWidgets and must not be deleted.

        @since 2.9.5
    */
    wxObject *GetEventUserData() const;

    /**
        Returns @true if the event handler should be skipped, @false otherwise.
    */
    bool GetSkipped() const;

    /**
        Gets the timestamp for the event. The timestamp is the time in milliseconds
        since some fixed moment (not necessarily the standard Unix Epoch, so only
        differences between the timestamps and not their absolute values usually make sense).

        @warning
        wxWidgets returns a non-NULL timestamp only for mouse and key events
        (see wxMouseEvent and wxKeyEvent).
    */
    long GetTimestamp() const;

    /**
        Returns @true if the event is or is derived from wxCommandEvent else it returns @false.

        @note exists only for optimization purposes.
    */
    bool IsCommandEvent() const;

    /**
        Sets the propagation level to the given value (for example returned from an
        earlier call to wxEvent::StopPropagation).
    */
    void ResumePropagation(int propagationLevel);

    /**
        Sets the originating object.
    */
    void SetEventObject(wxObject* object);

    /**
        Sets the event type.
    */
    void SetEventType(wxEventType type);

    /**
        Sets the identifier associated with this event, such as a button command id.
    */
    void SetId(int id);

    /**
        Sets the timestamp for the event.
    */
    void SetTimestamp(long timeStamp = 0);

    /**
        Test if this event should be propagated or not, i.e.\ if the propagation level
        is currently greater than 0.
    */
    bool ShouldPropagate() const;

    /**
        This method can be used inside an event handler to control whether further
        event handlers bound to this event will be called after the current one returns.

        Without Skip() (or equivalently if Skip(@false) is used), the event will not
        be processed any more. If Skip(@true) is called, the event processing system
        continues searching for a further handler function for this event, even though
        it has been processed already in the current handler.

        In general, it is recommended to skip all non-command events to allow the
        default handling to take place. The command events are, however, normally not
        skipped as usually a single command such as a button click or menu item
        selection must only be processed by one handler.
    */
    void Skip(bool skip = true);

    /**
        Stop the event from propagating to its parent window.

        Returns the old propagation level value which may be later passed to
        ResumePropagation() to allow propagating the event again.
    */
    int StopPropagation();

protected:
    /**
        Indicates how many levels the event can propagate.

        This member is protected and should typically only be set in the constructors
        of the derived classes. It may be temporarily changed by StopPropagation()
        and ResumePropagation() and tested with ShouldPropagate().

        The initial value is set to either @c wxEVENT_PROPAGATE_NONE (by default)
        meaning that the event shouldn't be propagated at all or to
        @c wxEVENT_PROPAGATE_MAX (for command events) meaning that it should be
        propagated as much as necessary.

        Any positive number means that the event should be propagated but no more than
        the given number of times. E.g. the propagation level may be set to 1 to
        propagate the event to its parent only, but not to its grandparent.
    */
    int m_propagationLevel;
};

#endif // wxUSE_BASE

#if wxUSE_GUI

/**
    @class wxEventBlocker

    This class is a special event handler which allows discarding
    any event (or a set of event types) directed to a specific window.

    Example:

    @code
    void MyWindow::DoSomething()
    {
        {
            // block all events directed to this window while
            // we do the 1000 FunctionWhichSendsEvents() calls
            wxEventBlocker blocker(this);

            for ( int i = 0; i  1000; i++ )
                FunctionWhichSendsEvents(i);

        } // ~wxEventBlocker called, old event handler is restored

        // the event generated by this call will be processed:
        FunctionWhichSendsEvents(0)
    }
    @endcode

    @library{wxcore}
    @category{events}

    @see @ref overview_events_processing, wxEvtHandler
*/
class wxEventBlocker : public wxEvtHandler
{
public:
    /**
        Constructs the blocker for the given window and for the given event type.

        If @a type is @c wxEVT_ANY, then all events for that window are blocked.
        You can call Block() after creation to add other event types to the list
        of events to block.

        Note that the @a win window @b must remain alive until the
        wxEventBlocker object destruction.
    */
    wxEventBlocker(wxWindow* win, wxEventType type = -1);

    /**
        Destructor. The blocker will remove itself from the chain of event handlers for
        the window provided in the constructor, thus restoring normal processing of events.
    */
    virtual ~wxEventBlocker();

    /**
        Adds to the list of event types which should be blocked the given @a eventType.
    */
    void Block(wxEventType eventType);
};



/**
   Helper class to temporarily change an event to not propagate.
*/
class wxPropagationDisabler
{
public:
    wxPropagationDisabler(wxEvent& event);
    ~wxPropagationDisabler();
};


/**
   Helper class to temporarily lower propagation level.
*/
class wxPropagateOnce
{
public:
    wxPropagateOnce(wxEvent& event);
    ~wxPropagateOnce();
};

#endif // wxUSE_GUI

#if wxUSE_BASE

/**
    @class wxEvtHandler

    A class that can handle events from the windowing system.
    wxWindow is (and therefore all window classes are) derived from this class.

    When events are received, wxEvtHandler invokes the method listed in the
    event table using itself as the object. When using multiple inheritance
    <b>it is imperative that the wxEvtHandler(-derived) class is the first
    class inherited</b> such that the @c this pointer for the overall object
    will be identical to the @c this pointer of the wxEvtHandler portion.

    @library{wxbase}
    @category{events}

    @see @ref overview_events_processing, wxEventBlocker, wxEventLoopBase
*/
class wxEvtHandler : public wxObject, public wxTrackable
{
public:
    /**
        Constructor.
    */
    wxEvtHandler();

    /**
        Destructor.

        If the handler is part of a chain, the destructor will unlink itself
        (see Unlink()).
    */
    virtual ~wxEvtHandler();


    /**
        @name Event queuing and processing
    */
    //@{

    /**
        Queue event for a later processing.

        This method is similar to ProcessEvent() but while the latter is
        synchronous, i.e. the event is processed immediately, before the
        function returns, this one is asynchronous and returns immediately
        while the event will be processed at some later time (usually during
        the next event loop iteration).

        Another important difference is that this method takes ownership of the
        @a event parameter, i.e. it will delete it itself. This implies that
        the event should be allocated on the heap and that the pointer can't be
        used any more after the function returns (as it can be deleted at any
        moment).

        QueueEvent() can be used for inter-thread communication from the worker
        threads to the main thread, it is safe in the sense that it uses
        locking internally and avoids the problem mentioned in AddPendingEvent()
        documentation by ensuring that the @a event object is not used by the
        calling thread any more. Care should still be taken to avoid that some
        fields of this object are used by it, notably any wxString members of
        the event object must not be shallow copies of another wxString object
        as this would result in them still using the same string buffer behind
        the scenes. For example:
        @code
            void FunctionInAWorkerThread(const wxString& str)
            {
                wxCommandEvent* evt = new wxCommandEvent;

                // NOT evt->SetString(str) as this would be a shallow copy
                evt->SetString(str.c_str()); // make a deep copy

                wxTheApp->QueueEvent( evt );
            }
        @endcode

        Note that you can use wxThreadEvent instead of wxCommandEvent
        to avoid this problem:
        @code
            void FunctionInAWorkerThread(const wxString& str)
            {
                wxThreadEvent evt;
                evt.SetString(str);

                // wxThreadEvent::Clone() makes sure that the internal wxString
                // member is not shared by other wxString instances:
                wxTheApp->QueueEvent( evt.Clone() );
            }
        @endcode

        Finally notice that this method automatically wakes up the event loop
        if it is currently idle by calling ::wxWakeUpIdle() so there is no need
        to do it manually when using it.

        @since 2.9.0

        @param event
            A heap-allocated event to be queued, QueueEvent() takes ownership
            of it. This parameter shouldn't be @c NULL.
     */
    virtual void QueueEvent(wxEvent *event);

    /**
        Post an event to be processed later.

        This function is similar to QueueEvent() but can't be used to post
        events from worker threads for the event objects with wxString fields
        (i.e. in practice most of them) because of an unsafe use of the same
        wxString object which happens because the wxString field in the
        original @a event object and its copy made internally by this function
        share the same string buffer internally. Use QueueEvent() to avoid
        this.

        A copy of @a event is made by the function, so the original can be deleted
        as soon as function returns (it is common that the original is created
        on the stack). This requires that the wxEvent::Clone() method be
        implemented by event so that it can be duplicated and stored until it
        gets processed.

        @param event
            Event to add to the pending events queue.
    */
    virtual void AddPendingEvent(const wxEvent& event);

    /**
         Asynchronously call the given method.

         Calling this function on an object schedules an asynchronous call to
         the method specified as CallAfter() argument at a (slightly) later
         time. This is useful when processing some events as certain actions
         typically can't be performed inside their handlers, e.g. you shouldn't
         show a modal dialog from a mouse click event handler as this would
         break the mouse capture state -- but you can call a method showing
         this message dialog after the current event handler completes.

         The method being called must be the method of the object on which
         CallAfter() itself is called.

         Notice that it is safe to use CallAfter() from other, non-GUI,
         threads, but that the method will be always called in the main, GUI,
         thread context.

         Example of use:
         @code
         class MyFrame : public wxFrame {
            void OnClick(wxMouseEvent& event) {
                CallAfter(&MyFrame::ShowPosition, event.GetPosition());
            }

            void ShowPosition(const wxPoint& pos) {
                if ( wxMessageBox(
                        wxString::Format("Perform click at (%d, %d)?",
                                         pos.x, pos.y), "", wxYES_NO) == wxYES )
                {
                    ... do take this click into account ...
                }
            }
         };
         @endcode

         @param method The method to call.
         @param x1 The (optional) first parameter to pass to the method.
            Currently, 0, 1 or 2 parameters can be passed. If you need to pass
            more than 2 arguments, you can use the CallAfter<T>(const T& fn)
            overload that can call any functor.

         @note This method is not available with Visual C++ before version 8
               (Visual Studio 2005) as earlier versions of the compiler don't
               have the required support for C++ templates to implement it.

         @since 2.9.5
     */
    template<typename T, typename T1, ...>
    void CallAfter(void (T::*method)(T1, ...), T1 x1, ...);

    /**
         Asynchronously call the given functor.

         Calling this function on an object schedules an asynchronous call to
         the functor specified as CallAfter() argument at a (slightly) later
         time. This is useful when processing some events as certain actions
         typically can't be performed inside their handlers, e.g. you shouldn't
         show a modal dialog from a mouse click event handler as this would
         break the mouse capture state -- but you can call a function showing
         this message dialog after the current event handler completes.

         Notice that it is safe to use CallAfter() from other, non-GUI,
         threads, but that the method will be always called in the main, GUI,
         thread context.

         This overload is particularly useful in combination with C++11 lambdas:
         @code
         wxGetApp().CallAfter([]{
             wxBell();
         });
         @endcode

         @param functor The functor to call.

         @note This method is not available with Visual C++ before version 8
               (Visual Studio 2005) as earlier versions of the compiler don't
               have the required support for C++ templates to implement it.

         @since 3.0
     */
    template<typename T>
    void CallAfter(const T& functor);

    /**
        Processes an event, searching event tables and calling zero or more suitable
        event handler function(s).

        Normally, your application would not call this function: it is called in the
        wxWidgets implementation to dispatch incoming user interface events to the
        framework (and application).

        However, you might need to call it if implementing new functionality
        (such as a new control) where you define new event types, as opposed to
        allowing the user to override virtual functions.

        Notice that you don't usually need to override ProcessEvent() to
        customize the event handling, overriding the specially provided
        TryBefore() and TryAfter() functions is usually enough. For example,
        wxMDIParentFrame may override TryBefore() to ensure that the menu
        events are processed in the active child frame before being processed
        in the parent frame itself.

        The normal order of event table searching is as follows:
        -# wxApp::FilterEvent() is called. If it returns anything but @c -1
           (default) the processing stops here.
        -# TryBefore() is called (this is where wxValidator are taken into
           account for wxWindow objects). If this returns @true, the function exits.
        -# If the object is disabled (via a call to wxEvtHandler::SetEvtHandlerEnabled)
           the function skips to step (7).
        -# Dynamic event table of the handlers bound using Bind<>() is
           searched in the most-recently-bound to the most-early-bound order.
           If a handler is found, it is executed and the function
           returns @true unless the handler used wxEvent::Skip() to indicate
           that it didn't handle the event in which case the search continues.
        -# Static events table of the handlers bound using event table
           macros is searched for this event handler in the order of appearance
           of event table macros in the source code. If this fails, the base
           class event table is tried, and so on until no more tables
           exist or an appropriate function was found. If a handler is found,
           the same logic as in the previous step applies.
        -# The search is applied down the entire chain of event handlers (usually the
           chain has a length of one). This chain can be formed using wxEvtHandler::SetNextHandler():
              @image html overview_events_chain.png
           (referring to the image, if @c A->ProcessEvent is called and it doesn't handle
            the event, @c B->ProcessEvent will be called and so on...).
           Note that in the case of wxWindow you can build a stack of event handlers
           (see wxWindow::PushEventHandler() for more info).
           If any of the handlers of the chain return @true, the function exits.
        -# TryAfter() is called: for the wxWindow object this may propagate the
           event to the window parent (recursively). If the event is still not
           processed, ProcessEvent() on wxTheApp object is called as the last
           step.

        Notice that steps (2)-(6) are performed in ProcessEventLocally()
        which is called by this function.

        @param event
            Event to process.
        @return
            @true if a suitable event handler function was found and executed,
            and the function did not call wxEvent::Skip.
    */
    virtual bool ProcessEvent(wxEvent& event);

    /**
        Try to process the event in this handler and all those chained to it.

        As explained in ProcessEvent() documentation, the event handlers may be
        chained in a doubly-linked list. This function tries to process the
        event in this handler (including performing any pre-processing done in
        TryBefore(), e.g. applying validators) and all those following it in
        the chain until the event is processed or the chain is exhausted.

        This function is called from ProcessEvent() and, in turn, calls
        TryBefore() and TryAfter(). It is not virtual and so cannot be
        overridden but can, and should, be called to forward an event to
        another handler instead of ProcessEvent() which would result in a
        duplicate call to TryAfter(), e.g. resulting in all unprocessed events
        being sent to the application object multiple times.

        @since 2.9.1

        @param event
            Event to process.
        @return
            @true if this handler of one of those chained to it processed the
            event.
     */
    bool ProcessEventLocally(wxEvent& event);

    /**
        Processes an event by calling ProcessEvent() and handles any exceptions
        that occur in the process.
        If an exception is thrown in event handler, wxApp::OnExceptionInMainLoop is called.

        @param event
            Event to process.

        @return @true if the event was processed, @false if no handler was found
                 or an exception was thrown.

        @see wxWindow::HandleWindowEvent
    */
    bool SafelyProcessEvent(wxEvent& event);

    /**
        Processes the pending events previously queued using QueueEvent() or
        AddPendingEvent(); you must call this function only if you are sure
        there are pending events for this handler, otherwise a @c wxCHECK
        will fail.

        The real processing still happens in ProcessEvent() which is called by this
        function.

        Note that this function needs a valid application object (see
        wxAppConsole::GetInstance()) because wxApp holds the list of the event
        handlers with pending events and this function manipulates that list.
    */
    void ProcessPendingEvents();

    /**
        Deletes all events queued on this event handler using QueueEvent() or
        AddPendingEvent().

        Use with care because the events which are deleted are (obviously) not
        processed and this may have unwanted consequences (e.g. user actions events
        will be lost).
    */
    void DeletePendingEvents();

    //@}


    /**
        @name Connecting and disconnecting
    */
    //@{

    /**
        Connects the given function dynamically with the event handler, id and
        event type.

        Notice that Bind() provides a more flexible and safer way to do the
        same thing as Connect(), please use it in any new code -- while
        Connect() is not formally deprecated due to its existing widespread
        usage, it has no advantages compared to Bind().

        This is an alternative to the use of static event tables. It is more
        flexible as it allows connecting events generated by some object to an
        event handler defined in a different object of a different class (which
        is impossible to do directly with the event tables -- the events can be
        only handled in another object if they are propagated upwards to it).
        Do make sure to specify the correct @a eventSink when connecting to an
        event of a different object.

        See @ref overview_events_bind for more detailed explanation
        of this function and the @ref page_samples_event sample for usage
        examples.

        This specific overload allows you to connect an event handler to a @e range
        of @e source IDs.
        Do not confuse @e source IDs with event @e types: source IDs identify the
        event generator objects (typically wxMenuItem or wxWindow objects) while the
        event @e type identify which type of events should be handled by the
        given @e function (an event generator object may generate many different
        types of events!).

        @param id
            The first ID of the identifier range to be associated with the event
            handler function.
        @param lastId
            The last ID of the identifier range to be associated with the event
            handler function.
        @param eventType
            The event type to be associated with this event handler.
        @param function
            The event handler function. Note that this function should
            be explicitly converted to the correct type which can be done using a macro
            called @c wxFooEventHandler for the handler for any @c wxFooEvent.
        @param userData
            Optional data to be associated with the event table entry.
            wxWidgets will take ownership of this pointer, i.e. it will be
            destroyed when the event handler is disconnected or at the program
            termination. This pointer can be retrieved using
            wxEvent::GetEventUserData() later.
        @param eventSink
            Object whose member function should be called. It must be specified
            when connecting an event generated by one object to a member
            function of a different object. If it is omitted, @c this is used.

        @beginWxPerlOnly
        In wxPerl this function takes 4 arguments: @a id, @a lastid,
        @a type, @a method; if @a method is undef, the handler is
        disconnected.
        @endWxPerlOnly

        @see Bind<>()
    */
    void Connect(int id, int lastId, wxEventType eventType,
                 wxObjectEventFunction function,
                 wxObject* userData = NULL,
                 wxEvtHandler* eventSink = NULL);

    /**
        See the Connect(int, int, wxEventType, wxObjectEventFunction, wxObject*, wxEvtHandler*)
        overload for more info.

        This overload can be used to attach an event handler to a single source ID:

        Example:
        @code
        frame->Connect( wxID_EXIT,
                        wxEVT_MENU,
                        wxCommandEventHandler(MyFrame::OnQuit) );
        @endcode

        @beginWxPerlOnly
        Not supported by wxPerl.
        @endWxPerlOnly
    */
    void Connect(int id, wxEventType eventType,
                 wxObjectEventFunction function,
                 wxObject* userData = NULL,
                 wxEvtHandler* eventSink = NULL);

    /**
        See the Connect(int, int, wxEventType, wxObjectEventFunction, wxObject*, wxEvtHandler*)
        overload for more info.

        This overload will connect the given event handler so that regardless of the
        ID of the event source, the handler will be called.

        @beginWxPerlOnly
        Not supported by wxPerl.
        @endWxPerlOnly
    */
    void Connect(wxEventType eventType,
                 wxObjectEventFunction function,
                 wxObject* userData = NULL,
                 wxEvtHandler* eventSink = NULL);

    /**
        Disconnects the given function dynamically from the event handler, using the
        specified parameters as search criteria and returning @true if a matching
        function has been found and removed.

        This method can only disconnect functions which have been added using the
        Connect() method. There is no way to disconnect functions connected using
        the (static) event tables.

        @param eventType
            The event type associated with this event handler.
        @param function
            The event handler function.
        @param userData
            Data associated with the event table entry.
        @param eventSink
            Object whose member function should be called.

        @beginWxPerlOnly
        Not supported by wxPerl.
        @endWxPerlOnly
    */
    bool Disconnect(wxEventType eventType,
                    wxObjectEventFunction function,
                    wxObject* userData = NULL,
                    wxEvtHandler* eventSink = NULL);

    /**
        See the Disconnect(wxEventType, wxObjectEventFunction, wxObject*, wxEvtHandler*)
        overload for more info.

        This overload takes the additional @a id parameter.

        @beginWxPerlOnly
        Not supported by wxPerl.
        @endWxPerlOnly
    */
    bool Disconnect(int id = wxID_ANY,
                    wxEventType eventType = wxEVT_NULL,
                    wxObjectEventFunction function = NULL,
                    wxObject* userData = NULL,
                    wxEvtHandler* eventSink = NULL);

    /**
        See the Disconnect(wxEventType, wxObjectEventFunction, wxObject*, wxEvtHandler*)
        overload for more info.

        This overload takes an additional range of source IDs.

        @beginWxPerlOnly
        In wxPerl this function takes 3 arguments: @a id,
        @a lastid, @a type.
        @endWxPerlOnly
    */
    bool Disconnect(int id, int lastId,
                    wxEventType eventType,
                    wxObjectEventFunction function = NULL,
                    wxObject* userData = NULL,
                    wxEvtHandler* eventSink = NULL);
    //@}


    /**
        @name Binding and Unbinding
    */
    //@{

    /**
        Binds the given function, functor or method dynamically with the event.

        This offers basically the same functionality as Connect(), but it is
        more flexible as it also allows you to use ordinary functions and
        arbitrary functors as event handlers. It is also less restrictive then
        Connect() because you can use an arbitrary method as an event handler,
        whereas Connect() requires a wxEvtHandler derived handler.

        See @ref overview_events_bind for more detailed explanation
        of this function and the @ref page_samples_event sample for usage
        examples.

        @param eventType
            The event type to be associated with this event handler.
        @param functor
            The event handler functor. This can be an ordinary function but also
            an arbitrary functor like boost::function<>.
        @param id
            The first ID of the identifier range to be associated with the event
            handler.
        @param lastId
            The last ID of the identifier range to be associated with the event
            handler.
        @param userData
            Optional data to be associated with the event table entry.
            wxWidgets will take ownership of this pointer, i.e. it will be
            destroyed when the event handler is disconnected or at the program
            termination. This pointer can be retrieved using
            wxEvent::GetEventUserData() later.

        @see @ref overview_cpp_rtti_disabled

        @since 2.9.0
    */
    template <typename EventTag, typename Functor>
    void Bind(const EventTag& eventType,
              Functor functor,
              int id = wxID_ANY,
              int lastId = wxID_ANY,
              wxObject *userData = NULL);

    /**
        See the Bind<>(const EventTag&, Functor, int, int, wxObject*) overload for
        more info.

        This overload will bind the given method as the event handler.

        @param eventType
            The event type to be associated with this event handler.
        @param method
            The event handler method. This can be an arbitrary method (doesn't need
            to be from a wxEvtHandler derived class).
        @param handler
            Object whose method should be called. It must always be specified
            so it can be checked at compile time whether the given method is an
            actual member of the given handler.
        @param id
            The first ID of the identifier range to be associated with the event
            handler.
        @param lastId
            The last ID of the identifier range to be associated with the event
            handler.
        @param userData
            Optional data to be associated with the event table entry.
            wxWidgets will take ownership of this pointer, i.e. it will be
            destroyed when the event handler is disconnected or at the program
            termination. This pointer can be retrieved using
            wxEvent::GetEventUserData() later.

        @see @ref overview_cpp_rtti_disabled

        @since 2.9.0
    */
    template <typename EventTag, typename Class, typename EventArg, typename EventHandler>
    void Bind(const EventTag &eventType,
              void (Class::*method)(EventArg &),
              EventHandler *handler,
              int id = wxID_ANY,
              int lastId = wxID_ANY,
              wxObject *userData = NULL);
    /**
        Unbinds the given function, functor or method dynamically from the
        event handler, using the specified parameters as search criteria and
        returning @true if a matching function has been found and removed.

        This method can only unbind functions, functors or methods which have
        been added using the Bind<>() method. There is no way to unbind
        functions bound using the (static) event tables.

        @note Currently functors are compared by their address which,
        unfortunately, doesn't work correctly if the same address is reused for
        two different functor objects. Because of this, using Unbind() is not
        recommended if there are multiple functors using the same @a eventType
        and @a id and @a lastId as a wrong one could be unbound.

        @param eventType
            The event type associated with this event handler.
        @param functor
            The event handler functor. This can be an ordinary function but also
            an arbitrary functor like boost::function<>.
        @param id
            The first ID of the identifier range associated with the event
            handler.
        @param lastId
            The last ID of the identifier range associated with the event
            handler.
        @param userData
            Data associated with the event table entry.

        @see @ref overview_cpp_rtti_disabled

        @since 2.9.0
    */
    template <typename EventTag, typename Functor>
    bool Unbind(const EventTag& eventType,
                Functor functor,
                int id = wxID_ANY,
                int lastId = wxID_ANY,
                wxObject *userData = NULL);

    /**
        See the Unbind<>(const EventTag&, Functor, int, int, wxObject*)
        overload for more info.

        This overload unbinds the given method from the event..

        @param eventType
            The event type associated with this event handler.
        @param method
            The event handler method associated with this event.
        @param handler
            Object whose method was called.
        @param id
            The first ID of the identifier range associated with the event
            handler.
        @param lastId
            The last ID of the identifier range associated with the event
            handler.
        @param userData
            Data associated with the event table entry.

        @see @ref overview_cpp_rtti_disabled

        @since 2.9.0
    */
    template <typename EventTag, typename Class, typename EventArg, typename EventHandler>
    bool Unbind(const EventTag &eventType,
                void (Class::*method)(EventArg&),
                EventHandler *handler,
                int id = wxID_ANY,
                int lastId = wxID_ANY,
                wxObject *userData = NULL );
    //@}
    /**
        @name User-supplied data
    */
    //@{

    /**
        Returns user-supplied client data.

        @remarks Normally, any extra data the programmer wishes to associate with
                 the object should be made available by deriving a new class with
                 new data members.

        @see SetClientData()
    */
    void* GetClientData() const;

    /**
        Returns a pointer to the user-supplied client data object.

        @see SetClientObject(), wxClientData
    */
    wxClientData* GetClientObject() const;

    /**
        Sets user-supplied client data.

        @param data
            Data to be associated with the event handler.

        @remarks Normally, any extra data the programmer wishes to associate
                 with the object should be made available by deriving a new
                 class with new data members. You must not call this method
                 and SetClientObject on the same class - only one of them.

        @see GetClientData()
    */
    void SetClientData(void* data);

    /**
        Set the client data object. Any previous object will be deleted.

        @see GetClientObject(), wxClientData
    */
    void SetClientObject(wxClientData* data);

    //@}


    /**
        @name Event handler chaining

        wxEvtHandler can be arranged in a double-linked list of handlers
        which is automatically iterated by ProcessEvent() if needed.
    */
    //@{

    /**
        Returns @true if the event handler is enabled, @false otherwise.

        @see SetEvtHandlerEnabled()
    */
    bool GetEvtHandlerEnabled() const;

    /**
        Returns the pointer to the next handler in the chain.

        @see SetNextHandler(), GetPreviousHandler(), SetPreviousHandler(),
             wxWindow::PushEventHandler, wxWindow::PopEventHandler
    */
    wxEvtHandler* GetNextHandler() const;

    /**
        Returns the pointer to the previous handler in the chain.

        @see SetPreviousHandler(), GetNextHandler(), SetNextHandler(),
             wxWindow::PushEventHandler, wxWindow::PopEventHandler
    */
    wxEvtHandler* GetPreviousHandler() const;

    /**
        Enables or disables the event handler.

        @param enabled
            @true if the event handler is to be enabled, @false if it is to be disabled.

        @remarks You can use this function to avoid having to remove the event
                 handler from the chain, for example when implementing a
                 dialog editor and changing from edit to test mode.

        @see GetEvtHandlerEnabled()
    */
    void SetEvtHandlerEnabled(bool enabled);

    /**
        Sets the pointer to the next handler.

        @remarks
        See ProcessEvent() for more info about how the chains of event handlers
        are internally used.
        Also remember that wxEvtHandler uses double-linked lists and thus if you
        use this function, you should also call SetPreviousHandler() on the
        argument passed to this function:
        @code
            handlerA->SetNextHandler(handlerB);
            handlerB->SetPreviousHandler(handlerA);
        @endcode

        @param handler
            The event handler to be set as the next handler.
            Cannot be @NULL.

        @see @ref overview_events_processing
    */
    virtual void SetNextHandler(wxEvtHandler* handler);

    /**
        Sets the pointer to the previous handler.
        All remarks about SetNextHandler() apply to this function as well.

        @param handler
            The event handler to be set as the previous handler.
            Cannot be @NULL.

        @see @ref overview_events_processing
    */
    virtual void SetPreviousHandler(wxEvtHandler* handler);

    /**
        Unlinks this event handler from the chain it's part of (if any);
        then links the "previous" event handler to the "next" one
        (so that the chain won't be interrupted).

        E.g. if before calling Unlink() you have the following chain:
            @image html evthandler_unlink_before.png
        then after calling @c B->Unlink() you'll have:
            @image html evthandler_unlink_after.png

        @since 2.9.0
    */
    void Unlink();

    /**
        Returns @true if the next and the previous handler pointers of this
        event handler instance are @NULL.

        @since 2.9.0

        @see SetPreviousHandler(), SetNextHandler()
    */
    bool IsUnlinked() const;

    //@}

    /**
        @name Global event filters.

        Methods for working with the global list of event filters.

        Event filters can be defined to pre-process all the events that happen
        in an application, see wxEventFilter documentation for more information.
     */
    //@{

    /**
        Add an event filter whose FilterEvent() method will be called for each
        and every event processed by wxWidgets.

        The filters are called in LIFO order and wxApp is registered as an
        event filter by default. The pointer must remain valid until it's
        removed with RemoveFilter() and is not deleted by wxEvtHandler.

        @since 2.9.3
     */
    static void AddFilter(wxEventFilter* filter);

    /**
        Remove a filter previously installed with AddFilter().

        It's an error to remove a filter that hadn't been previously added or
        was already removed.

        @since 2.9.3
     */
    static void RemoveFilter(wxEventFilter* filter);

    //@}

protected:
    /**
        Method called by ProcessEvent() before examining this object event
        tables.

        This method can be overridden to hook into the event processing logic
        as early as possible. You should usually call the base class version
        when overriding this method, even if wxEvtHandler itself does nothing
        here, some derived classes do use this method, e.g. wxWindow implements
        support for wxValidator in it.

        Example:
        @code
        class MyClass : public BaseClass // inheriting from wxEvtHandler
        {
        ...
        protected:
            virtual bool TryBefore(wxEvent& event)
            {
                if ( MyPreProcess(event) )
                    return true;

                return BaseClass::TryBefore(event);
            }
        };
        @endcode

        @see ProcessEvent()
     */
    virtual bool TryBefore(wxEvent& event);

    /**
        Method called by ProcessEvent() as last resort.

        This method can be overridden to implement post-processing for the
        events which were not processed anywhere else.

        The base class version handles forwarding the unprocessed events to
        wxApp at wxEvtHandler level and propagating them upwards the window
        child-parent chain at wxWindow level and so should usually be called
        when overriding this method:
        @code
        class MyClass : public BaseClass // inheriting from wxEvtHandler
        {
        ...
        protected:
            virtual bool TryAfter(wxEvent& event)
            {
                if ( BaseClass::TryAfter(event) )
                    return true;

                return MyPostProcess(event);
            }
        };
        @endcode

        @see ProcessEvent()
     */
    virtual bool TryAfter(wxEvent& event);
};

#endif // wxUSE_BASE

#if wxUSE_GUI

/**
    Flags for categories of keys.

    These values are used by wxKeyEvent::IsKeyInCategory(). They may be
    combined via the bitwise operators |, &, and ~.

    @since 2.9.1
*/
enum wxKeyCategoryFlags
{
    /// arrow keys, on and off numeric keypads
    WXK_CATEGORY_ARROW = 1,

    /// page up and page down keys, on and off numeric keypads
    WXK_CATEGORY_PAGING = 2,

    /// home and end keys, on and off numeric keypads
    WXK_CATEGORY_JUMP = 4,

    /// tab key, on and off numeric keypads
    WXK_CATEGORY_TAB = 8,

    /// backspace and delete keys, on and off numeric keypads
    WXK_CATEGORY_CUT = 16,

    /// union of WXK_CATEGORY_ARROW, WXK_CATEGORY_PAGING, and WXK_CATEGORY_JUMP categories
    WXK_CATEGORY_NAVIGATION = WXK_CATEGORY_ARROW | WXK_CATEGORY_PAGING | WXK_CATEGORY_JUMP
};


/**
    @class wxKeyEvent

    This event class contains information about key press and release events.

    The main information carried by this event is the key being pressed or
    released. It can be accessed using either GetKeyCode() function or
    GetUnicodeKey(). For the printable characters, the latter should be used as
    it works for any keys, including non-Latin-1 characters that can be entered
    when using national keyboard layouts. GetKeyCode() should be used to handle
    special characters (such as cursor arrows keys or @c HOME or @c INS and so
    on) which correspond to ::wxKeyCode enum elements above the @c WXK_START
    constant. While GetKeyCode() also returns the character code for Latin-1
    keys for compatibility, it doesn't work for Unicode characters in general
    and will return @c WXK_NONE for any non-Latin-1 ones. For this reason, it's
    recommended to always use GetUnicodeKey() and only fall back to GetKeyCode()
    if GetUnicodeKey() returned @c WXK_NONE meaning that the event corresponds
    to a non-printable special keys.

    While both of these functions can be used with the events of @c
    wxEVT_KEY_DOWN, @c wxEVT_KEY_UP and @c wxEVT_CHAR types, the values
    returned by them are different for the first two events and the last one.
    For the latter, the key returned corresponds to the character that would
    appear in e.g. a text zone if the user pressed the key in it. As such, its
    value depends on the current state of the Shift key and, for the letters,
    on the state of Caps Lock modifier. For example, if @c A key is pressed
    without Shift being held down, wxKeyEvent of type @c wxEVT_CHAR generated
    for this key press will return (from either GetKeyCode() or GetUnicodeKey()
    as their meanings coincide for ASCII characters) key code of 97
    corresponding the ASCII value of @c a. And if the same key is pressed but
    with Shift being held (or Caps Lock being active), then the key could would
    be 65, i.e. ASCII value of capital @c A.

    However for the key down and up events the returned key code will instead
    be @c A independently of the state of the modifier keys i.e. it depends
    only on physical key being pressed and is not translated to its logical
    representation using the current keyboard state. Such untranslated key
    codes are defined as follows:
        - For the letters they correspond to the @e upper case value of the
        letter.
        - For the other alphanumeric keys (e.g. @c 7 or @c +), the untranslated
        key code corresponds to the character produced by the key when it is
        pressed without Shift. E.g. in standard US keyboard layout the
        untranslated key code for the key @c =/+ in the upper right corner of
        the keyboard is 61 which is the ASCII value of @c =.
        - For the rest of the keys (i.e. special non-printable keys) it is the
        same as the normal key code as no translation is used anyhow.

    Notice that the first rule applies to all Unicode letters, not just the
    usual Latin-1 ones. However for non-Latin-1 letters only GetUnicodeKey()
    can be used to retrieve the key code as GetKeyCode() just returns @c
    WXK_NONE in this case.

    To summarize: you should handle @c wxEVT_CHAR if you need the translated
    key and @c wxEVT_KEY_DOWN if you only need the value of the key itself,
    independent of the current keyboard state.

    @note Not all key down events may be generated by the user. As an example,
        @c wxEVT_KEY_DOWN with @c = key code can be generated using the
        standard US keyboard layout but not using the German one because the @c
        = key corresponds to Shift-0 key combination in this layout and the key
        code for it is @c 0, not @c =. Because of this you should avoid
        requiring your users to type key events that might be impossible to
        enter on their keyboard.


    Another difference between key and char events is that another kind of
    translation is done for the latter ones when the Control key is pressed:
    char events for ASCII letters in this case carry codes corresponding to the
    ASCII value of Ctrl-Latter, i.e. 1 for Ctrl-A, 2 for Ctrl-B and so on until
    26 for Ctrl-Z. This is convenient for terminal-like applications and can be
    completely ignored by all the other ones (if you need to handle Ctrl-A it
    is probably a better idea to use the key event rather than the char one).
    Notice that currently no translation is done for the presses of @c [, @c
    \\, @c ], @c ^ and @c _ keys which might be mapped to ASCII values from 27
    to 31.
    Since version 2.9.2, the enum values @c WXK_CONTROL_A - @c WXK_CONTROL_Z
    can be used instead of the non-descriptive constant values 1-26.

    Finally, modifier keys only generate key events but no char events at all.
    The modifiers keys are @c WXK_SHIFT, @c WXK_CONTROL, @c WXK_ALT and various
    @c WXK_WINDOWS_XXX from ::wxKeyCode enum.

    Modifier keys events are special in one additional aspect: usually the
    keyboard state associated with a key press is well defined, e.g.
    wxKeyboardState::ShiftDown() returns @c true only if the Shift key was held
    pressed when the key that generated this event itself was pressed. There is
    an ambiguity for the key press events for Shift key itself however. By
    convention, it is considered to be already pressed when it is pressed and
    already released when it is released. In other words, @c wxEVT_KEY_DOWN
    event for the Shift key itself will have @c wxMOD_SHIFT in GetModifiers()
    and ShiftDown() will return true while the @c wxEVT_KEY_UP event for Shift
    itself will not have @c wxMOD_SHIFT in its modifiers and ShiftDown() will
    return false.


    @b Tip: You may discover the key codes and modifiers generated by all the
        keys on your system interactively by running the @ref
        page_samples_keyboard wxWidgets sample and pressing some keys in it.

    @note If a key down (@c EVT_KEY_DOWN) event is caught and the event handler
          does not call @c event.Skip() then the corresponding char event
          (@c EVT_CHAR) will not happen. This is by design and enables the
          programs that handle both types of events to avoid processing the
          same key twice. As a consequence, if you do not want to suppress the
          @c wxEVT_CHAR events for the keys you handle, always call @c
          event.Skip() in your @c wxEVT_KEY_DOWN handler. Not doing may also
          prevent accelerators defined using this key from working.

    @note If a key is maintained in a pressed state, you will typically get a
          lot of (automatically generated) key down events but only one key up
          one at the end when the key is released so it is wrong to assume that
          there is one up event corresponding to each down one.

    @note For Windows programmers: The key and char events in wxWidgets are
          similar to but slightly different from Windows @c WM_KEYDOWN and
          @c WM_CHAR events. In particular, Alt-x combination will generate a
          char event in wxWidgets (unless it is used as an accelerator) and
          almost all keys, including ones without ASCII equivalents, generate
          char events too.


    @beginEventTable{wxKeyEvent}
    @event{EVT_KEY_DOWN(func)}
        Process a @c wxEVT_KEY_DOWN event (any key has been pressed). If this
        event is handled and not skipped, @c wxEVT_CHAR will not be generated
        at all for this key press (but @c wxEVT_KEY_UP will be).
    @event{EVT_KEY_UP(func)}
        Process a @c wxEVT_KEY_UP event (any key has been released).
    @event{EVT_CHAR(func)}
        Process a @c wxEVT_CHAR event.
    @event{EVT_CHAR_HOOK(func)}
        Process a @c wxEVT_CHAR_HOOK event. Unlike all the other key events,
        this event is propagated upwards the window hierarchy which allows
        intercepting it in the parent window of the focused window to which it
        is sent initially (if there is no focused window, this event is sent to
        the wxApp global object). It is also generated before any other key
        events and so gives the parent window an opportunity to modify the
        keyboard handling of its children, e.g. it is used internally by
        wxWidgets in some ports to intercept pressing Esc key in any child of a
        dialog to close the dialog itself when it's pressed. By default, if
        this event is handled, i.e. the handler doesn't call wxEvent::Skip(),
        neither @c wxEVT_KEY_DOWN nor @c wxEVT_CHAR events will be generated
        (although @c wxEVT_KEY_UP still will be), i.e. it replaces the normal
        key events. However by calling the special DoAllowNextEvent() method
        you can handle @c wxEVT_CHAR_HOOK and still allow normal events
        generation. This is something that is rarely useful but can be required
        if you need to prevent a parent @c wxEVT_CHAR_HOOK handler from running
        without suppressing the normal key events. Finally notice that this
        event is not generated when the mouse is captured as it is considered
        that the window which has the capture should receive all the keyboard
        events too without allowing its parent wxTopLevelWindow to interfere
        with their processing.
    @endEventTable

    @see wxKeyboardState

    @library{wxcore}
    @category{events}
*/
class wxKeyEvent : public wxEvent,
                   public wxKeyboardState
{
public:
    /**
        Constructor.
        Currently, the only valid event types are @c wxEVT_CHAR and @c wxEVT_CHAR_HOOK.
    */
    wxKeyEvent(wxEventType keyEventType = wxEVT_NULL);

    /**
        Returns the key code of the key that generated this event.

        ASCII symbols return normal ASCII values, while events from special
        keys such as "left cursor arrow" (@c WXK_LEFT) return values outside of
        the ASCII range. See ::wxKeyCode for a full list of the virtual key
        codes.

        Note that this method returns a meaningful value only for special
        non-alphanumeric keys or if the user entered a Latin-1 character (this
        includes ASCII and the accented letters found in Western European
        languages but not letters of other alphabets such as e.g. Cyrillic).
        Otherwise it simply method returns @c WXK_NONE and GetUnicodeKey()
        should be used to obtain the corresponding Unicode character.

        Using GetUnicodeKey() is in general the right thing to do if you are
        interested in the characters typed by the user, GetKeyCode() should be
        only used for special keys (for which GetUnicodeKey() returns @c
        WXK_NONE). To handle both kinds of keys you might write:
        @code
            void MyHandler::OnChar(wxKeyEvent& event)
            {
                wxChar uc = event.GetUnicodeKey();
                if ( uc != WXK_NONE )
                {
                    // It's a "normal" character. Notice that this includes
                    // control characters in 1..31 range, e.g. WXK_RETURN or
                    // WXK_BACK, so check for them explicitly.
                    if ( uc >= 32 )
                    {
                        wxLogMessage("You pressed '%c'", uc);
                    }
                    else
                    {
                        // It's a control character
                        ...
                    }
                }
                else // No Unicode equivalent.
                {
                    // It's a special key, deal with all the known ones:
                    switch ( event.GetKeyCode() )
                    {
                        case WXK_LEFT:
                        case WXK_RIGHT:
                            ... move cursor ...
                            break;

                        case WXK_F1:
                            ... give help ...
                            break;
                    }
                }
            }
        @endcode
    */
    int GetKeyCode() const;

    /**
        Returns true if the key is in the given key category.

        @param category
            A bitwise combination of named ::wxKeyCategoryFlags constants.

        @since 2.9.1
    */
    bool IsKeyInCategory(int category) const;

    //@{
    /**
        Obtains the position (in client coordinates) at which the key was pressed.

        Notice that under most platforms this position is simply the current
        mouse pointer position and has no special relationship to the key event
        itself.

        @a x and @a y may be @NULL if the corresponding coordinate is not
        needed.
    */
    wxPoint GetPosition() const;
    void GetPosition(wxCoord* x, wxCoord* y) const;
    //@}

    /**
        Returns the raw key code for this event.

        The flags are platform-dependent and should only be used if the
        functionality provided by other wxKeyEvent methods is insufficient.

        Under MSW, the raw key code is the value of @c wParam parameter of the
        corresponding message.

        Under GTK, the raw key code is the @c keyval field of the corresponding
        GDK event.

        Under macOS, the raw key code is the @c keyCode field of the
        corresponding NSEvent.

        @note Currently the raw key codes are not supported by all ports, use
              @ifdef_ wxHAS_RAW_KEY_CODES to determine if this feature is available.
    */
    wxUint32 GetRawKeyCode() const;

    /**
        Returns the low level key flags for this event.

        The flags are platform-dependent and should only be used if the
        functionality provided by other wxKeyEvent methods is insufficient.

        Under MSW, the raw flags are just the value of @c lParam parameter of
        the corresponding message.

        Under GTK, the raw flags contain the @c hardware_keycode field of the
        corresponding GDK event.

        Under macOS, the raw flags contain the modifiers state.

        @note Currently the raw key flags are not supported by all ports, use
              @ifdef_ wxHAS_RAW_KEY_CODES to determine if this feature is available.
    */
    wxUint32 GetRawKeyFlags() const;

    /**
        Returns the Unicode character corresponding to this key event.

        If the key pressed doesn't have any character value (e.g. a cursor key)
        this method will return @c WXK_NONE. In this case you should use
        GetKeyCode() to retrieve the value of the key.

        This function is only available in Unicode build, i.e. when
        @c wxUSE_UNICODE is 1.
    */
    wxChar GetUnicodeKey() const;

    /**
        Returns the X position (in client coordinates) of the event.

        @see GetPosition()
    */
    wxCoord GetX() const;

    /**
        Returns the Y position (in client coordinates) of the event.

        @see GetPosition()
    */
    wxCoord GetY() const;

    /**
        Allow normal key events generation.

        Can be called from @c wxEVT_CHAR_HOOK handler to indicate that the
        generation of normal events should @em not be suppressed, as it happens
        by default when this event is handled.

        The intended use of this method is to allow some window object to
        prevent @c wxEVT_CHAR_HOOK handler in its parent window from running by
        defining its own handler for this event. Without calling this method,
        this would result in not generating @c wxEVT_KEY_DOWN nor @c wxEVT_CHAR
        events at all but by calling it you can ensure that these events would
        still be generated, even if @c wxEVT_CHAR_HOOK event was handled.

        @since 2.9.3
     */
    void DoAllowNextEvent();

    /**
        Returns @true if DoAllowNextEvent() had been called, @false by default.

        This method is used by wxWidgets itself to determine whether the normal
        key events should be generated after @c wxEVT_CHAR_HOOK processing.

        @since 2.9.3
     */
    bool IsNextEventAllowed() const;
};



enum
{
    wxJOYSTICK1,
    wxJOYSTICK2
};

// Which button is down?
enum
{
    wxJOY_BUTTON_ANY = -1,
    wxJOY_BUTTON1    = 1,
    wxJOY_BUTTON2    = 2,
    wxJOY_BUTTON3    = 4,
    wxJOY_BUTTON4    = 8
};


/**
    @class wxJoystickEvent

    This event class contains information about joystick events, particularly
    events received by windows.

    @beginEventTable{wxJoystickEvent}
    @event{EVT_JOY_BUTTON_DOWN(func)}
        Process a @c wxEVT_JOY_BUTTON_DOWN event.
    @event{EVT_JOY_BUTTON_UP(func)}
        Process a @c wxEVT_JOY_BUTTON_UP event.
    @event{EVT_JOY_MOVE(func)}
        Process a @c wxEVT_JOY_MOVE event.
    @event{EVT_JOY_ZMOVE(func)}
        Process a @c wxEVT_JOY_ZMOVE event.
    @event{EVT_JOYSTICK_EVENTS(func)}
        Processes all joystick events.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see wxJoystick
*/
class wxJoystickEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxJoystickEvent(wxEventType eventType = wxEVT_NULL, int state = 0,
                    int joystick = wxJOYSTICK1,
                    int change = 0);

    /**
        Returns @true if the event was a down event from the specified button
        (or any button).

        @param but
            Can be @c wxJOY_BUTTONn where @c n is 1, 2, 3 or 4; or @c wxJOY_BUTTON_ANY to
            indicate any button down event.
    */
    bool ButtonDown(int but = wxJOY_BUTTON_ANY) const;

    /**
        Returns @true if the specified button (or any button) was in a down state.

        @param but
            Can be @c wxJOY_BUTTONn where @c n is 1, 2, 3 or 4; or @c wxJOY_BUTTON_ANY to
            indicate any button down event.
    */
    bool ButtonIsDown(int but = wxJOY_BUTTON_ANY) const;

    /**
        Returns @true if the event was an up event from the specified button
        (or any button).

        @param button
            Can be @c wxJOY_BUTTONn where @c n is 1, 2, 3 or 4; or @c wxJOY_BUTTON_ANY to
            indicate any button down event.
    */
    bool ButtonUp(int but = wxJOY_BUTTON_ANY) const;

    /**
        Returns the identifier of the button changing state.

        The return value is @code 1 << n @endcode where @c n is the index of the
        button changing state, which can also be retrieved using GetButtonOrdinal().

        Note that for @c n equal to 1, 2, 3 or 4 there are predefined @c wxJOY_BUTTONn
        constants which can be used for more clarity, however these constants are not
        defined for the buttons beyond the first four.
    */
    int GetButtonChange() const;

    /**
        Returns the 0-indexed ordinal of the button changing state.

        @see GetButtonChange()

        @since 3.1.2.
    */
    int GetButtonOrdinal() const;

    /**
        Returns the down state of the buttons.

        This is a @c wxJOY_BUTTONn identifier, where @c n is one of 1, 2, 3, 4.
    */
    int GetButtonState() const;

    /**
        Returns the identifier of the joystick generating the event - one of
        wxJOYSTICK1 and wxJOYSTICK2.
    */
    int GetJoystick() const;

    /**
        Returns the x, y position of the joystick event.

        These coordinates are valid for all the events except wxEVT_JOY_ZMOVE.
    */
    wxPoint GetPosition() const;

    /**
        Returns the z position of the joystick event.

        This method can only be used for wxEVT_JOY_ZMOVE events.
    */
    int GetZPosition() const;

    /**
        Returns @true if this was a button up or down event
        (@e not 'is any button down?').
    */
    bool IsButton() const;

    /**
        Returns @true if this was an x, y move event.
    */
    bool IsMove() const;

    /**
        Returns @true if this was a z move event.
    */
    bool IsZMove() const;
};



/**
    @class wxScrollWinEvent

    A scroll event holds information about events sent from scrolling windows.

    Note that you can use the EVT_SCROLLWIN* macros for intercepting scroll window events
    from the receiving window.

    @beginEventTable{wxScrollWinEvent}
    @event{EVT_SCROLLWIN(func)}
        Process all scroll events.
    @event{EVT_SCROLLWIN_TOP(func)}
        Process @c wxEVT_SCROLLWIN_TOP scroll-to-top events.
    @event{EVT_SCROLLWIN_BOTTOM(func)}
        Process @c wxEVT_SCROLLWIN_BOTTOM scroll-to-bottom events.
    @event{EVT_SCROLLWIN_LINEUP(func)}
        Process @c wxEVT_SCROLLWIN_LINEUP line up events.
    @event{EVT_SCROLLWIN_LINEDOWN(func)}
        Process @c wxEVT_SCROLLWIN_LINEDOWN line down events.
    @event{EVT_SCROLLWIN_PAGEUP(func)}
        Process @c wxEVT_SCROLLWIN_PAGEUP page up events.
    @event{EVT_SCROLLWIN_PAGEDOWN(func)}
        Process @c wxEVT_SCROLLWIN_PAGEDOWN page down events.
    @event{EVT_SCROLLWIN_THUMBTRACK(func)}
        Process @c wxEVT_SCROLLWIN_THUMBTRACK thumbtrack events
        (frequent events sent as the user drags the thumbtrack).
    @event{EVT_SCROLLWIN_THUMBRELEASE(func)}
        Process @c wxEVT_SCROLLWIN_THUMBRELEASE thumb release events.
    @endEventTable


    @library{wxcore}
    @category{events}

    @see wxScrollEvent, @ref overview_events
*/
class wxScrollWinEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxScrollWinEvent(wxEventType commandType = wxEVT_NULL, int pos = 0,
                     int orientation = 0);

    /**
        Returns wxHORIZONTAL or wxVERTICAL, depending on the orientation of the
        scrollbar.

        @todo wxHORIZONTAL and wxVERTICAL should go in their own enum
    */
    int GetOrientation() const;

    /**
        Returns the position of the scrollbar for the thumb track and release events.

        Note that this field can't be used for the other events, you need to query
        the window itself for the current position in that case.
    */
    int GetPosition() const;

    void SetOrientation(int orient);
    void SetPosition(int pos);
};



/**
    @class wxSysColourChangedEvent

    This class is used for system colour change events, which are generated
    when the user changes the colour settings using the control panel.
    This is only appropriate under Windows.

    @remarks
        The default event handler for this event propagates the event to child windows,
        since Windows only sends the events to top-level windows.
        If intercepting this event for a top-level window, remember to call the base
        class handler, or to pass the event on to the window's children explicitly.

    @beginEventTable{wxSysColourChangedEvent}
    @event{EVT_SYS_COLOUR_CHANGED(func)}
        Process a @c wxEVT_SYS_COLOUR_CHANGED event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see @ref overview_events
*/
class wxSysColourChangedEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxSysColourChangedEvent();
};



/**
    @class wxCommandEvent

    This event class contains information about command events, which originate
    from a variety of simple controls.

    Note that wxCommandEvents and wxCommandEvent-derived event classes by default
    and unlike other wxEvent-derived classes propagate upward from the source
    window (the window which emits the event) up to the first parent which processes
    the event. Be sure to read @ref overview_events_propagation.

    More complex controls, such as wxTreeCtrl, have separate command event classes.

    @beginEventTable{wxCommandEvent}
    @event{EVT_COMMAND(id, event, func)}
        Process a command, supplying the window identifier, command event identifier,
        and member function.
    @event{EVT_COMMAND_RANGE(id1, id2, event, func)}
        Process a command for a range of window identifiers, supplying the minimum and
        maximum window identifiers, command event identifier, and member function.
    @event{EVT_BUTTON(id, func)}
        Process a @c wxEVT_BUTTON command, which is generated by a wxButton control.
    @event{EVT_CHECKBOX(id, func)}
        Process a @c wxEVT_CHECKBOX command, which is generated by a wxCheckBox control.
    @event{EVT_CHOICE(id, func)}
        Process a @c wxEVT_CHOICE command, which is generated by a wxChoice control.
    @event{EVT_COMBOBOX(id, func)}
        Process a @c wxEVT_COMBOBOX command, which is generated by a wxComboBox control.
    @event{EVT_LISTBOX(id, func)}
        Process a @c wxEVT_LISTBOX command, which is generated by a wxListBox control.
    @event{EVT_LISTBOX_DCLICK(id, func)}
        Process a @c wxEVT_LISTBOX_DCLICK command, which is generated by a wxListBox control.
    @event{EVT_CHECKLISTBOX(id, func)}
        Process a @c wxEVT_CHECKLISTBOX command, which is generated by a wxCheckListBox control.
    @event{EVT_MENU(id, func)}
        Process a @c wxEVT_MENU command, which is generated by a menu item.
    @event{EVT_MENU_RANGE(id1, id2, func)}
        Process a @c wxEVT_MENU command, which is generated by a range of menu items.
    @event{EVT_CONTEXT_MENU(func)}
        Process the event generated when the user has requested a popup menu to appear by
        pressing a special keyboard key (under Windows) or by right clicking the mouse.
    @event{EVT_RADIOBOX(id, func)}
        Process a @c wxEVT_RADIOBOX command, which is generated by a wxRadioBox control.
    @event{EVT_RADIOBUTTON(id, func)}
        Process a @c wxEVT_RADIOBUTTON command, which is generated by a wxRadioButton control.
    @event{EVT_SCROLLBAR(id, func)}
        Process a @c wxEVT_SCROLLBAR command, which is generated by a wxScrollBar
        control. This is provided for compatibility only; more specific scrollbar event macros
        should be used instead (see wxScrollEvent).
    @event{EVT_SLIDER(id, func)}
        Process a @c wxEVT_SLIDER command, which is generated by a wxSlider control.
    @event{EVT_TEXT(id, func)}
        Process a @c wxEVT_TEXT command, which is generated by a wxTextCtrl control.
    @event{EVT_TEXT_ENTER(id, func)}
        Process a @c wxEVT_TEXT_ENTER command, which is generated by a wxTextCtrl control.
        Note that you must use wxTE_PROCESS_ENTER flag when creating the control if you want it
        to generate such events.
    @event{EVT_TEXT_MAXLEN(id, func)}
        Process a @c wxEVT_TEXT_MAXLEN command, which is generated by a wxTextCtrl control
        when the user tries to enter more characters into it than the limit previously set
        with SetMaxLength().
    @event{EVT_TOGGLEBUTTON(id, func)}
        Process a @c wxEVT_TOGGLEBUTTON event.
    @event{EVT_TOOL(id, func)}
        Process a @c wxEVT_TOOL event (a synonym for @c wxEVT_MENU).
        Pass the id of the tool.
    @event{EVT_TOOL_RANGE(id1, id2, func)}
        Process a @c wxEVT_TOOL event for a range of identifiers. Pass the ids of the tools.
    @event{EVT_TOOL_RCLICKED(id, func)}
        Process a @c wxEVT_TOOL_RCLICKED event. Pass the id of the tool.  (Not available on wxOSX.)
    @event{EVT_TOOL_RCLICKED_RANGE(id1, id2, func)}
        Process a @c wxEVT_TOOL_RCLICKED event for a range of ids. Pass the ids of the tools.  (Not available on wxOSX.)
    @event{EVT_TOOL_ENTER(id, func)}
        Process a @c wxEVT_TOOL_ENTER event. Pass the id of the toolbar itself.
        The value of wxCommandEvent::GetSelection() is the tool id, or -1 if the mouse cursor
        has moved off a tool.  (Not available on wxOSX.)
    @event{EVT_COMMAND_LEFT_CLICK(id, func)}
        Process a @c wxEVT_COMMAND_LEFT_CLICK command, which is generated by a control (wxMSW only).
    @event{EVT_COMMAND_LEFT_DCLICK(id, func)}
        Process a @c wxEVT_COMMAND_LEFT_DCLICK command, which is generated by a control (wxMSW only).
    @event{EVT_COMMAND_RIGHT_CLICK(id, func)}
        Process a @c wxEVT_COMMAND_RIGHT_CLICK command, which is generated by a control (wxMSW only).
    @event{EVT_COMMAND_SET_FOCUS(id, func)}
        Process a @c wxEVT_COMMAND_SET_FOCUS command, which is generated by a control (wxMSW only).
    @event{EVT_COMMAND_KILL_FOCUS(id, func)}
        Process a @c wxEVT_COMMAND_KILL_FOCUS command, which is generated by a control (wxMSW only).
    @event{EVT_COMMAND_ENTER(id, func)}
        Process a @c wxEVT_COMMAND_ENTER command, which is generated by a control.
    @endEventTable

    @library{wxcore}
    @category{events}
*/
class wxCommandEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxCommandEvent(wxEventType commandEventType = wxEVT_NULL, int id = 0);

    /**
        Returns client data pointer for a listbox or choice selection event
        (not valid for a deselection).
    */
    void* GetClientData() const;

    /**
        Returns client object pointer for a listbox or choice selection event
        (not valid for a deselection).
    */
    wxClientData* GetClientObject() const;

    /**
        Returns extra information dependent on the event objects type.

        If the event comes from a listbox selection, it is a boolean
        determining whether the event was a selection (@true) or a
        deselection (@false). A listbox deselection only occurs for
        multiple-selection boxes, and in this case the index and string values
        are indeterminate and the listbox must be examined by the application.
    */
    long GetExtraLong() const;

    /**
        Returns the integer identifier corresponding to a listbox, choice or
        radiobox selection (only if the event was a selection, not a deselection),
        or a boolean value representing the value of a checkbox.

        For a menu item, this method returns -1 if the item is not checkable or
        a boolean value (true or false) for checkable items indicating the new
        state of the item.
    */
    int GetInt() const;

    /**
        Returns item index for a listbox or choice selection event (not valid for
        a deselection).
    */
    int GetSelection() const;

    /**
        Returns item string for a listbox or choice selection event. If one
        or several items have been deselected, returns the index of the first
        deselected item. If some items have been selected and others deselected
        at the same time, it will return the index of the first selected item.
    */
    wxString GetString() const;

    /**
        This method can be used with checkbox and menu events: for the checkboxes, the
        method returns @true for a selection event and @false for a deselection one.
        For the menu events, this method indicates if the menu item just has become
        checked or unchecked (and thus only makes sense for checkable menu items).

        Notice that this method cannot be used with wxCheckListBox currently.
    */
    bool IsChecked() const;

    /**
        For a listbox or similar event, returns @true if it is a selection, @false
        if it is a deselection. If some items have been selected and others deselected
        at the same time, it will return @true.
    */
    bool IsSelection() const;

    /**
        Sets the client data for this event.
    */
    void SetClientData(void* clientData);

    /**
        Sets the client object for this event. The client object is not owned by the
        event object and the event object will not delete the client object in its destructor.

        The client object must be owned and deleted by another object (e.g. a control)
        that has longer life time than the event object.
    */
    void SetClientObject(wxClientData* clientObject);

    /**
        Sets the @b m_extraLong member.
    */
    void SetExtraLong(long extraLong);

    /**
        Sets the @b m_commandInt member.
    */
    void SetInt(int intCommand);

    /**
        Sets the @b m_commandString member.
    */
    void SetString(const wxString& string);
};



/**
    @class wxWindowCreateEvent

    This event is sent just after the actual window associated with a wxWindow
    object has been created.

    Since it is derived from wxCommandEvent, the event propagates up
    the window hierarchy.

    @beginEventTable{wxWindowCreateEvent}
    @event{EVT_WINDOW_CREATE(func)}
        Process a @c wxEVT_CREATE event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see @ref overview_events, wxWindowDestroyEvent
*/
class wxWindowCreateEvent : public wxCommandEvent
{
public:
    /**
        Constructor.
    */
    wxWindowCreateEvent(wxWindow* win = NULL);

    /// Return the window being created.
    wxWindow *GetWindow() const;
};



/**
    @class wxPaintEvent

    A paint event is sent when a window's contents needs to be repainted.

    The handler of this event must create a wxPaintDC object and use it for
    painting the window contents. For example:
    @code
    void MyWindow::OnPaint(wxPaintEvent& event)
    {
        wxPaintDC dc(this);

        DrawMyDocument(dc);
    }
    @endcode

    Notice that you must @e not create other kinds of wxDC (e.g. wxClientDC or
    wxWindowDC) in EVT_PAINT handlers and also don't create wxPaintDC outside
    of this event handlers.


    You can optimize painting by retrieving the rectangles that have been damaged
    and only repainting these. The rectangles are in terms of the client area,
    and are unscrolled, so you will need to do some calculations using the current
    view position to obtain logical, scrolled units.
    Here is an example of using the wxRegionIterator class:
    @code
    // Called when window needs to be repainted.
    void MyWindow::OnPaint(wxPaintEvent& event)
    {
        wxPaintDC dc(this);

        // Find Out where the window is scrolled to
        int vbX,vbY;                     // Top left corner of client
        GetViewStart(&vbX,&vbY);

        int vX,vY,vW,vH;                 // Dimensions of client area in pixels
        wxRegionIterator upd(GetUpdateRegion()); // get the update rect list

        while (upd)
        {
            vX = upd.GetX();
            vY = upd.GetY();
            vW = upd.GetW();
            vH = upd.GetH();

            // Alternatively we can do this:
            // wxRect rect(upd.GetRect());

            // Repaint this rectangle
            ...some code...

            upd ++ ;
        }
    }
    @endcode

    @remarks
    Please notice that in general it is impossible to change the drawing of a
    standard control (such as wxButton) and so you shouldn't attempt to handle
    paint events for them as even if it might work on some platforms, this is
    inherently not portable and won't work everywhere.


    @beginEventTable{wxPaintEvent}
    @event{EVT_PAINT(func)}
        Process a @c wxEVT_PAINT event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see @ref overview_events
*/
class wxPaintEvent : public wxEvent
{
public:
    /**
        Constructor for exclusive use of wxWidgets itself.

        Note that the objects of this class can @em not be created from
        application code, they're only created by the library itself. If you
        need a window to be repainted, use wxWindow::Refresh() instead of
        trying to manually create an event of this class.
    */
    explicit wxPaintEvent(wxWindow* window);
};



/**
    @class wxMaximizeEvent

    An event being sent when a top level window is maximized. Notice that it is
    not sent when the window is restored to its original size after it had been
    maximized, only a normal wxSizeEvent is generated in this case.

    Currently this event is only generated in wxMSW, wxGTK and wxOSX/Cocoa
    ports so portable programs should only rely on receiving @c wxEVT_SIZE and
    not necessarily this event when the window is maximized.

    @beginEventTable{wxMaximizeEvent}
    @event{EVT_MAXIMIZE(func)}
        Process a @c wxEVT_MAXIMIZE event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see @ref overview_events, wxTopLevelWindow::Maximize,
         wxTopLevelWindow::IsMaximized
*/
class wxMaximizeEvent : public wxEvent
{
public:
    /**
        Constructor. Only used by wxWidgets internally.
    */
    wxMaximizeEvent(int id = 0);
};

/**
    The possibles modes to pass to wxUpdateUIEvent::SetMode().
*/
enum wxUpdateUIMode
{
        /** Send UI update events to all windows. */
    wxUPDATE_UI_PROCESS_ALL,

        /** Send UI update events to windows that have
            the wxWS_EX_PROCESS_UI_UPDATES flag specified. */
    wxUPDATE_UI_PROCESS_SPECIFIED
};


/**
    @class wxUpdateUIEvent

    This class is used for pseudo-events which are called by wxWidgets
    to give an application the chance to update various user interface elements.

    Without update UI events, an application has to work hard to check/uncheck,
    enable/disable, show/hide, and set the text for elements such as menu items
    and toolbar buttons. The code for doing this has to be mixed up with the code
    that is invoked when an action is invoked for a menu item or button.

    With update UI events, you define an event handler to look at the state of the
    application and change UI elements accordingly. wxWidgets will call your member
    functions in idle time, so you don't have to worry where to call this code.

    In addition to being a clearer and more declarative method, it also means you don't
    have to worry whether you're updating a toolbar or menubar identifier. The same
    handler can update a menu item and toolbar button, if the identifier is the same.
    Instead of directly manipulating the menu or button, you call functions in the event
    object, such as wxUpdateUIEvent::Check. wxWidgets will determine whether such a
    call has been made, and which UI element to update.

    These events will work for popup menus as well as menubars. Just before a menu is
    popped up, wxMenu::UpdateUI is called to process any UI events for the window that
    owns the menu.

    If you find that the overhead of UI update processing is affecting your application,
    you can do one or both of the following:
    @li Call wxUpdateUIEvent::SetMode with a value of wxUPDATE_UI_PROCESS_SPECIFIED,
        and set the extra style wxWS_EX_PROCESS_UI_UPDATES for every window that should
        receive update events. No other windows will receive update events.
    @li Call wxUpdateUIEvent::SetUpdateInterval with a millisecond value to set the delay
        between updates. You may need to call wxWindow::UpdateWindowUI at critical points,
        for example when a dialog is about to be shown, in case the user sees a slight
        delay before windows are updated.

    Note that although events are sent in idle time, defining a wxIdleEvent handler
    for a window does not affect this because the events are sent from wxWindow::OnInternalIdle
    which is always called in idle time.

    wxWidgets tries to optimize update events on some platforms.
    On Windows and GTK+, events for menubar items are only sent when the menu is about
    to be shown, and not in idle time.


    @beginEventTable{wxUpdateUIEvent}
    @event{EVT_UPDATE_UI(id, func)}
        Process a @c wxEVT_UPDATE_UI event for the command with the given id.
    @event{EVT_UPDATE_UI_RANGE(id1, id2, func)}
        Process a @c wxEVT_UPDATE_UI event for any command with id included in the given range.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see @ref overview_events
*/
class wxUpdateUIEvent : public wxCommandEvent
{
public:
    /**
        Constructor.
    */
    wxUpdateUIEvent(wxWindowID commandId = 0);

    /**
        Returns @true if it is appropriate to update (send UI update events to)
        this window.

        This function looks at the mode used (see wxUpdateUIEvent::SetMode),
        the wxWS_EX_PROCESS_UI_UPDATES flag in @a window, the time update events
        were last sent in idle time, and the update interval, to determine whether
        events should be sent to this window now. By default this will always
        return @true because the update mode is initially wxUPDATE_UI_PROCESS_ALL
        and the interval is set to 0; so update events will be sent as often as
        possible. You can reduce the frequency that events are sent by changing the
        mode and/or setting an update interval.

        @see ResetUpdateTime(), SetUpdateInterval(), SetMode()
    */
    static bool CanUpdate(wxWindow* window);

    /**
        Check or uncheck the UI element.
    */
    void Check(bool check);

    /**
        Enable or disable the UI element.
    */
    void Enable(bool enable);

    /**
        Returns @true if the UI element should be checked.
    */
    bool GetChecked() const;

    /**
        Returns @true if the UI element should be enabled.
    */
    bool GetEnabled() const;

    /**
        Static function returning a value specifying how wxWidgets will send update
        events: to all windows, or only to those which specify that they will process
        the events.

        @see SetMode()
    */
    static wxUpdateUIMode GetMode();

    /**
        Returns @true if the application has called Check().
        For wxWidgets internal use only.
    */
    bool GetSetChecked() const;

    /**
        Returns @true if the application has called Enable().
        For wxWidgets internal use only.
    */
    bool GetSetEnabled() const;

    /**
        Returns @true if the application has called Show().
        For wxWidgets internal use only.
    */
    bool GetSetShown() const;

    /**
        Returns @true if the application has called SetText().
        For wxWidgets internal use only.
    */
    bool GetSetText() const;

    /**
        Returns @true if the UI element should be shown.
    */
    bool GetShown() const;

    /**
        Returns the text that should be set for the UI element.
    */
    wxString GetText() const;

    /**
        Returns the current interval between updates in milliseconds.
        The value -1 disables updates, 0 updates as frequently as possible.

        @see SetUpdateInterval().
    */
    static long GetUpdateInterval();

    /**
        Used internally to reset the last-updated time to the current time.

        It is assumed that update events are normally sent in idle time, so this
        is called at the end of idle processing.

        @see CanUpdate(), SetUpdateInterval(), SetMode()
    */
    static void ResetUpdateTime();

    /**
        Specify how wxWidgets will send update events: to all windows, or only to
        those which specify that they will process the events.

        @param mode
            this parameter may be one of the ::wxUpdateUIMode enumeration values.
            The default mode is wxUPDATE_UI_PROCESS_ALL.
    */
    static void SetMode(wxUpdateUIMode mode);

    /**
        Sets the text for this UI element.
    */
    void SetText(const wxString& text);

    /**
        Sets the interval between updates in milliseconds.

        Set to -1 to disable updates, or to 0 to update as frequently as possible.
        The default is 0.

        Use this to reduce the overhead of UI update events if your application
        has a lot of windows. If you set the value to -1 or greater than 0,
        you may also need to call wxWindow::UpdateWindowUI at appropriate points
        in your application, such as when a dialog is about to be shown.
    */
    static void SetUpdateInterval(long updateInterval);

    /**
        Show or hide the UI element.
    */
    void Show(bool show);
};



/**
    @class wxClipboardTextEvent

    This class represents the events generated by a control (typically a
    wxTextCtrl but other windows can generate these events as well) when its
    content gets copied or cut to, or pasted from the clipboard.

    There are three types of corresponding events @c wxEVT_TEXT_COPY,
    @c wxEVT_TEXT_CUT and @c wxEVT_TEXT_PASTE.

    If any of these events is processed (without being skipped) by an event
    handler, the corresponding operation doesn't take place which allows
    preventing the text from being copied from or pasted to a control. It is also
    possible to examine the clipboard contents in the PASTE event handler and
    transform it in some way before inserting in a control -- for example,
    changing its case or removing invalid characters.

    Finally notice that a CUT event is always preceded by the COPY event which
    makes it possible to only process the latter if it doesn't matter if the
    text was copied or cut.

    @note
    These events are currently only generated by wxTextCtrl in wxGTK and wxOSX
    but are also generated by wxComboBox without wxCB_READONLY style in wxMSW.

    @beginEventTable{wxClipboardTextEvent}
    @event{EVT_TEXT_COPY(id, func)}
           Some or all of the controls content was copied to the clipboard.
    @event{EVT_TEXT_CUT(id, func)}
           Some or all of the controls content was cut (i.e. copied and
           deleted).
    @event{EVT_TEXT_PASTE(id, func)}
           Clipboard content was pasted into the control.
    @endEventTable


    @library{wxcore}
    @category{events}

    @see wxClipboard
*/
class wxClipboardTextEvent : public wxCommandEvent
{
public:
    /**
        Constructor.
    */
    wxClipboardTextEvent(wxEventType commandType = wxEVT_NULL, int id = 0);
};

/**
    Possible axis values for mouse wheel scroll events.

    @since 2.9.4
 */
enum wxMouseWheelAxis
{
    wxMOUSE_WHEEL_VERTICAL,     ///< Vertical scroll event.
    wxMOUSE_WHEEL_HORIZONTAL    ///< Horizontal scroll event.
};


/**
    @class wxMouseEvent

    This event class contains information about the events generated by the mouse:
    they include mouse buttons press and release events and mouse move events.

    All mouse events involving the buttons use @c wxMOUSE_BTN_LEFT for the
    left mouse button, @c wxMOUSE_BTN_MIDDLE for the middle one and
    @c wxMOUSE_BTN_RIGHT for the right one. And if the system supports more
    buttons, the @c wxMOUSE_BTN_AUX1 and @c wxMOUSE_BTN_AUX2 events
    can also be generated. Note that not all mice have even a middle button so a
    portable application should avoid relying on the events from it (but the right
    button click can be emulated using the left mouse button with the control key
    under Mac platforms with a single button mouse).

    For the @c wxEVT_ENTER_WINDOW and @c wxEVT_LEAVE_WINDOW events
    purposes, the mouse is considered to be inside the window if it is in the
    window client area and not inside one of its children. In other words, the
    parent window receives @c wxEVT_LEAVE_WINDOW event not only when the
    mouse leaves the window entirely but also when it enters one of its children.

    The position associated with a mouse event is expressed in the window
    coordinates of the window which generated the event, you can use
    wxWindow::ClientToScreen() to convert it to screen coordinates and possibly
    call wxWindow::ScreenToClient() next to convert it to window coordinates of
    another window.

    @note Note the difference between methods like wxMouseEvent::LeftDown and
          the inherited wxMouseState::LeftIsDown: the former returns @true when
          the event corresponds to the left mouse button click while the latter
          returns @true if the left mouse button is currently being pressed.
          For example, when the user is dragging the mouse you can use
          wxMouseEvent::LeftIsDown to test whether the left mouse button is
          (still) depressed. Also, by convention, if wxMouseEvent::LeftDown
          returns @true, wxMouseEvent::LeftIsDown will also return @true in
          wxWidgets whatever the underlying GUI behaviour is (which is
          platform-dependent). The same applies, of course, to other mouse
          buttons as well.


    @beginEventTable{wxMouseEvent}
    @event{EVT_LEFT_DOWN(func)}
        Process a @c wxEVT_LEFT_DOWN event. The handler of this event should normally
        call event.Skip() to allow the default processing to take place as otherwise
        the window under mouse wouldn't get the focus.
    @event{EVT_LEFT_UP(func)}
        Process a @c wxEVT_LEFT_UP event.
    @event{EVT_LEFT_DCLICK(func)}
        Process a @c wxEVT_LEFT_DCLICK event.
    @event{EVT_MIDDLE_DOWN(func)}
        Process a @c wxEVT_MIDDLE_DOWN event.
    @event{EVT_MIDDLE_UP(func)}
        Process a @c wxEVT_MIDDLE_UP event.
    @event{EVT_MIDDLE_DCLICK(func)}
        Process a @c wxEVT_MIDDLE_DCLICK event.
    @event{EVT_RIGHT_DOWN(func)}
        Process a @c wxEVT_RIGHT_DOWN event.
    @event{EVT_RIGHT_UP(func)}
        Process a @c wxEVT_RIGHT_UP event.
    @event{EVT_RIGHT_DCLICK(func)}
        Process a @c wxEVT_RIGHT_DCLICK event.
    @event{EVT_MOUSE_AUX1_DOWN(func)}
        Process a @c wxEVT_AUX1_DOWN event.
    @event{EVT_MOUSE_AUX1_UP(func)}
        Process a @c wxEVT_AUX1_UP event.
    @event{EVT_MOUSE_AUX1_DCLICK(func)}
        Process a @c wxEVT_AUX1_DCLICK event.
    @event{EVT_MOUSE_AUX2_DOWN(func)}
        Process a @c wxEVT_AUX2_DOWN event.
    @event{EVT_MOUSE_AUX2_UP(func)}
        Process a @c wxEVT_AUX2_UP event.
    @event{EVT_MOUSE_AUX2_DCLICK(func)}
        Process a @c wxEVT_AUX2_DCLICK event.
    @event{EVT_MOTION(func)}
        Process a @c wxEVT_MOTION event.
    @event{EVT_ENTER_WINDOW(func)}
        Process a @c wxEVT_ENTER_WINDOW event.
    @event{EVT_LEAVE_WINDOW(func)}
        Process a @c wxEVT_LEAVE_WINDOW event.
    @event{EVT_MOUSEWHEEL(func)}
        Process a @c wxEVT_MOUSEWHEEL event.
    @event{EVT_MOUSE_EVENTS(func)}
        Process all mouse events.
    @event{EVT_MAGNIFY(func)}
        Process a @c wxEVT_MAGNIFY event (new since wxWidgets 3.1.0).
    @endEventTable

    @library{wxcore}
    @category{events}

    @see wxKeyEvent
*/
class wxMouseEvent : public wxEvent,
                     public wxMouseState
{
public:
    /**
        Constructor. Valid event types are:

         @li @c wxEVT_ENTER_WINDOW
         @li @c wxEVT_LEAVE_WINDOW
         @li @c wxEVT_LEFT_DOWN
         @li @c wxEVT_LEFT_UP
         @li @c wxEVT_LEFT_DCLICK
         @li @c wxEVT_MIDDLE_DOWN
         @li @c wxEVT_MIDDLE_UP
         @li @c wxEVT_MIDDLE_DCLICK
         @li @c wxEVT_RIGHT_DOWN
         @li @c wxEVT_RIGHT_UP
         @li @c wxEVT_RIGHT_DCLICK
         @li @c wxEVT_AUX1_DOWN
         @li @c wxEVT_AUX1_UP
         @li @c wxEVT_AUX1_DCLICK
         @li @c wxEVT_AUX2_DOWN
         @li @c wxEVT_AUX2_UP
         @li @c wxEVT_AUX2_DCLICK
         @li @c wxEVT_MOTION
         @li @c wxEVT_MOUSEWHEEL
         @li @c wxEVT_MAGNIFY
    */
    wxMouseEvent(wxEventType mouseEventType = wxEVT_NULL);

    /**
        Returns @true if the event was a first extra button double click.
    */
    bool Aux1DClick() const;

    /**
        Returns @true if the first extra button mouse button changed to down.
    */
    bool Aux1Down() const;

    /**
        Returns @true if the first extra button mouse button changed to up.
    */
    bool Aux1Up() const;

    /**
        Returns @true if the event was a second extra button double click.
    */
    bool Aux2DClick() const;

    /**
        Returns @true if the second extra button mouse button changed to down.
    */
    bool Aux2Down() const;

    /**
        Returns @true if the second extra button mouse button changed to up.
    */
    bool Aux2Up() const;

    /**
        Returns @true if the event was generated by the specified button.

        @see wxMouseState::ButtoinIsDown()
    */
    bool Button(wxMouseButton but) const;

    /**
        If the argument is omitted, this returns @true if the event was a mouse
        double click event. Otherwise the argument specifies which double click event
        was generated (see Button() for the possible values).
    */
    bool ButtonDClick(wxMouseButton but = wxMOUSE_BTN_ANY) const;

    /**
        If the argument is omitted, this returns @true if the event was a mouse
        button down event. Otherwise the argument specifies which button-down event
        was generated (see Button() for the possible values).
    */
    bool ButtonDown(wxMouseButton but = wxMOUSE_BTN_ANY) const;

    /**
        If the argument is omitted, this returns @true if the event was a mouse
        button up event. Otherwise the argument specifies which button-up event
        was generated (see Button() for the possible values).
    */
    bool ButtonUp(wxMouseButton but = wxMOUSE_BTN_ANY) const;

    /**
        Returns @true if this was a dragging event (motion while a button is depressed).

        @see Moving()
    */
    bool Dragging() const;

    /**
        Returns @true if the mouse was entering the window.

        @see Leaving()
    */
    bool Entering() const;

    /**
        Returns the mouse button which generated this event or @c wxMOUSE_BTN_NONE
        if no button is involved (for mouse move, enter or leave event, for example).
        Otherwise @c wxMOUSE_BTN_LEFT is returned for the left button down, up and
        double click events, @c wxMOUSE_BTN_MIDDLE and @c wxMOUSE_BTN_RIGHT
        for the same events for the middle and the right buttons respectively.
    */
    int GetButton() const;

    /**
        Returns the number of mouse clicks for this event: 1 for a simple click, 2
        for a double-click, 3 for a triple-click and so on.

        Currently this function is implemented only in wxMac and returns -1 for the
        other platforms (you can still distinguish simple clicks from double-clicks as
        they generate different kinds of events however).

        @since 2.9.0
    */
    int GetClickCount() const;

    /**
        Returns the configured number of lines (or whatever) to be scrolled per
        wheel action.

        Default value under most platforms is three.

        @see GetColumnsPerAction()
    */
    int GetLinesPerAction() const;

    /**
        Returns the configured number of columns (or whatever) to be scrolled per
        wheel action.

        Default value under most platforms is three.

        @see GetLinesPerAction()

        @since 2.9.5
    */
    int GetColumnsPerAction() const;

    /**
        Returns the logical mouse position in pixels (i.e.\ translated according to the
        translation set for the DC, which usually indicates that the window has been
        scrolled).
    */
    wxPoint GetLogicalPosition(const wxDC& dc) const;

    /**
        For magnify (pinch to zoom) events: returns the change in magnification.

        A value of 0 means no change, a positive value means we should enlarge
        (or zoom in), a negative value means we should shrink (or zoom out).

        This method is only valid to call for @c wxEVT_MAGNIFY events which are
        currently only generated under macOS.

        @see Magnify()

        @since 3.1.0
    */
    float GetMagnification() const;

    /**
        Get wheel delta, normally 120.

        This is the threshold for action to be taken, and one such action
        (for example, scrolling one increment) should occur for each delta.
    */
    int GetWheelDelta() const;

    /**
        On Mac, has the user selected "Natural" scrolling in their System
        Preferences? Currently false on all other OS's.

        "Natural" scrolling means that content scrolling happens in the
        opposite direction, and if you are indeed scrolling content then
        you don't need to use this function because macOS has already
        inverted the scroll direction.
        But there can be special situations where you want the mouse wheel
        action to work always in the same direction and in that case you
        will need this function.

        @since 3.1.3
    */
    bool IsWheelInverted() const;

    /**
        Get wheel rotation, positive or negative indicates direction of rotation.

        Current devices all send an event when rotation is at least +/-WheelDelta, but
        finer resolution devices can be created in the future.

        Because of this you shouldn't assume that one event is equal to 1 line, but you
        should be able to either do partial line scrolling or wait until several
        events accumulate before scrolling.
    */
    int GetWheelRotation() const;

    /**
        Gets the axis the wheel operation concerns.

        Usually the mouse wheel is used to scroll vertically so @c
        wxMOUSE_WHEEL_VERTICAL is returned but some mice (and most trackpads)
        also allow to use the wheel to scroll horizontally in which case
        @c wxMOUSE_WHEEL_HORIZONTAL is returned.

        Notice that before wxWidgets 2.9.4 this method returned @c int.
    */
    wxMouseWheelAxis GetWheelAxis() const;

    /**
        Returns @true if the event was a mouse button event (not necessarily a button
        down event - that may be tested using ButtonDown()).
    */
    bool IsButton() const;

    /**
        Returns @true if the system has been setup to do page scrolling with
        the mouse wheel instead of line scrolling.
    */
    bool IsPageScroll() const;

    /**
        Returns @true if the mouse was leaving the window.

        @see Entering().
    */
    bool Leaving() const;

    /**
        Returns @true if the event was a left double click.
    */
    bool LeftDClick() const;

    /**
        Returns @true if the left mouse button changed to down.
    */
    bool LeftDown() const;

    /**
        Returns @true if the left mouse button changed to up.
    */
    bool LeftUp() const;

    /**
        Returns @true if the event is a magnify (i.e.\ pinch to zoom) event.

        Such events are currently generated only under macOS.

        @see GetMagnification()

        @since 3.1.0
    */
    bool Magnify() const;

    /**
        Returns @true if the Meta key was down at the time of the event.
    */
    bool MetaDown() const;

    /**
        Returns @true if the event was a middle double click.
    */
    bool MiddleDClick() const;

    /**
        Returns @true if the middle mouse button changed to down.
    */
    bool MiddleDown() const;

    /**
        Returns @true if the middle mouse button changed to up.
    */
    bool MiddleUp() const;

    /**
        Returns @true if this was a motion event and no mouse buttons were pressed.
        If any mouse button is held pressed, then this method returns @false and
        Dragging() returns @true.
    */
    bool Moving() const;

    /**
        Returns @true if the event was a right double click.
    */
    bool RightDClick() const;

    /**
        Returns @true if the right mouse button changed to down.
    */
    bool RightDown() const;

    /**
        Returns @true if the right mouse button changed to up.
    */
    bool RightUp() const;
};



/**
    @class wxDropFilesEvent

    This class is used for drop files events, that is, when files have been dropped
    onto the window.

    The window must have previously been enabled for dropping by calling
    wxWindow::DragAcceptFiles().

    Important note: this is a separate implementation to the more general drag and drop
    implementation documented in the @ref overview_dnd. It uses the older, Windows
    message-based approach of dropping files.

    @beginEventTable{wxDropFilesEvent}
    @event{EVT_DROP_FILES(func)}
        Process a @c wxEVT_DROP_FILES event.
    @endEventTable

    @remarks Windows only until version 2.8.9, available on all platforms
             since 2.8.10.

    @library{wxcore}
    @category{events}

    @see @ref overview_events, wxWindow::DragAcceptFiles()
*/
class wxDropFilesEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxDropFilesEvent(wxEventType id = 0, int noFiles = 0,
                     wxString* files = NULL);

    /**
        Returns an array of filenames.
    */
    wxString* GetFiles() const;

    /**
        Returns the number of files dropped.
    */
    int GetNumberOfFiles() const;

    /**
        Returns the position at which the files were dropped.
        Returns an array of filenames.
    */
    wxPoint GetPosition() const;
};



/**
    @class wxActivateEvent

    An activate event is sent when a window or application is being activated
    or deactivated.

    @beginEventTable{wxActivateEvent}
    @event{EVT_ACTIVATE(func)}
        Process a @c wxEVT_ACTIVATE event.
    @event{EVT_ACTIVATE_APP(func)}
        Process a @c wxEVT_ACTIVATE_APP event.
        This event is received by the wxApp-derived instance only.
    @event{EVT_HIBERNATE(func)}
        Process a hibernate event, supplying the member function. This event applies
        to wxApp only, and only on Windows SmartPhone and PocketPC.
        It is generated when the system is low on memory; the application should free
        up as much memory as possible, and restore full working state when it receives
        a @c wxEVT_ACTIVATE or @c wxEVT_ACTIVATE_APP event.
    @endEventTable

    @note Until wxWidgets 3.1.0 activation events could be sent by wxMSW when
    the window was minimized. This reflected the native MSW behaviour but was
    often surprising and unexpected, so starting from 3.1.0 such events are not
    sent any more when the window is in the minimized state.

    @library{wxcore}
    @category{events}

    @see @ref overview_events, wxApp::IsActive
*/
class wxActivateEvent : public wxEvent
{
public:
    /**
        Specifies the reason for the generation of this event.

        See GetActivationReason().

        @since 3.0
    */
    enum Reason
    {
        /// Window activated by mouse click.
        Reason_Mouse,
        /// Window was activated with some other method than mouse click.
        Reason_Unknown
    };

    /**
        Constructor.
    */
    wxActivateEvent(wxEventType eventType = wxEVT_NULL, bool active = true,
                    int id = 0, Reason ActivationReason = Reason_Unknown);

    /**
        Returns @true if the application or window is being activated, @false otherwise.
    */
    bool GetActive() const;

    /**
        Allows checking if the window was activated by clicking it with the
        mouse or in some other way.

        This method is currently only implemented in wxMSW and returns @c
        Reason_Mouse there if the window was activated by a mouse click and @c
        Reason_Unknown if it was activated in any other way (e.g. from
        keyboard or programmatically).

        Under all the other platforms, @c Reason_Unknown is always returned.

        @since 3.0
    */
    Reason GetActivationReason() const;
};



/**
    @class wxContextMenuEvent

    This class is used for context menu events, sent to give
    the application a chance to show a context (popup) menu for a wxWindow.

    Note that if wxContextMenuEvent::GetPosition returns wxDefaultPosition, this
    means that the event originated from a keyboard context button event, and you
    should compute a suitable position yourself, for example by calling wxGetMousePosition().

    Notice that the exact sequence of mouse events is different across the
    platforms. For example, under MSW the context menu event is generated after
    @c EVT_RIGHT_UP event and only if it was not handled but under GTK the
    context menu event is generated after @c EVT_RIGHT_DOWN event. This is
    correct in the sense that it ensures that the context menu is shown
    according to the current platform UI conventions and also means that you
    must not handle (or call wxEvent::Skip() in your handler if you do have
    one) neither right mouse down nor right mouse up event if you plan on
    handling @c EVT_CONTEXT_MENU event.

    @beginEventTable{wxContextMenuEvent}
    @event{EVT_CONTEXT_MENU(func)}
        A right click (or other context menu command depending on platform) has been detected.
    @endEventTable


    @library{wxcore}
    @category{events}

    @see wxCommandEvent, @ref overview_events
*/
class wxContextMenuEvent : public wxCommandEvent
{
public:
    /**
        Constructor.
    */
    wxContextMenuEvent(wxEventType type = wxEVT_NULL, int id = 0,
                       const wxPoint& pos = wxDefaultPosition);

    /**
        Returns the position in screen coordinates at which the menu should be shown.
        Use wxWindow::ScreenToClient to convert to client coordinates.

        You can also omit a position from  wxWindow::PopupMenu in order to use
        the current mouse pointer position.

        If the event originated from a keyboard event, the value returned from this
        function will be wxDefaultPosition.
    */
    const wxPoint& GetPosition() const;

    /**
        Sets the position at which the menu should be shown.
    */
    void SetPosition(const wxPoint& point);
};



/**
    @class wxEraseEvent

    An erase event is sent when a window's background needs to be repainted.

    On some platforms, such as GTK+, this event is simulated (simply generated just
    before the paint event) and may cause flicker. It is therefore recommended that
    you set the text background colour explicitly in order to prevent flicker.
    The default background colour under GTK+ is grey.

    To intercept this event, use the EVT_ERASE_BACKGROUND macro in an event table
    definition.

    You must use the device context returned by GetDC() to draw on, don't create
    a wxPaintDC in the event handler.

    @beginEventTable{wxEraseEvent}
    @event{EVT_ERASE_BACKGROUND(func)}
        Process a @c wxEVT_ERASE_BACKGROUND event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see @ref overview_events
*/
class wxEraseEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxEraseEvent(int id = 0, wxDC* dc = NULL);

    /**
        Returns the device context associated with the erase event to draw on.

        The returned pointer is never @NULL.
    */
    wxDC* GetDC() const;
};



/**
    @class wxFocusEvent

    A focus event is sent when a window's focus changes. The window losing focus
    receives a "kill focus" event while the window gaining it gets a "set focus" one.

    Notice that the set focus event happens both when the user gives focus to the
    window (whether using the mouse or keyboard) and when it is done from the
    program itself using wxWindow::SetFocus.

    The focus event handlers should almost invariably call wxEvent::Skip() on
    their event argument to allow the default handling to take place. Failure
    to do this may result in incorrect behaviour of the native controls. Also
    note that wxEVT_KILL_FOCUS handler must not call wxWindow::SetFocus() as
    this, again, is not supported by all native controls. If you need to do
    this, consider using the @ref sec_delayed_action described in wxIdleEvent
    documentation.

    @beginEventTable{wxFocusEvent}
    @event{EVT_SET_FOCUS(func)}
        Process a @c wxEVT_SET_FOCUS event.
    @event{EVT_KILL_FOCUS(func)}
        Process a @c wxEVT_KILL_FOCUS event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see @ref overview_events
*/
class wxFocusEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxFocusEvent(wxEventType eventType = wxEVT_NULL, int id = 0);

    /**
        Returns the window associated with this event, that is the window which had the
        focus before for the @c wxEVT_SET_FOCUS event and the window which is
        going to receive focus for the @c wxEVT_KILL_FOCUS one.

        Warning: the window pointer may be @NULL!
    */
    wxWindow *GetWindow() const;

    void SetWindow(wxWindow *win);
};



/**
    @class wxChildFocusEvent

    A child focus event is sent to a (parent-)window when one of its child windows
    gains focus, so that the window could restore the focus back to its corresponding
    child if it loses it now and regains later.

    Notice that child window is the direct child of the window receiving event.
    Use wxWindow::FindFocus() to retrieve the window which is actually getting focus.

    @beginEventTable{wxChildFocusEvent}
    @event{EVT_CHILD_FOCUS(func)}
        Process a @c wxEVT_CHILD_FOCUS event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see @ref overview_events
*/
class wxChildFocusEvent : public wxCommandEvent
{
public:
    /**
        Constructor.

        @param win
            The direct child which is (or which contains the window which is) receiving
            the focus.
    */
    wxChildFocusEvent(wxWindow* win = NULL);

    /**
        Returns the direct child which receives the focus, or a (grand-)parent of the
        control receiving the focus.

        To get the actually focused control use wxWindow::FindFocus.
    */
    wxWindow *GetWindow() const;
};



/**
    @class wxMouseCaptureLostEvent

    A mouse capture lost event is sent to a window that had obtained mouse capture,
    which was subsequently lost due to an "external" event (for example, when a dialog
    box is shown or if another application captures the mouse).

    If this happens, this event is sent to all windows that are on the capture stack
    (i.e. called CaptureMouse, but didn't call ReleaseMouse yet). The event is
    not sent if the capture changes because of a call to CaptureMouse or
    ReleaseMouse.

    This event is currently emitted under Windows only.

    @beginEventTable{wxMouseCaptureLostEvent}
    @event{EVT_MOUSE_CAPTURE_LOST(func)}
        Process a @c wxEVT_MOUSE_CAPTURE_LOST event.
    @endEventTable

    @onlyfor{wxmsw}

    @library{wxcore}
    @category{events}

    @see wxMouseCaptureChangedEvent, @ref overview_events,
         wxWindow::CaptureMouse, wxWindow::ReleaseMouse, wxWindow::GetCapture
*/
class wxMouseCaptureLostEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxMouseCaptureLostEvent(wxWindowID windowId = 0);
};

/**
    @class wxDisplayChangedEvent

    A display changed event is sent to top-level windows when the display resolution has changed.

    This event is currently emitted under Windows only.

    @beginEventTable{wxDisplayChangedEvent}
    @event{EVT_DISPLAY_CHANGED(func)}
        Process a @c wxEVT_DISPLAY_CHANGED event.
    @endEventTable

    @onlyfor{wxmsw}

    @library{wxcore}
    @category{events}

    @see wxDisplay
*/

class wxDisplayChangedEvent : public wxEvent
{
public:
    wxDisplayChangedEvent();
};



/**
    @class wxDPIChangedEvent

    Event sent when the display scale factor or pixel density (measured in
    dots-per-inch, or DPI) of the monitor a window is on changes.

    The event is sent to each wxTopLevelWindow affected by the change, and all
    its children recursively (post-order traversal). For example, this event is
    sent to the window when it is moved, by the user, from a display using some
    DPI value to another display using a different DPI value. It also sent to
    all program windows on the given display if its DPI changes due to a change
    in the system settings.

    Currently this event is generated by wxMSW port if only and only if the
    MSW application runs under Windows 10 Creators Update (v1703) or later and
    is marked as being "per-monitor DPI aware", i.e. contains a @c dpiAwareness
    tag with the value "PerMonitorV2" in its manifest (see Microsoft
    <a href="https://docs.microsoft.com/en-us/windows/desktop/sbscs/application-manifests">"Application Manifests" documentation</a>
    for more details).

    @note Per-monitor DPI support is an experimental feature that is still in
    development. It might not work correctly for some controls.

    @beginEventTable{wxDPIChangedEvent}
    @event{EVT_DPI_CHANGED(func)}
        Process a @c wxEVT_DPI_CHANGED event.
    @endEventTable

    @since 3.1.3

    @library{wxcore}
    @category{events}

    @see @ref overview_events
*/
class wxDPIChangedEvent : public wxEvent
{
public:
    /**
        Returns the old DPI.
    */
    wxSize GetOldDPI() const;

    /**
        Returns the new DPI.
    */
    wxSize GetNewDPI() const;
};



class wxPaletteChangedEvent : public wxEvent
{
public:
    wxPaletteChangedEvent(wxWindowID winid = 0);

    void SetChangedWindow(wxWindow* win);
    wxWindow* GetChangedWindow() const;
};


class wxQueryNewPaletteEvent : public wxEvent
{
public:
    wxQueryNewPaletteEvent(wxWindowID winid = 0);

    void SetPaletteRealized(bool realized);
    bool GetPaletteRealized();
};




/**
    @class wxNotifyEvent

    This class is not used by the event handlers by itself, but is a base class
    for other event classes (such as wxBookCtrlEvent).

    It (or an object of a derived class) is sent when the controls state is being
    changed and allows the program to wxNotifyEvent::Veto() this change if it wants
    to prevent it from happening.

    @library{wxcore}
    @category{events}

    @see wxBookCtrlEvent
*/
class wxNotifyEvent : public wxCommandEvent
{
public:
    /**
        Constructor (used internally by wxWidgets only).
    */
    wxNotifyEvent(wxEventType eventType = wxEVT_NULL, int id = 0);

    /**
        This is the opposite of Veto(): it explicitly allows the event to be processed.
        For most events it is not necessary to call this method as the events are allowed
        anyhow but some are forbidden by default (this will be mentioned in the corresponding
        event description).
    */
    void Allow();

    /**
        Returns @true if the change is allowed (Veto() hasn't been called) or @false
        otherwise (if it was).
    */
    bool IsAllowed() const;

    /**
        Prevents the change announced by this event from happening.

        It is in general a good idea to notify the user about the reasons for vetoing
        the change because otherwise the applications behaviour (which just refuses to
        do what the user wants) might be quite surprising.
    */
    void Veto();
};


/**
    @class wxThreadEvent

    This class adds some simple functionality to wxEvent to facilitate
    inter-thread communication.

    This event is not natively emitted by any control/class: it is just
    a helper class for the user.
    Its most important feature is the GetEventCategory() implementation which
    allows thread events @b NOT to be processed by wxEventLoopBase::YieldFor calls
    (unless the @c wxEVT_CATEGORY_THREAD is specified - which is never in wx code).

    @library{wxcore}
    @category{events,threading}

    @see @ref overview_thread, wxEventLoopBase::YieldFor

    @since 2.9.0
*/
class wxThreadEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxThreadEvent(wxEventType eventType = wxEVT_THREAD, int id = wxID_ANY);

    /**
        Clones this event making sure that all internal members which use
        COW (only @c m_commandString for now; see @ref overview_refcount)
        are unshared (see wxObject::UnShare).
    */
    virtual wxEvent *Clone() const;

    /**
        Returns @c wxEVT_CATEGORY_THREAD.

        This is important to avoid unwanted processing of thread events
        when calling wxEventLoopBase::YieldFor().
    */
    virtual wxEventCategory GetEventCategory() const;

    /**
        Sets custom data payload.

        The @a payload argument may be of any type that wxAny can handle
        (i.e. pretty much anything). Note that T's copy constructor must be
        thread-safe, i.e. create a copy that doesn't share anything with
        the original (see Clone()).

        @note This method is not available with Visual C++ 6.

        @since 2.9.1

        @see GetPayload(), wxAny
     */
    template<typename T>
    void SetPayload(const T& payload);

    /**
        Get custom data payload.

        Correct type is checked in debug builds.

        @note This method is not available with Visual C++ 6.

        @since 2.9.1

        @see SetPayload(), wxAny
     */
    template<typename T>
    T GetPayload() const;

    /**
        Returns extra information integer value.
    */
    long GetExtraLong() const;

    /**
        Returns stored integer value.
    */
    int GetInt() const;

    /**
        Returns stored string value.
    */
    wxString GetString() const;


    /**
        Sets the extra information value.
    */
    void SetExtraLong(long extraLong);

    /**
        Sets the integer value.
    */
    void SetInt(int intCommand);

    /**
        Sets the string value.
    */
    void SetString(const wxString& string);
};


/**
    @class wxHelpEvent

    A help event is sent when the user has requested context-sensitive help.
    This can either be caused by the application requesting context-sensitive help mode
    via wxContextHelp, or (on MS Windows) by the system generating a WM_HELP message when
    the user pressed F1 or clicked on the query button in a dialog caption.

    A help event is sent to the window that the user clicked on, and is propagated
    up the window hierarchy until the event is processed or there are no more event
    handlers.

    The application should call wxEvent::GetId to check the identity of the
    clicked-on window, and then either show some suitable help or call wxEvent::Skip()
    if the identifier is unrecognised.

    Calling Skip is important because it allows wxWidgets to generate further
    events for ancestors of the clicked-on window. Otherwise it would be impossible to
    show help for container windows, since processing would stop after the first window
    found.

    @beginEventTable{wxHelpEvent}
    @event{EVT_HELP(id, func)}
        Process a @c wxEVT_HELP event.
    @event{EVT_HELP_RANGE(id1, id2, func)}
        Process a @c wxEVT_HELP event for a range of ids.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see wxContextHelp, wxDialog, @ref overview_events
*/
class wxHelpEvent : public wxCommandEvent
{
public:
    /**
        Indicates how a wxHelpEvent was generated.
    */
    enum Origin
    {
        Origin_Unknown,    /**< unrecognized event source. */
        Origin_Keyboard,   /**< event generated from F1 key press. */

        /** event generated by wxContextHelp or from the [?] button on
            the title bar (Windows). */
        Origin_HelpButton
    };

    /**
        Constructor.
    */
    wxHelpEvent(wxEventType type = wxEVT_NULL,
                wxWindowID winid = 0,
                const wxPoint& pt = wxDefaultPosition,
                wxHelpEvent::Origin origin = Origin_Unknown);

    /**
        Returns the origin of the help event which is one of the wxHelpEvent::Origin
        values.

        The application may handle events generated using the keyboard or mouse
        differently, e.g. by using wxGetMousePosition() for the mouse events.

        @see SetOrigin()
    */
    wxHelpEvent::Origin GetOrigin() const;

    /**
        Returns the left-click position of the mouse, in screen coordinates.
        This allows the application to position the help appropriately.
    */
    const wxPoint& GetPosition() const;

    /**
        Set the help event origin, only used internally by wxWidgets normally.

        @see GetOrigin()
    */
    void SetOrigin(wxHelpEvent::Origin origin);

    /**
        Sets the left-click position of the mouse, in screen coordinates.
    */
    void SetPosition(const wxPoint& pt);
};



/**
    @class wxScrollEvent

    A scroll event holds information about events sent from stand-alone
    scrollbars (see wxScrollBar) and sliders (see wxSlider).

    Note that scrolled windows send the wxScrollWinEvent which does not derive from
    wxCommandEvent, but from wxEvent directly - don't confuse these two kinds of
    events and use the event table macros mentioned below only for the scrollbar-like
    controls.

    @section scrollevent_diff The difference between EVT_SCROLL_THUMBRELEASE and EVT_SCROLL_CHANGED

    The EVT_SCROLL_THUMBRELEASE event is only emitted when actually dragging the thumb
    using the mouse and releasing it (This EVT_SCROLL_THUMBRELEASE event is also followed
    by an EVT_SCROLL_CHANGED event).

    The EVT_SCROLL_CHANGED event also occurs when using the keyboard to change the thumb
    position, and when clicking next to the thumb (In all these cases the EVT_SCROLL_THUMBRELEASE
    event does not happen).

    In short, the EVT_SCROLL_CHANGED event is triggered when scrolling/ moving has finished
    independently of the way it had started. Please see the @ref page_samples_widgets ("Slider" page)
    to see the difference between EVT_SCROLL_THUMBRELEASE and EVT_SCROLL_CHANGED in action.

    @remarks
    Note that unless specifying a scroll control identifier, you will need to test for scrollbar
    orientation with wxScrollEvent::GetOrientation, since horizontal and vertical scroll events
    are processed using the same event handler.

    @beginEventTable{wxScrollEvent}
    You can use EVT_COMMAND_SCROLL... macros with window IDs for when intercepting
    scroll events from controls, or EVT_SCROLL... macros without window IDs for
    intercepting scroll events from the receiving window -- except for this, the
    macros behave exactly the same.
    @event{EVT_SCROLL(func)}
        Process all scroll events.
    @event{EVT_SCROLL_TOP(func)}
        Process @c wxEVT_SCROLL_TOP scroll-to-top events (minimum position).
    @event{EVT_SCROLL_BOTTOM(func)}
        Process @c wxEVT_SCROLL_BOTTOM scroll-to-bottom events (maximum position).
    @event{EVT_SCROLL_LINEUP(func)}
        Process @c wxEVT_SCROLL_LINEUP line up events.
    @event{EVT_SCROLL_LINEDOWN(func)}
        Process @c wxEVT_SCROLL_LINEDOWN line down events.
    @event{EVT_SCROLL_PAGEUP(func)}
        Process @c wxEVT_SCROLL_PAGEUP page up events.
    @event{EVT_SCROLL_PAGEDOWN(func)}
        Process @c wxEVT_SCROLL_PAGEDOWN page down events.
    @event{EVT_SCROLL_THUMBTRACK(func)}
        Process @c wxEVT_SCROLL_THUMBTRACK thumbtrack events (frequent events sent as the
        user drags the thumbtrack).
    @event{EVT_SCROLL_THUMBRELEASE(func)}
        Process @c wxEVT_SCROLL_THUMBRELEASE thumb release events.
    @event{EVT_SCROLL_CHANGED(func)}
        Process @c wxEVT_SCROLL_CHANGED end of scrolling events (MSW only).
    @event{EVT_COMMAND_SCROLL(id, func)}
        Process all scroll events.
    @event{EVT_COMMAND_SCROLL_TOP(id, func)}
        Process @c wxEVT_SCROLL_TOP scroll-to-top events (minimum position).
    @event{EVT_COMMAND_SCROLL_BOTTOM(id, func)}
        Process @c wxEVT_SCROLL_BOTTOM scroll-to-bottom events (maximum position).
    @event{EVT_COMMAND_SCROLL_LINEUP(id, func)}
        Process @c wxEVT_SCROLL_LINEUP line up events.
    @event{EVT_COMMAND_SCROLL_LINEDOWN(id, func)}
        Process @c wxEVT_SCROLL_LINEDOWN line down events.
    @event{EVT_COMMAND_SCROLL_PAGEUP(id, func)}
        Process @c wxEVT_SCROLL_PAGEUP page up events.
    @event{EVT_COMMAND_SCROLL_PAGEDOWN(id, func)}
        Process @c wxEVT_SCROLL_PAGEDOWN page down events.
    @event{EVT_COMMAND_SCROLL_THUMBTRACK(id, func)}
        Process @c wxEVT_SCROLL_THUMBTRACK thumbtrack events (frequent events sent
        as the user drags the thumbtrack).
    @event{EVT_COMMAND_SCROLL_THUMBRELEASE(func)}
        Process @c wxEVT_SCROLL_THUMBRELEASE thumb release events.
    @event{EVT_COMMAND_SCROLL_CHANGED(func)}
        Process @c wxEVT_SCROLL_CHANGED end of scrolling events (MSW only).
    @endEventTable

    @library{wxcore}
    @category{events}

    @see wxScrollBar, wxSlider, wxSpinButton, wxScrollWinEvent, @ref overview_events
*/
class wxScrollEvent : public wxCommandEvent
{
public:
    /**
        Constructor.
    */
    wxScrollEvent(wxEventType commandType = wxEVT_NULL, int id = 0, int pos = 0,
                  int orientation = 0);

    /**
        Returns wxHORIZONTAL or wxVERTICAL, depending on the orientation of the
        scrollbar.
    */
    int GetOrientation() const;

    /**
        Returns the position of the scrollbar.
    */
    int GetPosition() const;


    void SetOrientation(int orient);
    void SetPosition(int pos);
};



/** @class wxGestureEvent
    This is the base class for all supported gesture events.

    @note Gesture events are not generated by default, you must call
          wxWindow::EnableTouchEvents() with the appropriate parameter to
          request their generation.

    @library{wxcore}
    @category{events}

    @see wxPanGestureEvent, wxZoomGestureEvent, wxRotateGestureEvent

    @since 3.1.1
*/
class wxGestureEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxGestureEvent(wxWindowID winid = 0, wxEventType type = wxEVT_NULL);

    /**
        Returns the position where the event took effect, in client coordinates: position of Pan event,
        center of zoom for Zoom event, center of rotation for Rotate event, center of box formed by 2 fingers
        for Two Finger Tap event and position of the pressed finger for Press and Tap Event.
    */
    const wxPoint& GetPosition() const;

    /**
        Returns true if the event was the first in a gesture sequence.
    */
    bool IsGestureStart() const;

    /**
        Returns true if the event was the last in a gesture sequence.
    */
    bool IsGestureEnd() const;

    /**
        Sets the position where the event took effect, in client coordinates: position of Pan event,
        center of zoom for Zoom event, center of rotation for Rotate event.
    */
    void SetPosition(const wxPoint& pos);

    /**
        Sets the event to be the first in a gesture sequence.
    */
    void SetGestureStart(bool isStart = true);

    /**
        Sets the event to be the last in a gesture sequence.
    */
    void SetGestureEnd(bool isEnd = true);
};


/** @class wxPanGestureEvent

    This event is generated when the user moves a finger on the surface.

    wxGTK also generates this event during mouse dragging (mouse motion while a left mouse button is depressed).

    Note that OSX requires the primary mouse button to be pressed while performing the finger movement.

    @beginEventTable{wxPanGestureEvent}
    @event{EVT_GESTURE_PAN(id, func)}
        Process a @c wxEVT_GESTURE_PAN.
    @endEventTable

    @library{wxcore}
    @category{events}

    @since 3.1.1
*/
class wxPanGestureEvent : public wxGestureEvent
{
public:
    /**
        Constructor.
    */
    wxPanGestureEvent(wxWindowID winid = 0);

    /**
        Returns the distance covered since the previous panning event.
    */
    wxPoint GetDelta() const;

    /**
        Sets the distance covered since the previous panning event.
    */
    void SetDelta(const wxPoint& delta);
};


/** @class wxZoomGestureEvent

    This event is generated when two fingers pinch the surface to zoom in or out.

    @beginEventTable{wxZoomGestureEvent}
    @event{EVT_GESTURE_ZOOM(id, func)}
        Process a @c wxEVT_GESTURE_ZOOM.
    @endEventTable

    @library{wxcore}
    @category{events}

    @since 3.1.1
*/
class wxZoomGestureEvent : public wxGestureEvent
{
public:
    /**
        Constructor.
    */
    wxZoomGestureEvent(wxWindowID windid = 0);

    /**
        Returns the zoom Factor since the gesture started. Hence, starting of the gesture
        is considered as 1:1. A value greater than 1.0 means we should enlarge
        (or zoom in), a value less than 1.0 means we should shrink (or zoom out).
    */
    double GetZoomFactor() const;

    /**
        Sets the zoom Factor.
    */
    void SetZoomFactor(double zoomFactor);
};


/** @class wxRotateGestureEvent

    This event is generated when two fingers move in opposite directions on the surface.

    @beginEventTable{wxRotateGestureEvent}
    @event{EVT_GESTURE_ROTATE(id, func)}
        Process a @c wxEVT_GESTURE_ROTATE.
    @endEventTable

    @library{wxcore}
    @category{events}

    @since 3.1.1
*/
class wxRotateGestureEvent : public wxGestureEvent
{
public:
    /**
        Constructor.
    */
    wxRotateGestureEvent(wxWindowID windid = 0);

    /**
        Returns the total angle of rotation in radians in clockwise direction since the
        gesture was first started i.e. when IsGestureStart() returned true. This value is always
        greater than or equal to zero.
    */
    double GetRotationAngle() const;

     /**
        Sets the total angle of rotation in radians in clockwise direction since the
        gesture was first started i.e. when IsGestureStart() returned true. This value is always
        greater than or equal to zero.
    */
    void SetRotationAngle(double rotationAngle);
};

/** @class wxTwoFingerTapEvent

    This event is generated when two fingers touch the surface at the same time.

    @beginEventTable{wxTwoFingerTapEvent}
    @event{EVT_TWO_FINGER_TAP(id, func)}
        Process a @c wxEVT_TWO_FINGER_TAP.
    @endEventTable

    @library{wxcore}
    @category{events}

    @since 3.1.1
*/
class wxTwoFingerTapEvent : public wxGestureEvent
{
public:
    /**
        Constructor.
    */
    wxTwoFingerTapEvent(wxWindowID windid = 0);
};

/** @class wxLongPressEvent

    This event is generated when one finger touches the surface and remains stationary.

    Note that currently it is only generated under wxGTK and wxOSX.

    wxGTK also generates this event when left mouse button is being pressed for some minimum duration of time.

    @beginEventTable{wxLongPressEvent}
    @event{EVT_LONG_PRESS(id, func)}
        Process a @c wxEVT_LONG_PRESS.
    @endEventTable

    @library{wxcore}
    @category{events}

    @since 3.1.1
*/
class wxLongPressEvent : public wxGestureEvent
{
public:
    /**
        Constructor.
    */
    wxLongPressEvent(wxWindowID windid = 0);
};

/** @class wxPressAndTapEvent

    This event is generated when the user press the surface with one finger and taps with another.

    Note that once started the event will also be generated when the finger that was pressed moves on surface.

    @beginEventTable{wxPressAndTapEvent}
    @event{EVT_PRESS_AND_TAP(id, func)}
        Process a @c wxEVT_PRESS_AND_TAP.
    @endEventTable

    @library{wxcore}
    @category{events}

    @since 3.1.1
*/
class wxPressAndTapEvent : public wxGestureEvent
{
public:
    /**
        Constructor.
    */
    wxPressAndTapEvent(wxWindowID windid = 0);
};

#endif // wxUSE_GUI

#if wxUSE_BASE

/**
    See wxIdleEvent::SetMode() for more info.
*/
enum wxIdleMode
{
        /** Send idle events to all windows */
    wxIDLE_PROCESS_ALL,

        /** Send idle events to windows that have the wxWS_EX_PROCESS_IDLE flag specified */
    wxIDLE_PROCESS_SPECIFIED
};


/**
    @class wxIdleEvent

    This class is used for idle events, which are generated when the system becomes
    idle. Note that, unless you do something specifically, the idle events are not
    sent if the system remains idle once it has become it, e.g. only a single idle
    event will be generated until something else resulting in more normal events
    happens and only then is the next idle event sent again.

    If you need to ensure a continuous stream of idle events, you can either use
    wxIdleEvent::RequestMore method in your handler or call wxWakeUpIdle() periodically
    (for example from a timer event handler), but note that both of these approaches
    (and especially the first one) increase the system load and so should be avoided
    if possible.

    By default, idle events are sent to all windows, including even the hidden
    ones because they may be shown if some condition is met from their @c
    wxEVT_IDLE (or related @c wxEVT_UPDATE_UI) handler. The children of hidden
    windows do not receive idle events however as they can't change their state
    in any way noticeable by the user. Finally, the global wxApp object also
    receives these events, as usual, so it can be used for any global idle time
    processing.

    If sending idle events to all windows is causing a significant overhead in
    your application, you can call wxIdleEvent::SetMode with the value
    wxIDLE_PROCESS_SPECIFIED, and set the wxWS_EX_PROCESS_IDLE extra window
    style for every window which should receive idle events, all the other ones
    will not receive them in this case.

    @beginEventTable{wxIdleEvent}
    @event{EVT_IDLE(func)}
        Process a @c wxEVT_IDLE event.
    @endEventTable

    @library{wxbase}
    @category{events}

    @section sec_delayed_action Delayed Action Mechanism

    wxIdleEvent can be used to perform some action "at slightly later time".
    This can be necessary in several circumstances when, for whatever reason,
    something can't be done in the current event handler. For example, if a
    mouse event handler is called with the mouse button pressed, the mouse can
    be currently captured and some operations with it -- notably capturing it
    again -- might be impossible or lead to undesirable results. If you still
    want to capture it, you can do it from @c wxEVT_IDLE handler when it is
    called the next time instead of doing it immediately.

    This can be achieved in two different ways: when using static event tables,
    you will need a flag indicating to the (always connected) idle event
    handler whether the desired action should be performed. The originally
    called handler would then set it to indicate that it should indeed be done
    and the idle handler itself would reset it to prevent it from doing the
    same action again.

    Using dynamically connected event handlers things are even simpler as the
    original event handler can simply wxEvtHandler::Connect() or
    wxEvtHandler::Bind() the idle event handler which would only be executed
    then and could wxEvtHandler::Disconnect() or wxEvtHandler::Unbind() itself.


    @see @ref overview_events, wxUpdateUIEvent, wxWindow::OnInternalIdle
*/
class wxIdleEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxIdleEvent();

    /**
        Static function returning a value specifying how wxWidgets will send idle
        events: to all windows, or only to those which specify that they
        will process the events.

        @see SetMode().
    */
    static wxIdleMode GetMode();

    /**
        Returns @true if the OnIdle function processing this event requested more
        processing time.

        @see RequestMore()
    */
    bool MoreRequested() const;

    /**
        Tells wxWidgets that more processing is required.

        This function can be called by an OnIdle handler for a window or window event
        handler to indicate that wxApp::OnIdle should forward the OnIdle event once
        more to the application windows.

        If no window calls this function during OnIdle, then the application will
        remain in a passive event loop (not calling OnIdle) until a new event is
        posted to the application by the windowing system.

        @see MoreRequested()
    */
    void RequestMore(bool needMore = true);

    /**
        Static function for specifying how wxWidgets will send idle events: to
        all windows, or only to those which specify that they will process the events.

        @param mode
            Can be one of the ::wxIdleMode values.
            The default is wxIDLE_PROCESS_ALL.
    */
    static void SetMode(wxIdleMode mode);
};

#endif // wxUSE_BASE

#if wxUSE_GUI

/**
    @class wxInitDialogEvent

    A wxInitDialogEvent is sent as a dialog or panel is being initialised.
    Handlers for this event can transfer data to the window.

    The default handler calls wxWindow::TransferDataToWindow.

    @beginEventTable{wxInitDialogEvent}
    @event{EVT_INIT_DIALOG(func)}
        Process a @c wxEVT_INIT_DIALOG event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see @ref overview_events
*/
class wxInitDialogEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxInitDialogEvent(int id = 0);
};



/**
    @class wxWindowDestroyEvent

    This event is sent as early as possible during the window destruction
    process.

    For the top level windows, as early as possible means that this is done by
    wxFrame or wxDialog destructor, i.e. after the destructor of the derived
    class was executed and so any methods specific to the derived class can't
    be called any more from this event handler. If you need to do this, you
    must call wxWindow::SendDestroyEvent() from your derived class destructor.

    For the child windows, this event is generated just before deleting the
    window from wxWindow::Destroy() (which is also called when the parent
    window is deleted) or from the window destructor if operator @c delete was
    used directly (which is not recommended for this very reason).

    It is usually pointless to handle this event in the window itself but it ca
    be very useful to receive notifications about the window destruction in the
    parent window or in any other object interested in this window.

    @library{wxcore}
    @category{events}

    @see @ref overview_events, wxWindowCreateEvent
*/
class wxWindowDestroyEvent : public wxCommandEvent
{
public:
    /**
        Constructor.
    */
    wxWindowDestroyEvent(wxWindow* win = NULL);

    /// Return the window being destroyed.
    wxWindow *GetWindow() const;
};


/**
    @class wxNavigationKeyEvent

    This event class contains information about navigation events,
    generated by navigation keys such as tab and page down.

    This event is mainly used by wxWidgets implementations.
    A wxNavigationKeyEvent handler is automatically provided by wxWidgets
    when you enable keyboard navigation inside a window by inheriting it from
    wxNavigationEnabled<>.

    @beginEventTable{wxNavigationKeyEvent}
    @event{EVT_NAVIGATION_KEY(func)}
        Process a navigation key event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see wxWindow::Navigate, wxWindow::NavigateIn
*/
class wxNavigationKeyEvent : public wxEvent
{
public:
    /**
        Flags which can be used with wxNavigationKeyEvent.
    */
    enum wxNavigationKeyEventFlags
    {
        IsBackward = 0x0000,
        IsForward = 0x0001,
        WinChange = 0x0002,
        FromTab = 0x0004
    };

    wxNavigationKeyEvent();
    wxNavigationKeyEvent(const wxNavigationKeyEvent& event);

    /**
        Returns the child that has the focus, or @NULL.
    */
    wxWindow* GetCurrentFocus() const;

    /**
        Returns @true if the navigation was in the forward direction.
    */
    bool GetDirection() const;

    /**
        Returns @true if the navigation event was from a tab key.
        This is required for proper navigation over radio buttons.
    */
    bool IsFromTab() const;

    /**
        Returns @true if the navigation event represents a window change
        (for example, from Ctrl-Page Down in a notebook).
    */
    bool IsWindowChange() const;

    /**
        Sets the current focus window member.
    */
    void SetCurrentFocus(wxWindow* currentFocus);

    /**
        Sets the direction to forward if @a direction is @true, or backward
        if @false.
    */
    void SetDirection(bool direction);

    /**
        Sets the flags for this event.
        The @a flags can be a combination of the
        wxNavigationKeyEvent::wxNavigationKeyEventFlags values.
    */
    void SetFlags(long flags);

    /**
        Marks the navigation event as from a tab key.
    */
    void SetFromTab(bool fromTab);

    /**
        Marks the event as a window change event.
    */
    void SetWindowChange(bool windowChange);
};



/**
    @class wxMouseCaptureChangedEvent

    An mouse capture changed event is sent to a window that loses its
    mouse capture. This is called even if wxWindow::ReleaseMouse
    was called by the application code. Handling this event allows
    an application to cater for unexpected capture releases which
    might otherwise confuse mouse handling code.

    @onlyfor{wxmsw}

    @beginEventTable{wxMouseCaptureChangedEvent}
    @event{EVT_MOUSE_CAPTURE_CHANGED(func)}
        Process a @c wxEVT_MOUSE_CAPTURE_CHANGED event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see wxMouseCaptureLostEvent, @ref overview_events,
         wxWindow::CaptureMouse, wxWindow::ReleaseMouse, wxWindow::GetCapture
*/
class wxMouseCaptureChangedEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxMouseCaptureChangedEvent(wxWindowID windowId = 0,
                               wxWindow* gainedCapture = NULL);

    /**
        Returns the window that gained the capture, or @NULL if it was a
        non-wxWidgets window.
    */
    wxWindow* GetCapturedWindow() const;
};



/**
    @class wxCloseEvent

    This event class contains information about window and session close events.

    The handler function for EVT_CLOSE is called when the user has tried to close a
    a frame or dialog box using the window manager (X) or system menu (Windows).
    It can also be invoked by the application itself programmatically, for example by
    calling the wxWindow::Close function.

    You should check whether the application is forcing the deletion of the window
    using wxCloseEvent::CanVeto. If this is @false, you @e must destroy the window
    using wxWindow::Destroy.

    If the return value is @true, it is up to you whether you respond by destroying
    the window.

    If you don't destroy the window, you should call wxCloseEvent::Veto to
    let the calling code know that you did not destroy the window.
    This allows the wxWindow::Close function to return @true or @false depending
    on whether the close instruction was honoured or not.

    Example of a wxCloseEvent handler:

    @code
        void MyFrame::OnClose(wxCloseEvent& event)
        {
            if ( event.CanVeto() && m_bFileNotSaved )
            {
                if ( wxMessageBox("The file has not been saved... continue closing?",
                                  "Please confirm",
                                  wxICON_QUESTION | wxYES_NO) != wxYES )
                {
                    event.Veto();
                    return;
                }
            }

            Destroy();  // you may also do:  event.Skip();
                        // since the default event handler does call Destroy(), too
        }
    @endcode

    The EVT_END_SESSION event is slightly different as it is sent by the system
    when the user session is ending (e.g. because of log out or shutdown) and
    so all windows are being forcefully closed. At least under MSW, after the
    handler for this event is executed the program is simply killed by the
    system. Because of this, the default handler for this event provided by
    wxWidgets calls all the usual cleanup code (including wxApp::OnExit()) so
    that it could still be executed and exit()s the process itself, without
    waiting for being killed. If this behaviour is for some reason undesirable,
    make sure that you define a handler for this event in your wxApp-derived
    class and do not call @c event.Skip() in it (but be aware that the system
    will still kill your application).

    @beginEventTable{wxCloseEvent}
    @event{EVT_CLOSE(func)}
        Process a @c wxEVT_CLOSE_WINDOW command event, supplying the member function.
        This event applies to wxFrame and wxDialog classes.
    @event{EVT_QUERY_END_SESSION(func)}
        Process a @c wxEVT_QUERY_END_SESSION session event, supplying the member function.
        This event can be handled in wxApp-derived class only.
    @event{EVT_END_SESSION(func)}
        Process a @c wxEVT_END_SESSION session event, supplying the member function.
        This event can be handled in wxApp-derived class only.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see wxWindow::Close, @ref overview_windowdeletion
*/
class wxCloseEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxCloseEvent(wxEventType commandEventType = wxEVT_NULL, int id = 0);

    /**
        Returns @true if you can veto a system shutdown or a window close event.
        Vetoing a window close event is not possible if the calling code wishes to
        force the application to exit, and so this function must be called to check this.
    */
    bool CanVeto() const;

    /**
        Returns @true if the user is just logging off or @false if the system is
        shutting down. This method can only be called for end session and query end
        session events, it doesn't make sense for close window event.
    */
    bool GetLoggingOff() const;

    /**
        Sets the 'can veto' flag.
    */
    void SetCanVeto(bool canVeto);

    /**
        Sets the 'logging off' flag.
    */
    void SetLoggingOff(bool loggingOff);

    /**
        Call this from your event handler to veto a system shutdown or to signal
        to the calling application that a window close did not happen.

        You can only veto a shutdown if CanVeto() returns @true.
    */
    void Veto(bool veto = true);

    /**
       Returns whether the Veto flag was set.
    */
    bool GetVeto() const;
};



/**
    @class wxMenuEvent

    This class is used for a variety of menu-related events. Note that
    these do not include menu command events, which are
    handled using wxCommandEvent objects.

    Events of this class are generated by both menus that are part of a
    wxMenuBar, attached to wxFrame, and popup menus shown by
    wxWindow::PopupMenu(). They are sent to the following objects until one of
    them handles the event:

        -# The menu object itself, as returned by GetMenu(), if any.
        -# The wxMenuBar to which this menu is attached, if any.
        -# The window associated with the menu, e.g. the one calling
        PopupMenu() for the popup menus.
        -# The top level parent of that window if it's different from the
        window itself.

    This is similar to command events generated by the menu items, but, unlike
    them, wxMenuEvent are only sent to the window itself and its top level
    parent but not any intermediate windows in the hierarchy.

    The default handler for @c wxEVT_MENU_HIGHLIGHT in wxFrame displays help
    text in the status bar, see wxFrame::SetStatusBarPane().

    @beginEventTable{wxMenuEvent}
    @event{EVT_MENU_OPEN(func)}
        A menu is about to be opened. On Windows, this is only sent once for each
        navigation of the menubar (up until all menus have closed).
    @event{EVT_MENU_CLOSE(func)}
        A menu has been just closed. Notice that this event is currently being
        sent before the menu selection (@c wxEVT_MENU) event, if any.
    @event{EVT_MENU_HIGHLIGHT(id, func)}
        The menu item with the specified id has been highlighted: used to show
        help prompts in the status bar by wxFrame
    @event{EVT_MENU_HIGHLIGHT_ALL(func)}
        A menu item has been highlighted, i.e. the currently selected menu item has changed.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see wxCommandEvent, @ref overview_events
*/
class wxMenuEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxMenuEvent(wxEventType type = wxEVT_NULL, int id = 0, wxMenu* menu = NULL);

    /**
        Returns the menu which is being opened or closed, or the menu containing
        the highlighted item.

        Note that the returned value can be @NULL if the menu being opened
        doesn't have a corresponding wxMenu, e.g. this happens when opening the
        system menu in wxMSW port.

        @remarks Since 3.1.3 this function can be used with @c OPEN, @c CLOSE
        and @c HIGHLIGHT events. Before 3.1.3, this method can only be used
        with the @c OPEN and @c CLOSE events.
    */
    wxMenu* GetMenu() const;

    /**
        Returns the menu identifier associated with the event.
        This method should be only used with the @c HIGHLIGHT events.
    */
    int GetMenuId() const;

    /**
        Returns @true if the menu which is being opened or closed is a popup menu,
        @false if it is a normal one.

        This method should only be used with the @c OPEN and @c CLOSE events.
    */
    bool IsPopup() const;
};

/**
    @class wxShowEvent

    An event being sent when the window is shown or hidden.
    The event is triggered by calls to wxWindow::Show(), and any user
    action showing a previously hidden window or vice versa (if allowed by
    the current platform and/or window manager).
    Notice that the event is not triggered when the application is iconized
    (minimized) or restored under wxMSW.

    @beginEventTable{wxShowEvent}
    @event{EVT_SHOW(func)}
        Process a @c wxEVT_SHOW event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see @ref overview_events, wxWindow::Show,
         wxWindow::IsShown
*/

class wxShowEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxShowEvent(int winid = 0, bool show = false);

    /**
        Set whether the windows was shown or hidden.
    */
    void SetShow(bool show);

    /**
        Return @true if the window has been shown, @false if it has been
        hidden.
    */
    bool IsShown() const;

    /**
        @deprecated This function is deprecated in favour of IsShown().
    */
    bool GetShow() const;
};



/**
    @class wxIconizeEvent

    An event being sent when the frame is iconized (minimized) or restored.

    @beginEventTable{wxIconizeEvent}
    @event{EVT_ICONIZE(func)}
        Process a @c wxEVT_ICONIZE event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see @ref overview_events, wxTopLevelWindow::Iconize,
         wxTopLevelWindow::IsIconized
*/
class wxIconizeEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxIconizeEvent(int id = 0, bool iconized = true);

    /**
        Returns @true if the frame has been iconized, @false if it has been
        restored.
    */
    bool IsIconized() const;

    /**
        @deprecated This function is deprecated in favour of IsIconized().
    */
    bool Iconized() const;
};



/**
    @class wxMoveEvent

    A move event holds information about window position change.

    These events are currently generated for top level (see wxTopLevelWindow)
    windows in all ports, but are not generated for the child windows in
    wxGTK.

    @beginEventTable{wxMoveEvent}
    @event{EVT_MOVE(func)}
        Process a @c wxEVT_MOVE event, which is generated when a window is moved.
    @event{EVT_MOVE_START(func)}
        Process a @c wxEVT_MOVE_START event, which is generated when the user starts
        to move or size a window. wxMSW only.
    @event{EVT_MOVING(func)}
        Process a @c wxEVT_MOVING event, which is generated while the user is
        moving the window. wxMSW only.
    @event{EVT_MOVE_END(func)}
        Process a @c wxEVT_MOVE_END event, which is generated when the user stops
        moving or sizing a window. wxMSW only.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see wxPoint, @ref overview_events
*/
class wxMoveEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxMoveEvent(const wxPoint& pt, int id = 0);

    /**
        Returns the position of the window generating the move change event.
    */
    wxPoint GetPosition() const;

    wxRect GetRect() const;
    void SetRect(const wxRect& rect);
    void SetPosition(const wxPoint& pos);
};


/**
    @class wxSizeEvent

    A size event holds information about size change events of wxWindow.

    The EVT_SIZE handler function will be called when the window has been resized.

    You may wish to use this for frames to resize their child windows as appropriate.

    Note that the size passed is of the whole window: call wxWindow::GetClientSize()
    for the area which may be used by the application.

    When a window is resized, usually only a small part of the window is damaged
    and you  may only need to repaint that area. However, if your drawing depends on the
    size of the window, you may need to clear the DC explicitly and repaint the whole window.
    In which case, you may need to call wxWindow::Refresh to invalidate the entire window.

    @b Important : Sizers ( see @ref overview_sizer ) rely on size events to function
    correctly. Therefore, in a sizer-based layout, do not forget to call Skip on all
    size events you catch (and don't catch size events at all when you don't need to).

    @beginEventTable{wxSizeEvent}
    @event{EVT_SIZE(func)}
        Process a @c wxEVT_SIZE event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see wxSize, @ref overview_events
*/
class wxSizeEvent : public wxEvent
{
public:
    /**
        Constructor.
    */
    wxSizeEvent(const wxSize& sz, int id = 0);

    /**
        Returns the entire size of the window generating the size change event.

        This is the new total size of the window, i.e. the same size as would
        be returned by wxWindow::GetSize() if it were called now. Use
        wxWindow::GetClientSize() if you catch this event in a top level window
        such as wxFrame to find the size available for the window contents.
    */
    wxSize GetSize() const;
    void SetSize(wxSize size);

    wxRect GetRect() const;
    void SetRect(wxRect rect);
};



/**
    @class wxSetCursorEvent

    A wxSetCursorEvent is generated from wxWindow when the mouse cursor is about
    to be set as a result of mouse motion.

    This event gives the application the chance to perform specific mouse cursor
    processing based on the current position of the mouse within the window.
    Use wxSetCursorEvent::SetCursor to specify the cursor you want to be displayed.

    @beginEventTable{wxSetCursorEvent}
    @event{EVT_SET_CURSOR(func)}
        Process a @c wxEVT_SET_CURSOR event.
    @endEventTable

    @library{wxcore}
    @category{events}

    @see ::wxSetCursor, wxWindow::SetCursor
*/
class wxSetCursorEvent : public wxEvent
{
public:
    /**
        Constructor, used by the library itself internally to initialize the event
        object.
    */
    wxSetCursorEvent(wxCoord x = 0, wxCoord y = 0);

    /**
        Returns a reference to the cursor specified by this event.
    */
    const wxCursor& GetCursor() const;

    /**
        Returns the X coordinate of the mouse in client coordinates.
    */
    wxCoord GetX() const;

    /**
        Returns the Y coordinate of the mouse in client coordinates.
    */
    wxCoord GetY() const;

    /**
        Returns @true if the cursor specified by this event is a valid cursor.

        @remarks You cannot specify wxNullCursor with this event, as it is not
                 considered a valid cursor.
    */
    bool HasCursor() const;

    /**
        Sets the cursor associated with this event.
    */
    void SetCursor(const wxCursor& cursor);
};

#endif // wxUSE_GUI

// ============================================================================
// Global functions/macros
// ============================================================================

/** @addtogroup group_funcmacro_events */
//@{

#if wxUSE_BASE

/**
    A value uniquely identifying the type of the event.

    The values of this type should only be created using wxNewEventType().

    See the macro wxDEFINE_EVENT_TYPE() for more information.

    @see @ref overview_events
*/
typedef int wxEventType;

/**
    A special event type usually used to indicate that some wxEvent has yet
    no type assigned.
*/
wxEventType wxEVT_NULL;

wxEventType wxEVT_ANY;

/**
    Generates a new unique event type.

    Usually this function is only used by wxDEFINE_EVENT() and not called
    directly.
*/
wxEventType wxNewEventType();

/**
    Define a new event type associated with the specified event class.

    This macro defines a new unique event type @a name associated with the
    event class @a cls.

    For example:
    @code
    wxDEFINE_EVENT(MY_COMMAND_EVENT, wxCommandEvent);

    class MyCustomEvent : public wxEvent { ... };
    wxDEFINE_EVENT(MY_CUSTOM_EVENT, MyCustomEvent);
    @endcode

    @see wxDECLARE_EVENT(), @ref overview_events_custom
 */
#define wxDEFINE_EVENT(name, cls)

/**
    Declares a custom event type.

    This macro declares a variable called @a name which must be defined
    elsewhere using wxDEFINE_EVENT().

    The class @a cls must be the wxEvent-derived class associated with the
    events of this type and its full declaration must be visible from the point
    of use of this macro.

    For example:
    @code
    wxDECLARE_EVENT(MY_COMMAND_EVENT, wxCommandEvent);

    class MyCustomEvent : public wxEvent { ... };
    wxDECLARE_EVENT(MY_CUSTOM_EVENT, MyCustomEvent);
    @endcode
 */
#define wxDECLARE_EVENT(name, cls)

/**
    Variant of wxDECLARE_EVENT() used for event types defined inside a shared
    library.

    This is mostly used by wxWidgets internally, e.g.
    @code
    wxDECLARE_EXPORTED_EVENT(WXDLLIMPEXP_CORE, wxEVT_BUTTON, wxCommandEvent);
    @endcode
 */
#define wxDECLARE_EXPORTED_EVENT( expdecl, name, cls )

/**
    Helper macro for definition of custom event table macros.

    This macro casts the given event handler to the given function type using
    @c static_cast to ensure that the actual handler is indeed compatible with
    it, before (unsafely) casting it to a generic function pointer used by the
    event tables.

    See wx__DECLARE_EVT1 for an example of use.

    @see @ref overview_events_custom_ownclass
 */
#define wxEVENT_HANDLER_CAST(functype, func)

/**
    This macro is used to define event table macros for handling custom
    events.

    Example of use:
    @code
    class MyEvent : public wxEvent { ... };

    // note that this is not necessary unless using old compilers: for the
    // reasonably new ones just use &func instead of MyEventHandler(func)
    typedef void (wxEvtHandler::*MyEventFunction)(MyEvent&);
    #define MyEventHandler(func) wxEVENT_HANDLER_CAST(MyEventFunction, func)

    wxDEFINE_EVENT(MY_EVENT_TYPE, MyEvent);

    #define EVT_MY(id, func) \
        wx__DECLARE_EVT1(MY_EVENT_TYPE, id, MyEventHandler(func))

    ...

    wxBEGIN_EVENT_TABLE(MyFrame, wxFrame)
        EVT_MY(wxID_ANY, MyFrame::OnMyEvent)
    wxEND_EVENT_TABLE()
    @endcode

    @param evt
        The event type to handle.
    @param id
        The identifier of events to handle.
    @param fn
        The event handler method.
 */
#define wx__DECLARE_EVT1(evt, id, fn)

/**
    Generalized version of the wx__DECLARE_EVT1() macro taking a range of
    IDs instead of a single one.
    Argument @a id1 is the first identifier of the range, @a id2 is the
    second identifier of the range.
*/
#define wx__DECLARE_EVT2(evt, id1, id2, fn)

/**
    Simplified version of the wx__DECLARE_EVT1() macro, to be used when the
    event type must be handled regardless of the ID associated with the
    specific event instances.
*/
#define wx__DECLARE_EVT0(evt, fn)

/**
    Use this macro inside a class declaration to declare a @e static event table
    for that class.

    In the implementation file you'll need to use the wxBEGIN_EVENT_TABLE()
    and the wxEND_EVENT_TABLE() macros, plus some additional @c EVT_xxx macro
    to capture events.

    Note that this macro requires a final semicolon.

    @see @ref overview_events_eventtables
*/
#define wxDECLARE_EVENT_TABLE()

/**
    Use this macro in a source file to start listing @e static event handlers
    for a specific class.

    Use wxEND_EVENT_TABLE() to terminate the event-declaration block.

    @see @ref overview_events_eventtables
*/
#define wxBEGIN_EVENT_TABLE(theClass, baseClass)

/**
    Use this macro in a source file to end listing @e static event handlers
    for a specific class.

    Use wxBEGIN_EVENT_TABLE() to start the event-declaration block.

    @see @ref overview_events_eventtables
*/
#define wxEND_EVENT_TABLE()

/**
    In a GUI application, this function posts @a event to the specified @e dest
    object using wxEvtHandler::AddPendingEvent().

    Otherwise, it dispatches @a event immediately using
    wxEvtHandler::ProcessEvent(). See the respective documentation for details
    (and caveats). Because of limitation of wxEvtHandler::AddPendingEvent()
    this function is not thread-safe for event objects having wxString fields,
    use wxQueueEvent() instead.

    @header{wx/event.h}
*/
void wxPostEvent(wxEvtHandler* dest, const wxEvent& event);

/**
    Queue an event for processing on the given object.

    This is a wrapper around wxEvtHandler::QueueEvent(), see its documentation
    for more details.

    @header{wx/event.h}

    @param dest
        The object to queue the event on, can't be @c NULL.
    @param event
        The heap-allocated and non-@c NULL event to queue, the function takes
        ownership of it.
 */
void wxQueueEvent(wxEvtHandler* dest, wxEvent *event);

#endif // wxUSE_BASE

#if wxUSE_GUI

wxEventType wxEVT_BUTTON;
wxEventType wxEVT_CHECKBOX;
wxEventType wxEVT_CHOICE;
wxEventType wxEVT_LISTBOX;
wxEventType wxEVT_LISTBOX_DCLICK;
wxEventType wxEVT_CHECKLISTBOX;
wxEventType wxEVT_MENU;
wxEventType wxEVT_SLIDER;
wxEventType wxEVT_RADIOBOX;
wxEventType wxEVT_RADIOBUTTON;
wxEventType wxEVT_SCROLLBAR;
wxEventType wxEVT_VLBOX;
wxEventType wxEVT_COMBOBOX;
wxEventType wxEVT_TOOL_RCLICKED;
wxEventType wxEVT_TOOL_DROPDOWN;
wxEventType wxEVT_TOOL_ENTER;
wxEventType wxEVT_COMBOBOX_DROPDOWN;
wxEventType wxEVT_COMBOBOX_CLOSEUP;
wxEventType wxEVT_THREAD;
wxEventType wxEVT_LEFT_DOWN;
wxEventType wxEVT_LEFT_UP;
wxEventType wxEVT_MIDDLE_DOWN;
wxEventType wxEVT_MIDDLE_UP;
wxEventType wxEVT_RIGHT_DOWN;
wxEventType wxEVT_RIGHT_UP;
wxEventType wxEVT_MOTION;
wxEventType wxEVT_ENTER_WINDOW;
wxEventType wxEVT_LEAVE_WINDOW;
wxEventType wxEVT_LEFT_DCLICK;
wxEventType wxEVT_MIDDLE_DCLICK;
wxEventType wxEVT_RIGHT_DCLICK;
wxEventType wxEVT_SET_FOCUS;
wxEventType wxEVT_KILL_FOCUS;
wxEventType wxEVT_CHILD_FOCUS;
wxEventType wxEVT_MOUSEWHEEL;
wxEventType wxEVT_MAGNIFY;
wxEventType wxEVT_AUX1_DOWN;
wxEventType wxEVT_AUX1_UP;
wxEventType wxEVT_AUX1_DCLICK;
wxEventType wxEVT_AUX2_DOWN;
wxEventType wxEVT_AUX2_UP;
wxEventType wxEVT_AUX2_DCLICK;
wxEventType wxEVT_CHAR;
wxEventType wxEVT_CHAR_HOOK;
wxEventType wxEVT_NAVIGATION_KEY;
wxEventType wxEVT_KEY_DOWN;
wxEventType wxEVT_KEY_UP;
wxEventType wxEVT_HOTKEY;
wxEventType wxEVT_SET_CURSOR;
wxEventType wxEVT_SCROLL_TOP;
wxEventType wxEVT_SCROLL_BOTTOM;
wxEventType wxEVT_SCROLL_LINEUP;
wxEventType wxEVT_SCROLL_LINEDOWN;
wxEventType wxEVT_SCROLL_PAGEUP;
wxEventType wxEVT_SCROLL_PAGEDOWN;
wxEventType wxEVT_SCROLL_THUMBTRACK;
wxEventType wxEVT_SCROLL_THUMBRELEASE;
wxEventType wxEVT_SCROLL_CHANGED;
wxEventType wxEVT_SPIN_UP;
wxEventType wxEVT_SPIN_DOWN;
wxEventType wxEVT_SPIN;
wxEventType wxEVT_SCROLLWIN_TOP;
wxEventType wxEVT_SCROLLWIN_BOTTOM;
wxEventType wxEVT_SCROLLWIN_LINEUP;
wxEventType wxEVT_SCROLLWIN_LINEDOWN;
wxEventType wxEVT_SCROLLWIN_PAGEUP;
wxEventType wxEVT_SCROLLWIN_PAGEDOWN;
wxEventType wxEVT_SCROLLWIN_THUMBTRACK;
wxEventType wxEVT_SCROLLWIN_THUMBRELEASE;
wxEventType wxEVT_GESTURE_PAN;
wxEventType wxEVT_GESTURE_ZOOM;
wxEventType wxEVT_GESTURE_ROTATE;
wxEventType wxEVT_TWO_FINGER_TAP;
wxEventType wxEVT_LONG_PRESS;
wxEventType wxEVT_PRESS_AND_TAP;
wxEventType wxEVT_SIZE;
wxEventType wxEVT_MOVE;
wxEventType wxEVT_CLOSE_WINDOW;
wxEventType wxEVT_END_SESSION;
wxEventType wxEVT_QUERY_END_SESSION;
wxEventType wxEVT_ACTIVATE_APP;
wxEventType wxEVT_ACTIVATE;
wxEventType wxEVT_CREATE;
wxEventType wxEVT_DESTROY;
wxEventType wxEVT_SHOW;
wxEventType wxEVT_ICONIZE;
wxEventType wxEVT_MAXIMIZE;
wxEventType wxEVT_MOUSE_CAPTURE_CHANGED;
wxEventType wxEVT_MOUSE_CAPTURE_LOST;
wxEventType wxEVT_PAINT;
wxEventType wxEVT_ERASE_BACKGROUND;
wxEventType wxEVT_NC_PAINT;
wxEventType wxEVT_MENU_OPEN;
wxEventType wxEVT_MENU_CLOSE;
wxEventType wxEVT_MENU_HIGHLIGHT;
wxEventType wxEVT_CONTEXT_MENU;
wxEventType wxEVT_SYS_COLOUR_CHANGED;
wxEventType wxEVT_DISPLAY_CHANGED;
wxEventType wxEVT_DPI_CHANGED;
wxEventType wxEVT_QUERY_NEW_PALETTE;
wxEventType wxEVT_PALETTE_CHANGED;
wxEventType wxEVT_JOY_BUTTON_DOWN;
wxEventType wxEVT_JOY_BUTTON_UP;
wxEventType wxEVT_JOY_MOVE;
wxEventType wxEVT_JOY_ZMOVE;
wxEventType wxEVT_DROP_FILES;
wxEventType wxEVT_INIT_DIALOG;
wxEventType wxEVT_IDLE;
wxEventType wxEVT_UPDATE_UI;
wxEventType wxEVT_SIZING;
wxEventType wxEVT_MOVING;
wxEventType wxEVT_MOVE_START;
wxEventType wxEVT_MOVE_END;
wxEventType wxEVT_HIBERNATE;
wxEventType wxEVT_TEXT_COPY;
wxEventType wxEVT_TEXT_CUT;
wxEventType wxEVT_TEXT_PASTE;
wxEventType wxEVT_COMMAND_LEFT_CLICK;
wxEventType wxEVT_COMMAND_LEFT_DCLICK;
wxEventType wxEVT_COMMAND_RIGHT_CLICK;
wxEventType wxEVT_COMMAND_RIGHT_DCLICK;
wxEventType wxEVT_COMMAND_SET_FOCUS;
wxEventType wxEVT_COMMAND_KILL_FOCUS;
wxEventType wxEVT_COMMAND_ENTER;
wxEventType wxEVT_HELP;
wxEventType wxEVT_DETAILED_HELP;
wxEventType wxEVT_TOOL;
wxEventType wxEVT_WINDOW_MODAL_DIALOG_CLOSED;

#endif // wxUSE_GUI

//@}

