%%
%% %CopyrightBegin%
%% 
%% Copyright Ericsson AB 2018-2020. All Rights Reserved.
%% 
%% Licensed under the Apache License, Version 2.0 (the "License");
%% you may not use this file except in compliance with the License.
%% You may obtain a copy of the License at
%%
%%     http://www.apache.org/licenses/LICENSE-2.0
%%
%% Unless required by applicable law or agreed to in writing, software
%% distributed under the License is distributed on an "AS IS" BASIS,
%% WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%% See the License for the specific language governing permissions and
%% limitations under the License.
%% 
%% %CopyrightEnd%
%%

%% There are some environment variables that can be used to "manipulate"
%% the test suite: 
%%
%% Variable that controls which 'groups' are to run (with default values)
%%
%%         ESOCK_TEST_API:         include
%%         ESOCK_TEST_SOCK_CLOSE:  include
%%         ESOCK_TEST_TRAFFIC:     include
%%         ESOCK_TEST_TTEST:       exclude
%%
%% Variable that controls "verbosity" of the test case(s):
%%
%%         ESOCK_TEST_QUIET: true (default) | false
%%
%% Defines the runtime of the ttest cases
%% (This is the time during which "measurement" is performed. 
%%  the actual time it takes for the test case to complete
%%  will be longer; setup, completion, ...)
%%
%%          ESOCK_TEST_TTEST_RUNTIME: 10 seconds
%%              Format of values: <integer>[<unit>]
%%              Where unit is: ms | s | m
%%                 ms - milli seconds
%%                 s  - seconds (default)
%%                 m  - minutes
%%

%% Run the entire test suite: 
%% ts:run(emulator, socket_SUITE, [batch]).
%%
%% Run a specific group:
%% ts:run(emulator, socket_SUITE, {group, foo}, [batch]).
%%
%% Run a specific test case:
%% ts:run(emulator, socket_SUITE, foo, [batch]).

-module(socket_SUITE).

-include_lib("common_test/include/ct.hrl").
-include_lib("common_test/include/ct_event.hrl").
-include("socket_test_evaluator.hrl").

%% Suite exports
-export([suite/0, all/0, groups/0]).
-export([init_per_suite/1,    end_per_suite/1,
         init_per_group/2,    end_per_group/2,
         init_per_testcase/2, end_per_testcase/2]).

%% Test cases
-export([
         %% *** API Misc ***
         api_m_info/1,
         api_m_debug/1,

         %% *** API Basic ***
         api_b_open_and_info_udp4/1,
         api_b_open_and_info_udp6/1,
         api_b_open_and_info_tcp4/1,
         api_b_open_and_info_tcp6/1,
         api_b_open_and_close_udp4/1,
         api_b_open_and_close_udp6/1,
         api_b_open_and_close_tcp4/1,
         api_b_open_and_close_tcp6/1,
         api_b_open_and_close_udpL/1,
         api_b_open_and_close_tcpL/1,
         api_b_open_and_close_seqpL/1,
	 api_b_open_and_close_sctp4/1,
         api_b_open_and_maybe_close_raw/1,
         api_b_sendto_and_recvfrom_udp4/1,
         api_b_sendto_and_recvfrom_udpL/1,
         api_b_sendmsg_and_recvmsg_udp4/1,
         api_b_sendmsg_and_recvmsg_udpL/1,
         api_b_send_and_recv_tcp4/1,
         api_b_send_and_recv_tcpL/1,
         api_b_send_and_recv_seqpL/1,
         api_b_sendmsg_and_recvmsg_tcp4/1,
         api_b_sendmsg_and_recvmsg_tcpL/1,
         api_b_sendmsg_and_recvmsg_seqpL/1,
         api_b_sendmsg_and_recvmsg_sctp4/1,
         api_b_sendmsg_iov_dgram_inet/1,
         api_b_sendmsg_iov_dgram_inet6/1,
         api_b_sendmsg_iov_dgram_local/1,
         api_b_sendmsg_iov_stream_inet/1,
         api_b_sendmsg_iov_stream_inet6/1,
         api_b_sendmsg_iov_stream_local/1,

         %% *** API socket from FD ***
         api_ffd_open_wod_and_info_udp4/1,
         api_ffd_open_wod_and_info_udp6/1,
         api_ffd_open_wod_and_info_tcp4/1,
         api_ffd_open_wod_and_info_tcp6/1,
         api_ffd_open_wd_and_info_udp4/1,
         api_ffd_open_wd_and_info_udp6/1,
         api_ffd_open_wd_and_info_tcp4/1,
         api_ffd_open_wd_and_info_tcp6/1,
         api_ffd_open_and_open_wod_and_send_udp4/1,
         api_ffd_open_and_open_wod_and_send_udp6/1,
         api_ffd_open_and_open_wd_and_send_udp4/1,
         api_ffd_open_and_open_wd_and_send_udp6/1,
         api_ffd_open_connect_and_open_wod_and_send_tcp4/1,
         api_ffd_open_connect_and_open_wod_and_send_tcp6/1,
         api_ffd_open_connect_and_open_wd_and_send_tcp4/1,
         api_ffd_open_connect_and_open_wd_and_send_tcp6/1,


         %% *** API async ***
         api_a_connect_tcp4/1,
         api_a_connect_tcp6/1,
         api_a_sendto_and_recvfrom_udp4/1,
         api_a_sendto_and_recvfrom_udp6/1,
         api_a_sendmsg_and_recvmsg_udp4/1,
         api_a_sendmsg_and_recvmsg_udp6/1,
         api_a_send_and_recv_tcp4/1,
         api_a_send_and_recv_tcp6/1,
         api_a_sendmsg_and_recvmsg_tcp4/1,
         api_a_sendmsg_and_recvmsg_tcp6/1,
         api_a_recvfrom_cancel_udp4/1,
         api_a_recvfrom_cancel_udp6/1,
         api_a_recvmsg_cancel_udp4/1,
         api_a_recvmsg_cancel_udp6/1,
         api_a_accept_cancel_tcp4/1,
         api_a_accept_cancel_tcp6/1,
         api_a_recv_cancel_tcp4/1,
         api_a_recv_cancel_tcp6/1,
         api_a_recvmsg_cancel_tcp4/1,
         api_a_recvmsg_cancel_tcp6/1,
         api_a_mrecvfrom_cancel_udp4/1,
         api_a_mrecvfrom_cancel_udp6/1,
         api_a_mrecvmsg_cancel_udp4/1,
         api_a_mrecvmsg_cancel_udp6/1,
         api_a_maccept_cancel_tcp4/1,
         api_a_maccept_cancel_tcp6/1,
         api_a_mrecv_cancel_tcp4/1,
         api_a_mrecv_cancel_tcp6/1,
         api_a_mrecvmsg_cancel_tcp4/1,
         api_a_mrecvmsg_cancel_tcp6/1,


         %% *** API Options ***
         api_opt_simple_otp_options/1,
         api_opt_simple_otp_meta_option/1,
         api_opt_simple_otp_rcvbuf_option/1,
         api_opt_simple_otp_controlling_process/1,
         api_opt_sock_acceptconn_udp/1,
         api_opt_sock_acceptconn_tcp/1,
         api_opt_sock_acceptfilter/1,
         api_opt_sock_bindtodevice/1,
         api_opt_sock_broadcast/1,
         api_opt_sock_debug/1,
         api_opt_sock_domain/1,
         api_opt_sock_dontroute/1,
         api_opt_sock_error/1,
         api_opt_sock_keepalive/1,
         api_opt_sock_linger/1,
         api_opt_sock_mark/1,
         api_opt_sock_oobinline/1,
         api_opt_sock_passcred_tcp4/1,
         api_opt_sock_peek_off_tcpL/1,
         api_opt_sock_peercred_tcpL/1,
         api_opt_sock_priority_udp4/1,
         api_opt_sock_priority_tcp4/1,
         api_opt_sock_rcvbuf_udp4/1,
         api_opt_sock_rcvlowat_udp4/1,
         api_opt_sock_rcvtimeo_udp4/1,
         api_opt_sock_sndbuf_udp4/1,
         api_opt_sock_sndlowat_udp4/1,
         api_opt_sock_sndtimeo_udp4/1,
         api_opt_sock_timestamp_udp4/1,
         api_opt_sock_timestamp_tcp4/1,
         api_opt_ip_add_drop_membership/1,
         api_opt_ip_pktinfo_udp4/1,
         api_opt_ip_recvopts_udp4/1,
         api_opt_ip_recvorigdstaddr_udp4/1,
         api_opt_ip_recvtos_udp4/1,
         api_opt_ip_recvttl_udp4/1,
         api_opt_ip_tos_udp4/1,
         api_opt_ip_recverr_udp4/1,
         api_opt_ip_mopts_udp4/1,
         api_opt_ipv6_recvpktinfo_udp6/1,
	 api_opt_ipv6_flowinfo_udp6/1,
	 api_opt_ipv6_hoplimit_udp6/1,
	 api_opt_ipv6_tclass_udp6/1,
         api_opt_ipv6_recverr_udp6/1,
	 api_opt_ipv6_mopts_udp6/1,
         api_opt_tcp_congestion_tcp4/1,
         api_opt_tcp_cork_tcp4/1,
         api_opt_tcp_maxseg_tcp4/1,
         api_opt_tcp_nodelay_tcp4/1,
         api_opt_udp_cork_udp4/1,

         %% *** API Operation Timeout ***
         api_to_connect_tcp4/1,
         api_to_connect_tcp6/1,
         api_to_accept_tcp4/1,
         api_to_accept_tcp6/1,
         api_to_maccept_tcp4/1,
         api_to_maccept_tcp6/1,
         api_to_send_tcp4/1,
         api_to_send_tcp6/1,
         api_to_sendto_udp4/1,
         api_to_sendto_udp6/1,
         api_to_sendmsg_tcp4/1,
         api_to_sendmsg_tcp6/1,
         api_to_recv_udp4/1,
         api_to_recv_udp6/1,
         api_to_recv_tcp4/1,
         api_to_recv_tcp6/1,
         api_to_recvfrom_udp4/1,
         api_to_recvfrom_udp6/1,
         api_to_recvmsg_udp4/1,
         api_to_recvmsg_udp6/1,
         api_to_recvmsg_tcp4/1,
         api_to_recvmsg_tcp6/1,

         %% Socket Registry
         reg_s_single_open_and_close_and_count/1,
         reg_s_optional_open_and_close_and_count/1,
         

         %% *** Socket Closure ***
         sc_cpe_socket_cleanup_tcp4/1,
         sc_cpe_socket_cleanup_tcp6/1,
         sc_cpe_socket_cleanup_tcpL/1,
         sc_cpe_socket_cleanup_udp4/1,
         sc_cpe_socket_cleanup_udp6/1,
         sc_cpe_socket_cleanup_udpL/1,

         sc_lc_recv_response_tcp4/1,
         sc_lc_recv_response_tcp6/1,
         sc_lc_recv_response_tcpL/1,
         sc_lc_recvfrom_response_udp4/1,
         sc_lc_recvfrom_response_udp6/1,
         sc_lc_recvfrom_response_udpL/1,
         sc_lc_recvmsg_response_tcp4/1,
         sc_lc_recvmsg_response_tcp6/1,
         sc_lc_recvmsg_response_tcpL/1,
         sc_lc_recvmsg_response_udp4/1,
         sc_lc_recvmsg_response_udp6/1,
         sc_lc_recvmsg_response_udpL/1,
         sc_lc_acceptor_response_tcp4/1,
         sc_lc_acceptor_response_tcp6/1,
         sc_lc_acceptor_response_tcpL/1,

         sc_rc_recv_response_tcp4/1,
         sc_rc_recv_response_tcp6/1,
         sc_rc_recv_response_tcpL/1,
         sc_rc_recvmsg_response_tcp4/1,
         sc_rc_recvmsg_response_tcp6/1,
         sc_rc_recvmsg_response_tcpL/1,

         sc_rs_recv_send_shutdown_receive_tcp4/1,
         sc_rs_recv_send_shutdown_receive_tcp6/1,
         sc_rs_recv_send_shutdown_receive_tcpL/1,
         sc_rs_recvmsg_send_shutdown_receive_tcp4/1,
         sc_rs_recvmsg_send_shutdown_receive_tcp6/1,
         sc_rs_recvmsg_send_shutdown_receive_tcpL/1,

         %% *** Traffic ***
         traffic_send_and_recv_counters_tcp4/1,
         traffic_send_and_recv_counters_tcp6/1,
         traffic_send_and_recv_counters_tcpL/1,
         traffic_sendmsg_and_recvmsg_counters_tcp4/1,
         traffic_sendmsg_and_recvmsg_counters_tcp6/1,
         traffic_sendmsg_and_recvmsg_counters_tcpL/1,
         traffic_sendto_and_recvfrom_counters_udp4/1,
         traffic_sendto_and_recvfrom_counters_udp6/1,
         traffic_sendto_and_recvfrom_counters_udpL/1,
         traffic_sendmsg_and_recvmsg_counters_udp4/1,
         traffic_sendmsg_and_recvmsg_counters_udp6/1,
         traffic_sendmsg_and_recvmsg_counters_udpL/1,

         traffic_send_and_recv_chunks_tcp4/1,
         traffic_send_and_recv_chunks_tcp6/1,
         traffic_send_and_recv_chunks_tcpL/1,

         traffic_ping_pong_small_send_and_recv_tcp4/1,
         traffic_ping_pong_small_send_and_recv_tcp6/1,
         traffic_ping_pong_small_send_and_recv_tcpL/1,
         traffic_ping_pong_medium_send_and_recv_tcp4/1,
         traffic_ping_pong_medium_send_and_recv_tcp6/1,
         traffic_ping_pong_medium_send_and_recv_tcpL/1,
         traffic_ping_pong_large_send_and_recv_tcp4/1,
         traffic_ping_pong_large_send_and_recv_tcp6/1,
         traffic_ping_pong_large_send_and_recv_tcpL/1,

         traffic_ping_pong_small_sendto_and_recvfrom_udp4/1,
         traffic_ping_pong_small_sendto_and_recvfrom_udp6/1,
         traffic_ping_pong_small_sendto_and_recvfrom_udpL/1,
         traffic_ping_pong_medium_sendto_and_recvfrom_udp4/1,
         traffic_ping_pong_medium_sendto_and_recvfrom_udp6/1,
         traffic_ping_pong_medium_sendto_and_recvfrom_udpL/1,

         traffic_ping_pong_small_sendmsg_and_recvmsg_tcp4/1,
         traffic_ping_pong_small_sendmsg_and_recvmsg_tcp6/1,
         traffic_ping_pong_small_sendmsg_and_recvmsg_tcpL/1,
         traffic_ping_pong_medium_sendmsg_and_recvmsg_tcp4/1,
         traffic_ping_pong_medium_sendmsg_and_recvmsg_tcp6/1,
         traffic_ping_pong_medium_sendmsg_and_recvmsg_tcpL/1,
         traffic_ping_pong_large_sendmsg_and_recvmsg_tcp4/1,
         traffic_ping_pong_large_sendmsg_and_recvmsg_tcp6/1,
         traffic_ping_pong_large_sendmsg_and_recvmsg_tcpL/1,

         traffic_ping_pong_small_sendmsg_and_recvmsg_udp4/1,
         traffic_ping_pong_small_sendmsg_and_recvmsg_udp6/1,
         traffic_ping_pong_small_sendmsg_and_recvmsg_udpL/1,
         traffic_ping_pong_medium_sendmsg_and_recvmsg_udp4/1,
         traffic_ping_pong_medium_sendmsg_and_recvmsg_udp6/1,
         traffic_ping_pong_medium_sendmsg_and_recvmsg_udpL/1,

         %% *** Time Test ***
         %% Server: transport = gen_tcp, active = false
         %% Client: transport = gen_tcp
         ttest_sgenf_cgenf_small_tcp4/1,
         ttest_sgenf_cgenf_small_tcp6/1,
         ttest_sgenf_cgenf_medium_tcp4/1,
         ttest_sgenf_cgenf_medium_tcp6/1,
         ttest_sgenf_cgenf_large_tcp4/1,
         ttest_sgenf_cgenf_large_tcp6/1,

         ttest_sgenf_cgeno_small_tcp4/1,
         ttest_sgenf_cgeno_small_tcp6/1,
         ttest_sgenf_cgeno_medium_tcp4/1,
         ttest_sgenf_cgeno_medium_tcp6/1,
         ttest_sgenf_cgeno_large_tcp4/1,
         ttest_sgenf_cgeno_large_tcp6/1,

         ttest_sgenf_cgent_small_tcp4/1,
         ttest_sgenf_cgent_small_tcp6/1,
         ttest_sgenf_cgent_medium_tcp4/1,
         ttest_sgenf_cgent_medium_tcp6/1,
         ttest_sgenf_cgent_large_tcp4/1,
         ttest_sgenf_cgent_large_tcp6/1,

         %% Server: transport = gen_tcp, active = false
         %% Client: transport = socket(tcp)
         ttest_sgenf_csockf_small_tcp4/1,
         ttest_sgenf_csockf_small_tcp6/1,
         ttest_sgenf_csockf_medium_tcp4/1,
         ttest_sgenf_csockf_medium_tcp6/1,
         ttest_sgenf_csockf_large_tcp4/1,
         ttest_sgenf_csockf_large_tcp6/1,

         ttest_sgenf_csocko_small_tcp4/1,
         ttest_sgenf_csocko_small_tcp6/1,
         ttest_sgenf_csocko_medium_tcp4/1,
         ttest_sgenf_csocko_medium_tcp6/1,
         ttest_sgenf_csocko_large_tcp4/1,
         ttest_sgenf_csocko_large_tcp6/1,

         ttest_sgenf_csockt_small_tcp4/1,
         ttest_sgenf_csockt_small_tcp6/1,
         ttest_sgenf_csockt_medium_tcp4/1,
         ttest_sgenf_csockt_medium_tcp6/1,
         ttest_sgenf_csockt_large_tcp4/1,
         ttest_sgenf_csockt_large_tcp6/1,

         %% Server: transport = gen_tcp, active = once
         %% Client: transport = gen_tcp
         ttest_sgeno_cgenf_small_tcp4/1,
         ttest_sgeno_cgenf_small_tcp6/1,
         ttest_sgeno_cgenf_medium_tcp4/1,
         ttest_sgeno_cgenf_medium_tcp6/1,
         ttest_sgeno_cgenf_large_tcp4/1,
         ttest_sgeno_cgenf_large_tcp6/1,

         ttest_sgeno_cgeno_small_tcp4/1,
         ttest_sgeno_cgeno_small_tcp6/1,
         ttest_sgeno_cgeno_medium_tcp4/1,
         ttest_sgeno_cgeno_medium_tcp6/1,
         ttest_sgeno_cgeno_large_tcp4/1,
         ttest_sgeno_cgeno_large_tcp6/1,

         ttest_sgeno_cgent_small_tcp4/1,
         ttest_sgeno_cgent_small_tcp6/1,
         ttest_sgeno_cgent_medium_tcp4/1,
         ttest_sgeno_cgent_medium_tcp6/1,
         ttest_sgeno_cgent_large_tcp4/1,
         ttest_sgeno_cgent_large_tcp6/1,

         %% Server: transport = gen_tcp, active = once
         %% Client: transport = socket(tcp)
         ttest_sgeno_csockf_small_tcp4/1,
         ttest_sgeno_csockf_small_tcp6/1,
         ttest_sgeno_csockf_medium_tcp4/1,
         ttest_sgeno_csockf_medium_tcp6/1,
         ttest_sgeno_csockf_large_tcp4/1,
         ttest_sgeno_csockf_large_tcp6/1,

         ttest_sgeno_csocko_small_tcp4/1,
         ttest_sgeno_csocko_small_tcp6/1,
         ttest_sgeno_csocko_medium_tcp4/1,
         ttest_sgeno_csocko_medium_tcp6/1,
         ttest_sgeno_csocko_large_tcp4/1,
         ttest_sgeno_csocko_large_tcp6/1,

         ttest_sgeno_csockt_small_tcp4/1,
         ttest_sgeno_csockt_small_tcp6/1,
         ttest_sgeno_csockt_medium_tcp4/1,
         ttest_sgeno_csockt_medium_tcp6/1,
         ttest_sgeno_csockt_large_tcp4/1,
         ttest_sgeno_csockt_large_tcp6/1,

         %% Server: transport = gen_tcp, active = true
         %% Client: transport = gen_tcp
         ttest_sgent_cgenf_small_tcp4/1,
         ttest_sgent_cgenf_small_tcp6/1,
         ttest_sgent_cgenf_medium_tcp4/1,
         ttest_sgent_cgenf_medium_tcp6/1,
         ttest_sgent_cgenf_large_tcp4/1,
         ttest_sgent_cgenf_large_tcp6/1,

         ttest_sgent_cgeno_small_tcp4/1,
         ttest_sgent_cgeno_small_tcp6/1,
         ttest_sgent_cgeno_medium_tcp4/1,
         ttest_sgent_cgeno_medium_tcp6/1,
         ttest_sgent_cgeno_large_tcp4/1,
         ttest_sgent_cgeno_large_tcp6/1,

         ttest_sgent_cgent_small_tcp4/1,
         ttest_sgent_cgent_small_tcp6/1,
         ttest_sgent_cgent_medium_tcp4/1,
         ttest_sgent_cgent_medium_tcp6/1,
         ttest_sgent_cgent_large_tcp4/1,
         ttest_sgent_cgent_large_tcp6/1,

         %% Server: transport = gen_tcp, active = true
         %% Client: transport = socket(tcp)
         ttest_sgent_csockf_small_tcp4/1,
         ttest_sgent_csockf_small_tcp6/1,
         ttest_sgent_csockf_medium_tcp4/1,
         ttest_sgent_csockf_medium_tcp6/1,
         ttest_sgent_csockf_large_tcp4/1,
         ttest_sgent_csockf_large_tcp6/1,

         ttest_sgent_csocko_small_tcp4/1,
         ttest_sgent_csocko_small_tcp6/1,
         ttest_sgent_csocko_medium_tcp4/1,
         ttest_sgent_csocko_medium_tcp6/1,
         ttest_sgent_csocko_large_tcp4/1,
         ttest_sgent_csocko_large_tcp6/1,

         ttest_sgent_csockt_small_tcp4/1,
         ttest_sgent_csockt_small_tcp6/1,
         ttest_sgent_csockt_medium_tcp4/1,
         ttest_sgent_csockt_medium_tcp6/1,
         ttest_sgent_csockt_large_tcp4/1,
         ttest_sgent_csockt_large_tcp6/1,

         %% Server: transport = socket(tcp), active = false
         %% Client: transport = gen_tcp
         ttest_ssockf_cgenf_small_tcp4/1,
         ttest_ssockf_cgenf_small_tcp6/1,
         ttest_ssockf_cgenf_medium_tcp4/1,
         ttest_ssockf_cgenf_medium_tcp6/1,
         ttest_ssockf_cgenf_large_tcp4/1,
         ttest_ssockf_cgenf_large_tcp6/1,

         ttest_ssockf_cgeno_small_tcp4/1,
         ttest_ssockf_cgeno_small_tcp6/1,
         ttest_ssockf_cgeno_medium_tcp4/1,
         ttest_ssockf_cgeno_medium_tcp6/1,
         ttest_ssockf_cgeno_large_tcp4/1,
         ttest_ssockf_cgeno_large_tcp6/1,

         ttest_ssockf_cgent_small_tcp4/1,
         ttest_ssockf_cgent_small_tcp6/1,
         ttest_ssockf_cgent_medium_tcp4/1,
         ttest_ssockf_cgent_medium_tcp6/1,
         ttest_ssockf_cgent_large_tcp4/1,
         ttest_ssockf_cgent_large_tcp6/1,

         %% Server: transport = socket(tcp), active = false
         %% Client: transport = socket(tcp)
         ttest_ssockf_csockf_small_tcp4/1,
         ttest_ssockf_csockf_small_tcp6/1,
         ttest_ssockf_csockf_small_tcpL/1,
         ttest_ssockf_csockf_medium_tcp4/1,
         ttest_ssockf_csockf_medium_tcp6/1,
         ttest_ssockf_csockf_medium_tcpL/1,
         ttest_ssockf_csockf_large_tcp4/1,
         ttest_ssockf_csockf_large_tcp6/1,
         ttest_ssockf_csockf_large_tcpL/1,

         ttest_ssockf_csocko_small_tcp4/1,
         ttest_ssockf_csocko_small_tcp6/1,
         ttest_ssockf_csocko_small_tcpL/1,
         ttest_ssockf_csocko_medium_tcp4/1,
         ttest_ssockf_csocko_medium_tcp6/1,
         ttest_ssockf_csocko_medium_tcpL/1,
         ttest_ssockf_csocko_large_tcp4/1,
         ttest_ssockf_csocko_large_tcp6/1,
         ttest_ssockf_csocko_large_tcpL/1,

         ttest_ssockf_csockt_small_tcp4/1,
         ttest_ssockf_csockt_small_tcp6/1,
         ttest_ssockf_csockt_small_tcpL/1,
         ttest_ssockf_csockt_medium_tcp4/1,
         ttest_ssockf_csockt_medium_tcp6/1,
         ttest_ssockf_csockt_medium_tcpL/1,
         ttest_ssockf_csockt_large_tcp4/1,
         ttest_ssockf_csockt_large_tcp6/1,
         ttest_ssockf_csockt_large_tcpL/1,

         %% Server: transport = socket(tcp), active = once
         %% Client: transport = gen_tcp
         ttest_ssocko_cgenf_small_tcp4/1,
         ttest_ssocko_cgenf_small_tcp6/1,
         ttest_ssocko_cgenf_medium_tcp4/1,
         ttest_ssocko_cgenf_medium_tcp6/1,
         ttest_ssocko_cgenf_large_tcp4/1,
         ttest_ssocko_cgenf_large_tcp6/1,

         ttest_ssocko_cgeno_small_tcp4/1,
         ttest_ssocko_cgeno_small_tcp6/1,
         ttest_ssocko_cgeno_medium_tcp4/1,
         ttest_ssocko_cgeno_medium_tcp6/1,
         ttest_ssocko_cgeno_large_tcp4/1,
         ttest_ssocko_cgeno_large_tcp6/1,

         ttest_ssocko_cgent_small_tcp4/1,
         ttest_ssocko_cgent_small_tcp6/1,
         ttest_ssocko_cgent_medium_tcp4/1,
         ttest_ssocko_cgent_medium_tcp6/1,
         ttest_ssocko_cgent_large_tcp4/1,
         ttest_ssocko_cgent_large_tcp6/1,

         %% Server: transport = socket(tcp), active = once
         %% Client: transport = socket(tcp)
         ttest_ssocko_csockf_small_tcp4/1,
         ttest_ssocko_csockf_small_tcp6/1,
         ttest_ssocko_csockf_small_tcpL/1,
         ttest_ssocko_csockf_medium_tcp4/1,
         ttest_ssocko_csockf_medium_tcpL/1,
         ttest_ssocko_csockf_medium_tcp6/1,
         ttest_ssocko_csockf_large_tcp4/1,
         ttest_ssocko_csockf_large_tcp6/1,
         ttest_ssocko_csockf_large_tcpL/1,

         ttest_ssocko_csocko_small_tcp4/1,
         ttest_ssocko_csocko_small_tcp6/1,
         ttest_ssocko_csocko_small_tcpL/1,
         ttest_ssocko_csocko_medium_tcp4/1,
         ttest_ssocko_csocko_medium_tcp6/1,
         ttest_ssocko_csocko_medium_tcpL/1,
         ttest_ssocko_csocko_large_tcp4/1,
         ttest_ssocko_csocko_large_tcp6/1,
         ttest_ssocko_csocko_large_tcpL/1,

         ttest_ssocko_csockt_small_tcp4/1,
         ttest_ssocko_csockt_small_tcp6/1,
         ttest_ssocko_csockt_small_tcpL/1,
         ttest_ssocko_csockt_medium_tcp4/1,
         ttest_ssocko_csockt_medium_tcp6/1,
         ttest_ssocko_csockt_medium_tcpL/1,
         ttest_ssocko_csockt_large_tcp4/1,
         ttest_ssocko_csockt_large_tcp6/1,
         ttest_ssocko_csockt_large_tcpL/1,

         %% Server: transport = socket(tcp), active = true
         %% Client: transport = gen_tcp
         ttest_ssockt_cgenf_small_tcp4/1,
         ttest_ssockt_cgenf_small_tcp6/1,
         ttest_ssockt_cgenf_medium_tcp4/1,
         ttest_ssockt_cgenf_medium_tcp6/1,
         ttest_ssockt_cgenf_large_tcp4/1,
         ttest_ssockt_cgenf_large_tcp6/1,

         ttest_ssockt_cgeno_small_tcp4/1,
         ttest_ssockt_cgeno_small_tcp6/1,
         ttest_ssockt_cgeno_medium_tcp4/1,
         ttest_ssockt_cgeno_medium_tcp6/1,
         ttest_ssockt_cgeno_large_tcp4/1,
         ttest_ssockt_cgeno_large_tcp6/1,

         ttest_ssockt_cgent_small_tcp4/1,
         ttest_ssockt_cgent_small_tcp6/1,
         ttest_ssockt_cgent_medium_tcp4/1,
         ttest_ssockt_cgent_medium_tcp6/1,
         ttest_ssockt_cgent_large_tcp4/1,
         ttest_ssockt_cgent_large_tcp6/1,

         %% Server: transport = socket(tcp), active = true
         %% Client: transport = socket(tcp)
         ttest_ssockt_csockf_small_tcp4/1,
         ttest_ssockt_csockf_small_tcp6/1,
         ttest_ssockt_csockf_small_tcpL/1,
         ttest_ssockt_csockf_medium_tcp4/1,
         ttest_ssockt_csockf_medium_tcp6/1,
         ttest_ssockt_csockf_medium_tcpL/1,
         ttest_ssockt_csockf_large_tcp4/1,
         ttest_ssockt_csockf_large_tcp6/1,
         ttest_ssockt_csockf_large_tcpL/1,

         ttest_ssockt_csocko_small_tcp4/1,
         ttest_ssockt_csocko_small_tcp6/1,
         ttest_ssockt_csocko_small_tcpL/1,
         ttest_ssockt_csocko_medium_tcp4/1,
         ttest_ssockt_csocko_medium_tcp6/1,
         ttest_ssockt_csocko_medium_tcpL/1,
         ttest_ssockt_csocko_large_tcp4/1,
         ttest_ssockt_csocko_large_tcp6/1,
         ttest_ssockt_csocko_large_tcpL/1,

         ttest_ssockt_csockt_small_tcp4/1,
         ttest_ssockt_csockt_small_tcp6/1,
         ttest_ssockt_csockt_small_tcpL/1,
         ttest_ssockt_csockt_medium_tcp4/1,
         ttest_ssockt_csockt_medium_tcp6/1,
         ttest_ssockt_csockt_medium_tcpL/1,
         ttest_ssockt_csockt_large_tcp4/1,
         ttest_ssockt_csockt_large_tcp6/1,
         ttest_ssockt_csockt_large_tcpL/1,

         %% Tickets
         otp16359_maccept_tcp4/1,
         otp16359_maccept_tcp6/1,
         otp16359_maccept_tcpL/1
        ]).


%% Internal exports
%% -export([]).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

-define(LIB,        socket_test_lib).
-define(TTEST_LIB,  socket_test_ttest_lib).
-define(LOGGER,     socket_test_logger).

-define(BASIC_REQ,  <<"hejsan">>).
-define(BASIC_REP,  <<"hoppsan">>).

-define(DATA,       <<"HOPPSAN">>). % Temporary
-define(FAIL(R),    exit(R)).

-define(SLEEP(T),   receive after T -> ok end).

-define(MINS(M),    timer:minutes(M)).
-define(SECS(S),    timer:seconds(S)).

-define(TT(T),      ct:timetrap(T)).

-define(F(F, A),    ?LIB:f(F, A)).


-define(TPP_SMALL,  lists:seq(1, 8)).
-define(TPP_MEDIUM, lists:flatten(lists:duplicate(1024, ?TPP_SMALL))).
-define(TPP_LARGE,  lists:flatten(lists:duplicate(1024, ?TPP_MEDIUM))).

-define(TPP_SMALL_NUM,  5000).
-define(TPP_MEDIUM_NUM, 500).
-define(TPP_LARGE_NUM,  50).
-define(TPP_NUM(Config, Base), Base div lookup(esock_factor, 1, Config)).

-define(TTEST_RUNTIME,  ?SECS(10)).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

suite() ->
    [{ct_hooks, [ts_install_cth]},
     {timetrap, {minutes,1}}].

all() -> 
    Groups = [{api,          "ESOCK_TEST_API",        include},
              {reg,          undefined,               include},
	      {socket_close, "ESOCK_TEST_SOCK_CLOSE", include},
	      {traffic,      "ESOCK_TEST_TRAFFIC",    include},
	      {ttest,        "ESOCK_TEST_TTEST",      exclude},
	      {tickets,      "ESOCK_TEST_TICKETS",    include}],
    [use_group(Group, Env, Default) || {Group, Env, Default} <- Groups].

use_group(_Group, undefined, exclude) ->
    [];
use_group(Group, undefined, _Default) ->
    [{group, Group}];
use_group(Group, Env, Default) ->
	case os:getenv(Env) of
	    false when (Default =:= include) ->
		[{group, Group}];
	    false ->
		[];
	    Val ->
		case list_to_atom(string:to_lower(Val)) of
		    Use when (Use =:= include) orelse 
			     (Use =:= enable) orelse 
			     (Use =:= true) ->
			[{group, Group}];
		    _ ->
			[]
		end
	end.
    

groups() -> 
    [{api,                         [], api_cases()},
     {api_misc,                    [], api_misc_cases()},
     {api_basic,                   [], api_basic_cases()},
     {api_from_fd,                 [], api_from_fd_cases()},
     {api_async,                   [], api_async_cases()},
     {api_async_ref,               [], api_async_cases()},
     {api_options,                 [], api_options_cases()},
     {api_options_otp,             [], api_options_otp_cases()},
     {api_options_socket,          [], api_options_socket_cases()},
     {api_option_sock_acceptconn,  [], api_option_sock_acceptconn_cases()},
     {api_option_sock_passcred,    [], api_option_sock_passcred_cases()},
     {api_option_sock_priority,    [], api_option_sock_priority_cases()},
     {api_option_sock_buf,         [], api_option_sock_buf_cases()},
     {api_option_sock_lowat,       [], api_option_sock_lowat_cases()},
     {api_option_sock_timeo,       [], api_option_sock_timeo_cases()},
     {api_option_sock_timestamp,   [], api_option_sock_timestamp_cases()},
     {api_options_ip,              [], api_options_ip_cases()},
     {api_options_ipv6,            [], api_options_ipv6_cases()},
     {api_options_tcp,             [], api_options_tcp_cases()},
     {api_options_udp,             [], api_options_udp_cases()},
     %% {api_options_sctp,            [], api_options_sctp_cases()},
     {api_op_with_timeout,         [], api_op_with_timeout_cases()},
     {reg,                         [], reg_simple_cases()},
     {socket_close,                [], socket_close_cases()},
     {sc_ctrl_proc_exit,           [], sc_cp_exit_cases()},
     {sc_local_close,              [], sc_lc_cases()},
     {sc_remote_close,             [], sc_rc_cases()},
     {sc_remote_shutdown,          [], sc_rs_cases()},
     {traffic,                     [], traffic_cases()},
     {traffic_counters,            [], traffic_counters_cases()},
     {traffic_chunks,              [], traffic_chunks_cases()},
     {traffic_ping_pong,           [], traffic_ping_pong_cases()},
     {traffic_pp_send_recv,        [], traffic_pp_send_recv_cases()},
     {traffic_pp_sendto_recvfrom,  [], traffic_pp_sendto_recvfrom_cases()},
     {traffic_pp_sendmsg_recvmsg,  [], traffic_pp_sendmsg_recvmsg_cases()},
     {ttest,                       [], ttest_cases()},
     {ttest_sgenf,                 [], ttest_sgenf_cases()},
     {ttest_sgenf_cgen,            [], ttest_sgenf_cgen_cases()},
     {ttest_sgenf_cgenf,           [], ttest_sgenf_cgenf_cases()},
     {ttest_sgenf_cgeno,           [], ttest_sgenf_cgeno_cases()},
     {ttest_sgenf_cgent,           [], ttest_sgenf_cgent_cases()},
     {ttest_sgenf_csock,           [], ttest_sgenf_csock_cases()},
     {ttest_sgenf_csockf,          [], ttest_sgenf_csockf_cases()},
     {ttest_sgenf_csocko,          [], ttest_sgenf_csocko_cases()},
     {ttest_sgenf_csockt,          [], ttest_sgenf_csockt_cases()},
     {ttest_sgeno,                 [], ttest_sgeno_cases()},
     {ttest_sgeno_cgen,            [], ttest_sgeno_cgen_cases()},
     {ttest_sgeno_cgenf,           [], ttest_sgeno_cgenf_cases()},
     {ttest_sgeno_cgeno,           [], ttest_sgeno_cgeno_cases()},
     {ttest_sgeno_cgent,           [], ttest_sgeno_cgent_cases()},
     {ttest_sgeno_csock,           [], ttest_sgeno_csock_cases()},
     {ttest_sgeno_csockf,          [], ttest_sgeno_csockf_cases()},
     {ttest_sgeno_csocko,          [], ttest_sgeno_csocko_cases()},
     {ttest_sgeno_csockt,          [], ttest_sgeno_csockt_cases()},
     {ttest_sgent,                 [], ttest_sgent_cases()},
     {ttest_sgent_cgen,            [], ttest_sgent_cgen_cases()},
     {ttest_sgent_cgenf,           [], ttest_sgent_cgenf_cases()},
     {ttest_sgent_cgeno,           [], ttest_sgent_cgeno_cases()},
     {ttest_sgent_cgent,           [], ttest_sgent_cgent_cases()},
     {ttest_sgent_csock,           [], ttest_sgent_csock_cases()},
     {ttest_sgent_csockf,          [], ttest_sgent_csockf_cases()},
     {ttest_sgent_csocko,          [], ttest_sgent_csocko_cases()},
     {ttest_sgent_csockt,          [], ttest_sgent_csockt_cases()},
     {ttest_ssockf,                [], ttest_ssockf_cases()},
     {ttest_ssockf_cgen,           [], ttest_ssockf_cgen_cases()},
     {ttest_ssockf_cgenf,          [], ttest_ssockf_cgenf_cases()},
     {ttest_ssockf_cgeno,          [], ttest_ssockf_cgeno_cases()},
     {ttest_ssockf_cgent,          [], ttest_ssockf_cgent_cases()},
     {ttest_ssockf_csock,          [], ttest_ssockf_csock_cases()},
     {ttest_ssockf_csockf,         [], ttest_ssockf_csockf_cases()},
     {ttest_ssockf_csocko,         [], ttest_ssockf_csocko_cases()},
     {ttest_ssockf_csockt,         [], ttest_ssockf_csockt_cases()},
     {ttest_ssocko,                [], ttest_ssocko_cases()},
     {ttest_ssocko_cgen,           [], ttest_ssocko_cgen_cases()},
     {ttest_ssocko_cgenf,          [], ttest_ssocko_cgenf_cases()},
     {ttest_ssocko_cgeno,          [], ttest_ssocko_cgeno_cases()},
     {ttest_ssocko_cgent,          [], ttest_ssocko_cgent_cases()},
     {ttest_ssocko_csock,          [], ttest_ssocko_csock_cases()},
     {ttest_ssocko_csockf,         [], ttest_ssocko_csockf_cases()},
     {ttest_ssocko_csocko,         [], ttest_ssocko_csocko_cases()},
     {ttest_ssocko_csockt,         [], ttest_ssocko_csockt_cases()},
     {ttest_ssockt,                [], ttest_ssockt_cases()},
     {ttest_ssockt_cgen,           [], ttest_ssockt_cgen_cases()},
     {ttest_ssockt_cgenf,          [], ttest_ssockt_cgenf_cases()},
     {ttest_ssockt_cgeno,          [], ttest_ssockt_cgeno_cases()},
     {ttest_ssockt_cgent,          [], ttest_ssockt_cgent_cases()},
     {ttest_ssockt_csock,          [], ttest_ssockt_csock_cases()},
     {ttest_ssockt_csockf,         [], ttest_ssockt_csockf_cases()},
     {ttest_ssockt_csocko,         [], ttest_ssockt_csocko_cases()},
     {ttest_ssockt_csockt,         [], ttest_ssockt_csockt_cases()},

     %% Ticket groups
     {tickets,                     [], tickets_cases()},
     {otp16359,                    [], otp16359_cases()}
    ].
     
api_cases() ->
    [
     {group, api_misc},
     {group, api_basic},
     {group, api_async},
     {group, api_async_ref},
     {group, api_options},
     {group, api_op_with_timeout}
    ].

api_misc_cases() ->
    [
     api_m_info,
     api_m_debug
    ].

api_basic_cases() ->
    [
     api_b_open_and_info_udp4,
     api_b_open_and_info_udp6,
     api_b_open_and_info_tcp4,
     api_b_open_and_info_tcp6,
     api_b_open_and_close_udp4,
     api_b_open_and_close_udp6,
     api_b_open_and_close_tcp4,
     api_b_open_and_close_tcp6,
     api_b_open_and_close_udpL,
     api_b_open_and_close_tcpL,
     api_b_open_and_close_seqpL,
     api_b_open_and_close_sctp4,
     api_b_open_and_maybe_close_raw,
     api_b_sendto_and_recvfrom_udp4,
     api_b_sendto_and_recvfrom_udpL,
     api_b_sendmsg_and_recvmsg_udp4,
     api_b_sendmsg_and_recvmsg_udpL,
     api_b_send_and_recv_tcp4,
     api_b_send_and_recv_tcpL,
     api_b_send_and_recv_seqpL,
     api_b_sendmsg_and_recvmsg_tcp4,
     api_b_sendmsg_and_recvmsg_tcpL,
     api_b_sendmsg_and_recvmsg_seqpL,
     api_b_sendmsg_and_recvmsg_sctp4,
     api_b_sendmsg_iov_dgram_inet,
     api_b_sendmsg_iov_dgram_inet6,
     api_b_sendmsg_iov_dgram_local,
     api_b_sendmsg_iov_stream_inet,
     api_b_sendmsg_iov_stream_inet6,
     api_b_sendmsg_iov_stream_local
    ].

api_from_fd_cases() ->
    [
     api_ffd_open_wod_and_info_udp4,
     api_ffd_open_wod_and_info_udp6,
     api_ffd_open_wod_and_info_tcp4,
     api_ffd_open_wod_and_info_tcp6,
     api_ffd_open_wd_and_info_udp4,
     api_ffd_open_wd_and_info_udp6,
     api_ffd_open_wd_and_info_tcp4,
     api_ffd_open_wd_and_info_tcp6,
     api_ffd_open_and_open_wod_and_send_udp4,
     api_ffd_open_and_open_wod_and_send_udp6,
     api_ffd_open_and_open_wd_and_send_udp4,
     api_ffd_open_and_open_wd_and_send_udp6,
     api_ffd_open_connect_and_open_wod_and_send_tcp4,
     api_ffd_open_connect_and_open_wod_and_send_tcp6,
     api_ffd_open_connect_and_open_wd_and_send_tcp4,
     api_ffd_open_connect_and_open_wd_and_send_tcp6
    ].

api_async_cases() ->
    [
     api_a_connect_tcp4,
     api_a_connect_tcp6,
     api_a_sendto_and_recvfrom_udp4,
     api_a_sendto_and_recvfrom_udp6,
     api_a_sendmsg_and_recvmsg_udp4,
     api_a_sendmsg_and_recvmsg_udp6,
     api_a_send_and_recv_tcp4,
     api_a_send_and_recv_tcp6,
     api_a_sendmsg_and_recvmsg_tcp4,
     api_a_sendmsg_and_recvmsg_tcp6,
     api_a_recvfrom_cancel_udp4,
     api_a_recvfrom_cancel_udp6,
     api_a_recvmsg_cancel_udp4,
     api_a_recvmsg_cancel_udp6,
     api_a_accept_cancel_tcp4,
     api_a_accept_cancel_tcp6,
     api_a_recv_cancel_tcp4,
     api_a_recv_cancel_tcp6,
     api_a_recvmsg_cancel_tcp4,
     api_a_recvmsg_cancel_tcp6,
     api_a_mrecvfrom_cancel_udp4,
     api_a_mrecvfrom_cancel_udp6,
     api_a_mrecvmsg_cancel_udp4,
     api_a_mrecvmsg_cancel_udp6,
     api_a_maccept_cancel_tcp4,
     api_a_maccept_cancel_tcp6,
     api_a_mrecv_cancel_tcp4,
     api_a_mrecv_cancel_tcp6,
     api_a_mrecvmsg_cancel_tcp4,
     api_a_mrecvmsg_cancel_tcp6
    ].

api_options_cases() ->
    [
     {group, api_options_otp},
     {group, api_options_socket},
     {group, api_options_ip},
     {group, api_options_ipv6},
     {group, api_options_tcp},
     {group, api_options_udp}
     %% {group, api_options_sctp}
    ].

api_options_otp_cases() ->
    [
     api_opt_simple_otp_options,
     api_opt_simple_otp_meta_option,
     api_opt_simple_otp_rcvbuf_option,
     api_opt_simple_otp_controlling_process
    ].

api_options_socket_cases() ->
    [
     {group, api_option_sock_acceptconn},
     api_opt_sock_acceptfilter,
     api_opt_sock_bindtodevice,
     api_opt_sock_broadcast,
     api_opt_sock_debug,
     api_opt_sock_domain,
     api_opt_sock_dontroute,
     api_opt_sock_error,
     api_opt_sock_keepalive,
     api_opt_sock_linger,
     api_opt_sock_mark,
     api_opt_sock_oobinline,
     {group, api_option_sock_passcred},
     api_opt_sock_peek_off_tcpL,
     api_opt_sock_peercred_tcpL,
     {group, api_option_sock_priority},
     {group, api_option_sock_buf},
     {group, api_option_sock_lowat},
     {group, api_option_sock_timeo},
     {group, api_option_sock_timestamp}
    ].

api_option_sock_acceptconn_cases() ->
    [
     api_opt_sock_acceptconn_udp,
     api_opt_sock_acceptconn_tcp
    ].

api_option_sock_passcred_cases() ->
    [
     %% api_opt_sock_passcred_udp4,
     api_opt_sock_passcred_tcp4
    ].

api_option_sock_priority_cases() ->
    [
     api_opt_sock_priority_udp4,
     api_opt_sock_priority_tcp4%,
     %% api_opt_sock_priority_udp6,
     %% api_opt_sock_priority_tcp6
    ].

api_option_sock_buf_cases() ->
    [
     api_opt_sock_rcvbuf_udp4,
     api_opt_sock_sndbuf_udp4
    ].

api_option_sock_lowat_cases() ->
    [
     api_opt_sock_rcvlowat_udp4,
     api_opt_sock_sndlowat_udp4
    ].

api_option_sock_timeo_cases() ->
    [
     api_opt_sock_rcvtimeo_udp4,
     api_opt_sock_sndtimeo_udp4
    ].

api_option_sock_timestamp_cases() ->
    [
     api_opt_sock_timestamp_udp4,
     api_opt_sock_timestamp_tcp4
    ].

api_options_ip_cases() ->
    [
     api_opt_ip_add_drop_membership,
     api_opt_ip_pktinfo_udp4,
     api_opt_ip_recvopts_udp4,
     api_opt_ip_recvorigdstaddr_udp4,
     api_opt_ip_recvtos_udp4,
     api_opt_ip_recvttl_udp4,
     api_opt_ip_tos_udp4,
     api_opt_ip_recverr_udp4,

     %% Should be last!
     api_opt_ip_mopts_udp4
    ].

api_options_ipv6_cases() ->
    [
     api_opt_ipv6_recvpktinfo_udp6,
     api_opt_ipv6_flowinfo_udp6,
     api_opt_ipv6_hoplimit_udp6,
     api_opt_ipv6_tclass_udp6,
     api_opt_ipv6_recverr_udp6,

     %% Should be last!
     api_opt_ipv6_mopts_udp6
    ].

api_options_tcp_cases() ->
    [
     api_opt_tcp_congestion_tcp4,
     %% api_opt_tcp_congestion_tcp6,
     api_opt_tcp_cork_tcp4,
     %% api_opt_tcp_cork_tcp6,
     api_opt_tcp_maxseg_tcp4,
     %% api_opt_tcp_maxseg_tcp6,
     api_opt_tcp_nodelay_tcp4%,
     %% api_opt_tcp_nodelay_tcp6
    ].

api_options_udp_cases() ->
    [
     api_opt_udp_cork_udp4%,
     %% api_opt_udp_cork_udp6
    ].

api_op_with_timeout_cases() ->
    [
     api_to_connect_tcp4,
     api_to_connect_tcp6,
     api_to_accept_tcp4,
     api_to_accept_tcp6,
     api_to_maccept_tcp4,
     api_to_maccept_tcp6,
     api_to_send_tcp4,
     api_to_send_tcp6,
     api_to_sendto_udp4,
     api_to_sendto_udp6,
     api_to_sendmsg_tcp4,
     api_to_sendmsg_tcp6,
     api_to_recv_udp4,
     api_to_recv_udp6,
     api_to_recv_tcp4,
     api_to_recv_tcp6,
     api_to_recvfrom_udp4,
     api_to_recvfrom_udp6,
     api_to_recvmsg_udp4,
     api_to_recvmsg_udp6,
     api_to_recvmsg_tcp4,
     api_to_recvmsg_tcp6
    ].

%% Socket Registry "simple" test cases
reg_simple_cases() ->
    [
     reg_s_single_open_and_close_and_count,
     reg_s_optional_open_and_close_and_count
    ].


%% These cases tests what happens when the socket is closed/shutdown,
%% locally or remotely.
socket_close_cases() ->
    [
     {group, sc_ctrl_proc_exit},
     {group, sc_local_close},
     {group, sc_remote_close},
     {group, sc_remote_shutdown}
    ].

%% These cases are all about socket cleanup after the controlling process
%% exits *without* explicitly calling socket:close/1.
sc_cp_exit_cases() ->
    [
     sc_cpe_socket_cleanup_tcp4,
     sc_cpe_socket_cleanup_tcp6,
     sc_cpe_socket_cleanup_tcpL,
     sc_cpe_socket_cleanup_udp4,
     sc_cpe_socket_cleanup_udp6,
     sc_cpe_socket_cleanup_udpL
    ].

%% These cases tests what happens when the socket is closed locally.
sc_lc_cases() ->
    [
     sc_lc_recv_response_tcp4,
     sc_lc_recv_response_tcp6,
     sc_lc_recv_response_tcpL,

     sc_lc_recvfrom_response_udp4,
     sc_lc_recvfrom_response_udp6,
     sc_lc_recvfrom_response_udpL,

     sc_lc_recvmsg_response_tcp4,
     sc_lc_recvmsg_response_tcp6,
     sc_lc_recvmsg_response_tcpL,
     sc_lc_recvmsg_response_udp4,
     sc_lc_recvmsg_response_udp6,
     sc_lc_recvmsg_response_udpL,

     sc_lc_acceptor_response_tcp4,
     sc_lc_acceptor_response_tcp6,
     sc_lc_acceptor_response_tcpL
    ].

%% These cases tests what happens when the socket is closed remotely.
sc_rc_cases() ->
    [
     sc_rc_recv_response_tcp4,
     sc_rc_recv_response_tcp6,
     sc_rc_recv_response_tcpL,

     sc_rc_recvmsg_response_tcp4,
     sc_rc_recvmsg_response_tcp6,
     sc_rc_recvmsg_response_tcpL
    ].

%% These cases tests what happens when the socket is shutdown/closed remotely
%% after writing and reading is ongoing.
sc_rs_cases() ->
    [
     sc_rs_recv_send_shutdown_receive_tcp4,
     sc_rs_recv_send_shutdown_receive_tcp6,
     sc_rs_recv_send_shutdown_receive_tcpL,

     sc_rs_recvmsg_send_shutdown_receive_tcp4,
     sc_rs_recvmsg_send_shutdown_receive_tcp6,
     sc_rs_recvmsg_send_shutdown_receive_tcpL
    ].


traffic_cases() ->
    [
     {group, traffic_counters},
     {group, traffic_chunks},
     {group, traffic_ping_pong}
    ].

traffic_counters_cases() ->
    [
     traffic_send_and_recv_counters_tcp4,
     traffic_send_and_recv_counters_tcp6,
     traffic_send_and_recv_counters_tcpL,
     traffic_sendmsg_and_recvmsg_counters_tcp4,
     traffic_sendmsg_and_recvmsg_counters_tcp6,
     traffic_sendmsg_and_recvmsg_counters_tcpL,
     traffic_sendto_and_recvfrom_counters_udp4,
     traffic_sendto_and_recvfrom_counters_udp6,
     traffic_sendto_and_recvfrom_counters_udpL,
     traffic_sendmsg_and_recvmsg_counters_udp4,
     traffic_sendmsg_and_recvmsg_counters_udp6,
     traffic_sendmsg_and_recvmsg_counters_udpL
    ].

traffic_chunks_cases() ->
    [
     traffic_send_and_recv_chunks_tcp4,
     traffic_send_and_recv_chunks_tcp6,
     traffic_send_and_recv_chunks_tcpL
    ].

traffic_ping_pong_cases() ->
    [
     {group, traffic_pp_send_recv},
     {group, traffic_pp_sendto_recvfrom},
     {group, traffic_pp_sendmsg_recvmsg}
    ].

traffic_pp_send_recv_cases() ->
    [
     traffic_ping_pong_small_send_and_recv_tcp4,
     traffic_ping_pong_small_send_and_recv_tcp6,
     traffic_ping_pong_small_send_and_recv_tcpL,
     traffic_ping_pong_medium_send_and_recv_tcp4,
     traffic_ping_pong_medium_send_and_recv_tcp6,
     traffic_ping_pong_medium_send_and_recv_tcpL,
     traffic_ping_pong_large_send_and_recv_tcp4,
     traffic_ping_pong_large_send_and_recv_tcp6,
     traffic_ping_pong_large_send_and_recv_tcpL
    ].    

traffic_pp_sendto_recvfrom_cases() ->
    [
     traffic_ping_pong_small_sendto_and_recvfrom_udp4,
     traffic_ping_pong_small_sendto_and_recvfrom_udp6,
     traffic_ping_pong_small_sendto_and_recvfrom_udpL,
     traffic_ping_pong_medium_sendto_and_recvfrom_udp4,
     traffic_ping_pong_medium_sendto_and_recvfrom_udp6,
     traffic_ping_pong_medium_sendto_and_recvfrom_udpL
    ].

traffic_pp_sendmsg_recvmsg_cases() ->
    [    
     traffic_ping_pong_small_sendmsg_and_recvmsg_tcp4,
     traffic_ping_pong_small_sendmsg_and_recvmsg_tcp6,
     traffic_ping_pong_small_sendmsg_and_recvmsg_tcpL,
     traffic_ping_pong_medium_sendmsg_and_recvmsg_tcp4,
     traffic_ping_pong_medium_sendmsg_and_recvmsg_tcp6,
     traffic_ping_pong_medium_sendmsg_and_recvmsg_tcpL,
     traffic_ping_pong_large_sendmsg_and_recvmsg_tcp4,
     traffic_ping_pong_large_sendmsg_and_recvmsg_tcp6,
     traffic_ping_pong_large_sendmsg_and_recvmsg_tcpL,

     traffic_ping_pong_small_sendmsg_and_recvmsg_udp4,
     traffic_ping_pong_small_sendmsg_and_recvmsg_udp6,
     traffic_ping_pong_small_sendmsg_and_recvmsg_udpL,
     traffic_ping_pong_medium_sendmsg_and_recvmsg_udp4,
     traffic_ping_pong_medium_sendmsg_and_recvmsg_udp6,
     traffic_ping_pong_medium_sendmsg_and_recvmsg_udpL
    ].
    
ttest_cases() ->
    [
     %% Server: transport = gen_tcp, active = false
     {group, ttest_sgenf},

     %% Server: transport = gen_tcp, active = once
     {group, ttest_sgeno},

     %% Server: transport = gen_tcp, active = true
     {group, ttest_sgent},

     %% Server: transport = socket(tcp), active = false
     {group, ttest_ssockf},

     %% Server: transport = socket(tcp), active = once
     {group, ttest_ssocko},

     %% Server: transport = socket(tcp), active = true
     {group, ttest_ssockt}

    ].


%% Server: transport = gen_tcp, active = false
ttest_sgenf_cases() ->
    [
     {group, ttest_sgenf_cgen},
     {group, ttest_sgenf_csock}
    ].

%% Server: transport = gen_tcp, active = false
%% Client: transport = gen_tcp
ttest_sgenf_cgen_cases() ->
    [
     {group, ttest_sgenf_cgenf},
     {group, ttest_sgenf_cgeno},
     {group, ttest_sgenf_cgent}
    ].

%% Server: transport = gen_tcp, active = false
%% Client: transport = gen_tcp, active = false
ttest_sgenf_cgenf_cases() ->
    [
     ttest_sgenf_cgenf_small_tcp4,
     ttest_sgenf_cgenf_small_tcp6,

     ttest_sgenf_cgenf_medium_tcp4,
     ttest_sgenf_cgenf_medium_tcp6,

     ttest_sgenf_cgenf_large_tcp4,
     ttest_sgenf_cgenf_large_tcp6
    ].

%% Server: transport = gen_tcp, active = false
%% Client: transport = gen_tcp, active = once
ttest_sgenf_cgeno_cases() ->
    [
     ttest_sgenf_cgeno_small_tcp4,
     ttest_sgenf_cgeno_small_tcp6,

     ttest_sgenf_cgeno_medium_tcp4,
     ttest_sgenf_cgeno_medium_tcp6,

     ttest_sgenf_cgeno_large_tcp4,
     ttest_sgenf_cgeno_large_tcp6
    ].

%% Server: transport = gen_tcp, active = false
%% Client: transport = gen_tcp, active = true
ttest_sgenf_cgent_cases() ->
    [
     ttest_sgenf_cgent_small_tcp4,
     ttest_sgenf_cgent_small_tcp6,

     ttest_sgenf_cgent_medium_tcp4,
     ttest_sgenf_cgent_medium_tcp6,

     ttest_sgenf_cgent_large_tcp4,
     ttest_sgenf_cgent_large_tcp6
    ].

%% Server: transport = gen_tcp, active = false
%% Client: transport = socket(tcp)
ttest_sgenf_csock_cases() ->
    [
     {group, ttest_sgenf_csockf},
     {group, ttest_sgenf_csocko},
     {group, ttest_sgenf_csockt}
    ].

ttest_sgenf_csockf_cases() ->
    [
     ttest_sgenf_csockf_small_tcp4,
     ttest_sgenf_csockf_small_tcp6,

     ttest_sgenf_csockf_medium_tcp4,
     ttest_sgenf_csockf_medium_tcp6,

     ttest_sgenf_csockf_large_tcp4,
     ttest_sgenf_csockf_large_tcp6
    ].

ttest_sgenf_csocko_cases() ->
    [
     ttest_sgenf_csocko_small_tcp4,
     ttest_sgenf_csocko_small_tcp6,

     ttest_sgenf_csocko_medium_tcp4,
     ttest_sgenf_csocko_medium_tcp6,

     ttest_sgenf_csocko_large_tcp4,
     ttest_sgenf_csocko_large_tcp6
    ].

ttest_sgenf_csockt_cases() ->
    [
     ttest_sgenf_csockt_small_tcp4,
     ttest_sgenf_csockt_small_tcp6,

     ttest_sgenf_csockt_medium_tcp4,
     ttest_sgenf_csockt_medium_tcp6,

     ttest_sgenf_csockt_large_tcp4,
     ttest_sgenf_csockt_large_tcp6
    ].

%% Server: transport = gen_tcp, active = once
ttest_sgeno_cases() ->
    [
     {group, ttest_sgeno_cgen},
     {group, ttest_sgeno_csock}
    ].

%% Server: transport = gen_tcp, active = once
%% Client: transport = gen_tcp
ttest_sgeno_cgen_cases() ->
    [
     {group, ttest_sgeno_cgenf},
     {group, ttest_sgeno_cgeno},
     {group, ttest_sgeno_cgent}
    ].

%% Server: transport = gen_tcp, active = once
%% Client: transport = gen_tcp, active = false
ttest_sgeno_cgenf_cases() ->
    [
     ttest_sgeno_cgenf_small_tcp4,
     ttest_sgeno_cgenf_small_tcp6,

     ttest_sgeno_cgenf_medium_tcp4,
     ttest_sgeno_cgenf_medium_tcp6,

     ttest_sgeno_cgenf_large_tcp4,
     ttest_sgeno_cgenf_large_tcp6
    ].

%% Server: transport = gen_tcp, active = once
%% Client: transport = gen_tcp, active = once
ttest_sgeno_cgeno_cases() ->
    [
     ttest_sgeno_cgeno_small_tcp4,
     ttest_sgeno_cgeno_small_tcp6,

     ttest_sgeno_cgeno_medium_tcp4,
     ttest_sgeno_cgeno_medium_tcp6,

     ttest_sgeno_cgeno_large_tcp4,
     ttest_sgeno_cgeno_large_tcp6
    ].

%% Server: transport = gen_tcp, active = once
%% Client: transport = gen_tcp, active = true
ttest_sgeno_cgent_cases() ->
    [
     ttest_sgeno_cgent_small_tcp4,
     ttest_sgeno_cgent_small_tcp6,

     ttest_sgeno_cgent_medium_tcp4,
     ttest_sgeno_cgent_medium_tcp6,

     ttest_sgeno_cgent_large_tcp4,
     ttest_sgeno_cgent_large_tcp6
    ].

%% Server: transport = gen_tcp, active = once
%% Client: transport = socket(tcp)
ttest_sgeno_csock_cases() ->
    [
     {group, ttest_sgeno_csockf},
     {group, ttest_sgeno_csocko},
     {group, ttest_sgeno_csockt}
    ].

ttest_sgeno_csockf_cases() ->
    [
     ttest_sgeno_csockf_small_tcp4,
     ttest_sgeno_csockf_small_tcp6,

     ttest_sgeno_csockf_medium_tcp4,
     ttest_sgeno_csockf_medium_tcp6,

     ttest_sgeno_csockf_large_tcp4,
     ttest_sgeno_csockf_large_tcp6
    ].

ttest_sgeno_csocko_cases() ->
    [
     ttest_sgeno_csocko_small_tcp4,
     ttest_sgeno_csocko_small_tcp6,

     ttest_sgeno_csocko_medium_tcp4,
     ttest_sgeno_csocko_medium_tcp6,

     ttest_sgeno_csocko_large_tcp4,
     ttest_sgeno_csocko_large_tcp6
    ].

ttest_sgeno_csockt_cases() ->
    [
     ttest_sgeno_csockt_small_tcp4,
     ttest_sgeno_csockt_small_tcp6,

     ttest_sgeno_csockt_medium_tcp4,
     ttest_sgeno_csockt_medium_tcp6,

     ttest_sgeno_csockt_large_tcp4,
     ttest_sgeno_csockt_large_tcp6
    ].

%% Server: transport = gen_tcp, active = true
ttest_sgent_cases() ->
    [
     {group, ttest_sgent_cgen},
     {group, ttest_sgent_csock}
    ].

%% Server: transport = gen_tcp, active = true
%% Client: transport = gen_tcp
ttest_sgent_cgen_cases() ->
    [
     {group, ttest_sgent_cgenf},
     {group, ttest_sgent_cgeno},
     {group, ttest_sgent_cgent}
    ].

%% Server: transport = gen_tcp, active = true
%% Client: transport = gen_tcp, active = false
ttest_sgent_cgenf_cases() ->
    [
     ttest_sgent_cgenf_small_tcp4,
     ttest_sgent_cgenf_small_tcp6,

     ttest_sgent_cgenf_medium_tcp4,
     ttest_sgent_cgenf_medium_tcp6,

     ttest_sgent_cgenf_large_tcp4,
     ttest_sgent_cgenf_large_tcp6
    ].

%% Server: transport = gen_tcp, active = true
%% Client: transport = gen_tcp, active = once
ttest_sgent_cgeno_cases() ->
    [
     ttest_sgent_cgeno_small_tcp4,
     ttest_sgent_cgeno_small_tcp6,

     ttest_sgent_cgeno_medium_tcp4,
     ttest_sgent_cgeno_medium_tcp6,

     ttest_sgent_cgeno_large_tcp4,
     ttest_sgent_cgeno_large_tcp6
    ].

%% Server: transport = gen_tcp, active = true
%% Client: transport = gen_tcp, active = true
ttest_sgent_cgent_cases() ->
    [
     ttest_sgent_cgent_small_tcp4,
     ttest_sgent_cgent_small_tcp6,

     ttest_sgent_cgent_medium_tcp4,
     ttest_sgent_cgent_medium_tcp6,

     ttest_sgent_cgent_large_tcp4,
     ttest_sgent_cgent_large_tcp6
    ].

%% Server: transport = gen_tcp, active = true
%% Client: transport = socket(tcp)
ttest_sgent_csock_cases() ->
    [
     {group, ttest_sgent_csockf},
     {group, ttest_sgent_csocko},
     {group, ttest_sgent_csockt}
    ].

ttest_sgent_csockf_cases() ->
    [
     ttest_sgent_csockf_small_tcp4,
     ttest_sgent_csockf_small_tcp6,

     ttest_sgent_csockf_medium_tcp4,
     ttest_sgent_csockf_medium_tcp6,

     ttest_sgent_csockf_large_tcp4,
     ttest_sgent_csockf_large_tcp6
    ].

ttest_sgent_csocko_cases() ->
    [
     ttest_sgent_csocko_small_tcp4,
     ttest_sgent_csocko_small_tcp6,

     ttest_sgent_csocko_medium_tcp4,
     ttest_sgent_csocko_medium_tcp6,

     ttest_sgent_csocko_large_tcp4,
     ttest_sgent_csocko_large_tcp6
    ].

ttest_sgent_csockt_cases() ->
    [
     ttest_sgent_csockt_small_tcp4,
     ttest_sgent_csockt_small_tcp6,

     ttest_sgent_csockt_medium_tcp4,
     ttest_sgent_csockt_medium_tcp6,

     ttest_sgent_csockt_large_tcp4,
     ttest_sgent_csockt_large_tcp6
    ].

%% Server: transport = socket(tcp), active = false
ttest_ssockf_cases() ->
    [
     {group, ttest_ssockf_cgen},
     {group, ttest_ssockf_csock}
    ].

%% Server: transport = socket(tcp), active = false
%% Client: transport = gen_tcp
ttest_ssockf_cgen_cases() ->
    [
     {group, ttest_ssockf_cgenf},
     {group, ttest_ssockf_cgeno},
     {group, ttest_ssockf_cgent}
    ].

%% Server: transport = socket(tcp), active = false
%% Client: transport = gen_tcp, active = false
ttest_ssockf_cgenf_cases() ->
    [
     ttest_ssockf_cgenf_small_tcp4,
     ttest_ssockf_cgenf_small_tcp6,

     ttest_ssockf_cgenf_medium_tcp4,
     ttest_ssockf_cgenf_medium_tcp6,

     ttest_ssockf_cgenf_large_tcp4,
     ttest_ssockf_cgenf_large_tcp6
    ].

%% Server: transport = socket(tcp), active = false
%% Client: transport = gen_tcp, active = once
ttest_ssockf_cgeno_cases() ->
    [
     ttest_ssockf_cgeno_small_tcp4,
     ttest_ssockf_cgeno_small_tcp6,

     ttest_ssockf_cgeno_medium_tcp4,
     ttest_ssockf_cgeno_medium_tcp6,

     ttest_ssockf_cgeno_large_tcp4,
     ttest_ssockf_cgeno_large_tcp6
    ].

%% Server: transport = socket(tcp), active = false
%% Client: transport = gen_tcp, active = true
ttest_ssockf_cgent_cases() ->
    [
     ttest_ssockf_cgent_small_tcp4,
     ttest_ssockf_cgent_small_tcp6,

     ttest_ssockf_cgent_medium_tcp4,
     ttest_ssockf_cgent_medium_tcp6,

     ttest_ssockf_cgent_large_tcp4,
     ttest_ssockf_cgent_large_tcp6
    ].

%% Server: transport = socket(tcp), active = false
%% Client: transport = socket(tcp)
ttest_ssockf_csock_cases() ->
    [
     {group, ttest_ssockf_csockf},
     {group, ttest_ssockf_csocko},
     {group, ttest_ssockf_csockt}
    ].

%% Server: transport = socket(tcp), active = false
%% Client: transport = socket(tcp), active = false
ttest_ssockf_csockf_cases() ->
    [
     ttest_ssockf_csockf_small_tcp4,
     ttest_ssockf_csockf_small_tcp6,
     ttest_ssockf_csockf_small_tcpL,

     ttest_ssockf_csockf_medium_tcp4,
     ttest_ssockf_csockf_medium_tcp6,
     ttest_ssockf_csockf_medium_tcpL,

     ttest_ssockf_csockf_large_tcp4,
     ttest_ssockf_csockf_large_tcp6,
     ttest_ssockf_csockf_large_tcpL
    ].

%% Server: transport = socket(tcp), active = false
%% Client: transport = socket(tcp), active = once
ttest_ssockf_csocko_cases() ->
    [
     ttest_ssockf_csocko_small_tcp4,
     ttest_ssockf_csocko_small_tcp6,
     ttest_ssockf_csocko_small_tcpL,

     ttest_ssockf_csocko_medium_tcp4,
     ttest_ssockf_csocko_medium_tcp6,
     ttest_ssockf_csocko_medium_tcpL,

     ttest_ssockf_csocko_large_tcp4,
     ttest_ssockf_csocko_large_tcp6,
     ttest_ssockf_csocko_large_tcpL
    ].

%% Server: transport = socket(tcp), active = false
%% Client: transport = socket(tcp), active = true
ttest_ssockf_csockt_cases() ->
    [
     ttest_ssockf_csockt_small_tcp4,
     ttest_ssockf_csockt_small_tcp6,
     ttest_ssockf_csockt_small_tcpL,

     ttest_ssockf_csockt_medium_tcp4,
     ttest_ssockf_csockt_medium_tcp6,
     ttest_ssockf_csockt_medium_tcpL,

     ttest_ssockf_csockt_large_tcp4,
     ttest_ssockf_csockt_large_tcp6,
     ttest_ssockf_csockt_large_tcpL
    ].

%% Server: transport = socket(tcp), active = once
ttest_ssocko_cases() ->
    [
     {group, ttest_ssocko_cgen},
     {group, ttest_ssocko_csock}
    ].

%% Server: transport = socket(tcp), active = once
%% Client: transport = gen_tcp
ttest_ssocko_cgen_cases() ->
    [
     {group, ttest_ssocko_cgenf},
     {group, ttest_ssocko_cgeno},
     {group, ttest_ssocko_cgent}
    ].

%% Server: transport = socket(tcp), active = once
%% Client: transport = gen_tcp, active = false
ttest_ssocko_cgenf_cases() ->
    [
     ttest_ssocko_cgenf_small_tcp4,
     ttest_ssocko_cgenf_small_tcp6,

     ttest_ssocko_cgenf_medium_tcp4,
     ttest_ssocko_cgenf_medium_tcp6,

     ttest_ssocko_cgenf_large_tcp4,
     ttest_ssocko_cgenf_large_tcp6
    ].

%% Server: transport = socket(tcp), active = once
%% Client: transport = gen_tcp, active = once
ttest_ssocko_cgeno_cases() ->
    [
     ttest_ssocko_cgeno_small_tcp4,
     ttest_ssocko_cgeno_small_tcp6,

     ttest_ssocko_cgeno_medium_tcp4,
     ttest_ssocko_cgeno_medium_tcp6,

     ttest_ssocko_cgeno_large_tcp4,
     ttest_ssocko_cgeno_large_tcp6
    ].

%% Server: transport = socket(tcp), active = once
%% Client: transport = gen_tcp, active = true
ttest_ssocko_cgent_cases() ->
    [
     ttest_ssocko_cgent_small_tcp4,
     ttest_ssocko_cgent_small_tcp6,

     ttest_ssocko_cgent_medium_tcp4,
     ttest_ssocko_cgent_medium_tcp6,

     ttest_ssocko_cgent_large_tcp4,
     ttest_ssocko_cgent_large_tcp6
    ].

%% Server: transport = socket(tcp), active = once
%% Client: transport = socket(tcp)
ttest_ssocko_csock_cases() ->
    [
     {group, ttest_ssocko_csockf},
     {group, ttest_ssocko_csocko},
     {group, ttest_ssocko_csockt}
    ].

%% Server: transport = socket(tcp), active = once
%% Client: transport = socket(tcp), active = false
ttest_ssocko_csockf_cases() ->
    [
     ttest_ssocko_csockf_small_tcp4,
     ttest_ssocko_csockf_small_tcp6,
     ttest_ssocko_csockf_small_tcpL,

     ttest_ssocko_csockf_medium_tcp4,
     ttest_ssocko_csockf_medium_tcp6,
     ttest_ssocko_csockf_medium_tcpL,

     ttest_ssocko_csockf_large_tcp4,
     ttest_ssocko_csockf_large_tcp6,
     ttest_ssocko_csockf_large_tcpL
    ].

%% Server: transport = socket(tcp), active = once
%% Client: transport = socket(tcp), active = once
ttest_ssocko_csocko_cases() ->
    [
     ttest_ssocko_csocko_small_tcp4,
     ttest_ssocko_csocko_small_tcp6,
     ttest_ssocko_csocko_small_tcpL,

     ttest_ssocko_csocko_medium_tcp4,
     ttest_ssocko_csocko_medium_tcp6,
     ttest_ssocko_csocko_medium_tcpL,

     ttest_ssocko_csocko_large_tcp4,
     ttest_ssocko_csocko_large_tcp6,
     ttest_ssocko_csocko_large_tcpL
    ].

%% Server: transport = socket(tcp), active = once
%% Client: transport = socket(tcp), active = true
ttest_ssocko_csockt_cases() ->
    [
     ttest_ssocko_csockt_small_tcp4,
     ttest_ssocko_csockt_small_tcp6,
     ttest_ssocko_csockt_small_tcpL,

     ttest_ssocko_csockt_medium_tcp4,
     ttest_ssocko_csockt_medium_tcp6,
     ttest_ssocko_csockt_medium_tcpL,

     ttest_ssocko_csockt_large_tcp4,
     ttest_ssocko_csockt_large_tcp6,
     ttest_ssocko_csockt_large_tcpL
    ].

%% Server: transport = socket(tcp), active = true
ttest_ssockt_cases() ->
    [
     {group, ttest_ssockt_cgen},
     {group, ttest_ssockt_csock}
    ].

%% Server: transport = socket(tcp), active = true
%% Client: transport = gen_tcp
ttest_ssockt_cgen_cases() ->
    [
     {group, ttest_ssockt_cgenf},
     {group, ttest_ssockt_cgeno},
     {group, ttest_ssockt_cgent}
    ].

%% Server: transport = socket(tcp), active = true
%% Client: transport = gen_tcp, active = false
ttest_ssockt_cgenf_cases() ->
    [
     ttest_ssockt_cgenf_small_tcp4,
     ttest_ssockt_cgenf_small_tcp6,

     ttest_ssockt_cgenf_medium_tcp4,
     ttest_ssockt_cgenf_medium_tcp6,

     ttest_ssockt_cgenf_large_tcp4,
     ttest_ssockt_cgenf_large_tcp6
    ].

%% Server: transport = socket(tcp), active = true
%% Client: transport = gen_tcp, active = once
ttest_ssockt_cgeno_cases() ->
    [
     ttest_ssockt_cgeno_small_tcp4,
     ttest_ssockt_cgeno_small_tcp6,

     ttest_ssockt_cgeno_medium_tcp4,
     ttest_ssockt_cgeno_medium_tcp6,

     ttest_ssockt_cgeno_large_tcp4,
     ttest_ssockt_cgeno_large_tcp6
    ].

%% Server: transport = socket(tcp), active = true
%% Client: transport = gen_tcp, active = true
ttest_ssockt_cgent_cases() ->
    [
     ttest_ssockt_cgent_small_tcp4,
     ttest_ssockt_cgent_small_tcp6,

     ttest_ssockt_cgent_medium_tcp4,
     ttest_ssockt_cgent_medium_tcp6,

     ttest_ssockt_cgent_large_tcp4,
     ttest_ssockt_cgent_large_tcp6
    ].

%% Server: transport = socket(tcp), active = true
%% Client: transport = socket(tcp)
ttest_ssockt_csock_cases() ->
    [
     {group, ttest_ssockt_csockf},
     {group, ttest_ssockt_csocko},
     {group, ttest_ssockt_csockt}
    ].

%% Server: transport = socket(tcp), active = true
%% Client: transport = socket(tcp), active = false
ttest_ssockt_csockf_cases() ->
    [
     ttest_ssockt_csockf_small_tcp4,
     ttest_ssockt_csockf_small_tcp6,
     ttest_ssockt_csockf_small_tcpL,

     ttest_ssockt_csockf_medium_tcp4,
     ttest_ssockt_csockf_medium_tcp6,
     ttest_ssockt_csockf_medium_tcpL,

     ttest_ssockt_csockf_large_tcp4,
     ttest_ssockt_csockf_large_tcp6,
     ttest_ssockt_csockf_large_tcpL
    ].

%% Server: transport = socket(tcp), active = true
%% Client: transport = socket(tcp), active = once
ttest_ssockt_csocko_cases() ->
    [
     ttest_ssockt_csocko_small_tcp4,
     ttest_ssockt_csocko_small_tcp6,
     ttest_ssockt_csocko_small_tcpL,

     ttest_ssockt_csocko_medium_tcp4,
     ttest_ssockt_csocko_medium_tcp6,
     ttest_ssockt_csocko_medium_tcpL,

     ttest_ssockt_csocko_large_tcp4,
     ttest_ssockt_csocko_large_tcp6,
     ttest_ssockt_csocko_large_tcpL
    ].

%% Server: transport = socket(tcp), active = true
%% Client: transport = socket(tcp), active = true
ttest_ssockt_csockt_cases() ->
    [
     ttest_ssockt_csockt_small_tcp4,
     ttest_ssockt_csockt_small_tcp6,
     ttest_ssockt_csockt_small_tcpL,

     ttest_ssockt_csockt_medium_tcp4,
     ttest_ssockt_csockt_medium_tcp6,
     ttest_ssockt_csockt_medium_tcpL,

     ttest_ssockt_csockt_large_tcp4,
     ttest_ssockt_csockt_large_tcp6,
     ttest_ssockt_csockt_large_tcpL
    ].

tickets_cases() ->
    [
     {group, otp16359}
    ].

otp16359_cases() ->
    [
     otp16359_maccept_tcp4,
     otp16359_maccept_tcp6,
     otp16359_maccept_tcpL
    ].


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

init_per_suite(Config) ->
    ct:timetrap(?MINS(2)),
    Factor = analyze_and_print_host_info(),
    try socket:info() of
        #{} ->
            socket:use_registry(false),
            case quiet_mode(Config) of
                default ->
                    ?LOGGER:start(),
                    [{esock_factor, Factor} | Config];
                Quiet ->
                    ?LOGGER:start(Quiet),
                    [{esock_factor,     Factor},
                     {esock_test_quiet, Quiet} | Config]
            end
    catch
        error : notsup ->
            {skip, "esock not supported"}
    end.

end_per_suite(_) ->
    (catch ?LOGGER:stop()),
    ok.


init_per_group(GroupName, Config)
  when (GroupName =:= sc_remote_close) orelse
       (GroupName =:= sc_remote_shutdown) orelse
       (GroupName =:= traffic) ->
    io:format("init_per_group(~w) -> entry with"
              "~n   Config: ~p"
              "~n", [GroupName, Config]),
    %% Maybe we should skip the entire suite for this platform,
    %% but for now we just skip these groups, which seem to 
    %% have problems (slave node start).
    %% As stated elsewhere, its not really Fedora 16, but 
    %% the *really* slow VM that is the issue.
    try is_old_fedora16() of
        ok ->
            Config
    catch
        throw:{skip, _} = SKIP ->
            SKIP
    end;
init_per_group(ttest = _GroupName, Config) ->
    io:format("init_per_group(~w) -> entry with"
              "~n   Config: ~p"
              "~n", [_GroupName, Config]),
    ttest_manager_start(),
    case lists:keysearch(esock_test_ttest_runtime, 1, Config) of
        {value, _} ->
            Config;
        false ->
            [{esock_test_ttest_runtime, which_ttest_runtime_env()} | Config]
    end;
init_per_group(api_async_ref, Config) ->
    [{select_handle, true} | Config];
init_per_group(_GroupName, Config) ->
    Config.

end_per_group(ttest = _GroupName, Config) ->
    io:format("init_per_group(~w) -> entry with"
              "~n   Config: ~p"
              "~n", [_GroupName, Config]),
    ttest_manager_stop(),
    lists:keydelete(esock_test_ttest_runtime, 1, Config);
end_per_group(api_async_ref, Config) ->
    lists:keydelete(select_handle, 1, Config);
end_per_group(_GroupName, Config) ->
    Config.


init_per_testcase(_TC, Config) ->
    io:format("init_per_testcase(~w) -> entry with"
              "~n   Config: ~p"
              "~n", [_TC, Config]),
    %% case quiet_mode(Config) of
    %%     default ->
    %%         ?LOGGER:start();
    %%     Quiet ->
    %%         ?LOGGER:start(Quiet)
    %% end,
    Config.

end_per_testcase(_TC, Config) ->
    %% ?LOGGER:stop(),
    Config.


quiet_mode(Config) ->
    case lists:keysearch(esock_test_quiet, 1, Config) of
        {value, {esock_test_quiet, Quiet}} ->
            Quiet;
        false ->
            case os:getenv("ESOCK_TEST_QUIET") of
                "true"  -> true;
                "false" -> false;
                _       -> default
            end
    end.


                
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                                                                     %%
%%                           API MISC                                  %%
%%                                                                     %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% This is an extremely rudimentary test case, that just tests
%% that we can call the "global" info function and that it returns
%% a non-empty map...

api_m_info(suite) ->
    [];
api_m_info(doc) ->
    [];
api_m_info(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_m_info,
           fun() ->
                   ok = api_m_info()
           end).

api_m_info() ->
    case socket:info() of
        Info when is_map(Info) ->
            Sz = maps:size(Info),
            if
                (Sz > 0) ->
                    ok;
                true ->
                    ?FAIL(no_info)
            end;
        Info ->
            ?FAIL({invalid_info, Info})
    end.



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% A simple test case that tests that the global debug can be changed.
%% At the same time, it will test the info function (since it uses it
%% for verification).

api_m_debug(suite) ->
    [];
api_m_debug(doc) ->
    [];
api_m_debug(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_m_debug,
           fun() -> has_bugfree_gcc() end,
           fun() ->
                   ok = api_m_debug()
           end).

%% For some reason this test case triggers a gcc bug, which causes
%% a segfault, on an ancient Fedora 16 VM. So, check the version of gcc...
%% Not pretty, but the simplest way to skip (without actually testing
%% for the host).
has_bugfree_gcc() ->
    has_bugfree_gcc(os:type()).

%% Make sure we are on linux
has_bugfree_gcc({unix, linux}) ->
    has_bugfree_gcc2(string:trim(os:cmd("cat /etc/issue")));
has_bugfree_gcc(_) ->
    ok.

%% Make sure we are on Fedora 16
has_bugfree_gcc2("Fedora release 16 " ++ _) ->
    has_bugfree_gcc3(os:cmd("gcc --version"));
has_bugfree_gcc2("Welcome to SUSE Linux " ++ _) ->
    has_bugfree_gcc4(os:cmd("gcc --version"));
has_bugfree_gcc2(_) ->
    ok.

has_bugfree_gcc3("gcc (GCC) 4.6.3 20120306 (Red Hat 4.6.3-2" ++ _) ->
    skip("Buggy GCC");
has_bugfree_gcc3(_) ->
    ok.

has_bugfree_gcc4("gcc (SUSE Linux) 4.3.2" ++ _) ->
    skip("Buggy GCC");
has_bugfree_gcc4(_) ->
    ok.

api_m_debug() ->
    i("get initial info"),
    #{debug := D0} = socket:info(),
    D1 = not D0,
    i("set new debug (~w => ~w)", [D0, D1]),
    ok = socket:debug(D1),
    i("get updated info (~w)", [D1]),
    #{debug := D1} = socket:info(),
    D2 = not D1,
    i("set new debug (~w => ~w)", [D1, D2]),
    ok = socket:debug(D2),
    i("get updated info (~w)", [D2]),
    #{debug := D2} = socket:info(),
    i("ok"),
    ok.
    


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                                                                     %%
%%                           API BASIC                                 %%
%%                                                                     %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and info of an IPv4 UDP (dgram) socket.
%% With some extra checks...
api_b_open_and_info_udp4(suite) ->
    [];
api_b_open_and_info_udp4(doc) ->
    [];
api_b_open_and_info_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_info_udp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => dgram,
                                 protocol => udp},
                   ok = api_b_open_and_info(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and info of an IPv6 UDP (dgram) socket.
%% With some extra checks...
api_b_open_and_info_udp6(suite) ->
    [];
api_b_open_and_info_udp6(doc) ->
    [];
api_b_open_and_info_udp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_info_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet6,
                                 type     => dgram,
                                 protocol => udp},
                   ok = api_b_open_and_info(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and info of an IPv4 TCP (stream) socket.
%% With some extra checks...
api_b_open_and_info_tcp4(suite) ->
    [];
api_b_open_and_info_tcp4(doc) ->
    [];
api_b_open_and_info_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_info_tcp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => stream,
                                 protocol => tcp},
                   ok = api_b_open_and_info(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and info of an IPv6 TCP (stream) socket.
%% With some extra checks...
api_b_open_and_info_tcp6(suite) ->
    [];
api_b_open_and_info_tcp6(doc) ->
    [];
api_b_open_and_info_tcp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_info_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet6,
                                 type     => stream,
                                 protocol => tcp},
                   ok = api_b_open_and_info(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_b_open_and_info(InitState) ->
    Seq = 
        [
         #{desc => "open",
           cmd  => fun(#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol} = State) -> 
                           case socket:open(Domain, Type, Protocol) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "get socket info",
           cmd  => fun(#{sock := Sock} = State) ->
                           Info = socket:info(Sock),
                           ?SEV_IPRINT("Got (some) Info: "
                                       "~n   ~p", [Info]),
                           {ok, State#{info => Info}}
                   end},
         #{desc => "validate socket info",
           cmd  => fun(#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol,
                         info     := #{domain        := Domain,
                                       type          := Type,
                                       protocol      := Protocol,
                                       counters      := _,
                                       num_readers   := 0,
                                       num_writers   := 0,
                                       num_acceptors := 0}}) ->
                           ok;
                      (#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol,
                         info     := Info}) ->
                           ?SEV_EPRINT("Unexpected Info: "
                                       "~n   (expected) Domain:   ~p"
                                       "~n   (expected) Type:     ~p"
                                       "~n   (expected) Protocol: ~p"
                                       "~n   ~p",
                                       [Domain, Type, Protocol, Info]),
                           {error, unexpected_infio}
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = _State) ->
                           socket:close(Sock)
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and close an IPv4 UDP (dgram) socket.
%% With some extra checks...
api_b_open_and_close_udp4(suite) ->
    [];
api_b_open_and_close_udp4(doc) ->
    [];
api_b_open_and_close_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_close_udp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => dgram,
                                 protocol => udp},
                   ok = api_b_open_and_close(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and close an IPv6 UDP (dgram) socket.
%% With some extra checks...
api_b_open_and_close_udp6(suite) ->
    [];
api_b_open_and_close_udp6(doc) ->
    [];
api_b_open_and_close_udp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_close_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet6,
                                 type     => dgram,
                                 protocol => udp},
                   ok = api_b_open_and_close(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and close an IPv4 TCP (stream) socket.
%% With some extra checks...
api_b_open_and_close_tcp4(suite) ->
    [];
api_b_open_and_close_tcp4(doc) ->
    [];
api_b_open_and_close_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_close_tcp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => stream,
                                 protocol => tcp},
                   ok = api_b_open_and_close(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and close an IPv6 TCP (stream) socket.
%% With some extra checks...
api_b_open_and_close_tcp6(suite) ->
    [];
api_b_open_and_close_tcp6(doc) ->
    [];
api_b_open_and_close_tcp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_close_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => stream,
                                 protocol => tcp},
                   ok = api_b_open_and_close(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and close an Unix Domain dgram (UDP) socket.
%% With some extra checks...
api_b_open_and_close_udpL(suite) ->
    [];
api_b_open_and_close_udpL(doc) ->
    [];
api_b_open_and_close_udpL(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_close_udpL,
           fun() -> has_support_unix_domain_socket() end,
           fun() ->
                   InitState = #{domain   => local,
                                 type     => dgram,
                                 protocol => default},
                   ok = api_b_open_and_close(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and close an Unix Domain stream (TCP) socket.
%% With some extra checks...
api_b_open_and_close_tcpL(suite) ->
    [];
api_b_open_and_close_tcpL(doc) ->
    [];
api_b_open_and_close_tcpL(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_close_tcpL,
           fun() -> has_support_unix_domain_socket() end,
           fun() ->
                   InitState = #{domain   => local,
                                 type     => stream,
                                 protocol => default},
                   ok = api_b_open_and_close(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and close an Unix Domain dgram (UDP) socket.
%% With some extra checks...
api_b_open_and_close_seqpL(suite) ->
    [];
api_b_open_and_close_seqpL(doc) ->
    [];
api_b_open_and_close_seqpL(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(?FUNCTION_NAME,
           fun() -> has_support_unix_domain_socket() end,
           fun() ->
                   InitState = #{domain   => local,
                                 type     => seqpacket,
                                 protocol => default},
                   ok = api_b_open_and_close(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and close an IPv4 SCTP (seqpacket) socket.
%% With some extra checks...
api_b_open_and_close_sctp4(suite) ->
    [];
api_b_open_and_close_sctp4(doc) ->
    [];
api_b_open_and_close_sctp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_close_sctp4,
           fun() -> has_support_sctp() end,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => seqpacket,
                                 protocol => sctp},
                   ok = api_b_open_and_close(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_b_open_and_close(InitState) ->
    Seq = 
        [
         #{desc => "open",
           cmd  => fun(#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol} = S) -> 
                           Res = socket:open(Domain, Type, Protocol), 
                           {ok, {S, Res}} 
                   end},
         #{desc => "validate open",
           cmd  => fun({S, {ok, Sock}}) -> 
                           NewS = S#{socket => Sock},
                           {ok, NewS};
                      ({_, {error, epfnosupport = Reason}}) ->
                           {skip, Reason};
                      ({_, {error, eprotonosupport = Reason}}) ->
                           {skip, Reason};
                      ({_, {error, esocktnosupport = Reason}}) ->
                           {skip, Reason};
                      ({_, {error, _} = ERROR}) ->
                           ERROR
                   end},
         #{desc => "get domain (maybe)",
           cmd  => fun(#{socket := Sock} = S) ->
                           Res = socket:getopt(Sock, socket, domain),
                           {ok, {S, Res}}
                   end},
         #{desc => "validate domain (maybe)",
           cmd  => fun({#{domain := Domain} = S, {ok, Domain}}) -> 
                           {ok, S};
                      ({#{domain := ExpDomain}, {ok, Domain}}) ->
                           {error, {unexpected_domain, ExpDomain, Domain}};
                      %% Some platforms do not support this option
                      ({S, {error, {invalid, _}} = ERROR}) ->
                           case
                               socket:is_supported(options, socket, domain)
                           of
                               true ->
                                   ERROR;
                               false ->
                                   ?SEV_IPRINT("socket option 'domain' "
                                               "not supported"),
                                   {ok, S}
                           end;
                      ({_, {error, _} = ERROR}) ->
                           ERROR
                   end},
         #{desc => "get type",
           cmd  => fun(#{socket := Sock} = State) ->
                           Res = socket:getopt(Sock, socket, type), 
                           {ok, {State, Res}}
                   end},
         #{desc => "validate type",
           cmd  => fun({#{type := Type} = State, {ok, Type}}) ->
                           {ok, State};
                      ({#{type := ExpType}, {ok, Type}}) ->
                           {error, {unexpected_type, ExpType, Type}};
                      ({_, {error, _} = ERROR}) ->
                           ERROR
                   end},
         #{desc => "get protocol (maybe)",
           cmd  => fun(#{socket := Sock} = State) ->
                           Res = socket:getopt(Sock, socket, protocol),
                           {ok, {State, Res}}
                   end},
         #{desc => "validate protocol",
           cmd  => fun({#{protocol := Protocol} = State, {ok, Protocol}}) ->
                           {ok, State};
                      ({#{domain   := Domain,
			  protocol := ExpProtocol}, {ok, Protocol}}) ->
			   %% On OpenBSD (at least 6.6) something screwy happens
			   %% when domain = local.
			   %% It will report a completly different protocol (icmp)
			   %% but everything still works. So we skip if this happens
			   %% on OpenBSD...
			   case os:type() of
			       {unix, openbsd} when (Domain =:= local) ->
				   {skip, ?F("Unexpected protocol: ~p instead of ~p",
					     [Protocol, ExpProtocol])};
			       _ ->
				   {error, {unexpected_protocol,
					    ExpProtocol, Protocol}}
			   end;
                      %% Some platforms do not support this option
                      ({State, {error, {invalid, _}} = ERROR}) ->
			   case socket:is_supported(options, socket, protocol) of
			       true ->
                                   ERROR;
			       false ->
                                   ?SEV_IPRINT("socket option 'protocol' "
                                               "not supported"),
                                   {ok, State}
			   end;
                      ({_, {error, _} = ERROR}) ->
                           ERROR
                   end},
         #{desc => "get controlling-process",
           cmd  => fun(#{socket := Sock} = State) ->
                           Res = socket:getopt(Sock, otp, controlling_process),
                           {ok, {State, Res}}
                   end},
         #{desc => "validate controlling-process",
           cmd  => fun({State, {ok, Pid}}) ->
                           case self() of
                               Pid ->
                                   {ok, State};
                               _ ->
                                   {error, {unexpected_owner, Pid}}
                           end;
                      ({_, {error, _} = ERROR}) ->
                           ERROR
                   end},
         #{desc => "close socket",
           cmd  => fun(#{socket := Sock} = State) ->
                           Res = socket:close(Sock),
                           {ok, {State, Res}}
                   end},
         #{desc => "validate socket close",
           cmd  => fun({_, ok}) ->
                           ok;
                      ({_, {error, _} = ERROR}) ->
                           ERROR
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) and (maybe) close an RAW socket.

api_b_open_and_maybe_close_raw(suite) ->
    [];
api_b_open_and_maybe_close_raw(doc) ->
    [];
api_b_open_and_maybe_close_raw(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_open_and_maybe_close_raw,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => raw,
                                 protocol => {raw, 255}},
                   ok = do_api_b_open_and_maybe_close_raw(InitState)
           end).

do_api_b_open_and_maybe_close_raw(InitState) ->
    Tester = 
        [
         #{desc => "open",
           cmd  => fun(#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol} = State) -> 
                           ?SEV_IPRINT("try open with:"
                                       "~n   Domain:   ~p"
                                       "~n   Type:     ~p"
                                       "~n   Protocol: ~p",
                                       [Domain, Type, Protocol]),
                           case socket:open(Domain, Type, Protocol) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, Reason} when (Reason =:= eperm) orelse
                                                    (Reason =:= eacces) ->
                                   ?SEV_IPRINT("not allowed (~w) => SKIP",
                                               [Reason]),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("open failed:"
                                               "~n   Reason: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{socket := Sock} = State) ->
                           ?SEV_IPRINT("try socket close"),
                           case socket:close(Sock) of
                               ok ->
                                   ?SEV_IPRINT("socket closed"),
                                   {ok, maps:remote(sock, State)};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("close failed:"
                                               "~n   Reason: ~p", [Reason]),
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Tester, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive on an IPv4 UDP (dgram) socket using
%% sendto and recvfrom..
api_b_sendto_and_recvfrom_udp4(suite) ->
    [];
api_b_sendto_and_recvfrom_udp4(doc) ->
    [];
api_b_sendto_and_recvfrom_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_sendto_and_recvfrom_udp4,
           fun() ->
                   Send = fun(Sock, Data, Dest) ->
                                  socket:sendto(Sock, Data, Dest)
                          end,
                   Recv = fun(Sock) ->
                                  socket:recvfrom(Sock)
                          end,
                   InitState = #{domain => inet,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv},
                   ok = api_b_send_and_recv_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive on an IPv4 UDP (dgram) socket using
%% sendto and recvfrom.
api_b_sendto_and_recvfrom_udpL(suite) ->
    [];
api_b_sendto_and_recvfrom_udpL(doc) ->
    [];
api_b_sendto_and_recvfrom_udpL(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_sendto_and_recvfrom_udpL,
           fun() ->
                   has_support_unix_domain_socket(),
                   unix_domain_socket_host_cond()
           end,
           fun() ->
                   Send = fun(Sock, Data, Dest) ->
                                  socket:sendto(Sock, Data, Dest)
                          end,
                   Recv = fun(Sock) ->
                                  socket:recvfrom(Sock)
                          end,
                   InitState = #{domain => local,
                                 proto  => default,
                                 send   => Send,
                                 recv   => Recv},
                   ok = api_b_send_and_recv_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive on an IPv4 UDP (dgram) socket
%% using sendmsg and recvmsg.
api_b_sendmsg_and_recvmsg_udp4(suite) ->
    [];
api_b_sendmsg_and_recvmsg_udp4(doc) ->
    [];
api_b_sendmsg_and_recvmsg_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_sendmsg_and_recvmsg_udp4,
           fun() ->
                   Send = fun(Sock, Data, Dest) ->
                                  %% We need tests for this,
                                  %% but this is not the place it.
                                  %% CMsg  = #{level => ip,
                                  %%              type  => tos,
                                  %%              data  => reliability},
                                  %% CMsgs = [CMsg],
                                  Msg = #{addr => Dest,
                                             %% ctrl => CMsgs,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  %% We have some issues on old darwin...
                                  %% ok = socket:setopt(Sock, {otp,debug}, true),
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr  := Source,
                                             iov   := [Data]}} ->
                                          %% socket:setopt(Sock, otp, debug, false),
                                          {ok, {Source, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv},
                   ok = api_b_send_and_recv_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive on an IPv4 UDP (dgram) socket
%% using sendmsg and recvmsg.
api_b_sendmsg_and_recvmsg_udpL(suite) ->
    [];
api_b_sendmsg_and_recvmsg_udpL(doc) ->
    [];
api_b_sendmsg_and_recvmsg_udpL(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_sendmsg_and_recvmsg_udpL,
           fun() ->
                   has_support_unix_domain_socket(),
                   unix_domain_socket_host_cond()
           end,
           fun() ->
                   Send = fun(Sock, Data, Dest) ->
                                  %% We need tests for this,
                                  %% but this is not the place it.
                                  %% CMsg  = #{level => ip,
                                  %%              type  => tos,
                                  %%              data  => reliability},
                                  %% CMsgs = [CMsg],
                                  Msg = #{addr => Dest,
                                             %% ctrl => CMsgs,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  %% We have some issues on old darwing...
                                  %% socket:setopt(Sock, otp, debug, true),
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr  := Source,
                                             iov   := [Data]}} ->
                                          %% socket:setopt(Sock, otp, debug, false),
                                          {ok, {Source, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => local,
                                 proto  => default,
                                 send   => Send,
                                 recv   => Recv},
                   ok = api_b_send_and_recv_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_b_send_and_recv_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := local = Domain} = State) ->
                           LSASrc = which_local_socket_addr(Domain),
                           LSADst = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSASrc,
                                       lsa_dst => LSADst}};
                      (#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src socket",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   ?SEV_IPRINT("src bound (to ~p)", [Port]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst socket",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   ?SEV_IPRINT("src bound (to ~p)", [Port]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},
         #{desc => "send req (to dst)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, ?BASIC_REQ}} ->
                                   ok;
                               {ok, UnexpData} ->
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},
         #{desc => "send rep (to src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, send := Send}) ->
                           Send(Sock, ?BASIC_REP, Src)
                   end},
         #{desc => "recv rep (from dst)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Dst, ?BASIC_REP}} ->
                                   ok;
                               {ok, UnexpData} ->
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},
         #{desc => "close src socket",
           cmd  => fun(#{domain   := local,
                         sock_src := Sock,
                         lsa_src  := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() -> maps:remove(lsa_src, State) end,
                                           fun() -> State end),
                           {ok, maps:remove(sock_src, State1)};
                      (#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{domain   := local,
                         sock_dst := Sock,
                         lsa_dst  := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() -> maps:remove(lsa_dst, State) end,
                                           fun() -> State end),
                           {ok, maps:remove(sock_dst, State1)};
                      (#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive using the "common" functions (send and recv)
%% on an IPv4 TCP (stream) socket.
api_b_send_and_recv_tcp4(suite) ->
    [];
api_b_send_and_recv_tcp4(doc) ->
    [];
api_b_send_and_recv_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(api_b_send_and_recv_tcp4,
           fun() ->
                   Send = fun(Sock, Data) ->
                                  socket:send(Sock, Data)
                          end,
                   Recv = fun(Sock) ->
                                  socket:recv(Sock)
                          end,
                   InitState = #{domain => inet,
                                 type   => stream,
                                 proto  => tcp,
                                 send   => Send,
                                 recv   => Recv},
                   ok = api_b_send_and_recv_conn(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive using the "common" functions (send and recv)
%% on an Unix Domain (stream) socket (TCP).
api_b_send_and_recv_tcpL(suite) ->
    [];
api_b_send_and_recv_tcpL(doc) ->
    [];
api_b_send_and_recv_tcpL(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(api_b_send_and_recv_tcpL,
           fun() -> has_support_unix_domain_socket() end,
           fun() ->
                   Send = fun(Sock, Data) ->
                                  socket:send(Sock, Data)
                          end,
                   Recv = fun(Sock) ->
                                  socket:recv(Sock)
                          end,
                   InitState = #{domain => local,
                                 type   => stream,
                                 proto  => default,
                                 send   => Send,
                                 recv   => Recv},
                   ok = api_b_send_and_recv_conn(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive using the "common" functions (send and recv)
%% on an Unix Domain seqpacket socket.
api_b_send_and_recv_seqpL(suite) ->
    [];
api_b_send_and_recv_seqpL(doc) ->
    [];
api_b_send_and_recv_seqpL(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(?FUNCTION_NAME,
           fun() -> has_support_unix_domain_socket() end,
           fun() ->
                   Send = fun(Sock, Data) ->
                                  socket:send(Sock, Data)
                          end,
                   Recv = fun(Sock) ->
                                  socket:recv(Sock)
                          end,
                   InitState = #{domain => local,
                                 type   => seqpacket,
                                 proto  => default,
                                 send   => Send,
                                 recv   => Recv},
                   ok = api_b_send_and_recv_conn(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive using the msg functions (sendmsg and recvmsg)
%% on an IPv4 TCP (stream) socket.
api_b_sendmsg_and_recvmsg_tcp4(suite) ->
    [];
api_b_sendmsg_and_recvmsg_tcp4(doc) ->
    [];
api_b_sendmsg_and_recvmsg_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(api_b_sendmsg_and_recvmsg_tcp4,
           fun() ->
                   Send = fun(Sock, Data) ->
                                  Msg = #{iov => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr  := _} = Addr} ->
                                          {error, {addr, Addr}};
                                      {ok, #{iov   := [Data]}} ->
                                          {ok, Data};
                                      {ok, Msg} ->
                                          {error, {msg, Msg}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 type   => stream,
                                 proto  => tcp,
                                 send   => Send,
                                 recv   => Recv},
                   ok = api_b_send_and_recv_conn(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive using the msg functions (sendmsg and recvmsg)
%% on an Unix Domain (stream) socket (TCP).
api_b_sendmsg_and_recvmsg_tcpL(suite) ->
    [];
api_b_sendmsg_and_recvmsg_tcpL(doc) ->
    [];
api_b_sendmsg_and_recvmsg_tcpL(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(api_b_sendmsg_and_recvmsg_tcpL,
           fun() -> has_support_unix_domain_socket() end,
           fun() ->
                   Send = fun(Sock, Data) ->
                                  Msg = #{iov => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      %% On some platforms, the address
                                      %% *is* provided (e.g. linux)
                                      {ok, #{addr  := #{family := local},
                                             iov   := [Data]}} ->
                                          socket:setopt(Sock, 
                                                        otp, 
                                                        debug, 
                                                        false),
                                          {ok, Data};
                                      {ok, #{addr := _} = Msg} ->
                                          {error, {msg, Msg}};
                                      %% On some platforms, the address
                                      %% is *not* provided (e.g. FreeBSD)
                                      {ok, #{iov   := [Data]}} ->
                                          {ok, Data};
                                      {ok, Msg} ->
                                          {error, {msg, Msg}};
                                      {error, _} = ERROR ->
                                          socket:setopt(Sock, 
                                                        otp, 
                                                        debug, 
                                                        false),
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => local,
                                 type   => stream,
                                 proto  => default,
                                 send   => Send,
                                 recv   => Recv},
                   ok = api_b_send_and_recv_conn(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive using the msg functions (sendmsg and recvmsg)
%% on an Unix Domain (stream) socket (TCP).
api_b_sendmsg_and_recvmsg_seqpL(suite) ->
    [];
api_b_sendmsg_and_recvmsg_seqpL(doc) ->
    [];
api_b_sendmsg_and_recvmsg_seqpL(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(?FUNCTION_NAME,
           fun() -> has_support_unix_domain_socket() end,
           fun() ->
                   Send =
                       fun(Sock, Data) ->
                               Msg =
                                   #{iov => [Data]},
                               socket:sendmsg(Sock, Msg)
                       end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      %% On some platforms, the address
                                      %% *is* provided (e.g. linux)
                                      {ok,
                                       #{addr  := #{family := local},
                                         iov   := [Data]}} ->
                                          socket:setopt(
                                            Sock, otp, debug, false),
                                          {ok, Data};
                                      {ok, #{addr := _} = Msg} ->
                                          {error, {msg, Msg}};
                                      %% On some platforms, the address
                                      %% is *not* provided (e.g. FreeBSD)
                                      {ok,
                                       #{iov   := [Data]}} ->
                                          {ok, Data};
                                      {ok, Msg} ->
                                          {error, {msg, Msg}};
                                      {error, _} = ERROR ->
                                          socket:setopt(
                                            Sock, otp, debug, false),
                                          ERROR
                                  end
                          end,
                   InitState =
                       #{domain => local,
                         type   => seqpacket,
                         proto  => default,
                         send   => Send,
                         recv   => Recv},
                   ok = api_b_send_and_recv_conn(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_b_send_and_recv_conn(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain,
                         type := Type,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, Type, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, eprotonosupport = Reason} ->
                                   {skip, Reason};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{domain := local,
                         lsock  := LSock,
                         lsa    := LSA} = _State) ->
                           case socket:bind(LSock, LSA) of
                               ok ->
                                   %% We do not care about the port for local
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end;
                      (#{lsock := LSock, lsa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   ?SEV_IPRINT("bound to port: ~w", [Port]),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{domain := local,
                         tester := Tester, lsa := #{path := Path}}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Path),
                           ok;
                      (#{tester := Tester, lport := Port}) ->
                           %% This is actually not used for unix domain socket
                           ?SEV_ANNOUNCE_READY(Tester, init, Port),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:accept(LSock) of
                               {ok, Sock} ->
                                   ?SEV_IPRINT("accepted: ~n   ~p", [Sock]),
                                   {ok, State#{csock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept),
                           ok
                   end},
         #{desc => "await (recv) request",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, ?BASIC_REQ} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready (send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close connection socket",
           cmd  => fun(#{csock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(csock, State)}
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{domain   := local,
                         lsock    := Sock,
                         local_sa := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() ->
                                                   maps:remove(local_sa, State)
                                           end,
                                           fun() -> State end),
                           {ok, maps:remove(lsock, State1)};
                      (#{lsock := LSock} = State) ->
                           case socket:close(LSock) of
                               ok ->
                                   {ok, maps:remove(lsock, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    ClientSeq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(#{domain := local} = State) ->
                           {Tester, Path} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_path => Path}};
                      (State) ->
                           {Tester, Port} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_port => Port}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain      := local = Domain,
                         server_path := Path} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = #{family => Domain, path => Path},
                           {ok, State#{local_sa => LSA, server_sa => SSA}};
                      (#{domain := Domain, server_port := Port} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = LSA#{port => Port},
                           {ok, State#{local_sa => LSA, server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain,
                         type := Type,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, Type, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "await continue (connect)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, connect)
                   end},
         #{desc => "connect to server",
           cmd  => fun(#{sock := Sock, server_sa := SSA}) ->
                           socket:connect(Sock, SSA)
                   end},
         #{desc => "announce ready (connect)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect),
                           ok
                   end},
         #{desc => "await continue (send request)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply (from server)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           {ok, ?BASIC_REP} = Recv(Sock),
                           ok
                   end},
         #{desc => "announce ready (recv reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{domain   := local,
                         sock     := Sock,
                         local_sa := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() ->
                                                   maps:remove(local_sa, State)
                                           end,
                                           fun() -> State end),
                           {ok, maps:remove(sock, State1)};
                      (#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% Start the client
         #{desc => "order client start",
           cmd  => fun(#{client := Pid, server_port := Port} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Port),
                           ok
                   end},
         #{desc => "await client ready (init)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, init)
                   end},

         %% *** The actual test ***
         #{desc => "order server to continue (with accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
         ?SEV_SLEEP(?SECS(1)),
         #{desc => "order client to continue (with connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, connect),
                           ok
                   end},
         #{desc => "await client ready (connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, connect)
                   end},
         #{desc => "await server ready (accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, accept)
                   end},
         #{desc => "order client to continue (with send request)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client ready (with send request)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send_req)
                   end},
         #{desc => "await server ready (request recv)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply sent)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client ready (reply recv)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, recv_reply)
                   end},


         %% *** Termination ***
         #{desc => "order client to terminate",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client termination",
           cmd  => fun(#{client := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client, State),
                           {ok, State1}
                   end},
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start client evaluator"),
    Client = ?SEV_START("client", ClientSeq, InitState),
    i("await evaluator(s)"),

    i("start tester evaluator"),
    TesterInitState = #{server => Server#ev.pid,
                        client => Client#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive on an IPv4 SCTP (seqpacket) socket
%% using sendmsg and recvmsg.
api_b_sendmsg_and_recvmsg_sctp4(suite) ->
    [];
api_b_sendmsg_and_recvmsg_sctp4(doc) ->
    [];
api_b_sendmsg_and_recvmsg_sctp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_b_sendmsg_and_recvmsg_sctp4,
           fun() ->
		   has_support_sctp(),
		   not_yet_implemented()
	   end,
           fun() ->
                   Send = fun(Sock, Data) ->
                                  %% CMsg  = #{level => sctp,
                                  %%              type  => tos,
                                  %%              data  => reliability},
                                  %% CMsgs = [CMsg],
                                  Msg = #{%% ctrl => CMsgs,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  %% We have some issues on old darwing...
                                  %% socket:setopt(Sock, otp, debug, true),
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr  := Source,
                                             iov   := [Data]}} ->
                                          %% socket:setopt(Sock, otp, debug, false),
                                          {ok, {Source, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 proto  => sctp,
                                 send   => Send,
                                 recv   => Recv},
                   ok = api_b_send_and_recv_sctp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_b_send_and_recv_sctp(_InitState) ->
%%     Seq = 
%%         [
%%          #{desc => "local address",
%%            cmd  => fun(#{domain := Domain} = State) ->
%%                            LSA = which_local_socket_addr(Domain),
%%                            {ok, State#{lsa_src => LSA,
%%                                        lsa_dst => LSA}}
%%                    end},

%%          #{desc => "open src socket",
%%            cmd  => fun(#{domain := Domain,
%%                          proto  := Proto} = State) ->
%%                            Sock = sock_open(Domain, seqpacket, Proto),
%%                            {ok, State#{sock_src => Sock}}
%%                    end},
%%          #{desc => "bind src",
%%            cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
%%                            case socket:bind(Sock, LSA) of
%%                                ok ->
%%                                    ?SEV_IPRINT("src bound"),
%%                                    ok;
%%                                {error, Reason} = ERROR ->
%%                                    ?SEV_EPRINT("src bind failed: ~p", [Reason]),
%%                                    ERROR
%%                            end
%%                    end},
%%          #{desc => "sockname src socket",
%%            cmd  => fun(#{sock_src := Sock} = State) ->
%%                            SASrc = sock_sockname(Sock),
%%                            ?SEV_IPRINT("src sockaddr: "
%%                                        "~n   ~p", [SASrc]),
%%                            {ok, State#{sa_src => SASrc}}
%%                    end},

%%          #{desc => "open dst socket",
%%            cmd  => fun(#{domain := Domain,
%%                          proto  := Proto} = State) ->
%%                            Sock = sock_open(Domain, seqpacket, Proto),
%%                            {ok, State#{sock_dst => Sock}}
%%                    end},
%%          #{desc => "bind dst",
%%            cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
%%                            case socket:bind(Sock, LSA) of
%%                                ok ->
%%                                    ?SEV_IPRINT("src bound"),
%%                                    ok;
%%                                {error, Reason} = ERROR ->
%%                                    ?SEV_EPRINT("src bind failed: ~p", [Reason]),
%%                                    ERROR
%%                            end
%%                    end},
%%          #{desc => "sockname dst socket",
%%            cmd  => fun(#{sock_dst := Sock} = State) ->
%%                            SADst = sock_sockname(Sock),
%%                            ?SEV_IPRINT("dst sockaddr: "
%%                                        "~n   ~p", [SADst]),
%%                            {ok, State#{sa_dst => SADst}}
%%                    end},
%%          #{desc => "send req (to dst)",
%%            cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
%%                            Send(Sock, ?BASIC_REQ, Dst)
%%                    end},
%%          #{desc => "recv req (from src)",
%%            cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
%%                            case Recv(Sock) of
%%                                {ok, {Src, ?BASIC_REQ}} ->
%%                                    ok;
%%                                {ok, UnexpData} ->
%%                                    {error, {unexpected_data, UnexpData}};
%%                                {error, _} = ERROR ->
%%                                    %% At the moment there is no way to get
%%                                    %% status or state for the socket...
%%                                    ERROR
%%                            end
%%                    end},
%%          #{desc => "send rep (to src)",
%%            cmd  => fun(#{sock_dst := Sock, sa_src := Src, send := Send}) ->
%%                            Send(Sock, ?BASIC_REP, Src)
%%                    end},
%%          #{desc => "recv rep (from dst)",
%%            cmd  => fun(#{sock_src := Sock, sa_dst := Dst, recv := Recv}) ->
%%                            case Recv(Sock) of
%%                                {ok, {Dst, ?BASIC_REP}} ->
%%                                    ok;
%%                                {ok, UnexpData} ->
%%                                    {error, {unexpected_data, UnexpData}};
%%                                {error, _} = ERROR ->
%%                                    %% At the moment there is no way to get
%%                                    %% status or state for the socket...
%%                                    ERROR
%%                            end
%%                    end},
%%          #{desc => "close src socket",
%%            cmd  => fun(#{sock_src := Sock} = State) ->
%%                            ok = socket:close(Sock),
%%                            {ok, maps:remove(sock_src, State)}
%%                    end},
%%          #{desc => "close dst socket",
%%            cmd  => fun(#{sock_dst := Sock} = State) ->
%%                            ok = socket:close(Sock),
%%                            {ok, maps:remove(sock_dst, State)}
%%                    end},

%%          %% *** We are done ***
%%          ?SEV_FINISH_NORMAL
%%         ],
%%     Evaluator = ?SEV_START("tester", Seq, InitState),
%%     ok = ?SEV_AWAIT_FINISH([Evaluator]).

%%     process_flag(trap_exit, true),
%%     ServerSeq = 
%%         [
%%          %% *** Wait for start order ***
%%          #{desc => "await start (from tester)",
%%            cmd  => fun(State) ->
%%                            Tester = ?SEV_AWAIT_START(),
%%                            {ok, State#{tester => Tester}}
%%                    end},
%%          #{desc => "monitor tester",
%%            cmd  => fun(#{tester := Tester}) ->
%%                            _MRef = erlang:monitor(process, Tester),
%%                            ok
%%                    end},

%%          %% *** Init part ***
%%          #{desc => "which local address",
%%            cmd  => fun(#{domain := Domain} = State) ->
%%                            LSA = which_local_socket_addr(Domain),
%%                            {ok, State#{lsa => LSA}}
%%                    end},
%%          #{desc => "create (listen) socket",
%%            cmd  => fun(#{domain := Domain,
%%                          proto  := Proto} = State) ->
%%                            case socket:open(Domain, seqpacket, Proto) of
%%                                {ok, Sock} ->
%%                                    {ok, State#{lsock => Sock}};
%%                                {error, _} = ERROR ->
%%                                    ERROR
%%                            end
%%                    end},
%%          #{desc => "bind to local address",
%%            cmd  => fun(#{lsock := LSock, lsa := LSA} = State) ->
%%                            case socket:bind(LSock, LSA) of
%%                                ok ->
%%                                    Port = sock_port(LSock),
%%                                    ?SEV_IPRINT("bound to port: ~w", [Port]),
%%                                    {ok, State#{lport => Port}};
%%                                {error, _} = ERROR ->
%%                                    ERROR
%%                            end
%%                    end},
%%          #{desc => "make listen socket",
%%            cmd  => fun(#{lsock := LSock}) ->
%%                            socket:listen(LSock)
%%                    end},
%%          #{desc => "announce ready (init)",
%%            cmd  => fun(#{tester := Tester, lport := Port}) ->
%%                            ?SEV_ANNOUNCE_READY(Tester, init, Port),
%%                            ok
%%                    end},

%%          %% The actual test
%%          #{desc => "await continue (accept)",
%%            cmd  => fun(#{tester := Tester}) ->
%%                            ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
%%                    end},
%%          #{desc => "accepting (await connection) FAKE",
%%            cmd  => fun(#{lsock := LSock} = State) ->
%% 			   {ok, State#{csock => LSock}}
%%                    end},
%% %%          #{desc => "accepting (await connection)",
%% %%            cmd  => fun(#{lsock := LSock} = State) ->
%% %% 			   socket:setopt(LSock, otp, debug, true),
%% %%                            case socket:accept(LSock) of
%% %%                                {ok, Sock} ->
%% %% 				   socket:setopt(LSock, otp, debug, false),
%% %%                                    ?SEV_IPRINT("accepted: ~n   ~p", [Sock]),
%% %%                                    {ok, State#{csock => Sock}};
%% %%                                {error, Reason} = ERROR ->
%% %% 				   socket:setopt(LSock, otp, debug, false),
%% %% 				   ?SEV_EPRINT("Failed accepting: "
%% %% 					       "~n   ~p", [Reason]),
%% %%                                    ERROR
%% %%                            end
%% %%                    end},
%%          #{desc => "announce ready (accept)",
%%            cmd  => fun(#{tester := Tester}) ->
%%                            ?SEV_ANNOUNCE_READY(Tester, accept),
%%                            ok
%%                    end},
%%          #{desc => "await (recv) request",
%%            cmd  => fun(#{csock := Sock, recv := Recv}) ->
%%                            case Recv(Sock) of
%%                                {ok, ?BASIC_REQ} ->
%%                                    ok;
%%                                {error, _} = ERROR ->
%%                                    ERROR
%%                            end
%%                    end},
%%          #{desc => "announce ready (recv request)",
%%            cmd  => fun(#{tester := Tester}) ->
%%                            ?SEV_ANNOUNCE_READY(Tester, recv_req),
%%                            ok
%%                    end},
%%          #{desc => "await continue (with send reply)",
%%            cmd  => fun(#{tester := Tester}) ->
%%                            ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
%%                    end},
%%          #{desc => "send reply",
%%            cmd  => fun(#{csock := Sock, send := Send}) ->
%%                            Send(Sock, ?BASIC_REP)
%%                    end},
%%          #{desc => "announce ready (send reply)",
%%            cmd  => fun(#{tester := Tester}) ->
%%                            ?SEV_ANNOUNCE_READY(Tester, send_reply),
%%                            ok
%%                    end},

%%          %% *** Termination ***
%%          #{desc => "await terminate",
%%            cmd  => fun(#{tester := Tester} = State) ->
%%                            case ?SEV_AWAIT_TERMINATE(Tester, tester) of
%%                                ok ->
%%                                    {ok, maps:remove(tester, State)};
%%                                {error, _} = ERROR ->
%%                                    ERROR
%%                            end
%%                    end},
%%          #{desc => "close connection socket",
%%            cmd  => fun(#{csock := Sock} = State) ->
%%                            ok = socket:close(Sock),
%%                            {ok, maps:remove(csock, State)}
%%                    end},
%%          #{desc => "close listen socket",
%%            cmd  => fun(#{lsock := LSock} = State) ->
%%                            case socket:close(LSock) of
%%                                ok ->
%%                                    {ok, maps:remove(lsock, State)};
%%                                {error, _} = ERROR ->
%%                                    ERROR
%%                            end
%%                    end},

%%          %% *** We are done ***
%%          ?SEV_FINISH_NORMAL
%%         ],

%%     ClientSeq = 
%%         [
%%          %% *** Wait for start order ***
%%          #{desc => "await start (from tester)",
%%            cmd  => fun(State) ->
%%                            {Tester, Port} = ?SEV_AWAIT_START(),
%%                            {ok, State#{tester => Tester, server_port => Port}}
%%                    end},
%%          #{desc => "monitor tester",
%%            cmd  => fun(#{tester := Tester}) ->
%%                            _MRef = erlang:monitor(process, Tester),
%%                            ok
%%                    end},

%%          %% *** The init part ***
%%          #{desc => "which server (local) address",
%%            cmd  => fun(#{domain := Domain, server_port := Port} = State) ->
%%                            LSA = which_local_socket_addr(Domain),
%%                            SSA = LSA#{port => Port},
%%                            {ok, State#{local_sa => LSA, server_sa => SSA}}
%%                    end},
%%          #{desc => "create socket",
%%            cmd  => fun(#{domain := Domain,
%%                          proto  := Proto} = State) ->
%%                            case socket:open(Domain, seqpacket, Proto) of
%%                                {ok, Sock} ->
%%                                    {ok, State#{sock => Sock}};
%%                                {error, _} = ERROR ->
%%                                    ERROR
%%                            end
%%                    end},
%%          #{desc => "bind to local address",
%%            cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
%%                            case socket:bind(Sock, LSA) of
%%                                ok ->
%%                                    ok;
%%                                {error, _} = ERROR ->
%%                                    ERROR
%%                            end
%%                    end},
%%          #{desc => "announce ready (init)",
%%            cmd  => fun(#{tester := Tester}) ->
%%                            ?SEV_ANNOUNCE_READY(Tester, init),
%%                            ok
%%                    end},

%%          %% *** The actual test ***
%%          #{desc => "await continue (connect)",
%%            cmd  => fun(#{tester := Tester} = _State) ->
%%                            ?SEV_AWAIT_CONTINUE(Tester, tester, connect)
%%                    end},
%%          #{desc => "connect to server",
%%            cmd  => fun(#{sock := Sock, server_sa := SSA}) ->
%%                            case socket:connect(Sock, SSA) of
%% 			       ok ->
%% 				   ?SEV_IPRINT("connected"),
%% 				   ok;
%% 			       {error, Reason} = ERROR ->
%% 				   ?SEV_EPRINT("Failed connect: "
%% 					       "~n   ~p", [Reason]),
%% 				   ERROR
%% 			   end
%%                    end},
%%          #{desc => "announce ready (connect)",
%%            cmd  => fun(#{tester := Tester}) ->
%%                            ?SEV_ANNOUNCE_READY(Tester, connect),
%%                            ok
%%                    end},
%%          #{desc => "await continue (send request)",
%%            cmd  => fun(#{tester := Tester} = _State) ->
%%                            ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
%%                    end},
%%          #{desc => "send request (to server)",
%%            cmd  => fun(#{sock := Sock, send := Send}) ->
%%                            Send(Sock, ?BASIC_REQ)
%%                    end},
%%          #{desc => "announce ready (send request)",
%%            cmd  => fun(#{tester := Tester}) ->
%%                            ?SEV_ANNOUNCE_READY(Tester, send_req),
%%                            ok
%%                    end},
%%          #{desc => "await recv reply (from server)",
%%            cmd  => fun(#{sock := Sock, recv := Recv}) ->
%%                            {ok, ?BASIC_REP} = Recv(Sock),
%%                            ok
%%                    end},
%%          #{desc => "announce ready (recv reply)",
%%            cmd  => fun(#{tester := Tester}) ->
%%                            ?SEV_ANNOUNCE_READY(Tester, recv_reply),
%%                            ok
%%                    end},

%%          %% *** Termination ***
%%          #{desc => "await terminate",
%%            cmd  => fun(#{tester := Tester} = State) ->
%%                            case ?SEV_AWAIT_TERMINATE(Tester, tester) of
%%                                ok ->
%%                                    {ok, maps:remove(tester, State)};
%%                                {error, _} = ERROR ->
%%                                    ERROR
%%                            end
%%                    end},
%%          #{desc => "close socket",
%%            cmd  => fun(#{sock := Sock} = State) ->
%%                            ok = socket:close(Sock),
%%                            {ok, maps:remove(sock, State)}
%%                    end},

%%          %% *** We are done ***
%%          ?SEV_FINISH_NORMAL
%%         ],

%%     TesterSeq =
%%         [
%%          %% *** Init part ***
%%          #{desc => "monitor server",
%%            cmd  => fun(#{server := Pid} = _State) ->
%%                            _MRef = erlang:monitor(process, Pid),
%%                            ok
%%                    end},
%%          #{desc => "monitor client",
%%            cmd  => fun(#{client := Pid} = _State) ->
%%                            _MRef = erlang:monitor(process, Pid),
%%                            ok
%%                    end},

%%          %% Start the server
%%          #{desc => "order server start",
%%            cmd  => fun(#{server := Pid} = _State) ->
%%                            ?SEV_ANNOUNCE_START(Pid),
%%                            ok
%%                    end},
%%          #{desc => "await server ready (init)",
%%            cmd  => fun(#{server := Pid} = State) ->
%%                            {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
%%                            {ok, State#{server_port => Port}}
%%                    end},

%%          %% Start the client
%%          #{desc => "order client start",
%%            cmd  => fun(#{client := Pid, server_port := Port} = _State) ->
%%                            ?SEV_ANNOUNCE_START(Pid, Port),
%%                            ok
%%                    end},
%%          #{desc => "await client ready (init)",
%%            cmd  => fun(#{client := Pid} = _State) ->
%%                            ok = ?SEV_AWAIT_READY(Pid, client, init)
%%                    end},

%%          %% *** The actual test ***
%%          #{desc => "order server to continue (with accept)",
%%            cmd  => fun(#{server := Server} = _State) ->
%%                            ?SEV_ANNOUNCE_CONTINUE(Server, accept),
%%                            ok
%%                    end},
%%          ?SEV_SLEEP(?SECS(1)),
%%          #{desc => "order client to continue (with connect)",
%%            cmd  => fun(#{client := Client} = _State) ->
%%                            ?SEV_ANNOUNCE_CONTINUE(Client, connect),
%%                            ok
%%                    end},
%%          #{desc => "await client ready (connect)",
%%            cmd  => fun(#{client := Client} = _State) ->
%%                            ?SEV_AWAIT_READY(Client, client, connect)
%%                    end},
%%          #{desc => "await server ready (accept)",
%%            cmd  => fun(#{server := Server} = _State) ->
%%                            ?SEV_AWAIT_READY(Server, server, accept)
%%                    end},
%%          #{desc => "order client to continue (with send request)",
%%            cmd  => fun(#{client := Client} = _State) ->
%%                            ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
%%                            ok
%%                    end},
%%          #{desc => "await client ready (with send request)",
%%            cmd  => fun(#{client := Client} = _State) ->
%%                            ?SEV_AWAIT_READY(Client, client, send_req)
%%                    end},
%%          #{desc => "await server ready (request recv)",
%%            cmd  => fun(#{server := Server} = _State) ->
%%                            ?SEV_AWAIT_READY(Server, server, recv_req)
%%                    end},
%%          #{desc => "order server to continue (with send reply)",
%%            cmd  => fun(#{server := Server} = _State) ->
%%                            ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
%%                            ok
%%                    end},
%%          #{desc => "await server ready (with reply sent)",
%%            cmd  => fun(#{server := Server} = _State) ->
%%                            ?SEV_AWAIT_READY(Server, server, send_reply)
%%                    end},
%%          #{desc => "await client ready (reply recv)",
%%            cmd  => fun(#{client := Client} = _State) ->
%%                            ?SEV_AWAIT_READY(Client, client, recv_reply)
%%                    end},


%%          %% *** Termination ***
%%          #{desc => "order client to terminate",
%%            cmd  => fun(#{client := Client} = _State) ->
%%                            ?SEV_ANNOUNCE_TERMINATE(Client),
%%                            ok
%%                    end},
%%          #{desc => "await client termination",
%%            cmd  => fun(#{client := Client} = State) ->
%%                            ?SEV_AWAIT_TERMINATION(Client),
%%                            State1 = maps:remove(client, State),
%%                            {ok, State1}
%%                    end},
%%          #{desc => "order server to terminate",
%%            cmd  => fun(#{server := Server} = _State) ->
%%                            ?SEV_ANNOUNCE_TERMINATE(Server),
%%                            ok
%%                    end},
%%          #{desc => "await server termination",
%%            cmd  => fun(#{server := Server} = State) ->
%%                            ?SEV_AWAIT_TERMINATION(Server),
%%                            State1 = maps:remove(server, State),
%%                            {ok, State1}
%%                    end},

%%          %% *** We are done ***
%%          ?SEV_FINISH_NORMAL
%%         ],

%%     i("start server evaluator"),
%%     Server = ?SEV_START("server", ServerSeq, InitState),

%%     i("start client evaluator"),
%%     Client = ?SEV_START("client", ClientSeq, InitState),
%%     i("await evaluator(s)"),

%%     i("start tester evaluator"),
%%     TesterInitState = #{server => Server#ev.pid,
%%                         client => Client#ev.pid},
%%     Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

%%     ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).

    ok.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                                                                     %%
%%                           API IOV                                   %%
%%                                                                     %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_b_sendmsg_iov_dgram_inet(Config) when is_list(Config) ->
    api_b_sendmsg_iov_dgram(inet).
%%
api_b_sendmsg_iov_dgram_inet6(Config) when is_list(Config) ->
    has_support_ipv6(),
    api_b_sendmsg_iov_dgram(inet6).
%%
api_b_sendmsg_iov_dgram_local(Config) when is_list(Config) ->
    has_support_unix_domain_socket(),
    api_b_sendmsg_iov_dgram(local).

api_b_sendmsg_iov_stream_inet(Config) when is_list(Config) ->
    api_b_sendmsg_iov_stream(inet).
%%
api_b_sendmsg_iov_stream_inet6(Config) when is_list(Config) ->
    has_support_ipv6(),
    api_b_sendmsg_iov_stream(inet6).
%%
api_b_sendmsg_iov_stream_local(Config) when is_list(Config) ->
    has_support_unix_domain_socket(),
    api_b_sendmsg_iov_stream(local).


api_b_sendmsg_iov_dgram(Domain) ->
    ?TT(?SECS(5)),
    #{iov_max := IOVMax} = socket:info(),
    IOV =
        lists:map(
          fun (N) -> <<(rand:uniform(N) - 1)>> end,
          lists:duplicate(IOVMax, 256)),
    IOVTooLarge = IOV ++ IOV,
    Data = erlang:iolist_to_binary(IOV),
    {ok, Sa} = socket:open(Domain, dgram),
    try
        {ok, Sb} = socket:open(Domain, dgram),
        try
            ok = socket:bind(Sa, which_local_socket_addr(Domain)),
            ok = socket:bind(Sb, which_local_socket_addr(Domain)),
            {ok, Aa} = socket:sockname(Sa),
            {ok, Ab} = socket:sockname(Sb),
            %%
            ok = socket:sendmsg(Sa, #{addr => Ab, iov => IOV}),
            {ok, {Aa, Data}} = socket:recvfrom(Sb),
            %%
            {error, {invalid, _}} =
                socket:sendmsg(Sb, #{addr => Aa, iov => IOVTooLarge}),
            ok
        after
            socket:close(Sb)
        end
    after
        socket:close(Sa)
    end.

api_b_sendmsg_iov_stream(Domain) ->
    ?TT(?SECS(5)),
    #{iov_max := IOVMax} = socket:info(),
    IOV =
        lists:map(
          fun (N) -> <<(rand:uniform(N) - 1)>> end,
          lists:duplicate(IOVMax, 256)),
    IOVTooLarge = IOV ++ IOV,
    Data = erlang:iolist_to_binary(IOV),
    DataTooLarge = erlang:iolist_to_binary(IOVTooLarge),
    {ok, Sa} = socket:open(Domain, stream),
    try
        {ok, Sb} = socket:open(Domain, stream),
        try
            ok = socket:bind(Sb, which_local_socket_addr(Domain)),
            {ok, Ab} = socket:sockname(Sb),
            ok = socket:listen(Sb),
            ok = socket:connect(Sa, Ab),
            {ok, Aa} = socket:sockname(Sa),
            {ok, Ab} = socket:peername(Sa),
            {ok, Sc} = socket:accept(Sb),
            try
                {ok, Ab} = socket:sockname(Sc),
                {ok, Aa} = socket:peername(Sc),
                %%
                ok = socket:sendmsg(Sa, #{iov => IOV}),
                {ok, Data} =
                    socket:recv(Sc, byte_size(Data)),
                ok = socket:sendmsg(Sc, #{iov => IOVTooLarge}),
                {ok, DataTooLarge} =
                    socket:recv(Sa, byte_size(DataTooLarge)),
                ok
            after
                socket:close(Sc)
            end
        after
            socket:close(Sb)
        end
    after
        socket:close(Sa)
    end.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                                                                     %%
%%                           API FROM FD                               %%
%%                                                                     %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) a socket from an already existing 
%% file descriptor (FD) and info of an IPv4 UDP (dgram) socket.
%% With some extra checks...
%% IPv4
%% Without dup
api_ffd_open_wod_and_info_udp4(suite) ->
    [];
api_ffd_open_wod_and_info_udp4(doc) ->
    [];
api_ffd_open_wod_and_info_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_wod_and_info_udp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => dgram,
                                 protocol => udp,
                                 dup      => false},
                   ok = api_ffd_open_and_info(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) a socket from an already existing 
%% file descriptor (FD) and info of an IPv6 UDP (dgram) socket.
%% With some extra checks...
%% IPv6
%% Without dup
api_ffd_open_wod_and_info_udp6(suite) ->
    [];
api_ffd_open_wod_and_info_udp6(doc) ->
    [];
api_ffd_open_wod_and_info_udp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_wod_and_info_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet6,
                                 type     => dgram,
                                 protocol => udp,
                                 dup      => false},
                   ok = api_ffd_open_and_info(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) a socket from an already existing 
%% file descriptor (FD) and info of an IPv4 UDP (dgram) socket.
%% With some extra checks...
%% IPv4
%% With dup
api_ffd_open_wd_and_info_udp4(suite) ->
    [];
api_ffd_open_wd_and_info_udp4(doc) ->
    [];
api_ffd_open_wd_and_info_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_wd_open_and_info_udp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => dgram,
                                 protocol => udp,
                                 dup      => true},
                   ok = api_ffd_open_and_info(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) a socket from an already existing 
%% file descriptor (FD) and info of an IPv4 UDP (dgram) socket.
%% With some extra checks...
%% IPv6
%% With dup
api_ffd_open_wd_and_info_udp6(suite) ->
    [];
api_ffd_open_wd_and_info_udp6(doc) ->
    [];
api_ffd_open_wd_and_info_udp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_wd_open_and_info_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => dgram,
                                 protocol => udp,
                                 dup      => true},
                   ok = api_ffd_open_and_info(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) a socket from an already existing 
%% file descriptor (FD) and info of an IPv4 TCP (stream) socket.
%% With some extra checks...
%% IPv6
%% Without dup
api_ffd_open_wod_and_info_tcp4(suite) ->
    [];
api_ffd_open_wod_and_info_tcp4(doc) ->
    [];
api_ffd_open_wod_and_info_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_wod_and_info_tcp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => stream,
                                 protocol => tcp,
                                 dup      => false},
                   ok = api_ffd_open_and_info(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) a socket from an already existing 
%% file descriptor (FD) and info of an IPv6 TCP (stream) socket.
%% With some extra checks...
%% IPv6
%% Without dup
api_ffd_open_wod_and_info_tcp6(suite) ->
    [];
api_ffd_open_wod_and_info_tcp6(doc) ->
    [];
api_ffd_open_wod_and_info_tcp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_wod_and_info_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet6,
                                 type     => stream,
                                 protocol => tcp,
                                 dup      => false},
                   ok = api_ffd_open_and_info(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) a socket from an already existing 
%% file descriptor (FD) and info of an IPv4 TCP (stream) socket.
%% With some extra checks...
%% IPv6
%% With dup
api_ffd_open_wd_and_info_tcp4(suite) ->
    [];
api_ffd_open_wd_and_info_tcp4(doc) ->
    [];
api_ffd_open_wd_and_info_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_wd_and_info_tcp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => stream,
                                 protocol => tcp,
                                 dup      => true},
                   ok = api_ffd_open_and_info(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open (create) a socket from an already existing 
%% file descriptor (FD) and info of an IPv6 TCP (stream) socket.
%% With some extra checks...
%% IPv6
%% With dup
api_ffd_open_wd_and_info_tcp6(suite) ->
    [];
api_ffd_open_wd_and_info_tcp6(doc) ->
    [];
api_ffd_open_wd_and_info_tcp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_wd_and_info_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet6,
                                 type     => stream,
                                 protocol => tcp,
                                 dup      => true},
                   ok = api_ffd_open_and_info(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_ffd_open_and_info(InitState) ->
    Seq = 
        [
         #{desc => "open",
           cmd  => fun(#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol} = State) -> 
                           case socket:open(Domain, Type, Protocol) of
                               {ok, Sock1} ->
                                   {ok, State#{sock1 => Sock1}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "get socket (1) FD",
           cmd  => fun(#{sock1 := Sock1} = State) ->
                           case socket:getopt(Sock1, otp, fd) of
                               {ok, FD} ->
                                   ?SEV_IPRINT("FD: ~w", [FD]),
                                   {ok, State#{fd => FD}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("failed get FD: "
                                               "~n   ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "check if we need to provide protocol or not",
           cmd  => fun(#{sock1 := Sock1} = State) ->
                           case socket:getopt(Sock1, socket, protocol) of
                               {ok, _} ->
                                   ?SEV_IPRINT("protocol accessible"),
                                   {ok, State#{provide_protocol => false}};
                               {error, Reason} ->
                                   ?SEV_IPRINT("failed get protocol: "
                                               "~n   ~p", [Reason]),
                                   {ok, State#{provide_protocol => true}}
                           end
                   end},
         #{desc => "open with FD",
           cmd  => fun(#{fd               := FD,
                         dup              := DUP,
                         provide_protocol := true,
                         protocol         := Protocol} = State) -> 
                           case socket:open(FD, #{dup      => DUP,
                                                  protocol => Protocol}) of
                               {ok, Sock2} ->
                                   ?SEV_IPRINT("socket 2 open"),
                                   {ok, State#{sock2 => Sock2}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("failed open socket with FD (~w): "
                                               "~n   ~p", [FD, Reason]),
                                   ERROR
                           end;
                      (#{fd               := FD,
                         dup              := DUP,
                         provide_protocol := false} = State) -> 
                           case socket:open(FD, #{dup => DUP}) of
                               {ok, Sock2} ->
                                   ?SEV_IPRINT("socket 2 open"),
                                   {ok, State#{sock2 => Sock2}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("failed open socket with FD (~w): "
                                               "~n   ~p", [FD, Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "get socket (1) info",
           cmd  => fun(#{sock1 := Sock} = State) ->
                           %% socket:setopt(Sock, otp, debug, true),
                           Info = socket:info(Sock),
                           %% socket:setopt(Sock, otp, debug, false),
                           ?SEV_IPRINT("Got Info: "
                                       "~n   ~p", [Info]),
                           {ok, State#{info1 => Info}}
                   end},
         #{desc => "get socket (2) info",
           cmd  => fun(#{sock2 := Sock} = State) ->
                           %% socket:setopt(Sock, otp, debug, true),
                           Info = socket:info(Sock),
                           %% socket:setopt(Sock, otp, debug, false),
                           ?SEV_IPRINT("Got Info: "
                                       "~n   ~p", [Info]),
                           {ok, State#{info2 => Info}}
                   end},
         #{desc => "validate socket (1) info",
           cmd  => fun(#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol,
                         info1    := #{domain        := Domain,
                                       type          := Type,
                                       protocol      := Protocol,
                                       ctype         := normal,
                                       counters      := _,
                                       num_readers   := 0,
                                       num_writers   := 0,
                                       num_acceptors := 0}}) ->
                           ok;
                      (#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol,
                         info     := Info}) ->
                           ?SEV_EPRINT("Unexpected Info for socket 1: "
                                       "~n   (expected) Domain:      ~p"
                                       "~n   (expected) Type:        ~p"
                                       "~n   (expected) Protocol:    ~p"
                                       "~n   (expected) Create Type: ~p"
                                       "~n   ~p",
                                       [Domain, Type, Protocol, normal, Info]),
                           {error, unexpected_infio}
                   end},
         #{desc => "validate socket (2) info",
           cmd  => fun(#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol,
                         fd       := _FD,
                         dup      := false,
                         info2    := #{domain        := Domain,
                                       type          := Type,
                                       protocol      := Protocol,
                                       ctype         := fromfd,
                                       counters      := _,
                                       num_readers   := 0,
                                       num_writers   := 0,
                                       num_acceptors := 0}}) ->
                           ok;
                      (#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol,
                         fd       := _FD,
                         dup      := false,
                         info     := Info}) ->
                           ?SEV_EPRINT("Unexpected Info for socket 2: "
                                       "~n   (expected) Domain:      ~p"
                                       "~n   (expected) Type:        ~p"
                                       "~n   (expected) Protocol:    ~p"
                                       "~n   (expected) Create Type: ~p"
                                       "~n   ~p",
                                       [Domain, Type, Protocol,
                                        fromfd, Info]),
                           {error, unexpected_info};
                      (#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol,
                         fd       := FD,
                         dup      := true,
                         info2    := #{domain        := Domain,
                                       type          := Type,
                                       protocol      := Protocol,
                                       ctype         := {fromfd, FD},
                                       counters      := _,
                                       num_readers   := 0,
                                       num_writers   := 0,
                                       num_acceptors := 0}}) ->
                           ok;
                      (#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol,
                         fd       := FD,
                         dup      := true,
                         info     := Info}) ->
                           ?SEV_EPRINT("Unexpected Info for socket 2: "
                                       "~n   (expected) Domain:      ~p"
                                       "~n   (expected) Type:        ~p"
                                       "~n   (expected) Protocol:    ~p"
                                       "~n   (expected) Create Type: ~p"
                                       "~n   ~p",
                                       [Domain, Type, Protocol,
                                        {fromfd, FD}, Info]),
                           {error, unexpected_info}
                   end},
         #{desc => "close socket (1)",
           cmd  => fun(#{sock1 := Sock} = _State) ->
                           socket:close(Sock)
                   end},
         #{desc => "close socket (2)",
           cmd  => fun(#{sock2 := Sock} = _State) ->
                           socket:close(Sock)
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open a socket (1) and then create another socket (2) from
%% its file descriptor *without* dup.
%% Exchange som data from via both "client" sockets.
%% Finally close the second socket. Ensure that the original socket
%% has not been closed (test by sending some data).
%% IPv4 UDP (dgram) socket.
%%
%% <WARNING>
%%
%% This is *not* how its intended to be used.
%% That an erlang process creating a socket and then handing over the
%% file descriptor to another erlang process. *But* its a convient way
%% to test it!
%%
%% </WARNING>
%%
api_ffd_open_and_open_wod_and_send_udp4(suite) ->
    [];
api_ffd_open_and_open_wod_and_send_udp4(doc) ->
    [];
api_ffd_open_and_open_wod_and_send_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_and_open_wod_and_send_udp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => dgram,
                                 protocol => udp,
                                 dup      => false},
                   ok = api_ffd_open_and_open_and_send_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open a socket (1) and then create another socket (2) from
%% its file descriptor *without* dup.
%% Exchange som data from via both "client" sockets.
%% Finally close the second socket. Ensure that the original socket
%% has not been closed (test by sending some data).
%% IPv6 UDP (dgram) socket.
%%
%% <WARNING>
%%
%% This is *not* how its intended to be used.
%% That an erlang process creating a socket and then handing over the
%% file descriptor to another erlang process. *But* its a convient way
%% to test it!
%%
%% </WARNING>
%%
api_ffd_open_and_open_wod_and_send_udp6(suite) ->
    [];
api_ffd_open_and_open_wod_and_send_udp6(doc) ->
    [];
api_ffd_open_and_open_wod_and_send_udp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_and_open_wod_and_send_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet6,
                                 type     => dgram,
                                 protocol => udp,
                                 dup      => false},
                   ok = api_ffd_open_and_open_and_send_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open a socket (1) and then create another socket (2) from
%% its file descriptor *with* dup.
%% Exchange som data from via both "client" sockets.
%% Finally close the second socket. Ensure that the original socket
%% has not been closed (test by sending some data).
%% IPv4 UDP (dgram) socket.
%%
api_ffd_open_and_open_wd_and_send_udp4(suite) ->
    [];
api_ffd_open_and_open_wd_and_send_udp4(doc) ->
    [];
api_ffd_open_and_open_wd_and_send_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_and_open_wd_and_send_udp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => dgram,
                                 protocol => udp,
                                 dup      => true},
                   ok = api_ffd_open_and_open_and_send_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open a socket (1) and then create another socket (2) from
%% its file descriptor *with* dup.
%% Exchange som data from via both "client" sockets.
%% Finally close the second socket. Ensure that the original socket
%% has not been closed (test by sending some data).
%% IPv6 UDP (dgram) socket.
%%
api_ffd_open_and_open_wd_and_send_udp6(suite) ->
    [];
api_ffd_open_and_open_wd_and_send_udp6(doc) ->
    [];
api_ffd_open_and_open_wd_and_send_udp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_and_open_wd_and_send_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet6,
                                 type     => dgram,
                                 protocol => udp,
                                 dup      => true},
                   ok = api_ffd_open_and_open_and_send_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_ffd_open_and_open_and_send_udp(InitState) ->
    Send = fun(Sock, Data, Dest) ->
                   socket:sendto(Sock, Data, Dest)
           end,
    Recv = fun(Sock) ->
                   socket:recvfrom(Sock)
           end,
    api_ffd_open_and_open_and_send_udp2(InitState#{send   => Send,
                                                           recv   => Recv}).

api_ffd_open_and_open_and_send_udp2(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain   := Domain,
                         protocol := Proto} = State) ->
                           case socket:open(Domain, dgram, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, lsa := LSA} = State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   ?SEV_IPRINT("bound to port: ~w", [Port]),
                                   {ok, State#{port => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, port := Port}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Port),
                           ok
                   end},

         #{desc => "await request 1 (recv)",
           cmd  => fun(#{sock := Sock, recv := Recv} = State) ->
                           case Recv(Sock) of
                               {ok, {Source, ?BASIC_REQ}} ->
                                   ?SEV_IPRINT("received request (1) from: "
                                               "~n   ~p", [Source]),
                                   {ok, State#{source => Source}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 1 (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue 1 (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply 1",
           cmd  => fun(#{sock := Sock, send := Send, source := Source}) ->
                           Send(Sock, ?BASIC_REP, Source)
                   end},
         #{desc => "announce ready 1 (send reply)",
           cmd  => fun(#{tester := Tester} = State) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           {ok, maps:remove(source, State)}
                   end},

         #{desc => "await request 2 (recv)",
           cmd  => fun(#{sock := Sock, recv := Recv} = State) ->
                           case Recv(Sock) of
                               {ok, {Source, ?BASIC_REQ}} -> 
                                   ?SEV_IPRINT("received request (2) from: "
                                               "~n   ~p", [Source]),
                                   {ok, State#{source => Source}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 2 (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue 2 (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply 2",
           cmd  => fun(#{sock := Sock, send := Send, source := Source}) ->
                           Send(Sock, ?BASIC_REP, Source)
                   end},
         #{desc => "announce ready 2 (send reply)",
           cmd  => fun(#{tester := Tester} = State) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           {ok, maps:remove(source, State)}
                   end},

         #{desc => "await request 3 (recv)",
           cmd  => fun(#{sock := Sock, recv := Recv} = State) ->
                           case Recv(Sock) of
                               {ok, {Source, ?BASIC_REQ}} ->
                                   ?SEV_IPRINT("received request (2) from: "
                                               "~n   ~p", [Source]),
                                   {ok, State#{source => Source}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 3 (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue 3 (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply 3",
           cmd  => fun(#{sock := Sock, send := Send, source := Source}) ->
                           Send(Sock, ?BASIC_REP, Source)
                   end},
         #{desc => "announce ready 3 (send reply)",
           cmd  => fun(#{tester := Tester} = State) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           {ok, maps:remove(source, State)}
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    Client1Seq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           {Tester, Port} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_port => Port}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain := Domain, server_port := Port} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = LSA#{port => Port},
                           {ok, State#{local_sa => LSA, server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain   := Domain,
                         protocol := Proto} = State) ->
                           case socket:open(Domain, dgram, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "get socket FD",
           cmd  => fun(#{sock := Sock} = State) ->
                           case socket:getopt(Sock, otp, fd) of
                               {ok, FD} ->
                                   ?SEV_IPRINT("FD: ~w", [FD]),
                                   {ok, State#{fd => FD}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("failed get FD: "
                                               "~n   ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester,
                         fd     := FD}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, FD),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "await continue (send request 1)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 1 (to server)",
           cmd  => fun(#{sock := Sock, send := Send, server_sa := SSA}) ->
                           Send(Sock, ?BASIC_REQ, SSA)
                   end},
         #{desc => "announce ready (send request 1)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 1 (from server)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           {ok, {_, ?BASIC_REP}} = Recv(Sock),
                           ok
                   end},
         #{desc => "announce ready (recv reply 1)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         #{desc => "await continue (send request 3)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 3 (to server)",
           cmd  => fun(#{sock := Sock, send := Send, server_sa := SSA}) ->
                           Send(Sock, ?BASIC_REQ, SSA)
                   end},
         #{desc => "announce ready (send request 3)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 3 (from server)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           {ok, {_, ?BASIC_REP}} = Recv(Sock),
                           ok
                   end},
         #{desc => "announce ready (recv reply 3)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    Client2Seq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           {Tester, {Port, FD}} = ?SEV_AWAIT_START(),
                           {ok, State#{tester      => Tester,
                                       server_port => Port,
                                       fd          => FD}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain := Domain, server_port := Port} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = LSA#{port => Port},
                           {ok, State#{server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{fd  := FD,
                         dup := DUP} = State) ->
                           case socket:open(FD, #{dup => DUP}) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "await continue (send request 2)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 2 (to server)",
           cmd  => fun(#{sock := Sock, send := Send, server_sa := SSA}) ->
                           Send(Sock, ?BASIC_REQ, SSA)
                   end},
         #{desc => "announce ready (send request 2)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 2 (from server)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           {ok, {_, ?BASIC_REP}} = Recv(Sock),
                           ok
                   end},
         #{desc => "announce ready (recv reply 2)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client 1",
           cmd  => fun(#{client1 := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client 2",
           cmd  => fun(#{client2 := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% Start the client 1
         #{desc => "order client 1 start",
           cmd  => fun(#{client1 := Pid, server_port := Port} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Port),
                           ok
                   end},
         #{desc => "await client 1 ready (init)",
           cmd  => fun(#{client1 := Pid} = State) ->
                           case ?SEV_AWAIT_READY(Pid, client1, init) of
                               {ok, FD} ->
                                   {ok, State#{fd => FD}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Client 1 init error: "
                                               "~n   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         %% Start the client 2
         #{desc => "order client 2 start",
           cmd  => fun(#{client2     := Pid,
                         server_port := Port,
                         fd          := FD} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, {Port, FD}),
                           ok
                   end},
         #{desc => "await client 2 ready (init)",
           cmd  => fun(#{client2 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client2, init)
                   end},

         %% *** The actual test ***

         #{desc => "order client 1 to continue (with send request 1)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client 1 ready (with send request 1)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client1, send_req)
                   end},
         #{desc => "await server ready (request recv 1)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply 1)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply 1 sent)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client 1 ready (reply recv 1)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client1, recv_reply)
                   end},


         #{desc => "order client 2 to continue (with send request 2)",
           cmd  => fun(#{client2 := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client 2 ready (with send request 2)",
           cmd  => fun(#{client2 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client2, send_req)
                   end},
         #{desc => "await server ready (request recv 2)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply 2)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply 2 sent)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client 2 ready (reply recv 2)",
           cmd  => fun(#{client2 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client2, recv_reply)
                   end},


         #{desc => "order client 2 to terminate",
           cmd  => fun(#{client2 := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client 2 termination",
           cmd  => fun(#{client2 := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client2, State),
                           {ok, State1}
                   end},


         #{desc => "order client 1 to continue (with send request 3)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client 1 ready (with send request 3)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client1, send_req)
                   end},
         #{desc => "await server ready (request recv 3)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply 3)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply 3 sent)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client 1 ready (reply recv 3)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client1, recv_reply)
                   end},


         %% *** Termination ***
         #{desc => "order client 1 to terminate",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client 1 termination",
           cmd  => fun(#{client1 := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client1, State),
                           {ok, State1}
                   end},
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, maps:remove(dup, InitState)),

    i("start (socket origin) client 1 evaluator"),
    Client1 = ?SEV_START("client-1", Client1Seq, maps:remove(dup, InitState)),
    i("await evaluator(s)"),

    i("start client 2 evaluator"),
    Client2 = ?SEV_START("client-2", Client2Seq, InitState),
    i("await evaluator(s)"),

    i("start tester evaluator"),
    TesterInitState = #{server  => Server#ev.pid,
                        client1 => Client1#ev.pid,
                        client2 => Client2#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    ok = ?SEV_AWAIT_FINISH([Server, Client1, Client2, Tester]).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open a socket (1), connect to a server and then create
%% another socket (2) from its file descriptor *without* dup.
%% Exchange som data from via both "client" sockets.
%% Finally close the second socket. Ensure that the original socket
%% has not been closed (test by sending some data).
%% IPv4 TCP (stream) socket.
%%
%% <WARNING>
%%
%% This is *not* how its intended to be used.
%% That an erlang process creating a socket and then handing over the
%% file descriptor to another erlang process. *But* its a convient way
%% to test it!
%%
%% </WARNING>
%%
api_ffd_open_connect_and_open_wod_and_send_tcp4(suite) ->
    [];
api_ffd_open_connect_and_open_wod_and_send_tcp4(doc) ->
    [];
api_ffd_open_connect_and_open_wod_and_send_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_connect_and_open_wod_and_send_tcp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => stream,
                                 protocol => tcp,
                                 dup      => false},
                   ok = api_ffd_open_connect_and_open_and_send_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open a socket (1), connect to a server and then create
%% another socket (2) from its file descriptor *without* dup.
%% Exchange som data from via both "client" sockets.
%% Finally close the second socket. Ensure that the original socket
%% has not been closed (test by sending some data).
%% IPv6 TCP (stream) socket.
%%
%% <WARNING>
%%
%% This is *not* how its intended to be used.
%% That an erlang process creating a socket and then handing over the
%% file descriptor to another erlang process. *But* its a convient way
%% to test it!
%%
%% </WARNING>
%%
api_ffd_open_connect_and_open_wod_and_send_tcp6(suite) ->
    [];
api_ffd_open_connect_and_open_wod_and_send_tcp6(doc) ->
    [];
api_ffd_open_connect_and_open_wod_and_send_tcp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_connect_and_open_wod_and_send_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet6,
                                 type     => stream,
                                 protocol => tcp,
                                 dup      => false},
                   ok = api_ffd_open_connect_and_open_and_send_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open a socket (1), connect to a server and then create
%% another socket (2) from its file descriptor *with* dup.
%% Exchange som data from via both "client" sockets.
%% Finally close the second socket. Ensure that the original socket
%% has not been closed (test by sending some data).
%% IPv4 TCP (stream) socket.
api_ffd_open_connect_and_open_wd_and_send_tcp4(suite) ->
    [];
api_ffd_open_connect_and_open_wd_and_send_tcp4(doc) ->
    [];
api_ffd_open_connect_and_open_wd_and_send_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_connect_and_open_wd_and_send_tcp4,
           fun() ->
                   InitState = #{domain   => inet,
                                 type     => stream,
                                 protocol => tcp,
                                 dup      => true},
                   ok = api_ffd_open_connect_and_open_and_send_tcp(InitState)
           end).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically open a socket (1), connect to a server and then create
%% another socket (2) from its file descriptor *with* dup.
%% Exchange som data from via both "client" sockets.
%% Finally close the second socket. Ensure that the original socket
%% has not been closed (test by sending some data).
%% IPv6 TCP (stream) socket.
api_ffd_open_connect_and_open_wd_and_send_tcp6(suite) ->
    [];
api_ffd_open_connect_and_open_wd_and_send_tcp6(doc) ->
    [];
api_ffd_open_connect_and_open_wd_and_send_tcp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_ffd_open_connect_and_open_wd_and_send_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   InitState = #{domain   => inet6,
                                 type     => stream,
                                 protocol => tcp,
                                 dup      => true},
                   ok = api_ffd_open_connect_and_open_and_send_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_ffd_open_connect_and_open_and_send_tcp(InitState) ->
    Send = fun(Sock, Data) ->
                   socket:send(Sock, Data)
           end,
    Recv = fun(Sock) ->
                   socket:recv(Sock)
           end,
    api_ffd_open_connect_and_open_and_send_tcp2(InitState#{send   => Send,
                                                           recv   => Recv}).

api_ffd_open_connect_and_open_and_send_tcp2(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain   := Domain,
                         protocol := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{lsock := LSock, lsa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   ?SEV_IPRINT("bound to port: ~w", [Port]),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, lport := Port}) ->
                           %% This is actually not used for unix domain socket
                           ?SEV_ANNOUNCE_READY(Tester, init, Port),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:accept(LSock) of
                               {ok, Sock} ->
                                   ?SEV_IPRINT("accepted: ~n   ~p", [Sock]),
                                   {ok, State#{csock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept),
                           ok
                   end},

         #{desc => "await request 1 (recv)",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, ?BASIC_REQ} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 1 (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue 1 (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply 1",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready 1 (send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           ok
                   end},

         #{desc => "await request 2 (recv)",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, ?BASIC_REQ} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 2 (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue 2 (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply 2",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready 2 (send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           ok
                   end},

         #{desc => "await request 3 (recv)",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, ?BASIC_REQ} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 3 (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue 3 (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply 3",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready 3 (send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close connection socket",
           cmd  => fun(#{csock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(csock, State)}
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:close(LSock) of
                               ok ->
                                   {ok, maps:remove(lsock, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    Client1Seq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           {Tester, Port} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_port => Port}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain := Domain, server_port := Port} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = LSA#{port => Port},
                           {ok, State#{local_sa => LSA, server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain   := Domain,
                         protocol := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},


         #{desc => "await continue (connect)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, connect)
                   end},
         #{desc => "connect to server",
           cmd  => fun(#{sock := Sock, server_sa := SSA}) ->
                           socket:connect(Sock, SSA)
                   end},
         #{desc => "get socket FD",
           cmd  => fun(#{sock := Sock} = State) ->
                           case socket:getopt(Sock, otp, fd) of
                               {ok, FD} ->
                                   ?SEV_IPRINT("FD: ~w", [FD]),
                                   {ok, State#{fd => FD}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("failed get FD: "
                                               "~n   ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (connect)",
           cmd  => fun(#{tester := Tester,
                         fd     := FD}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect, FD),
                           ok
                   end},


         %% *** The actual test ***
         #{desc => "await continue (send request 1)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 1 (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send request 1)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 1 (from server)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           {ok, ?BASIC_REP} = Recv(Sock),
                           ok
                   end},
         #{desc => "announce ready (recv reply 1)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         #{desc => "await continue (send request 3)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 3 (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send request 3)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 3 (from server)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           {ok, ?BASIC_REP} = Recv(Sock),
                           ok
                   end},
         #{desc => "announce ready (recv reply 3)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    Client2Seq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           {Tester, FD} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, fd => FD}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "create socket",
           cmd  => fun(#{fd  := FD,
                         dup := DUP} = State) ->
                           case socket:open(FD, #{dup => DUP}) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "await continue (send request 2)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 2 (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send request 2)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 2 (from server)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           {ok, ?BASIC_REP} = Recv(Sock),
                           ok
                   end},
         #{desc => "announce ready (recv reply 2)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client 1",
           cmd  => fun(#{client1 := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client 2",
           cmd  => fun(#{client2 := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% Start the client 1
         #{desc => "order client 1 start",
           cmd  => fun(#{client1 := Pid, server_port := Port} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Port),
                           ok
                   end},
         #{desc => "await client 1 ready (init)",
           cmd  => fun(#{client1 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client1, init)
                   end},

         #{desc => "order server to continue (with accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
         ?SEV_SLEEP(?SECS(1)),
         #{desc => "order client 1 to continue (with connect)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, connect),
                           ok
                   end},
         #{desc => "await client 1 ready (connect)",
           cmd  => fun(#{client1 := Pid} = State) ->
                           {ok, FD} = ?SEV_AWAIT_READY(Pid, client1, connect),
                           {ok, State#{fd => FD}}
                   end},
         #{desc => "await server ready (accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, accept)
                   end},         

         %% Start the client 2
         #{desc => "order client 2 start",
           cmd  => fun(#{client2 := Pid, fd := FD} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, FD),
                           ok
                   end},
         #{desc => "await client 2 ready (init)",
           cmd  => fun(#{client2 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client2, init)
                   end},

         %% *** The actual test ***

         #{desc => "order client 1 to continue (with send request 1)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client 1 ready (with send request 1)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client1, send_req)
                   end},
         #{desc => "await server ready (request recv 1)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply 1)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply 1 sent)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client 1 ready (reply recv 1)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client1, recv_reply)
                   end},


         #{desc => "order client 2 to continue (with send request 2)",
           cmd  => fun(#{client2 := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client 2 ready (with send request 2)",
           cmd  => fun(#{client2 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client2, send_req)
                   end},
         #{desc => "await server ready (request recv 2)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply 2)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply 2 sent)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client 2 ready (reply recv 2)",
           cmd  => fun(#{client2 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client2, recv_reply)
                   end},


         #{desc => "order client 2 to terminate",
           cmd  => fun(#{client2 := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client 2 termination",
           cmd  => fun(#{client2 := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client2, State),
                           {ok, State1}
                   end},


         #{desc => "order client 1 to continue (with send request 3)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client 1 ready (with send request 3)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client1, send_req)
                   end},
         #{desc => "await server ready (request recv 3)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply 3)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply 3 sent)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client 1 ready (reply recv 3)",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client1, recv_reply)
                   end},


         %% *** Termination ***
         #{desc => "order client 1 to terminate",
           cmd  => fun(#{client1 := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client 1 termination",
           cmd  => fun(#{client1 := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client1, State),
                           {ok, State1}
                   end},
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, maps:remove(dup, InitState)),

    i("start (socket origin) client 1 evaluator"),
    Client1 = ?SEV_START("client-1", Client1Seq, maps:remove(dup, InitState)),
    i("await evaluator(s)"),

    i("start client 2 evaluator"),
    Client2 = ?SEV_START("client-2", Client2Seq, InitState),
    i("await evaluator(s)"),

    i("start tester evaluator"),
    TesterInitState = #{server  => Server#ev.pid,
                        client1 => Client1#ev.pid,
                        client2 => Client2#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    ok = ?SEV_AWAIT_FINISH([Server, Client1, Client2, Tester]).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                                                                     %%
%%                           API ASYNC                                 %%
%%                                                                     %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically establish a TCP connection via an async connect. IPv4.

api_a_connect_tcp4(suite) ->
    [];
api_a_connect_tcp4(doc) ->
    [];
api_a_connect_tcp4(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    tc_try(api_a_connect_tcp4,
           fun() ->
                   ok = api_a_connect_tcpD(inet, nowait(Config))
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically establish a TCP connection via an async connect. IPv6.

api_a_connect_tcp6(suite) ->
    [];
api_a_connect_tcp6(doc) ->
    [];
api_a_connect_tcp6(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    tc_try(api_a_connect_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   ok = api_a_connect_tcpD(inet6, nowait(Config))
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_a_connect_tcpD(Domain, Nowait) ->
    Connect = fun(Sock, SockAddr) ->
                      socket:connect(Sock, SockAddr, Nowait)
              end,
    Send = fun(Sock, Data) ->
                   socket:send(Sock, Data)
           end,
    Recv = fun(Sock) ->
                   socket:recv(Sock)
           end,
    InitState = #{domain => Domain,
                  connect => Connect,
                  send => Send,
                  recv => Recv,
                  connect_sref => Nowait},
    api_a_connect_tcp(InitState).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_a_connect_tcp(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{lsock := LSock, lsa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, lport := Port}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Port),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:accept(LSock) of
                               {ok, Sock} ->
                                   ?SEV_IPRINT("accepted: ~n   ~p", [Sock]),
                                   {ok, State#{csock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept),
                           ok
                   end},

         #{desc => "await continue (recv_req)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv_req)
                   end},
         #{desc => "recv req",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, ?BASIC_REQ} ->
                                   ok;
                               {ok, UnexpData} ->
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_req)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue (send_rep)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_rep)
                   end},
         #{desc => "send rep",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready (send_rep)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_rep),
                           ok
                   end},
	 

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close connection socket",
           cmd  => fun(#{csock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(csock, State)}
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{lsock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(lsock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    ClientSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           {Tester, Port} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_port => Port}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain := Domain, server_port := Port} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = LSA#{port => Port},
                           {ok, State#{local_sa => LSA, server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "await continue (async connect)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, async_connect)
                   end},
         #{desc => "connect (async) to server",
           cmd  => fun(#{sock := Sock,
                         server_sa := SSA,
                         connect := Connect,
                         connect_sref := SR} = State) ->
                           case Connect(Sock, SSA) of
                               ok ->
                                   ?SEV_IPRINT("ok -> "
					       "unexpected success => SKIP", 
                                               []),
                                   {skip, unexpected_success};
                               {select, {select_info, ST, SelectRef}}
                                 when SR =:= nowait ->
                                   ?SEV_IPRINT("select nowait ->"
                                               "~n   tag: ~p"
                                               "~n   ref: ~p",
                                               [ST, SelectRef]),
                                   {ok, State#{connect_stag => ST,
                                               connect_sref => SelectRef}};
                               {select, {select_info, ST, SR}}
                                 when is_reference(SR) ->
                                   ?SEV_IPRINT("select ref ->"
                                               "~n   tag: ~p"
                                               "~n   ref: ~p", [ST, SR]),
                                   {ok, State#{connect_stag => ST}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (connect select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect_select),
                           ok
                   end},
         #{desc => "await select message",
           cmd  => fun(#{sock := Sock, connect_sref := Ref}) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   ?SEV_IPRINT("select message ->"
                                               "~n   ref: ~p", [Ref]),
                                   ok
                           after 5000 ->
                                   ?SEV_EPRINT("timeout: "
                                               "~n   message queue: ~p",
                                               [mq()]),
                                   {error, timeout}
                           end
                   end},
         #{desc => "announce ready (select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, select),
                           ok
                   end},
         #{desc => "connect (async) to server",
           cmd  => fun(#{sock := Sock, server_sa := SSA, connect := Connect}) ->
                           case Connect(Sock, SSA) of
                               ok ->
                                   ok;
                               {select, SelectInfo} ->
                                   {error, {unexpected_select, SelectInfo}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (connect)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect),
                           ok
                   end},
         #{desc => "get peername",
           cmd  => fun(#{sock := Sock} = _State) ->
                           case socket:peername(Sock) of
                               {ok, SockAddr} ->
                                   ?SEV_IPRINT("Peer Name: ~p", [SockAddr]),
                                   ok;
                                {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "await continue (send_req)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send req",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send_req)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await continue (recv_rep)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv_rep)
                   end},
         #{desc => "recv rep",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, ?BASIC_REP} ->
                                   ok;
                               {ok, UnexpData} ->
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_rep)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_rep),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           State2 = maps:remove(sock,         State),
                           State3 = maps:remove(connect_stag, State2),
                           State4 = maps:remove(connect_sref, State3),
                           {ok, State4}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% Start the client
         #{desc => "order client start",
           cmd  => fun(#{client := Pid, server_port := Port} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Port),
                           ok
                   end},
         #{desc => "await client ready (init)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, init)
                   end},


         %% *** The actual test ***
         #{desc => "order client to continue (async connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, async_connect),
                           ok
                   end},
         #{desc => "await client ready (connect select)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, connect_select)
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "order server to continue (with accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
         #{desc => "await client ready (select)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, select)
                   end},
         #{desc => "await client ready (connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, connect)
                   end},
         #{desc => "await server ready (accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, accept)
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "order server to recv test req (recv req)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, recv_req),
                           ok
                   end},
         #{desc => "order client to send test req (send req)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client ready (send_req)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send_req)
                   end},
         #{desc => "await server ready (recv_req)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order client to recv test rep (send rep)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, recv_rep),
                           ok
                   end},
         #{desc => "order server to send test rep (send rep)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_rep),
                           ok
                   end},
         #{desc => "await server ready (send_rep)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_rep)
                   end},
         #{desc => "await client ready (recv_rep)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, recv_rep)
                   end},


         %% *** Termination ***
         #{desc => "order client to terminate",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client termination",
           cmd  => fun(#{client := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client, State),
                           {ok, State1}
                   end},
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start client evaluator"),
    Client = ?SEV_START("client", ClientSeq, InitState),
    i("await evaluator(s)"),

    i("start tester evaluator"),
    TesterInitState = #{server => Server#ev.pid,
                        client => Client#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive on an IPv4 UDP (dgram) socket using
%% sendto and recvfrom. But we try to be async. That is, we use
%% the 'nowait' value for the Timeout argument (and await the eventual
%% select message). Note that we only do this for the recvfrom,
%% since its much more difficult to "arrange" for sendto.
%%
api_a_sendto_and_recvfrom_udp4(suite) ->
    [];
api_a_sendto_and_recvfrom_udp4(doc) ->
    [];
api_a_sendto_and_recvfrom_udp4(Config) when is_list(Config) ->
    ?TT(?SECS(5)),
    Nowait = nowait(Config),
    tc_try(api_a_sendto_and_recvfrom_udp4,
           fun() ->
                   Send = fun(Sock, Data, Dest) ->
                                  socket:sendto(Sock, Data, Dest)
                          end,
                   Recv = fun(Sock) ->
                                  socket:recvfrom(Sock, 0, Nowait)
                          end,
                   InitState = #{domain => inet,
                                 send => Send,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_send_and_recv_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive on an IPv6 UDP (dgram) socket using
%% sendto and recvfrom. But we try to be async. That is, we use
%% the 'nowait' value for the Timeout argument (and await the eventual
%% select message). Note that we only do this for the recvfrom,
%% since its much more difficult to "arrange" for sendto.
%%
api_a_sendto_and_recvfrom_udp6(suite) ->
    [];
api_a_sendto_and_recvfrom_udp6(doc) ->
    [];
api_a_sendto_and_recvfrom_udp6(Config) when is_list(Config) ->
    ?TT(?SECS(5)),
    Nowait = nowait(Config),
    tc_try(api_a_sendto_and_recvfrom_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Send = fun(Sock, Data, Dest) ->
                                  socket:sendto(Sock, Data, Dest)
                          end,
                   Recv = fun(Sock) ->
                                  socket:recvfrom(Sock, 0, Nowait)
                          end,
                   InitState = #{domain => inet6,
                                 send => Send,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_send_and_recv_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive on an IPv4 UDP (dgram) socket using
%% sendto and recvfrom. But we try to be async. That is, we use
%% the 'nowait' value for the Timeout argument (and await the eventual
%% select message). Note that we only do this for the recvmsg,
%% since its much more difficult to "arrange" for sendmsg.
%%
api_a_sendmsg_and_recvmsg_udp4(suite) ->
    [];
api_a_sendmsg_and_recvmsg_udp4(doc) ->
    [];
api_a_sendmsg_and_recvmsg_udp4(Config) when is_list(Config) ->
    ?TT(?SECS(5)),
    Nowait = nowait(Config),
    tc_try(api_a_sendmsg_and_recvmsg_udp4,
           fun() ->
                   Send = fun(Sock, Data, Dest) ->
                                  Msg = #{addr => Dest,
                                             %% ctrl => CMsgs,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock, Nowait) of
                                      {ok, #{addr  := Source,
                                             iov   := [Data]}} ->
                                          {ok, {Source, Data}};
                                      {ok, _} = OK ->
                                          OK;
                                      {select, _} = SELECT ->
                                          SELECT;
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 send => Send,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_send_and_recv_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive on an IPv6 UDP (dgram) socket using
%% sendto and recvfrom. But we try to be async. That is, we use
%% the 'nowait' value for the Timeout argument (and await the eventual
%% select message). Note that we only do this for the recvmsg,
%% since its much more difficult to "arrange" for sendmsg.
%%
api_a_sendmsg_and_recvmsg_udp6(suite) ->
    [];
api_a_sendmsg_and_recvmsg_udp6(doc) ->
    [];
api_a_sendmsg_and_recvmsg_udp6(Config) when is_list(Config) ->
    ?TT(?SECS(5)),
    Nowait = nowait(Config),
    tc_try(api_a_sendmsg_and_recvmsg_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Send = fun(Sock, Data, Dest) ->
                                  Msg = #{addr => Dest,
                                             %% ctrl => CMsgs,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock, Nowait) of
                                      {ok, #{addr  := Source,
                                             iov   := [Data]}} ->
                                          {ok, {Source, Data}};
                                      {ok, _} = OK ->
                                          OK;
                                      {select, _} = SELECT ->
                                          SELECT;
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet6,
                                 send => Send,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_send_and_recv_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_a_send_and_recv_udp(InitState) ->
    ServerSeq = 
        [
         %% *** Wait for start order part ***
         #{desc => "await start",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester} = _State) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{local_sa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind socket (to local address)",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   {ok, State#{port => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, local_sa := LSA, port := Port}) ->
                           ServerSA = LSA#{port => Port},
                           ?SEV_ANNOUNCE_READY(Tester, init, ServerSA),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (recv)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv)
                   end},
         #{desc => "try recv request (with nowait, expect select)",
           cmd  => fun(#{sock := Sock,
                         recv := Recv,
                         recv_sref := SR} = State) ->
                           case Recv(Sock) of
                               {select, {select_info, Tag, RecvRef}}
                                 when SR =:= nowait ->
                                   ?SEV_IPRINT("expected select nowait: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [Tag, RecvRef]),
                                   {ok, State#{recv_stag => Tag,
                                               recv_sref => RecvRef}};
                               {select, {select_info, Tag, SR}}
                                 when is_reference(SR) ->
                                   ?SEV_IPRINT("expected select ref: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [Tag, SR]),
                                   {ok, State#{recv_stag => Tag}};
                               {ok, X} ->
                                   {error, {unexpected_succes, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_select),
                           ok
                   end},
         #{desc => "await select message",
           cmd  => fun(#{sock := Sock, recv_sref := RecvRef}) ->
                           receive
                               {'$socket', Sock, select, RecvRef} ->
                                   ok
                           after 5000 ->
                                   ?SEV_EPRINT("message queue: ~p", [mq()]),
                                   {error, timeout}
                           end
                   end},
         #{desc => "announce ready (select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, select),
                           ok
                   end},
         #{desc => "now read the data (request)",
           cmd  => fun(#{sock := Sock, recv := Recv} = State) ->
                           case Recv(Sock) of
                               {ok, {Src, ?BASIC_REQ}} ->
                                   {ok, State#{req_src => Src}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},

         #{desc => "await continue (send reply)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply",
           cmd  => fun(#{sock := Sock, req_src := Src, send := Send}) ->
                           Send(Sock, ?BASIC_REP, Src)
                   end},
         #{desc => "announce ready (send)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send),
                           ok
                   end},

         %% Termination
         #{desc => "await terminate (from tester)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   State2 = maps:remove(tester,    State),
                                   State3 = maps:remove(recv_stag, State2),
                                   State4 = maps:remove(recv_sref, State3),
                                   State5 = maps:remove(req_src,   State4),
                                   {ok, State5};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    ClientSeq = 
        [
         %% *** Wait for start order part ***
         #{desc => "await start",
           cmd  => fun(State) ->
                           {Tester, ServerSA} = ?SEV_AWAIT_START(),
                           {ok, State#{tester    => Tester, 
                                       server_sa => ServerSA}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester} = _State) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "open socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           Sock = sock_open(Domain, dgram, udp),
                           SA   = sock_sockname(Sock),
                           {ok, State#{sock => Sock, sa => SA}}
                   end},
         #{desc => "bind socket (to local address)",
           cmd  => fun(#{sock := Sock, lsa := LSA}) ->
                          case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (send request)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request",
           cmd  => fun(#{sock := Sock, server_sa := Server, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Server)
                   end},
         #{desc => "announce ready (send request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await continue (recv)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv)
                   end},
         #{desc => "try recv reply (with nowait)",
           cmd  => fun(#{sock := Sock,
                         recv := Recv,
                         recv_sref := SR} = State) ->
                           case Recv(Sock) of
                               {select, {select_info, Tag, RecvRef}}
                                 when SR =:= nowait ->
                                   {ok, State#{recv_stag => Tag,
                                               recv_sref => RecvRef}};
                               {select, {select_info, Tag, SR}}
                                 when is_reference(SR) ->
                                   {ok, State#{recv_stag => Tag}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_select),
                           ok
                   end},
         #{desc => "await select message",
           cmd  => fun(#{sock := Sock, recv_sref := RecvRef}) ->
                           receive
                               {'$socket', Sock, select, RecvRef} ->
                                   ok
                           end
                   end},
         #{desc => "announce ready (select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, select),
                           ok
                   end},
         #{desc => "now read the data (reply)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {_Src, ?BASIC_REP}} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_rep),
                           ok
                   end},

         %% Termination
         #{desc => "await terminate (from tester)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   State2 = maps:remove(tester,    State),
                                   State3 = maps:remove(recv_stag, State2),
                                   State4 = maps:remove(recv_sref, State3),
                                   {ok, State4};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    TesterSeq = 
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, ServerSA} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_sa => ServerSA}}
                   end},

         %% Start the client
         #{desc => "order client start",
           cmd  => fun(#{client    := Pid, 
                         server_sa := ServerSA} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, ServerSA),
                           ok
                   end},
         #{desc => "await client ready (init)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, init)
                   end},
 
         %% The actual test
         #{desc => "order server continue (recv)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv),
                           ok
                   end},
         #{desc => "await server ready (recv_select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, recv_select)
                   end},

         #{desc => "order client continue (send request)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, send_req),
                           ok
                   end},
         #{desc => "await client ready (send request)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, send_req)
                   end},
         #{desc => "await server ready (select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, select)
                   end},
         #{desc => "await server ready (recv request)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, recv_req)
                   end},

         #{desc => "order client continue (recv)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv),
                           ok
                   end},
         #{desc => "await client ready (recv_select)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, recv_select)
                   end},
         #{desc => "order server continue (send reply)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, send_reply),
                           ok
                   end},
         #{desc => "await server ready (send)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, send)
                   end},
         #{desc => "await client ready (select)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, select)
                   end},
         #{desc => "await client ready (recv reply)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, recv_rep)
                   end},

         %% Terminations
         #{desc => "order client to terminate",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Pid),
                           ok
                   end},
         #{desc => "await client termination",
           cmd  => fun(#{client := Pid} = State) ->
                           case ?SEV_AWAIT_TERMINATION(Pid) of
                               ok ->
                                   State1 = maps:remove(client, State),
                                   {ok, State1};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Pid),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Pid} = State) ->
                           case ?SEV_AWAIT_TERMINATION(Pid) of
                               ok ->
                                   State1 = maps:remove(server, State),
                                   {ok, State1};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    ServerInitState = InitState,
    Server = ?SEV_START("server", ServerSeq, ServerInitState),

    i("start client evaluator(s)"),
    ClientInitState = InitState,
    Client = ?SEV_START("client", ClientSeq, ClientInitState),

    i("start 'tester' evaluator"),
    TesterInitState = #{server => Server#ev.pid,
                        client => Client#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    i("await evaluator"),
    ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive using the "common" functions (send and recv)
%% on an IPv4 TCP (stream) socket. But we try to be async. That is, we use
%% the 'nowait' value for the Timeout argument (and await the eventual
%% select message). Note that we only do this for the recv,
%% since its much more difficult to "arrange" for send.
%% We *also* test async for accept.
api_a_send_and_recv_tcp4(suite) ->
    [];
api_a_send_and_recv_tcp4(doc) ->
    [];
api_a_send_and_recv_tcp4(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_send_and_recv_tcp4,
           fun() ->
                   Send = fun(Sock, Data) ->
                                  socket:send(Sock, Data)
                          end,
                   Recv = fun(Sock) ->
                                  socket:recv(Sock, 0, Nowait)
                          end,
                   InitState = #{domain => inet,
                                 send => Send,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_send_and_recv_tcp(Config, InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive using the "common" functions (send and recv)
%% on an IPv6 TCP (stream) socket. But we try to be async. That is, we use
%% the 'nowait' value for the Timeout argument (and await the eventual
%% select message). Note that we only do this for the recv,
%% since its much more difficult to "arrange" for send.
%% We *also* test async for accept.
api_a_send_and_recv_tcp6(suite) ->
    [];
api_a_send_and_recv_tcp6(doc) ->
    [];
api_a_send_and_recv_tcp6(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_send_and_recv_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Send = fun(Sock, Data) ->
                                  socket:send(Sock, Data)
                          end,
                   Recv = fun(Sock) ->
                                  socket:recv(Sock, 0, Nowait)
                          end,
                   InitState = #{domain => inet6,
                                 send => Send,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_send_and_recv_tcp(Config, InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive using the msg functions (sendmsg and recvmsg)
%% on an IPv4 TCP (stream) socket. But we try to be async. That is, we use
%% the 'nowait' value for the Timeout argument (and await the eventual
%% select message). Note that we only do this for the recvmsg,
%% since its much more difficult to "arrange" for sendmsg.
%% We *also* test async for accept.
api_a_sendmsg_and_recvmsg_tcp4(suite) ->
    [];
api_a_sendmsg_and_recvmsg_tcp4(doc) ->
    [];
api_a_sendmsg_and_recvmsg_tcp4(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_sendmsg_and_recvmsg_tcp4,
           fun() ->
                   Send = fun(Sock, Data) ->
                                  Msg = #{iov => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock, Nowait) of
                                      {ok, #{iov   := [Data]}} ->
                                          {ok, Data};
                                      {select, _} = SELECT ->
                                          SELECT;
				      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 send => Send,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_send_and_recv_tcp(Config, InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically send and receive using the msg functions (sendmsg and recvmsg)
%% on an IPv6 TCP (stream) socket. But we try to be async. That is, we use
%% the 'nowait' value for the Timeout argument (and await the eventual
%% select message). Note that we only do this for the recvmsg,
%% since its much more difficult to "arrange" for sendmsg.
%% We *also* test async for accept.
api_a_sendmsg_and_recvmsg_tcp6(suite) ->
    [];
api_a_sendmsg_and_recvmsg_tcp6(doc) ->
    [];
api_a_sendmsg_and_recvmsg_tcp6(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_sendmsg_and_recvmsg_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Send = fun(Sock, Data) ->
                                  Msg = #{iov => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock, Nowait) of
                                      {ok, #{iov   := [Data]}} ->
                                          {ok, Data};
                                      {select, _} = SELECT ->
                                          SELECT;
				      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet6,
                                 send => Send,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_send_and_recv_tcp(Config, InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_a_send_and_recv_tcp(Config, InitState) ->
    process_flag(trap_exit, true),
    ServerSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{lsock := LSock, lsa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, lport := Port}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Port),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection (nowait)",
           cmd  => fun(#{lsock := LSock} = State) ->
                           Nowait = nowait(Config),
                           case socket:accept(LSock, Nowait) of
                               {select, {select_info, Tag, Ref}}
                                 when Nowait =:= nowait ->
                                   ?SEV_IPRINT("accept select nowait: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [Tag, Ref]),
                                   {ok, State#{accept_stag => Tag,
                                               accept_sref => Ref}};
                               {select, {select_info, Tag, Nowait}}
                                 when is_reference(Nowait) ->
                                   ?SEV_IPRINT("accept select ref: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [Tag, Nowait]),
                                   {ok, State#{accept_stag => Tag,
                                               accept_sref => Nowait}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept_select),
                           ok
                   end},
         #{desc => "await select message",
           cmd  => fun(#{lsock := Sock, accept_sref := Ref}) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   ok
                           end
                   end},
         #{desc => "announce ready (select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, select),
                           ok
                   end},
         #{desc => "await connection (again)",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:accept(LSock, nowait) of
                               {ok, Sock} ->
                                   ?SEV_IPRINT("accepted: "
                                               "~n   Sock: ~p", [Sock]),
                                   {ok, State#{csock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept),
                           ok
                   end},

         #{desc => "await continue (recv request)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv_req)
                   end},
         #{desc => "try recv request (with nowait, expect select)",
           cmd  => fun(#{csock := Sock,
                         recv := Recv,
                         recv_sref := SR} = State) ->
                           case Recv(Sock) of
                               {select, {select_info, Tag, Ref}}
                                 when SR =:= nowait ->
                                   ?SEV_IPRINT("recv select nowait: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [Tag, Ref]),
                                   {ok, State#{recv_stag => Tag,
                                               recv_sref => Ref}};
                               {select, {select_info, Tag, SR}}
                                 when is_reference(SR) ->
                                   ?SEV_IPRINT("recv select ref: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [Tag, SR]),
                                   {ok, State#{recv_stag => Tag}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_select),
                           ok
                   end},
         #{desc => "await select message",
           cmd  => fun(#{csock := Sock, recv_sref := RecvRef}) ->
                           receive
                               {'$socket', Sock, select, RecvRef} ->
                                   ok
                           end
                   end},
         #{desc => "announce ready (select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, select),
                           ok
                   end},
         #{desc => "now read the data (request)",
           cmd  => fun(#{csock := Sock, recv := Recv} = _State) ->
                           case Recv(Sock) of
                               {ok, ?BASIC_REQ} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},

         #{desc => "await continue (send reply)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_rep)
                   end},
         #{desc => "send reply",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready (send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_rep),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close connection socket",
           cmd  => fun(#{csock := Sock}) ->
                           socket:close(Sock)
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{lsock := Sock}) ->
                           socket:close(Sock)
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    ClientSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           {Tester, Port} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_port => Port}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain := Domain, server_port := Port} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = LSA#{port => Port},
                           {ok, State#{local_sa => LSA, server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "await continue (connect)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, connect)
                   end},
         #{desc => "connect to server",
           cmd  => fun(#{sock := Sock, server_sa := SSA}) ->
                           socket:connect(Sock, SSA)
                   end},
         #{desc => "announce ready (connect)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect),
                           ok
                   end},

         #{desc => "await continue (send request)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           ok = Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},

         #{desc => "try recv reply (with nowait, expect select)",
           cmd  => fun(#{sock := Sock,
                         recv := Recv,
                         recv_sref := SR} = State) ->
                           case Recv(Sock) of
                               {select, {select_info, Tag, Ref}}
                                 when SR =:= nowait ->
                                   ?SEV_IPRINT("recv select nowait: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [Tag, Ref]),
                                   {ok, State#{recv_stag => Tag,
                                               recv_sref => Ref}};
                               {select, {select_info, Tag, SR}}
                                 when is_reference(SR) ->
                                   ?SEV_IPRINT("recv select ref: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [Tag, SR]),
                                   {ok, State#{recv_stag => Tag}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_select),
                           ok
                   end},
         #{desc => "await select message",
           cmd  => fun(#{sock := Sock, recv_sref := RecvRef}) ->
                           receive
                               {'$socket', Sock, select, RecvRef} ->
                                   ok
                           end
                   end},
         #{desc => "announce ready (select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, select),
                           ok
                   end},
         #{desc => "now read the data (reply)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           {ok, ?BASIC_REP} = Recv(Sock),
                           ok
                   end},
         #{desc => "announce ready (recv reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_rep),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock}) ->
                           socket:close(Sock)
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% Start the client
         #{desc => "order client start",
           cmd  => fun(#{client := Pid, server_port := Port} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Port),
                           ok
                   end},
         #{desc => "await client ready (init)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, init)
                   end},

         %% *** The actual test ***
         #{desc => "order server to continue (with accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
         #{desc => "await server ready (accept select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, accept_select)
                   end},
         #{desc => "order client to continue (with connect)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, connect),
                           ok
                   end},
         #{desc => "await server ready (select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, select)
                   end},
         #{desc => "await server ready (accept)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, accept)
                   end},
         #{desc => "await client ready (connect)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, client, connect)
                   end},

         #{desc => "order server to continue (recv request)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv_req),
                           ok
                   end},
         #{desc => "await server ready (recv select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, recv_select)
                   end},
         #{desc => "order client to continue (send request)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, send_req),
                           ok
                   end},
         #{desc => "await client ready (send request)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send_req)
                   end},
         #{desc => "await server ready (select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, select)
                   end},
         #{desc => "await server ready (recv request)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, recv_req)
                   end},

         #{desc => "order client to continue (recv reply)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv_rep),
                           ok
                   end},
         #{desc => "await client ready (recv select)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, client, recv_select)
                   end},
         #{desc => "order server to continue (send reply)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, send_rep),
                           ok
                   end},
         #{desc => "await server ready (send reply)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, send_rep)
                   end},
         #{desc => "await client ready (select)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, client, select)
                   end},
         #{desc => "await client ready (reply recv)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, recv_rep)
                   end},


         %% *** Termination ***
         #{desc => "order client to terminate",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client termination",
           cmd  => fun(#{client := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client, State),
                           {ok, State1}
                   end},
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start client evaluator"),
    Client = ?SEV_START("client", ClientSeq, InitState),

    i("start tester evaluator"),
    TesterInitState = #{server => Server#ev.pid,
                        client => Client#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).





%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make an async (Timeout = nowait) call to recvfrom,
%% wait some time and then cancel. IPv4
%%
api_a_recvfrom_cancel_udp4(suite) ->
    [];
api_a_recvfrom_cancel_udp4(doc) ->
    [];
api_a_recvfrom_cancel_udp4(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_recvfrom_cancel_udp4,
           fun() ->
                   Recv = fun(Sock) ->
                                  case socket:recvfrom(Sock, 0, Nowait) of
                                      {ok, _} = OK ->
                                          OK;
                                      {select, _} = SELECT ->
                                          SELECT;
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_recv_cancel_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make an async (Timeout = nowait) call to recvfrom,
%% wait some time and then cancel. IPv6
%%
api_a_recvfrom_cancel_udp6(suite) ->
    [];
api_a_recvfrom_cancel_udp6(doc) ->
    [];
api_a_recvfrom_cancel_udp6(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_recvfrom_cancel_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Recv = fun(Sock) ->
                                  case socket:recvfrom(Sock, 0, Nowait) of
                                      {ok, _} = OK ->
                                          OK;
                                      {select, _} = SELECT ->
                                          SELECT;
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet6,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_recv_cancel_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make an async (Timeout = nowait) call to recvmsg,
%% wait some time and then cancel. IPv4
%%
api_a_recvmsg_cancel_udp4(suite) ->
    [];
api_a_recvmsg_cancel_udp4(doc) ->
    [];
api_a_recvmsg_cancel_udp4(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_recvmsg_cancel_udp4,
           fun() ->
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock, Nowait) of
                                      {ok, _} = OK ->
                                          OK;
                                      {select, _} = SELECT ->
                                          SELECT;
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_recv_cancel_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make an async (Timeout = nowait) call to recvmsg,
%% wait some time and then cancel. IPv6
%%
api_a_recvmsg_cancel_udp6(suite) ->
    [];
api_a_recvmsg_cancel_udp6(doc) ->
    [];
api_a_recvmsg_cancel_udp6(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_recvmsg_cancel_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock, Nowait) of
                                      {ok, _} = OK ->
                                          OK;
                                      {select, _} = SELECT ->
                                          SELECT;
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet6,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_recv_cancel_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_a_recv_cancel_udp(InitState) ->
    ServerSeq = 
        [
         %% *** Wait for start order part ***
         #{desc => "await start",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester} = _State) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{local_sa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind socket (to local address)",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   {ok, State#{port => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, local_sa := LSA, port := Port}) ->
                           ServerSA = LSA#{port => Port},
                           ?SEV_ANNOUNCE_READY(Tester, init, ServerSA),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (recv)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv)
                   end},
         #{desc => "try recv request (with nowait, expect select)",
           cmd  => fun(#{sock := Sock,
                         recv := Recv,
                         recv_sref := SR} = State) ->
                           case Recv(Sock) of
                               {select, SelectInfo} when SR =:= nowait ->
                                   {ok, State#{recv_select_info => SelectInfo}};
                               {select,
                                {select_info, _Tag, SR} = SelectInfo}
                                 when is_reference(SR) ->
                                   {ok, State#{recv_select_info => SelectInfo}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_select),
                           ok
                   end},
         #{desc => "await select message (without success)",
           cmd  => fun(#{sock := Sock}) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   {error, {unexpected_select, Ref}}
                           after 5000 ->
                                   ok
                           end
                   end},
         #{desc => "announce ready (no select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, no_select),
                           ok
                   end},
         #{desc => "await continue (cancel)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, cancel)
                   end},
         #{desc => "cancel",
           cmd  => fun(#{sock := Sock, recv_select_info := SelectInfo}) ->
                           ok = socket:cancel(Sock, SelectInfo)
                   end},
         #{desc => "announce ready (cancel)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, cancel),
                           ok
                   end},

         %% Termination
         #{desc => "await terminate (from tester)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   State2 = maps:remove(tester,    State),
                                   State3 = maps:remove(recv_stag, State2),
                                   State4 = maps:remove(recv_sref, State3),
                                   State5 = maps:remove(req_src,   State4),
                                   {ok, State5};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    TesterSeq = 
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, ServerSA} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_sa => ServerSA}}
                   end},

         %% The actual test
         #{desc => "order server continue (recv)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv),
                           ok
                   end},
         #{desc => "await server ready (recv select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, recv_select)
                   end},
         #{desc => "await server ready (no select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, no_select)
                   end},
         #{desc => "order server continue (cancel)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, cancel),
                           ok
                   end},
         #{desc => "await server ready (cancel)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, cancel)
                   end},

         %% Terminations
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Pid),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Pid} = State) ->
                           case ?SEV_AWAIT_TERMINATION(Pid) of
                               ok ->
                                   State1 = maps:remove(server, State),
                                   {ok, State1};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    ServerInitState = InitState,
    Server = ?SEV_START("server", ServerSeq, ServerInitState),

    i("start 'tester' evaluator"),
    TesterInitState = #{server => Server#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    i("await evaluator"),
    ok = ?SEV_AWAIT_FINISH([Server, Tester]).





%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make an async (Timeout = nowait) call to accept,
%% wait some time and then cancel. IPv4
%%
api_a_accept_cancel_tcp4(suite) ->
    [];
api_a_accept_cancel_tcp4(doc) ->
    [];
api_a_accept_cancel_tcp4(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_accept_cancel_tcp4,
           fun() ->
                   Accept = fun(Sock) ->
                                    case socket:accept(Sock, Nowait) of
                                        {ok, _} = OK ->
                                            OK;
                                        {select, _} = SELECT ->
                                            SELECT;
                                        {error, _} = ERROR ->
                                            ERROR
                                    end
                            end,
                   InitState = #{domain => inet,
                                 accept => Accept,
                                 accept_sref => Nowait},
                   ok = api_a_accept_cancel_tcp(InitState)
           end).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make an async (Timeout = nowait) call to accept,
%% wait some time and then cancel. IPv6
%%
api_a_accept_cancel_tcp6(suite) ->
    [];
api_a_accept_cancel_tcp6(doc) ->
    [];
api_a_accept_cancel_tcp6(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_accept_cancel_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Accept = fun(Sock) ->
                                    case socket:accept(Sock, Nowait) of
                                        {ok, _} = OK ->
                                            OK;
                                        {select, _} = SELECT ->
                                            SELECT;
                                        {error, _} = ERROR ->
                                            ERROR
                                    end
                            end,
                   InitState = #{domain => inet6,
                                 accept => Accept,
                                 accept_sref => Nowait},
                   ok = api_a_accept_cancel_tcp(InitState)
           end).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_a_accept_cancel_tcp(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{lsock := LSock, lsa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, lport := Port}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Port),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection (nowait)",
           cmd  => fun(#{lsock := LSock,
                         accept := Accept,
                         accept_sref := SR} = State) ->
                           case Accept(LSock) of
                               {select, {select_info, T, R} = SelectInfo}
                                 when SR =:= nowait ->
                                   ?SEV_IPRINT("accept select nowait: "
                                               "~n   T: ~p"
                                               "~n   R: ~p", [T, R]),
                                   {ok,
                                    State#{accept_select_info =>
                                               SelectInfo}};
                               {select, {select_info, T, SR} = SelectInfo}
                                 when is_reference(SR) ->
                                   ?SEV_IPRINT("accept select ref: "
                                               "~n   T: ~p"
                                               "~n   R: ~p", [T, SR]),
                                   {ok,
                                    State#{accept_select_info =>
                                               SelectInfo}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept_select),
                           ok
                   end},
         #{desc => "await select message (without success)",
           cmd  => fun(#{lsock := Sock}) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   {error, {unexpected_select, Ref}}
                           after 5000 ->
                                   ok
                           end
                   end},
         #{desc => "announce ready (no select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, no_select),
                           ok
                   end},
         #{desc => "await continue (cancel)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, cancel)
                   end},
         #{desc => "cancel",
           cmd  => fun(#{lsock := Sock, accept_select_info := SelectInfo}) ->
                           ok = socket:cancel(Sock, SelectInfo)
                   end},
         #{desc => "announce ready (cancel)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, cancel),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{lsock := Sock}) ->
                           socket:close(Sock)
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% *** The actual test ***
         #{desc => "order server to continue (with accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
         #{desc => "await server ready (accept select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, accept_select)
                   end},
         #{desc => "await server ready (no select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, no_select)
                   end},
         #{desc => "order server to continue (cancel)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, cancel),
                           ok
                   end},
         #{desc => "await server ready (cancel)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, cancel)
                   end},

         %% *** Termination ***
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start tester evaluator"),
    TesterInitState = #{server => Server#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Server, Tester]).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make an async (Timeout = nowait) call to recv,
%% wait some time and then cancel. IPv4
%%
api_a_recv_cancel_tcp4(suite) ->
    [];
api_a_recv_cancel_tcp4(doc) ->
    [];
api_a_recv_cancel_tcp4(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_recv_cancel_tcp4,
           fun() ->
                   Recv = fun(Sock) ->
                                  socket:recv(Sock, 0, Nowait)
                          end,
                   InitState = #{domain => inet,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_recv_cancel_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make an async (Timeout = nowait) call to recv,
%% wait some time and then cancel. IPv6
%%
api_a_recv_cancel_tcp6(suite) ->
    [];
api_a_recv_cancel_tcp6(doc) ->
    [];
api_a_recv_cancel_tcp6(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_recv_cancel_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Recv = fun(Sock) ->
                                  socket:recv(Sock, 0, Nowait)
                          end,
                   InitState = #{domain => inet6,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_recv_cancel_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make an async (Timeout = nowait) call to recvmsg,
%% wait some time and then cancel. IPv4
%%
api_a_recvmsg_cancel_tcp4(suite) ->
    [];
api_a_recvmsg_cancel_tcp4(doc) ->
    [];
api_a_recvmsg_cancel_tcp4(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_recvmsg_cancel_tcp4,
           fun() ->
                   Recv = fun(Sock) ->
                                  socket:recvmsg(Sock, Nowait)
                          end,
                   InitState = #{domain => inet,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_recv_cancel_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make an async (Timeout = nowait) call to recvmsg,
%% wait some time and then cancel. IPv6
%%
api_a_recvmsg_cancel_tcp6(suite) ->
    [];
api_a_recvmsg_cancel_tcp6(doc) ->
    [];
api_a_recvmsg_cancel_tcp6(Config) when is_list(Config) ->
    ?TT(?SECS(10)),
    Nowait = nowait(Config),
    tc_try(api_a_recvmsg_cancel_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Recv = fun(Sock) ->
                                  socket:recvmsg(Sock, Nowait)
                          end,
                   InitState = #{domain => inet6,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_recv_cancel_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_a_recv_cancel_tcp(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{lsock := LSock, lsa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, lport := Port}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Port),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:accept(LSock) of
                               {ok, CSock} ->
                                   {ok, State#{csock => CSock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept),
                           ok
                   end},

         #{desc => "await continue (nowait recv)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv)
                   end},
         #{desc => "try recv request (with nowait, expect select)",
           cmd  => fun(#{csock := Sock,
                         recv := Recv,
                         recv_sref := SR} = State) ->
                           case Recv(Sock) of
                               {select, {select_info, T, R} = SelectInfo}
                                 when SR =:= nowait ->
                                   ?SEV_IPRINT("recv select nowait: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [T, R]),
                                   {ok,
                                    State#{recv_select_info => SelectInfo}};
                               {select, {select_info, T, SR} = SelectInfo}
                                 when is_reference(SR) ->
                                   ?SEV_IPRINT("recv select ref: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [T, SR]),
                                   {ok,
                                    State#{recv_select_info => SelectInfo}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_select),
                           ok
                   end},
         #{desc => "await select message",
           cmd  => fun(#{csock := Sock}) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   {error, {unexpected_select, Ref}}
                           after 5000 ->
                                   ok
                           end
                   end},
         #{desc => "announce ready (no select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, no_select),
                           ok
                   end},
         #{desc => "await continue (cancel)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, cancel)
                   end},
         #{desc => "cancel",
           cmd  => fun(#{csock := Sock, recv_select_info := SelectInfo}) ->
                           ok = socket:cancel(Sock, SelectInfo)
                   end},
         #{desc => "announce ready (cancel)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, cancel),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close connection socket",
           cmd  => fun(#{csock := Sock}) ->
                           socket:close(Sock)
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{lsock := Sock}) ->
                           socket:close(Sock)
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    ClientSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           {Tester, Port} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_port => Port}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain := Domain, server_port := Port} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = LSA#{port => Port},
                           {ok, State#{local_sa => LSA, server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "await continue (connect)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, connect)
                   end},
         #{desc => "connect to server",
           cmd  => fun(#{sock := Sock, server_sa := SSA}) ->
                           socket:connect(Sock, SSA)
                   end},
         #{desc => "announce ready (connect)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock}) ->
                           socket:close(Sock)
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% Start the client
         #{desc => "order client start",
           cmd  => fun(#{client := Pid, server_port := Port} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Port),
                           ok
                   end},
         #{desc => "await client ready (init)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, init)
                   end},

         %% *** The actual test ***
         #{desc => "order server to continue (with accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
         #{desc => "order client to continue (connect)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, connect),
                           ok
                   end},
         #{desc => "await client ready (connect)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, client, connect)
                   end},
         #{desc => "await server ready (accept)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, accept)
                   end},

         #{desc => "order server to continue (recv)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv),
                           ok
                   end},
         #{desc => "await server ready (recv select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, recv_select)
                   end},
         #{desc => "await server ready (no select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, no_select)
                   end},
         #{desc => "order server to continue (send request)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, cancel),
                           ok
                   end},
         #{desc => "await server ready (cancel)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, server, cancel)
                   end},

         %% *** Termination ***
         #{desc => "order client to terminate",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client termination",
           cmd  => fun(#{client := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client, State),
                           {ok, State1}
                   end},
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start client evaluator"),
    Client = ?SEV_START("client", ClientSeq, InitState),

    i("start tester evaluator"),
    TesterInitState = #{server => Server#ev.pid,
                        client => Client#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make multiple async (Timeout = nowait) call(s) to recvfrom
%% (from *several* processes), wait some time and then cancel.
%% This should result in abort messages to the 'other' processes. IPv4
%%
api_a_mrecvfrom_cancel_udp4(suite) ->
    [];
api_a_mrecvfrom_cancel_udp4(doc) ->
    [];
api_a_mrecvfrom_cancel_udp4(Config) when is_list(Config) ->
    ?TT(?SECS(20)),
    Nowait = nowait(Config),
    tc_try(api_a_mrecvfrom_cancel_udp4,
           fun() ->
                   Recv = fun(Sock) ->
                                  case socket:recvfrom(Sock, 0, Nowait) of
                                      {ok, _} = OK ->
                                          OK;
                                      {select, _} = SELECT ->
                                          SELECT;
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_mrecv_cancel_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make multiple async (Timeout = nowait) call(s) to recvfrom
%% (from *several* processes), wait some time and then cancel.
%% This should result in abort messages to the 'other' processes. IPv6
%%
api_a_mrecvfrom_cancel_udp6(suite) ->
    [];
api_a_mrecvfrom_cancel_udp6(doc) ->
    [];
api_a_mrecvfrom_cancel_udp6(Config) when is_list(Config) ->
    ?TT(?SECS(20)),
    Nowait = nowait(Config),
    tc_try(api_a_mrecvfrom_cancel_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Recv = fun(Sock) ->
                                  case socket:recvfrom(Sock, 0, Nowait) of
                                      {ok, _} = OK ->
                                          OK;
                                      {select, _} = SELECT ->
                                          SELECT;
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet6,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_mrecv_cancel_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make multiple async (Timeout = nowait) call(s) to recvmsg
%% (from *several* processes), wait some time and then cancel.
%% This should result in abort messages to the 'other' processes. IPv4
%%
api_a_mrecvmsg_cancel_udp4(suite) ->
    [];
api_a_mrecvmsg_cancel_udp4(doc) ->
    [];
api_a_mrecvmsg_cancel_udp4(Config) when is_list(Config) ->
    ?TT(?SECS(20)),
    Nowait = nowait(Config),
    tc_try(api_a_mrecvmsg_cancel_udp4,
           fun() ->
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock, Nowait) of
                                      {ok, _} = OK ->
                                          OK;
                                      {select, _} = SELECT ->
                                          SELECT;
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_mrecv_cancel_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make multiple async (Timeout = nowait) call(s) to recvmsg
%% (from *several* processes), wait some time and then cancel.
%% This should result in abort messages to the 'other' processes. IPv6
%%
api_a_mrecvmsg_cancel_udp6(suite) ->
    [];
api_a_mrecvmsg_cancel_udp6(doc) ->
    [];
api_a_mrecvmsg_cancel_udp6(Config) when is_list(Config) ->
    ?TT(?SECS(20)),
    Nowait = nowait(Config),
    tc_try(api_a_mrecvmsg_cancel_udp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock, Nowait) of
                                      {ok, _} = OK ->
                                          OK;
                                      {select, _} = SELECT ->
                                          SELECT;
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet6,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_mrecv_cancel_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_a_mrecv_cancel_udp(InitState) ->
    ServerSeq = 
        [
         %% *** Wait for start order part ***
         #{desc => "await start",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester} = _State) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{local_sa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind socket (to local address)",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   {ok, State#{port => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, sock := Sock}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Sock),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (recv)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv)
                   end},
         #{desc => "try recv request (with nowait, expect select)",
           cmd  => fun(#{sock := Sock,
                         recv := Recv,
                         recv_sref := SR} = State) ->
                           case Recv(Sock) of
                               {select, SelectInfo}
                                 when SR =:= nowait ->
                                   {ok,
                                    State#{recv_select_info => SelectInfo}};
                               {select,
                                {select_info, _Tag, SR} = SelectInfo}
                                 when is_reference(SR) ->
                                   {ok,
                                    State#{recv_select_info => SelectInfo}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_select),
                           ok
                   end},
         #{desc => "await abort message",
           cmd  => fun(#{sock             := Sock,
                         recv_select_info := {select_info, _, Ref}} = State) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   {error, {unexpected_select, Ref}};
                               {'$socket', Sock, abort, {Ref, closed}} ->
                                   {ok, maps:remove(sock, State)}
                           after 5000 ->
                                   ?SEV_EPRINT("message queue: ~p", [mq()]),
                                   {error, timeout}
                           end
                   end},
         #{desc => "announce ready (abort)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, abort),
                           ok
                   end},

         %% Termination
         #{desc => "await terminate (from tester)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   State2 = maps:remove(tester,    State),
                                   State3 = maps:remove(recv_stag, State2),
                                   State4 = maps:remove(recv_sref, State3),
                                   State5 = maps:remove(req_src,   State4),
                                   {ok, State5};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    AltServerSeq = 
        [
         %% *** Wait for start order part ***
         #{desc => "await start",
           cmd  => fun(State) ->
                           {Tester, Sock} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, sock => Sock}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester} = _State) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (recv)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv)
                   end},
         #{desc => "try recv request (with nowait, expect select)",
           cmd  => fun(#{sock := Sock,
                         recv := Recv,
                         recv_sref := SR} = State) ->
                           case Recv(Sock) of
                               {select, SelectInfo} when SR =:= nowait ->
                                   {ok,
                                    State#{recv_select_info => SelectInfo}};
                               {select,
                                {select_info, _Tag, SR} = SelectInfo}
                                 when is_reference(SR) ->
                                   {ok,
                                    State#{recv_select_info => SelectInfo}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_select),
                           ok
                   end},
         #{desc => "await abort message",
           cmd  => fun(#{sock             := Sock,
                         recv_select_info := {select_info, _, Ref}} = State) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   {error, {unexpected_select, Ref}};
                               {'$socket', Sock, abort, {Ref, closed}} ->
                                   {ok, maps:remove(sock, State)}
                           after 5000 ->
                                   ?SEV_EPRINT("message queue: ~p", [mq()]),
                                   {error, timeout}
                           end
                   end},
         #{desc => "announce ready (abort)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, abort),
                           ok
                   end},

         %% Termination
         #{desc => "await terminate (from tester)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   ?SEV_IPRINT("terminating"),
                                   State1 = maps:remove(recv_select_info, State),
                                   State2 = maps:remove(tester,           State1),
                                   State3 = maps:remove(sock,             State2),
                                   {ok, State3};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],



    TesterSeq = 
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor alt-server 1",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor alt-server 2",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Sock} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{sock => Sock}}
                   end},

         %% Start the alt-server 1
         #{desc => "order alt-server 1 start",
           cmd  => fun(#{alt_server1 := Pid, sock := Sock} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Sock),
                           ok
                   end},
         #{desc => "await alt-server 1 ready (init)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, alt_server1, init)
                   end},

         %% Start the alt-server 2
         #{desc => "order alt-server 2 start",
           cmd  => fun(#{alt_server2 := Pid, sock := Sock} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Sock),
                           ok
                   end},
         #{desc => "await alt-server 2 ready (init)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, alt_server2, init)
                   end},


         %% The actual test
         #{desc => "order server continue (recv)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv),
                           ok
                   end},
         #{desc => "await server ready (recv select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, recv_select)
                   end},

         #{desc => "order alt-server 1 continue (recv)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv),
                           ok
                   end},
         #{desc => "await alt-server 1 ready (recv select)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server1, recv_select)
                   end},

         #{desc => "order alt-server 2 continue (recv)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv),
                           ok
                   end},
         #{desc => "await alt-server 2 ready (recv select)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server2, recv_select)
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "close the socket",
           cmd  => fun(#{sock := Sock} = _State) ->
                           socket:close(Sock)
                   end},

         #{desc => "await server ready (abort)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, abort)
                   end},
         #{desc => "await alt-server 1 ready (abort)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server1, abort)
                   end},
         #{desc => "await alt-server 2 ready (abort)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server2, abort)
                   end},

         %% Terminations
         #{desc => "order alt-server 2 to terminate",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Pid),
                           ok
                   end},
         #{desc => "await alt-server 2 termination",
           cmd  => fun(#{alt_server2 := Pid} = State) ->
                           case ?SEV_AWAIT_TERMINATION(Pid) of
                               ok ->
                                   State1 = maps:remove(alt_server2, State),
                                   {ok, State1};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "order alt-server 1 to terminate",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Pid),
                           ok
                   end},
         #{desc => "await alt-server 1 termination",
           cmd  => fun(#{alt_server1 := Pid} = State) ->
                           case ?SEV_AWAIT_TERMINATION(Pid) of
                               ok ->
                                   State1 = maps:remove(alt_server1, State),
                                   {ok, State1};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "order server to terminate",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Pid),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Pid} = State) ->
                           case ?SEV_AWAIT_TERMINATION(Pid) of
                               ok ->
                                   State1 = maps:remove(server, State),
                                   {ok, State1};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start alt-server 1 evaluator"),
    AltServer1 = ?SEV_START("alt_server1", AltServerSeq, InitState),

    i("start alt-server 2 evaluator"),
    AltServer2 = ?SEV_START("alt_server2", AltServerSeq, InitState),

    i("start 'tester' evaluator"),
    TesterInitState = #{server      => Server#ev.pid,
                        alt_server1 => AltServer1#ev.pid,
                        alt_server2 => AltServer2#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Server, AltServer1, AltServer2, Tester]).






%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make multiple async (Timeout = nowait) call(s) to accept
%% (from *several* processes), wait some time and then cancel,
%% This should result in abort messages to the 'other' processes. IPv4
%%
api_a_maccept_cancel_tcp4(suite) ->
    [];
api_a_maccept_cancel_tcp4(doc) ->
    [];
api_a_maccept_cancel_tcp4(Config) when is_list(Config) ->
    ?TT(?SECS(20)),
    Nowait = nowait(Config),
    tc_try(api_a_maccept_cancel_tcp4,
           fun() ->
                   Accept = fun(Sock) ->
                                    case socket:accept(Sock, Nowait) of
                                        {ok, _} = OK ->
                                            OK;
                                        {select, _} = SELECT ->
                                            SELECT;
                                        {error, _} = ERROR ->
                                            ERROR
                                    end
                            end,
                   InitState = #{domain => inet,
                                 accept => Accept,
                                 accept_sref => Nowait},
                   ok = api_a_maccept_cancel_tcp(InitState)
           end).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make multiple async (Timeout = nowait) call(s) to accept
%% (from *several* processes), wait some time and then cancel,
%% This should result in abort messages to the 'other' processes. IPv6
%%
api_a_maccept_cancel_tcp6(suite) ->
    [];
api_a_maccept_cancel_tcp6(doc) ->
    [];
api_a_maccept_cancel_tcp6(Config) when is_list(Config) ->
    ?TT(?SECS(20)),
    Nowait = nowait(Config),
    tc_try(api_a_maccept_cancel_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Accept = fun(Sock) ->
                                    case socket:accept(Sock, Nowait) of
                                        {ok, _} = OK ->
                                            OK;
                                        {select, _} = SELECT ->
                                            SELECT;
                                        {error, _} = ERROR ->
                                            ERROR
                                    end
                            end,
                   InitState = #{domain => inet6,
                                 accept => Accept,
                                 accept_sref => Nowait},
                   ok = api_a_maccept_cancel_tcp(InitState)
           end).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_a_maccept_cancel_tcp(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{lsock := LSock, lsa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, lsock := Sock}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Sock),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection (nowait)",
           cmd  => fun(#{lsock := LSock,
                         accept := Accept,
                         accept_sref := SR} = State) ->
                           case Accept(LSock) of
                               {select, {select_info, T, R} = SelectInfo}
                                 when SR =:= nowait ->
                                   ?SEV_IPRINT("accept select nowait: "
                                               "~n   T: ~p"
                                               "~n   R: ~p", [T, R]),
                                   {ok,
                                    State#{accept_select_info =>
                                               SelectInfo}};
                               {select, {select_info, T, SR} = SelectInfo}
                                 when is_reference(SR) ->
                                   ?SEV_IPRINT("accept select ref: "
                                               "~n   T: ~p"
                                               "~n   R: ~p", [T, SR]),
                                   {ok,
                                    State#{accept_select_info =>
                                               SelectInfo}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept_select),
                           ok
                   end},
         #{desc => "await select message (without success)",
           cmd  => fun(#{lsock              := Sock,
                         accept_select_info := {select_info, _, Ref}} = State) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   {error, {unexpected_select, Ref}};
                               {'$socket', Sock, abort, {Ref, closed}} ->
                                   {ok, maps:remove(lsock, State)}
                           after 5000 ->
                                   ?SEV_EPRINT("message queue: ~p", [mq()]),
                                   {error, timeout}
                           end
                   end},
         #{desc => "announce ready (abort)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, abort),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    AltServerSeq =
        [
         %% *** Wait for start order part ***
         #{desc => "await start",
           cmd  => fun(State) ->
                           {Tester, Sock} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, lsock => Sock}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester} = _State) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "try accept request (with nowait, expect select)",
           cmd  => fun(#{lsock := Sock,
                         accept := Accept,
                         accept_sref := SR} = State) ->
                           case Accept(Sock) of
                               {select, SelectInfo} when SR =:= nowait ->
                                   {ok,
                                    State#{accept_select_info =>
                                               SelectInfo}};
                               {select,
                                {select_info, _Tag, SR} = SelectInfo}
                                 when is_reference(SR) ->
                                   {ok,
                                    State#{accept_select_info =>
                                               SelectInfo}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept_select),
                           ok
                   end},
         #{desc => "await abort message",
           cmd  => fun(#{lsock              := Sock,
                         accept_select_info := {select_info, _, Ref}} = State) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   {error, {unexpected_select, Ref}};
                               {'$socket', Sock, abort, {Ref, closed}} ->
                                   {ok, maps:remove(sock, State)}
                           after 5000 ->
                                   ?SEV_EPRINT("message queue: ~p", [mq()]),
                                   {error, timeout}
                           end
                   end},
         #{desc => "announce ready (abort)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, abort),
                           ok
                   end},

         %% Termination
         #{desc => "await terminate (from tester)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   ?SEV_IPRINT("terminating"),
                                   State1 = maps:remove(tester,             State),
                                   State2 = maps:remove(accept_select_info, State1),
                                   State3 = maps:remove(lsock,              State2),
                                   {ok, State3};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor alt-server 1",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor alt-server 2",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Sock} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{sock => Sock}}
                   end},

         %% Start the alt-server 1
         #{desc => "order alt-server 1 start",
           cmd  => fun(#{alt_server1 := Pid, sock := Sock} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Sock),
                           ok
                   end},
         #{desc => "await alt-server 1 ready (init)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, alt_server1, init)
                   end},

         %% Start the alt-server 2
         #{desc => "order alt-server 2 start",
           cmd  => fun(#{alt_server2 := Pid, sock := Sock} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Sock),
                           ok
                   end},
         #{desc => "await alt-server 2 ready (init)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, alt_server2, init)
                   end},


         %% *** The actual test ***
         #{desc => "order server continue (accept)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, accept),
                           ok
                   end},
         #{desc => "await server ready (accept select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, accept_select)
                   end},

         #{desc => "order alt-server 1 continue (accept)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, accept),
                           ok
                   end},
         #{desc => "await alt-server 1 ready (accept select)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server1, accept_select)
                   end},

         #{desc => "order alt-server 2 continue (accept)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, accept),
                           ok
                   end},
         #{desc => "await alt-server 2 ready (accept select)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server2, accept_select)
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "close the socket",
           cmd  => fun(#{sock := Sock} = _State) ->
                           socket:close(Sock)
                   end},

         #{desc => "await server ready (abort)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, abort)
                   end},
         #{desc => "await alt-server 1 ready (abort)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server1, abort)
                   end},
         #{desc => "await alt-server 2 ready (abort)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server2, abort)
                   end},


         %% *** Termination ***
         #{desc => "order alt-server 2 to terminate",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Pid),
                           ok
                   end},
         #{desc => "await alt-server 2 termination",
           cmd  => fun(#{alt_server2 := Pid} = State) ->
                           case ?SEV_AWAIT_TERMINATION(Pid) of
                               ok ->
                                   State1 = maps:remove(alt_server2, State),
                                   {ok, State1};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "order alt-server 1 to terminate",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Pid),
                           ok
                   end},
         #{desc => "await alt-server 1 termination",
           cmd  => fun(#{alt_server1 := Pid} = State) ->
                           case ?SEV_AWAIT_TERMINATION(Pid) of
                               ok ->
                                   State1 = maps:remove(alt_server1, State),
                                   {ok, State1};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start alt-server 1 evaluator"),
    AltServer1 = ?SEV_START("alt_server1", AltServerSeq, InitState),

    i("start alt-server 2 evaluator"),
    AltServer2 = ?SEV_START("alt_server2", AltServerSeq, InitState),

    i("start tester evaluator"),
    TesterInitState = #{server      => Server#ev.pid,
                        alt_server1 => AltServer1#ev.pid,
                        alt_server2 => AltServer2#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Server, AltServer1, AltServer2, Tester]).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make multiple async (Timeout = nowait) call(s) to recv
%% (from *several* processes), wait some time and then cancel,
%% This should result in abort messages to the 'other' processes. IPv4
%%
api_a_mrecv_cancel_tcp4(suite) ->
    [];
api_a_mrecv_cancel_tcp4(doc) ->
    [];
api_a_mrecv_cancel_tcp4(Config) when is_list(Config) ->
    ?TT(?SECS(20)),
    Nowait = nowait(Config),
    tc_try(api_a_mrecv_cancel_tcp4,
           fun() ->
                   Recv = fun(Sock) ->
                                  socket:recv(Sock, 0, Nowait)
                          end,
                   InitState = #{domain => inet,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_mrecv_cancel_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make multiple async (Timeout = nowait) call(s) to recv
%% (from *several* processes), wait some time and then cancel,
%% This should result in abort messages to the 'other' processes. IPv6
%%
api_a_mrecv_cancel_tcp6(suite) ->
    [];
api_a_mrecv_cancel_tcp6(doc) ->
    [];
api_a_mrecv_cancel_tcp6(Config) when is_list(Config) ->
    ?TT(?SECS(20)),
    Nowait = nowait(Config),
    tc_try(api_a_mrecv_cancel_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Recv = fun(Sock) ->
                                  socket:recv(Sock, 0, Nowait)
                          end,
                   InitState = #{domain => inet6,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_mrecv_cancel_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make multiple async (Timeout = nowait) call(s) to recvmsg
%% (from *several* processes), wait some time and then cancel,
%% This should result in abort messages to the 'other' processes. IPv4
%%
api_a_mrecvmsg_cancel_tcp4(suite) ->
    [];
api_a_mrecvmsg_cancel_tcp4(doc) ->
    [];
api_a_mrecvmsg_cancel_tcp4(Config) when is_list(Config) ->
    ?TT(?SECS(20)),
    Nowait = nowait(Config),
    tc_try(api_a_mrecvmsg_cancel_tcp4,
           fun() ->
                   Recv = fun(Sock) ->
                                  socket:recvmsg(Sock, Nowait)
                          end,
                   InitState = #{domain => inet,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_mrecv_cancel_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Basically we make multiple async (Timeout = nowait) call(s) to recvmsg
%% (from *several* processes), wait some time and then cancel,
%% This should result in abort messages to the 'other' processes. IPv6
%%
api_a_mrecvmsg_cancel_tcp6(suite) ->
    [];
api_a_mrecvmsg_cancel_tcp6(doc) ->
    [];
api_a_mrecvmsg_cancel_tcp6(Config) when is_list(Config) ->
    ?TT(?SECS(20)),
    Nowait = nowait(Config),
    tc_try(api_a_mrecvmsg_cancel_tcp6,
           fun() -> has_support_ipv6() end,
           fun() ->
                   Recv = fun(Sock) ->
                                  socket:recvmsg(Sock, Nowait)
                          end,
                   InitState = #{domain => inet6,
                                 recv => Recv,
                                 recv_sref => Nowait},
                   ok = api_a_mrecv_cancel_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_a_mrecv_cancel_tcp(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{lsock := LSock, lsa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, lport := Port}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Port),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:accept(LSock) of
                               {ok, CSock} ->
                                   {ok, State#{csock => CSock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept)",
           cmd  => fun(#{tester := Tester, csock := Sock}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept, Sock),
                           ok
                   end},

         #{desc => "await continue (nowait recv)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv)
                   end},
         #{desc => "try recv request (with nowait, expect select)",
           cmd  => fun(#{csock := Sock,
                         recv := Recv,
                         recv_sref := SR} = State) ->
                           case Recv(Sock) of
                               {select, {select_info, T, R} = SelectInfo}
                                 when SR =:= nowait ->
                                   ?SEV_IPRINT("recv select nowait: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [T, R]),
                                   {ok, State#{recv_select_info => SelectInfo}};
                               {select, {select_info, T, SR} = SelectInfo}
                                 when is_reference(SR) ->
                                   ?SEV_IPRINT("recv select nowait: "
                                               "~n   Tag: ~p"
                                               "~n   Ref: ~p", [T, SR]),
                                   {ok, State#{recv_select_info => SelectInfo}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_select),
                           ok
                   end},
         #{desc => "await select message",
           cmd  => fun(#{csock            := Sock,
                         recv_select_info := {select_info, _, Ref}} = State) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   {error, {unexpected_select, Ref}};
                               {'$socket', Sock, abort, {Ref, closed}} ->
                                   {ok, maps:remove(sock, State)}
                           after 5000 ->
                                   ok
                           end
                   end},
         #{desc => "announce ready (abort)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, abort),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{lsock := Sock}) ->
                           socket:close(Sock)
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    AltServerSeq =
        [
         %% *** Wait for start order part ***
         #{desc => "await start",
           cmd  => fun(State) ->
                           {Tester, Sock} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, sock => Sock}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester} = _State) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (recv)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv)
                   end},
         #{desc => "try recv request (with nowait, expect select)",
           cmd  => fun(#{sock := Sock,
                         recv := Recv,
                         recv_sref := SR} = State) ->
                           case Recv(Sock) of
                               {select, SelectInfo} when SR =:= nowait ->
                                   {ok,
                                    State#{recv_select_info => SelectInfo}};
                               {select,
                                {select_info, _Tag, SR} = SelectInfo}
                                 when is_reference(SR) ->
                                   {ok,
                                    State#{recv_select_info => SelectInfo}};
                               {ok, X} ->
                                   {error, {unexpected_select_info, X}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv_select)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_select),
                           ok
                   end},
         #{desc => "await abort message",
           cmd  => fun(#{sock             := Sock,
                         recv_select_info := {select_info, _, Ref}} = State) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   {error, {unexpected_select, Ref}};
                               {'$socket', Sock, abort, {Ref, closed}} ->
                                   {ok, maps:remove(sock, State)}
                           after 5000 ->
                                   ?SEV_EPRINT("message queue: ~p", [mq()]),
                                   {error, timeout}
                           end
                   end},
         #{desc => "announce ready (abort)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, abort),
                           ok
                   end},

         %% Termination
         #{desc => "await terminate (from tester)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   ?SEV_IPRINT("terminating"),
                                   State1 = maps:remove(recv_select_info, State),
                                   State2 = maps:remove(tester,           State1),
                                   State3 = maps:remove(sock,             State2),
                                   {ok, State3};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    ClientSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           {Tester, Port} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_port => Port}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain := Domain, server_port := Port} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = LSA#{port => Port},
                           {ok, State#{local_sa => LSA, server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "await continue (connect)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, connect)
                   end},
         #{desc => "connect to server",
           cmd  => fun(#{sock := Sock, server_sa := SSA}) ->
                           socket:connect(Sock, SSA)
                   end},
         #{desc => "announce ready (connect)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock}) ->
                           socket:close(Sock)
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor alt-server 1",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor alt-server 2",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% Start the client
         #{desc => "order client start",
           cmd  => fun(#{client := Pid, server_port := Port} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Port),
                           ok
                   end},
         #{desc => "await client ready (init)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, init)
                   end},

         #{desc => "order server to continue (with accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
         #{desc => "order client to continue (connect)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, connect),
                           ok
                   end},
         #{desc => "await client ready (connect)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ?SEV_AWAIT_READY(Pid, client, connect)
                   end},
         #{desc => "await server ready (accept)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Sock} = ?SEV_AWAIT_READY(Pid, server, accept),
                           {ok, State#{sock => Sock}}
                   end},

         %% Start the alt server 1
         #{desc => "order alt-server 1 start",
           cmd  => fun(#{alt_server1 := Pid, sock := Sock} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Sock),
                           ok
                   end},
         #{desc => "await alt-server 1 ready (init)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server1, init)
                   end},

         %% Start the alt server 2
         #{desc => "order alt-server 2 start",
           cmd  => fun(#{alt_server2 := Pid, sock := Sock} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Sock),
                           ok
                   end},
         #{desc => "await alt-server 2 ready (init)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server2, init)
                   end},


         %% *** The actual test ***
         #{desc => "order server continue (recv)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv),
                           ok
                   end},
         #{desc => "await server ready (recv select)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, recv_select)
                   end},

         #{desc => "order alt-server 1 continue (recv)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv),
                           ok
                   end},
         #{desc => "await alt-server 1 ready (recv select)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server1, recv_select)
                   end},

         #{desc => "order alt-server 2 continue (recv)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Pid, recv),
                           ok
                   end},
         #{desc => "await alt-server 2 ready (recv select)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server2, recv_select)
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "close the socket",
           cmd  => fun(#{sock := Sock} = _State) ->
                           socket:close(Sock)
                   end},

         #{desc => "await server ready (abort)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, abort)
                   end},
         #{desc => "await alt-server 1 ready (abort)",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server1, abort)
                   end},
         #{desc => "await alt-server 2 ready (abort)",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, alt_server2, abort)
                   end},

         %% Terminations
         #{desc => "order client to terminate",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client termination",
           cmd  => fun(#{client := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client, State),
                           {ok, State1}
                   end},

         #{desc => "order alt-server 2 to terminate",
           cmd  => fun(#{alt_server2 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Pid),
                           ok
                   end},
         #{desc => "await alt-server 2 termination",
           cmd  => fun(#{alt_server2 := Pid} = State) ->
                           case ?SEV_AWAIT_TERMINATION(Pid) of
                               ok ->
                                   State1 = maps:remove(alt_server2, State),
                                   {ok, State1};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "order alt-server 1 to terminate",
           cmd  => fun(#{alt_server1 := Pid} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Pid),
                           ok
                   end},
         #{desc => "await alt-server 1 termination",
           cmd  => fun(#{alt_server1 := Pid} = State) ->
                           case ?SEV_AWAIT_TERMINATION(Pid) of
                               ok ->
                                   State1 = maps:remove(alt_server1, State),
                                   {ok, State1};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "order server to terminate",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Pid),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Pid} = State) ->
                           case ?SEV_AWAIT_TERMINATION(Pid) of
                               ok ->
                                   State1 = maps:remove(server, State),
                                   {ok, State1};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start alt-server 1 evaluator"),
    AltServer1 = ?SEV_START("alt_server1", AltServerSeq, InitState),

    i("start alt-server 2 evaluator"),
    AltServer2 = ?SEV_START("alt_server2", AltServerSeq, InitState),

    i("start client evaluator"),
    Client = ?SEV_START("client", ClientSeq, InitState),

    i("start tester evaluator"),
    TesterInitState = #{server      => Server#ev.pid,
                        alt_server1 => AltServer1#ev.pid,
                        alt_server2 => AltServer2#ev.pid,
                        client      => Client#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Server, AltServer1, AltServer2, Client, Tester]).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                                                                     %%
%%                           API OPTIONS                               %%
%%                                                                     %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Perform some simple getopt and setopt with the level = otp options
api_opt_simple_otp_options(suite) ->
    [];
api_opt_simple_otp_options(doc) ->
    [];
api_opt_simple_otp_options(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_simple_otp_options,
           fun() -> api_opt_simple_otp_options() end).

api_opt_simple_otp_options() ->
    Get = fun(S, Key) ->
                  socket:getopt(S, otp, Key)
          end,
    Set = fun(S, Key, Val) ->
                  socket:setopt(S, otp, Key, Val)
          end,

    Seq = 
        [
         %% *** Init part ***
         #{desc => "create socket",
           cmd  => fun(#{domain   := Domain, 
                         type     := Type,
                         protocol := Protocol} = State) ->
                           Sock = sock_open(Domain, Type, Protocol),
                           {ok, State#{sock => Sock}}
                   end},
         #{desc => "create dummy process",
           cmd  => fun(State) ->
                           Pid =  spawn_link(fun() -> 
                                                     put(sname, "dummy"),
                                                     receive
                                                         die -> 
                                                             exit(normal) 
                                                     end 
                                             end),
                           {ok, State#{dummy => Pid}}
                   end},

         %% *** Check iow part ***
         #{desc => "get iow",
           cmd  => fun(#{sock := Sock} = State) ->
                           case Get(Sock, iow) of
                               {ok, IOW} when is_boolean(IOW) ->
                                   {ok, State#{iow => IOW}};
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% #{desc => "enable debug",
         %%   cmd  => fun(#{sock := Sock}) ->
         %%                   ok = socket:setopt(Sock, otp, debug, true)
         %%           end},

         #{desc => "set (new) iow",
           cmd  => fun(#{sock := Sock, iow := OldIOW} = State) ->
                           NewIOW = not OldIOW,
                           case Set(Sock, iow, NewIOW) of
                               ok ->
                                   {ok, State#{iow => NewIOW}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "get (new) iow",
           cmd  => fun(#{sock := Sock, iow := IOW}) ->
                           case Get(Sock, iow) of
                               {ok, IOW} ->
                                   ok;
                               {ok, _} = OK->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** Check rcvbuf part ***
         #{desc => "get rcvbuf",
           cmd  => fun(#{sock := Sock} = State) ->
                           case Get(Sock, rcvbuf) of
                               {ok, RcvBuf} when is_integer(RcvBuf) ->
                                   {ok, State#{rcvbuf => RcvBuf}};
                               {ok, {N, RcvBuf} = V} when is_integer(N) andalso 
                                                          is_integer(RcvBuf) ->
                                   {ok, State#{rcvbuf => V}};
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "set (new) rcvbuf",
           cmd  => fun(#{sock := Sock, rcvbuf := {OldN, OldRcvBuf}} = State) ->
                           NewRcvBuf = {OldN+2, OldRcvBuf + 1024},
                           case Set(Sock, rcvbuf, NewRcvBuf) of
                               ok ->
                                   {ok, State#{rcvbuf => NewRcvBuf}};
                               {error, _} = ERROR ->
                                   ERROR
                           end;
                      (#{sock := Sock, rcvbuf := OldRcvBuf} = State) when is_integer(OldRcvBuf) ->
                           NewRcvBuf = 2 * OldRcvBuf,
                           case Set(Sock, rcvbuf, NewRcvBuf) of
                               ok ->
                                   {ok, State#{rcvbuf => NewRcvBuf}};
                               {error, _} = ERROR ->
                                   ERROR
                           end;
                      (#{sock := Sock, rcvbuf := OldRcvBuf,
                         type := stream,
                         protocol := tcp} = State) when is_integer(OldRcvBuf) ->
                           NewRcvBuf = {2, OldRcvBuf},
                           case Set(Sock, rcvbuf, NewRcvBuf) of
                               ok ->
                                   {ok, State#{rcvbuf => NewRcvBuf}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "get (new) rcvbuf",
           cmd  => fun(#{sock := Sock, rcvbuf := RcvBuf}) ->
                           case Get(Sock, rcvbuf) of
                               {ok, RcvBuf} ->
                                   ok;
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** Check rcvctrlbuf part ***
         #{desc => "get rcvctrlbuf",
           cmd  => fun(#{sock := Sock} = State) ->
                           case Get(Sock, rcvctrlbuf) of
                               {ok, RcvCtrlBuf} when is_integer(RcvCtrlBuf) ->
                                   {ok, State#{rcvctrlbuf => RcvCtrlBuf}};
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "set (new) rcvctrlbuf",
           cmd  => fun(#{sock := Sock, rcvctrlbuf := OldRcvCtrlBuf} = State) ->
                           NewRcvCtrlBuf = 2 * OldRcvCtrlBuf,
                           case Set(Sock, rcvctrlbuf, NewRcvCtrlBuf) of
                               ok ->
                                   {ok, State#{rcvctrlbuf => NewRcvCtrlBuf}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "get (new) rcvctrlbuf",
           cmd  => fun(#{sock := Sock, rcvctrlbuf := RcvCtrlBuf}) ->
                           case Get(Sock, rcvctrlbuf) of
                               {ok, RcvCtrlBuf} ->
                                   ok;
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         %% *** Check rcvctrlbuf part ***
         #{desc => "get rcvctrlbuf",
           cmd  => fun(#{sock := Sock} = State) ->
                           case Get(Sock, rcvctrlbuf) of
                               {ok, RcvCtrlBuf} when is_integer(RcvCtrlBuf) ->
                                   {ok, State#{rcvctrlbuf => RcvCtrlBuf}};
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "set (new) rcvctrlbuf",
           cmd  => fun(#{sock := Sock, rcvctrlbuf := OldRcvCtrlBuf} = State) ->
                           NewRcvCtrlBuf = 2 * OldRcvCtrlBuf,
                           case Set(Sock, rcvctrlbuf, NewRcvCtrlBuf) of
                               ok ->
                                   {ok, State#{rcvctrlbuf => NewRcvCtrlBuf}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "get (new) rcvctrlbuf",
           cmd  => fun(#{sock := Sock, rcvctrlbuf := RcvCtrlBuf}) ->
                           case Get(Sock, rcvctrlbuf) of
                               {ok, RcvCtrlBuf} ->
                                   ok;
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% *** Check sndctrlbuf part ***
         #{desc => "get sndctrlbuf",
           cmd  => fun(#{sock := Sock} = State) ->
                           case Get(Sock, sndctrlbuf) of
                               {ok, SndCtrlBuf} when is_integer(SndCtrlBuf) ->
                                   {ok, State#{sndctrlbuf => SndCtrlBuf}};
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "set (new) sndctrlbuf",
           cmd  => fun(#{sock := Sock, sndctrlbuf := OldSndCtrlBuf} = State) ->
                           NewSndCtrlBuf = 2 * OldSndCtrlBuf,
                           case Set(Sock, sndctrlbuf, NewSndCtrlBuf) of
                               ok ->
                                   {ok, State#{sndctrlbuf => NewSndCtrlBuf}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "get (new) sndctrlbuf",
           cmd  => fun(#{sock := Sock, sndctrlbuf := SndCtrlBuf}) ->
                           case Get(Sock, sndctrlbuf) of
                               {ok, SndCtrlBuf} ->
                                   ok;
                               {ok, _} = OK->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** Check controlling-process part ***
         #{desc => "verify self as controlling-process",
           cmd  => fun(#{sock := Sock}) ->
                           Self = self(),
                           case Get(Sock, controlling_process) of
                               {ok, Self} ->
                                   ok;
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "set dummy as controlling-process",
           cmd  => fun(#{sock := Sock, dummy := Dummy}) ->
                           Set(Sock, controlling_process, Dummy)
                   end},
         #{desc => "verify dummy as controlling-process",
           cmd  => fun(#{sock := Sock, dummy := Dummy}) ->
                           case Get(Sock, controlling_process) of
                               {ok, Dummy} ->
                                   ok;
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         #{desc => "finish",
           cmd  => fun(_) ->
                           {ok, normal}
                   end}
        ],

    i("start tcp (stream) evaluator"),
    InitState1 = #{domain => inet, type => stream, protocol => tcp},
    Tester1 = ?SEV_START("tcp-tester", Seq, InitState1),
    i("await tcp evaluator"),
    ok = ?SEV_AWAIT_FINISH([Tester1]),

    i("start udp (dgram) socket"),
    InitState2 = #{domain => inet, type => dgram, protocol => udp},
    Tester2 = ?SEV_START("udp-tester", Seq, InitState2),
    i("await udp evaluator"),
    ok = ?SEV_AWAIT_FINISH([Tester2]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Perform some simple getopt and setopt otp meta option
api_opt_simple_otp_meta_option(suite) ->
    [];
api_opt_simple_otp_meta_option(doc) ->
    [];
api_opt_simple_otp_meta_option(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_simple_otp_meta_option,
           fun() -> api_opt_simple_otp_meta_option() end).

api_opt_simple_otp_meta_option() ->
    Get = fun(S) ->
                  socket:getopt(S, otp, meta)
          end,
    Set = fun(S, Val) ->
                  socket:setopt(S, otp, meta, Val)
          end,

    MainSeq =
        [
         #{desc => "monitor helper",
           cmd => fun(#{helper := Pid}) ->
                          _ = erlang:monitor(process, Pid),
                          ok
                  end},

         #{desc => "create socket",
           cmd  => fun(#{domain   := Domain,
                         type     := Type,
                         protocol := Protocol} = State) ->
                           Sock = sock_open(Domain, Type, Protocol),
                           {ok, State#{sock => Sock}}
                   end},

         #{desc => "get default",
           cmd => fun(#{sock := Sock}) ->
                          case Get(Sock) of
                              {ok, undefined} ->
                                  ok;
                              {ok, _} = OK ->
                                  {error, OK};
                              {error, _} = ERROR ->
                                  ERROR
                          end
                  end},

         #{desc => "set value",
           cmd => fun(#{sock := Sock} = State) ->
                          Value = make_ref(),
                          case Set(Sock, Value) of
                              ok ->
                                  {ok, State#{value => Value}};
                              {error, _} = ERROR ->
                                  ERROR
                          end
                  end},

         #{desc => "get value",
           cmd => fun(#{sock := Sock, value := Value}) ->
                          case Get(Sock) of
                              {ok, Value} ->
                                  ok;
                              {ok, _} = OK ->
                                  {error, OK};
                              {error, _} = ERROR ->
                                  ERROR
                          end
                  end},

         #{desc => "set complex value",
           cmd => fun(#{sock := Sock} = State) ->
                          Value =
                              #{a => 1,
                                b => {2, 3},
                                c => make_ref(),
                                d => self(),
                                e => State},
                          case Set(Sock, Value) of
                              ok ->
                                  {ok, State#{value := Value}};
                              {error, _} = ERROR ->
                                  ERROR
                          end
                  end},

         #{desc => "get complex value",
           cmd => fun(#{sock := Sock, value := Value}) ->
                          case Get(Sock) of
                              {ok, Value} ->
                                  ok;
                              {ok, _} = OK->
                                  {error, OK};
                              {error, _} = ERROR ->
                                  ERROR
                          end
                  end},

         #{desc => "start helper",
           cmd => fun(#{helper := Pid,  sock := Sock, value := Value}) ->
                          ?SEV_ANNOUNCE_START(Pid, {Sock, Value}),
                          ok
                  end},

         #{desc => "wait for helper ready",
           cmd => fun(#{helper := Pid}) ->
                          ?SEV_AWAIT_READY(Pid, helper, test)
                  end},

         #{desc => "socket close",
           cmd => fun(#{sock := Sock}) ->
                          socket:close(Sock)
                  end},

        ?SEV_FINISH_NORMAL],

    HelperSeq =
        [#{desc => "await start",
           cmd => fun (State) ->
                          {Main, {Sock, Value}} = ?SEV_AWAIT_START(),
                          {ok, State#{main => Main,
                                      sock => Sock,
                                      value => Value}}
                  end},
         #{desc => "monitor main",
           cmd => fun(#{main := Main}) ->
                          _ = erlang:monitor(process, Main),
                          ok
                  end}

         #{desc => "get value",
           cmd => fun(#{sock := Sock, value := Value}) ->
                          case Get(Sock) of
                              {ok, Value} ->
                                  ok;
                              {ok, _} = OK->
                                  {error, OK};
                              {error, _} = ERROR ->
                                  ERROR
                          end
                  end},

         #{desc => "set and fail",
           cmd => fun(#{sock := Sock}) ->
                          Value = self(),
                          case Set(Sock, Value) of
                              ok ->
                                  {error, only_owner_may_set};
                              {error, {invalid, not_owner}} ->
                                  ok;
                              {error, _} = ERROR ->
                                  ERROR
                          end
                  end},

         #{desc => "announce ready (test)",
           cmd  => fun(#{main := Main}) ->
                           ?SEV_ANNOUNCE_READY(Main, test),
                           ok
                   end},

         ?SEV_FINISH_NORMAL],


    i("start tcp helper evaluator"),
    Helper = ?SEV_START("tcp-helper", HelperSeq, #{}),

    i("start tcp main evaluator"),
    MainState = #{domain => inet, type => stream, protocol => tcp,
                  helper => Helper#ev.pid},
    Main = ?SEV_START("tcp-main", MainSeq, MainState),

    i("await tcp evaluators"),
    ok = ?SEV_AWAIT_FINISH([Helper, Main]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Perform some simple operations with the rcvbuf otp option
%% The operations we test here are only for type = stream and
%% protocol = tcp.
api_opt_simple_otp_rcvbuf_option(suite) ->
    [];
api_opt_simple_otp_rcvbuf_option(doc) ->
    [];
api_opt_simple_otp_rcvbuf_option(_Config) when is_list(_Config) ->
    ?TT(?SECS(15)),
    tc_try(api_opt_simple_otp_rcvbuf_option,
           fun() -> api_opt_simple_otp_rcvbuf_option() end).

api_opt_simple_otp_rcvbuf_option() ->
    Get = fun(S) ->
                  socket:getopt(S, otp, rcvbuf)
          end,
    Set = fun(S, Val) ->
                  socket:setopt(S, otp, rcvbuf, Val)
          end,

    ServerSeq = 
        [
         %% *** Wait for start order part ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester} = _State) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{local_sa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{lsock := LSock, local_sa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester   := Tester,
                         local_sa := LocalSA,
                         lport    := Port}) ->
                           ServerSA = LocalSA#{port => Port},
                           ?SEV_ANNOUNCE_READY(Tester, init, ServerSA),
                           ok
                   end},


         %% *** The actual test part ***
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "attempt to accept",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:accept(LSock) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept),
                           ok
                   end},

         %% Recv with default size for (otp) rcvbuf
         #{desc => "await continue (recv initial)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_CONTINUE(Tester, tester, recv) of
                               {ok, MsgSz} ->
                                   ?SEV_IPRINT("MsgSz: ~p", [MsgSz]),
                                   {ok, State#{msg_sz => MsgSz}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "attempt to recv",
           cmd  => fun(#{sock := Sock, msg_sz := MsgSz} = _State) ->
                           ?SEV_IPRINT("try recv ~w bytes when rcvbuf is ~s", 
                                       [MsgSz,
                                        case Get(Sock) of
                                            {ok, RcvBuf} -> f("~w", [RcvBuf]);
                                            {error, _}   -> "-"
                                        end]),
                           case socket:recv(Sock) of
                               {ok, Data} when (size(Data) =:= MsgSz) ->
                                   ok;
                               {ok, Data} ->
                                   {error, {invalid_msg_sz, MsgSz, size(Data)}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv initial)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv),
                           ok
                   end},

         %% Recv with new size (1) for (otp) rcvbuf
         #{desc => "await continue (recv 1)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_CONTINUE(Tester, tester, recv) of
                               {ok, NewRcvBuf} ->
                                   ?SEV_IPRINT("set new rcvbuf: ~p", [NewRcvBuf]),
                                   {ok, State#{rcvbuf => NewRcvBuf}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "attempt to setopt rcvbuf",
           cmd  => fun(#{sock := Sock, rcvbuf := NewRcvBuf} = _State) ->
                           case Set(Sock, NewRcvBuf) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "attempt to recv",
           cmd  => fun(#{sock := Sock, msg_sz := MsgSz} = _State) ->
                           case socket:recv(Sock) of
                               {ok, Data} when (size(Data) =:= MsgSz) ->
                                   ok;
                               {ok, Data} ->
                                   {error, {invalid_msg_sz, MsgSz, size(Data)}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv 1)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv),
                           ok
                   end},

         %% Recv with new size (2) for (otp) rcvbuf
         #{desc => "await continue (recv 2)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_CONTINUE(Tester, tester, recv) of
                               {ok, NewRcvBuf} ->
                                   ?SEV_IPRINT("set new rcvbuf: ~p", [NewRcvBuf]),
                                   {ok, State#{rcvbuf => NewRcvBuf}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "attempt to setopt rcvbuf",
           cmd  => fun(#{sock := Sock, rcvbuf := NewRcvBuf} = _State) ->
                           case Set(Sock, NewRcvBuf) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "attempt to recv",
           cmd  => fun(#{sock := Sock, msg_sz := MsgSz} = _State) ->
                           case socket:recv(Sock) of
                               {ok, Data} when (size(Data) =:= MsgSz) ->
                                   ok;
                               {ok, Data} ->
                                   {error, {invalid_msg_sz, MsgSz, size(Data)}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv 2)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv),
                           ok
                   end},

         %% Recv with new size (3) for (otp) rcvbuf
         #{desc => "await continue (recv 3, truncated)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_CONTINUE(Tester, tester, recv) of
                               {ok, {ExpSz, NewRcvBuf}} ->
                                   {ok, State#{msg_sz => ExpSz,
                                               rcvbuf => NewRcvBuf}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "attempt to setopt rcvbuf",
           cmd  => fun(#{sock := Sock, rcvbuf := NewRcvBuf} = _State) ->
                           case Set(Sock, NewRcvBuf) of
                               ok ->
                                   ?SEV_IPRINT("set new rcvbuf: ~p", [NewRcvBuf]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "attempt to recv",
           cmd  => fun(#{sock := Sock, msg_sz := MsgSz} = _State) ->
                           ?SEV_IPRINT("try recv ~w bytes of data", [MsgSz]),
                           case socket:recv(Sock) of
                               {ok, Data} when (size(Data) =:= MsgSz) ->
                                   ok;
                               {ok, Data} ->
                                   {error, {invalid_msg_sz, MsgSz, size(Data)}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv),
                           ok
                   end},


         %% Termination
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket(s)",
           cmd  => fun(#{lsock := LSock, sock := Sock} = State) ->
                           sock_close(Sock),
                           sock_close(LSock),
                           State1 = maps:remove(sock,  State),
                           State2 = maps:remove(lport, State1),
                           State3 = maps:remove(lsock, State2),
                           {ok, State3}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    ClientSeq =
        [
         %% *** Wait for start order part ***
         #{desc => "await start",
           cmd  => fun(State) ->
                           {Tester, ServerSA} = ?SEV_AWAIT_START(),
                           {ok, State#{tester    => Tester,
                                       server_sa => ServerSA}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester} = _State) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{local_sa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (connect)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, connect)
                   end},
         #{desc => "connect to server",
           cmd  => fun(#{sock := Sock, server_sa := SSA}) ->
                           socket:connect(Sock, SSA)
                   end},
         #{desc => "announce ready (connect)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect),
                           ok
                   end},

         #{desc => "await continue (send initial)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_CONTINUE(Tester, tester, send) of
                               {ok, Data} ->
                                   {ok, State#{data => Data}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "send (initial) data to server",
           cmd  => fun(#{sock := Sock, data := Data} = _State) ->
                           ?SEV_IPRINT("try send ~w bytes", [size(Data)]),
                           socket:send(Sock, Data)
                   end},
         #{desc => "announce ready (send initial)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send),
                           ok
                   end},

         #{desc => "await continue (send 1)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send)
                   end},
         #{desc => "send (1) data to server",
           cmd  => fun(#{sock := Sock, data := Data}) ->
                           ?SEV_IPRINT("try send ~w bytes", [size(Data)]),
                           socket:send(Sock, Data)
                   end},
         #{desc => "announce ready (send 1)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send),
                           ok
                   end},

         #{desc => "await continue (send 2)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send)
                   end},
         #{desc => "send (2) data to server",
           cmd  => fun(#{sock := Sock, data := Data}) ->
                           ?SEV_IPRINT("try send ~w bytes", [size(Data)]),
                           socket:send(Sock, Data)
                   end},
         #{desc => "announce ready (send 2)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send),
                           ok
                   end},

         #{desc => "await continue (send 3)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send)
                   end},
         #{desc => "send (3) data to server",
           cmd  => fun(#{sock := Sock, data := Data}) ->
                           ?SEV_IPRINT("try send ~w bytes", [size(Data)]),
                           socket:send(Sock, Data)
                   end},
         #{desc => "announce ready (send 3)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send),
                           ok
                   end},


         %% Termination
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock}) ->
                           socket:close(Sock)
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Server} = _State) ->
                           _MRef = erlang:monitor(process, Server),
                           ok
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Client} = _State) ->
                           _MRef = erlang:monitor(process, Client),
                           ok
                   end},
         #{desc => "order server start",
           cmd  => fun(#{server := Server}) ->
                           ?SEV_ANNOUNCE_START(Server)
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Server} = State) ->
                           {ok, ServerSA} = ?SEV_AWAIT_READY(Server, server, init),
                           {ok, State#{server_sa => ServerSA}}
                   end},
         #{desc => "order client start",
           cmd  => fun(#{client    := Client,
                         server_sa := ServerSA}) ->
                           ?SEV_ANNOUNCE_START(Client, ServerSA),
                           ok
                   end},
         #{desc => "await client ready (init)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, init)
                   end},


         %% The actual test (connecting)
         #{desc => "order server accept (accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
         ?SEV_SLEEP(?SECS(1)),
         #{desc => "order client continue (connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, connect),
                           ok
                   end},
         #{desc => "await client ready (connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, connect)
                   end},
         #{desc => "await server ready (accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, accept)
                   end},

         %% The actual test (initial part)
         #{desc => "order client continue (send initial)",
           cmd  => fun(#{client := Client, data := Data} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send, Data),
                           ok
                   end},
         ?SEV_SLEEP(?SECS(1)),
         #{desc => "order server continue (recv initial)",
           cmd  => fun(#{server := Server, data := Data} = _State) ->
                           ExpMsgSz = size(Data),
                           ?SEV_ANNOUNCE_CONTINUE(Server, recv, ExpMsgSz),
                           ok
                   end},
         #{desc => "await client ready (send initial)",
           cmd  => fun(#{server := Server,
                         client := Client} = _State) ->
                           case ?SEV_AWAIT_READY(Client, client, send,
                                                 [{server, Server}]) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "await server ready (recv initial)",
           cmd  => fun(#{server := Server,
                         client := Client} = _State) ->
                           case ?SEV_AWAIT_READY(Server, client, recv,
                                                 [{client, Client}]) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% The actual test (part 1)
         #{desc => "order client continue (send 1)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send),
                           ok
                   end},
         ?SEV_SLEEP(?SECS(1)),
         #{desc => "order server continue (recv 1)",
           cmd  => fun(#{server := Server, data := Data} = _State) ->
                           MsgSz     = size(Data),
                           NewRcvBuf = {2 + (MsgSz div 1024), 1024},
                           ?SEV_ANNOUNCE_CONTINUE(Server, recv, NewRcvBuf),
                           ok
                   end},
         #{desc => "await client ready (send 1)",
           cmd  => fun(#{server := Server,
                         client := Client} = _State) ->
                           case ?SEV_AWAIT_READY(Client, client, send,
                                                 [{server, Server}]) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "await server ready (recv 1)",
           cmd  => fun(#{server := Server,
                         client := Client} = _State) ->
                           case ?SEV_AWAIT_READY(Server, client, recv,
                                                 [{client, Client}]) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% The actual test (part 2)
         #{desc => "order client continue (send 2)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send),
                           ok
                   end},
         ?SEV_SLEEP(?SECS(1)),
         #{desc => "order server continue (recv 2)",
           cmd  => fun(#{server := Server, data := Data} = _State) ->
                           MsgSz     = size(Data),
                           NewRcvBuf = {2 + (MsgSz div 2048), 2048},
                           ?SEV_ANNOUNCE_CONTINUE(Server, recv, NewRcvBuf),
                           ok
                   end},
         #{desc => "await client ready (send 2)",
           cmd  => fun(#{server := Server,
                         client := Client} = _State) ->
                           case ?SEV_AWAIT_READY(Client, client, send,
                                                 [{server, Server}]) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "await server ready (recv 2)",
           cmd  => fun(#{server := Server,
                         client := Client} = _State) ->
                           case ?SEV_AWAIT_READY(Server, client, recv,
                                                 [{client, Client}]) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% The actual test (part 3)
         #{desc => "order client continue (send 3)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send),
                           ok
                   end},
         ?SEV_SLEEP(?SECS(1)),
         #{desc => "order server continue (recv 3)",
           cmd  => fun(#{server := Server, data := Data} = _State) ->
                           MsgSz     = size(Data),
                           BufSz     = 2048,
                           N         = MsgSz div BufSz - 1,
                           NewRcvBuf = {N, BufSz},
                           ?SEV_ANNOUNCE_CONTINUE(Server, recv,
                                                  {N*BufSz, NewRcvBuf})
                   end},
         #{desc => "await client ready (send 3)",
           cmd  => fun(#{server := Server,
                         client := Client} = _State) ->
                           case ?SEV_AWAIT_READY(Client, client, send,
                                                 [{server, Server}]) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "await server ready (recv 3)",
           cmd  => fun(#{server := Server,
                         client := Client} = _State) ->
                           case ?SEV_AWAIT_READY(Server, client, recv,
                                                 [{client, Client}]) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         ?SEV_SLEEP(?SECS(1)),

         %% *** Terminate server ***
         #{desc => "order client terminate",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client down",
           cmd  => fun(#{client := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client,    State),
                           {ok, State1}
                   end},
         #{desc => "order server terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server down",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server,    State),
                           State2 = maps:remove(server_sa, State1),
                           {ok, State2}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    %% Create a data binary of 6*1024 bytes
    Data      = list_to_binary(lists:duplicate(6*4, lists:seq(0, 255))),
    InitState = #{domain => inet,
                  data   => Data},

    i("create server evaluator"),
    ServerInitState = #{domain => maps:get(domain, InitState)},
    Server          = ?SEV_START("server", ServerSeq, ServerInitState),

    i("create client evaluator"),
    ClientInitState = #{host   => local_host(),
                        domain => maps:get(domain, InitState)},
    Client          = ?SEV_START("client", ClientSeq, ClientInitState),

    i("create tester evaluator"),
    TesterInitState = InitState#{server => Server#ev.pid,
                                 client => Client#ev.pid},
    Tester          = ?SEV_START("tester", TesterSeq, TesterInitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Perform some simple getopt and setopt with the level = otp options
api_opt_simple_otp_controlling_process(suite) ->
    [];
api_opt_simple_otp_controlling_process(doc) ->
    [];
api_opt_simple_otp_controlling_process(_Config) when is_list(_Config) ->
    ?TT(?SECS(30)),
    tc_try(api_opt_simple_otp_controlling_process,
           fun() -> api_opt_simple_otp_controlling_process() end).

api_opt_simple_otp_controlling_process() ->
    Get = fun(S, Key) ->
                  socket:getopt(S, otp, Key)
          end,
    Set = fun(S, Key, Val) ->
                  socket:setopt(S, otp, Key, Val)
          end,

    ClientSeq =
        [
         %% *** Init part ***
         #{desc => "await start",
           cmd  => fun(State) ->
                           {Tester, Sock} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester,
                                       sock   => Sock}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "verify tester as controlling-process",
           cmd  => fun(#{tester := Tester, sock := Sock} = _State) ->
                           case Get(Sock, controlling_process) of
                               {ok, Tester} ->
                                   ok;
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "attempt invalid controlling-process transfer (to self)",
           cmd  => fun(#{sock := Sock} = _State) ->
                           case Set(Sock, controlling_process, self()) of
                               {error, {invalid, not_owner}} ->
                                   ok;
                               ok ->
                                   {error, unexpected_success};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (not owner)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_ANNOUNCE_READY(Tester, not_owner),
                           ok
                   end},
         #{desc => "await continue (owner)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, owner)
                   end},
         #{desc => "verify self as controlling-process",
           cmd  => fun(#{sock := Sock} = _State) ->
                           Self = self(),
                           case Get(Sock, controlling_process) of
                               {ok, Self} ->
                                   ok;
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "attempt controlling-process transfer to tester",
           cmd  => fun(#{tester := Tester, sock := Sock} = _State) ->
                           Set(Sock, controlling_process, Tester)
                   end},
         #{desc => "attempt invalid controlling-process transfer (to self)",
           cmd  => fun(#{sock := Sock} = _State) ->
                           case Set(Sock, controlling_process, self()) of
                               {error, {invalid, not_owner}} ->
                                   ok;
                               ok ->
                                   {error, unexpected_success};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (owner)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_ANNOUNCE_READY(Tester, owner),
                           ok

                   end},
         
         %% *** Termination ***
         #{desc => "await termination",
           cmd  => fun(#{tester := Tester} = State) ->
                           ?SEV_AWAIT_TERMINATE(Tester, tester),
                           State1 = maps:remove(tester, State),
                           State2 = maps:remove(sock, State1),
                           {ok, State2}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "create socket",
           cmd  => fun(#{domain   := Domain, 
                         type     := Type,
                         protocol := Protocol} = State) ->
                           Sock = sock_open(Domain, Type, Protocol),
                           {ok, State#{sock => Sock}}
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Client} = _State) ->
                           _MRef = erlang:monitor(process, Client),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "verify self as controlling-process",
           cmd  => fun(#{sock := Sock} = _State) ->
                           Self = self(),
                           case Get(Sock, controlling_process) of
                               {ok, Self} ->
                                   ok;
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "order (client) start",
           cmd  => fun(#{client := Client, sock := Sock} = _State) ->
                           ?SEV_ANNOUNCE_START(Client, Sock),
                           ok
                   end},
         #{desc => "await (client) ready (not owner)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, not_owner)
                   end},
         #{desc => "attempt controlling-process transfer to client",
           cmd  => fun(#{client := Client, sock := Sock} = _State) ->
                           Set(Sock, controlling_process, Client)
                   end},
         #{desc => "verify client as controlling-process",
           cmd  => fun(#{client := Client, sock := Sock} = _State) ->
                           case Get(Sock, controlling_process) of
                               {ok, Client} ->
                                   ok;
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "attempt invalid controlling-process transfer (to self)",
           cmd  => fun(#{sock := Sock} = _State) ->
                           case Set(Sock, controlling_process, self()) of
                               {error, {invalid, not_owner}} ->
                                   ok;
                               ok ->
                                   {error, unexpected_success};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "order (client) continue (owner)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, owner),
                           ok
                   end},
         #{desc => "await (client) ready (2)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, owner),
                           ok
                   end},
         #{desc => "verify self as controlling-process",
           cmd  => fun(#{sock := Sock} = _State) ->
                           Self = self(),
                           case Get(Sock, controlling_process) of
                               {ok, Self} ->
                                   ok;
                               {ok, _} = OK ->
                                   {error, OK};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** Termination ***
         #{desc => "order (client) terminate",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client termination",
           cmd  => fun(#{client := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           {ok, maps:remove(client, State)}
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           sock_close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start tcp (stream) client evaluator"),
    ClientInitState1 = #{},
    Client1 = ?SEV_START("tcp-client", ClientSeq, ClientInitState1),

    i("start tcp (stream) tester evaluator"),
    TesterInitState1 = #{domain   => inet, 
                         type     => stream, 
                         protocol => tcp,
                         client   => Client1#ev.pid},
    Tester1 = ?SEV_START("tcp-tester", TesterSeq, TesterInitState1),

    i("await tcp evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Tester1, Client1]),

    i("start udp (dgram) client evaluator"),
    ClientInitState2 = #{},
    Client2 = ?SEV_START("udp-client", ClientSeq, ClientInitState2),

    i("start udp (dgram) tester evaluator"),
    TesterInitState2 = #{domain   => inet, 
                         type     => dgram, 
                         protocol => udp,
                         client   => Client2#ev.pid},
    Tester2 = ?SEV_START("udp-tester", TesterSeq, TesterInitState2),

    i("await udp evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Tester2, Client2]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option acceptconn for UDP.
%% This should be possible to get but not set.

api_opt_sock_acceptconn_udp(suite) ->
    [];
api_opt_sock_acceptconn_udp(doc) ->
    [];
api_opt_sock_acceptconn_udp(_Config) when is_list(_Config) ->
    ?TT(?SECS(30)),
    tc_try(api_opt_sock_acceptconn_udp,
           fun() ->
                   has_support_sock_acceptconn()
           end,
           fun() -> api_opt_sock_acceptconn_udp() end).



api_opt_sock_acceptconn_udp() ->
    Opt = acceptconn,
    Set = fun(S, Val) ->
                  socket:setopt(S, socket, Opt, Val)
          end,
    Get = fun(S) ->
                  socket:getopt(S, socket, Opt)
          end,

    TesterSeq =
        [
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{local_sa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "[get] verify socket (before bind)",
           cmd  => fun(#{sock := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, false} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "Not accepting connections"),
                                   ok;
                               {ok, true} ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Accepting connections"),
                                   {error, {unexpected_success, {Opt, true}}};
                               {error, enoprotoopt = Reason} ->
                                   %% On some platforms this is not accepted
                                   %% for UDP, so skip this part (UDP).
                                   ?SEV_EPRINT("Expected Failure: "
                                               "~p => SKIP", [Reason]),
                                   (catch socket:close(Sock)),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[set] verify socket (before bind)",
           cmd  => fun(#{sock := Sock} = _State) ->
                           case Set(Sock, true) of
                               {error, Reason} ->
                                   ?SEV_IPRINT("Expected Failure: ~p",
                                               [Reason]),
                                   ok;
                               ok ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Set acceptconn (=true)"),
                                   {error, unexpected_success}
                           end
                   end},

         #{desc => "bind socket to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "[get] verify socket (after bind)",
           cmd  => fun(#{sock := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, false} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "Not accepting connections"),
                                   ok;
                               {ok, true} ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Accepting connections"),
                                   {error, {unexpected_success, {Opt, true}}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[set] verify socket (after bind)",
           cmd  => fun(#{sock := Sock} = _State) ->
                           case Set(Sock, true) of
                               {error, Reason} ->
                                   ?SEV_IPRINT("Expected Failure: ~p",
                                               [Reason]),
                                   ok;
                               ok ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Set acceptconn (=true)"),
                                   {error, unexpected_success}
                           end
                   end},

         %% *** Termination ***
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    Domain = inet,

    i("start tester evaluator"),
    InitState = #{domain  => Domain},
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option acceptconn for TCP.
%% This should be possible to get but not set.

api_opt_sock_acceptconn_tcp(suite) ->
    [];
api_opt_sock_acceptconn_tcp(doc) ->
    [];
api_opt_sock_acceptconn_tcp(_Config) when is_list(_Config) ->
    ?TT(?SECS(30)),
    tc_try(api_opt_sock_acceptconn_tcp,
           fun() ->
                   has_support_sock_acceptconn()
           end,
           fun() -> api_opt_sock_acceptconn_tcp() end).



api_opt_sock_acceptconn_tcp() ->
    Opt = acceptconn,
    Set = fun(S, Val) ->
                  socket:setopt(S, socket, Opt, Val)
          end,
    Get = fun(S) ->
                  socket:getopt(S, socket, Opt)
          end,

    TesterSeq =
        [
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{local_sa => LSA}}
                   end},

         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "[get] verify listen socket (before bind)",
           cmd  => fun(#{lsock := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, false} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "Not accepting connections"),
                                   ok;
                               {ok, true} ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Accepting connections"),
                                   {error, {unexpected_success, {Opt, true}}};
                               {error, enoprotoopt = Reason} ->
                                   ?SEV_EPRINT("Expected Failure: "
                                               "~p => SKIP", [Reason]),
                                   (catch socket:close(Sock)),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[set] verify listen socket (before bind)",
           cmd  => fun(#{lsock := Sock} = _State) ->
                           case Set(Sock, true) of
                               {error, Reason} ->
                                   ?SEV_IPRINT("Expected Failure: ~p", [Reason]),
                                   ok;
                               ok ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Set acceptconn (=true)"),
                                   {error, unexpected_success}
                           end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "bind listen socket to local address",
           cmd  => fun(#{lsock := Sock, local_sa := LSA} = State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   {ok, State#{server_sa => LSA#{port => Port}}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "[get] verify listen socket (after bind)",
           cmd  => fun(#{lsock := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, false} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "Not accepting connections"),
                                   ok;
                               {ok, true} ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Accepting connections"),
                                   {error, {unexpected_success, {Opt, true}}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[set] verify listen socket (after bind)",
           cmd  => fun(#{lsock := Sock} = _State) ->
                           case Set(Sock, true) of
                               {error, Reason} ->
                                   ?SEV_IPRINT("Expected Failure: ~p", [Reason]),
                                   ok;
                               ok ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Set acceptconn (=true)"),
                                   {error, unexpected_success}
                           end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "make listen socket accept connections",
           cmd  => fun(#{lsock := Sock} = _State) ->
                           case socket:listen(Sock) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "[get] verify listen socket (after listen)",
           cmd  => fun(#{lsock := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, true} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "Accepting connections"),
                                   ok;
                               {ok, false} ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Not accepting connections"),
                                   {error, {unexpected_success, {Opt, false}}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[set] verify listen socket (after listen)",
           cmd  => fun(#{lsock := Sock} = _State) ->
                           case Set(Sock, false) of
                               {error, Reason} ->
                                   ?SEV_IPRINT("Expected Failure: ~p", [Reason]),
                                   ok;
                               ok ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Set acceptconn (=false)"),
                                   {error, unexpected_success}
                           end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "create (connecting) socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{csockc => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "bind connecting socket to local address",
           cmd  => fun(#{csockc := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "[get] verify connecting socket (before connect)",
           cmd  => fun(#{csockc := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, false} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "Not accepting connections"),
                                   ok;
                               {ok, true} ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Accepting connections"),
                                   {error, {unexpected_success, {Opt, true}}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[set] verify connecting socket (before connect)",
           cmd  => fun(#{csockc := Sock} = _State) ->
                           case Set(Sock, true) of
                               {error, Reason} ->
                                   ?SEV_IPRINT("Expected Failure: ~p", [Reason]),
                                   ok;
                               ok ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Set acceptconn (=true)"),
                                   {error, unexpected_success}
                           end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "connect to server",
           cmd  => fun(#{csockc := Sock, server_sa := SSA} = _State) ->
                           case socket:connect(Sock, SSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "accept connection",
           cmd  => fun(#{lsock := Sock} = State) ->
                           case socket:accept(Sock) of
                               {ok, CSock} ->
                                   {ok, State#{csocks => CSock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "[get] verify connecting socket (after connect)",
           cmd  => fun(#{csockc := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, false} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "Not accepting connections"),
                                   ok;
                               {ok, true} ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Accepting connections"),
                                   {error, {unexpected_success, {Opt, true}}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[set] verify connecting socket (after connect)",
           cmd  => fun(#{csockc := Sock} = _State) ->
                           case Set(Sock, true) of
                               {error, Reason} ->
                                   ?SEV_IPRINT("Expected Failure: ~p", [Reason]),
                                   ok;
                               ok ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Set acceptconn (=true)"),
                                   {error, unexpected_success}
                           end
                   end},

         #{desc => "[get] verify connected socket",
           cmd  => fun(#{csocks := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, false} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "Not accepting connections"),
                                   ok;
                               {ok, true} ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Accepting connections"),
                                   {error, {unexpected_success, {Opt, true}}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[set] verify connected socket",
           cmd  => fun(#{csocks := Sock} = _State) ->
                           case Set(Sock, true) of
                               {error, Reason} ->
                                   ?SEV_IPRINT("Expected Failure: ~p", [Reason]),
                                   ok;
                               ok ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Set acceptconn (=true)"),
                                   {error, unexpected_success}
                           end
                   end},

         #{desc => "[get] verify listen socket (after connect)",
           cmd  => fun(#{lsock := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, true} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "Accepting connections"),
                                   ok;
                               {ok, false} ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Not accepting connections"),
                                   {error, {unexpected_success, {Opt, false}}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[set] verify listen socket (after connect)",
           cmd  => fun(#{lsock := Sock} = _State) ->
                           case Set(Sock, false) of
                               {error, Reason} ->
                                   ?SEV_IPRINT("Expected Failure: ~p", [Reason]),
                                   ok;
                               ok ->
                                   ?SEV_EPRINT("Unexpected Success: "
                                               "Set acceptconn (=false)"),
                                   {error, unexpected_success}
                           end
                   end},

         %% *** Termination ***
         #{desc => "close connecting socket(s)",
           cmd  => fun(#{csockc := Sock} = State0) ->
                           socket:close(Sock),
                           State1 = maps:remove(csockc, State0),
                           State2 = maps:remove(csocks, State1), %% Auto-close
                           {ok, maps:remove(csockc, State2)}
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{lsock := Sock} = State) ->
                           socket:close(Sock),
                           {ok, maps:remove(lsock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    Domain = inet,

    i("start tester evaluator"),
    InitState = #{domain  => Domain},
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option acceptfilter. PLACEHOLDER!

api_opt_sock_acceptfilter(suite) ->
    [];
api_opt_sock_acceptfilter(doc) ->
    [];
api_opt_sock_acceptfilter(_Config) when is_list(_Config) ->
    ?TT(?SECS(30)),
    tc_try(api_opt_sock_acceptfilter,
           fun() -> not_yet_implemented() end,
           fun() -> ok end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option bindtodevice.
%% It has not always been possible to 'get' this option
%% (atleast on linux).

api_opt_sock_bindtodevice(suite) ->
    [];
api_opt_sock_bindtodevice(doc) ->
    [];
api_opt_sock_bindtodevice(_Config) when is_list(_Config) ->
    ?TT(?SECS(30)),
    tc_try(api_opt_sock_bindtodevice,
           fun() -> has_support_sock_bindtodevice() end,
           fun() -> api_opt_sock_bindtodevice() end).


api_opt_sock_bindtodevice() ->
    Opt = bindtodevice,
    Set = fun(S, Val) ->
                  socket:setopt(S, socket, Opt, Val)
          end,
    Get = fun(S) ->
                  socket:getopt(S, socket, Opt)
          end,

    TesterSeq =
        [
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           case ?LIB:which_local_host_info(Domain) of
                               {ok, #{name := Name, addr := Addr}} ->
                                   ?SEV_IPRINT("local host info (~p): "
                                               "~n   Name: ~p"
                                               "~n   Addr: ~p",
                                               [Domain, Name, Addr]),
                                   LSA = #{family => Domain,
                                           addr   => Addr},
                                   {ok, State#{dev      => Name,
                                               local_sa => LSA}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "create UDP socket 1",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{usock1 => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "create UDP socket 2",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{usock2 => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "create TCP socket 1",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{tsock1 => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "create TCP socket 2",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{tsock2 => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "[get] verify UDP socket 1 (before bindtodevice)",
           cmd  => fun(#{usock1 := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, Dev} ->
                                   ?SEV_IPRINT("Expected Success: ~p", [Dev]),
                                   ok;
                               {error, enoprotoopt = Reason} ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p => SKIP",
					       [Reason]),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[get] verify UDP socket 2 (before bind)",
           cmd  => fun(#{usock2 := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, Dev} ->
                                   ?SEV_IPRINT("Expected Success: ~p", [Dev]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[get] verify TCP socket 1 (before bindtodevice)",
           cmd  => fun(#{tsock1 := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, Dev} ->
                                   ?SEV_IPRINT("Expected Success: ~p", [Dev]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[get] verify TCP socket 2 (before bind)",
           cmd  => fun(#{tsock2 := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, Dev} ->
                                   ?SEV_IPRINT("Expected Success: ~p", [Dev]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "Bind UDP socket 1 to device",
           cmd  => fun(#{usock1 := Sock, dev := Dev} = State) ->
                           case Set(Sock, Dev) of
                               ok ->
                                   ?SEV_IPRINT("Expected Success"),
                                   ok;
                               {error, eperm = Reason} ->
                                   ?SEV_IPRINT("Expected Failure: ~p", [Reason]),
                                   (catch socket:close(Sock)),
                                   {ok, State#{usock1 => skip}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "Bind UDP socket 2 to local address",
           cmd  => fun(#{usock2 := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("Expected Success"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "Bind TCP socket 1 to device",
           cmd  => fun(#{usock1 := USock1,
                         tsock1 := Sock, dev := Dev} = State) ->
                           case Set(Sock, Dev) of
                               ok ->
                                   ?SEV_IPRINT("Expected Success"),
                                   ok;
                               {error, eperm = Reason} when (USock1 =:= skip) ->
                                   ?SEV_IPRINT("Expected Failure: ~p", [Reason]),
                                   {skip, Reason};
                               {error, eperm = Reason} ->
                                   ?SEV_IPRINT("Expected Failure: ~p", [Reason]),
                                   (catch socket:close(Sock)),
                                   {ok, State#{tsock1 => skip}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "Bind TCP socket 2 to local address",
           cmd  => fun(#{tsock2 := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("Expected Success"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "[get] verify UDP socket 1 (after bindtodevice)",
           cmd  => fun(#{usock1 := skip} = _State) ->
                           ?SEV_IPRINT("SKIP'ed (previous eperm)"),
                           ok;
                      (#{usock1 := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, Dev} ->
                                   ?SEV_IPRINT("Expected Success: ~p", [Dev]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[get] verify UDP socket 2 (after bind)",
           cmd  => fun(#{usock2 := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, Dev} ->
                                   ?SEV_IPRINT("Expected Success: ~p", [Dev]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[get] verify TCP socket 1 (after bindtodevice)",
           cmd  => fun(#{tsock1 := skip} = _State) ->
                           ?SEV_IPRINT("SKIP'ed (previous eperm)"),
                           ok;
                      (#{tsock1 := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, Dev} ->
                                   ?SEV_IPRINT("Expected Success: ~p", [Dev]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[get] verify TCP socket 2 (after bind)",
           cmd  => fun(#{tsock2 := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, Dev} ->
                                   ?SEV_IPRINT("Expected Success: ~p", [Dev]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p", [Reason]),
                                   ERROR
                           end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         %% *** Termination ***
         #{desc => "close UDP socket 1",
           cmd  => fun(#{usock1 := skip} = State) ->
                           ?SEV_IPRINT("SKIP'ed (already closed)"),
                           {ok, maps:remove(usock1, State)};
                      (#{usock1 := Sock} = State) ->
                           socket:close(Sock),
                           {ok, maps:remove(usock1, State)}
                   end},
         #{desc => "close UDP socket 2",
           cmd  => fun(#{usock2 := Sock} = State) ->
                           socket:close(Sock),
                           {ok, maps:remove(usock2, State)}
                   end},
         #{desc => "close TCP socket 1",
           cmd  => fun(#{tsock1 := skip} = State) ->
                           ?SEV_IPRINT("SKIP'ed (already closed)"),
                           {ok, maps:remove(tsock1, State)};
                      (#{tsock1 := Sock} = State) ->
                           socket:close(Sock),
                           {ok, maps:remove(tsock1, State)}
                   end},
         #{desc => "close TCP socket 2",
           cmd  => fun(#{tsock2 := Sock} = State) ->
                           socket:close(Sock),
                           {ok, maps:remove(tsock2, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    Domain = inet,

    i("start tester evaluator"),
    InitState = #{domain  => Domain},
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option broadcast.
%% Make it possible for datagram sockets to send packets to a broadcast
%% address (IPv4 only).

api_opt_sock_broadcast(suite) ->
    [];
api_opt_sock_broadcast(doc) ->
    [];
api_opt_sock_broadcast(_Config) when is_list(_Config) ->
    ?TT(?SECS(30)),
    tc_try(api_opt_sock_broadcast,
           fun() -> has_support_sock_broadcast() end,
           fun() -> api_opt_sock_broadcast() end).


api_opt_sock_broadcast() ->
    Opt    = broadcast,
    Set    = fun(S, Val) when is_boolean(Val) ->
                     socket:setopt(S, socket, Opt, Val)
             end,
    Get    = fun(S) ->
                     socket:getopt(S, socket, Opt)
             end,

    TesterSeq =
        [
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           case ?LIB:which_local_host_info(Domain) of
                               {ok, #{name      := Name,
                                      addr      := Addr,
                                      broadaddr := BAddr}} ->
                                   ?SEV_IPRINT("local host info: "
                                               "~n   Name:           ~p"
                                               "~n   Addr:           ~p"
                                               "~n   Broadcast Addr: ~p",
                                               [Name, Addr, BAddr]),
                                   LSA = #{family => Domain,
                                           addr   => Addr},
                                   BSA = #{family => Domain,
                                           addr   => BAddr},
                                   {ok, State#{lsa => LSA,
                                               bsa => BSA}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "[socket 1] create UDP socket (listening 1)",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{sock1 => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "[socket 1] Bind UDP socket (to limited broadcast address)",
           cmd  => fun(#{sock1 := Sock} = State) ->
			   BSA = #{family => inet,
				   addr   => broadcast},
                           ?SEV_IPRINT("Try bind (socket 1) to: "
                                       "~n   ~p", [BSA]),
                           case socket:bind(Sock, BSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   ?SEV_IPRINT("Expected Success (bound): ~p",
                                               [Port]),
                                   {ok, State#{sa1 => BSA#{port => Port}}};
                               {error, eaddrnotavail = Reason} ->
                                   ?SEV_IPRINT("~p => "
					       "SKIP limited broadcast test",
					       [Reason]),
                                   {ok, State#{sa1 => skip}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[socket 1] UDP socket sockname",
           cmd  => fun(#{sa1 := skip} = _State) ->
                           ?SEV_IPRINT("SKIP limited broadcast test"),
                           ok;
		      (#{sock1 := Sock} = _State) ->
			   case socket:sockname(Sock) of
			       {ok, SA} ->
				   ?SEV_IPRINT("SA: ~p", [SA]),
				   ok;
			       {error, _} = ERROR ->
				   ERROR
			   end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "[socket 2] create UDP socket (listening 2)",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{sock2 => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "[socket 2] Bind UDP socket (to subnet-directed broadcast address)",
           cmd  => fun(#{sock2 := Sock,
			 bsa   := BSA} = State) ->
                           ?SEV_IPRINT("Try bind (socket 1) to: "
                                       "~n   ~p", [BSA]),
                           case socket:bind(Sock, BSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   ?SEV_IPRINT("Expected Success (bound): ~p",
                                               [Port]),
                                   {ok, State#{sa2 => BSA#{port => Port}}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[socket 2] UDP socket sockname",
           cmd  => fun(#{sock2 := Sock} = _State) ->
                           case socket:sockname(Sock) of
                               {ok, SA} ->
				   ?SEV_IPRINT("SA: ~p", [SA]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         ?SEV_SLEEP(?SECS(1)),

         #{desc => "[socket 3] create UDP socket (sender)",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{sock3 => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "[socket 3][get] verify UDP socket (before bind and set)",
           cmd  => fun(#{sock3 := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, false} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "broadcast not allowed"),
                                   ok;
                               {ok, true} ->
                                   ?SEV_IPRINT("Unexpected Success result: "
                                               "broadcast already allowed"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[socket 3] Try make broadcast allowed",
           cmd  => fun(#{sock3 := Sock} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "broadcast now allowed"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[socket 3] verify UDP socket broadcast allowed",
           cmd  => fun(#{sock3 := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, true} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "broadcast allowed"),
                                   ok;
                               {ok, false} ->
                                   ?SEV_IPRINT("Unexpected Success result: "
                                               "broadcast *not* allowed"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[socket 3] Bind UDP socket (to local address)",
           cmd  => fun(#{sock3 := Sock, lsa := LSA} = State) ->
                           ?SEV_IPRINT("Try bind (socket 2) to: "
                                       "~n   ~p", [LSA]),
                           case socket:bind(Sock, LSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   ?SEV_IPRINT("Expected Success (bound): ~p",
                                               [Port]),
                                   {ok, State#{sa3 => LSA#{port => Port}}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[socket 3] verify UDP socket (after set)",
           cmd  => fun(#{sock3 := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, true} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "broadcast allowed"),
                                   ok;
                               {ok, false} ->
                                   ?SEV_IPRINT("Unexpected Success result: "
                                               "broadcast not allowed"),
                                   {error, not_allowed};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},

	 ?SEV_SLEEP(?SECS(1)),

         #{desc => "[socket 3] try send to limited broadcast address",
           cmd  => fun(#{sa1 := skip} = _State) ->
                           ?SEV_IPRINT("SKIP limited broadcast test"),
			   ok;
		      (#{sock3 := Sock,
			 sa1   := Dest} = _State) ->
			   Data = list_to_binary("hejsan"),
			   ?SEV_IPRINT("try send to bradcast address: "
				       "~n   ~p", [Dest]),
			   case socket:sendto(Sock, Data, Dest) of
			       ok ->
				   ?SEV_IPRINT("Expected Success: "
					       "broadcast message sent"),
				   ok;
			       {error, Reason} = ERROR ->
				   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
				   ERROR
			   end
		   end},
         #{desc => "[socket 1] try recv",
           cmd  => fun(#{sa1 := skip} = _State) ->
			   ?SEV_IPRINT("SKIP limited broadcast test"),
			   ok;
		      (#{sock1 := Sock} = State) ->
                           case socket:recvfrom(Sock, 0, 5000) of
                               {ok, _} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "received message"),
                                   ok;
                               {error, timeout = Reason} ->
                                   %% Some platforms seem to balk at this.
                                   %% It spossible to bind to this, and
                                   %% send to it, but no data is received.
                                   %% At some point we should investigate...
                                   %% For now, we just skip this part of
                                   %% the test...
                                   ?SEV_IPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   {ok, State#{sa1 => skip}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},

	 ?SEV_SLEEP(?SECS(1)),

         #{desc => "[socket 3] try send to subnet-directed broadcast address",
           cmd  => fun(#{sock3 := Sock,
                         sa2   := Dest} = _State) ->
                           Data = list_to_binary("hejsan"),
                           ?SEV_IPRINT("try send to bradcast address: "
                                       "~n   ~p", [Dest]),
                           case socket:sendto(Sock, Data, Dest) of
                               ok ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "broadcast message sent"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "[socket 2] try recv",
           cmd  => fun(#{sock2 := Sock, sa1 := SA1} = _State) ->
                           case socket:recvfrom(Sock, 0, 5000) of
                               {ok, _} ->
                                   ?SEV_IPRINT("Expected Success: "
                                               "received message"),
                                   ok;
                               {error, timeout = Reason} when (SA1 =:= skip) ->
                                   ?SEV_IPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   {skip, "receive timeout"};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},

         %% *** Termination ***
         #{desc => "[socket 3] close UDP socket (sender)",
           cmd  => fun(#{sock3 := Sock} = State0) ->
                           socket:close(Sock),
			   State1 = maps:remove(sock3, State0),
			   State2 = maps:remove(sa3,   State1),
			   {ok, State2}
                   end},
         #{desc => "[socket 2] close UDP socket (listener 2)",
           cmd  => fun(#{sock2 := Sock} = State0) ->
                           socket:close(Sock),
			   State1 = maps:remove(sock2, State0),
			   State2 = maps:remove(sa2,   State1),
                           {ok, State2}
                   end},
         #{desc => "[socket 1] close UDP socket (listener 1)",
           cmd  => fun(#{sock1 := Sock} = State0) ->
                           socket:close(Sock),
			   State1 = maps:remove(sock1, State0),
			   State2 = maps:remove(sa1,   State1),
                           {ok, State2}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    Domain = inet,

    i("start tester evaluator"),
    InitState = #{domain => Domain},
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option debug.
%% On linux, this test requires that the user running the test to have
%% CAP_NET_ADMIN capabilities or be root (effective user ID of 0), 
%% therefor we explicitly test for the result eacces when attempting to
%% set, and skip if we get it.

api_opt_sock_debug(suite) ->
    [];
api_opt_sock_debug(doc) ->
    [];
api_opt_sock_debug(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(api_opt_sock_debug,
           fun() -> has_support_sock_debug() end,
           fun() -> api_opt_sock_debug() end).


api_opt_sock_debug() ->
    Opt    = debug,
    Set    = fun(S, Val) when is_integer(Val) ->
                     socket:setopt(S, socket, Opt, Val)
             end,
    Get    = fun(S) ->
                     socket:getopt(S, socket, Opt)
             end,

    TesterSeq =
        [
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           case ?LIB:which_local_host_info(Domain) of
                               {ok, #{name      := Name,
                                      addr      := Addr,
                                      broadaddr := BAddr}} ->
                                   ?SEV_IPRINT("local host info: "
                                               "~n   Name:           ~p"
                                               "~n   Addr:           ~p"
                                               "~n   Broadcast Addr: ~p",
                                               [Name, Addr, BAddr]),
                                   LSA = #{family => Domain,
                                           addr   => Addr},
                                   BSA = #{family => Domain,
                                           addr   => BAddr},
                                   {ok, State#{lsa => LSA,
                                               bsa => BSA}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "create UDP socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "Get current debug value",
           cmd  => fun(#{sock := Sock} = State) ->
                           case Get(Sock) of
                               {ok, Debug} when is_integer(Debug) ->
                                   ?SEV_IPRINT("Success: ~p", [Debug]),
                                   {ok, State#{debug => Debug}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected failure: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "Try enable socket debug",
           cmd  => fun(#{sock := Sock, debug := Debug} = State) ->
			   NewDebug = Debug + 1,
                           case Set(Sock, NewDebug) of
                               ok ->
                                   ?SEV_IPRINT("Expected Success"),
                                   {ok, State#{debug => NewDebug}};
                               {error, eacces = Reason} ->
                                   ?SEV_EPRINT("NO ACCESS => SKIP"),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "Get current (new) debug value",
           cmd  => fun(#{sock := Sock, debug := Debug} = _State) ->
                           case Get(Sock) of
                               {ok, Debug} when is_integer(Debug) ->
                                   ?SEV_IPRINT("Success: ~p", [Debug]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected failure: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},

         %% *** Termination ***
         #{desc => "close UDP socket",
           cmd  => fun(#{sock := Sock} = State0) ->
                           socket:close(Sock),
			   State1 = maps:remove(sock, State0),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    Domain = inet,

    i("start tester evaluator"),
    InitState = #{domain => Domain},
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option domain.
%% This is a read only option. Also not available on all platforms.

api_opt_sock_domain(suite) ->
    [];
api_opt_sock_domain(doc) ->
    [];
api_opt_sock_domain(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(api_opt_sock_domain,
           fun() -> has_support_sock_domain() end,
           fun() -> api_opt_sock_domain() end).


api_opt_sock_domain() ->
    Opt = domain,
    Get = fun(S) ->
                  socket:getopt(S, socket, Opt)
          end,

    TesterSeq =
        [
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           case ?LIB:which_local_host_info(Domain) of
                               {ok, #{name      := Name,
                                      addr      := Addr,
                                      broadaddr := BAddr}} ->
                                   ?SEV_IPRINT("local host info: "
                                               "~n   Name:           ~p"
                                               "~n   Addr:           ~p"
                                               "~n   Broadcast Addr: ~p",
                                               [Name, Addr, BAddr]),
                                   LSA = #{family => Domain,
                                           addr   => Addr},
                                   BSA = #{family => Domain,
                                           addr   => BAddr},
                                   {ok, State#{lsa => LSA,
                                               bsa => BSA}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "create IPv4 UDP socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{usock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "Get domain for the UDP socket",
           cmd  => fun(#{domain := Domain, usock := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, Domain} ->
                                   ?SEV_IPRINT("Success: ~p", [Domain]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected failure: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "create TCP socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{tsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "Get domain for the TCP socket",
           cmd  => fun(#{domain := Domain, tsock := Sock} = _State) ->
                           case Get(Sock) of
                               {ok, Domain} ->
                                   ?SEV_IPRINT("Success: ~p", [Domain]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected failure: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},

         %% *** Termination ***
         #{desc => "close UDP socket",
           cmd  => fun(#{usock := Sock} = State0) ->
                           socket:close(Sock),
			   State1 = maps:remove(usock, State0),
                           {ok, State1}
                   end},
         #{desc => "close TCP socket",
           cmd  => fun(#{tsock := Sock} = State0) ->
                           socket:close(Sock),
			   State1 = maps:remove(tsock, State0),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    Domain = inet,

    i("start tester evaluator"),
    InitState = #{domain => Domain},
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option dontroute.
%% The man page has the following to say:
%% "Don't send via a gateway, send only to directly connected hosts.
%%  The same effect can be achieved by setting the MSG_DONTROUTE
%%  flag on a socket send(2) operation."
%% Since its "kind of" difficult to check if it actually takes an 
%% effect (you would need a gateway for that and a machine "on the
%% other side"), we only test if we can set and get the value.
%% Better then nothing.

api_opt_sock_dontroute(suite) ->
    [];
api_opt_sock_dontroute(doc) ->
    [];
api_opt_sock_dontroute(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(api_opt_sock_dontroute,
           fun() -> has_support_sock_dontroute() end,
           fun() -> api_opt_sock_dontroute() end).


api_opt_sock_dontroute() ->
    Opt    = dontroute,
    Set    = fun(S, Val) when is_boolean(Val) ->
                     socket:setopt(S, socket, Opt, Val)
             end,
    Get    = fun(S) ->
                     socket:getopt(S, socket, Opt)
             end,

    TesterSeq =
        [
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           case ?LIB:which_local_host_info(Domain) of
                               {ok, #{name      := Name,
                                      addr      := Addr,
                                      broadaddr := BAddr}} ->
                                   ?SEV_IPRINT("local host info: "
                                               "~n   Name:           ~p"
                                               "~n   Addr:           ~p"
                                               "~n   Broadcast Addr: ~p",
                                               [Name, Addr, BAddr]),
                                   LSA = #{family => Domain,
                                           addr   => Addr},
                                   BSA = #{family => Domain,
                                           addr   => BAddr},
                                   {ok, State#{lsa => LSA,
                                               bsa => BSA}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "create UDP socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "Get current value",
           cmd  => fun(#{sock := Sock} = State) ->
                           case Get(Sock) of
                               {ok, Val} when is_boolean(Val) ->
                                   ?SEV_IPRINT("Success: ~p", [Val]),
                                   {ok, State#{dontroute => Val}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected failure: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "Try change value",
           cmd  => fun(#{sock := Sock, dontroute := Current} = State) ->
			   New = not Current,
                           ?SEV_IPRINT("Change from ~p to ~p", [Current, New]),
                           case Set(Sock, New) of
                               ok ->
                                   ?SEV_IPRINT("Expected Success"),
                                   {ok, State#{dontroute => New}};
                               {error, eopnotsupp = Reason} ->
                                   ?SEV_EPRINT("Expected Failure: ~p",
					       [Reason]),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "Verify changed value",
           cmd  => fun(#{sock := Sock, dontroute := Val} = _State) ->
                           case Get(Sock) of
                               {ok, Val} ->
                                   ?SEV_IPRINT("Expected Success"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected failure: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},

         %% *** Termination ***
         #{desc => "close UDP socket",
           cmd  => fun(#{sock := Sock} = State0) ->
                           socket:close(Sock),
			   State1 = maps:remove(sock, State0),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    Domain = inet,

    i("start tester evaluator"),
    InitState = #{domain => Domain},
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option error. PLACEHOLDER!

api_opt_sock_error(suite) ->
    [];
api_opt_sock_error(doc) ->
    [];
api_opt_sock_error(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(api_opt_sock_error,
           fun() -> not_yet_implemented() end,
           fun() -> ok end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option keepalive.
%% This is bit tricky to test, partly because we have no control over
%% the underlying TCP timeouts. So, for now, we just test that we can
%% change the value.

api_opt_sock_keepalive(suite) ->
    [];
api_opt_sock_keepalive(doc) ->
    [];
api_opt_sock_keepalive(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(api_opt_sock_keepalive,
           fun() -> has_support_sock_keepalive() end,
           fun() -> api_opt_sock_keepalive() end).


api_opt_sock_keepalive() ->
    Opt    = keepalive,
    Set    = fun(S, Val) when is_boolean(Val) ->
                     socket:setopt(S, socket, Opt, Val)
             end,
    Get    = fun(S) ->
                     socket:getopt(S, socket, Opt)
             end,

    TesterSeq =
        [
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           case ?LIB:which_local_host_info(Domain) of
                               {ok, #{name      := Name,
                                      addr      := Addr,
                                      broadaddr := BAddr}} ->
                                   ?SEV_IPRINT("local host info: "
                                               "~n   Name:           ~p"
                                               "~n   Addr:           ~p"
                                               "~n   Broadcast Addr: ~p",
                                               [Name, Addr, BAddr]),
                                   LSA = #{family => Domain,
                                           addr   => Addr},
                                   BSA = #{family => Domain,
                                           addr   => BAddr},
                                   {ok, State#{lsa => LSA,
                                               bsa => BSA}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         #{desc => "create TCP socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, stream, tcp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "Get current value",
           cmd  => fun(#{sock := Sock} = State) ->
                           case Get(Sock) of
                               {ok, Val} when is_boolean(Val) ->
                                   ?SEV_IPRINT("Success: ~p", [Val]),
                                   {ok, State#{keepalive => Val}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected failure: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "Try change the value",
           cmd  => fun(#{sock := Sock, keepalive := Current} = State) ->
			   New = not Current,
                           ?SEV_IPRINT("Try change value from ~p to ~p", 
                                       [Current, New]),
                           case Set(Sock, New) of
                               ok ->
                                   ?SEV_IPRINT("Expected Success"),
                                   {ok, State#{keepalive => New}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected Failure: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "Verify (new) current value",
           cmd  => fun(#{sock := Sock, keepalive := Val} = _State) ->
                           case Get(Sock) of
                               {ok, Val} ->
                                   ?SEV_IPRINT("Expected Success (~p)", [Val]),
                                   ok;
                               {ok, OtherVal} ->
                                   ?SEV_IPRINT("Unexpected Success: ~p",
                                               [OtherVal]),
                                   {error, {unexpected_success_value,
                                            Val, OtherVal}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Unexpected failure: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},

         %% *** Termination ***
         #{desc => "close UDP socket",
           cmd  => fun(#{sock := Sock} = State0) ->
                           socket:close(Sock),
			   State1 = maps:remove(sock, State0),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    Domain = inet,

    i("start tester evaluator"),
    InitState = #{domain => Domain},
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option linger. PLACEHOLDER!

api_opt_sock_linger(suite) ->
    [];
api_opt_sock_linger(doc) ->
    [];
api_opt_sock_linger(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(api_opt_sock_linger,
           fun() -> not_yet_implemented() end,
           fun() -> ok end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the socket option mark. PLACEHOLDER!

api_opt_sock_mark(suite) ->
    [];
api_opt_sock_mark(doc) ->
    [];
api_opt_sock_mark(_Config) when is_list(_Config) ->
    ?TT(?SECS(10)),
    tc_try(api_opt_sock_mark,
           fun() -> not_yet_implemented() end,
           fun() -> ok end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% This test case tries to test that the oobinline socket 'socket' option
%% works.
%% So, this is done on the receiving side: 
%%
%%               socket:setopt(Sock, socket, oobinline, boolean()).
%%
%% This works on linux of some version (atleast linux kernel 4.15.0),
%% but not on FreeBSD (12) for some reason. Until we have figured out
%% exctly why, we skip a bunch of OSs...
%%
%% Do we need to make sure the two entities does not run in the same
%% process? This test case does not currently do that (which works in'
%% linux but maybe not in, say, FreeBSD).
%%

api_opt_sock_oobinline(suite) ->
    [];
api_opt_sock_oobinline(doc) ->
    [];
api_opt_sock_oobinline(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_ooinline,
           fun() ->
                   has_support_sock_oobinline(),
                   has_support_msg_flag(oob),
                   is_valid_oobinline_platform()
           end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, socket, oobinline, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, socket, oobinline)
                          end,
                   Send = fun(Sock, Data, true) ->
                                  socket:send(Sock, Data, [oob]);
                             (Sock, Data, false) ->
                                     socket:send(Sock, Data)
                          end,
                   Recv = fun(Sock, true) ->
                                  socket:recv(Sock, 0, [oob]);
                             (Sock, false) ->
                                  socket:recv(Sock)
                          end,
                   InitState = #{domain => inet,
                                 proto  => tcp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = do_api_opt_sock_oobinline(InitState)
           end).

%% Hopefully this is a temporary solution...
is_valid_oobinline_platform() ->
    case os:type() of
        {unix, linux} ->
            ok;

        Type ->
            %% Actually, all we know is that the
            %% test case only work for linux, but
            %% it *should* for FreeBSD and Solaris
            %% also...
            not_supported(Type)
    end.
    



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

do_api_opt_sock_oobinline(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{domain := local,
                         lsock  := LSock,
                         lsa    := LSA} = _State) ->
                           case socket:bind(LSock, LSA) of
                               ok ->
                                   ok; % We do not care about the port for local
                               {error, _} = ERROR ->
                                   ERROR
                           end;
                      (#{lsock := LSock, lsa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   ?SEV_IPRINT("bound to port: ~w", [Port]),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{domain := local,
                         tester := Tester, lsa := #{path := Path}}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Path),
                           ok;
                      (#{tester := Tester, lport := Port}) ->
                           %% This is actually not used for unix domain socket
                           ?SEV_ANNOUNCE_READY(Tester, init, Port),
                           ok
                   end},


         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:accept(LSock) of
                               {ok, Sock} ->
                                   ?SEV_IPRINT("accepted: ~n   ~p", [Sock]),
                                   {ok, State#{csock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept),
                           ok
                   end},

         %% *** no oobinline ***

         #{desc => "await continue (verify no oobinline)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, verify_oobinline)
                   end},
         #{desc => "verify no oobinline",
           cmd  => fun(#{csock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = _Value} ->
                                   ?SEV_IPRINT("oobinline: ~p", [_Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected oobinline: ~p",
                                               [Unexpected]),
                                   {error, {unexpected_oobinline, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting oobinline:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end                                   
                   end},
         #{desc => "announce ready (no oobinline)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, no_oobinline),
                           ok
                   end},

         #{desc => "await continue (recv no oobinline)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv)
                   end},
         #{desc => "await plain data",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock, false) of
                               {ok, <<"a">>} ->
                                   ?SEV_IPRINT("received plain data"),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (plain data)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_plain),
                           ok
                   end},
         #{desc => "await oob data",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock, true) of
                               {ok, <<"b">>} ->
                                   ?SEV_IPRINT("received oob data"),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (oob data)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_oob),
                           ok
                   end},

         %% *** oobinline ***

          #{desc => "await continue (enable oobinline)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, enable_oobinline)
                   end},
         #{desc => "enable oobinline",
           cmd  => fun(#{csock := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("oobinline enabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed enable oobinline:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end                                   
                   end},
         #{desc => "announce ready (oobinline)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, oobinline),
                           ok
                   end},

         #{desc => "await continue (recv)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, recv)
                   end},
       #{desc => "await (recv) data",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock, false) of
                               {ok, <<"ba">>} ->
                                   ?SEV_IPRINT("received expected message: "
                                               "both plain and oob data"),
                                   ok;
                               {ok, BadMsg} ->
                                   ?SEV_EPRINT("received unexpected message: ~p",
                                               [BadMsg]),
                                   {error, {unexpected_msg, BadMsg}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close connection socket",
           cmd  => fun(#{csock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(csock, State)}
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{domain   := local,
                         lsock    := Sock,
                         local_sa := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() ->
                                                   maps:remove(local_sa, State)
                                           end,
                                           fun() -> State end),
                           {ok, maps:remove(lsock, State1)};
                      (#{lsock := LSock} = State) ->
                           case socket:close(LSock) of
                               ok ->
                                   {ok, maps:remove(lsock, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    ClientSeq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(#{domain := local} = State) ->
                           {Tester, Path} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_path => Path}};
                      (State) ->
                           {Tester, Port} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_port => Port}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain      := local = Domain,
                         server_path := Path} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = #{family => Domain, path => Path},
                           {ok, State#{local_sa => LSA, server_sa => SSA}};
                      (#{domain := Domain, server_port := Port} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = LSA#{port => Port},
                           {ok, State#{local_sa => LSA, server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "await continue (connect)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, connect)
                   end},
         #{desc => "connect to server",
           cmd  => fun(#{sock := Sock, server_sa := SSA}) ->
                           socket:connect(Sock, SSA)
                   end},
         #{desc => "announce ready (connect)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect),
                           ok
                   end},

         %% *** First batch of data (no oobinline) ***

         #{desc => "await continue (send data)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send)
                   end},
         #{desc => "send plain data",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, <<"a">>, false)
                   end},
         %% #{desc => "enable (socket & global) debug",
         %%   cmd  => fun(#{sock := Sock}) ->
         %%                   ok = socket:setopt(Sock, otp, debug, true),
         %%                   ok = socket:debug(true)
         %%           end},
         #{desc => "send oob data",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, <<"b">>, true)
                   end},
         %% #{desc => "disable (socket) debug",
         %%   cmd  => fun(#{sock := Sock}) ->
         %%                   ok = socket:debug(true),
         %%                   ok = socket:setopt(Sock, otp, debug, false)
         %%           end},
         #{desc => "announce ready (send)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send),
                           ok
                   end},

         %% *** Second batch of data (oobinline) ***

         #{desc => "await continue (send data)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send)
                   end},
         #{desc => "send plain data",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, <<"a">>, false)
                   end},
         #{desc => "send oob data",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, <<"b">>, true)
                   end},
         #{desc => "announce ready (send)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{domain   := local,
                         sock     := Sock,
                         local_sa := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() ->
                                                   maps:remove(local_sa, State)
                                           end,
                                           fun() -> State end),
                           {ok, maps:remove(sock, State1)};
                      (#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% Start the client
         #{desc => "order client start",
           cmd  => fun(#{client := Pid, server_port := Port} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Port),
                           ok
                   end},
         #{desc => "await client ready (init)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, init)
                   end},

         %% *** The actual test ***
         #{desc => "order server to continue (with accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
         ?SEV_SLEEP(?SECS(1)),
         #{desc => "order client to continue (with connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, connect),
                           ok
                   end},
         #{desc => "await client ready (connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, connect)
                   end},
         #{desc => "await server ready (accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, accept)
                   end},

         %% *** First batch of data (no oobinline) ***

         #{desc => "order server to continue (with verify no oobinline)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, verify_oobinline),
                           ok
                   end},
         #{desc => "await server ready (no oobinline)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, no_oobinline)
                   end},

         #{desc => "order client to continue (with send)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send),
                           ok
                   end},
         #{desc => "await client ready (with send)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send)
                   end},
         #{desc => "order server to continue (with recv plain)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, recv),
                           ok
                   end},
         #{desc => "await server ready (recv plain)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_plain)
                   end},
         #{desc => "await server ready (recv oob)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_oob)
                   end},

         %% Second message (w timestamp)

         #{desc => "order server to continue (with enable oobinline)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, enable_oobinline),
                           ok
                   end},
         #{desc => "await server ready (oobinline)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, oobinline)
                   end},

         #{desc => "order client to continue (with send)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send),
                           ok
                   end},
         #{desc => "await client ready (with send)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send)
                   end},
         #{desc => "order server to continue (with recv)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, recv),
                           ok
                   end},
         #{desc => "await server ready (recv plain)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv)
                   end},


         %% *** Termination ***
         #{desc => "order client to terminate",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client termination",
           cmd  => fun(#{client := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client, State),
                           {ok, State1}
                   end},
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start client evaluator"),
    Client = ?SEV_START("client", ClientSeq, InitState),

    i("start tester evaluator"),
    TesterInitState = #{server => Server#ev.pid,
                        client => Client#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the credentials control message header is received when
%% setting the socket 'socket' option true when using sendmsg/recvmsg
%% on an IPv4 TCP (stream) socket.
%% So, this is done on the receiving side: 
%%
%%               socket:setopt(Sock, socket, passcred, boolean()).
%%
%% We *may* need to run the different entities (server and client) in 
%% separate VM (os processes) for this to actually work.
%% As it is now, the client does *not* get any credentials!
%% Until this has been done, this case is skipped!.

api_opt_sock_passcred_tcp4(suite) ->
    [];
api_opt_sock_passcred_tcp4(doc) ->
    [];
api_opt_sock_passcred_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_passcred_tcp4,
           fun() -> has_support_sock_passcred(),
                    not_yet_implemented()
           end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, socket, passcred, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, socket, passcred)
                          end,
                   Send = fun(Sock, Data) ->
                                  Msg = #{iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 proto  => tcp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_sock_passcred_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_sock_passcred_tcp(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{domain := local,
                         lsock  := LSock,
                         lsa    := LSA} = _State) ->
                           case socket:bind(LSock, LSA) of
                               ok ->
                                   ok; % We do not care about the port for local
                               {error, _} = ERROR ->
                                   ERROR
                           end;
                      (#{lsock := LSock, lsa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   ?SEV_IPRINT("bound to port: ~w", [Port]),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{domain := local,
                         tester := Tester, lsa := #{path := Path}}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Path),
                           ok;
                      (#{tester := Tester, lport := Port}) ->
                           %% This is actually not used for unix domain socket
                           ?SEV_ANNOUNCE_READY(Tester, init, Port),
                           ok
                   end},


         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:accept(LSock) of
                               {ok, Sock} ->
                                   ?SEV_IPRINT("accepted: ~n   ~p", [Sock]),
                                   {ok, State#{csock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept),
                           ok
                   end},

         %% *** First message ***

         #{desc => "await (recv) request 1",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {[], ?BASIC_REQ}} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready (send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           ok
                   end},

         %% *** Second message ***

         #{desc => "await (recv) request 2",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {[], ?BASIC_REQ}} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready (send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           ok
                   end},

         %% *** Third message ***

         #{desc => "await (recv) request 3",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {[], ?BASIC_REQ}} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready (send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close connection socket",
           cmd  => fun(#{csock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(csock, State)}
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{domain   := local,
                         lsock    := Sock,
                         local_sa := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() ->
                                                   maps:remove(local_sa, State)
                                           end,
                                           fun() -> State end),
                           {ok, maps:remove(lsock, State1)};
                      (#{lsock := LSock} = State) ->
                           case socket:close(LSock) of
                               ok ->
                                   {ok, maps:remove(lsock, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    ClientSeq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(#{domain := local} = State) ->
                           {Tester, Path} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_path => Path}};
                      (State) ->
                           {Tester, Port} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_port => Port}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain      := local = Domain,
                         server_path := Path} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = #{family => Domain, path => Path},
                           {ok, State#{local_sa => LSA, server_sa => SSA}};
                      (#{domain := Domain, server_port := Port} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = LSA#{port => Port},
                           {ok, State#{local_sa => LSA, server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "await continue (connect)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, connect)
                   end},
         #{desc => "connect to server",
           cmd  => fun(#{sock := Sock, server_sa := SSA}) ->
                           socket:connect(Sock, SSA)
                   end},
         #{desc => "announce ready (connect)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect),
                           ok
                   end},

         %% *** First message (default=wo passcred) ***

         #{desc => "await continue (verify timestamp off)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, verify_passcred)
                   end},
         #{desc => "verify passcred off",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = _Value} ->
                                   ?SEV_IPRINT("passcred: ~p", [_Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected passcred: ~p",
                                               [Unexpected]),
                                   {error, {unexpected_passcred, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting passcred:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end                                   
                   end},
         #{desc => "announce ready (passcred off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, passcred_off),
                           ok
                   end},

         #{desc => "await continue (send request)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 1 (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 1 (from server, wo passcred)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {[], ?BASIC_REP}} ->
                                   ok;
                               {ok, {[], UnexpData}} ->
                                   {error, {unexpected_reply_data, UnexpData}};
                               {ok, {BadCMsgs, ?BASIC_REP}} ->
                                   {error, {unexpected_reply_cmsgs,
                                            BadCMsgs}};
                               {ok, BadReply} ->
                                   {error, {unexpected_reply, BadReply}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 1 (recv reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         %% *** Second message (w passcred) ***

         #{desc => "await continue (enable passcred)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, enable_passcred)
                   end},
         #{desc => "enable passcred",
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("passcred enabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed enable passcred:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end                                   
                   end},
         #{desc => "announce ready (passcred on)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, passcred_on),
                           ok
                   end},

         #{desc => "await continue (send request 2)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 2 (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send request 2)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 2 (from server, w passcred)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           %% socket:setopt(Sock, otp, debug, true),
                           case Recv(Sock) of
                               {ok, {[#{level := socket,
                                        type  := passcred,
                                        value := Cred}], ?BASIC_REP}} ->
                                   %% socket:setopt(Sock, otp, debug, false),
                                   ?SEV_IPRINT("received reply *with* "
                                               "expected passcred: "
                                               "~n   ~p", [Cred]),
                                   ok;
                               {ok, {BadCMsgs, ?BASIC_REP}} ->
                                   %% socket:setopt(Sock, otp, debug, false),
                                   {error, {unexpected_reply_cmsgs,
                                            BadCMsgs}};
                               {ok, {[#{level := socket,
                                        type  := passcred,
                                        value := _Cred}], BadData}} ->
                                   %% socket:setopt(Sock, otp, debug, false),
                                   {error, {unexpected_reply_data,
                                            BadData}};
                               {ok, BadReply} ->
                                   %% socket:setopt(Sock, otp, debug, false),
                                   {error, {unexpected_reply, BadReply}};
                               {error, _} = ERROR ->
                                   %% socket:setopt(Sock, otp, debug, false),
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 2 (recv reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         %% *** Third message (wo passcred) ***

         #{desc => "await continue (disable passcred)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, disable_passcred)
                   end},
         #{desc => "disable passcred",
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           case Set(Sock, false) of
                               ok ->
                                   ?SEV_IPRINT("passcred disabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed disable timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end                                   
                   end},
         #{desc => "announce ready (passcred off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, passcred_off),
                           ok
                   end},

         #{desc => "await continue (send request 3)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 3 (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send request 3)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 3 (from server, wo passcred)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {[], ?BASIC_REP}} ->
                                   ?SEV_IPRINT("received reply *without* "
                                               "passcred"),
                                   ok;
                               {ok, {BadCMsgs, ?BASIC_REP}} ->
                                   {error, {unexpected_reply_cmsgs,
                                            BadCMsgs}};
                               {ok, {[], BadData}} ->
                                   {error, {unexpected_reply_data,
                                            BadData}};
                               {ok, BadReply} ->
                                   {error, {unexpected_reply, BadReply}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 3 (recv reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{domain   := local,
                         sock     := Sock,
                         local_sa := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() ->
                                                   maps:remove(local_sa, State)
                                           end,
                                           fun() -> State end),
                           {ok, maps:remove(sock, State1)};
                      (#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% Start the client
         #{desc => "order client start",
           cmd  => fun(#{client := Pid, server_port := Port} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Port),
                           ok
                   end},
         #{desc => "await client ready (init)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, init)
                   end},

         %% *** The actual test ***
         #{desc => "order server to continue (with accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
         ?SEV_SLEEP(?SECS(1)),
         #{desc => "order client to continue (with connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, connect),
                           ok
                   end},
         #{desc => "await client ready (connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, connect)
                   end},
         #{desc => "await server ready (accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, accept)
                   end},

         %% *** First message (default=wo passcred) ***

         #{desc => "order client to continue (with verify timestamp off)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, verify_passcred),
                           ok
                   end},
         #{desc => "await client ready (passcred off)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, passcred_off)
                   end},

         #{desc => "order client to continue (with send request 1)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client ready (with send request 1)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send_req)
                   end},
         #{desc => "await server ready (request recv 1)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply sent)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client ready (reply recv)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, recv_reply)
                   end},

         %% Second message (w passcred)

         #{desc => "order client to continue (with enable passcred)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, enable_passcred),
                           ok
                   end},
         #{desc => "await client ready (passcred on)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, passcred_on)
                   end},

         #{desc => "order client to continue (with send request 2)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client ready (with send request 2)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send_req)
                   end},
         #{desc => "await server ready (request recv 2)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply 2)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply sent 2)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client ready (reply recv 2)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, recv_reply)
                   end},

         %% Third message (wo passcred)

         #{desc => "order client to continue (with disable passcred)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, disable_passcred),
                           ok
                   end},
         #{desc => "await client ready (passcred off)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, passcred_off)
                   end},

         #{desc => "order client to continue (with send request 3)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client ready (with send request 3)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send_req)
                   end},
         #{desc => "await server ready (request recv 3)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply 3)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply sent 3)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client ready (reply recv 3)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, recv_reply)
                   end},


         %% *** Termination ***
         #{desc => "order client to terminate",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client termination",
           cmd  => fun(#{client := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client, State),
                           {ok, State1}
                   end},
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start client evaluator"),
    Client = ?SEV_START("client", ClientSeq, InitState),

    i("start tester evaluator"),
    TesterInitState = #{server => Server#ev.pid,
                        client => Client#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the the peek-off socket option for a unix domain socket
%% (stream TCP in this case).
%%
%% THIS IS A PLACEHOLDER!!
%%
%%

api_opt_sock_peek_off_tcpL(suite) ->
    [];
api_opt_sock_peek_off_tcpL(doc) ->
    [];
api_opt_sock_peek_off_tcpL(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_peek_off_tcpL,
           fun() ->
                   has_support_unix_domain_socket(),
                   has_support_sock_peek_off(),
                   has_support_msg_flag(peek)
           end,
           fun() ->
                   Set  = fun(Sock, Val) when is_integer(Val) ->
                                  socket:setopt(Sock, socket, peek_off, Val)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, socket, peek_off)
                          end,
                   Send = fun(Sock, Data) ->
                                  socket:send(Sock, Data)
                          end,
                   Recv = fun(Sock, L, false) ->
                                  socket:recv(Sock, L);
                             (Sock, L, true) ->
                                  socket:recv(Sock, L, [peek])
                          end,
                   InitState = #{domain => local,
                                 proto  => default, % Type = stream => tcp
                                 set    => Set,
                                 get    => Get,
                                 send   => Send,
                                 recv   => Recv},
                   ok = api_opt_sock_peek_off(InitState)
           end).

api_opt_sock_peek_off(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{lsock := LSock,
                         lsa   := LSA} = _State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   %% We do not care about the port for local
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester, lsa := #{path := Path}}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Path),
                           ok
                   end},

         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:accept(LSock) of
                               {ok, Sock} ->
                                   ?SEV_IPRINT("accepted: ~n   ~p", [Sock]),
                                   {ok, State#{csock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept),
                           ok
                   end},


         %% The actual test

         %% 1) peek (0 = everything: 1,2,3,4,5,6,7,8)
         #{desc => "1a: await continue (peek)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, peek)
                   end},
         #{desc => "1a: peek read",
           cmd  => fun(#{csock := Sock,
                         recv  := Recv} = _State) ->
                           case Recv(Sock, 0, true) of
                               {ok, <<1,2,3,4,5,6,7,8>>} ->
                                   ?SEV_IPRINT("peek'ed expected data"),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "1a: announce ready (peek)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, peek),
                           ok
                   end},

         #{desc => "1b: await continue (verify peek-off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, verify_peek_off)
                   end},
         #{desc => "1b: verify peek-off",
           cmd  => fun(#{csock := Sock,
                         get   := Get} = _State) ->
                           case Get(Sock) of
                               {ok, DefaultPeekOff} ->
                                   ?SEV_IPRINT("verify peek-off: ~w",
                                               [DefaultPeekOff]),
                                   ok;
                               {error, {not_supported, {socket, peek_off}}} ->
                                   {skip, "Not supported"};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "1b: announce ready (verify peek-off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, verify_peek_off),
                           ok
                   end},


         %% 2) set peek-off to 4
         #{desc => "2a: await continue (set peek-off: 4)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, set_peek_off)
                   end},
         #{desc => "2a: set peek-off: 4",
           cmd  => fun(#{csock := Sock,
                         set   := Set} = _State) ->
                           case Set(Sock, 4) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "2a: announce ready (set peek-off: 4)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, set_peek_off),
                           ok
                   end},

         #{desc => "2b: await continue (verify peek-off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, verify_peek_off)
                   end},
         #{desc => "2b: verify peek-off",
           cmd  => fun(#{csock := Sock,
                         get   := Get} = _State) ->
                           case Get(Sock) of
                               {ok, 4 = PeekOff} ->
                                   ?SEV_IPRINT("verify peek-off: ~w", [PeekOff]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "2b: announce ready (verify peek-off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, verify_peek_off),
                           ok
                   end},


         %% 3) peek (0 = everything: 5,6,7,8)
         %%    NOTE THAT THIS WILL MOVE THE PEEK-OFF "POINTER" TO THE END OF 
         %%    THE *CURRENT* DATA POSITION IN THE BUFFER (READY FOR NEXT BATCH
         %%    OF DATA).
         #{desc => "3a: await continue (peek)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, peek)
                   end},
         #{desc => "3a: peek read",
           cmd  => fun(#{csock := Sock,
                         recv  := Recv} = _State) ->
                           case Recv(Sock, 0, true) of
                               {ok, <<5,6,7,8>>} ->
                                   ?SEV_IPRINT("peek'ed expected data"),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "3a: announce ready (peek)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, peek),
                           ok
                   end},

         #{desc => "3b: await continue (verify peek-off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, verify_peek_off)
                   end},
         #{desc => "3b: verify peek-off",
           cmd  => fun(#{csock := Sock,
                         get   := Get} = _State) ->
                           case Get(Sock) of
                               {ok, 8 = PeekOff} ->
                                   ?SEV_IPRINT("verify peek-off: ~w", [PeekOff]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "3b: announce ready (verify peek-off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, verify_peek_off),
                           ok
                   end},


         %% 4) read two byte(s): 1,2
         #{desc => "4a: await continue (read 2 byte)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, read)
                   end},
         #{desc => "4a: read (2 bytes)",
           cmd  => fun(#{csock := Sock,
                         recv  := Recv} = _State) ->
                           case Recv(Sock, 2, false) of
                               {ok, <<1,2>>} ->
                                   ?SEV_IPRINT("read expected data"),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "4a: announce ready (read)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, read),
                           ok
                   end},

         #{desc => "4b: await continue (verify peek-off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, verify_peek_off)
                   end},
         #{desc => "4b: verify peek-off",
           cmd  => fun(#{csock := Sock,
                         get   := Get} = _State) ->
                           case Get(Sock) of
                               {ok, 6 = PeekOff} ->
                                   ?SEV_IPRINT("verify peek-off: ~w", [PeekOff]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "4b: announce ready (verify peek-off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, verify_peek_off),
                           ok
                   end},


         %% 5) read the rest: 3,4,5,6,7,8)
         #{desc => "5a: await continue (read the rest)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, read)
                   end},
         #{desc => "5a: read (the rest)",
           cmd  => fun(#{csock := Sock,
                         recv  := Recv} = _State) ->
                           case Recv(Sock, 0, false) of
                               {ok, <<3,4,5,6,7,8>>} ->
                                   ?SEV_IPRINT("read expected data"),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "5a: announce ready (read)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, read),
                           ok
                   end},

         #{desc => "5b: await continue (verify peek-off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, verify_peek_off)
                   end},
         #{desc => "5b: verify peek-off",
           cmd  => fun(#{csock := Sock,
                         get   := Get} = _State) ->
                           case Get(Sock) of
                               {ok, PeekOff} ->
                                   ?SEV_IPRINT("verify peek-off: ~w", [PeekOff]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "5b: announce ready (verify peek-off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, verify_peek_off),
                           ok
                   end},


         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close connection socket",
           cmd  => fun(#{csock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(csock, State)}
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{lsock := Sock,
                         lsa   := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() ->
                                                   maps:remove(local_sa, State)
                                           end,
                                           fun() -> State end),
                           {ok, maps:remove(lsock, State1)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    ClientSeq = 
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(#{domain := local} = State) ->
                           {Tester, Path} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_path => Path}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain      := local = Domain,
                         server_path := Path} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = #{family => Domain, path => Path},
                           {ok, State#{local_sa => LSA, server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         #{desc => "await continue (connect)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, connect)
                   end},
         #{desc => "connect to server",
           cmd  => fun(#{sock := Sock, server_sa := SSA}) ->
                           socket:connect(Sock, SSA)
                   end},
         #{desc => "announce ready (connect)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect),
                           ok
                   end},


         %% *** The actual test ***
         #{desc => "await continue (send data)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_data)
                   end},
         #{desc => "send data (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, <<1:8/integer,
                                        2:8/integer,
                                        3:8/integer,
                                        4:8/integer,
                                        5:8/integer,
                                        6:8/integer,
                                        7:8/integer,
                                        8:8/integer>>)
                   end},
         #{desc => "announce ready (send data)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_data),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock     := Sock,
                         local_sa := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() ->
                                                   maps:remove(local_sa, State)
                                           end,
                                           fun() -> State end),
                           {ok, maps:remove(sock, State1)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% Start the client
         #{desc => "order client start",
           cmd  => fun(#{client := Pid, server_port := Port} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Port),
                           ok
                   end},
         #{desc => "await client ready (init)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, init)
                   end},

         %% Establish the connection
         #{desc => "order server to continue (with accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
         #{desc => "order client to continue (with connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, connect),
                           ok
                   end},
         #{desc => "await client ready (connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, connect)
                   end},
         #{desc => "await server ready (accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, accept)
                   end},


         %% *** The actual test ***
         #{desc => "order client to continue (with send data)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_data),
                           ok
                   end},
         #{desc => "await client ready (with send data)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send_data)
                   end},

         %% There is no way to be sure that the data has actually arrived,
         %% and with no data on the server side, the peek will fail.
         %% Hopfully a sleep will take care of this...
         ?SEV_SLEEP(?SECS(1)),

         %% 1) peek
         #{desc => "1a: order server to continue (peek)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, peek),
                           ok
                   end},
         #{desc => "1a: await server ready (peek)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, peek)
                   end},

         #{desc => "1b: order server to continue (verify peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, verify_peek_off),
                           ok
                   end},
         #{desc => "1b: await server ready (verify peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, verify_peek_off)
                   end},


         %% 2) set peek-off
         #{desc => "2a: order server to continue (set peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, set_peek_off),
                           ok
                   end},
         #{desc => "2a: await server ready (set peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, set_peek_off)
                   end},

         #{desc => "2b: order server to continue (verify peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, verify_peek_off),
                           ok
                   end},
         #{desc => "2b: await server ready (verify peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, verify_peek_off)
                   end},



         %% 3) peek
         #{desc => "3a: order server to continue (peek)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, peek),
                           ok
                   end},
         #{desc => "3a: await server ready (peek)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, peek)
                   end},

         #{desc => "3b: order server to continue (verify peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, verify_peek_off),
                           ok
                   end},
         #{desc => "3b: await server ready (verify peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, verify_peek_off)
                   end},



         %% 4) read part
         #{desc => "4a: order server to continue (read part)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, read),
                           ok
                   end},
         #{desc => "4a: await server ready (peek)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, read)
                   end},

         #{desc => "4b: order server to continue (verify peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, verify_peek_off),
                           ok
                   end},
         #{desc => "4b: await server ready (verify peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, verify_peek_off)
                   end},


         %% 5) read (the rest)
         #{desc => "5a: order server to continue (read the rest)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, read),
                           ok
                   end},
         #{desc => "5a: await server ready (peek)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, read)
                   end},

         #{desc => "5b: order server to continue (verify peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, verify_peek_off),
                           ok
                   end},
         #{desc => "5b: await server ready (verify peek-off)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, verify_peek_off)
                   end},


         %% *** Termination ***
         #{desc => "order client to terminate",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client termination",
           cmd  => fun(#{client := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client, State),
                           {ok, State1}
                   end},
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start client evaluator"),
    Client = ?SEV_START("client", ClientSeq, InitState),
    i("await evaluator(s)"),

    i("start tester evaluator"),
    TesterInitState = #{server => Server#ev.pid,
                        client => Client#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that we get the peer credentials for a connected unix domain
%% TCP (stream) socket.
%% That is, all we need to do is to create a slave node, and have 
%% process connect from that to a local (unix domain socket) socket.
%%
%% THIS IS A PLACEHOLDER!!
%%
%% We need to figure out what the ucred structure looks like,
%% and decode it...
%%

api_opt_sock_peercred_tcpL(suite) ->
    [];
api_opt_sock_peercred_tcpL(doc) ->
    [];
api_opt_sock_peercred_tcpL(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_peercred_tcpL,
           fun() ->
                   has_support_unix_domain_socket(),
                   has_support_sock_peercred(),
                   not_yet_implemented()
           end,
           fun() ->
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, socket, peercred)
                          end,
                   InitState = #{domain => local,
                                 proto  => default, % Type = stream => tcp
                                 get    => Get},
                   ok = api_opt_sock_peercred_tcp(InitState)
           end).


api_opt_sock_peercred_tcp(_InitState) ->
    %% ServerSeq =
    %%     [
    %%      %% *** Wait for start order part ***
    %%      #{desc => "await start (from tester)",
    %%        cmd  => fun(State) ->
    %%                        {Tester, Backlog} = ?SEV_AWAIT_START(),
    %%                        {ok, State#{tester  => Tester,
    %%                                    backlog => Backlog}}
    %%                end},
    %%      #{desc => "monitor tester",
    %%        cmd  => fun(#{tester := Tester} = _State) ->
    %%                        _MRef = erlang:monitor(process, Tester),
    %%                        ok
    %%                end},

    %%      %% *** Init part ***
    %%      #{desc => "which local address",
    %%        cmd  => fun(#{domain := Domain} = State) ->
    %%                        LSA = which_local_socket_addr(Domain),
    %%                        {ok, State#{lsa => LSA}}
    %%                end},
    %%      #{desc => "create listen socket",
    %%        cmd  => fun(#{domain := Domain, proto := Proto} = State) ->
    %%                        case socket:open(Domain, stream, Proto) of
    %%                            {ok, Sock} ->
    %%                                {ok, State#{lsock => Sock}};
    %%                            {error, _} = ERROR ->
    %%                                ERROR
    %%                        end
    %%                end},
    %%      #{desc => "bind to local address",
    %%        cmd  => fun(#{domain := local,
    %%                      lsock  := LSock,
    %%                      lsa    := LSA} = _State) ->
    %%                        case socket:bind(LSock, LSA) of
    %%                            ok ->
    %%                                ok;
    %%                            {error, _} = ERROR ->
    %%                                ERROR
    %%                        end
    %%                end},
    %%      #{desc => "make listen socket",
    %%        cmd  => fun(#{lsock := LSock}) ->
    %%                        socket:listen(LSock)
    %%                end},
    %%      #{desc => "announce ready (init)",
    %%        cmd  => fun(#{domain := local,
    %%                      tester := Tester, lsa := #{path := Path}}) ->
    %%                        ?SEV_ANNOUNCE_READY(Tester, init, Path),
    %%                        ok
    %%                end},


    %%      %% The actual test
    %%      #{desc => "await continue (accept)",
    %%        cmd  => fun(#{tester := Tester}) ->
    %%                        ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
    %%                end},
    %%      #{desc => "await connection",
    %%        cmd  => fun(#{lsock := LSock} = State) ->
    %%                        case socket:accept(LSock) of
    %%                            {ok, Sock} ->
    %%                                ?SEV_IPRINT("accepted: ~n   ~p", [Sock]),
    %%                                {ok, State#{csock => Sock}};
    %%                            {error, _} = ERROR ->
    %%                                ERROR
    %%                        end
    %%                end},
    %%      #{desc => "announce ready (accept)",
    %%        cmd  => fun(#{tester := Tester}) ->
    %%                        ?SEV_ANNOUNCE_READY(Tester, accept),
    %%                        ok
    %%                end},

    %%      #{desc => "await continue (peercred)",
    %%        cmd  => fun(#{tester := Tester}) ->
    %%                        ?SEV_AWAIT_CONTINUE(Tester, tester, peercred)
    %%                end},
    %%      #{desc => "get peercred",
    %%        cmd  => fun(#{csock := Sock, get := Get} = _State) ->
    %%                        case Get(Sock) of
    %%                            {ok, PeerCred} ->
    %%                                ?SEV_IPRINT("PeerCred: ~n   ~p", [PeerCred]),
    %%                                ok;
    %%                            {error, _} = ERROR ->
    %%                                ERROR
    %%                        end
    %%                end},
    %%      #{desc => "announce ready (peercred)",
    %%        cmd  => fun(#{tester := Tester}) ->
    %%                        ?SEV_ANNOUNCE_READY(Tester, peercred),
    %%                        ok
    %%                end},


    %%      %% Termination
    %%      #{desc => "await terminate",
    %%        cmd  => fun(#{tester := Tester} = State) ->
    %%                        case ?SEV_AWAIT_TERMINATE(Tester, tester) of
    %%                            ok ->
    %%                                {ok, maps:remove(tester, State)};
    %%                            {error, _} = ERROR ->
    %%                                ERROR
    %%                        end
    %%                end},
    %%      #{desc => "close connection socket",
    %%        cmd  => fun(#{csock := Sock} = State) ->
    %%                        ok = socket:close(Sock),
    %%                        {ok, maps:remove(csock, State)}
    %%                end},
    %%      #{desc => "close listen socket",
    %%        cmd  => fun(#{domain := local,
    %%                      lsock  := Sock,
    %%                      lsa    := #{path := Path}} = State) ->
    %%                        ok = socket:close(Sock),
    %%                        State1 =
    %%                            unlink_path(Path,
    %%                                        fun() ->
    %%                                                maps:remove(lsa, State)
    %%                                        end,
    %%                                        fun() -> State end),
    %%                        {ok, maps:remove(lsock, State1)}
    %%                end},

    %%      %% *** We are done ***
    %%      ?SEV_FINISH_NORMAL
    %%     ],


    %% ClientSeq =
    %%     [
    %%      %% *** Wait for start order part ***
    %%      #{desc => "await start",
    %%        cmd  => fun(#{domain := local} = State) ->
    %%                        {Tester, Path} = ?SEV_AWAIT_START(),
    %%                        {ok, State#{tester    => Tester,
    %%                                    server_path => Path}}
    %%                end},
    %%      #{desc => "monitor tester",
    %%        cmd  => fun(#{tester := Tester} = _State) ->
    %%                        _MRef = erlang:monitor(process, Tester),
    %%                        ok
    %%                end},


    %%      %% *** Init part ***
    %%      #{desc => "which local address",
    %%        cmd  => fun(#{domain      := local = Domain,
    %%                      server_path := Path} = State) ->
    %%                        LSA = which_local_socket_addr(Domain),
    %%                        SSA = #{family => Domain, path => Path},
    %%                        {ok, State#{local_sa => LSA, server_sa => SSA}}
    %%                end},
    %%      #{desc => "create node",
    %%        cmd  => fun(#{host := Host} = State) ->
    %%     		   ?SEV_IPRINT("try create node on ~p", [Host]),
    %%                        case start_node(Host, client) of
    %%                            {ok, Node} ->
    %%                                ?SEV_IPRINT("client node ~p started",
    %%                                            [Node]),
    %%                                {ok, State#{node => Node}};
    %%                            {error, Reason} ->
    %%                                {skip, Reason}
    %%                        end
    %%                end},
    %%       #{desc => "monitor client node",
    %%        cmd  => fun(#{node := Node} = _State) ->
    %%                        true = erlang:monitor_node(Node, true),
    %%                        ok
    %%                end},
    %%      #{desc => "start remote client on client node",
    %%        cmd  => fun(#{node := Node} = State) ->
    %%                        Pid = api_opt_sock_peercred_tcp_client_start(Node),
    %%                        ?SEV_IPRINT("remote client ~p started", [Pid]),
    %%                        {ok, State#{rclient => Pid}}
    %%                end},
    %%      #{desc => "monitor remote client",
    %%        cmd  => fun(#{rclient := Pid}) ->
    %%                        _MRef = erlang:monitor(process, Pid),
    %%                        ok
    %%                end},
    %%      #{desc => "order remote client to start",
    %%        cmd  => fun(#{rclient   := Client,
    %%                     proto      := Proto,
    %%                      server_sa := ServerSA}) ->
    %%                        ?SEV_ANNOUNCE_START(Client, {Proto, ServerSA}),
    %%                        ok
    %%                end},
    %%      #{desc => "await remote client ready",
    %%        cmd  => fun(#{tester  := Tester,
    %%                      rclient := Client} = _State) ->
    %%                        ?SEV_AWAIT_READY(Client, rclient, init,
    %%                                         [{tester, Tester}])
    %%                end},
    %%      #{desc => "announce ready (init)",
    %%        cmd  => fun(#{tester := Tester}) ->
    %%                        ?SEV_ANNOUNCE_READY(Tester, init),
    %%                        ok
    %%                end},


    %%      %% The actual test
    %%      #{desc => "await continue (connect)",
    %%        cmd  => fun(#{tester  := Tester,
    %%                      rclient := Client} = State) ->
    %%                        case ?SEV_AWAIT_CONTINUE(Tester, tester, connect,
    %%                                                 [{rclient, Client}]) of
    %%                            {ok, {ConTimeout, ConLimit}} ->
    %%                                {ok, State#{connect_timeout => ConTimeout,
    %%                                            connect_limit   => ConLimit}};
    %%                            {error, _} = ERROR ->
    %%                                ERROR
    %%                        end
    %%                end},
    %%      #{desc => "order remote client to continue (connect)",
    %%        cmd  => fun(#{rclient         := RClient,
    %%                      connect_timeout := ConTimeout,
    %%                      connect_limit   := ConLimit}) ->
    %%                        ?SEV_ANNOUNCE_CONTINUE(RClient, connect,
    %%                                               {ConTimeout, ConLimit}),
    %%                        ok
    %%                end},
    %%      #{desc => "await remote client ready (connect)",
    %%        cmd  => fun(#{tester  := Tester,
    %%                      rclient := RClient} = State) ->
    %%                        case ?SEV_AWAIT_READY(RClient, rclient, connect,
    %%                                              [{tester, Tester}]) of
    %%                            {ok, ok = _Result} ->
    %%                                {ok, maps:remove(connect_limit, State)};
    %%                            {ok, {error, {connect_limit_reached,R,L}}} ->
    %%                                {skip,
    %%                                 ?LIB:f("Connect limit reached ~w: ~w",
    %%                                        [L, R])};
    %%                            {ok, Result} ->
    %%                                Result;
    %%                            {error, _} = ERROR ->
    %%                                ERROR
    %%                        end
    %%                end},
    %%      #{desc => "announce ready (connect)",
    %%        cmd  => fun(#{tester := Tester}) ->
    %%                        ?SEV_ANNOUNCE_READY(Tester, connect),
    %%                        ok
    %%                end},

    %%      %% Termination
    %%      #{desc => "await terminate (from tester)",
    %%        cmd  => fun(#{tester  := Tester,
    %%                      rclient := RClient} = State) ->
    %%                        case ?SEV_AWAIT_TERMINATE(Tester, tester,
    %%                                                  [{rclient, RClient}]) of
    %%                            ok ->
    %%                                {ok, maps:remove(tester, State)};
    %%                            {error, _} = ERROR ->
    %%                                ERROR
    %%                        end
    %%                end},
    %%      #{desc => "kill remote client",
    %%        cmd  => fun(#{rclient := Client}) ->
    %%                        ?SEV_ANNOUNCE_TERMINATE(Client),
    %%                        ok
    %%                end},
    %%      #{desc => "await remote client termination",
    %%        cmd  => fun(#{rclient := Client} = State) ->
    %%                        ?SEV_AWAIT_TERMINATION(Client),
    %%                        State1 = maps:remove(rclient, State),
    %%                        {ok, State1}
    %%                end},
    %%      #{desc => "stop client node",
    %%        cmd  => fun(#{node := Node} = _State) ->
    %%                        stop_node(Node)
    %%                end},
    %%      #{desc => "await client node termination",
    %%        cmd  => fun(#{node := Node} = State) ->
    %%                        receive
    %%                            {nodedown, Node} ->
    %%                                State1 = maps:remove(node_id, State),
    %%                                State2 = maps:remove(node,    State1),
    %%                                {ok, State2}
    %%                        end
    %%                end},

    %%      %% *** We are done ***
    %%      ?SEV_FINISH_NORMAL
    %%    ],

    %% TesterSeq =
    %%     [
    %%      %% *** Init part ***
    %%      #{desc => "monitor server",
    %%        cmd  => fun(#{server := Server} = _State) ->
    %%                        _MRef = erlang:monitor(process, Server),
    %%                        ok
    %%                end},
    %%      #{desc => "monitor client",
    %%        cmd  => fun(#{client := Client} = _State) ->
    %%                        _MRef = erlang:monitor(process, Client),
    %%                        ok
    %%                end},
    %%      #{desc => "which local address",
    %%        cmd  => fun(#{domain := Domain} = State) ->
    %%                        LSA = which_local_socket_addr(Domain),
    %%                        {ok, State#{local_sa => LSA}}
    %%                end},
    %%      #{desc => "order server start",
    %%        cmd  => fun(#{server  := Server,
    %%                      backlog := Backlog}) ->
    %%                        ?SEV_ANNOUNCE_START(Server, Backlog),
    %%                        ok
    %%                end},
    %%      #{desc => "await server ready (init)",
    %%        cmd  => fun(#{server := Server, local_sa := LSA} = State) ->
    %%                        {ok, Port} = ?SEV_AWAIT_READY(Server, server, init),
    %%                        ServerSA = LSA#{port => Port},
    %%                        {ok, State#{server_sa => ServerSA}}
    %%                end},
    %%      #{desc => "order client start",
    %%        cmd  => fun(#{client    := Client,
    %%                      server_sa := ServerSA}) ->
    %%                        ?SEV_ANNOUNCE_START(Client, ServerSA),
    %%                        ok
    %%                end},
    %%      #{desc => "await client ready (init)",
    %%        cmd  => fun(#{client := Client} = _State) ->
    %%                        ?SEV_AWAIT_READY(Client, client, init),
    %%                        ok
    %%                end},


    %%      %% The actual test
    %%      %% The server accepts the connect from the client, announces
    %%      %% this to us (accept) and then attempts to get peercred.
    %%      #{desc => "order client continue (connect)",
    %%        cmd  => fun(#{client        := Client,
    %%                      timeout       := Timeout,
    %%                      connect_limit := ConLimit} = _State) ->
    %%                        ?SEV_ANNOUNCE_CONTINUE(Client, connect,
    %%                                               {Timeout, ConLimit}),
    %%                        ok
    %%                end},
    %%      #{desc => "await client ready (connect)",
    %%        cmd  => fun(#{server := Server,
    %%                      client := Client} = _State) ->
    %%                        case ?SEV_AWAIT_READY(Client, client, connect,
    %%                                              [{server, Server}]) of
    %%                            ok ->
    %%                                ok;
    %%                            {error, _} = ERROR ->
    %%                                ERROR
    %%                        end
    %%                end},
    %%      #{desc => "await server ready (accept)",
    %%        cmd  => fun(#{server := Server,
    %%                      client := Client} = _State) ->
    %%                        case ?SEV_AWAIT_READY(Server, server, accept,
    %%                                              [{client, Client}]) of
    %%                            ok ->
    %%                                ok;
    %%                            {error, _} = ERROR ->
    %%                                ERROR
    %%                        end
    %%                end},
    %%      #{desc => "await server ready (peercred)",
    %%        cmd  => fun(#{server := Server,
    %%                      client := Client} = _State) ->
    %%                        case ?SEV_AWAIT_READY(Server, server, peercred,
    %%                                              [{client, Client}]) of
    %%                            ok ->
    %%                                ok;
    %%                            {error, _} = ERROR ->
    %%                                ERROR
    %%                        end
    %%                end},


    %%      %% *** Terminate server ***
    %%      #{desc => "order client terminate",
    %%        cmd  => fun(#{client := Client} = _State) ->
    %%                        ?SEV_ANNOUNCE_TERMINATE(Client),
    %%                        ok
    %%                end},
    %%      #{desc => "await client down",
    %%        cmd  => fun(#{client := Client} = State) ->
    %%                        ?SEV_AWAIT_TERMINATION(Client),
    %%                        State1 = maps:remove(client,    State),
    %%                        {ok, State1}
    %%                end},
    %%      #{desc => "order server terminate",
    %%        cmd  => fun(#{server := Server} = _State) ->
    %%                        ?SEV_ANNOUNCE_TERMINATE(Server),
    %%                        ok
    %%                end},
    %%      #{desc => "await server down",
    %%        cmd  => fun(#{server := Server} = State) ->
    %%                        ?SEV_AWAIT_TERMINATION(Server),
    %%                        State1 = maps:remove(server,    State),
    %%                        State2 = maps:remove(server_sa, State1),
    %%                        {ok, State2}
    %%                end},

    %%      %% *** We are done ***
    %%      ?SEV_FINISH_NORMAL
    %%     ],

    %% i("create server evaluator"),
    %% ServerInitState = #{domain => maps:get(domain, InitState)},
    %% Server          = ?SEV_START("server", ServerSeq, ServerInitState),

    %% i("create client evaluator"),
    %% ClientInitState = #{host   => local_host(),
    %%                     domain => maps:get(domain, InitState)},
    %% Client          = ?SEV_START("client", ClientSeq, ClientInitState),

    %% i("create tester evaluator"),
    %% TesterInitState = InitState#{server => Server#ev.pid,
    %%                              client => Client#ev.pid},
    %% Tester          = ?SEV_START("tester", TesterSeq, TesterInitState),

    %% i("await evaluator(s)"),
    %% ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).


    %% This should actually never be called (the conditions should cause a skip),
    %% but just to be on the safe side...
    skip.


%% api_opt_sock_peercred_tcp_client_start(Node) ->
%%     Self = self(),
%%     Fun  = fun() -> api_opt_sock_peercred_tcp_client(Self) end,
%%     erlang:spawn(Node, Fun).

%% api_opt_sock_peercred_tcp_client(Parent) ->
%%     api_opt_sock_peercred_tcp_client_init(Parent),
%%     {Proto, ServerSA} = api_opt_sock_peercred_tcp_client_await_start(Parent),
%%     Domain   = maps:get(family, ServerSA),
%%     api_opt_sock_peercred_tcp_client_announce_ready(Parent, init),
%%     api_opt_sock_peercred_tcp_client_await_continue(Parent, connect),
%%     Result = api_opt_sock_peercred_tcp_client_connect(Domain, Proto, ServerSA),
%%     ?SEV_IPRINT("result: ~p", [Result]),
%%     api_opt_sock_peercred_tcp_client_announce_ready(Parent, connect, Result),
%%     Reason = api_opt_sock_peercred_tcp_client_await_terminate(Parent),
%%     api_opt_sock_peercred_tcp_client_close(Result),
%%     exit(Reason).

%% api_opt_sock_peercred_tcp_client_init(Parent) ->
%%     put(sname, "rclient"),
%%     _MRef = erlang:monitor(process, Parent),
%%     ok.

%% api_opt_sock_peercred_tcp_client_await_start(Parent) ->
%%     ?SEV_AWAIT_START(Parent).

%% api_opt_sock_peercred_tcp_client_announce_ready(Parent, Slogan) ->
%%     ?SEV_ANNOUNCE_READY(Parent, Slogan).
%% api_opt_sock_peercred_tcp_client_announce_ready(Parent, Slogan, Result) ->
%%     ?SEV_ANNOUNCE_READY(Parent, Slogan, Result).

%% api_opt_sock_peercred_tcp_client_await_continue(Parent, Slogan) ->
%%     case ?SEV_AWAIT_CONTINUE(Parent, parent, Slogan) of
%%         ok ->
%%             ok;
%%         {ok, Extra} ->
%%             Extra;
%%         {error, Reason} ->
%%             exit({await_continue, Slogan, Reason})
%%     end.

%% api_opt_sock_peercred_tcp_client_await_terminate(Parent) ->
%%     case ?SEV_AWAIT_TERMINATE(Parent, parent) of
%%         ok ->
%%             ok;
%%         {error, Reason} ->
%%             Reason
%%     end.

%% api_opt_sock_peercred_tcp_client_connect(Domain, Proto, ServerSA) ->
%%     LSA  = which_local_socket_addr(Domain),
%%     Sock = case socket:open(Domain, stream, Proto) of
%%                {ok, S} ->
%%                    S;
%%                {error, OReason} ->
%%                    ?FAIL({open, OReason})
%%            end,
%%     case socket:bind(Sock, LSA) of
%%         ok ->
%%             ok;
%%         {error, BReason} ->
%%             (catch socket:close(Sock)),
%%             ?FAIL({bind, BReason})
%%     end,
%%     case socket:connect(Sock, ServerSA) of
%%         ok ->
%%             {ok, Sock};
%%         {error, Reason} ->
%%             (catch socket:close(Sock)),
%%             ?FAIL({connect, Reason})
%%     end.

%% api_opt_sock_peercred_tcp_client_close({ok, Sock}) ->
%%     (catch socket:close(Sock));
%% api_opt_sock_peercred_tcp_client_close(_) ->
%%     ok.




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the 'PRIORITY' socket 'socket' option with IPv4 UDP:
%%
%%               socket:setopt(Sock, socket, priority, integer()).
%%
%%

api_opt_sock_priority_udp4(suite) ->
    [];
api_opt_sock_priority_udp4(doc) ->
    [];
api_opt_sock_priority_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_priority_udp4,
           fun() -> has_support_sock_priority() end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, socket, priority, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, socket, priority)
                          end,
                   InitState = #{domain => inet,
                                 type   => dgram,
                                 proto  => udp,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_sock_priority(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the 'PRIORITY' socket 'socket' option with IPv4 TCP:
%%
%%               socket:setopt(Sock, socket, priority, integer()).
%%
%%

api_opt_sock_priority_tcp4(suite) ->
    [];
api_opt_sock_priority_tcp4(doc) ->
    [];
api_opt_sock_priority_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_priority_tcp4,
           fun() -> has_support_sock_priority() end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, socket, priority, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, socket, priority)
                          end,
                   InitState = #{domain => inet,
                                 type   => stream,
                                 proto  => tcp,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_sock_priority(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_sock_priority(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},

         #{desc => "open socket",
           cmd  => fun(#{domain := Domain,
                         type   := Type,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, Type, Proto),
                           {ok, State#{sock => Sock}}
                   end},
         #{desc => "bind",
           cmd  => fun(#{sock := Sock, lsa := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "get current (default) priority",
           cmd  => fun(#{sock := Sock, get := Get} = State) ->
                           case Get(Sock) of
                               {ok, Prio} ->
                                   ?SEV_IPRINT("(default) priority: ~p",
                                               [Prio]),
                                   {ok, State#{default => Prio}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) "
                                               "priority:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "change priority (to within non-root range)",
           cmd  => fun(#{sock    := Sock,
                         default := DefaultPrio,
                         set     := Set} = _State) ->
                           NewPrio =
                               if
                                   (DefaultPrio =< 0) andalso
                                   (DefaultPrio < 6) ->
                                       DefaultPrio+1;
                                   (DefaultPrio =:= 6) ->
                                       DefaultPrio-1;
                                   true ->
                                       3 % ...
                               end,
                           ?SEV_IPRINT("try set new priority (to ~p)",
                                       [NewPrio]),
                           case Set(Sock, NewPrio) of
                               ok ->
                                   ?SEV_IPRINT("priority changed (to ~p)",
                                               [NewPrio]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "change priority (to outside root-range)",
           cmd  => fun(#{sock := Sock,
                         set  := Set} = _State) ->
                           NewPrio = 42,
                           ?SEV_IPRINT("try set new priority (to ~p)",
                                       [NewPrio]),
                           case Set(Sock, NewPrio) of
                               ok ->
                                   ?SEV_IPRINT("priority changed (to ~p)",
                                               [NewPrio]),
                                   ok;
                               {error, eperm} ->
                                   ?SEV_IPRINT("priority change not allowed"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the 'SNDBUF' socket 'socket' option with IPv4 UDP:
%%
%%               socket:setopt(Sock, socket, sndbuf, integer()).
%%
%%

api_opt_sock_rcvbuf_udp4(suite) ->
    [];
api_opt_sock_rcvbuf_udp4(doc) ->
    [];
api_opt_sock_rcvbuf_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_rcvbuf_udp4,
           fun() -> has_support_sock_rcvbuf() end,
           fun() ->
                   ok = api_opt_sock_buf_udp4(rcvbuf)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the 'RCVBUF' socket 'socket' option with IPv4 UDP:
%%
%%               socket:setopt(Sock, socket, rcvbuf, integer()).
%%
%%

api_opt_sock_sndbuf_udp4(suite) ->
    [];
api_opt_sock_sndbuf_udp4(doc) ->
    [];
api_opt_sock_sndbuf_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_sndbuf_udp4,
           fun() -> has_support_sock_sndbuf() end,
           fun() ->
                   ok = api_opt_sock_buf_udp4(sndbuf)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_sock_buf_udp4(Opt) ->
    Set  = fun(Sock, Value) ->
                   socket:setopt(Sock, socket, Opt, Value)
           end,
    Get  = fun(Sock) ->
                   socket:getopt(Sock, socket, Opt)
           end,
    InitState = #{domain => inet,
                  type   => dgram,
                  proto  => udp,
                  set    => Set,
                  get    => Get},
    ok = api_opt_sock_buf(InitState).


api_opt_sock_buf(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},

         #{desc => "open socket",
           cmd  => fun(#{domain := Domain,
                         type   := Type,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, Type, Proto),
                           {ok, State#{sock => Sock}}
                   end},
         #{desc => "bind",
           cmd  => fun(#{sock := Sock, lsa := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "get current (default) buffer size",
           cmd  => fun(#{sock := Sock, get := Get} = State) ->
                           case Get(Sock) of
                               {ok, Sz} ->
                                   ?SEV_IPRINT("(default) buffer: ~p",
                                               [Sz]),
                                   {ok, State#{default_sz => Sz}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) "
                                               "buffer size:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "change buffer size (default + 1024)",
           cmd  => fun(#{sock       := Sock,
                         default_sz := DefaultSz,
                         set        := Set} = State) ->
                           NewSz = DefaultSz + 1024,
                           ?SEV_IPRINT("try set new buffer size to ~w", [NewSz]),
                           case Set(Sock, NewSz) of
                               ok ->
                                   ?SEV_IPRINT("Buffer size change success", []),
                                   {ok, State#{new_sz => NewSz}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed changing buffer size:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "validate buffer change",
           cmd  => fun(#{sock   := Sock,
                         get    := Get,
                         new_sz := ExpSz} = _State) ->
                           ?SEV_IPRINT("try validate buffer size (~w)", [ExpSz]),
                           case Get(Sock) of
                               {ok, Sz} when (Sz >= ExpSz) ->
                                   ?SEV_IPRINT("buffer size validated:"
                                               "~n   Sz: ~w (~w)", [Sz, ExpSz]),
                                   ok;
                               {ok, Sz} ->
                                   ?SEV_EPRINT("buffer size invalid:"
                                               "~n   Sz:          ~w"
                                               "~n   Expected Sz: ~w", [Sz, ExpSz]),
                                   {error, {invalid_size, Sz, ExpSz}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed get buffer size:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the 'RCVTIMEO' socket 'socket' option with IPv4 UDP:
%%
%%               socket:setopt(Sock, socket, rcvtimeo, #{sec  => integer(),
%%                                                       usec => integer()}).
%%
%% We should really test that the receive behaves as expected,
%% but we don't (we just set the value and read it back...)
%%

api_opt_sock_rcvtimeo_udp4(suite) ->
    [];
api_opt_sock_rcvtimeo_udp4(doc) ->
    [];
api_opt_sock_rcvtimeo_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_rcvtimeo_udp4,
           fun() -> has_support_sock_rcvtimeo() end,
           fun() ->
                   ok = api_opt_sock_timeo_udp4(rcvtimeo)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the 'SNDTIMEO' socket 'socket' option with IPv4 UDP:
%%
%%               socket:setopt(Sock, socket, sndtimeo, integer()).
%%
%%

api_opt_sock_sndtimeo_udp4(suite) ->
    [];
api_opt_sock_sndtimeo_udp4(doc) ->
    [];
api_opt_sock_sndtimeo_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_sndtimeo_udp4,
           fun() -> has_support_sock_sndtimeo() end,
           fun() ->
                   ok = api_opt_sock_timeo_udp4(sndtimeo)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_sock_timeo_udp4(Opt) ->
    Set  = fun(Sock, Value) ->
                   socket:setopt(Sock, socket, Opt, Value)
           end,
    Get  = fun(Sock) ->
                   socket:getopt(Sock, socket, Opt)
           end,
    InitState = #{domain => inet,
                  type   => dgram,
                  proto  => udp,
		  opt   => Opt,
                  set    => Set,
                  get    => Get},
    ok = api_opt_sock_timeo(InitState).


api_opt_sock_timeo(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},

         #{desc => "open socket",
           cmd  => fun(#{domain := Domain,
                         type   := Type,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, Type, Proto),
                           {ok, State#{sock => Sock}}
                   end},
         #{desc => "bind",
           cmd  => fun(#{sock := Sock, lsa := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "get current (default) timeout",
           cmd  => fun(#{sock := Sock, get := Get} = State) ->
                           case Get(Sock) of
                               {ok, #{sec := _, usec := _} = TO} ->
                                   ?SEV_IPRINT("(default) timeout: ~p", [TO]),
                                   {ok, State#{default_timeo => TO}};
                               {error, enoprotoopt = Reason} ->
                                   ?SEV_IPRINT("Failed getting (default) timeout:"
                                               "   ~p", [Reason]),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) timeout:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "change timeout",
           cmd  => fun(#{sock          := Sock,
                         default_timeo := #{sec := DefaultSec} = DefaultTO,
                         set           := Set} = State) ->
                           NewTO = DefaultTO#{sec => DefaultSec + 100},
                           ?SEV_IPRINT("try set new timeout to ~w", [NewTO]),
                           case Set(Sock, NewTO) of
                               ok ->
                                   ?SEV_IPRINT("Timeout change success", []),
                                   {ok, State#{new_timeo => NewTO}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed changing timeout:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "validate timeout change",
           cmd  => fun(#{sock      := Sock,
                         get       := Get,
                         new_timeo := #{sec := ExpSec} = ExpTO} = _State) ->
                           ?SEV_IPRINT("try validate timeout (~w)", [ExpTO]),
                           case Get(Sock) of
                               {ok, ExpTO} ->
                                   ?SEV_IPRINT("timeout (exactly) validated"),
                                   ok;
                               {ok, #{sec := Sec}} when (ExpSec =:= Sec) ->
				   %% For some reason OpenBSD "adjusts" the timeout,
				   %% so that usec does not (allways match)
                                   ?SEV_IPRINT("timeout (approx) validated"),
                                   ok;
                               {ok, TO} ->
                                   ?SEV_EPRINT("timeout invalid:"
                                               "~n   Timeout:          ~w"
                                               "~n   Expected Timeout: ~w",
                                               [TO, ExpTO]),
                                   {error, {invalid_timeo, TO, ExpTO}};
                               {error, edom = Reason} ->
				   %% On OpenBSD (at least) its possible that if the value
				   %% is too far "out of bounds", this will be the result:
				   %%
				   %%     "Numerical argument out of domain"
				   %%
                                   ?SEV_IPRINT("Failed get timeout:"
                                               "   ~p", [Reason]),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed get timeout:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the 'RCVLOWAT' socket 'socket' option with IPv4 UDP:
%%
%%               socket:setopt(Sock, socket, rcvlowat, integer()).
%%
%%

api_opt_sock_rcvlowat_udp4(suite) ->
    [];
api_opt_sock_rcvlowat_udp4(doc) ->
    [];
api_opt_sock_rcvlowat_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_rcvlowat_udp4,
           fun() -> has_support_sock_rcvlowat() end,
           fun() ->
                   ok = api_opt_sock_lowat_udp4(rcvlowat)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the 'SNDLOWAT' socket 'socket' option with IPv4 UDP:
%%
%%               socket:setopt(Sock, socket, sndlowat, integer()).
%%
%% This is (currently) not changeable on linux (among others),
%% so we skip if we get ENOPROTOOPT when attempting a change.
%%

api_opt_sock_sndlowat_udp4(suite) ->
    [];
api_opt_sock_sndlowat_udp4(doc) ->
    [];
api_opt_sock_sndlowat_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_sndlowat_udp4,
           fun() -> has_support_sock_sndlowat() end,
           fun() ->
                   ok = api_opt_sock_lowat_udp4(sndlowat)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_sock_lowat_udp4(Opt) ->
    Set  = fun(Sock, Value) ->
                   socket:setopt(Sock, socket, Opt, Value)
           end,
    Get  = fun(Sock) ->
                   socket:getopt(Sock, socket, Opt)
           end,
    InitState = #{domain => inet,
                  type   => dgram,
                  proto  => udp,
                  set    => Set,
                  get    => Get},
    ok = api_opt_sock_lowat(InitState).


api_opt_sock_lowat(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},

         #{desc => "open socket",
           cmd  => fun(#{domain := Domain,
                         type   := Type,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, Type, Proto),
                           {ok, State#{sock => Sock}}
                   end},
         #{desc => "bind",
           cmd  => fun(#{sock := Sock, lsa := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "get current (default) lowat",
           cmd  => fun(#{sock := Sock, get := Get} = State) ->
                           case Get(Sock) of
                               {ok, LOWAT} ->
                                   ?SEV_IPRINT("(default) lowat: ~p",
                                               [LOWAT]),
                                   {ok, State#{default_lowat => LOWAT}};
                               {error, enoprotoopt = Reason} ->
                                   ?SEV_IPRINT("Failed getting (default) lowat:"
                                               "   ~p", [Reason]),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) lowat:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "change lowat ( + 1 )",
           cmd  => fun(#{sock          := Sock,
                         default_lowat := DefaultLOWAT,
                         set           := Set} = State) ->
                           NewLOWAT = DefaultLOWAT + 1,
                           ?SEV_IPRINT("try set new lowat to ~w", [NewLOWAT]),
                           case Set(Sock, NewLOWAT) of
                               ok ->
                                   ?SEV_IPRINT("LOWAT change success", []),
                                   {ok, State#{new_lowat => NewLOWAT}};
                               {error, enoprotoopt} ->
                                   ?SEV_IPRINT("LOWAT not changeable", []),
                                   {skip, "Not changeable"};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed changing buffer size:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "validate lowat",
           cmd  => fun(#{sock      := Sock,
                         get       := Get,
                         new_lowat := ExpLOWAT} = _State) ->
                           ?SEV_IPRINT("try validate lowat (~w)", [ExpLOWAT]),
                           case Get(Sock) of
                               {ok, ExpLOWAT} ->
                                   ?SEV_IPRINT("lowat validated:"
                                               "~n   LOWAT: ~w", [ExpLOWAT]),
                                   ok;
                               {ok, LOWAT} ->
                                   ?SEV_EPRINT("lowat invalid:"
                                               "~n   LOWAT:          ~w"
                                               "~n   Expected LOWAT: ~w",
                                               [LOWAT, ExpLOWAT]),
                                   {error, {invalid_lowat, LOWAT, ExpLOWAT}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed get lowat:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the timestamp control message header is received when
%% setting the socket 'socket' option true when using sendmsg/recvmsg
%% on an IPv4 UDP (dgram) socket.
%% So, this is done on the receiving side: 
%%
%%               socket:setopt(Sock, socket, timestamp, boolean()).
%%
%% All subsequent *received* messages will be timestamped.
%%

api_opt_sock_timestamp_udp4(suite) ->
    [];
api_opt_sock_timestamp_udp4(doc) ->
    [];
api_opt_sock_timestamp_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_timestamp_udp4,
           fun() -> has_support_sock_timestamp() end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, socket, timestamp, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, socket, timestamp)
                          end,
                   Send = fun(Sock, Data, Dest) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_sock_timestamp_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_sock_timestamp_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := local = Domain} = State) ->
                           LSASrc = which_local_socket_addr(Domain),
                           LSADst = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSASrc,
                                       lsa_dst => LSADst}};
                      (#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},
         #{desc => "get current (default) timestamp for src socket",
           cmd  => fun(#{sock_src := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("src timestamp: ~p", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected src timestamp: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},

         #{desc => "send req (to dst) (WO TIMESTAMP)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},
         #{desc => "send rep (to src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, send := Send}) ->
                           Send(Sock, ?BASIC_REP, Src)
                   end},
         #{desc => "recv rep (from dst)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Dst, [], ?BASIC_REP}} ->
                                   ok;
                               {ok, UnexpData} ->
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},


         #{desc => "enable timestamp on dst socket",
           cmd  => fun(#{sock_dst := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("dst timestamp enabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},


         #{desc => "send req 1 (to dst) (W TIMESTAMP)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := socket,
                                             type  := timestamp,
                                             value := TS}], ?BASIC_REQ}} ->
                                   ?SEV_IPRINT("received req *with* "
                                               "expected timestamp: "
                                               "~n   ~p", [TS]),
                                   ok;
                               {ok, {Src, CMsgs, ?BASIC_REQ}} ->
                                   ?SEV_EPRINT("Unexpected control message(s):"
                                               "~n   CMsgs: ~p",
                                               [CMsgs]),
                                   {error, {unexpected_cmsgs, CMsgs}};
                               {ok, UnexpData} ->
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},
         #{desc => "send rep (to src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, send := Send}) ->
                           Send(Sock, ?BASIC_REP, Src)
                   end},
         #{desc => "recv rep (from dst)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Dst, [], ?BASIC_REP}} ->
                                   ok;
                               {ok, UnexpData} ->
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},


         #{desc => "send req 2 (to dst) (W TIMESTAMP)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := socket,
                                             type  := timestamp,
                                             value := TS}], ?BASIC_REQ}} ->
                                   ?SEV_IPRINT("received req *with* "
                                               "expected timestamp: "
                                               "~n   ~p", [TS]),
                                   ok;
                               {ok, UnexpData} ->
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},
         #{desc => "send rep (to src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, send := Send}) ->
                           Send(Sock, ?BASIC_REP, Src)
                   end},
         #{desc => "recv rep (from dst)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Dst, [], ?BASIC_REP}} ->
                                   ok;
                               {ok, UnexpData} ->
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},


         #{desc => "disable timestamps on dst socket",
           cmd  => fun(#{sock_dst := Sock, set := Set} = _State) ->
                           case Set(Sock, false) of
                               ok ->
                                   ?SEV_IPRINT("dst timestamp disabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},


         #{desc => "send req (to dst) (WO TIMESTAMP)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ?SEV_IPRINT("received req *without* timestamp"),
                                   ok;
                               {ok, UnexpData} ->
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},
         #{desc => "send rep (to src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, send := Send}) ->
                           Send(Sock, ?BASIC_REP, Src)
                   end},
         #{desc => "recv rep (from dst)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Dst, [], ?BASIC_REP}} ->
                                   ok;
                               {ok, UnexpData} ->
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},


         #{desc => "close src socket",
           cmd  => fun(#{domain   := local,
                         sock_src := Sock,
                         lsa_src  := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() -> maps:remove(lsa_src, State) end,
                                           fun() -> State end),
                           {ok, maps:remove(sock_src, State1)};
                      (#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{domain   := local,
                         sock_dst := Sock,
                         lsa_dst  := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() -> maps:remove(lsa_dst, State) end,
                                           fun() -> State end),
                           {ok, maps:remove(sock_dst, State1)};
                      (#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the timestamp control message header is received when
%% setting the socket 'socket' option true when using sendmsg/recvmsg
%% on an IPv4 TCP (stream) socket.
%% So, this is done on the receiving side: 
%%
%%               socket:setopt(Sock, socket, timestamp, boolean()).
%%
%% All subsequent *received* messages will be timestamped.
%%
%% There is no mention of this not working for TCP in the man page
%% on a SLES 11 SP4 machine (=> 3.0.101-108.87), but it does not
%% (we don't get a timestamp control message header when its enabled).
%% It also does not work on SLES 12 SP2 (=> 4.4.120-92.70), 
%% so we start by skipping from that version (4.4.120) or older!
%% Don't actually know if its the distro or the (kernel) version...
%%

api_opt_sock_timestamp_tcp4(suite) ->
    [];
api_opt_sock_timestamp_tcp4(doc) ->
    [];
api_opt_sock_timestamp_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_sock_timestamp_tcp4,
           fun() ->
                   has_support_sock_timestamp(),
                   is_good_enough_linux({4,4,120}),
                   is_not_freebsd(),
                   is_not_openbsd(),
                   is_not_netbsd(),
                   is_not_darwin()
           end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, socket, timestamp, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, socket, timestamp)
                          end,
                   Send = fun(Sock, Data) ->
                                  Msg = #{iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 proto  => tcp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_sock_timestamp_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_sock_timestamp_tcp(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create listen socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{lsock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{domain := local,
                         lsock  := LSock,
                         lsa    := LSA} = _State) ->
                           case socket:bind(LSock, LSA) of
                               ok ->
                                   ok; % We do not care about the port for local
                               {error, _} = ERROR ->
                                   ERROR
                           end;
                      (#{lsock := LSock, lsa := LSA} = State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   ?SEV_IPRINT("bound to port: ~w", [Port]),
                                   {ok, State#{lport => Port}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make listen socket",
           cmd  => fun(#{lsock := LSock}) ->
                           socket:listen(LSock)
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{domain := local,
                         tester := Tester, lsa := #{path := Path}}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init, Path),
                           ok;
                      (#{tester := Tester, lport := Port}) ->
                           %% This is actually not used for unix domain socket
                           ?SEV_ANNOUNCE_READY(Tester, init, Port),
                           ok
                   end},


         %% The actual test
         #{desc => "await continue (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, accept)
                   end},
         #{desc => "await connection",
           cmd  => fun(#{lsock := LSock} = State) ->
                           case socket:accept(LSock) of
                               {ok, Sock} ->
                                   ?SEV_IPRINT("accepted: ~n   ~p", [Sock]),
                                   {ok, State#{csock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (accept)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, accept),
                           ok
                   end},

         %% *** First message ***

         #{desc => "await (recv) request 1",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {[], ?BASIC_REQ}} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready (send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           ok
                   end},

         %% *** Second message ***

         #{desc => "await (recv) request 2",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {[], ?BASIC_REQ}} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready (send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           ok
                   end},

         %% *** Third message ***

         #{desc => "await (recv) request 3",
           cmd  => fun(#{csock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {[], ?BASIC_REQ}} ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (recv request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_req),
                           ok
                   end},
         #{desc => "await continue (with send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_reply)
                   end},
         #{desc => "send reply",
           cmd  => fun(#{csock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REP)
                   end},
         #{desc => "announce ready (send reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_reply),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close connection socket",
           cmd  => fun(#{csock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(csock, State)}
                   end},
         #{desc => "close listen socket",
           cmd  => fun(#{domain   := local,
                         lsock    := Sock,
                         local_sa := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() ->
                                                   maps:remove(local_sa, State)
                                           end,
                                           fun() -> State end),
                           {ok, maps:remove(lsock, State1)};
                      (#{lsock := LSock} = State) ->
                           case socket:close(LSock) of
                               ok ->
                                   {ok, maps:remove(lsock, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    ClientSeq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(#{domain := local} = State) ->
                           {Tester, Path} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_path => Path}};
                      (State) ->
                           {Tester, Port} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, server_port => Port}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** The init part ***
         #{desc => "which server (local) address",
           cmd  => fun(#{domain      := local = Domain,
                         server_path := Path} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = #{family => Domain, path => Path},
                           {ok, State#{local_sa => LSA, server_sa => SSA}};
                      (#{domain := Domain, server_port := Port} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           SSA = LSA#{port => Port},
                           {ok, State#{local_sa => LSA, server_sa => SSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, local_sa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "await continue (connect)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, connect)
                   end},
         #{desc => "connect to server",
           cmd  => fun(#{sock := Sock, server_sa := SSA}) ->
                           socket:connect(Sock, SSA)
                   end},
         #{desc => "announce ready (connect)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, connect),
                           ok
                   end},

         %% *** First message (default=wo timestamp) ***

         #{desc => "await continue (verify timestamp off)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, verify_timestamp)
                   end},
         #{desc => "verify timestamp off",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = _Value} ->
                                   ?SEV_IPRINT("timestamp: ~p", [_Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected timestamp: ~p",
                                               [Unexpected]),
                                   {error, {unexpected_timestamp, Unexpected}};
                               {error, enoprotoopt = Reason} ->
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end                                   
                   end},
         #{desc => "announce ready (timestamp off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, timestamp_off),
                           ok
                   end},

         #{desc => "await continue (send request)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 1 (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send request)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 1 (from server, wo timestamp)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {[], ?BASIC_REP}} ->
                                   ok;
                               {ok, {[], UnexpData}} ->
                                   {error, {unexpected_reply_data, UnexpData}};
                               {ok, {BadCMsgs, ?BASIC_REP}} ->
                                   {error, {unexpected_reply_cmsgs,
                                            BadCMsgs}};
                               {ok, BadReply} ->
                                   {error, {unexpected_reply, BadReply}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 1 (recv reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         %% *** Second message (w timestamp) ***

         #{desc => "await continue (enable timestamp)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, enable_timestamp)
                   end},
         #{desc => "enable timestamp",
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("timestamp enabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed enable timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end                                   
                   end},
         %% Linux pecularity observed here...
         %% Detected on Kernel 4.15.0-72 x96_64.
         %% The option set to enable receiving timestamps just above
         %% has failed to be effective down in "await recv reply 2
         %% (from server, w timestamp)" below, unless we put the
         %% sleep between setting the option and informing
         %% the writer that it shall write to the other socket end.
         %% A sleep 1 ms improves a lot but does not remove
         %% problem completely. Believe it or not.
         ?SEV_SLEEP(100),
         #{desc => "announce ready (timestamp on)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, timestamp_on),
                           ok
                   end},

         #{desc => "await continue (send request 2)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 2 (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send request 2)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 2 (from server, w timestamp)",
           cmd  => fun(#{sock := Sock, recv := Recv, get := Get}) ->
                           case Recv(Sock) of
                               {ok, {[#{level := socket,
                                        type  := timestamp,
                                        value := TS}], ?BASIC_REP}} ->
                                   ?SEV_IPRINT("received reply *with* "
                                               "expected timestamp: "
                                               "~n   ~p", [TS]),
                                   ok;
                               {ok, {BadCMsgs, ?BASIC_REP}} ->
                                   ?SEV_EPRINT("received reply *with* "
                                               "unexpected cmsg headers:"
                                               "~n   ~p"
                                               "Current timestamp value: "
                                               "~n   ~p",
                                               [BadCMsgs, Get(Sock)]),
                                   {error, {unexpected_reply_cmsgs,
                                            BadCMsgs}};
                               {ok, BadReply} ->
                                   {error, {unexpected_reply, BadReply}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 2 (recv reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         %% *** Third message (wo timestamp) ***

         #{desc => "await continue (disable timestamp)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, disable_timestamp)
                   end},
         #{desc => "disable timestamp",
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           case Set(Sock, false) of
                               ok ->
                                   ?SEV_IPRINT("timestamp disabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed disable timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end                                   
                   end},
         #{desc => "announce ready (timestamp off)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, timestamp_off),
                           ok
                   end},

         #{desc => "await continue (send request 3)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, send_req)
                   end},
         #{desc => "send request 3 (to server)",
           cmd  => fun(#{sock := Sock, send := Send}) ->
                           Send(Sock, ?BASIC_REQ)
                   end},
         #{desc => "announce ready (send request 3)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, send_req),
                           ok
                   end},
         #{desc => "await recv reply 3 (from server, wo timestamp)",
           cmd  => fun(#{sock := Sock, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {[], ?BASIC_REP}} ->
                                   ?SEV_IPRINT("received reply *without* "
                                               "timestamp"),
                                   ok;
                               {ok, {BadCMsgs, ?BASIC_REP}} ->
                                   {error, {unexpected_reply_cmsgs,
                                            BadCMsgs}};
                               {ok, {[], BadData}} ->
                                   {error, {unexpected_reply_data,
                                            BadData}};
                               {ok, BadReply} ->
                                   {error, {unexpected_reply, BadReply}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready 3 (recv reply)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, recv_reply),
                           ok
                   end},

         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{domain   := local,
                         sock     := Sock,
                         local_sa := #{path := Path}} = State) ->
                           ok = socket:close(Sock),
                           State1 =
                               unlink_path(Path,
                                           fun() ->
                                                   maps:remove(local_sa, State)
                                           end,
                                           fun() -> State end),
                           {ok, maps:remove(sock, State1)};
                      (#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},
         #{desc => "monitor client",
           cmd  => fun(#{client := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = State) ->
                           {ok, Port} = ?SEV_AWAIT_READY(Pid, server, init),
                           {ok, State#{server_port => Port}}
                   end},

         %% Start the client
         #{desc => "order client start",
           cmd  => fun(#{client := Pid, server_port := Port} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, Port),
                           ok
                   end},
         #{desc => "await client ready (init)",
           cmd  => fun(#{client := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, client, init)
                   end},

         %% *** The actual test ***
         #{desc => "order server to continue (with accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, accept),
                           ok
                   end},
%%%         ?SEV_SLEEP(?SECS(1)),
         #{desc => "order client to continue (with connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, connect),
                           ok
                   end},
         #{desc => "await client ready (connect)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, connect)
                   end},
         #{desc => "await server ready (accept)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, accept)
                   end},

         %% *** First message (default=wo timestamp) ***

         #{desc => "order client to continue (with verify timestamp off)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, verify_timestamp),
                           ok
                   end},
         #{desc => "await client ready (timestamp off)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, timestamp_off)
                   end},

         #{desc => "order client to continue (with send request 1)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client ready (with send request 1)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send_req)
                   end},
         #{desc => "await server ready (request recv 1)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply sent)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client ready (reply recv)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, recv_reply)
                   end},

         %% Second message (w timestamp)

         #{desc => "order client to continue (with enable timestamp)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, enable_timestamp),
                           ok
                   end},
         #{desc => "await client ready (timestamp on)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, timestamp_on)
                   end},

         #{desc => "order client to continue (with send request 2)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client ready (with send request 2)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send_req)
                   end},
         #{desc => "await server ready (request recv 2)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply 2)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply sent 2)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client ready (reply recv 2)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, recv_reply)
                   end},

         %% Third message (wo timestamp)

         #{desc => "order client to continue (with disable timestamp)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, disable_timestamp),
                           ok
                   end},
         #{desc => "await client ready (timestamp off)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, timestamp_off)
                   end},

         #{desc => "order client to continue (with send request 3)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Client, send_req),
                           ok
                   end},
         #{desc => "await client ready (with send request 3)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, send_req)
                   end},
         #{desc => "await server ready (request recv 3)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, recv_req)
                   end},
         #{desc => "order server to continue (with send reply 3)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, send_reply),
                           ok
                   end},
         #{desc => "await server ready (with reply sent 3)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, send_reply)
                   end},
         #{desc => "await client ready (reply recv 3)",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_AWAIT_READY(Client, client, recv_reply)
                   end},


         %% *** Termination ***
         #{desc => "order client to terminate",
           cmd  => fun(#{client := Client} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Client),
                           ok
                   end},
         #{desc => "await client termination",
           cmd  => fun(#{client := Client} = State) ->
                           ?SEV_AWAIT_TERMINATION(Client),
                           State1 = maps:remove(client, State),
                           {ok, State1}
                   end},
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start client evaluator"),
    Client = ?SEV_START("client", ClientSeq, InitState),

    i("start tester evaluator"),
    TesterInitState = #{server => Server#ev.pid,
                        client => Client#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    ok = ?SEV_AWAIT_FINISH([Server, Client, Tester]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the add_mambership and drop_membership ip options work.
%% We create one server and two clients. The server only send messages,
%% the clients only receives messages.
%% An UDP datagram is forbidden (RFC 1122) from having a source address
%% that is a multicast address (or a broadcast address).
%% So, the server create a socket "for sending" and the clients sockets
%% "for receiving".
%% Sending socket:   Bound to the local address (and any port).
%%                   When sending, the dest will be the multicast address
%%                   and port of the receiving socket.
%% Receiving socket: Bound to the multicast address and port.
api_opt_ip_add_drop_membership(suite) ->
    [];
api_opt_ip_add_drop_membership(doc) ->
    ["OTP-15908 (ERL-980)"];
api_opt_ip_add_drop_membership(_Config) when is_list(_Config) ->
    ?TT(?SECS(30)),
    tc_try(api_opt_ip_add_drop_membership,
           fun() ->
                   has_support_ip_add_membership(),
                   has_support_ip_drop_membership(),
                   has_support_ip_multicast()
           end,
           fun() -> api_opt_ip_add_drop_membership() end).


api_opt_ip_add_drop_membership() ->
    Set = fun(S, Key, Val) ->
                  socket:setopt(S, ip, Key, Val)
          end,
    AddMembership  = fun(S, Val) -> Set(S, add_membership,  Val) end,
    DropMembership = fun(S, Val) -> Set(S, drop_membership, Val) end,

    ServerSeq =
        [
         %% *** Wait for start order part ***
         #{desc => "await start",
           cmd  => fun(State) ->
                           {Tester, MSA} = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester, msa => MSA}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester} = _State) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{local_sa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "make recv socket reuse addr",
           cmd  => fun(#{sock := Sock} = _State) ->
                           case socket:setopt(Sock, socket, reuseaddr, true) of
                               ok ->
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed set reuseaddr: "
                                               "~n   ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "bind recv socket to multicast address",
           cmd  => fun(#{sock := Sock, msa := MSA} = State) ->
                           case sock_bind(Sock, MSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   ?SEV_IPRINT("bound to:"
                                               "~n   ~p", [Port]),
                                   {ok, State#{msa => MSA#{port => Port}}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},

         %% The actual test
         #{desc => "await continue (add_membership)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, add_membership)
                   end},
         #{desc => "add membership",
           cmd  => fun(#{sock     := Sock,
                         msa      := #{addr := MAddr},
                         local_sa := #{addr := Addr}} = State) ->
                           MReq = #{multiaddr => MAddr,
                                    interface => Addr},
                           ?SEV_IPRINT("try add membership to:"
                                       "~n   ~p", [MReq]),
                           case AddMembership(Sock, MReq) of
                               ok ->
                                   ?SEV_IPRINT("membership added"),
                                   {ok, State#{mreq => MReq}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed adding membership to: "
                                               "~n   ~p"
                                               "~n   Reason:  ~p",
                                               [MReq, Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (add-membership)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, add_membership),
                           ok
                   end},

         #{desc => "await continue (drop_membership)",
           cmd  => fun(#{tester := Tester} = _State) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, drop_membership)
                   end},
         #{desc => "drop membership",
           cmd  => fun(#{sock := Sock,
                         mreq := MReq} = State) ->
                           ?SEV_IPRINT("try drop membership from:"
                                       "~n   ~p", [MReq]),
                           case DropMembership(Sock, MReq) of
                               ok ->
                                   ?SEV_IPRINT("membership dropped"),
                                   {ok, maps:remove(mreq, State)};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed drop membership from: "
                                               "~n   ~p"
                                               "~n   Reason:  ~p",
                                               [MReq, Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (drop-membership)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, drop_membership),
                           ok
                   end},


         %% Termination
         #{desc => "await terminate (from tester)",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Server} = _State) ->
                           _MRef = erlang:monitor(process, Server),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid, msa := MSA} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid, MSA),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = _State) ->
                           case ?SEV_AWAIT_READY(Pid, server, init) of
                               ok ->
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Start of server failed: "
                                               "~n   ~p", [Reason]),
                                   ERROR
                           end
                   end},


         %% *** The actual test ***
         #{desc => "order server to continue (add-membership)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, add_membership),
                           ok
                   end},
         #{desc => "await server ready (add-membership)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, add_membership)
                   end},

         #{desc => "order server to continue (drop-membership)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, drop_membership),
                           ok
                   end},
         #{desc => "await server ready (drop-membership)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, drop_membership)
                   end},

         ?SEV_SLEEP(?SECS(1)),

         %% *** Termination ***
         #{desc => "order server terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           {ok, maps:remove(server, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    Domain = inet,
    i("get multicast address"),
    MAddr  = which_ip_multicast_address(),
    MSA    = #{family => Domain, addr => MAddr},

    i("start server evaluator"),
    ServerInitState = #{domain => Domain},
    Server = ?SEV_START("server", ServerSeq, ServerInitState),

    i("start tester evaluator"),
    TesterInitState = #{domain  => Domain,
                        msa     => MSA,
                        server  => Server#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    i("await evaluator(s)"),
    ok = ?SEV_AWAIT_FINISH([Tester, Server]).



which_ip_multicast_address() ->
    which_multicast_address(inet).

which_multicast_address(Domain) ->
    case os:type() of
        {unix, linux} ->
            WhichMAddr = fun([_, _, MAddr]) -> MAddr end,
            which_multicast_address2(Domain, WhichMAddr);

        {unix, sunos} ->
            WhichMAddr = fun([_, MAddr, _]) -> MAddr end,
            which_multicast_address2(Domain, WhichMAddr);

        Type ->
            %% Actually, what is "not supported". is netstat!
            not_supported({multicast, Type})
    end.

%% Note that the 'netstat -g' table looks different on linux and SunOS
%% Linux: IfName - RefCnt - Group
%% SunOS: IfName - Group  - RefCnt

which_multicast_address2(Domain, WhichMAddr) ->
    IfName = which_local_host_ifname(Domain),
    %% On some platforms the netstat barfs out some crap on stderr
    %% before the actual info...
    case os:cmd("netstat -g 2>/dev/null | grep " ++ IfName) of
        [] ->
            %% Can't figure out if we support multicast or not...
            not_supported(no_netstat);
        NetstatGroupsStr ->
            try
                begin
                    NetstatGroups0   = string:tokens(NetstatGroupsStr, [$\n]),
                    NetstatGroups    = [string:tokens(G, [$ ]) || 
                                           G <- NetstatGroups0],
                    MAddrs           = [WhichMAddr(NetstatGroup) || 
                                           NetstatGroup <- NetstatGroups],
                    which_multicast_address3(Domain, MAddrs)
                end
            catch
                throw:E:_ ->
                    throw(E);
                C:E:S ->
                    not_supported({multicast, {C,E,S}})
            end
    end.

which_multicast_address3(_Domain, []) ->
    not_supported({multicast, no_valid_addrs});
which_multicast_address3(Domain, [MAddrStr|MAddrs]) ->
    %% Even on linux some of these are not actually addresses, but
    %% "host names", such as all-systems.mcast.net. But both
    %% address strings, such as "224.0.0.251" and host name strings
    %% gets translated into an address by the inet:inet:getaddr/2.
    case inet:getaddr(MAddrStr, Domain) of
        {ok, MAddr} ->
            MAddr;
        {error, _} ->
            which_multicast_address3(Domain, MAddrs)
    end.
    
which_local_host_ifname(Domain) ->
    case ?LIB:which_local_host_info(Domain) of
        {ok, #{name := Name}} ->
            Name;
        {error, Reason} ->
            not_supported({multicast, Reason})
    end.

    

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the pktinfo control message header is received when
%% setting the socket 'ip' option pktinfo is set to true when using
%% sendmsg/recvmsg on an IPv4 UDP (dgram) socket.
%% So, this is done on the receiving side: 
%%
%%               socket:setopt(Sock, ip, pktinfo, boolean()).
%%
%% For all subsequent *received* messages, the pktinfo control message
%% header will be with the message.
%%
%% Note that it *should* be possible to explicitly send pktinfo also,
%% but this have not yet been implemented (in socket), so that part
%% we do not test!!
%%

api_opt_ip_pktinfo_udp4(suite) ->
    [];
api_opt_ip_pktinfo_udp4(doc) ->
    [];
api_opt_ip_pktinfo_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ip_pktinfo_udp4,
           fun() -> has_support_ip_pktinfo() end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, ip, pktinfo, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, ip, pktinfo)
                          end,
                   Send = fun(Sock, Data, Dest, default) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg);
                             (Sock, Data, Dest, Info) ->
                                  %% We do not support this at the moment!!!
                                  CMsg = #{level => ip,
                                              type  => pktinfo,
                                              data  => Info},
                                  Msg  = #{addr => Dest,
                                              ctrl => [CMsg],
                                              iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_ip_pktinfo_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ip_pktinfo_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := local = Domain} = State) ->
                           LSASrc = which_local_socket_addr(Domain),
                           LSADst = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSASrc,
                                       lsa_dst => LSADst}};
                      (#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},
         #{desc => "default pktinfo for dst socket",
           cmd  => fun(#{sock_dst := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("dst recvttl: ~p", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected src recvtos: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (wo (explicit) pktinfo)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},


         %% *** We do not *yet* support sending pktinfo ***

         %% #{desc => "send req (to dst) (w explicit pktinfo)",
         %%   cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
         %%                   Send(Sock, ?BASIC_REQ, Dst, PktInfo)
         %%           end},
         %% #{desc => "recv req (from src) - wo pktinfo",
         %%   cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
         %%                   case Recv(Sock) of
         %%                       {ok, {Src, [], ?BASIC_REQ}} ->
         %%                           ok;
         %%                       {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
         %%                           ?SEV_EPRINT("Unexpected msg: "
         %%                                       "~n   Expect Source: ~p"
         %%                                       "~n   Recv Source:   ~p"
         %%                                       "~n   Expect CHdrs:  ~p"
         %%                                       "~n   Recv CHdrs:    ~p"
         %%                                       "~n   Expect Msg:    ~p"
         %%                                       "~n   Recv Msg:      ~p",
         %%                                       [Src, BadSrc,
         %%                                        [], BadCHdrs,
         %%                                        ?BASIC_REQ, BadReq]),
         %%                           {error, {unexpected_data, UnexpData}};
         %%                       {ok, UnexpData} ->
         %%                           ?SEV_EPRINT("Unexpected msg: "
         %%                                       "~n   Expect Source: ~p"
         %%                                       "~n   Expect CHdrs:  ~p"
         %%                                       "~n   Expect Msg:    ~p"
         %%                                       "~n   Unexp Data:    ~p",
         %%                                       [Src, [], ?BASIC_REQ, UnexpData]),
         %%                           {error, {unexpected_data, UnexpData}};
         %%                       {error, _} = ERROR ->
         %%                           %% At the moment there is no way to get
         %%                           %% status or state for the socket...
         %%                           ERROR
         %%                   end
         %%           end},

         #{desc => "enable pktinfo on dst socket",
           cmd  => fun(#{sock_dst := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("dst pktinfo enabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting pktinfo:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (wo explicit pktinfo)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src) - w default pktinfo",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := ip,
                                             type  := pktinfo,
                                             value := #{addr     := Addr,
                                                        ifindex  := IfIdx,
                                                        spec_dst := SpecDst}}],
                                     ?BASIC_REQ}} ->
                                   ?SEV_IPRINT("Got (default) Pkt Info: "
                                               "~n   Addr:      ~p"
                                               "~n   If Index:  ~p"
                                               "~n   Spec Dst:  ~p",
                                               [Addr, IfIdx, SpecDst]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},


         %% *** We do not *yet* support sending pktinfo ***

         %% #{desc => "send req (to dst) (w explicit pktinfo)",
         %%   cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
         %%                   Send(Sock, ?BASIC_REQ, Dst, PktInfo)
         %%           end},
         %% #{desc => "recv req (from src) - w ttl = 100",
         %%   cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
         %%                   case Recv(Sock) of
         %%                       {ok, {Src, [#{level := ip,
         %%                                     type  := ttl,
         %%                                     data  := PktInfo}], ?BASIC_REQ}} ->
         %%                           ?SEV_IPRINT("Got Pkt Info: "
         %%                                       "~n   ~p", [Info]),
         %%                           ok;
         %%                       {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
         %%                           ?SEV_EPRINT("Unexpected msg: "
         %%                                       "~n   Expect Source: ~p"
         %%                                       "~n   Recv Source:   ~p"
         %%                                       "~n   Expect CHdrs:  ~p"
         %%                                       "~n   Recv CHdrs:    ~p"
         %%                                       "~n   Expect Msg:    ~p"
         %%                                       "~n   Recv Msg:      ~p",
         %%                                       [Src, BadSrc,
         %%                                        [], BadCHdrs,
         %%                                        ?BASIC_REQ, BadReq]),
         %%                           {error, {unexpected_data, UnexpData}};
         %%                       {ok, UnexpData} ->
         %%                           ?SEV_EPRINT("Unexpected msg: "
         %%                                       "~n   Expect Source: ~p"
         %%                                       "~n   Expect CHdrs:  ~p"
         %%                                       "~n   Expect Msg:    ~p"
         %%                                       "~n   Unexp Data:    ~p",
         %%                                       [Src, [], ?BASIC_REQ, UnexpData]),
         %%                           {error, {unexpected_data, UnexpData}};
         %%                       {error, _} = ERROR ->
         %%                           %% At the moment there is no way to get
         %%                           %% status or state for the socket...
         %%                           ERROR
         %%                   end
         %%           end},

         #{desc => "close src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the options control message header is received when
%% setting the socket 'ip' option recvopts is set to true when using
%% sendmsg/recvmsg on an IPv4 UDP (dgram) socket.
%% So, this is done on the receiving side: 
%%
%%               socket:setopt(Sock, ip, recvopts, boolean()).
%%
%% For all subsequent *received* messages, the options control message
%% header will be with the message.
%%
%% Note that it *should* be possible to explicitly send options also,
%% but this have not yet been implemented (in socket), so that part
%% we do not test!!
%%
%%
%% <NOTE>
%%
%% This test does not currently work. The recvopts is supposed to
%% result in a IP_OPTIONS control message header but does not!
%% So, exactly how we are suppose to use this option is unknown.
%% So, let the test code remain, but skip until we have figured out
%% how to test this.
%%
%% </NOTE>
%%

api_opt_ip_recvopts_udp4(suite) ->
    [];
api_opt_ip_recvopts_udp4(doc) ->
    [];
api_opt_ip_recvopts_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ip_recvopts_udp4,
           fun() ->
                   has_support_ip_recvopts(),
                   %% We also use the recvtos and timestamp options
                   %% in this test, so at least one of them must
                   %% be supported
                   has_support_ip_recvtos_and_or_sock_timestamp(),
                   not_yet_implemented()

           end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, ip, recvopts, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, ip, recvopts)
                          end,
                   Send = fun(Sock, Data, Dest, default) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg);
                             (Sock, Data, Dest, Info) ->
                                  %% We do not support this at the moment!!!
                                  CMsg = #{level => ip,
                                              type  => options,
                                              data  => Info},
                                  Msg  = #{addr => Dest,
                                              ctrl => [CMsg],
                                              iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_ip_recvopts_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ip_recvopts_udp(InitState) ->
    Seq = 
        [
         %% Start by figure out which of the ip:recvtos and/or socket:timestamp
         %% options we can use.
         #{desc => "test for ip:recvtos",
           cmd  => fun(State) ->
                           ?SEV_IPRINT("test for ip:recvtos"),
                           case socket:is_supported(options, ip, recvtos) of
                               true ->
                                   ?SEV_IPRINT("use ip:recvtos"),
                                   {ok, State#{recvtos => true}};
                               false -> 
                                   ?SEV_IPRINT("do *not* use ip:recvtos"),
                                   {ok, State#{recvtos => false}}
                           end
                   end},
         #{desc => "test for socket:timestamp",
           cmd  => fun(State) ->
                           ?SEV_IPRINT("test for socket:timestamp"),
                           case socket:is_supported(options, socket, timestamp) of
                               true ->
                                   ?SEV_IPRINT("use socket:timestamp"),
                                   {ok, State#{timestamp => true}};
                               false -> 
                                   ?SEV_IPRINT("do *not* use socket:timestamp"),
                                   {ok, State#{timestamp => false}}
                           end
                   end},

         #{desc => "local address",
           cmd  => fun(#{domain := local = Domain} = State) ->
                           LSASrc = which_local_socket_addr(Domain),
                           LSADst = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSASrc,
                                       lsa_dst => LSADst}};
                      (#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},
         #{desc => "default recvopts for dst socket",
           cmd  => fun(#{sock_dst := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("dst recvopts: ~p", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected src recvtos: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (wo (explicit) options)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},


         %% *** We do not *yet* support sending options ***

         %% #{desc => "send req (to dst) (w explicit options)",
         %%   cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
         %%                   Send(Sock, ?BASIC_REQ, Dst, Opts)
         %%           end},
         %% #{desc => "recv req (from src) - wo options",
         %%   cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
         %%                   case Recv(Sock) of
         %%                       {ok, {Src, [], ?BASIC_REQ}} ->
         %%                           ok;
         %%                       {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
         %%                           ?SEV_EPRINT("Unexpected msg: "
         %%                                       "~n   Expect Source: ~p"
         %%                                       "~n   Recv Source:   ~p"
         %%                                       "~n   Expect CHdrs:  ~p"
         %%                                       "~n   Recv CHdrs:    ~p"
         %%                                       "~n   Expect Msg:    ~p"
         %%                                       "~n   Recv Msg:      ~p",
         %%                                       [Src, BadSrc,
         %%                                        [], BadCHdrs,
         %%                                        ?BASIC_REQ, BadReq]),
         %%                           {error, {unexpected_data, UnexpData}};
         %%                       {ok, UnexpData} ->
         %%                           ?SEV_EPRINT("Unexpected msg: "
         %%                                       "~n   Expect Source: ~p"
         %%                                       "~n   Expect CHdrs:  ~p"
         %%                                       "~n   Expect Msg:    ~p"
         %%                                       "~n   Unexp Data:    ~p",
         %%                                       [Src, [], ?BASIC_REQ, UnexpData]),
         %%                           {error, {unexpected_data, UnexpData}};
         %%                       {error, _} = ERROR ->
         %%                           %% At the moment there is no way to get
         %%                           %% status or state for the socket...
         %%                           ERROR
         %%                   end
         %%           end},

         #{desc => "enable recvopts on dst socket",
           cmd  => fun(#{sock_dst := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("dst recvopts enabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting recvopts:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         %% This specific option, recvtos, is tested in another test case
         %% Note that this may not actually be supported here!!
         #{desc => "maybe enable ip:recvtos on dst socket",
           cmd  => fun(#{recvtos := true, sock_dst := Sock} = _State) ->
                           ?SEV_IPRINT("enable ip:recvtos"),
                           ok = socket:setopt(Sock, ip, recvtos, true);
                      (#{recvtos := false} = _State) ->
                           ok
                   end},
         %% This specific option, timestamp, is tested in another test case
         #{desc => "maybe enable socket:timestamp on dst socket",
           cmd  => fun(#{timestamp := true, sock_dst := Sock} = _State) ->
                           ?SEV_IPRINT("enable socket:timestamp"),
                           ok = socket:setopt(Sock, socket, timestamp, true);
                      (#{timestamp := false} = _State) ->
                           ok
                   end},

         #{desc => "send req (to dst) (wo explicit options)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src) - w default options",
           cmd  => fun(#{recvtos := true, timestamp := true,
                         sock_dst := Sock,
                         sa_src   := Src,
                         recv     := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, Opts, ?BASIC_REQ}} 
                                 when (length(Opts) =:= 2) ->
                                   ?SEV_IPRINT("Got (default) Options: "
                                               "~n   Opts:  ~p", [Opts]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end;
                      (#{timestamp := true,
                         sock_dst := Sock,
                         sa_src   := Src,
                         recv     := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, Opts, ?BASIC_REQ}} 
                                 when (length(Opts) =:= 1) ->
                                   ?SEV_IPRINT("Got (default) Options: "
                                               "~n   Opts:  ~p", [Opts]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end;
                      (#{recvtos := true,
                         sock_dst := Sock,
                         sa_src   := Src,
                         recv     := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, Opts, ?BASIC_REQ}} 
                                 when (length(Opts) =:= 1) ->
                                   ?SEV_IPRINT("Got (default) Options: "
                                               "~n   Opts:  ~p", [Opts]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},


         %% *** We do not *yet* support sending options ***

         %% #{desc => "send req (to dst) (w explicit options)",
         %%   cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
         %%                   Send(Sock, ?BASIC_REQ, Dst, Opts)
         %%           end},
         %% #{desc => "recv req (from src) - w options",
         %%   cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
         %%                   case Recv(Sock) of
         %%                       {ok, {Src, Opts, ?BASIC_REQ}} ->
         %%                           ?SEV_IPRINT("Got Options: "
         %%                                       "~n   ~p", [Opts]),
         %%                           ok;
         %%                       {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
         %%                           ?SEV_EPRINT("Unexpected msg: "
         %%                                       "~n   Expect Source: ~p"
         %%                                       "~n   Recv Source:   ~p"
         %%                                       "~n   Expect CHdrs:  ~p"
         %%                                       "~n   Recv CHdrs:    ~p"
         %%                                       "~n   Expect Msg:    ~p"
         %%                                       "~n   Recv Msg:      ~p",
         %%                                       [Src, BadSrc,
         %%                                        [], BadCHdrs,
         %%                                        ?BASIC_REQ, BadReq]),
         %%                           {error, {unexpected_data, UnexpData}};
         %%                       {ok, UnexpData} ->
         %%                           ?SEV_EPRINT("Unexpected msg: "
         %%                                       "~n   Expect Source: ~p"
         %%                                       "~n   Expect CHdrs:  ~p"
         %%                                       "~n   Expect Msg:    ~p"
         %%                                       "~n   Unexp Data:    ~p",
         %%                                       [Src, [], ?BASIC_REQ, UnexpData]),
         %%                           {error, {unexpected_data, UnexpData}};
         %%                       {error, _} = ERROR ->
         %%                           %% At the moment there is no way to get
         %%                           %% status or state for the socket...
         %%                           ERROR
         %%                   end
         %%           end},

         #{desc => "close src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the origdstaddr control message header is received when
%% setting the socket 'ip' option recvorigdstaddr is set to true when
%% using sendmsg/recvmsg on an IPv4 UDP (dgram) socket.
%% So, this is done on the receiving side: 
%%
%%               socket:setopt(Sock, ip, recvorigdstaddr, boolean()).
%%
%% For all subsequent *received* messages, the origdstaddr control
%% message header will be with the message.
%%
%%

api_opt_ip_recvorigdstaddr_udp4(suite) ->
    [];
api_opt_ip_recvorigdstaddr_udp4(doc) ->
    [];
api_opt_ip_recvorigdstaddr_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ip_recvorigdstaddr_udp4,
           fun() -> has_support_ip_recvorigdstaddr() end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, ip, recvorigdstaddr, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, ip, recvorigdstaddr)
                          end,
                   Send = fun(Sock, Data, Dest) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_ip_recvorigdstaddr_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ip_recvorigdstaddr_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := local = Domain} = State) ->
                           LSASrc = which_local_socket_addr(Domain),
                           LSADst = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSASrc,
                                       lsa_dst => LSADst}};
                      (#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},
         #{desc => "get default recvorigdstaddr for dst socket",
           cmd  => fun(#{sock_dst := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("dst recvorigdstaddr: ~p", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected src recvorigdstaddr: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) "
                                               "recvorigdstaddr:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (when recvorigdstaddr disabled)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst)
                   end},
         #{desc => "recv req (from src) - wo origdstaddr",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "enable recvorigdstaddr on dst socket",
           cmd  => fun(#{sock_dst := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("dst recvorigdstaddr enabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed enable recvorigdstaddr:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (when recvorigdstaddr enabled)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst)
                   end},
         #{desc => "recv req (from src) - w origdstaddr",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := ip,
                                             type  := origdstaddr,
                                             value := Addr}], ?BASIC_REQ}} ->
                                   ?SEV_IPRINT("got origdstaddr "
                                               "control message header: "
                                               "~n   ~p", [Addr]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [#{level => ip,
                                                   type  => origdstaddr,
                                                   data  => "something"}], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src,
                                                [#{level => ip,
                                                   type  => origdstaddr,
                                                   data  => "something"}],
                                                ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "close src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the tos control message header is received when
%% setting the socket 'ip' option recvtos is set to true when using
%% sendmsg/recvmsg on an IPv4 UDP (dgram) socket.
%% So, this is done on the receiving side: 
%%
%%               socket:setopt(Sock, ip, recvtos, boolean()).
%%
%% For all subsequent *received* messages, the tos control message
%% header will be with the message.
%%
%% On some platforms it works sending TOS with the message (sendmsg with 
%% a control message header), but since its not universal, we can't use
%% that method. Instead, set tos (true) on the sending socket.
%%

api_opt_ip_recvtos_udp4(suite) ->
    [];
api_opt_ip_recvtos_udp4(doc) ->
    [];
api_opt_ip_recvtos_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ip_recvtos_udp4,
           fun() -> has_support_ip_recvtos() end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, ip, recvtos, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, ip, recvtos)
                          end,
                   Send = fun(Sock, Data, Dest, default) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg);
                             (Sock, Data, Dest, TOS) ->
                                  CMsg = #{level => ip,
                                              type  => tos,
                                              data  => TOS},
                                  Msg  = #{addr => Dest,
                                              ctrl => [CMsg],
                                              iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_ip_recvtos_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ip_recvtos_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := local = Domain} = State) ->
                           LSASrc = which_local_socket_addr(Domain),
                           LSADst = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSASrc,
                                       lsa_dst => LSADst}};
                      (#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},
         #{desc => "default recvtos for dst socket",
           cmd  => fun(#{sock_dst := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("dst recvtos: ~p", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected src recvtos: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (wo explicit tos)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "set tos = mincost on src sock",
           cmd  => fun(#{sock_src := Sock}) ->
                           ok = socket:setopt(Sock, ip, tos, mincost)
                   end},
         #{desc => "send req (to dst) (w tos = mincost)",
           cmd  => fun(#{sock_src := Sock,
                         sa_dst   := Dst,
                         send     := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},

         %% #{desc => "send req (to dst) (w explicit tos = mincost)",
         %%   cmd  => fun(#{sock_src := Sock,
         %%                 sa_dst   := Dst,
         %%                 send     := Send}) ->
         %%                   socket:setopt(Sock, otp, debug, true),
         %%                   case Send(Sock, ?BASIC_REQ, Dst, mincost) of
         %%                       ok ->
         %%                           socket:setopt(Sock, otp, debug, false),
         %%                           ok;
         %%                       {error, Reason} ->
         %%                           ?SEV_EPRINT("Failed sending message with tos: "
         %%                                       "~n   Reason: ~p", [Reason]),
         %%                           socket:setopt(Sock, otp, debug, false),
         %%                           {skip, "Failed sending message with TOS"}
         %%                   end
         %%           end},

         #{desc => "recv req (from src) - wo explicit tos",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "set tos = 0 on src sock (\"disabled\")",
           cmd  => fun(#{sock_src := Sock}) ->
                           ok = socket:setopt(Sock, ip, tos, 0)
                   end},

         #{desc => "enable recvtos on dst socket",
           cmd  => fun(#{sock_dst := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("dst recvtos enabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting recvtos:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (wo explicit tos)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src) - w default tos",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := ip,
                                             type  := TOS,
                                             value := 0}], ?BASIC_REQ}}
                                 when ((TOS =:= tos) orelse (TOS =:= recvtos)) ->
                                   ?SEV_IPRINT("got default TOS (~w) "
                                               "control message header", [TOS]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "set tos = mincost on src sock",
           cmd  => fun(#{sock_src := Sock}) ->
                           ok = socket:setopt(Sock, ip, tos, mincost)
                   end},

         #{desc => "send req (to dst) (w tos = mincost)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src) - w tos = mincost",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := ip,
                                             type  := TOS,
                                             value := mincost = TOSData}],
                                     ?BASIC_REQ}} 
                                 when ((TOS =:= tos) orelse (TOS =:= recvtos)) ->
                                   ?SEV_IPRINT("got expected TOS (~w) = ~w "
                                               "control message header", 
                                               [TOS, TOSData]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "close src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the ttl control message header is received when
%% setting the socket 'ip' option recvttl is set to true when using
%% sendmsg/recvmsg on an IPv4 UDP (dgram) socket.
%% So, this is done on the receiving side: 
%%
%%               socket:setopt(Sock, ip, recvttl, boolean()).
%%
%% For all subsequent *received* messages, the ttl control message
%% header will be with the message.
%%
%% On darwin we don't actually get the TTL we send even after we have
%% enabled TTL. Instead we get the default value (which was 64).
%% Possibly this is because we run the test in the same OS process and
%% even the same erlang process....
%% The same issue on OpenBSD (6.6).
%% Maybe we should send and receive from different VMs, until then
%% skip darwin and OpenBSD.
%%

api_opt_ip_recvttl_udp4(suite) ->
    [];
api_opt_ip_recvttl_udp4(doc) ->
    [];
api_opt_ip_recvttl_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ip_recvttl_udp4,
           fun() ->
		   has_support_ip_recvttl(),
		   is_not_openbsd(),
		   is_not_darwin()
	   end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, ip, recvttl, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, ip, recvttl)
                          end,
                   Send = fun(Sock, Data, Dest, default) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg);
                             (Sock, Data, Dest, TTL) ->
                                  CMsg = #{level => ip,
                                              type  => ttl,
                                              value => TTL},
                                  Msg  = #{addr => Dest,
                                              ctrl => [CMsg],
                                              iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_ip_recvttl_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ip_recvttl_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := local = Domain} = State) ->
                           LSASrc = which_local_socket_addr(Domain),
                           LSADst = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSASrc,
                                       lsa_dst => LSADst}};
                      (#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},
         #{desc => "default recvttl for dst socket",
           cmd  => fun(#{sock_dst := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("dst recvttl: ~p", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected src recvttl: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) timestamp:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (wo explicit ttl)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (w explicit ttl = 100)",
           cmd  => fun(#{sock_src := SSock,
                         sock_dst := DSock, sa_dst := Dst,
                         send     := Send}) ->
                           case Send(SSock, ?BASIC_REQ, Dst, 100) of
                               ok ->
                                   ok;
                               {error, einval = Reason} ->
                                   %% IF we can't send it the test will not work
                                   ?SEV_EPRINT("Cannot send TTL: "
                                               "~p => SKIP", [Reason]),
                                   (catch socket:close(SSock)),
                                   (catch socket:close(DSock)),
                                   {skip,
				    ?F("Cannot send with TTL: ~p", [Reason])};
                               {error, enoprotoopt = Reason} ->
                                   %% On some platforms this is not
                                   %% accepted (FreeBSD), so skip.
                                   ?SEV_EPRINT("Expected Failure: "
                                               "~p => SKIP", [Reason]),
                                   (catch socket:close(SSock)),
                                   (catch socket:close(DSock)),
                                   {skip, Reason};
                               {error, _Reason} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "recv req (from src) - wo ttl",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {Src, [#{level := ip,
                                             type  := TTLType,
                                             value := TTL}], ?BASIC_REQ}}
                               when ((TTLType =:= recvttl) andalso
                                     (TTL =:= 255)) ->
                                   %% This is the behaviopur on Solaris (11)
                                   %% and maybe on other platforms...
                                   ?SEV_IPRINT("Got (default) TTL (~w): ~p",
                                               [TTLType, TTL]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "enable recvttl on dst socket",
           cmd  => fun(#{sock_dst := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("dst recvttl enabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed enabling recvttl:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (wo explicit ttl)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src) - w default ttl",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := ip,
                                             type  := TTLType,
                                             value := TTL}], ?BASIC_REQ}}
                               when ((TTLType =:= ttl) orelse 
                                     (TTLType =:= recvttl)) ->
                                   ?SEV_IPRINT("Got (default) TTL (~w): ~p",
                                               [TTLType, TTL]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [#{level => ip,
						   type  => ttl,
						   value => "something"}],
						BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [#{level => ip,
							type  => ttl,
							value => "something"}],
						?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (w explicit ttl = 100)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, 100)
                   end},
         #{desc => "recv req (from src) - w ttl = 100",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := ip,
                                             type  := TTLType,
                                             value := 100 = TTL}], ?BASIC_REQ}}
                               when ((TTLType =:= ttl) orelse
                                     (TTLType =:= recvttl)) ->
                                   ?SEV_IPRINT("Got TTL (~w): ~p",
                                               [TTLType, TTL]),
                                   ok;
                               {ok, {Src, [#{level := ip,
                                             type  := TTLType,
                                             value := BadTTL}], ?BASIC_REQ}}
                               when ((TTLType =:= ttl) orelse
                                     (TTLType =:= recvttl)) ->
                                   ?SEV_EPRINT("Unexpected TTL: "
                                               "~n   Expect TTL: ~p"
                                               "~n   Recv TTL:   ~p",
                                               [100, BadTTL]),
                                   {error, {unexpected_ttl, {100, BadTTL}}};
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [#{level => ip,
						   type  => ttl,
						   value => 100}], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [#{level => ip,
							type  => ttl,
							value => 100}],
						?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "close src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the ip socket option 'tos' can be set and retrieved from a
%% the socket its set on. It sets the type-of-server field in the IP
%% header for a TCP or UDP socket.
%% There is no way to fetch the value a received IP datagram.
%% Default value is supposed to be '0'.
%% 

api_opt_ip_tos_udp4(suite) ->
    [];
api_opt_ip_tos_udp4(doc) ->
    [];
api_opt_ip_tos_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ip_tos_udp4,
           fun() -> has_support_ip_tos() end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, ip, tos, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, ip, tos)
                          end,
                   InitState = #{set => Set,
                                 get => Get},
                   ok = api_opt_ip_tos_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ip_tos_udp(InitState) ->
    process_flag(trap_exit, true),
    %% mincost is not supported on all platforms.
    %% For instance, Solaris 10, does not have that constant.
    %% Instead it has two others with, what appers to be,
    %% completely different meanings...
    %% So, avoid the complication by not using this value...
    %% TOS1 = mincost,     TOS1Str = atom_to_list(TOS1),
    TOS2 = throughput,  TOS2Str = atom_to_list(TOS2),
    TOS3 = reliability, TOS3Str = atom_to_list(TOS3),
    TOS4 = lowdelay,    TOS4Str = atom_to_list(TOS4),
    TOS5 = 42,          TOS5Str = integer_to_list(TOS5),
    Seq =
        [
         #{desc => "local address",
           cmd  => fun(State) ->
                           LSA = which_local_socket_addr(inet),
                           {ok, State#{lsa => LSA}}
                   end},

         #{desc => "open socket",
           cmd  => fun(State) ->
                           Sock = sock_open(inet, dgram, udp),
                           {ok, State#{sock => Sock}}
                   end},
         #{desc => "bind",
           cmd  => fun(#{sock := Sock, lsa := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("socket bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "get default tos",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, 0 = Value} ->
                                   ?SEV_IPRINT("expected default tos: ~p", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected default tos: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) tos:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         %% #{desc => "set tos " ++ TOS1Str,
         %%   cmd  => fun(#{sock := Sock, set := Set} = _State) ->
         %%                   socket:setopt(Sock, otp, debug, true),
         %%                   case Set(Sock, TOS1) of
         %%                       ok ->
         %%                           socket:setopt(Sock, otp, debug, false),
         %%                           ?SEV_IPRINT("tos set to ~p", [TOS1]),
         %%                           ok;
         %%                       {error, Reason} = ERROR ->
         %%                           socket:setopt(Sock, otp, debug, false),
         %%                           ?SEV_EPRINT("Failed setting tos:"
         %%                                       "   ~p", [Reason]),
         %%                           ERROR
         %%                   end
         %%           end},
         %% #{desc => "get tos (expect " ++ TOS1Str ++ ")",
         %%   cmd  => fun(#{sock := Sock, get := Get} = _State) ->
         %%                   case Get(Sock) of
         %%                       {ok, TOS1 = Value} ->
         %%                           ?SEV_IPRINT("expected tos (~p)", [Value]),
         %%                           ok;
         %%                       {ok, Unexpected} ->
         %%                           ?SEV_EPRINT("Unexpected tos: ~p",
         %%                                       [Unexpected]),
         %%                           {error, {unexpected, Unexpected}};
         %%                       {error, Reason} = ERROR ->
         %%                           ?SEV_EPRINT("Failed getting (default) tos:"
         %%                                       "   ~p", [Reason]),
         %%                           ERROR
         %%                   end
         %%           end},

         #{desc => "set tos " ++ TOS2Str,
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           case Set(Sock, TOS2) of
                               ok ->
                                   ?SEV_IPRINT("tos set to ~p", [TOS2]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting tos:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "get tos (expect " ++ TOS2Str ++ ")",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, TOS2 = Value} ->
                                   ?SEV_IPRINT("expected tos (~p)", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected tos: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) tos:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "set tos " ++ TOS3Str,
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           case Set(Sock, TOS3) of
                               ok ->
                                   ?SEV_IPRINT("tos set to ~p", [TOS3]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting tos:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "get tos (expect " ++ TOS3Str ++ ")",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, TOS3 = Value} ->
                                   ?SEV_IPRINT("expected tos (~p)", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected tos: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) tos:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "set tos " ++ TOS4Str,
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           case Set(Sock, TOS4) of
                               ok ->
                                   ?SEV_IPRINT("tos set to ~p", [TOS4]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting tos:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "get tos (expect " ++ TOS4Str ++ ")",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, TOS4 = Value} ->
                                   ?SEV_IPRINT("expected tos (~p)", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected tos: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) tos:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "set tos " ++ TOS5Str,
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           case Set(Sock, TOS5) of
                               ok ->
                                   ?SEV_IPRINT("tos set to ~p", [TOS5]),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting tos:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "get tos (expect " ++ TOS5Str ++ ")",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, TOS5 = Value} ->
                                   ?SEV_IPRINT("expected tos (~p)", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected tos: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) tos:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the ip socket option 'recverr' can be set and that the error
%% queue can be read.
%% 

api_opt_ip_recverr_udp4(suite) ->
    [];
api_opt_ip_recverr_udp4(doc) ->
    [];
api_opt_ip_recverr_udp4(Config) when is_list(Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ip_recverr_udp4,
           fun() ->
                   has_support_ip_recverr()
           end,
           fun() ->
                   Set  = fun(Sock, Key, Value) ->
                                  socket:setopt(Sock, ip, Key, Value)
                          end,
                   Get  = fun(Sock, Key) ->
                                  socket:getopt(Sock, ip, Key)
                          end,
                   Send = fun(Sock, Data, Dest, Tmo) ->
                                  socket:sendto(Sock, Data, Dest, [], Tmo)
                          end,
                   Recv = fun(Sock, Tmo) ->
                                  socket:recvfrom(Sock, 0, [], Tmo)
                          end,
                   InitState = #{domain => inet,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_recverr_udp(Config, InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests the ipv6 socket option 'recverr' can be set and that the error
%% queue can be read.
%% 

api_opt_ipv6_recverr_udp6(suite) ->
    [];
api_opt_ipv6_recverr_udp6(doc) ->
    [];
api_opt_ipv6_recverr_udp6(Config) when is_list(Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ipv6_recverr_udp6,
           fun() ->
                   has_support_ipv6(),
                   has_support_ipv6_recverr()
           end,
           fun() ->
                   Set  = fun(Sock, Key, Value) ->
                                  socket:setopt(Sock, ipv6, Key, Value)
                          end,
                   Get  = fun(Sock, Key) ->
                                  socket:getopt(Sock, ipv6, Key)
                          end,
                   Send = fun(Sock, Data, Dest, Tmo) ->
                                  socket:sendto(Sock, Data, Dest, [], Tmo)
                          end,
                   Recv = fun(Sock, Tmo) ->
                                  socket:recvfrom(Sock, 0, [], Tmo)
                          end,
                   InitState = #{domain => inet6,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_recverr_udp(Config, InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_recverr_udp(Config, InitState) ->
    Seq = 
        [
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain} = State) ->
                           ?SEV_IPRINT("test for ip:recvtos"),
                           case socket:open(Domain, dgram, udp) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR -> 
                                   ERROR
                           end
                   end},
         #{desc => "bind (to loopback)",
           cmd  => fun(#{sock := Sock} = State) ->
                           case socket:bind(Sock, loopback) of
                               ok ->
                                   case socket:sockname(Sock) of
                                       {ok, Addr} ->
                                           ?SEV_IPRINT(
                                              "bound to ~p", [Addr]),
                                           {ok, State#{addr => Addr}};
                                       {error, _} = ERROR_1 ->
                                           ERROR_1
                                   end;
                               {error, _} = ERROR_2 ->
                                   ERROR_2
                           end
                   end},

         #{desc => "enable recverr",
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           Set(Sock, recverr, true)
                   end},

         #{desc => "disable mtu_discover",
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           Set(Sock, mtu_discover, dont)
                   end},

         #{desc => "try (async) read (=> select)",
           cmd  => fun(#{sock := Sock,
                         recv := Recv} = State) ->
                           RecvRef = nowait(Config),
                           case Recv(Sock, RecvRef) of
                               {select, SelectInfo} when RecvRef =:= nowait ->
                                   ?SEV_IPRINT("expected select nowait: "
					       "~n   ~p", [SelectInfo]),
                                   {ok, State#{rselect => SelectInfo}};
                               {select,
                                {select_info, _Tag, RecvRef} = SelectInfo}
                                 when is_reference(RecvRef) ->
                                   ?SEV_IPRINT("expected select ref: "
					       "~n   ~p", [SelectInfo]),
                                   {ok, State#{rselect => SelectInfo}};
                               {ok, _} ->
                                   ?SEV_EPRINT("unexpected successs"),
                                   {error, unexpected_success};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("unexpected error: ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "try send to nowhere",
           cmd  => fun(#{domain := Domain,
                         sock := Sock,
                         send := Send} = State) ->
                           SendRef = nowait(Config),
                           Dest = #{family => Domain,
                                    addr   => if
                                                  (Domain =:= inet) ->
                                                      {127,0,0,1};
                                                  (Domain =:= inet6) ->
                                                      {0,0,0,0,0,0,0,1}
                                              end,
                                    port   => 44444},
                           case Send(Sock, <<"ping">>, Dest, SendRef) of
                               ok ->
                                   ?SEV_IPRINT("sent"),
                                   ok;
                               {select, SelectInfo}
                                 when SendRef =:= nowait ->
                                   ?SEV_IPRINT("expected select nowait: ~p",
					       [SelectInfo]),
                                   {ok, State#{sselect => SelectInfo}};
                               {select,
                                {select_info, _Tag, SendRef} = SelectInfo}
                                 when is_reference(SendRef) ->
                                   ?SEV_IPRINT("expected select ref: ~p",
					       [SelectInfo]),
                                   {ok, State#{sselect => SelectInfo}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("unexpected error: ~p",
					       [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "await receive select message",
           cmd  => fun(#{sock    := Sock,
                         rselect := {select_info, _, Ref}} = _State) ->
                           receive
                               {'$socket', Sock, select, Ref} ->
                                   ?SEV_IPRINT("received expected (read) select message: "
                                               "~n   ~p", [Ref]),
                                   ok
                           end
                   end},

         #{desc => "try recv - expect econnrefused",
           cmd  => fun(#{sock := Sock, recv := Recv} = _State) ->
                           case Recv(Sock, infinity) of
                               {error, econnrefused = Reason} ->
                                   ?SEV_IPRINT("expected failure: ~p", [Reason]),
                                   ok;
                               {ok, _} ->
                                   ?SEV_EPRINT("unexpected successs"),
                                   {error, unexpected_success};
                               {select, SelectInfo} ->
                                   ?SEV_EPRINT("unexpected select: ~p",
                                               [SelectInfo]),
                                   {error, unexpected_success};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("unexpected error: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send to self",
           cmd  =>
               fun(#{sock := Sock,
                     send := Send,
                     addr := Addr} = _State) ->
                       case Send(Sock, <<"ring">>, Addr, infinity) of
                           ok ->
                               ?SEV_IPRINT("sent to self"),
                               ok;
                           {error, _} = Error ->
                               Error
                       end
               end},

         #{desc => "try recv - expect data sent to self",
           cmd  => fun(#{sock := Sock,
                         addr := Addr,
                         recv := Recv} = _State) ->
                           case Recv(Sock, infinity) of
                               {ok, {Addr, <<"ring">>}} ->
                                   ?SEV_IPRINT("receive expected"),
                                   ok;
                               {select, SelectInfo} ->
                                   ?SEV_EPRINT("unexpected select: ~p",
                                               [SelectInfo]),
                                   {error, unexpected_success};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("unexpected error: ~p",
                                               [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "try recv error queue",
           cmd  => fun(#{domain := Domain, sock := Sock}) ->
                           %% Note that not all platforms that support
                           %% recverr, actually supports "encoding" the data
                           %% part, so we need to adjust for that.
			   Origin = 
			       if (Domain =:= inet)  -> icmp;
				  (Domain =:= inet6) -> icmp6
			       end,
			   Level =
			       if (Domain =:= inet)  -> ip;
				  (Domain =:= inet6) -> ipv6
			       end,
                           case socket:recvmsg(Sock, [errqueue], 0) of
                               {ok, #{addr  := #{family := Domain,
						 addr   := Addr},
                                      flags := [errqueue],
                                      iov   := [<<"ping">>],
                                      ctrl  := [#{level := Level,
                                                  type  := recverr,
                                                  value :=
                                                      #{code     := port_unreach,
                                                        data     := 0,
                                                        error    := econnrefused,
                                                        info     := 0,
                                                        offender := #{family := Domain,
								      addr   := Addr},
                                                        origin   := Origin,
                                                        type     := dest_unreach}
                                                 }]} = Msg} ->
                                   ?SEV_IPRINT("expected error queue (decoded): "
                                               "~n   ~p", [Msg]),
                                   ok;
                               {ok, #{addr  := #{family := Domain,
						 addr   := _Addr},
                                      flags := [errqueue],
                                      iov   := [<<"ping">>],
                                      value := [#{level := Level,
                                                  type  := recverr}]} = _Msg} ->
                                   ?SEV_IPRINT("expected error queue"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("failed reading error queue: "
                                               "~n   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% This intended to test "all" of the (currently) supported IPv4
%% options that results in control message header(s).
%% So, this is done on the receiving side:
%%
%%      socket:setopt(Sock, ip, Flag, boolean()).
%%
%% For all subsequent *received* messages, a control message header
%% for each of the enabled options will be received with the message.
%%
%% Only allowed for dgram and raw,
%% although we only test this with dgram.
%%
%% Currently we *try* to use the following opts:
%%
%%      pktinfo         => pktinfo
%%      recvorigdstaddr => origdstaddr
%%      recvtos         => tos
%%      recvttl         => ttl
%%
%%
%% Every time we add a test case for a new option (that results in
%% a control message hedare), we should also add it here.
%%
%% Even though this is a IPv4 test case, we add the 'socket' timestamp
%% option (just to fill up), but in the test to see if we should run
%% the test (since its a IPv4 test case).
%%

api_opt_ip_mopts_udp4(suite) ->
    [];
api_opt_ip_mopts_udp4(doc) ->
    [];
api_opt_ip_mopts_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ip_mopts_udp4,
           fun() ->
		   case is_any_options_supported(
			  [{ip, pktinfo},
			   {ip, recvorigdstaddr},
			   {ip, recvtos},
			   {ip, recvttl}]) of
		       true ->
			   ok;
		       false ->
			   skip("None of the needed options are supported")
		   end
           end,
           fun() ->
		   %% If we get this far, we *know* that at least one of the
		   %% options are available.

		   %% This is list of all the options and there resulting
		   %% control message header type(s):
		   %%   [{level,
		   %%     'ipv6 socket option',
                   %%     'control message header type',
		   %%     default | value()}]
		   Opts =
		       case socket:is_supported(options, socket, timestamp) of
			   true ->
			       [{socket, timestamp, timestamp, default}];
			   false ->
			       []
		       end ++
		       case socket:is_supported(options, ip, pktinfo) of
			   true ->
			       [{ip, pktinfo, pktinfo, default}];
			   false ->
			       []
		       end ++
		       case socket:is_supported(options, ip, recvorigdstaddr) of
			   true ->
			       [{ip, recvorigdstaddr, origdstaddr, default}];
			   false ->
			       []
		       end ++
		       case socket:is_supported(options, ip, recvtos) of
			   true ->
                               %% It seems that sending any of the
                               %% TOS or TTL values will fail on: 
			       %%    FreeBSD
			       %%    Linux when
			       %%      version =< 3.12.60 (at least)
			       %%      Don't know when this starts working,
			       %%      but it works on: 
			       %%           Ubunto 16.04.6 => 4.15.0-65
			       %%           SLES 12 SP2    => 4.4.120-92.70
			       %% so don't!
                               %%
                               %% The latest we know it not to work was a
                               %% SLES 12 (plain) at 3.12.50-52.54
                               %%
			       [{ip, recvtos, tos, 
                                 case os:type() of
                                     {unix, freebsd} ->
                                         default;
				     {unix, linux} ->
					 case os:version() of
					     Vsn when Vsn > {3,12,60} ->
						 42;
					     _ ->
						 default
					 end;
                                     _ -> 
                                         42
                                 end}];
			   false ->
			       []
		       end ++
                       case os:type() of
                           {unix, darwin} ->
                               [];
                           _ ->
                               case socket:is_supported(options, ip, recvttl) of
                                   true ->
				       %% It seems that sending any of the
				       %% TOS or TTL values will fail on: 
				       %%    FreeBSD and NetBSD
				       %%    Linux when
				       %%      version =< 3.12.60 (at least)
				       %% so don't!
                                       %% See recvtos above for more info.
                                       [{ip, recvttl, ttl,
                                         case os:type() of
                                             {unix, BSD} 
                                               when (BSD =:= freebsd) orelse
                                                    (BSD =:= netbsd) ->
                                                 default;
                                             {unix, netbsd} ->
                                                 default;
					     {unix, linux} ->
						 case os:version() of
						     Vsn when Vsn > {3,12,60} ->
							 42;
						     _ ->
							 default
						 end;
                                             _ -> 
                                                 42
                                         end}];
                                   false ->
                                       []
                               end
		       end,

                   Enable = fun(Sock, Level, Opt) ->
				    ?SEV_IPRINT("try enable [~w] ~p", [Level, Opt]),
				    socket:setopt(Sock, Level, Opt, true)
                            end,
                   Send = fun(Sock, Data, Dest, []) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg);
			     (Sock, Data, Dest, Hdrs) when is_list(Hdrs) ->
				  CMsgs = [#{level => Level,
						type  => Type,
						value => Val} ||
						 {Level, Type, Val} <- Hdrs],
                                  Msg   = #{addr => Dest,
					       ctrl => CMsgs,
					       iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet,
                                 proto  => udp,
				 opts   => Opts,
                                 send   => Send,
                                 recv   => Recv,
                                 enable => Enable},
                   ok = api_opt_ip_mopts_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ip_mopts_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},

         #{desc => "enable options on dst socket",
           cmd  => fun(#{sock_dst := DSock,
			 sock_src := SSock,
			 opts     := Opts,
			 enable   := Enable} = _State) ->
			   %% If we fail to enable *any* of the options,
			   %% we give up.
			   E = fun({Level, Opt, _, _}) -> 
                                       case Enable(DSock, Level, Opt) of
                                           ok ->
                                               ?SEV_IPRINT("dst [~w] ~w enabled",
                                                           [Level, Opt]),
                                               ok;
                                           {error, enoprotoopt = Reason} ->
                                               ?SEV_EPRINT("Expected "
                                                           "Failure: "
                                                           "~p => SKIP",
                                                           [Reason]),
                                               (catch socket:close(DSock)),
                                               (catch socket:close(SSock)),
                                               {skip, Reason};
                                           {error, Reason} = ERROR ->
                                               ?SEV_EPRINT("Failed "
                                                           "setting ~w:"
                                                           "   ~p",
                                                           [Opt, Reason]),
                                               throw(ERROR)
                                       end
			       end,
			   lists:foreach(E, Opts),
			   ok
                   end},

         #{desc => "send req (to dst)",
           cmd  => fun(#{sock_src := Sock,
			 sa_dst   := Dst,
			 opts     := Opts,
			 send     := Send}) ->
			   Hdrs = [{Level, Type, Data} ||
				      {Level, _, Type, Data} <- 
					  Opts, (Data =/= default)],
                           Send(Sock, ?BASIC_REQ, Dst, Hdrs)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock,
			 sa_src   := Src,
			 recv     := Recv,
			 opts     := Opts}) ->
                           case Recv(Sock) of
                               {ok, {Src, CMsgs, ?BASIC_REQ}}
                                 when length(CMsgs) =:= length(Opts)  ->
                                   ?SEV_IPRINT("Got (expected) cmsg headers: "
					       "~n   ~p", [CMsgs]),
				   %% We should really verify the headers:
				   %% values, types and so on...
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [{Level, Type} ||
                                                    {Level, _, Type, _} <- Opts],
						BadCHdrs,
						?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src,
                                                [{Level, Type} ||
                                                    {Level, _, Type, _} <- Opts],
						?BASIC_REQ,
                                                UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "close src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the IPv6 pktinfo control message header is received on
%% incoming datagrams (UDP and RAW) when setting the socket 'ipv6'
%% option recvpktinfo is set to true when using.
%% So, this is done on the receiving side: 
%%
%%               socket:setopt(Sock, ipv6, recvpktinfo, boolean()).
%%
%% For all subsequent *received* messages, the pktinfo control message
%% header will be with the message.
%%
%% Only allowed for dgram and raw,
%% although we only test this with dgram.
%%

api_opt_ipv6_recvpktinfo_udp6(suite) ->
    [];
api_opt_ipv6_recvpktinfo_udp6(doc) ->
    [];
api_opt_ipv6_recvpktinfo_udp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ipv6_recvpktinfo_udp6,
           fun() ->
                   has_support_ipv6(),
                   has_support_ipv6_recvpktinfo()
           end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, ipv6, recvpktinfo, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, ipv6, recvpktinfo)
                          end,
                   Send = fun(Sock, Data, Dest, default) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg);
                             (Sock, Data, Dest, Info) ->
                                  %% We do not support this at the moment!!!
                                  CMsg = #{level => ipv6,
                                              type  => pktinfo,
                                              data  => Info},
                                  Msg  = #{addr => Dest,
                                              ctrl => [CMsg],
                                              iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet6,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_ipv6_recvpktinfo_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ipv6_recvpktinfo_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},
         #{desc => "default pktinfo for dst socket",
           cmd  => fun(#{sock_dst := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("dst recvttl: ~p", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected src recvtos: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) recvtos:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (wo (explicit) pktinfo)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "enable pktinfo on dst socket",
           cmd  => fun(#{sock_dst := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("dst pktinfo enabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting pktinfo:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (wo explicit pktinfo)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src) - w default pktinfo",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := ipv6,
                                             type  := pktinfo,
                                             value := #{addr    := Addr,
                                                        ifindex := IfIdx}}],
                                     ?BASIC_REQ}} ->
                                   ?SEV_IPRINT("Got (default) Pkt Info: "
                                               "~n   Addr:      ~p"
                                               "~n   If Index:  ~p",
                                               [Addr, IfIdx]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                #{level => ipv6,
                                                  type  => pktinfo,
                                                  value => "something"} , BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},


         #{desc => "close src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the 'flow info' control message header is received when
%% setting the socket 'ipv6' option flowinfo  is set to true when using
%% sendmsg/recvmsg on an IPv6 UDP (dgram) socket.
%% So, this is done on the receiving side: 
%%
%%               socket:setopt(Sock, ipv6, flowinfo, boolean()).
%%
%% For all subsequent *received* messages, the 'flow info' control message
%% header will be with the message.
%%
%% Only allowed for dgram and raw,
%% although we only test this with dgram.
%%
%% There seem to be some weirdness with the definition of this
%% option, so its defined in an include file we don't include
%% (directly or indirectly). And since some of the defines
%% are occure in a file we *do* include (via netinet/in.h), we
%% leave it as is for now...
%%

api_opt_ipv6_flowinfo_udp6(suite) ->
    [];
api_opt_ipv6_flowinfo_udp6(doc) ->
    [];
api_opt_ipv6_flowinfo_udp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ipv6_flowinfo_udp6,
           fun() ->
                   has_support_ipv6(),
                   has_support_ipv6_flowinfo()
           end,
           fun() ->
                   Set  = fun(Sock, Value) ->
                                  socket:setopt(Sock, ipv6, flowinfo, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, ipv6, flowinfo)
                          end,
                   Send = fun(Sock, Data, Dest) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet6,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_ipv6_flowinfo_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ipv6_flowinfo_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},
         #{desc => "default flowinfo for dst socket",
           cmd  => fun(#{sock_dst := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("dst flowinfo: ~p", [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected src flowinfo: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) flowinfo:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "enable flowinfo on dst socket",
           cmd  => fun(#{sock_dst := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("dst flowinfo enabled"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting flowinfo:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := ipv6,
                                             type  := flowinfo,
                                             value := FlowID}], ?BASIC_REQ}} ->
                                   ?SEV_IPRINT("Got flow info: "
					       "~n   Flow ID: ~p", [FlowID]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                #{level => ipv6,
						  type  => flowinfo,
						  value => "something"},
						BadCHdrs,
						?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "close src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the 'hop limit' control message header is received when
%% setting the socket 'ipv6' hoplimit or recvhoplimit option is set to 
%% true when using sendmsg/recvmsg on an IPv6 UDP (dgram) socket.
%% So, this is done on the receiving side: 
%%
%%      socket:setopt(Sock, ipv6, recvhoplimit | hoplimit, boolean()).
%%
%% For all subsequent *received* messages, the 'hop limit' control message
%% header will be with the message.
%% We make the assumption, that if 'recvhoplimit' is supported, then
%% that option is used to order the hoplimit control message, otherwise
%% hoplimit is used.
%%
%% Only allowed for dgram and raw,
%% although we only test this with dgram.
%%
%% <Note>
%%
%% There is also an IPV6_RECVHOPLIMIT option defined in the header
%% file (bits/in.h) with a different value. This is not mentioned
%% in the man page. Deprecated? More testing needed...
%%
%% </Note>
%%

api_opt_ipv6_hoplimit_udp6(suite) ->
    [];
api_opt_ipv6_hoplimit_udp6(doc) ->
    [];
api_opt_ipv6_hoplimit_udp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ipv6_hoplimit_udp6,
           fun() ->
                   has_support_ipv6(),
                   has_support_ipv6_hoplimit_or_recvhoplimit(),
		   is_good_enough_darwin({9,8,0}),
                   is_good_enough_montavista("4.0.1")
           end,
           fun() ->
		   %% Begin by choosing which of the options we shall use
		   Opt = case socket:is_supported(options, ipv6, recvhoplimit) of
			     true  -> recvhoplimit;
			     false -> hoplimit
			 end,
                   Set  = fun(Sock, Value) ->
				  ?SEV_IPRINT("try set ~p: ~p", [Opt, Value]),
                                  socket:setopt(Sock, ipv6, Opt, Value)
                          end,
                   Get  = fun(Sock) ->
				  ?SEV_IPRINT("try get ~p", [Opt]),
                                  socket:getopt(Sock, ipv6, Opt)
                          end,
                   Send = fun(Sock, Data, Dest) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet6,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_ipv6_hoplimit_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ipv6_hoplimit_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},
         #{desc => "default [recv]hoplimit for dst socket",
           cmd  => fun(#{sock_dst := Sock, get := Get} = State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("dst [recv]hoplimit: ~p",
					       [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected src [recv]hoplimit: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, enoprotoopt = Reason} ->
                                   %% On some platforms this is not accepted
                                   %% for UDP, so skip this part (UDP).
                                   ?SEV_EPRINT("Expected Failure: "
                                               "~p => SKIP", [Reason]),
                                   (catch socket:close(Sock)),
                                   (catch socket:close(maps:get_value(sock_src,
								      State))),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) hoplimit:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "enable [recv]hoplimit on dst socket",
           cmd  => fun(#{sock_dst := Sock, set := Set} = State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("dst [recv]hoplimit enabled"),
                                   ok;
                               {error, enoprotoopt = Reason} ->
                                   %% On some platforms this is not accepted
                                   %% for UDP, so skip this part (UDP).
                                   ?SEV_EPRINT("Expected Failure: "
                                               "~p => SKIP", [Reason]),
                                   (catch socket:close(Sock)),
                                   (catch socket:close(maps:get_value(sock_src,
								      State))),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting hoplimit:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (wo explicit ttl)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := ipv6,
                                             type  := hoplimit,
                                             value := HL}], ?BASIC_REQ}}
                                 when is_integer(HL) ->
                                   ?SEV_IPRINT("Got hop limit: "
					       "~n   Hop Limit: ~p", [HL]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                #{level => ipv6,
						  type  => hoplimit,
						  value => "something"},
						BadCHdrs,
						?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, #{level => ipv6,
						       type  => hoplimit,
						       value => "something"},
						?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "close src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the 'tclass' control message header is received when
%% setting the socket 'ipv6' tclass or recvtclass option is set to 
%% true when using sendmsg/recvmsg on an IPv6 UDP (dgram) socket.
%% So, this is done on the receiving side: 
%%
%%      socket:setopt(Sock, ipv6, recvtclass | tclass, boolean()).
%%
%% For all subsequent *received* messages, the 'tclass' control message
%% header will be with the message.
%% We make the assumption, that if 'recvtclass' is supported, then
%% that option is used to order the tclass control message, otherwise
%% tclass is used.
%%
%% Only allowed for dgram and raw,
%% although we only test this with dgram.
%%
%% <Note>
%%
%% There is also an IPV6_RECVTCLASS option defined in the header
%% file (bits/in.h) with a different value. This is not mentioned
%% in the man page. Deprecated? More testing needed...
%%
%% </Note>
%%

api_opt_ipv6_tclass_udp6(suite) ->
    [];
api_opt_ipv6_tclass_udp6(doc) ->
    [];
api_opt_ipv6_tclass_udp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ipv6_tclass_udp6,
           fun() ->
                   has_support_ipv6(),
                   has_support_ipv6_tclass_or_recvtclass()
           end,
           fun() ->
		   %% Begin by choosing which of the options we shall use
		   Opt = case socket:is_supported(options, ipv6, recvtclass) of
			     true  -> recvtclass;
			     false -> tclass
			 end,
                   Set  = fun(Sock, Value) ->
				  ?SEV_IPRINT("try set ~p: ~p", [Opt, Value]),
                                  socket:setopt(Sock, ipv6, Opt, Value)
                          end,
                   Get  = fun(Sock) ->
				  ?SEV_IPRINT("try get ~p", [Opt]),
                                  socket:getopt(Sock, ipv6, Opt)
                          end,
                   Send = fun(Sock, Data, Dest, default) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg);
			     (Sock, Data, Dest, TC) ->
                                  TCHdr    = #{level => ipv6,
					       type  => tclass,
					       data  => TC},
				  CMsgs = [TCHdr],
                                  Msg   = #{addr => Dest,
					       ctrl => CMsgs,
					       iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet6,
                                 proto  => udp,
                                 send   => Send,
                                 recv   => Recv,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_ipv6_tclass_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ipv6_tclass_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},
         #{desc => "default [recv]tclass for dst socket",
           cmd  => fun(#{sock_dst := Sock, get := Get} = State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("dst [recv]tclass: ~p",
					       [Value]),
                                   ok;
                               {ok, Unexpected} ->
                                   ?SEV_EPRINT("Unexpected src [recv]tclass: ~p",
                                               [Unexpected]),
                                   {error, {unexpected, Unexpected}};
                               {error, enoprotoopt = Reason} ->
                                   %% On some platforms this is not accepted
                                   %% for UDP, so skip this part (UDP).
                                   ?SEV_EPRINT("Expected Failure: "
                                               "~p => SKIP", [Reason]),
                                   (catch socket:close(Sock)),
                                   (catch socket:close(maps:get_value(sock_src,
								      State))),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed getting (default) tclass:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [], ?BASIC_REQ}} ->
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [], BadCHdrs,
                                                ?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, [], ?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "enable [recv]tclass on dst socket",
           cmd  => fun(#{sock_dst := Sock, set := Set} = State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("dst [recv]tclass enabled"),
                                   ok;
                               {error, enoprotoopt = Reason} ->
                                   %% On some platforms this is not accepted
                                   %% for UDP, so skip this part (UDP).
                                   ?SEV_EPRINT("Expected Failure: "
                                               "~p => SKIP", [Reason]),
                                   (catch socket:close(Sock)),
                                   (catch socket:close(maps:get_value(sock_src,
								      State))),
                                   {skip, Reason};
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("Failed setting tclass:"
                                               "   ~p", [Reason]),
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (wo explicit tc)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, default)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := ipv6,
                                             type  := tclass,
                                             value := TClass}], ?BASIC_REQ}}
			       when is_integer(TClass) ->
                                   ?SEV_IPRINT("Got tclass: "
					       "~n   TClass: ~p", [TClass]),
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                #{level => ipv6,
						  type  => tclass,
						  value => "something"},
						BadCHdrs,
						?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, #{level => ipv6,
						       type  => tclass,
						       value => "something"},
						?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "send req (to dst) (w explicit tc = 1)",
           cmd  => fun(#{sock_src := Sock, sa_dst := Dst, send := Send}) ->
                           Send(Sock, ?BASIC_REQ, Dst, 1)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock, sa_src := Src, recv := Recv}) ->
                           case Recv(Sock) of
                               {ok, {Src, [#{level := ipv6,
                                             type  := tclass,
                                             value := 1 = TClass}], ?BASIC_REQ}}
			       when is_integer(TClass) ->
                                   ?SEV_IPRINT("Got (expected) tclass: "
					       "~n   TClass: ~p", [TClass]),
                                   ok;
                               {ok, {_Src, [#{level := ipv6,
					      type  := tclass,
					      value := TClass}], ?BASIC_REQ}}
			       when is_integer(TClass) ->
                                   ?SEV_EPRINT("Unexpected tclass: "
                                               "~n   Expect TClass: ~p"
                                               "~n   Recv TClass:   ~p",
                                               [1, TClass]),
                                   {error, {unexpected_tclass, TClass}};
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                #{level => ipv6,
						  type  => tclass,
						  value => "something"},
						BadCHdrs,
						?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src, #{level => ipv6,
						       type  => tclass,
						       value => "something"},
						?BASIC_REQ, UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "close src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% This intended to test "all" of the (currently) supported IPv6
%% options that results in control message header(s).
%% So, this is done on the receiving side: 
%%
%%      socket:setopt(Sock, ipv6, Flag, boolean()).
%%
%% For all subsequent *received* messages, a control message header
%% for each of the enabled options will be received with the message.
%%
%% Only allowed for dgram and raw,
%% although we only test this with dgram.
%%
%% Currently we *try* to use the following opts:
%%
%%      recvpktinfo | pktinfo   => pktinfo
%%      flowinfo                => flowinfo
%%      recvhoplimit | hoplimit => hoplimit
%%      recvtclass | tclass     => tclass
%%
%%
%% Every time we add a test case for a new option (that results in
%% a control message hedare), we should also add it here.
%%
%% Even though this is a IPv6 test case, we add the 'socket' timestamp
%% option (just to fill up), but in the test to see if we should run
%% the test (since its a IPv6 test case).
%%

api_opt_ipv6_mopts_udp6(suite) ->
    [];
api_opt_ipv6_mopts_udp6(doc) ->
    [];
api_opt_ipv6_mopts_udp6(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_ipv6_mopts_udp6,
           fun() ->
                   has_support_ipv6(),
		   case is_any_options_supported(
			  [{ipv6, recvpktinfo},
			   {ipv6, flowinfo},
			   {ipv6, recvhoplimit},
			   {ipv6, hoplimit},
			   {ipv6, recvtclass},
			   {ipv6, tclass}]) of
		       true ->
			   ok;
		       false ->
			   skip("None of the needed options are supported")
		   end,
		   %% The problem here is hoplimit on darwin 9.8.0,
		   %% but I can't be bothered to adjust the test case,
		   %% just skip on that machine (there is only one)...
		   is_good_enough_darwin({9,8,0}),
                   is_good_enough_montavista("4.0.1")
           end,
           fun() ->
		   %% If we get this far, we *know* that at least one of the
		   %% options are available.

		   %% This is list of all the options and there resulting
		   %% control message header type(s):
		   %%   [{'ipv6 socket option', 'control message header type'}]
		   Opts =
		       case socket:is_supported(options, socket, timestamp) of
			   true ->
			       [{socket, timestamp, timestamp, default}];
			   false ->
			       []
		       end ++
		       case socket:is_supported(options, ipv6, recvpktinfo) of
			   true ->
			       [{ipv6, recvpktinfo, pktinfo, default}];
			   false ->
			       []
		       end ++
		       case socket:is_supported(options, ipv6, flowinfo) of
			   true ->
			       [{ipv6, flowinfo, flowinfo, default}];
			   false ->
			       []
		       end ++
		       case socket:is_supported(options, ipv6, recvhoplimit) of
			   true ->
			       [{ipv6, recvhoplimit, hoplimit, default}];
			   false ->
			       case socket:is_supported(options, ipv6, hoplimit) of
				   true ->
				       [{ipv6, hoplimit, hoplimit, default}];
				   false ->
				       []
			       end
		       end ++
		       case socket:is_supported(options, ipv6, recvtclass) of
			   true ->
			       [{ipv6, recvtclass, tclass, 42}];
			   false ->
			       case socket:is_supported(options, ipv6, tclass) of
				   true ->
				       [{ipv6, tclass, tclass, 42}];
				   false ->
				       []
			       end
		       end,

                   Enable = fun(Sock, Level, Opt) ->
				    ?SEV_IPRINT("try enable [~w] ~p", [Level, Opt]),
				    socket:setopt(Sock, Level, Opt, true)
                            end,
                   Send = fun(Sock, Data, Dest, []) ->
                                  Msg = #{addr => Dest,
                                             iov  => [Data]},
                                  socket:sendmsg(Sock, Msg);
			     (Sock, Data, Dest, Hdrs) when is_list(Hdrs) ->
				  CMsgs = [#{level => Level,
						type  => Type,
						data  => Val} ||
						 {Level, Type, Val} <- Hdrs],
                                  Msg   = #{addr => Dest,
					       ctrl => CMsgs,
					       iov  => [Data]},
                                  socket:sendmsg(Sock, Msg)
                          end,
                   Recv = fun(Sock) ->
                                  case socket:recvmsg(Sock) of
                                      {ok, #{addr := Source,
                                             ctrl := CMsgs,
                                             iov  := [Data]}} ->
                                          {ok, {Source, CMsgs, Data}};
                                      {error, _} = ERROR ->
                                          ERROR
                                  end
                          end,
                   InitState = #{domain => inet6,
                                 proto  => udp,
				 opts   => Opts,
                                 send   => Send,
                                 recv   => Recv,
                                 enable => Enable},
                   ok = api_opt_ipv6_mopts_udp(InitState)
           end).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_ipv6_mopts_udp(InitState) ->
    Seq = 
        [
         #{desc => "local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa_src => LSA,
                                       lsa_dst => LSA}}
                   end},

         #{desc => "open src socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_src => Sock}}
                   end},
         #{desc => "bind src",
           cmd  => fun(#{sock_src := Sock, lsa_src := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           SASrc = sock_sockname(Sock),
                           ?SEV_IPRINT("src sockaddr: "
                                       "~n   ~p", [SASrc]),
                           {ok, State#{sa_src => SASrc}}
                   end},

         #{desc => "open dst socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           Sock = sock_open(Domain, dgram, Proto),
                           {ok, State#{sock_dst => Sock}}
                   end},
         #{desc => "bind dst",
           cmd  => fun(#{sock_dst := Sock, lsa_dst := LSA}) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   ?SEV_IPRINT("src bound"),
                                   ok;
                               {error, Reason} = ERROR ->
                                   ?SEV_EPRINT("src bind failed: ~p", [Reason]),
                                   ERROR
                           end
                   end},
         #{desc => "sockname dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           SADst = sock_sockname(Sock),
                           ?SEV_IPRINT("dst sockaddr: "
                                       "~n   ~p", [SADst]),
                           {ok, State#{sa_dst => SADst}}
                   end},

         #{desc => "enable options on dst socket",
           cmd  => fun(#{sock_dst := DSock,
			 sock_src := SSock,
			 opts     := Opts,
			 enable   := Enable} = _State) ->
			   %% If we fail to enable *any* of the options,
			   %% we give up.
			   E = fun({Level, Opt, _, _}) -> 
                                       case Enable(DSock, Level, Opt) of
                                           ok ->
                                               ?SEV_IPRINT("dst [~w] ~w enabled",
                                                           [Level, Opt]),
                                               ok;
                                           {error, enoprotoopt = Reason} ->
                                               ?SEV_EPRINT("Expected "
                                                           "Failure: "
                                                           "~p => SKIP",
                                                           [Reason]),
                                               (catch socket:close(DSock)),
                                               (catch socket:close(SSock)),
                                               {skip, Reason};
                                           {error, Reason} = ERROR ->
                                               ?SEV_EPRINT("Failed "
                                                           "setting ~w:"
                                                           "   ~p",
                                                           [Opt, Reason]),
                                               throw(ERROR)
                                       end
			       end,
			   lists:foreach(E, Opts),
			   ok
                   end},

         #{desc => "send req (to dst)",
           cmd  => fun(#{sock_src := Sock,
			 sa_dst   := Dst,
			 opts     := Opts,
			 send     := Send}) ->
			   Hdrs = [{Level, Type, Data} ||
				      {Level, _, Type, Data} <- 
					  Opts, (Data =/= default)],
                           Send(Sock, ?BASIC_REQ, Dst, Hdrs)
                   end},
         #{desc => "recv req (from src)",
           cmd  => fun(#{sock_dst := Sock,
			 sa_src   := Src,
			 recv     := Recv,
			 opts     := Opts}) ->
                           case Recv(Sock) of
                               {ok, {Src, CMsgs, ?BASIC_REQ}}
                                 when length(CMsgs) =:= length(Opts)  ->
                                   ?SEV_IPRINT("Got (expected) cmsg headers: "
					       "~n   ~p", [CMsgs]),
				   %% We should really verify the headers:
				   %% values, types and so on...
                                   ok;
                               {ok, {BadSrc, BadCHdrs, BadReq} = UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Recv Source:   ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Recv CHdrs:    ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Recv Msg:      ~p",
                                               [Src, BadSrc,
                                                [{Level, Type} ||
                                                    {Level, _, Type, _} <- Opts],
						BadCHdrs,
						?BASIC_REQ, BadReq]),
                                   {error, {unexpected_data, UnexpData}};
                               {ok, UnexpData} ->
                                   ?SEV_EPRINT("Unexpected msg: "
                                               "~n   Expect Source: ~p"
                                               "~n   Expect CHdrs:  ~p"
                                               "~n   Expect Msg:    ~p"
                                               "~n   Unexp Data:    ~p",
                                               [Src,
                                                [{Level, Type} ||
                                                    {Level, _, Type, _} <- Opts],
						?BASIC_REQ,
                                                UnexpData]),
                                   {error, {unexpected_data, UnexpData}};
                               {error, _} = ERROR ->
                                   %% At the moment there is no way to get
                                   %% status or state for the socket...
                                   ERROR
                           end
                   end},

         #{desc => "close src socket",
           cmd  => fun(#{sock_src := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_src, State)}
                   end},
         #{desc => "close dst socket",
           cmd  => fun(#{sock_dst := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock_dst, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],
    Evaluator = ?SEV_START("tester", Seq, InitState),
    ok = ?SEV_AWAIT_FINISH([Evaluator]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the congestion tcp socket option.
%%
%% According to the man page (on linux) for this option it *should* be
%% possible to both get and set *allowed* algorithms. But when we attempt
%% to set, we get 'enoent'.
%% Accoring to /proc/sys/net/ipv4/tcp_allowed_congestion_control that
%% allgorithm was allowed, so...
%% For now, we only test that we can get (it could be a bug in our code)

api_opt_tcp_congestion_tcp4(suite) ->
    [];
api_opt_tcp_congestion_tcp4(doc) ->
    [];
api_opt_tcp_congestion_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_tcp_congestion_tcp4,
           fun() -> has_support_tcp_congestion() end,
           fun() ->
                   Set  = fun(Sock, Value) when is_list(Value) ->
                                  socket:setopt(Sock, tcp, congestion, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, tcp, congestion)
                          end,
                   InitState = #{domain => inet,
                                 proto  => tcp,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_tcp_congestion_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_tcp_congestion_tcp(InitState) ->
    process_flag(trap_exit, true),
    ServerSeq =
        [
         %% *** Wait for start order ***
         #{desc => "await start (from tester)",
           cmd  => fun(State) ->
                           Tester = ?SEV_AWAIT_START(),
                           {ok, State#{tester => Tester}}
                   end},
         #{desc => "monitor tester",
           cmd  => fun(#{tester := Tester}) ->
                           _MRef = erlang:monitor(process, Tester),
                           ok
                   end},

         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := LSock, lsa := LSA} = _State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   ?SEV_IPRINT("bound to port: ~w", [Port]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (init)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, init),
                           ok
                   end},


         %% The actual test
         #{desc => "await continue (get congestion)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_AWAIT_CONTINUE(Tester, tester, get_congestion)
                   end},
         #{desc => "get congestion",
           cmd  => fun(#{sock := Sock, get := Get} = State) ->
                           case Get(Sock) of
                               {ok, Algorithm} ->
                                   ?SEV_IPRINT("algorithm: ~s", [Algorithm]),
                                   {ok, State#{alg => Algorithm}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "announce ready (get congestion)",
           cmd  => fun(#{tester := Tester}) ->
                           ?SEV_ANNOUNCE_READY(Tester, get_congestion),
                           ok
                   end},


         %% *** Termination ***
         #{desc => "await terminate",
           cmd  => fun(#{tester := Tester} = State) ->
                           case ?SEV_AWAIT_TERMINATE(Tester, tester) of
                               ok ->
                                   {ok, maps:remove(tester, State)};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "close connection socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "monitor server",
           cmd  => fun(#{server := Pid} = _State) ->
                           _MRef = erlang:monitor(process, Pid),
                           ok
                   end},

         %% Start the server
         #{desc => "order server start",
           cmd  => fun(#{server := Pid} = _State) ->
                           ?SEV_ANNOUNCE_START(Pid),
                           ok
                   end},
         #{desc => "await server ready (init)",
           cmd  => fun(#{server := Pid} = _State) ->
                           ok = ?SEV_AWAIT_READY(Pid, server, init),
                           ok
                   end},

         %% *** The actual test ***
         #{desc => "order server to continue (with get-congestion)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_CONTINUE(Server, get_congestion),
                           ok
                   end},
         #{desc => "await server ready (get-congestion)",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_AWAIT_READY(Server, server, get_congestion)
                   end},


         %% *** Termination ***
         #{desc => "order server to terminate",
           cmd  => fun(#{server := Server} = _State) ->
                           ?SEV_ANNOUNCE_TERMINATE(Server),
                           ok
                   end},
         #{desc => "await server termination",
           cmd  => fun(#{server := Server} = State) ->
                           ?SEV_AWAIT_TERMINATION(Server),
                           State1 = maps:remove(server, State),
                           {ok, State1}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],


    i("start server evaluator"),
    Server = ?SEV_START("server", ServerSeq, InitState),

    i("start tester evaluator"),
    TesterInitState = #{server => Server#ev.pid},
    Tester = ?SEV_START("tester", TesterSeq, TesterInitState),

    ok = ?SEV_AWAIT_FINISH([Server, Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the cork tcp socket option.
%%
%% This is a very simple test. We simple set and get the value.
%% To test that it has an effect is just "to much work"...
%%
%% Reading the man page it seems like (on linux) that the
%% value resets itself after some (short) time...

api_opt_tcp_cork_tcp4(suite) ->
    [];
api_opt_tcp_cork_tcp4(doc) ->
    [];
api_opt_tcp_cork_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_tcp_cork_tcp4,
           fun() -> has_support_tcp_cork() end,
           fun() ->
                   Set  = fun(Sock, Value) when is_boolean(Value) ->
                                  socket:setopt(Sock, tcp, cork, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, tcp, cork)
                          end,
                   InitState = #{domain => inet,
                                 proto  => tcp,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_tcp_cork_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_tcp_cork_tcp(InitState) ->
    process_flag(trap_exit, true),
    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := LSock, lsa := LSA} = _State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   ?SEV_IPRINT("bound to port: ~w", [Port]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% The actual test
         #{desc => "get (default) cork (= false)",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("cork default: ~p", [Value]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "enable cork (=> true)",
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("cork enabled"),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "get cork (= true)",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, true = Value} ->
                                   ?SEV_IPRINT("cork: ~p", [Value]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% *** Termination ***
         #{desc => "close connection socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start tester evaluator"),
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    ok = ?SEV_AWAIT_FINISH([Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the maxseg tcp socket option.
%%
%% This is a very simple test. We simple set and get the value.
%% To test that it has an effect is just "to much work"...
%%
%% Note that there is no point in reading this value back,
%% since the kernel imposes its own rules with regard
%% to what is an acceptible value.
%%

api_opt_tcp_maxseg_tcp4(suite) ->
    [];
api_opt_tcp_maxseg_tcp4(doc) ->
    [];
api_opt_tcp_maxseg_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_tcp_maxseg_tcp4,
           fun() -> has_support_tcp_maxseg() end,
           fun() ->
                   Set  = fun(Sock, Value) when is_integer(Value) ->
                                  socket:setopt(Sock, tcp, maxseg, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, tcp, maxseg)
                          end,
                   InitState = #{domain => inet,
                                 proto  => tcp,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_tcp_maxseg_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_tcp_maxseg_tcp(InitState) ->
    process_flag(trap_exit, true),
    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := LSock, lsa := LSA} = _State) ->
                           case sock_bind(LSock, LSA) of
                               ok ->
                                   Port = sock_port(LSock),
                                   ?SEV_IPRINT("bound to port: ~w", [Port]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% The actual test
         #{desc => "get (default) maxseg",
           cmd  => fun(#{sock := Sock, get := Get} = State) ->
                           case Get(Sock) of
                               {ok, DefMaxSeg} ->
                                   ?SEV_IPRINT("maxseg default: ~p", [DefMaxSeg]),
                                   {ok, State#{def_maxseg => DefMaxSeg}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         
         %% Note that there is no point in reading this value back,
         %% since the kernel imposes its own rules with regard
         %% to what is an acceptible value.
         %% So, even if the set operation is a success, the value
         %% still might not have changed.
         %%
         %% Note that not all platforms allow this to be set!
         %% Since this is the *last* operation in the test sequence
         %% (before termination) we also accept error reason = einval
         %% as success (rather then skip).
         %% The same goes for the error reason = enoprotoopt (Solaris).
         #{desc => "(maybe) change maxseg (default + 16)",
           cmd  => fun(#{sock       := Sock,
                         set        := Set,
                         def_maxseg := DefMaxSeg} = _State) ->
                           NewMaxSeg = DefMaxSeg + 16,
                           case Set(Sock, NewMaxSeg) of
                               ok ->
                                   ?SEV_IPRINT("maxseg (maybe) changed (to ~w)",
                                               [NewMaxSeg]),
                                   ok;
                               {error, Reason} when (Reason =:= einval) orelse
                                                    (Reason =:= enoprotoopt) ->
                                   ?SEV_IPRINT("change not allowed (~w)",
                                               [Reason]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},

         %% *** Termination ***
         #{desc => "close connection socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start tester evaluator"),
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    ok = ?SEV_AWAIT_FINISH([Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the nodelay tcp socket option.
%%
%% This is a very simple test. We simple set and get the value.
%% To test that it has an effect is just "to much work"...

api_opt_tcp_nodelay_tcp4(suite) ->
    [];
api_opt_tcp_nodelay_tcp4(doc) ->
    [];
api_opt_tcp_nodelay_tcp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_tcp_nodelay_tcp4,
           fun() -> has_support_tcp_nodelay() end,
           fun() ->
                   Set  = fun(Sock, Value) when is_boolean(Value) ->
                                  socket:setopt(Sock, tcp, nodelay, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, tcp, nodelay)
                          end,
                   InitState = #{domain => inet,
                                 proto  => tcp,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_tcp_nodelay_tcp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_tcp_nodelay_tcp(InitState) ->
    process_flag(trap_exit, true),
    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, stream, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, lsa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   ?SEV_IPRINT("bound to port: ~w", [Port]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% The actual test
         #{desc => "get (default) nodelay (= false)",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("nodelay default: ~p", [Value]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "enable nodelay (=> true)",
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("nodelay enabled"),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "get nodelay (= true)",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, true = Value} ->
                                   ?SEV_IPRINT("nodelay: ~p", [Value]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% *** Termination ***
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start tester evaluator"),
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    ok = ?SEV_AWAIT_FINISH([Tester]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Tests that the cork udp socket option.
%%
%% This is a very simple test. We simple set and get the value.
%% To test that it has an effect is just "to much work"...
%%

api_opt_udp_cork_udp4(suite) ->
    [];
api_opt_udp_cork_udp4(doc) ->
    [];
api_opt_udp_cork_udp4(_Config) when is_list(_Config) ->
    ?TT(?SECS(5)),
    tc_try(api_opt_udp_cork_udp4,
           fun() -> has_support_udp_cork() end,
           fun() ->
                   Set  = fun(Sock, Value) when is_boolean(Value) ->
                                  socket:setopt(Sock, udp, cork, Value)
                          end,
                   Get  = fun(Sock) ->
                                  socket:getopt(Sock, udp, cork)
                          end,
                   InitState = #{domain => inet,
                                 proto  => udp,
                                 set    => Set,
                                 get    => Get},
                   ok = api_opt_udp_cork_udp(InitState)
           end).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

api_opt_udp_cork_udp(InitState) ->
    process_flag(trap_exit, true),
    TesterSeq =
        [
         %% *** Init part ***
         #{desc => "which local address",
           cmd  => fun(#{domain := Domain} = State) ->
                           LSA = which_local_socket_addr(Domain),
                           {ok, State#{lsa => LSA}}
                   end},
         #{desc => "create socket",
           cmd  => fun(#{domain := Domain,
                         proto  := Proto} = State) ->
                           case socket:open(Domain, dgram, Proto) of
                               {ok, Sock} ->
                                   {ok, State#{sock => Sock}};
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "bind to local address",
           cmd  => fun(#{sock := Sock, lsa := LSA} = _State) ->
                           case sock_bind(Sock, LSA) of
                               ok ->
                                   Port = sock_port(Sock),
                                   ?SEV_IPRINT("bound to port: ~w", [Port]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% The actual test
         #{desc => "get (default) cork (= false)",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, false = Value} ->
                                   ?SEV_IPRINT("cork default: ~p", [Value]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "enable cork (=> true)",
           cmd  => fun(#{sock := Sock, set := Set} = _State) ->
                           case Set(Sock, true) of
                               ok ->
                                   ?SEV_IPRINT("cork enabled"),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},
         #{desc => "get cork (= true)",
           cmd  => fun(#{sock := Sock, get := Get} = _State) ->
                           case Get(Sock) of
                               {ok, true = Value} ->
                                   ?SEV_IPRINT("cork: ~p", [Value]),
                                   ok;
                               {error, _} = ERROR ->
                                   ERROR
                           end
                   end},


         %% *** Termination ***
         #{desc => "close socket",
           cmd  => fun(#{sock := Sock} = State) ->
                           ok = socket:close(Sock),
                           {ok, maps:remove(sock, State)}
                   end},

         %% *** We are done ***
         ?SEV_FINISH_NORMAL
        ],

    i("start tester evaluator"),
    Tester = ?SEV_START("tester", TesterSeq, InitState),

    ok = ?SEV_AWAIT_FINISH([Tester]).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                                                                     %%
%%                  API OPERATIONS WITH TIMEOUT                        %%
%%                                                                     %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% This test case is intended to test the connect timeout option
%% on an IPv4 TCP (stream) socket.
api_to_connect_tcp4(suite) ->
    [