/*
 * Copyright 2021 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef LYRA_CODEC_PACKET_LOSS_HANDLER_H_
#define LYRA_CODEC_PACKET_LOSS_HANDLER_H_

#include <memory>
#include <vector>

#include "absl/types/optional.h"
#include "noise_estimator_interface.h"
#include "packet_loss_handler_interface.h"
#include "spectrogram_predictor_interface.h"

namespace chromemedia {
namespace codec {

// When the decoder is missing a packet, PacketLossHandler either
// returns a generated prediction of the next packet or noise, based on the
// number of consecutive packets missing.
class PacketLossHandler : public PacketLossHandlerInterface {
 public:
  static std::unique_ptr<PacketLossHandler> Create(int sample_rate_hz,
                                                   int num_features,
                                                   float seconds_per_frame);

  // Called for every new packet received by the decoder. Updates both the
  // Spectrogram Predictor and the Noise Estimator with the received features.
  // Returns true on success and false on failure.
  bool SetReceivedFeatures(const std::vector<float>& features) override;

  // Provides estimated features from which a packet can be reconstructed.
  // Requires knowing the number of samples that the decoder is looking to
  // reconstruct in order to keep track of how many samples have been lost.
  // Estimated features are returned from the Spectrogram Predictor if
  // |num_samples| is below the maximum allowed and are returned from the Noise
  // Estimator otherwise. Returns a nullopt if |num_samples| is out of bounds.
  absl::optional<std::vector<float>> EstimateLostFeatures(
      int num_samples) override;

  // Returns true if the last returned features are generated by the background
  // noise estimator.
  bool is_comfort_noise() const override;

 private:
  explicit PacketLossHandler(
      int sample_rate_hz,
      std::unique_ptr<NoiseEstimatorInterface> noise_estimator,
      std::unique_ptr<SpectrogramPredictorInterface> spectrogram_predictor);

  int consecutive_lost_samples_;
  int max_lost_samples_;
  const std::unique_ptr<NoiseEstimatorInterface> noise_estimator_;
  std::unique_ptr<SpectrogramPredictorInterface> spectrogram_predictor_;

  friend class PacketLossHandlerPeer;
};
}  // namespace codec
}  // namespace chromemedia

#endif  // LYRA_CODEC_PACKET_LOSS_HANDLER_H_
