# Command Hook Examples

## Overview

Command hooks in Backrest enable you to execute shell commands in response to specific lifecycle events, extending Backrest's functionality. This cookbook provides practical examples of command hooks for various use cases.

## Error Behavior Configuration

When using command hooks with `CONDITION_SNAPSHOT_START`, you can control how Backrest responds to script exit statuses:

| Behavior          | Description                                                             |
| ----------------- | ----------------------------------------------------------------------- |
| `ON_ERROR_CANCEL` | Cancels the backup operation if script exits with non-zero status       |
| `ON_ERROR_FATAL`  | Treats non-zero exit as backup failure and triggers error notifications |
| `ON_ERROR_IGNORE` | Continues backup operation regardless of script exit status             |

## Unix/MacOS Examples

### Health Monitoring

#### Healthcheck Service Integration
Notify a health monitoring service (e.g., healthchecks.io) about backup status.

**Event:** `CONDITION_SNAPSHOT_END`  
**Error Behavior:** `ON_ERROR_IGNORE`

```bash
#!/bin/bash
{{ if .Error -}}
curl -fsS --retry 3 https://hc-ping.com/your-uuid/fail
{{ else -}}
curl -fsS --retry 3 https://hc-ping.com/your-uuid
{{ end -}}
```

### System Notifications

#### MacOS System Notifications
Display system notifications for backup events.

**Events:** `CONDITION_SNAPSHOT_END`, `CONDITION_PRUNE_ERROR`, `CONDITION_CHECK_ERROR`  
**Error Behavior:** `ON_ERROR_IGNORE`

```bash
#!/bin/bash
{{ if .Error -}}
osascript -e 'display notification "{{ .ShellEscape .Task }} failed" with title "Backrest"'
{{ else -}}
osascript -e 'display notification "{{ .ShellEscape .Task }} succeeded" with title "Backrest"'
{{ end -}}
```

### Pre-backup Checks

#### Internet Connectivity Check
Verify internet connection before starting backup.

**Event:** `CONDITION_SNAPSHOT_START`  
**Error Behavior:** `ON_ERROR_CANCEL`

```bash
#!/bin/bash
if ping -q -c 1 -W 1 google.com >/dev/null; then
  echo "Internet connection is up"
  exit 0
else
  echo "Internet connection is down"
  exit 1
fi
```

#### Directory Existence Check
Ensure backup target directory exists.

**Event:** `CONDITION_SNAPSHOT_START`  
**Error Behavior:** `ON_ERROR_CANCEL`

```bash
#!/bin/bash
if [ -d /path/to/backup ]; then
  echo "Backup directory exists"
  exit 0
else
  echo "Backup directory does not exist"
  exit 1
fi
```

#### Battery Level Check
Verify sufficient battery level before backup.

**Event:** `CONDITION_SNAPSHOT_START`  
**Error Behavior:** `ON_ERROR_CANCEL`

```bash
#!/bin/bash
if [ $(cat /sys/class/power_supply/BAT0/capacity) -gt 80 ]; then
  echo "Battery level is above 20%"
  exit 0
else
  echo "Battery level is below 20%"
  exit 1
fi
```

### Filesystem Operations

#### BTRFS Snapshot Management
Create and manage BTRFS snapshots for consistent backups.

**Pre-backup Snapshot**  
**Event:** `CONDITION_SNAPSHOT_START`  
**Error Behavior:** `ON_ERROR_FATAL`

```bash
#!/bin/bash
btrfs subvolume snapshot -r /your-btrfs-filesystem /your-btrfs-filesystem/.backrest-snapshot
```

**Post-backup Cleanup**  
**Event:** `CONDITION_SNAPSHOT_END`  
**Error Behavior:** `ON_ERROR_IGNORE`

```bash
#!/bin/bash
btrfs subvolume delete /your-btrfs-filesystem/.backrest-snapshot
```

## Windows Examples

### GUI Notifications

#### Error Message Box
Display persistent error messages requiring user acknowledgment.

**Event:** `CONDITION_ANY_ERROR`

```sh
Add-Type -AssemblyName System.Windows.Forms
$options = [System.Windows.Forms.MessageBoxOptions]::ServiceNotification
$defbutton = [System.Windows.Forms.MessageBoxDefaultButton]::Button1
$buttons = [System.Windows.Forms.MessageBoxButtons]::OK
$icon = [System.Windows.Forms.MessageBoxIcon]::Error
$title = "Backrest"
$message = '{{ .Summary }}'
[System.Windows.Forms.MessageBox]::Show($message, $title, $buttons, $icon, $defbutton, $options)
```

#### Warning Toast Notification
Show temporary warning notifications.

**Event:** `CONDITION_SNAPSHOT_WARNING`

```sh
Add-Type -AssemblyName System.Windows.Forms
$balloon = New-Object System.Windows.Forms.NotifyIcon
$balloon.Icon = [System.Drawing.SystemIcons]::Warning
$balloon.BalloonTipIcon = [System.Windows.Forms.ToolTipIcon]::Warning
$balloon.BalloonTipText = '{{ .Summary }}'
$balloon.BalloonTipTitle = "Backrest"
$balloon.Visible = $true
$balloon.ShowBalloonTip(5000)
Start-Sleep -Seconds(5)
$balloon.Visible = $false
$balloon.Icon.Dispose()
$balloon.Dispose()
```

#### Success Toast Notification
Display temporary success notifications.

**Event:** `CONDITION_SNAPSHOT_SUCCESS`

```sh
Add-Type -AssemblyName System.Windows.Forms
$balloon = New-Object System.Windows.Forms.NotifyIcon
$balloon.Icon = [System.Drawing.SystemIcons]::Information
$balloon.BalloonTipIcon = [System.Windows.Forms.ToolTipIcon]::Info
$balloon.BalloonTipText = '{{ .Summary }}'
$balloon.BalloonTipTitle = "Backrest"
$balloon.Visible = $true
$balloon.ShowBalloonTip(5000)
Start-Sleep -Seconds(5)
$balloon.Visible = $false
$balloon.Icon.Dispose()
$balloon.Dispose()
```
