defmodule Stemmer.Step4 do
  alias Stemmer.Rules

  def apply(word) do
    word
    |> remove_suffix_in_r2()
  end

  @doc """
  Search for the longest among the following suffixes, and, if found and in R2,
  perform the action indicated.

  ## Examples

      iex> Stemmer.Step4.remove_suffix_in_r2("national")
      "nation"

      iex> Stemmer.Step4.remove_suffix_in_r2("association")
      "associat"

      iex> Stemmer.Step4.remove_suffix_in_r2("apprehension")
      "apprehens"

      iex> Stemmer.Step4.remove_suffix_in_r2("concepcion")
      "concepcion"

      iex> Stemmer.Step4.remove_suffix_in_r2("addition")
      "addit"

      iex> Stemmer.Step4.remove_suffix_in_r2("agreement")
      "agreement"
  """
  def remove_suffix_in_r2(word) do
    {_, word} =
      with {:next, _word} <- remove_suffix(word),
           {:next, _word} <- remove_suffix_ion(word)
        do {:found, word}
      end

    word
  end

  defp remove_suffix(word) do
    r_suffix = ~r/(al|ance|ence|er|ic|able|ible|ant|ement|ment|ent|ism|ate|iti|ous|ive|ize)$/

    if matches = Regex.run(r_suffix, word) do
      suffix = List.first(matches)
      remove_suffix_in_r2(word, suffix)
    else
      {:next, word}
    end
  end

  defp remove_suffix_in_r2(word, suffix) do
    if Rules.r2(word) =~ suffix do
      {:found, String.replace_suffix(word, suffix, "")}
    else
      {:next, word}
    end
  end

  defp remove_suffix_ion(word) do
    if Rules.r2(word) =~ ~r/ion$/ && word =~ ~r/(s|t)ion$/ do
      {:found, String.replace_suffix(word, "ion", "")}
    else
      {:next, word}
    end
  end
end
