# Быстрый старт

Electron позволяет Вам делать приложения для рабочего стола на чистом JavaScript,
предоставляя среду с богатым API. Можете представлять его как Node.js приложение, которое
ориентировано для рабочего стола, а не для веб сервера.

Однако это не значит, что Electron — лишь привязка к GUI билиотекам. На деле
Electron использует веб-страницы как интерфейс, так что Вы можете считать его
небольшим Chroumium браузером, который контролируется с помощью JavaScript.

### Главный процесс

В Electron процесс, который запускает `main` из `package.json` называется
__главным процессом__. Скрипт, который работает в главном процессе может
показывать GUI, создавая веб-страницы.

### Процесс-рендерер

Так как Electron использует Chromium для показа веб-страниц,
мульти-процессовая архитектура показа страниц Chromium тоже используется.
Каждая веб-страница в Electron работает в своём собственном процессе,
который называется  __процесс-рендерер__.

В обычных браузерах веб-страницы обычно запускаются в "песочнице" и им недоступны
реальные ресурсы компьютера. Пользователи Electron напротив могут использовать API
Node.js на страницах, что допускает более низкоуровневую работу с операционной системой.

### Разница между главным процессом и процессом-рендерером

Главный процесс создаёт веб-страницы используя `BrowserWindow`. Каждый экземпляр
`BrowserWindow` показывает веб-страницу через свой собственный процесс-рендерер.
Если экземпляр `BrowserWindow` уничтожается, то и соответствующий процесс-рендерер тоже
завершается.

Главный процесс управляет всеми веб-страницами и соответствующими им процессами-редерерами.
Каждый процесс-рендерер изолирован и управляет только своей веб-страницей.

На веб-страницах не позволяется вызывать нативные API, которые управляют GUI,
потому что это очень опасно и может легко вызвать утечку ресурсов. Если вы хотите
выполнить действия с GUI на странице, процесс-рендерер этой страницы должен
"попросить" главный процесс сделать эти действия.

В Electron есть несолько способов общения между процессам. Например, модули
[`ipcRenderer`](../api/ipc-renderer.md) и [`ipcMain`](../api/ipc-main.md) используются
для отправки сообщений, а [remote](../api/remote.md) - для коммуникации в RPC стиле.
В FAQ также есть пункт о том, [как разделять информацию между страницами][share-data]

## Первое приложение на Electron

Как правило, приложение Electron структурировано следующим образом:

```text
your-app/
├── package.json
├── main.js
└── index.html
```

Формат `package.json` точно такой же, как у модулей Node и скрипт, объявленый
как `main`, будет выполняться при запуске вашего приложения, работая в
главном процессе. Например, Ваш `package.json` может выглядеть вот так:

```javascripton
{
  "name"    : "your-app",
  "version" : "0.1.0",
  "main"    : "main.js"
}
```

__Заметка__: Если поле `main` отсутствует в `package.json`, Electron попробует
загрузить `index.js`.


`main.js` должен создавать окно и управлять системными событиями,
типичный пример:

```javascript
const electron = require('electron')
// Модуль, контролирующий основное: сам Electron.
const app = electron.app
// Модуль, создающий окно приложения.
const BrowserWindow = electron.BrowserWindow

// Удерживайте глобальное обращение к объекту окна, если Вы так не сделаете, то
// окно само закроется после того, как объект будет собран сборщиком мусора.
let mainWindow

function createWindow () {
  // Создаём окно браузера
  mainWindow = new BrowserWindow({width: 800, height: 600})

  // и загружаем index.html приложения.
  mainWindow.loadURL(`file://${__dirname}/index.html`)

  // Открываем DevTools.
  mainWindow.webContents.openDevTools()

  // Будет выполнено, когда пользователь закроет окно
  mainWindow.on('closed', function () {
    // Убрать обращение на объект окна, обычно стоит хранить окна в массиве,
    // если ваше приложение поддерживает несколько, сейчас стоит удалить
    // соответствующий элемент.
    mainWindow = null
  })
}

// Этот метод будет вызван, когда Electron закончит инициализацию
// и будет готов создавать окна браузера.
// Некоторые API возможно использовать только после того, как
// это произойдёт.
app.on('ready', createWindow)

// Выйти, если все окна закрыты
app.on('window-all-closed', function () {
  // На macOS приложение и его строка меню обычно остаются активными,
  // пока пользователь не завершит их с помощью `Cmd + Q`.
  if (process.platform !== 'darwin') {
    app.quit()
  }
})

app.on('activate', function () {
  // На macOS приложение обычно пересоздаёт окно, когда
  // пользователь кликает на его иконку в доке, если не открыто
  // других окон.
  if (mainWindow === null) {
    createWindow()
  }
})

// В этот файл Вы можете включить остальной код вашего главного процесса.
// Вы также можете разложить его по отдельным файлам и подключить с помощью require.

```

Наконец, `index.html`, страница, которую Вы хотите показать:

```html
<!DOCTYPE html>
<html>
  <head>
    <meta charset="UTF-8">
    <title>Hello World!</title>
  </head>
  <body>
    <h1>Привет, мир!</h1>
    Мы используем Node <script>document.write(process.versions.node)</script>,
    Chrome <script>document.write(process.versions.chrome)</script>,
    и Electron <script>document.write(process.versions.electron)</script>.
  </body>
</html>
```

## Запуск вашего приложения

После того как Вы создали `main.js`, `index.html` и `package.json` Вам скорее всего захочется
запустить приложение, чтобы проверить, что оно работает так, как надо.

### electron-prebuilt

[`electron-prebuilt`](https://github.com/electron-userland/electron-prebuilt) — `npm` модуль,
который содержит прекомпилированную версию Electron.

Если вы установили Electron глобально через `npm`, то Вам нужно будет всего лишь
запустить сдедующее в папке вашего проекта:

```bash
electron .
```

Если Вы установили Electron локально, то выполните это:

```bash
./node_modules/.bin/electron .
```

### Исполняемые файлы Electron, скачанные вручную

Если Вы скачали Electron вручную, то Вы можете использовать
исполняемые файлы прямо в папке вашего проекта.

#### Windows

```bash
$ .\electron\electron.exe your-app\
```

#### Linux

```bash
$ ./electron/electron your-app/
```

#### macOS

```bash
$ ./Electron.app/Contents/MacOS/Electron your-app/
```

`Electron.app` — часть реализного пакета Electron, Вы можете скачать его
[тут](https://github.com/electron/electron/releases).

### Запустить как дистрибутив

Когда Вы закончили написание вашего приложения, Вы можете создать
дистрибутив, следуя инструкциям [отсюда](./application-distribution.md) и
затем запустить полученное приложение.

### Попробуйте этот пример

Скопируйте и запустите этот обучающий код, используя репозиторий [`atom/electron-quick-start`](https://github.com/electron/electron-quick-start)

**Заметка**: Для запуска требуется [Git](https://git-scm.com) и [Node.js](https://nodejs.org/en/download/) (который включает в себя [npm](https://npmjs.org)).

```bash
# Клонируем репозиторий
$ git clone https://github.com/electron/electron-quick-start
# Переходим в папку скачанного репозитория
$ cd electron-quick-start
# Устанавливаем зависимости и запускаем
$ npm install && npm start
```

[share-data]: ../faq/electron-faq.md#how-to-share-data-between-web-pages
