# Copyright 2013-2019 Lawrence Livermore National Security, LLC and other
# Spack Project Developers. See the top-level COPYRIGHT file for details.
#
# SPDX-License-Identifier: (Apache-2.0 OR MIT)

from spack import *


class Subversion(AutotoolsPackage):
    """Apache Subversion - an open source version control system."""

    homepage = 'https://subversion.apache.org/'
    url      = 'https://archive.apache.org/dist/subversion/subversion-1.12.2.tar.gz'

    version('1.12.2', sha256='f4927d6603d96c5ddabebbafe9a0f6833c18a891ff0ce1ea6ffd186ce9bc21f3')
    version('1.9.7',  sha256='c72a209c883e20245f14c4e644803f50ae83ae24652e385ff5e82300a0d06c3c')
    version('1.9.6',  sha256='a400cbc46d05cb29f2d7806405bb539e9e045b24013b0f12f8f82688513321a7')
    version('1.9.5',  sha256='280ba586c5d51d7b976b65d22d5e8e42f3908ed1c968d71120dcf534ce857a83')
    version('1.9.3',  sha256='74cd21d2f8a2a54e4dbd2389fe1605a19dbda8ba88ffc4bb0edc9a66e143cc93')
    version('1.8.17', sha256='1b2cb9a0ca454035e55b114ee91c6433b9ede6c2893f2fb140939094d33919e4')
    version('1.8.13', sha256='17e8900a877ac9f0d5ef437c20df437fec4eb2c5cb9882609d2277e2312da52c')

    variant('serf', default=True,  description='Serf HTTP client library')
    variant('perl', default=False, description='Build with Perl bindings')

    depends_on('apr')
    depends_on('apr-util')
    depends_on('zlib')
    depends_on('sqlite@3.8.2:')
    depends_on('expat')
    depends_on('lz4', when='@1.10:')
    depends_on('utf8proc', when='@1.10:')
    depends_on('serf', when='+serf')

    extends('perl', when='+perl')
    depends_on('swig@1.3.24:3.0.0', when='+perl')
    depends_on('perl-term-readkey', when='+perl')

    # Installation has race cases.
    parallel = False

    # http://www.linuxfromscratch.org/blfs/view/svn/general/subversion.html
    def configure_args(self):
        spec = self.spec
        args = [
            '--with-apr={0}'.format(spec['apr'].prefix),
            '--with-apr-util={0}'.format(spec['apr-util'].prefix),
            '--with-sqlite={0}'.format(spec['sqlite'].prefix),
            '--with-expat={0}:{1}:{2}'.format(
                spec['expat'].headers.directories[0],
                spec['expat'].libs.directories[0],
                spec['expat'].libs.names[0]
            ),
            '--with-zlib={0}'.format(spec['zlib'].prefix),
            '--without-apxs',
            '--without-trang',
            '--without-doxygen',
            '--without-berkeley-db',
            '--without-sasl',
            '--without-libmagic',
            '--without-kwallet',
            '--without-jdk',
            '--without-boost',
        ]

        if spec.satisfies('@1.10:'):
            args.extend([
                '--with-lz4={0}'.format(spec['lz4'].prefix),
                '--with-utf8proc={0}'.format(spec['utf8proc'].prefix),
            ])

        if '+serf' in spec:
            args.append('--with-serf={0}'.format(spec['serf'].prefix))
        else:
            args.append('--without-serf')

        if 'swig' in spec:
            args.append('--with-swig={0}'.format(spec['swig'].prefix))
        else:
            args.append('--without-swig')

        if '+perl' in spec:
            args.append('PERL={0}'.format(spec['perl'].command.path))

        return args

    def build(self, spec, prefix):
        make()
        if '+perl' in spec:
            make('swig-pl')
            with working_dir(join_path(
                    'subversion', 'bindings', 'swig', 'perl', 'native')):
                perl = spec['perl'].command
                perl('Makefile.PL', 'INSTALL_BASE={0}'.format(prefix))

    def test(self):
        make('check')
        if '+perl' in self.spec:
            make('check-swig-pl')

    def install(self, spec, prefix):
        make('install')
        if '+perl' in spec:
            make('install-swig-pl-lib')
            with working_dir(join_path(
                    'subversion', 'bindings', 'swig', 'perl', 'native')):
                make('install')
