# Copyright 2013-2019 Lawrence Livermore National Security, LLC and other
# Spack Project Developers. See the top-level COPYRIGHT file for details.
#
# SPDX-License-Identifier: (Apache-2.0 OR MIT)

from spack import *


class Nco(AutotoolsPackage):
    """The NCO toolkit manipulates and analyzes data stored in
    netCDF-accessible formats"""

    homepage = "http://nco.sourceforge.net/"
    url      = "https://github.com/nco/nco/archive/4.6.7.tar.gz"

    version('4.7.9', sha256='048f6298bceb40913c3ae433f875dea1e9129b1c86019128e7271d08f274a879')
    version('4.6.7', sha256='2fe2dabf14a60bface694307cbe719df57103682b715348e9d77bfe8d31487f3')
    version('4.6.6', sha256='079d83f800b73d9b12b8de1634a88c2cbe40a639aaf7bc056cd2e836c6047697')
    version('4.6.5', sha256='d5b18c9ada25d062a539e2995be445db39e8021c56cd4b20c88485cb2452c7ae')
    version('4.6.4', sha256='1c2ab906fc81f91bf8aff3e6da27ae7a4c89821c5836d787188fff5262418062')
    version('4.6.3', sha256='414ccb349ed25cb37b669fb87f9e2e4ca8d58c2f45538feda199bf895b982bf8')
    version('4.6.2', sha256='cec82e35d47a6bbf8ab9301d5ff4cf08051f489b49e8529ebf780380f2c21ed3')
    version('4.6.1', sha256='7433fe5901f48eb5170f24c6d53b484161e1c63884d9350600070573baf8b8b0')
    version('4.5.5', sha256='bc6f5b976fdfbdec51f2ebefa158fa54672442c2fd5f042ba884f9f32c2ad666')

    # https://github.com/nco/nco/issues/43
    patch('NUL-0-NULL.patch', when='@:4.6.7')

    variant('doc', default=False, description='Build/install NCO TexInfo-based documentation')

    # See "Compilation Requirements" at:
    # http://nco.sourceforge.net/#bld
    depends_on('netcdf')
    depends_on('antlr@2.7.7+cxx')  # required for ncap2
    depends_on('gsl')              # desirable for ncap2
    depends_on('udunits2')         # allows dimensional unit transformations

    depends_on('flex', type='build')
    depends_on('bison', type='build')
    depends_on('texinfo@4.12:', type='build', when='+doc')

    conflicts('%gcc@9:', when='@:4.7.8')

    def configure_args(self):
        spec = self.spec
        return ['--{0}-doc'.format('enable' if '+doc' in spec else 'disable')]

    def setup_environment(self, spack_env, run_env):
        spec = self.spec
        spack_env.set('NETCDF_INC', spec['netcdf'].prefix.include)
        spack_env.set('NETCDF_LIB', spec['netcdf'].prefix.lib)
        spack_env.set('ANTLR_ROOT', spec['antlr'].prefix)
        spack_env.set('UDUNITS2_PATH', spec['udunits2'].prefix)
