# Copyright 2013-2018 Lawrence Livermore National Security, LLC and other
# Spack Project Developers. See the top-level COPYRIGHT file for details.
#
# SPDX-License-Identifier: (Apache-2.0 OR MIT)

from spack import *
import os


class PyMatplotlib(PythonPackage):
    """matplotlib is a python 2D plotting library which produces publication
    quality figures in a variety of hardcopy formats and interactive
    environments across platforms."""

    homepage = "https://pypi.python.org/pypi/matplotlib"
    url      = "https://pypi.io/packages/source/m/matplotlib/matplotlib-2.0.2.tar.gz"

    version('3.0.0', '39c7f44c8fa0f24cbf684137371ce4ae')
    version('2.2.3', '403b0bddd751d71187416f20d4cff100')
    version('2.2.2', 'dd1e49e041309a7fd4e32be8bf17c3b6')
    version('2.0.2', '061111784278bde89b5d4987014be4ca')
    version('2.0.0', '7aa54b06327f0e1c4f3877fc2f7d6b17')
    version('1.5.3', 'ba993b06113040fee6628d74b80af0fd')
    version('1.5.1', 'f51847d8692cb63df64cd0bd0304fd20')
    version('1.4.3', '86af2e3e3c61849ac7576a6f5ca44267')
    version('1.4.2', '7d22efb6cce475025733c50487bd8898')

    # See: http://matplotlib.org/users/installing.html

    # Variants enabled by default for a standard configuration
    variant('tk', default=True, description='Enable Tk GUI')
    variant('image', default=True,
            description='Enable reading/saving JPEG, BMP and TIFF files')

    # Variants optionally available to user
    variant('ipython', default=False, description='Enable ipython support')
    variant('qt', default=False, description='Enable Qt GUI')
    variant('latex', default=False,
            description='Enable LaTeX text rendering support')
    variant('animation', default=False,
            description='Enable animation support')

    # Python 2.7, 3.4, or 3.5
    extends('python', ignore=r'bin/nosetests.*$|bin/pbr$')

    # ------ Required dependencies
    # Per Github issue #3813, setuptools is required at runtime in order
    # to make mpl_toolkits a namespace package that can span multiple
    # directories (i.e., matplotlib and basemap)
    depends_on('py-setuptools', type=('build', 'run'))

    depends_on('python@3.5:', when='@3:')
    depends_on('libpng@1.2:')
    depends_on('freetype@2.3:')
    patch('freetype-include-path.patch', when='@2.2.2:2.9.9')  # Patch to pick up correct freetype headers

    depends_on('py-numpy@1.6:', type=('build', 'run'))
    depends_on('py-dateutil@1.1:', type=('build', 'run'))
    depends_on('py-pyparsing', type=('build', 'run'))
    depends_on('py-pytz', type=('build', 'run'))
    depends_on('py-cycler@0.9:', type=('build', 'run'))
    depends_on('py-subprocess32', type=('build', 'run'), when='^python@:2.7')
    depends_on('py-functools32', type=('build', 'run'), when='^python@2.7')
    depends_on('py-kiwisolver', type=('build', 'run'), when='@2.2.0:')

    # ------ Optional GUI frameworks
    depends_on('tk@8.3:', when='+tk')  # not 8.6.0 or 8.6.1
    depends_on('qt', when='+qt')
    depends_on('py-pyside', when='+qt', type=('build', 'run'))

    # --------- Optional external programs
    # ffmpeg/avconv or mencoder
    depends_on('image-magick', when='+animation')

    # --------- Optional dependencies
    depends_on('pkgconfig', type='build')    # why not...
    depends_on('pil', when='+image', type=('build', 'run'))
    depends_on('py-ipython', when='+ipython', type=('build', 'run'))
    depends_on('ghostscript', when='+latex', type='run')
    depends_on('texlive', when='+latex', type='run')

    # Testing dependencies
    depends_on('py-nose', type='test')
    depends_on('py-mock', type='test')

    # Required libraries that ship with matplotlib
    # depends_on('agg@2.4:')
    depends_on('qhull@2012.1:')
    # depends_on('ttconv')
    depends_on('py-six@1.9.0:', type=('build', 'run'))

    @run_before('build')
    def set_cc(self):
        if self.spec.satisfies('%intel'):
            env['CC'] = spack_cxx

    @run_after('install')
    def set_backend(self):
        spec = self.spec
        prefix = self.prefix

        if '+qt' in spec or '+tk' in spec:
            # Set backend in matplotlib configuration file
            config_file = None
            for p, d, f in os.walk(prefix.lib):
                for file in f:
                    if file.find('matplotlibrc') != -1:
                        config_file = join_path(p, 'matplotlibrc')
            if not config_file:
                raise InstallError('Could not find matplotlibrc')

            kwargs = {'ignore_absent': False, 'backup': False, 'string': False}
            rc = FileFilter(config_file)

            # Only make Qt4 be the default backend if Tk is turned off
            if '+qt' in spec and '+tk' not in spec:
                rc.filter('^backend.*', 'backend     : Qt4Agg', **kwargs)

            # Additional options in case user is doing Qt4:
            if '+qt' in spec:
                rc.filter('^#backend.qt4.*', 'backend.qt4 : PySide', **kwargs)
