# Copyright 2013-2018 Lawrence Livermore National Security, LLC and other
# Spack Project Developers. See the top-level COPYRIGHT file for details.
#
# SPDX-License-Identifier: (Apache-2.0 OR MIT)

from spack import *


class Ninja(Package):
    """Ninja is a small build system with a focus on speed. It differs from
    other build systems in two major respects: it is designed to have its input
    files generated by a higher-level build system, and it is designed to run
    builds as fast as possible."""

    homepage = "https://ninja-build.org/"
    url      = "https://github.com/ninja-build/ninja/archive/v1.7.2.tar.gz"

    version('1.8.2', '5fdb04461cc7f5d02536b3bfc0300166')
    version('1.7.2', '7b482218757acbaeac4d4d54a3cd94e1')
    version('1.6.0', '254133059f2da79d8727f654d7198f43')

    depends_on('python', type=('build', 'run'))

    phases = ['configure', 'install']

    def configure(self, spec, prefix):
        python('configure.py', '--bootstrap')

    @run_after('configure')
    @on_package_attributes(run_tests=True)
    def test(self):
        ninja = Executable('./ninja')
        ninja('-j{0}'.format(make_jobs), 'ninja_test')
        ninja_test = Executable('./ninja_test')
        ninja_test()

    def install(self, spec, prefix):
        mkdir(prefix.bin)
        install('ninja', prefix.bin)
        install_tree('misc', prefix.misc)

        # Some distros like Fedora install a 'ninja-build' executable
        # instead of 'ninja'. Install both for uniformity.
        with working_dir(prefix.bin):
            symlink('ninja', 'ninja-build')
