﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using Xunit;

namespace System.Windows.Forms.Tests
{
    public class ToolStripDropDownTests
    {
        [Fact]
        public void ToolStripDropDown_Constructor()
        {
            var tsdd = new ToolStripDropDown();

            Assert.NotNull(tsdd);
        }

        [Fact]
        public void ToolStripDropDown_ConstructorOwnerItem()
        {
            var owner = new ToolStripButton();

            var tsdd = new ToolStripDropDown(owner);

            Assert.NotNull(tsdd);
            Assert.Equal(owner, tsdd.OwnerItem);
        }

        [Fact]
        public void ToolStripDropDown_ConstructorOwnerItemBool()
        {
            var owner = new ToolStripButton();
            var isAutoGenerated = true;

            var tsdd = new ToolStripDropDown(owner, isAutoGenerated);

            Assert.NotNull(tsdd);
            Assert.Equal(owner, tsdd.OwnerItem);
            Assert.True(tsdd.IsAutoGenerated);
        }

        [StaFact]
        public void ToolStripDropDown_KeyboardArrowNavigation_Test()
        {
            ToolStripMenuItem toolStrip = new ToolStripMenuItem();
            ToolStripItemCollection items = toolStrip.DropDown.DisplayedItems;
            items.Add("First item");
            items.Add("Second item");
            items.Add("Third item");
            items.Add("Fourth item");
            items.Add("Fifth item");

            for (int i = 1; i < items.Count; i++) //it needs for correct work of ToolStrip.GetNextItemVertical method
            {
                items[i].SetBounds(0, items[i - 1].Bounds.Bottom + 1, items[i].Bounds.Width, items[i].Bounds.Height);
            }

            ToolStripItem expected = items[4];
            ToolStripItem actual = toolStrip.DropDown.GetNextItem(start: null, direction: ArrowDirection.Up);
            Assert.Equal(expected, actual);

            expected = items[4];
            actual = toolStrip.DropDown.GetNextItem(start: items[0], direction: ArrowDirection.Up);
            Assert.Equal(expected, actual);

            expected = items[0];
            actual = toolStrip.DropDown.GetNextItem(start: null, direction: ArrowDirection.Down);
            Assert.Equal(expected, actual);

            expected = items[1];
            actual = toolStrip.DropDown.GetNextItem(start: items[0], direction: ArrowDirection.Down);
            Assert.Equal(expected, actual);

            expected = items[0];
            actual = toolStrip.DropDown.GetNextItem(start: items[4], direction: ArrowDirection.Down);
            Assert.Equal(expected, actual);
        }

        [StaFact]
        public void ToolStripDropDown_KeyboardAccelerators_Test()
        {
            TestForm testForm = new TestForm();
            Application.Run(testForm); //it needs for correct work of ToolStripDropDown.CanProcessMnemonic method

            //TestResult property is made as separate for the reason that 
            //when the Assert is inside FormLoaded method and it fails, 
            //the Application doesn't exit and the process freezes.
            Assert.True(testForm.TestResult);
        }

        class TestForm : Form
        {
            private SubToolStripDropDown toolStrip;
            private bool _result = true;

            public TestForm()
            {
                toolStrip = new SubToolStripDropDown();
                Load += FormLoaded;
            }

            public bool TestResult => _result;

            private void FormLoaded(object sender, EventArgs e)
            {
                toolStrip.Enabled = true; // it needs for correct work of Control.CanProcessMnemonic method
                toolStrip.Visible = true; // 

                _result &= !(toolStrip.ProcessDialogCharTest('F'));

                toolStrip.DisplayedItems.Add("&First item");
                toolStrip.DisplayedItems.Add("&Second item");
                toolStrip.DisplayedItems.Add("Third item");
                toolStrip.Visible = true; // it needs for correct work of Control.CanProcessMnemonic method

                _result &= toolStrip.ProcessDialogCharTest('F');
                _result &= toolStrip.ProcessDialogCharTest('S');
                _result &= !(toolStrip.ProcessDialogCharTest('T'));

                Application.Exit();
            }
        }

        class SubToolStripDropDown : ToolStripDropDown
        {
            public bool ProcessDialogCharTest(char charCode)
            {
                return base.ProcessDialogChar(charCode);
            }
        }
    }
}
