﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System.Collections;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Globalization;
using System.Runtime.CompilerServices;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms.Layout;
using System.Windows.Forms.VisualStyles;
using Microsoft.Win32;
using static Interop;

namespace System.Windows.Forms
{
    public partial class DataGridView
    {
        protected virtual void AccessibilityNotifyCurrentCellChanged(Point cellAddress)
        {
            if (cellAddress.X < 0 || cellAddress.X >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(cellAddress));
            }

            if (cellAddress.Y < 0 || cellAddress.Y >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(cellAddress));
            }

            int visibleRowIndex = Rows.GetRowCount(DataGridViewElementStates.Visible, 0, cellAddress.Y);
            int visibleColumnIndex = Columns.ColumnIndexToActualDisplayIndex(cellAddress.X, DataGridViewElementStates.Visible);

            int topHeaderRowIncrement = ColumnHeadersVisible ? 1 : 0;
            int rowHeaderIncrement = RowHeadersVisible ? 1 : 0;

            int objectID = visibleRowIndex + topHeaderRowIncrement  // + 1 because the top header row acc obj is at index 0
                                           + 1;                     // + 1 because objectID's need to be positive and non-zero

            int childID = visibleColumnIndex + rowHeaderIncrement;  // + 1 because the column header cell is at index 0 in top header row acc obj
                                                                    //     same thing for the row header cell in the data grid view row acc obj

            if (ContainsFocus)
            {
                AccessibilityNotifyClients(AccessibleEvents.Focus, objectID, childID);

                CurrentCell?.AccessibilityObject.SetFocus();
            }
            AccessibilityNotifyClients(AccessibleEvents.Selection, objectID, childID);
        }

        internal void ActivateToolTip(bool activate, string toolTipText, int columnIndex, int rowIndex)
        {
            toolTipCaption = toolTipText;
            ptToolTipCell = new Point(columnIndex, rowIndex);
            toolTipControl.Activate(activate);
        }

        internal void AddNewRow(bool createdByEditing)
        {
            Debug.Assert(Columns.Count > 0);
            Debug.Assert(newRowIndex == -1);

            Rows.AddInternal(true /*newRow*/, null /*values*/);
            newRowIndex = Rows.Count - 1;
            dataGridViewState1[DATAGRIDVIEWSTATE1_newRowCreatedByEditing] = createdByEditing;

            if (createdByEditing)
            {
                DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs(Rows[newRowIndex]);
                OnUserAddedRow(dgvre);
            }
        }

        [
            EditorBrowsable(EditorBrowsableState.Advanced)
        ]
        public virtual DataGridViewAdvancedBorderStyle AdjustColumnHeaderBorderStyle(DataGridViewAdvancedBorderStyle dataGridViewAdvancedBorderStyleInput,
                                                                         DataGridViewAdvancedBorderStyle dataGridViewAdvancedBorderStylePlaceholder,
                                                                         bool isFirstDisplayedColumn,
                                                                         bool isLastVisibleColumn)
        {
            if (ApplyVisualStylesToHeaderCells)
            {
                switch (dataGridViewAdvancedBorderStyleInput.All)
                {
                    case DataGridViewAdvancedCellBorderStyle.OutsetPartial:
                    case DataGridViewAdvancedCellBorderStyle.OutsetDouble:
                        if (RightToLeftInternal)
                        {
                            dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.None;
                            if (isFirstDisplayedColumn)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = RowHeadersVisible ? DataGridViewAdvancedCellBorderStyle.None : DataGridViewAdvancedCellBorderStyle.Outset;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.None;
                            }
                        }
                        else
                        {
                            if (isFirstDisplayedColumn)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = RowHeadersVisible ? DataGridViewAdvancedCellBorderStyle.None : DataGridViewAdvancedCellBorderStyle.OutsetDouble;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.None;
                            }
                            dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.None;
                        }
                        dataGridViewAdvancedBorderStylePlaceholder.TopInternal = DataGridViewAdvancedCellBorderStyle.OutsetDouble;
                        dataGridViewAdvancedBorderStylePlaceholder.BottomInternal = DataGridViewAdvancedCellBorderStyle.Outset;
                        return dataGridViewAdvancedBorderStylePlaceholder;

                    case DataGridViewAdvancedCellBorderStyle.InsetDouble:
                        if (RightToLeftInternal)
                        {
                            dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.None;
                            if (isFirstDisplayedColumn)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = RowHeadersVisible ? DataGridViewAdvancedCellBorderStyle.None : DataGridViewAdvancedCellBorderStyle.Inset;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.None;
                            }
                        }
                        else
                        {
                            if (isFirstDisplayedColumn)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = RowHeadersVisible ? DataGridViewAdvancedCellBorderStyle.None : DataGridViewAdvancedCellBorderStyle.InsetDouble;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.None;
                            }
                            dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.None;
                        }
                        dataGridViewAdvancedBorderStylePlaceholder.TopInternal = DataGridViewAdvancedCellBorderStyle.InsetDouble;
                        dataGridViewAdvancedBorderStylePlaceholder.BottomInternal = DataGridViewAdvancedCellBorderStyle.Inset;
                        return dataGridViewAdvancedBorderStylePlaceholder;

                    case DataGridViewAdvancedCellBorderStyle.Single:
                    case DataGridViewAdvancedCellBorderStyle.Outset:
                    case DataGridViewAdvancedCellBorderStyle.Inset:
                        if (!isFirstDisplayedColumn || RowHeadersVisible)
                        {
                            dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.None;
                            dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.None;
                            dataGridViewAdvancedBorderStylePlaceholder.TopInternal = dataGridViewAdvancedBorderStyleInput.All;
                            dataGridViewAdvancedBorderStylePlaceholder.BottomInternal = dataGridViewAdvancedBorderStyleInput.All;
                            return dataGridViewAdvancedBorderStylePlaceholder;
                        }
                        else
                        {
                            // isFirstDisplayedColumn == true && this.RowHeadersVisible == false
                            if (RightToLeftInternal)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.None;
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = dataGridViewAdvancedBorderStyleInput.All;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = dataGridViewAdvancedBorderStyleInput.All;
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.None;
                            }
                            dataGridViewAdvancedBorderStylePlaceholder.TopInternal = dataGridViewAdvancedBorderStyleInput.All;
                            dataGridViewAdvancedBorderStylePlaceholder.BottomInternal = dataGridViewAdvancedBorderStyleInput.All;
                            return dataGridViewAdvancedBorderStylePlaceholder;
                        }
                }
            }
            else
            {
                switch (dataGridViewAdvancedBorderStyleInput.All)
                {
                    case DataGridViewAdvancedCellBorderStyle.OutsetPartial:
                        if (RightToLeftInternal)
                        {
                            dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = isLastVisibleColumn ? DataGridViewAdvancedCellBorderStyle.Outset : DataGridViewAdvancedCellBorderStyle.OutsetPartial;
                            if (isFirstDisplayedColumn)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = RowHeadersVisible ? DataGridViewAdvancedCellBorderStyle.Outset : DataGridViewAdvancedCellBorderStyle.OutsetDouble;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.OutsetPartial;
                            }
                        }
                        else
                        {
                            if (isFirstDisplayedColumn)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = RowHeadersVisible ? DataGridViewAdvancedCellBorderStyle.Outset : DataGridViewAdvancedCellBorderStyle.OutsetDouble;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.OutsetPartial;
                            }
                            dataGridViewAdvancedBorderStylePlaceholder.RightInternal = isLastVisibleColumn ? DataGridViewAdvancedCellBorderStyle.Outset : DataGridViewAdvancedCellBorderStyle.OutsetPartial;
                        }
                        dataGridViewAdvancedBorderStylePlaceholder.TopInternal = DataGridViewAdvancedCellBorderStyle.OutsetDouble;
                        dataGridViewAdvancedBorderStylePlaceholder.BottomInternal = DataGridViewAdvancedCellBorderStyle.Outset;
                        return dataGridViewAdvancedBorderStylePlaceholder;

                    case DataGridViewAdvancedCellBorderStyle.OutsetDouble:
                        if (RightToLeftInternal)
                        {
                            dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.Outset;
                            if (isFirstDisplayedColumn)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = RowHeadersVisible ? DataGridViewAdvancedCellBorderStyle.Outset : DataGridViewAdvancedCellBorderStyle.OutsetDouble;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.Outset;
                            }
                        }
                        else
                        {
                            if (isFirstDisplayedColumn)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = RowHeadersVisible ? DataGridViewAdvancedCellBorderStyle.Outset : DataGridViewAdvancedCellBorderStyle.OutsetDouble;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.Outset;
                            }
                            dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.Outset;
                        }
                        dataGridViewAdvancedBorderStylePlaceholder.TopInternal = DataGridViewAdvancedCellBorderStyle.OutsetDouble;
                        dataGridViewAdvancedBorderStylePlaceholder.BottomInternal = DataGridViewAdvancedCellBorderStyle.Outset;
                        return dataGridViewAdvancedBorderStylePlaceholder;

                    case DataGridViewAdvancedCellBorderStyle.InsetDouble:
                        if (RightToLeftInternal)
                        {
                            dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.Inset;
                            if (isFirstDisplayedColumn)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = RowHeadersVisible ? DataGridViewAdvancedCellBorderStyle.Inset : DataGridViewAdvancedCellBorderStyle.InsetDouble;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.Inset;
                            }
                        }
                        else
                        {
                            if (isFirstDisplayedColumn)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = RowHeadersVisible ? DataGridViewAdvancedCellBorderStyle.Inset : DataGridViewAdvancedCellBorderStyle.InsetDouble;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.Inset;
                            }
                            dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.Inset;
                        }
                        dataGridViewAdvancedBorderStylePlaceholder.TopInternal = DataGridViewAdvancedCellBorderStyle.InsetDouble;
                        dataGridViewAdvancedBorderStylePlaceholder.BottomInternal = DataGridViewAdvancedCellBorderStyle.Inset;
                        return dataGridViewAdvancedBorderStylePlaceholder;

                    case DataGridViewAdvancedCellBorderStyle.Single:
                        if (!isFirstDisplayedColumn || RowHeadersVisible)
                        {
                            if (RightToLeftInternal)
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.Single;
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.None;
                            }
                            else
                            {
                                dataGridViewAdvancedBorderStylePlaceholder.LeftInternal = DataGridViewAdvancedCellBorderStyle.None;
                                dataGridViewAdvancedBorderStylePlaceholder.RightInternal = DataGridViewAdvancedCellBorderStyle.Single;
                            }
                            dataGridViewAdvancedBorderStylePlaceholder.TopInternal = DataGridViewAdvancedCellBorderStyle.Single;
                            dataGridViewAdvancedBorderStylePlaceholder.BottomInternal = DataGridViewAdvancedCellBorderStyle.Single;
                            return dataGridViewAdvancedBorderStylePlaceholder;
                        }
                        break;
                }
            }

            return dataGridViewAdvancedBorderStyleInput;
        }

        private bool AdjustExpandingColumn(DataGridViewColumn dataGridViewColumn, int rowIndex)
        {
            Debug.Assert(dataGridViewColumn != null);
            Debug.Assert(rowIndex > -1);
            Debug.Assert(rowIndex < Rows.Count);

            if (!IsHandleCreated)
            {
                // auto sizing causes handle creation.
                // don't create the handle inside InitializeComponent because that causes problems w/ data binding
                dataGridViewState2[DATAGRIDVIEWSTATE2_autoSizedWithoutHandle] = true;
                return false;
            }

            bool ret = false; // No autosizing occurs by default.
            try
            {
                noAutoSizeCount++;
                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                int preferredThickness = dataGridViewRow.Cells[dataGridViewColumn.Index].GetPreferredWidth(rowIndex, dataGridViewRow.GetHeight(rowIndex));
                if (preferredThickness > DataGridViewBand.MaxBandThickness)
                {
                    preferredThickness = DataGridViewBand.MaxBandThickness;
                }
                if (dataGridViewColumn.Width < preferredThickness)
                {
                    // Column needs to be expanded
                    dataGridViewColumn.ThicknessInternal = preferredThickness;
                    ret = true;
                }
            }
            finally
            {
                Debug.Assert(noAutoSizeCount > 0);
                noAutoSizeCount--;
            }

            return ret;
        }

        private bool AdjustExpandingColumns(DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaFilter, int rowIndex)
        {
            Debug.Assert(autoSizeColumnCriteriaFilter != DataGridViewAutoSizeColumnCriteriaInternal.None);
            Debug.Assert((autoSizeColumnCriteriaFilter & DataGridViewAutoSizeColumnCriteriaInternal.Fill) == 0);

            bool ret = false; // No column autosizes by default
            DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
            while (dataGridViewColumn != null)
            {
                DataGridViewAutoSizeColumnCriteriaInternal inheritedAutoSizeColumnCriteria = (DataGridViewAutoSizeColumnCriteriaInternal)dataGridViewColumn.InheritedAutoSizeMode;
                DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaFiltered = (inheritedAutoSizeColumnCriteria & autoSizeColumnCriteriaFilter);
                if (autoSizeColumnCriteriaFiltered != 0)
                {
                    ret |= AdjustExpandingColumn(dataGridViewColumn, rowIndex);
                }
                dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                    DataGridViewElementStates.Visible,
                    DataGridViewElementStates.None);
            }
            return ret;
        }

        private int AdjustExpandingRow(int rowIndex, int columnIndex, bool fixedWidth)
        {
            Debug.Assert(columnIndex >= -1 && columnIndex < Columns.Count);
            Debug.Assert(rowIndex >= 0 && rowIndex < Rows.Count);
            Debug.Assert(autoSizeRowsMode == DataGridViewAutoSizeRowsMode.DisplayedHeaders ||
                         autoSizeRowsMode == DataGridViewAutoSizeRowsMode.DisplayedCellsExceptHeaders ||
                         autoSizeRowsMode == DataGridViewAutoSizeRowsMode.DisplayedCells);

            // Use of WindowsFormsUtils.CreateMeasurementGraphics() avoid use of this.Handle
            // IntPtr handle = this.Handle; // Force creation of control's handle because for databound grids, handle creation wipes out and recreates the columns/rows.
            int width = 0;
            DataGridViewCell dataGridViewCell;
            if (columnIndex > -1 && (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0)
            {
                dataGridViewCell = Rows.SharedRow(rowIndex).Cells[columnIndex];
                if (fixedWidth)
                {
                    width = Columns[columnIndex].Thickness;
                }
            }
            else
            {
                Debug.Assert(columnIndex == -1);
                Debug.Assert((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.Header) != 0);
                dataGridViewCell = Rows.SharedRow(rowIndex).HeaderCell;
                if (fixedWidth)
                {
                    Debug.Assert(RowHeadersWidth > 0);
                    Debug.Assert(RowHeadersVisible);
                    width = RowHeadersWidth;
                }
            }
            int preferredThickness;
            if (fixedWidth)
            {
                preferredThickness = dataGridViewCell.GetPreferredHeight(rowIndex, width);
            }
            else
            {
                preferredThickness = dataGridViewCell.GetPreferredSize(rowIndex).Height;
            }
            Rows.SharedRow(rowIndex).GetHeightInfo(rowIndex, out int height, out int minimumHeight);
            if (preferredThickness < height)
            {
                preferredThickness = height;
            }
            Debug.Assert(preferredThickness >= minimumHeight);
            if (preferredThickness > DataGridViewBand.MaxBandThickness)
            {
                preferredThickness = DataGridViewBand.MaxBandThickness;
            }
            if (height != preferredThickness)
            {
                Debug.Assert(autoSizeRowsMode != DataGridViewAutoSizeRowsMode.None);
                Rows[rowIndex].Thickness = preferredThickness;   // unsharing the resized row
            }
            return preferredThickness;
        }

        private void AdjustExpandingRows(int columnIndex, bool fixedWidth)
        {
            if ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0 ||
                ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.Header) != 0 && RowHeadersVisible))
            {
                if (!IsHandleCreated)
                {
                    // auto sizing causes handle creation.
                    // don't create the handle inside InitializeComponent because that causes problems w/ data binding
                    dataGridViewState2[DATAGRIDVIEWSTATE2_autoSizedWithoutHandle] = true;
                    return;
                }

                // Very expensive processing - the developer should avoid this scenario.
                // Switch to batch operation
                inBulkPaintCount++;
                try
                {
                    if ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllRows) != 0)
                    {
                        inBulkLayoutCount++;
                        try
                        {
                            for (int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                                rowIndex != -1;
                                rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible))
                            {
                                int width = 0;
                                DataGridViewCell dataGridViewCell;
                                if (columnIndex > -1 && (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0)
                                {
                                    dataGridViewCell = Rows.SharedRow(rowIndex).Cells[columnIndex];
                                    if (fixedWidth)
                                    {
                                        width = Columns[columnIndex].Thickness;
                                    }
                                }
                                else
                                {
                                    //Debug.Assert(columnIndex == -1);
                                    Debug.Assert((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.Header) != 0);
                                    dataGridViewCell = Rows.SharedRow(rowIndex).HeaderCell;
                                    if (fixedWidth)
                                    {
                                        Debug.Assert(RowHeadersWidth > 0);
                                        Debug.Assert(RowHeadersVisible);
                                        width = RowHeadersWidth;
                                    }
                                }
                                int preferredHeight;
                                if (fixedWidth)
                                {
                                    preferredHeight = dataGridViewCell.GetPreferredHeight(rowIndex, width);
                                }
                                else
                                {
                                    preferredHeight = dataGridViewCell.GetPreferredSize(rowIndex).Height;
                                }
                                if (Rows.SharedRow(rowIndex).Height < preferredHeight)
                                {
                                    Rows[rowIndex].Height = preferredHeight;  // unsharing the row to be resized
                                }
                            }
                        }
                        finally
                        {
                            ExitBulkLayout(false /*invalidInAdjustFillingColumns*/);
                        }
                    }
                    else
                    {
                        Debug.Assert((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.DisplayedRows) != 0);
                        int displayHeight = layout.Data.Height;
                        int cy = 0;

                        int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                        while (rowIndex != -1 && cy < displayHeight)
                        {
                            cy += AdjustExpandingRow(rowIndex, columnIndex, fixedWidth);
                            rowIndex = Rows.GetNextRow(rowIndex,
                                DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                        }

                        if (cy < displayHeight)
                        {
                            rowIndex = displayedBandsInfo.FirstDisplayedScrollingRow;
                            while (rowIndex != -1 && cy < displayHeight)
                            {
                                cy += AdjustExpandingRow(rowIndex, columnIndex, fixedWidth);
                                rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                            }
                        }
                    }
                }
                finally
                {
                    ExitBulkPaint(-1, -1);
                }
            }
        }

        internal void AdjustFillingColumn(DataGridViewColumn dataGridViewColumn, int width)
        {
            if (InAdjustFillingColumns)
            {
                throw new InvalidOperationException(SR.DataGridView_CannotAlterAutoFillColumnParameter);
            }

            dataGridViewOper[DATAGRIDVIEWOPER_inAdjustFillingColumn] = true;

            try
            {
                Debug.Assert(dataGridViewColumn != null);
                Debug.Assert(dataGridViewColumn.Visible);
                Debug.Assert(!dataGridViewColumn.Frozen);
                Debug.Assert(dataGridViewColumn.MinimumWidth <= width);
                Debug.Assert(dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill);
                Debug.Assert(!layout.dirty);

                if (Columns.GetColumnsWidth(DataGridViewElementStates.Visible) > layout.Data.Width)
                {
                    // Columns are scrolling - this means that all columns have reached their minimum width.
                    // Do not affect their width or fill weight
                    Debug.Assert(dataGridViewColumn.MinimumWidth == dataGridViewColumn.Width);
                    return;
                }

                int availableWidth = layout.Data.Width;  // Width available for auto-filled columns

                // Check if the column is the first or last visible scrolling column
                if (DesignMode ||
                    dataGridViewColumn == Columns.GetFirstColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen) ||
                    dataGridViewColumn == Columns.GetLastColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen))
                {
                    // Changing the width is equivalent to adjusting the FillWeight when:
                    //  - the column is not scrolling and is the first non-frozen visible column
                    //  - the column is not scrolling and is the last non-frozen visible column

                    float weightSum = 0;        // Weights of all auto filled visible columns.
                    int widthSum = 0;           // Sum of widths of visible auto filled columns.
                    int imposedWidthSum = 0;    // Minimum width required for all other columns.
                    bool otherFillingColumnExists = false;
                    foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                    {
                        if (dataGridViewColumnTmp.Visible)
                        {
                            if (dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                            {
                                Debug.Assert(!dataGridViewColumnTmp.Frozen);
                                widthSum += dataGridViewColumnTmp.Width;
                                if (dataGridViewColumnTmp.Index != dataGridViewColumn.Index)
                                {
                                    imposedWidthSum += dataGridViewColumnTmp.MinimumWidth;
                                    otherFillingColumnExists = true;
                                }
                                weightSum += dataGridViewColumnTmp.FillWeight;
                            }
                            else
                            {
                                imposedWidthSum += dataGridViewColumnTmp.Width;
                                availableWidth -= dataGridViewColumnTmp.Width;
                            }
                        }
                    }

                    if (!otherFillingColumnExists)
                    {
                        // The resized column is the only one that is filling. This is a no-op operation.
                        // Neither the width nor the fill weight can change
                        return;
                    }

                    int maximumPossibleWidth = layout.Data.Width - imposedWidthSum;
                    if (width > maximumPossibleWidth)
                    {
                        width = maximumPossibleWidth;
                    }

                    // Determine fill weight equivalent to 'width'
                    float oldWeight = dataGridViewColumn.FillWeight;
                    float newWeight = (float)(width * weightSum) / (float)widthSum;
                    bool desiredWidthTooSmall = false;
                    foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                    {
                        if (dataGridViewColumnTmp.Index != dataGridViewColumn.Index &&
                            dataGridViewColumnTmp.Visible &&
                            dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                        {
                            dataGridViewColumnTmp.FillWeightInternal = (weightSum - newWeight) * dataGridViewColumnTmp.FillWeight / (weightSum - oldWeight);

                            if (dataGridViewColumnTmp.FillWeight < (dataGridViewColumnTmp.MinimumWidth * weightSum) / (float)widthSum)
                            {
                                desiredWidthTooSmall = true;
                                dataGridViewColumnTmp.DesiredFillWidth = -1;
                            }
                            else
                            {
                                dataGridViewColumnTmp.DesiredFillWidth = 0;
                            }
                        }
                    }

                    dataGridViewColumn.FillWeightInternal = newWeight;

                    if (desiredWidthTooSmall)
                    {
                        // At least one column hits its minimum width
                        // Adjust UsedFillWeight values are adjusted FillWeight values
                        float usedWeightSumNoneMinimal = weightSum;
                        float weightSumNoneMinimal = weightSum;
                        float usedFillWeights = 0F;
                        foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                        {
                            if (dataGridViewColumnTmp.Visible &&
                                dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                            {
                                if (dataGridViewColumnTmp.Index == dataGridViewColumn.Index)
                                {
                                    dataGridViewColumnTmp.UsedFillWeight = dataGridViewColumnTmp.FillWeight;
                                    usedWeightSumNoneMinimal -= dataGridViewColumnTmp.UsedFillWeight;
                                    weightSumNoneMinimal -= dataGridViewColumnTmp.FillWeight;
                                    usedFillWeights += dataGridViewColumnTmp.UsedFillWeight;
                                }
                                else if (dataGridViewColumnTmp.DesiredFillWidth == -1)
                                {
                                    dataGridViewColumnTmp.UsedFillWeight = weightSum * dataGridViewColumnTmp.MinimumWidth / widthSum;
                                    usedWeightSumNoneMinimal -= dataGridViewColumnTmp.UsedFillWeight;
                                    weightSumNoneMinimal -= dataGridViewColumnTmp.FillWeight;
                                    usedFillWeights += dataGridViewColumnTmp.UsedFillWeight;
                                }
                            }
                        }
                        foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                        {
                            if (dataGridViewColumnTmp.Index != dataGridViewColumn.Index &&
                                dataGridViewColumnTmp.Visible &&
                                dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill &&
                                dataGridViewColumnTmp.DesiredFillWidth != -1)
                            {
                                dataGridViewColumnTmp.UsedFillWeight = Math.Max(dataGridViewColumnTmp.FillWeight * usedWeightSumNoneMinimal / weightSumNoneMinimal,
                                                                                weightSum * dataGridViewColumnTmp.MinimumWidth / widthSum);
                                usedFillWeights += dataGridViewColumnTmp.UsedFillWeight;
                            }
                        }
                        dataGridViewColumn.UsedFillWeight += weightSum - usedFillWeights;
                    }
                    else
                    {
                        // No column hits its minimum width
                        // Each UsedFillWeight simply equals the FillWeight
                        foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                        {
                            if (dataGridViewColumnTmp.Visible &&
                                dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                            {
                                dataGridViewColumnTmp.UsedFillWeight = dataGridViewColumnTmp.FillWeight;
                            }
                        }
                    }
#if DEBUG
                    float weightSumDbg = 0F;
                    foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                    {
                        if (dataGridViewColumnTmp.Visible &&
                            dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                        {
                            weightSumDbg += dataGridViewColumnTmp.UsedFillWeight;
                        }
                    }
                    Debug.Assert(Math.Abs(weightSum - weightSumDbg) < 1.0F);
#endif
                }
                else
                {
                    // This column is not the first nor the last visible non-frozen column
                    // Changing its width only affects the width and weight of the columns displayed after it

                    // First figure out the maximum possible width
                    int imposedWidthSum = 0;    // Minimum width required for all other columns
                    float weightSum = 0;        // Weights of all auto filled visible columns.
                    float oldWeightSum = 0F;    // Fill weights of the columns displayed after this resized column
                    bool otherFillingColumnExists = false;
                    foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                    {
                        if (dataGridViewColumnTmp.Visible)
                        {
                            if (dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                            {
                                Debug.Assert(!dataGridViewColumnTmp.Frozen);
                                if (dataGridViewColumnTmp.Index != dataGridViewColumn.Index)
                                {
                                    if (Columns.DisplayInOrder(dataGridViewColumn.Index, dataGridViewColumnTmp.Index))
                                    {
                                        imposedWidthSum += dataGridViewColumnTmp.MinimumWidth;   // Column is allowed to shrink down to its minimum
                                        oldWeightSum += dataGridViewColumnTmp.FillWeight;
                                    }
                                    else
                                    {
                                        // Column is displayed before 'dataGridViewColumn', it is not allowed to shrink at all
                                        imposedWidthSum += dataGridViewColumnTmp.Width;
                                    }
                                    otherFillingColumnExists = true;
                                }
                                weightSum += dataGridViewColumnTmp.FillWeight;
                            }
                            else
                            {
                                imposedWidthSum += dataGridViewColumnTmp.Width;
                                availableWidth -= dataGridViewColumnTmp.Width;
                            }
                        }
                    }

                    if (!otherFillingColumnExists)
                    {
                        // The resized column is the only one that is filling. This is a no-op operation.
                        // Neither the width nor the fill weight can change
                        return;
                    }

                    int maximumPossibleWidth = layout.Data.Width - imposedWidthSum;
                    if (width > maximumPossibleWidth)
                    {
                        width = maximumPossibleWidth;
                    }

                    // Then figure out the target fill weights
                    float oldWeight = dataGridViewColumn.FillWeight;
                    float newWeight = weightSum * width / availableWidth;
                    float newWeightSum = oldWeightSum + oldWeight - newWeight;
                    Debug.Assert(newWeightSum > 0);
                    foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                    {
                        if (dataGridViewColumnTmp.Index != dataGridViewColumn.Index &&
                            dataGridViewColumnTmp.Visible &&
                            dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill &&
                            Columns.DisplayInOrder(dataGridViewColumn.Index, dataGridViewColumnTmp.Index))
                        {
                            dataGridViewColumnTmp.FillWeightInternal = dataGridViewColumnTmp.FillWeight * newWeightSum / oldWeightSum;
                        }
                    }

                    dataGridViewColumn.FillWeightInternal = newWeight;

                    bool desiredWidthTooSmall = false;
                    foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                    {
                        if (dataGridViewColumnTmp.Visible &&
                            dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                        {
                            if (dataGridViewColumnTmp.FillWeight < (dataGridViewColumnTmp.MinimumWidth * weightSum) / (float)availableWidth)
                            {
                                desiredWidthTooSmall = true;
                                dataGridViewColumnTmp.DesiredFillWidth = -1;
                            }
                            else
                            {
                                dataGridViewColumnTmp.DesiredFillWidth = 0;
                            }
                        }
                    }

                    if (desiredWidthTooSmall)
                    {
                        // At least one column hits its minimum width
                        // Adjust UsedFillWeight values are adjusted FillWeight values
                        float usedWeightSumNoneMinimal = weightSum;
                        float weightSumNoneMinimal = weightSum;
                        float usedFillWeights = 0F;
                        foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                        {
                            if (dataGridViewColumnTmp.Visible &&
                                dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                            {
                                if (dataGridViewColumnTmp.Index == dataGridViewColumn.Index ||
                                    Columns.DisplayInOrder(dataGridViewColumnTmp.Index, dataGridViewColumn.Index))
                                {
                                    if (dataGridViewColumnTmp.Index == dataGridViewColumn.Index)
                                    {
                                        dataGridViewColumnTmp.UsedFillWeight = dataGridViewColumnTmp.FillWeight;
                                    }
                                    else
                                    {
                                        dataGridViewColumnTmp.UsedFillWeight = weightSum * dataGridViewColumnTmp.Width / availableWidth;
                                    }
                                    usedWeightSumNoneMinimal -= dataGridViewColumnTmp.UsedFillWeight;
                                    weightSumNoneMinimal -= dataGridViewColumnTmp.FillWeight;
                                    usedFillWeights += dataGridViewColumnTmp.UsedFillWeight;
                                }
                                else if (dataGridViewColumnTmp.DesiredFillWidth == -1)
                                {
                                    dataGridViewColumnTmp.UsedFillWeight = weightSum * dataGridViewColumnTmp.MinimumWidth / availableWidth;
                                    usedWeightSumNoneMinimal -= dataGridViewColumnTmp.UsedFillWeight;
                                    weightSumNoneMinimal -= dataGridViewColumnTmp.FillWeight;
                                    usedFillWeights += dataGridViewColumnTmp.UsedFillWeight;
                                }
                            }
                        }
                        foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                        {
                            if (Columns.DisplayInOrder(dataGridViewColumn.Index, dataGridViewColumnTmp.Index) &&
                                dataGridViewColumnTmp.Visible &&
                                dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill &&
                                dataGridViewColumnTmp.DesiredFillWidth != -1)
                            {
                                dataGridViewColumnTmp.UsedFillWeight = Math.Max(dataGridViewColumnTmp.FillWeight * usedWeightSumNoneMinimal / weightSumNoneMinimal,
                                                                                weightSum * dataGridViewColumnTmp.MinimumWidth / availableWidth);
                                usedFillWeights += dataGridViewColumnTmp.UsedFillWeight;
                            }
                        }
                        dataGridViewColumn.UsedFillWeight += weightSum - usedFillWeights;
                    }
                    else
                    {
                        // No column hits its minimum width
                        // Each UsedFillWeight simply equals the FillWeight
                        foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                        {
                            if (dataGridViewColumnTmp.Visible &&
                                dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                            {
                                dataGridViewColumnTmp.UsedFillWeight = dataGridViewColumnTmp.FillWeight;
                            }
                        }
                    }
#if DEBUG
                    float weightSumDbg = 0F;
                    foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                    {
                        if (dataGridViewColumnTmp.Visible &&
                            dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                        {
                            weightSumDbg += dataGridViewColumnTmp.UsedFillWeight;
                        }
                    }
                    Debug.Assert(Math.Abs(weightSum - weightSumDbg) < 1.0F);
#endif
                }

                // UsedFillWeight properties are up-to-date
                dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty] = false;
                availableWidthForFillColumns = availableWidth;
                // AdjustFillingColumns() will resize columns based on the UsedFillWeight values
                PerformLayoutPrivate(false /*useRowShortcut*/, true /*computeVisibleRows*/, false /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
            }
            finally
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inAdjustFillingColumn] = false;
            }
        }

        private bool AdjustFillingColumns()
        {
            if (dataGridViewOper[DATAGRIDVIEWOPER_inAdjustFillingColumns])
            {
                // No need to auto fill columns while we're already doing it.
                return false;
            }

            dataGridViewOper[DATAGRIDVIEWOPER_inAdjustFillingColumns] = true;

            bool columnsAdjusted = false;
            try
            {
                // Determine free space for filling columns.
                int numVisibleFillColumns = 0;  // number of visible columns that are auto filled.
                int imposedWidthSum = 0;        // total width of columns that don't have a flexible width.
                int requiredWidthSum = 0;       // total of the minimum widths of the visible auto filled columns.
                float weightSum = 0F;           // total of the weights of the visible auto filled columns.
                ArrayList autoFillColumns = null;
                foreach (DataGridViewColumn dataGridViewColumn in Columns)
                {
                    if (dataGridViewColumn.Visible)
                    {
                        if (dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                        {
                            Debug.Assert(!dataGridViewColumn.Frozen);
                            numVisibleFillColumns++;
                            requiredWidthSum += dataGridViewColumn.DesiredMinimumWidth > 0 ? dataGridViewColumn.DesiredMinimumWidth : dataGridViewColumn.MinimumWidth;
                            weightSum += dataGridViewColumn.FillWeight;
                            if (autoFillColumns == null)
                            {
                                autoFillColumns = new ArrayList(Columns.Count);
                            }
                            autoFillColumns.Add(dataGridViewColumn);
                        }
                        else
                        {
                            imposedWidthSum += dataGridViewColumn.Width;
                        }
                    }
                }

                if (numVisibleFillColumns > 0)
                {
                    // Assuming no vertical scrollbar has been accounted for yet
                    Debug.Assert(layout.Data.Width == layout.Inside.Width - layout.RowHeaders.Width - (SingleVerticalBorderAdded ? 1 : 0));
                    int availableWidth = layout.Data.Width - imposedWidthSum;
                    if ((scrollBars == ScrollBars.Both || scrollBars == ScrollBars.Vertical) /*&&
                        (availableWidth > requiredWidthSum || this.dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty])*/)
                    {
                        int totalVisibleRowCount = Rows.GetRowCount(DataGridViewElementStates.Visible);
                        int totalVisibleHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible);
                        int totalVisibleFrozenHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);

                        // Assuming there is no horizontal scrollbar, is a vertical scrollbar required?
                        ComputeVisibleRows(); // Make sure this.displayedBandsInfo.FirstDisplayedScrollingRow and other row count info variables have been set

                        if (displayedBandsInfo.NumTotallyDisplayedFrozenRows == Rows.GetRowCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen) &&
                            displayedBandsInfo.NumTotallyDisplayedScrollingRows != totalVisibleRowCount - Rows.GetRowCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen) &&
                            (totalVisibleHeight - totalVisibleFrozenHeight != ComputeHeightOfFittingTrailingScrollingRows(totalVisibleFrozenHeight)) &&
                            layout.Data.Height > totalVisibleFrozenHeight &&
                            SystemInformation.VerticalScrollBarWidth <= layout.Data.Width)
                        {
                            // Vertical scrollbar is required, even when there is not horizontal one.
                            availableWidth -= SystemInformation.VerticalScrollBarWidth;
                        }
                    }

                    int columnEntry;

                    if (availableWidth <= requiredWidthSum)
                    {
                        // All auto filled columns need to take their minimum width. If (availableWidth < requiredWidthSum) a horizontal scrollbar appears.
                        availableWidth = 0;
                        for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                        {
                            DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                            int minimumWidth = dataGridViewColumn.DesiredMinimumWidth > 0 ? dataGridViewColumn.DesiredMinimumWidth : dataGridViewColumn.MinimumWidth;
                            if (dataGridViewColumn.Thickness != minimumWidth)
                            {
                                columnsAdjusted = true;
                                dataGridViewColumn.ThicknessInternal = minimumWidth;
                            }
                            availableWidth += dataGridViewColumn.Thickness;
                        }
                        //if (this.dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty])
                        {
                            for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                            {
                                // Make sure the UsedFillWeight correspond to the actual column width
                                DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                dataGridViewColumn.UsedFillWeight = dataGridViewColumn.Width * weightSum / availableWidth;
                            }
                            dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty] = false;
                            availableWidthForFillColumns = availableWidth;
                        }
                        return columnsAdjusted;
                    }

                    // Auto filling columns can share some real estate.

                    int usedWidth = 0;

                    // Update the UsedFillWeight value if dirty
                    if (dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty])
                    {
                        // Assign desired widths
                        Debug.Assert(weightSum > 0);
                        bool desiredWidthTooSmall = false;
                        for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                        {
                            DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                            if (columnEntry == autoFillColumns.Count - 1)
                            {
                                dataGridViewColumn.DesiredFillWidth = availableWidth - usedWidth;
                            }
                            else
                            {
                                float desiredFillWidth = (dataGridViewColumn.FillWeight / weightSum) * availableWidth;
                                dataGridViewColumn.DesiredFillWidth = (int)Math.Round(desiredFillWidth, MidpointRounding.AwayFromZero);
                                usedWidth += dataGridViewColumn.DesiredFillWidth;
                            }
                            int minimumWidth = dataGridViewColumn.DesiredMinimumWidth > 0 ? dataGridViewColumn.DesiredMinimumWidth : dataGridViewColumn.MinimumWidth;
                            if (dataGridViewColumn.DesiredFillWidth < minimumWidth)
                            {
                                desiredWidthTooSmall = true;
                                dataGridViewColumn.DesiredFillWidth = -1;
                            }
                        }

                        if (desiredWidthTooSmall)
                        {
                            // At least one column hits its minimum width
                            // Adjust UsedFillWeight values are adjusted FillWeight values
                            float usedWeightSumNoneMinimal = weightSum;
                            float weightSumNoneMinimal = weightSum;
                            for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                            {
                                DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                if (dataGridViewColumn.DesiredFillWidth == -1)
                                {
                                    int minimumWidth = dataGridViewColumn.DesiredMinimumWidth > 0 ? dataGridViewColumn.DesiredMinimumWidth : dataGridViewColumn.MinimumWidth;
                                    dataGridViewColumn.UsedFillWeight = weightSum * minimumWidth / availableWidth;
                                    usedWeightSumNoneMinimal -= dataGridViewColumn.UsedFillWeight;
                                    weightSumNoneMinimal -= dataGridViewColumn.FillWeight;
                                }
                            }
                            for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                            {
                                DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                if (dataGridViewColumn.DesiredFillWidth != -1)
                                {
                                    dataGridViewColumn.UsedFillWeight = dataGridViewColumn.FillWeight * usedWeightSumNoneMinimal / weightSumNoneMinimal;
                                }
                            }
                        }
                        else
                        {
                            // No column hits its minimum width
                            // Each UsedFillWeight simply equals the FillWeight
                            for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                            {
                                DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                dataGridViewColumn.UsedFillWeight = dataGridViewColumn.FillWeight;
                            }
                        }

                        dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty] = false;
                        availableWidthForFillColumns = availableWidth;
                    }
                    else if (availableWidth != availableWidthForFillColumns)
                    {
                        // The available width for auto-filled columns has changed - UsedFillWeight values need to be adjusted.
                        if (availableWidth > availableWidthForFillColumns)
                        {
                            // Available width increased
                            int widthGain = availableWidth - availableWidthForFillColumns;
                            // Allocate additional width according to UsedFillWeight and FillWeight values
                            for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                            {
                                DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                dataGridViewColumn.DesiredFillWidth = dataGridViewColumn.Width;
                            }
                            float[] floatDesiredWidths = new float[autoFillColumns.Count];
                            for (int gain = 0; gain < widthGain; gain++)
                            {
                                float fillWeightRatioSum = 0F;
                                bool minimumColumnExists = false;
                                for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                                {
                                    DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                    fillWeightRatioSum += dataGridViewColumn.FillWeight / dataGridViewColumn.UsedFillWeight;
                                    if (dataGridViewColumn.DesiredFillWidth <= dataGridViewColumn.MinimumWidth)
                                    {
                                        minimumColumnExists = true;
                                    }
                                }
                                for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                                {
                                    DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                    if (gain == 0)
                                    {
                                        floatDesiredWidths[columnEntry] = availableWidthForFillColumns * dataGridViewColumn.UsedFillWeight / weightSum;
                                    }
                                    if (minimumColumnExists)
                                    {
                                        floatDesiredWidths[columnEntry] += dataGridViewColumn.FillWeight / dataGridViewColumn.UsedFillWeight / fillWeightRatioSum;
                                    }
                                    else
                                    {
                                        floatDesiredWidths[columnEntry] += dataGridViewColumn.FillWeight / weightSum;
                                    }
                                }
                            }
                            for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                            {
                                DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                dataGridViewColumn.UsedFillWeight = weightSum / availableWidth * floatDesiredWidths[columnEntry];
                            }
                        }
                        else
                        {
                            // availableWidth < this.availableWidthForFillColumns - Available width decreased
                            int totalWidthLoss = availableWidthForFillColumns - availableWidth;
                            int cumulatedWidthLoss = 0;
                            for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                            {
                                DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                dataGridViewColumn.DesiredFillWidth = dataGridViewColumn.Width;
                            }
                            // the width loss is accounted for in steps of 10%
                            do
                            {
                                int stepDownAvailableWidthForFillColumns = availableWidthForFillColumns - cumulatedWidthLoss;
                                int widthLoss = Math.Min(stepDownAvailableWidthForFillColumns - availableWidth, Math.Max(1, (int)(stepDownAvailableWidthForFillColumns * 0.1F)));
                                cumulatedWidthLoss += widthLoss;
                                bool changeDone;
                                do
                                {
                                    changeDone = false;
                                    // Determine which column deserves to shrink the most
                                    float biggestWeightDeficiency = 0F, fillWeightRatioSum = 0F, fillWeightRatio;
                                    DataGridViewColumn mostDeservingDataGridViewColumn = null;
                                    for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                                    {
                                        DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                        if (dataGridViewColumn.DesiredFillWidth > dataGridViewColumn.MinimumWidth)
                                        {
                                            fillWeightRatio = dataGridViewColumn.UsedFillWeight / dataGridViewColumn.FillWeight;
                                            fillWeightRatioSum += fillWeightRatio;
                                            if (fillWeightRatio > biggestWeightDeficiency)
                                            {
                                                mostDeservingDataGridViewColumn = dataGridViewColumn;
                                                biggestWeightDeficiency = fillWeightRatio;
                                            }
                                        }
                                    }
                                    if (mostDeservingDataGridViewColumn != null)
                                    {
                                        float floatDesiredWidth = (stepDownAvailableWidthForFillColumns * mostDeservingDataGridViewColumn.UsedFillWeight / weightSum) - widthLoss * mostDeservingDataGridViewColumn.UsedFillWeight / mostDeservingDataGridViewColumn.FillWeight / fillWeightRatioSum;
                                        if (floatDesiredWidth < (float)mostDeservingDataGridViewColumn.MinimumWidth)
                                        {
                                            floatDesiredWidth = (int)mostDeservingDataGridViewColumn.MinimumWidth;
                                        }
                                        int oldDesiredWidth = mostDeservingDataGridViewColumn.DesiredFillWidth;
                                        mostDeservingDataGridViewColumn.DesiredFillWidth = Math.Min(oldDesiredWidth, (int)Math.Round(floatDesiredWidth, MidpointRounding.AwayFromZero));
                                        changeDone = (oldDesiredWidth != mostDeservingDataGridViewColumn.DesiredFillWidth);
                                        if (!changeDone && widthLoss == 1 && oldDesiredWidth > mostDeservingDataGridViewColumn.MinimumWidth)
                                        {
                                            mostDeservingDataGridViewColumn.DesiredFillWidth--;
                                            changeDone = true;
                                        }
                                        Debug.Assert(oldDesiredWidth >= mostDeservingDataGridViewColumn.DesiredFillWidth);
                                        widthLoss -= oldDesiredWidth - mostDeservingDataGridViewColumn.DesiredFillWidth;
                                        if (changeDone)
                                        {
                                            stepDownAvailableWidthForFillColumns -= oldDesiredWidth - mostDeservingDataGridViewColumn.DesiredFillWidth;
                                            for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                                            {
                                                DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                                dataGridViewColumn.UsedFillWeight = weightSum / stepDownAvailableWidthForFillColumns * dataGridViewColumn.DesiredFillWidth;
                                            }
                                        }
                                        Debug.Assert(widthLoss >= 0);
                                    }
                                }
                                while (changeDone && widthLoss > 0);
                            }
                            while (cumulatedWidthLoss < totalWidthLoss);
                        }
                        availableWidthForFillColumns = availableWidth;
                    }

#if DEBUG
                    float weightSumDbg = 0F;
                    for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                    {
                        DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                        weightSumDbg += dataGridViewColumn.UsedFillWeight;

                    }
                    Debug.Assert(Math.Abs(weightSum - weightSumDbg) < 1.0F);
#endif

                    // Finally update the columns' width according the UsedFillWeight values.
                    try
                    {
                        dataGridViewState2[DATAGRIDVIEWSTATE2_allowHorizontalScrollbar] = false;
                        usedWidth = 0;
                        float carryover = 0F;
                        while (autoFillColumns.Count > 0)
                        {
                            DataGridViewColumn mostDeservingDataGridViewColumn = null;
                            if (autoFillColumns.Count == 1)
                            {
                                mostDeservingDataGridViewColumn = (DataGridViewColumn)autoFillColumns[0];
                                mostDeservingDataGridViewColumn.DesiredFillWidth = Math.Max(availableWidth - usedWidth, mostDeservingDataGridViewColumn.MinimumWidth);
                                autoFillColumns.Clear();
                            }
                            else
                            {
                                float biggestWeightDiscrepancy = 0F, weightDiscrepancy;
                                for (columnEntry = 0; columnEntry < autoFillColumns.Count; columnEntry++)
                                {
                                    DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)autoFillColumns[columnEntry];
                                    weightDiscrepancy = Math.Abs(dataGridViewColumn.UsedFillWeight - dataGridViewColumn.FillWeight) / dataGridViewColumn.FillWeight;
                                    if (weightDiscrepancy > biggestWeightDiscrepancy || mostDeservingDataGridViewColumn == null)
                                    {
                                        mostDeservingDataGridViewColumn = dataGridViewColumn;
                                        biggestWeightDiscrepancy = weightDiscrepancy;
                                    }
                                }
                                float floatDesiredWidth = (mostDeservingDataGridViewColumn.UsedFillWeight * availableWidth / weightSum) + carryover;
                                mostDeservingDataGridViewColumn.DesiredFillWidth = Math.Max(mostDeservingDataGridViewColumn.MinimumWidth, (int)Math.Round(floatDesiredWidth, MidpointRounding.AwayFromZero));
                                carryover = floatDesiredWidth - mostDeservingDataGridViewColumn.DesiredFillWidth;
                                usedWidth += mostDeservingDataGridViewColumn.DesiredFillWidth;
                                autoFillColumns.Remove(mostDeservingDataGridViewColumn);
                            }
                            if (mostDeservingDataGridViewColumn.DesiredFillWidth != mostDeservingDataGridViewColumn.Thickness)
                            {
                                columnsAdjusted = true;
                                mostDeservingDataGridViewColumn.ThicknessInternal = mostDeservingDataGridViewColumn.DesiredFillWidth;
                            }
                        }
                    }
                    finally
                    {
                        dataGridViewState2[DATAGRIDVIEWSTATE2_allowHorizontalScrollbar] = true;
                    }
                }
#if DEBUG
                if (!dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty])
                {
                    foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                    {
                        if (dataGridViewColumnTmp.Visible &&
                            dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                        {
                            Debug.Assert(Math.Abs(dataGridViewColumnTmp.UsedFillWeight * availableWidthForFillColumns - weightSum * dataGridViewColumnTmp.Width) / weightSum / dataGridViewColumnTmp.Width <= 1.25F / dataGridViewColumnTmp.Width);
                        }
                    }
                }

                bool nonMinColumnExists = false;
                int widthSum = 0;
                foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                {
                    if (dataGridViewColumnTmp.Visible &&
                        dataGridViewColumnTmp.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                    {
                        widthSum += dataGridViewColumnTmp.Width;
                        if (dataGridViewColumnTmp.Width > dataGridViewColumnTmp.MinimumWidth)
                        {
                            nonMinColumnExists = true;
                        }
                    }
                }
                if (nonMinColumnExists)
                {
                    Debug.Assert(widthSum == availableWidthForFillColumns);
                }
#endif
            }
            finally
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inAdjustFillingColumns] = false;
            }

            return columnsAdjusted;
        }

        private void AdjustShrinkingRows(DataGridViewAutoSizeRowsMode autoSizeRowsMode, bool fixedWidth, bool internalAutosizing)
        {
            if ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0 ||
                ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.Header) != 0 && RowHeadersVisible))
            {
                // Switch to batch operation
                inBulkPaintCount++;
                try
                {
                    if ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllRows) != 0)
                    {
                        // Very expensive processing - the developer should avoid this scenario.
                        inBulkLayoutCount++;
                        try
                        {
                            for (int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                                rowIndex != -1;
                                rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible))
                            {
                                AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), fixedWidth, internalAutosizing);
                            }
                        }
                        finally
                        {
                            ExitBulkLayout(false /*invalidInAdjustFillingColumns*/);
                        }
                    }
                    else
                    {
                        Debug.Assert((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.DisplayedRows) != 0);
                        int displayHeight = layout.Data.Height;
                        int cy = 0;

                        int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                        while (rowIndex != -1 && cy < displayHeight)
                        {
                            AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), fixedWidth, internalAutosizing);
                            cy += Rows.SharedRow(rowIndex).GetHeight(rowIndex);
                            rowIndex = Rows.GetNextRow(rowIndex,
                                DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                        }

                        if (cy < displayHeight)
                        {
                            int cyFrozen = cy;
                            int oldFirstVisibleScrollingRow = displayedBandsInfo.FirstDisplayedScrollingRow;
                            rowIndex = oldFirstVisibleScrollingRow;
                            while (rowIndex != -1 &&
                                cy < displayHeight &&
                                oldFirstVisibleScrollingRow == displayedBandsInfo.FirstDisplayedScrollingRow)
                            {
                                AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), fixedWidth, internalAutosizing);
                                cy += Rows.SharedRow(rowIndex).GetHeight(rowIndex);
                                rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                            }

                            do
                            {
                                oldFirstVisibleScrollingRow = displayedBandsInfo.FirstDisplayedScrollingRow;
                                if (cy < displayHeight)
                                {
                                    int rowAboveFirstVisibleScrollingRow = Rows.GetPreviousRow(displayedBandsInfo.FirstDisplayedScrollingRow, DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                                    if (rowAboveFirstVisibleScrollingRow != -1)
                                    {
                                        AutoResizeRowInternal(rowAboveFirstVisibleScrollingRow, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), fixedWidth, internalAutosizing);
                                    }
                                }
                                cy = cyFrozen;
                                rowIndex = displayedBandsInfo.FirstDisplayedScrollingRow;
                                while (rowIndex != -1 && cy < displayHeight)
                                {
                                    AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), fixedWidth, internalAutosizing);
                                    cy += Rows.SharedRow(rowIndex).GetHeight(rowIndex);
                                    rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                                }
                            }
                            while (oldFirstVisibleScrollingRow != displayedBandsInfo.FirstDisplayedScrollingRow);
                        }
                    }
                }
                finally
                {
                    ExitBulkPaint(-1, -1);
                }
            }
        }

        public bool AreAllCellsSelected(bool includeInvisibleCells)
        {
            if (Columns.Count == 0 && Rows.Count == 0)
            {
                return true;
            }
            if (!includeInvisibleCells &&
                (Rows.GetFirstRow(DataGridViewElementStates.Visible) == -1 ||
                 Columns.GetFirstColumn(DataGridViewElementStates.Visible) == null))
            {
                return true;
            }

            DataGridViewRow dataGridViewRow = null;
            bool allCellsSelected;
            switch (SelectionMode)
            {
                case DataGridViewSelectionMode.CellSelect:
                    {
                        allCellsSelected = individualSelectedCells.Count == Columns.Count * Rows.Count;
                        if (allCellsSelected || includeInvisibleCells)
                        {
                            return allCellsSelected;
                        }
                        else
                        {
                            for (int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                                 rowIndex != -1;
                                 rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible))
                            {
                                dataGridViewRow = Rows[rowIndex]; // unshares this row

                                DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
                                while (dataGridViewColumn != null)
                                {
                                    if (!dataGridViewRow.Cells[dataGridViewColumn.Index].Selected)
                                    {
                                        return false;
                                    }
                                    dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                                        DataGridViewElementStates.Visible,
                                        DataGridViewElementStates.None);
                                }
                            }
                            return true;
                        }
                    }
                case DataGridViewSelectionMode.FullColumnSelect:
                case DataGridViewSelectionMode.ColumnHeaderSelect:
                    {
                        allCellsSelected = selectedBandIndexes.Count * Rows.Count + individualSelectedCells.Count == Columns.Count * Rows.Count;
                        if (allCellsSelected || includeInvisibleCells)
                        {
                            return allCellsSelected;
                        }
                        else
                        {
                            DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
                            while (dataGridViewColumn != null)
                            {
                                if (!selectedBandIndexes.Contains(dataGridViewColumn.Index))
                                {
                                    for (int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                                        rowIndex != -1;
                                        rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible))
                                    {
                                        dataGridViewRow = Rows[rowIndex]; // unshares this row
                                        if (!dataGridViewRow.Cells[dataGridViewColumn.Index].Selected)
                                        {
                                            return false;
                                        }
                                    }
                                }
                                dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                                    DataGridViewElementStates.Visible,
                                    DataGridViewElementStates.None);
                            }
                            return true;
                        }
                    }
                case DataGridViewSelectionMode.FullRowSelect:
                case DataGridViewSelectionMode.RowHeaderSelect:
                    {
                        allCellsSelected = selectedBandIndexes.Count * Columns.Count + individualSelectedCells.Count == Columns.Count * Rows.Count;
                        if (allCellsSelected || includeInvisibleCells)
                        {
                            return allCellsSelected;
                        }
                        else
                        {
                            for (int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                                rowIndex != -1;
                                rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible))
                            {
                                Debug.Assert(selectedBandIndexes.Contains(rowIndex) ==
                                             ((Rows.GetRowState(rowIndex) & DataGridViewElementStates.Selected) != 0));
                                if ((Rows.GetRowState(rowIndex) & DataGridViewElementStates.Selected) == 0)
                                {
                                    dataGridViewRow = Rows[rowIndex]; // unshares this row
                                    DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
                                    while (dataGridViewColumn != null)
                                    {
                                        if (!dataGridViewRow.Cells[dataGridViewColumn.Index].Selected)
                                        {
                                            return false;
                                        }
                                        dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                                            DataGridViewElementStates.Visible,
                                            DataGridViewElementStates.None);
                                    }
                                }
                            }
                            return true;
                        }
                    }
            }
            Debug.Fail("DataGridView.AreAllCellsSelected - Shouldn't reach this code");
            return false;
        }

        /// <summary>
        ///  Assigns a new parent control to the DataGridView.
        /// </summary>
        internal override void AssignParent(Control value)
        {
            if (toolTipControl.Activated)
            {
                toolTipControl.Activate(false /*activate*/);
            }
            base.AssignParent(value);
        }

        private void AutoGenerateDataBoundColumns(DataGridViewColumn[] boundColumns)
        {
            DataGridViewColumnCollection dataGridViewCols = Columns;
            DataGridViewColumn[] clonedColumns = new DataGridViewColumn[dataGridViewCols.Count];
            int clonedColumnsCount = 0;
            int i;

            // 1. Clone all the columns which are currently bound and will also be bound under the new DataSource/DataMember combination.
            // 2. Delete all the currently bound columns.
            // 3. Sort the cloned columns in order of their DisplayIndex.
            // 4. Add the new data bound columns. Here are the rules:
            //      a. if the cloned collection contains (possible multiple) columns with DataProperyHame == current data field,
            //          add the first cloned column that maps to the data field.
            //      b. other wise add the new bound column.
            // 5. Add the remaining cloned columns in the order of their DisplayIndex.

            // 1. Clone all the currently bound columns.
            //
            // We can't do 1. and 2. in the same loop because we need to save the DisplayIndex.
            for (i = 0; i < dataGridViewCols.Count; i++)
            {
                if (DataSource != null &&
                    !string.IsNullOrEmpty(dataGridViewCols[i].DataPropertyName) &&
                    !dataGridViewCols[i].IsDataBound)
                {
                    MapDataGridViewColumnToDataBoundField(dataGridViewCols[i]);
                }

                if (dataGridViewCols[i].IsDataBound)
                {
                    // We only clone columns which are data bound w/ the new DataSource/DataMember combination.
                    if (dataConnection != null && dataConnection.BoundColumnIndex(dataGridViewCols[i].DataPropertyName) != -1)
                    {
                        clonedColumns[clonedColumnsCount] = (DataGridViewColumn)dataGridViewCols[i].Clone();
                        clonedColumns[clonedColumnsCount].DisplayIndex = dataGridViewCols[i].DisplayIndex;
                        clonedColumnsCount++;
                    }
                }
            }

            i = 0;
            // 2. Delete all the currently bound columns.
            while (i < dataGridViewCols.Count)
            {
                if (dataGridViewCols[i].IsDataBound)
                {
                    dataGridViewCols.RemoveAtInternal(i, true /*force*/);
                }
                else
                {
                    i++;
                }
            }

            // 3. Sort the cloned columns in the order of their DisplayIndex.

            // Sort the cloned columns array by the display index.
            // We need to copy the cloned columns into a possibly smaller array.
            DataGridViewColumn[] finalClonedColumns;
            if (clonedColumns.Length == clonedColumnsCount)
            {
                finalClonedColumns = clonedColumns;
            }
            else
            {
                finalClonedColumns = new DataGridViewColumn[clonedColumnsCount];
                Array.Copy(clonedColumns, finalClonedColumns, clonedColumnsCount);
            }

            // Sort the array.
            Array.Sort(finalClonedColumns, System.Windows.Forms.DataGridViewColumnCollection.ColumnCollectionOrderComparer);

            // 4. Add new columns for the Fields which were not data bound previously ( ie, for fields which do not have a clone ).
            if (boundColumns != null)
            {
                for (int j = 0; j < boundColumns.Length; j++)
                {
                    if (boundColumns[j] != null && boundColumns[j].IsBrowsableInternal)
                    {
                        bool addNewColumn = true;
                        // Go thru the list of cloned columns and see if there is another column w/ the same data property name.
                        int clonedColIndex = 0;
                        for (; clonedColIndex < clonedColumnsCount; clonedColIndex++)
                        {
                            if (finalClonedColumns[clonedColIndex] != null &&
                                string.Compare(finalClonedColumns[clonedColIndex].DataPropertyName,
                                    boundColumns[j].DataPropertyName,
                                    true /*ignoreCase*/,
                                    CultureInfo.InvariantCulture) == 0)
                            {
                                addNewColumn = false;
                                break;
                            }
                        }

                        if (addNewColumn)
                        {
                            dataGridViewCols.Add(boundColumns[j]);
                        }
                        else
                        {
                            // add the cloned column.
                            dataGridViewCols.Add(finalClonedColumns[clonedColIndex]);
                            MapDataGridViewColumnToDataBoundField(finalClonedColumns[clonedColIndex]);
                            Debug.Assert(finalClonedColumns[clonedColIndex].IsDataBound);
                            finalClonedColumns[clonedColIndex] = null;
                        }
                    }
                }
            }
#if DEBUG
            else
            {
                // If there are no data bound columns then there are no cloned columns either.
                Debug.Assert(finalClonedColumns.Length == 0);
                Debug.Assert(clonedColumnsCount == 0);
            }
#endif // DEBUG

            // 5. Add remaining cloned columns.
            if (clonedColumnsCount > 0)
            {
                for (int k = 0; k < finalClonedColumns.Length; k++)
                {
                    if (finalClonedColumns[k] != null)
                    {
                        dataGridViewCols.Add(finalClonedColumns[k]);
                        MapDataGridViewColumnToDataBoundField(finalClonedColumns[k]);
                        Debug.Assert(finalClonedColumns[k].IsDataBound);
                    }
                }
            }
        }

        private bool AutoResizeAllVisibleColumnsInternal(DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaFilter, bool fixedHeight)
        {
            Debug.Assert(autoSizeColumnCriteriaFilter != DataGridViewAutoSizeColumnCriteriaInternal.None);
            Debug.Assert((autoSizeColumnCriteriaFilter & DataGridViewAutoSizeColumnCriteriaInternal.Fill) == 0);

            bool ret = false; // No column autosizes by default
            DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
            while (dataGridViewColumn != null)
            {
                DataGridViewAutoSizeColumnCriteriaInternal inheritedAutoSizeColumnCriteria = (DataGridViewAutoSizeColumnCriteriaInternal)dataGridViewColumn.InheritedAutoSizeMode;
                DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaFiltered = (inheritedAutoSizeColumnCriteria & autoSizeColumnCriteriaFilter);
                if (autoSizeColumnCriteriaFiltered != 0)
                {
                    ret |= AutoResizeColumnInternal(dataGridViewColumn.Index, inheritedAutoSizeColumnCriteria, fixedHeight);
                }
                dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                    DataGridViewElementStates.Visible,
                    DataGridViewElementStates.None);
            }
            return ret;
        }

        public void AutoResizeColumn(int columnIndex)
        {
            AutoResizeColumn(columnIndex, DataGridViewAutoSizeColumnMode.AllCells);
        }

        public void AutoResizeColumn(int columnIndex, DataGridViewAutoSizeColumnMode autoSizeColumnMode)
        {
            AutoResizeColumn(columnIndex, autoSizeColumnMode, true);
        }

        protected void AutoResizeColumn(int columnIndex, DataGridViewAutoSizeColumnMode autoSizeColumnMode, bool fixedHeight)
        {
            if (autoSizeColumnMode == DataGridViewAutoSizeColumnMode.NotSet ||
                autoSizeColumnMode == DataGridViewAutoSizeColumnMode.None ||
                autoSizeColumnMode == DataGridViewAutoSizeColumnMode.Fill)
            {
                throw new ArgumentException(string.Format(SR.DataGridView_NeedColumnAutoSizingCriteria, "autoSizeColumnMode"));
            }

            switch (autoSizeColumnMode)
            {
                case DataGridViewAutoSizeColumnMode.NotSet:
                case DataGridViewAutoSizeColumnMode.None:
                case DataGridViewAutoSizeColumnMode.ColumnHeader:
                case DataGridViewAutoSizeColumnMode.AllCellsExceptHeader:
                case DataGridViewAutoSizeColumnMode.AllCells:
                case DataGridViewAutoSizeColumnMode.DisplayedCellsExceptHeader:
                case DataGridViewAutoSizeColumnMode.DisplayedCells:
                case DataGridViewAutoSizeColumnMode.Fill:
                    break;
                default:
                    throw new InvalidEnumArgumentException(nameof(autoSizeColumnMode), (int)autoSizeColumnMode, typeof(DataGridViewAutoSizeColumnMode));
            }

            if (columnIndex < 0 || columnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(columnIndex));
            }

            if (autoSizeColumnMode == DataGridViewAutoSizeColumnMode.ColumnHeader && !ColumnHeadersVisible)
            {
                throw new InvalidOperationException(SR.DataGridView_CannotAutoSizeInvisibleColumnHeader);
            }

            AutoResizeColumnInternal(columnIndex, (DataGridViewAutoSizeColumnCriteriaInternal)autoSizeColumnMode, fixedHeight);
        }

        public void AutoResizeColumnHeadersHeight()
        {
            AutoResizeColumnHeadersHeight(true, true);
        }

        public void AutoResizeColumnHeadersHeight(int columnIndex)
        {
            AutoResizeColumnHeadersHeight(columnIndex, true, true);
        }

        protected void AutoResizeColumnHeadersHeight(bool fixedRowHeadersWidth, bool fixedColumnsWidth)
        {
            if (!ColumnHeadersVisible)
            {
                return;
            }

            if (!IsHandleCreated)
            {
                // auto sizing causes handle creation.
                // don't create the handle inside InitializeComponent because that causes problems w/ data binding
                dataGridViewState2[DATAGRIDVIEWSTATE2_autoSizedWithoutHandle] = true;
                return;
            }

            try
            {
                noAutoSizeCount++;
                // Use of WindowsFormsUtils.CreateMeasurementGraphics() avoid use of this.Handle
                // IntPtr handle = this.Handle; // Force creation of control's handle because for databound grids, handle creation wipes out and recreates the columns/rows.
                int preferredHeight = 0;
                if (layout.TopLeftHeader.Width > 0)
                {
                    if (fixedRowHeadersWidth)
                    {
                        preferredHeight = TopLeftHeaderCell.GetPreferredHeight(-1, layout.TopLeftHeader.Width);
                    }
                    else
                    {
                        preferredHeight = TopLeftHeaderCell.GetPreferredSize(-1).Height;
                    }
                }
                int columnsCount = Columns.Count;
                for (int columnIndex = 0; columnIndex < columnsCount; columnIndex++)
                {
                    if (Columns[columnIndex].Visible)
                    {
                        if (fixedColumnsWidth)
                        {
                            preferredHeight = Math.Max(preferredHeight, Columns[columnIndex].HeaderCell.GetPreferredHeight(-1, Columns[columnIndex].Thickness));
                        }
                        else
                        {
                            preferredHeight = Math.Max(preferredHeight, Columns[columnIndex].HeaderCell.GetPreferredSize(-1).Height);
                        }
                    }
                }
                if (preferredHeight < minimumColumnHeadersHeight)
                {
                    preferredHeight = minimumColumnHeadersHeight;
                }
                if (preferredHeight > maxHeadersThickness)
                {
                    preferredHeight = maxHeadersThickness;
                }
                if (preferredHeight != ColumnHeadersHeight)
                {
                    SetColumnHeadersHeightInternal(preferredHeight, !fixedColumnsWidth /*invalidInAdjustFillingColumns*/);
                }
            }
            finally
            {
                Debug.Assert(noAutoSizeCount > 0);
                noAutoSizeCount--;
            }
        }

        protected void AutoResizeColumnHeadersHeight(int columnIndex, bool fixedRowHeadersWidth, bool fixedColumnWidth)
        {
            if (columnIndex < -1 || columnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(columnIndex));
            }

            if (!ColumnHeadersVisible)
            {
                return;
            }

            if (!IsHandleCreated)
            {
                // auto sizing causes handle creation.
                // don't create the handle inside InitializeComponent because that causes problems w/ data binding
                dataGridViewState2[DATAGRIDVIEWSTATE2_autoSizedWithoutHandle] = true;
                return;
            }

            try
            {
                noAutoSizeCount++;
                // Use of WindowsFormsUtils.CreateMeasurementGraphics() avoid use of this.Handle
                // IntPtr handle = this.Handle; // Force creation of control's handle because for databound grids, handle creation wipes out and recreates the columns/rows.
                int preferredHeight = 0;
                if (layout.TopLeftHeader.Width > 0)
                {
                    if (columnIndex != -1 || fixedRowHeadersWidth)
                    {
                        preferredHeight = TopLeftHeaderCell.GetPreferredHeight(-1, layout.TopLeftHeader.Width);
                    }
                    else
                    {
                        preferredHeight = TopLeftHeaderCell.GetPreferredSize(-1).Height;
                    }
                }
                int columnsCount = Columns.Count;
                for (int columnIndexTmp = 0; columnIndexTmp < columnsCount; columnIndexTmp++)
                {
                    if (Columns[columnIndexTmp].Visible)
                    {
                        if (columnIndex != columnIndexTmp || fixedColumnWidth)
                        {
                            preferredHeight = Math.Max(preferredHeight, Columns[columnIndexTmp].HeaderCell.GetPreferredHeight(-1, Columns[columnIndexTmp].Thickness));
                        }
                        else
                        {
                            preferredHeight = Math.Max(preferredHeight, Columns[columnIndexTmp].HeaderCell.GetPreferredSize(-1).Height);
                        }
                    }
                }
                if (preferredHeight < minimumColumnHeadersHeight)
                {
                    preferredHeight = minimumColumnHeadersHeight;
                }
                if (preferredHeight > maxHeadersThickness)
                {
                    preferredHeight = maxHeadersThickness;
                }
                if (preferredHeight != ColumnHeadersHeight)
                {
                    SetColumnHeadersHeightInternal(preferredHeight, !fixedColumnWidth /*invalidInAdjustFillingColumns*/);
                }
            }
            finally
            {
                Debug.Assert(noAutoSizeCount > 0);
                noAutoSizeCount--;
            }
        }

        private bool AutoResizeColumnInternal(int columnIndex, DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaInternal, bool fixedHeight)
        {
            Debug.Assert(autoSizeColumnCriteriaInternal == DataGridViewAutoSizeColumnCriteriaInternal.Header ||
                autoSizeColumnCriteriaInternal == DataGridViewAutoSizeColumnCriteriaInternal.AllRows ||
                autoSizeColumnCriteriaInternal == DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows ||
                autoSizeColumnCriteriaInternal == (DataGridViewAutoSizeColumnCriteriaInternal.Header | DataGridViewAutoSizeColumnCriteriaInternal.AllRows) ||
                autoSizeColumnCriteriaInternal == (DataGridViewAutoSizeColumnCriteriaInternal.Header | DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows));
            Debug.Assert(columnIndex >= 0 && columnIndex < Columns.Count);
            Debug.Assert(autoSizeColumnCriteriaInternal != DataGridViewAutoSizeColumnCriteriaInternal.Header || ColumnHeadersVisible);

            if (!IsHandleCreated)
            {
                // auto sizing causes handle creation.
                // don't create the handle inside InitializeComponent because that causes problems w/ data binding
                dataGridViewState2[DATAGRIDVIEWSTATE2_autoSizedWithoutHandle] = true;
                return false;
            }

            bool ret = false; // No autosizing occurs by default.
            try
            {
                noAutoSizeCount++;
                // Use of WindowsFormsUtils.CreateMeasurementGraphics() avoid use of this.Handle
                // IntPtr handle = this.Handle; // Force creation of control's handle because for databound grids, handle creation wipes out and recreates the columns.
                // Note: Even none-resizable column can programmatically be resized.
                DataGridViewColumn dataGridViewColumn = Columns[columnIndex];
                int preferredColumnWidth = dataGridViewColumn.GetPreferredWidth((DataGridViewAutoSizeColumnMode)autoSizeColumnCriteriaInternal, fixedHeight);
                if (preferredColumnWidth < dataGridViewColumn.MinimumThickness)
                {
                    preferredColumnWidth = dataGridViewColumn.MinimumThickness;
                }
                if (preferredColumnWidth > DataGridViewBand.MaxBandThickness)
                {
                    preferredColumnWidth = DataGridViewBand.MaxBandThickness;
                }
                if (preferredColumnWidth != dataGridViewColumn.Thickness)
                {
                    if (dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                    {
                        AdjustFillingColumn(dataGridViewColumn, preferredColumnWidth);
                    }
                    else
                    {
                        Columns[columnIndex].ThicknessInternal = preferredColumnWidth;
                    }
                    ret = true;
                }
            }
            finally
            {
                Debug.Assert(noAutoSizeCount > 0);
                noAutoSizeCount--;
            }
            return ret;
        }

        public void AutoResizeColumns()
        {
            AutoResizeColumns(DataGridViewAutoSizeColumnsMode.AllCells);
        }

        public void AutoResizeColumns(DataGridViewAutoSizeColumnsMode autoSizeColumnsMode)
        {
            AutoResizeColumns(autoSizeColumnsMode, true);
        }

        protected void AutoResizeColumns(DataGridViewAutoSizeColumnsMode autoSizeColumnsMode, bool fixedHeight)
        {
            for (int columnIndex = 0; columnIndex < Columns.Count; columnIndex++)
            {
                AutoResizeColumn(columnIndex, (DataGridViewAutoSizeColumnMode)autoSizeColumnsMode, fixedHeight);
            }
        }

        public void AutoResizeRow(int rowIndex)
        {
            AutoResizeRow(rowIndex, DataGridViewAutoSizeRowMode.AllCells);
        }

        public void AutoResizeRow(int rowIndex, DataGridViewAutoSizeRowMode autoSizeRowMode)
        {
            AutoResizeRow(rowIndex, autoSizeRowMode, true /*fixedWidth*/);
        }

        protected void AutoResizeRow(int rowIndex, DataGridViewAutoSizeRowMode autoSizeRowMode, bool fixedWidth)
        {
            if (rowIndex < 0 || rowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(rowIndex));
            }
            // not using ClientUtils here because it's a flags enum, masking instead.
            if (((DataGridViewAutoSizeRowCriteriaInternal)autoSizeRowMode & invalidDataGridViewAutoSizeRowCriteriaInternalMask) != 0)
            {
                throw new InvalidEnumArgumentException(nameof(autoSizeRowMode), (int)autoSizeRowMode, typeof(DataGridViewAutoSizeRowMode));
            }
            if (autoSizeRowMode == DataGridViewAutoSizeRowMode.RowHeader && !RowHeadersVisible)
            {
                throw new InvalidOperationException(SR.DataGridView_CannotAutoSizeRowInvisibleRowHeader);
            }
            AutoResizeRowInternal(rowIndex, autoSizeRowMode, fixedWidth, false /*internalAutosizing*/);
        }

        // User can override this if there is a quicker way to determine preferred row headers width
        public void AutoResizeRowHeadersWidth(DataGridViewRowHeadersWidthSizeMode rowHeadersWidthSizeMode)
        {
            AutoResizeRowHeadersWidth(rowHeadersWidthSizeMode,
                                      true /*fixedColumnHeadersHeight*/,
                                      true /*fixedRowsHeight*/);
        }

        // User can override this if there is a quicker way to determine preferred row headers width
        protected void AutoResizeRowHeadersWidth(DataGridViewRowHeadersWidthSizeMode rowHeadersWidthSizeMode,
                                                 bool fixedColumnHeadersHeight,
                                                 bool fixedRowsHeight)
        {
            if (rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing ||
                rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.DisableResizing)
            {
                throw new ArgumentException(string.Format(SR.DataGridView_NeedAutoSizingCriteria, "rowHeadersWidthSizeMode"));
            }
            // custom range checking, not using ClientUtils.
            if (rowHeadersWidthSizeMode < DataGridViewRowHeadersWidthSizeMode.AutoSizeToAllHeaders || rowHeadersWidthSizeMode > DataGridViewRowHeadersWidthSizeMode.AutoSizeToFirstHeader)
            {
                throw new InvalidEnumArgumentException(nameof(rowHeadersWidthSizeMode), (int)rowHeadersWidthSizeMode, typeof(DataGridViewRowHeadersWidthSizeMode));
            }

            if (!RowHeadersVisible)
            {
                return;
            }

            if (!IsHandleCreated)
            {
                // auto sizing causes handle creation.
                // don't create the handle inside InitializeComponent because that causes problems w/ data binding
                dataGridViewState2[DATAGRIDVIEWSTATE2_autoSizedWithoutHandle] = true;
                return;
            }

            try
            {
                noAutoSizeCount++;
                // Use of WindowsFormsUtils.CreateMeasurementGraphics() avoid use of this.Handle
                // IntPtr handle = this.Handle; // Force creation of control's handle because for databound grids, handle creation wipes out and recreates the columns/rows.
                int preferredWidth = 0, rowIndex;
                if (layout.TopLeftHeader.Width > 0)
                {
                    if (fixedColumnHeadersHeight)
                    {
                        preferredWidth = TopLeftHeaderCell.GetPreferredWidth(-1, layout.TopLeftHeader.Height);
                    }
                    else
                    {
                        preferredWidth = TopLeftHeaderCell.GetPreferredSize(-1).Width;
                    }
                }
                switch (rowHeadersWidthSizeMode)
                {
                    case DataGridViewRowHeadersWidthSizeMode.AutoSizeToFirstHeader:
                        {
                            rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                            if (rowIndex != -1)
                            {
                                if (fixedRowsHeight)
                                {
                                    preferredWidth = Math.Max(preferredWidth, Rows.SharedRow(rowIndex).HeaderCell.GetPreferredWidth(rowIndex, Rows.SharedRow(rowIndex).GetHeight(rowIndex)));
                                }
                                else
                                {
                                    preferredWidth = Math.Max(preferredWidth, Rows.SharedRow(rowIndex).HeaderCell.GetPreferredSize(rowIndex).Width);
                                }
                            }
                            break;
                        }
                    case DataGridViewRowHeadersWidthSizeMode.AutoSizeToDisplayedHeaders:
                        {
                            int displayHeight = layout.Data.Height, cy = 0;
                            rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                            while (rowIndex != -1 && cy < displayHeight)
                            {
                                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                                int dataGridViewRowHeight = dataGridViewRow.GetHeight(rowIndex);
                                cy += dataGridViewRowHeight;
                                if (fixedRowsHeight)
                                {
                                    preferredWidth = Math.Max(preferredWidth, dataGridViewRow.HeaderCell.GetPreferredWidth(rowIndex, dataGridViewRowHeight));
                                }
                                else
                                {
                                    preferredWidth = Math.Max(preferredWidth, dataGridViewRow.HeaderCell.GetPreferredSize(rowIndex).Width);
                                }
                                rowIndex = Rows.GetNextRow(rowIndex,
                                                                DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                            }
                            if (cy < displayHeight)
                            {
                                rowIndex = displayedBandsInfo.FirstDisplayedScrollingRow;
                                while (rowIndex != -1 && cy < displayHeight)
                                {
                                    DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                                    int dataGridViewRowHeight = dataGridViewRow.GetHeight(rowIndex);
                                    cy += dataGridViewRowHeight;
                                    if (fixedRowsHeight)
                                    {
                                        preferredWidth = Math.Max(preferredWidth, dataGridViewRow.HeaderCell.GetPreferredWidth(rowIndex, dataGridViewRowHeight));
                                    }
                                    else
                                    {
                                        preferredWidth = Math.Max(preferredWidth, dataGridViewRow.HeaderCell.GetPreferredSize(rowIndex).Width);
                                    }
                                    rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                                }
                            }
                            break;
                        }
                    case DataGridViewRowHeadersWidthSizeMode.AutoSizeToAllHeaders:
                        {
                            for (rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                                rowIndex != -1;
                                rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible))
                            {
                                if (fixedRowsHeight)
                                {
                                    preferredWidth = Math.Max(preferredWidth, Rows.SharedRow(rowIndex).HeaderCell.GetPreferredWidth(rowIndex, Rows.SharedRow(rowIndex).GetHeight(rowIndex)));
                                }
                                else
                                {
                                    preferredWidth = Math.Max(preferredWidth, Rows.SharedRow(rowIndex).HeaderCell.GetPreferredSize(rowIndex).Width);
                                }
                            }
                            break;
                        }
                    default:
                        {
                            Debug.Fail("Unexpected rowHeadersWidthSizeMode value in AutoResizeRowHeadersWidth");
                            break;
                        }
                }
                if (preferredWidth < minimumRowHeadersWidth)
                {
                    preferredWidth = minimumRowHeadersWidth;
                }
                if (preferredWidth != RowHeadersWidth)
                {
                    RowHeadersWidthInternal = preferredWidth;
                }
            }
            finally
            {
                Debug.Assert(noAutoSizeCount > 0);
                noAutoSizeCount--;
            }
        }

        public void AutoResizeRowHeadersWidth(int rowIndex, DataGridViewRowHeadersWidthSizeMode rowHeadersWidthSizeMode)
        {
            AutoResizeRowHeadersWidth(rowIndex,
                                      rowHeadersWidthSizeMode,
                                      true /*fixedColumnHeadersHeight*/,
                                      true /*fixedRowHeight*/);
        }

        protected void AutoResizeRowHeadersWidth(int rowIndex,
                                                 DataGridViewRowHeadersWidthSizeMode rowHeadersWidthSizeMode,
                                                 bool fixedColumnHeadersHeight,
                                                 bool fixedRowHeight)
        {
            if (rowIndex < -1 || rowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(rowIndex));
            }
            if (rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing ||
                rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.DisableResizing)
            {
                throw new ArgumentException(string.Format(SR.DataGridView_NeedAutoSizingCriteria, "rowHeadersWidthSizeMode"));
            }
            if (rowHeadersWidthSizeMode < DataGridViewRowHeadersWidthSizeMode.AutoSizeToAllHeaders || rowHeadersWidthSizeMode > DataGridViewRowHeadersWidthSizeMode.AutoSizeToFirstHeader)
            {
                throw new InvalidEnumArgumentException(nameof(rowHeadersWidthSizeMode), (int)rowHeadersWidthSizeMode, typeof(DataGridViewRowHeadersWidthSizeMode));
            }

            if (!RowHeadersVisible)
            {
                return;
            }
            if (rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.AutoSizeToFirstHeader &&
                rowIndex != -1 &&
                rowIndex != Rows.GetFirstRow(DataGridViewElementStates.Visible))
            {
                return;
            }
            if (rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.AutoSizeToDisplayedHeaders &&
                rowIndex != -1)
            {
                DataGridViewElementStates rowState = Rows.GetRowState(rowIndex);
                bool rowDisplayed = (rowState & DataGridViewElementStates.Displayed) != 0;
                if (!rowDisplayed)
                {
                    return;
                }
            }

            if (!IsHandleCreated)
            {
                // auto sizing causes handle creation.
                // don't create the handle inside InitializeComponent because that causes problems w/ data binding
                dataGridViewState2[DATAGRIDVIEWSTATE2_autoSizedWithoutHandle] = true;
                return;
            }

            try
            {
                noAutoSizeCount++;
                // Use of WindowsFormsUtils.CreateMeasurementGraphics() avoid use of this.Handle
                // IntPtr handle = this.Handle; // Force creation of control's handle because for databound grids, handle creation wipes out and recreates the columns/rows.
                int preferredWidth = 0, rowIndexTmp;
                if (layout.TopLeftHeader.Width > 0)
                {
                    if (rowIndex != -1 || fixedColumnHeadersHeight)
                    {
                        preferredWidth = TopLeftHeaderCell.GetPreferredWidth(-1, layout.TopLeftHeader.Height);
                    }
                    else
                    {
                        preferredWidth = TopLeftHeaderCell.GetPreferredSize(-1).Width;
                    }
                }
                switch (rowHeadersWidthSizeMode)
                {
                    case DataGridViewRowHeadersWidthSizeMode.AutoSizeToFirstHeader:
                        {
                            rowIndexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                            if (rowIndexTmp != -1)
                            {
                                if (rowIndex != rowIndexTmp || fixedRowHeight)
                                {
                                    preferredWidth = Math.Max(preferredWidth, Rows.SharedRow(rowIndexTmp).HeaderCell.GetPreferredWidth(rowIndexTmp, Rows.SharedRow(rowIndexTmp).GetHeight(rowIndexTmp)));
                                }
                                else
                                {
                                    preferredWidth = Math.Max(preferredWidth, Rows.SharedRow(rowIndexTmp).HeaderCell.GetPreferredSize(rowIndexTmp).Width);
                                }
                            }
                            break;
                        }
                    case DataGridViewRowHeadersWidthSizeMode.AutoSizeToDisplayedHeaders:
                        {
                            int displayHeight = layout.Data.Height, cy = 0;
                            rowIndexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                            while (rowIndexTmp != -1 && cy < displayHeight)
                            {
                                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndexTmp);
                                int dataGridViewRowHeight = dataGridViewRow.GetHeight(rowIndexTmp);
                                cy += dataGridViewRowHeight;
                                if (rowIndex != rowIndexTmp || fixedRowHeight)
                                {
                                    preferredWidth = Math.Max(preferredWidth, dataGridViewRow.HeaderCell.GetPreferredWidth(rowIndexTmp, dataGridViewRowHeight));
                                }
                                else
                                {
                                    preferredWidth = Math.Max(preferredWidth, dataGridViewRow.HeaderCell.GetPreferredSize(rowIndexTmp).Width);
                                }
                                rowIndexTmp = Rows.GetNextRow(rowIndexTmp,
                                                                DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                            }
                            if (cy < displayHeight)
                            {
                                rowIndexTmp = displayedBandsInfo.FirstDisplayedScrollingRow;
                                while (rowIndexTmp != -1 && cy < displayHeight)
                                {
                                    DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndexTmp);
                                    int dataGridViewRowHeight = dataGridViewRow.GetHeight(rowIndexTmp);
                                    cy += dataGridViewRowHeight;
                                    if (rowIndex != rowIndexTmp || fixedRowHeight)
                                    {
                                        preferredWidth = Math.Max(preferredWidth, dataGridViewRow.HeaderCell.GetPreferredWidth(rowIndexTmp, dataGridViewRowHeight));
                                    }
                                    else
                                    {
                                        preferredWidth = Math.Max(preferredWidth, dataGridViewRow.HeaderCell.GetPreferredSize(rowIndexTmp).Width);
                                    }
                                    rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible);
                                }
                            }
                            break;
                        }
                    case DataGridViewRowHeadersWidthSizeMode.AutoSizeToAllHeaders:
                        {
                            for (rowIndexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                                rowIndexTmp != -1;
                                rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible))
                            {
                                if (rowIndex != rowIndexTmp || fixedRowHeight)
                                {
                                    preferredWidth = Math.Max(preferredWidth, Rows.SharedRow(rowIndexTmp).HeaderCell.GetPreferredWidth(rowIndexTmp, Rows.SharedRow(rowIndexTmp).GetHeight(rowIndexTmp)));
                                }
                                else
                                {
                                    preferredWidth = Math.Max(preferredWidth, Rows.SharedRow(rowIndexTmp).HeaderCell.GetPreferredSize(rowIndexTmp).Width);
                                }
                            }
                            break;
                        }
                    default:
                        {
                            Debug.Fail("Unexpected rowHeadersWidthSizeMode value in AutoResizeRowHeadersWidth");
                            break;
                        }
                }
                if (preferredWidth < minimumRowHeadersWidth)
                {
                    preferredWidth = minimumRowHeadersWidth;
                }
                if (preferredWidth != RowHeadersWidth)
                {
                    RowHeadersWidthInternal = preferredWidth;
                }
            }
            finally
            {
                Debug.Assert(noAutoSizeCount > 0);
                noAutoSizeCount--;
            }
        }

        private void AutoResizeRowInternal(int rowIndex, DataGridViewAutoSizeRowMode autoSizeRowMode, bool fixedWidth, bool internalAutosizing)
        {
            Debug.Assert(rowIndex >= 0 && rowIndex < Rows.Count);
            Debug.Assert(((DataGridViewAutoSizeRowCriteriaInternal)autoSizeRowMode & invalidDataGridViewAutoSizeRowCriteriaInternalMask) == 0);

            if (!IsHandleCreated)
            {
                // auto sizing causes handle creation.
                // don't create the handle inside InitializeComponent because that causes problems w/ data binding
                dataGridViewState2[DATAGRIDVIEWSTATE2_autoSizedWithoutHandle] = true;
                return;
            }

            try
            {
                noAutoSizeCount++;
                // Use of WindowsFormsUtils.CreateMeasurementGraphics() avoid use of this.Handle
                // IntPtr handle = this.Handle; // Force creation of control's handle because for databound grids, handle creation wipes out and recreates the columns/rows.
                // Note: Even none-resizable row can programmatically be resized.
                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                dataGridViewRow.GetHeightInfo(rowIndex, out int height, out int minimumHeight);
                int preferredThickness = dataGridViewRow.GetPreferredHeight(rowIndex, autoSizeRowMode, fixedWidth);
                if (preferredThickness < minimumHeight)
                {
                    preferredThickness = minimumHeight;
                }
                if (preferredThickness > DataGridViewBand.MaxBandThickness)
                {
                    preferredThickness = DataGridViewBand.MaxBandThickness;
                }
                if (height != preferredThickness)
                {
                    if (autoSizeRowsMode == DataGridViewAutoSizeRowsMode.None)
                    {
                        if (!OnRowHeightInfoPushed(rowIndex, preferredThickness, minimumHeight))
                        {
                            Rows[rowIndex].ThicknessInternal = preferredThickness;   // unsharing the resized row
                        }
                    }
                    else
                    {
                        if (internalAutosizing)
                        {
                            Rows[rowIndex].ThicknessInternal = preferredThickness;   // unsharing the resized row
                        }
                        else
                        {
                            Rows[rowIndex].Thickness = preferredThickness;   // unsharing the resized row
                        }
                    }
                }
            }
            finally
            {
                Debug.Assert(noAutoSizeCount > 0);
                noAutoSizeCount--;
            }
        }

        public void AutoResizeRows()
        {
            AutoResizeRows(DataGridViewAutoSizeRowsMode.AllCells);
        }

        public void AutoResizeRows(DataGridViewAutoSizeRowsMode autoSizeRowsMode)
        {
            AutoResizeRows(autoSizeRowsMode, true /*fixedWidth*/);
        }

        protected void AutoResizeRows(DataGridViewAutoSizeRowsMode autoSizeRowsMode, bool fixedWidth)
        {
            switch (autoSizeRowsMode)
            {
                case DataGridViewAutoSizeRowsMode.None:
                case DataGridViewAutoSizeRowsMode.AllHeaders:
                case DataGridViewAutoSizeRowsMode.AllCellsExceptHeaders:
                case DataGridViewAutoSizeRowsMode.AllCells:
                case DataGridViewAutoSizeRowsMode.DisplayedHeaders:
                case DataGridViewAutoSizeRowsMode.DisplayedCellsExceptHeaders:
                case DataGridViewAutoSizeRowsMode.DisplayedCells:
                    break;
                default:
                    throw new InvalidEnumArgumentException(nameof(autoSizeRowsMode), (int)autoSizeRowsMode, typeof(DataGridViewAutoSizeRowsMode));
            }

            if (autoSizeRowsMode == DataGridViewAutoSizeRowsMode.None)
            {
                throw new ArgumentException(string.Format(SR.DataGridView_NeedAutoSizingCriteria, "autoSizeRowsMode"));
            }

            if ((autoSizeRowsMode == DataGridViewAutoSizeRowsMode.AllHeaders || autoSizeRowsMode == DataGridViewAutoSizeRowsMode.DisplayedHeaders) &&
                !RowHeadersVisible)
            {
                throw new InvalidOperationException(SR.DataGridView_CannotAutoSizeRowsInvisibleRowHeader);
            }

            AdjustShrinkingRows(autoSizeRowsMode, fixedWidth, false /*internalAutosizing*/);
        }

        protected void AutoResizeRows(int rowIndexStart, int rowsCount, DataGridViewAutoSizeRowMode autoSizeRowMode, bool fixedWidth)
        {
            // not using ClientUtils.IsEnumValid here because DataGridViewAutoSizeRowCriteriaInternal is a flags enum.
            if (((DataGridViewAutoSizeRowCriteriaInternal)autoSizeRowMode & invalidDataGridViewAutoSizeRowCriteriaInternalMask) != 0)
            {
                throw new InvalidEnumArgumentException(nameof(autoSizeRowMode), (int)autoSizeRowMode, typeof(DataGridViewAutoSizeRowMode));
            }

            if (autoSizeRowMode == DataGridViewAutoSizeRowMode.RowHeader && !RowHeadersVisible)
            {
                throw new InvalidOperationException(SR.DataGridView_CannotAutoSizeRowsInvisibleRowHeader);
            }

            if (rowsCount < 0)
            {
                throw new ArgumentOutOfRangeException(nameof(rowsCount));
            }

            if (rowIndexStart < 0)
            {
                throw new ArgumentOutOfRangeException(nameof(rowIndexStart));
            }

            if (!IsHandleCreated)
            {
                // auto sizing causes handle creation.
                // don't create the handle inside InitializeComponent because that causes problems w/ data binding
                dataGridViewState2[DATAGRIDVIEWSTATE2_autoSizedWithoutHandle] = true;
                return;
            }

            inBulkPaintCount++;
            inBulkLayoutCount++;
            try
            {
                int rowIndex = Rows.GetNextRow(rowIndexStart - 1, DataGridViewElementStates.Visible);
                int autoSizedCount = 0;
                while (rowIndex != -1 && autoSizedCount < rowsCount)
                {
                    AutoResizeRowInternal(rowIndex, autoSizeRowMode, fixedWidth, false /*internalAutosizing*/);
                    autoSizedCount++;
                    if (autoSizedCount < rowsCount)
                    {
                        rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                    }
                }
            }
            finally
            {
                ExitBulkLayout(true /*invalidInAdjustFillingColumns*/);
                ExitBulkPaint(-1, -1);
            }
        }

        private void BeginColumnHeadersResize(int mouseY, int mouseBarOffset)
        {
            if (IsKeyboardOperationActive())
            {
                return;
            }

            Rectangle clip = Rectangle.Union(layout.ColumnHeaders, layout.Data);
            if (layout.TopLeftHeader.Width > 0)
            {
                clip = Rectangle.Union(layout.TopLeftHeader, clip);
            }
            clip.Y += minimumColumnHeadersHeight - mouseBarOffset - 1;
            // No need to limit the bottom edge of the cursor clip since maxHeadersThickness is very large.
            CaptureMouse(clip);
            dataGridViewOper[DATAGRIDVIEWOPER_trackColHeadersResize] = true;
            trackRowAnchor = mouseY;
            this.mouseBarOffset = mouseBarOffset;
            Debug.Assert(lastRowSplitBar == -1);
            currentRowSplitBar = mouseY;
            Invalidate(CalcRowResizeFeedbackRect(currentRowSplitBar));
        }

        private void BeginColumnRelocation(int mouseX, int index)
        {
            if (IsKeyboardOperationActive())
            {
                return;
            }

            Rectangle cursorClip = layout.ColumnHeaders;
            int frozenWidth = Columns.GetColumnsWidth(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
            int scrollingWidth = Columns.GetColumnsWidth(DataGridViewElementStates.Visible) - frozenWidth;
            if (Columns[index].Frozen)
            {
                // A frozen column cannot be relocated into an unfrozen area
                if (RightToLeftInternal)
                {
                    cursorClip.X += cursorClip.Width - frozenWidth;
                }
                cursorClip.Width = Math.Min(frozenWidth, layout.Data.Width);
            }
            else
            {
                // An unfrozen column cannot be relocated into a frozen area
                if (!RightToLeftInternal)
                {
                    cursorClip.X += frozenWidth;
                }
                else if (layout.Data.Width > frozenWidth + scrollingWidth)
                {
                    cursorClip.X += layout.Data.Width - frozenWidth - scrollingWidth;
                }
                cursorClip.Width = Math.Min(scrollingWidth, layout.Data.Width);
            }
            CaptureMouse(cursorClip);
            dataGridViewOper[DATAGRIDVIEWOPER_trackColRelocation] = true;
            dataGridViewState2[DATAGRIDVIEWSTATE2_showColumnRelocationInsertion] = false;
            trackColumn = index;
            trackColumnEdge = -1;

            mouseBarOffset = GetColumnXFromIndex(index) - mouseX;
            lastHeaderShadow = mouseX;
            Invalidate(layout.ColumnHeaders);
        }

        private void BeginColumnResize(int x, int columnIndex)
        {
            trackColAnchor = x;
            trackColumn = columnIndex;

            currentColSplitBar = x;
            Invalidate(CalcColResizeFeedbackRect(currentColSplitBar));
        }

        private void BeginMouseColumnResize(int mouseX, int mouseBarOffset, int index)
        {
            if (IsKeyboardOperationActive())
            {
                return;
            }

            dataGridViewOper[DATAGRIDVIEWOPER_trackColResize] = true;
            this.mouseBarOffset = mouseBarOffset;
            resizeClipRectangle = GetResizeClipRectangle(index);
            CaptureMouse(resizeClipRectangle);

            BeginColumnResize(mouseX, index);
        }

        private void BeginKeyboardColumnResize(int columnIndex)
        {
            if (IsMouseOperationActive())
            {
                return;
            }

            dataGridViewOper[DATAGRIDVIEWOPER_trackKeyboardColResize] = true;
            mouseBarOffset = 0;
            resizeClipRectangle = GetResizeClipRectangle(columnIndex);
            keyboardResizeStep = ScaleToCurrentDpi(RightToLeftInternal ? -1 : 1);
            int x = GetColumnXFromIndex(columnIndex);
            x += RightToLeftInternal ? -Columns[columnIndex].Width : Columns[columnIndex].Width;

            BeginColumnResize(x, columnIndex);
        }

        private Rectangle GetResizeClipRectangle(int columnIndex)
        {
            Rectangle clip = Rectangle.Union(layout.ColumnHeaders, layout.Data);
            int leftEdge = GetColumnXFromIndex(columnIndex);
            if (RightToLeftInternal)
            {
                clip.X = layout.Data.X - mouseBarOffset - 1;
                clip.Width = leftEdge - Columns[columnIndex].MinimumThickness - layout.Data.X + 3;
                int overflowWidth = leftEdge - mouseBarOffset - clip.Left - DataGridViewBand.MaxBandThickness + 1;
                if (overflowWidth > 0)
                {
                    clip.X += overflowWidth;
                    clip.Width -= overflowWidth;
                }
            }
            else
            {
                clip.X = leftEdge + Columns[columnIndex].MinimumThickness - mouseBarOffset - 1;
                clip.Width = layout.Data.Right - leftEdge - 1;
                int overflowWidth = clip.Right + mouseBarOffset - leftEdge - DataGridViewBand.MaxBandThickness;
                if (overflowWidth > 0)
                {
                    clip.Width -= overflowWidth;
                }
            }

            return clip;
        }

        public virtual bool BeginEdit(bool selectAll)
        {
            if (ptCurrentCell.X == -1)
            {
                throw new InvalidOperationException(SR.DataGridView_NoCurrentCell);
            }

            if (IsCurrentCellInEditMode)
            {
                return true;
            }

            return BeginEditInternal(selectAll);
        }

        private bool BeginEditInternal(bool selectAll)
        {
            if (dataGridViewOper[DATAGRIDVIEWOPER_inBeginEdit])
            {
                throw new InvalidOperationException(SR.DataGridView_BeginEditNotReentrant);
            }
            try
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inBeginEdit] = true;
                Debug.Assert(ptCurrentCell.X >= 0 && ptCurrentCell.X < Columns.Count);
                Debug.Assert(ptCurrentCell.Y >= 0 && ptCurrentCell.Y < Rows.Count);
                Debug.Assert(!IsCurrentCellInEditMode);

                DataGridViewCell dataGridViewCell = CurrentCellInternal;
                Debug.Assert(dataGridViewCell != null);

                if (IsSharedCellReadOnly(dataGridViewCell, ptCurrentCell.Y) ||
                    !ColumnEditable(ptCurrentCell.X))
                {
                    return false;
                }

                Type editControlType = dataGridViewCell.EditType;
                if (editControlType == null)
                {
                    // Current cell does not have an editing control. Does it implement IDataGridViewEditingCell?
                    Type editingCellInterface = dataGridViewCell.GetType().GetInterface("System.Windows.Forms.IDataGridViewEditingCell");
                    if (editingCellInterface == null)
                    {
                        return false;
                    }
                }

                DataGridViewCellCancelEventArgs dgvcce = new DataGridViewCellCancelEventArgs(ptCurrentCell.X, ptCurrentCell.Y);
                OnCellBeginEdit(dgvcce);
                if (dgvcce.Cancel)
                {
                    return false;
                }
                Debug.Assert(!IsCurrentCellInEditMode);

                if (ptCurrentCell.X > -1)
                {
                    DataGridViewCell previousCurrentCell = dataGridViewCell;
                    dataGridViewCell = CurrentCellInternal;
                    if (previousCurrentCell != dataGridViewCell)
                    {
                        // The new current cell can be a whole different cell.
                        // In that case, all tests previously done are no longer valid.
                        if (IsSharedCellReadOnly(dataGridViewCell, ptCurrentCell.Y) ||
                            !ColumnEditable(ptCurrentCell.X))
                        {
                            return false;
                        }

                        editControlType = dataGridViewCell.EditType;
                        if (editControlType == null)
                        {
                            // Current cell does not have an editing control. Does it implement IDataGridViewEditingCell?
                            Type editingCellInterface = dataGridViewCell.GetType().GetInterface("System.Windows.Forms.IDataGridViewEditingCell");
                            if (editingCellInterface == null)
                            {
                                return false;
                            }
                        }
                    }
                }
                else
                {
                    return false;
                }

                DataGridViewCellStyle dataGridViewCellStyle = dataGridViewCell.GetInheritedStyle(null, ptCurrentCell.Y, true);

                if (editControlType == null)
                {
                    dataGridViewState1[DATAGRIDVIEWSTATE1_currentCellInEditMode] = true;
                    InitializeEditingCellValue(ref dataGridViewCellStyle, ref dataGridViewCell);
                    ((IDataGridViewEditingCell)dataGridViewCell).PrepareEditingCellForEdit(selectAll);
                    return true;
                }

                Type editingCtrlInterface = editControlType.GetInterface("System.Windows.Forms.IDataGridViewEditingControl");
                if (!editControlType.IsSubclassOf(Type.GetType("System.Windows.Forms.Control")) ||
                    editingCtrlInterface == null)
                {
                    throw new InvalidCastException(SR.DataGridView_InvalidEditingControl);
                }
                if (latestEditingControl != null &&
                    editControlType.IsInstanceOfType(latestEditingControl) &&
                    !latestEditingControl.GetType().IsSubclassOf(editControlType))
                {
                    editingControl = latestEditingControl;
                    Debug.Assert(((IDataGridViewEditingControl)editingControl).EditingControlDataGridView == this);
                }
                else
                {
                    Debug.Assert(editingControl == null);
                    editingControl = (Control)Activator.CreateInstance(editControlType);
                    Debug.Assert(editingControl != null);

                    ((IDataGridViewEditingControl)editingControl).EditingControlDataGridView = this;
                    if (latestEditingControl != null)
                    {
                        latestEditingControl.Dispose();
                        latestEditingControl = null;
                    }
                }

                Debug.Assert(editingControl != null);
                if (string.IsNullOrEmpty(editingControl.AccessibleName))
                {
                    editingControl.AccessibleName = SR.DataGridView_AccEditingControlAccName;
                }
                editingControl.ImeMode = ImeMode;

                ((IDataGridViewEditingControl)editingControl).EditingControlRowIndex = ptCurrentCell.Y;

                InitializeEditingControlValue(ref dataGridViewCellStyle, dataGridViewCell);

                WireEditingControlEvents();

                Debug.Assert(editingControl != null);
                Debug.Assert(editingPanel != null);
                DataGridViewEditingControlShowingEventArgs dgvese = new DataGridViewEditingControlShowingEventArgs(editingControl, dataGridViewCellStyle);
                OnEditingControlShowing(dgvese);
                Debug.Assert(dgvese.CellStyle != null);
                if (editingPanel == null || editingControl == null)
                {
                    return false;
                }
                editingPanel.BackColor = dgvese.CellStyle.BackColor;
                ((IDataGridViewEditingControl)editingControl).ApplyCellStyleToEditingControl(dgvese.CellStyle);

                // Get rid of the tooltip if it's showing for the current cell
                if (toolTipControl.Activated && ptToolTipCell == ptCurrentCell)
                {
                    toolTipControl.Activate(false /*activate*/);
                }

                PositionEditingControl(true, true, true);

                // Guarding against bugs in customer code.
                // For example setting the CurrentCell to null in DataGridView_OnLostFocus(...) causes this.editingControl
                // to become null.
                if (editingPanel == null || editingControl == null)
                {
                    return false;
                }
                else
                {
                    ((IDataGridViewEditingControl)editingControl).PrepareEditingControlForEdit(selectAll);
                    InvalidateCellPrivate(ptCurrentCell.X, ptCurrentCell.Y);
                    return true;
                }
            }
            finally
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inBeginEdit] = false;
            }
        }

        private void BeginRowHeadersResize(int mouseX, int mouseBarOffset)
        {
            if (IsKeyboardOperationActive())
            {
                return;
            }

            Rectangle clip = Rectangle.Union(layout.RowHeaders, layout.Data);
            if (layout.TopLeftHeader.Width > 0)
            {
                clip = Rectangle.Union(layout.TopLeftHeader, clip);
            }
            if (RightToLeftInternal)
            {
                clip.X -= mouseBarOffset + 1;
                clip.Width -= minimumRowHeadersWidth - 1;
                // No need to limit the left edge of the cursor clip since maxHeadersThickness is very large.
            }
            else
            {
                clip.X += minimumRowHeadersWidth - mouseBarOffset - 1;
                // No need to limit the right edge of the cursor clip since maxHeadersThickness is very large.
            }
            CaptureMouse(clip);

            dataGridViewOper[DATAGRIDVIEWOPER_trackRowHeadersResize] = true;
            trackColAnchor = mouseX;
            this.mouseBarOffset = mouseBarOffset;
            Debug.Assert(lastColSplitBar == -1);
            currentColSplitBar = mouseX;
            Invalidate(CalcColResizeFeedbackRect(currentColSplitBar));
        }

        private void BeginRowResize(int mouseY, int mouseBarOffset, int index)
        {
            if (IsKeyboardOperationActive())
            {
                return;
            }

            Rectangle clip = Rectangle.Union(layout.RowHeaders, layout.Data);
            int topEdge = GetRowYFromIndex(index);
            clip.Y = topEdge + Rows.SharedRow(index).GetMinimumHeight(index) - mouseBarOffset - 1;
            clip.Height = layout.Data.Y + layout.Data.Height - topEdge - 1;
            CaptureMouse(clip);

            dataGridViewOper[DATAGRIDVIEWOPER_trackRowResize] = true;
            trackRowAnchor = mouseY;
            trackRow = index;

            this.mouseBarOffset = mouseBarOffset;
            Debug.Assert(lastRowSplitBar == -1);
            currentRowSplitBar = mouseY;
            Invalidate(CalcRowResizeFeedbackRect(currentRowSplitBar));
        }

        private void BuildInheritedColumnHeaderCellStyle(DataGridViewCellStyle inheritedCellStyle, DataGridViewCell cell)
        {
            Debug.Assert(inheritedCellStyle != null);

            DataGridViewCellStyle cellStyle = null;
            if (cell.HasStyle)
            {
                cellStyle = cell.Style;
                Debug.Assert(cellStyle != null);
            }

            DataGridViewCellStyle columnHeadersStyle = ColumnHeadersDefaultCellStyle;
            Debug.Assert(columnHeadersStyle != null);

            DataGridViewCellStyle dataGridViewStyle = DefaultCellStyle;
            Debug.Assert(dataGridViewStyle != null);

            if (cellStyle != null && !cellStyle.BackColor.IsEmpty)
            {
                inheritedCellStyle.BackColor = cellStyle.BackColor;
            }
            else if (!columnHeadersStyle.BackColor.IsEmpty)
            {
                inheritedCellStyle.BackColor = columnHeadersStyle.BackColor;
            }
            else
            {
                inheritedCellStyle.BackColor = dataGridViewStyle.BackColor;
            }

            if (cellStyle != null && !cellStyle.ForeColor.IsEmpty)
            {
                inheritedCellStyle.ForeColor = cellStyle.ForeColor;
            }
            else if (!columnHeadersStyle.ForeColor.IsEmpty)
            {
                inheritedCellStyle.ForeColor = columnHeadersStyle.ForeColor;
            }
            else
            {
                inheritedCellStyle.ForeColor = dataGridViewStyle.ForeColor;
            }

            if (cellStyle != null && !cellStyle.SelectionBackColor.IsEmpty)
            {
                inheritedCellStyle.SelectionBackColor = cellStyle.SelectionBackColor;
            }
            else if (!columnHeadersStyle.SelectionBackColor.IsEmpty)
            {
                inheritedCellStyle.SelectionBackColor = columnHeadersStyle.SelectionBackColor;
            }
            else
            {
                inheritedCellStyle.SelectionBackColor = dataGridViewStyle.SelectionBackColor;
            }

            if (cellStyle != null && !cellStyle.SelectionForeColor.IsEmpty)
            {
                inheritedCellStyle.SelectionForeColor = cellStyle.SelectionForeColor;
            }
            else if (!columnHeadersStyle.SelectionForeColor.IsEmpty)
            {
                inheritedCellStyle.SelectionForeColor = columnHeadersStyle.SelectionForeColor;
            }
            else
            {
                inheritedCellStyle.SelectionForeColor = dataGridViewStyle.SelectionForeColor;
            }

            if (cellStyle != null && cellStyle.Font != null)
            {
                inheritedCellStyle.Font = cellStyle.Font;
            }
            else if (columnHeadersStyle.Font != null)
            {
                inheritedCellStyle.Font = columnHeadersStyle.Font;
            }
            else
            {
                inheritedCellStyle.Font = dataGridViewStyle.Font;
            }

            if (cellStyle != null && !cellStyle.IsNullValueDefault)
            {
                inheritedCellStyle.NullValue = cellStyle.NullValue;
            }
            else if (!columnHeadersStyle.IsNullValueDefault)
            {
                inheritedCellStyle.NullValue = columnHeadersStyle.NullValue;
            }
            else
            {
                inheritedCellStyle.NullValue = dataGridViewStyle.NullValue;
            }

            if (cellStyle != null && !cellStyle.IsDataSourceNullValueDefault)
            {
                inheritedCellStyle.DataSourceNullValue = cellStyle.DataSourceNullValue;
            }
            else if (!columnHeadersStyle.IsDataSourceNullValueDefault)
            {
                inheritedCellStyle.DataSourceNullValue = columnHeadersStyle.DataSourceNullValue;
            }
            else
            {
                inheritedCellStyle.DataSourceNullValue = dataGridViewStyle.DataSourceNullValue;
            }

            if (cellStyle != null && cellStyle.Format.Length != 0)
            {
                inheritedCellStyle.Format = cellStyle.Format;
            }
            else if (columnHeadersStyle.Format.Length != 0)
            {
                inheritedCellStyle.Format = columnHeadersStyle.Format;
            }
            else
            {
                inheritedCellStyle.Format = dataGridViewStyle.Format;
            }

            if (cellStyle != null && !cellStyle.IsFormatProviderDefault)
            {
                inheritedCellStyle.FormatProvider = cellStyle.FormatProvider;
            }
            else if (!columnHeadersStyle.IsFormatProviderDefault)
            {
                inheritedCellStyle.FormatProvider = columnHeadersStyle.FormatProvider;
            }
            else
            {
                inheritedCellStyle.FormatProvider = dataGridViewStyle.FormatProvider;
            }

            if (cellStyle != null && cellStyle.Alignment != DataGridViewContentAlignment.NotSet)
            {
                inheritedCellStyle.AlignmentInternal = cellStyle.Alignment;
            }
            else if (columnHeadersStyle != null && columnHeadersStyle.Alignment != DataGridViewContentAlignment.NotSet)
            {
                inheritedCellStyle.AlignmentInternal = columnHeadersStyle.Alignment;
            }
            else
            {
                Debug.Assert(dataGridViewStyle.Alignment != DataGridViewContentAlignment.NotSet);
                inheritedCellStyle.AlignmentInternal = dataGridViewStyle.Alignment;
            }

            if (cellStyle != null && cellStyle.WrapMode != DataGridViewTriState.NotSet)
            {
                inheritedCellStyle.WrapModeInternal = cellStyle.WrapMode;
            }
            else if (columnHeadersStyle != null && columnHeadersStyle.WrapMode != DataGridViewTriState.NotSet)
            {
                inheritedCellStyle.WrapModeInternal = columnHeadersStyle.WrapMode;
            }
            else
            {
                Debug.Assert(dataGridViewStyle.WrapMode != DataGridViewTriState.NotSet);
                inheritedCellStyle.WrapModeInternal = dataGridViewStyle.WrapMode;
            }

            if (cellStyle != null && cellStyle.Tag != null)
            {
                inheritedCellStyle.Tag = cellStyle.Tag;
            }
            else if (columnHeadersStyle.Tag != null)
            {
                inheritedCellStyle.Tag = columnHeadersStyle.Tag;
            }
            else
            {
                inheritedCellStyle.Tag = dataGridViewStyle.Tag;
            }

            if (cellStyle != null && cellStyle.Padding != Padding.Empty)
            {
                inheritedCellStyle.PaddingInternal = cellStyle.Padding;
            }
            else if (columnHeadersStyle.Padding != Padding.Empty)
            {
                inheritedCellStyle.PaddingInternal = columnHeadersStyle.Padding;
            }
            else
            {
                inheritedCellStyle.PaddingInternal = dataGridViewStyle.Padding;
            }
        }

        private Rectangle CalcColRelocationFeedbackRect(int mouseX)
        {
            Rectangle r, inside = layout.ColumnHeaders;
            if (layout.TopLeftHeader.Width > 0)
            {
                inside = Rectangle.Union(layout.TopLeftHeader, inside);
            }
            if (RightToLeftInternal)
            {
                r = new Rectangle(mouseX + mouseBarOffset - Columns[trackColumn].Thickness + 1,
                                  inside.Y,
                                  Columns[trackColumn].Thickness,
                                  inside.Height);
                r.X = Math.Max(inside.Left, r.X);
                r.X = Math.Min(r.X, inside.Right - r.Width);
            }
            else
            {
                r = new Rectangle(mouseX + mouseBarOffset - 1, inside.Y, Columns[trackColumn].Thickness, inside.Height);
                r.X = Math.Min(inside.Right - r.Width, r.X);
                r.X = Math.Max(r.X, inside.Left);
            }
            return r;
        }

        private Rectangle CalcColResizeFeedbackRect(int mouseX)
        {
            Rectangle inside = layout.Data;
            Rectangle r = new Rectangle(mouseX + mouseBarOffset - 1, inside.Y, 3, inside.Height);
            if (RightToLeftInternal)
            {
                r.X = Math.Max(inside.Left, r.X);
            }
            else
            {
                r.X = Math.Min(inside.Right - 3, r.X);
                r.X = Math.Max(r.X, 0);
            }
            return r;
        }

        private Rectangle CalcRowResizeFeedbackRect(int mouseY)
        {
            Rectangle inside = layout.Data;
            Rectangle r = new Rectangle(inside.X, mouseY + mouseBarOffset - 1, inside.Width, 3);
            r.Y = Math.Min(inside.Bottom - 3, r.Y);
            r.Y = Math.Max(r.Y, 0);
            return r;
        }

        public bool CancelEdit()
        {
            return CancelEdit(false /*endEdit, DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.InitialValueRestoration*/);
        }

        private bool CancelEdit(bool endEdit /*, DataGridViewDataErrorContexts context*/)
        {
            if (ptCurrentCell.X != -1)
            {
                Debug.Assert(ptCurrentCell.Y != -1);

                int oldCurrentCellX = ptCurrentCell.X;
                DataGridViewDataErrorEventArgs dgvdee = CancelEditPrivate(/*ref dataGridViewCurrentCell, context*/);

                if (null != dgvdee)
                {
                    if (dgvdee.ThrowException)
                    {
                        throw dgvdee.Exception;
                    }

                    if (dgvdee.Cancel)
                    {
                        return false;
                    }
                }

                if (IsCurrentCellInEditMode)
                {
                    if (endEdit && EditMode != DataGridViewEditMode.EditOnEnter && editingControl != null)
                    {
                        bool success = EndEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.InitialValueRestoration,
                                               DataGridViewValidateCellInternal.Never /*validateCell*/,
                                               false /*fireCellLeave*/,
                                               false /*fireCellEnter*/,
                                               false /*fireRowLeave*/,
                                               false /*fireRowEnter*/,
                                               false /*fireLeave*/,
                                               true /*keepFocus*/,
                                               true /*resetCurrentCell unused here*/,
                                               true /*resetAnchorCell unused here*/);
                        Debug.Assert(success);
                    }
                    else
                    {
                        DataGridViewDataErrorEventArgs dgvdee2 = null;
                        IDataGridViewEditingCell dataGridViewEditingCell = null;
                        try
                        {
                            dataGridViewState1[DATAGRIDVIEWSTATE1_ignoringEditingChanges] = true;
                            if (editingControl != null)
                            {
                                ((IDataGridViewEditingControl)editingControl).EditingControlFormattedValue = uneditedFormattedValue;
                                ((IDataGridViewEditingControl)editingControl).EditingControlValueChanged = false;
                            }
                            else
                            {
                                Debug.Assert(dataGridViewState1[DATAGRIDVIEWSTATE1_currentCellInEditMode]);
                                dataGridViewEditingCell = CurrentCellInternal as IDataGridViewEditingCell;
                                Debug.Assert(dataGridViewEditingCell != null);
                                dataGridViewEditingCell.EditingCellFormattedValue = uneditedFormattedValue;
                                dataGridViewEditingCell.EditingCellValueChanged = false;
                            }
                        }
                        catch (Exception exception)
                        {
                            if (ClientUtils.IsCriticalException(exception))
                            {
                                throw;
                            }
                            dgvdee2 = new DataGridViewDataErrorEventArgs(exception, ptCurrentCell.X,
                                ptCurrentCell.Y,
                                DataGridViewDataErrorContexts.InitialValueRestoration);
                        }
                        finally
                        {
                            dataGridViewState1[DATAGRIDVIEWSTATE1_ignoringEditingChanges] = false;
                        }
                        if (dgvdee2 != null)
                        {
                            OnDataErrorInternal(dgvdee2);
                            if (dgvdee2.ThrowException)
                            {
                                throw dgvdee2.Exception;
                            }
                        }

                        if (editingControl != null)
                        {
                            ((IDataGridViewEditingControl)editingControl).PrepareEditingControlForEdit(true /*selectAll*/);
                        }
                        else
                        {
                            Debug.Assert(dataGridViewEditingCell != null);
                            dataGridViewEditingCell.PrepareEditingCellForEdit(true /*selectAll*/);
                            InvalidateCellPrivate(ptCurrentCell.X, ptCurrentCell.Y);
                        }
                    }
                }
                else if (ptCurrentCell.X == -1 && Focused)
                {
                    Debug.Assert((AllowUserToAddRowsInternal && Rows.Count == 1) ||
                                 (!AllowUserToAddRowsInternal && Rows.Count == 0));
                    if (Rows.Count > 0)
                    {
                        if (Columns.Count > oldCurrentCellX && Columns[oldCurrentCellX].Visible)
                        {
                            int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                            if (rowIndex != -1)
                            {
                                bool success = SetAndSelectCurrentCellAddress(oldCurrentCellX,
                                                                              rowIndex,
                                                                              true /*setAnchorCellAddress*/,
                                                                              false /*validateCurrentCell*/,
                                                                              false /*throughMouseClick*/,
                                                                              false /*clearSelection*/,
                                                                              false /*forceCurrentCellSelection*/);
                                Debug.Assert(success);
                            }
                        }
                        else
                        {
                            MakeFirstDisplayedCellCurrentCell(true /*includeNewRow*/);
                        }
                    }
                }
            }
            return true;
        }

        private DataGridViewDataErrorEventArgs CancelEditPrivate(/*ref DataGridViewCell dataGridViewCurrentCell, DataGridViewDataErrorContexts context*/)
        {
            bool currentCellDirty = IsCurrentCellDirty;
            bool currentRowDirty = IsCurrentRowDirty;

            if (IsCurrentCellInEditMode)
            {
                /* Do not push original value back into the cell

*/
                if (editingControl != null)
                {
                    ((IDataGridViewEditingControl)editingControl).EditingControlValueChanged = false;
                }
                else
                {
                    Debug.Assert(dataGridViewState1[DATAGRIDVIEWSTATE1_currentCellInEditMode]);
                    ((IDataGridViewEditingCell)CurrentCellInternal).EditingCellValueChanged = false;
                }
                IsCurrentCellDirtyInternal = false;
            }

            if (DataSource != null || VirtualMode)
            {
                if ((currentRowDirty && !currentCellDirty) ||
                    (dataGridViewState1[DATAGRIDVIEWSTATE1_newRowEdited] &&
                    !dataGridViewState1[DATAGRIDVIEWSTATE1_editedRowChanged]))
                {
                    bool discardNewRow = dataGridViewState1[DATAGRIDVIEWSTATE1_newRowEdited];
                    IsCurrentRowDirtyInternal = false;
                    if (VirtualMode)
                    {
                        QuestionEventArgs qe = new QuestionEventArgs(discardNewRow);
                        OnCancelRowEdit(qe);
                        discardNewRow &= qe.Response;
                    }
                    if (DataSource != null)
                    {
                        int oldCurrentCellX = ptCurrentCell.X;
                        dataConnection.CancelRowEdit(true /*restoreRow*/, dataGridViewState1[DATAGRIDVIEWSTATE1_newRowEdited]/*addNewFinished*/);
                        if (dataConnection.List.Count == 0)
                        {
                            // There are no rows left in the back end.
                            if (currentCellDirty || ptCurrentCell.Y == -1 || ptCurrentCell.X == -1)
                            {
                                if (!IsColumnOutOfBounds(oldCurrentCellX) && Columns[oldCurrentCellX].Visible)
                                {
                                    Debug.Assert(0 == Rows.GetFirstRow(DataGridViewElementStates.Visible));
                                    // Setting the current cell to the current column in the first row
                                    // will create the new row if the user was editing the cell.
                                    SetAndSelectCurrentCellAddress(oldCurrentCellX,
                                                                   0,
                                                                   true,  /*setAnchorCellAddress*/
                                                                   false, /*validateCurrentCell*/
                                                                   false,  /*throughMouseClick*/
                                                                   true /*clearSelecttion*/,
                                                                   false /*forceCurrentCellSelection (unused)*/);
                                }
                            }
                            else
                            {
                                // Else, simply add a new row.
                                dataConnection.OnNewRowNeeded();
                            }
                        }

                        // CancelRowEdit discarded the new row if we were editing the new row.
                        discardNewRow = false;
                    }
                    if (ptCurrentCell.Y > -1)
                    {
                        InvalidateRowPrivate(ptCurrentCell.Y);
                        DataGridViewCell dataGridViewCell = CurrentCellInternal;
                        if (IsCurrentCellInEditMode)
                        {
                            DataGridViewCellStyle dataGridViewCellStyle = dataGridViewCell.GetInheritedStyle(null, ptCurrentCell.Y, true);
                            if (editingControl != null)
                            {
                                InitializeEditingControlValue(ref dataGridViewCellStyle, dataGridViewCell);
                                if (((IDataGridViewEditingControl)editingControl).RepositionEditingControlOnValueChange)
                                {
                                    PositionEditingControl(true /*setLocation*/, true /*setSize*/, false /*setFocus*/);
                                }
                            }
                            else
                            {
                                Debug.Assert(dataGridViewState1[DATAGRIDVIEWSTATE1_currentCellInEditMode]);
                                InitializeEditingCellValue(ref dataGridViewCellStyle, ref dataGridViewCell);
                            }
                        }
                    }
                    if (discardNewRow && ptCurrentCell.Y == newRowIndex - 1)
                    {
                        DiscardNewRow();
                    }
                }
            }
            else
            {
                if (!IsCurrentRowDirty &&
                    ptCurrentCell.Y == newRowIndex - 1 &&
                    dataGridViewState1[DATAGRIDVIEWSTATE1_newRowCreatedByEditing])
                {
                    DiscardNewRow();
                }
            }
            return null;
        }

        internal bool CancelToolTipPopup(ToolTip toolTip)
        {
            if (toolTipControl.ToolTip == toolTip || KeyboardToolTip == toolTip)
            {
                // Our own tool tip wants to show its text.
                return false;
            }
            else
            {
                // This is an external tool tip control which wants to show a tool tip over the DataGridView.
                // ToolTips from the data Grid view ( the error text, or the formatted text that does not fit in, or the tool tip text from the cell)
                // and the ShowCellToolTips take precedence over the external tool tip.
                return string.IsNullOrEmpty(toolTipCaption) && ShowCellToolTips;
            }
        }

        private bool CanSort(DataGridViewColumn dataGridViewColumn)
        {
            return dataGridViewColumn.SortMode == DataGridViewColumnSortMode.Automatic && (!VirtualMode || dataGridViewColumn.IsDataBound);
        }

        private bool IsSortable(DataGridViewColumn dataGridViewColumn)
        {
            return dataGridViewColumn.SortMode != DataGridViewColumnSortMode.NotSortable && (!VirtualMode || dataGridViewColumn.IsDataBound);
        }

        // determines if a data bound cell can be validated or not
        private bool CanValidateDataBoundDataGridViewCell(DataGridViewCell dataGridViewCurrentCell)
        {
            if (dataGridViewCurrentCell == null)
            {
                if (ptCurrentCell.X > -1)
                {
                    dataGridViewCurrentCell = CurrentCellInternal;
                }
            }

            if (dataGridViewCurrentCell == null)
            {
                return true;
            }

            Debug.Assert(dataGridViewCurrentCell.OwningColumn != null);

            if (!dataGridViewCurrentCell.OwningColumn.IsDataBoundInternal)
            {
                // we are not data bound so it's not up to us to decide to stop validation
                return true;
            }

            if (dataGridViewOper[DATAGRIDVIEWOPER_inDispose])
            {
                // Dispose is not the place to validate data. Also, chances are that the data source is also disposing itself.
                return false;
            }

            if (dataConnection == null)
            {
                // if there is no dataConnection then it is not up to this function to stop validation.
                return true;
            }

            /////////////////////////////////////////////////////////////////////////////////////////////////
            //                                                                                             //
            // FROM HERE DOWN THE DATA CONNECTION DETERMINES IF THE DATAGRIDVIEW SHOULD VALIDATE THE CELL. //
            //                                                                                             //
            /////////////////////////////////////////////////////////////////////////////////////////////////
            if (dataConnection.ProcessingMetaDataChanges)
            {
                // don't validate a cell in a data bound column while the property descriptors change under us
                return false;
            }

            if (dataConnection.CancellingRowEdit && !dataConnection.RestoreRow)
            {
                // don't validate a cell in a data bound column while we are cancelling a row edit and the old row is not restored
                return false;
            }

            if (dataConnection.CurrencyManager.Count <= ptCurrentCell.Y)
            {
                // don't validate a row beyond the last row in the back end list
                return false;
            }

            if (dataConnection.PositionChangingOutsideDataGridView)
            {
                // the position changed outside the data grid view and we haven't validated the data grid view cell already
                // we can't validate it now because if the user cancels validation then we end up
                // with a position different than the position in the currency manager
                return false;
            }

            if (dataConnection.ListWasReset)
            {
                // The list was reset outside data grid view.
                // We can't validate it now because we would be pushing a value into a different object ( possibly located in a different list ).
                return false;
            }

            return true;
        }

        private void CaptureMouse(Rectangle cursorClip)
        {
            CaptureInternal = true;
            Cursor.ClipInternal = RectangleToScreen(cursorClip);
        }

        private void ClearRegionCache()
        {
            cachedScrollableRegion = null;
        }

        public void ClearSelection()
        {
            noDimensionChangeCount++;
            noSelectionChangeCount++;

            bool switchedToBulkPaint = false;

            if (selectedBandIndexes.Count > DATAGRIDVIEW_bulkPaintThreshold ||
                individualSelectedCells.Count > DATAGRIDVIEW_bulkPaintThreshold)
            {
                inBulkPaintCount++;
                switchedToBulkPaint = true;
            }
            try
            {
                RemoveIndividuallySelectedCells();
                switch (SelectionMode)
                {
                    case DataGridViewSelectionMode.CellSelect:
                        {
                            // If we change the design and start using this.selectedBandIndexes in this SelectionMode,
                            // we'll have to clear those selections too.
                            break;
                        }

                    case DataGridViewSelectionMode.FullRowSelect:
                    case DataGridViewSelectionMode.RowHeaderSelect:
                        {
                            while (selectedBandIndexes.Count > 0)
                            {
                                SetSelectedRowCore(selectedBandIndexes.HeadInt, false);
                            }

                            // Force repainting of the current collumn's header cell to remove highlighting
                            if (ptCurrentCell.X != -1 &&
                                SelectionMode == DataGridViewSelectionMode.FullRowSelect)
                            {
                                InvalidateCellPrivate(ptCurrentCell.X, -1);
                            }
                            break;
                        }

                    case DataGridViewSelectionMode.FullColumnSelect:
                    case DataGridViewSelectionMode.ColumnHeaderSelect:
                        {
                            while (selectedBandIndexes.Count > 0)
                            {
                                SetSelectedColumnCore(selectedBandIndexes.HeadInt, false);
                            }
                            break;
                        }
                }
            }
            finally
            {
                noDimensionChangeCount--;
                Debug.Assert(noDimensionChangeCount >= 0);
                NoSelectionChangeCount--;
                if (switchedToBulkPaint)
                {
                    ExitBulkPaint(-1, -1);
                }
            }
        }

        protected void ClearSelection(int columnIndexException, int rowIndexException, bool selectExceptionElement)
        {
            switch (SelectionMode)
            {
                case DataGridViewSelectionMode.CellSelect:
                case DataGridViewSelectionMode.FullColumnSelect:
                case DataGridViewSelectionMode.ColumnHeaderSelect:
                    {
                        if (columnIndexException < 0 || columnIndexException >= Columns.Count)
                        {
                            throw new ArgumentOutOfRangeException(nameof(columnIndexException));
                        }
                        break;
                    }

                case DataGridViewSelectionMode.FullRowSelect:
                case DataGridViewSelectionMode.RowHeaderSelect:
                    {
                        if (columnIndexException < -1 || columnIndexException >= Columns.Count)
                        {
                            throw new ArgumentOutOfRangeException(nameof(columnIndexException));
                        }
                        break;
                    }
            }

            switch (SelectionMode)
            {
                case DataGridViewSelectionMode.CellSelect:
                case DataGridViewSelectionMode.FullRowSelect:
                case DataGridViewSelectionMode.RowHeaderSelect:
                    {
                        if (rowIndexException < 0 || rowIndexException >= Rows.Count)
                        {
                            throw new ArgumentOutOfRangeException(nameof(rowIndexException));
                        }
                        break;
                    }

                case DataGridViewSelectionMode.FullColumnSelect:
                case DataGridViewSelectionMode.ColumnHeaderSelect:
                    {
                        if (rowIndexException < -1 || rowIndexException >= Rows.Count)
                        {
                            throw new ArgumentOutOfRangeException(nameof(rowIndexException));
                        }
                        break;
                    }
            }

            // Clears all selection except the row/column/cell specified as parameter
            noDimensionChangeCount++;
            noSelectionChangeCount++;

            bool switchedToBulkPaint = false;

            if (selectedBandIndexes.Count > DATAGRIDVIEW_bulkPaintThreshold ||
                individualSelectedCells.Count > DATAGRIDVIEW_bulkPaintThreshold)
            {
                inBulkPaintCount++;
                switchedToBulkPaint = true;
            }
            try
            {
                switch (SelectionMode)
                {
                    case DataGridViewSelectionMode.CellSelect:
                        {
                            // If we change the design and start using this.selectedBandIndexes in this SelectionMode,
                            // we'll have to clear those selections too.
                            RemoveIndividuallySelectedCells(columnIndexException, rowIndexException);
                            break;
                        }

                    case DataGridViewSelectionMode.FullRowSelect:
                    case DataGridViewSelectionMode.RowHeaderSelect:
                        {
                            int bandIndex = 0;
                            while (bandIndex < selectedBandIndexes.Count)
                            {
                                if (selectedBandIndexes[bandIndex] != rowIndexException)
                                {
                                    // deselect currently selected row
                                    SetSelectedRowCore(selectedBandIndexes[bandIndex], false);
                                }
                                else
                                {
                                    bandIndex++;
                                }
                            }
                            if (SelectionMode == DataGridViewSelectionMode.RowHeaderSelect)
                            {
                                RemoveIndividuallySelectedCells(columnIndexException, rowIndexException);
                            }
                            break;
                        }

                    case DataGridViewSelectionMode.FullColumnSelect:
                    case DataGridViewSelectionMode.ColumnHeaderSelect:
                        {
                            int bandIndex = 0;
                            while (bandIndex < selectedBandIndexes.Count)
                            {
                                if (selectedBandIndexes[bandIndex] != columnIndexException)
                                {
                                    // deselect currently selected column
                                    SetSelectedColumnCore(selectedBandIndexes[bandIndex], false);
                                }
                                else
                                {
                                    bandIndex++;
                                }
                            }
                            if (SelectionMode == DataGridViewSelectionMode.ColumnHeaderSelect)
                            {
                                RemoveIndividuallySelectedCells(columnIndexException, rowIndexException);
                            }
                            break;
                        }
                }
                if (selectExceptionElement)
                {
                    SetSelectedElementCore(columnIndexException, rowIndexException, true);
                }
            }
            finally
            {
                noDimensionChangeCount--;
                Debug.Assert(noDimensionChangeCount >= 0);
                NoSelectionChangeCount--;
                if (switchedToBulkPaint)
                {
                    ExitBulkPaint(-1, -1);
                }
            }
        }

        private bool ColumnEditable(int columnIndex)
        {
            Debug.Assert(columnIndex >= 0 && columnIndex < Columns.Count, "Invalid columnIndex: " + columnIndex);
            if (Columns[columnIndex].IsDataBound &&
                dataConnection != null &&
                !dataConnection.AllowEdit)
            {
                return false;
            }
            return true;
        }

        private bool ColumnNeedsDisplayedState(DataGridViewColumn dataGridViewColumn)
        {
            Debug.Assert(dataGridViewColumn != null);

            if (!dataGridViewColumn.Visible)
            {
                return false;
            }

            if (dataGridViewColumn.Frozen)
            {
                DataGridViewColumn firstVisibleFrozenColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                Debug.Assert(firstVisibleFrozenColumn != null);
                if (firstVisibleFrozenColumn.Index == dataGridViewColumn.Index)
                {
                    return displayedBandsInfo.NumDisplayedFrozenCols > 0;
                }
                Debug.Assert(Columns.DisplayInOrder(firstVisibleFrozenColumn.Index, dataGridViewColumn.Index));
                return Columns.GetColumnCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen, firstVisibleFrozenColumn.Index, dataGridViewColumn.Index) < displayedBandsInfo.NumDisplayedFrozenCols;
            }
            else
            {
                int firstDisplayedScrollingColumnIndex = displayedBandsInfo.FirstDisplayedScrollingCol;
                if (firstDisplayedScrollingColumnIndex != -1)
                {
                    if (firstDisplayedScrollingColumnIndex == dataGridViewColumn.Index)
                    {
                        return displayedBandsInfo.NumDisplayedScrollingCols > 0;
                    }
                    if (Columns.DisplayInOrder(firstDisplayedScrollingColumnIndex, dataGridViewColumn.Index))
                    {
                        return Columns.GetColumnCount(DataGridViewElementStates.Visible, firstDisplayedScrollingColumnIndex, dataGridViewColumn.Index) < displayedBandsInfo.NumDisplayedScrollingCols;
                    }
                }
            }
            return false;
        }

        private bool ColumnRelocationTarget(MouseEventArgs e, HitTestInfo hti, out int previousColumnIndex)
        {
            previousColumnIndex = -1;
            if (hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnHeadersResizeBottom ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnHeader ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnResizeLeft ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnResizeRight ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnHeaderLeft ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnHeaderRight)
            {
                Debug.Assert(hti.col != -1);
                if (hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnHeadersResizeBottom ||
                    hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnHeader)
                {
                    int xColumnLeftEdge = GetColumnXFromIndex(hti.col);
                    int wColumn = Columns[hti.col].Width;
                    if ((RightToLeftInternal && e.X < xColumnLeftEdge - wColumn / 2) ||
                        (!RightToLeftInternal && e.X > xColumnLeftEdge + wColumn / 2))
                    {
                        // Insert column on the right of hti.col
                        previousColumnIndex = hti.col;
                    }
                    else
                    {
                        // Insert column on the left of hti.col
                        DataGridViewColumn dataGridViewColumnPrev = Columns.GetPreviousColumn(Columns[hti.col],
                                                                                                            DataGridViewElementStates.Visible,
                                                                                                            DataGridViewElementStates.None);
                        if (dataGridViewColumnPrev != null)
                        {
                            previousColumnIndex = dataGridViewColumnPrev.Index;
                        }
                    }
                }
                else
                {
                    previousColumnIndex = (hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnResizeRight || hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnHeaderRight) ?
                                          hti.col : hti.adjacentCol;
                }
                DataGridViewColumn dataGridViewColumnNext = null;
                if (previousColumnIndex != -1)
                {
                    dataGridViewColumnNext = Columns.GetNextColumn(Columns[previousColumnIndex],
                                                                                 DataGridViewElementStates.Visible,
                                                                                 DataGridViewElementStates.None);
                }
                if (trackColumn != previousColumnIndex &&
                    !(previousColumnIndex == -1 && hti.col == trackColumn) &&
                    (dataGridViewColumnNext == null || trackColumn != dataGridViewColumnNext.Index))
                {
                    return true;
                }
            }
            else if (hti.typeInternal == DataGridViewHitTestTypeInternal.FirstColumnHeaderLeft ||
                     hti.typeInternal == DataGridViewHitTestTypeInternal.TopLeftHeaderResizeRight)
            {
                Debug.Assert(hti.col != -1);
                if (hti.col != trackColumn)
                {
                    return true;
                }
            }
            return false;
        }

        private static bool ColumnsDisplayInOrder(int columnIndex1,
            int columnDisplayIndex1,
            int columnIndex2,
            int columnDisplayIndex2)
        {
            return columnDisplayIndex1 < columnDisplayIndex2 ||
                (columnDisplayIndex1 == columnDisplayIndex2 && columnIndex1 < columnIndex2);
        }

        public bool CommitEdit(DataGridViewDataErrorContexts context)
        {
            if (IsCurrentCellInEditMode)
            {
                DataGridViewCell dataGridViewCurrentCell = CurrentCellInternal;
                DataGridViewDataErrorEventArgs dgvdee = CommitEdit(ref dataGridViewCurrentCell,
                    context,
                    DataGridViewValidateCellInternal.Never,
                    false /*fireCellLeave*/,
                    false /*fireCellEnter*/,
                    false /*fireRowLeave*/,
                    false /*fireRowEnter*/,
                    false /*fireLeave*/);
                if (null != dgvdee)
                {
                    if (dgvdee.ThrowException)
                    {
                        throw dgvdee.Exception;
                    }
                    if (dgvdee.Cancel)
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        private DataGridViewDataErrorEventArgs CommitEdit(ref DataGridViewCell dataGridViewCurrentCell,
            DataGridViewDataErrorContexts context,
            DataGridViewValidateCellInternal validateCell,
            bool fireCellLeave,
            bool fireCellEnter,
            bool fireRowLeave,
            bool fireRowEnter,
            bool fireLeave)
        {
            if (validateCell == DataGridViewValidateCellInternal.Always)
            {
                Debug.Assert(ptCurrentCell.X > -1);
                if (fireCellLeave)
                {
                    if (ptCurrentCell.X == -1)
                    {
                        return null;
                    }
                    OnCellLeave(ref dataGridViewCurrentCell, ptCurrentCell.X, ptCurrentCell.Y);
                }
                if (fireRowLeave)
                {
                    if (ptCurrentCell.X == -1)
                    {
                        return null;
                    }
                    OnRowLeave(ref dataGridViewCurrentCell, ptCurrentCell.X, ptCurrentCell.Y);
                }
                if (fireLeave)
                {
                    base.OnLeave(EventArgs.Empty);

                    // Microsoft: can we be smarter about this? What if validating the current cell below forces a repaint on the cell?
                    // we would end up repainting the current cell twice.
                    //
                    // invalidate the current cell so the data grid view does not paint the focus rectangle any longer
                    if (ptCurrentCell.X > -1 && ptCurrentCell.Y > -1)
                    {
                        InvalidateCellPrivate(ptCurrentCell.X /*columnIndex*/, ptCurrentCell.Y /*rowIndex*/);
                    }
                }
                // OnCellValidating returns true if the dev cancelled the validation.
                bool validateFormattedValue = CanValidateDataBoundDataGridViewCell(dataGridViewCurrentCell);
                if (validateFormattedValue)
                {
                    if (ptCurrentCell.X == -1)
                    {
                        return null;
                    }
                    if (OnCellValidating(ref dataGridViewCurrentCell, ptCurrentCell.X, ptCurrentCell.Y, context))
                    {
                        if (fireRowEnter)
                        {
                            if (ptCurrentCell.X == -1)
                            {
                                return null;
                            }
                            OnRowEnter(ref dataGridViewCurrentCell, ptCurrentCell.X, ptCurrentCell.Y, true /*canCreateNewRow*/, true /*validationFailureOccurred*/);
                        }
                        if (fireCellEnter)
                        {
                            if (ptCurrentCell.X == -1)
                            {
                                return null;
                            }
                            OnCellEnter(ref dataGridViewCurrentCell, ptCurrentCell.X, ptCurrentCell.Y);
                        }
                        if (ptCurrentCell.X == -1)
                        {
                            return null;
                        }
                        DataGridViewDataErrorEventArgs dgvdee = new DataGridViewDataErrorEventArgs(null,
                                                                                    ptCurrentCell.X,
                                                                                    ptCurrentCell.Y,
                                                                                    // null,
                                                                                    // null,
                                                                                    context)
                        {
                            Cancel = true
                        };
                        return dgvdee;
                    }

                    if (!IsCurrentCellInEditMode || !IsCurrentCellDirty)
                    {
                        if (ptCurrentCell.X == -1)
                        {
                            return null;
                        }
                        OnCellValidated(ref dataGridViewCurrentCell, ptCurrentCell.X, ptCurrentCell.Y);
                    }
                }
            }

            if (ptCurrentCell.X == -1 || !IsCurrentCellInEditMode)
            {
                return null;
            }

            Debug.Assert(
                 (
                  (editingControl != null && ((IDataGridViewEditingControl)editingControl).EditingControlValueChanged) ||
                  (dataGridViewState1[DATAGRIDVIEWSTATE1_currentCellInEditMode] && ((IDataGridViewEditingCell)CurrentCellInternal).EditingCellValueChanged)
                 ) == IsCurrentCellDirty ||
                 dataGridViewState1[DATAGRIDVIEWSTATE1_ignoringEditingChanges]);

            if (IsCurrentCellDirty)
            {
                bool validateAndPushFormattedValue = CanValidateDataBoundDataGridViewCell(dataGridViewCurrentCell);
                if (validateAndPushFormattedValue)
                {
                    if (validateCell == DataGridViewValidateCellInternal.WhenChanged)
                    {
                        Debug.Assert(ptCurrentCell.X > -1);
                        if (ptCurrentCell.X == -1)
                        {
                            return null;
                        }
                        if (OnCellValidating(ref dataGridViewCurrentCell, ptCurrentCell.X, ptCurrentCell.Y, context))
                        {
                            if (ptCurrentCell.X == -1)
                            {
                                return null;
                            }
                            DataGridViewDataErrorEventArgs dgvdee = new DataGridViewDataErrorEventArgs(null,
                                                                                                       ptCurrentCell.X,
                                                                                                       ptCurrentCell.Y,
                                                                                                       context)
                            {
                                Cancel = true
                            };
                            return dgvdee;
                        }
                    }

                    object formattedValue;

                    if (editingControl != null)
                    {
                        formattedValue = ((IDataGridViewEditingControl)editingControl).GetEditingControlFormattedValue(context);
                    }
                    else
                    {
                        Debug.Assert(dataGridViewState1[DATAGRIDVIEWSTATE1_currentCellInEditMode]);
                        formattedValue = ((IDataGridViewEditingCell)CurrentCellInternal).GetEditingCellFormattedValue(context);
                    }

                    if (!PushFormattedValue(ref dataGridViewCurrentCell, formattedValue, out Exception exception))
                    {
                        if (ptCurrentCell.X == -1)
                        {
                            return null;
                        }
                        DataGridViewDataErrorEventArgs dgvdee = new DataGridViewDataErrorEventArgs(exception,
                                                                                    ptCurrentCell.X,
                                                                                    ptCurrentCell.Y,
                                                                                    // dataGridViewCurrentCell.Value,
                                                                                    // formattedValue,
                                                                                    context)
                        {
                            Cancel = true
                        };
                        OnDataErrorInternal(dgvdee);
                        return dgvdee;
                    }
                    if (!IsCurrentCellInEditMode)
                    {
                        return null;
                    }
                    uneditedFormattedValue = formattedValue;
                }

                if (editingControl != null)
                {
                    ((IDataGridViewEditingControl)editingControl).EditingControlValueChanged = false;
                }
                else
                {
                    Debug.Assert(dataGridViewState1[DATAGRIDVIEWSTATE1_currentCellInEditMode]);
                    ((IDataGridViewEditingCell)CurrentCellInternal).EditingCellValueChanged = false;
                }
                IsCurrentCellDirtyInternal = false;
                IsCurrentRowDirtyInternal = true;

                if (validateAndPushFormattedValue)
                {
                    if (validateCell == DataGridViewValidateCellInternal.Always ||
                        validateCell == DataGridViewValidateCellInternal.WhenChanged)
                    {
                        if (ptCurrentCell.X == -1)
                        {
                            return null;
                        }
                        OnCellValidated(ref dataGridViewCurrentCell, ptCurrentCell.X, ptCurrentCell.Y);
                    }
                }
            }
            return null;
        }

        private bool CommitEdit(DataGridViewDataErrorContexts context,
                                bool forCurrentCellChange,
                                bool forCurrentRowChange)
        {
            // If we're already within a CellValidating event handler, don't try to commit the cell again.
            if (dataGridViewOper[DATAGRIDVIEWOPER_inCellValidating])
            {
                return false;
            }

            DataGridViewCell dataGridViewCurrentCell = CurrentCellInternal;
            DataGridViewDataErrorEventArgs dgvdee = CommitEdit(ref dataGridViewCurrentCell,
                context,
                forCurrentCellChange ? DataGridViewValidateCellInternal.Always : DataGridViewValidateCellInternal.WhenChanged /*validateCell*/,
                forCurrentCellChange /*fireCellLeave*/,
                forCurrentCellChange /*fireCellEnter*/,
                forCurrentRowChange /*fireRowLeave*/,
                forCurrentRowChange /*fireRowEnter*/,
                false /*fireLeave*/);
            if (null != dgvdee)
            {
                if (dgvdee.ThrowException)
                {
                    throw dgvdee.Exception;
                }
                if (dgvdee.Cancel)
                {
                    return false;
                }
                dgvdee = CancelEditPrivate(/*ref dataGridViewCurrentCell,
                                           DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit | DataGridViewDataErrorContexts.Scroll*/);    // restore old value
                if (null != dgvdee)
                {
                    if (dgvdee.ThrowException)
                    {
                        throw dgvdee.Exception;
                    }
                    if (dgvdee.Cancel)
                    {
                        return false;
                    }
                }
            }

            // See if we can leave the row
            if (forCurrentRowChange && forCurrentCellChange)
            {
                Debug.Assert(ptCurrentCell.X > -1);
                if (ptCurrentCell.X == -1)
                {
                    return false;
                }
                int columnIndex = ptCurrentCell.X;
                int rowIndex = ptCurrentCell.Y;
                // OnRowValidating returns true when the row validation was cancelled.
                if (OnRowValidating(ref dataGridViewCurrentCell, columnIndex, rowIndex))
                {
                    if (IsInnerCellOutOfBounds(columnIndex, rowIndex))
                    {
                        return false;
                    }
                    OnRowEnter(ref dataGridViewCurrentCell, columnIndex, rowIndex, true /*canCreateNewRow*/, true /*validationFailureOccurred*/);
                    if (IsInnerCellOutOfBounds(columnIndex, rowIndex))
                    {
                        return false;
                    }
                    OnCellEnter(ref dataGridViewCurrentCell, columnIndex, rowIndex);
                    return false;
                }
                if (IsInnerCellOutOfBounds(columnIndex, rowIndex))
                {
                    return false;
                }
                OnRowValidated(ref dataGridViewCurrentCell, columnIndex, rowIndex);
            }
            return true;
        }

        private bool CommitEditForOperation(int columnIndex, int rowIndex, bool forCurrentCellChange)
        {
            if (forCurrentCellChange)
            {
                if (!EndEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit | DataGridViewDataErrorContexts.CurrentCellChange,
                                DataGridViewValidateCellInternal.Always /*validateCell*/,
                                true /*fireCellLeave*/,
                                true /*fireCellEnter*/,
                                ptCurrentCell.Y != rowIndex /*fireRowLeave*/,
                                ptCurrentCell.Y != rowIndex /*fireRowEnter*/,
                                false /*fireLeave*/,
                                EditMode != DataGridViewEditMode.EditOnEnter /*keepFocus*/,
                                false /*resetCurrentCell*/,
                                false /*resetAnchorCell unused here*/))
                {
                    return false;
                }
                if (ptCurrentCell.Y != rowIndex && ptCurrentCell.Y != -1)
                {
                    DataGridViewCell dataGridViewCellTmp = null;
                    int columnIndexTmp = ptCurrentCell.X;
                    int rowIndexTmp = ptCurrentCell.Y;
                    if (OnRowValidating(ref dataGridViewCellTmp, columnIndexTmp, rowIndexTmp))
                    {
                        // Row validation was cancelled
                        if (IsInnerCellOutOfBounds(columnIndexTmp, rowIndexTmp))
                        {
                            return false;
                        }
                        OnRowEnter(ref dataGridViewCellTmp, columnIndexTmp, rowIndexTmp, true /*canCreateNewRow*/, true /*validationFailureOccurred*/);
                        if (IsInnerCellOutOfBounds(columnIndexTmp, rowIndexTmp))
                        {
                            return false;
                        }
                        OnCellEnter(ref dataGridViewCellTmp, columnIndexTmp, rowIndexTmp);
                        if (IsInnerCellOutOfBounds(columnIndexTmp, rowIndexTmp))
                        {
                            return false;
                        }

                        // Re-enter editing mode if needed
                        if (Focused &&
                            (!IsCurrentCellInEditMode && (EditMode == DataGridViewEditMode.EditOnEnter ||
                            (EditMode != DataGridViewEditMode.EditProgrammatically && CurrentCellInternal.EditType == null))))
                        {
                            BeginEditInternal(true /*selectAll*/);
                        }

                        return false;
                    }
                    if (IsInnerCellOutOfBounds(columnIndexTmp, rowIndexTmp))
                    {
                        return false;
                    }
                    OnRowValidated(ref dataGridViewCellTmp, columnIndexTmp, rowIndexTmp);
                }
            }
            else
            {
                if (!CommitEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit | DataGridViewDataErrorContexts.Scroll,
                                false /*forCurrentCellChange*/,
                                ptCurrentCell.Y != rowIndex /*forCurrentRowChange*/))
                {
                    return false;
                }
            }

            // Row validation was not cancelled, but operation needs to be re-evaluated.
            Debug.Assert(columnIndex < Columns.Count);
            if (IsColumnOutOfBounds(columnIndex))
            {
                return false;
            }
            if (rowIndex >= Rows.Count)
            {
                // CurrentCell was reset because the commit deleted row(s).
                // Since the user wants to change the current cell, we don't
                // want to end up with no CurrentCell. We pick the last visible
                // row in the grid which may be the 'new row'.
                int lastVisibleRowIndex = Rows.GetLastRow(DataGridViewElementStates.Visible);
                if (forCurrentCellChange &&
                    ptCurrentCell.X == -1 &&
                    lastVisibleRowIndex != -1)
                {
                    bool success = SetAndSelectCurrentCellAddress(columnIndex, lastVisibleRowIndex, true, false, false, false /*clearSelection*/, false /*forceCurrentCellSelection*/);
                    Debug.Assert(success);
                }
                // Interrupt operation because it has become invalid.
                return false;
            }
            if (rowIndex > -1 && (Rows.GetRowState(rowIndex) & DataGridViewElementStates.Visible) == 0)
            {
                // Interrupt operation because target row has become invisible.
                return false;
            }
            return true;
        }

        internal void CompleteCellsCollection(DataGridViewRow dataGridViewRow)
        {
            Debug.Assert(dataGridViewRow != null);
            int cellsInCollection = dataGridViewRow.Cells.Count;
            if (Columns.Count > cellsInCollection)
            {
                int cellCount = 0;
                DataGridViewCell[] cells = new DataGridViewCell[Columns.Count - cellsInCollection];
                for (int columnIndex = cellsInCollection; columnIndex < Columns.Count; columnIndex++)
                {
                    if (Columns[columnIndex].CellTemplate == null)
                    {
                        throw new InvalidOperationException(SR.DataGridView_AColumnHasNoCellTemplate);
                    }
                    DataGridViewCell dgvcNew = (DataGridViewCell)Columns[columnIndex].CellTemplate.Clone();
                    cells[cellCount] = dgvcNew;
                    cellCount++;
                }
                dataGridViewRow.Cells.AddRange(cells);
            }
        }

        /// <summary>
        ///  Determines which column is the first visible scrolling
        ///  column given the object's horizontalOffset.
        /// </summary>
        private int ComputeFirstVisibleScrollingColumn()
        {
            if (Columns.GetColumnsWidth(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen) >= layout.Data.Width)
            {
                // Not enough room for scrolling columns.
                negOffset = 0;
                return -1;
            }

            DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);

            if (horizontalOffset == 0)
            {
                negOffset = 0;
                return (dataGridViewColumn == null) ? -1 : dataGridViewColumn.Index;
            }

            int cx = 0;
            while (dataGridViewColumn != null)
            {
                cx += dataGridViewColumn.Thickness;
                if (cx > horizontalOffset)
                {
                    break;
                }
                dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                    DataGridViewElementStates.Visible,
                    DataGridViewElementStates.None);
            }

            if (dataGridViewColumn == null)
            {
                Debug.Assert(cx <= horizontalOffset);
                dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                if (dataGridViewColumn == null)
                {
                    negOffset = 0;
                    return -1;
                }
                else
                {
                    if (negOffset != horizontalOffset)
                    {
                        negOffset = 0;
                    }
                    return dataGridViewColumn.Index;
                }
            }
            else
            {
                negOffset = dataGridViewColumn.Thickness - (cx - horizontalOffset);
                return dataGridViewColumn.Index;
            }
        }

        private int ComputeHeightOfFittingTrailingScrollingRows(int totalVisibleFrozenHeight)
        {
            //
            int displayHeight = layout.Data.Height - totalVisibleFrozenHeight;
            int rowHeight = 0, rowHeights = 0;
            int indexTmp = Rows.Count;

            if (indexTmp == 0 || displayHeight <= 0)
            {
                return 0;
            }
            else
            {
                indexTmp--;
            }

            DataGridViewElementStates rowState = Rows.GetRowState(indexTmp);
            if ((rowState & DataGridViewElementStates.Frozen) != 0)
            {
                return 0;
            }
            if ((rowState & DataGridViewElementStates.Visible) == 0)
            {
                indexTmp = Rows.GetPreviousRow(indexTmp,
                    DataGridViewElementStates.Visible,
                    DataGridViewElementStates.Frozen);
            }

            if (indexTmp != -1)
            {
                rowHeight = Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                if (rowHeight > displayHeight)
                {
                    return rowHeight;
                }
            }

            while (indexTmp != -1 && rowHeights + rowHeight <= displayHeight)
            {
                rowHeights += rowHeight;
                indexTmp = Rows.GetPreviousRow(indexTmp,
                    DataGridViewElementStates.Visible,
                    DataGridViewElementStates.Frozen);
                if (indexTmp != -1)
                {
                    rowHeight = Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                }
            }

            return rowHeights;
        }

        private int ComputeHeightOfScrolledOffRows()
        {
            //
            int height = 0;
            if (displayedBandsInfo.FirstDisplayedScrollingRow >= 0)
            {
                int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                if (rowIndex != -1)
                {
                    while (rowIndex != displayedBandsInfo.FirstDisplayedScrollingRow)
                    {
                        Debug.Assert(rowIndex < displayedBandsInfo.FirstDisplayedScrollingRow);
                        height += Rows.SharedRow(rowIndex).GetHeight(rowIndex);
                        rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                    }
                }
            }
            return height;
        }

        private int ComputeHeightOfTrailingScrollingRows()
        {
            if (displayedBandsInfo.FirstDisplayedScrollingRow >= 0)
            {
                int lastVisibleRowIndex = Rows.GetLastRow(DataGridViewElementStates.Visible);
                return Rows.GetRowsHeight(DataGridViewElementStates.Visible, displayedBandsInfo.FirstDisplayedScrollingRow, lastVisibleRowIndex) +
                       Rows.SharedRow(lastVisibleRowIndex).GetHeight(lastVisibleRowIndex);
            }
            return 0;
        }

        private bool ComputeLayout()
        {
            ClearRegionCache();

            LayoutData newLayout = new LayoutData(layout);
            Rectangle oldResizeRect = layout.ResizeBoxRect;

            // Inside region
            if (normalClientRectangle.Width > 0 || normalClientRectangle.Height > 0)
            {
                newLayout.Inside = normalClientRectangle;
            }
            else
            {
                newLayout.Inside = ClientRectangle;
            }
            Rectangle inside = newLayout.Inside;
            int borderWidth = BorderWidth;
            inside.Inflate(-borderWidth, -borderWidth);
            if (inside.Height < 0)
            {
                inside.Height = 0;
            }
            if (inside.Width < 0)
            {
                inside.Width = 0;
            }

            Rectangle insideLeft = inside;

            // Headers
            if (layout.ColumnHeadersVisible)
            {
                Rectangle colHeaders = insideLeft;
                colHeaders.Height = Math.Min(columnHeadersHeight, colHeaders.Height);
                insideLeft.Y += colHeaders.Height;
                insideLeft.Height -= colHeaders.Height;
                Debug.Assert(insideLeft.Height >= 0);
                newLayout.ColumnHeaders = colHeaders;
            }
            else
            {
                newLayout.ColumnHeaders = Rectangle.Empty;
            }

            if (layout.RowHeadersVisible)
            {
                Rectangle rowHeaders = insideLeft;
                rowHeaders.Width = Math.Min(rowHeadersWidth, rowHeaders.Width);
                if (RightToLeftInternal)
                {
                    rowHeaders.X += insideLeft.Width - rowHeaders.Width;
                }
                else
                {
                    insideLeft.X += rowHeaders.Width;
                }
                insideLeft.Width -= rowHeaders.Width;
                Debug.Assert(insideLeft.Width >= 0);
                newLayout.RowHeaders = rowHeaders;

                if (layout.ColumnHeadersVisible)
                {
                    Rectangle topLeft;
                    Rectangle colHeaders = newLayout.ColumnHeaders;
                    topLeft = colHeaders;
                    topLeft.Width = Math.Min(rowHeadersWidth, topLeft.Width);
                    colHeaders.Width -= topLeft.Width;
                    if (RightToLeftInternal)
                    {
                        topLeft.X += insideLeft.Width;
                    }
                    else
                    {
                        colHeaders.X += topLeft.Width;
                    }
                    Debug.Assert(colHeaders.Width >= 0);
                    newLayout.TopLeftHeader = topLeft;
                    newLayout.ColumnHeaders = colHeaders;
                }
                else
                {
                    newLayout.TopLeftHeader = Rectangle.Empty;
                }
            }
            else
            {
                newLayout.RowHeaders = Rectangle.Empty;
                newLayout.TopLeftHeader = Rectangle.Empty;
            }

            // Adjust insideLeft in case static top / left edge needs to be painted
            if (SingleVerticalBorderAdded)
            {
                if (!RightToLeftInternal)
                {
                    insideLeft.X++;
                }
                if (insideLeft.Width > 0)
                {
                    insideLeft.Width--;
                }
            }
            if (SingleHorizontalBorderAdded)
            {
                insideLeft.Y++;
                if (insideLeft.Height > 0)
                {
                    insideLeft.Height--;
                }
            }

            // Data region
            newLayout.Data = insideLeft;
            newLayout.Inside = inside;

            Debug.Assert(newLayout.Data.X >= 0);
            Debug.Assert(newLayout.Data.Y >= 0);
            Debug.Assert(newLayout.Data.Width >= 0);
            Debug.Assert(newLayout.Data.Height >= 0);

            layout = newLayout;
            layout.dirty = false;

            bool columnsAdjusted = AdjustFillingColumns();

            layout = newLayout;
            Debug.Assert(!layout.dirty);
            LayoutScrollBars();

            // if the user shrank the grid client area, then OnResize invalidated the old
            // resize area. however, we need to invalidate the left upper corner in the new ResizeArea
            // note that we can't take the Invalidate call from the OnResize method, because if the
            // user enlarges the form then the old area will not be invalidated.
            //
            if (!oldResizeRect.Equals(layout.ResizeBoxRect) && !layout.ResizeBoxRect.IsEmpty)
            {
                Invalidate(layout.ResizeBoxRect);
            }

            return columnsAdjusted;
        }

        private void ComputeLayoutShortcut(bool computeVisibleRows)
        {
            // Called instead of ComputeLayout when a row is added, inserted or deleted beyond the limits of
            // the layout.Data area.
            // this.layout is unchanged - only the potential vertical scrollbar is affected.

            if (computeVisibleRows)
            {
                ComputeVisibleRows();
            }
#if DEBUG
            else
            {
                int oldNumTotallyVisibleFrozenRows = displayedBandsInfo.NumTotallyDisplayedFrozenRows;
                int oldNumVisibleScrollingRows = displayedBandsInfo.NumDisplayedScrollingRows;
                int oldNumTotallyVisibleScrollingRows = displayedBandsInfo.NumTotallyDisplayedScrollingRows;
                int oldFirstVisibleScrollingRow = displayedBandsInfo.FirstDisplayedScrollingRow;
                ComputeVisibleRows();
                Debug.Assert(oldNumTotallyVisibleFrozenRows == displayedBandsInfo.NumTotallyDisplayedFrozenRows);
                Debug.Assert(oldNumVisibleScrollingRows == displayedBandsInfo.NumDisplayedScrollingRows);
                Debug.Assert(oldNumTotallyVisibleScrollingRows == displayedBandsInfo.NumTotallyDisplayedScrollingRows);
                Debug.Assert(oldFirstVisibleScrollingRow == displayedBandsInfo.FirstDisplayedScrollingRow);
            }
#endif

#if DEBUG
            int newFirstVisibleScrollingCol = ComputeFirstVisibleScrollingColumn();
            Debug.Assert(newFirstVisibleScrollingCol == displayedBandsInfo.FirstDisplayedScrollingCol);

            int oldLastTotallyVisibleScrollingCol = displayedBandsInfo.LastTotallyDisplayedScrollingCol;
            int oldFirstVisibleScrollingCol = displayedBandsInfo.FirstDisplayedScrollingCol;
            ComputeVisibleColumns();
            Debug.Assert(oldLastTotallyVisibleScrollingCol == displayedBandsInfo.LastTotallyDisplayedScrollingCol);
            Debug.Assert(oldFirstVisibleScrollingCol == displayedBandsInfo.FirstDisplayedScrollingCol);
#endif

            if (vertScrollBar.Enabled)
            {
                int totalVisibleHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible);
                int totalVisibleFrozenHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                int oldVertScrollBarValue = vertScrollBar.Value;
                int oldThumbHeight = Math.Max(((vertScrollBar.Height - 2 * SystemInformation.VerticalScrollBarArrowHeight) * vertScrollBar.LargeChange) / vertScrollBar.Maximum, 8);

                vertScrollBar.Maximum = totalVisibleHeight - totalVisibleFrozenHeight;
                Debug.Assert(vertScrollBar.Maximum > 0);
                vertScrollBar.Value = ComputeHeightOfScrolledOffRows();
                vertScrollBar.LargeChange = layout.Data.Height - totalVisibleFrozenHeight;
                verticalOffset = vertScrollBar.Value;

                if (vertScrollBar.Visible &&
                    (oldVertScrollBarValue != verticalOffset ||
                     oldThumbHeight != Math.Max(((vertScrollBar.Height - 2 * SystemInformation.VerticalScrollBarArrowHeight) * vertScrollBar.LargeChange) / vertScrollBar.Maximum, 8)))
                {
                    // Only update the vertical scroll bar is the thumb moved or resized.
                    vertScrollBar.Invalidate();
                }
                Debug.Assert(verticalOffset == vertScrollBar.Value);
            }
        }

        /* Unused for now
        private int ComputeScrolledOffRowCount(int scrolledOffRowsHeight)
        {
            int rowIndex = this.Rows.GetFirstRow(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
            if (rowIndex == -1)
            {
                // No scrolling rows
                return 0;
            }
            else
            {
                int height = 0;
                int rowCount = 0;
                while (rowIndex != -1 && height < scrolledOffRowsHeight)
                {
                    height += this.Rows.SharedRow(rowIndex).GetHeight(rowIndex);
                    if (height <= scrolledOffRowsHeight)
                    {
                        rowCount++;
                    }
                    rowIndex = this.Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                }
                return rowCount;
            }
        }
        */

        private void ComputeVisibleColumns()
        {
            DataGridViewColumn dataGridViewColumn = null;
            int numVisibleScrollingCols = 0, visibleScrollingColumnsTmp = 0;
            int displayWidth = layout.Data.Width, cx = 0;
            int numDisplayedFrozenCols = 0, firstDisplayedFrozenCol = -1, lastDisplayedFrozenCol = -1;
            int firstDisplayedScrollingCol = displayedBandsInfo.FirstDisplayedScrollingCol;

            // the same problem with negative numbers:
            // if the width passed in is negative, then return 0
            if (displayWidth <= 0 || Columns.GetColumnCount(DataGridViewElementStates.Visible) == 0)
            {
                displayedBandsInfo.FirstDisplayedFrozenCol = -1;
                displayedBandsInfo.NumDisplayedFrozenCols = 0;
                displayedBandsInfo.FirstDisplayedScrollingCol = -1;
                displayedBandsInfo.NumDisplayedScrollingCols = 0;
                displayedBandsInfo.LastDisplayedFrozenCol = -1;
                displayedBandsInfo.LastTotallyDisplayedScrollingCol = -1;
                return;
            }

            dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.None);
            while (dataGridViewColumn != null)
            {
                if (!dataGridViewColumn.Frozen && dataGridViewColumn.Visible)
                {
                    break;
                }
                if (dataGridViewColumn.Visible)
                {
                    if (firstDisplayedFrozenCol == -1)
                    {
                        firstDisplayedFrozenCol = dataGridViewColumn.Index;
                    }
                    cx += dataGridViewColumn.Width;
                    numDisplayedFrozenCols++;
                    lastDisplayedFrozenCol = dataGridViewColumn.Index;
                    if (cx >= displayWidth)
                    {
                        break;
                    }
                }
                dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.None, DataGridViewElementStates.None);
            }

            Debug.Assert(cx <= Columns.GetColumnsWidth(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen));

            if (cx < displayWidth && firstDisplayedScrollingCol >= 0)
            {
                dataGridViewColumn = Columns[firstDisplayedScrollingCol];
                if (dataGridViewColumn.Frozen)
                {
                    dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible,
                        DataGridViewElementStates.Frozen);
                    negOffset = 0;
                    if (dataGridViewColumn == null)
                    {
                        displayedBandsInfo.FirstDisplayedFrozenCol = firstDisplayedFrozenCol;
                        displayedBandsInfo.LastDisplayedFrozenCol = lastDisplayedFrozenCol;
                        displayedBandsInfo.NumDisplayedFrozenCols = numDisplayedFrozenCols;
                        displayedBandsInfo.FirstDisplayedScrollingCol = displayedBandsInfo.LastTotallyDisplayedScrollingCol = -1;
                        displayedBandsInfo.NumDisplayedScrollingCols = 0;
                        return;
                    }
                    else
                    {
                        firstDisplayedScrollingCol = dataGridViewColumn.Index;
                    }
                }

                cx -= negOffset;
                while (cx < displayWidth && dataGridViewColumn != null)
                {
                    cx += dataGridViewColumn.Thickness;
                    visibleScrollingColumnsTmp++;
                    dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                        DataGridViewElementStates.Visible,
                        DataGridViewElementStates.None);
                }
                numVisibleScrollingCols = visibleScrollingColumnsTmp;

                // if we inflate the data area then we paint columns to the left of firstDisplayedScrollingCol
                if (cx < displayWidth)
                {
                    bool invalidate = false;
                    Debug.Assert(firstDisplayedScrollingCol >= 0);
                    //first minimize value of this.negOffset
                    if (negOffset > 0)
                    {
                        invalidate = true;
                        if (displayWidth - cx > negOffset)
                        {
                            cx += negOffset;
                            horizontalOffset -= negOffset;
                            negOffset = 0;
                        }
                        else
                        {
                            horizontalOffset -= displayWidth - cx;
                            negOffset -= displayWidth - cx;
                            cx = displayWidth;
                        }
                    }
                    // second try to scroll entire columns
                    if (cx < displayWidth && horizontalOffset > 0)
                    {
                        Debug.Assert(negOffset == 0);
                        dataGridViewColumn = Columns.GetPreviousColumn((Columns[firstDisplayedScrollingCol]),
                            DataGridViewElementStates.Visible,
                            DataGridViewElementStates.Frozen);
                        while (dataGridViewColumn != null && cx + dataGridViewColumn.Thickness <= displayWidth)
                        {
                            cx += dataGridViewColumn.Thickness;
                            visibleScrollingColumnsTmp++;
                            invalidate = true;
                            firstDisplayedScrollingCol = dataGridViewColumn.Index;
                            horizontalOffset -= dataGridViewColumn.Thickness;
                            dataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn,
                                DataGridViewElementStates.Visible,
                                DataGridViewElementStates.Frozen);
                        }
                    }
                    // third try to partially scroll in first scrolled off column
                    if (cx < displayWidth && horizontalOffset > 0 && firstDisplayedScrollingCol != 0)
                    {
                        Debug.Assert(negOffset == 0);
                        dataGridViewColumn = Columns.GetPreviousColumn((Columns[firstDisplayedScrollingCol]),
                                                                            DataGridViewElementStates.Visible,
                                                                            DataGridViewElementStates.Frozen);
                        Debug.Assert(dataGridViewColumn != null);
                        Debug.Assert(dataGridViewColumn.Thickness > displayWidth - cx);
                        firstDisplayedScrollingCol = dataGridViewColumn.Index;
                        negOffset = dataGridViewColumn.Thickness - displayWidth + cx;
                        horizontalOffset -= displayWidth - cx;
                        visibleScrollingColumnsTmp++;
                        invalidate = true;
                        cx = displayWidth;
                        Debug.Assert(negOffset == GetNegOffsetFromHorizontalOffset(horizontalOffset));
                    }

                    // update the number of visible columns to the new reality
                    Debug.Assert(numVisibleScrollingCols <= visibleScrollingColumnsTmp, "the number of displayed columns can only grow");
                    numVisibleScrollingCols = visibleScrollingColumnsTmp;

                    if (invalidate)
                    {
                        InvalidateData();
                        Invalidate(layout.ColumnHeaders);
                    }
                }

                int jumpFromFirstVisibleScrollingCol = numVisibleScrollingCols - 1;
                if (cx > displayWidth)
                {
                    jumpFromFirstVisibleScrollingCol--;
                }

                Debug.Assert(jumpFromFirstVisibleScrollingCol >= -1);

                if (jumpFromFirstVisibleScrollingCol < 0)
                {
                    displayedBandsInfo.LastTotallyDisplayedScrollingCol = -1; // no totally visible scrolling column at all
                }
                else
                {
                    Debug.Assert(firstDisplayedScrollingCol >= 0);
                    dataGridViewColumn = Columns[firstDisplayedScrollingCol];
                    for (int jump = 0; jump < jumpFromFirstVisibleScrollingCol; jump++)
                    {
                        dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                            DataGridViewElementStates.Visible,
                            DataGridViewElementStates.None);
                        Debug.Assert(dataGridViewColumn != null);
                    }
                    displayedBandsInfo.LastTotallyDisplayedScrollingCol = dataGridViewColumn.Index;
                }
            }
            else
            {
                displayedBandsInfo.LastTotallyDisplayedScrollingCol = -1;
            }
            displayedBandsInfo.FirstDisplayedFrozenCol = firstDisplayedFrozenCol;
            displayedBandsInfo.LastDisplayedFrozenCol = lastDisplayedFrozenCol;
            displayedBandsInfo.NumDisplayedFrozenCols = numDisplayedFrozenCols;
            displayedBandsInfo.FirstDisplayedScrollingCol = firstDisplayedScrollingCol;
            displayedBandsInfo.NumDisplayedScrollingCols = numVisibleScrollingCols;
            Debug.Assert((displayedBandsInfo.NumDisplayedScrollingCols > 0 && displayedBandsInfo.FirstDisplayedScrollingCol != -1) ||
                         (displayedBandsInfo.NumDisplayedScrollingCols == 0 && displayedBandsInfo.FirstDisplayedScrollingCol == -1));
        }

        private void ComputeVisibleRows()
        {
            int firstDisplayedFrozenRow = -1;
            int firstDisplayedScrollingRow = displayedBandsInfo.FirstDisplayedScrollingRow;
            int lastDisplayedFrozenRow = -1, lastDisplayedScrollingRow = -1;
            int numTotallyDisplayedFrozenRows = 0;
            int displayHeight = layout.Data.Height;
            int cy = 0;
            int visibleScrollingRows = 0;
            int nRows = Rows.Count;
            int rowIndex;

            // when minimizing the dataGridView window, we will get negative values for the
            // layout.Data.Width and layout.Data.Height

            if (displayHeight <= 0 || nRows == 0)
            {
                displayedBandsInfo.NumDisplayedFrozenRows = displayedBandsInfo.NumTotallyDisplayedFrozenRows =
                    displayedBandsInfo.NumDisplayedScrollingRows = displayedBandsInfo.NumTotallyDisplayedScrollingRows = 0;
                displayedBandsInfo.FirstDisplayedFrozenRow = displayedBandsInfo.FirstDisplayedScrollingRow =
                    displayedBandsInfo.LastDisplayedFrozenRow = displayedBandsInfo.LastDisplayedScrollingRow = -1;
                return;
            }

            for (rowIndex = 0; rowIndex < nRows; rowIndex++)
            {
                Debug.Assert(cy < displayHeight);
                DataGridViewElementStates rowState = Rows.GetRowState(rowIndex);
                if ((rowState & DataGridViewElementStates.Frozen) == 0 &&
                    (rowState & DataGridViewElementStates.Visible) != 0)
                {
                    break;
                }
                if ((rowState & DataGridViewElementStates.Visible) != 0)
                {
                    cy += Rows.SharedRow(rowIndex).GetHeight(rowIndex);
                    if (firstDisplayedFrozenRow == -1)
                    {
                        firstDisplayedFrozenRow = rowIndex;
                    }
                    lastDisplayedFrozenRow = rowIndex;
                    if (cy <= displayHeight)
                    {
                        numTotallyDisplayedFrozenRows++;
                    }
                    if (cy >= displayHeight)
                    {
                        break;
                    }
                }
            }

            if (cy > displayHeight)
            {
                displayedBandsInfo.NumDisplayedFrozenRows = numTotallyDisplayedFrozenRows + 1;
            }
            else
            {
                displayedBandsInfo.NumDisplayedFrozenRows = numTotallyDisplayedFrozenRows;
            }

            // loop exited when:
            // - all rows are frozen and fit in displayHeight: rowIndex == nRows, cy <= displayHeight
            // - rowIndex is not frozen: rowIndex < nRows, cy <= displayHeight
            // - there are more frozen rows than can fit in displayHeight: rowIndex <= nRows, cy > displayHeight

            if (cy < displayHeight && rowIndex < nRows)
            {
                if (firstDisplayedScrollingRow == -1)
                {
                    firstDisplayedScrollingRow = rowIndex;
                }

                while (firstDisplayedScrollingRow < nRows &&
                    (
                    (Rows.GetRowState(firstDisplayedScrollingRow) & DataGridViewElementStates.Frozen) != 0 ||
                    (Rows.GetRowState(firstDisplayedScrollingRow) & DataGridViewElementStates.Visible) == 0
                    )
                    )
                {
                    firstDisplayedScrollingRow++;
                }

                for (int i = firstDisplayedScrollingRow; i < nRows; i++)
                {
                    if ((Rows.GetRowState(i) & DataGridViewElementStates.Visible) != 0)
                    {
                        cy += Rows.SharedRow(i).GetHeight(i);
                        visibleScrollingRows++;
                        lastDisplayedScrollingRow = i;
                    }
                    if (cy >= displayHeight)
                    {
                        break;
                    }
                }

                if (cy < displayHeight)
                {
                    for (int i = firstDisplayedScrollingRow - 1; i >= numTotallyDisplayedFrozenRows; i--)
                    {
                        if ((Rows.GetRowState(i) & (DataGridViewElementStates.Frozen | DataGridViewElementStates.Visible)) == DataGridViewElementStates.Visible)
                        {
                            int height = Rows.SharedRow(i).GetHeight(i);
                            if (cy + height > displayHeight)
                            {
                                break;
                            }
                            cy += height;
                            firstDisplayedScrollingRow = i;
                            visibleScrollingRows++;
                            lastDisplayedScrollingRow = i;
                        }
                    }
                }

                displayedBandsInfo.NumDisplayedScrollingRows = visibleScrollingRows;
                if (cy > displayHeight)
                {
                    displayedBandsInfo.NumTotallyDisplayedScrollingRows = visibleScrollingRows - 1;
                }
                else
                {
                    displayedBandsInfo.NumTotallyDisplayedScrollingRows = visibleScrollingRows;
                }
                if (visibleScrollingRows == 0)
                {
                    firstDisplayedScrollingRow = -1;
                    Debug.Assert(lastDisplayedScrollingRow == -1);
                }
            }
            else
            {
                displayedBandsInfo.NumDisplayedScrollingRows = displayedBandsInfo.NumTotallyDisplayedScrollingRows = 0;
                firstDisplayedScrollingRow = -1;
            }

            Debug.Assert(firstDisplayedFrozenRow < nRows, "firstDisplayedFrozenRow larger than number of rows");
            Debug.Assert(lastDisplayedFrozenRow < nRows, "lastDisplayedFrozenRow larger than number of rows");
            Debug.Assert(lastDisplayedScrollingRow < nRows, "lastDisplayedScrollingRow larger than number of rows");

            displayedBandsInfo.FirstDisplayedFrozenRow = firstDisplayedFrozenRow;
            displayedBandsInfo.FirstDisplayedScrollingRow = firstDisplayedScrollingRow;
            displayedBandsInfo.NumTotallyDisplayedFrozenRows = numTotallyDisplayedFrozenRows;
            displayedBandsInfo.LastDisplayedFrozenRow = lastDisplayedFrozenRow;
            displayedBandsInfo.LastDisplayedScrollingRow = lastDisplayedScrollingRow;

            Debug.Assert(displayedBandsInfo.NumTotallyDisplayedFrozenRows >= 0, "the number of visible frozen rows can't be negative");
            Debug.Assert(displayedBandsInfo.NumDisplayedScrollingRows >= 0, "the number of visible scrolling rows can't be negative");
            Debug.Assert(displayedBandsInfo.NumTotallyDisplayedScrollingRows >= 0, "the number of totally visible scrolling rows can't be negative");
            Debug.Assert(displayedBandsInfo.FirstDisplayedScrollingRow < nRows, "firstDisplayedScrollingRow larger than number of rows");
        }

        private Point ConvertCellToGridCoord(int columnIndex, int rowIndex, int x, int y)
        {
            int columnX, rowY;
            if (columnIndex > -1)
            {
                columnX = GetColumnXFromIndex(columnIndex);
                if (RightToLeftInternal)
                {
                    columnX -= Columns[columnIndex].Width;
                }
            }
            else
            {
                if (RightToLeftInternal)
                {
                    columnX = layout.RowHeaders.Left - 1;
                }
                else
                {
                    columnX = layout.RowHeaders.Left;
                }
            }

            if (rowIndex > -1)
            {
                rowY = GetRowYFromIndex(rowIndex);
            }
            else
            {
                rowY = layout.ColumnHeaders.Top;
            }

            return new Point(columnX + x, rowY + y);
        }

        private void CorrectColumnDisplayIndexesAfterDeletion(DataGridViewColumn dataGridViewColumn)
        {
            // Column indexes have already been adjusted.
            // This column has already been detached and has retained its old Index and DisplayIndex

            Debug.Assert(dataGridViewColumn != null);
            Debug.Assert(dataGridViewColumn.DataGridView == null);
            Debug.Assert(dataGridViewColumn.Index >= 0);
            Debug.Assert(dataGridViewColumn.DisplayIndex >= 0);

            try
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inDisplayIndexAdjustments] = true;

                // All remaining columns with a DisplayIndex greater than dataGridViewColumn.DisplayIndex need to be decremented
                foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                {
                    if (dataGridViewColumnTmp.DisplayIndex > dataGridViewColumn.DisplayIndex)
                    {
                        dataGridViewColumnTmp.DisplayIndexInternal = dataGridViewColumnTmp.DisplayIndex - 1;
                        dataGridViewColumnTmp.DisplayIndexHasChanged = true; // OnColumnDisplayIndexChanged needs to be raised later on
                    }
                }

#if DEBUG
                Debug.Assert(Columns.VerifyColumnDisplayIndexes());
#endif
                // Now raise all the OnColumnDisplayIndexChanged events
                FlushDisplayIndexChanged(true /*raiseEvent*/);
            }
            finally
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inDisplayIndexAdjustments] = false;
                FlushDisplayIndexChanged(false /*raiseEvent*/);
            }
        }

        private void CorrectColumnDisplayIndexesAfterInsertion(DataGridViewColumn dataGridViewColumn)
        {
            Debug.Assert(dataGridViewColumn != null);
            Debug.Assert(dataGridViewColumn.DataGridView == this);
            // dataGridViewColumn.DisplayIndex has been set already.
            Debug.Assert(dataGridViewColumn.DisplayIndex >= 0);
            Debug.Assert(dataGridViewColumn.DisplayIndex < Columns.Count);

            try
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inDisplayIndexAdjustments] = true;

                // All other columns with a DisplayIndex equal or greater than dataGridViewColumn.DisplayIndex need to be incremented
                foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                {
                    if (dataGridViewColumnTmp != dataGridViewColumn && dataGridViewColumnTmp.DisplayIndex >= dataGridViewColumn.DisplayIndex)
                    {
                        dataGridViewColumnTmp.DisplayIndexInternal = dataGridViewColumnTmp.DisplayIndex + 1;
                        dataGridViewColumnTmp.DisplayIndexHasChanged = true; // OnColumnDisplayIndexChanged needs to be raised later on
                    }
                }

#if DEBUG
                Debug.Assert(Columns.VerifyColumnDisplayIndexes());
#endif
                // Now raise all the OnColumnDisplayIndexChanged events
                FlushDisplayIndexChanged(true /*raiseEvent*/);
            }
            finally
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inDisplayIndexAdjustments] = false;
                FlushDisplayIndexChanged(false /*raiseEvent*/);
            }
        }

        private void CorrectColumnFrozenState(DataGridViewColumn dataGridViewColumn, int anticipatedColumnIndex)
        {
            Debug.Assert(dataGridViewColumn != null);
            Debug.Assert(anticipatedColumnIndex >= 0 && anticipatedColumnIndex <= Columns.Count);

            int anticipatedColumnDisplayIndex;
            if (dataGridViewColumn.DisplayIndex == -1 || dataGridViewColumn.DisplayIndex > Columns.Count)
            {
                anticipatedColumnDisplayIndex = anticipatedColumnIndex; // By default, we pick the Index as the DisplayIndex.
            }
            else
            {
                Debug.Assert(dataGridViewColumn.DisplayIndex >= 0 && dataGridViewColumn.DisplayIndex <= Columns.Count);
                anticipatedColumnDisplayIndex = dataGridViewColumn.DisplayIndex; // The specified DisplayIndex is just fine.
            }

            DataGridViewColumn dataGridViewColumnPrev;
            int displayIndex = anticipatedColumnDisplayIndex - 1;
            do
            {
                dataGridViewColumnPrev = Columns.GetColumnAtDisplayIndex(displayIndex);
                displayIndex--;
            }
            while (displayIndex >= 0 && (dataGridViewColumnPrev == null || !dataGridViewColumnPrev.Visible));
            if (dataGridViewColumnPrev != null && !dataGridViewColumnPrev.Frozen && dataGridViewColumn.Frozen)
            {
                throw new InvalidOperationException(SR.DataGridView_CannotAddFrozenColumn);
            }
            else
            {
                DataGridViewColumn dataGridViewColumnNext;
                displayIndex = anticipatedColumnDisplayIndex;
                do
                {
                    dataGridViewColumnNext = Columns.GetColumnAtDisplayIndex(displayIndex);
                    displayIndex++;
                }
                while (displayIndex < Columns.Count && (dataGridViewColumnNext == null || !dataGridViewColumnNext.Visible));
                if (dataGridViewColumnNext != null && dataGridViewColumnNext.Frozen && !dataGridViewColumn.Frozen)
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAddNonFrozenColumn);
                }
            }
        }

        private void CorrectColumnFrozenStates(DataGridViewColumn[] dataGridViewColumns)
        {
            DataGridView dataGridViewTmp = new DataGridView();
            DataGridViewColumn dataGridViewColumnClone;
            foreach (DataGridViewColumn dataGridViewColumn in Columns)
            {
                dataGridViewColumnClone = (DataGridViewColumn)dataGridViewColumn.Clone();
                // DataGridViewColumn.Clone does not replicate the DisplayIndex value.
                dataGridViewColumnClone.DisplayIndex = dataGridViewColumn.DisplayIndex;
                dataGridViewTmp.Columns.Add(dataGridViewColumnClone);
            }
            foreach (DataGridViewColumn dataGridViewColumn in dataGridViewColumns)
            {
                dataGridViewColumnClone = (DataGridViewColumn)dataGridViewColumn.Clone();
                dataGridViewColumnClone.DisplayIndex = dataGridViewColumn.DisplayIndex;
                dataGridViewTmp.Columns.Add(dataGridViewColumnClone);
            }
        }

        private void CorrectColumnFrozenStates(DataGridViewColumn dataGridViewColumn, bool frozenStateChanging)
        {
            Debug.Assert(dataGridViewColumn != null);
            DataGridViewColumn dataGridViewColumnTmp;
            if ((dataGridViewColumn.Frozen && !frozenStateChanging) ||
                (!dataGridViewColumn.Frozen && frozenStateChanging))
            {
                // make sure the previous visible columns are frozen as well
                dataGridViewColumnTmp = Columns.GetPreviousColumn(dataGridViewColumn,
                    DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen,
                    DataGridViewElementStates.None);
                if (dataGridViewColumnTmp == null)
                {
                    DataGridViewColumn dataGridViewColumnFirst = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
                    if (dataGridViewColumnFirst != dataGridViewColumn)
                    {
                        dataGridViewColumnTmp = dataGridViewColumnFirst;
                    }
                }
                while (dataGridViewColumnTmp != null && Columns.DisplayInOrder(dataGridViewColumnTmp.Index, dataGridViewColumn.Index))
                {
                    dataGridViewColumnTmp.Frozen = true;
                    dataGridViewColumnTmp = Columns.GetNextColumn(dataGridViewColumnTmp,
                        DataGridViewElementStates.Visible,
                        DataGridViewElementStates.Frozen);
                }
            }
            else
            {
                // make sure the next visible columns are not frozen
                dataGridViewColumnTmp = Columns.GetNextColumn(dataGridViewColumn,
                    DataGridViewElementStates.Visible,
                    DataGridViewElementStates.Frozen);
                if (dataGridViewColumnTmp == null)
                {
                    DataGridViewColumn dataGridViewColumnLast = dataGridViewColumn;
                    do
                    {
                        dataGridViewColumnTmp = Columns.GetNextColumn(dataGridViewColumnLast,
                            DataGridViewElementStates.Visible,
                            DataGridViewElementStates.None);
                        if (dataGridViewColumnTmp != null)
                        {
                            dataGridViewColumnLast = dataGridViewColumnTmp;
                        }
                    }
                    while (dataGridViewColumnTmp != null);
                    if (dataGridViewColumnLast != dataGridViewColumn)
                    {
                        dataGridViewColumnTmp = dataGridViewColumnLast;
                    }
                }
                while (dataGridViewColumnTmp != null && Columns.DisplayInOrder(dataGridViewColumn.Index, dataGridViewColumnTmp.Index))
                {
                    dataGridViewColumnTmp.Frozen = false;
                    dataGridViewColumnTmp = Columns.GetPreviousColumn(dataGridViewColumnTmp,
                        DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen,
                        DataGridViewElementStates.None);
                }
            }
        }

        private void CorrectColumnFrozenStatesForMove(DataGridViewColumn dataGridViewColumn, int newDisplayIndex)
        {
            Debug.Assert(dataGridViewColumn != null);
            Debug.Assert(newDisplayIndex != dataGridViewColumn.DisplayIndex);
            Debug.Assert(!dataGridViewOper[DATAGRIDVIEWOPER_inDisplayIndexAdjustments]);

            // No check necessary when:
            // - column is invisible.
            // - DisplayIndex decreases and column is frozen.
            // - DisplayIndex increases and column is unfrozen.

            if (!dataGridViewColumn.Visible ||
                (newDisplayIndex < dataGridViewColumn.DisplayIndex && dataGridViewColumn.Frozen) ||
                (newDisplayIndex > dataGridViewColumn.DisplayIndex && !dataGridViewColumn.Frozen))
            {
                return;
            }

            int colCount = Columns.Count, displayIndex;

            if (newDisplayIndex < dataGridViewColumn.DisplayIndex)
            {
                // DisplayIndex decreases.
                // Throw an exception if the visible unfrozen column is placed before a frozen column
                // Get the closest visible column placed after the displaced column
                DataGridViewColumn dataGridViewColumnNext;
                displayIndex = newDisplayIndex;
                do
                {
                    dataGridViewColumnNext = Columns.GetColumnAtDisplayIndex(displayIndex);
                    displayIndex++;
                }
                while (displayIndex < colCount && (dataGridViewColumnNext == null || dataGridViewColumnNext == dataGridViewColumn || !dataGridViewColumnNext.Visible));

                if (dataGridViewColumnNext != null && dataGridViewColumnNext.Frozen)
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotMoveNonFrozenColumn);
                }
            }
            else
            {
                // DisplayIndex increases.
                // Throw an exception if the visible frozen column is placed after a non-frozen column
                // Get the closest visible column placed before the displaced column
                DataGridViewColumn dataGridViewColumnPrev;
                displayIndex = newDisplayIndex;
                do
                {
                    dataGridViewColumnPrev = Columns.GetColumnAtDisplayIndex(displayIndex);
                    displayIndex--;
                }
                while (displayIndex >= 0 && (dataGridViewColumnPrev == null || !dataGridViewColumnPrev.Visible));

                if (dataGridViewColumnPrev != null && !dataGridViewColumnPrev.Frozen)
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotMoveFrozenColumn);
                }
            }
        }

        private void CorrectColumnIndexesAfterDeletion(DataGridViewColumn dataGridViewColumn)
        {
            Debug.Assert(dataGridViewColumn != null);
            for (int columnIndex = dataGridViewColumn.Index; columnIndex < Columns.Count; columnIndex++)
            {
                Columns[columnIndex].Index = Columns[columnIndex].Index - 1;
                Debug.Assert(Columns[columnIndex].Index == columnIndex);
            }
        }

        private void CorrectColumnIndexesAfterInsertion(DataGridViewColumn dataGridViewColumn, int insertionCount)
        {
            Debug.Assert(dataGridViewColumn != null);
            Debug.Assert(insertionCount > 0);
            for (int columnIndex = dataGridViewColumn.Index + insertionCount; columnIndex < Columns.Count; columnIndex++)
            {
                Columns[columnIndex].Index = columnIndex;
            }
        }

        private void CorrectFocus(bool onlyIfGridHasFocus)
        {
            if ((!onlyIfGridHasFocus || Focused) && editingControl != null)
            {
                Debug.Assert(CurrentCellInternal != null);
                //Debug.Assert(this.editingControl.CanFocus);
                editingControl.Focus();
            }
        }

        private void CorrectRowFrozenState(DataGridViewRow dataGridViewRow, DataGridViewElementStates rowState, int anticipatedRowIndex)
        {
            Debug.Assert(dataGridViewRow != null);
            Debug.Assert(anticipatedRowIndex >= 0 && anticipatedRowIndex <= Rows.Count);

            int previousRowIndex = Rows.GetPreviousRow(anticipatedRowIndex,
                                                            DataGridViewElementStates.Visible,
                                                            DataGridViewElementStates.None);
            if (previousRowIndex != -1 &&
                (Rows.GetRowState(previousRowIndex) & DataGridViewElementStates.Frozen) == 0 &&
                (rowState & DataGridViewElementStates.Frozen) != 0)
            {
                throw new InvalidOperationException(SR.DataGridView_CannotAddFrozenRow);
            }
            else
            {
                int nextRowIndex = Rows.GetNextRow((previousRowIndex == -1) ? anticipatedRowIndex - 1 : previousRowIndex,
                                                        DataGridViewElementStates.Visible,
                                                        DataGridViewElementStates.None);
                if (nextRowIndex != -1 &&
                    (Rows.GetRowState(nextRowIndex) & DataGridViewElementStates.Frozen) != 0 &&
                    (rowState & DataGridViewElementStates.Frozen) == 0)
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAddNonFrozenRow);
                }
            }
        }

        private void CorrectRowFrozenStates(DataGridViewRow[] dataGridViewRows, int rowIndexInserted)
        {
            bool nextVisibleRowPresent = false, previousRowFrozen = true, nextRowFrozen = false, currentRowFrozen;

            // Check if there is a visible row before the insertion point, and if it's frozen
            int rowIndexTmp = Rows.GetPreviousRow(rowIndexInserted, DataGridViewElementStates.Visible);
            if (rowIndexTmp != -1)
            {
                previousRowFrozen = (Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Frozen) == DataGridViewElementStates.Frozen;
            }

            // Check if there is a visible row at or after the insertion point, and if it's frozen
            rowIndexTmp = Rows.GetNextRow(rowIndexInserted - 1, DataGridViewElementStates.Visible);
            if (rowIndexTmp != -1)
            {
                nextVisibleRowPresent = true;
                nextRowFrozen = (Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Frozen) == DataGridViewElementStates.Frozen;
            }

            for (int arrayIndex = 0; arrayIndex < dataGridViewRows.Length; arrayIndex++)
            {
                currentRowFrozen = ((DataGridViewRow)dataGridViewRows[arrayIndex]).Frozen;
                if (!previousRowFrozen && currentRowFrozen)
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAddFrozenRow);
                }
                previousRowFrozen = currentRowFrozen;
                if (arrayIndex == dataGridViewRows.Length - 1 &&
                    !currentRowFrozen &&
                    nextVisibleRowPresent &&
                    nextRowFrozen)
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAddNonFrozenRow);
                }
            }
        }

        private void CorrectRowFrozenStates(DataGridViewRow dataGridViewRow, int rowIndex, bool frozenStateChanging)
        {
            Debug.Assert(dataGridViewRow != null);
            int rowIndexTmp;
            if (((Rows.GetRowState(rowIndex) & DataGridViewElementStates.Frozen) != 0 && !frozenStateChanging) ||
                ((Rows.GetRowState(rowIndex) & DataGridViewElementStates.Frozen) == 0 && frozenStateChanging))
            {
                // make sure the previous visible rows are frozen as well
                rowIndexTmp = Rows.GetPreviousRow(rowIndex,
                    DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                if (rowIndexTmp == -1)
                {
                    int dataGridViewRowFirst = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                    if (dataGridViewRowFirst != rowIndex)
                    {
                        rowIndexTmp = dataGridViewRowFirst;
                    }
                }
                while (rowIndexTmp != -1 && rowIndexTmp < rowIndex)
                {
                    Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Frozen, true);
                    rowIndexTmp = Rows.GetNextRow(rowIndexTmp,
                        DataGridViewElementStates.Visible,
                        DataGridViewElementStates.Frozen);
                }
            }
            else
            {
                // make sure the next visible rows are not frozen
                rowIndexTmp = Rows.GetNextRow(rowIndex,
                    DataGridViewElementStates.Visible,
                    DataGridViewElementStates.Frozen);
                if (rowIndexTmp == -1)
                {
                    int dataGridViewRowLast = rowIndex;
                    do
                    {
                        rowIndexTmp = Rows.GetNextRow(dataGridViewRowLast,
                            DataGridViewElementStates.Visible);
                        if (rowIndexTmp != -1)
                        {
                            dataGridViewRowLast = rowIndexTmp;
                        }
                    }
                    while (rowIndexTmp != -1);
                    if (dataGridViewRowLast != rowIndex)
                    {
                        rowIndexTmp = dataGridViewRowLast;
                    }
                }
                while (rowIndexTmp != -1 && rowIndexTmp > rowIndex)
                {
                    Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Frozen, false);
                    rowIndexTmp = Rows.GetPreviousRow(rowIndexTmp,
                        DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                }
            }
        }

        private void CorrectRowIndexesAfterDeletion(int rowIndexDeleted)
        {
            Debug.Assert(rowIndexDeleted >= 0);
            int rowsCount = Rows.Count;
            for (int rowIndex = rowIndexDeleted; rowIndex < rowsCount; rowIndex++)
            {
                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                if (dataGridViewRow.Index >= 0)
                {
                    dataGridViewRow.Index = dataGridViewRow.Index - 1;
                    Debug.Assert(dataGridViewRow.Index == rowIndex);
                }
            }

            // Fix 'new' row index if existant
            if (newRowIndex == rowIndexDeleted)
            {
                newRowIndex = -1; // No more 'new' row.
            }
            else if (newRowIndex != -1)
            {
                newRowIndex--;
            }
        }

        private void CorrectRowIndexesAfterInsertion(int rowIndexInserted, int insertionCount)
        {
            Debug.Assert(rowIndexInserted >= 0);
            Debug.Assert(insertionCount > 0);
            int rowsCount = Rows.Count;
            for (int rowIndex = rowIndexInserted + insertionCount; rowIndex < rowsCount; rowIndex++)
            {
                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                if (dataGridViewRow.Index >= 0)
                {
                    dataGridViewRow.Index = dataGridViewRow.Index + insertionCount;
                    Debug.Assert(dataGridViewRow.Index == rowIndex);
                }
            }

            // Lastly update the 'new' row index if needed.
            if (newRowIndex != -1)
            {
                newRowIndex += insertionCount;
            }
        }

        protected override AccessibleObject CreateAccessibilityInstance()
        {
            return new DataGridViewAccessibleObject(this);
        }

        protected override ControlCollection CreateControlsInstance()
        {
            return new DataGridViewControlCollection(this);
        }

        /// <summary>
        ///  Constructs the new instance of the Columns collection objects. Subclasses
        ///  should not call base.CreateColumnsInstance.
        /// </summary>
        [EditorBrowsable(EditorBrowsableState.Advanced)]
        protected virtual DataGridViewColumnCollection CreateColumnsInstance()
        {
            return new DataGridViewColumnCollection(this);
        }

        /// <summary>
        ///  Constructs the new instance of the Rows collection objects. Subclasses
        ///  should not call base.CreateRowsInstance.
        /// </summary>
        [EditorBrowsable(EditorBrowsableState.Advanced)]
        protected virtual DataGridViewRowCollection CreateRowsInstance()
        {
            return new DataGridViewRowCollection(this);
        }

        private RECT[] CreateScrollableRegion(Rectangle scroll)
        {
            if (cachedScrollableRegion != null)
            {
                return cachedScrollableRegion;
            }

            using (Region region = new Region(scroll))
            {
                IntPtr handle = IntPtr.Zero;
                using (Graphics graphics = CreateGraphicsInternal())
                {
                    handle = region.GetHrgn(graphics);
                }
                if (handle != IntPtr.Zero)
                {
                    cachedScrollableRegion = UnsafeNativeMethods.GetRectsFromRegion(handle);

                    region.ReleaseHrgn(handle);
                }
            }
            return cachedScrollableRegion;
        }

        private void DiscardNewRow()
        {
            Debug.Assert(Rows.Count > 1);
            Debug.Assert(newRowIndex != -1);

            DataGridViewRowCancelEventArgs dgvrce = new DataGridViewRowCancelEventArgs(Rows[newRowIndex]);
            OnUserDeletingRow(dgvrce);
            if (dgvrce.Cancel)
            {
                return;
            }

            // Delete the 'new' row
            Debug.Assert(newRowIndex == Rows.Count - 1);
            DataGridViewRow dataGridViewRow = Rows[newRowIndex];
            Rows.RemoveAtInternal(newRowIndex, false /*force*/);
            Debug.Assert(dataGridViewRow.Index == -1);
            DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs(dataGridViewRow);
            OnUserDeletedRow(dgvre);
            // CorrectRowIndexesAfterDeletion resets this.newRowIndex to -1.
            Debug.Assert(newRowIndex == -1);

            if (AllowUserToAddRowsInternal)
            {
                newRowIndex = Rows.Count - 1;
                Debug.Assert((Rows.GetRowState(newRowIndex) & DataGridViewElementStates.Visible) != 0);
                Debug.Assert(ptCurrentCell.Y == newRowIndex);

                OnDefaultValuesNeeded(new DataGridViewRowEventArgs(Rows[newRowIndex]));
                InvalidateRowPrivate(newRowIndex);
            }
        }

        private void DiscardZonesInScrollingArea(ref Rectangle rectScrollingArea,
                                                 int emptyBackgroundWidth,
                                                 int emptyBackgroundHeight,
                                                 int frozenVisibleRowsHeight,
                                                 bool discardFrozenColumns,
                                                 bool discardFrozenRows)
        {
            // Discard empty background
            rectScrollingArea.Width -= emptyBackgroundWidth;
            rectScrollingArea.Height -= emptyBackgroundHeight;
            if (RightToLeftInternal)
            {
                rectScrollingArea.X += emptyBackgroundWidth;
            }

            if (discardFrozenColumns)
            {
                // Discard frozen columns
                int frozenVisibleColumnsWidth = Columns.GetColumnsWidth(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                if (!RightToLeftInternal)
                {
                    rectScrollingArea.X += frozenVisibleColumnsWidth;
                }
                rectScrollingArea.Width -= frozenVisibleColumnsWidth;
            }
            if (discardFrozenRows)
            {
                // Discard frozen rows
                rectScrollingArea.Y += frozenVisibleRowsHeight;
                rectScrollingArea.Height -= frozenVisibleRowsHeight;
            }
        }

        public int DisplayedColumnCount(bool includePartialColumns)
        {
            int cxMax = layout.Data.Width, cx = 0;
            int completeColumns = 0, partialColumns = 0;
            DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
            while (dataGridViewColumn != null && cx < cxMax)
            {
                partialColumns++;
                cx += dataGridViewColumn.Thickness;
                if (cx <= cxMax)
                {
                    completeColumns++;
                    dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                        DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen,
                        DataGridViewElementStates.None);
                }
            }

            if (cx < cxMax && displayedBandsInfo.FirstDisplayedScrollingCol >= 0)
            {
                if (negOffset > 0)
                {
                    cx -= negOffset;
                    completeColumns--;
                }
                dataGridViewColumn = (DataGridViewColumn)Columns[displayedBandsInfo.FirstDisplayedScrollingCol];
                Debug.Assert(dataGridViewColumn.Visible && !dataGridViewColumn.Frozen);

                while (dataGridViewColumn != null && cx < cxMax)
                {
                    partialColumns++;
                    cx += dataGridViewColumn.Thickness;
                    if (cx <= cxMax)
                    {
                        completeColumns++;
                        dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                            DataGridViewElementStates.Visible,
                            DataGridViewElementStates.None);
                    }
                }
            }

            return includePartialColumns ? partialColumns : completeColumns;
        }

        public int DisplayedRowCount(bool includePartialRow)
        {
            return includePartialRow ? (displayedBandsInfo.NumDisplayedFrozenRows + displayedBandsInfo.NumDisplayedScrollingRows) :
                (displayedBandsInfo.NumTotallyDisplayedFrozenRows + displayedBandsInfo.NumTotallyDisplayedScrollingRows);
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inDispose] = true;
                try
                {
                    // Remove any Columns contained in this control
                    for (int i = 0; i < Columns.Count; i++)
                    {
                        Columns[i].Dispose();
                    }

                    Columns.Clear();

                    UnwireScrollBarsEvents();
                    if (vertScrollBar != null)
                    {
                        vertScrollBar.Dispose();
                        vertScrollBar = null;
                    }
                    if (horizScrollBar != null)
                    {
                        horizScrollBar.Dispose();
                        horizScrollBar = null;
                    }

                    if (pens != null)
                    {
                        int nPenEntries = pens.Count;
                        if (nPenEntries > 0)
                        {
                            foreach (Pen pen in pens.Values)
                            {
                                pen.Dispose();
                            }
                            pens.Clear();
                        }
                        pens = null;
                    }

                    if (brushes != null)
                    {
                        int nBrushEntries = brushes.Count;
                        if (nBrushEntries > 0)
                        {
                            foreach (SolidBrush brush in brushes.Values)
                            {
                                brush.Dispose();
                            }
                            brushes.Clear();
                        }
                        brushes = null;
                    }

                    if (placeholderStringFormat != null)
                    {
                        placeholderStringFormat.Dispose();
                        placeholderStringFormat = null;
                    }

                    if (latestEditingControl != null)
                    {
                        latestEditingControl.Dispose();
                        latestEditingControl = null;
                    }
                    if (editingControl != null)
                    {
                        editingControl.Dispose();
                        editingControl = null;
                    }
                    if (editingPanel != null)
                    {
                        editingPanel.Dispose();
                        editingPanel = null;
                    }
                    if (gridPen != null)
                    {
                        gridPen.Dispose();
                        gridPen = null;
                    }
                    Debug.Assert(noSelectionChangeCount == 0);

                    if (dataConnection != null)
                    {
                        dataConnection.Dispose();
                    }

                    // DGV should dispose the tool tip control before disposing itself.
                    toolTipControl.Dispose();
                }
                finally
                {
                    dataGridViewOper[DATAGRIDVIEWOPER_inDispose] = false;
                }
            }
            base.Dispose(disposing);
        }

        private void DrawColHeaderShadow(Graphics g, int mouseX)
        {
            Rectangle r = CalcColRelocationFeedbackRect(mouseX);
            DrawShadowRect(r);
            if (dataGridViewState2[DATAGRIDVIEWSTATE2_showColumnRelocationInsertion])
            {
                Rectangle rectInsertionBar = new Rectangle(0, layout.ColumnHeaders.Top, DATAGRIDVIEW_insertionBarWidth, layout.ColumnHeaders.Height);
                // this.trackColumnEdge is the column preceeding the insertion point
                if (trackColumnEdge == -1)
                {
                    // Insert as first column
                    rectInsertionBar.X = GetColumnXFromIndex(Columns.GetFirstColumn(DataGridViewElementStates.Visible).Index);
                    if (RightToLeftInternal)
                    {
                        rectInsertionBar.X -= DATAGRIDVIEW_insertionBarWidth;
                    }
                }
                else
                {
                    int offsetFromCenter = 0;
                    if (Columns.GetNextColumn(Columns[trackColumnEdge], DataGridViewElementStates.Visible, DataGridViewElementStates.None) == null)
                    {
                        if (!RightToLeftInternal)
                        {
                            offsetFromCenter = DATAGRIDVIEW_insertionBarWidth;
                        }
                    }
                    else
                    {
                        if (RightToLeftInternal)
                        {
                            offsetFromCenter = DATAGRIDVIEW_insertionBarWidth / 2 - 1;
                        }
                        else
                        {
                            offsetFromCenter = DATAGRIDVIEW_insertionBarWidth / 2 + 1;
                        }
                    }
                    if (RightToLeftInternal)
                    {
                        rectInsertionBar.X = Math.Max(layout.ColumnHeaders.X,
                                                      GetColumnXFromIndex(trackColumnEdge) - Columns[trackColumnEdge].Width - offsetFromCenter);
                    }
                    else
                    {
                        rectInsertionBar.X = Math.Min(GetColumnXFromIndex(trackColumnEdge) + Columns[trackColumnEdge].Width - offsetFromCenter,
                                                      layout.ColumnHeaders.Right - DATAGRIDVIEW_insertionBarWidth);
                    }
                }
                if (ApplyVisualStylesToHeaderCells)
                {
                    g.FillRectangle(GetCachedBrush(SystemColors.HotTrack), rectInsertionBar);
                }
                else
                {
                    ControlPaint.FillReversibleRectangle(RectangleToScreen(rectInsertionBar), Color.White);
                }
            }
        }

        /// <summary>
        ///  Draws an XOR region to give UI feedback for Column Resizing.
        ///  This looks just like the Splitter control's UI when resizing.
        /// </summary>
        private void DrawColSplitBar(int mouseX)
        {
            Rectangle r = CalcColResizeFeedbackRect(mouseX);
            DrawSplitBar(r);
        }

        /// <summary>
        ///  Draws an XOR region to give UI feedback for Row Resizing.
        ///  This looks just like the Splitter control's UI when resizing.
        /// </summary>
        private void DrawRowSplitBar(int mouseY)
        {
            Rectangle r = CalcRowResizeFeedbackRect(mouseY);
            DrawSplitBar(r);
        }

        private void DrawShadowRect(Rectangle r)
        {
            const byte DATAGRIDVIEW_shadowEdgeThickness = 3;

            IntPtr parentHandle = Handle;
            IntPtr dc = UnsafeNativeMethods.GetDCEx(new HandleRef(this, parentHandle), NativeMethods.NullHandleRef, NativeMethods.DCX_CACHE | NativeMethods.DCX_LOCKWINDOWUPDATE);
            IntPtr halftone = ControlPaint.CreateHalftoneHBRUSH();
            IntPtr saveBrush = Gdi32.SelectObject(dc, halftone);

            SafeNativeMethods.PatBlt(new HandleRef(this, dc), r.X, r.Y, r.Width, DATAGRIDVIEW_shadowEdgeThickness, NativeMethods.PATINVERT);
            SafeNativeMethods.PatBlt(new HandleRef(this, dc), r.X, r.Y + r.Height - DATAGRIDVIEW_shadowEdgeThickness, r.Width, DATAGRIDVIEW_shadowEdgeThickness, NativeMethods.PATINVERT);
            SafeNativeMethods.PatBlt(new HandleRef(this, dc), r.X, r.Y + DATAGRIDVIEW_shadowEdgeThickness, DATAGRIDVIEW_shadowEdgeThickness, r.Height - 2 * DATAGRIDVIEW_shadowEdgeThickness, NativeMethods.PATINVERT);
            SafeNativeMethods.PatBlt(new HandleRef(this, dc), r.X + r.Width - DATAGRIDVIEW_shadowEdgeThickness, r.Y + DATAGRIDVIEW_shadowEdgeThickness, DATAGRIDVIEW_shadowEdgeThickness, r.Height - 2 * DATAGRIDVIEW_shadowEdgeThickness, NativeMethods.PATINVERT);

            Gdi32.SelectObject(dc, saveBrush);
            Gdi32.DeleteObject(halftone);
            User32.ReleaseDC(new HandleRef(this, parentHandle), dc);
        }

        /// <summary>
        ///  Draws an XOR region to give UI feedback for Column/Row Resizing.
        ///  This looks just like the Splitter control's UI when resizing.
        /// </summary>
        private void DrawSplitBar(Rectangle r)
        {
            IntPtr parentHandle = Handle;
            IntPtr dc = UnsafeNativeMethods.GetDCEx(new HandleRef(this, parentHandle), NativeMethods.NullHandleRef, NativeMethods.DCX_CACHE | NativeMethods.DCX_LOCKWINDOWUPDATE);
            IntPtr halftone = ControlPaint.CreateHalftoneHBRUSH();
            IntPtr saveBrush = Gdi32.SelectObject(dc, halftone);
            SafeNativeMethods.PatBlt(new HandleRef(this, dc), r.X, r.Y, r.Width, r.Height, NativeMethods.PATINVERT);
            Gdi32.SelectObject(dc, saveBrush);
            Gdi32.DeleteObject(halftone);
            User32.ReleaseDC(new HandleRef(this, parentHandle), dc);
        }

        private void EditingControls_CommonMouseEventHandler(object sender, MouseEventArgs e, DataGridViewMouseEvent dgvme)
        {
            Debug.Assert(ptCurrentCell.X != -1);
            int adjustedX = editingPanel.Location.X + e.X;
            int adjustedY = editingPanel.Location.Y + e.Y;
            if (sender == editingControl)
            {
                adjustedX += editingControl.Location.X;
                adjustedY += editingControl.Location.Y;
            }

            if (dgvme == DataGridViewMouseEvent.MouseDown && e.Clicks == 1)
            {
                // Reset the flag that records single-click exposed as double-click.
                dataGridViewOper[DATAGRIDVIEWOPER_lastEditCtrlClickDoubled] = false;
            }

            MouseEventArgs me = new MouseEventArgs(e.Button,
                                                   e.Clicks,
                                                   adjustedX,
                                                   adjustedY,
                                                   e.Delta);

            HitTestInfo hti = HitTest(me.X, me.Y);
            int mouseX = me.X - hti.ColumnX;
            if (RightToLeftInternal)
            {
                mouseX += ((hti.col == -1) ? RowHeadersWidth : Columns[hti.col].Thickness);
            }
            DataGridViewCellMouseEventArgs dgvcme = new DataGridViewCellMouseEventArgs(hti.col, hti.row, mouseX, me.Y - hti.RowY, me);

            try
            {
                dataGridViewState2[DATAGRIDVIEWSTATE2_messageFromEditingCtrls] = true;
                // Check to see if this is a CellMouseDoubleClick situation
                if ((dgvme == DataGridViewMouseEvent.MouseDown ||
                     dgvme == DataGridViewMouseEvent.Click ||
                     dgvme == DataGridViewMouseEvent.MouseClick) &&
                    (DateTime.Now.Ticks - lastMouseClickInfo.timeStamp) / 10000 <= SystemInformation.DoubleClickTime &&
                    e.Button == lastMouseClickInfo.button &&
                    e.Clicks == 1 &&
                    dgvcme.ColumnIndex == lastMouseClickInfo.col &&
                    dgvcme.RowIndex == lastMouseClickInfo.row)
                {
                    Size hotDoubleClickZone = SystemInformation.DoubleClickSize;
                    if (Math.Abs(dgvcme.X - lastMouseClickInfo.x) <= hotDoubleClickZone.Width / 2 &&
                        Math.Abs(dgvcme.Y - lastMouseClickInfo.y) <= hotDoubleClickZone.Height / 2)
                    {
                        me = new MouseEventArgs(e.Button,
                                                2,
                                                adjustedX,
                                                adjustedY,
                                                e.Delta);
                        dgvcme = new DataGridViewCellMouseEventArgs(dgvcme.ColumnIndex, dgvcme.RowIndex, dgvcme.X, dgvcme.Y, me);
                        switch (dgvme)
                        {
                            case DataGridViewMouseEvent.MouseDown:
                                {
                                    OnMouseDown(me);
                                    if (dgvcme.ColumnIndex < Columns.Count &&
                                        dgvcme.RowIndex < Rows.Count)
                                    {
                                        OnCellMouseDown(dgvcme);
                                    }
                                    break;
                                }
                            case DataGridViewMouseEvent.Click:
                                {
                                    OnDoubleClick(me);
                                    if (e.Button == MouseButtons.Left &&
                                        dgvcme.ColumnIndex < Columns.Count &&
                                        dgvcme.RowIndex < Rows.Count)
                                    {
                                        OnCellDoubleClick(new DataGridViewCellEventArgs(dgvcme.ColumnIndex, dgvcme.RowIndex));
                                    }
                                    break;
                                }
                            case DataGridViewMouseEvent.MouseClick:
                                {
                                    // Set the flag that prevents the triple-click to be exposed as a double-click
                                    dataGridViewOper[DATAGRIDVIEWOPER_lastEditCtrlClickDoubled] = true;

                                    OnMouseDoubleClick(me);
                                    if (dgvcme.ColumnIndex < Columns.Count && dgvcme.RowIndex < Rows.Count)
                                    {
                                        OnCellMouseDoubleClick(dgvcme);
                                    }
                                    break;
                                }
                        }
                        return;
                    }
                }

                if (dataGridViewOper[DATAGRIDVIEWOPER_lastEditCtrlClickDoubled])
                {
                    // Make sure that the triple-click is exposed as a single-click and not a double-click.
                    if (e.Clicks == 2)
                    {
                        me = new MouseEventArgs(e.Button,
                                                1,
                                                adjustedX,
                                                adjustedY,
                                                e.Delta);
                        dgvcme = new DataGridViewCellMouseEventArgs(hti.col, hti.row, mouseX, me.Y - hti.RowY, me);
                    }
                    switch (dgvme)
                    {
                        case DataGridViewMouseEvent.DoubleClick:
                            dgvme = DataGridViewMouseEvent.Click;
                            break;
                        case DataGridViewMouseEvent.MouseDoubleClick:
                            dgvme = DataGridViewMouseEvent.MouseClick;
                            break;
                    }
                }

                switch (dgvme)
                {
                    case DataGridViewMouseEvent.Click:
                        OnClick(me);
                        if (e.Button == MouseButtons.Left &&
                            dgvcme.ColumnIndex < Columns.Count &&
                            dgvcme.RowIndex < Rows.Count)
                        {
                            OnCellClick(new DataGridViewCellEventArgs(dgvcme.ColumnIndex, dgvcme.RowIndex));
                        }
                        break;
                    case DataGridViewMouseEvent.DoubleClick:
                        OnDoubleClick(me);
                        if (e.Button == MouseButtons.Left &&
                            dgvcme.ColumnIndex < Columns.Count &&
                            dgvcme.RowIndex < Rows.Count)
                        {
                            OnCellDoubleClick(new DataGridViewCellEventArgs(dgvcme.ColumnIndex, dgvcme.RowIndex));
                        }
                        break;
                    case DataGridViewMouseEvent.MouseClick:
                        OnMouseClick(me);
                        if (dgvcme.ColumnIndex < Columns.Count &&
                            dgvcme.RowIndex < Rows.Count)
                        {
                            OnCellMouseClick(dgvcme);
                        }
                        break;
                    case DataGridViewMouseEvent.MouseDoubleClick:
                        OnMouseDoubleClick(me);
                        if (dgvcme.ColumnIndex < Columns.Count &&
                            dgvcme.RowIndex < Rows.Count)
                        {
                            OnCellMouseDoubleClick(dgvcme);
                        }
                        break;
                    case DataGridViewMouseEvent.MouseDown:
                        OnMouseDown(me);
                        if (dgvcme.ColumnIndex < Columns.Count &&
                            dgvcme.RowIndex < Rows.Count)
                        {
                            OnCellMouseDown(dgvcme);
                        }
                        break;
                    case DataGridViewMouseEvent.MouseUp:
                        if (dataGridViewState2[DATAGRIDVIEWSTATE2_nextMouseUpIsDouble])
                        {
                            MouseEventArgs meTmp = new MouseEventArgs(e.Button,
                                                                      2,
                                                                      adjustedX,
                                                                      adjustedY,
                                                                      e.Delta);
                            dgvcme = new DataGridViewCellMouseEventArgs(dgvcme.ColumnIndex, dgvcme.RowIndex, dgvcme.X, dgvcme.Y, meTmp);
                        }
                        OnCellMouseUp(dgvcme);
                        OnMouseUp(me);
                        break;
                    case DataGridViewMouseEvent.MouseMove:
                        OnCellMouseMove(dgvcme);
                        break;
                }
            }
            finally
            {
                dataGridViewState2[DATAGRIDVIEWSTATE2_messageFromEditingCtrls] = false;
            }
        }

        private void EditingControls_Click(object sender, EventArgs e)
        {
            Debug.Assert(sender == editingControl || sender == editingPanel);
            Debug.Assert(ptCurrentCell.X != -1);
            if (e is MouseEventArgs me)
            {
                EditingControls_CommonMouseEventHandler(sender, me, DataGridViewMouseEvent.Click);
            }
        }

        private void EditingControls_DoubleClick(object sender, EventArgs e)
        {
            Debug.Assert(sender == editingControl || sender == editingPanel);
            Debug.Assert(ptCurrentCell.X != -1);
            if (e is MouseEventArgs me)
            {
                EditingControls_CommonMouseEventHandler(sender, me, DataGridViewMouseEvent.DoubleClick);
            }
        }

        private void EditingControls_MouseClick(object sender, MouseEventArgs e)
        {
            Debug.Assert(sender == editingControl || sender == editingPanel);
            EditingControls_CommonMouseEventHandler(sender, e, DataGridViewMouseEvent.MouseClick);
        }

        private void EditingControls_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            Debug.Assert(sender == editingControl || sender == editingPanel);
            EditingControls_CommonMouseEventHandler(sender, e, DataGridViewMouseEvent.MouseDoubleClick);
        }

        private void EditingControls_MouseDown(object sender, MouseEventArgs e)
        {
            Debug.Assert(sender == editingControl || sender == editingPanel);
            EditingControls_CommonMouseEventHandler(sender, e, DataGridViewMouseEvent.MouseDown);
        }

        private void EditingControls_MouseEnter(object sender, EventArgs e)
        {
            Debug.Assert(sender == editingControl || sender == editingPanel);
            if (sender == editingPanel)
            {
                Debug.Assert(editingControl != null);
                Debug.Assert(!dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet]);
                dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet] = true;
                oldCursor = Cursor;
                CursorInternal = ((IDataGridViewEditingControl)editingControl).EditingPanelCursor;
            }

            if (dataGridViewState2[DATAGRIDVIEWSTATE2_mouseEnterExpected])
            {
                OnMouseEnter(EventArgs.Empty);
            }
            UpdateMouseEnteredCell(null /*HitTestInfo*/, null /*MouseEventArgs*/);
        }

        private void EditingControls_MouseLeave(object sender, EventArgs e)
        {
            Debug.Assert(sender == editingControl || sender == editingPanel);
            if (sender == editingPanel)
            {
                Debug.Assert(editingControl != null);
                if (dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet])
                {
                    dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet] = false;
                    CursorInternal = oldCursor;
                }
            }

            UpdateMouseEnteredCell(null /*HitTestInfo*/, null /*MouseEventArgs*/);

            Point ptMouse = PointToClient(Control.MousePosition);
            if (!ClientRectangle.Contains(ptMouse))
            {
                OnMouseLeave(EventArgs.Empty);
            }
        }

        private void EditingControls_MouseMove(object sender, MouseEventArgs e)
        {
            Debug.Assert(sender == editingControl || sender == editingPanel);
            EditingControls_CommonMouseEventHandler(sender, e, DataGridViewMouseEvent.MouseMove);
        }

        private void EditingControls_MouseUp(object sender, MouseEventArgs e)
        {
            Debug.Assert(sender == editingControl || sender == editingPanel);
            EditingControls_CommonMouseEventHandler(sender, e, DataGridViewMouseEvent.MouseUp);
        }

        private void EndColumnHeadersResize(MouseEventArgs e)
        {
            try
            {
                if (currentRowSplitBar != -1)
                {
                    Invalidate(CalcRowResizeFeedbackRect(currentRowSplitBar), true);
                    lastRowSplitBar = currentRowSplitBar = -1;
                }

                int y = Math.Min(e.Y + mouseBarOffset, layout.Data.Bottom - 1);
                int delta = y - layout.ColumnHeaders.Y - ColumnHeadersHeight + 1;
                if (trackRowAnchor != y && delta != 0)
                {
                    ColumnHeadersHeight += delta;
                }
            }
            finally
            {
                RealeaseMouse();
            }
        }

        private void EndColumnRelocation(MouseEventArgs e, HitTestInfo hti)
        {
            try
            {
                if (lastHeaderShadow != -1)
                {
                    dataGridViewState2[DATAGRIDVIEWSTATE2_showColumnRelocationInsertion] = false;
                    trackColumnEdge = -1;
                    lastHeaderShadow = -1;
                    Invalidate(Rectangle.Union(layout.TopLeftHeader, layout.ColumnHeaders));
                }

                if (ColumnRelocationTarget(e, hti, out int previousColumnIndex))
                {
                    if (previousColumnIndex == -1)
                    {
                        Columns[trackColumn].DisplayIndex = 0;
                    }
                    else if (Columns[trackColumn].DisplayIndex > Columns[previousColumnIndex].DisplayIndex)
                    {
                        Columns[trackColumn].DisplayIndex = Columns[previousColumnIndex].DisplayIndex + 1;
                    }
                    else
                    {
                        Columns[trackColumn].DisplayIndex = Columns[previousColumnIndex].DisplayIndex;
                    }
                }
            }
            finally
            {
                RealeaseMouse();
            }
        }

        private void EndColumnResize(MouseEventArgs e)
        {
            try
            {
                EndColumnResize(e.X);
            }
            finally
            {
                RealeaseMouse();
            }
        }

        private void EndColumnResize(int x)
        {
            int newX, delta;
            if (RightToLeftInternal)
            {
                newX = Math.Max(x + mouseBarOffset, layout.Data.X);
                delta = GetColumnXFromIndex(trackColumn) - Columns[trackColumn].Thickness - newX + 1;
            }
            else
            {
                newX = Math.Min(x + mouseBarOffset, layout.Data.Right - 1);
                delta = newX - (GetColumnXFromIndex(trackColumn) + Columns[trackColumn].Thickness) + 1;
            }

            if (trackColAnchor != newX && delta != 0)
            {
                int proposed = Columns[trackColumn].Thickness + delta;
                Debug.Assert(proposed >= Columns[trackColumn].MinimumThickness);
                Debug.Assert(proposed <= DataGridViewBand.MaxBandThickness);
                Columns[trackColumn].Thickness = proposed;
            }
        }

        public bool EndEdit()
        {
            return EndEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit);
        }

        public bool EndEdit(DataGridViewDataErrorContexts context)
        {
            if (EditMode == DataGridViewEditMode.EditOnEnter)
            {
                return CommitEdit(context);
            }
            else
            {
                return EndEdit(context,
                    DataGridViewValidateCellInternal.Never /*validateCell*/,
                    false /*fireCellLeave*/,
                    false /*fireCellEnter*/,
                    false /*fireRowLeave*/,
                    false /*fireRowEnter*/,
                    false /*fireLeave*/,
                    true /*keepFocus*/,
                    true /*resetCurrentCell*/,
                    true /*resetAnchorCell*/);
            }
        }

        private bool EndEdit(DataGridViewDataErrorContexts context,
            DataGridViewValidateCellInternal validateCell,
            bool fireCellLeave,
            bool fireCellEnter,
            bool fireRowLeave,
            bool fireRowEnter,
            bool fireLeave,
            bool keepFocus,
            bool resetCurrentCell,
            bool resetAnchorCell)
        {
            if (ptCurrentCell.X == -1)
            {
                return true;
            }

            dataGridViewOper[DATAGRIDVIEWOPER_inEndEdit] = true;
            try
            {
                int curRowIndex = ptCurrentCell.Y;
                int curColIndex = ptCurrentCell.X;
                DataGridViewCell dataGridViewCurrentCell = CurrentCellInternal;
                DataGridViewDataErrorEventArgs dgvdee = CommitEdit(ref dataGridViewCurrentCell, context, validateCell,
                    fireCellLeave, fireCellEnter, fireRowLeave, fireRowEnter, fireLeave);
                if (dgvdee != null)
                {
                    if (dgvdee.ThrowException)
                    {
                        throw dgvdee.Exception;
                    }
                    if (dgvdee.Cancel)
                    {
                        return false;
                    }
                    dgvdee = CancelEditPrivate(/*ref dataGridViewCurrentCell, context*/);    // restore old value
                    if (null != dgvdee)
                    {
                        if (dgvdee.ThrowException)
                        {
                            throw dgvdee.Exception;
                        }
                        if (dgvdee.Cancel)
                        {
                            return false;
                        }
                    }
                }

                if (!IsCurrentCellInEditMode)
                {
                    return true;
                }

                if (curRowIndex != ptCurrentCell.Y || curColIndex != ptCurrentCell.X)
                {
                    return true;
                }

                if (editingControl != null)
                {
                    UnwireEditingControlEvents();
                    dataGridViewState2[DATAGRIDVIEWSTATE2_mouseOverRemovedEditingCtrl] = MouseOverEditingControl;
                    dataGridViewState2[DATAGRIDVIEWSTATE2_mouseOverRemovedEditingPanel] = MouseOverEditingPanel;
                    dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlChanging] = true;
                    try
                    {
                        dataGridViewCurrentCell.DetachEditingControl();
                    }
                    finally
                    {
                        dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlChanging] = false;
                    }

                    ImeMode editingControlImeMode = editingControl.CachedImeMode; // If in restricted mode, ImeMode will be Disable.
                    latestEditingControl = editingControl;
                    Debug.Assert(editingPanel == null || editingPanel.Controls.Count == 0);
                    editingControl = null;
                    InvalidateCellPrivate(ptCurrentCell.X, ptCurrentCell.Y);

                    if (EditMode == DataGridViewEditMode.EditOnEnter)
                    {
                        if (resetCurrentCell)
                        {
                            bool success = SetCurrentCellAddressCore(-1, -1, resetAnchorCell, false, false);
                            Debug.Assert(success);
                        }
                    }
                    if (keepFocus)
                    {
                        // Debug.Assert(this.CanFocus || this.Focused); // Invalid assertion
                        Focus();
                    }
                    ImeMode = editingControlImeMode;
                }
                else
                {
                    Debug.Assert(dataGridViewState1[DATAGRIDVIEWSTATE1_currentCellInEditMode]);
                    dataGridViewState1[DATAGRIDVIEWSTATE1_currentCellInEditMode] = false;
                    InvalidateCellPrivate(ptCurrentCell.X, ptCurrentCell.Y);
                }

                if (!IsInnerCellOutOfBounds(curColIndex, curRowIndex))
                {
                    DataGridViewCellEventArgs dgvce = new DataGridViewCellEventArgs(curColIndex, curRowIndex);
                    OnCellEndEdit(dgvce);
                }

                return true;
            }
            finally
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inEndEdit] = false;
            }
        }

        private void EndRowHeadersResize(MouseEventArgs e)
        {
            try
            {
                if (currentColSplitBar != -1)
                {
                    Invalidate(CalcColResizeFeedbackRect(currentColSplitBar), true);
                    lastColSplitBar = currentColSplitBar = -1;
                }

                int x, delta;
                if (RightToLeftInternal)
                {
                    x = Math.Max(e.X + mouseBarOffset, layout.Data.Left - 1);
                    delta = layout.RowHeaders.Right - RowHeadersWidth - x - 1;
                }
                else
                {
                    x = Math.Min(e.X + mouseBarOffset, layout.Data.Right - 1);
                    delta = x - layout.RowHeaders.X - RowHeadersWidth + 1;
                }
                if (trackColAnchor != x && delta != 0)
                {
                    RowHeadersWidth += delta;
                }
            }
            finally
            {
                RealeaseMouse();
            }
        }

        private void EndRowResize(MouseEventArgs e)
        {
            try
            {
                if (currentRowSplitBar != -1)
                {
                    Invalidate(CalcRowResizeFeedbackRect(currentRowSplitBar), true);
                    lastRowSplitBar = currentRowSplitBar = -1;
                }

                DataGridViewRow dataGridViewRow = Rows.SharedRow(trackRow);
                dataGridViewRow.GetHeightInfo(trackRow, out int height, out int minimumHeight);
                int y = Math.Min(e.Y + mouseBarOffset, layout.Data.Bottom - 1);
                int delta = y - (GetRowYFromIndex(trackRow) + height) + 1;
                if (trackRowAnchor != y && delta != 0)
                {
                    int proposedHeight = height + delta;
                    proposedHeight = Math.Max(proposedHeight, minimumHeight);
                    if (!OnRowHeightInfoPushed(trackRow, proposedHeight, minimumHeight))
                    {
                        if (dataGridViewRow.Index == -1)
                        {
                            dataGridViewRow = Rows[trackRow];  // Unsharing row
                        }
                        Debug.Assert(autoSizeRowsMode == DataGridViewAutoSizeRowsMode.None);
                        dataGridViewRow.ThicknessInternal = proposedHeight;
                    }
                }
            }
            finally
            {
                RealeaseMouse();
            }
        }

        private void ExitBulkLayout(bool invalidInAdjustFillingColumns)
        {
            if (inBulkLayoutCount > 0)
            {
                inBulkLayoutCount--;
                if (inBulkLayoutCount == 0)
                {
                    PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, invalidInAdjustFillingColumns, false /*repositionEditingControl*/);
                }
            }
        }

        private void ExitBulkPaint(int columnIndex, int rowIndex)
        {
            if (inBulkPaintCount > 0)
            {
                inBulkPaintCount--;
                if (inBulkPaintCount == 0)
                {
                    if (columnIndex >= 0)
                    {
                        InvalidateColumnInternal(columnIndex);
                    }
                    else if (rowIndex >= 0)
                    {
                        InvalidateRowPrivate(rowIndex);
                    }
                    else
                    {
                        Invalidate();
                    }
                }
            }
        }

        private void FirstVisibleScrollingRowTempted(int rowIndex)
        {
            Debug.Assert(rowIndex >= 0);
            Debug.Assert(rowIndex < Rows.Count);
            Debug.Assert((Rows.GetRowState(rowIndex) & DataGridViewElementStates.Visible) != 0);
            Debug.Assert((Rows.GetRowState(rowIndex) & DataGridViewElementStates.Frozen) == 0);

            int displayHeight = layout.Data.Height;
            if (displayHeight <= 0)
            {
                return;
            }

            int totalVisibleFrozenHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
            if (totalVisibleFrozenHeight < displayHeight)
            {
                displayedBandsInfo.FirstDisplayedScrollingRow = rowIndex;
            }
        }

        private void FlushDisplayedChanged()
        {
            if (displayedBandsInfo.Dirty && Visible)
            {
                // Handle the rows

                if (!RowHeadersVisible && Columns.GetColumnCount(DataGridViewElementStates.Visible) == 0)
                {
                    // All rows are hidden
                    UpdateRowsDisplayedState(false /*displayed*/);
                }
                else
                {
                    Rectangle rectScreen = Screen.FromControl(this).WorkingArea;
                    int maxDisplayedRows = (int)(rectScreen.Height / DataGridViewBand.MinBandThickness);

                    // Make sure all displayed scrolling rows have the Displayed state.
                    int rowIndexTmp = displayedBandsInfo.FirstDisplayedScrollingRow;
                    if (rowIndexTmp != -1)
                    {
                        int numDisplayedScrollingRows = displayedBandsInfo.NumDisplayedScrollingRows;
                        Debug.Assert(numDisplayedScrollingRows > 0);
                        while (numDisplayedScrollingRows > 0)
                        {
                            Debug.Assert(rowIndexTmp != -1);
                            if ((Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) == 0)
                            {
                                Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Displayed, true);
                            }
                            rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible);
                            numDisplayedScrollingRows--;
                        }

                        int rowIndexTmp2 = rowIndexTmp;

                        // Make sure all scrolling rows before FirstDisplayedScrollingRow have their Displayed state set to false
                        Debug.Assert(displayedBandsInfo.FirstDisplayedScrollingRow != -1);
                        rowIndexTmp = Rows.GetPreviousRow(displayedBandsInfo.FirstDisplayedScrollingRow, DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                        while (rowIndexTmp != -1 && (Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) != 0)
                        {
                            Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Displayed, false);
                            rowIndexTmp = Rows.GetPreviousRow(rowIndexTmp, DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                        }

                        // Make sure all rows below last displayed scrolling row have Displayed state set to false (next loop)
                        rowIndexTmp = rowIndexTmp2;
                    }
                    else
                    {
                        // No displayed scrolling rows. Make sure all non-frozen rows have their Displayed state set to false (next loop)
                        rowIndexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                    }
                    while (rowIndexTmp != -1 && (Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) != 0)
                    {
                        Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Displayed, false);
                        rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible);
                    }

                    // Make sure all displayed frozen rows have their Displayed state set to true
                    int numDisplayedFrozenRows = displayedBandsInfo.NumDisplayedFrozenRows;
                    rowIndexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                    while (numDisplayedFrozenRows > 0)
                    {
                        Debug.Assert(rowIndexTmp != -1);
                        if ((Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) == 0)
                        {
                            Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Displayed, true);
                        }
                        rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                        numDisplayedFrozenRows--;
                    }

                    // Make sure all non-displayed frozen rows have their Displayed state set to false
                    while (rowIndexTmp != -1 && (Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) != 0)
                    {
                        Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Displayed, false);
                        rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                    }

                    // Treat the cases where the old displayed rows are completely disjoint from the new displayed rows
                    int lastDisplayedFrozenRowIndex = -1;
                    int lastDisplayedScrollingRowIndex = -1;

                    if (displayedBandsInfo.NumDisplayedFrozenRows > 0)
                    {
                        int firstDisplayedFrozenRowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                        Debug.Assert(firstDisplayedFrozenRowIndex != -1);
                        if (displayedBandsInfo.NumDisplayedFrozenRows > 1)
                        {
                            lastDisplayedFrozenRowIndex = Rows.GetNextRow(firstDisplayedFrozenRowIndex, DataGridViewElementStates.Visible, displayedBandsInfo.NumDisplayedFrozenRows - 2 /*skipRows*/);
                        }
                        else
                        {
                            lastDisplayedFrozenRowIndex = firstDisplayedFrozenRowIndex;
                        }
                    }

                    if (displayedBandsInfo.FirstDisplayedScrollingRow != -1)
                    {
                        if (displayedBandsInfo.NumDisplayedScrollingRows > 1)
                        {
                            lastDisplayedScrollingRowIndex = Rows.GetNextRow(displayedBandsInfo.FirstDisplayedScrollingRow, DataGridViewElementStates.Visible, displayedBandsInfo.NumDisplayedScrollingRows - 2 /*skipRows*/);
                        }
                        else
                        {
                            lastDisplayedScrollingRowIndex = displayedBandsInfo.FirstDisplayedScrollingRow;
                        }
                    }

                    rowIndexTmp = displayedBandsInfo.OldFirstDisplayedScrollingRow;
                    while (rowIndexTmp != -1 &&
                        rowIndexTmp < displayedBandsInfo.FirstDisplayedScrollingRow &&
                        !RowNeedsDisplayedState(rowIndexTmp, lastDisplayedFrozenRowIndex, lastDisplayedScrollingRowIndex))
                    {
                        if ((Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) == 0)
                        {
                            break;
                        }
                        else
                        {
                            Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Displayed, false);
                            rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible);
                        }
                    }

                    rowIndexTmp = displayedBandsInfo.OldFirstDisplayedScrollingRow;
                    if (rowIndexTmp != -1 &&
                        rowIndexTmp < Rows.Count &&
                        (displayedBandsInfo.FirstDisplayedScrollingRow == -1 || displayedBandsInfo.FirstDisplayedScrollingRow < rowIndexTmp) &&
                        !RowNeedsDisplayedState(rowIndexTmp, lastDisplayedFrozenRowIndex, lastDisplayedScrollingRowIndex))
                    {
                        while (rowIndexTmp != -1)
                        {
                            if ((Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) == 0)
                            {
                                break;
                            }
                            else
                            {
                                Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Displayed, false);
                                rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible);
                            }
                        }
                    }

                    if (displayedBandsInfo.RowInsertionOccurred)
                    {
                        // Adjust the scrolling rows that were pushed down by the rows insertion
                        rowIndexTmp = displayedBandsInfo.OldFirstDisplayedScrollingRow;
                        if (rowIndexTmp != -1)
                        {
                            rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible, displayedBandsInfo.OldNumDisplayedScrollingRows - 1);
                            if (rowIndexTmp == -1)
                            {
                                rowIndexTmp = Rows.GetLastRow(DataGridViewElementStates.Visible);
                            }
                            int rowCount = 0;
                            while (rowIndexTmp != -1 &&
                                   rowCount <= maxDisplayedRows &&
                                   !RowNeedsDisplayedState(rowIndexTmp, lastDisplayedFrozenRowIndex, lastDisplayedScrollingRowIndex))
                            {
                                if ((Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) != 0)
                                {
                                    Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Displayed, false);
                                }
                                rowIndexTmp = Rows.GetPreviousRow(rowIndexTmp, DataGridViewElementStates.Visible);
                                rowCount++;
                            }
                        }

                        // Adjust the frozen rows that were pushed down by the rows insertion
                        rowIndexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                        if (rowIndexTmp != -1)
                        {
                            rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen, displayedBandsInfo.OldNumDisplayedFrozenRows - 1);
                            if (rowIndexTmp == -1)
                            {
                                rowIndexTmp = Rows.GetLastRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                            }
                            int rowCount = 0;
                            while (rowIndexTmp != -1 &&
                                   rowCount <= maxDisplayedRows &&
                                   !RowNeedsDisplayedState(rowIndexTmp, lastDisplayedFrozenRowIndex, lastDisplayedScrollingRowIndex))
                            {
                                if ((Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) != 0)
                                {
                                    Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Displayed, false);
                                }
                                rowIndexTmp = Rows.GetPreviousRow(rowIndexTmp, DataGridViewElementStates.Visible);
                                rowCount++;
                            }
                        }
                    }

#if DEBUG
                    for (rowIndexTmp = 0; rowIndexTmp < Rows.Count; rowIndexTmp++)
                    {
                        DataGridViewElementStates rowStateDbg = Rows.GetRowState(rowIndexTmp);
                        bool rowNeedsDisplayedState = RowNeedsDisplayedState(rowIndexTmp, lastDisplayedFrozenRowIndex, lastDisplayedScrollingRowIndex);
                        if (((rowStateDbg & DataGridViewElementStates.Displayed) != 0) != rowNeedsDisplayedState)
                        {
                            Debug.Fail("Unexpected Displayed state for row");
                        }
                    }
#endif
                }

                // Handle the columns

                if (!ColumnHeadersVisible && Rows.GetRowCount(DataGridViewElementStates.Visible) == 0)
                {
                    // All columns are hidden
                    UpdateColumnsDisplayedState(false /*displayed*/);
                }
                else
                {
                    // Make sure all displayed scrolling columns have the Displayed state.
                    DataGridViewColumn dataGridViewColumnTmp;
                    int columnIndexTmp = displayedBandsInfo.FirstDisplayedScrollingCol;
                    if (columnIndexTmp != -1)
                    {
                        int numDisplayedScrollingCols = displayedBandsInfo.NumDisplayedScrollingCols;
                        Debug.Assert(numDisplayedScrollingCols > 0);
                        dataGridViewColumnTmp = Columns[columnIndexTmp];
                        while (numDisplayedScrollingCols > 0)
                        {
                            Debug.Assert(dataGridViewColumnTmp != null);
                            if (!dataGridViewColumnTmp.Displayed)
                            {
                                dataGridViewColumnTmp.Displayed = true;
                            }
                            dataGridViewColumnTmp = Columns.GetNextColumn(dataGridViewColumnTmp, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                            numDisplayedScrollingCols--;
                        }

                        DataGridViewColumn dataGridViewColumnTmp2 = dataGridViewColumnTmp;

                        // Make sure all scrolling columns before FirstDisplayedScrollingCol have their Displayed state set to false
                        Debug.Assert(displayedBandsInfo.FirstDisplayedScrollingCol != -1);
                        dataGridViewColumnTmp = Columns.GetPreviousColumn(Columns[displayedBandsInfo.FirstDisplayedScrollingCol], DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                        while (dataGridViewColumnTmp != null && dataGridViewColumnTmp.Displayed)
                        {
                            dataGridViewColumnTmp.Displayed = false;
                            dataGridViewColumnTmp = Columns.GetPreviousColumn(dataGridViewColumnTmp, DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                        }

                        // Make sure all columns after last displayed scrolling column have Displayed state set to false (next loop)
                        dataGridViewColumnTmp = dataGridViewColumnTmp2;
                    }
                    else
                    {
                        // No displayed scrolling columns. Make sure all non-frozen columns have their Displayed state set to false (next loop)
                        dataGridViewColumnTmp = Columns.GetFirstColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                    }
                    while (dataGridViewColumnTmp != null && dataGridViewColumnTmp.Displayed)
                    {
                        dataGridViewColumnTmp.Displayed = false;
                        dataGridViewColumnTmp = Columns.GetNextColumn(dataGridViewColumnTmp, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                    }

                    // Make sure all displayed frozen columns have their Displayed state set to true
                    int numDisplayedFrozenCols = displayedBandsInfo.NumDisplayedFrozenCols;
                    dataGridViewColumnTmp = Columns.GetFirstColumn(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                    while (numDisplayedFrozenCols > 0)
                    {
                        Debug.Assert(dataGridViewColumnTmp != null);
                        if (!dataGridViewColumnTmp.Displayed)
                        {
                            dataGridViewColumnTmp.Displayed = true;
                        }
                        dataGridViewColumnTmp = Columns.GetNextColumn(dataGridViewColumnTmp, DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen, DataGridViewElementStates.None);
                        numDisplayedFrozenCols--;
                    }

                    // Make sure all non-displayed frozen columns have their Displayed state set to false
                    while (dataGridViewColumnTmp != null && dataGridViewColumnTmp.Displayed)
                    {
                        dataGridViewColumnTmp.Displayed = false;
                        dataGridViewColumnTmp = Columns.GetNextColumn(dataGridViewColumnTmp, DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen, DataGridViewElementStates.None);
                    }

                    // Treat the cases where the old displayed columns are completely disjoint from the new displayed columns

                    columnIndexTmp = displayedBandsInfo.OldFirstDisplayedScrollingCol;
                    while (columnIndexTmp != -1 &&
                        columnIndexTmp < Columns.Count &&
                        displayedBandsInfo.FirstDisplayedScrollingCol != -1 &&
                        columnIndexTmp != displayedBandsInfo.FirstDisplayedScrollingCol &&
                        Columns.DisplayInOrder(columnIndexTmp, displayedBandsInfo.FirstDisplayedScrollingCol) &&
                        !ColumnNeedsDisplayedState(Columns[columnIndexTmp]))
                    {
                        dataGridViewColumnTmp = Columns[columnIndexTmp];
                        if (!dataGridViewColumnTmp.Displayed)
                        {
                            break;
                        }
                        else
                        {
                            dataGridViewColumnTmp.Displayed = false;
                            dataGridViewColumnTmp = Columns.GetNextColumn(dataGridViewColumnTmp, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                            columnIndexTmp = (dataGridViewColumnTmp == null) ? -1 : dataGridViewColumnTmp.Index;
                        }
                    }

                    columnIndexTmp = displayedBandsInfo.OldFirstDisplayedScrollingCol;
                    if (columnIndexTmp != -1 &&
                        columnIndexTmp < Columns.Count &&
                        (displayedBandsInfo.FirstDisplayedScrollingCol == -1 || (displayedBandsInfo.FirstDisplayedScrollingCol != columnIndexTmp && Columns.DisplayInOrder(displayedBandsInfo.FirstDisplayedScrollingCol, columnIndexTmp))) &&
                        !ColumnNeedsDisplayedState(Columns[columnIndexTmp]))
                    {
                        dataGridViewColumnTmp = Columns[columnIndexTmp];
                        while (dataGridViewColumnTmp != null)
                        {
                            if (!dataGridViewColumnTmp.Displayed)
                            {
                                break;
                            }
                            else
                            {
                                dataGridViewColumnTmp.Displayed = false;
                                dataGridViewColumnTmp = Columns.GetNextColumn(dataGridViewColumnTmp, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                            }
                        }
                    }

                    if (displayedBandsInfo.ColumnInsertionOccurred)
                    {
                        dataGridViewColumnTmp = Columns[Columns.Count - 1];
                        while (dataGridViewColumnTmp != null && !ColumnNeedsDisplayedState(dataGridViewColumnTmp))
                        {
                            if (dataGridViewColumnTmp.Displayed)
                            {
                                dataGridViewColumnTmp.Displayed = false;
                            }
                            dataGridViewColumnTmp = Columns.GetPreviousColumn(dataGridViewColumnTmp, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                        }
                    }

#if DEBUG
                    for (columnIndexTmp = 0; columnIndexTmp < Columns.Count; columnIndexTmp++)
                    {
                        DataGridViewElementStates colStateDbg = Columns[columnIndexTmp].State;
                        bool columnNeedsDisplayedState = ColumnNeedsDisplayedState(Columns[columnIndexTmp]);
                        if (((colStateDbg & DataGridViewElementStates.Displayed) != 0) != columnNeedsDisplayedState)
                        {
                            Debug.Fail("Unexpected Displayed state for column");
                        }
                    }
#endif
                }

                displayedBandsInfo.Dirty = false;
            }
        }

        private void FlushDisplayIndexChanged(bool raiseEvent)
        {
            foreach (DataGridViewColumn dataGridViewColumn in Columns)
            {
                if (dataGridViewColumn.DisplayIndexHasChanged)
                {
                    dataGridViewColumn.DisplayIndexHasChanged = false;
                    if (raiseEvent)
                    {
                        OnColumnDisplayIndexChanged(dataGridViewColumn);
                    }
                }
            }
        }

        private void FlushSelectionChanged()
        {
            if (dataGridViewState2[DATAGRIDVIEWSTATE2_raiseSelectionChanged])
            {
                OnSelectionChanged(EventArgs.Empty);
            }
        }

        protected override AccessibleObject GetAccessibilityObjectById(int objectId)
        {
            // decrement the objectId because in our implementation of AccessibilityClient notitification objectId's are 1 - based.
            // 0 == NativeMethods.CHILDID_SELF corresponds to the AccessibleObject itself
            return AccessibilityObject.GetChild(objectId - 1);
        }

        internal SolidBrush GetCachedBrush(Color color)
        {
            SolidBrush brush = (SolidBrush)brushes[color];
            if (brush == null)
            {
                brush = new SolidBrush(color);
                brushes.Add(color, brush);
            }
            return brush;
        }

#if DGV_GDI
        internal WindowsSolidBrush GetCachedWindowsBrush(Color color)
        {
            WindowsSolidBrush brush = (WindowsSolidBrush)this.brushes[color];
            if (brush == null)
            {
                brush = new WindowsSolidBrush(color);
                this.brushes.Add(color, brush);
            }
            return brush;
        }
#endif // DGV_GDI

        internal Pen GetCachedPen(Color color)
        {
            Pen pen = (Pen)pens[color];
            if (pen == null)
            {
                pen = new Pen(color);
                pens.Add(color, pen);
            }
            return pen;
        }

#if DGV_GDI
        internal WindowsPen GetCachedWindowsPen(Color color)
        {
            WindowsPen pen = (WindowsPen)this.pens[color];
            if (pen == null)
            {
                pen = new WindowsPen(color);
                this.pens.Add(color, pen);
            }
            return pen;
        }
#endif // DGV_GDI

        internal TypeConverter GetCachedTypeConverter(Type type)
        {
            if (converters.ContainsKey(type))
            {
                return (TypeConverter)converters[type];
            }

            TypeConverter converter = TypeDescriptor.GetConverter(type);
            converters.Add(type, converter);
            return converter;
        }

        internal Rectangle GetCellAdjustedDisplayRectangle(int columnIndex, int rowIndex, bool cutOverflow)
        {
            Rectangle rect = GetCellDisplayRectangle(columnIndex, rowIndex, cutOverflow);
            if (!rect.IsEmpty)
            {
                if (SingleVerticalBorderAdded && columnIndex == FirstDisplayedColumnIndex)
                {
                    if (!RightToLeftInternal)
                    {
                        rect.X--;
                    }
                    rect.Width++;
                }
                if (SingleHorizontalBorderAdded && rowIndex == FirstDisplayedRowIndex)
                {
                    rect.Y--;
                    rect.Height++;
                }
            }
            return rect;
        }

        public int GetCellCount(DataGridViewElementStates includeFilter)
        {
            if ((includeFilter & ~(DataGridViewElementStates.Displayed | DataGridViewElementStates.Frozen | DataGridViewElementStates.Resizable |
                                   DataGridViewElementStates.ReadOnly | DataGridViewElementStates.Selected | DataGridViewElementStates.Visible)) != 0)
            {
                throw new ArgumentException(string.Format(SR.DataGridView_InvalidDataGridViewElementStateCombination, nameof(includeFilter)));
            }

            int cellCount = 0;
            bool displayedRequired, frozenRequired, resizableRequired, readOnlyRequired, visibleRequired;

            if ((includeFilter & DataGridViewElementStates.Selected) == DataGridViewElementStates.Selected)
            {
                if (includeFilter == DataGridViewElementStates.Selected)
                {
                    cellCount = individualSelectedCells.Count;
                    switch (SelectionMode)
                    {
                        case DataGridViewSelectionMode.CellSelect:
                            {
                                // If we change the design and decide that SelectAll() should use band selection,
                                // we need to take the bands into account.
                                return cellCount;
                            }
                        case DataGridViewSelectionMode.FullColumnSelect:
                        case DataGridViewSelectionMode.ColumnHeaderSelect:
                            {
                                return cellCount + selectedBandIndexes.Count * Rows.Count;
                            }
                        case DataGridViewSelectionMode.FullRowSelect:
                        case DataGridViewSelectionMode.RowHeaderSelect:
                            {
                                return cellCount + selectedBandIndexes.Count * Columns.Count;
                            }
                    }
                }

                displayedRequired = (includeFilter & DataGridViewElementStates.Displayed) == DataGridViewElementStates.Displayed;
                frozenRequired = (includeFilter & DataGridViewElementStates.Frozen) == DataGridViewElementStates.Frozen;
                resizableRequired = (includeFilter & DataGridViewElementStates.Resizable) == DataGridViewElementStates.Resizable;
                readOnlyRequired = (includeFilter & DataGridViewElementStates.ReadOnly) == DataGridViewElementStates.ReadOnly;
                visibleRequired = (includeFilter & DataGridViewElementStates.Visible) == DataGridViewElementStates.Visible;

                foreach (DataGridViewCell dataGridViewCell in individualSelectedCells)
                {
                    if (GetCellCount_CellIncluded(dataGridViewCell, dataGridViewCell.RowIndex, displayedRequired, frozenRequired, resizableRequired, readOnlyRequired, visibleRequired))
                    {
                        cellCount++;
                    }
                }
                switch (SelectionMode)
                {
                    case DataGridViewSelectionMode.CellSelect:
                        {
                            // If we change the design and decide that SelectAll() should use band selection,
                            // we need to take the bands into account.
                            return cellCount;
                        }
                    case DataGridViewSelectionMode.FullColumnSelect:
                    case DataGridViewSelectionMode.ColumnHeaderSelect:
                        {
                            for (int rowIndex = 0; rowIndex < Rows.Count; rowIndex++)
                            {
                                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                                foreach (int columnIndex in selectedBandIndexes)
                                {
                                    DataGridViewCell dataGridViewCell = dataGridViewRow.Cells[columnIndex];
                                    if (GetCellCount_CellIncluded(dataGridViewCell, rowIndex, displayedRequired, frozenRequired, resizableRequired, readOnlyRequired, visibleRequired))
                                    {
                                        cellCount++;
                                    }
                                }
                            }
                            return cellCount;
                        }
                    case DataGridViewSelectionMode.FullRowSelect:
                    case DataGridViewSelectionMode.RowHeaderSelect:
                        {
                            foreach (int rowIndex in selectedBandIndexes)
                            {
                                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                                foreach (DataGridViewCell dataGridViewCell in dataGridViewRow.Cells)
                                {
                                    if (GetCellCount_CellIncluded(dataGridViewCell, rowIndex, displayedRequired, frozenRequired, resizableRequired, readOnlyRequired, visibleRequired))
                                    {
                                        cellCount++;
                                    }
                                }
                            }
                            return cellCount;
                        }
                }
            }

            if ((includeFilter == DataGridViewElementStates.ReadOnly && ReadOnly) ||
                includeFilter == DataGridViewElementStates.None)
            {
                return Rows.Count * Columns.Count;
            }

            displayedRequired = (includeFilter & DataGridViewElementStates.Displayed) == DataGridViewElementStates.Displayed;
            frozenRequired = (includeFilter & DataGridViewElementStates.Frozen) == DataGridViewElementStates.Frozen;
            resizableRequired = (includeFilter & DataGridViewElementStates.Resizable) == DataGridViewElementStates.Resizable;
            readOnlyRequired = (includeFilter & DataGridViewElementStates.ReadOnly) == DataGridViewElementStates.ReadOnly;
            visibleRequired = (includeFilter & DataGridViewElementStates.Visible) == DataGridViewElementStates.Visible;

            for (int rowIndex = 0; rowIndex < Rows.Count; rowIndex++)
            {
                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                if (!visibleRequired || (Rows.GetRowState(rowIndex) & DataGridViewElementStates.Visible) != 0)
                {
                    foreach (DataGridViewCell dataGridViewCell in dataGridViewRow.Cells)
                    {
                        if (GetCellCount_CellIncluded(dataGridViewCell, rowIndex, displayedRequired, frozenRequired, resizableRequired, readOnlyRequired, visibleRequired))
                        {
                            cellCount++;
                        }
                    }
                }
            }
            return cellCount;
        }

        private bool GetCellCount_CellIncluded(DataGridViewCell dataGridViewCell,
                                               int rowIndex,
                                               bool displayedRequired,
                                               bool frozenRequired,
                                               bool resizableRequired,
                                               bool readOnlyRequired,
                                               bool visibleRequired)
        {
            Debug.Assert(dataGridViewCell != null);
            Debug.Assert(rowIndex >= 0);
            DataGridViewElementStates rowState = Rows.GetRowState(rowIndex);
            if (displayedRequired)
            {
                bool cellDisplayed = (rowState & DataGridViewElementStates.Displayed) != 0 &&
                                     dataGridViewCell.OwningColumn.Displayed;
                if (!cellDisplayed)
                {
                    return false;
                }
            }
            if (frozenRequired)
            {
                bool cellFrozen = (rowState & DataGridViewElementStates.Frozen) != 0 ||
                                  dataGridViewCell.OwningColumn.Frozen ||
                                  dataGridViewCell.StateIncludes(DataGridViewElementStates.Frozen);
                if (!cellFrozen)
                {
                    return false;
                }
            }
            if (resizableRequired)
            {
                if (!RowIsResizable(rowIndex) && dataGridViewCell.OwningColumn.Resizable != DataGridViewTriState.True)
                {
                    return false;
                }
            }
            if (readOnlyRequired)
            {
                bool cellReadOnly = ReadOnly ||
                                    (rowState & DataGridViewElementStates.ReadOnly) != 0 ||
                                    dataGridViewCell.OwningColumn.ReadOnly ||
                                    dataGridViewCell.StateIncludes(DataGridViewElementStates.ReadOnly);
                if (!cellReadOnly)
                {
                    return false;
                }
            }
            if (visibleRequired)
            {
                bool cellVisible = (rowState & DataGridViewElementStates.Visible) != 0 &&
                                   dataGridViewCell.OwningColumn.Visible;
                if (!cellVisible)
                {
                    return false;
                }
            }
            return true;
        }

        public Rectangle GetCellDisplayRectangle(int columnIndex, int rowIndex, bool cutOverflow)
        {
            Rectangle rowRect;
            Rectangle columnRect;

            if (columnIndex >= 0)
            {
                if (columnIndex >= Columns.Count)
                {
                    throw new ArgumentOutOfRangeException(nameof(columnIndex));
                }
                columnRect = GetColumnDisplayRectanglePrivate(columnIndex, cutOverflow);
            }
            else
            {
                if (columnIndex != -1)
                {
                    throw new ArgumentOutOfRangeException(nameof(columnIndex));
                }
                if (rowIndex >= 0)
                {
                    columnRect = layout.RowHeaders;
                }
                else
                {
                    columnRect = layout.TopLeftHeader;
                }
            }

            if (rowIndex >= 0)
            {
                if (rowIndex >= Rows.Count)
                {
                    throw new ArgumentOutOfRangeException(nameof(rowIndex));
                }
                rowRect = GetRowDisplayRectanglePrivate(rowIndex, cutOverflow);
            }
            else
            {
                if (rowIndex != -1)
                {
                    throw new ArgumentOutOfRangeException(nameof(rowIndex));
                }
                if (columnIndex >= 0)
                {
                    rowRect = layout.ColumnHeaders;
                }
                else
                {
                    rowRect = layout.TopLeftHeader;
                }
            }

            if (!cutOverflow)
            {
                int height = rowRect.Bottom - columnRect.Bottom;
                if (height > 0)
                {
                    columnRect.Height += height;
                }
                int width;
                if (RightToLeftInternal)
                {
                    width = rowRect.X - columnRect.X;
                    if (width > 0)
                    {
                        rowRect.Width += width;
                        rowRect.X -= width;
                    }
                }
                else
                {
                    width = columnRect.Right - rowRect.Right;
                    if (width > 0)
                    {
                        rowRect.Width += width;
                    }
                }
            }

            rowRect.Intersect(columnRect);

            return rowRect;
        }

        internal DataGridViewCell GetCellInternal(int columnIndex, int rowIndex)
        {
            Debug.Assert(columnIndex >= -1 && columnIndex < Columns.Count);
            Debug.Assert(rowIndex >= -1 && rowIndex < Rows.Count);
            if (rowIndex >= 0)
            {
                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                Debug.Assert(dataGridViewRow != null);
                if (columnIndex >= 0)
                {
                    return dataGridViewRow.Cells[columnIndex];
                }
                else
                {
                    return dataGridViewRow.HeaderCell;
                }
            }
            else
            {
                if (columnIndex >= 0)
                {
                    return Columns[columnIndex].HeaderCell;
                }
                else
                {
                    return TopLeftHeaderCell;
                }
            }
        }

        public virtual DataObject GetClipboardContent()
        {
            if (ClipboardCopyMode == DataGridViewClipboardCopyMode.Disable)
            {
                throw new NotSupportedException(SR.DataGridView_DisabledClipboardCopy);
            }

            if (CurrentCellIsEditedAndOnlySelectedCell)
            {
                return null;
            }

            string[] formats = new string[] { DataFormats.Html, DataFormats.Text, DataFormats.UnicodeText, DataFormats.CommaSeparatedValue };
            DataObject dataObject = new DataObject();
            bool includeColumnHeaders = false, includeRowHeaders = false;
            string cellContent = null;
            StringBuilder sbContent = null;
            DataGridViewColumn dataGridViewColumn, prevDataGridViewColumn, nextDataGridViewColumn;

            switch (SelectionMode)
            {
                case DataGridViewSelectionMode.FullRowSelect:
                    if (Rows.GetRowCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Selected) == 0)
                    {
                        return null;
                    }

                    if (clipboardCopyMode == DataGridViewClipboardCopyMode.EnableWithAutoHeaderText)
                    {
                        includeColumnHeaders = (Rows.GetFirstRow(DataGridViewElementStates.Visible, DataGridViewElementStates.Selected) == -1);
                        includeRowHeaders = true;
                    }
                    else
                    {
                        includeColumnHeaders = includeRowHeaders = (clipboardCopyMode == DataGridViewClipboardCopyMode.EnableAlwaysIncludeHeaderText);
                    }

                    includeColumnHeaders &= ColumnHeadersVisible;
                    includeRowHeaders &= RowHeadersVisible;

                    foreach (string format in formats)
                    {
                        /* if (!String.Equals(format, DataFormats.Html, StringComparison.OrdinalIgnoreCase) &&
                            !String.Equals(format, DataFormats.Text, StringComparison.OrdinalIgnoreCase) &&
                            !String.Equals(format, DataFormats.UnicodeText, StringComparison.OrdinalIgnoreCase) &&
                            !String.Equals(format, DataFormats.CommaSeparatedValue, StringComparison.OrdinalIgnoreCase))
                        {
                            continue;
                        }*/

                        if (sbContent == null)
                        {
                            sbContent = new StringBuilder(1024);
                        }
                        else
                        {
                            sbContent.Length = 0;
                        }

                        if (includeColumnHeaders)
                        {
                            if (RightToLeftInternal)
                            {
                                // Cycle through the visible columns in their reverse display order
                                dataGridViewColumn = Columns.GetLastColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                if (dataGridViewColumn != null)
                                {
                                    prevDataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                    cellContent = dataGridViewColumn.HeaderCell.GetClipboardContentInternal(-1,
                                                                                                            true /*firstCell*/,
                                                                                                            !includeRowHeaders && prevDataGridViewColumn == null /*lastCell*/,
                                                                                                            true /*inFirstRow*/,
                                                                                                            false /*inLastRow*/,
                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    while (prevDataGridViewColumn != null)
                                    {
                                        dataGridViewColumn = prevDataGridViewColumn;
                                        prevDataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                        cellContent = dataGridViewColumn.HeaderCell.GetClipboardContentInternal(-1,
                                                                                                                false /*firstCell*/,
                                                                                                                !includeRowHeaders && prevDataGridViewColumn == null /*lastCell*/,
                                                                                                                true /*inFirstRow*/,
                                                                                                                false /*inLastRow*/,
                                                                                                                format) as string;
                                        if (cellContent != null)
                                        {
                                            sbContent.Append(cellContent);
                                        }
                                    }
                                }
                                if (includeRowHeaders)
                                {
                                    cellContent = TopLeftHeaderCell.GetClipboardContentInternal(-1,
                                                                                                     Columns.GetColumnCount(DataGridViewElementStates.Visible) == 0 /*firstCell*/,
                                                                                                     true /*lastCell*/,
                                                                                                     true /*inFirstRow*/,
                                                                                                     false /*inLastRow*/,
                                                                                                     format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }
                            }
                            else
                            {
                                if (includeRowHeaders)
                                {
                                    cellContent = TopLeftHeaderCell.GetClipboardContentInternal(-1,
                                                                                                     true /*firstCell*/,
                                                                                                     Columns.GetColumnCount(DataGridViewElementStates.Visible) == 0 /*lastCell*/,
                                                                                                     true /*inFirstRow*/,
                                                                                                     false /*inLastRow*/,
                                                                                                     format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }
                                // Cycle through the visible columns in their display order
                                dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
                                if (dataGridViewColumn != null)
                                {
                                    nextDataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                    cellContent = dataGridViewColumn.HeaderCell.GetClipboardContentInternal(-1,
                                                                                                            !includeRowHeaders /*firstCell*/,
                                                                                                            nextDataGridViewColumn == null /*lastCell*/,
                                                                                                            true /*inFirstRow*/,
                                                                                                            false /*inLastRow*/,
                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    while (nextDataGridViewColumn != null)
                                    {
                                        dataGridViewColumn = nextDataGridViewColumn;
                                        nextDataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                        cellContent = dataGridViewColumn.HeaderCell.GetClipboardContentInternal(-1,
                                                                                                                false /*firstCell*/,
                                                                                                                nextDataGridViewColumn == null /*lastCell*/,
                                                                                                                true /*inFirstRow*/,
                                                                                                                false /*inLastRow*/,
                                                                                                                format) as string;
                                        if (cellContent != null)
                                        {
                                            sbContent.Append(cellContent);
                                        }
                                    }
                                }
                            }
                        }
                        // Cycle through the visible selected rows.
                        bool firstRowIndex = true;
                        int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Selected);
                        Debug.Assert(rowIndex != -1);
                        int nextRowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible | DataGridViewElementStates.Selected);
                        while (rowIndex != -1)
                        {
                            if (RightToLeftInternal)
                            {
                                dataGridViewColumn = Columns.GetLastColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.None);

                                // Cycle through the visible columns in their reverse display order
                                if (dataGridViewColumn != null)
                                {
                                    prevDataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                    cellContent = Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].GetClipboardContentInternal(rowIndex,
                                                                                                                                            true /*firstCell*/,
                                                                                                                                            !includeRowHeaders && prevDataGridViewColumn == null /*lastCell*/,
                                                                                                                                            !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                                            nextRowIndex == -1 /*inLastRow*/,
                                                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    while (prevDataGridViewColumn != null)
                                    {
                                        dataGridViewColumn = prevDataGridViewColumn;
                                        prevDataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                        cellContent = Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].GetClipboardContentInternal(rowIndex,
                                                                                                                                                false /*firstCell*/,
                                                                                                                                                !includeRowHeaders && prevDataGridViewColumn == null /*lastCell*/,
                                                                                                                                                !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                                                nextRowIndex == -1 /*inLastRow*/,
                                                                                                                                                format) as string;
                                        if (cellContent != null)
                                        {
                                            sbContent.Append(cellContent);
                                        }
                                    }
                                }

                                // Get the row header clipboard content
                                if (includeRowHeaders)
                                {
                                    cellContent = Rows.SharedRow(rowIndex).HeaderCell.GetClipboardContentInternal(rowIndex,
                                                                                                                       Columns.GetColumnCount(DataGridViewElementStates.Visible) == 0 /*firstCell*/,
                                                                                                                       true /*lastCell*/,
                                                                                                                       !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                       nextRowIndex == -1 /*inLastRow*/,
                                                                                                                       format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }
                            }
                            else
                            {
                                dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);

                                // Get the row header clipboard content
                                if (includeRowHeaders)
                                {
                                    cellContent = Rows.SharedRow(rowIndex).HeaderCell.GetClipboardContentInternal(rowIndex,
                                                                                                                       true /*firstCell*/,
                                                                                                                       dataGridViewColumn == null /*lastCell*/,
                                                                                                                       !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                       nextRowIndex == -1 /*inLastRow*/,
                                                                                                                       format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }

                                // Cycle through the visible columns in their display order
                                if (dataGridViewColumn != null)
                                {
                                    nextDataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                    cellContent = Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].GetClipboardContentInternal(rowIndex,
                                                                                                                                            !includeRowHeaders /*firstCell*/,
                                                                                                                                            nextDataGridViewColumn == null /*lastCell*/,
                                                                                                                                            !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                                            nextRowIndex == -1 /*inLastRow*/,
                                                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    while (nextDataGridViewColumn != null)
                                    {
                                        dataGridViewColumn = nextDataGridViewColumn;
                                        nextDataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                        cellContent = Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].GetClipboardContentInternal(rowIndex,
                                                                                                                                                false /*firstCell*/,
                                                                                                                                                nextDataGridViewColumn == null /*lastCell*/,
                                                                                                                                                !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                                                nextRowIndex == -1 /*inLastRow*/,
                                                                                                                                                format) as string;
                                        if (cellContent != null)
                                        {
                                            sbContent.Append(cellContent);
                                        }
                                    }
                                }
                            }
                            rowIndex = nextRowIndex;
                            if (rowIndex != -1)
                            {
                                nextRowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible | DataGridViewElementStates.Selected);
                            }
                            firstRowIndex = false;
                        }
                        if (string.Equals(format, DataFormats.Html, StringComparison.OrdinalIgnoreCase))
                        {
                            GetClipboardContentForHtml(sbContent, out IO.MemoryStream utf8Stream);
                            dataObject.SetData(format, false /*autoConvert*/, utf8Stream);
                        }
                        else
                        {
                            dataObject.SetData(format, false /*autoConvert*/, sbContent.ToString());
                        }
                    }
                    break;

                case DataGridViewSelectionMode.FullColumnSelect:
                    if (Columns.GetColumnCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Selected) == 0)
                    {
                        return null;
                    }

                    if (clipboardCopyMode == DataGridViewClipboardCopyMode.EnableWithAutoHeaderText)
                    {
                        includeColumnHeaders = true;
                        includeRowHeaders = (Columns.GetFirstColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.Selected) == null);
                    }
                    else
                    {
                        includeColumnHeaders = includeRowHeaders = (clipboardCopyMode == DataGridViewClipboardCopyMode.EnableAlwaysIncludeHeaderText);
                    }

                    includeColumnHeaders &= ColumnHeadersVisible;
                    includeRowHeaders &= RowHeadersVisible;

                    int firstVisibleRowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);

                    foreach (string format in formats)
                    {
                        /* if (!String.Equals(format, DataFormats.Html, StringComparison.OrdinalIgnoreCase) &&
                            !String.Equals(format, DataFormats.Text, StringComparison.OrdinalIgnoreCase) &&
                            !String.Equals(format, DataFormats.UnicodeText, StringComparison.OrdinalIgnoreCase) &&
                            !String.Equals(format, DataFormats.CommaSeparatedValue, StringComparison.OrdinalIgnoreCase))
                        {
                            continue;
                        }*/

                        if (sbContent == null)
                        {
                            sbContent = new StringBuilder(1024);
                        }
                        else
                        {
                            sbContent.Length = 0;
                        }

                        if (includeColumnHeaders)
                        {
                            if (RightToLeftInternal)
                            {
                                // Cycle through the visible & selected columns in their display order
                                DataGridViewColumn lastDataGridViewColumn = Columns.GetLastColumn(DataGridViewElementStates.Visible | DataGridViewElementStates.Selected, DataGridViewElementStates.None);
                                dataGridViewColumn = lastDataGridViewColumn;
                                Debug.Assert(dataGridViewColumn != null);
                                if (dataGridViewColumn != null)
                                {
                                    prevDataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn, DataGridViewElementStates.Visible | DataGridViewElementStates.Selected, DataGridViewElementStates.None);
                                    cellContent = dataGridViewColumn.HeaderCell.GetClipboardContentInternal(-1,
                                                                                                            true /*firstCell*/,
                                                                                                            !includeRowHeaders && prevDataGridViewColumn == null /*lastCell*/,
                                                                                                            true /*inFirstRow*/,
                                                                                                            firstVisibleRowIndex == -1 /*inLastRow*/,
                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    while (prevDataGridViewColumn != null)
                                    {
                                        dataGridViewColumn = prevDataGridViewColumn;
                                        prevDataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn, DataGridViewElementStates.Visible | DataGridViewElementStates.Selected, DataGridViewElementStates.None);
                                        cellContent = dataGridViewColumn.HeaderCell.GetClipboardContentInternal(-1,
                                                                                                                false /*firstCell*/,
                                                                                                                !includeRowHeaders && prevDataGridViewColumn == null /*lastCell*/,
                                                                                                                true /*inFirstRow*/,
                                                                                                                firstVisibleRowIndex == -1 /*inLastRow*/,
                                                                                                                format) as string;
                                        if (cellContent != null)
                                        {
                                            sbContent.Append(cellContent);
                                        }
                                    }
                                }
                                if (includeRowHeaders)
                                {
                                    cellContent = TopLeftHeaderCell.GetClipboardContentInternal(-1,
                                                                                                     lastDataGridViewColumn == null /*firstCell*/,
                                                                                                     true /*lastCell*/,
                                                                                                     true /*inFirstRow*/,
                                                                                                     firstVisibleRowIndex == -1 /*inLastRow*/,
                                                                                                     format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }
                            }
                            else
                            {
                                dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible | DataGridViewElementStates.Selected);

                                if (includeRowHeaders)
                                {
                                    cellContent = TopLeftHeaderCell.GetClipboardContentInternal(-1,
                                                                                                     true /*firstCell*/,
                                                                                                     dataGridViewColumn == null /*lastCell*/,
                                                                                                     true /*inFirstRow*/,
                                                                                                     firstVisibleRowIndex == -1 /*inLastRow*/,
                                                                                                     format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }
                                // Cycle through the visible & selected columns in their display order
                                Debug.Assert(dataGridViewColumn != null);
                                if (dataGridViewColumn != null)
                                {
                                    nextDataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible | DataGridViewElementStates.Selected, DataGridViewElementStates.None);
                                    cellContent = dataGridViewColumn.HeaderCell.GetClipboardContentInternal(-1,
                                                                                                            !includeRowHeaders /*firstCell*/,
                                                                                                            nextDataGridViewColumn == null /*lastCell*/,
                                                                                                            true /*inFirstRow*/,
                                                                                                            firstVisibleRowIndex == -1 /*inLastRow*/,
                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    while (nextDataGridViewColumn != null)
                                    {
                                        dataGridViewColumn = nextDataGridViewColumn;
                                        nextDataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible | DataGridViewElementStates.Selected, DataGridViewElementStates.None);
                                        cellContent = dataGridViewColumn.HeaderCell.GetClipboardContentInternal(-1,
                                                                                                                false /*firstCell*/,
                                                                                                                nextDataGridViewColumn == null /*lastCell*/,
                                                                                                                true /*inFirstRow*/,
                                                                                                                firstVisibleRowIndex == -1 /*inLastRow*/,
                                                                                                                format) as string;
                                        if (cellContent != null)
                                        {
                                            sbContent.Append(cellContent);
                                        }
                                    }
                                }
                            }
                        }
                        // Cycle through the visible rows.
                        bool firstRowIndex = true;
                        int rowIndex = firstVisibleRowIndex;
                        int nextRowIndex = -1;
                        if (rowIndex != -1)
                        {
                            nextRowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                        }
                        while (rowIndex != -1)
                        {
                            if (RightToLeftInternal)
                            {
                                DataGridViewColumn lastDataGridViewColumn = Columns.GetLastColumn(DataGridViewElementStates.Visible | DataGridViewElementStates.Selected, DataGridViewElementStates.None);

                                // Cycle through the visible & selected columns in their reverse display order
                                dataGridViewColumn = lastDataGridViewColumn;
                                if (dataGridViewColumn != null)
                                {
                                    prevDataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn, DataGridViewElementStates.Visible | DataGridViewElementStates.Selected, DataGridViewElementStates.None);
                                    cellContent = Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].GetClipboardContentInternal(rowIndex,
                                                                                                                                            true /*firstCell*/,
                                                                                                                                            !includeRowHeaders && prevDataGridViewColumn == null /*lastCell*/,
                                                                                                                                            !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                                            nextRowIndex == -1 /*inLastRow*/,
                                                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    while (prevDataGridViewColumn != null)
                                    {
                                        dataGridViewColumn = prevDataGridViewColumn;
                                        prevDataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn, DataGridViewElementStates.Visible | DataGridViewElementStates.Selected, DataGridViewElementStates.None);
                                        cellContent = Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].GetClipboardContentInternal(rowIndex,
                                                                                                                                                false /*firstCell*/,
                                                                                                                                                !includeRowHeaders && prevDataGridViewColumn == null /*lastCell*/,
                                                                                                                                                !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                                                nextRowIndex == -1 /*inLastRow*/,
                                                                                                                                                format) as string;
                                        if (cellContent != null)
                                        {
                                            sbContent.Append(cellContent);
                                        }
                                    }
                                }

                                // Get the row header clipboard content
                                if (includeRowHeaders)
                                {
                                    cellContent = Rows.SharedRow(rowIndex).HeaderCell.GetClipboardContentInternal(rowIndex,
                                                                                                                       lastDataGridViewColumn == null /*firstCell*/,
                                                                                                                       true /*lastCell*/,
                                                                                                                       !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                       nextRowIndex == -1 /*inLastRow*/,
                                                                                                                       format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }
                            }
                            else
                            {
                                dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible | DataGridViewElementStates.Selected);

                                // Get the row header clipboard content
                                if (includeRowHeaders)
                                {
                                    cellContent = Rows.SharedRow(rowIndex).HeaderCell.GetClipboardContentInternal(rowIndex,
                                                                                                                       true /*firstCell*/,
                                                                                                                       dataGridViewColumn == null /*lastCell*/,
                                                                                                                       !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                       nextRowIndex == -1 /*inLastRow*/,
                                                                                                                       format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }

                                // Cycle through the visible & selected columns in their display order
                                if (dataGridViewColumn != null)
                                {
                                    nextDataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible | DataGridViewElementStates.Selected, DataGridViewElementStates.None);
                                    cellContent = Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].GetClipboardContentInternal(rowIndex,
                                                                                                                                            !includeRowHeaders /*firstCell*/,
                                                                                                                                            nextDataGridViewColumn == null /*lastCell*/,
                                                                                                                                            !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                                            nextRowIndex == -1 /*inLastRow*/,
                                                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    while (nextDataGridViewColumn != null)
                                    {
                                        dataGridViewColumn = nextDataGridViewColumn;
                                        nextDataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible | DataGridViewElementStates.Selected, DataGridViewElementStates.None);
                                        cellContent = Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].GetClipboardContentInternal(rowIndex,
                                                                                                                                                false /*firstCell*/,
                                                                                                                                                nextDataGridViewColumn == null /*lastCell*/,
                                                                                                                                                !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                                                nextRowIndex == -1 /*inLastRow*/,
                                                                                                                                                format) as string;
                                        if (cellContent != null)
                                        {
                                            sbContent.Append(cellContent);
                                        }
                                    }
                                }
                            }
                            rowIndex = nextRowIndex;
                            if (rowIndex != -1)
                            {
                                nextRowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                            }
                            firstRowIndex = false;
                        }
                        if (string.Equals(format, DataFormats.Html, StringComparison.OrdinalIgnoreCase))
                        {
                            GetClipboardContentForHtml(sbContent, out IO.MemoryStream utf8Stream);
                            dataObject.SetData(format, false /*autoConvert*/, utf8Stream);
                        }
                        else
                        {
                            dataObject.SetData(format, false /*autoConvert*/, sbContent.ToString());
                        }
                    }
                    break;

                case DataGridViewSelectionMode.CellSelect:
                case DataGridViewSelectionMode.RowHeaderSelect:
                case DataGridViewSelectionMode.ColumnHeaderSelect:
                    bool selectedVisibleCellExists = false;
                    bool selectedVisibleColumnExists = false;
                    bool selectedVisibleRowExists = false;
                    if (SelectionMode == DataGridViewSelectionMode.RowHeaderSelect)
                    {
                        selectedVisibleRowExists = Rows.GetRowCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Selected) != 0;
                        selectedVisibleCellExists = selectedVisibleRowExists && Columns.GetColumnCount(DataGridViewElementStates.Visible) != 0;
                    }
                    else if (SelectionMode == DataGridViewSelectionMode.ColumnHeaderSelect)
                    {
                        selectedVisibleColumnExists = Columns.GetColumnCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Selected) != 0;
                        selectedVisibleCellExists = selectedVisibleColumnExists && Rows.GetRowCount(DataGridViewElementStates.Visible) != 0;
                    }
                    if (!selectedVisibleCellExists && individualSelectedCells.Count > 0)
                    {
                        foreach (DataGridViewCell dataGridViewCell in individualSelectedCells)
                        {
                            if (dataGridViewCell.Visible)
                            {
                                selectedVisibleCellExists = true;
                                break;
                            }
                        }
                    }
                    if (!selectedVisibleCellExists)
                    {
                        return null;
                    }

                    // There is at least one selected visible cell.
                    if (SelectionMode == DataGridViewSelectionMode.CellSelect)
                    {
                        includeColumnHeaders = includeRowHeaders = (clipboardCopyMode == DataGridViewClipboardCopyMode.EnableAlwaysIncludeHeaderText);
                        includeColumnHeaders &= ColumnHeadersVisible;
                        includeRowHeaders &= RowHeadersVisible;
                    }
                    else
                    {
                        includeColumnHeaders = includeRowHeaders = false;
                        if (ColumnHeadersVisible)
                        {
                            if (clipboardCopyMode == DataGridViewClipboardCopyMode.EnableWithAutoHeaderText)
                            {
                                if (selectedVisibleColumnExists)
                                {
                                    includeColumnHeaders = true;
                                }
                                /* Use this code if column headers should be included when all cells are selected in a visible column.
                                else
                                {
                                    // Include column headers if there is a column where all visible cells are selected
                                    DataGridViewColumn dataGridViewColumn = this.Columns.GetFirstColumn(DataGridViewElementStates.Visible);
                                    Debug.Assert(dataGridViewColumn != null);
                                    while (dataGridViewColumn != null)
                                    {
                                        // Cycle through the visible rows, see if the cell in that column is selected
                                        int rowIndex = this.Rows.GetFirstRow(DataGridViewElementStates.Visible);
                                        while (rowIndex != -1)
                                        {
                                            if (!this.Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].Selected)
                                            {
                                                break;
                                            }
                                            rowIndex = this.Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                                        }
                                        if (rowIndex == -1)
                                        {
                                            // All visible cells in column are selected
                                            includeColumnHeaders = true;
                                            break;
                                        }
                                        dataGridViewColumn = this.Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                    }
                                }
                                */
                            }
                            else
                            {
                                includeColumnHeaders = (clipboardCopyMode == DataGridViewClipboardCopyMode.EnableAlwaysIncludeHeaderText);
                            }
                        }

                        if (RowHeadersVisible)
                        {
                            if (clipboardCopyMode == DataGridViewClipboardCopyMode.EnableWithAutoHeaderText)
                            {
                                if (selectedVisibleRowExists)
                                {
                                    includeRowHeaders = true;
                                }
                                /* Use this code if row headers should be included when all cells are selected in a visible row.
                                else
                                {
                                    // Include row headers if there is a row where all visible cells are selected
                                    int rowIndex = this.Rows.GetFirstRow(DataGridViewElementStates.Visible);
                                    Debug.Assert(rowIndex != -1);
                                    while (rowIndex != -1)
                                    {
                                        // Cycle through the visible columns, see if the cell in that row is selected
                                        DataGridViewColumn dataGridViewColumn = this.Columns.GetFirstColumn(DataGridViewElementStates.Visible);
                                        Debug.Assert(dataGridViewColumn != null);
                                        while (dataGridViewColumn != null)
                                        {
                                            if (!this.Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].Selected)
                                            {
                                                break;
                                            }
                                            dataGridViewColumn = this.Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                        }
                                        if (dataGridViewColumn == null)
                                        {
                                            // All visible cells in row are selected
                                            includeRowHeaders = true;
                                            break;
                                        }
                                        rowIndex = this.Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                                    }
                                }
                                */
                            }
                            else
                            {
                                includeRowHeaders = (clipboardCopyMode == DataGridViewClipboardCopyMode.EnableAlwaysIncludeHeaderText);
                            }
                        }
                    }

                    // Get the four corners of the 'selected table'
                    int lRowIndex = int.MaxValue;
                    int uRowIndex = -1;
                    DataGridViewColumn lColumn = null, uColumn = null;

                    if (SelectionMode == DataGridViewSelectionMode.RowHeaderSelect)
                    {
                        DataGridViewColumn firstVisibleColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
                        DataGridViewColumn lastVisibleColumn = Columns.GetLastColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.None);

                        Debug.Assert(firstVisibleColumn != null);
                        Debug.Assert(lastVisibleColumn != null);
                        foreach (int rowIndex in selectedBandIndexes)
                        {
                            if ((Rows.GetRowState(rowIndex) & DataGridViewElementStates.Visible) != 0)
                            {
                                if (rowIndex < lRowIndex)
                                {
                                    lRowIndex = rowIndex;
                                }
                                if (rowIndex > uRowIndex)
                                {
                                    uRowIndex = rowIndex;
                                }
                                lColumn = firstVisibleColumn;
                                uColumn = lastVisibleColumn;
                            }
                        }
                    }
                    else if (SelectionMode == DataGridViewSelectionMode.ColumnHeaderSelect)
                    {
                        firstVisibleRowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                        int lastVisibleRowIndex = Rows.GetLastRow(DataGridViewElementStates.Visible);
                        Debug.Assert(firstVisibleRowIndex != -1);
                        Debug.Assert(lastVisibleRowIndex != -1);
                        foreach (int columnIndex in selectedBandIndexes)
                        {
                            if (Columns[columnIndex].Visible)
                            {
                                if (lColumn == null || Columns.DisplayInOrder(columnIndex, lColumn.Index))
                                {
                                    lColumn = Columns[columnIndex];
                                }
                                if (uColumn == null || Columns.DisplayInOrder(uColumn.Index, columnIndex))
                                {
                                    uColumn = Columns[columnIndex];
                                }
                                lRowIndex = firstVisibleRowIndex;
                                uRowIndex = lastVisibleRowIndex;
                            }
                        }
                    }

                    // Go through the individually selected cells to potentially stretch the current 'selected table'.
                    foreach (DataGridViewCell dataGridViewCell in individualSelectedCells)
                    {
                        if (dataGridViewCell.Visible)
                        {
                            if (dataGridViewCell.RowIndex < lRowIndex)
                            {
                                lRowIndex = dataGridViewCell.RowIndex;
                            }
                            if (dataGridViewCell.RowIndex > uRowIndex)
                            {
                                uRowIndex = dataGridViewCell.RowIndex;
                            }
                            if (lColumn == null || Columns.DisplayInOrder(dataGridViewCell.ColumnIndex, lColumn.Index))
                            {
                                lColumn = dataGridViewCell.OwningColumn;
                            }
                            if (uColumn == null || Columns.DisplayInOrder(uColumn.Index, dataGridViewCell.ColumnIndex))
                            {
                                uColumn = dataGridViewCell.OwningColumn;
                            }
                        }
                    }

                    Debug.Assert(lRowIndex != -1);
                    Debug.Assert(uRowIndex != -1);
                    Debug.Assert(lColumn != null);
                    Debug.Assert(uColumn != null);
                    Debug.Assert(lColumn.Index == uColumn.Index || Columns.DisplayInOrder(lColumn.Index, uColumn.Index));
                    Debug.Assert(lRowIndex <= uRowIndex);

                    foreach (string format in formats)
                    {
                        /* if (!String.Equals(format, DataFormats.Html, StringComparison.OrdinalIgnoreCase) &&
                            !String.Equals(format, DataFormats.Text, StringComparison.OrdinalIgnoreCase) &&
                            !String.Equals(format, DataFormats.UnicodeText, StringComparison.OrdinalIgnoreCase) &&
                            !String.Equals(format, DataFormats.CommaSeparatedValue, StringComparison.OrdinalIgnoreCase))
                        {
                            continue;
                        }*/

                        if (sbContent == null)
                        {
                            sbContent = new StringBuilder(1024);
                        }
                        else
                        {
                            sbContent.Length = 0;
                        }

                        if (includeColumnHeaders)
                        {
                            if (RightToLeftInternal)
                            {
                                // Cycle through the visible columns from uColumn to lColumn
                                dataGridViewColumn = uColumn;
                                Debug.Assert(dataGridViewColumn != null);
                                while (dataGridViewColumn != null)
                                {
                                    if (dataGridViewColumn != lColumn)
                                    {
                                        prevDataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                        Debug.Assert(prevDataGridViewColumn != null);
                                    }
                                    else
                                    {
                                        prevDataGridViewColumn = null;
                                    }
                                    cellContent = dataGridViewColumn.HeaderCell.GetClipboardContentInternal(-1,
                                                                                                            dataGridViewColumn == uColumn /*firstCell*/,
                                                                                                            !includeRowHeaders && prevDataGridViewColumn == null /*lastCell*/,
                                                                                                            true /*inFirstRow*/,
                                                                                                            false /*inLastRow*/,
                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    dataGridViewColumn = prevDataGridViewColumn;
                                }
                                if (includeRowHeaders)
                                {
                                    cellContent = TopLeftHeaderCell.GetClipboardContentInternal(-1,
                                                                                                     false /*firstCell*/,
                                                                                                     true /*lastCell*/,
                                                                                                     true  /*inFirstRow*/,
                                                                                                     false /*inLastRow*/,
                                                                                                     format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }
                            }
                            else
                            {
                                if (includeRowHeaders)
                                {
                                    cellContent = TopLeftHeaderCell.GetClipboardContentInternal(-1,
                                                                                                     true  /*firstCell*/,
                                                                                                     false /*lastCell*/,
                                                                                                     true  /*inFirstRow*/,
                                                                                                     false /*inLastRow*/,
                                                                                                     format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }
                                // Cycle through the visible columns from lColumn to uColumn
                                dataGridViewColumn = lColumn;
                                Debug.Assert(dataGridViewColumn != null);
                                while (dataGridViewColumn != null)
                                {
                                    if (dataGridViewColumn != uColumn)
                                    {
                                        nextDataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                        Debug.Assert(nextDataGridViewColumn != null);
                                    }
                                    else
                                    {
                                        nextDataGridViewColumn = null;
                                    }
                                    cellContent = dataGridViewColumn.HeaderCell.GetClipboardContentInternal(-1,
                                                                                                            !includeRowHeaders && dataGridViewColumn == lColumn /*firstCell*/,
                                                                                                            nextDataGridViewColumn == null /*lastCell*/,
                                                                                                            true /*inFirstRow*/,
                                                                                                            false /*inLastRow*/,
                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    dataGridViewColumn = nextDataGridViewColumn;
                                }
                            }
                        }
                        // Cycle through the visible rows from lRowIndex to uRowIndex.
                        bool firstRowIndex = true;
                        int rowIndex = lRowIndex;
                        int nextRowIndex = -1;
                        Debug.Assert(rowIndex != -1);
                        while (rowIndex != -1)
                        {
                            if (rowIndex != uRowIndex)
                            {
                                nextRowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                                Debug.Assert(nextRowIndex != -1);
                            }
                            else
                            {
                                nextRowIndex = -1;
                            }

                            if (RightToLeftInternal)
                            {
                                // Cycle through the visible columns from uColumn to lColumn
                                dataGridViewColumn = uColumn;
                                Debug.Assert(dataGridViewColumn != null);
                                while (dataGridViewColumn != null)
                                {
                                    if (dataGridViewColumn != lColumn)
                                    {
                                        prevDataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                        Debug.Assert(prevDataGridViewColumn != null);
                                    }
                                    else
                                    {
                                        prevDataGridViewColumn = null;
                                    }

                                    cellContent = Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].GetClipboardContentInternal(rowIndex,
                                                                                                                                            dataGridViewColumn == uColumn /*firstCell*/,
                                                                                                                                            !includeRowHeaders && prevDataGridViewColumn == null /*lastCell*/,
                                                                                                                                            !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                                            nextRowIndex == -1 /*inLastRow*/,
                                                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    dataGridViewColumn = prevDataGridViewColumn;
                                }

                                if (includeRowHeaders)
                                {
                                    // Get the row header clipboard content
                                    cellContent = Rows.SharedRow(rowIndex).HeaderCell.GetClipboardContentInternal(rowIndex,
                                                                                                                       false /*firstCell*/,
                                                                                                                       true /*lastCell*/,
                                                                                                                       !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                       nextRowIndex == -1 /*inLastRow*/,
                                                                                                                       format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }
                            }
                            else
                            {
                                if (includeRowHeaders)
                                {
                                    // Get the row header clipboard content
                                    cellContent = Rows.SharedRow(rowIndex).HeaderCell.GetClipboardContentInternal(rowIndex,
                                                                                                                       true /*firstCell*/,
                                                                                                                       false /*lastCell*/,
                                                                                                                       !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                       nextRowIndex == -1 /*inLastRow*/,
                                                                                                                       format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                }

                                // Cycle through the visible columns from lColumn to uColumn
                                dataGridViewColumn = lColumn;
                                Debug.Assert(dataGridViewColumn != null);
                                while (dataGridViewColumn != null)
                                {
                                    if (dataGridViewColumn != uColumn)
                                    {
                                        nextDataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                                        Debug.Assert(nextDataGridViewColumn != null);
                                    }
                                    else
                                    {
                                        nextDataGridViewColumn = null;
                                    }

                                    cellContent = Rows.SharedRow(rowIndex).Cells[dataGridViewColumn.Index].GetClipboardContentInternal(rowIndex,
                                                                                                                                            !includeRowHeaders && dataGridViewColumn == lColumn /*firstCell*/,
                                                                                                                                            nextDataGridViewColumn == null /*lastCell*/,
                                                                                                                                            !includeColumnHeaders && firstRowIndex /*inFirstRow*/,
                                                                                                                                            nextRowIndex == -1 /*inLastRow*/,
                                                                                                                                            format) as string;
                                    if (cellContent != null)
                                    {
                                        sbContent.Append(cellContent);
                                    }
                                    dataGridViewColumn = nextDataGridViewColumn;
                                }
                            }
                            rowIndex = nextRowIndex;
                            firstRowIndex = false;
                        }
                        if (string.Equals(format, DataFormats.Html, StringComparison.OrdinalIgnoreCase))
                        {
                            GetClipboardContentForHtml(sbContent, out IO.MemoryStream utf8Stream);
                            dataObject.SetData(format, false /*autoConvert*/, utf8Stream);
                        }
                        else
                        {
                            dataObject.SetData(format, false /*autoConvert*/, sbContent.ToString());
                        }
                    }
                    break;
            }
            return dataObject;
        }

        private static void GetClipboardContentForHtml(StringBuilder sbContent, out IO.MemoryStream utf8Stream)
        {
            byte[] sourceBytes = Encoding.Unicode.GetBytes(sbContent.ToString());
            byte[] destinationBytes = Encoding.Convert(Encoding.Unicode, Encoding.UTF8, sourceBytes);

            int bytecountEndOfFragment = 135 + destinationBytes.Length;
            int bytecountEndOfHtml = bytecountEndOfFragment + 36;
            string prefix = string.Format(CultureInfo.InvariantCulture, DATAGRIDVIEW_htmlPrefix, bytecountEndOfHtml.ToString("00000000", CultureInfo.InvariantCulture), bytecountEndOfFragment.ToString("00000000", CultureInfo.InvariantCulture)) + DATAGRIDVIEW_htmlStartFragment;
            sbContent.Insert(0, prefix);
            sbContent.Append(DATAGRIDVIEW_htmlEndFragment);

            sourceBytes = Encoding.Unicode.GetBytes(sbContent.ToString());
            destinationBytes = Encoding.Convert(Encoding.Unicode, Encoding.UTF8, sourceBytes);

            utf8Stream = new IO.MemoryStream(bytecountEndOfHtml + 1);
            utf8Stream.Write(destinationBytes, 0, bytecountEndOfHtml);
            utf8Stream.WriteByte((byte)0);

#if DEBUG
            Debug.Assert(destinationBytes[97] == '<');
            Debug.Assert(destinationBytes[bytecountEndOfHtml - 1] == '>');
            Debug.Assert(destinationBytes[133] == '<');
            Debug.Assert(destinationBytes[bytecountEndOfFragment] == '<');
#endif
        }

        // Rectangle returned includes the potential column header
        public Rectangle GetColumnDisplayRectangle(int columnIndex, bool cutOverflow)
        {
            if (columnIndex < 0 || columnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(columnIndex));
            }
            return GetColumnDisplayRectanglePrivate(columnIndex, cutOverflow);
        }

        private Rectangle GetColumnDisplayRectanglePrivate(int columnIndex, bool cutOverflow)
        {
            Debug.Assert(columnIndex >= 0 && columnIndex < Columns.Count);

            if (!Columns[columnIndex].Displayed)
            {
                return Rectangle.Empty;
            }

            Rectangle data = layout.Data;
            int cx;
            bool columnFound = false;
            DataGridViewColumn dataGridViewColumn;
            if (RightToLeftInternal)
            {
                cx = data.Right;
            }
            else
            {
                cx = data.X;
            }
            for (dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                dataGridViewColumn != null && !columnFound;
                )
            {
                if ((RightToLeftInternal && cx < data.X) ||
                    (!RightToLeftInternal && cx > data.Right))
                {
                    break;
                }
                if (dataGridViewColumn.Index == columnIndex)
                {
                    columnFound = true;
                }
                else
                {
                    if (RightToLeftInternal)
                    {
                        cx -= dataGridViewColumn.Thickness;
                    }
                    else
                    {
                        cx += dataGridViewColumn.Thickness;
                    }
                    dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                        DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen,
                        DataGridViewElementStates.None);
                }
            }

            if (!columnFound && displayedBandsInfo.FirstDisplayedScrollingCol >= 0)
            {
                for (dataGridViewColumn = Columns[displayedBandsInfo.FirstDisplayedScrollingCol];
                    dataGridViewColumn != null && !columnFound;
                    )
                {
                    if ((RightToLeftInternal && cx < data.X) ||
                        (!RightToLeftInternal && cx > data.Right))
                    {
                        break;
                    }
                    if (dataGridViewColumn.Index == columnIndex)
                    {
                        columnFound = true;
                    }
                    else
                    {
                        if (RightToLeftInternal)
                        {
                            cx -= dataGridViewColumn.Thickness;
                        }
                        else
                        {
                            cx += dataGridViewColumn.Thickness;
                        }
                        if (dataGridViewColumn.Index == displayedBandsInfo.FirstDisplayedScrollingCol)
                        {
                            if (RightToLeftInternal)
                            {
                                cx += negOffset;
                            }
                            else
                            {
                                cx -= negOffset;
                            }
                        }
                        dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                            DataGridViewElementStates.Visible,
                            DataGridViewElementStates.None);
                    }
                }
            }

            if (columnFound)
            {
                Debug.Assert(dataGridViewColumn != null);
                int displayWidth, viewedColumnWidth = dataGridViewColumn.Thickness;
                if (dataGridViewColumn.Index == displayedBandsInfo.FirstDisplayedScrollingCol)
                {
                    viewedColumnWidth -= negOffset;
                }
                if (cutOverflow &&
                    ((!RightToLeftInternal && cx + viewedColumnWidth > data.Right) ||
                     (RightToLeftInternal && cx - viewedColumnWidth < data.X)))
                {
                    if (RightToLeftInternal)
                    {
                        displayWidth = cx - data.X;
                    }
                    else
                    {
                        displayWidth = data.Right - cx;
                    }
                }
                else
                {
                    displayWidth = viewedColumnWidth;
                }

                Rectangle columnRect;
                if (RightToLeftInternal)
                {
                    columnRect = new Rectangle(cx - displayWidth, data.Y, displayWidth, data.Height);
                }
                else
                {
                    columnRect = new Rectangle(cx, data.Y, displayWidth, data.Height);
                }
                if (layout.ColumnHeadersVisible)
                {
                    columnRect.Height += layout.ColumnHeaders.Height;
                    columnRect.Y -= layout.ColumnHeaders.Height;
                }
                return columnRect;
            }

            return Rectangle.Empty;
        }

        // xColumnLeftEdge returns the left edge of the column when RightToLeft is false.
        // xColumnLeftEdge returns the right edge of the column when RightToLeft is true.
        private int GetColumnIndexFromX(int x, out int xColumnLeftEdge)
        {
            Rectangle data = layout.Data;
            Debug.Assert(RightToLeftInternal || (x >= data.X - 1 && x < data.Right), "x must be inside the horizontal bounds of this.layout.Data");
            Debug.Assert(!RightToLeftInternal || (x >= data.X && x <= data.Right), "x must be inside the horizontal bounds of this.layout.Data");

            if (!RightToLeftInternal && x == data.X - 1)
            {
                x++;
            }
            else if (RightToLeftInternal && x == data.Right)
            {
                x--;
            }

            int cx;
            if (RightToLeftInternal)
            {
                cx = data.Right - 1;
            }
            else
            {
                cx = data.X;
            }

            // first try to match x against a frozen column
            DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
            while (dataGridViewColumn != null &&
                   ((!RightToLeftInternal && cx < data.Right) || (RightToLeftInternal && cx >= data.X)))
            {
                if (RightToLeftInternal)
                {
                    cx -= dataGridViewColumn.Thickness;
                }
                else
                {
                    cx += dataGridViewColumn.Thickness;
                }
                if (!RightToLeftInternal && cx > x)
                {
                    xColumnLeftEdge = cx - dataGridViewColumn.Thickness;
                    return dataGridViewColumn.Index;
                }
                else if (RightToLeftInternal && cx < x)
                {
                    xColumnLeftEdge = cx + dataGridViewColumn.Thickness;
                    return dataGridViewColumn.Index;
                }
                dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                    DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen,
                    DataGridViewElementStates.None);
            }

            if (RightToLeftInternal)
            {
                cx += negOffset;
            }
            else
            {
                cx -= negOffset;
            }

            // second try to match x against a scrolling column
            if (displayedBandsInfo.FirstDisplayedScrollingCol >= 0)
            {
                dataGridViewColumn = Columns[displayedBandsInfo.FirstDisplayedScrollingCol];
                while (dataGridViewColumn != null &&
                       ((!RightToLeftInternal && cx < data.Right) || (RightToLeftInternal && cx >= data.X)))
                {
                    Debug.Assert(dataGridViewColumn.Visible && !dataGridViewColumn.Frozen);
                    if (RightToLeftInternal)
                    {
                        cx -= dataGridViewColumn.Thickness;
                    }
                    else
                    {
                        cx += dataGridViewColumn.Thickness;
                    }
                    if (!RightToLeftInternal && cx > x)
                    {
                        xColumnLeftEdge = cx - dataGridViewColumn.Thickness;
                        return dataGridViewColumn.Index;
                    }
                    else if (RightToLeftInternal && cx < x)
                    {
                        xColumnLeftEdge = cx + dataGridViewColumn.Thickness;
                        return dataGridViewColumn.Index;
                    }
                    dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                        DataGridViewElementStates.Visible,
                        DataGridViewElementStates.None);
                }
            }

            xColumnLeftEdge = -1;
            return -1;
        }

        private static int GetColumnScrollRate(int xOffset)
        {
            Debug.Assert(xOffset > 0);
            // Counting 20ms for executing the actual column scrolling
            if (xOffset <= 10)
            {
                return 480;     // Two columns per second
            }
            if (xOffset <= 15)
            {
                return 313;     // Three columns per second
            }
            if (xOffset <= 25)
            {
                return 180;     // Five columns per second
            }
            if (xOffset <= 35)
            {
                return 123;     // Seven columns per second
            }
            return Math.Max(1, 4000 / xOffset);
        }

        /// <summary>
        ///  Returns the coordinate of the left edge of the given column. Note that
        ///  the column does not need to be completely visible on the display area.
        ///  Value returned is not necessarily within layout.Data because of the
        ///  this.negOffset value, or because the column may start to the right of
        ///  data area, or behind the frozen area, or completely on the left of the control.
        ///  The right edge is returned in RightToLeft mode.
        /// </summary>
        internal int GetColumnXFromIndex(int index)
        {
            Debug.Assert(index < Columns.Count);
            Debug.Assert(Columns[index].Visible);

            int x;
            if (RightToLeftInternal)
            {
                x = layout.Data.Right - 1;
            }
            else
            {
                x = layout.Data.X;
            }

            DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
            while (dataGridViewColumn != null)
            {
                if (index == dataGridViewColumn.Index)
                {
                    return x;
                }
                if (RightToLeftInternal)
                {
                    x -= dataGridViewColumn.Thickness;
                }
                else
                {
                    x += dataGridViewColumn.Thickness;
                }
                dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                    DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen,
                    DataGridViewElementStates.None);
            }

            if (RightToLeftInternal)
            {
                x += negOffset;
            }
            else
            {
                x -= negOffset;
            }

            int xFirstVisibleScrollingCol = x;

            if (displayedBandsInfo.FirstDisplayedScrollingCol >= 0)
            {
                dataGridViewColumn = Columns[displayedBandsInfo.FirstDisplayedScrollingCol];
            }
            else
            {
                dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
            }
            Debug.Assert(dataGridViewColumn.Visible && !dataGridViewColumn.Frozen);

            while (dataGridViewColumn != null)
            {
                if (index == dataGridViewColumn.Index)
                {
                    return x;
                }
                if (RightToLeftInternal)
                {
                    x -= dataGridViewColumn.Thickness;
                }
                else
                {
                    x += dataGridViewColumn.Thickness;
                }
                dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                    DataGridViewElementStates.Visible,
                    DataGridViewElementStates.None);
            }

            // The column is completely hidden on the left/right of the dataGridView
            x = xFirstVisibleScrollingCol;
            dataGridViewColumn = Columns[displayedBandsInfo.FirstDisplayedScrollingCol];
            dataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn,
                DataGridViewElementStates.Visible,
                DataGridViewElementStates.Frozen);
            while (dataGridViewColumn != null)
            {
                if (RightToLeftInternal)
                {
                    x += dataGridViewColumn.Thickness;
                }
                else
                {
                    x -= dataGridViewColumn.Thickness;
                }
                if (index == dataGridViewColumn.Index)
                {
                    return x;
                }
                dataGridViewColumn = Columns.GetPreviousColumn(dataGridViewColumn,
                    DataGridViewElementStates.Visible,
                    DataGridViewElementStates.Frozen);
            }

            Debug.Fail("Could not find column in GetColumnXFromIndex");
            return 0;
        }

        private int GetNegOffsetFromHorizontalOffset(int horizontalOffset)
        {
            DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
            while (dataGridViewColumn != null && dataGridViewColumn.Thickness <= horizontalOffset)
            {
                horizontalOffset -= dataGridViewColumn.Thickness;
                dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn, DataGridViewElementStates.Visible, DataGridViewElementStates.None);
            }
            return horizontalOffset;
        }

        private bool GetOutOfBoundCorrectedHitTestInfo(ref HitTestInfo hti, ref int mouseX, ref int mouseY, out int xOffset, out int yOffset)
        {
            xOffset = yOffset = 0;
            Rectangle rectScrollingArea = layout.Data;
            int visibleRowsHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible);
            int frozenVisibleRowsHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
            int fittingTrailingScrollingRowsHeight = ComputeHeightOfFittingTrailingScrollingRows(frozenVisibleRowsHeight);
            int trailingScrollingRowsHeight = ComputeHeightOfTrailingScrollingRows();
            int emptyBackgroundWidth = Math.Max(0, layout.Data.Width - Columns.GetColumnsWidth(DataGridViewElementStates.Visible));
            int emptyBackgroundHeight = Math.Max(0, layout.Data.Height - frozenVisibleRowsHeight - trailingScrollingRowsHeight);

            Debug.Assert(!vertScrollBar.Enabled ||
                         !vertScrollBar.Visible ||
                         vertScrollBar.Maximum == visibleRowsHeight - frozenVisibleRowsHeight);

            //Debug.Assert(!this.vertScrollBar.Enabled ||
            //             !this.vertScrollBar.Visible ||
            //             this.vertScrollBar.Value >= this.verticalOffset);

            if (dataGridViewOper[DATAGRIDVIEWOPER_trackRowSelect])
            {
                if (layout.RowHeadersVisible)
                {
                    // Include row headers
                    rectScrollingArea = Rectangle.Union(rectScrollingArea, layout.RowHeaders);
                }
                // Discard frozen rows
                DiscardZonesInScrollingArea(ref rectScrollingArea, emptyBackgroundWidth, emptyBackgroundHeight, frozenVisibleRowsHeight,
                                            false /*discardFrozenColumns*/, true /*discardFrozenRows*/);

                if (mouseY >= rectScrollingArea.Top && mouseY <= rectScrollingArea.Bottom)
                {
                    // Mouse's Y is in-bound -- correct X value
                    hti = HitTest(RightToLeftInternal ? rectScrollingArea.Right - 1 : rectScrollingArea.Left, mouseY);
                    if (ptAnchorCell.Y != -1 &&
                        (Rows.GetRowState(ptAnchorCell.Y) & DataGridViewElementStates.Frozen) != 0 &&
                        trackRowEdge != -1 &&
                        (Rows.GetRowState(trackRowEdge) & DataGridViewElementStates.Frozen) != 0 &&
                        hti.row >= 0 &&
                        (Rows.GetRowState(hti.row) & DataGridViewElementStates.Frozen) == 0)
                    {
                        // Anchor cell is in frozen row and target cell is in unfrozen row. Make sure no row is scrolled off.
                        Debug.Assert(displayedBandsInfo.FirstDisplayedScrollingRow >= 0);
                        int firstUnfrozenRowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                        int firstColumnIndex;
                        if (hti.col >= 0)
                        {
                            firstColumnIndex = hti.col;
                        }
                        else
                        {
                            DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
                            firstColumnIndex = (dataGridViewColumn == null) ? -1 : dataGridViewColumn.Index;
                        }
                        if (firstColumnIndex >= 0 && firstUnfrozenRowIndex >= 0)
                        {
                            if (!ScrollIntoView(firstColumnIndex, firstUnfrozenRowIndex, false /*forCurrentCellChange*/))
                            {
                                return false;
                            }
                            hti = HitTest(RightToLeftInternal ? rectScrollingArea.Right : rectScrollingArea.Left, mouseY);
                        }
                    }
                    return true;
                }

                // Mouse's Y is outside of scrolling bands
                if (mouseY < rectScrollingArea.Top)
                {
                    if (ptAnchorCell.Y != -1 &&
                        ((Rows.GetRowState(ptAnchorCell.Y) & DataGridViewElementStates.Frozen) == 0 ||
                         (trackRowEdge != -1 && (Rows.GetRowState(trackRowEdge) & DataGridViewElementStates.Frozen) == 0)) &&
                        verticalOffset != 0)
                    {
                        // Up scrolling is required because the anchor's row is unfrozen
                        Debug.Assert(displayedBandsInfo.FirstDisplayedScrollingRow >= 0);
                        yOffset = mouseY - rectScrollingArea.Top;   // yOffset strictly negative
                        if (RightToLeftInternal)
                        {
                            mouseX = rectScrollingArea.Right - 1;
                        }
                        else
                        {
                            mouseX = rectScrollingArea.Left + 1;
                        }
                    }
                    else
                    {
                        hti = HitTest(RightToLeftInternal ? rectScrollingArea.Right : rectScrollingArea.Left, mouseY);
                    }
                }
                else
                {
                    Debug.Assert(mouseY > rectScrollingArea.Bottom);
                    if (displayedBandsInfo.FirstDisplayedScrollingRow >= 0)
                    {
                        if (verticalOffset + Rows.SharedRow(displayedBandsInfo.FirstDisplayedScrollingRow).GetHeight(displayedBandsInfo.FirstDisplayedScrollingRow) <=
                            visibleRowsHeight - frozenVisibleRowsHeight - fittingTrailingScrollingRowsHeight)
                        {
                            // Down scrolling is required
                            yOffset = mouseY - rectScrollingArea.Bottom;   // yOffset strictly positive
                            if (RightToLeftInternal)
                            {
                                mouseX = rectScrollingArea.Right - 1;
                            }
                            else
                            {
                                mouseX = rectScrollingArea.Left + 1;
                            }
                        }
                    }
                }
                return true;
            }

            if (dataGridViewOper[DATAGRIDVIEWOPER_trackColSelect])
            {
                if (layout.ColumnHeadersVisible)
                {
                    // Include column headers
                    rectScrollingArea = Rectangle.Union(rectScrollingArea, layout.ColumnHeaders);
                }

                // Discard frozen columns
                DiscardZonesInScrollingArea(ref rectScrollingArea, emptyBackgroundWidth, emptyBackgroundHeight, frozenVisibleRowsHeight,
                                            true /*discardFrozenColumns*/, false /*discardFrozenRows*/);

                if (mouseX >= rectScrollingArea.Left && mouseX <= rectScrollingArea.Right)
                {
                    // Mouse's X is in-bound -- correct Y value
                    hti = HitTest(mouseX, rectScrollingArea.Top);
                    if (ptAnchorCell.X != -1 &&
                        Columns[ptAnchorCell.X].Frozen &&
                        trackColumnEdge != -1 &&
                        Columns[trackColumnEdge].Frozen &&
                        hti.col >= 0 &&
                        !Columns[hti.col].Frozen)
                    {
                        // Anchor cell is in frozen column and target cell is in unfrozen column. Make sure no column is scrolled off.
                        Debug.Assert(displayedBandsInfo.FirstDisplayedScrollingCol >= 0);
                        int firstUnfrozenColumnIndex = Columns.GetFirstColumn(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen).Index;
                        int firstRowIndex;
                        if (hti.row >= 0)
                        {
                            firstRowIndex = hti.row;
                        }
                        else
                        {
                            firstRowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                        }
                        if (firstRowIndex >= 0 && firstUnfrozenColumnIndex >= 0)
                        {
                            if (!ScrollIntoView(firstUnfrozenColumnIndex, firstRowIndex, false /*forCurrentCellChange*/))
                            {
                                return false;
                            }
                            hti = HitTest(mouseX, rectScrollingArea.Top);
                        }
                    }
                    return true;
                }

                // Mouse's X is outside of scrolling bands
                if ((!RightToLeftInternal && mouseX < rectScrollingArea.Left) ||
                    (RightToLeftInternal && mouseX > rectScrollingArea.Right))
                {
                    if (ptAnchorCell.X != -1 &&
                        (!Columns[ptAnchorCell.X].Frozen ||
                         (trackColumnEdge != -1 && !Columns[trackColumnEdge].Frozen)) &&
                        displayedBandsInfo.FirstDisplayedScrollingCol >= 0 &&
                        (negOffset > 0 ||
                         Columns.GetPreviousColumn(Columns[displayedBandsInfo.FirstDisplayedScrollingCol], DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen) != null))
                    {
                        // xOffset strictly negative
                        if (RightToLeftInternal)
                        {
                            // Right scrolling is required
                            xOffset = rectScrollingArea.Right - mouseX;
                        }
                        else
                        {
                            // Left scrolling is required
                            xOffset = mouseX - rectScrollingArea.Left;
                        }
                        mouseY = rectScrollingArea.Top + 1;
                    }
                    else
                    {
                        hti = HitTest(mouseX, rectScrollingArea.Top);
                    }
                }
                else
                {
                    Debug.Assert((!RightToLeftInternal && mouseX > rectScrollingArea.Right) || (RightToLeftInternal && mouseX < rectScrollingArea.Left));
                    if (displayedBandsInfo.FirstDisplayedScrollingCol >= 0)
                    {
                        if (displayedBandsInfo.LastTotallyDisplayedScrollingCol != -1 &&
                            Columns.GetNextColumn(Columns[displayedBandsInfo.LastTotallyDisplayedScrollingCol], DataGridViewElementStates.Visible, DataGridViewElementStates.None) == null)
                        {
                            // No more columns to scroll
                            return true;
                        }

                        DataGridViewColumn newFirstVisibleScrollingCol = Columns.GetNextColumn(Columns[displayedBandsInfo.FirstDisplayedScrollingCol],
                                                                                                             DataGridViewElementStates.Visible,
                                                                                                             DataGridViewElementStates.None);
                        int newColOffset = 0;
                        for (DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible,
                                                                                                          DataGridViewElementStates.Frozen);
                            dataGridViewColumn != newFirstVisibleScrollingCol;
                            dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                            DataGridViewElementStates.Visible,
                            DataGridViewElementStates.None))
                        {
                            newColOffset += dataGridViewColumn.Thickness;
                        }

                        if (HorizontalOffset != newColOffset)
                        {
                            // xOffset strictly positive
                            if (RightToLeftInternal)
                            {
                                // Left scrolling is required
                                xOffset = rectScrollingArea.Left - mouseX;
                            }
                            else
                            {
                                // Right scrolling is required
                                xOffset = mouseX - rectScrollingArea.Right;
                            }
                            mouseY = rectScrollingArea.Top + 1;
                        }
                    }
                }
                return true;
            }

            if (dataGridViewOper[DATAGRIDVIEWOPER_trackCellSelect])
            {
                bool recomputeHitTestInfo = false;

                // Discard frozen columns and rows
                DiscardZonesInScrollingArea(ref rectScrollingArea, emptyBackgroundWidth, emptyBackgroundHeight, frozenVisibleRowsHeight,
                                            true /*discardFrozenColumns*/, true /*discardFrozenRows*/);

                if (mouseY < rectScrollingArea.Top)
                {
                    // Mouse's Y is above scrolling bands
                    if (
                        (
                         (ptAnchorCell.Y != -1 && (Rows.GetRowState(ptAnchorCell.Y) & DataGridViewElementStates.Frozen) == 0)
                         ||
                         (ptCurrentCell.Y != -1 && (Rows.GetRowState(ptCurrentCell.Y) & DataGridViewElementStates.Frozen) == 0)
                        )
                        &&
                        verticalOffset != 0
                       )
                    {
                        // Up scrolling is required - the anchor's row is unfrozen
                        Debug.Assert(displayedBandsInfo.FirstDisplayedScrollingRow >= 0);
                        yOffset = mouseY - rectScrollingArea.Top;   // yOffset strictly negative
                    }
                    else
                    {
                        // Correct mouse's Y - no scrolling can be performed
                        if (mouseY < layout.Data.Top)
                        {
                            mouseY = layout.Data.Top + 1;
                            recomputeHitTestInfo = true;
                        }
                    }
                }
                else if (mouseY > rectScrollingArea.Bottom)
                {
                    // Mouse's Y is below scrolling bands
                    if (displayedBandsInfo.FirstDisplayedScrollingRow >= 0)
                    {
                        if (verticalOffset + Rows.SharedRow(displayedBandsInfo.FirstDisplayedScrollingRow).GetHeight(displayedBandsInfo.FirstDisplayedScrollingRow) <=
                            visibleRowsHeight - frozenVisibleRowsHeight - fittingTrailingScrollingRowsHeight)
                        {
                            // Down scrolling is required
                            yOffset = mouseY - rectScrollingArea.Bottom;   // yOffset strictly positive
                        }
                        else
                        {
                            // Correct mouse's Y - no scrolling can be performed
                            mouseY = rectScrollingArea.Bottom - 1;
                            recomputeHitTestInfo = true;
                        }
                    }
                    else
                    {
                        // Correct mouse's Y - no scrolling can be performed
                        mouseY = rectScrollingArea.Bottom - 1;
                        recomputeHitTestInfo = true;
                    }
                }
#if DEBUG
                else
                {
                    // Mouse's Y is in-bound
                    Debug.Assert(mouseY >= rectScrollingArea.Top && mouseY <= rectScrollingArea.Bottom);
                }
#endif
                if ((!RightToLeftInternal && mouseX < rectScrollingArea.Left) ||
                    (RightToLeftInternal && mouseX > rectScrollingArea.Right))
                {
                    // Mouse's X is on the left of scrolling bands (LTR)
                    if (
                        (
                         (ptAnchorCell.X != -1 && !Columns[ptAnchorCell.X].Frozen)
                         ||
                         (ptCurrentCell.X != -1 && !Columns[ptCurrentCell.X].Frozen)
                        )
                        &&
                        displayedBandsInfo.FirstDisplayedScrollingCol >= 0 &&
                        (negOffset > 0 ||
                         Columns.GetPreviousColumn(Columns[displayedBandsInfo.FirstDisplayedScrollingCol], DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen) != null)
                       )
                    {
                        // xOffset strictly negative
                        if (RightToLeftInternal)
                        {
                            // Right scrolling is required - anchor's column is unfrozen
                            xOffset = rectScrollingArea.Right - mouseX;
                        }
                        else
                        {
                            // Left scrolling is required - anchor's column is unfrozen
                            xOffset = mouseX - rectScrollingArea.Left;
                        }
                    }
                    else
                    {
                        // Correct mouse's X - no scrolling can be performed
                        if (!RightToLeftInternal && mouseX < layout.Data.Left)
                        {
                            mouseX = layout.Data.Left + 1;
                            recomputeHitTestInfo = true;
                        }
                        else if (RightToLeftInternal && mouseX > layout.Data.Right)
                        {
                            mouseX = layout.Data.Right - 1;
                            recomputeHitTestInfo = true;
                        }
                    }
                }
                else if ((!RightToLeftInternal && mouseX > rectScrollingArea.Right) ||
                         (RightToLeftInternal && mouseX < rectScrollingArea.Left))
                {
                    // Mouse's X is on the right of scrolling bands (LTR)
                    if (displayedBandsInfo.FirstDisplayedScrollingCol >= 0 &&
                        (displayedBandsInfo.LastTotallyDisplayedScrollingCol == -1 ||
                         Columns.GetNextColumn(Columns[displayedBandsInfo.LastTotallyDisplayedScrollingCol], DataGridViewElementStates.Visible, DataGridViewElementStates.None) != null))
                    {
                        DataGridViewColumn newFirstVisibleScrollingCol = Columns.GetNextColumn(Columns[displayedBandsInfo.FirstDisplayedScrollingCol],
                                                                                                            DataGridViewElementStates.Visible,
                                                                                                            DataGridViewElementStates.None);
                        int newColOffset = 0;
                        for (DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible,
                                                                                                          DataGridViewElementStates.Frozen);
                            dataGridViewColumn != newFirstVisibleScrollingCol;
                            dataGridViewColumn = Columns.GetNextColumn(dataGridViewColumn,
                            DataGridViewElementStates.Visible,
                            DataGridViewElementStates.None))
                        {
                            newColOffset += dataGridViewColumn.Thickness;
                        }
                        if (HorizontalOffset != newColOffset)
                        {
                            // xOffset strictly positive
                            if (RightToLeftInternal)
                            {
                                // Left scrolling is required
                                xOffset = rectScrollingArea.Left - mouseX;
                            }
                            else
                            {
                                // Right scrolling is required
                                xOffset = mouseX - rectScrollingArea.Right;
                            }
                        }
                        else
                        {
                            // Correct mouse's X - no scrolling can be performed
                            if (RightToLeftInternal)
                            {
                                mouseX = rectScrollingArea.Left + 1;
                            }
                            else
                            {
                                mouseX = rectScrollingArea.Right - 1;
                            }
                            recomputeHitTestInfo = true;
                        }
                    }
                    else
                    {
                        // Correct mouse's X - no scrolling can be performed
                        if (RightToLeftInternal)
                        {
                            mouseX = rectScrollingArea.Left + 1;
                        }
                        else
                        {
                            mouseX = rectScrollingArea.Right - 1;
                        }
                        recomputeHitTestInfo = true;
                    }
                }
#if DEBUG
                else
                {
                    // Mouse's X is in-bound
                    Debug.Assert(mouseX >= rectScrollingArea.Left && mouseX <= rectScrollingArea.Right);
                }
#endif
                if (recomputeHitTestInfo)
                {
                    hti = HitTest(mouseX, mouseY);
                }
            }
            return true;
        }

        internal override Size GetPreferredSizeCore(Size proposedConstraints)
        {
            int bordersAndPaddingWidth = 2 * (BorderWidth + Padding.Size.Width);
            int bordersAndPaddingHeight = 2 * (BorderWidth + Padding.Size.Height);

            bool allowHorizScrollbar = (scrollBars == ScrollBars.Both) || (scrollBars == ScrollBars.Horizontal);
            bool allowVertScrollbar = (scrollBars == ScrollBars.Both) || (scrollBars == ScrollBars.Vertical);

            int minimumWidth = 16 + bordersAndPaddingWidth;
            if (allowVertScrollbar)
            {
                minimumWidth += vertScrollBar.Width;
            }
            if (RowHeadersVisible)
            {
                minimumWidth += RowHeadersWidth;
            }
            int preferredWidth = Math.Min(minimumWidth + Columns.GetColumnsWidth(DataGridViewElementStates.Visible), proposedConstraints.Width);
            if (preferredWidth < minimumWidth)
            {
                preferredWidth = minimumWidth;
            }

            int minimumHeight = 16 + bordersAndPaddingHeight;
            if (allowHorizScrollbar)
            {
                minimumHeight += horizScrollBar.Height;
            }
            if (ColumnHeadersVisible)
            {
                minimumHeight += ColumnHeadersHeight;
            }
            int preferredHeight = Math.Min(minimumHeight + Rows.GetRowsHeight(DataGridViewElementStates.Visible), proposedConstraints.Height);
            if (preferredHeight < minimumHeight)
            {
                preferredHeight = minimumHeight;
            }

            return new Size(preferredWidth, preferredHeight);
        }

        // Rectangle returned includes the potential row header
        public Rectangle GetRowDisplayRectangle(int rowIndex, bool cutOverflow)
        {
            if (rowIndex < 0 || rowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(rowIndex));
            }
            return GetRowDisplayRectanglePrivate(rowIndex, cutOverflow);
        }

        private Rectangle GetRowDisplayRectanglePrivate(int rowIndex, bool cutOverflow)
        {
            Debug.Assert(rowIndex >= 0 && rowIndex < Rows.Count);

            if ((Rows.GetRowState(rowIndex) & DataGridViewElementStates.Displayed) == 0)
            {
                return Rectangle.Empty;
            }

            Rectangle data = layout.Data;
            int cy = data.Y;
            bool rowFound = false;
            int indexTmp;
            for (indexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                indexTmp != -1 && !rowFound;
                )
            {
                if (cy > data.Bottom)
                {
                    break;
                }
                if (indexTmp == rowIndex)
                {
                    rowFound = true;
                }
                else
                {
                    cy += Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                    indexTmp = Rows.GetNextRow(indexTmp, DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                }
            }

            if (!rowFound && displayedBandsInfo.FirstDisplayedScrollingRow >= 0)
            {
                for (indexTmp = displayedBandsInfo.FirstDisplayedScrollingRow;
                    indexTmp != -1 && !rowFound;
                    )
                {
                    if (cy > data.Bottom)
                    {
                        break;
                    }
                    if (indexTmp == rowIndex)
                    {
                        rowFound = true;
                    }
                    else
                    {
                        cy += Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                        indexTmp = Rows.GetNextRow(indexTmp, DataGridViewElementStates.Visible);
                    }
                }
            }

            if (rowFound)
            {
                int displayHeight;
                if (cutOverflow && cy + Rows.SharedRow(indexTmp).GetHeight(indexTmp) > data.Bottom)
                {
                    displayHeight = data.Bottom - cy;
                }
                else
                {
                    displayHeight = Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                }

                Rectangle rowRect = new Rectangle(data.X,
                    cy,
                    data.Width,
                    displayHeight);
                if (layout.RowHeadersVisible)
                {
                    rowRect.Width += layout.RowHeaders.Width;
                    if (!RightToLeftInternal)
                    {
                        rowRect.X -= layout.RowHeaders.Width;
                    }
                }
                return rowRect;
            }

            return Rectangle.Empty;
        }

        private int GetRowIndexFromY(int y, out int yRowTopEdge)
        {
            Rectangle data = layout.Data;
            Debug.Assert(y >= data.Y - 1 && y < data.Bottom, "y must be inside the vertical bounds of the data area.");

            if (y == data.Y - 1)
            {
                y++;
            }

            int rowHeight;
            int cy = data.Y;

            // first try to match y against a frozen rows
            int indexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
            while (indexTmp != -1 && cy < data.Bottom)
            {
                rowHeight = Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                cy += rowHeight;
                if (cy > y)
                {
                    yRowTopEdge = cy - rowHeight;
                    return indexTmp;
                }
                indexTmp = Rows.GetNextRow(indexTmp,
                    DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
            }

            // second try to match y against a scrolling row
            if (displayedBandsInfo.FirstDisplayedScrollingRow >= 0)
            {
                indexTmp = displayedBandsInfo.FirstDisplayedScrollingRow;
                Debug.Assert((Rows.GetRowState(indexTmp) & DataGridViewElementStates.Visible) != 0 &&
                    (Rows.GetRowState(indexTmp) & DataGridViewElementStates.Frozen) == 0);

                while (indexTmp != -1 && cy < data.Bottom)
                {
                    rowHeight = Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                    cy += rowHeight;
                    if (cy > y)
                    {
                        yRowTopEdge = cy - rowHeight;
                        return indexTmp;
                    }
                    indexTmp = Rows.GetNextRow(indexTmp,
                        DataGridViewElementStates.Visible);
                }
            }

            yRowTopEdge = -1;
            return -1;
        }

        private static int GetRowScrollRate(int yOffset)
        {
            Debug.Assert(yOffset > 0);
            // Counting 10ms for executing the actual row scrolling
            if (yOffset <= 10)
            {
                return 90;      // Ten rows per second
            }
            if (yOffset <= 15)
            {
                return 57;     // Fifteen rows per second
            }
            if (yOffset <= 25)
            {
                return 30;     // Twenty-five rows per second
            }
            if (yOffset <= 35)
            {
                return 18;     // Thirty-five rows per second
            }
            return Math.Max(1, 600 / yOffset);
        }

        /// <summary>
        ///  Returns the coordinate of the upper edge of the given row. Note that
        ///  the row does not need to be completely visible on the display area.
        ///  Value returned is not necessarily within layout.Data because the row
        ///  may start below the data area.
        /// </summary>
        internal int GetRowYFromIndex(int index)
        {
            Debug.Assert(index >= 0 && index < Rows.Count);
            Debug.Assert((Rows.GetRowState(index) & DataGridViewElementStates.Visible) != 0);

            int y = layout.Data.Y;

            int indexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
            while (indexTmp != -1)
            {
                if (index == indexTmp)
                {
                    return y;
                }
                y += Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                indexTmp = Rows.GetNextRow(indexTmp,
                    DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
            }

            int yFirstVisibleScrollingRow = y;

            if (displayedBandsInfo.FirstDisplayedScrollingRow >= 0)
            {
                if (index >= displayedBandsInfo.FirstDisplayedScrollingRow)
                {
                    // index is part of the scrolling rows below the frozen rows
                    indexTmp = displayedBandsInfo.FirstDisplayedScrollingRow;
                }
                else
                {
                    // index is part of the rows hidden behind the frozen rows or on top of the control
                    indexTmp = -1;
                }
            }
            else
            {
                // frozen rows cover all the rows real-estate. Look for index starting at the first visible non-frozen row.
                indexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                Debug.Assert(indexTmp != -1);
            }

            if (indexTmp != -1)
            {
                Debug.Assert((Rows.GetRowState(indexTmp) & DataGridViewElementStates.Visible) != 0 &&
                    (Rows.GetRowState(indexTmp) & DataGridViewElementStates.Frozen) == 0);

                while (indexTmp != -1)
                {
                    if (index == indexTmp)
                    {
                        return y;
                    }
                    y += Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                    indexTmp = Rows.GetNextRow(indexTmp, DataGridViewElementStates.Visible);
                }
            }

            // Row is completely hidden behind frozen rows or on top of control
            y = yFirstVisibleScrollingRow;
            Debug.Assert(displayedBandsInfo.FirstDisplayedScrollingRow != -1);
            indexTmp = Rows.GetPreviousRow(displayedBandsInfo.FirstDisplayedScrollingRow,
                DataGridViewElementStates.Visible,
                DataGridViewElementStates.Frozen);
            while (indexTmp != -1)
            {
                y -= Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                if (index == indexTmp)
                {
                    return y;
                }
                indexTmp = Rows.GetPreviousRow(indexTmp,
                    DataGridViewElementStates.Visible,
                    DataGridViewElementStates.Frozen);
            }

            Debug.Fail("Could not find row in GetRowYFromIndex");
            return 0;
        }

        private bool GetTabKeyEffective(bool shift, bool ctrl)
        {
            if (StandardTab)
            {
                return ctrl &&
                       !((!shift && (!VisibleCellExists || CurrentCellIsLastVisibleCell)) ||
                         (shift && (!VisibleCellExists || CurrentCellIsFirstVisibleCell)));
            }
            else
            {
                return !ctrl &&
                       !((!shift && (!VisibleCellExists || CurrentCellIsLastVisibleCell)) ||
                         (shift && (!VisibleCellExists || CurrentCellIsFirstVisibleCell)));
            }
        }

        public HitTestInfo HitTest(int x, int y)
        {
            HitTestInfo hti = new HitTestInfo();

            if (!layout.Inside.Contains(x, y))
            {
                return hti;
            }

            if (horizScrollBar != null && horizScrollBar.Visible && horizScrollBar.Bounds.Contains(x, y))
            {
                hti.type = DataGridViewHitTestType.HorizontalScrollBar;
                return hti;
            }

            if (vertScrollBar != null && vertScrollBar.Visible && vertScrollBar.Bounds.Contains(x, y))
            {
                hti.type = DataGridViewHitTestType.VerticalScrollBar;
                return hti;
            }

            if (layout.TopLeftHeader.Contains(x, y))
            {
                hti.type = DataGridViewHitTestType.TopLeftHeader;
                hti.typeInternal = DataGridViewHitTestTypeInternal.TopLeftHeader;
                if (RightToLeftInternal)
                {
                    hti.colStart = layout.TopLeftHeader.Right - 1;
                }
                else
                {
                    hti.colStart = layout.TopLeftHeader.Left;
                }
                hti.rowStart = layout.TopLeftHeader.Top;
                if ((!RightToLeftInternal && layout.TopLeftHeader.Right - x < DATAGRIDVIEW_columnSizingHotZone) ||
                    (RightToLeftInternal && x - layout.TopLeftHeader.Left < DATAGRIDVIEW_columnSizingHotZone))
                {
                    //hti.edge = DataGridViewHitTestTypeCloseEdge.Right;
                    if (RowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing)
                    {
                        hti.typeInternal = DataGridViewHitTestTypeInternal.TopLeftHeaderResizeLeft;
                        if (RightToLeftInternal)
                        {
                            hti.mouseBarOffset = layout.TopLeftHeader.Left - x - 1;
                        }
                        else
                        {
                            hti.mouseBarOffset = layout.TopLeftHeader.Right - x - 1;
                        }
                    }
                }
                else if (layout.TopLeftHeader.Top + layout.TopLeftHeader.Height - y < DATAGRIDVIEW_rowSizingHotZone)
                {
                    //hti.edge = DataGridViewHitTestTypeCloseEdge.Bottom;
                    if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.EnableResizing)
                    {
                        hti.typeInternal = DataGridViewHitTestTypeInternal.TopLeftHeaderResizeTop;
                        hti.mouseBarOffset = layout.TopLeftHeader.Top + layout.TopLeftHeader.Height - y - 1;
                    }
                }
                return hti;
            }

            // check for column resize / insertion
            if (layout.ColumnHeaders.Contains(x, y))
            {
                // this is actually the right edge in RTL mode
                hti.col = GetColumnIndexFromX(x, out int xColumnLeftEdge);
                if (hti.col < 0)
                {
                    return HitTestInfo.Nowhere;
                }
                Debug.Assert(xColumnLeftEdge == GetColumnXFromIndex(hti.col));
                hti.type = DataGridViewHitTestType.ColumnHeader;
                hti.typeInternal = DataGridViewHitTestTypeInternal.ColumnHeader;
                hti.rowStart = layout.ColumnHeaders.Top;
                hti.colStart = xColumnLeftEdge;
                int columnWidth = Columns[hti.col].Thickness;
                if ((!RightToLeftInternal && xColumnLeftEdge + columnWidth - x < DATAGRIDVIEW_columnSizingHotZone) ||
                    (RightToLeftInternal && x - xColumnLeftEdge + columnWidth < DATAGRIDVIEW_columnSizingHotZone))
                {
                    //hti.edge = DataGridViewHitTestTypeCloseEdge.Right;
                    if (RightToLeftInternal)
                    {
                        hti.mouseBarOffset = xColumnLeftEdge - columnWidth - x + 1;
                    }
                    else
                    {
                        hti.mouseBarOffset = xColumnLeftEdge + columnWidth - x - 1;
                    }
                    DataGridViewColumn dataGridViewColumn = Columns[hti.col];
                    if (dataGridViewColumn.Resizable == DataGridViewTriState.True &&
                        (dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.None || dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill))
                    {
                        hti.typeInternal = DataGridViewHitTestTypeInternal.ColumnResizeRight;
                    }
                    else
                    {
                        hti.typeInternal = DataGridViewHitTestTypeInternal.ColumnHeaderRight;
                    }
                }
                else if ((!RightToLeftInternal && x - xColumnLeftEdge < DATAGRIDVIEW_columnSizingHotZone) ||
                         (RightToLeftInternal && xColumnLeftEdge - x < DATAGRIDVIEW_columnSizingHotZone))
                {
                    //hti.edge = DataGridViewHitTestTypeCloseEdge.Left;
                    DataGridViewColumn dataGridViewColumn = null;

                    dataGridViewColumn = Columns.GetPreviousColumn(Columns[hti.col],
                                                                                 DataGridViewElementStates.Visible,
                                                                                 DataGridViewElementStates.None);
                    //}
                    if (dataGridViewColumn != null)
                    {
                        hti.adjacentCol = dataGridViewColumn.Index;
                        if (RightToLeftInternal)
                        {
                            hti.mouseBarOffset = xColumnLeftEdge - x + 1;
                        }
                        else
                        {
                            hti.mouseBarOffset = xColumnLeftEdge - x - 1;
                        }
                        if (dataGridViewColumn.Resizable == DataGridViewTriState.True &&
                            (dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.None || dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill))
                        {
                            hti.typeInternal = DataGridViewHitTestTypeInternal.ColumnResizeLeft;
                        }
                        else
                        {
                            hti.typeInternal = DataGridViewHitTestTypeInternal.ColumnHeaderLeft;
                        }
                    }
                    else
                    {
                        if (RowHeadersVisible && RowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing)
                        {
                            hti.typeInternal = DataGridViewHitTestTypeInternal.TopLeftHeaderResizeRight;
                            if (RightToLeftInternal)
                            {
                                hti.mouseBarOffset = xColumnLeftEdge - x;
                            }
                            else
                            {
                                hti.mouseBarOffset = xColumnLeftEdge - x - 1;
                            }
                        }
                        else
                        {
                            hti.typeInternal = DataGridViewHitTestTypeInternal.FirstColumnHeaderLeft;
                        }
                    }
                }
                else if (layout.ColumnHeaders.Bottom - y < DATAGRIDVIEW_rowSizingHotZone)
                {
                    //hti.edge = DataGridViewHitTestTypeCloseEdge.Bottom;
                    if (/*!this.RowHeadersVisible &&*/ ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.EnableResizing)
                    {
                        hti.typeInternal = DataGridViewHitTestTypeInternal.ColumnHeadersResizeBottom;
                        hti.mouseBarOffset = layout.ColumnHeaders.Bottom - y - 1;
                    }
                }
            }

            // check for row resize
            if (layout.RowHeaders.Contains(x, y))
            {
                hti.row = GetRowIndexFromY(y, out int yRowTopEdge);
                if (hti.row < 0)
                {
                    return HitTestInfo.Nowhere;
                }
                Debug.Assert(yRowTopEdge == GetRowYFromIndex(hti.row));
                hti.type = DataGridViewHitTestType.RowHeader;
                hti.typeInternal = DataGridViewHitTestTypeInternal.RowHeader;
                hti.rowStart = yRowTopEdge;
                if (RightToLeftInternal)
                {
                    hti.colStart = layout.RowHeaders.Right - 1;
                }
                else
                {
                    hti.colStart = layout.RowHeaders.Left;
                }
                int rowHeight = Rows.SharedRow(hti.row).GetHeight(hti.row);
                if (yRowTopEdge + rowHeight - y < DATAGRIDVIEW_rowSizingHotZone)
                {
                    //hti.edge = DataGridViewHitTestTypeCloseEdge.Bottom;

                    if (RowIsResizable(hti.row) && AutoSizeRowsMode == DataGridViewAutoSizeRowsMode.None)
                    {
                        hti.typeInternal = DataGridViewHitTestTypeInternal.RowResizeBottom;
                        hti.mouseBarOffset = yRowTopEdge + rowHeight - y - 1;
                    }
                }
                else if (y - yRowTopEdge < DATAGRIDVIEW_rowSizingHotZone)
                {
                    //hti.edge = DataGridViewHitTestTypeCloseEdge.Top;
                    int indexTmp = -1;
                    if (hti.row != displayedBandsInfo.FirstDisplayedScrollingRow || displayedBandsInfo.NumDisplayedFrozenRows > 0)
                    {
                        indexTmp = Rows.GetPreviousRow(hti.row, DataGridViewElementStates.Visible);
                    }
                    if (indexTmp != -1)
                    {
                        if (RowIsResizable(indexTmp) && AutoSizeRowsMode == DataGridViewAutoSizeRowsMode.None)
                        {
                            hti.typeInternal = DataGridViewHitTestTypeInternal.RowResizeTop;
                            hti.adjacentRow = indexTmp;
                            hti.mouseBarOffset = yRowTopEdge - y - 1;
                        }
                    }
                    else
                    {
                        if (ColumnHeadersVisible && ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.EnableResizing)
                        {
                            hti.typeInternal = DataGridViewHitTestTypeInternal.TopLeftHeaderResizeBottom;
                            hti.mouseBarOffset = yRowTopEdge - y - 1;
                        }
                    }
                }
                else if ((!RightToLeftInternal && layout.RowHeaders.Right - x < DATAGRIDVIEW_columnSizingHotZone) ||
                         (RightToLeftInternal && x - layout.RowHeaders.Left < DATAGRIDVIEW_columnSizingHotZone))
                {
                    //hti.edge = DataGridViewHitTestTypeCloseEdge.Right;
                    if (RowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing)
                    {
                        hti.typeInternal = DataGridViewHitTestTypeInternal.RowHeadersResizeRight;
                        if (RightToLeftInternal)
                        {
                            hti.mouseBarOffset = layout.RowHeaders.Left - x - 1;
                        }
                        else
                        {
                            hti.mouseBarOffset = layout.RowHeaders.Right - x - 1;
                        }
                    }
                }
            }

            if (layout.Data.Contains(x, y))
            {
                hti.col = GetColumnIndexFromX(x, out int xColumnLeftEdge);
                hti.row = GetRowIndexFromY(y, out int yRowTopEdge);
                if (hti.col < 0 || hti.row < 0)
                {
                    return HitTestInfo.Nowhere;
                }
                Debug.Assert(xColumnLeftEdge == GetColumnXFromIndex(hti.col));
                Debug.Assert(yRowTopEdge == GetRowYFromIndex(hti.row));
                hti.type = DataGridViewHitTestType.Cell;
                hti.typeInternal = DataGridViewHitTestTypeInternal.Cell;
                hti.rowStart = yRowTopEdge;
                hti.colStart = xColumnLeftEdge;
                if (!ColumnHeadersVisible)
                {
                    int columnWidth = Columns[hti.col].Thickness;
                    if ((!RightToLeftInternal && xColumnLeftEdge + columnWidth - x < DATAGRIDVIEW_columnSizingHotZone) ||
                        (RightToLeftInternal && x - xColumnLeftEdge + columnWidth < DATAGRIDVIEW_columnSizingHotZone))
                    {
                        if (RightToLeftInternal)
                        {
                            hti.mouseBarOffset = xColumnLeftEdge - columnWidth - x + 1;
                        }
                        else
                        {
                            hti.mouseBarOffset = xColumnLeftEdge + columnWidth - x - 1;
                        }
                        DataGridViewColumn dataGridViewColumn = Columns[hti.col];
                        if (dataGridViewColumn.Resizable == DataGridViewTriState.True &&
                            (dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.None || dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill))
                        {
                            hti.typeInternal = DataGridViewHitTestTypeInternal.ColumnResizeRight;
                        }
                        return hti;
                    }
                    else if ((!RightToLeftInternal && x - xColumnLeftEdge < DATAGRIDVIEW_columnSizingHotZone) ||
                             (RightToLeftInternal && xColumnLeftEdge - x < DATAGRIDVIEW_columnSizingHotZone))
                    {
                        DataGridViewColumn dataGridViewColumn = null;
                        if (hti.col != displayedBandsInfo.FirstDisplayedScrollingCol || displayedBandsInfo.LastTotallyDisplayedScrollingCol >= 0)
                        {
                            dataGridViewColumn = Columns.GetPreviousColumn(Columns[hti.col],
                                DataGridViewElementStates.Visible,
                                DataGridViewElementStates.None);
                        }
                        if (dataGridViewColumn != null)
                        {
                            hti.adjacentCol = dataGridViewColumn.Index;
                            if (RightToLeftInternal)
                            {
                                hti.mouseBarOffset = xColumnLeftEdge - x + 1;
                            }
                            else
                            {
                                hti.mouseBarOffset = xColumnLeftEdge - x - 1;
                            }
                            if (dataGridViewColumn.Resizable == DataGridViewTriState.True &&
                                (dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.None || dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill))
                            {
                                hti.typeInternal = DataGridViewHitTestTypeInternal.ColumnResizeLeft;
                            }
                            return hti;
                        }
                        else
                        {
                            if (RowHeadersVisible && RowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing)
                            {
                                hti.typeInternal = DataGridViewHitTestTypeInternal.RowHeadersResizeLeft;
                                if (RightToLeftInternal)
                                {
                                    hti.mouseBarOffset = xColumnLeftEdge - x;
                                }
                                else
                                {
                                    hti.mouseBarOffset = xColumnLeftEdge - x - 1;
                                }
                                return hti;
                            }
                        }
                    }
                }
                else if ((!RightToLeftInternal && x - xColumnLeftEdge < DATAGRIDVIEW_columnSizingHotZone) ||
                         (RightToLeftInternal && xColumnLeftEdge - x < DATAGRIDVIEW_columnSizingHotZone))
                {
                    DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
                    Debug.Assert(dataGridViewColumn != null);
                    if (hti.col == dataGridViewColumn.Index &&
                        RowHeadersVisible &&
                        RowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing)
                    {
                        hti.typeInternal = DataGridViewHitTestTypeInternal.RowHeadersResizeLeft;
                        if (RightToLeftInternal)
                        {
                            hti.mouseBarOffset = xColumnLeftEdge - x;
                        }
                        else
                        {
                            hti.mouseBarOffset = xColumnLeftEdge - x - 1;
                        }
                        return hti;
                    }
                }

                if (!RowHeadersVisible)
                {
                    int rowHeight = Rows.SharedRow(hti.row).GetHeight(hti.row);
                    if (yRowTopEdge + rowHeight - y < DATAGRIDVIEW_rowSizingHotZone)
                    {
                        if (RowIsResizable(hti.row) && AutoSizeRowsMode == DataGridViewAutoSizeRowsMode.None)
                        {
                            hti.typeInternal = DataGridViewHitTestTypeInternal.RowResizeBottom;
                            hti.mouseBarOffset = yRowTopEdge + rowHeight - y - 1;
                        }
                    }
                    else if (y - yRowTopEdge < DATAGRIDVIEW_rowSizingHotZone)
                    {
                        int indexTmp = -1;
                        if (hti.row != displayedBandsInfo.FirstDisplayedScrollingRow || displayedBandsInfo.NumDisplayedFrozenRows > 0)
                        {
                            indexTmp = Rows.GetPreviousRow(hti.row,
                                DataGridViewElementStates.Visible);
                        }
                        if (indexTmp != -1)
                        {
                            if (RowIsResizable(indexTmp) && AutoSizeRowsMode == DataGridViewAutoSizeRowsMode.None)
                            {
                                hti.typeInternal = DataGridViewHitTestTypeInternal.RowResizeTop;
                                hti.adjacentRow = indexTmp;
                                hti.mouseBarOffset = yRowTopEdge - y - 1;
                            }
                        }
                        else
                        {
                            if (ColumnHeadersVisible && ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.EnableResizing)
                            {
                                hti.typeInternal = DataGridViewHitTestTypeInternal.ColumnHeadersResizeTop;
                                hti.mouseBarOffset = yRowTopEdge - y - 1;
                            }
                        }
                    }
                }
                else if (y - yRowTopEdge < DATAGRIDVIEW_rowSizingHotZone)
                {
                    int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                    Debug.Assert(rowIndex >= 0);
                    if (hti.row == rowIndex &&
                        ColumnHeadersVisible &&
                        ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.EnableResizing)
                    {
                        hti.typeInternal = DataGridViewHitTestTypeInternal.ColumnHeadersResizeTop;
                        hti.mouseBarOffset = yRowTopEdge - y - 1;
                    }
                }
            }

            return hti;
        }

        private void HorizScrollTimer_Tick(object sender, EventArgs e)
        {
            BeginInvoke(new MethodInvoker(HorizScrollTimerHandler));
        }

        private void HorizScrollTimerHandler()
        {
            Debug.Assert(dataGridViewOper[DATAGRIDVIEWOPER_trackColSelect] || dataGridViewOper[DATAGRIDVIEWOPER_trackCellSelect]);

            Point ptMouse = PointToClient(Control.MousePosition);
            HitTestInfo hti = HitTest(ptMouse.X, ptMouse.Y);
            int mouseX = ptMouse.X, mouseY = ptMouse.Y;
            if (GetOutOfBoundCorrectedHitTestInfo(ref hti, ref mouseX, ref mouseY, out int xOffset, out int yOffset))
            {
                if (xOffset != 0)
                {
                    int absXOffset = Math.Abs(xOffset), normOffset = xOffset / absXOffset;
                    ScrollColumns(normOffset);
                    horizScrollTimer.Interval = GetColumnScrollRate(absXOffset);
                    if (dataGridViewOper[DATAGRIDVIEWOPER_trackColSelect])
                    {
                        hti = HitTest(ptMouse.X + (RightToLeftInternal ? 1 : -1) * (xOffset + normOffset), mouseY);
                        if (hti.col >= 0)
                        {
                            OnColumnSelectMouseMove(hti);
                        }
                    }
                    else if (dataGridViewOper[DATAGRIDVIEWOPER_trackCellSelect])
                    {
                        if (yOffset != 0)
                        {
                            hti = HitTest(ptMouse.X + (RightToLeftInternal ? 1 : -1) * (xOffset + normOffset), ptMouse.Y - yOffset - (yOffset / Math.Abs(yOffset)));
                        }
                        else
                        {
                            hti = HitTest(ptMouse.X + (RightToLeftInternal ? 1 : -1) * (xOffset + normOffset), mouseY);
                        }
                        if (hti.col >= 0 && hti.row >= 0)
                        {
                            OnCellSelectMouseMove(hti);
                        }
                    }
                }
                else
                {
                    if (dataGridViewOper[DATAGRIDVIEWOPER_trackColSelect] && hti.col >= 0)
                    {
                        OnColumnSelectMouseMove(hti);
                    }
                    else if (dataGridViewOper[DATAGRIDVIEWOPER_trackCellSelect] && hti.col >= 0 && hti.row >= 0)
                    {
                        OnCellSelectMouseMove(hti);
                    }
                    HorizScrollTimer.Enabled = false;
                }
            }
        }

        // Returns true for success, returns false when the OnDataError event cancels the operation.
        private bool InitializeEditingCellValue(ref DataGridViewCellStyle dataGridViewCellStyle, ref DataGridViewCell dataGridViewCell)
        {
            DataGridViewDataErrorEventArgs dgvdee = null;
            // Save unedited value so we can restore it later if parsing of new value fails
            uneditedFormattedValue = dataGridViewCell.GetFormattedValue(ptCurrentCell.Y, ref dataGridViewCellStyle, DataGridViewDataErrorContexts.Formatting);
            dataGridViewState1[DATAGRIDVIEWSTATE1_ignoringEditingChanges] = true;
            try
            {
                IDataGridViewEditingCell dataGridViewEditingCell = dataGridViewCell as IDataGridViewEditingCell;
                Debug.Assert(dataGridViewEditingCell != null);
                object currentFormattedValue = dataGridViewEditingCell.GetEditingCellFormattedValue(DataGridViewDataErrorContexts.Formatting);
                if ((currentFormattedValue == null && uneditedFormattedValue != null) ||
                    (currentFormattedValue != null && uneditedFormattedValue == null) ||
                    (currentFormattedValue != null && !uneditedFormattedValue.Equals(currentFormattedValue)))
                {
                    Debug.Assert(ptCurrentCell.X == dataGridViewCell.ColumnIndex);
                    dataGridViewCell = Rows[ptCurrentCell.Y].Cells[ptCurrentCell.X]; // unshare the edited cell
                    dataGridViewEditingCell = dataGridViewCell as IDataGridViewEditingCell;
                    dataGridViewEditingCell.EditingCellFormattedValue = uneditedFormattedValue;
                    dataGridViewEditingCell.EditingCellValueChanged = false;
                }
            }
            catch (Exception exception)
            {
                if (ClientUtils.IsCriticalException(exception))
                {
                    throw;
                }
                dgvdee = new DataGridViewDataErrorEventArgs(exception, ptCurrentCell.X,
                    ptCurrentCell.Y,
                    DataGridViewDataErrorContexts.InitialValueRestoration);
                OnDataErrorInternal(dgvdee);
            }
            finally
            {
                dataGridViewState1[DATAGRIDVIEWSTATE1_ignoringEditingChanges] = false;
            }
            if (dgvdee != null)
            {
                if (dgvdee.ThrowException)
                {
                    throw dgvdee.Exception;
                }
                return !dgvdee.Cancel;
            }
            return true;
        }

        // Returns true for success, returns false when the OnDataError event cancels the operation.
        private bool InitializeEditingControlValue(ref DataGridViewCellStyle dataGridViewCellStyle, DataGridViewCell dataGridViewCell)
        {
            Debug.Assert(dataGridViewCell != null);
            Debug.Assert(editingControl != null);

            DataGridViewDataErrorEventArgs dgvdee = null;
            object initialFormattedValue = dataGridViewCell.GetFormattedValue(ptCurrentCell.Y, ref dataGridViewCellStyle, DataGridViewDataErrorContexts.Formatting);
            dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlChanging] = true;
            dataGridViewState1[DATAGRIDVIEWSTATE1_ignoringEditingChanges] = true;
            try
            {
                dataGridViewCell.InitializeEditingControl(ptCurrentCell.Y, initialFormattedValue, dataGridViewCellStyle);
                ((IDataGridViewEditingControl)editingControl).EditingControlValueChanged = false;
            }
            catch (Exception exception)
            {
                if (ClientUtils.IsCriticalException(exception))
                {
                    throw;
                }
                dgvdee = new DataGridViewDataErrorEventArgs(exception, ptCurrentCell.X,
                    ptCurrentCell.Y,
                    DataGridViewDataErrorContexts.InitialValueRestoration);
                OnDataErrorInternal(dgvdee);
            }
            finally
            {
                dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlChanging] = false;
                dataGridViewState1[DATAGRIDVIEWSTATE1_ignoringEditingChanges] = false;
            }
            if (dgvdee != null)
            {
                if (dgvdee.ThrowException)
                {
                    throw dgvdee.Exception;
                }
                return !dgvdee.Cancel;
            }

            // Save unedited value so we can restore it later if parsing of new value fails
            uneditedFormattedValue = initialFormattedValue;
            return true;
        }

        public void InvalidateCell(DataGridViewCell dataGridViewCell)
        {
            if (dataGridViewCell == null)
            {
                throw new ArgumentNullException(nameof(dataGridViewCell));
            }
            if (dataGridViewCell.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_CellDoesNotBelongToDataGridView);
            }
            InvalidateCellPrivate(dataGridViewCell);
        }

        private void InvalidateCellPrivate(DataGridViewCell dataGridViewCell)
        {
            Debug.Assert(dataGridViewCell != null);
            Debug.Assert(dataGridViewCell.DataGridView == this);
            InvalidateCell(dataGridViewCell.ColumnIndex, dataGridViewCell.RowIndex);
        }

        public void InvalidateCell(int columnIndex, int rowIndex)
        {
            if (columnIndex < -1 || columnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(columnIndex));
            }
            if (rowIndex < -1 || rowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(rowIndex));
            }
            InvalidateCellPrivate(columnIndex, rowIndex);
        }

        private void InvalidateCellPrivate(int columnIndex, int rowIndex)
        {
            if (IsHandleCreated)
            {
                Rectangle cellDisplayRect = GetCellAdjustedDisplayRectangle(columnIndex, rowIndex, true);
                if (!cellDisplayRect.IsEmpty)
                {
                    Invalidate(cellDisplayRect);
                }
            }
        }

        /// <summary>
        ///  Invalidate the painting region for the column specified.
        /// </summary>
        public void InvalidateColumn(int columnIndex)
        {
            if (columnIndex < 0 || columnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(columnIndex));
            }
            InvalidateColumnInternal(columnIndex);
        }

        internal void InvalidateColumnInternal(int columnIndex)
        {
            Debug.Assert(columnIndex >= 0 && columnIndex < Columns.Count);

            if (IsHandleCreated)
            {
                Rectangle columnDisplayRect = GetColumnDisplayRectanglePrivate(columnIndex, true);
                if (!columnDisplayRect.IsEmpty)
                {
                    Invalidate(columnDisplayRect);
                }
            }
        }

        private void InvalidateData()
        {
            if (IsHandleCreated)
            {
                Invalidate(layout.Data);
            }
        }

        /// <summary>
        ///  Invalidates the scrollable area of the DataGridView.
        /// </summary>
        private void InvalidateInside()
        {
            if (IsHandleCreated)
            {
                Invalidate(layout.Inside);
            }
        }

        /// <summary>
        ///  Invalidate the painting region for the row specified.
        /// </summary>
        public void InvalidateRow(int rowIndex)
        {
            if (rowIndex < 0 || rowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException(nameof(rowIndex));
            }
            InvalidateRowPrivate(rowIndex);
        }

        private void InvalidateRowPrivate(int rowIndex)
        {
            Debug.Assert(rowIndex >= 0 && rowIndex < Rows.Count);

            if (IsHandleCreated)
            {
                Rectangle rowDisplayRect = GetRowDisplayRectanglePrivate(rowIndex, true);
                if (!rowDisplayRect.IsEmpty)
                {
                    Invalidate(rowDisplayRect);
                }
            }
        }

        private void InvalidateRowHeights()
        {
            Rows.InvalidateCachedRowsHeights();
            if (IsHandleCreated)
            {
                PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, false /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
                Invalidate();
            }
        }

        private void InvalidateRows(int lo, int hi)
        {
            Debug.Assert(lo <= hi);
            Debug.Assert(lo < Rows.Count);
            Debug.Assert(hi < Rows.Count);

            if (Rows.GetRowCount(DataGridViewElementStates.Visible) == 0)
            {
                return;
            }

            Rectangle rowDisplayRect, data;
            int top, bottom;

            data = layout.Data;

            // If "lo" is not visible, then get the next visible row
            if ((Rows.GetRowState(lo) & DataGridViewElementStates.Visible) == 0)
            {
                lo = Rows.GetNextRow(lo, DataGridViewElementStates.Visible);
            }

            if (lo == -1)
            {
                // there are no visible rows below "lo" so there is nothing to invalidate.
                return;
            }

            // If "hi" is not visible, then get the previous visible row
            if ((Rows.GetRowState(hi) & DataGridViewElementStates.Visible) == 0)
            {
                hi = Rows.GetPreviousRow(hi, DataGridViewElementStates.Visible);
            }

            Debug.Assert(lo <= hi);
            Debug.Assert(lo > -1);

            rowDisplayRect = GetRowDisplayRectangle(lo, true /*cutOverflow*/);

            if (rowDisplayRect.IsEmpty)
            {
                // The top row is offscreen
                if ((Rows.GetRowState(lo) & DataGridViewElementStates.Frozen) != 0)
                {
                    // "lo" is a frozen row which is offscreen.
                    // This means that "lo" and any other row below it are offscreen.
                    return;
                }
                else if (displayedBandsInfo.NumDisplayedScrollingRows == 0)
                {
                    // "lo" and all the rows below are scrolling rows but no scrolling rows are displayed.
                    return;
                }
                else if (lo >= displayedBandsInfo.FirstDisplayedScrollingRow &&
                    Rows.GetRowCount(DataGridViewElementStates.Visible,
                                          displayedBandsInfo.FirstDisplayedScrollingRow,
                                          lo) >= displayedBandsInfo.NumDisplayedScrollingRows)
                {
                    // "lo" is a scrolling row whose coordinates are below the last visible row.
                    return;
                }
                else
                {
                    // "lo" is a scrolling row "behind" frozen rows.
                    // Start invalidating at the top of the first displayed scrolling row.
                    top = GetRowDisplayRectangle(displayedBandsInfo.FirstDisplayedScrollingRow, true /*cutOverflow*/).Top;
                }
            }
            else
            {
                top = rowDisplayRect.Top;
            }

            rowDisplayRect = GetRowDisplayRectangle(hi, true /*cutOverflow*/);

            if (rowDisplayRect.IsEmpty)
            {
                // The bottom row is offscreen.
                if ((Rows.GetRowState(hi) & DataGridViewElementStates.Frozen) == DataGridViewElementStates.Frozen)
                {
                    // "hi" is a frozen row offscreen and "lo" is a frozen row on screen.
                    // Invalidate all the way to the bottom
                    bottom = data.Bottom;
                }
                else if (hi > displayedBandsInfo.FirstDisplayedScrollingRow)
                {
                    // "hi" is a scrolling row offscreen which is beyond the firstDisplayedScrollingRow
                    // Invalidate all the way to the bottom again.
                    bottom = data.Bottom;
                }
                else if (Rows.GetRowCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen) == 0)
                {
                    // "hi" is a scrolling row above the first displayed scrolling row and there are no frozen rows.
                    // There is nothing to invalidate.
                    return;
                }
                else
                {
                    // "hi" is a scrolling row which is "behind" the frozen rows.
                    // Invalidate all the way to the bottom of the frozen rows
                    // Compute the bottom of the last displayed frozen row.
                    // There may be invisible rows between the frozen rows.
                    bottom = 0;
                    for (int i = 0; i < displayedBandsInfo.NumDisplayedFrozenRows;)
                    {
                        if ((Rows.GetRowState(i) & DataGridViewElementStates.Visible) == 0)
                        {
                            continue;
                        }

                        if (i == displayedBandsInfo.NumDisplayedFrozenRows - 1)
                        {
                            bottom = GetRowDisplayRectangle(i, true /*cutOverflow*/).Bottom;
                            break;
                        }

                        i++;
                    }

                    if (bottom <= top)
                    {
                        // In this case both "lo" and "hi" are two scrolling rows behind the frozen rows.
                        // Nothing to invalidate.
                        return;
                    }
                }
            }
            else
            {
                bottom = rowDisplayRect.Bottom;
            }

            Invalidate(new Rectangle(data.X, top, data.Width, bottom - top));
        }

        private void InvalidateScrollBars()
        {
            // invalidate the horizontal and the vertical scrollbars
            // note that the scrollbars can be null - this happens when
            // the control has been disposed.
            if (horizScrollBar != null && horizScrollBar.Visible)
            {
                horizScrollBar.Invalidate();
            }
            if (vertScrollBar != null && vertScrollBar.Visible)
            {
                vertScrollBar.Invalidate();
            }
        }

        private bool IsColumnOutOfBounds(int columnIndex)
        {
            return columnIndex >= Columns.Count || columnIndex == -1;
        }

        private bool IsInnerCellOutOfBounds(int columnIndex, int rowIndex)
        {
            return columnIndex >= Columns.Count || rowIndex >= Rows.Count || columnIndex == -1 || rowIndex == -1;
        }

        private bool IsRowOutOfBounds(int rowIndex)
        {
            return rowIndex >= Rows.Count || rowIndex == -1;
        }

        protected override bool IsInputChar(char charCode)
        {
            if (editingControl != null &&
                dataGridViewState1[DATAGRIDVIEWSTATE1_forwardCharMessage])
            {
                // Do not process key press in ProcessDialogChar.
                return true;
            }
            else
            {
                return base.IsInputChar(charCode);
            }
        }

        protected override bool IsInputKey(Keys keyData)
        {
            if ((keyData & Keys.Alt) == Keys.Alt)
            {
                return false;
            }
            switch (keyData & Keys.KeyCode)
            {
                case Keys.Escape:
                    {
                        return IsEscapeKeyEffective;
                    }

                case Keys.Tab:
                    {
                        return GetTabKeyEffective((keyData & Keys.Shift) == Keys.Shift, (keyData & Keys.Control) == Keys.Control);
                    }

                case Keys.A:
                    {
                        if ((keyData & (Keys.Control | Keys.Shift | Keys.Alt)) == Keys.Control)
                        {
                            return true;
                        }
                        break;
                    }

                case Keys.C:
                case Keys.Insert:
                    {
                        if ((keyData & (Keys.Control | Keys.Shift | Keys.Alt)) == Keys.Control)
                        {
                            return true;
                        }
                        break;
                    }

                case Keys.Space:
                    {
                        if ((keyData & (Keys.Control | Keys.Shift | Keys.Alt)) == Keys.Shift &&
                            (SelectionMode == DataGridViewSelectionMode.ColumnHeaderSelect ||
                             SelectionMode == DataGridViewSelectionMode.RowHeaderSelect) &&
                            ptCurrentCell.X != -1)
                        {
                            return true;
                        }
                        break;
                    }

                case Keys.Up:
                case Keys.Down:
                case Keys.Left:
                case Keys.Right:
                case Keys.Home:
                case Keys.End:
                case Keys.Next:
                case Keys.Prior:
                case Keys.Enter:
                case Keys.Delete:
                case Keys.D0:
                case Keys.NumPad0:
                case Keys.F2:
                case Keys.F3:
                    {
                        return true;
                    }
            }
            return base.IsInputKey(keyData);
        }

        /// <summary>
        ///  Determines if Scrollbars should be visible,
        ///  updates their bounds and the bounds of all
        ///  other regions in the dataGridView's Layout.
        /// </summary>
        private void LayoutScrollBars()
        {
            SuspendLayout();
            try
            {
                // Scrollbars are a tricky issue.
                // We need to see if we can cram our columns and rows
                // in without scrollbars and if they don't fit, we make
                // scrollbars visible and then fixup our regions for the
                // data.
                bool allowHorizScrollbar = ((scrollBars == ScrollBars.Both) || (scrollBars == ScrollBars.Horizontal)) &&
                                           dataGridViewState2[DATAGRIDVIEWSTATE2_allowHorizontalScrollbar];
                bool allowVertScrollbar = (scrollBars == ScrollBars.Both) || (scrollBars == ScrollBars.Vertical);
                bool needHorizScrollbarWithoutVertScrollbar = false;
                bool needHorizScrollbar = false;
                bool needVertScrollbar = false;
                bool rightToLeftInternal = RightToLeftInternal;
                int oldfirstDisplayedScrollingRow;

                int totalVisibleColCount = Columns.GetColumnCount(DataGridViewElementStates.Visible);
                int totalVisibleRowCount = Rows.GetRowCount(DataGridViewElementStates.Visible);
                int totalVisibleWidth = Columns.GetColumnsWidth(DataGridViewElementStates.Visible);
                int totalVisibleFrozenWidth = Columns.GetColumnsWidth(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);

                // Expensive call - dataGridView could have a mode where no row is resizable which would result in better perfs
                int totalVisibleHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible);
                int totalVisibleFrozenHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);

                int horizScrollBarHeight = horizScrollBar.Height = SystemInformation.HorizontalScrollBarHeight;
                int vertScrollBarWidth = vertScrollBar.Width = SystemInformation.VerticalScrollBarWidth;

                if (allowHorizScrollbar &&
                    totalVisibleWidth > layout.Data.Width && totalVisibleFrozenWidth < layout.Data.Width &&
                    horizScrollBarHeight <= layout.Data.Height)
                {
                    int oldDataHeight = layout.Data.Height;
                    layout.Data.Height -= horizScrollBarHeight;
                    Debug.Assert(layout.Data.Height >= 0);
                    needHorizScrollbarWithoutVertScrollbar = needHorizScrollbar = true;
                    if (totalVisibleWidth - layout.Data.Width <= vertScrollBarWidth ||
                        layout.Data.Width - totalVisibleFrozenWidth <= vertScrollBarWidth)
                    {
                        // Would we still need a horizontal scrollbar if there were a vertical one?
                        oldfirstDisplayedScrollingRow = displayedBandsInfo.FirstDisplayedScrollingRow;
                        ComputeVisibleRows();
                        if (displayedBandsInfo.NumTotallyDisplayedFrozenRows == Rows.GetRowCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen) &&
                            displayedBandsInfo.NumTotallyDisplayedScrollingRows != totalVisibleRowCount - Rows.GetRowCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen) &&
                            (totalVisibleHeight - totalVisibleFrozenHeight != ComputeHeightOfFittingTrailingScrollingRows(totalVisibleFrozenHeight)))
                        {
                            needHorizScrollbar = (totalVisibleFrozenWidth < layout.Data.Width - vertScrollBarWidth);
                        }
                        displayedBandsInfo.FirstDisplayedScrollingRow = oldfirstDisplayedScrollingRow;
                    }

                    if (needHorizScrollbar)
                    {
                        if (layout.RowHeadersVisible)
                        {
                            layout.RowHeaders.Height -= horizScrollBarHeight;
                            Debug.Assert(layout.RowHeaders.Height >= 0);
                        }
                    }
                    else
                    {
                        // Restore old data height because turns out a horizontal scroll bar wouldn't make sense
                        layout.Data.Height = oldDataHeight;
                    }
                }

                oldfirstDisplayedScrollingRow = displayedBandsInfo.FirstDisplayedScrollingRow;

                ComputeVisibleRows();
                if (allowVertScrollbar &&
                    displayedBandsInfo.NumTotallyDisplayedFrozenRows == Rows.GetRowCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen) &&
                    displayedBandsInfo.NumTotallyDisplayedScrollingRows != totalVisibleRowCount - Rows.GetRowCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen) &&
                    (totalVisibleHeight - totalVisibleFrozenHeight != ComputeHeightOfFittingTrailingScrollingRows(totalVisibleFrozenHeight)) &&
                    layout.Data.Height > totalVisibleFrozenHeight &&
                    vertScrollBarWidth <= layout.Data.Width)
                {
                    layout.Data.Width -= vertScrollBarWidth;
                    Debug.Assert(layout.Data.Width >= 0);
                    if (rightToLeftInternal)
                    {
                        layout.Data.X += vertScrollBarWidth;
                    }
                    if (layout.ColumnHeadersVisible)
                    {
                        layout.ColumnHeaders.Width -= vertScrollBarWidth;
                        Debug.Assert(layout.ColumnHeaders.Width >= 0);
                        if (rightToLeftInternal)
                        {
                            layout.ColumnHeaders.X += vertScrollBarWidth;
                        }
                    }
                    needVertScrollbar = true;
                }

                displayedBandsInfo.FirstDisplayedScrollingCol = ComputeFirstVisibleScrollingColumn();
                // we compute the number of visible columns only after we set up the vertical scroll bar.
                ComputeVisibleColumns();

                if (allowHorizScrollbar &&
                    needVertScrollbar && !needHorizScrollbar &&
                    totalVisibleWidth > layout.Data.Width && totalVisibleFrozenWidth < layout.Data.Width &&
                    horizScrollBarHeight <= layout.Data.Height)
                {
                    displayedBandsInfo.FirstDisplayedScrollingRow = oldfirstDisplayedScrollingRow;
                    if (layout.ColumnHeadersVisible)
                    {
                        layout.ColumnHeaders.Width += vertScrollBarWidth;
                        if (rightToLeftInternal)
                        {
                            layout.ColumnHeaders.X -= vertScrollBarWidth;
                        }
                    }
                    layout.Data.Width += vertScrollBarWidth;
                    if (rightToLeftInternal)
                    {
                        layout.Data.X -= vertScrollBarWidth;
                    }
                    layout.Data.Height -= horizScrollBarHeight;
                    Debug.Assert(layout.Data.Height >= 0);
                    needVertScrollbar = false;

                    ComputeVisibleRows();
                    if (displayedBandsInfo.NumTotallyDisplayedFrozenRows == Rows.GetRowCount(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen) &&
                        displayedBandsInfo.NumTotallyDisplayedScrollingRows != totalVisibleRowCount &&
                        (totalVisibleHeight - totalVisibleFrozenHeight != ComputeHeightOfFittingTrailingScrollingRows(totalVisibleFrozenHeight)) &&
                        layout.Data.Height > totalVisibleFrozenHeight &&
                        vertScrollBarWidth <= layout.Data.Width)
                    {
                        layout.Data.Width -= vertScrollBarWidth;
                        Debug.Assert(layout.Data.Width >= 0);
                        if (rightToLeftInternal)
                        {
                            layout.Data.X += vertScrollBarWidth;
                        }
                        if (layout.ColumnHeadersVisible)
                        {
                            layout.ColumnHeaders.Width -= vertScrollBarWidth;
                            Debug.Assert(layout.ColumnHeaders.Width >= 0);
                            if (rightToLeftInternal)
                            {
                                layout.ColumnHeaders.X += vertScrollBarWidth;
                            }
                        }
                        needVertScrollbar = true;
                    }
                    if (needVertScrollbar)
                    {
                        needHorizScrollbar = true;
                    }
                    else
                    {
                        needHorizScrollbar = needHorizScrollbarWithoutVertScrollbar;
                    }
                }

                layout.ResizeBoxRect = new Rectangle();
                if (needVertScrollbar && needHorizScrollbar)
                {
                    layout.ResizeBoxRect = new Rectangle(
                        rightToLeftInternal ? layout.Data.X - vertScrollBar.Width : layout.Data.Right,
                        layout.Data.Bottom,
                        vertScrollBar.Width,
                        horizScrollBar.Height);
                }

                if (needHorizScrollbar && totalVisibleColCount > 0)
                {
                    int widthNotVisible = totalVisibleWidth - layout.Data.Width;

                    horizScrollBar.Minimum = 0;
                    horizScrollBar.Maximum = totalVisibleWidth - totalVisibleFrozenWidth;
                    Debug.Assert(horizScrollBar.Maximum > 0);
                    horizScrollBar.SmallChange = 1;
                    horizScrollBar.LargeChange = Math.Max(totalVisibleWidth - totalVisibleFrozenWidth - widthNotVisible, 0);
                    horizScrollBar.Enabled = Enabled;
                    horizScrollBar.Bounds = new Rectangle(
                        rightToLeftInternal ? layout.Inside.X + layout.ResizeBoxRect.Width : layout.Inside.X,
                        layout.Data.Bottom,
                        layout.Inside.Width - layout.ResizeBoxRect.Width,
                        horizScrollBar.Height);
                    horizScrollBar.Visible = true;
                    horizScrollBar.Invalidate();
                }
                else
                {
                    horizScrollBar.Visible = false;
                    HorizontalOffset = 0;

                    horizScrollBar.Enabled = false;
                    horizScrollBar.Minimum = 0;
                    horizScrollBar.Maximum = 1;
                    horizScrollBar.SmallChange = 1;
                    horizScrollBar.LargeChange = 1;
                    horizScrollBar.Value = 0;
                }

                if (needVertScrollbar)
                {
                    int vertScrollBarTop = layout.Data.Y;
                    int vertScrollBarHeight = layout.Data.Height;
                    if (layout.ColumnHeadersVisible)
                    {
                        vertScrollBarTop = layout.ColumnHeaders.Y;
                        vertScrollBarHeight += layout.ColumnHeaders.Height;
                    }
                    else if (SingleHorizontalBorderAdded)
                    {
                        vertScrollBarTop--;
                        vertScrollBarHeight++;
                    }

                    vertScrollBar.Minimum = 0;
                    vertScrollBar.Maximum = totalVisibleHeight - totalVisibleFrozenHeight;
                    Debug.Assert(vertScrollBar.Maximum > 0);
                    vertScrollBar.Value = ComputeHeightOfScrolledOffRows();
                    vertScrollBar.LargeChange = layout.Data.Height - totalVisibleFrozenHeight;
                    vertScrollBar.Bounds = new Rectangle(
                        rightToLeftInternal ? layout.Data.X - vertScrollBar.Width : layout.Data.Right,
                        vertScrollBarTop,
                        vertScrollBar.Width,
                        vertScrollBarHeight);
                    vertScrollBar.Enabled = Enabled;
                    vertScrollBar.Visible = true;
                    vertScrollBar.Invalidate();

                    verticalOffset = vertScrollBar.Value;
                }
                else
                {
                    vertScrollBar.Visible = false;
                    verticalOffset = ComputeHeightOfScrolledOffRows();

                    vertScrollBar.Enabled = false;
                    vertScrollBar.Minimum = 0;
                    vertScrollBar.Maximum = 1;
                    vertScrollBar.LargeChange = 1;
                    vertScrollBar.Value = 0;
                }
            }
            finally
            {
                ResumeLayout(false);
            }
        }

        private void MakeFirstDisplayedCellCurrentCell(bool includeNewRow)
        {
            // No current cell - try to set the first displayed cell to be the current one.
            Point firstDisplayedCellAddress = FirstDisplayedCellAddress;
            if (firstDisplayedCellAddress.X != -1 &&
                (includeNewRow ||
                 !AllowUserToAddRowsInternal ||
                 firstDisplayedCellAddress.Y != Rows.Count - 1))
            {
                bool success = SetAndSelectCurrentCellAddress(firstDisplayedCellAddress.X,
                                                              firstDisplayedCellAddress.Y,
                                                              true /*setAnchorCellAddress*/,
                                                              false /*validateCurrentCell*/,
                                                              false /*throughMouseClick*/,
                                                              true /*clearSelection*/,
                                                              false /*forceCurrentCellSelection (unused)*/);
                Debug.Assert(success);
            }
        }

        private static DataGridViewAutoSizeRowMode MapAutoSizeRowsModeToRowMode(DataGridViewAutoSizeRowsMode autoSizeRowsMode)
        {
            switch (autoSizeRowsMode)
            {
                case DataGridViewAutoSizeRowsMode.AllHeaders:
                    return DataGridViewAutoSizeRowMode.RowHeader;
                case DataGridViewAutoSizeRowsMode.DisplayedHeaders:
                    return DataGridViewAutoSizeRowMode.RowHeader;
                case DataGridViewAutoSizeRowsMode.AllCellsExceptHeaders:
                    return DataGridViewAutoSizeRowMode.AllCellsExceptHeader;
                case DataGridViewAutoSizeRowsMode.DisplayedCellsExceptHeaders:
                    return DataGridViewAutoSizeRowMode.AllCellsExceptHeader;
                case DataGridViewAutoSizeRowsMode.AllCells:
                    return DataGridViewAutoSizeRowMode.AllCells;
                case DataGridViewAutoSizeRowsMode.DisplayedCells:
                    return DataGridViewAutoSizeRowMode.AllCells;
                default:
                    Debug.Fail("Unexpected autoSizeRowsMode value in MapAutoSizeRowsModeToRowMode");
                    return DataGridViewAutoSizeRowMode.RowHeader;
            }
        }

        private void MoveColumnHeadersOrRowResize(MouseEventArgs e)
        {
            lastRowSplitBar = currentRowSplitBar;
            currentRowSplitBar = e.Y;
            Rectangle lastSplitBarRect = CalcRowResizeFeedbackRect(lastRowSplitBar);
            if (editingControl != null &&
                !dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlHidden] &&
                editingPanel.Bounds.IntersectsWith(lastSplitBarRect))
            {
                editingPanel.Invalidate();
                editingPanel.Update();
                editingControl.Invalidate();
                editingControl.Update();
            }
            Invalidate(lastSplitBarRect);
            Invalidate(CalcRowResizeFeedbackRect(currentRowSplitBar));
        }

        private void MapDataGridViewColumnToDataBoundField(DataGridViewColumn dataGridViewColumn)
        {
            Debug.Assert(DataSource != null, "this method should only be called when we have a data connection");
            Debug.Assert(dataGridViewColumn.DataPropertyName.Length != 0, "this method should be called only for columns which have DataPropertyName set");
            DataGridViewDataConnection conn = DataConnection;

            int boundColumnIndex = ((conn == null) ? -1 : conn.BoundColumnIndex(dataGridViewColumn.DataPropertyName));
            if (boundColumnIndex != -1)
            {
                dataGridViewColumn.IsDataBoundInternal = true;
                dataGridViewColumn.BoundColumnIndex = boundColumnIndex;
                dataGridViewColumn.BoundColumnConverter = conn.BoundColumnConverter(boundColumnIndex);
                dataGridViewColumn.ValueType = conn.BoundColumnValueType(boundColumnIndex);
                dataGridViewColumn.ReadOnly = conn.DataFieldIsReadOnly(dataGridViewColumn.BoundColumnIndex) || dataGridViewColumn.ReadOnly;
                InvalidateColumnInternal(dataGridViewColumn.Index);

                // Set the Sorting information on the data grid view according to the new DataPropertyName.
                // RefreshColumns() has its own routine for setting the Sorting information so don't do this step
                // if we are in RefreshColumns();
                if (dataGridViewColumn.SortMode != DataGridViewColumnSortMode.NotSortable &&
                    !dataGridViewOper[DATAGRIDVIEWOPER_inRefreshColumns])
                {
                    dataGridViewColumn.HeaderCell.SortGlyphDirection = conn.BoundColumnSortOrder(boundColumnIndex);
                    if (sortedColumn == null && dataGridViewColumn.HeaderCell.SortGlyphDirection != SortOrder.None)
                    {
                        sortedColumn = dataGridViewColumn;
                        sortOrder = dataGridViewColumn.HeaderCell.SortGlyphDirection;
                        // no need to sort because the back end is already sorted....
                    }
                }
            }
            else
            {
                dataGridViewColumn.IsDataBoundInternal = false;
                dataGridViewColumn.BoundColumnIndex = -1;
                dataGridViewColumn.BoundColumnConverter = null;
                InvalidateColumnInternal(dataGridViewColumn.Index);
            }
        }

        private void MoveColumnRelocation(MouseEventArgs e, HitTestInfo hti)
        {
            lastHeaderShadow = e.X;
            dataGridViewState2[DATAGRIDVIEWSTATE2_showColumnRelocationInsertion] = ColumnRelocationTarget(e, hti, out trackColumnEdge);
            Invalidate(Rectangle.Union(layout.TopLeftHeader, layout.ColumnHeaders));
        }

        private void MoveRowHeadersOrColumnResize(int x)
        {
            lastColSplitBar = currentColSplitBar;
            currentColSplitBar = x;
            Rectangle lastSplitBarRect = CalcColResizeFeedbackRect(lastColSplitBar);
            if (editingControl != null &&
                !dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlHidden] &&
                editingPanel.Bounds.IntersectsWith(lastSplitBarRect))
            {
                editingPanel.Invalidate();
                editingPanel.Update();
                editingControl.Invalidate();
                editingControl.Update();
            }
            Invalidate(lastSplitBarRect);
            Invalidate(CalcColResizeFeedbackRect(currentColSplitBar));
        }

        public virtual void NotifyCurrentCellDirty(bool dirty)
        {
            Debug.Assert(ptCurrentCell.X >= 0 && ptCurrentCell.X < Columns.Count);
            Debug.Assert(ptCurrentCell.Y >= 0 && ptCurrentCell.Y < Rows.Count);

            if (dataGridViewState1[DATAGRIDVIEWSTATE1_ignoringEditingChanges] == false)
            {
                // autosizing has no effect since edited value hasn't been committed
                // and autosizing code only looks at committed values.

                IsCurrentCellDirtyInternal = dirty;
                if (dirty && editingControl != null && ((IDataGridViewEditingControl)editingControl).RepositionEditingControlOnValueChange)
                {
                    PositionEditingControl(true /*setLocation*/, true /*setSize*/, false /*setFocus*/);
                }
            }
        }

        internal void OnAddedColumn(DataGridViewColumn dataGridViewColumn)
        {
            Debug.Assert(dataGridViewColumn.Index >= 0);
            Debug.Assert(dataGridViewColumn.Index < Columns.Count);
            Debug.Assert(dataGridViewColumn.DataGridView == this);

            if (dataGridViewColumn.DisplayIndex == -1 || dataGridViewColumn.DisplayIndex >= Columns.Count)
            {
                // Developer did not assign a DisplayIndex or picked a large number.
                // Choose the Index as the DisplayIndex.
                dataGridViewColumn.DisplayIndexInternal = dataGridViewColumn.Index;
                Columns.InvalidateCachedColumnsOrder();
            }

            CorrectColumnDisplayIndexesAfterInsertion(dataGridViewColumn);

            if (dataGridViewColumn.HasHeaderCell)
            {
                dataGridViewColumn.HeaderCell.DataGridView = this;
            }

            AdjustExpandingRows(dataGridViewColumn.Index, false /*fixedWidth*/);
            DataGridViewAutoSizeColumnMode autoSizeColumnMode = dataGridViewColumn.InheritedAutoSizeMode;
            Debug.Assert(autoSizeColumnMode != DataGridViewAutoSizeColumnMode.NotSet);
            bool fixedColumnWidth = autoSizeColumnMode == DataGridViewAutoSizeColumnMode.None ||
                                    autoSizeColumnMode == DataGridViewAutoSizeColumnMode.Fill;
            if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
            {
                AutoResizeColumnHeadersHeight(dataGridViewColumn.Index, true /*fixedRowHeadersWidth*/, fixedColumnWidth);
            }

            if (!fixedColumnWidth)
            {
                // This is the first time the column autosizes. Save current column width for later reuse.
                dataGridViewColumn.CachedThickness = dataGridViewColumn.Thickness;
                AutoResizeColumnInternal(dataGridViewColumn.Index, (DataGridViewAutoSizeColumnCriteriaInternal)autoSizeColumnMode, true /*fixedHeight*/);

                if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
                {
                    // Second round of column headers autosizing
                    AutoResizeColumnHeadersHeight(dataGridViewColumn.Index, true /*fixedRowHeadersWidth*/, true /*fixedColumnWidth*/);
                }
            }

            // Raise the ColumnAdded event
            OnColumnAdded(new DataGridViewColumnEventArgs(dataGridViewColumn));
        }

        internal void OnAddedRow_PreNotification(int rowIndex)
        {
            Debug.Assert(rowIndex >= 0);

            if (AllowUserToAddRowsInternal && newRowIndex == -1)
            {
                // The added row is necessarily the 'new row'
                // Set the this.newRowIndex variable as early as possible.
                Debug.Assert(rowIndex == Rows.Count - 1);
                newRowIndex = rowIndex;
            }

            DataGridViewElementStates rowState = Rows.GetRowState(rowIndex);

#if DEBUG
            DataGridViewRow dataGridViewRowDebug = Rows.SharedRow(rowIndex);
            foreach (DataGridViewCell dataGridViewCell in dataGridViewRowDebug.Cells)
            {
                Debug.Assert(!dataGridViewCell.Selected);
                Debug.Assert(dataGridViewRowDebug.Index != -1 || !dataGridViewCell.HasValue);
            }
#endif

            // Update this.individualReadOnlyCells
            if ((rowState & DataGridViewElementStates.ReadOnly) == 0 &&
                !ReadOnly)
            {
                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                foreach (DataGridViewCell dataGridViewCell in dataGridViewRow.Cells)
                {
                    if (!dataGridViewCell.OwningColumn.ReadOnly && IsSharedCellReadOnly(dataGridViewCell, rowIndex))
                    {
                        individualReadOnlyCells.Add(dataGridViewCell);
                    }
                }
            }
        }

        internal void OnAddedRow_PostNotification(int rowIndex)
        {
            Debug.Assert(rowIndex >= 0);

            DataGridViewElementStates rowState = Rows.GetRowState(rowIndex);

            if ((rowState & DataGridViewElementStates.Visible) != 0)
            {
                bool rowDisplayed = (rowState & DataGridViewElementStates.Displayed) != 0;

                DataGridViewAutoSizeRowsModeInternal autoSizeRowsModeInternal = (DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode;

                bool autoSizeRow = false;
                // Auto size row if needed
                if (autoSizeRowsModeInternal != DataGridViewAutoSizeRowsModeInternal.None &&
                    !((autoSizeRowsModeInternal & DataGridViewAutoSizeRowsModeInternal.DisplayedRows) != 0 && !rowDisplayed))
                {
                    // this call may unshare the row.
                    int rowHeight = Rows.SharedRow(rowIndex).GetHeight(rowIndex);
                    Rows.SharedRow(rowIndex).CachedThickness = rowHeight;
                    AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), false /*fixedWidth*/, true /*internalAutosizing*/);
                    autoSizeRow = true;
                }

                // Auto size columms also if needed
                DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaFilter = DataGridViewAutoSizeColumnCriteriaInternal.AllRows;
                if (rowDisplayed)
                {
                    autoSizeColumnCriteriaFilter |= DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows;
                }
                bool columnAutoSized;
                if (Rows.GetRowCount(DataGridViewElementStates.Visible) > 1)
                {
                    // Columns can only expand, and not collapse.
                    columnAutoSized = AdjustExpandingColumns(autoSizeColumnCriteriaFilter, rowIndex);
                }
                else
                {
                    columnAutoSized = AutoResizeAllVisibleColumnsInternal(autoSizeColumnCriteriaFilter, true /*fixedHeight*/);
                }
                bool fixedColumnHeadersHeight = ColumnHeadersHeightSizeMode != DataGridViewColumnHeadersHeightSizeMode.AutoSize;
                bool rowHeadersAutoSize = rowHeadersWidthSizeMode != DataGridViewRowHeadersWidthSizeMode.EnableResizing &&
                                          rowHeadersWidthSizeMode != DataGridViewRowHeadersWidthSizeMode.DisableResizing;

                if (!rowHeadersAutoSize && !columnAutoSized)
                {
                    // No need to autosize the column headers when the row headers and columns don't change.
                    fixedColumnHeadersHeight = true;
                }

                // Auto size row headers
                if (rowHeadersAutoSize)
                {
                    AutoResizeRowHeadersWidth(rowIndex, rowHeadersWidthSizeMode, fixedColumnHeadersHeight, true /*fixedRowsHeight*/);
                }

                // Auto size column headers
                if (!fixedColumnHeadersHeight)
                {
                    AutoResizeColumnHeadersHeight(true /*fixedRowHeadersWidth*/, true /*fixedColumnsWidth*/);
                }

                if (autoSizeRow)
                {
                    // Second round of row autosizing
                    AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), true /*fixedWidth*/, true /*internalAutosizing*/);
                }

                if (rowHeadersAutoSize && !fixedColumnHeadersHeight)
                {
                    // Second round of row headers autosizing
                    AutoResizeRowHeadersWidth(rowIndex, rowHeadersWidthSizeMode, true /*fixedColumnHeadersHeight*/, true /*fixedRowsHeight*/);
                }
            }
        }

        internal void OnAddedRows_PreNotification(DataGridViewRow[] dataGridViewRows)
        {
            // Note: no row can be added that breaks the frozen row packing on the top

            foreach (DataGridViewRow dataGridViewRow in dataGridViewRows)
            {
                OnAddedRow_PreNotification(dataGridViewRow.Index);
            }
        }

        internal void OnAddedRows_PostNotification(DataGridViewRow[] dataGridViewRows)
        {
            foreach (DataGridViewRow dataGridViewRow in dataGridViewRows)
            {
                OnAddedRow_PostNotification(dataGridViewRow.Index);
            }
        }

        internal void OnAddingColumn(DataGridViewColumn dataGridViewColumn)
        {
            // throw an exception if the column to be added breaks the rules
            if (dataGridViewColumn == null)
            {
                throw new ArgumentNullException(nameof(dataGridViewColumn));
            }
            if (dataGridViewColumn.DataGridView != null)
            {
                throw new InvalidOperationException(SR.DataGridView_ColumnAlreadyBelongsToDataGridView);
            }

            if (!InInitialization &&
                dataGridViewColumn.SortMode == DataGridViewColumnSortMode.Automatic &&
                (SelectionMode == DataGridViewSelectionMode.FullColumnSelect ||
                 SelectionMode == DataGridViewSelectionMode.ColumnHeaderSelect))
            {
                throw new InvalidOperationException(string.Format(SR.DataGridViewColumn_SortModeAndSelectionModeClash, DataGridViewColumnSortMode.Automatic.ToString(), SelectionMode.ToString()));
            }

            if (dataGridViewColumn.Visible)
            {
                // Note that dataGridViewColumn.DataGridView is set later on, so dataGridViewColumn.InheritedAutoSizeMode should not be used

                // Make sure the column does not autosize based only on header while column headers are invisible
                if (!ColumnHeadersVisible &&
                    (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.ColumnHeader || (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.NotSet && AutoSizeColumnsMode == DataGridViewAutoSizeColumnsMode.ColumnHeader)))
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAddAutoSizedColumn);
                }

                // Make sure the column is not frozen and auto fills
                if (dataGridViewColumn.Frozen &&
                    (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.Fill || (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.NotSet && AutoSizeColumnsMode == DataGridViewAutoSizeColumnsMode.Fill)))
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAddAutoFillColumn);
                }

                // UsedFillWeight values need to be updated
                dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty] = true;
            }

            // Make sure the sum of the column weights does not exceed ushort.MaxValue
            float weightSum = Columns.GetColumnsFillWeight(DataGridViewElementStates.None) + dataGridViewColumn.FillWeight;
            if (weightSum > (float)ushort.MaxValue)
            {
                throw new InvalidOperationException(string.Format(SR.DataGridView_WeightSumCannotExceedLongMaxValue, ushort.MaxValue));
            }

            // check for correctness of frozen state - throws exception if state is incorrect.
            CorrectColumnFrozenState(dataGridViewColumn, Columns.Count);

            // prepare the existing rows by appending cells of correct type
            if (Rows.Count > 0)
            {
                // Only require a default cell type when there are rows to fill
                if (dataGridViewColumn.CellType == null)
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAddUntypedColumn);
                }

                if (dataGridViewColumn.CellTemplate.DefaultNewRowValue != null && newRowIndex != -1)
                {
                    // New row needs to be unshared before addition of new cell with a Value != null
                    DataGridViewRow newRow = Rows[newRowIndex];
                }

                int newColumnCount = Columns.Count + 1;

                try
                {
                    for (int rowIndex = 0; rowIndex < Rows.Count; rowIndex++)
                    {
                        DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                        if (dataGridViewRow.Cells.Count < newColumnCount)
                        {
                            DataGridViewCell dataGridViewCellNew = (DataGridViewCell)dataGridViewColumn.CellTemplate.Clone();
                            dataGridViewRow.Cells.AddInternal(dataGridViewCellNew);
                            if (rowIndex == newRowIndex)
                            {
                                dataGridViewCellNew.SetValueInternal(rowIndex, dataGridViewCellNew.DefaultNewRowValue);
                            }
                            dataGridViewCellNew.DataGridView = this;
                            dataGridViewCellNew.OwningRow = dataGridViewRow;
                            dataGridViewCellNew.OwningColumn = dataGridViewColumn;

                            KeyboardToolTipStateMachine.Instance.Hook(dataGridViewCellNew, KeyboardToolTip);
                        }
                    }
                }
                catch
                {
                    for (int rowIndex = 0; rowIndex < Rows.Count; rowIndex++)
                    {
                        DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                        if (dataGridViewRow.Cells.Count == newColumnCount)
                        {
                            dataGridViewRow.Cells.RemoveAtInternal(newColumnCount - 1);
                        }
                        else
                        {
                            Debug.Assert(dataGridViewRow.Cells.Count < newColumnCount);
                            break;
                        }
                    }
                    throw;
                }
            }
        }

        internal void OnAddingColumns(DataGridViewColumn[] dataGridViewColumns)
        {
            // Make sure the sum of the column weights does not exceed ushort.MaxValue
            float weightSum = Columns.GetColumnsFillWeight(DataGridViewElementStates.None);
            Debug.Assert(weightSum <= (float)ushort.MaxValue);

            // throw an exception if any of the columns to be added breaks the rules
            Debug.Assert(dataGridViewColumns != null);
            foreach (DataGridViewColumn dataGridViewColumn in dataGridViewColumns)
            {
                if (dataGridViewColumn == null)
                {
                    throw new InvalidOperationException(SR.DataGridView_AtLeastOneColumnIsNull);
                }
                if (dataGridViewColumn.DataGridView != null)
                {
                    throw new InvalidOperationException(SR.DataGridView_ColumnAlreadyBelongsToDataGridView);
                }
                // Only require a default cell type when there are rows to fill
                if (Rows.Count > 0 && dataGridViewColumn.CellType == null)
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAddUntypedColumn);
                }

                if (!InInitialization &&
                    dataGridViewColumn.SortMode == DataGridViewColumnSortMode.Automatic &&
                    (SelectionMode == DataGridViewSelectionMode.FullColumnSelect ||
                     SelectionMode == DataGridViewSelectionMode.ColumnHeaderSelect))
                {
                    throw new InvalidOperationException(string.Format(SR.DataGridViewColumn_SortModeAndSelectionModeClash, DataGridViewColumnSortMode.Automatic.ToString(), SelectionMode.ToString()));
                }

                if (dataGridViewColumn.Visible)
                {
                    // Note that dataGridViewColumn.DataGridView is set later on, so dataGridViewColumn.InheritedAutoSizeMode should not be used

                    // Make sure the column does not autosize based only on header while column headers are invisible
                    if (!ColumnHeadersVisible &&
                        (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.ColumnHeader || (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.NotSet && AutoSizeColumnsMode == DataGridViewAutoSizeColumnsMode.ColumnHeader)))
                    {
                        throw new InvalidOperationException(SR.DataGridView_CannotAddAutoSizedColumn);
                    }

                    // Make sure the column is not frozen and auto fills
                    if (dataGridViewColumn.Frozen &&
                        (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.Fill || (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.NotSet && AutoSizeColumnsMode == DataGridViewAutoSizeColumnsMode.Fill)))
                    {
                        throw new InvalidOperationException(SR.DataGridView_CannotAddAutoFillColumn);
                    }

                    // UsedFillWeight values need to be updated
                    dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty] = true;
                }

                weightSum += dataGridViewColumn.FillWeight;
                if (weightSum > (float)ushort.MaxValue)
                {
                    throw new InvalidOperationException(string.Format(SR.DataGridView_WeightSumCannotExceedLongMaxValue, ushort.MaxValue));
                }
            }

            Debug.Assert(weightSum <= (float)ushort.MaxValue);

            // make sure no two columns are identical
            int columnCount = dataGridViewColumns.Length;
            for (int column1 = 0; column1 < columnCount - 1; column1++)
            {
                for (int column2 = column1 + 1; column2 < columnCount; column2++)
                {
                    if (dataGridViewColumns[column1] == dataGridViewColumns[column2])
                    {
                        throw new InvalidOperationException(SR.DataGridView_CannotAddIdenticalColumns);
                    }
                }
            }

            // check for correctness of frozen states - throws exception if any state is incorrect.
            CorrectColumnFrozenStates(dataGridViewColumns);

            // prepare the existing rows by appending cells of correct type
            if (Rows.Count > 0)
            {
                foreach (DataGridViewColumn dataGridViewColumn in dataGridViewColumns)
                {
                    Debug.Assert(dataGridViewColumn.CellType != null);
                    if (dataGridViewColumn.CellTemplate.DefaultNewRowValue != null && newRowIndex != -1)
                    {
                        // New row needs to be unshared before addition of new cell with a Value != null
                        DataGridViewRow newRow = Rows[newRowIndex];
                        break;
                    }
                }

                int previousColumnCount = Columns.Count;
                int addedColumnCount = 0;

                try
                {
                    foreach (DataGridViewColumn dataGridViewColumn in dataGridViewColumns)
                    {
                        addedColumnCount++;
                        Debug.Assert(dataGridViewColumn.CellType != null);
                        for (int rowIndex = 0; rowIndex < Rows.Count; rowIndex++)
                        {
                            DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                            if (dataGridViewRow.Cells.Count < previousColumnCount + addedColumnCount)
                            {
                                DataGridViewCell dataGridViewCellNew = (DataGridViewCell)dataGridViewColumn.CellTemplate.Clone();
                                int indexCell = dataGridViewRow.Cells.AddInternal(dataGridViewCellNew);
                                if (rowIndex == newRowIndex)
                                {
                                    dataGridViewCellNew.Value = dataGridViewCellNew.DefaultNewRowValue;
                                }
                                dataGridViewCellNew.DataGridView = this;
                                dataGridViewCellNew.OwningRow = dataGridViewRow;
                                dataGridViewCellNew.OwningColumn = dataGridViewColumn;
                            }
                        }
                    }
                }
                catch
                {
                    for (int rowIndex = 0; rowIndex < Rows.Count; rowIndex++)
                    {
                        DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                        while (dataGridViewRow.Cells.Count > previousColumnCount)
                        {
                            dataGridViewRow.Cells.RemoveAtInternal(dataGridViewRow.Cells.Count - 1);
                        }
                    }
                    throw;
                }
            }
        }

        internal void OnAddingRow(DataGridViewRow dataGridViewRow, DataGridViewElementStates rowState, bool checkFrozenState)
        {
            // Note dataGridViewRow.DataGridView != null for duplication of shared rows.

            // throw an exception if the row to be added breaks the rules
            if (dataGridViewRow == null)
            {
                throw new ArgumentNullException(nameof(dataGridViewRow));
            }

            // !Do not check for dataGridViewRow.Selected flag. Caller does it instead!
            // !Do not check for dataGridViewRow.DataGridView != null. Caller does it instead!

            if (checkFrozenState)
            {
                // check for correctness of frozen state - throws exception if state is incorrect.
                CorrectRowFrozenState(dataGridViewRow, rowState, Rows.Count);
            }

            if (ReadOnly && dataGridViewRow.DataGridView == null && dataGridViewRow.ReadOnly)
            {
                // Clear the superfluous flag since the whole dataGridView is read-only
                dataGridViewRow.ReadOnly = false;
            }

            int columnIndex = 0;
            foreach (DataGridViewColumn dataGridViewColumn in Columns)
            {
                DataGridViewCell dataGridViewCell = dataGridViewRow.Cells[columnIndex];
                if ((ReadOnly || dataGridViewColumn.ReadOnly) && dataGridViewCell.StateIncludes(DataGridViewElementStates.ReadOnly))
                {
                    // Clear superfluous flag since the whole dataGridView or column is ReadOnly
                    dataGridViewCell.ReadOnlyInternal = false;
                }

                KeyboardToolTipStateMachine.Instance.Hook(dataGridViewCell, KeyboardToolTip);
                columnIndex++;
            }
        }

        internal void OnAddingRows(DataGridViewRow[] dataGridViewRows, bool checkFrozenStates)
        {
            // throw an exception if any of the rows to be added breaks the rules
            Debug.Assert(dataGridViewRows != null);

            foreach (DataGridViewRow dataGridViewRow in dataGridViewRows)
            {
                if (dataGridViewRow == null)
                {
                    throw new InvalidOperationException(SR.DataGridView_AtLeastOneRowIsNull);
                }

                if (dataGridViewRow.DataGridView != null)
                {
                    throw new InvalidOperationException(SR.DataGridView_RowAlreadyBelongsToDataGridView);
                }

                if (dataGridViewRow.Selected)
                {
                    throw new InvalidOperationException(SR.DataGridViewRowCollection_CannotAddOrInsertSelectedRow);
                }

                if (dataGridViewRow.Cells.Count > Columns.Count)
                {
                    throw new InvalidOperationException(SR.DataGridViewRowCollection_TooManyCells);
                }
            }

            // make sure no two rows are identical
            int rowCount = dataGridViewRows.Length;
            for (int row1 = 0; row1 < rowCount - 1; row1++)
            {
                for (int row2 = row1 + 1; row2 < rowCount; row2++)
                {
                    if (dataGridViewRows[row1] == dataGridViewRows[row2])
                    {
                        throw new InvalidOperationException(SR.DataGridView_CannotAddIdenticalRows);
                    }
                }
            }

            if (checkFrozenStates)
            {
                Debug.Assert(!AllowUserToAddRowsInternal);
                CorrectRowFrozenStates(dataGridViewRows, Rows.Count /*rowIndexInserted*/);
            }

            foreach (DataGridViewRow dataGridViewRow in dataGridViewRows)
            {
                CompleteCellsCollection(dataGridViewRow);
                OnAddingRow(dataGridViewRow, dataGridViewRow.State, false /*checkFrozenState*/);
            }
        }

        internal void OnAdvancedBorderStyleChanged(DataGridViewAdvancedBorderStyle dgvabs)
        {
            if (!dataGridViewOper[DATAGRIDVIEWOPER_inBorderStyleChange])
            {
                if (dgvabs == advancedCellBorderStyle)
                {
                    OnCellBorderStyleChanged(EventArgs.Empty);
                }
                else if (dgvabs == advancedColumnHeadersBorderStyle)
                {
                    OnColumnHeadersBorderStyleChanged(EventArgs.Empty);
                }
                else if (dgvabs == advancedRowHeadersBorderStyle)
                {
                    OnRowHeadersBorderStyleChanged(EventArgs.Empty);
                }
            }
        }

        protected virtual void OnAllowUserToAddRowsChanged(EventArgs e)
        {
            PushAllowUserToAddRows();

            if (Events[EVENT_DATAGRIDVIEWALLOWUSERTOADDROWSCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnAllowUserToDeleteRowsChanged(EventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWALLOWUSERTODELETEROWSCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnAllowUserToOrderColumnsChanged(EventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWALLOWUSERTOORDERCOLUMNSCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnAllowUserToResizeColumnsChanged(EventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWALLOWUSERTORESIZECOLUMNSCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnAllowUserToResizeRowsChanged(EventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWALLOWUSERTORESIZEROWSCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnAlternatingRowsDefaultCellStyleChanged(EventArgs e)
        {
            if (e is DataGridViewCellStyleChangedEventArgs dgvcsce && !dgvcsce.ChangeAffectsPreferredSize)
            {
                InvalidateData();
            }
            else
            {
                OnRowsGlobalAutoSize();
                if (editingControl != null)
                {
                    PositionEditingControl(true /*setLocation*/, true /*setSize*/, false /*setFocus*/);
                }
            }

            if (Events[EVENT_DATAGRIDVIEWALTERNATINGROWSDEFAULTCELLSTYLECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnAutoGenerateColumnsChanged(EventArgs e)
        {
            if (AutoGenerateColumns && DataSource != null)
            {
                // refresh the list of columns and the rows
                RefreshColumnsAndRows();
            }
            if (Events[EVENT_DATAGRIDVIEWAUTOGENERATECOLUMNSCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnAutoSizeColumnModeChanged(DataGridViewColumn dataGridViewColumn, DataGridViewAutoSizeColumnMode previousInheritedMode)
        {
            Debug.Assert(dataGridViewColumn != null);
            DataGridViewAutoSizeColumnModeEventArgs dgvascme = new DataGridViewAutoSizeColumnModeEventArgs(dataGridViewColumn, previousInheritedMode);
            OnAutoSizeColumnModeChanged(dgvascme);
        }

        protected virtual void OnAutoSizeColumnModeChanged(DataGridViewAutoSizeColumnModeEventArgs e)
        {
            DataGridViewColumn dataGridViewColumn = e.Column;
            if (e.Column == null)
            {
                throw new InvalidOperationException(string.Format(SR.InvalidNullArgument, "e.Column"));
            }
            DataGridViewAutoSizeColumnMode autoSizeColumnMode = dataGridViewColumn.InheritedAutoSizeMode;
            Debug.Assert(autoSizeColumnMode != DataGridViewAutoSizeColumnMode.NotSet);

            DataGridViewAutoSizeColumnMode previousInheritedMode = e.PreviousMode;
            bool previousInheritedModeAutoSized = previousInheritedMode != DataGridViewAutoSizeColumnMode.Fill &&
                                                  previousInheritedMode != DataGridViewAutoSizeColumnMode.None &&
                                                  previousInheritedMode != DataGridViewAutoSizeColumnMode.NotSet;

            if (autoSizeColumnMode == DataGridViewAutoSizeColumnMode.Fill ||
                previousInheritedMode == DataGridViewAutoSizeColumnMode.Fill)
            {
                // UsedFillWeight values need to be updated
                dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty] = true;
            }

            bool fixedHeight = (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) == 0;

            if (autoSizeColumnMode != DataGridViewAutoSizeColumnMode.None)
            {
                if (autoSizeColumnMode != DataGridViewAutoSizeColumnMode.Fill)
                {
                    if (!previousInheritedModeAutoSized)
                    {
                        // Save current column width for later reuse
                        dataGridViewColumn.CachedThickness = dataGridViewColumn.Thickness;
                    }
                    AutoResizeColumnInternal(dataGridViewColumn.Index,
                                             (DataGridViewAutoSizeColumnCriteriaInternal)autoSizeColumnMode,
                                             fixedHeight);
                }
            }
            else if (dataGridViewColumn.Thickness != dataGridViewColumn.CachedThickness && previousInheritedModeAutoSized)
            {
                // Restoring cached column width
                dataGridViewColumn.ThicknessInternal = Math.Max(dataGridViewColumn.MinimumWidth, dataGridViewColumn.CachedThickness);
            }

            // Auto fill columns if needed
            PerformLayoutPrivate(false /*useRowShortcut*/, true /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);

            // Autosize rows and column headers if needed
            if (!fixedHeight)
            {
                AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
                if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
                {
                    AutoResizeColumnHeadersHeight(dataGridViewColumn.Index, true /*fixedRowHeadersWidth*/, true /*fixedColumnWidth*/);
                }

                // Column gets autosized with 1 degree of freedom this time.
                if (autoSizeColumnMode != DataGridViewAutoSizeColumnMode.None && autoSizeColumnMode != DataGridViewAutoSizeColumnMode.Fill)
                {
                    AutoResizeColumnInternal(dataGridViewColumn.Index,
                                             (DataGridViewAutoSizeColumnCriteriaInternal)autoSizeColumnMode,
                                             true /*fixedHeight*/);
                }
            }

            if (Events[EVENT_DATAGRIDVIEWAUTOSIZECOLUMNMODECHANGED] is DataGridViewAutoSizeColumnModeEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnAutoSizeColumnsModeChanged(DataGridViewAutoSizeColumnsModeEventArgs e)
        {
            DataGridViewAutoSizeColumnMode[] previousModes = e.PreviousModes;
            if (previousModes == null)
            {
                throw new ArgumentNullException(nameof(e.PreviousModes));
            }
            if (previousModes.Length != Columns.Count)
            {
                throw new ArgumentException(SR.DataGridView_PreviousModesHasWrongLength);
            }

            foreach (DataGridViewColumn dataGridViewColumn in Columns)
            {
                if (dataGridViewColumn.Visible)
                {
                    DataGridViewAutoSizeColumnMode autoSizeColumnMode = dataGridViewColumn.InheritedAutoSizeMode;
                    Debug.Assert(autoSizeColumnMode != DataGridViewAutoSizeColumnMode.NotSet);
                    DataGridViewAutoSizeColumnMode previousInheritedMode = previousModes[dataGridViewColumn.Index];
                    bool previousInheritedModeAutoSized = previousInheritedMode != DataGridViewAutoSizeColumnMode.Fill &&
                                                          previousInheritedMode != DataGridViewAutoSizeColumnMode.None &&
                                                          previousInheritedMode != DataGridViewAutoSizeColumnMode.NotSet;
                    if (autoSizeColumnMode == DataGridViewAutoSizeColumnMode.Fill ||
                        previousInheritedMode == DataGridViewAutoSizeColumnMode.Fill)
                    {
                        // UsedFillWeight values need to be updated
                        dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty] = true;
                    }

                    if (autoSizeColumnMode != DataGridViewAutoSizeColumnMode.None)
                    {
                        if (autoSizeColumnMode != DataGridViewAutoSizeColumnMode.Fill)
                        {
                            if (!previousInheritedModeAutoSized)
                            {
                                // Save current column width for later reuse
                                dataGridViewColumn.CachedThickness = dataGridViewColumn.Thickness;
                            }
                            AutoResizeColumnInternal(dataGridViewColumn.Index,
                                                     (DataGridViewAutoSizeColumnCriteriaInternal)autoSizeColumnMode,
                                                     (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) == 0 /*fixedHeight*/);
                        }
                    }
                    else if (dataGridViewColumn.Thickness != dataGridViewColumn.CachedThickness && previousInheritedModeAutoSized)
                    {
                        // Restoring cached column width
                        dataGridViewColumn.ThicknessInternal = Math.Max(dataGridViewColumn.MinimumWidth, dataGridViewColumn.CachedThickness);
                    }
                }
            }

            // Auto fill columns if needed
            PerformLayoutPrivate(false /*useRowShortcut*/, true /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);

            // Autosize rows and column headers if needed
            if ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0)
            {
                AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
                if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
                {
                    AutoResizeColumnHeadersHeight(true /*fixedRowHeadersWidth*/, true /*fixedColumnWidth*/);
                }

                // Second pass of column autosizing with 1 degree of freedom
                foreach (DataGridViewColumn dataGridViewColumn in Columns)
                {
                    DataGridViewAutoSizeColumnMode autoSizeColumnMode = dataGridViewColumn.InheritedAutoSizeMode;
                    Debug.Assert(autoSizeColumnMode != DataGridViewAutoSizeColumnMode.NotSet);

                    if (autoSizeColumnMode != DataGridViewAutoSizeColumnMode.None && autoSizeColumnMode != DataGridViewAutoSizeColumnMode.Fill)
                    {
                        AutoResizeColumnInternal(dataGridViewColumn.Index,
                                                 (DataGridViewAutoSizeColumnCriteriaInternal)autoSizeColumnMode,
                                                 true /*fixedHeight*/);
                    }
                }
            }

            if (Events[EVENT_DATAGRIDVIEWAUTOSIZECOLUMNSMODECHANGED] is DataGridViewAutoSizeColumnsModeEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnAutoSizeRowsModeChanged(DataGridViewAutoSizeModeEventArgs e)
        {
            if (autoSizeRowsMode == DataGridViewAutoSizeRowsMode.None)
            {
                // restore cached rows thickness
                RestoreRowsCachedThickness();
            }
            else
            {
                if (!e.PreviousModeAutoSized)
                {
                    // Save the rows thickness for later reuse
                    // Note that only visible rows are affected, contrary to columns in OnAutoSizeColumnsModeChanged where all columns are affected.
                    for (int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                        rowIndex != -1;
                        rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible))
                    {
                        // this call may unshare the row.
                        int rowHeight = Rows.SharedRow(rowIndex).GetHeight(rowIndex);
                        Rows.SharedRow(rowIndex).CachedThickness = rowHeight;
                    }
                }
                AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
            }

            if (Events[EVENT_DATAGRIDVIEWAUTOSIZEROWSMODECHANGED] is DataGridViewAutoSizeModeEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnBackgroundColorChanged(EventArgs e)
        {
            InvalidateInside();

            if (Events[EVENT_DATAGRIDVIEWBACKGROUNDCOLORCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnBandContextMenuStripChanged(DataGridViewBand dataGridViewBand)
        {
            if (dataGridViewBand is DataGridViewColumn dataGridViewColumn)
            {
                DataGridViewColumnEventArgs dgvce = new DataGridViewColumnEventArgs(dataGridViewColumn);
                OnColumnContextMenuStripChanged(dgvce);
            }
            else
            {
                DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs((DataGridViewRow)dataGridViewBand);
                OnRowContextMenuStripChanged(dgvre);
            }
        }

        internal void OnBandDefaultCellStyleChanged(DataGridViewBand dataGridViewBand)
        {
            if (dataGridViewBand is DataGridViewColumn dataGridViewColumn)
            {
                DataGridViewColumnEventArgs dgvce = new DataGridViewColumnEventArgs(dataGridViewColumn);
                OnColumnDefaultCellStyleChanged(dgvce);
            }
            else
            {
                DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs((DataGridViewRow)dataGridViewBand);
                OnRowDefaultCellStyleChanged(dgvre);
            }
        }

        internal void OnBandDividerThicknessChanged(DataGridViewBand dataGridViewBand)
        {
            if (dataGridViewBand is DataGridViewColumn dataGridViewColumn)
            {
                DataGridViewColumnEventArgs dgvce = new DataGridViewColumnEventArgs(dataGridViewColumn);
                OnColumnDividerWidthChanged(dgvce);
            }
            else
            {
                DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs((DataGridViewRow)dataGridViewBand);
                OnRowDividerHeightChanged(dgvre);
            }
        }

        internal void OnBandHeaderCellChanged(DataGridViewBand dataGridViewBand)
        {
            if (dataGridViewBand is DataGridViewColumn dataGridViewColumn)
            {
                DataGridViewColumnEventArgs dgvce = new DataGridViewColumnEventArgs(dataGridViewColumn);
                OnColumnHeaderCellChanged(dgvce);
            }
            else
            {
                DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs((DataGridViewRow)dataGridViewBand);
                OnRowHeaderCellChanged(dgvre);
            }
        }

        internal void OnBandMinimumThicknessChanged(DataGridViewBand dataGridViewBand)
        {
            if (dataGridViewBand is DataGridViewColumn dataGridViewColumn)
            {
                DataGridViewColumnEventArgs dgvce = new DataGridViewColumnEventArgs(dataGridViewColumn);
                OnColumnMinimumWidthChanged(dgvce);
            }
            else
            {
                DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs((DataGridViewRow)dataGridViewBand);
                OnRowMinimumHeightChanged(dgvre);
            }
        }

        internal void OnBandThicknessChanged(DataGridViewBand dataGridViewBand)
        {
            if (dataGridViewBand is DataGridViewColumn dataGridViewColumn)
            {
                DataGridViewColumnEventArgs dgvce = new DataGridViewColumnEventArgs(dataGridViewColumn);
                OnColumnWidthChanged(dgvce);
            }
            else
            {
                DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs((DataGridViewRow)dataGridViewBand);
                OnRowHeightChanged(dgvre);
            }
        }

        internal void OnBandThicknessChanging()
        {
            if (InAdjustFillingColumns)
            {
                throw new InvalidOperationException(SR.DataGridView_CannotAlterAutoFillColumnParameter);
            }
        }

        protected override void OnBindingContextChanged(EventArgs e)
        {
            if (dataGridViewState2[DATAGRIDVIEWSTATE2_inBindingContextChanged])
            {
                return;
            }

            dataGridViewState2[DATAGRIDVIEWSTATE2_inBindingContextChanged] = true;
            try
            {
                if (dataConnection != null)
                {
                    CurrentCell = null;
                    try
                    {
                        dataConnection.SetDataConnection(DataSource, DataMember);
                    }
                    catch (ArgumentException)
                    {
                        if (DesignMode)
                        {
                            // If the DataMember became invalid at DesignTime then set it to String.Empty,
                            // regenerate the column collection and DO NOT send BindingContextChanged event.
                            DataMember = string.Empty;
                            RefreshColumnsAndRows();
                            return;
                        }
                        else
                        {
                            throw;
                        }
                    }
                    RefreshColumnsAndRows();
                    base.OnBindingContextChanged(e);
                    if (dataConnection.CurrencyManager != null)
                    {
                        OnDataBindingComplete(ListChangedType.Reset);
                    }
                }
                else
                {
                    base.OnBindingContextChanged(e);
                }
            }
            finally
            {
                dataGridViewState2[DATAGRIDVIEWSTATE2_inBindingContextChanged] = false;
            }
        }

        protected virtual void OnBorderStyleChanged(EventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWBORDERSTYLECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnCancelRowEdit(QuestionEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWCANCELROWEDIT] is QuestionEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
                CorrectFocus(true /*onlyIfGridHasFocus*/);
            }
        }

        protected virtual void OnCellBeginEdit(DataGridViewCellCancelEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            if (Events[EVENT_DATAGRIDVIEWCELLBEGINEDIT] is DataGridViewCellCancelEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnCellBorderStyleChanged(EventArgs e)
        {
            PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
            Invalidate();

            if (Events[EVENT_DATAGRIDVIEWCELLBORDERSTYLECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnCellClickInternal(DataGridViewCellEventArgs e)
        {
            OnCellClick(e);
        }

        protected virtual void OnCellClick(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            DataGridViewCell dataGridViewCell = GetCellInternal(e.ColumnIndex, e.RowIndex);
            Debug.Assert(dataGridViewCell != null);
            if (e.RowIndex >= 0 && dataGridViewCell.ClickUnsharesRowInternal(e))
            {
                DataGridViewRow dataGridViewRow = Rows[e.RowIndex];
                GetCellInternal(e.ColumnIndex, e.RowIndex).OnClickInternal(e);
            }
            else
            {
                dataGridViewCell.OnClickInternal(e);
            }

            if (Events[EVENT_DATAGRIDVIEWCELLCLICK] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnCellCommonChange(int columnIndex, int rowIndex)
        {
            if (columnIndex == -1)
            {
                // row or topleft header characteristic has changed
                OnRowHeaderGlobalAutoSize(rowIndex);
            }
            else
            {
                if (rowIndex == -1)
                {
                    // column header characteristic has changed
                    OnColumnHeaderGlobalAutoSize(columnIndex);
                }
                else
                {
                    // regular cell characteristic changed
                    InvalidateCellPrivate(columnIndex, rowIndex);

                    bool rowDisplayed = false;
                    if (rowIndex != -1)
                    {
                        rowDisplayed = (Rows.GetRowState(rowIndex) & DataGridViewElementStates.Displayed) != 0;
                    }

                    DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaInternal = (DataGridViewAutoSizeColumnCriteriaInternal)Columns[columnIndex].InheritedAutoSizeMode;
                    bool autoSizeColumn = (autoSizeColumnCriteriaInternal & DataGridViewAutoSizeColumnCriteriaInternal.AllRows) != 0;
                    if (rowDisplayed)
                    {
                        autoSizeColumn |= (autoSizeColumnCriteriaInternal & DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows) != 0;
                    }

                    bool autoSizeRow = (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0;
                    if (autoSizeRow)
                    {
                        AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), !autoSizeColumn /*fixedWidth*/, true /*internalAutosizing*/);
                    }
                    if (autoSizeColumn)
                    {
                        AutoResizeColumnInternal(columnIndex, autoSizeColumnCriteriaInternal, true /*fixedHeight*/);
                        if (autoSizeRow)
                        {
                            // Second round of row autosizing with 1 degree of freedom.
                            AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), true /*fixedWidth*/, true /*internalAutosizing*/);
                        }
                    }
                }
            }
        }

        internal void OnCellContentClickInternal(DataGridViewCellEventArgs e)
        {
            OnCellContentClick(e);
        }

        protected virtual void OnCellContentClick(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            DataGridViewCell dataGridViewCell = GetCellInternal(e.ColumnIndex, e.RowIndex);
            Debug.Assert(dataGridViewCell != null);
            if (e.RowIndex >= 0 && dataGridViewCell.ContentClickUnsharesRowInternal(e))
            {
                DataGridViewRow dataGridViewRow = Rows[e.RowIndex];
                GetCellInternal(e.ColumnIndex, e.RowIndex).OnContentClickInternal(e);
            }
            else
            {
                dataGridViewCell.OnContentClickInternal(e);
            }

            if (Events[EVENT_DATAGRIDVIEWCELLCONTENTCLICK] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnCellContentDoubleClickInternal(DataGridViewCellEventArgs e)
        {
            OnCellContentDoubleClick(e);
        }

        protected virtual void OnCellContentDoubleClick(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            DataGridViewCell dataGridViewCell = GetCellInternal(e.ColumnIndex, e.RowIndex);
            Debug.Assert(dataGridViewCell != null);
            if (e.RowIndex >= 0 && dataGridViewCell.ContentDoubleClickUnsharesRowInternal(e))
            {
                DataGridViewRow dataGridViewRow = Rows[e.RowIndex];
                GetCellInternal(e.ColumnIndex, e.RowIndex).OnContentDoubleClickInternal(e);
            }
            else
            {
                dataGridViewCell.OnContentDoubleClickInternal(e);
            }

            if (Events[EVENT_DATAGRIDVIEWCELLCONTENTDOUBLECLICK] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnCellContextMenuStripChanged(DataGridViewCell dataGridViewCell)
        {
            DataGridViewCellEventArgs dgvce = new DataGridViewCellEventArgs(dataGridViewCell);
            OnCellContextMenuStripChanged(dgvce);
        }

        protected virtual void OnCellContextMenuStripChanged(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            if (Events[EVENT_DATAGRIDVIEWCELLCONTEXTMENUSTRIPCHANGED] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal ContextMenuStrip OnCellContextMenuStripNeeded(int columnIndex, int rowIndex, ContextMenuStrip contextMenuStrip)
        {
            DataGridViewCellContextMenuStripNeededEventArgs dgvccmsne = new DataGridViewCellContextMenuStripNeededEventArgs(columnIndex, rowIndex, contextMenuStrip);
            OnCellContextMenuStripNeeded(dgvccmsne);
            return dgvccmsne.ContextMenuStrip;
        }

        protected virtual void OnCellContextMenuStripNeeded(DataGridViewCellContextMenuStripNeededEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            if (Events[EVENT_DATAGRIDVIEWCELLCONTEXTMENUSTRIPNEEDED] is DataGridViewCellContextMenuStripNeededEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnCellDoubleClick(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            DataGridViewCell dataGridViewCell = GetCellInternal(e.ColumnIndex, e.RowIndex);
            Debug.Assert(dataGridViewCell != null);
            if (e.RowIndex >= 0 && dataGridViewCell.DoubleClickUnsharesRowInternal(e))
            {
                DataGridViewRow dataGridViewRow = Rows[e.RowIndex];
                GetCellInternal(e.ColumnIndex, e.RowIndex).OnDoubleClickInternal(e);
            }
            else
            {
                dataGridViewCell.OnDoubleClickInternal(e);
            }

            if (Events[EVENT_DATAGRIDVIEWCELLDOUBLECLICK] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnCellEndEdit(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            if (Events[EVENT_DATAGRIDVIEWCELLENDEDIT] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnCellEnter(ref DataGridViewCell dataGridViewCell, int columnIndex, int rowIndex)
        {
            OnCellEnter(new DataGridViewCellEventArgs(columnIndex, rowIndex));
            if (dataGridViewCell != null)
            {
                if (IsInnerCellOutOfBounds(columnIndex, rowIndex))
                {
                    dataGridViewCell = null;
                }
                else
                {
                    Debug.Assert(rowIndex < Rows.Count && columnIndex < Columns.Count);
                    dataGridViewCell = Rows.SharedRow(rowIndex).Cells[columnIndex];
                }
            }
        }

        protected virtual void OnCellEnter(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            try
            {
                noDimensionChangeCount++;

                if (Events[EVENT_DATAGRIDVIEWCELLENTER] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
                {
                    eh(this, e);
                }
            }
            finally
            {
                noDimensionChangeCount--;
                Debug.Assert(noDimensionChangeCount >= 0);
            }
        }

        internal void OnCellErrorTextChanged(DataGridViewCell dataGridViewCell)
        {
            Debug.Assert(dataGridViewCell.RowIndex >= -1);
            Debug.Assert(dataGridViewCell.ColumnIndex >= -1);
            DataGridViewCellEventArgs dgvce = new DataGridViewCellEventArgs(dataGridViewCell);
            OnCellErrorTextChanged(dgvce);
        }

        protected virtual void OnCellErrorTextChanged(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            UpdateCellErrorText(e.ColumnIndex, e.RowIndex);

            if (Events[EVENT_DATAGRIDVIEWCELLERRORTEXTCHANGED] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal string OnCellErrorTextNeeded(int columnIndex, int rowIndex, string errorText)
        {
            Debug.Assert(columnIndex >= 0);
            Debug.Assert(rowIndex >= 0);
            DataGridViewCellErrorTextNeededEventArgs dgvcetne = new DataGridViewCellErrorTextNeededEventArgs(columnIndex, rowIndex, errorText);
            OnCellErrorTextNeeded(dgvcetne);
            return dgvcetne.ErrorText;
        }

        protected virtual void OnCellErrorTextNeeded(DataGridViewCellErrorTextNeededEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            if (Events[EVENT_DATAGRIDVIEWCELLERRORTEXTNEEDED] is DataGridViewCellErrorTextNeededEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal DataGridViewCellFormattingEventArgs OnCellFormatting(int columnIndex, int rowIndex, object val, Type formattedValueType, DataGridViewCellStyle cellStyle)
        {
            DataGridViewCellFormattingEventArgs dgvcfe = new DataGridViewCellFormattingEventArgs(columnIndex,
                                                                                                 rowIndex,
                                                                                                 val,
                                                                                                 formattedValueType,
                                                                                                 cellStyle);
            OnCellFormatting(dgvcfe);
            return dgvcfe;
        }

        protected virtual void OnCellFormatting(DataGridViewCellFormattingEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            if (Events[EVENT_DATAGRIDVIEWCELLFORMATTING] is DataGridViewCellFormattingEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnCellLeave(ref DataGridViewCell dataGridViewCell, int columnIndex, int rowIndex)
        {
            OnCellLeave(new DataGridViewCellEventArgs(columnIndex, rowIndex));
            if (dataGridViewCell != null)
            {
                if (IsInnerCellOutOfBounds(columnIndex, rowIndex))
                {
                    dataGridViewCell = null;
                }
                else
                {
                    Debug.Assert(rowIndex < Rows.Count && columnIndex < Columns.Count);
                    dataGridViewCell = Rows.SharedRow(rowIndex).Cells[columnIndex];
                }
            }
        }

        protected virtual void OnCellLeave(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            try
            {
                noDimensionChangeCount++;

                if (Events[EVENT_DATAGRIDVIEWCELLLEAVE] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
                {
                    eh(this, e);
                }
            }
            finally
            {
                noDimensionChangeCount--;
                Debug.Assert(noDimensionChangeCount >= 0);
            }
        }

        protected virtual void OnCellMouseClick(DataGridViewCellMouseEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            DataGridViewCell dataGridViewCell = GetCellInternal(e.ColumnIndex, e.RowIndex);
            Debug.Assert(dataGridViewCell != null);
            if (e.RowIndex >= 0 && dataGridViewCell.MouseClickUnsharesRowInternal(e))
            {
                DataGridViewRow dataGridViewRow = Rows[e.RowIndex];
                GetCellInternal(e.ColumnIndex, e.RowIndex).OnMouseClickInternal(e);
            }
            else
            {
                dataGridViewCell.OnMouseClickInternal(e);
            }
            dataGridViewState2[DATAGRIDVIEWSTATE2_nextMouseUpIsDouble] = false;

            if (Events[EVENT_DATAGRIDVIEWCELLMOUSECLICK] is DataGridViewCellMouseEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnCellMouseDoubleClick(DataGridViewCellMouseEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            DataGridViewCell dataGridViewCell = GetCellInternal(e.ColumnIndex, e.RowIndex);
            Debug.Assert(dataGridViewCell != null);
            if (e.RowIndex >= 0 && dataGridViewCell.MouseDoubleClickUnsharesRowInternal(e))
            {
                DataGridViewRow dataGridViewRow = Rows[e.RowIndex];
                GetCellInternal(e.ColumnIndex, e.RowIndex).OnMouseDoubleClickInternal(e);
            }
            else
            {
                dataGridViewCell.OnMouseDoubleClickInternal(e);
            }
            dataGridViewState2[DATAGRIDVIEWSTATE2_nextMouseUpIsDouble] = true;

            if (Events[EVENT_DATAGRIDVIEWCELLMOUSEDOUBLECLICK] is DataGridViewCellMouseEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnCellMouseDown(DataGridViewCellMouseEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }

            DataGridViewCell dataGridViewCell = GetCellInternal(e.ColumnIndex, e.RowIndex);
            Debug.Assert(dataGridViewCell != null);

            // Only left clicks for now
            Keys nModifier = ModifierKeys;
            bool isControlDown = (nModifier & Keys.Control) == Keys.Control && (nModifier & Keys.Alt) == 0;
            bool isShiftDown = (nModifier & Keys.Shift) == Keys.Shift;
            bool isAltDown = (nModifier & Keys.Alt) == Keys.Alt;

            Point ptGridCoord = ConvertCellToGridCoord(e.ColumnIndex, e.RowIndex, e.X, e.Y);

            HitTestInfo hti = HitTest(ptGridCoord.X, ptGridCoord.Y);

            if (!dataGridViewState2[DATAGRIDVIEWSTATE2_messageFromEditingCtrls] && e.Button == MouseButtons.Left)
            {
                Debug.Assert(hti.Type != DataGridViewHitTestType.None &&
                             hti.Type != DataGridViewHitTestType.HorizontalScrollBar &&
                             hti.Type != DataGridViewHitTestType.VerticalScrollBar);
                Debug.Assert(ptMouseDownCell.X == hti.col);
                Debug.Assert(ptMouseDownCell.Y == hti.row);

                switch (hti.typeInternal)
                {
                    // Check for column/row (headers) resize
                    case DataGridViewHitTestTypeInternal.ColumnResizeLeft:
                    case DataGridViewHitTestTypeInternal.ColumnResizeRight:
                    case DataGridViewHitTestTypeInternal.RowResizeBottom:
                    case DataGridViewHitTestTypeInternal.RowResizeTop:
                    case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeLeft:
                    case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeRight:
                    case DataGridViewHitTestTypeInternal.RowHeadersResizeLeft:
                    case DataGridViewHitTestTypeInternal.RowHeadersResizeRight:
                    case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeTop:
                    case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeBottom:
                    case DataGridViewHitTestTypeInternal.ColumnHeadersResizeTop:
                    case DataGridViewHitTestTypeInternal.ColumnHeadersResizeBottom:
                        {
                            dataGridViewOper[DATAGRIDVIEWOPER_resizingOperationAboutToStart] = (e.Clicks == 1);
                            break;
                        }
                }
            }

            try
            {
                if (e.RowIndex >= 0 && dataGridViewCell.MouseDownUnsharesRowInternal(e))
                {
                    DataGridViewRow dataGridViewRow = Rows[e.RowIndex];
                    GetCellInternal(e.ColumnIndex, e.RowIndex).OnMouseDownInternal(e);
                }
                else
                {
                    dataGridViewCell.OnMouseDownInternal(e);
                }

                if (Events[EVENT_DATAGRIDVIEWCELLMOUSEDOWN] is DataGridViewCellMouseEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
                {
                    eh(this, e);
                }

                if (!dataGridViewState2[DATAGRIDVIEWSTATE2_messageFromEditingCtrls] && e.Button == MouseButtons.Left)
                {
                    switch (hti.typeInternal)
                    {
                        // Check column resize
                        case DataGridViewHitTestTypeInternal.ColumnResizeLeft:
                        case DataGridViewHitTestTypeInternal.ColumnResizeRight:
                            {
                                int columnIndex = (hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnResizeRight) ? hti.col : hti.adjacentCol;
                                Debug.Assert(Columns[columnIndex].Resizable == DataGridViewTriState.True);
                                if (e.Clicks == 1)
                                {
                                    BeginMouseColumnResize(ptGridCoord.X, hti.mouseBarOffset, columnIndex);
                                }
                                break;
                            }

                        // Check row resize
                        case DataGridViewHitTestTypeInternal.RowResizeBottom:
                        case DataGridViewHitTestTypeInternal.RowResizeTop:
                            {
                                int rowIndex = (hti.typeInternal == DataGridViewHitTestTypeInternal.RowResizeBottom) ? hti.row : hti.adjacentRow;
                                if (e.Clicks == 1)
                                {
                                    BeginRowResize(ptGridCoord.Y, hti.mouseBarOffset, rowIndex);
                                }
                                break;
                            }

                        // Check for column header mouse down
                        case DataGridViewHitTestTypeInternal.ColumnHeader:
                        case DataGridViewHitTestTypeInternal.ColumnHeaderLeft:
                        case DataGridViewHitTestTypeInternal.ColumnHeaderRight:
                        case DataGridViewHitTestTypeInternal.FirstColumnHeaderLeft:
                            {
                                if (isAltDown && AllowUserToOrderColumns &&
                                    (SelectionMode == DataGridViewSelectionMode.FullColumnSelect ||
                                     SelectionMode == DataGridViewSelectionMode.ColumnHeaderSelect))
                                {
                                    BeginColumnRelocation(ptGridCoord.X, hti.col);
                                }
                                else
                                {
                                    OnColumnHeaderMouseDown(hti, isShiftDown, isControlDown);
                                }
                                break;
                            }

                        // Check for row header mouse down
                        case DataGridViewHitTestTypeInternal.RowHeader:
                            {
                                OnRowHeaderMouseDown(hti, isShiftDown, isControlDown);
                                break;
                            }

                        // Check for cell mouse down
                        case DataGridViewHitTestTypeInternal.Cell:
                            {
                                OnCellMouseDown(hti, isShiftDown, isControlDown);
                                break;
                            }

                        // Check for top/left header mouse down
                        case DataGridViewHitTestTypeInternal.TopLeftHeader:
                            {
                                OnTopLeftHeaderMouseDown();
                                break;
                            }

                        // Check for row headers resize
                        case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeLeft:
                        case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeRight:
                        case DataGridViewHitTestTypeInternal.RowHeadersResizeLeft:
                        case DataGridViewHitTestTypeInternal.RowHeadersResizeRight:
                            {
                                if (e.Clicks == 1)
                                {
                                    BeginRowHeadersResize(ptGridCoord.X, hti.mouseBarOffset);
                                }
                                break;
                            }

                        // Check for column headers resize
                        case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeTop:
                        case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeBottom:
                        case DataGridViewHitTestTypeInternal.ColumnHeadersResizeTop:
                        case DataGridViewHitTestTypeInternal.ColumnHeadersResizeBottom:
                            {
                                if (e.Clicks == 1)
                                {
                                    BeginColumnHeadersResize(ptGridCoord.Y, hti.mouseBarOffset);
                                }
                                break;
                            }
                    }
                    // Make sure that there is a current cell after this mouse down event.
                    if (ptCurrentCell.X == -1)
                    {
                        MakeFirstDisplayedCellCurrentCell(true /*includeNewRow*/);
                    }
                }
            }
            finally
            {
                dataGridViewOper[DATAGRIDVIEWOPER_resizingOperationAboutToStart] = false;
            }
        }

        private void OnCellMouseDown(HitTestInfo hti, bool isShiftDown, bool isControlDown)
        {
            Debug.Assert(hti.Type == DataGridViewHitTestType.Cell);
            // Only commit cell if the target cell is different from the current one.
            if (ptCurrentCell.X >= 0 &&
                (ptCurrentCell.X != hti.col || ptCurrentCell.Y != hti.row))
            {
                Point ptOriginalCurrentCell = ptCurrentCell;
                if (!CommitEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit,
                    ptCurrentCell.X != hti.col || ptCurrentCell.Y != hti.row /*forCurrentCellChange*/,
                    ptCurrentCell.Y != hti.row /*forCurrentRowChange*/))
                {
                    // Return silently if validating/commit/abort failed
                    return;
                }
                if (ptCurrentCell != ptOriginalCurrentCell)
                {
                    // Somehow the fact that the current cell was committed altered the current cell value.
                    // To avoid unintentional multi-selections, we act as if Shift and Control keys were up.
                    isShiftDown = isControlDown = false;
                }
            }

            if (hti.col >= Columns.Count)
            {
                DataGridViewColumn dataGridViewLastVisibleColumn = Columns.GetLastColumn(DataGridViewElementStates.Visible,
                                                                                              DataGridViewElementStates.None);
                if (ptCurrentCell.X == -1 && dataGridViewLastVisibleColumn != null)
                {
                    // CurrentCell was reset because CommitEdit deleted column(s).
                    // Since the user clicked on a cell, we don't want to end up
                    // with no CurrentCell. We pick the last visible column in the grid.
                    hti.col = dataGridViewLastVisibleColumn.Index;
                }
                else
                {
                    return;
                }
            }
            if (hti.row >= Rows.Count)
            {
                int lastVisibleRowIndex = Rows.GetLastRow(DataGridViewElementStates.Visible);
                if (ptCurrentCell.X == -1 && lastVisibleRowIndex != -1)
                {
                    // CurrentCell was reset because CommitEdit deleted row(s).
                    // Since the user clicked on a cell, we don't want to end up
                    // with no CurrentCell. We pick the last visible row in the
                    // grid which may be the 'new row'.
                    hti.row = lastVisibleRowIndex;
                }
                else
                {
                    return;
                }
            }

            bool select = true;
            noSelectionChangeCount++;
            try
            {
                switch (SelectionMode)
                {
                    case DataGridViewSelectionMode.CellSelect:
                        {
                            if (isControlDown &&
                                IsSharedCellSelected(Rows.SharedRow(hti.row).Cells[hti.col], hti.row) &&
                                (!isShiftDown || !MultiSelect))
                            {
                                select = false;
                            }
                            if (select)
                            {
                                if ((!MultiSelect || !isControlDown) && !(MultiSelect && isShiftDown))
                                {
                                    Debug.Assert(MultiSelect || individualSelectedCells.Count <= 1);
                                    RemoveIndividuallySelectedCells(hti.col, hti.row);
                                }
                                if (MultiSelect)
                                {
                                    if (dataGridViewOper[DATAGRIDVIEWOPER_trackMouseMoves])
                                    {
                                        dataGridViewOper[DATAGRIDVIEWOPER_trackCellSelect] = true;
                                    }
                                    if (isShiftDown)
                                    {
                                        int oldEdgeColumnIndex = ptCurrentCell.X;
                                        int oldEdgeRowIndex = ptCurrentCell.Y;
                                        if (ptAnchorCell.X == -1)
                                        {
                                            return;
                                        }
                                        UpdateSelectedCellsBlock(ptAnchorCell.X, ref oldEdgeColumnIndex, hti.col,
                                            ptAnchorCell.Y, ref oldEdgeRowIndex, hti.row);
                                    }
                                    else
                                    {
                                        SetSelectedCellCore(hti.col, hti.row, true);
                                    }
                                }
                                else
                                {
                                    SetSelectedCellCore(hti.col, hti.row, true);
                                }
                            }
                            else
                            {
                                SetSelectedCellCore(hti.col, hti.row, false);
                            }
                            bool success = SetCurrentCellAddressCore(hti.col, hti.row, !isShiftDown, false, true);
                            Debug.Assert(success);
                            break;
                        }

                    case DataGridViewSelectionMode.FullColumnSelect:
                        {
                            if (isControlDown && Columns[hti.col].Selected)
                            {
                                select = false;
                            }
                            if (select)
                            {
                                bool selectColumnRange = false;
                                trackColumn = hti.col;
                                trackColumnEdge = -1;
                                if (MultiSelect &&
                                    isShiftDown &&
                                    ptAnchorCell.X > -1 &&
                                    Columns[ptAnchorCell.X].Selected)
                                {
                                    selectColumnRange = true;
                                }
                                if (!MultiSelect || !isControlDown || isShiftDown)
                                {
                                    Debug.Assert(MultiSelect || selectedBandIndexes.Count <= 1);
                                    int bandIndex = 0;
                                    bool switchedToBulkPaint = false;
                                    if (selectedBandIndexes.Count > DATAGRIDVIEW_bulkPaintThreshold)
                                    {
                                        inBulkPaintCount++;
                                        switchedToBulkPaint = true;
                                    }
                                    try
                                    {
                                        while (bandIndex < selectedBandIndexes.Count)
                                        {
                                            if (selectedBandIndexes[bandIndex] != hti.col)
                                            {
                                                // deselect currently selected column
                                                SetSelectedColumnCore(selectedBandIndexes[bandIndex], false);
                                            }
                                            else
                                            {
                                                bandIndex++;
                                            }
                                        }
                                    }
                                    finally
                                    {
                                        if (switchedToBulkPaint)
                                        {
                                            ExitBulkPaint(-1, -1);
                                        }
                                    }
                                }
                                if (MultiSelect && dataGridViewOper[DATAGRIDVIEWOPER_trackMouseMoves])
                                {
                                    dataGridViewOper[DATAGRIDVIEWOPER_trackColSelect] = true;
                                }
                                if (selectColumnRange)
                                {
                                    if (Columns.DisplayInOrder(ptAnchorCell.X, hti.col))
                                    {
                                        SelectColumnRange(ptAnchorCell.X, hti.col, true);
                                    }
                                    else
                                    {
                                        SelectColumnRange(hti.col, ptAnchorCell.X, true);
                                    }
                                }
                                else if (!selectedBandIndexes.Contains(hti.col))
                                {
                                    SetSelectedColumnCore(hti.col, true);
                                }
                            }
                            else
                            {
                                Debug.Assert(selectedBandIndexes.Contains(hti.col));
                                SetSelectedColumnCore(hti.col, false);
                            }
                            bool success = SetCurrentCellAddressCore(hti.col, hti.row, !isShiftDown, false, true);
                            Debug.Assert(success);
                            break;
                        }

                    case DataGridViewSelectionMode.ColumnHeaderSelect:
                        {
                            if (isControlDown &&
                                (Columns[hti.col].Selected || IsSharedCellSelected(Rows.SharedRow(hti.row).Cells[hti.col], hti.row)) &&
                                (!isShiftDown || !MultiSelect))
                            {
                                select = false;
                            }
                            if (select)
                            {
                                if (!MultiSelect)
                                {
                                    Debug.Assert(selectedBandIndexes.Count <= 1);
                                    if (selectedBandIndexes.Count > 0)
                                    {
                                        SetSelectedColumnCore(selectedBandIndexes.HeadInt, false);
                                    }
                                    else
                                    {
                                        RemoveIndividuallySelectedCells();
                                    }
                                    SetSelectedCellCore(hti.col, hti.row, true);
                                }
                                else
                                {
                                    // this.MultiSelect == true
                                    if (!isControlDown && !isShiftDown)
                                    {
                                        bool switchedToBulkPaint = false;
                                        if (selectedBandIndexes.Count > DATAGRIDVIEW_bulkPaintThreshold)
                                        {
                                            inBulkPaintCount++;
                                            switchedToBulkPaint = true;
                                        }
                                        try
                                        {
                                            while (selectedBandIndexes.Count > 0)
                                            {
                                                SetSelectedColumnCore(selectedBandIndexes.HeadInt, false);
                                            }
                                            RemoveIndividuallySelectedCells(hti.col, hti.row);
                                        }
                                        finally
                                        {
                                            if (switchedToBulkPaint)
                                            {
                                                ExitBulkPaint(-1, -1);
                                            }
                                        }
                                    }
                                    if (dataGridViewOper[DATAGRIDVIEWOPER_trackMouseMoves])
                                    {
                                        dataGridViewOper[DATAGRIDVIEWOPER_trackCellSelect] = true;
                                    }
                                    if (isShiftDown)
                                    {
                                        int oldEdgeColumnIndex = ptCurrentCell.X;
                                        int oldEdgeRowIndex = ptCurrentCell.Y;
                                        if (ptAnchorCell.X == -1)
                                        {
                                            return;
                                        }
                                        UpdateSelectedCellsBlock(ptAnchorCell.X, ref oldEdgeColumnIndex, hti.col,
                                            ptAnchorCell.Y, ref oldEdgeRowIndex, hti.row);
                                    }
                                    else
                                    {
                                        SetSelectedCellCore(hti.col, hti.row, true);
                                    }
                                }
                            }
                            else
                            {
                                if (!MultiSelect)
                                {
                                    Debug.Assert(selectedBandIndexes.Count <= 1);
                                    if (selectedBandIndexes.Count > 0)
                                    {
                                        SetSelectedColumnCore(selectedBandIndexes.HeadInt, false);
                                    }
                                    else
                                    {
                                        SetSelectedCellCore(hti.col, hti.row, false);
                                    }
                                }
                                else
                                {
                                    SetSelectedCellCore(hti.col, hti.row, false);
                                }
                            }
                            bool success = SetCurrentCellAddressCore(hti.col, hti.row, !isShiftDown, false, true);
                            Debug.Assert(success);
                            break;
                        }

                    case DataGridViewSelectionMode.FullRowSelect:
                        {
                            if (isControlDown &&
                                ((Rows.GetRowState(hti.row) & DataGridViewElementStates.Selected) != 0))
                            {
                                select = false;
                            }
                            if (select)
                            {
                                bool selectRowRange = false;
                                trackRow = hti.row;
                                trackRowEdge = -1;
                                if (MultiSelect &&
                                    isShiftDown &&
                                    ptAnchorCell.Y > -1 && (Rows.GetRowState(ptAnchorCell.Y) & DataGridViewElementStates.Selected) != 0)
                                {
                                    selectRowRange = true;
                                }

                                if (!MultiSelect || !isControlDown || isShiftDown)
                                {
                                    Debug.Assert(MultiSelect || selectedBandIndexes.Count <= 1);
                                    int bandIndex = 0;
                                    bool switchedToBulkPaint = false;
                                    if (selectedBandIndexes.Count > DATAGRIDVIEW_bulkPaintThreshold)
                                    {
                                        inBulkPaintCount++;
                                        switchedToBulkPaint = true;
                                    }
                                    try
                                    {
                                        while (bandIndex < selectedBandIndexes.Count)
                                        {
                                            if (selectedBandIndexes[bandIndex] != hti.row)
                                            {
                                                // deselect currently selected row
                                                SetSelectedRowCore(selectedBandIndexes[bandIndex], false);
                                            }
                                            else
                                            {
                                                bandIndex++;
                                            }
                                        }
                                    }
                                    finally
                                    {
                                        if (switchedToBulkPaint)
                                        {
                                            ExitBulkPaint(-1, -1);
                                        }
                                    }
                                }
                                if (MultiSelect && dataGridViewOper[DATAGRIDVIEWOPER_trackMouseMoves])
                                {
                                    dataGridViewOper[DATAGRIDVIEWOPER_trackRowSelect] = true;
                                }
                                if (selectRowRange)
                                {
                                    if (hti.row >= ptAnchorCell.Y)
                                    {
                                        SelectRowRange(ptAnchorCell.Y, hti.row, true);
                                    }
                                    else
                                    {
                                        SelectRowRange(hti.row, ptAnchorCell.Y, true);
                                    }
                                }
                                else if ((Rows.GetRowState(hti.row) & DataGridViewElementStates.Selected) == 0)
                                {
                                    Debug.Assert(selectedBandIndexes.Contains(hti.row) ==
                                                 ((Rows.GetRowState(hti.row) & DataGridViewElementStates.Selected) != 0));
                                    SetSelectedRowCore(hti.row, true);
                                }
                            }
                            else
                            {
                                Debug.Assert(selectedBandIndexes.Contains(hti.row));
                                SetSelectedRowCore(hti.row, false);
                            }
                            bool success = SetCurrentCellAddressCore(hti.col, hti.row, !isShiftDown, false, true);
                            Debug.Assert(success);
                            break;
                        }

                    case DataGridViewSelectionMode.RowHeaderSelect:
                        {
                            if (isControlDown &&
                                (((Rows.GetRowState(hti.row) & DataGridViewElementStates.Selected) != 0) ||
                                IsSharedCellSelected(Rows.SharedRow(hti.row).Cells[hti.col], hti.row)) &&
                                (!isShiftDown || !MultiSelect))
                            {
                                select = false;
                            }
                            if (select)
                            {
                                if (!MultiSelect)
                                {
                                    Debug.Assert(selectedBandIndexes.Count <= 1);
                                    if (selectedBandIndexes.Count > 0)
                                    {
                                        SetSelectedRowCore(selectedBandIndexes.HeadInt, false);
                                    }
                                    else
                                    {
                                        RemoveIndividuallySelectedCells();
                                    }
                                    SetSelectedCellCore(hti.col, hti.row, true);
                                }
                                else
                                {
                                    // this.MultiSelect == true
                                    if (!isControlDown && !isShiftDown)
                                    {
                                        bool switchedToBulkPaint = false;
                                        if (selectedBandIndexes.Count > DATAGRIDVIEW_bulkPaintThreshold)
                                        {
                                            inBulkPaintCount++;
                                            switchedToBulkPaint = true;
                                        }
                                        try
                                        {
                                            while (selectedBandIndexes.Count > 0)
                                            {
                                                SetSelectedRowCore(selectedBandIndexes.HeadInt, false);
                                            }
                                            RemoveIndividuallySelectedCells(hti.col, hti.row);
                                        }
                                        finally
                                        {
                                            if (switchedToBulkPaint)
                                            {
                                                ExitBulkPaint(-1, -1);
                                            }
                                        }
                                    }
                                    if (dataGridViewOper[DATAGRIDVIEWOPER_trackMouseMoves])
                                    {
                                        dataGridViewOper[DATAGRIDVIEWOPER_trackCellSelect] = true;
                                    }
                                    if (isShiftDown)
                                    {
                                        int oldEdgeColumnIndex = ptCurrentCell.X;
                                        int oldEdgeRowIndex = ptCurrentCell.Y;
                                        if (ptAnchorCell.X == -1)
                                        {
                                            return;
                                        }
                                        UpdateSelectedCellsBlock(ptAnchorCell.X, ref oldEdgeColumnIndex, hti.col,
                                            ptAnchorCell.Y, ref oldEdgeRowIndex, hti.row);
                                    }
                                    else
                                    {
                                        SetSelectedCellCore(hti.col, hti.row, true);
                                    }
                                }
                            }
                            else
                            {
                                if (!MultiSelect)
                                {
                                    Debug.Assert(selectedBandIndexes.Count <= 1);
                                    if (selectedBandIndexes.Count > 0)
                                    {
                                        SetSelectedRowCore(selectedBandIndexes.HeadInt, false);
                                    }
                                    else
                                    {
                                        SetSelectedCellCore(hti.col, hti.row, false);
                                    }
                                }
                                else
                                {
                                    SetSelectedCellCore(hti.col, hti.row, false);
                                }
                            }
                            bool success = SetCurrentCellAddressCore(hti.col, hti.row, !isShiftDown, false, true);

                            // Microsoft: SetCurrentCellAddressCore can fail if by navigating to a cell the list under the
                            // DataGridView changes.
                            // Debug.Assert(success);
                            break;
                        }
                }
            }
            finally
            {
                NoSelectionChangeCount--;
            }
        }

        protected virtual void OnCellMouseEnter(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            ptMouseEnteredCell.X = e.ColumnIndex;
            ptMouseEnteredCell.Y = e.RowIndex;

            DataGridViewCell dataGridViewCell = GetCellInternal(e.ColumnIndex, e.RowIndex);
            Debug.Assert(dataGridViewCell != null);
            if (e.RowIndex >= 0 && dataGridViewCell.MouseEnterUnsharesRowInternal(e.RowIndex))
            {
                DataGridViewRow dataGridViewRow = Rows[e.RowIndex];
                GetCellInternal(e.ColumnIndex, e.RowIndex).OnMouseEnterInternal(e.RowIndex);
            }
            else
            {
                dataGridViewCell.OnMouseEnterInternal(e.RowIndex);
            }

            if (Events[EVENT_DATAGRIDVIEWCELLMOUSEENTER] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnCellMouseLeave(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            ptMouseEnteredCell.X = -2;
            ptMouseEnteredCell.Y = -2;

            DataGridViewCell dataGridViewCell = GetCellInternal(e.ColumnIndex, e.RowIndex);
            Debug.Assert(dataGridViewCell != null);
            if (e.RowIndex >= 0 && dataGridViewCell.MouseLeaveUnsharesRowInternal(e.RowIndex))
            {
                DataGridViewRow dataGridViewRow = Rows[e.RowIndex];
                GetCellInternal(e.ColumnIndex, e.RowIndex).OnMouseLeaveInternal(e.RowIndex);
            }
            else
            {
                dataGridViewCell.OnMouseLeaveInternal(e.RowIndex);
            }

            if (Events[EVENT_DATAGRIDVIEWCELLMOUSELEAVE] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnCellMouseMove(DataGridViewCellMouseEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            DataGridViewCell dataGridViewCell = GetCellInternal(e.ColumnIndex, e.RowIndex);
            Debug.Assert(dataGridViewCell != null);
            if (e.RowIndex >= 0 && dataGridViewCell.MouseMoveUnsharesRowInternal(e))
            {
                DataGridViewRow dataGridViewRow = Rows[e.RowIndex];
                GetCellInternal(e.ColumnIndex, e.RowIndex).OnMouseMoveInternal(e);
            }
            else
            {
                dataGridViewCell.OnMouseMoveInternal(e);
            }

            if (Events[EVENT_DATAGRIDVIEWCELLMOUSEMOVE] is DataGridViewCellMouseEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            if (!dataGridViewState1[DATAGRIDVIEWSTATE1_scrolledSinceMouseDown] &&
                !IsMouseOperationActive() &&
                AllowUserToOrderColumns &&
                SelectionMode != DataGridViewSelectionMode.FullColumnSelect &&
                SelectionMode != DataGridViewSelectionMode.ColumnHeaderSelect &&
                e.Button == MouseButtons.Left &&
                ptMouseDownCell.Y == -1 &&
                ptMouseDownCell.X >= 0 &&
                ptMouseDownCell.X < Columns.Count)
            {
                Point ptGridCoord = ConvertCellToGridCoord(e.ColumnIndex, e.RowIndex, e.X, e.Y);

                HitTestInfo hti = HitTest(ptGridCoord.X, ptGridCoord.Y);

                Debug.Assert(hti.Type != DataGridViewHitTestType.None &&
                             hti.Type != DataGridViewHitTestType.HorizontalScrollBar &&
                             hti.Type != DataGridViewHitTestType.VerticalScrollBar);

                switch (hti.typeInternal)
                {
                    // Check for column header mouse down
                    case DataGridViewHitTestTypeInternal.ColumnHeader:
                    case DataGridViewHitTestTypeInternal.ColumnHeaderLeft:
                    case DataGridViewHitTestTypeInternal.ColumnHeaderRight:
                    case DataGridViewHitTestTypeInternal.FirstColumnHeaderLeft:
                        {
                            Debug.Assert(!dataGridViewState2[DATAGRIDVIEWSTATE2_messageFromEditingCtrls]);
                            if (Math.Abs(ptMouseDownGridCoord.X - ptGridCoord.X) >= DataGridView.DragSize.Width ||
                                Math.Abs(ptMouseDownGridCoord.Y - ptGridCoord.Y) >= DataGridView.DragSize.Height)
                            {
                                BeginColumnRelocation(ptMouseDownGridCoord.X, ptMouseDownCell.X);
                            }
                            break;
                        }
                }
            }
        }

        protected virtual void OnCellMouseUp(DataGridViewCellMouseEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            DataGridViewCell dataGridViewCell = GetCellInternal(e.ColumnIndex, e.RowIndex);
            Debug.Assert(dataGridViewCell != null);
            if (e.RowIndex >= 0 && dataGridViewCell.MouseUpUnsharesRowInternal(e))
            {
                DataGridViewRow dataGridViewRow = Rows[e.RowIndex];
                GetCellInternal(e.ColumnIndex, e.RowIndex).OnMouseUpInternal(e);
            }
            else
            {
                dataGridViewCell.OnMouseUpInternal(e);
            }

            if (Events[EVENT_DATAGRIDVIEWCELLMOUSEUP] is DataGridViewCellMouseEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected internal virtual void OnCellPainting(DataGridViewCellPaintingEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            if (Events[EVENT_DATAGRIDVIEWCELLPAINTING] is DataGridViewCellPaintingEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal DataGridViewCellParsingEventArgs OnCellParsing(int rowIndex, int columnIndex, object formattedValue, Type valueType, DataGridViewCellStyle cellStyle)
        {
            DataGridViewCellParsingEventArgs dgvcpe = new DataGridViewCellParsingEventArgs(rowIndex, columnIndex,
                                                                                           formattedValue,
                                                                                           valueType,
                                                                                           cellStyle);
            OnCellParsing(dgvcpe);
            return dgvcpe;
        }

        protected virtual void OnCellParsing(DataGridViewCellParsingEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            if (Events[EVENT_DATAGRIDVIEWCELLPARSING] is DataGridViewCellParsingEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnCellSelectMouseMove(HitTestInfo hti)
        {
            Debug.Assert(MultiSelect);
            int oldEdgeColumnIndex = ptCurrentCell.X;
            int oldEdgeRowIndex = ptCurrentCell.Y;
            if ((hti.col != ptCurrentCell.X || hti.row != ptCurrentCell.Y) &&
                !CommitEditForOperation(hti.col, hti.row, true))
            {
                // Return silently if validating/commit/abort failed
                return;
            }
            noSelectionChangeCount++;
            try
            {
                if (ptAnchorCell.X == -1 || IsInnerCellOutOfBounds(hti.col, hti.row))
                {
                    return;
                }
                UpdateSelectedCellsBlock(ptAnchorCell.X, ref oldEdgeColumnIndex, hti.col,
                    ptAnchorCell.Y, ref oldEdgeRowIndex, hti.row);
                if (hti.col != ptCurrentCell.X || hti.row != ptCurrentCell.Y)
                {
                    bool success = SetCurrentCellAddressCore(hti.col, hti.row, false, false, false);
                    Debug.Assert(success);
                }
            }
            finally
            {
                NoSelectionChangeCount--;
            }
        }

        protected virtual void OnCellStateChanged(DataGridViewCellStateChangedEventArgs e)
        {
            // At this point we assume that only the Selected state has an influence on the rendering of the cell.
            // If there is a scenario where another state has an effect, then the dev will have to invalidate the cell by hand.
            DataGridViewCell dataGridViewCell = e.Cell;
            if (e.StateChanged == DataGridViewElementStates.Selected)
            {
                Debug.Assert(dataGridViewCell.RowIndex >= 0);
                if (inBulkPaintCount == 0)
                {
                    InvalidateCellPrivate(dataGridViewCell);
                }
            }

            if (Events[EVENT_DATAGRIDVIEWCELLSTATECHANGED] is DataGridViewCellStateChangedEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            if (e.StateChanged == DataGridViewElementStates.ReadOnly &&
                ptCurrentCell.X == dataGridViewCell.ColumnIndex &&
                ptCurrentCell.Y == dataGridViewCell.RowIndex &&
                dataGridViewCell.RowIndex > -1 &&
                !dataGridViewOper[DATAGRIDVIEWOPER_inReadOnlyChange])
            {
                VerifyImeRestrictedModeChanged();

                if (!dataGridViewCell.ReadOnly &&
                    ColumnEditable(ptCurrentCell.X) &&
                    !IsCurrentCellInEditMode &&
                    (EditMode == DataGridViewEditMode.EditOnEnter ||
                    (EditMode != DataGridViewEditMode.EditProgrammatically && CurrentCellInternal.EditType == null)))
                {
                    // Current cell becomes read/write. Enter editing mode.
                    BeginEditInternal(true /*selectAll*/);
                }
            }
        }

        internal void OnCellStyleChanged(DataGridViewCell dataGridViewCell)
        {
            DataGridViewCellEventArgs dgvce = new DataGridViewCellEventArgs(dataGridViewCell);
            OnCellStyleChanged(dgvce);
        }

        protected virtual void OnCellStyleChanged(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            OnCellCommonChange(e.ColumnIndex, e.RowIndex);

            if (Events[EVENT_DATAGRIDVIEWCELLSTYLECHANGED] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnCellStyleContentChanged(DataGridViewCellStyle dataGridViewCellStyle, DataGridViewCellStyle.DataGridViewCellStylePropertyInternal property)
        {
            Debug.Assert(dataGridViewCellStyle != null);
            switch (property)
            {
                case DataGridViewCellStyle.DataGridViewCellStylePropertyInternal.Font:
                    if ((dataGridViewCellStyle.Scope & DataGridViewCellStyleScopes.DataGridView) != 0 && dataGridViewState1[DATAGRIDVIEWSTATE1_ambientFont])
                    {
                        dataGridViewState1[DATAGRIDVIEWSTATE1_ambientFont] = false;
                    }
                    if ((dataGridViewCellStyle.Scope & DataGridViewCellStyleScopes.ColumnHeaders) != 0 && dataGridViewState1[DATAGRIDVIEWSTATE1_ambientColumnHeadersFont])
                    {
                        dataGridViewState1[DATAGRIDVIEWSTATE1_ambientColumnHeadersFont] = false;
                    }
                    if ((dataGridViewCellStyle.Scope & DataGridViewCellStyleScopes.RowHeaders) != 0 && dataGridViewState1[DATAGRIDVIEWSTATE1_ambientRowHeadersFont])
                    {
                        dataGridViewState1[DATAGRIDVIEWSTATE1_ambientRowHeadersFont] = false;
                    }
                    break;

                case DataGridViewCellStyle.DataGridViewCellStylePropertyInternal.ForeColor:
                    if ((dataGridViewCellStyle.Scope & DataGridViewCellStyleScopes.DataGridView) != 0 && dataGridViewState1[DATAGRIDVIEWSTATE1_ambientForeColor])
                    {
                        dataGridViewState1[DATAGRIDVIEWSTATE1_ambientForeColor] = false;
                    }
                    break;
            }

            DataGridViewCellStyleContentChangedEventArgs dgvcscce = new DataGridViewCellStyleContentChangedEventArgs(dataGridViewCellStyle,
                property != DataGridViewCellStyle.DataGridViewCellStylePropertyInternal.Color &&
                property != DataGridViewCellStyle.DataGridViewCellStylePropertyInternal.ForeColor /*changeAffectsPreferredSize*/);
            OnCellStyleContentChanged(dgvcscce);
        }

        protected virtual void OnCellStyleContentChanged(DataGridViewCellStyleContentChangedEventArgs e)
        {
            // We assume that when a color changes, it has no effect on the autosize of elements
            bool repositionEditingControl = false;

            if ((e.CellStyleScope & DataGridViewCellStyleScopes.Cell) == DataGridViewCellStyleScopes.Cell && (e.CellStyleScope & DataGridViewCellStyleScopes.DataGridView) == 0)
            {
                // Same processing as in OnDefaultCellStyleChanged
                if (e.ChangeAffectsPreferredSize)
                {
                    repositionEditingControl = true;
                    OnGlobalAutoSize();
                }
                else
                {
                    Invalidate();
                }
            }

            if ((e.CellStyleScope & DataGridViewCellStyleScopes.Column) == DataGridViewCellStyleScopes.Column)
            {
                if (e.ChangeAffectsPreferredSize)
                {
                    repositionEditingControl = true;
                    OnColumnsGlobalAutoSize();
                }
                else
                {
                    InvalidateData();
                }
            }

            if ((e.CellStyleScope & DataGridViewCellStyleScopes.Row) == DataGridViewCellStyleScopes.Row && (e.CellStyleScope & DataGridViewCellStyleScopes.Rows) == 0 && (e.CellStyleScope & DataGridViewCellStyleScopes.AlternatingRows) == 0)
            {
                // Same processing as in OnRowsDefaultCellStyleChanged
                InvalidateData();
                if (e.ChangeAffectsPreferredSize)
                {
                    repositionEditingControl = true;
                    // Autosize rows if needed
                    if ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0)
                    {
                        AdjustShrinkingRows(autoSizeRowsMode, false /*fixedWidth*/, true /*internalAutosizing*/);
                    }
                    // Auto size columms also if needed
                    // Impossible to figure out if DisplayedRows filter should be added or not. Adding it to be on the conservative side.
                    AutoResizeAllVisibleColumnsInternal(DataGridViewAutoSizeColumnCriteriaInternal.AllRows |
                                                        DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows,
                                                        true /*fixedHeight*/);
                    // Second round of rows autosizing
                    if ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0)
                    {
                        AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
                    }
                }
            }

            if ((e.CellStyleScope & DataGridViewCellStyleScopes.DataGridView) == DataGridViewCellStyleScopes.DataGridView)
            {
                CellStyleChangedEventArgs.ChangeAffectsPreferredSize = e.ChangeAffectsPreferredSize;
                if (e.ChangeAffectsPreferredSize)
                {
                    repositionEditingControl = false;
                    // OnDefaultCellStyleChanged will reposition the editing control.
                }
                OnDefaultCellStyleChanged(CellStyleChangedEventArgs);
            }

            if ((e.CellStyleScope & DataGridViewCellStyleScopes.ColumnHeaders) == DataGridViewCellStyleScopes.ColumnHeaders)
            {
                CellStyleChangedEventArgs.ChangeAffectsPreferredSize = e.ChangeAffectsPreferredSize;
                if (e.ChangeAffectsPreferredSize)
                {
                    repositionEditingControl = false;
                    // OnColumnHeadersDefaultCellStyleChanged will reposition the editing control.
                }
                OnColumnHeadersDefaultCellStyleChanged(CellStyleChangedEventArgs);
            }

            if ((e.CellStyleScope & DataGridViewCellStyleScopes.RowHeaders) == DataGridViewCellStyleScopes.RowHeaders)
            {
                CellStyleChangedEventArgs.ChangeAffectsPreferredSize = e.ChangeAffectsPreferredSize;
                if (e.ChangeAffectsPreferredSize)
                {
                    repositionEditingControl = false;
                    // OnRowHeadersDefaultCellStyleChanged will reposition the editing control.
                }
                OnRowHeadersDefaultCellStyleChanged(CellStyleChangedEventArgs);
            }

            if ((e.CellStyleScope & DataGridViewCellStyleScopes.Rows) == DataGridViewCellStyleScopes.Rows)
            {
                CellStyleChangedEventArgs.ChangeAffectsPreferredSize = e.ChangeAffectsPreferredSize;
                if (e.ChangeAffectsPreferredSize)
                {
                    repositionEditingControl = false;
                    // OnRowsDefaultCellStyleChanged will reposition the editing control.
                }
                OnRowsDefaultCellStyleChanged(CellStyleChangedEventArgs);
            }

            if ((e.CellStyleScope & DataGridViewCellStyleScopes.AlternatingRows) == DataGridViewCellStyleScopes.AlternatingRows)
            {
                CellStyleChangedEventArgs.ChangeAffectsPreferredSize = e.ChangeAffectsPreferredSize;
                if (e.ChangeAffectsPreferredSize)
                {
                    repositionEditingControl = false;
                    // OnAlternatingRowsDefaultCellStyleChanged will reposition the editing control.
                }
                OnAlternatingRowsDefaultCellStyleChanged(CellStyleChangedEventArgs);
            }

            if (repositionEditingControl && editingControl != null)
            {
                PositionEditingControl(true /*setLocation*/, true /*setSize*/, false /*setFocus*/);
            }

            if (Events[EVENT_DATAGRIDVIEWCELLSTYLECONTENTCHANGED] is DataGridViewCellStyleContentChangedEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnCellToolTipTextChanged(DataGridViewCell dataGridViewCell)
        {
            DataGridViewCellEventArgs dgvce = new DataGridViewCellEventArgs(dataGridViewCell);
            OnCellToolTipTextChanged(dgvce);
        }

        protected virtual void OnCellToolTipTextChanged(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            if (Events[EVENT_DATAGRIDVIEWCELLTOOLTIPTEXTCHANGED] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal string OnCellToolTipTextNeeded(int columnIndex, int rowIndex, string toolTipText)
        {
            DataGridViewCellToolTipTextNeededEventArgs dgvctttne = new DataGridViewCellToolTipTextNeededEventArgs(columnIndex, rowIndex, toolTipText);
            OnCellToolTipTextNeeded(dgvctttne);
            return dgvctttne.ToolTipText;
        }

        protected virtual void OnCellToolTipTextNeeded(DataGridViewCellToolTipTextNeededEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            if (Events[EVENT_DATAGRIDVIEWCELLTOOLTIPTEXTNEEDED] is DataGridViewCellToolTipTextNeededEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnCellValidated(ref DataGridViewCell dataGridViewCell, int columnIndex, int rowIndex)
        {
            OnCellValidated(new DataGridViewCellEventArgs(columnIndex, rowIndex));
            if (dataGridViewCell != null)
            {
                if (IsInnerCellOutOfBounds(columnIndex, rowIndex))
                {
                    dataGridViewCell = null;
                }
                else
                {
                    Debug.Assert(rowIndex < Rows.Count && columnIndex < Columns.Count);
                    dataGridViewCell = Rows.SharedRow(rowIndex).Cells[columnIndex];
                }
            }
        }

        protected virtual void OnCellValidated(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            try
            {
                noDimensionChangeCount++;

                if (Events[EVENT_DATAGRIDVIEWCELLVALIDATED] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
                {
                    eh(this, e);
                    CorrectFocus(true /*onlyIfGridHasFocus*/);
                }
            }
            finally
            {
                noDimensionChangeCount--;
                Debug.Assert(noDimensionChangeCount >= 0);
            }
        }

        internal bool OnCellValidating(ref DataGridViewCell dataGridViewCell, int columnIndex, int rowIndex, DataGridViewDataErrorContexts context)
        {
            DataGridViewCell currentCell = dataGridViewCell ?? CurrentCellInternal;
            DataGridViewCellStyle dataGridViewCellStyle = currentCell.GetInheritedStyle(null, rowIndex, false);
            object val = currentCell.GetValueInternal(rowIndex);
            object editedFormattedValue = currentCell.GetEditedFormattedValue(val, rowIndex, ref dataGridViewCellStyle, context);
            DataGridViewCellValidatingEventArgs dgvcfvce = new DataGridViewCellValidatingEventArgs(columnIndex, rowIndex, editedFormattedValue);
            OnCellValidating(dgvcfvce);
            if (dataGridViewCell != null)
            {
                if (IsInnerCellOutOfBounds(columnIndex, rowIndex))
                {
                    dataGridViewCell = null;
                }
                else
                {
                    Debug.Assert(rowIndex < Rows.Count && columnIndex < Columns.Count);
                    dataGridViewCell = Rows.SharedRow(rowIndex).Cells[columnIndex];
                }
            }
            return dgvcfvce.Cancel;
        }

        protected virtual void OnCellValidating(DataGridViewCellValidatingEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            try
            {
                noDimensionChangeCount++;
                dataGridViewOper[DATAGRIDVIEWOPER_inCellValidating] = true;

                if (Events[EVENT_DATAGRIDVIEWCELLVALIDATING] is DataGridViewCellValidatingEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
                {
                    eh(this, e);
                    CorrectFocus(true /*onlyIfGridHasFocus*/);
                }
            }
            finally
            {
                noDimensionChangeCount--;
                Debug.Assert(noDimensionChangeCount >= 0);
                dataGridViewOper[DATAGRIDVIEWOPER_inCellValidating] = false;
            }
        }

        internal void OnCellValueChangedInternal(DataGridViewCellEventArgs e)
        {
            // For now, same effect as if the cell style had changed.
            OnCellValueChanged(e);
        }

        protected virtual void OnCellValueChanged(DataGridViewCellEventArgs e)
        {
            if (e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            OnCellCommonChange(e.ColumnIndex, e.RowIndex);

            if (Events[EVENT_DATAGRIDVIEWCELLVALUECHANGED] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal object OnCellValueNeeded(int columnIndex, int rowIndex)
        {
            DataGridViewCellValueEventArgs dgvcve = CellValueEventArgs;
            dgvcve.SetProperties(columnIndex, rowIndex, null);
            OnCellValueNeeded(dgvcve);
            return dgvcve.Value;
        }

        protected virtual void OnCellValueNeeded(DataGridViewCellValueEventArgs e)
        {
            if (e.ColumnIndex < 0 || e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex < 0 || e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            //#if DEBUG
            // Some customer scenarios may result in accessing cell values while this.dataStoreAccessAllowed is false. This is bad practice,
            // but since we're late in Whidbey, throwing an exception would be destabilizing our internal customers.
            // Debug.Assert(this.dataStoreAccessAllowed);
            //#endif
            if (Events[EVENT_DATAGRIDVIEWCELLVALUENEEDED] is DataGridViewCellValueEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnCellValuePushed(int columnIndex, int rowIndex, object value)
        {
            DataGridViewCellValueEventArgs dgvcve = CellValueEventArgs;
            dgvcve.SetProperties(columnIndex, rowIndex, value);
            OnCellValuePushed(dgvcve);
        }

        protected virtual void OnCellValuePushed(DataGridViewCellValueEventArgs e)
        {
            if (e.ColumnIndex < 0 || e.ColumnIndex >= Columns.Count)
            {
                throw new ArgumentOutOfRangeException("e.ColumnIndex");
            }
            if (e.RowIndex < 0 || e.RowIndex >= Rows.Count)
            {
                throw new ArgumentOutOfRangeException("e.RowIndex");
            }
            if (Events[EVENT_DATAGRIDVIEWCELLVALUEPUSHED] is DataGridViewCellValueEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnClearedRows()
        {
            // Raise the RowStateChanged(Displayed->false) events
            foreach (DataGridViewRow dataGridViewRow in lstRows)
            {
                if (dataGridViewRow.Displayed)
                {
                    dataGridViewRow.Displayed = false;
                    DataGridViewRowStateChangedEventArgs dgvrsce = new DataGridViewRowStateChangedEventArgs(dataGridViewRow, DataGridViewElementStates.Displayed);
                    OnRowStateChanged(-1 /*rowIndex*/, dgvrsce);
                }
            }
            lstRows.Clear();
        }

        internal void OnClearingColumns()
        {
            CurrentCell = null;

            // Rows need to be cleared first. There cannot be rows without also having columns.
            Rows.ClearInternal(false /*recreateNewRow*/);

            // Reset sort related variables.
            sortedColumn = null;
            sortOrder = SortOrder.None;

            // selectedBandIndexes, individualSelectedCells & individualReadOnlyCells cleared in OnClearingRows.
        }

        internal void OnClearingRows()
        {
            // Build a list of displayed rows in order to be able to raise their RowStateChanged(Displayed->false) events later on
            lstRows.Clear();
            int numDisplayedRows = displayedBandsInfo.NumDisplayedFrozenRows + displayedBandsInfo.NumDisplayedScrollingRows;
            if (numDisplayedRows > 0)
            {
                lstRows.Capacity = numDisplayedRows;
                int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Displayed);
                while (numDisplayedRows > 0 && rowIndex != -1)
                {
                    lstRows.Add(Rows[rowIndex]);
                    numDisplayedRows--;
                    if (numDisplayedRows > 0)
                    {
                        rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Displayed);
                    }
                }
            }

            CurrentCell = null;

            newRowIndex = -1;
            dataGridViewState2[DATAGRIDVIEWSTATE2_raiseSelectionChanged] = selectedBandIndexes.Count > 0 ||
                                                                                individualSelectedCells.Count > 0;
            selectedBandIndexes.Clear();
            if (selectedBandSnapshotIndexes != null)
            {
                selectedBandSnapshotIndexes.Clear();
            }
            individualSelectedCells.Clear();
            individualReadOnlyCells.Clear();
        }

        protected virtual void OnColumnAdded(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }
            if (Events[EVENT_DATAGRIDVIEWCOLUMNADDED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnColumnCollectionChanged_PreNotification(CollectionChangeEventArgs ccea)
        {
            // we need to map columns w/ DataPropertyName to bound columns
            if (DataSource != null && !dataGridViewOper[DATAGRIDVIEWOPER_inRefreshColumns])
            {
                if (ccea.Action == CollectionChangeAction.Add)
                {
                    DataGridViewColumn dataGridViewColumn = (DataGridViewColumn)ccea.Element;
                    if (dataGridViewColumn.DataPropertyName.Length != 0)
                    {
                        MapDataGridViewColumnToDataBoundField(dataGridViewColumn);
                    }
                }
                else if (ccea.Action == CollectionChangeAction.Refresh)
                {
                    for (int i = 0; i < Columns.Count; i++)
                    {
                        if (Columns[i].DataPropertyName.Length != 0)
                        {
                            MapDataGridViewColumnToDataBoundField(Columns[i]);
                        }
                    }
                }
            }

            ResetUIState(false /*useRowShortcut*/, false /*computeVisibleRows*/);
        }

        internal void OnColumnCollectionChanged_PostNotification(DataGridViewColumn dataGridViewColumn)
        {
            if (Columns.Count != 0 && Rows.Count == 0)
            {
                if (DataSource != null && !dataGridViewOper[DATAGRIDVIEWOPER_inRefreshColumns])
                {
                    // this will create the 'add new row' when AllowUserToAddRowsInternal == true
                    RefreshRows(true /*scrollIntoView*/);
                }
                else if (AllowUserToAddRowsInternal)
                {
                    AddNewRow(false);
                }
            }
            if (AutoSize && (dataGridViewColumn == null || dataGridViewColumn.Visible))
            {
                LayoutTransaction.DoLayout(ParentInternal, this, PropertyNames.Columns);
            }
        }

        internal void OnColumnCommonChange(int columnIndex)
        {
            OnColumnGlobalAutoSize(columnIndex);
        }

        protected virtual void OnColumnContextMenuStripChanged(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }
            if (Events[EVENT_DATAGRIDVIEWCOLUMNCONTEXTMENUSTRIPCHANGED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnColumnDataPropertyNameChanged(DataGridViewColumn dataGridViewColumn)
        {
            OnColumnDataPropertyNameChanged(new DataGridViewColumnEventArgs(dataGridViewColumn));
        }

        protected virtual void OnColumnDataPropertyNameChanged(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }
            // map the dataGridView column to some data field
            if (DataSource != null && e.Column.DataPropertyName.Length != 0 && !dataGridViewOper[DATAGRIDVIEWOPER_inRefreshColumns])
            {
                MapDataGridViewColumnToDataBoundField(e.Column);
            }
            else if (DataSource != null && e.Column.DataPropertyName.Length == 0)
            {
                if (e.Column.IsDataBound)
                {
                    e.Column.IsDataBoundInternal = false;
                    e.Column.BoundColumnIndex = -1;
                    e.Column.BoundColumnConverter = null;
                    InvalidateColumnInternal(e.Column.Index);
                }
            }

            if (Events[EVENT_DATAGRIDVIEWCOLUMNDATAPROPERTYNAMECHANGED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnColumnDefaultCellStyleChanged(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }
            OnColumnGlobalAutoSize(e.Column.Index);

            if (Events[EVENT_DATAGRIDVIEWCOLUMNDEFAULTCELLSTYLECHANGED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnColumnDisplayIndexChanged(DataGridViewColumn dataGridViewColumn)
        {
            Debug.Assert(dataGridViewColumn != null);
            DataGridViewColumnEventArgs dgvce = new DataGridViewColumnEventArgs(dataGridViewColumn);
            OnColumnDisplayIndexChanged(dgvce);
        }

        internal void OnColumnDisplayIndexChanging(DataGridViewColumn dataGridViewColumn, int newDisplayIndex)
        {
            Debug.Assert(dataGridViewColumn != null);
            Debug.Assert(newDisplayIndex != dataGridViewColumn.DisplayIndex);

            if (dataGridViewOper[DATAGRIDVIEWOPER_inDisplayIndexAdjustments])
            {
                // We are within columns display indexes adjustments. We do not allow changing display indexes while adjusting them.
                throw new InvalidOperationException(SR.DataGridView_CannotAlterDisplayIndexWithinAdjustments);
            }

            // Throws an exception if the requested move is illegal
            CorrectColumnFrozenStatesForMove(dataGridViewColumn, newDisplayIndex);

            try
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inDisplayIndexAdjustments] = true;

                // Move is legal - let's adjust the affected display indexes.
                if (newDisplayIndex < dataGridViewColumn.DisplayIndex)
                {
                    // DisplayIndex decreases. All columns with newDisplayIndex <= DisplayIndex < dataGridViewColumn.DisplayIndex
                    // get their DisplayIndex incremented.
                    foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                    {
                        if (newDisplayIndex <= dataGridViewColumnTmp.DisplayIndex && dataGridViewColumnTmp.DisplayIndex < dataGridViewColumn.DisplayIndex)
                        {
                            dataGridViewColumnTmp.DisplayIndexInternal = dataGridViewColumnTmp.DisplayIndex + 1;
                            dataGridViewColumnTmp.DisplayIndexHasChanged = true; // OnColumnDisplayIndexChanged needs to be raised later on
                        }
                    }
                }
                else
                {
                    // DisplayIndex increases. All columns with dataGridViewColumn.DisplayIndex < DisplayIndex <= newDisplayIndex
                    // get their DisplayIndex incremented.
                    foreach (DataGridViewColumn dataGridViewColumnTmp in Columns)
                    {
                        if (dataGridViewColumn.DisplayIndex < dataGridViewColumnTmp.DisplayIndex && dataGridViewColumnTmp.DisplayIndex <= newDisplayIndex)
                        {
                            dataGridViewColumnTmp.DisplayIndexInternal = dataGridViewColumnTmp.DisplayIndex - 1;
                            dataGridViewColumnTmp.DisplayIndexHasChanged = true; // OnColumnDisplayIndexChanged needs to be raised later on
                        }
                    }
                }
            }
            finally
            {
                dataGridViewOper[DATAGRIDVIEWOPER_inDisplayIndexAdjustments] = false;
            }

            // Note that displayIndex of moved column is updated by caller.
        }

        protected virtual void OnColumnDisplayIndexChanged(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }

            Debug.Assert(dataGridViewOper[DATAGRIDVIEWOPER_inDisplayIndexAdjustments]);
#if DEBUG
            Debug.Assert(Columns.VerifyColumnDisplayIndexes());

#endif
            if (Events[EVENT_DATAGRIDVIEWCOLUMNDISPLAYINDEXCHANGED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnColumnDisplayIndexChanged_PreNotification()
        {
            Debug.Assert(dataGridViewOper[DATAGRIDVIEWOPER_inDisplayIndexAdjustments]);

            // column.DisplayIndex changed - this may require a complete re-layout of the control
            Columns.InvalidateCachedColumnsOrder();

            PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
            if (editingControl != null)
            {
                PositionEditingControl(true, true, false);
            }
            Invalidate(Rectangle.Union(layout.ColumnHeaders, layout.Data));
        }

        internal void OnColumnDisplayIndexChanged_PostNotification()
        {
            // Notifications for adjusted display indexes.
            FlushDisplayIndexChanged(true /*raiseEvent*/);
        }

        protected virtual void OnColumnDividerDoubleClick(DataGridViewColumnDividerDoubleClickEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWCOLUMNDIVIDERDOUBLECLICK] is DataGridViewColumnDividerDoubleClickEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            if (!e.Handled && e.Button == MouseButtons.Left && e.ColumnIndex < Columns.Count)
            {
                if (e.ColumnIndex == -1)
                {
                    AutoResizeRowHeadersWidth(DataGridViewRowHeadersWidthSizeMode.AutoSizeToDisplayedHeaders,
                                              true /*fixedColumnHeadersHeight*/,
                                              true /*fixedRowsHeight*/);
                }
                else
                {
                    DataGridViewAutoSizeColumnMode inheritedAutoSizeMode = Columns[e.ColumnIndex].InheritedAutoSizeMode;
                    if (inheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.None || inheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                    {
                        AutoResizeColumnInternal(e.ColumnIndex, DataGridViewAutoSizeColumnCriteriaInternal.Header | DataGridViewAutoSizeColumnCriteriaInternal.AllRows, true /*fixedHeight*/);
                    }
                    else
                    {
                        AutoResizeColumnInternal(e.ColumnIndex, (DataGridViewAutoSizeColumnCriteriaInternal)inheritedAutoSizeMode, true /*fixedHeight*/);
                    }
                }
            }
        }

        protected virtual void OnColumnDividerWidthChanged(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }
            OnColumnGlobalAutoSize(e.Column.Index);

            if (Events[EVENT_DATAGRIDVIEWCOLUMNDIVIDERWIDTHCHANGED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnColumnFillWeightChanged(DataGridViewColumn dataGridViewColumn)
        {
            if (dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
            {
                // UsedFillWeight properties need to be re-evaluated
                dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty] = true;
                // Adjust filling columns based on new weight of this column
                PerformLayoutPrivate(false /*useRowShortcut*/, true /*computeVisibleRows*/, false /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
            }
        }

        internal void OnColumnFillWeightChanging(DataGridViewColumn dataGridViewColumn, float fillWeight)
        {
            if (InAdjustFillingColumns)
            {
                throw new InvalidOperationException(SR.DataGridView_CannotAlterAutoFillColumnParameter);
            }

            // Make sure the sum of the column weights does not exceed ushort.MaxValue
            float weightSum = Columns.GetColumnsFillWeight(DataGridViewElementStates.None) - dataGridViewColumn.FillWeight + fillWeight;
            if (weightSum > (float)ushort.MaxValue)
            {
                throw new InvalidOperationException(string.Format(SR.DataGridView_WeightSumCannotExceedLongMaxValue, ushort.MaxValue));
            }
        }

        private void OnColumnGlobalAutoSize(int columnIndex)
        {
            Debug.Assert(columnIndex >= 0 && columnIndex < Columns.Count);

            if (!Columns[columnIndex].Visible)
            {
                return;
            }

            InvalidateColumnInternal(columnIndex);

            if (noAutoSizeCount > 0)
            {
                return;
            }

            bool fixedHeight = (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) == 0;

            DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaInternal = (DataGridViewAutoSizeColumnCriteriaInternal)Columns[columnIndex].InheritedAutoSizeMode;
            if (autoSizeColumnCriteriaInternal != DataGridViewAutoSizeColumnCriteriaInternal.None &&
                autoSizeColumnCriteriaInternal != DataGridViewAutoSizeColumnCriteriaInternal.Fill)
            {
                AutoResizeColumnInternal(columnIndex, autoSizeColumnCriteriaInternal, fixedHeight);
            }

            // Autosize rows and column headers if needed
            if (!fixedHeight)
            {
                AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
            }
            if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
            {
                AutoResizeColumnHeadersHeight(columnIndex, true /*fixedRowHeadersWidth*/, true /*fixedColumnWidth*/);
            }
            if (!fixedHeight &&
                autoSizeColumnCriteriaInternal != DataGridViewAutoSizeColumnCriteriaInternal.None &&
                autoSizeColumnCriteriaInternal != DataGridViewAutoSizeColumnCriteriaInternal.Fill)
            {
                // Second round of column autosizing with 1 degree of freedom
                AutoResizeColumnInternal(columnIndex, autoSizeColumnCriteriaInternal, true /*fixedHeight*/);
            }
        }

        protected virtual void OnColumnHeaderCellChanged(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }

            OnColumnHeaderGlobalAutoSize(e.Column.Index);

            if (Events[EVENT_DATAGRIDVIEWCOLUMNHEADERCELLCHANGED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnColumnHeaderGlobalAutoSize(int columnIndex)
        {
            if (!ColumnHeadersVisible)
            {
                return;
            }

            InvalidateCellPrivate(columnIndex, -1);

            if (noAutoSizeCount > 0)
            {
                return;
            }

            DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaInternal = (DataGridViewAutoSizeColumnCriteriaInternal)Columns[columnIndex].InheritedAutoSizeMode;
            DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaFiltered = autoSizeColumnCriteriaInternal & DataGridViewAutoSizeColumnCriteriaInternal.Header;
            bool fixedColumnWidth = autoSizeColumnCriteriaFiltered == 0;

            if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
            {
                AutoResizeColumnHeadersHeight(columnIndex, true /*fixedRowHeadersWidth*/, fixedColumnWidth);
            }

            if (!fixedColumnWidth)
            {
                Debug.Assert(autoSizeColumnCriteriaInternal != DataGridViewAutoSizeColumnCriteriaInternal.None);
                Debug.Assert(autoSizeColumnCriteriaInternal != DataGridViewAutoSizeColumnCriteriaInternal.Fill);
                bool fixedHeight = (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) == 0;
                AutoResizeColumnInternal(columnIndex, autoSizeColumnCriteriaInternal, fixedHeight);
                if (!fixedHeight)
                {
                    AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
                    // Second round of column autosizing with 1 degree of freedom
                    AutoResizeColumnInternal(columnIndex, autoSizeColumnCriteriaInternal, true /*fixedHeight*/);
                }
                if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
                {
                    // Second round of column headers autosizing with 1 degree of freedom
                    AutoResizeColumnHeadersHeight(columnIndex, true /*fixedRowHeadersWidth*/, true /*fixedColumnWidth*/);
                }
            }
        }

        protected virtual void OnColumnHeaderMouseClick(DataGridViewCellMouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left &&
                SelectionMode != DataGridViewSelectionMode.FullColumnSelect &&
                SelectionMode != DataGridViewSelectionMode.ColumnHeaderSelect)
            {
                DataGridViewColumn dataGridViewColumn = Columns[e.ColumnIndex];

                if (CanSort(dataGridViewColumn))
                {
                    ListSortDirection direction = ListSortDirection.Ascending;

                    if (sortedColumn == dataGridViewColumn)
                    {
                        Debug.Assert(sortOrder != SortOrder.None);
                        if (sortOrder == SortOrder.Ascending)
                        {
                            direction = ListSortDirection.Descending;
                        }
                    }

                    if ((DataSource == null) ||
                        (DataSource != null &&
                         (dataConnection.List is IBindingList) &&
                         ((IBindingList)dataConnection.List).SupportsSorting &&
                         dataGridViewColumn.IsDataBound))
                    {
                        Sort(dataGridViewColumn, direction);
                    }
                }
            }

            if (Events[EVENT_DATAGRIDVIEWCOLUMNHEADERMOUSECLICK] is DataGridViewCellMouseEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnColumnHeaderMouseDoubleClick(DataGridViewCellMouseEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWCOLUMNHEADERMOUSEDOUBLECLICK] is DataGridViewCellMouseEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnColumnHeaderMouseDown(HitTestInfo hti, bool isShiftDown, bool isControlDown)
        {
            Debug.Assert(hti.Type == DataGridViewHitTestType.ColumnHeader);
            noSelectionChangeCount++;
            try
            {
                switch (SelectionMode)
                {
                    case DataGridViewSelectionMode.CellSelect:
                    case DataGridViewSelectionMode.FullRowSelect:
                    case DataGridViewSelectionMode.RowHeaderSelect:
                        break;

                    case DataGridViewSelectionMode.FullColumnSelect:
                    case DataGridViewSelectionMode.ColumnHeaderSelect:
                        {
                            bool select = true;
                            if (isControlDown && Columns[hti.col].Selected)
                            {
                                select = false;
                            }
                            if (select)
                            {
                                int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible);
                                if (rowIndex > -1 && hti.col != ptCurrentCell.X)
                                {
                                    // Make sure we will be able to scroll into view
                                    int oldCurrentCellX = ptCurrentCell.X;
                                    int oldCurrentCellY = ptCurrentCell.Y;
                                    if (!EndEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit | DataGridViewDataErrorContexts.CurrentCellChange,
                                                DataGridViewValidateCellInternal.Always /*validateCell*/,
                                                true /*fireCellLeave*/,
                                                true /*fireCellEnter*/,
                                                rowIndex != ptCurrentCell.Y /*fireRowLeave*/,
                                                rowIndex != ptCurrentCell.Y /*fireRowEnter*/,
                                                false /*fireLeave*/,
                                                EditMode != DataGridViewEditMode.EditOnEnter /*keepFocus*/,
                                                true /*resetCurrentCell*/,
                                                false /*resetAnchorCell*/))
                                    {
                                        // Just cancel operation silently instead of throwing InvalidOperationException
                                        return;
                                    }
                                    if (rowIndex != oldCurrentCellY && oldCurrentCellY != -1)
                                    {
                                        DataGridViewCell dataGridViewCellTmp = null;
                                        if (IsInnerCellOutOfBounds(oldCurrentCellX, oldCurrentCellY))
                                        {
                                            return;
                                        }
                                        if (OnRowValidating(ref dataGridViewCellTmp, oldCurrentCellX, oldCurrentCellY))
                                        {
                                            // Row validation was cancelled
                                            if (IsInnerCellOutOfBounds(oldCurrentCellX, oldCurrentCellY))
                                            {
                                                return;
                                            }
                                            OnRowEnter(ref dataGridViewCellTmp, oldCurrentCellX, oldCurrentCellY, true /*canCreateNewRow*/, true /*validationFailureOccurred*/);
                                            if (IsInnerCellOutOfBounds(oldCurrentCellX, oldCurrentCellY))
                                            {
                                                return;
                                            }
                                            OnCellEnter(ref dataGridViewCellTmp, oldCurrentCellX, oldCurrentCellY);
                                            return;
                                        }
                                        if (IsInnerCellOutOfBounds(oldCurrentCellX, oldCurrentCellY))
                                        {
                                            return;
                                        }
                                        OnRowValidated(ref dataGridViewCellTmp, oldCurrentCellX, oldCurrentCellY);
                                    }
                                }
                                if (IsColumnOutOfBounds(hti.col))
                                {
                                    return;
                                }

                                bool selectColumnRange = false;
                                trackColumn = hti.col;
                                trackColumnEdge = -1;
                                if (MultiSelect &&
                                    isShiftDown &&
                                    ptAnchorCell.X > -1 &&
                                    Columns[ptAnchorCell.X].Selected)
                                {
                                    selectColumnRange = true;
                                }
                                if (!MultiSelect || !isControlDown || isShiftDown)
                                {
                                    Debug.Assert(MultiSelect || selectedBandIndexes.Count <= 1);
                                    int bandIndex = 0;
                                    bool switchedToBulkPaint = false;
                                    if (selectedBandIndexes.Count > DATAGRIDVIEW_bulkPaintThreshold)
                                    {
                                        inBulkPaintCount++;
                                        switchedToBulkPaint = true;
                                    }
                                    try
                                    {
                                        while (bandIndex < selectedBandIndexes.Count)
                                        {
                                            if (selectedBandIndexes[bandIndex] != hti.col)
                                            {
                                                // deselect currently selected column
                                                SetSelectedColumnCore(selectedBandIndexes[bandIndex], false);
                                            }
                                            else
                                            {
                                                bandIndex++;
                                            }
                                        }
                                        if (SelectionMode == DataGridViewSelectionMode.ColumnHeaderSelect)
                                        {
                                            RemoveIndividuallySelectedCells();
                                        }
                                        else
                                        {
                                            Debug.Assert(individualSelectedCells.Count == 0);
                                        }
                                    }
                                    finally
                                    {
                                        if (switchedToBulkPaint)
                                        {
                                            ExitBulkPaint(-1, -1);
                                        }
                                    }
                                }
                                if (MultiSelect && dataGridViewOper[DATAGRIDVIEWOPER_trackMouseMoves])
                                {
                                    dataGridViewOper[DATAGRIDVIEWOPER_trackColSelect] = true;
                                }
                                if (selectColumnRange)
                                {
                                    if (Columns.DisplayInOrder(ptAnchorCell.X, hti.col))
                                    {
                                        SelectColumnRange(ptAnchorCell.X, hti.col, true);
                                    }
                                    else
                                    {
                                        SelectColumnRange(hti.col, ptAnchorCell.X, true);
                                    }
                                }
                                else if (!selectedBandIndexes.Contains(hti.col))
                                {
                                    SetSelectedColumnCore(hti.col, true);
                                }
                                // set current cell to the top most visible cell in the column
                                if (rowIndex != -1)
                                {
                                    if (hti.col != ptCurrentCell.X)
                                    {
                                        if (IsInnerCellOutOfBounds(hti.col, rowIndex))
                                        {
                                            return;
                                        }
                                        bool success = ScrollIntoView(hti.col, rowIndex, false);
                                        Debug.Assert(success);
                                        if (IsInnerCellOutOfBounds(hti.col, rowIndex))
                                        {
                                            return;
                                        }
                                        success = SetCurrentCellAddressCore(hti.col, rowIndex, !isShiftDown, false, true);
                                        Debug.Assert(success);
                                    }
                                    else if (-1 != ptCurrentCell.X)
                                    {
                                        // Potentially have to give focus back to the current edited cell.
                                        bool success = SetCurrentCellAddressCore(ptCurrentCell.X, ptCurrentCell.Y, false /*setAnchorCellAddress*/, false /*validateCurrentCell*/, false /*throughMouseClick*/);
                                        Debug.Assert(success);
                                    }
                                }
                                else
                                {
                                    Debug.Assert(CurrentCellAddress == new Point(-1, -1));
                                }
                            }
                            else
                            {
                                Debug.Assert(selectedBandIndexes.Contains(hti.col));
                                SetSelectedColumnCore(hti.col, false);
                            }
                            break;
                        }
                }
            }
            finally
            {
                NoSelectionChangeCount--;
            }
        }

        protected virtual void OnColumnHeadersBorderStyleChanged(EventArgs e)
        {
            PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
            Invalidate();

            if (Events[EVENT_DATAGRIDVIEWCOLUMNHEADERSBORDERSTYLECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnColumnHeadersDefaultCellStyleChanged(EventArgs e)
        {
            if (ColumnHeadersVisible)
            {
                Invalidate(Rectangle.Union(layout.TopLeftHeader, layout.ColumnHeaders));

                if (!(e is DataGridViewCellStyleChangedEventArgs dgvcsce) || dgvcsce.ChangeAffectsPreferredSize)
                {
                    OnColumnHeadersGlobalAutoSize();
                    if (editingControl != null)
                    {
                        PositionEditingControl(true /*setLocation*/, true /*setSize*/, false /*setFocus*/);
                    }
                }
            }

            if (Events[EVENT_DATAGRIDVIEWCOLUMNHEADERSDEFAULTCELLSTYLECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnColumnHeadersGlobalAutoSize()
        {
            if (noAutoSizeCount > 0)
            {
                return;
            }

            bool fixedRowHeadersWidth = rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing ||
                                        rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.DisableResizing;
            bool fixedColumnHeadersHeight = ColumnHeadersHeightSizeMode != DataGridViewColumnHeadersHeightSizeMode.AutoSize;

            if (!fixedColumnHeadersHeight)
            {
                AutoResizeColumnHeadersHeight(fixedRowHeadersWidth, false /*fixedColumnsWidth*/);
            }

            if (!fixedRowHeadersWidth)
            {
                AutoResizeRowHeadersWidth(rowHeadersWidthSizeMode, true /*fixedColumnHeadersHeight*/, false /*fixedRowsHeight*/);
            }

            // Autosize columns
            bool columnAutoSized = AutoResizeAllVisibleColumnsInternal(DataGridViewAutoSizeColumnCriteriaInternal.Header, false /*fixedHeight*/);

            if (!fixedRowHeadersWidth || columnAutoSized)
            {
                // Autosize rows
                AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
            }

            if (!fixedColumnHeadersHeight)
            {
                // Second round of column headers autosizing
                AutoResizeColumnHeadersHeight(true /*fixedRowHeadersWidth*/, true /*fixedColumnsWidth*/);
            }

            if (!fixedRowHeadersWidth)
            {
                // Second round of row headers autosizing
                AutoResizeRowHeadersWidth(rowHeadersWidthSizeMode, true /*fixedColumnHeadersHeight*/, true /*fixedRowsHeight*/);
            }

            // Second round of columns autosizing
            AutoResizeAllVisibleColumnsInternal(DataGridViewAutoSizeColumnCriteriaInternal.Header, true /*fixedHeight*/);
        }

        protected virtual void OnColumnHeadersHeightChanged(EventArgs e)
        {
            if (editingControl != null)
            {
                PositionEditingControl(true, false, false);
            }

            UpdateMouseEnteredCell(null /*HitTestInfo*/, null /*MouseEventArgs*/);

            OnColumnHeadersGlobalAutoSize();

            if (Events[EVENT_DATAGRIDVIEWCOLUMNHEADERSHEIGHTCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnColumnHeadersHeightSizeModeChanged(DataGridViewAutoSizeModeEventArgs e)
        {
            if (columnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
            {
                if (!e.PreviousModeAutoSized)
                {
                    // Save current column headers height for later reuse
                    cachedColumnHeadersHeight = ColumnHeadersHeight;
                }
                AutoResizeColumnHeadersHeight(true /*fixedRowHeadersWidth*/, true /*fixedColumnsWidth*/);
            }
            else if (e.PreviousModeAutoSized)
            {
                ColumnHeadersHeight = cachedColumnHeadersHeight;
            }

            if (Events[EVENT_DATAGRIDVIEWCOLUMNHEADERSHEIGHTSIZEMODECHANGED] is DataGridViewAutoSizeModeEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnColumnHidden(DataGridViewColumn dataGridViewColumn)
        {
            Debug.Assert(dataGridViewColumn != null);
            if (dataGridViewColumn.Displayed)
            {
                dataGridViewColumn.Displayed = false;
                DataGridViewColumnStateChangedEventArgs dgvrsce = new DataGridViewColumnStateChangedEventArgs(dataGridViewColumn, DataGridViewElementStates.Displayed);
                OnColumnStateChanged(dgvrsce);
            }
        }

        internal void OnColumnMinimumWidthChanging(DataGridViewColumn dataGridViewColumn, int minimumWidth)
        {
            if (dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill && dataGridViewColumn.Width < minimumWidth)
            {
                // Force the filled column's width to be minimumWidth
                try
                {
                    dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty] = true;
                    dataGridViewColumn.DesiredMinimumWidth = minimumWidth;
                    PerformLayoutPrivate(false /*useRowShortcut*/, true /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
                }
                finally
                {
                    dataGridViewColumn.DesiredMinimumWidth = 0;
                }
                Debug.Assert(dataGridViewColumn.Width == minimumWidth);
            }
        }

        protected virtual void OnColumnMinimumWidthChanged(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }
            if (e.Column.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
            {
                // Column's width may adjust smaller
                PerformLayoutPrivate(false /*useRowShortcut*/, true /*computeVisibleRows*/, false /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
            }

            if (Events[EVENT_DATAGRIDVIEWCOLUMNMINIMUMWIDTHCHANGED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnColumnNameChanged(DataGridViewColumn dataGridViewColumn)
        {
            Debug.Assert(dataGridViewColumn != null);
            DataGridViewColumnEventArgs dgvce = new DataGridViewColumnEventArgs(dataGridViewColumn);
            OnColumnNameChanged(dgvce);
        }

        protected virtual void OnColumnNameChanged(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }
            // Check if the column name is used as is in the column header
            DataGridViewColumn dataGridViewColumn = e.Column;

            if (dataGridViewColumn.HasHeaderCell && dataGridViewColumn.HeaderCell.Value is string &&
                string.Compare((string)dataGridViewColumn.HeaderCell.Value, dataGridViewColumn.Name, false, CultureInfo.InvariantCulture) == 0)
            {
                InvalidateCellPrivate(dataGridViewColumn.Index, -1);

                DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaInternal = (DataGridViewAutoSizeColumnCriteriaInternal)dataGridViewColumn.InheritedAutoSizeMode;
                DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaFiltered = autoSizeColumnCriteriaInternal & DataGridViewAutoSizeColumnCriteriaInternal.Header;
                bool fixedColumnWidth = autoSizeColumnCriteriaFiltered == 0 || !ColumnHeadersVisible;
                if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
                {
                    AutoResizeColumnHeadersHeight(dataGridViewColumn.Index, true /*fixedRowHeadersWidth*/, fixedColumnWidth);
                }
                if (!fixedColumnWidth)
                {
                    bool fixedHeight = (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) == 0;
                    AutoResizeColumnInternal(dataGridViewColumn.Index, autoSizeColumnCriteriaInternal, fixedHeight);
                    if (!fixedHeight)
                    {
                        AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
                        // Second round of column autosizing
                        AutoResizeColumnInternal(dataGridViewColumn.Index, autoSizeColumnCriteriaInternal, true /*fixedHeight*/);
                    }
                    if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
                    {
                        // Second round of column headers autosizing
                        AutoResizeColumnHeadersHeight(dataGridViewColumn.Index, true /*fixedRowHeadersWidth*/, true /*fixedColumnWidth*/);
                    }
                }
            }

            if (Events[EVENT_DATAGRIDVIEWCOLUMNNAMECHANGED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnColumnRemoved(DataGridViewColumn dataGridViewColumn)
        {
            Debug.Assert(dataGridViewColumn != null);
            Debug.Assert(dataGridViewColumn.DataGridView == null);
            OnColumnRemoved(new DataGridViewColumnEventArgs(dataGridViewColumn));
        }

        protected virtual void OnColumnRemoved(DataGridViewColumnEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWCOLUMNREMOVED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnColumnSelectMouseMove(HitTestInfo hti)
        {
            Debug.Assert(hti.col >= 0);
            Debug.Assert(MultiSelect);

            if (ptCurrentCell.X != -1 &&
                hti.col != ptCurrentCell.X &&
                !CommitEditForOperation(hti.col, ptCurrentCell.Y, true))
            {
                // Return silently if validating/commit/abort failed
                return;
            }
            if (IsColumnOutOfBounds(hti.col))
            {
                return;
            }

            noSelectionChangeCount++;
            try
            {
                if (trackColumnEdge >= 0 && (Columns.DisplayInOrder(trackColumn, trackColumnEdge) || trackColumnEdge == trackColumn) && Columns.DisplayInOrder(trackColumnEdge, hti.col))
                {
                    DataGridViewColumn dataGridViewColumn = Columns.GetNextColumn(Columns[trackColumnEdge], DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                    Debug.Assert(dataGridViewColumn != null);
                    SelectColumnRange(dataGridViewColumn.Index, hti.col, true);
                    trackColumnEdge = hti.col;
                }
                else if (trackColumnEdge >= 0 && Columns.DisplayInOrder(trackColumn, trackColumnEdge) && Columns.DisplayInOrder(hti.col, trackColumnEdge) && (Columns.DisplayInOrder(trackColumn, hti.col) || hti.col == trackColumn))
                {
                    DataGridViewColumn dataGridViewColumn = Columns.GetNextColumn(Columns[hti.col], DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                    Debug.Assert(dataGridViewColumn != null);
                    SelectColumnRange(dataGridViewColumn.Index, trackColumnEdge, false);
                    trackColumnEdge = hti.col;
                }
                else if (trackColumnEdge == -1 && Columns.DisplayInOrder(trackColumn, hti.col))
                {
                    DataGridViewColumn dataGridViewColumn = Columns.GetNextColumn(Columns[trackColumn], DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                    Debug.Assert(dataGridViewColumn != null);
                    SelectColumnRange(dataGridViewColumn.Index, hti.col, true);
                    trackColumnEdge = hti.col;
                }
                else if (trackColumnEdge >= 0 && (Columns.DisplayInOrder(trackColumnEdge, trackColumn) || trackColumnEdge == trackColumn) && Columns.DisplayInOrder(hti.col, trackColumnEdge))
                {
                    DataGridViewColumn dataGridViewColumn = Columns.GetPreviousColumn(Columns[trackColumnEdge], DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                    Debug.Assert(dataGridViewColumn != null);
                    SelectColumnRange(hti.col, dataGridViewColumn.Index, true);
                    trackColumnEdge = hti.col;
                }
                else if (trackColumnEdge >= 0 && Columns.DisplayInOrder(trackColumnEdge, trackColumn) && Columns.DisplayInOrder(trackColumnEdge, hti.col) && (Columns.DisplayInOrder(hti.col, trackColumn) || hti.col == trackColumn))
                {
                    DataGridViewColumn dataGridViewColumn = Columns.GetPreviousColumn(Columns[hti.col], DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                    Debug.Assert(dataGridViewColumn != null);
                    SelectColumnRange(trackColumnEdge, dataGridViewColumn.Index, false);
                    trackColumnEdge = hti.col;
                }
                else if (trackColumnEdge == -1 && Columns.DisplayInOrder(hti.col, trackColumn))
                {
                    DataGridViewColumn dataGridViewColumn = Columns.GetPreviousColumn(Columns[trackColumn], DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                    Debug.Assert(dataGridViewColumn != null);
                    SelectColumnRange(hti.col, dataGridViewColumn.Index, true);
                    trackColumnEdge = hti.col;
                }
                else if (trackColumnEdge >= 0 && Columns.DisplayInOrder(trackColumn, trackColumnEdge) && Columns.DisplayInOrder(hti.col, trackColumn))
                {
                    DataGridViewColumn dataGridViewColumn = Columns.GetNextColumn(Columns[trackColumn], DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                    Debug.Assert(dataGridViewColumn != null);
                    SelectColumnRange(dataGridViewColumn.Index, trackColumnEdge, false);
                    dataGridViewColumn = Columns.GetPreviousColumn(Columns[trackColumn], DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                    Debug.Assert(dataGridViewColumn != null);
                    SelectColumnRange(hti.col, dataGridViewColumn.Index, true);
                    trackColumnEdge = hti.col;
                }
                else if (trackColumnEdge >= 0 && Columns.DisplayInOrder(trackColumn, hti.col) && Columns.DisplayInOrder(trackColumnEdge, trackColumn))
                {
                    DataGridViewColumn dataGridViewColumn = Columns.GetPreviousColumn(Columns[trackColumn], DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                    Debug.Assert(dataGridViewColumn != null);
                    SelectColumnRange(trackColumnEdge, dataGridViewColumn.Index, false);
                    dataGridViewColumn = Columns.GetNextColumn(Columns[trackColumn], DataGridViewElementStates.Visible, DataGridViewElementStates.None);
                    Debug.Assert(dataGridViewColumn != null);
                    SelectColumnRange(dataGridViewColumn.Index, hti.col, true);
                    trackColumnEdge = hti.col;
                }
            }
            finally
            {
                NoSelectionChangeCount--;
            }

            if (ptCurrentCell.X != -1 && hti.col != ptCurrentCell.X)
            {
                if (ptCurrentCell.Y == -1 || IsColumnOutOfBounds(hti.col))
                {
                    return;
                }
                bool success = SetCurrentCellAddressCore(hti.col,
                    ptCurrentCell.Y,
                    false /*setAnchorCellAddress*/,
                    false /*validateCurrentCell*/,
                    false /*throughMouseClick*/);
                Debug.Assert(success);
            }
        }

        private void OnColumnsGlobalAutoSize()
        {
            InvalidateData();

            if (noAutoSizeCount > 0)
            {
                return;
            }

            // Auto-size columms if needed
            bool fixedHeight = (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) == 0;
            bool columnAutoSized = AutoResizeAllVisibleColumnsInternal(DataGridViewAutoSizeColumnCriteriaInternal.AllRows | DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows, fixedHeight);
            // Autosize rows if needed
            if (!fixedHeight)
            {
                if (columnAutoSized)
                {
                    AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
                }
                // Second round of columns autosizing
                AutoResizeAllVisibleColumnsInternal(DataGridViewAutoSizeColumnCriteriaInternal.AllRows | DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows, true /*fixedHeight*/);
            }
        }

        internal void OnColumnSortModeChanged(DataGridViewColumn dataGridViewColumn)
        {
            Debug.Assert(dataGridViewColumn != null);
            DataGridViewColumnEventArgs dgvce = new DataGridViewColumnEventArgs(dataGridViewColumn);
            OnColumnSortModeChanged(dgvce);
        }

        protected virtual void OnColumnSortModeChanged(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }
            DataGridViewColumn dataGridViewColumn = e.Column;

            if (dataGridViewColumn.HasHeaderCell)
            {
                if (dataGridViewColumn.SortMode == DataGridViewColumnSortMode.NotSortable ||
                    (dataGridViewColumn.SortMode == DataGridViewColumnSortMode.Programmatic && SortedColumn == dataGridViewColumn))
                {
                    dataGridViewColumn.HeaderCell.SortGlyphDirection = SortOrder.None;
                    // This call will trigger OnSortGlyphDirectionChanged which in turn does
                    // this.sortedColumn = null; and InvalidateCellPrivate(e.Column.Index, -1);
                }
                // Potential resizing of the column headers and/or affected column.
                DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaInternal = (DataGridViewAutoSizeColumnCriteriaInternal)dataGridViewColumn.InheritedAutoSizeMode;
                DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaFiltered = autoSizeColumnCriteriaInternal & DataGridViewAutoSizeColumnCriteriaInternal.Header;
                bool fixedColumnWidth = autoSizeColumnCriteriaFiltered == 0 || !ColumnHeadersVisible;
                if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
                {
                    AutoResizeColumnHeadersHeight(dataGridViewColumn.Index, true /*fixedRowHeadersWidth*/, fixedColumnWidth);
                }
                if (!fixedColumnWidth)
                {
                    bool fixedHeight = (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) == 0;
                    AutoResizeColumnInternal(dataGridViewColumn.Index, autoSizeColumnCriteriaInternal, fixedHeight);
                    if (!fixedHeight)
                    {
                        AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
                        // Second round of column autosizing
                        AutoResizeColumnInternal(dataGridViewColumn.Index, autoSizeColumnCriteriaInternal, true /*fixedHeight*/);
                    }
                    if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
                    {
                        // Second round of column headers autosizing
                        AutoResizeColumnHeadersHeight(dataGridViewColumn.Index, true /*fixedRowHeadersWidth*/, true /*fixedColumnWidth*/);
                    }
                }
            }

            if (Events[EVENT_DATAGRIDVIEWCOLUMNSORTMODECHANGED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnColumnStateChanged(DataGridViewColumnStateChangedEventArgs e)
        {
            // column.Frozen | .Visible changed - this may require a complete re-layout of the control
            DataGridViewColumn dataGridViewColumn = e.Column;
            switch (e.StateChanged)
            {
                // At this point we assume that only the Selected state has an influence on the rendering of the column.
                // If there is a customer scenario where another state has an influence, the dev must invalidate the column by hand.
                // case DataGridViewElementStates.ReadOnly:
                // case DataGridViewElementStates.Resizable:

                case DataGridViewElementStates.Selected:
                    if (dataGridViewColumn.Visible && inBulkPaintCount == 0)
                    {
                        InvalidateColumnInternal(dataGridViewColumn.Index);
                    }
                    break;

                case DataGridViewElementStates.Frozen:
                    if (dataGridViewColumn.Visible)
                    {
                        if (dataGridViewColumn.Frozen)
                        {
                            // visible column became frozen
                            if (horizontalOffset >= dataGridViewColumn.Thickness)
                            {
                                Debug.Assert(Columns.DisplayInOrder(dataGridViewColumn.Index, displayedBandsInfo.FirstDisplayedScrollingCol));
                                horizontalOffset -= dataGridViewColumn.Thickness;
                            }
                            else
                            {
                                horizontalOffset = negOffset = 0;
                            }
                        }
                        else
                        {
                            // column was unfrozen - make it the first visible scrolling column if there is room
                            horizontalOffset = negOffset = 0;
                        }
                        if (horizScrollBar.Enabled)
                        {
                            horizScrollBar.Value = horizontalOffset;
                        }

                        // UsedFillWeight values need to be updated
                        dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty] = true;

                        PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, true /*repositionEditingControl*/);
                        Invalidate();
                    }
                    break;

                case DataGridViewElementStates.Visible:
                    if (!dataGridViewColumn.Visible && dataGridViewColumn.Displayed)
                    {
                        // Displayed column becomes invisible. Turns off the Displayed state.
                        dataGridViewColumn.Displayed = false;
                    }

                    // UsedFillWeight values need to be updated
                    dataGridViewState2[DATAGRIDVIEWSTATE2_usedFillWeightsDirty] = true;

                    PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, true /*repositionEditingControl*/);

                    bool autoSizeRows = (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0 ||
                                        ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.Header) != 0 &&
                                        RowHeadersVisible);
                    bool autoSizeColumn = false;
                    DataGridViewAutoSizeColumnMode autoSizeColumnMode = dataGridViewColumn.InheritedAutoSizeMode;
                    Debug.Assert(autoSizeColumnMode != DataGridViewAutoSizeColumnMode.NotSet);
                    if (autoSizeColumnMode != DataGridViewAutoSizeColumnMode.None &&
                        autoSizeColumnMode != DataGridViewAutoSizeColumnMode.Fill)
                    {
                        // Column autosizes
                        int width = dataGridViewColumn.ThicknessInternal;
                        if (dataGridViewColumn.Visible)
                        {
                            // Cache column's width before potential autosizing occurs
                            dataGridViewColumn.CachedThickness = width;
                            AutoResizeColumnInternal(dataGridViewColumn.Index, (DataGridViewAutoSizeColumnCriteriaInternal)autoSizeColumnMode, !autoSizeRows /*fixedHeight*/);
                            autoSizeColumn = true;
                        }
                        else if (width != dataGridViewColumn.CachedThickness)
                        {
                            // Columns that are made invisible in the collection take their non-autosized width
                            dataGridViewColumn.ThicknessInternal = Math.Max(dataGridViewColumn.MinimumWidth, dataGridViewColumn.CachedThickness);
                        }
                    }

                    if (autoSizeRows)
                    {
                        if (dataGridViewColumn.Visible)
                        {
                            AdjustExpandingRows(dataGridViewColumn.Index, true /*fixedWidth*/);
                        }
                        else
                        {
                            AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
                        }
                        if (autoSizeColumn)
                        {
                            // Second round of column autosizing
                            AutoResizeColumnInternal(dataGridViewColumn.Index, (DataGridViewAutoSizeColumnCriteriaInternal)autoSizeColumnMode, true /*fixedHeight*/);
                        }
                    }
                    else
                    {
                        Invalidate();
                    }
                    break;
            }

            if (Events[EVENT_DATAGRIDVIEWCOLUMNSTATECHANGED] is DataGridViewColumnStateChangedEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            if (e.StateChanged == DataGridViewElementStates.ReadOnly &&
                dataGridViewColumn.Index == ptCurrentCell.X &&
                !dataGridViewOper[DATAGRIDVIEWOPER_inReadOnlyChange])
            {
                VerifyImeRestrictedModeChanged();

                if (!dataGridViewColumn.ReadOnly &&
                    ColumnEditable(ptCurrentCell.X) &&
                    (Rows.GetRowState(ptCurrentCell.Y) & DataGridViewElementStates.ReadOnly) == 0 &&
                     !IsCurrentCellInEditMode &&
                     (EditMode == DataGridViewEditMode.EditOnEnter ||
                      (EditMode != DataGridViewEditMode.EditProgrammatically && CurrentCellInternal.EditType == null)))
                {
                    // Current column becomes read/write. Enter editing mode.
                    BeginEditInternal(true /*selectAll*/);
                }
            }
        }

        internal void OnColumnToolTipTextChanged(DataGridViewColumn dataGridViewColumn)
        {
            OnColumnToolTipTextChanged(new DataGridViewColumnEventArgs(dataGridViewColumn));
        }

        protected virtual void OnColumnToolTipTextChanged(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }
            if (Events[EVENT_DATAGRIDVIEWCOLUMNTOOLTIPTEXTCHANGED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnColumnWidthChanged(DataGridViewColumnEventArgs e)
        {
            if (e.Column.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_ColumnDoesNotBelongToDataGridView);
            }
            Columns.InvalidateCachedColumnsWidths();

            // don't do any layout logic if the handle was not created already
            if (e.Column.Visible && IsHandleCreated)
            {
                PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, false /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);

                Rectangle rightArea = layout.Data;
                if (layout.ColumnHeadersVisible)
                {
                    rightArea = Rectangle.Union(rightArea, layout.ColumnHeaders);
                }
                else if (SingleHorizontalBorderAdded)
                {
                    rightArea.Y--;
                    rightArea.Height++;
                }
                if (rightArea.Width > 0 && rightArea.Height > 0)
                {
                    int leftEdge = GetColumnXFromIndex(e.Column.Index);
                    if (RightToLeftInternal)
                    {
                        rightArea.Width -= rightArea.Right - leftEdge;
                    }
                    else
                    {
                        rightArea.Width -= leftEdge - rightArea.X;
                        rightArea.X = leftEdge;
                    }
                    if (rightArea.Width > 0 && rightArea.Height > 0)
                    {
                        Invalidate(rightArea);
                    }
                }

                if (editingControl != null)
                {
                    PositionEditingControl(ptCurrentCell.X != e.Column.Index, true, false);
                }

                UpdateMouseEnteredCell(null /*HitTestInfo*/, null /*MouseEventArgs*/);

                if (AutoSize)
                {
                    LayoutTransaction.DoLayout(ParentInternal, this, PropertyNames.Columns);
                }
            }

            if (Events[EVENT_DATAGRIDVIEWCOLUMNWIDTHCHANGED] is DataGridViewColumnEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            if (!InAdjustFillingColumns)
            {
                // Autosize rows and column headers if needed
                OnColumnGlobalAutoSize(e.Column.Index);
            }
        }

        internal void OnCommonCellContentClick(int columnIndex, int rowIndex, bool doubleClick)
        {
            if (ptMouseDownCell.X == -2 ||
                (dataGridViewState2[DATAGRIDVIEWSTATE2_cellMouseDownInContentBounds] &&
                 ptMouseDownCell.X == columnIndex && ptMouseDownCell.Y == rowIndex &&
                 (ptMouseDownCell.X == -1 || ptMouseDownCell.Y == -1 ||
                  (columnIndex == ptCurrentCell.X && rowIndex == ptCurrentCell.Y))))
            {
                DataGridViewCellEventArgs dgvce = new DataGridViewCellEventArgs(columnIndex, rowIndex);
                if (doubleClick)
                {
                    OnCellContentDoubleClick(dgvce);
                }
                else
                {
                    OnCellContentClick(dgvce);
                }
            }
        }

        protected virtual void OnCurrentCellChanged(EventArgs e)
        {
            VerifyImeRestrictedModeChanged();
            if (Events[EVENT_DATAGRIDVIEWCURRENTCELLCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            if (CurrentCell != null && (ShowCellToolTips || (ShowCellErrors && !string.IsNullOrEmpty(CurrentCell?.ErrorText))))
            {
                ActivateToolTip(false /*activate*/, String.Empty, CurrentCell.ColumnIndex, CurrentCell.RowIndex);
                KeyboardToolTipStateMachine.Instance.NotifyAboutGotFocus(CurrentCell);
            }
        }

        protected virtual void OnCurrentCellDirtyStateChanged(EventArgs e)
        {
            if (RowHeadersVisible && ShowEditingIcon)
            {
                // Force the pencil to appear in the row header
                Debug.Assert(ptCurrentCell.Y >= 0);
                InvalidateCellPrivate(-1, ptCurrentCell.Y);
            }
            if (IsCurrentCellDirty && newRowIndex == ptCurrentCell.Y)
            {
                Debug.Assert(newRowIndex != -1);
                Debug.Assert(AllowUserToAddRowsInternal);
                // First time the 'new' row gets edited.
                // It becomes a regular row and a new 'new' row is appened.
                newRowIndex = -1;
                AddNewRow(true /*createdByEditing*/);
            }

            if (Events[EVENT_DATAGRIDVIEWCURRENTCELLDIRTYSTATECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected override void OnCursorChanged(EventArgs e)
        {
            base.OnCursorChanged(e);
            if (!dataGridViewState2[DATAGRIDVIEWSTATE2_ignoreCursorChange])
            {
                oldCursor = Cursor;
            }
        }

        internal void OnDataBindingComplete(ListChangedType listChangedType)
        {
            OnDataBindingComplete(new DataGridViewBindingCompleteEventArgs(listChangedType));
        }

        protected virtual void OnDataBindingComplete(DataGridViewBindingCompleteEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWDATABINDINGCOMPLETE] is DataGridViewBindingCompleteEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnDataError(bool displayErrorDialogIfNoHandler, DataGridViewDataErrorEventArgs e)
        {
            if (!dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                if (!(Events[EVENT_DATAGRIDVIEWDATAERROR] is DataGridViewDataErrorEventHandler eh))
                {
                    if (displayErrorDialogIfNoHandler)
                    {
                        string errorText;
                        if (e.Exception == null)
                        {
                            errorText = SR.DataGridView_ErrorMessageText_NoException;
                        }
                        else
                        {
                            errorText = string.Format(SR.DataGridView_ErrorMessageText_WithException, e.Exception);
                        }
                        if (RightToLeftInternal)
                        {
                            MessageBox.Show(errorText, SR.DataGridView_ErrorMessageCaption, MessageBoxButtons.OK, MessageBoxIcon.Error, MessageBoxDefaultButton.Button1, MessageBoxOptions.RightAlign | MessageBoxOptions.RtlReading);
                        }
                        else
                        {
                            MessageBox.Show(errorText, SR.DataGridView_ErrorMessageCaption, MessageBoxButtons.OK, MessageBoxIcon.Error);
                        }
                        CorrectFocus(true /*onlyIfGridHasFocus*/);
                    }
                }
                else
                {
                    eh(this, e);
                    CorrectFocus(true /*onlyIfGridHasFocus*/);
                }
            }
        }

        internal void OnDataErrorInternal(DataGridViewDataErrorEventArgs e)
        {
            OnDataError(!DesignMode /*displayErrorDialogIfNoHandler*/, e);
        }

        internal void OnDataGridViewElementStateChanged(DataGridViewElement element, int index, DataGridViewElementStates elementState)
        {
            if (element is DataGridViewColumn dataGridViewColumn)
            {
                DataGridViewColumnStateChangedEventArgs dgvcsce = new DataGridViewColumnStateChangedEventArgs(dataGridViewColumn, elementState);

                OnColumnStateChanged(dgvcsce);
            }
            else
            {
                if (element is DataGridViewRow dataGridViewRow)
                {
                    if (Events[EVENT_DATAGRIDVIEWROWSTATECHANGED] is DataGridViewRowStateChangedEventHandler eh && dataGridViewRow.DataGridView != null && dataGridViewRow.Index == -1)
                    {
                        dataGridViewRow = Rows[index];
                    }

                    DataGridViewRowStateChangedEventArgs dgvrsce = new DataGridViewRowStateChangedEventArgs(dataGridViewRow, elementState);

                    OnRowStateChanged(dataGridViewRow.Index == -1 ? index : dataGridViewRow.Index, dgvrsce);
                }
                else
                {
                    if (element is DataGridViewCell dataGridViewCell)
                    {
                        DataGridViewCellStateChangedEventArgs dgvcsce = new DataGridViewCellStateChangedEventArgs(dataGridViewCell, elementState);

                        OnCellStateChanged(dgvcsce);
                    }
                }
            }

            if ((elementState & DataGridViewElementStates.Selected) == DataGridViewElementStates.Selected)
            {
                if (noSelectionChangeCount > 0)
                {
                    dataGridViewState2[DATAGRIDVIEWSTATE2_raiseSelectionChanged] = true;
                }
                else
                {
                    OnSelectionChanged(EventArgs.Empty);
                }
            }
        }

        internal void OnDataGridViewElementStateChanging(DataGridViewElement element, int index, DataGridViewElementStates elementState)
        {
            if (element is DataGridViewColumn dataGridViewColumn)
            {
                // column.Frozen | .Visible | .ReadOnly changing
                switch (elementState)
                {
                    case DataGridViewElementStates.Frozen:
                    case DataGridViewElementStates.Visible:
                        if (elementState == DataGridViewElementStates.Visible)
                        {
                            if (!dataGridViewColumn.Visible &&
                                dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.ColumnHeader &&
                                !ColumnHeadersVisible)
                            {
                                throw new InvalidOperationException(SR.DataGridView_CannotMakeAutoSizedColumnVisible);
                            }
                            else if (!dataGridViewColumn.Visible &&
                                dataGridViewColumn.Frozen &&
                                dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                            {
                                // alternative: throw new InvalidOperationException(SR.DataGridView_CannotMakeAutoFillColumnVisible);
                                //              DataGridView_CannotMakeAutoFillColumnVisible=The column cannot be made visible because its autosizing mode is Fill and it is frozen.
                                // Removing the Fill auto size mode when frozen column becomes visible (instead of throwing an exception)
                                dataGridViewColumn.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
                            }
                            else if (dataGridViewColumn.Visible && ptCurrentCell.X == dataGridViewColumn.Index)
                            {
                                // Column of the current cell is made invisible. Trying to reset the current cell. May throw an exception.
                                ResetCurrentCell();
                                // Microsoft: Should the current cell be set to some cell after the operation?
                            }
                        }
                        if (elementState == DataGridViewElementStates.Frozen &&
                            !dataGridViewColumn.Frozen &&
                            dataGridViewColumn.Visible &&
                            dataGridViewColumn.InheritedAutoSizeMode == DataGridViewAutoSizeColumnMode.Fill)
                        {
                            // Removing the Fill auto size mode when visible column becomes frozen (instead of throwing an exception)
                            dataGridViewColumn.AutoSizeMode = DataGridViewAutoSizeColumnMode.None;
                        }
                        CorrectColumnFrozenStates(dataGridViewColumn, elementState == DataGridViewElementStates.Frozen);
                        break;

                    case DataGridViewElementStates.ReadOnly:
                        if (ptCurrentCell.X == dataGridViewColumn.Index &&
                            IsCurrentCellInEditMode &&
                            !dataGridViewColumn.ReadOnly &&
                            !dataGridViewOper[DATAGRIDVIEWOPER_inReadOnlyChange])
                        {
                            Debug.Assert(!ReadOnly);
                            // Column becomes read-only. Exit editing mode.
                            if (!EndEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit,
                                        DataGridViewValidateCellInternal.Always /*validateCell*/,
                                        false /*fireCellLeave*/,
                                        false /*fireCellEnter*/,
                                        false /*fireRowLeave*/,
                                        false /*fireRowEnter*/,
                                        false /*fireLeave*/,
                                        true /*keepFocus*/,
                                        false /*resetCurrentCell*/,
                                        false /*resetAnchorCell*/))
                            {
                                throw new InvalidOperationException(SR.DataGridView_CommitFailedCannotCompleteOperation);
                            }
                        }
                        break;

                    default:
                        Debug.Fail("Unexpected DataGridViewElementStates param in DataGridView.OnDataGridViewElementStateChanging");
                        break;
                }
            }
            else
            {
                if (element is DataGridViewRow dataGridViewRow)
                {
                    // row.Frozen | .Visible | .ReadOnly changing
                    int rowIndex = ((dataGridViewRow.Index > -1) ? dataGridViewRow.Index : index);
                    switch (elementState)
                    {
                        case DataGridViewElementStates.Frozen:
                        case DataGridViewElementStates.Visible:
                            if (elementState == DataGridViewElementStates.Visible && ptCurrentCell.Y == rowIndex)
                            {
                                Debug.Assert((Rows.GetRowState(rowIndex) & DataGridViewElementStates.Visible) != 0);
                                // Row of the current cell is made invisible.
                                if (DataSource != null)
                                {
                                    Debug.Assert(dataConnection != null);
                                    Debug.Assert(dataConnection.CurrencyManager != null);
                                    Debug.Assert(dataConnection.CurrencyManager.Position == ptCurrentCell.Y);
                                    // the row associated with the currency manager's position cannot be made invisble.
                                    throw new InvalidOperationException(SR.DataGridView_CurrencyManagerRowCannotBeInvisible);
                                }
                                // Trying to reset the current cell. May throw an exception.
                                ResetCurrentCell();
                                // Microsoft: Should the current cell be set to some cell after the operation?
                            }
                            CorrectRowFrozenStates(dataGridViewRow, rowIndex, elementState == DataGridViewElementStates.Frozen);
                            break;

                        case DataGridViewElementStates.ReadOnly:
                            if (ptCurrentCell.Y == rowIndex &&
                                (Rows.GetRowState(rowIndex) & DataGridViewElementStates.ReadOnly) == 0 &&
                                !ReadOnly &&
                                IsCurrentCellInEditMode &&
                                !dataGridViewOper[DATAGRIDVIEWOPER_inReadOnlyChange])
                            {
                                // Row becomes read-only. Exit editing mode.
                                if (!EndEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit,
                                            DataGridViewValidateCellInternal.Always /*validateCell*/,
                                            false /*fireCellLeave*/,
                                            false /*fireCellEnter*/,
                                            false /*fireRowLeave*/,
                                            false /*fireRowEnter*/,
                                            false /*fireLeave*/,
                                            true /*keepFocus*/,
                                            false /*resetCurrentCell*/,
                                            false /*resetAnchorCell*/))
                                {
                                    throw new InvalidOperationException(SR.DataGridView_CommitFailedCannotCompleteOperation);
                                }
                            }
                            break;

                        default:
                            Debug.Fail("Unexpected DataGridViewElementStates param in DataGridView.OnDataGridViewElementStateChanging");
                            break;
                    }
                }
                else
                {
                    if (element is DataGridViewCell dataGridViewCell)
                    {
                        // cell.ReadOnly changing
                        switch (elementState)
                        {
                            case DataGridViewElementStates.ReadOnly:
                                if (ptCurrentCell.X == dataGridViewCell.ColumnIndex &&
                                    ptCurrentCell.Y == dataGridViewCell.RowIndex &&
                                    IsCurrentCellInEditMode &&
                                    !dataGridViewCell.ReadOnly &&
                                    !dataGridViewOper[DATAGRIDVIEWOPER_inReadOnlyChange])
                                {
                                    Debug.Assert(!Columns[dataGridViewCell.ColumnIndex].ReadOnly);
                                    Debug.Assert(!Rows[dataGridViewCell.RowIndex].ReadOnly);
                                    Debug.Assert(!ReadOnly);
                                    // Current cell becomes read-only. Exit editing mode.
                                    if (!EndEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit,
                                                DataGridViewValidateCellInternal.Always /*validateCell*/,
                                                false /*fireCellLeave*/,
                                                false /*fireCellEnter*/,
                                                false /*fireRowLeave*/,
                                                false /*fireRowEnter*/,
                                                false /*fireLeave*/,
                                                true /*keepFocus*/,
                                                false /*resetCurrentCell*/,
                                                false /*resetAnchorCell*/))
                                    {
                                        throw new InvalidOperationException(SR.DataGridView_CommitFailedCannotCompleteOperation);
                                    }
                                }
                                break;

                            default:
                                Debug.Fail("Unexpected DataGridViewElementStates param in DataGridView.OnDataGridViewElementStateChanging");
                                break;
                        }
                    }
                    else
                    {
                        Debug.Fail("Unexpected DataGridViewElement type in DataGridView.OnDataGridViewElementStateChanging");
                    }
                }
            }
        }

        protected virtual void OnDataMemberChanged(EventArgs e)
        {
            RefreshColumnsAndRows();

            if (Events[EVENT_DATAGRIDVIEWDATAMEMBERCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            if (dataConnection != null && dataConnection.CurrencyManager != null)
            {
                OnDataBindingComplete(ListChangedType.Reset);
            }
        }

        protected virtual void OnDataSourceChanged(EventArgs e)
        {
            RefreshColumnsAndRows();
            InvalidateRowHeights();

            if (Events[EVENT_DATAGRIDVIEWDATASOURCECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            if (dataConnection != null && dataConnection.CurrencyManager != null)
            {
                OnDataBindingComplete(ListChangedType.Reset);
            }
        }

        protected virtual void OnDefaultCellStyleChanged(EventArgs e)
        {
            if (e is DataGridViewCellStyleChangedEventArgs dgvcsce && !dgvcsce.ChangeAffectsPreferredSize)
            {
                Invalidate();
            }
            else
            {
                OnGlobalAutoSize();
                if (editingControl != null)
                {
                    PositionEditingControl(true /*setLocation*/, true /*setSize*/, false /*setFocus*/);
                }
            }

            if (Events[EVENT_DATAGRIDVIEWDEFAULTCELLSTYLECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnDefaultValuesNeeded(DataGridViewRowEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWDEFAULTVALUESNEEDED] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected override void OnDoubleClick(EventArgs e)
        {
            base.OnDoubleClick(e);

            if (!dataGridViewState2[DATAGRIDVIEWSTATE2_messageFromEditingCtrls] && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                if (e is MouseEventArgs me)
                {
                    HitTestInfo hti = HitTest(me.X, me.Y);
                    if (hti.Type != DataGridViewHitTestType.None &&
                        hti.Type != DataGridViewHitTestType.HorizontalScrollBar &&
                        hti.Type != DataGridViewHitTestType.VerticalScrollBar &&
                        me.Button == MouseButtons.Left)
                    {
                        OnCellDoubleClick(new DataGridViewCellEventArgs(hti.col, hti.row));
                    }
                }
            }
        }

        protected virtual void OnEditingControlShowing(DataGridViewEditingControlShowingEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWEDITINGCONTROLSHOWING] is DataGridViewEditingControlShowingEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnEditModeChanged(EventArgs e)
        {
            if (Focused &&
                EditMode == DataGridViewEditMode.EditOnEnter &&
                ptCurrentCell.X > -1 &&
                !IsCurrentCellInEditMode)
            {
                // New edit mode is EditOnEnter and there is an editable current cell, try to go to edit mode.
                BeginEditInternal(true /*selectAll*/);
            }

            if (Events[EVENT_DATAGRIDVIEWEDITMODECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected override void OnEnabledChanged(EventArgs e)
        {
            base.OnEnabledChanged(e);
            if (GetAnyDisposingInHierarchy())
            {
                return;
            }

            if (IsHandleCreated && Enabled)
            {
                if (vertScrollBar != null && vertScrollBar.Visible)
                {
                    int totalVisibleHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible);
                    int totalVisibleFrozenHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                    vertScrollBar.Maximum = totalVisibleHeight - totalVisibleFrozenHeight;

                    vertScrollBar.Enabled = true;
                }
                if (horizScrollBar != null && horizScrollBar.Visible)
                {
                    horizScrollBar.Enabled = true;
                }
            }
        }

        protected override void OnEnter(EventArgs e)
        {
            if (editingControl != null && editingControl.ContainsFocus)
            {
                return;
            }

            base.OnEnter(e);

            // There are cases when a control is on the designer and it still receives the OnEnter event.
            // Guard against this.
            if (DesignMode)
            {
                return;
            }

            dataGridViewState1[DATAGRIDVIEWSTATE1_leavingWithTabKey] = false;

            if (ptCurrentCell.X > -1)
            {
                DataGridViewCell dataGridViewCell = null;
                // We're re-entering a row we already entered earlier. The first time we entered the row,
                // the DataGridView didn't have focus. This time it does. We don't want to recreate the new row a second time.
                OnRowEnter(ref dataGridViewCell, ptCurrentCell.X, ptCurrentCell.Y, false /*canCreateNewRow*/, false /*validationFailureOccurred*/);
                if (ptCurrentCell.X == -1)
                {
                    return;
                }
                OnCellEnter(ref dataGridViewCell, ptCurrentCell.X, ptCurrentCell.Y);

                // Force repainting of the current collumn's header cell to highlight it
                if (SelectionMode == DataGridViewSelectionMode.FullRowSelect)
                {
                    InvalidateCellPrivate(ptCurrentCell.X, -1);
                }
            }
            else if (!dataGridViewOper[DATAGRIDVIEWOPER_inMouseDown])
            {
                // Focus is given to the DataGridView control via a the TAB key.
                MakeFirstDisplayedCellCurrentCell(true /*includeNewRow*/);
            }

            if (ptCurrentCell.X > -1 &&
                !IsCurrentCellInEditMode)
            {
                if (EditMode == DataGridViewEditMode.EditOnEnter ||
                   (EditMode != DataGridViewEditMode.EditProgrammatically && CurrentCellInternal.EditType == null))
                {
                    BeginEditInternal(true /*selectAll*/);
                    if (ptCurrentCell.X > -1 && CurrentCellInternal.EditType == null && !dataGridViewOper[DATAGRIDVIEWOPER_inMouseDown])
                    {
                        // The current cell does not have an edit type so the data grid view did not put an edit control on top.
                        // We should invalidate the current cell so that the dataGridView repaints the focus around the current cell.
                        // But do that only if the dataGridView did not get the focus via mouse.
                        InvalidateCellPrivate(ptCurrentCell.X /*columnIndex*/, ptCurrentCell.Y /*rowIndex*/);
                    }
                }
                else if (!dataGridViewOper[DATAGRIDVIEWOPER_inMouseDown])
                {
                    // When the focus is given to the DataGridView control via mouse
                    // the dataGridView changes selection so it invalidates the current cell anyway
                    //
                    // In any other case Invalidate the current cell so the dataGridView repaints the focus around the current cell
                    InvalidateCellPrivate(ptCurrentCell.X /*columnIndex*/, ptCurrentCell.Y /*rowIndex*/);
                }
            }

            // Draw focus rectangle around the grid
            if (IsGridFocusRectangleEnabled())
            {
                InvalidateRectangleEdges(GetGridFocusRectangle());
            }
        }

        protected override void OnFontChanged(EventArgs e)
        {
            base.OnFontChanged(e);

            if (GetAnyDisposingInHierarchy())
            {
                return;
            }

            // Change may be due to an ambient font change.
            if (dataGridViewState1[DATAGRIDVIEWSTATE1_ambientColumnHeadersFont] &&
                ColumnHeadersDefaultCellStyle.Font != base.Font)
            {
                ColumnHeadersDefaultCellStyle.Font = base.Font;
                dataGridViewState1[DATAGRIDVIEWSTATE1_ambientColumnHeadersFont] = true;
                CellStyleChangedEventArgs.ChangeAffectsPreferredSize = true;
                OnColumnHeadersDefaultCellStyleChanged(CellStyleChangedEventArgs);
            }

            if (dataGridViewState1[DATAGRIDVIEWSTATE1_ambientRowHeadersFont] &&
                RowHeadersDefaultCellStyle.Font != base.Font)
            {
                RowHeadersDefaultCellStyle.Font = base.Font;
                dataGridViewState1[DATAGRIDVIEWSTATE1_ambientRowHeadersFont] = true;
                CellStyleChangedEventArgs.ChangeAffectsPreferredSize = true;
                OnRowHeadersDefaultCellStyleChanged(CellStyleChangedEventArgs);
            }

            if (dataGridViewState1[DATAGRIDVIEWSTATE1_ambientFont] &&
                DefaultCellStyle.Font != base.Font)
            {
                DefaultCellStyle.Font = base.Font;
                dataGridViewState1[DATAGRIDVIEWSTATE1_ambientFont] = true;
                CellStyleChangedEventArgs.ChangeAffectsPreferredSize = true;
                OnDefaultCellStyleChanged(CellStyleChangedEventArgs);
            }
        }

        protected override void OnForeColorChanged(EventArgs e)
        {
            base.OnForeColorChanged(e);
            if (GetAnyDisposingInHierarchy())
            {
                return;
            }

            // Change may be due to an ambient forecolor change.
            if (dataGridViewState1[DATAGRIDVIEWSTATE1_ambientForeColor] && DefaultCellStyle.ForeColor != base.ForeColor)
            {
                DefaultCellStyle.ForeColor = base.ForeColor;
                dataGridViewState1[DATAGRIDVIEWSTATE1_ambientForeColor] = true;
                CellStyleChangedEventArgs.ChangeAffectsPreferredSize = false;
                OnDefaultCellStyleChanged(CellStyleChangedEventArgs);
            }
        }

        private void OnGlobalAutoSize()
        {
            Invalidate();

            if (noAutoSizeCount > 0)
            {
                return;
            }

            bool autoSizeRowHeaders = rowHeadersWidthSizeMode != DataGridViewRowHeadersWidthSizeMode.EnableResizing &&
                                      rowHeadersWidthSizeMode != DataGridViewRowHeadersWidthSizeMode.DisableResizing;
            if (autoSizeRowHeaders)
            {
                AutoResizeRowHeadersWidth(rowHeadersWidthSizeMode,
                                          ColumnHeadersHeightSizeMode != DataGridViewColumnHeadersHeightSizeMode.AutoSize /*fixedColumnHeadersHeight*/,
                                          autoSizeRowsMode == DataGridViewAutoSizeRowsMode.None /*fixedRowsHeight*/);
            }
            if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
            {
                AutoResizeColumnHeadersHeight(true /*fixedRowHeadersWidth*/, false /*fixedColumnsWidth*/);
            }
            if (autoSizeRowsMode != DataGridViewAutoSizeRowsMode.None)
            {
                AdjustShrinkingRows(autoSizeRowsMode, false /*fixedWidth*/, true /*internalAutosizing*/);
            }
            AutoResizeAllVisibleColumnsInternal(DataGridViewAutoSizeColumnCriteriaInternal.Header | DataGridViewAutoSizeColumnCriteriaInternal.AllRows | DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows, true /*fixedHeight*/);

            if (autoSizeRowHeaders &&
                (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize || autoSizeRowsMode != DataGridViewAutoSizeRowsMode.None))
            {
                // Second round of row headers autosizing
                AutoResizeRowHeadersWidth(rowHeadersWidthSizeMode, true /*fixedColumnHeadersHeight*/, true /*fixedRowsHeight*/);
            }
            if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
            {
                // Second round of column headers autosizing
                AutoResizeColumnHeadersHeight(true /*fixedRowHeadersWidth*/, true /*fixedColumnsWidth*/);
            }
            if (autoSizeRowsMode != DataGridViewAutoSizeRowsMode.None)
            {
                // Second round of rows autosizing
                AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
            }
        }

        protected override void OnGotFocus(EventArgs e)
        {
            base.OnGotFocus(e);
            if (ptCurrentCell.X != -1)
            {
                InvalidateCell(ptCurrentCell.X, ptCurrentCell.Y);
            }

            // Inform Accessibility that our current cell contains the focus.
            if (!dataGridViewOper[DATAGRIDVIEWOPER_inCurrentCellChange] &&
                (!dataGridViewOper[DATAGRIDVIEWOPER_inEndEdit] || EditMode != DataGridViewEditMode.EditOnEnter) &&
                (!dataGridViewOper[DATAGRIDVIEWOPER_inMouseDown] || EditMode != DataGridViewEditMode.EditOnEnter) &&
                ptCurrentCell.X > -1)
            {
                // The name is misleading ( the current cell did not change ).
                // However, AccessibilityNotifyCurrentCellChanged is now a public method so we can't change its name
                // to better reflect its purpose.
                AccessibilityNotifyCurrentCellChanged(ptCurrentCell);
                if (CurrentCell != null && (ShowCellToolTips || (ShowCellErrors && !string.IsNullOrEmpty(CurrentCell.ErrorText))))
                {
                    ActivateToolTip(false /*activate*/, String.Empty, CurrentCell.ColumnIndex, CurrentCell.RowIndex);
                    KeyboardToolTipStateMachine.Instance.NotifyAboutGotFocus(CurrentCell);
                }
            }
        }

        protected virtual void OnGridColorChanged(EventArgs e)
        {
            InvalidateInside();

            if (Events[EVENT_DATAGRIDVIEWGRIDCOLORCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);

            if (layout.dirty)
            {
                PerformLayoutPrivate(false /*useRowShortcut*/, true /*computeVisibleRows*/, false /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
            }
            if (ptCurrentCell.X == -1)
            {
                MakeFirstDisplayedCellCurrentCell(false /*includeNewRow*/);
            }
            else
            {
                ScrollIntoView(ptCurrentCell.X, ptCurrentCell.Y, false /*forCurrentCellChange*/);
            }

            // do the AutoSize work that was skipped during initialization
            if (dataGridViewState2[DATAGRIDVIEWSTATE2_autoSizedWithoutHandle])
            {
                dataGridViewState2[DATAGRIDVIEWSTATE2_autoSizedWithoutHandle] = false;
                OnGlobalAutoSize();
            }
            SystemEvents.UserPreferenceChanged += new UserPreferenceChangedEventHandler(OnUserPreferenceChanged);
        }

        protected override void OnHandleDestroyed(EventArgs e)
        {
            SystemEvents.UserPreferenceChanged -= new UserPreferenceChangedEventHandler(OnUserPreferenceChanged);
            base.OnHandleDestroyed(e);
        }

        internal void OnInsertedColumn_PreNotification(DataGridViewColumn dataGridViewColumn)
        {
            // Fix the OldFirstDisplayedScrollingCol
            displayedBandsInfo.CorrectColumnIndexAfterInsertion(dataGridViewColumn.Index, 1);

            // Fix the Index of all following columns
            CorrectColumnIndexesAfterInsertion(dataGridViewColumn, 1);

            // Same effect as appending a column
            OnAddedColumn(dataGridViewColumn);
        }

        internal void OnInsertedColumn_PostNotification(Point newCurrentCell)
        {
            // Update current cell if needed
            if (newCurrentCell.X != -1)
            {
                Debug.Assert(ptCurrentCell.X == -1);
                bool success = SetAndSelectCurrentCellAddress(newCurrentCell.X,
                                                              newCurrentCell.Y,
                                                              true,
                                                              false,
                                                              false,
                                                              false /*clearSelection*/,
                                                              Columns.GetColumnCount(DataGridViewElementStates.Visible) == 1 /*forceCurrentCellSelection*/);
                Debug.Assert(success);
            }
        }

        internal void OnInsertedRow_PreNotification(int rowIndex, int insertionCount)
        {
            Debug.Assert(rowIndex >= 0);
            Debug.Assert(insertionCount > 0);

            // Fix the OldFirstDisplayedScrollingRow
            displayedBandsInfo.CorrectRowIndexAfterInsertion(rowIndex, insertionCount);

            // Fix the Index of all following rows
            CorrectRowIndexesAfterInsertion(rowIndex, insertionCount);

            // Next, same effect as adding a row
            OnAddedRow_PreNotification(rowIndex);
        }

        internal void OnInsertedRow_PostNotification(int rowIndex, Point newCurrentCell, bool lastInsertion)
        {
            Debug.Assert(rowIndex >= 0);

            // Same effect as adding a row
            OnAddedRow_PostNotification(rowIndex);

            // Update current cell if needed
            if (lastInsertion && newCurrentCell.Y != -1)
            {
                Debug.Assert(ptCurrentCell.X == -1);
                bool success = SetAndSelectCurrentCellAddress(newCurrentCell.X,
                                                              newCurrentCell.Y,
                                                              true,
                                                              false,
                                                              false,
                                                              false /*clearSelection*/,
                                                              Rows.GetRowCount(DataGridViewElementStates.Visible) == 1 /*forceCurrentCellSelection*/);
                Debug.Assert(success);
            }
        }

        internal void OnInsertedRows_PreNotification(int rowIndex, DataGridViewRow[] dataGridViewRows)
        {
            Debug.Assert(rowIndex >= 0);
            Debug.Assert(dataGridViewRows != null);
            Debug.Assert(dataGridViewRows.Length > 0);

            // Fix the OldFirstDisplayedScrollingRow
            displayedBandsInfo.CorrectRowIndexAfterInsertion(rowIndex, dataGridViewRows.Length);

            // Fix the Index of all following rows
            CorrectRowIndexesAfterInsertion(rowIndex, dataGridViewRows.Length);

            // Next, same effect as adding the rows
            OnAddedRows_PreNotification(dataGridViewRows);
        }

        internal void OnInsertedRows_PostNotification(DataGridViewRow[] dataGridViewRows, Point newCurrentCell)
        {
            Debug.Assert(dataGridViewRows != null);
            Debug.Assert(dataGridViewRows.Length > 0);

            // Same effect as adding the rows
            OnAddedRows_PostNotification(dataGridViewRows);

            // Update current cell if needed
            if (newCurrentCell.Y != -1)
            {
                Debug.Assert(ptCurrentCell.X == -1);
                bool success = SetAndSelectCurrentCellAddress(newCurrentCell.X, newCurrentCell.Y, true, false, false, false /*clearSelection*/, false /*forceCurrentCellSelection*/);
                Debug.Assert(success);
            }
        }

        internal void OnInsertingColumn(int columnIndexInserted, DataGridViewColumn dataGridViewColumn, out Point newCurrentCell)
        {
            Debug.Assert(dataGridViewColumn != null);

            if (dataGridViewColumn.DataGridView != null)
            {
                throw new InvalidOperationException(SR.DataGridView_ColumnAlreadyBelongsToDataGridView);
            }

            if (!InInitialization &&
                dataGridViewColumn.SortMode == DataGridViewColumnSortMode.Automatic &&
                (SelectionMode == DataGridViewSelectionMode.FullColumnSelect ||
                 SelectionMode == DataGridViewSelectionMode.ColumnHeaderSelect))
            {
                throw new InvalidOperationException(string.Format(SR.DataGridViewColumn_SortModeAndSelectionModeClash, DataGridViewColumnSortMode.Automatic.ToString(), SelectionMode.ToString()));
            }

            if (dataGridViewColumn.Visible)
            {
                // Note that dataGridViewColumn.DataGridView is set later on, so dataGridViewColumn.InheritedAutoSizeMode should not be used

                // Make sure the column does not autosize based only on header while column headers are invisible
                if (!ColumnHeadersVisible &&
                    (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.ColumnHeader || (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.NotSet && AutoSizeColumnsMode == DataGridViewAutoSizeColumnsMode.ColumnHeader)))
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAddAutoSizedColumn);
                }

                // Make sure the column is not frozen and auto fills
                if (dataGridViewColumn.Frozen &&
                    (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.Fill || (dataGridViewColumn.AutoSizeMode == DataGridViewAutoSizeColumnMode.NotSet && AutoSizeColumnsMode == DataGridViewAutoSizeColumnsMode.Fill)))
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAddAutoFillColumn);
                }
            }

            // check for correctness of frozen state - throws exception if state is incorrect.
            CorrectColumnFrozenState(dataGridViewColumn, columnIndexInserted);

            // Reset current cell if there is one, no matter the relative position of the columns involved
            if (ptCurrentCell.X != -1)
            {
                newCurrentCell = new Point(columnIndexInserted <= ptCurrentCell.X ? ptCurrentCell.X + 1 : ptCurrentCell.X,
                     ptCurrentCell.Y);
                ResetCurrentCell();
            }
            else
            {
                newCurrentCell = new Point(-1, -1);
            }

            // prepare the existing rows by inserting cells of correct type
            if (Rows.Count > 0)
            {
                // Only require a default cell type when there are rows to fill
                if (dataGridViewColumn.CellType == null)
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAddUntypedColumn);
                }

                if (dataGridViewColumn.CellTemplate.DefaultNewRowValue != null && newRowIndex != -1)
                {
                    // New row needs to be unshared before addition of new cell with a Value != null
                    DataGridViewRow newRow = Rows[newRowIndex];
                }

                int newColumnCount = Columns.Count + 1;

                try
                {
                    for (int rowIndex = 0; rowIndex < Rows.Count; rowIndex++)
                    {
                        DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                        if (dataGridViewRow.Cells.Count < newColumnCount)
                        {
                            DataGridViewCell dataGridViewCellNew = (DataGridViewCell)dataGridViewColumn.CellTemplate.Clone();
                            dataGridViewRow.Cells.InsertInternal(columnIndexInserted, dataGridViewCellNew);
                            if (rowIndex == newRowIndex)
                            {
                                dataGridViewCellNew.Value = dataGridViewCellNew.DefaultNewRowValue;
                            }
                            dataGridViewCellNew.DataGridView = this;
                            dataGridViewCellNew.OwningRow = dataGridViewRow;
                            dataGridViewCellNew.OwningColumn = dataGridViewColumn;
                        }
                    }
                }
                catch
                {
                    // An error occurred while inserting the cells. Revert all the insertions.
                    for (int rowIndex = 0; rowIndex < Rows.Count; rowIndex++)
                    {
                        DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                        if (dataGridViewRow.Cells.Count == newColumnCount)
                        {
                            dataGridViewRow.Cells.RemoveAtInternal(columnIndexInserted);
                        }
                        else
                        {
                            Debug.Assert(dataGridViewRow.Cells.Count < newColumnCount);
                            break;
                        }
                    }
                    throw;
                }
            }

            // Update the indexes of selected columns to compensate for the insertion of this column
            switch (SelectionMode)
            {
                case DataGridViewSelectionMode.FullColumnSelect:
                case DataGridViewSelectionMode.ColumnHeaderSelect:
                    int columnEntries = selectedBandIndexes.Count;
                    int columnEntry = 0;
                    while (columnEntry < columnEntries)
                    {
                        int columnIndex = selectedBandIndexes[columnEntry];
                        if (columnIndexInserted <= columnIndex)
                        {
                            selectedBandIndexes[columnEntry] = columnIndex + 1;
                        }
                        columnEntry++;
                    }
                    if (selectedBandSnapshotIndexes != null)
                    {
                        columnEntries = selectedBandSnapshotIndexes.Count;
                        columnEntry = 0;
                        while (columnEntry < columnEntries)
                        {
                            int columnIndex = selectedBandSnapshotIndexes[columnEntry];
                            if (columnIndexInserted <= columnIndex)
                            {
                                selectedBandSnapshotIndexes[columnEntry] = columnIndex + 1;
                            }
                            columnEntry++;
                        }
                    }
                    break;
            }
        }

        internal void OnInsertingRow(int rowIndexInserted,
                                     DataGridViewRow dataGridViewRow,
                                     DataGridViewElementStates rowState,
                                     ref Point newCurrentCell,
                                     bool firstInsertion,
                                     int insertionCount,
                                     bool force)
        {
            // Reset the current cell's address if it's after the inserted row.
            if (firstInsertion)
            {
                if (ptCurrentCell.Y != -1 && rowIndexInserted <= ptCurrentCell.Y)
                {
                    newCurrentCell = new Point(ptCurrentCell.X, ptCurrentCell.Y + insertionCount);
                    if (force)
                    {
                        // When force is true, the underlying data was already added, therefore we need to avoid accessing any back-end data
                        // since we might be off by 1 row.
                        dataGridViewState1[DATAGRIDVIEWSTATE1_temporarilyResetCurrentCell] = true;
                        bool success = SetCurrentCellAddressCore(-1, -1, true /*setAnchorCellAddress*/, false /*validateCurrentCell*/, false);
                        Debug.Assert(success);
                    }
                    else
                    {
                        ResetCurrentCell();
                    }
                }
                else
                {
                    newCurrentCell = new Point(-1, -1);
                }
            }
            else
            {
                if (newCurrentCell.Y != -1)
                {
                    newCurrentCell.Y += insertionCount;
                }
            }

            // For now same checks as for OnAddingRow
            OnAddingRow(dataGridViewRow, rowState, false /*checkFrozenState*/);

            // check for correctness of frozen state - throws exception if state is incorrect.
            CorrectRowFrozenState(dataGridViewRow, rowState, rowIndexInserted);

            // Update the indexes of selected rows to compensate for the insertion of this row
            switch (SelectionMode)
            {
                case DataGridViewSelectionMode.FullRowSelect:
                case DataGridViewSelectionMode.RowHeaderSelect:
                    int rowEntries = selectedBandIndexes.Count;
                    int rowEntry = 0;
                    while (rowEntry < rowEntries)
                    {
                        int rowIndex = selectedBandIndexes[rowEntry];
                        if (rowIndexInserted <= rowIndex)
                        {
                            selectedBandIndexes[rowEntry] = rowIndex + insertionCount;
                        }
                        rowEntry++;
                    }
                    if (selectedBandSnapshotIndexes != null)
                    {
                        rowEntries = selectedBandSnapshotIndexes.Count;
                        rowEntry = 0;
                        while (rowEntry < rowEntries)
                        {
                            int rowIndex = selectedBandSnapshotIndexes[rowEntry];
                            if (rowIndexInserted <= rowIndex)
                            {
                                selectedBandSnapshotIndexes[rowEntry] = rowIndex + insertionCount;
                            }
                            rowEntry++;
                        }
                    }
                    break;
            }
        }

        internal void OnInsertingRows(int rowIndexInserted, DataGridViewRow[] dataGridViewRows, ref Point newCurrentCell)
        {
            Debug.Assert(dataGridViewRows != null);

            // Reset the current cell's address if it's after the inserted row.
            if (ptCurrentCell.Y != -1 && rowIndexInserted <= ptCurrentCell.Y)
            {
                newCurrentCell = new Point(ptCurrentCell.X, ptCurrentCell.Y + dataGridViewRows.Length);
                ResetCurrentCell();
            }
            else
            {
                newCurrentCell = new Point(-1, -1);
            }

            // For now almost same checks as for OnAddingRows
            // OnAddingRows checks for Selected status of rows.
            OnAddingRows(dataGridViewRows, false /*checkFrozenStates*/);

            // Check for Frozen state correctness
            CorrectRowFrozenStates(dataGridViewRows, rowIndexInserted);

            // Update the indexes of selected rows to compensate for the insertion of this row
            switch (SelectionMode)
            {
                case DataGridViewSelectionMode.FullRowSelect:
                case DataGridViewSelectionMode.RowHeaderSelect:
                    int rowEntries = selectedBandIndexes.Count;
                    int rowEntry = 0;
                    while (rowEntry < rowEntries)
                    {
                        int rowIndex = selectedBandIndexes[rowEntry];
                        if (rowIndexInserted <= rowIndex)
                        {
                            selectedBandIndexes[rowEntry] = rowIndex + dataGridViewRows.Length;
                        }
                        rowEntry++;
                    }
                    if (selectedBandSnapshotIndexes != null)
                    {
                        rowEntries = selectedBandSnapshotIndexes.Count;
                        rowEntry = 0;
                        while (rowEntry < rowEntries)
                        {
                            int rowIndex = selectedBandSnapshotIndexes[rowEntry];
                            if (rowIndexInserted <= rowIndex)
                            {
                                selectedBandSnapshotIndexes[rowEntry] = rowIndex + dataGridViewRows.Length;
                            }
                            rowEntry++;
                        }
                    }
                    break;
            }
        }

        [EditorBrowsable(EditorBrowsableState.Advanced)]
        protected override void OnKeyDown(KeyEventArgs e)
        {
            base.OnKeyDown(e);
            if (e.Handled)
            {
                return;
            }

            // Forward key down to current cell if any
            if (ptCurrentCell.X != -1)
            {
                DataGridViewCell dataGridViewCell = CurrentCellInternal;
                Debug.Assert(dataGridViewCell != null);
                if (dataGridViewCell.KeyDownUnsharesRowInternal(e, ptCurrentCell.Y))
                {
                    DataGridViewRow dataGridViewRow = Rows[ptCurrentCell.Y];
                    CurrentCellInternal.OnKeyDownInternal(e, ptCurrentCell.Y);
                }
                else
                {
                    dataGridViewCell.OnKeyDownInternal(e, ptCurrentCell.Y);
                }
            }

            if (!e.Handled)
            {
                switch (e.KeyData & Keys.KeyCode)
                {
                    case Keys.A:
                    case Keys.C:
                    case Keys.D0:
                    case Keys.NumPad0:
                    case Keys.Delete:
                    case Keys.Down:
                    case Keys.F2:
                    case Keys.F3:
                    case Keys.F10:
                    case Keys.End:
                    case Keys.Enter:
                    case Keys.Escape:
                    case Keys.Home:
                    case Keys.Insert:
                    case Keys.Left:
                    case Keys.Next:
                    case Keys.Prior:
                    case Keys.Right:
                    case Keys.Space:
                    case Keys.Tab:
                    case Keys.Up:
                        {
                            e.Handled = ProcessDataGridViewKey(e);
                            break;
                        }
                }
            }
        }

        [EditorBrowsable(EditorBrowsableState.Advanced)]
        protected override void OnKeyPress(KeyPressEventArgs e)
        {
            base.OnKeyPress(e);
            if (e.Handled)
            {
                return;
            }

            // Forward key press to current cell if any
            if (ptCurrentCell.X != -1)
            {
                DataGridViewCell dataGridViewCell = CurrentCellInternal;
                Debug.Assert(dataGridViewCell != null);
                if (dataGridViewCell.KeyPressUnsharesRowInternal(e, ptCurrentCell.Y))
                {
                    DataGridViewRow dataGridViewRow = Rows[ptCurrentCell.Y];
                    CurrentCellInternal.OnKeyPressInternal(e, ptCurrentCell.Y);
                }
                else
                {
                    dataGridViewCell.OnKeyPressInternal(e, ptCurrentCell.Y);
                }
            }
        }

        [EditorBrowsable(EditorBrowsableState.Advanced)]
        protected override void OnKeyUp(KeyEventArgs e)
        {
            base.OnKeyUp(e);
            if (e.Handled)
            {
                return;
            }

            if (dataGridViewOper[DATAGRIDVIEWOPER_trackKeyboardColResize] && (e.KeyData & Keys.Alt) != Keys.Alt)
            {
                EndColumnResize(currentColSplitBar);
                ResetKeyboardTrackingState();
                return;
            }

            // Forward key up to current cell if any
            if (ptCurrentCell.X != -1)
            {
                DataGridViewCell dataGridViewCell = CurrentCellInternal;
                Debug.Assert(dataGridViewCell != null);
                if (dataGridViewCell.KeyUpUnsharesRowInternal(e, ptCurrentCell.Y))
                {
                    DataGridViewRow dataGridViewRow = Rows[ptCurrentCell.Y];
                    CurrentCellInternal.OnKeyUpInternal(e, ptCurrentCell.Y);
                }
                else
                {
                    dataGridViewCell.OnKeyUpInternal(e, ptCurrentCell.Y);
                }
            }
        }

        protected override void OnLayout(LayoutEventArgs e)
        {
            if (dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlChanging])
            {
                return;
            }

            base.OnLayout(e);
            PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, false /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
            if (RightToLeftInternal)
            {
                Invalidate();
            }
            if (editingControl != null)
            {
                PositionEditingControl(true, true, false);
            }
        }

        protected override void OnLeave(EventArgs e)
        {
            if (ptCurrentCell.X > -1 && !dataGridViewState1[DATAGRIDVIEWSTATE1_leavingWithTabKey])
            {
                DataGridViewCell dataGridViewCell = null;
                OnCellLeave(ref dataGridViewCell, ptCurrentCell.X, ptCurrentCell.Y);
                if (ptCurrentCell.X == -1)
                {
                    return;
                }
                OnRowLeave(ref dataGridViewCell, ptCurrentCell.X, ptCurrentCell.Y);
            }

            if (!dataGridViewState1[DATAGRIDVIEWSTATE1_leavingWithTabKey])
            {
                base.OnLeave(e);

                // invalidate the current cell so the data grid view does not paint the focus rectangle any longer
                if (ptCurrentCell.X > -1 && ptCurrentCell.Y > -1)
                {
                    InvalidateCellPrivate(ptCurrentCell.X /*columnIndex*/, ptCurrentCell.Y /*rowIndex*/);
                }
            }

            // Erase focus rectangle around the grid
            if (IsGridFocusRectangleEnabled())
            {
                InvalidateRectangleEdges(GetGridFocusRectangle());
            }
        }

        protected override void OnLostFocus(EventArgs e)
        {
            base.OnLostFocus(e);
            if (ptCurrentCell.X != -1)
            {
                InvalidateCell(ptCurrentCell.X, ptCurrentCell.Y);
            }

            if (CurrentCell != null)
            {
                KeyboardToolTipStateMachine.Instance.NotifyAboutLostFocus(CurrentCell);
            }
        }

        protected override void OnMouseClick(MouseEventArgs e)
        {
            bool mouseClickRaised = false;

            if (!dataGridViewState2[DATAGRIDVIEWSTATE2_messageFromEditingCtrls] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackColResize] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackRowResize] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackColHeadersResize] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackRowHeadersResize])
            {
                HitTestInfo hti = HitTest(e.X, e.Y);

                if (ptMouseDownCell.X == hti.col &&
                    ptMouseDownCell.Y == hti.row &&
                    (e.Button != MouseButtons.Left ||
                     ptMouseDownCell.X == -1 ||
                     ptMouseDownCell.Y == -1 ||
                     (ptMouseDownCell.X == ptCurrentCell.X &&
                      ptMouseDownCell.Y == ptCurrentCell.Y)))
                {
                    DataGridViewCellMouseEventArgs dgvcme = null;
                    if (hti.Type != DataGridViewHitTestType.None &&
                        hti.Type != DataGridViewHitTestType.HorizontalScrollBar &&
                        hti.Type != DataGridViewHitTestType.VerticalScrollBar)
                    {
                        int mouseX = e.X - hti.ColumnX;
                        if (RightToLeftInternal)
                        {
                            mouseX += ((hti.col == -1) ? RowHeadersWidth : Columns[hti.col].Thickness);
                        }
                        dgvcme = new DataGridViewCellMouseEventArgs(hti.col, hti.row, mouseX, e.Y - hti.RowY, e);
                        RecordCellMouseClick(dgvcme);
                        if (e.Button == MouseButtons.Left)
                        {
                            OnCellClick(new DataGridViewCellEventArgs(hti.col, hti.row));
                        }
                        base.OnMouseClick(e);
                        mouseClickRaised = true;
                        if (dgvcme.ColumnIndex < Columns.Count && dgvcme.RowIndex < Rows.Count)
                        {
                            OnCellMouseClick(dgvcme);
                        }
                    }
                    else
                    {
                        base.OnMouseClick(e);
                        mouseClickRaised = true;
                    }

                    if (!dataGridViewOper[DATAGRIDVIEWOPER_trackColRelocation])
                    {
                        switch (hti.typeInternal)
                        {
                            case DataGridViewHitTestTypeInternal.ColumnHeader:
                            case DataGridViewHitTestTypeInternal.ColumnHeaderLeft:
                            case DataGridViewHitTestTypeInternal.ColumnHeaderRight:
                            case DataGridViewHitTestTypeInternal.FirstColumnHeaderLeft:
                                {
                                    Debug.Assert(dgvcme != null);
                                    if (dgvcme.ColumnIndex < Columns.Count && dgvcme.RowIndex < Rows.Count)
                                    {
                                        OnColumnHeaderMouseClick(dgvcme);
                                    }
                                    break;
                                }

                            case DataGridViewHitTestTypeInternal.RowHeader:
                                {
                                    Debug.Assert(dgvcme != null);
                                    if (dgvcme.ColumnIndex < Columns.Count && dgvcme.RowIndex < Rows.Count)
                                    {
                                        OnRowHeaderMouseClick(dgvcme);
                                    }
                                    break;
                                }
                        }
                    }
                }
            }
            if (!mouseClickRaised)
            {
                base.OnMouseClick(e);
            }
        }

        protected override void OnMouseDoubleClick(MouseEventArgs e)
        {
            base.OnMouseDoubleClick(e);

            if (!dataGridViewState2[DATAGRIDVIEWSTATE2_messageFromEditingCtrls] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackColResize] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackRowResize] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackColHeadersResize] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackRowHeadersResize])
            {
                HitTestInfo hti = HitTest(e.X, e.Y);

                if (ptMouseDownCell.X == hti.col && ptMouseDownCell.Y == hti.row)
                {
                    DataGridViewCellMouseEventArgs dgvcme = null;
                    if (hti.Type != DataGridViewHitTestType.None &&
                        hti.Type != DataGridViewHitTestType.HorizontalScrollBar &&
                        hti.Type != DataGridViewHitTestType.VerticalScrollBar)
                    {
                        int mouseX = e.X - hti.ColumnX;
                        if (RightToLeftInternal)
                        {
                            mouseX += ((hti.col == -1) ? RowHeadersWidth : Columns[hti.col].Thickness);
                        }
                        dgvcme = new DataGridViewCellMouseEventArgs(hti.col, hti.row, mouseX, e.Y - hti.RowY, e);
                        OnCellMouseDoubleClick(dgvcme);
                    }

                    if (!dataGridViewOper[DATAGRIDVIEWOPER_trackColRelocation])
                    {
                        switch (hti.typeInternal)
                        {
                            case DataGridViewHitTestTypeInternal.ColumnHeader:
                            case DataGridViewHitTestTypeInternal.ColumnHeaderLeft:
                            case DataGridViewHitTestTypeInternal.ColumnHeaderRight:
                            case DataGridViewHitTestTypeInternal.FirstColumnHeaderLeft:
                                {
                                    Debug.Assert(dgvcme != null);
                                    if (dgvcme.ColumnIndex < Columns.Count && dgvcme.RowIndex < Rows.Count)
                                    {
                                        OnColumnHeaderMouseDoubleClick(dgvcme);
                                    }
                                    break;
                                }

                            case DataGridViewHitTestTypeInternal.ColumnResizeLeft:
                            case DataGridViewHitTestTypeInternal.ColumnResizeRight:
                                {
                                    int columnIndex = (hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnResizeRight) ? hti.col : hti.adjacentCol;
                                    if (columnIndex < Columns.Count)
                                    {
                                        HandledMouseEventArgs hme = new HandledMouseEventArgs(e.Button, e.Clicks, e.X, e.Y, e.Delta, false /*defaultHandledValue*/);
                                        DataGridViewColumnDividerDoubleClickEventArgs dgvcddce = new DataGridViewColumnDividerDoubleClickEventArgs(columnIndex, hme);
                                        Debug.Assert(Columns[columnIndex].Resizable == DataGridViewTriState.True);
                                        OnColumnDividerDoubleClick(dgvcddce);
                                    }
                                    break;
                                }

                            case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeTop:
                            case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeBottom:
                            case DataGridViewHitTestTypeInternal.ColumnHeadersResizeTop:
                            case DataGridViewHitTestTypeInternal.ColumnHeadersResizeBottom:
                                {
                                    HandledMouseEventArgs hme = new HandledMouseEventArgs(e.Button, e.Clicks, e.X, e.Y, e.Delta, false /*defaultHandledValue*/);
                                    DataGridViewRowDividerDoubleClickEventArgs dgvrddce = new DataGridViewRowDividerDoubleClickEventArgs(-1, hme);
                                    Debug.Assert(columnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.EnableResizing);
                                    OnRowDividerDoubleClick(dgvrddce);
                                    break;
                                }

                            case DataGridViewHitTestTypeInternal.RowHeader:
                                {
                                    Debug.Assert(dgvcme != null);
                                    if (dgvcme.ColumnIndex < Columns.Count && dgvcme.RowIndex < Rows.Count)
                                    {
                                        OnRowHeaderMouseDoubleClick(dgvcme);
                                    }
                                    break;
                                }

                            case DataGridViewHitTestTypeInternal.RowResizeBottom:
                            case DataGridViewHitTestTypeInternal.RowResizeTop:
                                {
                                    int rowIndex = (hti.typeInternal == DataGridViewHitTestTypeInternal.RowResizeBottom) ? hti.row : hti.adjacentRow;
                                    if (rowIndex < Rows.Count)
                                    {
                                        HandledMouseEventArgs hme = new HandledMouseEventArgs(e.Button, e.Clicks, e.X, e.Y, e.Delta, false /*defaultHandledValue*/);
                                        DataGridViewRowDividerDoubleClickEventArgs dgvrddce = new DataGridViewRowDividerDoubleClickEventArgs(rowIndex, hme);
                                        Debug.Assert(Rows[rowIndex].Resizable == DataGridViewTriState.True);
                                        OnRowDividerDoubleClick(dgvrddce);
                                    }
                                    break;
                                }

                            case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeLeft:
                            case DataGridViewHitTestTypeInternal.TopLeftHeaderResizeRight:
                            case DataGridViewHitTestTypeInternal.RowHeadersResizeLeft:
                            case DataGridViewHitTestTypeInternal.RowHeadersResizeRight:
                                {
                                    HandledMouseEventArgs hme = new HandledMouseEventArgs(e.Button, e.Clicks, e.X, e.Y, e.Delta, false /*defaultHandledValue*/);
                                    DataGridViewColumnDividerDoubleClickEventArgs dgvcddce = new DataGridViewColumnDividerDoubleClickEventArgs(-1, hme);
                                    Debug.Assert(rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing);
                                    OnColumnDividerDoubleClick(dgvcddce);
                                    break;
                                }
                        }
                    }
                }
            }
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            if (!dataGridViewState2[DATAGRIDVIEWSTATE2_messageFromEditingCtrls])
            {
                dataGridViewOper[DATAGRIDVIEWOPER_trackMouseMoves] = true;
            }

            base.OnMouseDown(e);

            if (dataGridViewState2[DATAGRIDVIEWSTATE2_messageFromEditingCtrls])
            {
                return;
            }

            if (ptMouseDownCell.X != -2)
            {
                // Happens when user pushes the mouse wheel down while the left mouse button is already down
                Debug.Assert(ptMouseDownCell.Y != -2);
                return;
            }

            HitTestInfo hti = HitTest(e.X, e.Y);

            if (hti.Type != DataGridViewHitTestType.None &&
                hti.Type != DataGridViewHitTestType.HorizontalScrollBar &&
                hti.Type != DataGridViewHitTestType.VerticalScrollBar)
            {
                ptMouseDownCell.X = hti.col;
                ptMouseDownCell.Y = hti.row;
                ptMouseDownGridCoord = new Point(e.X, e.Y);
                int mouseX = e.X - hti.ColumnX;
                if (RightToLeftInternal)
                {
                    mouseX += ((hti.col == -1) ? RowHeadersWidth : Columns[hti.col].Thickness);
                }
                DataGridViewCellMouseEventArgs dgvcme = new DataGridViewCellMouseEventArgs(hti.col, hti.row, mouseX, e.Y - hti.RowY, e);
                OnCellMouseDown(dgvcme);
            }
        }

        protected override void OnMouseEnter(EventArgs e)
        {
            if (!dataGridViewState2[DATAGRIDVIEWSTATE2_mouseOverRemovedEditingCtrl] &&
                !dataGridViewState2[DATAGRIDVIEWSTATE2_mouseOverRemovedEditingPanel] &&
                dataGridViewState2[DATAGRIDVIEWSTATE2_mouseEnterExpected] &&
                !toolTipControl.Activated)
            {
                base.OnMouseEnter(e);
            }

            dataGridViewState2[DATAGRIDVIEWSTATE2_mouseOverRemovedEditingCtrl] = false;
            dataGridViewState2[DATAGRIDVIEWSTATE2_mouseOverRemovedEditingPanel] = false;
            dataGridViewState2[DATAGRIDVIEWSTATE2_mouseEnterExpected] = false;
        }

        protected override void OnMouseLeave(EventArgs e)
        {
            // when the mouse leaves the dataGridView control, reset the cursor to the previously cached one
            if (dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet])
            {
                dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet] = false;
                CursorInternal = oldCursor;
            }

            bool mouseOverEditingControl = MouseOverEditingControl;
            bool mouseOverEditingPanel = MouseOverEditingPanel;
            bool mouseOverToolTipControl = toolTipControl.Activated && ClientRectangle.Contains(PointToClient(Control.MousePosition));

            if (!mouseOverEditingPanel && !mouseOverEditingControl && !mouseOverToolTipControl && ptMouseEnteredCell.X != -2)
            {
                if (ptMouseEnteredCell.X >= -1 && ptMouseEnteredCell.X < Columns.Count &&
                    ptMouseEnteredCell.Y >= -1 && ptMouseEnteredCell.Y < Rows.Count)
                {
                    DataGridViewCellEventArgs dgvce = new DataGridViewCellEventArgs(ptMouseEnteredCell.X, ptMouseEnteredCell.Y);
                    OnCellMouseLeave(dgvce);
                }
                else
                {
                    ptMouseEnteredCell.X = ptMouseEnteredCell.Y = -2;
                }
            }

            ResetTrackingState();
            dataGridViewOper[DATAGRIDVIEWOPER_trackMouseMoves] = false;

            if (!mouseOverEditingPanel && !mouseOverEditingControl && !mouseOverToolTipControl && !MouseOverScrollBar)
            {
                toolTipControl.Activate(false /*activate*/);
                base.OnMouseLeave(e);
                dataGridViewState2[DATAGRIDVIEWSTATE2_mouseEnterExpected] = true;
            }
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            HitTestInfo hti = HitTest(e.X, e.Y);

            UpdateMouseEnteredCell(hti, e);

            // We need to give UI feedback when the user is resizing a column
            if (dataGridViewOper[DATAGRIDVIEWOPER_trackColResize])
            {
                MoveRowHeadersOrColumnResize(e.X);
            }
            else if (dataGridViewOper[DATAGRIDVIEWOPER_trackRowResize])
            {
                MoveColumnHeadersOrRowResize(e);
            }
            else if (dataGridViewOper[DATAGRIDVIEWOPER_trackColRelocation])
            {
                MoveColumnRelocation(e, hti);
            }
            else if (dataGridViewOper[DATAGRIDVIEWOPER_trackColHeadersResize])
            {
                MoveColumnHeadersOrRowResize(e);
            }
            else if (dataGridViewOper[DATAGRIDVIEWOPER_trackRowHeadersResize])
            {
                MoveRowHeadersOrColumnResize(e.X);
            }

            if (dataGridViewOper[DATAGRIDVIEWOPER_trackColResize] ||
                dataGridViewOper[DATAGRIDVIEWOPER_trackRowHeadersResize] ||
                ((hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnResizeLeft ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.TopLeftHeaderResizeLeft ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.TopLeftHeaderResizeRight ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnResizeRight ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.RowHeadersResizeLeft ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.RowHeadersResizeRight) &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackColHeadersResize] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackColRelocation] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackColSelect] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackRowSelect] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackCellSelect]))
            {
                if (!dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet])
                {
                    dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet] = true;
                    oldCursor = Cursor;
                }
                CursorInternal = Cursors.SizeWE;
                return;
            }
            else if (dataGridViewOper[DATAGRIDVIEWOPER_trackRowResize] ||
                dataGridViewOper[DATAGRIDVIEWOPER_trackColHeadersResize] ||
                ((hti.typeInternal == DataGridViewHitTestTypeInternal.RowResizeBottom ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.TopLeftHeaderResizeTop ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.TopLeftHeaderResizeBottom ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.RowResizeTop ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnHeadersResizeTop ||
                hti.typeInternal == DataGridViewHitTestTypeInternal.ColumnHeadersResizeBottom) &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackRowHeadersResize] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackColRelocation] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackColSelect] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackRowSelect] &&
                !dataGridViewOper[DATAGRIDVIEWOPER_trackCellSelect]))
            {
                if (!dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet])
                {
                    dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet] = true;
                    oldCursor = Cursor;
                }
                CursorInternal = Cursors.SizeNS;
                return;
            }
            /* Whidbey

*/
            else if (dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet])
            {
                dataGridViewState1[DATAGRIDVIEWSTATE1_customCursorSet] = false;
                CursorInternal = oldCursor;
            }

            if (dataGridViewOper[DATAGRIDVIEWOPER_trackColSelect] ||
                dataGridViewOper[DATAGRIDVIEWOPER_trackRowSelect] ||
                dataGridViewOper[DATAGRIDVIEWOPER_trackCellSelect])
            {
                int mouseX = e.X, mouseY = e.Y;
                if (GetOutOfBoundCorrectedHitTestInfo(ref hti, ref mouseX, ref mouseY, out int xOffset, out int yOffset))
                {
                    if (xOffset == 0)
                    {
                        if (horizScrollTimer != null && horizScrollTimer.Enabled)
                        {
                            // Mouse's X came in-bound - need to stop the horizontal scroll timer
                            horizScrollTimer.Enabled = false;
                        }
                    }
                    else if (horizScrollTimer == null || !horizScrollTimer.Enabled)
                    {
                        // Need to start delayed horizontal scroll
                        HorizScrollTimer.Interval = GetColumnScrollRate(Math.Abs(xOffset));
                        HorizScrollTimer.Enabled = true;
                    }

                    if (yOffset == 0)
                    {
                        if (vertScrollTimer != null && vertScrollTimer.Enabled)
                        {
                            // Mouse's Y came in-bound - need to stop the vertical scroll timer
                            vertScrollTimer.Enabled = false;
                        }
                    }
                    else if (vertScrollTimer == null || !vertScrollTimer.Enabled)
                    {
                        // Need to start delayed vertical scroll
                        VertScrollTimer.Interval = GetRowScrollRate(Math.Abs(yOffset));
                        VertScrollTimer.Enabled = true;
                    }

                    if (HorizScrollTimer.Enabled || VertScrollTimer.Enabled)
                    {
                        return;
                    }

                    if (/*!this.dataGridViewState1[DATAGRIDVIEWSTATE1_scrolledSinceMouseDown] && */
                        hti.Type != DataGridViewHitTestType.None &&
                        hti.Type != DataGridViewHitTestType.HorizontalScrollBar &&
                        hti.Type != DataGridViewHitTestType.VerticalScrollBar)
                    {
                        if (dataGridViewOper[DATAGRIDVIEWOPER_trackColSelect] && hti.col >= 0)
                        {
                            OnColumnSelectMouseMove(hti);
                        }
                        else if (dataGridViewOper[DATAGRIDVIEWOPER_trackRowSelect] && hti.row >= 0)
                        {
                            OnRowSelectMouseMove(hti);
                        }
                        else if (dataGridViewOper[DATAGRIDVIEWOPER_trackCellSelect] && hti.col >= 0 && hti.row >= 0)
                        {
                            OnCellSelectMouseMove(hti);
                        }
                    }
                }
            }
#if DEBUG
            else
            {
                Debug.Assert(vertScrollTimer == null || !vertScrollTimer.Enabled);
                Debug.Assert(horizScrollTimer == null || !horizScrollTimer.Enabled);
            }
#endif
            if (!toolTipControl.Activated)
            {
                //
                // After processing the MouseMove event, the tool tip is still not activated.
                // Reset the tool tip cell.
                ptToolTipCell = new Point(-1, -1);
            }
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            if (!dataGridViewState2[DATAGRIDVIEWSTATE2_messageFromEditingCtrls])
            {
                dataGridViewState1[DATAGRIDVIEWSTATE1_scrolledSinceMouseDown] = false;

                HitTestInfo hti = HitTest(e.X, e.Y);

                if (!IsMouseOperationActive())
                {
                    if (hti.Type != DataGridViewHitTestType.None &&
                        hti.Type != DataGridViewHitTestType.HorizontalScrollBar &&
                        hti.Type != DataGridViewHitTestType.VerticalScrollBar)
                    {
                        int mouseX = e.X - hti.ColumnX;
                        if (RightToLeftInternal)
                        {
                            mouseX += ((hti.col == -1) ? RowHeadersWidth : Columns[hti.col].Thickness);
                        }
                        DataGridViewCellMouseEventArgs dgvcme;
                        if (dataGridViewState2[DATAGRIDVIEWSTATE2_nextMouseUpIsDouble])
                        {
                            MouseEventArgs meTmp = new MouseEventArgs(e.Button, 2, e.X, e.Y, e.Delta);
                            dgvcme = new DataGridViewCellMouseEventArgs(hti.col, hti.row, mouseX, e.Y - hti.RowY, meTmp);
                        }
                        else
                        {
                            dgvcme = new DataGridViewCellMouseEventArgs(hti.col, hti.row, mouseX, e.Y - hti.RowY, e);
                        }
                        if (hti.col >= 0 && ptMouseDownCell.X == hti.col &&
                            hti.row >= 0 && ptMouseDownCell.Y == hti.row &&
                            EditMode == DataGridViewEditMode.EditOnEnter &&
                            editingControl != null)
                        {
                            OnClick(e);
                            OnMouseClick(e);
                        }

                        CorrectFocus(true /*onlyIfGridHasFocus*/);

                        if (dgvcme.ColumnIndex < Columns.Count && dgvcme.RowIndex < Rows.Count)
                        {
                            OnCellMouseUp(dgvcme);
                        }
                    }
                    else if (hti.Type == DataGridViewHitTestType.None)
                    {
                        // VS Whidbey
                        CorrectFocus(true /*onlyIfGridHasFocus*/);
                    }
                }
                else
                {
                    if (dataGridViewOper[DATAGRIDVIEWOPER_trackColResize])
                    {
                        EndColumnResize(e);
                    }

                    if (dataGridViewOper[DATAGRIDVIEWOPER_trackRowResize])
                    {
                        EndRowResize(e);
                    }

                    if (dataGridViewOper[DATAGRIDVIEWOPER_trackColRelocation])
                    {
                        EndColumnRelocation(e, hti);
                    }

                    if (dataGridViewOper[DATAGRIDVIEWOPER_trackColHeadersResize])
                    {
                        EndColumnHeadersResize(e);
                    }

                    if (dataGridViewOper[DATAGRIDVIEWOPER_trackRowHeadersResize])
                    {
                        EndRowHeadersResize(e);
                    }

                    // VS Whidbey
                    CorrectFocus(true /*onlyIfGridHasFocus*/);

                    // Updating the hit test info since the EndXXX operation above may have invalidated the previously
                    // determined hti.
                    hti = HitTest(e.X, e.Y);
                    if (hti.Type != DataGridViewHitTestType.None &&
                        hti.Type != DataGridViewHitTestType.HorizontalScrollBar &&
                        hti.Type != DataGridViewHitTestType.VerticalScrollBar)
                    {
                        int mouseX = e.X - hti.ColumnX;
                        if (RightToLeftInternal)
                        {
                            mouseX += ((hti.col == -1) ? RowHeadersWidth : Columns[hti.col].Thickness);
                        }
                        OnCellMouseUp(new DataGridViewCellMouseEventArgs(hti.col, hti.row, mouseX, e.Y - hti.RowY, e));
                    }
                }

                ResetTrackingState();
            }
            base.OnMouseUp(e);
        }

        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            HandledMouseEventArgs hme = e as HandledMouseEventArgs;
            if (hme != null && hme.Handled)
            {
                // The application event handler handled the scrolling - don't do anything more.
                return;
            }

            if ((ModifierKeys & (Keys.Shift | Keys.Alt)) != 0 || MouseButtons != MouseButtons.None)
            {
                return; // Do not scroll when Shift or Alt key is down, or when a mouse button is down.
            }

            bool verticalScroll = ((ModifierKeys & Keys.Control) == 0);

            ScrollBar sb = (verticalScroll ? (ScrollBar)vertScrollBar : (ScrollBar)horizScrollBar);

            if (!sb.Visible || !sb.Enabled)
            {
                return; // Do not scroll when the corresponding scrollbar is invisible or disabled
            }

            if (hme != null)
            {
                hme.Handled = true;
            }

            int wheelScrollLines = SystemInformation.MouseWheelScrollLines;
            if (wheelScrollLines == 0)
            {
                return; // Do not scroll when the user system setting is 0 lines per notch
            }

            Debug.Assert(cumulativeVerticalWheelDelta > -NativeMethods.WHEEL_DELTA);
            Debug.Assert(cumulativeVerticalWheelDelta < NativeMethods.WHEEL_DELTA);
            Debug.Assert(cumulativeHorizontalWheelDelta > -NativeMethods.WHEEL_DELTA);
            Debug.Assert(cumulativeHorizontalWheelDelta < NativeMethods.WHEEL_DELTA);

            float partialNotches;

            if (verticalScroll)
            {
                cumulativeVerticalWheelDelta += e.Delta;
                partialNotches = (float)cumulativeVerticalWheelDelta / (float)NativeMethods.WHEEL_DELTA;
            }
            else
            {
                cumulativeHorizontalWheelDelta += e.Delta;
                partialNotches = (float)cumulativeHorizontalWheelDelta / (float)NativeMethods.WHEEL_DELTA;
            }

            int fullNotches = (int)partialNotches;

            if (wheelScrollLines == -1)
            {
                // Equivalent to large change scrolls
                if (fullNotches != 0)
                {
                    if (ptCurrentCell.X >= 0 &&
                        !CommitEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit | DataGridViewDataErrorContexts.Scroll,
                                    false /*forCurrentCellChange*/, false /*forCurrentRowChange*/))
                    {
                        // Could not commit edited cell value
                        return;
                    }
                    if (verticalScroll)
                    {
                        int originalVerticalOffset = VerticalOffset;
                        VerticalOffset -= fullNotches * vertScrollBar.LargeChange;
                        if (Math.Abs(VerticalOffset - originalVerticalOffset) >= Math.Abs(fullNotches * vertScrollBar.LargeChange))
                        {
                            cumulativeVerticalWheelDelta -= fullNotches * NativeMethods.WHEEL_DELTA;
                        }
                        else
                        {
                            cumulativeVerticalWheelDelta = 0;
                        }
                    }
                    else
                    {
                        int originalHorizontalOffset = HorizontalOffset;
                        HorizontalOffset -= fullNotches * horizScrollBar.LargeChange;
                        if (Math.Abs(HorizontalOffset - originalHorizontalOffset) >= Math.Abs(fullNotches * horizScrollBar.LargeChange))
                        {
                            cumulativeHorizontalWheelDelta -= fullNotches * NativeMethods.WHEEL_DELTA;
                        }
                        else
                        {
                            cumulativeHorizontalWheelDelta = 0;
                        }
                    }
                }
            }
            else
            {
                // Evaluate number of bands to scroll
                int scrollBands = (int)((float)wheelScrollLines * partialNotches);
                if (scrollBands != 0)
                {
                    if (ptCurrentCell.X >= 0 &&
                        !CommitEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit | DataGridViewDataErrorContexts.Scroll,
                                    false /*forCurrentCellChange*/, false /*forCurrentRowChange*/))
                    {
                        // Could not commit edited cell value
                        return;
                    }
                    int absScrollBands;
                    if (verticalScroll)
                    {
                        if (scrollBands > 0)
                        {
                            absScrollBands = scrollBands;
                            while (vertScrollBar.Value != vertScrollBar.Minimum && absScrollBands > 0)
                            {
                                ScrollRowsByCount(-1, ScrollEventType.SmallDecrement);
                                absScrollBands--;
                            }
                            if (vertScrollBar.Value == vertScrollBar.Minimum)
                            {
                                cumulativeVerticalWheelDelta = 0;
                            }
                            else
                            {
                                cumulativeVerticalWheelDelta -= (int)((float)scrollBands * ((float)NativeMethods.WHEEL_DELTA / (float)wheelScrollLines));
                            }
                        }
                        else
                        {
                            absScrollBands = -scrollBands;
                            Debug.Assert(displayedBandsInfo.FirstDisplayedScrollingRow >= 0);
                            int totalVisibleFrozenHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                            while (absScrollBands > 0 &&
                                   vertScrollBar.Value + Rows.SharedRow(displayedBandsInfo.FirstDisplayedScrollingRow).GetHeight(displayedBandsInfo.FirstDisplayedScrollingRow) <=
                                   vertScrollBar.Maximum - ComputeHeightOfFittingTrailingScrollingRows(totalVisibleFrozenHeight))
                            {
                                ScrollRowsByCount(1, ScrollEventType.SmallIncrement);
                                // Assuming totalVisibleFrozenHeight is unchanged by scrolling operation
                                Debug.Assert(totalVisibleFrozenHeight == Rows.GetRowsHeight(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen));
                                Debug.Assert(displayedBandsInfo.FirstDisplayedScrollingRow >= 0);
                                absScrollBands--;
                            }
                            if (vertScrollBar.Value + Rows.SharedRow(displayedBandsInfo.FirstDisplayedScrollingRow).GetHeight(displayedBandsInfo.FirstDisplayedScrollingRow) >
                                vertScrollBar.Maximum - ComputeHeightOfFittingTrailingScrollingRows(totalVisibleFrozenHeight))
                            {
                                cumulativeVerticalWheelDelta = 0;
                            }
                            else
                            {
                                cumulativeVerticalWheelDelta -= (int)((float)scrollBands * ((float)NativeMethods.WHEEL_DELTA / (float)wheelScrollLines));
                            }
                        }
                    }
                    else
                    {
                        int extremeScrollBarValue, scrollBand;
                        if (scrollBands > 0)
                        {
                            extremeScrollBarValue = horizScrollBar.Minimum;
                            scrollBand = -1;
                        }
                        else
                        {
                            extremeScrollBarValue = horizScrollBar.Maximum;
                            scrollBand = 1;
                        }
                        absScrollBands = Math.Abs(scrollBands);
                        while (horizScrollBar.Value != extremeScrollBarValue && absScrollBands > 0)
                        {
                            ScrollColumns(scrollBand);
                            absScrollBands--;
                        }
                        if (horizScrollBar.Value == extremeScrollBarValue)
                        {
                            cumulativeHorizontalWheelDelta = 0;
                        }
                        else
                        {
                            cumulativeHorizontalWheelDelta -= (int)((float)scrollBands * ((float)NativeMethods.WHEEL_DELTA / (float)wheelScrollLines));
                        }
                    }
                }
            }
        }

        internal void OnMouseWheelInternal(MouseEventArgs e)
        {
            // Notification forwarded from editing control
            OnMouseWheel(e);
        }

        protected virtual void OnMultiSelectChanged(EventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWMULTISELECTCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnNewRowNeeded(DataGridViewRowEventArgs e)
        {
            if (e.Row.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_RowDoesNotBelongToDataGridView, "e.Row");
            }
            if (Events[EVENT_DATAGRIDVIEWNEWROWNEEDED] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            try
            {
                // We can't paint if we are disposed.
                if (dataGridViewOper[DATAGRIDVIEWOPER_inDispose] || IsDisposed)
                {
                    return;
                }

                if (layout.dirty)
                {
                    PerformLayoutPrivate(false /*useRowShortcut*/, true /*computeVisibleRows*/, false /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
                }

                Graphics g = e.Graphics;
                Rectangle clipRect = e.ClipRectangle;
                Rectangle gridRect = GetGridRectangle();

                if (currentRowSplitBar != -1)
                {
                    clipRect = Rectangle.Union(clipRect, CalcRowResizeFeedbackRect(currentRowSplitBar));
                }
                else if (currentColSplitBar != -1)
                {
                    clipRect = Rectangle.Union(clipRect, CalcColResizeFeedbackRect(currentColSplitBar));
                }

                if (clipRect.IntersectsWith(gridRect))
                {
                    using (Region clipRegion = g.Clip)
                    {
                        g.SetClip(gridRect);
                        PaintBackground(g, clipRect, gridRect);
                        PaintGrid(g, gridRect, clipRect, SingleVerticalBorderAdded, SingleHorizontalBorderAdded);
                        g.Clip = clipRegion;
                    }
                }

                PaintBorder(g, clipRect, layout.ClientRectangle);
                if (clipRect.IntersectsWith(layout.ResizeBoxRect))
                {
                    g.FillRectangle(SystemBrushes.Control, layout.ResizeBoxRect);
                }

                // Draw focus rectangle around the grid
                if (Focused && IsGridFocusRectangleEnabled())
                {
                    if (SystemInformation.HighContrast)
                    {
                        ControlPaint.DrawHighContrastFocusRectangle(g, GetGridFocusRectangle(), SystemColors.ActiveCaptionText);
                    }
                    else
                    {
                        ControlPaint.DrawFocusRectangle(g, GetGridFocusRectangle());
                    }
                }

                base.OnPaint(e); // raise paint event
            }
            catch (Exception ex)
            {
#if DEBUG
                Debug.Fail("DataGridView.OnPaint exception: " + ex.Message + " stack trace " + ex.StackTrace);
#endif
                if (ClientUtils.IsCriticalException(ex))
                {
                    throw;
                }
            }
        }

        // Determines if a focus rectangle may be drawn along the perimeter of the DataGridView control
        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        private bool IsGridFocusRectangleEnabled()
        {
            return ShowFocusCues && CurrentCell == null;
        }

        // Creates a rectangle by merging row headers, column headers
        // and cells rectangles (from layout data)
        private Rectangle GetGridRectangle()
        {
            Rectangle gridRect = layout.Data;
            if (layout.RowHeadersVisible)
            {
                gridRect = Rectangle.Union(gridRect, layout.RowHeaders);
            }
            else if (SingleVerticalBorderAdded)
            {
                if (!RightToLeftInternal)
                {
                    gridRect.X--;
                }
                gridRect.Width++;
            }

            if (layout.ColumnHeadersVisible)
            {
                gridRect = Rectangle.Union(gridRect, layout.ColumnHeaders);
            }
            else if (SingleHorizontalBorderAdded)
            {
                if (gridRect.Y == layout.Data.Y)
                {
                    gridRect.Y--;
                    gridRect.Height++;
                }
            }

            return gridRect;
        }

        // Creates a grid focus rectangle
        private Rectangle GetGridFocusRectangle()
        {
            Rectangle focusRect = GetGridRectangle();
            focusRect.Inflate(1 - FOCUS_RECT_OFFSET, 1 - FOCUS_RECT_OFFSET);
            return focusRect;
        }

        private void InvalidateGridFocusOnScroll(int change, ScrollOrientation orientation)
        {
            if (change == 0)
            {
                return;
            }

            Rectangle focusRect = GetGridFocusRectangle();

            if (orientation == ScrollOrientation.HorizontalScroll)
            {
                // Scroll right
                if (change > 0)
                {
                    focusRect.Width -= change;
                }
                // Scroll left
                else
                {
                    focusRect.X -= change;
                    focusRect.Width += change;
                }
            }
            else
            {
                // Scroll down
                if (change > 0)
                {
                    focusRect.Height -= change;
                }
                // Scroll up
                else
                {
                    focusRect.Y -= change;
                    focusRect.Height += change;
                }
            }

            InvalidateRectangleEdges(focusRect);
        }

        private void InvalidateRectangleEdges(Rectangle rect)
        {
            // Left edge
            Rectangle edge = rect;
            edge.Width = 1;
            Invalidate(edge);

            // Right edge
            edge.X += rect.Width - 1;
            Invalidate(edge);

            // Top edge
            edge = rect;
            edge.Height = 1;
            Invalidate(edge);

            // Bottom edge
            edge.Y += rect.Height - 1;
            Invalidate(edge);
        }

        internal override void OnParentBecameInvisible()
        {
            base.OnParentBecameInvisible();
            if (GetState(STATE_VISIBLE))
            {
                // This control became invisible too - Update the Displayed properties of the bands.
                OnVisibleChangedPrivate();
            }
        }

        protected virtual void OnReadOnlyChanged(EventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWREADONLYCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            VerifyImeRestrictedModeChanged();

            if (!ReadOnly &&
                ptCurrentCell.X != -1 &&
                ColumnEditable(ptCurrentCell.X) &&
                !IsCurrentCellInEditMode &&
                (EditMode == DataGridViewEditMode.EditOnEnter ||
                 (EditMode != DataGridViewEditMode.EditProgrammatically && CurrentCellInternal.EditType == null)) &&
                !Rows[ptCurrentCell.Y].Cells[ptCurrentCell.X].ReadOnly) // Unshares the row
            {
                // Current cell becomes read/write. Enter editing mode.
                BeginEditInternal(true /*selectAll*/);
            }
        }

        internal void OnRemovedColumn_PreNotification(DataGridViewColumn dataGridViewColumn)
        {
            Debug.Assert(dataGridViewColumn.Index >= 0);
            Debug.Assert(dataGridViewColumn.DataGridView == null);

            // Clear the potential header sort glyph
            if (dataGridViewColumn.HasHeaderCell)
            {
                dataGridViewColumn.HeaderCell.SortGlyphDirectionInternal = SortOrder.None;
            }
            // Intentionally keep the DisplayIndex intact after detaching the column.
            CorrectColumnIndexesAfterDeletion(dataGridViewColumn);

            CorrectColumnDisplayIndexesAfterDeletion(dataGridViewColumn);

            // Fix the OldFirstDisplayedScrollingCol
            displayedBandsInfo.CorrectRowIndexAfterDeletion(dataGridViewColumn.Index);

            // Raise the ColumnRemoved event
            OnColumnRemoved(dataGridViewColumn);
        }

        internal void OnRemovedColumn_PostNotification(DataGridViewColumn dataGridViewColumn, Point newCurrentCell)
        {
            // Update current cell if needed
            if (newCurrentCell.X != -1)
            {
                Debug.Assert(ptCurrentCell.X == -1);
                bool success = SetAndSelectCurrentCellAddress(newCurrentCell.X,
                                                              newCurrentCell.Y,
                                                              true,
                                                              false,
                                                              false,
                                                              false /*clearSelection*/,
                                                              false /*forceCurrentCellSelection*/);
                Debug.Assert(success);
            }

            // Raise SelectionChanged event if needed
            FlushSelectionChanged();

            // Raise ColumnStateChanged event for Displayed state of deleted column
            OnColumnHidden(dataGridViewColumn);

            // Columns that are removed from the collection take their non-autosized width
            // Note that in some edge cases, the dev could have changed:
            //     - the grid's AutoSizeColumnsMode
            //     - the column's Width or AutoSizeMode
            // in a ColumnRemoved event handler for example, in which case using the CachedThickness may be wrong.
            // At least we make sure the column is not sized smaller than its minimum width.
            DataGridViewAutoSizeColumnMode autoSizeColumnMode = dataGridViewColumn.GetInheritedAutoSizeMode(this);
            Debug.Assert(autoSizeColumnMode != DataGridViewAutoSizeColumnMode.NotSet);
            if (autoSizeColumnMode != DataGridViewAutoSizeColumnMode.None &&
                autoSizeColumnMode != DataGridViewAutoSizeColumnMode.Fill &&
                dataGridViewColumn.ThicknessInternal != dataGridViewColumn.CachedThickness)
            {
                dataGridViewColumn.ThicknessInternal = Math.Max(dataGridViewColumn.MinimumWidth, dataGridViewColumn.CachedThickness);
            }

            // Autosize rows if needed
            AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
        }

        internal void OnRemovedRow_PreNotification(int rowIndexDeleted)
        {
            // Fix the OldFirstDisplayedScrollingRow
            displayedBandsInfo.CorrectRowIndexAfterDeletion(rowIndexDeleted);

            CorrectRowIndexesAfterDeletion(rowIndexDeleted);
            ComputeVisibleRows();
        }

        internal void OnRemovedRow_PostNotification(DataGridViewRow dataGridViewRow, Point newCurrentCell)
        {
            // Update current cell if needed
            if (newCurrentCell.Y != -1)
            {
                Debug.Assert(ptCurrentCell.X == -1);
                bool success = SetAndSelectCurrentCellAddress(newCurrentCell.X,
                                                              newCurrentCell.Y,
                                                              true /*setAnchorCellAddress*/,
                                                              false /*validateCurrentCell*/,
                                                              false /*throughMouseClick*/,
                                                              false /*clearSelection*/,
                                                              false /*forceCurrentCellSelection*/);
                Debug.Assert(success);
            }

            // Raise SelectionChange event if needed
            FlushSelectionChanged();

            bool rowDisplayed = dataGridViewRow.DataGridView == null && dataGridViewRow.Displayed;

            // Raise RowStateChanged event for Displayed state of deleted row
            if (rowDisplayed)
            {
                dataGridViewRow.Displayed = false;
                DataGridViewRowStateChangedEventArgs dgvrsce = new DataGridViewRowStateChangedEventArgs(dataGridViewRow, DataGridViewElementStates.Displayed);
                OnRowStateChanged(-1 /*rowIndex*/, dgvrsce);
            }

            // Rows that are removed from the collection take their non-autosized height
            // Note that in some edge cases, the dev could have changed:
            //     - the grid's AutoSizeRowsMode
            //     - the row's Height
            // in a RowsRemoved event handler for example, in which case using the CachedThickness may be wrong.
            // At least we make sure the row is not sized smaller than its minimum height.
            if (autoSizeRowsMode != DataGridViewAutoSizeRowsMode.None && dataGridViewRow.ThicknessInternal != dataGridViewRow.CachedThickness)
            {
                dataGridViewRow.ThicknessInternal = Math.Max(dataGridViewRow.MinimumHeight, dataGridViewRow.CachedThickness);
            }

            // Auto size columms also if needed
            DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaFilter = DataGridViewAutoSizeColumnCriteriaInternal.AllRows;
            if (rowDisplayed)
            {
                autoSizeColumnCriteriaFilter |= DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows;
            }
            bool columnAutoSized = AutoResizeAllVisibleColumnsInternal(autoSizeColumnCriteriaFilter, true /*fixedHeight*/);
            bool fixedColumnHeadersHeight = ColumnHeadersHeightSizeMode != DataGridViewColumnHeadersHeightSizeMode.AutoSize;
            bool fixedRowHeadersWidth = rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing ||
                                        rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.DisableResizing;

            if (fixedRowHeadersWidth && !columnAutoSized)
            {
                // No need to autosize the column headers when the row headers and columns don't change.
                fixedColumnHeadersHeight = true;
            }

            // Auto size column headers
            if (!fixedColumnHeadersHeight)
            {
                AutoResizeColumnHeadersHeight(fixedRowHeadersWidth, true /*fixedColumnsWidth*/);
            }

            // Auto size row headers
            if (!fixedRowHeadersWidth)
            {
                AutoResizeRowHeadersWidth(rowHeadersWidthSizeMode, true /*fixedColumnHeadersHeight*/, true /*fixedRowsHeight*/);
            }

            if (!fixedColumnHeadersHeight && !fixedRowHeadersWidth)
            {
                // Second round of column headers autosizing
                AutoResizeColumnHeadersHeight(true /*fixedRowHeadersWidth*/, true /*fixedColumnsWidth*/);
            }
        }

        internal void OnRemovingColumn(DataGridViewColumn dataGridViewColumn, out Point newCurrentCell, bool force)
        {
            Debug.Assert(dataGridViewColumn != null);
            Debug.Assert(dataGridViewColumn.Index >= 0 && dataGridViewColumn.Index < Columns.Count);

            dataGridViewState1[DATAGRIDVIEWSTATE1_temporarilyResetCurrentCell] = false;
            int columnIndex = dataGridViewColumn.Index;

            // Reset the current cell's address if there is one.
            if (ptCurrentCell.X != -1)
            {
                int newX = ptCurrentCell.X;
                if (columnIndex == ptCurrentCell.X)
                {
                    DataGridViewColumn dataGridViewColumnNext = Columns.GetNextColumn(
                        Columns[columnIndex],
                        DataGridViewElementStates.Visible,
                        DataGridViewElementStates.None);
                    if (dataGridViewColumnNext != null)
                    {
                        if (dataGridViewColumnNext.Index > columnIndex)
                        {
                            newX = dataGridViewColumnNext.Index - 1;
                        }
                        else
                        {
                            newX = dataGridViewColumnNext.Index;
                        }
                    }
                    else
                    {
                        DataGridViewColumn dataGridViewColumnPrevious = Columns.GetPreviousColumn(
                            Columns[columnIndex],
                            DataGridViewElementStates.Visible,
                            DataGridViewElementStates.None);
                        if (dataGridViewColumnPrevious != null)
                        {
                            if (dataGridViewColumnPrevious.Index > columnIndex)
                            {
                                newX = dataGridViewColumnPrevious.Index - 1;
                            }
                            else
                            {
                                newX = dataGridViewColumnPrevious.Index;
                            }
                        }
                        else
                        {
                            newX = -1;
                        }
                    }
                }
                else if (columnIndex < ptCurrentCell.X)
                {
                    newX = ptCurrentCell.X - 1;
                }
                newCurrentCell = new Point(newX, (newX == -1) ? -1 : ptCurrentCell.Y);
                if (columnIndex == ptCurrentCell.X)
                {
                    // Left cell is not validated since cancelling validation wouldn't have any effect anyways.
                    bool success = SetCurrentCellAddressCore(-1, -1, true /*setAnchorCellAddress*/, false /*validateCurrentCell*/, false);
                    Debug.Assert(success);
                }
                else if (force)
                {
                    // Underlying data of deleted column is gone. It cannot be accessed anymore.
                    // Do not end editing mode so that CellValidation doesn't get raised, since that event needs the current formatted value.
                    dataGridViewState1[DATAGRIDVIEWSTATE1_temporarilyResetCurrentCell] = true;
                    bool success = SetCurrentCellAddressCore(-1, -1, true /*setAnchorCellAddress*/, false /*validateCurrentCell*/, false);
                    Debug.Assert(success);
                }
                else
                {
                    // Quit editing mode and set the current cell to its new location once everything is in sync again.
                    ResetCurrentCell();
                }
            }
            else
            {
                newCurrentCell = new Point(-1, -1);
            }

            // If the last column is removed, delete all the rows first.
            if (Columns.Count == 1)
            {
                Debug.Assert(columnIndex == 0);
                Rows.ClearInternal(false /*recreateNewRow*/);
            }

            // Prepare the existing rows by removing cells at correct index
            int newColumnCount = Columns.Count - 1;

            for (int rowIndex = 0; rowIndex < Rows.Count; rowIndex++)
            {
                DataGridViewRow dataGridViewRow = Rows.SharedRow(rowIndex);
                if (dataGridViewRow.Cells.Count > newColumnCount)
                {
                    KeyboardToolTipStateMachine.Instance.Unhook(dataGridViewRow.Cells[columnIndex], KeyboardToolTip);
                    dataGridViewRow.Cells.RemoveAtInternal(columnIndex);
                }
            }

            // Detach column header cell
            if (dataGridViewColumn.HasHeaderCell)
            {
                dataGridViewColumn.HeaderCell.DataGridView = null;
            }

            // Reset sort related variables.
            if (dataGridViewColumn == sortedColumn)
            {
                sortedColumn = null;
                sortOrder = SortOrder.None;

                if (dataGridViewColumn.IsDataBound)
                {
                    // If the column being removed is the sorted column and it is also the dataBound column
                    // then see if there is another dataBound column which has the same property name as the sorted column.
                    // If so, then make that dataGridViewColumn the sorted column in the data grid view.
                    for (int i = 0; i < Columns.Count; i++)
                    {
                        if (dataGridViewColumn != Columns[i] &&
                            Columns[i].SortMode != DataGridViewColumnSortMode.NotSortable &&
                            string.Compare(dataGridViewColumn.DataPropertyName,
                                           Columns[i].DataPropertyName,
                                           true /*ignoreCase*/,
                                           CultureInfo.InvariantCulture) == 0)
                        {
                            Debug.Assert(Columns[i].IsDataBound, "two columns w/ the same DataPropertyName should be DataBound at the same time");
                            Debug.Assert(Columns[i].HeaderCell.SortGlyphDirection == dataGridViewColumn.HeaderCell.SortGlyphDirection, "DataBound columns should have the same SortGlyphDirection as the one on the DataGridView");
                            sortedColumn = Columns[i];
                            sortOrder = Columns[i].HeaderCell.SortGlyphDirection;
                            break;
                        }
                    }
                }
            }

            // Is deleted column scrolled off screen?
            if (dataGridViewColumn.Visible &&
                !dataGridViewColumn.Frozen &&
                displayedBandsInfo.FirstDisplayedScrollingCol >= 0)
            {
                // Deleted column is part of scrolling columns.
                if (displayedBandsInfo.FirstDisplayedScrollingCol == dataGridViewColumn.Index)
                {
                    // Deleted column is first scrolling column
                    horizontalOffset -= negOffset;
                    negOffset = 0;
                }
                else if (Columns.DisplayInOrder(displayedBandsInfo.FirstDisplayedScrollingCol, dataGridViewColumn.Index))
                {
                    // Deleted column is displayed after first scrolling column
                    if (horizScrollBar.Enabled)
                    {
                        int newHorizontalOffset = horizScrollBar.Maximum - horizScrollBar.LargeChange - dataGridViewColumn.Thickness;
                        if (newHorizontalOffset >= 0 && newHorizontalOffset < horizScrollBar.Value)
                        {
                            horizontalOffset = newHorizontalOffset;
                            negOffset = GetNegOffsetFromHorizontalOffset(horizontalOffset);
                        }
                    }
                    else
                    {
                        horizontalOffset = negOffset = 0;
                    }
                }
                else
                {
                    // Deleted column is displayed before first scrolling column
                    Debug.Assert(horizontalOffset >= dataGridViewColumn.Thickness);
                    horizontalOffset -= dataGridViewColumn.Thickness;
                }

                if (horizScrollBar.Enabled)
                {
                    horizScrollBar.Value = horizontalOffset;
                }
            }

            bool raiseSelectionChanged = false;

            // Update the indexes of selected columns or individual cells to compensate for the removal of this column
            switch (SelectionMode)
            {
                case DataGridViewSelectionMode.FullColumnSelect:
                case DataGridViewSelectionMode.ColumnHeaderSelect:
                    int columnEntries = selectedBandIndexes.Count;
                    int columnEntry = 0;
                    while (columnEntry < columnEntries)
                    {
                        int columnIndexSelected = selectedBandIndexes[columnEntry];
                        if (columnIndex == columnIndexSelected)
                        {
                            selectedBandIndexes.RemoveAt(columnEntry);
                            columnEntries--;
                            raiseSelectionChanged = true;
                        }
                        else
                        {
                            if (columnIndex < columnIndexSelected)
                            {
                                selectedBandIndexes[columnEntry] = columnIndexSelected - 1;
                            }
                            columnEntry++;
                        }
                    }
                    break;
            }

            dataGridViewState2[DATAGRIDVIEWSTATE2_raiseSelectionChanged] |= individualSelectedCells.RemoveAllCellsAtBand(true /*column*/, columnIndex) > 0 ||
                                                                                 raiseSelectionChanged;
            individualReadOnlyCells.RemoveAllCellsAtBand(true /*column*/, columnIndex);
        }

        internal void OnRemovingRow(int rowIndexDeleted, out Point newCurrentCell, bool force)
        {
            // if force is true, the row needs to be deleted no matter what. The underlying data row was already deleted.

            Debug.Assert(rowIndexDeleted >= 0 && rowIndexDeleted < Rows.Count);

            if (rowIndexDeleted >= 0 && rowIndexDeleted < Rows.Count)
            {
                foreach (DataGridViewCell cell in Rows[rowIndexDeleted].Cells)
                {
                    KeyboardToolTipStateMachine.Instance.Unhook(cell, KeyboardToolTip);
                }
            }

            dataGridViewState1[DATAGRIDVIEWSTATE1_temporarilyResetCurrentCell] = false;
            newCurrentCell = new Point(-1, -1);

            // Reset the current cell's address if it's on the deleted row, or after it.
            if (ptCurrentCell.Y != -1 && rowIndexDeleted <= ptCurrentCell.Y)
            {
                int newY;
                if (rowIndexDeleted == ptCurrentCell.Y)
                {
                    int rowIndexPrevious = Rows.GetPreviousRow(rowIndexDeleted, DataGridViewElementStates.Visible);
                    int rowIndexNext = Rows.GetNextRow(rowIndexDeleted, DataGridViewElementStates.Visible);
                    if (rowIndexPrevious > -1 && AllowUserToAddRowsInternal)
                    {
                        Debug.Assert(newRowIndex != -1);
                        Debug.Assert(newRowIndex == Rows.Count - 1);
                        if (rowIndexNext > -1 && rowIndexNext < Rows.Count - 1)
                        {
                            newY = rowIndexNext - 1;
                        }
                        else
                        {
                            newY = rowIndexPrevious;
                        }
                    }
                    else
                    {
                        if (rowIndexNext > -1)
                        {
                            newY = rowIndexNext - 1;
                        }
                        else
                        {
                            newY = rowIndexPrevious;
                        }
                    }
                    // Since the current row is deleted, the dirty states need to be reset
                    IsCurrentCellDirtyInternal = false;
                    IsCurrentRowDirtyInternal = false;
                }
                else
                {
                    Debug.Assert(rowIndexDeleted < ptCurrentCell.Y);
                    newY = ptCurrentCell.Y - 1;
                }
                newCurrentCell = new Point(ptCurrentCell.X, newY);
                if (rowIndexDeleted == ptCurrentCell.Y)
                {
                    // Left cell is not validated since cancelling validation wouldn't have any effect anyways.
                    bool success = SetCurrentCellAddressCore(-1, -1, true /*setAnchorCellAddress*/, false /*validateCurrentCell*/, false);
                    Debug.Assert(success);
                }
                else if (force)
                {
                    // Underlying data of deleted row is gone. It cannot be accessed anymore.
                    // Do not end editing mode so that CellValidation doesn't get raised, since that event needs the current formatted value.
                    dataGridViewState1[DATAGRIDVIEWSTATE1_temporarilyResetCurrentCell] = true;
                    bool success = SetCurrentCellAddressCore(-1, -1, true /*setAnchorCellAddress*/, false /*validateCurrentCell*/, false);
                    Debug.Assert(success);
                }
                else
                {
                    // Quit editing mode and set the current cell to its new location once everything is in sync again.
                    ResetCurrentCell();
                }
            }

            bool raiseSelectionChanged = false;

            // Update the indexes of selected rows to compensate for the removal of this row
            switch (SelectionMode)
            {
                case DataGridViewSelectionMode.FullRowSelect:
                case DataGridViewSelectionMode.RowHeaderSelect:
                    int rowEntries = selectedBandIndexes.Count;
                    int rowEntry = 0;
                    while (rowEntry < rowEntries)
                    {
                        int rowIndex = selectedBandIndexes[rowEntry];
                        if (rowIndexDeleted == rowIndex)
                        {
                            raiseSelectionChanged = true;
                            selectedBandIndexes.RemoveAt(rowEntry);
                            rowEntries--;
                        }
                        else
                        {
                            if (rowIndexDeleted < rowIndex)
                            {
                                selectedBandIndexes[rowEntry] = rowIndex - 1;
                            }
                            rowEntry++;
                        }
                    }
                    if (selectedBandSnapshotIndexes != null)
                    {
                        rowEntries = selectedBandSnapshotIndexes.Count;
                        rowEntry = 0;
                        while (rowEntry < rowEntries)
                        {
                            int rowIndex = selectedBandSnapshotIndexes[rowEntry];
                            if (rowIndexDeleted == rowIndex)
                            {
                                selectedBandSnapshotIndexes.RemoveAt(rowEntry);
                                rowEntries--;
                            }
                            else
                            {
                                if (rowIndexDeleted < rowIndex)
                                {
                                    selectedBandSnapshotIndexes[rowEntry] = rowIndex - 1;
                                }
                                rowEntry++;
                            }
                        }
                    }
                    break;
            }

            dataGridViewState2[DATAGRIDVIEWSTATE2_raiseSelectionChanged] |= individualSelectedCells.RemoveAllCellsAtBand(false /*column*/, rowIndexDeleted) > 0 ||
                                                                                 raiseSelectionChanged;
            individualReadOnlyCells.RemoveAllCellsAtBand(false /*column*/, rowIndexDeleted);
        }

        internal void OnReplacedCell(DataGridViewRow dataGridViewRow, int columnIndex)
        {
            DataGridViewCell dataGridViewCell = dataGridViewRow.Cells[columnIndex];
            if (dataGridViewState2[DATAGRIDVIEWSTATE2_replacedCellSelected])
            {
                individualSelectedCells.Add(dataGridViewCell);
            }
            if (dataGridViewState2[DATAGRIDVIEWSTATE2_replacedCellReadOnly])
            {
                individualReadOnlyCells.Add(dataGridViewCell);
            }

            // AutoSize column and row if needed
            OnCellCommonChange(columnIndex, dataGridViewRow.Index);

            if (ptCurrentCellCache.X != -1)
            {
                if (!IsInnerCellOutOfBounds(ptCurrentCellCache.X, ptCurrentCellCache.Y))
                {
                    SetCurrentCellAddressCore(ptCurrentCellCache.X, ptCurrentCellCache.Y, false, false, false);
                }
                ptCurrentCellCache.X = -1;
                ptCurrentCellCache.Y = -1;
            }
        }

        internal void OnReplacingCell(DataGridViewRow dataGridViewRow, int columnIndex)
        {
            if (ptCurrentCell.X == dataGridViewRow.Index &&
                ptCurrentCell.Y == columnIndex)
            {
                // Trying to replace the current cell. Exiting editing mode first (if needed).
                // Remember to reset the current cell in OnReplacedCell notification
                ptCurrentCellCache.X = ptCurrentCell.X;
                ptCurrentCellCache.Y = ptCurrentCell.Y;
                // This may fail and throw an exception
                ResetCurrentCell();
            }
            else
            {
                ptCurrentCellCache.X = -1;
                ptCurrentCellCache.Y = -1;
            }
            DataGridViewCell dataGridViewCell = dataGridViewRow.Cells[columnIndex];
            dataGridViewState2[DATAGRIDVIEWSTATE2_replacedCellSelected] = individualSelectedCells.Contains(dataGridViewCell);
            if (dataGridViewState2[DATAGRIDVIEWSTATE2_replacedCellSelected])
            {
                individualSelectedCells.Remove(dataGridViewCell);
            }
            dataGridViewState2[DATAGRIDVIEWSTATE2_replacedCellReadOnly] = individualReadOnlyCells.Contains(dataGridViewCell);
            if (dataGridViewState2[DATAGRIDVIEWSTATE2_replacedCellReadOnly])
            {
                individualReadOnlyCells.Remove(dataGridViewCell);
            }
        }

        protected override void OnResize(EventArgs e)
        {
            int borderWidth = BorderWidth;
            Rectangle right;
            Rectangle bottom;
            Rectangle oldClientRectangle = layout.ClientRectangle;
            Rectangle oldGridFocusRectangle = GetGridFocusRectangle();

            right = new Rectangle(oldClientRectangle.X + oldClientRectangle.Width - borderWidth,
                oldClientRectangle.Y,
                borderWidth,
                oldClientRectangle.Height);
            bottom = new Rectangle(oldClientRectangle.X,
                oldClientRectangle.Y + oldClientRectangle.Height - borderWidth,
                oldClientRectangle.Width,
                borderWidth);

            if (!IsMinimized)
            {
                // When owning form is minimized, act as if it had a normal size
                normalClientRectangle = ClientRectangle;
            }

            Rectangle newClientRectangle = normalClientRectangle;
            Rectangle newGridFocusRectangle = DisplayRectangle;
            newGridFocusRectangle.Inflate(1 - borderWidth - FOCUS_RECT_OFFSET, 1 - borderWidth - FOCUS_RECT_OFFSET);

            if (newClientRectangle.Width != oldClientRectangle.Width)
            {
                Invalidate(right);
                right = new Rectangle(newClientRectangle.X + newClientRectangle.Width - borderWidth,
                    newClientRectangle.Y,
                    borderWidth,
                    newClientRectangle.Height);
                Invalidate(right);
            }
            if (newClientRectangle.Height != oldClientRectangle.Height)
            {
                Invalidate(bottom);
                bottom = new Rectangle(newClientRectangle.X,
                    newClientRectangle.Y + newClientRectangle.Height - borderWidth,
                    newClientRectangle.Width,
                    borderWidth);
                Invalidate(bottom);
            }

            // Invalidate grid focus rectangle
            if (Focused && IsGridFocusRectangleEnabled() && oldGridFocusRectangle != newGridFocusRectangle)
            {
                right = new Rectangle(oldGridFocusRectangle.X + oldGridFocusRectangle.Width - 1,
                    oldGridFocusRectangle.Y,
                    1,
                    oldGridFocusRectangle.Height);
                Invalidate(right);

                bottom = new Rectangle(oldGridFocusRectangle.X,
                    oldGridFocusRectangle.Y + oldGridFocusRectangle.Height - 1,
                    oldGridFocusRectangle.Width,
                    1);
                Invalidate(bottom);

                InvalidateRectangleEdges(newGridFocusRectangle);
            }

            //also, invalidate the ResizeBoxRect
            if (!layout.ResizeBoxRect.IsEmpty)
            {
                Invalidate(layout.ResizeBoxRect);
            }
            layout.ClientRectangle = newClientRectangle;

            int oldfirstDisplayedScrollingRow = displayedBandsInfo.FirstDisplayedScrollingRow;
            base.OnResize(e);
            if (oldfirstDisplayedScrollingRow != displayedBandsInfo.FirstDisplayedScrollingRow)
            {
                Invalidate();
            }
        }

        protected override void OnRightToLeftChanged(EventArgs e)
        {
            dataGridViewState2[DATAGRIDVIEWSTATE2_rightToLeftValid] = false;
            base.OnRightToLeftChanged(e);
            CorrectFocus(true /*onlyIfGridHasFocus*/);
            PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
        }

        internal void OnRowCollectionChanged_PostNotification(bool recreateNewRow,
                                                              bool allowSettingCurrentCell,
                                                              CollectionChangeAction cca,
                                                              DataGridViewRow dataGridViewRow,
                                                              int rowIndex)
        {
            if (recreateNewRow &&
                cca == CollectionChangeAction.Refresh &&
                Columns.Count != 0 &&
                Rows.Count == 0 &&
                AllowUserToAddRowsInternal)
            {
                AddNewRow(false);
            }

            if (cca == CollectionChangeAction.Refresh)
            {
                FlushSelectionChanged();
            }

            if ((cca == CollectionChangeAction.Refresh || cca == CollectionChangeAction.Add) &&
                ptCurrentCell.X == -1 && allowSettingCurrentCell && !InSortOperation)
            {
                MakeFirstDisplayedCellCurrentCell(false /*includeNewRow*/);
            }

            if (AutoSize)
            {
                bool invalidatePreferredSizeCache = true;
                switch (cca)
                {
                    case CollectionChangeAction.Add:
                        Debug.Assert(rowIndex >= 0);
                        DataGridViewElementStates rowState = Rows.GetRowState(rowIndex);
                        invalidatePreferredSizeCache = ((rowState & DataGridViewElementStates.Visible) != 0);
                        break;
                    case CollectionChangeAction.Remove:
                        invalidatePreferredSizeCache = dataGridViewRow.DataGridView == null && dataGridViewRow.Visible;
                        break;
                        // case CollectionChangeAction.Refresh: invalidatePreferredSizeCache stays true
                }
                if (invalidatePreferredSizeCache)
                {
                    LayoutTransaction.DoLayout(ParentInternal, this, PropertyNames.Rows);
                }
            }
        }

        protected virtual void OnRowContextMenuStripChanged(DataGridViewRowEventArgs e)
        {
            if (e.Row.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_RowDoesNotBelongToDataGridView, "e.Row");
            }
            if (Events[EVENT_DATAGRIDVIEWROWCONTEXTMENUSTRIPCHANGED] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal ContextMenuStrip OnRowContextMenuStripNeeded(int rowIndex, ContextMenuStrip contextMenuStrip)
        {
            DataGridViewRowContextMenuStripNeededEventArgs dgvrcmsne = new DataGridViewRowContextMenuStripNeededEventArgs(rowIndex, contextMenuStrip);
            OnRowContextMenuStripNeeded(dgvrcmsne);
            return dgvrcmsne.ContextMenuStrip;
        }

        protected virtual void OnRowContextMenuStripNeeded(DataGridViewRowContextMenuStripNeededEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWCONTEXTMENUSTRIPNEEDED] is DataGridViewRowContextMenuStripNeededEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnRowDefaultCellStyleChanged(DataGridViewRowEventArgs e)
        {
            if (e.Row.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_RowDoesNotBelongToDataGridView, "e.Row");
            }

            OnRowGlobalAutoSize(e.Row.Index);

            if (Events[EVENT_DATAGRIDVIEWROWDEFAULTCELLSTYLECHANGED] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnRowDirtyStateNeeded(QuestionEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWDIRTYSTATENEEDED] is QuestionEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnRowDividerDoubleClick(DataGridViewRowDividerDoubleClickEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWDIVIDERDOUBLECLICK] is DataGridViewRowDividerDoubleClickEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            if (!e.Handled && e.Button == MouseButtons.Left && e.RowIndex < Rows.Count)
            {
                if (e.RowIndex == -1)
                {
                    AutoResizeColumnHeadersHeight(true /*fixedRowHeadersWidth*/, true /*fixedColumnsWidth*/);
                }
                else
                {
                    if (autoSizeRowsMode == DataGridViewAutoSizeRowsMode.None)
                    {
                        AutoResizeRowInternal(e.RowIndex, DataGridViewAutoSizeRowMode.AllCells, true /*fixedWidth*/, true /*internalAutosizing*/);
                    }
                    else
                    {
                        AutoResizeRowInternal(e.RowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), true /*fixedWidth*/, true /*internalAutosizing*/);
                    }
                }
            }
        }

        protected virtual void OnRowDividerHeightChanged(DataGridViewRowEventArgs e)
        {
            if (e.Row.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_RowDoesNotBelongToDataGridView, "e.Row");
            }

            OnRowGlobalAutoSize(e.Row.Index);

            if (Events[EVENT_DATAGRIDVIEWROWDIVIDERHEIGHTCHANGED] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnRowEnter(ref DataGridViewCell dataGridViewCell, int columnIndex, int rowIndex, bool canCreateNewRow, bool validationFailureOccurred)
        {
            Debug.Assert(columnIndex >= 0 && rowIndex >= 0);

            if (!validationFailureOccurred)
            {
                dataGridViewState1[DATAGRIDVIEWSTATE1_newRowEdited] = false;
            }
            if (rowIndex < Rows.Count &&
                columnIndex < Columns.Count)
            {
                bool calledAddNewOnTheDataConnection = false;
                if (!validationFailureOccurred && AllowUserToAddRowsInternal && newRowIndex == rowIndex)
                {
                    dataGridViewState1[DATAGRIDVIEWSTATE1_newRowEdited] = true;

                    if (canCreateNewRow)
                    {
                        DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs(Rows[newRowIndex]);
                        if (VirtualMode || DataSource != null)
                        {
                            if (dataConnection != null && dataConnection.InterestedInRowEvents)
                            {
                                dataConnection.OnNewRowNeeded();
                                calledAddNewOnTheDataConnection = true;
                            }
                            if (VirtualMode)
                            {
                                OnNewRowNeeded(dgvre);
                            }
                        }

                        // AllowUserToAddRowsInternal can become FALSE while adding a row.
                        // NOTE: we don't need to invalidate if AllowUserToAddRowsInternal changed to FALSE.
                        //
                        if (AllowUserToAddRowsInternal)
                        {
                            OnDefaultValuesNeeded(dgvre);
                            InvalidateRowPrivate(newRowIndex);
                        }
#if DEBUG
                        else
                        {
                            Debug.Assert(newRowIndex == -1, "newRowIndex and AllowUserToAddRowsInternal became out of sync");
                        }
#endif //
                    }
                }

                if (calledAddNewOnTheDataConnection && rowIndex > Rows.Count - 1)
                {
                    // Calling AddNew on the DataConnection can result in the entire list being wiped out.
                    //
                    rowIndex = Math.Min(rowIndex, Rows.Count - 1);
                }

                DataGridViewCellEventArgs dgvce = new DataGridViewCellEventArgs(columnIndex, rowIndex);
                OnRowEnter(dgvce);
                if (dataConnection != null &&
                    dataConnection.InterestedInRowEvents &&
                    !dataConnection.PositionChangingOutsideDataGridView &&
                    !dataConnection.ListWasReset &&
                    (!calledAddNewOnTheDataConnection || dataConnection.List.Count > 0))
                {
                    dataConnection.OnRowEnter(dgvce);
                }

                if (dataGridViewCell != null)
                {
                    if (IsInnerCellOutOfBounds(columnIndex, rowIndex))
                    {
                        dataGridViewCell = null;
                    }
                    else
                    {
                        Debug.Assert(rowIndex < Rows.Count && columnIndex < Columns.Count);
                        dataGridViewCell = Rows.SharedRow(rowIndex).Cells[columnIndex];
                    }
                }
            }
        }

        protected virtual void OnRowEnter(DataGridViewCellEventArgs e)
        {
            try
            {
                noDimensionChangeCount++;

                if (Events[EVENT_DATAGRIDVIEWROWENTER] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
                {
                    eh(this, e);
                    CorrectFocus(true /*onlyIfGridHasFocus*/);
                }
            }
            finally
            {
                noDimensionChangeCount--;
                Debug.Assert(noDimensionChangeCount >= 0);
            }
        }

        internal void OnRowErrorTextChanged(DataGridViewRow dataGridViewRow)
        {
            DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs(dataGridViewRow);
            OnRowErrorTextChanged(dgvre);
        }

        protected virtual void OnRowErrorTextChanged(DataGridViewRowEventArgs e)
        {
            if (e.Row.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_RowDoesNotBelongToDataGridView, "e.Row");
            }
            UpdateRowErrorText(e.Row.Index);

            if (Events[EVENT_DATAGRIDVIEWROWERRORTEXTCHANGED] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal string OnRowErrorTextNeeded(int rowIndex, string errorText)
        {
            Debug.Assert(rowIndex >= 0);
            DataGridViewRowErrorTextNeededEventArgs dgvretne = new DataGridViewRowErrorTextNeededEventArgs(rowIndex, errorText);
            OnRowErrorTextNeeded(dgvretne);
            return dgvretne.ErrorText;
        }

        protected virtual void OnRowErrorTextNeeded(DataGridViewRowErrorTextNeededEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWERRORTEXTNEEDED] is DataGridViewRowErrorTextNeededEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnRowGlobalAutoSize(int rowIndex)
        {
            DataGridViewElementStates rowState = Rows.GetRowState(rowIndex);
            if ((rowState & DataGridViewElementStates.Visible) == 0)
            {
                return;
            }

            InvalidateRowPrivate(rowIndex);

            if (noAutoSizeCount > 0)
            {
                return;
            }

            DataGridViewAutoSizeRowsModeInternal autoSizeRowsModeInternal = (DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode;
            bool autoSizeRow = false;
            bool rowDisplayed = (rowState & DataGridViewElementStates.Displayed) != 0;
            if (autoSizeRowsModeInternal != DataGridViewAutoSizeRowsModeInternal.None &&
                !((autoSizeRowsModeInternal & DataGridViewAutoSizeRowsModeInternal.DisplayedRows) != 0 && !rowDisplayed))
            {
                AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), false /*fixedWidth*/, true /*internalAutosizing*/);
                autoSizeRow = true;
            }

            // Auto size columms also if needed
            DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaFilter = DataGridViewAutoSizeColumnCriteriaInternal.AllRows;
            if (rowDisplayed)
            {
                autoSizeColumnCriteriaFilter |= DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows;
            }
            AutoResizeAllVisibleColumnsInternal(autoSizeColumnCriteriaFilter, true /*fixedHeight*/);

            bool fixedRowHeadersWidth = rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing ||
                                        rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.DisableResizing;
            // Auto size column headers
            if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
            {
                AutoResizeColumnHeadersHeight(fixedRowHeadersWidth, true /*fixedColumnsWidth*/);
            }

            // Auto size row headers
            if (!fixedRowHeadersWidth)
            {
                AutoResizeRowHeadersWidth(rowHeadersWidthSizeMode, true /*fixedColumnHeadersHeight*/, true /*fixedRowsHeight*/);
            }

            if (autoSizeRow)
            {
                // Second round of row autosizing
                AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), true /*fixedWidth*/, true /*internalAutosizing*/);
            }

            if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize && !fixedRowHeadersWidth)
            {
                // Second round of column headers autosizing
                AutoResizeColumnHeadersHeight(true /*fixedRowHeadersWidth*/, true /*fixedColumnsWidth*/);
            }
        }

        protected virtual void OnRowHeaderCellChanged(DataGridViewRowEventArgs e)
        {
            if (e.Row.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_RowDoesNotBelongToDataGridView, "e.Row");
            }

            OnRowHeaderGlobalAutoSize(e.Row.Index);

            if (Events[EVENT_DATAGRIDVIEWROWHEADERCELLCHANGED] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnRowHeaderGlobalAutoSize(int rowIndex)
        {
            if (!RowHeadersVisible)
            {
                return;
            }

            InvalidateCellPrivate(-1, rowIndex);

            if (noAutoSizeCount > 0)
            {
                return;
            }

            bool rowDisplayed = false;
            if (rowIndex != -1)
            {
                rowDisplayed = (Rows.GetRowState(rowIndex) & DataGridViewElementStates.Displayed) != 0;
            }

            bool fixedColumnHeadersHeight = rowIndex != -1 || ColumnHeadersHeightSizeMode != DataGridViewColumnHeadersHeightSizeMode.AutoSize;
            bool fixedRowHeight = rowIndex == -1 ||
                                  ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.Header) == 0) ||
                                  ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.DisplayedRows) != 0 && rowIndex != -1 && !rowDisplayed);

            bool autoSizeRowHeaders = false;
            if (rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.AutoSizeToAllHeaders ||
                (rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.AutoSizeToDisplayedHeaders && rowIndex != -1 && rowDisplayed) ||
                (rowHeadersWidthSizeMode != DataGridViewRowHeadersWidthSizeMode.EnableResizing && rowHeadersWidthSizeMode != DataGridViewRowHeadersWidthSizeMode.DisableResizing && rowIndex == -1) ||
                (rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.AutoSizeToFirstHeader && rowIndex != -1 && rowIndex == Rows.GetFirstRow(DataGridViewElementStates.Visible)))
            {
                AutoResizeRowHeadersWidth(rowIndex,
                                          rowHeadersWidthSizeMode,
                                          fixedColumnHeadersHeight,
                                          fixedRowHeight);
                autoSizeRowHeaders = true;
            }
            if (!fixedColumnHeadersHeight)
            {
                AutoResizeColumnHeadersHeight(-1, true /*fixedRowHeadersWidth*/, true /*fixedColumnsWidth*/);
            }
            if (!fixedRowHeight)
            {
                AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), true /*fixedWidth*/, true /*internalAutosizing*/);
            }

            if (autoSizeRowHeaders && (!fixedColumnHeadersHeight || !fixedRowHeight))
            {
                // Second round of row headers autosizing
                AutoResizeRowHeadersWidth(rowIndex,
                                          rowHeadersWidthSizeMode,
                                          true /*fixedColumnHeadersHeight*/,
                                          true /*fixedRowHeight*/);
            }
        }

        protected virtual void OnRowHeaderMouseClick(DataGridViewCellMouseEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWHEADERMOUSECLICK] is DataGridViewCellMouseEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnRowHeaderMouseDoubleClick(DataGridViewCellMouseEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWHEADERMOUSEDOUBLECLICK] is DataGridViewCellMouseEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnRowHeaderMouseDown(HitTestInfo hti, bool isShiftDown, bool isControlDown)
        {
            Debug.Assert(hti.Type == DataGridViewHitTestType.RowHeader);
            noSelectionChangeCount++;
            try
            {
                switch (SelectionMode)
                {
                    case DataGridViewSelectionMode.CellSelect:
                    case DataGridViewSelectionMode.FullColumnSelect:
                    case DataGridViewSelectionMode.ColumnHeaderSelect:
                        break;

                    case DataGridViewSelectionMode.FullRowSelect:
                    case DataGridViewSelectionMode.RowHeaderSelect:
                        {
                            bool select = true;
                            if (isControlDown &&
                                ((Rows.GetRowState(hti.row) & DataGridViewElementStates.Selected) != 0))
                            {
                                select = false;
                            }
                            if (select)
                            {
                                DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
                                if (dataGridViewColumn != null && hti.row != ptCurrentCell.Y)
                                {
                                    int oldCurrentCellX = ptCurrentCell.X;
                                    int oldCurrentCellY = ptCurrentCell.Y;
                                    // Make sure we will be able to scroll into view
                                    if (!EndEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit | DataGridViewDataErrorContexts.CurrentCellChange,
                                        DataGridViewValidateCellInternal.Always /*validateCell*/,
                                        true /*fireCellLeave*/,
                                        true /*fireCellEnter*/,
                                        hti.row != ptCurrentCell.Y /*fireRowLeave*/,
                                        hti.row != ptCurrentCell.Y /*fireRowEnter*/,
                                        false /*fireLeave*/,
                                        EditMode != DataGridViewEditMode.EditOnEnter /*keepFocus*/,
                                        true /*resetCurrentCell*/,
                                        false /*resetAnchorCell*/))
                                    {
                                        // Just cancel operation silently instead of throwing InvalidOperationException
                                        return;
                                    }
                                    if (oldCurrentCellY != -1)
                                    {
                                        DataGridViewCell dataGridViewCellTmp = null;
                                        if (IsInnerCellOutOfBounds(oldCurrentCellX, oldCurrentCellY))
                                        {
                                            return;
                                        }
                                        if (OnRowValidating(ref dataGridViewCellTmp, oldCurrentCellX, oldCurrentCellY))
                                        {
                                            // Row validation was cancelled
                                            if (IsInnerCellOutOfBounds(oldCurrentCellX, oldCurrentCellY))
                                            {
                                                return;
                                            }
                                            OnRowEnter(ref dataGridViewCellTmp, oldCurrentCellX, oldCurrentCellY, true /*canCreateNewRow*/, true /*validationFailureOccurred*/);
                                            if (IsInnerCellOutOfBounds(oldCurrentCellX, oldCurrentCellY))
                                            {
                                                return;
                                            }
                                            OnCellEnter(ref dataGridViewCellTmp, oldCurrentCellX, oldCurrentCellY);
                                            return;
                                        }
                                        if (IsInnerCellOutOfBounds(oldCurrentCellX, oldCurrentCellY))
                                        {
                                            return;
                                        }
                                        OnRowValidated(ref dataGridViewCellTmp, oldCurrentCellX, oldCurrentCellY);

                                        // Row validation was not cancelled, but operation needs to be re-evaluated.
                                        if (hti.row >= Rows.Count)
                                        {
                                            int lastVisibleRowIndex = Rows.GetLastRow(DataGridViewElementStates.Visible);
                                            if (ptCurrentCell.X == -1 && lastVisibleRowIndex != -1)
                                            {
                                                // CurrentCell was reset because commit deleted row(s).
                                                // Since the user wants to select a row, we don't want to
                                                // end up with no CurrentCell. We pick the last visible
                                                // row in the grid which may be the 'new row'.
                                                if (IsColumnOutOfBounds(oldCurrentCellX))
                                                {
                                                    return;
                                                }
                                                bool success = SetAndSelectCurrentCellAddress(oldCurrentCellX,
                                                                                              lastVisibleRowIndex,
                                                                                              true,
                                                                                              false,
                                                                                              false,
                                                                                              false /*clearSelection*/,
                                                                                              false /*forceCurrentCellSelection*/);
                                                Debug.Assert(success);
                                            }
                                            return;
                                        }
                                        else if ((Rows.GetRowState(hti.row) & DataGridViewElementStates.Visible) == 0)
                                        {
                                            return;
                                        }
                                    }
                                }
                                bool selectRowRange = false;
                                trackRow = hti.row;
                                trackRowEdge = -1;
                                if (MultiSelect &&
                                    isShiftDown &&
                                    ptAnchorCell.Y > -1 &&
                                    (Rows.GetRowState(ptAnchorCell.Y) & DataGridViewElementStates.Selected) != 0)
                                {
                                    selectRowRange = true;
                                }
                                if (!MultiSelect || !isControlDown || isShiftDown)
                                {
                                    Debug.Assert(MultiSelect || selectedBandIndexes.Count <= 1);
                                    int bandIndex = 0;
                                    bool switchedToBulkPaint = false;
                                    if (selectedBandIndexes.Count > DATAGRIDVIEW_bulkPaintThreshold)
                                    {
                                        inBulkPaintCount++;
                                        switchedToBulkPaint = true;
                                    }
                                    try
                                    {
                                        while (bandIndex < selectedBandIndexes.Count)
                                        {
                                            if (selectedBandIndexes[bandIndex] != hti.row)
                                            {
                                                // deselect currently selected row
                                                SetSelectedRowCore(selectedBandIndexes[bandIndex], false);
                                            }
                                            else
                                            {
                                                bandIndex++;
                                            }
                                        }

                                        if (SelectionMode == DataGridViewSelectionMode.RowHeaderSelect)
                                        {
                                            RemoveIndividuallySelectedCells();
                                        }
                                        else
                                        {
                                            Debug.Assert(individualSelectedCells.Count == 0);
                                        }
                                    }
                                    finally
                                    {
                                        if (switchedToBulkPaint)
                                        {
                                            ExitBulkPaint(-1, -1);
                                        }
                                    }
                                }
                                if (MultiSelect && dataGridViewOper[DATAGRIDVIEWOPER_trackMouseMoves])
                                {
                                    dataGridViewOper[DATAGRIDVIEWOPER_trackRowSelect] = true;
                                }
                                if (selectRowRange)
                                {
                                    if (hti.row >= ptAnchorCell.Y)
                                    {
                                        SelectRowRange(ptAnchorCell.Y, hti.row, true);
                                    }
                                    else
                                    {
                                        SelectRowRange(hti.row, ptAnchorCell.Y, true);
                                    }
                                }
                                else if ((Rows.GetRowState(hti.row) & DataGridViewElementStates.Selected) == 0)
                                {
                                    Debug.Assert(selectedBandIndexes.Contains(hti.row) ==
                                                 ((Rows.GetRowState(hti.row) & DataGridViewElementStates.Selected) != 0));
                                    SetSelectedRowCore(hti.row, true);
                                }
                                if (dataGridViewColumn != null)
                                {
                                    if (hti.row != ptCurrentCell.Y)
                                    {
                                        if (IsInnerCellOutOfBounds(dataGridViewColumn.Index, hti.row))
                                        {
                                            return;
                                        }
                                        // set current cell to the left most visible cell in the row
                                        bool success = ScrollIntoView(dataGridViewColumn.Index, hti.row, false);
                                        Debug.Assert(success);
                                        if (IsInnerCellOutOfBounds(dataGridViewColumn.Index, hti.row))
                                        {
                                            return;
                                        }
                                        success = SetCurrentCellAddressCore(dataGridViewColumn.Index, hti.row, !selectRowRange, false, true);
                                        Debug.Assert(success);
                                    }
                                    else if (-1 != ptCurrentCell.Y)
                                    {
                                        // Potentially have to give focus back to the current edited cell.
                                        bool success = SetCurrentCellAddressCore(ptCurrentCell.X, ptCurrentCell.Y,
                                                                                false /*setAnchorCellAddress*/,
                                                                                false /*validateCurrentCell*/,
                                                                                false /*throughMouseClick*/);
                                        Debug.Assert(success);
                                    }
                                }
                                else
                                {
                                    Debug.Assert(CurrentCellAddress == new Point(-1, -1));
                                }
                            }
                            else
                            {
                                Debug.Assert(selectedBandIndexes.Contains(hti.row));
                                SetSelectedRowCore(hti.row, false);
                            }
                            break;
                        }
                }
            }
            finally
            {
                NoSelectionChangeCount--;
            }
        }

        protected virtual void OnRowHeadersBorderStyleChanged(EventArgs e)
        {
            PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, false /*repositionEditingControl*/);
            Invalidate();

            if (Events[EVENT_DATAGRIDVIEWROWHEADERSBORDERSTYLECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnRowHeadersDefaultCellStyleChanged(EventArgs e)
        {
            if (RowHeadersVisible)
            {
                Invalidate(Rectangle.Union(layout.TopLeftHeader, layout.RowHeaders));

                if (!(e is DataGridViewCellStyleChangedEventArgs dgvcsce) || dgvcsce.ChangeAffectsPreferredSize)
                {
                    OnRowHeadersGlobalAutoSize(false /*expandingRows*/);
                    if (editingControl != null)
                    {
                        PositionEditingControl(true /*setLocation*/, true /*setSize*/, false /*setFocus*/);
                    }
                }
            }

            if (Events[EVENT_DATAGRIDVIEWROWHEADERSDEFAULTCELLSTYLECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnRowHeadersGlobalAutoSize(bool expandingRows)
        {
            if (noAutoSizeCount > 0)
            {
                return;
            }

            bool fixedRowsHeight = (((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.Header) == 0 ||
                                   !RowHeadersVisible;
            bool autoSizeRowHeaders = rowHeadersWidthSizeMode != DataGridViewRowHeadersWidthSizeMode.EnableResizing &&
                                      rowHeadersWidthSizeMode != DataGridViewRowHeadersWidthSizeMode.DisableResizing;
            if (autoSizeRowHeaders)
            {
                AutoResizeRowHeadersWidth(rowHeadersWidthSizeMode, true /*fixedColumnHeadersHeight*/, fixedRowsHeight);
            }

            if (!fixedRowsHeight)
            {
                if (expandingRows)
                {
                    AdjustExpandingRows(-1, true /*fixedWidth*/);
                }
                else
                {
                    AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
                }

                if (autoSizeRowHeaders)
                {
                    // Second round of row headers autosizing
                    AutoResizeRowHeadersWidth(rowHeadersWidthSizeMode, true /*fixedColumnHeadersHeight*/, true /*fixedRowsHeight*/);
                }
            }
        }

        protected virtual void OnRowHeadersWidthChanged(EventArgs e)
        {
            if (RowHeadersVisible)
            {
                if (editingControl != null)
                {
                    PositionEditingControl(true, false, false);
                }

                if (IsHandleCreated)
                {
                    UpdateMouseEnteredCell(null /*HitTestInfo*/, null /*MouseEventArgs*/);
                }

                OnRowHeadersGlobalAutoSize(false /*expandingRows*/);
            }

            if (Events[EVENT_DATAGRIDVIEWROWHEADERSWIDTHCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnRowHeadersWidthSizeModeChanged(DataGridViewAutoSizeModeEventArgs e)
        {
            if (rowHeadersWidthSizeMode != DataGridViewRowHeadersWidthSizeMode.EnableResizing &&
                rowHeadersWidthSizeMode != DataGridViewRowHeadersWidthSizeMode.DisableResizing)
            {
                if (!e.PreviousModeAutoSized)
                {
                    // Save current row headers width for later reuse
                    cachedRowHeadersWidth = RowHeadersWidth;
                }
                AutoResizeRowHeadersWidth(rowHeadersWidthSizeMode,
                                          true /*fixedColumnHeadersHeight*/,
                                          true /*fixedRowsHeight*/);
            }
            else if (e.PreviousModeAutoSized)
            {
                RowHeadersWidth = cachedRowHeadersWidth;
            }

            if (Events[EVENT_DATAGRIDVIEWROWHEADERSWIDTHSIZEMODECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnRowHeightChanged(DataGridViewRowEventArgs e)
        {
            if (e.Row.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_RowDoesNotBelongToDataGridView, "e.Row");
            }
            UpdateRowHeightInfoPrivate(e.Row.Index, false, false /*invalidInAdjustFillingColumns*/);

            if (Events[EVENT_DATAGRIDVIEWROWHEIGHTCHANGED] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            OnRowGlobalAutoSize(e.Row.Index);
        }

        internal DataGridViewRowHeightInfoNeededEventArgs OnRowHeightInfoNeeded(int rowIndex, int height, int minimumHeight)
        {
            DataGridViewRowHeightInfoNeededEventArgs dgvrhine = RowHeightInfoNeededEventArgs;
            dgvrhine.SetProperties(rowIndex, height, minimumHeight);
            OnRowHeightInfoNeeded(dgvrhine);
            return dgvrhine;
        }

        protected virtual void OnRowHeightInfoNeeded(DataGridViewRowHeightInfoNeededEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWHEIGHTINFONEEDED] is DataGridViewRowHeightInfoNeededEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private bool OnRowHeightInfoPushed(int rowIndex, int height, int minimumHeight)
        {
            Debug.Assert(rowIndex != -1);
            Debug.Assert(autoSizeRowsMode == DataGridViewAutoSizeRowsMode.None);
            if (VirtualMode || DataSource != null)
            {
                DataGridViewRowHeightInfoPushedEventArgs dgvrhipe = new DataGridViewRowHeightInfoPushedEventArgs(rowIndex, height, minimumHeight);
                OnRowHeightInfoPushed(dgvrhipe);
                if (dgvrhipe.Handled)
                {
                    UpdateRowHeightInfoPrivate(rowIndex, false, true /*invalidInAdjustFillingColumns*/);
                    return true;
                }
            }
            return false;
        }

        protected virtual void OnRowHeightInfoPushed(DataGridViewRowHeightInfoPushedEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWHEIGHTINFOPUSHED] is DataGridViewRowHeightInfoPushedEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnRowLeave(ref DataGridViewCell dataGridViewCell, int columnIndex, int rowIndex)
        {
            Debug.Assert(columnIndex >= 0 && rowIndex >= 0);
            if (rowIndex < Rows.Count && columnIndex < Columns.Count)
            {
                DataGridViewCellEventArgs dgvce = new DataGridViewCellEventArgs(columnIndex, rowIndex);
                OnRowLeave(dgvce);
                if (dataGridViewCell != null)
                {
                    if (IsInnerCellOutOfBounds(columnIndex, rowIndex))
                    {
                        dataGridViewCell = null;
                    }
                    else
                    {
                        Debug.Assert(rowIndex < Rows.Count && columnIndex < Columns.Count);
                        dataGridViewCell = Rows.SharedRow(rowIndex).Cells[columnIndex];
                    }
                }
            }
        }

        protected virtual void OnRowLeave(DataGridViewCellEventArgs e)
        {
            try
            {
                noDimensionChangeCount++;

                if (Events[EVENT_DATAGRIDVIEWROWLEAVE] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
                {
                    eh(this, e);
                    CorrectFocus(true /*onlyIfGridHasFocus*/);
                }
            }
            finally
            {
                noDimensionChangeCount--;
                Debug.Assert(noDimensionChangeCount >= 0);
            }
        }

        protected virtual void OnRowMinimumHeightChanged(DataGridViewRowEventArgs e)
        {
            if (e.Row.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_RowDoesNotBelongToDataGridView, "e.Row");
            }
            if (Events[EVENT_DATAGRIDVIEWROWMINIMUMHEIGHTCHANGED] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected internal virtual void OnRowPostPaint(DataGridViewRowPostPaintEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWPOSTPAINT] is DataGridViewRowPostPaintEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected internal virtual void OnRowPrePaint(DataGridViewRowPrePaintEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWPREPAINT] is DataGridViewRowPrePaintEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnRowsAddedInternal(int rowIndex, int rowCount)
        {
            OnRowsAdded(new DataGridViewRowsAddedEventArgs(rowIndex, rowCount));
        }

        protected virtual void OnRowsAdded(DataGridViewRowsAddedEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWSADDED] is DataGridViewRowsAddedEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnRowsDefaultCellStyleChanged(EventArgs e)
        {
            if (e is DataGridViewCellStyleChangedEventArgs dgvcsce && !dgvcsce.ChangeAffectsPreferredSize)
            {
                InvalidateData();
            }
            else
            {
                OnRowsGlobalAutoSize();
                if (editingControl != null)
                {
                    PositionEditingControl(true /*setLocation*/, true /*setSize*/, false /*setFocus*/);
                }
            }

            if (Events[EVENT_DATAGRIDVIEWROWSDEFAULTCELLSTYLECHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnRowSelectMouseMove(HitTestInfo hti)
        {
            Debug.Assert(hti.row >= 0);
            Debug.Assert(MultiSelect);

            if (ptCurrentCell.Y != -1 &&
                hti.row != ptCurrentCell.Y &&
                !CommitEditForOperation(ptCurrentCell.X, hti.row, true))
            {
                // Return silently if validating/commit/abort failed
                return;
            }
            if (IsRowOutOfBounds(hti.row))
            {
                return;
            }

            noSelectionChangeCount++;
            try
            {
                if (trackRowEdge >= trackRow && hti.row > trackRowEdge && trackRowEdge >= 0)
                {
                    SelectRowRange(Rows.GetNextRow(trackRowEdge, DataGridViewElementStates.Visible),
                        hti.row, true);
                    trackRowEdge = hti.row;
                }
                else if (trackRowEdge > trackRow && hti.row < trackRowEdge && hti.row >= trackRow && trackRowEdge >= 0)
                {
                    SelectRowRange(Rows.GetNextRow(hti.row, DataGridViewElementStates.Visible),
                        trackRowEdge, false);
                    trackRowEdge = hti.row;
                }
                else if (hti.row > trackRow && trackRowEdge == -1)
                {
                    SelectRowRange(Rows.GetNextRow(trackRow, DataGridViewElementStates.Visible),
                        hti.row, true);
                    trackRowEdge = hti.row;
                }
                else if (trackRowEdge <= trackRow && hti.row < trackRowEdge && trackRowEdge >= 0)
                {
                    SelectRowRange(hti.row,
                        Rows.GetPreviousRow(trackRowEdge, DataGridViewElementStates.Visible),
                        true);
                    trackRowEdge = hti.row;
                }
                else if (trackRowEdge < trackRow && hti.row > trackRowEdge && hti.row <= trackRow && trackRowEdge >= 0)
                {
                    SelectRowRange(trackRowEdge,
                        Rows.GetPreviousRow(hti.row, DataGridViewElementStates.Visible),
                        false);
                    trackRowEdge = hti.row;
                }
                else if (hti.row < trackRow && trackRowEdge == -1)
                {
                    SelectRowRange(hti.row,
                        Rows.GetPreviousRow(trackRow, DataGridViewElementStates.Visible),
                        true);
                    trackRowEdge = hti.row;
                }
                else if (trackRowEdge > trackRow && hti.row < trackRow)
                {
                    SelectRowRange(Rows.GetNextRow(trackRow, DataGridViewElementStates.Visible),
                        trackRowEdge, false);
                    SelectRowRange(hti.row,
                        Rows.GetPreviousRow(trackRow, DataGridViewElementStates.Visible),
                        true);
                    trackRowEdge = hti.row;
                }
                else if (hti.row > trackRow && trackRowEdge < trackRow && trackRowEdge >= 0)
                {
                    SelectRowRange(trackRowEdge,
                        Rows.GetPreviousRow(trackRow, DataGridViewElementStates.Visible),
                        false);
                    SelectRowRange(Rows.GetNextRow(trackRow, DataGridViewElementStates.Visible),
                        hti.row, true);
                    trackRowEdge = hti.row;
                }
            }
            finally
            {
                NoSelectionChangeCount--;
            }

            if (ptCurrentCell.Y != -1 && hti.row != ptCurrentCell.Y)
            {
                if (IsRowOutOfBounds(hti.row))
                {
                    return;
                }
                bool success = SetCurrentCellAddressCore(ptCurrentCell.X,
                    hti.row,
                    false /*setAnchorCellAddress*/,
                    false /*validateCurrentCell*/,
                    false /*throughMouseClick*/);
                Debug.Assert(success);
            }
        }

        private void OnRowsGlobalAutoSize()
        {
            InvalidateData();

            if (noAutoSizeCount > 0)
            {
                return;
            }

            // Autosize rows if needed
            if ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0)
            {
                AdjustShrinkingRows(autoSizeRowsMode, false /*fixedWidth*/, true /*internalAutosizing*/);
            }

            // Auto size columms also if needed
            AutoResizeAllVisibleColumnsInternal(DataGridViewAutoSizeColumnCriteriaInternal.AllRows |
                                                DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows,
                                                true /*fixedHeight*/);

            bool fixedRowHeadersWidth = rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.EnableResizing ||
                                        rowHeadersWidthSizeMode == DataGridViewRowHeadersWidthSizeMode.DisableResizing;
            // Auto size column headers
            if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
            {
                AutoResizeColumnHeadersHeight(fixedRowHeadersWidth, true /*fixedColumnsWidth*/);
            }

            // Auto size row headers
            if (!fixedRowHeadersWidth)
            {
                AutoResizeRowHeadersWidth(rowHeadersWidthSizeMode, true /*fixedColumnHeadersHeight*/, true /*fixedRowsHeight*/);
            }

            // Second round of rows autosizing
            if ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0)
            {
                AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
            }

            // Second round of column headers autosizing
            if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize && !fixedRowHeadersWidth)
            {
                AutoResizeColumnHeadersHeight(true /*fixedRowHeadersWidth*/, true /*fixedColumnsWidth*/);
            }
        }

        internal void OnRowsRemovedInternal(int rowIndex, int rowCount)
        {
            OnRowsRemoved(new DataGridViewRowsRemovedEventArgs(rowIndex, rowCount));
        }

        protected virtual void OnRowsRemoved(DataGridViewRowsRemovedEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWROWSREMOVED] is DataGridViewRowsRemovedEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnRowStateChanged(int rowIndex, DataGridViewRowStateChangedEventArgs e)
        {
            Debug.Assert(rowIndex >= -1);

            // If row.Frozen changed, we need to update the vertical scroll bar
            // A hidden row may become visible and vice-versa, we'd better repaint the whole control
            DataGridViewRow dataGridViewRow = e.Row;
            DataGridViewElementStates newState = DataGridViewElementStates.None;
            bool rowVisible = false;
            if (rowIndex >= 0)
            {
                newState = Rows.GetRowState(rowIndex);
                rowVisible = ((newState & DataGridViewElementStates.Visible) != 0);
            }
            switch (e.StateChanged)
            {
                // At this point we assume that only the Selected state has an influence on the rendering of the row.
                // If there is a customer scenario where another state has an influence, the dev will have to invalidate the row by hand.
                // case DataGridViewElementStates.ReadOnly:
                // case DataGridViewElementStates.Resizable:

                case DataGridViewElementStates.Selected:
                    if (rowVisible && inBulkPaintCount == 0)
                    {
                        InvalidateRowPrivate(rowIndex);
                    }
                    break;

                case DataGridViewElementStates.Frozen:
                    if (rowVisible)
                    {
                        if ((newState & DataGridViewElementStates.Frozen) == 0)
                        {
                            // row was unfrozen - make it the first visible scrolling row if there is room
                            FirstVisibleScrollingRowTempted(rowIndex);
                        }
                        PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, true /*repositionEditingControl*/);
                        Invalidate();
                    }
                    break;

                case DataGridViewElementStates.Visible:
                    {
                        if (!rowVisible && (newState & DataGridViewElementStates.Displayed) != 0)
                        {
                            // Displayed row becomes invisible. Turns off the Displayed state.
                            Rows.SetRowState(rowIndex, DataGridViewElementStates.Displayed, false);
                        }

                        PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, true /*invalidInAdjustFillingColumns*/, true /*repositionEditingControl*/);
                        Invalidate();

                        bool rowDisplayed = (Rows.GetRowState(rowIndex) & DataGridViewElementStates.Displayed) != 0;
                        DataGridViewAutoSizeRowsModeInternal autoSizeRowsModeInternal = (DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode;

                        bool autoSizeRow = false;

                        if (autoSizeRowsModeInternal != DataGridViewAutoSizeRowsModeInternal.None)
                        {
                            int height = dataGridViewRow.ThicknessInternal;
                            if (rowVisible)
                            {
                                // Cache row's height before potential autosizing occurs
                                // Valid operation even for shared rows
                                dataGridViewRow.CachedThickness = height;
                                if (!((autoSizeRowsModeInternal & DataGridViewAutoSizeRowsModeInternal.DisplayedRows) != 0 && !rowDisplayed))
                                {
                                    AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), false /*fixedWidth*/, true /*internalAutosizing*/);
                                    autoSizeRow = true;
                                }
                            }
                            else if (height != dataGridViewRow.CachedThickness)
                            {
                                // Rows that are made invisible in the collection take their non-autosized height
                                // Not calling OnRowHeightInfoPushed(...) because rows are autosized
                                // Make sure the affected row is unshared
                                if (dataGridViewRow.Index == -1)
                                {
                                    dataGridViewRow = Rows[rowIndex];
                                }
                                dataGridViewRow.ThicknessInternal = Math.Max(dataGridViewRow.MinimumHeight, dataGridViewRow.CachedThickness);
                            }
                        }

                        // Auto size columms also if needed
                        DataGridViewAutoSizeColumnCriteriaInternal autoSizeColumnCriteriaFilter = DataGridViewAutoSizeColumnCriteriaInternal.AllRows;
                        if (rowDisplayed)
                        {
                            autoSizeColumnCriteriaFilter |= DataGridViewAutoSizeColumnCriteriaInternal.DisplayedRows;
                        }
                        if (rowVisible && Rows.GetRowCount(DataGridViewElementStates.Visible) > 1)
                        {
                            // Columns can only expand, and not collapse.
                            AdjustExpandingColumns(autoSizeColumnCriteriaFilter, rowIndex);
                        }
                        else
                        {
                            AutoResizeAllVisibleColumnsInternal(autoSizeColumnCriteriaFilter, true /*fixedHeight*/);
                        }

                        if (autoSizeRow)
                        {
                            // Second round of row autosizing
                            AutoResizeRowInternal(rowIndex, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), true /*fixedWidth*/, true /*internalAutosizing*/);
                        }
                        break;
                    }
            }

            if (Events[EVENT_DATAGRIDVIEWROWSTATECHANGED] is DataGridViewRowStateChangedEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }

            if (e.StateChanged == DataGridViewElementStates.ReadOnly &&
                rowIndex == ptCurrentCell.Y &&
                !dataGridViewOper[DATAGRIDVIEWOPER_inReadOnlyChange])
            {
                VerifyImeRestrictedModeChanged();

                if ((newState & DataGridViewElementStates.ReadOnly) == 0 &&
                    !ReadOnly &&
                    !Columns[ptCurrentCell.X].ReadOnly &&
                    ColumnEditable(ptCurrentCell.X) &&
                    !IsCurrentCellInEditMode &&
                    (EditMode == DataGridViewEditMode.EditOnEnter ||
                    (EditMode != DataGridViewEditMode.EditProgrammatically && CurrentCellInternal.EditType == null)))
                {
                    // Current row becomes read/write. Enter editing mode.
                    BeginEditInternal(true /*selectAll*/);
                }
            }
        }

        internal void OnRowUnshared(DataGridViewRow dataGridViewRow)
        {
            if (-1 != ptCurrentCell.X && dataGridViewRow.Index == ptCurrentCell.Y && editingControl != null)
            {
                CurrentCellInternal.CacheEditingControl();
            }
            DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs(dataGridViewRow);
            OnRowUnshared(dgvre);
        }

        protected virtual void OnRowUnshared(DataGridViewRowEventArgs e)
        {
            if (e.Row.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_RowDoesNotBelongToDataGridView, "e.Row");
            }
            if (Events[EVENT_DATAGRIDVIEWROWUNSHARED] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private bool OnRowValidating(ref DataGridViewCell dataGridViewCell, int columnIndex, int rowIndex)
        {
            DataGridViewCellCancelEventArgs dgvcce = new DataGridViewCellCancelEventArgs(columnIndex, rowIndex);
            OnRowValidating(dgvcce);
            if (!dgvcce.Cancel)
            {
                if (dataConnection != null &&
                    dataConnection.InterestedInRowEvents &&
                    !dataConnection.PositionChangingOutsideDataGridView &&
                    !dataConnection.ListWasReset)
                {
                    dataConnection.OnRowValidating(dgvcce);
                }
            }
            if (dataGridViewCell != null && rowIndex < Rows.Count && columnIndex < Columns.Count)
            {
                dataGridViewCell = Rows.SharedRow(rowIndex).Cells[columnIndex];
            }
            return dgvcce.Cancel;
        }

        protected virtual void OnRowValidating(DataGridViewCellCancelEventArgs e)
        {
            try
            {
                noDimensionChangeCount++;

                if (Events[EVENT_DATAGRIDVIEWROWVALIDATING] is DataGridViewCellCancelEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
                {
                    eh(this, e);
                    CorrectFocus(true /*onlyIfGridHasFocus*/);
                }
            }
            finally
            {
                noDimensionChangeCount--;
                Debug.Assert(noDimensionChangeCount >= 0);
            }
        }

        private void OnRowValidated(ref DataGridViewCell dataGridViewCell, int columnIndex, int rowIndex)
        {
            IsCurrentRowDirtyInternal = false;
            dataGridViewState1[DATAGRIDVIEWSTATE1_newRowCreatedByEditing] = false;
            if (rowIndex == newRowIndex)
            {
                // Stop displaying the default cell values on the 'new row'.
                InvalidateRowPrivate(rowIndex);
            }

            DataGridViewCellEventArgs dgvce = new DataGridViewCellEventArgs(columnIndex, rowIndex);
            OnRowValidated(dgvce);
            if (dataGridViewCell != null)
            {
                if (IsInnerCellOutOfBounds(columnIndex, rowIndex))
                {
                    dataGridViewCell = null;
                }
                else
                {
                    Debug.Assert(rowIndex < Rows.Count && columnIndex < Columns.Count);
                    dataGridViewCell = Rows.SharedRow(rowIndex).Cells[columnIndex];
                }
            }
        }

        protected virtual void OnRowValidated(DataGridViewCellEventArgs e)
        {
            try
            {
                noDimensionChangeCount++;

                if (Events[EVENT_DATAGRIDVIEWROWVALIDATED] is DataGridViewCellEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
                {
                    eh(this, e);
                    CorrectFocus(true /*onlyIfGridHasFocus*/);
                }
            }
            finally
            {
                noDimensionChangeCount--;
                Debug.Assert(noDimensionChangeCount >= 0);
            }
        }

        private void RefreshByCurrentPos(int oldValue, int newValue)
        {
            Point pt = PointToScreen(Location);
            int step = newValue - oldValue;

            // horizontal scrool left
            if (pt.X < 0 && step < 0)
            {
                Invalidate(new Rectangle(new Point(-pt.X, ColumnHeadersHeight),
                                         new Size(-step, ClientSize.Height)));
            }

            pt.X += Width;
            pt.Y += Height;
            Rectangle rect = Screen.GetBounds(pt);

            // horizontal scrool right
            if (pt.X > rect.Right && step > 0)
            {
                Invalidate(new Rectangle(new Point(ClientSize.Width - (pt.X - rect.Right) - step, ColumnHeadersHeight),
                                         new Size(step, ClientSize.Height)));
            }

            // vertical scrool up
            if (pt.Y < 0 && step < 0)
            {
                Invalidate(new Rectangle(new Point(0, -pt.Y),
                                         new Size(-step, ClientSize.Width)));
            }

            // vertical scrool down
            if (pt.Y > rect.Bottom && step > 0)
            {
                Invalidate(new Rectangle(new Point(0, ColumnHeadersHeight),
                                         new Size(ClientSize.Width, ClientSize.Height - (pt.Y - rect.Bottom) - step)));
            }
        }

        private void OnScroll(ScrollEventType scrollEventType, int oldValue, int newValue, ScrollOrientation orientation)
        {
            ScrollEventArgs se = new ScrollEventArgs(scrollEventType, oldValue, newValue, orientation);
            OnScroll(se);
            RefreshByCurrentPos(oldValue, newValue);
            if (Focused && IsGridFocusRectangleEnabled())
            {
                InvalidateGridFocusOnScroll(newValue - oldValue, orientation);
            }
            if (ScrollOrientation.VerticalScroll == orientation)
            {
                if (se.NewValue != newValue)
                {
                    try
                    {
                        dataGridViewState2[DATAGRIDVIEWSTATE2_stopRaisingVerticalScroll] = true;
                        int rowIndex = Rows.GetFirstRow(DataGridViewElementStates.Visible, DataGridViewElementStates.Frozen);
                        int rowIndexPrevious = rowIndex;
                        newValue = se.NewValue;
                        while (rowIndex != -1 && newValue > 0)
                        {
                            rowIndexPrevious = rowIndex;
                            rowIndex = Rows.GetNextRow(rowIndex, DataGridViewElementStates.Visible);
                            newValue--;
                        }
                        if (rowIndex != -1)
                        {
                            rowIndexPrevious = rowIndex;
                        }
                        if (rowIndexPrevious != -1)
                        {
                            FirstDisplayedScrollingRowIndex = rowIndexPrevious;
                        }
                    }
                    finally
                    {
                        dataGridViewState2[DATAGRIDVIEWSTATE2_stopRaisingVerticalScroll] = false;
                    }
                }
            }
            else
            {
                if (se.NewValue != newValue)
                {
                    try
                    {
                        dataGridViewState2[DATAGRIDVIEWSTATE2_stopRaisingHorizontalScroll] = true;
                        HorizontalOffset = se.NewValue;
                    }
                    finally
                    {
                        dataGridViewState2[DATAGRIDVIEWSTATE2_stopRaisingHorizontalScroll] = false;
                    }
                }
            }
        }

        protected virtual void OnScroll(ScrollEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWSCROLL] is ScrollEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnSelectionChanged(EventArgs e)
        {
            dataGridViewState2[DATAGRIDVIEWSTATE2_raiseSelectionChanged] = false;

            if (Events[EVENT_DATAGRIDVIEWSELECTIONCHANGED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal bool OnSortCompare(DataGridViewColumn dataGridViewSortedColumn, object value1, object value2, int rowIndex1, int rowIndex2, out int sortResult)
        {
            DataGridViewSortCompareEventArgs dgvsce = new DataGridViewSortCompareEventArgs(dataGridViewSortedColumn, value1, value2, rowIndex1, rowIndex2);
            OnSortCompare(dgvsce);
            sortResult = dgvsce.SortResult;
            return dgvsce.Handled;
        }

        protected virtual void OnSortCompare(DataGridViewSortCompareEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWSORTCOMPARE] is DataGridViewSortCompareEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnSorted(EventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWSORTED] is EventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        internal void OnSortGlyphDirectionChanged(DataGridViewColumnHeaderCell dataGridViewColumnHeaderCell)
        {
            Debug.Assert(dataGridViewColumnHeaderCell != null);

            if (dataGridViewColumnHeaderCell.OwningColumn == SortedColumn)
            {
                if (dataGridViewColumnHeaderCell.SortGlyphDirection == SortOrder.None)
                {
                    sortedColumn = null;
                    DataGridViewColumn dataGridViewColumn = dataGridViewColumnHeaderCell.OwningColumn;

                    if (dataGridViewColumn.IsDataBound)
                    {
                        // If the column whose SortGlyphChanges is the sorted column and it is also the dataBound column
                        // then see if there is another dataBound column which has the same property name as the sorted column.
                        // If so, then make that dataGridViewColumn the sorted column in the data grid view.
                        for (int i = 0; i < Columns.Count; i++)
                        {
                            if (dataGridViewColumn != Columns[i] &&
                                Columns[i].SortMode != DataGridViewColumnSortMode.NotSortable &&
                                string.Compare(dataGridViewColumn.DataPropertyName,
                                               Columns[i].DataPropertyName,
                                               true /*ignoreCase*/,
                                               CultureInfo.InvariantCulture) == 0)
                            {
                                Debug.Assert(Columns[i].IsDataBound, "two columns w/ the same DataPropertyName should be DataBound at the same time");
                                sortedColumn = Columns[i];
                                break;
                            }
                        }
                    }
                }

                sortOrder = sortedColumn != null ? sortedColumn.HeaderCell.SortGlyphDirection : SortOrder.None;
            }

            InvalidateCellPrivate(dataGridViewColumnHeaderCell);
        }

        private void OnTopLeftHeaderMouseDown()
        {
            if (MultiSelect)
            {
                SelectAll();
                if (-1 != ptCurrentCell.X)
                {
                    // Potentially have to give focus back to the current edited cell.
                    bool success = SetCurrentCellAddressCore(ptCurrentCell.X, ptCurrentCell.Y,
                                                             false /*setAnchorCellAddress*/,
                                                             false /*validateCurrentCell*/,
                                                             false /*throughMouseClick*/);
                    Debug.Assert(success);
                }
            }
        }

        protected virtual void OnUserAddedRow(DataGridViewRowEventArgs e)
        {
            if (e.Row.DataGridView != this)
            {
                throw new ArgumentException(SR.DataGridView_RowDoesNotBelongToDataGridView, "e.Row");
            }
            if (Events[EVENT_DATAGRIDVIEWUSERADDEDROW] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnUserDeletedRow(DataGridViewRowEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWUSERDELETEDROW] is DataGridViewRowEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        protected virtual void OnUserDeletingRow(DataGridViewRowCancelEventArgs e)
        {
            if (Events[EVENT_DATAGRIDVIEWUSERDELETINGROW] is DataGridViewRowCancelEventHandler eh && !dataGridViewOper[DATAGRIDVIEWOPER_inDispose] && !IsDisposed)
            {
                eh(this, e);
            }
        }

        private void OnUserPreferenceChanged(object sender, UserPreferenceChangedEventArgs e)
        {
            if (e.Category == UserPreferenceCategory.Color ||
                e.Category == UserPreferenceCategory.Locale ||
                e.Category == UserPreferenceCategory.General ||
                e.Category == UserPreferenceCategory.Window ||
                e.Category == UserPreferenceCategory.VisualStyle)
            {
                OnGlobalAutoSize();
                if (e.Category == UserPreferenceCategory.Window)
                {
                    cachedEditingControl = null;
                    if (editingControl != null)
                    {
                        // The editing control may not adapt well to the new system rendering,
                        // so instead of caching it into the this.cachedEditingControl variable
                        // next time editing mode is exited, simply discard the control.
                        dataGridViewState2[DATAGRIDVIEWSTATE2_discardEditingControl] = true;
                    }
                    PerformLayoutPrivate(false /*useRowShortcut*/, false /*computeVisibleRows*/, false /*invalidInAdjustFillingColumns*/, true /*repositionEditingControl*/);
                }
            }
        }

        protected override void OnValidating(CancelEventArgs e)
        {
            // Avoid Cell/Row Validation events when the grid or its editing control gets the focus
            if (!BecomingActiveControl && (editingControl == null || !editingControl.BecomingActiveControl))
            {
                if (!dataGridViewState1[DATAGRIDVIEWSTATE1_leavingWithTabKey])
                {
                    if (!EndEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit | DataGridViewDataErrorContexts.LeaveControl,
                                 DataGridViewValidateCellInternal.Always,
                                 false /*fireCellLeave*/,
                                 false /*fireCellEnter*/,
                                 false /*fireRowLeave*/,
                                 false /*fireRowEnter*/,
                                 false /*fireLeave*/,
                                 false /*keepFocus*/,
                                 false /*resetCurrentCell*/,
                                 false /*resetAnchorCell unused here*/))
                    {
                        e.Cancel = true;
                        return;
                    }
                }

                if (ptCurrentCell.X >= 0)
                {
                    DataGridViewCell dataGridViewCellTmp = null;
                    if (OnRowValidating(ref dataGridViewCellTmp, ptCurrentCell.X, ptCurrentCell.Y))
                    {
                        // Row validation was cancelled
                        e.Cancel = true;
                        return;
                    }
                    if (ptCurrentCell.X == -1)
                    {
                        return;
                    }
                    OnRowValidated(ref dataGridViewCellTmp, ptCurrentCell.X, ptCurrentCell.Y);
                    // Row validation was not cancelled, but does operation need to be re-evaluated.
                    if (DataSource != null &&
                        ptCurrentCell.X >= 0 &&
                        AllowUserToAddRowsInternal &&
                        newRowIndex == ptCurrentCell.Y)
                    {
                        // Current cell needs to be moved to the row just above the 'new row' if possible.
                        int rowIndex = Rows.GetPreviousRow(ptCurrentCell.Y, DataGridViewElementStates.Visible);
                        if (rowIndex > -1)
                        {
                            bool success = SetAndSelectCurrentCellAddress(ptCurrentCell.X, rowIndex,
                                true /*setAnchorCellAddress*/,
                                false /*validateCurrentCell*/,
                                false /*throughMouseClick*/,
                                false /*clearSelection*/,
                                false /*forceCurrentCellSelection*/);
                            Debug.Assert(success);
                        }
                        else
                        {
                            bool success = SetCurrentCellAddressCore(-1, -1,
                                true /*setAnchorCellAddress*/,
                                false /*validateCurrentCell*/,
                                false /*throughMouseClick*/);
                            Debug.Assert(success);
                        }
                    }
                }
            }
            base.OnValidating(e);
        }

        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);
            OnVisibleChangedPrivate();
        }

        private void OnVisibleChangedPrivate()
        {
            // Debug.Assert(!this.displayedBandsInfo.Dirty);   Not valid because EnsureDirtyState can potentially be called
            //                                                 for example when RowHeadersVisible is changed while the control is invisible.
            int rowIndexTmp;

            if (Visible)
            {
                // Make sure all displayed bands get the Displayed state: 1 & 2 for rows

                // 1. Make sure all displayed frozen rows have their Displayed state set to true
                int numDisplayedFrozenRows = displayedBandsInfo.NumDisplayedFrozenRows;
                if (numDisplayedFrozenRows > 0)
                {
                    rowIndexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                    while (numDisplayedFrozenRows > 0)
                    {
                        Debug.Assert(rowIndexTmp != -1);
                        if ((Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) != 0)
                        {
#if DEBUG
                            int numDisplayedFrozenRowsDbg = numDisplayedFrozenRows;
                            while (numDisplayedFrozenRowsDbg > 0)
                            {
                                Debug.Assert(rowIndexTmp != -1);
                                Debug.Assert((Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) != 0);
                                rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                                numDisplayedFrozenRowsDbg--;
                            }
#endif
                            return; // rows' Displayed states are already up-to-date. OnHandleCreated already did the job.
                        }
                        else
                        {
                            Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Displayed, true);
                            rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                            numDisplayedFrozenRows--;
                        }
                    }
                }

                // 2. Make sure all displayed scrolling rows have their Displayed state set to true
                rowIndexTmp = displayedBandsInfo.FirstDisplayedScrollingRow;
                if (rowIndexTmp > -1)
                {
                    Debug.Assert((Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Visible) != 0);
                    int numDisplayedScrollingRows = displayedBandsInfo.NumDisplayedScrollingRows;
                    Debug.Assert(numDisplayedScrollingRows > 0);
                    while (numDisplayedScrollingRows > 0)
                    {
                        Debug.Assert(rowIndexTmp != -1);
                        if ((Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) != 0)
                        {
#if DEBUG
                            int numDisplayedScrollingRowsDbg = numDisplayedScrollingRows;
                            while (numDisplayedScrollingRowsDbg > 0)
                            {
                                Debug.Assert(rowIndexTmp != -1);
                                Debug.Assert((Rows.GetRowState(rowIndexTmp) & DataGridViewElementStates.Displayed) != 0);
                                rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible);
                                numDisplayedScrollingRowsDbg--;
                            }
#endif
                            return; // rows' Displayed states are already up-to-date. OnHandleCreated already did the job.
                        }
                        else
                        {
                            Rows.SetRowState(rowIndexTmp, DataGridViewElementStates.Displayed, true);
                            rowIndexTmp = Rows.GetNextRow(rowIndexTmp, DataGridViewElementStates.Visible);
                            numDisplayedScrollingRows--;
                        }
                    }
                }
            }
            else
            {
                // Make sure all displayed bands lose the Displayed state
                UpdateRowsDisplayedState(false /*displayed*/);
            }

            UpdateColumnsDisplayedState(Visible /*displayed*/);
        }

        protected virtual void PaintBackground(Graphics graphics, Rectangle clipBounds, Rectangle gridBounds)
        {
            // Paint potential block below rows
            Rectangle rcBelowRows = gridBounds;

            // Remaining cell images on DataGridView
            int visibleRowsHeight = Rows.GetRowsHeight(DataGridViewElementStates.Displayed);
            if (layout.ColumnHeadersVisible)
            {
                rcBelowRows.Y += layout.ColumnHeaders.Height;
                rcBelowRows.Height -= layout.ColumnHeaders.Height;
            }
            else if (SingleHorizontalBorderAdded && visibleRowsHeight > 0)
            {
                rcBelowRows.Y++;
                rcBelowRows.Height--;
            }

            rcBelowRows.Y += visibleRowsHeight;
            rcBelowRows.Height -= visibleRowsHeight;
            if (rcBelowRows.Width > 0 && rcBelowRows.Height > 0)
            {
                graphics.FillRectangle(backgroundBrush, rcBelowRows);
            }

            // Paint potential block next to column headers and rows
            // Remaining cell images on DataGridView
            int visibleColumnsWidth = Columns.GetColumnsWidth(DataGridViewElementStates.Displayed);
            Rectangle rcNextRows = gridBounds;
            if (Columns.Count > 0)
            {
                if (layout.RowHeadersVisible)
                {
                    if (!RightToLeftInternal)
                    {
                        rcNextRows.X += layout.RowHeaders.Width;
                    }
                    rcNextRows.Width -= layout.RowHeaders.Width;
                }
                else if (SingleVerticalBorderAdded && visibleColumnsWidth > 0)
                {
                    if (!RightToLeftInternal)
                    {
                        rcNextRows.X++;
                    }
                    rcNextRows.Width--;
                }
            }
            int rowsWidth = visibleColumnsWidth - horizontalOffset;
            if (!RightToLeftInternal)
            {
                rcNextRows.X += rowsWidth;
            }
            rcNextRows.Width -= rowsWidth;
            if (rcBelowRows.Height > 0)
            {
                rcNextRows.Height = gridBounds.Height - rcBelowRows.Height;
            }
            if (rcNextRows.Width > 0 && rcNextRows.Height > 0)
            {
                graphics.FillRectangle(backgroundBrush, rcNextRows);
            }
        }

        private void PaintBorder(Graphics g, Rectangle clipRect, Rectangle bounds)
        {
            Debug.Assert(bounds.Left == 0);
            Debug.Assert(bounds.Top == 0);
            if (BorderStyle == BorderStyle.None)
            {
                return;
            }
            bool paintingNeeded = false;
            int borderWidth = BorderWidth;
            // Does the clipRect intersect with the top edge?
            Rectangle edge = new Rectangle(0, 0, bounds.Width, borderWidth);
            paintingNeeded = clipRect.IntersectsWith(edge);
            if (!paintingNeeded)
            {
                // Does the clipRect intersect with the bottom edge?
                edge.Y = bounds.Height - borderWidth;
                paintingNeeded = clipRect.IntersectsWith(edge);
                if (!paintingNeeded)
                {
                    // Does the clipRect intersect with the left edge?
                    edge.Y = 0;
                    edge.Height = bounds.Height;
                    edge.Width = borderWidth;
                    paintingNeeded = clipRect.IntersectsWith(edge);
                    if (!paintingNeeded)
                    {
                        // Does the clipRect intersect with the right edge?
                        edge.X = bounds.Width - borderWidth;
                        paintingNeeded = clipRect.IntersectsWith(edge);
                    }
                }
            }

            if (paintingNeeded)
            {
                if (BorderStyle == BorderStyle.Fixed3D)
                {
                    if (Application.RenderWithVisualStyles)
                    {
                        Pen pen = GetCachedPen(VisualStyleInformation.TextControlBorder);
                        g.DrawRectangle(pen, new Rectangle(0, 0, bounds.Width - 1, bounds.Height - 1));
                    }
                    else
                    {
                        ControlPaint.DrawBorder3D(g, bounds, Border3DStyle.Sunken);
                    }
                }
                else if (BorderStyle == BorderStyle.FixedSingle)
                {
                    Pen pen = GetCachedPen(SystemColors.ControlText);
                    g.DrawRectangle(pen, new Rectangle(0, 0, bounds.Width - 1, bounds.Height - 1));
                }
                else
                {
                    Debug.Fail("DataGridView.PaintBorder - Unexpected BorderStyle value.");
                }
            }
        }

        private void PaintColumnHeaders(Graphics g, Rectangle clipBounds, bool singleBorderAdded)
        {
            if (g.IsVisible(layout.ColumnHeaders))
            {
                Rectangle bandBounds, cellBounds;
                bandBounds = cellBounds = layout.ColumnHeaders;
                bandBounds.Height = cellBounds.Height = columnHeadersHeight;
                int cx = 0;
                bool isFirstDisplayedColumn = true, isLastVisibleColumn = false;
                DataGridViewCell cell;
                DataGridViewCellStyle inheritedCellStyle = new DataGridViewCellStyle();
                DataGridViewAdvancedBorderStyle dataGridViewAdvancedBorderStylePlaceholder = new DataGridViewAdvancedBorderStyle(), dgvabsEffective;
                DataGridViewColumn dataGridViewColumnNext = null;

                // first paint the visible frozen columns
                DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                while (dataGridViewColumn != null)
                {
                    cell = dataGridViewColumn.HeaderCell;
                    cellBounds.Width = dataGridViewColumn.Thickness;
                    if (singleBorderAdded && isFirstDisplayedColumn)
                    {
                        cellBounds.Width++;
                    }
                    Debug.Assert(cellBounds.Width > 0);
                    if (RightToLeftInternal)
                    {
                        cellBounds.X = bandBounds.Right - cx - cellBounds.Width;
                    }
                    else
                    {
                        cellBounds.X = bandBounds.X + cx;
                    }

                    BuildInheritedColumnHeaderCellStyle(inheritedCellStyle, cell);

                    dataGridViewColumnNext = Columns.GetNextColumn(dataGridViewColumn,
                        DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen,
                        DataGridViewElementStates.None);
                    if (dataGridViewColumnNext == null)
                    {
                        isLastVisibleColumn = (displayedBandsInfo.FirstDisplayedScrollingCol < 0);
                    }

                    dgvabsEffective = AdjustColumnHeaderBorderStyle(AdvancedColumnHeadersBorderStyle, dataGridViewAdvancedBorderStylePlaceholder,
                                                                    isFirstDisplayedColumn, isLastVisibleColumn);

                    // Microsoft: should paintSelectionBackground be dev-settable?
                    cell.PaintWork(g,
                                   clipBounds,
                                   cellBounds,
                                   -1,
                                   dataGridViewColumn.State,
                                   inheritedCellStyle,
                                   dgvabsEffective,
                                   DataGridViewPaintParts.Background | DataGridViewPaintParts.Border | DataGridViewPaintParts.ContentBackground | DataGridViewPaintParts.ContentForeground | DataGridViewPaintParts.ErrorIcon | DataGridViewPaintParts.SelectionBackground);

                    cx += cellBounds.Width;
                    if (cx >= bandBounds.Width)
                    {
                        break;
                    }

                    dataGridViewColumn = dataGridViewColumnNext;
                    isFirstDisplayedColumn = false;
                }

                // then paint the visible scrolling ones
                Rectangle scrollingBounds = bandBounds;
                if (!RightToLeftInternal)
                {
                    scrollingBounds.X -= negOffset;
                }
                scrollingBounds.Width += negOffset;

                if (displayedBandsInfo.FirstDisplayedScrollingCol >= 0 && cx < scrollingBounds.Width)
                {
                    Region clipRegion = null;
                    if (negOffset > 0)
                    {
                        clipRegion = g.Clip;
                        Rectangle rowRect = bandBounds;
                        if (!RightToLeftInternal)
                        {
                            rowRect.X += cx;
                        }
                        rowRect.Width -= cx;
                        g.SetClip(rowRect);
                    }

                    dataGridViewColumn = Columns[displayedBandsInfo.FirstDisplayedScrollingCol];
                    while (dataGridViewColumn != null)
                    {
                        Debug.Assert(dataGridViewColumn.Visible && !dataGridViewColumn.Frozen);

                        cell = dataGridViewColumn.HeaderCell;
                        cellBounds.Width = dataGridViewColumn.Thickness;
                        if (singleBorderAdded && isFirstDisplayedColumn)
                        {
                            cellBounds.Width++;
                        }
                        Debug.Assert(cellBounds.Width > 0);
                        if (RightToLeftInternal)
                        {
                            cellBounds.X = scrollingBounds.Right - cx - cellBounds.Width;
                        }
                        else
                        {
                            cellBounds.X = scrollingBounds.X + cx;
                        }

                        BuildInheritedColumnHeaderCellStyle(inheritedCellStyle, cell);

                        dataGridViewColumnNext = Columns.GetNextColumn(dataGridViewColumn,
                            DataGridViewElementStates.Visible,
                            DataGridViewElementStates.None);
                        isLastVisibleColumn = (dataGridViewColumnNext == null);

                        dgvabsEffective = AdjustColumnHeaderBorderStyle(AdvancedColumnHeadersBorderStyle, dataGridViewAdvancedBorderStylePlaceholder,
                                                                        isFirstDisplayedColumn, isLastVisibleColumn);

                        cell.PaintWork(g,
                                       clipBounds,
                                       cellBounds,
                                       -1,
                                       dataGridViewColumn.State,
                                       inheritedCellStyle,
                                       dgvabsEffective,
                                       DataGridViewPaintParts.Background | DataGridViewPaintParts.Border | DataGridViewPaintParts.ContentBackground | DataGridViewPaintParts.ContentForeground | DataGridViewPaintParts.ErrorIcon | DataGridViewPaintParts.SelectionBackground);

                        cx += cellBounds.Width;
                        if (cx >= scrollingBounds.Width)
                        {
                            break;
                        }

                        dataGridViewColumn = dataGridViewColumnNext;
                        isFirstDisplayedColumn = false;
                    }

                    if (negOffset > 0)
                    {
                        Debug.Assert(clipRegion != null);
                        g.Clip = clipRegion;
                        clipRegion.Dispose();
                    }
                }
            }
        }

        private void PaintGrid(Graphics g,
            Rectangle gridBounds,
            Rectangle clipRect,
            bool singleVerticalBorderAdded,
            bool singleHorizontalBorderAdded)
        {
            Rectangle rc = gridBounds;

            if (layout.TopLeftHeader.Width > 0 &&
                (clipRect.IntersectsWith(layout.TopLeftHeader) || lastHeaderShadow != -1))
            {
                if (Columns.Count > 0 || Rows.Count > 0)
                {
                    using (Region clipRegion = g.Clip)
                    {
                        g.SetClip(layout.TopLeftHeader);
                        PaintTopLeftHeaderCell(g);
                        g.Clip = clipRegion;
                    }
                }
            }

            if (layout.ColumnHeadersVisible)
            {
                Rectangle columnHeadersClip = new Rectangle();
                columnHeadersClip = layout.ColumnHeaders;
                if (singleVerticalBorderAdded)
                {
                    columnHeadersClip.Width++;
                }
                if (clipRect.IntersectsWith(columnHeadersClip) || lastHeaderShadow != -1)
                {
                    using (Region clipRegion = g.Clip)
                    {
                        g.SetClip(columnHeadersClip);
                        PaintColumnHeaders(g, columnHeadersClip, singleVerticalBorderAdded);
                        g.Clip = clipRegion;
                    }
                }
                int columnHeadersHeight = layout.ColumnHeaders.Height;
                rc.Y += columnHeadersHeight;
                rc.Height -= columnHeadersHeight;

                if (lastHeaderShadow != -1)
                {
                    DrawColHeaderShadow(g, lastHeaderShadow);
                }
            }

            if (rc.Height > 0)
            {
                PaintRows(g, rc, clipRect, /*singleVerticalBorderAdded, */ singleHorizontalBorderAdded);
            }

            if (currentRowSplitBar != -1)
            {
                Debug.Assert(dataGridViewOper[DATAGRIDVIEWOPER_trackColHeadersResize] || dataGridViewOper[DATAGRIDVIEWOPER_trackRowResize]);
                DrawRowSplitBar(currentRowSplitBar);
            }
            else if (currentColSplitBar != -1)
            {
                Debug.Assert(dataGridViewOper[DATAGRIDVIEWOPER_trackRowHeadersResize] ||
                    dataGridViewOper[DATAGRIDVIEWOPER_trackColResize] ||
                    dataGridViewOper[DATAGRIDVIEWOPER_trackKeyboardColResize]);
                DrawColSplitBar(currentColSplitBar);
            }
        }

        private void PaintRows(Graphics g,
            Rectangle boundingRect,
            Rectangle clipRect,
            /*bool singleVerticalBorderAdded,*/
            bool singleHorizontalBorderAdded)
        {
            int cy = 0;
            Rectangle rowBounds;
            DataGridViewRow dataGridViewRow;
            bool isFirstDisplayedRow = true;
            int indexTmp, indexTmpNext;

            // paint visible none-scrolling rows
            indexTmp = Rows.GetFirstRow(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
            while (indexTmp != -1)
            {
                rowBounds = boundingRect;

                // DataGridView AutoSizeRowsMode does not work properly after column sort
                // Should unshared the row and set the thickness to a perfect value
                // every time user scroll to display the specific row.

                DataGridViewAutoSizeRowsModeInternal autoSizeRowsModeInternal = (DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode;
                // Auto size row if needed
                if (autoSizeRowsModeInternal != DataGridViewAutoSizeRowsModeInternal.None)
                {
                    // this call may unshare the row.
                    int rowHeight = Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                    Rows.SharedRow(indexTmp).CachedThickness = rowHeight;
                    AutoResizeRowInternal(indexTmp, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), false /*fixedWidth*/, true /*internalAutosizing*/);
                }

                rowBounds.Height = Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                if (isFirstDisplayedRow && singleHorizontalBorderAdded)
                {
                    rowBounds.Height++;
                }
                rowBounds.Y = boundingRect.Y + cy;

                indexTmpNext = Rows.GetNextRow(indexTmp, DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);

                if (clipRect.IntersectsWith(rowBounds))
                {
                    dataGridViewRow = Rows.SharedRow(indexTmp);
                    dataGridViewRow.Paint(g,
                        clipRect,
                        rowBounds,
                        indexTmp,
                        Rows.GetRowState(indexTmp),
                        isFirstDisplayedRow,
                        (indexTmpNext == -1) && (displayedBandsInfo.FirstDisplayedScrollingRow == -1));
                }
                cy += rowBounds.Height;
                if (cy >= boundingRect.Height)
                {
                    break;
                }
                indexTmp = indexTmpNext;
                isFirstDisplayedRow = false;
            }

            // paint scrolling rows
            if (displayedBandsInfo.FirstDisplayedScrollingRow >= 0 && cy < boundingRect.Height)
            {
                indexTmp = displayedBandsInfo.FirstDisplayedScrollingRow;
                Debug.Assert((Rows.GetRowState(indexTmp) & DataGridViewElementStates.Frozen) == 0);
                Debug.Assert((Rows.GetRowState(indexTmp) & DataGridViewElementStates.Visible) != 0);
                while (indexTmp != -1)
                {
                    rowBounds = boundingRect;

                    // DataGridView AutoSizeRowsMode does not work properly after column sort
                    // Should unshared the row and set the thickness to a perfect value
                    // every time user scroll to display the specific row.

                    DataGridViewAutoSizeRowsModeInternal autoSizeRowsModeInternal = (DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode;
                    // Auto size row if needed
                    if (autoSizeRowsModeInternal != DataGridViewAutoSizeRowsModeInternal.None)
                    {
                        // this call may unshare the row.
                        int rowHeight = Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                        Rows.SharedRow(indexTmp).CachedThickness = rowHeight;
                        AutoResizeRowInternal(indexTmp, MapAutoSizeRowsModeToRowMode(autoSizeRowsMode), false /*fixedWidth*/, true /*internalAutosizing*/);
                    }

                    rowBounds.Height = Rows.SharedRow(indexTmp).GetHeight(indexTmp);
                    if (isFirstDisplayedRow && singleHorizontalBorderAdded)
                    {
                        rowBounds.Height++;
                    }
                    rowBounds.Y = boundingRect.Y + cy;

                    indexTmpNext = Rows.GetNextRow(indexTmp, DataGridViewElementStates.Visible);

                    if (clipRect.IntersectsWith(rowBounds))
                    {
                        dataGridViewRow = Rows.SharedRow(indexTmp);
                        dataGridViewRow.Paint(g,
                            clipRect,
                            rowBounds,
                            indexTmp,
                            Rows.GetRowState(indexTmp),
                            isFirstDisplayedRow,
                            indexTmpNext == -1);
                    }
                    cy += rowBounds.Height;
                    if (cy >= boundingRect.Height)
                    {
                        break;
                    }
                    indexTmp = indexTmpNext;
                    isFirstDisplayedRow = false;
                }
            }
        }

        private void PaintTopLeftHeaderCell(Graphics g)
        {
            if (g.IsVisible(layout.TopLeftHeader))
            {
                DataGridViewCell cell = TopLeftHeaderCell;
                DataGridViewCellStyle inheritedCellStyle = new DataGridViewCellStyle();
                BuildInheritedColumnHeaderCellStyle(inheritedCellStyle, cell);
                Rectangle cellBounds = layout.TopLeftHeader;
                cellBounds.Width = rowHeadersWidth;
                cellBounds.Height = columnHeadersHeight;
                // Microsoft: Should paintSelectionBackground be dev-settable?
                cell.PaintWork(g,
                               layout.TopLeftHeader,
                               cellBounds,
                               -1,
                               cell.State,
                               inheritedCellStyle,
                               AdjustedTopLeftHeaderBorderStyle,
                               DataGridViewPaintParts.Background | DataGridViewPaintParts.Border | DataGridViewPaintParts.ContentBackground | DataGridViewPaintParts.ContentForeground | DataGridViewPaintParts.ErrorIcon | DataGridViewPaintParts.SelectionBackground);
            }
        }

        private void PerformLayoutPrivate(bool useRowShortcut,
                                          bool computeVisibleRows,
                                          bool invalidInAdjustFillingColumns,
                                          bool repositionEditingControl)
        {
            inPerformLayoutCount++;
            try
            {
                if (invalidInAdjustFillingColumns && InAdjustFillingColumns)
                {
                    throw new InvalidOperationException(SR.DataGridView_CannotAlterAutoFillColumnParameter);
                }

                if (IsHandleCreated)
                {
                    bool columnsAdjusted = false;
                    if (useRowShortcut)
                    {
                        ComputeLayoutShortcut(computeVisibleRows);
                    }
                    else
                    {
                        columnsAdjusted = ComputeLayout();
                    }
                    FlushDisplayedChanged();
                    if (columnsAdjusted && inPerformLayoutCount < 3)
                    {
                        // Some columns were auto-filled, the rows and column headers may need to be autosized.
                        if ((((DataGridViewAutoSizeRowsModeInternal)autoSizeRowsMode) & DataGridViewAutoSizeRowsModeInternal.AllColumns) != 0)
                        {
                            AdjustShrinkingRows(autoSizeRowsMode, true /*fixedWidth*/, true /*internalAutosizing*/);
                        }
                        if (ColumnHeadersHeightSizeMode == DataGridViewColumnHeadersHeightSizeMode.AutoSize)
                        {
                            AutoResizeColumnHeadersHeight(true /*fixedRowHeadersWidth*/, true /*fixedColumnWidth*/);
                        }
                    }
                    if (repositionEditingControl && editingControl != null)
                    {
                        PositionEditingControl(true /*setLocation*/, false /*setSize*/, false /*setFocus*/);
                    }
                }
                else
                {
                    displayedBandsInfo.FirstDisplayedFrozenCol = -1;
                    displayedBandsInfo.FirstDisplayedFrozenRow = -1;
                    displayedBandsInfo.FirstDisplayedScrollingRow = -1;
                    displayedBandsInfo.FirstDisplayedScrollingCol = -1;
                    displayedBandsInfo.NumDisplayedFrozenRows = 0;
                    displayedBandsInfo.NumDisplayedFrozenCols = 0;
                    displayedBandsInfo.NumDisplayedScrollingRows = 0;
                    displayedBandsInfo.NumDisplayedScrollingCols = 0;
                    displayedBandsInfo.NumTotallyDisplayedFrozenRows = 0;
                    displayedBandsInfo.NumTotallyDisplayedScrollingRows = 0;
                    displayedBandsInfo.LastDisplayedScrollingRow = -1;
                    displayedBandsInfo.LastTotallyDisplayedScrollingCol = -1;
                    if (layout != null)
                    {
                        layout.dirty = true;
                    }
                }
            }
            finally
            {
                inPerformLayoutCount--;
                Debug.Assert(inPerformLayoutCount >= 0);
            }
        }

        private void PopulateNewRowWithDefaultValues()
        {
            if (newRowIndex != -1)
            {
                DataGridViewRow newRow = Rows.SharedRow(newRowIndex);
                DataGridViewCellCollection newRowCells = newRow.Cells;
                foreach (DataGridViewCell dataGridViewCell in newRowCells)
                {
                    if (dataGridViewCell.DefaultNewRowValue != null)
                    {
                        newRow = Rows[newRowIndex]; // unshare the 'new row'.
                        newRowCells = newRow.Cells;
                        break;
                    }
                }
                foreach (DataGridViewCell dataGridViewCell in newRowCells)
                {
                    dataGridViewCell.SetValueInternal(newRowIndex, dataGridViewCell.DefaultNewRowValue);
                }
            }
        }

        private void PositionEditingControl(bool setLocation, bool setSize, bool setFocus)
        {
            Debug.Assert(editingControl != null);

            if (!IsHandleCreated)
            {
                return;
            }

#if DEBUG
            DataGridViewCell dataGridViewCell = CurrentCellInternal;
            Debug.Assert(dataGridViewCell != null);
            Debug.Assert(dataGridViewCell.ColumnIndex == ptCurrentCell.X);
            Debug.Assert(dataGridViewCell.RowIndex == ptCurrentCell.Y || dataGridViewCell.RowIndex == -1);
#endif

            Rectangle editingZone = layout.Data;
            if (editingZone.Width == 0 || editingZone.Height == 0)
            {
                return;
            }

            dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlChanging] = true;

            try
            {
                int leftEdge = GetColumnXFromIndex(ptCurrentCell.X);
                if (RightToLeftInternal)
                {
                    leftEdge -= Columns[ptCurrentCell.X].Width - 1;
                }
                Rectangle cellBounds = new Rectangle(leftEdge, GetRowYFromIndex(ptCurrentCell.Y),
                                                     Columns[ptCurrentCell.X].Width, Rows.SharedRow(ptCurrentCell.Y).GetHeight(ptCurrentCell.Y));
                Rectangle cellClip = cellBounds;
                // Need to clip the zones of the frozen columns and rows and headers.
                if (!Columns[ptCurrentCell.X].Frozen)
                {
                    int totalVisibleFrozenWidth = Columns.GetColumnsWidth(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                    if (!RightToLeftInternal)
                    {
                        editingZone.X += totalVisibleFrozenWidth;
                    }
                    editingZone.Width = Math.Max(0, editingZone.Width - totalVisibleFrozenWidth);
                }
                if ((Rows.GetRowState(ptCurrentCell.Y) & DataGridViewElementStates.Frozen) == 0)
                {
                    int totalVisibleFrozenHeight = Rows.GetRowsHeight(DataGridViewElementStates.Visible | DataGridViewElementStates.Frozen);
                    editingZone.Y += totalVisibleFrozenHeight;
                }
                cellClip.Intersect(editingZone);

                if (cellClip.Width == 0 || cellClip.Height == 0)
                {
                    // we cannot simply make the control invisible because we want it to keep the focus.
                    // (and Control::CanFocus returns false if the control is not visible).
                    // So we place the editing control to the right of the DataGridView.
                    Debug.Assert(editingControl != null);
                    editingPanel.Location = new Point(Width + 1, 0);
                    dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlHidden] = true;
                }
                else
                {
                    bool singleVerticalBorderAdded = SingleVerticalBorderAdded;
                    bool singleHorizontalBorderAdded = SingleHorizontalBorderAdded;
                    bool isFirstDisplayedColumn = FirstDisplayedColumnIndex == ptCurrentCell.X;
                    bool isFirstDisplayedRow = FirstDisplayedRowIndex == ptCurrentCell.Y;

                    if (singleVerticalBorderAdded && isFirstDisplayedColumn)
                    {
                        if (!RightToLeftInternal)
                        {
                            cellBounds.X--;
                            cellClip.X--;
                        }
                        cellBounds.Width++;
                        cellClip.Width++;
                    }
                    if (singleHorizontalBorderAdded && isFirstDisplayedRow)
                    {
                        cellBounds.Y--;
                        cellClip.Y--;
                        cellBounds.Height++;
                        cellClip.Height++;
                    }

                    CurrentCellInternal.PositionEditingControl(
                        setLocation || dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlHidden],
                        setSize || dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlHidden],
                        cellBounds, cellClip, InheritedEditingCellStyle,
                        singleVerticalBorderAdded, singleHorizontalBorderAdded,
                        isFirstDisplayedColumn, isFirstDisplayedRow);
                    dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlHidden] = false;
                }
                editingPanel.Visible = true;
                if (setFocus)
                {
                    CorrectFocus(false /*onlyIfGridHasFocus*/);
                }
            }
            finally
            {
                dataGridViewState1[DATAGRIDVIEWSTATE1_editingControlChanging] = false;
            }
        }

        protected bool ProcessAKey(Keys keyData)
        {
            if ((keyData & (Keys.Shift | Keys.Control | Keys.Alt)) == Keys.Control &&
                MultiSelect)
            {
                SelectAll();
                return true;
            }
            return false;
        }

        protected bool ProcessDeleteKey(Keys keyData)
        {
            if (AllowUserToDeleteRowsInternal)
            {
                if (editingControl != null)
                {
                    // editing control gets a chance to handle the Delete key first
                    return false;
                }
                switch (SelectionMode)
                {
                    case DataGridViewSelectionMode.FullRowSelect:
                    case DataGridViewSelectionMode.RowHeaderSelect:
                        int remainingSelectedRows = 0;
                        try
                        {
                            selectedBandSnapshotIndexes = new DataGridViewIntLinkedList(selectedBandIndexes);
                            while (selectedBandSnapshotIndexes.Count > remainingSelectedRows)
                            {
                                int rowIndex = selectedBandSnapshotIndexes[remainingSelectedRows];
                                Debug.Assert(rowIndex >= 0);
                                if (rowIndex == newRowIndex || rowIndex >= Rows.Count)
                                {
                                    remainingSelectedRows++;
                                }
                                else
                                {
                                    DataGridViewRowCancelEventArgs dgvrce = new DataGridViewRowCancelEventArgs(Rows[rowIndex]);
                                    OnUserDeletingRow(dgvrce);
                                    if (!dgvrce.Cancel)
                                    {
                                        DataGridViewRow dataGridViewRow = Rows[rowIndex];
                                        if (DataSource != null)
                                        {
                                            int dataGridRowsCount = Rows.Count;
#if DEBUG
                                            int dataGridViewRowsCount = Rows.Count;           // the number of rows in the dataGridView row collection not counting the AddNewRow
                                            int rowCount = dataConnection.CurrencyManager.List.Count;
                                            if (AllowUserToAddRowsInternal)
                                            {
                                                if (newRowIndex < rowCount)
                                                {
                                                    // the user did not type inside the 'add new row'
                                                    Debug.Assert(rowCount == dataGridViewRowsCount, "out of sync in AddNewTransaction when the user did not type in the 'add new row'");
                                                }
                                                else
                                                {
                                                    dataGridViewRowsCount--;
                                                }
                                            }

                                            Debug.Assert(rowCount == dataGridViewRowsCount, "out of sync");
#endif
                                            DataGridViewDataErrorEventArgs dgvdee = null;
                                            try
                                            {
                                                DataConnection.DeleteRow(rowIndex);
                                            }
                                            catch (Exception exception)
                                            {
                                                if (ClientUtils.IsCriticalException(exception))
                                                {
                                                    throw;
                                                }
                                                // this is tricky.
                                                // the back-end threw an exception. At that stage, we did not delete the dataGridView row
                                                // from our collection of dataGridView rows.
                                                // So all we do is to throw the exception if the user wants. Otherwise we don't do anything.
                                                dgvdee = new DataGridViewDataErrorEventArgs(exception,
                                                                                            -1,
                                                                                            rowIndex,
                                                                                            // null,
                                                                                            // null,
                                                                                            DataGridViewDataErrorContexts.RowDeletion);
                                                OnDataErrorInternal(dgvdee);

                                                if (dgvdee.ThrowException)
                                                {
                                                    throw dgvdee.Exception;
                                                }
                                                else
                                                {
                                                    remainingSelectedRows++;
                                                }
                                            }

                                            if (dataGridRowsCount != Rows.Count)
                                            {
                                                Debug.Assert(dataGridViewRow.Index == -1);
                                                DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs(dataGridViewRow);
                                                OnUserDeletedRow(dgvre);
                                            }
                                            else if (dgvdee == null)
                                            {
                                                remainingSelectedRows++;
                                            }
                                        }
                                        else
                                        {
                                            Rows.RemoveAtInternal(rowIndex, false /*force*/);
                                            Debug.Assert(dataGridViewRow.Index == -1);
                                            DataGridViewRowEventArgs dgvre = new DataGridViewRowEventArgs(dataGridViewRow);
                                            OnUserDeletedRow(dgvre);
                                        }
                                    }
                                    else
                                    {
                                        remainingSelectedRows++;
                                    }
                                }
                            }
                        }
                        finally
                        {
                            selectedBandSnapshotIndexes = null;
                        }
                        return true;
                }
            }
            return false;
        }

        /// <summary>
        ///  Gets or sets a value that indicates whether a key should be processed
        ///  further.
        /// </summary>
        protected override bool ProcessDialogKey(Keys keyData)
        {
            Keys key = (keyData & Keys.KeyCode);

            if (key == Keys.Enter)
            {
                if (ProcessEnterKey(keyData))
                {
                    return true;
                }
            }
            else if (key == Keys.Escape)
            {
                bool keyEffective = IsEscapeKeyEffective;
                bool ret = base.ProcessDialogKey(keyData);
                if (!keyEffective)
                {
                    // This call may perform Click of Cancel button of form.
                    if (Focused)
                    {
                        // Make sure the current cell is in editing mode if needed.
                        if (ptCurrentCell.X > -1 &&
                            !IsCurrentCellInEditMode &&
                            (EditMode == DataGridViewEditMode.EditOnEnter ||
                            (EditMode != DataGridViewEditMode.EditProgrammatically && CurrentCellInternal.EditType == null)))
                        {
                            BeginEditInternal(true /*selectAll*/);
                        }
                    }
                }
                return ret;
            }
            else if (key == Keys.D0 || key == Keys.NumPad0)
            {
                if (ProcessZeroKey(keyData))
                {
                    return true;
                }
            }
            else if (key == Keys.C || key == Keys.Insert)
            {
                if (ProcessInsertKey(keyData))
                {
                    return true;
                }
            }
            else if (key == Keys.Tab)
            {
                if (ProcessTabKey(keyData))
                {
                    return true;
                }
                else
                {
                    if (editingControl != null)
                    {
                        dataGridViewState1[DATAGRIDVIEWSTATE1_leavingWithTabKey] = true;
                        if (!EndEdit(DataGridViewDataErrorContexts.Parsing | DataGridViewDataErrorContexts.Commit | DataGridViewDataErrorContexts.LeaveControl,
                                     DataGridViewValidateCellInternal.Always,
                                     true /*fireCellLeave*/,
                                     true /*fireCellEnter*/,
                                     true /*fireRowLeave*/,
                                     true /*fireRowEnter*/,
                                     true /*fireLeave*/,
                                     false /*keepFocus*/,
                                     false /*resetCurrentCell*/,
                                     false /*resetAnchorCell unused here*/))
                        {
                            return true;
                        }
                    }

                    keyData &= ~Keys.Control;
                    bool ret = false;

                    ret = base.ProcessDialogKey(keyData);

                    if (dataGridViewState1[DATAGRIDVIEWSTATE1_leavingWithTabKey] && Focused)
                    {
                        // There was no other control to tab to. The CellLeave, RowLeave, Leave events were raised.
                        // Since the DataGridView control still has the focus, Enter, RowEnter, CellEnter events need to be raised.
                        OnEnter(EventArgs.Empty);
                    }

                    return ret;
                }
            }
            return base.ProcessDialogKey(keyData);
        }

        protected bool ProcessDownKey(Keys keyData)
        {
            return ProcessDownKeyInternal(keyData, out bool moved);
        }

        private bool ProcessDownKeyInternal(Keys keyData, out bool moved)
        {
            bool success;
            DataGridViewColumn dataGridViewColumn = Columns.GetFirstColumn(DataGridViewElementStates.Visible);
            int firstVisibleColumnIndex = (dataGridViewColumn == null) ? -1 : dataGridViewColumn.Index;
            int lastVisibleRowIndex = Rows.GetLastRow(DataGridViewElementStates.Visible);
            if (firstVisibleColumnIndex == -1 || lastVisibleRowIndex == -1)
            {
                moved = false;
                return false;
            }
            int nextVisibleRowIndex = -1;
            if (ptCurrentCell.Y != -1)
            {
                nextVisibleRowIndex = Rows.GetNextRow(ptCurrentCell.Y, DataGridViewElementStates.Visible);
            }
            moved = true;

            noSelectionChangeCount++;
            try
            {
                switch (SelectionMode)
                {
                    case DataGridViewSelectionMode.CellSelect:
                    case DataGridViewSelectionMode.ColumnHeaderSelect:
                        if ((keyData & Keys.Control) == Keys.Control)
                        {
                            if ((keyData & Keys.Shift) == Keys.Shift)
                            {
                                if (ptCurrentCell.X == -1)
                                {
                                    ClearSelection();
                                    SetSelectedCellCore(firstVisibleColumnIndex, lastVisibleRowIndex, true);
                                    success = ScrollIntoView(firstVisibleColumnIndex, lastVisibleRowIndex, false);
                                    Debug.Assert(success);
                                    if (IsInnerCellOutOfBounds(firstVisibleColumnIndex, lastVisibleRowIndex))
                                    {
                                        moved = false;
                                        return true;
                                    }
                                    success = SetCurrentCellAddressCore(firstVisibleColumnIndex, lastVisibleRowIndex, true, false, false);
                                    if (!success)
                                    {
                                        // Microsoft: SetCurrentCellAddressCore can fail if by navigating to a cell
                                        // the list under the DataGridView changes.
                                        // In this case set moved to false so the users that call ProcessDownKey
                                        // will commit the data.
                                        moved = false;
                                    }
                                }
                                else
                                {
                                    if (MultiSelect)
                                    {
                                        if (!ScrollIntoView(ptCurrentCell.X, lastVisibleRowIndex, true))
                                        {
                                            return true;
                                        }
                                        //ClearSelection();
                                        Debug.Assert(ptAnchorCell.Y >= 0);
                                        //SelectCellRange(this.ptCurrentCell.X, this.ptAnchorCell.Y, this.ptCurrentCell.X, lastVisibleRowIndex, true);
                                        int oldEdgeColumnIndex = ptCurrentCell.X;
                                        int oldEdgeRowIndex = ptCurrentCell.Y;
                                        if (ptCurrentCell.X == -1 || ptAnchorCell.X == -1 ||
                                            IsRowOutOfBounds(lastVisibleRowIndex))
                                        {
                                            moved = false;
                                            return true;
                                        }
                                        UpdateSelectedCellsBlock(ptAnchorCell.X, ref oldEdgeColumnIndex, oldEdgeColumnIndex,
                                            ptAnchorCell.Y, ref oldEdgeRowIndex, lastVisibleRowIndex);
                                        success = SetCurrentCellAddressCore(ptCurrentCell.X, lastVisibleRowIndex, false, false, false);
                                        if (!success)
                                        {
                                            moved = false;
                                        }
                                    }
                                    else
                                    {
                                        if (!ScrollIntoView(ptCurrentCell.X, lastVisibleRowIndex, true))
                                        {
                                            return true;
                                        }
                                        if (ptCurrentCell.X == -1 || IsRowOutOfBounds(lastVisibleRowIndex))
                                        {
                                            moved = false;
                                            return true;
                                        }
                                        //SetSelectedCellCore(this.ptCurrentCell.X, this.ptCurrentCell.Y, false);
                                        ClearSelection();
                                        SetSelectedCellCore(ptCurrentCell.X, lastVisibleRowIndex, true);
                                        success = SetCurrentCellAddressCore(ptCurrentCell.X, lastVisibleRowIndex, true, false, false);
                                        if (!success)
                                        {
                                            moved = false;
                                        }
                                    }
                                }
                            }
                            else
                            {
                                if (ptCurrentCell.X == -1)
                                {
                                    ClearSelection();
                                    SetSelectedCellCore(firstVisibleColumnIndex, lastVisibleRowIndex, true);
                                    success = ScrollIntoView(firstVisibleColumnIndex, lastVisibleRowIndex, false);
                                    Debug.Assert(success);
                                    if (IsInnerCellOutOfBounds(firstVisibleColumnIndex, lastVisibleRowIndex))
                                    {
                                        moved = false;
                                        return true;
                                    }
                                    success = SetCurrentCellAddressCore(firstVisibleColumnIndex, lastVisibleRowIndex, true, false, false);
                                    if (!success)
                                    {
                                        moved = false;
                                    }
                                }
                                else
                                {
                                    if (!ScrollIntoView(ptCurrentCell.X, lastVisibleRowIndex, true))
                                    {
                                        return true;
                                    }
                                    if (ptCurrentCell.X == -1 || IsRowOutOfBounds(lastVisibleRowIndex))
                                    {
                                        moved = false;
                                        return true;
                                    }
                                    ClearSelection();
                                    SetSelectedCellCore(ptCurrentCell.X, lastVisibleRowIndex, true);
                                    success = SetCurrentCellAddressCore(ptCurrentCell.X, lastVisibleRowIndex, true, false, false);
                                    if (!success)
                                    {
                                        moved = false;
                                    }
                                }
                            }
                        }
                        else
                        {
                            if ((keyData & Keys.Shift) == Keys.Shift)
                            {
                                if (ptCurrentCell.X == -1)
                                {
                                    ClearSelection();
                                    SetSelectedCellCore(firstVisibleColumnIndex, lastVisibleRowIndex, true);
                                    success = ScrollIntoView(firstVisibleColumnIndex, lastVisibleRowIndex, false);
                                    Debug.Assert(success);
                                    if (IsInnerCellOutOfBounds(firstVisibleColumnIndex, lastVisibleRowIndex))
                                    {
                                        moved = false;
                                        return true;
                                    }
                                    success = SetCurrentCellAddressCore(firstVisibleColumnIndex, lastVisibleRowIndex, true, false, false);
                                    if (!success)
                                    {
                                        moved = false;
                                    }
                                }
                                else
                                {
                                    if (nextVisibleRowIndex == -1)
                                    {
                                        moved = false;
                                        return true;
                                    }
                                    if (!ScrollIntoView(ptCurrentCell.X, nextVisibleRowIndex, true))
                                    {
                                        return true;
                                    }
                                    if (ptCurrentCell.X == -1 || IsRowOutOfBounds(nextVisibleRowIndex))
                                    {
                                        moved = false;
                                        return true;
                                    }
                                    if (MultiSelect)
                                    {
                                        //SelectCellUnorderedRange(this.ptCurrentCell.X, this.ptAnchorCell.Y, this.ptCurrentCell.X, nextVisibleRowIndex, true);
                                        int oldEdgeColumnIndex = ptCurrentCell.X;
                                        int oldEdgeRowIndex = ptCurrentCell.Y;
                                        if (ptAnchorCell.X == -1)
                                        {
                                            moved = false;
                                            return true;
                                        }
                                        UpdateSelectedCellsBlock(ptAnchorCell.X, ref oldEdgeColumnIndex, oldEdgeColumnIndex,
                                            ptAnchorCell.Y, ref oldEdgeRowIndex, nextVisibleRowIndex);
                                    }
                                    else
                                    {
                                        ClearSelection();
                                        SetSelectedCellCore(ptCurrentCell.X, nextVisibleRowIndex, true);
                                    }
                                    success = SetCurrentCellAddressCore(ptCurrentCell.X, nextVisibleRowIndex, !MultiSelect, false, false);
                                    if (!success)
                                    {
                                        moved = false;
                                    }
                                }
                            }
                            else
                            {
                                if (ptCurrentCell.X == -1)
                                {
                                    ClearSelection();
                                    SetSelectedCellCore(firstVisibleColumnIndex, lastVisibleRowIndex, true);
                                    success = ScrollIntoView(firstVisibleColumnIndex, lastVisibleRowIndex, false);
                                    Debug.Assert(success);
                                    if (IsInnerCellOutOfBounds(firstVisibleColumnIndex, lastVisibleRowIndex))
                                    {
                                        moved = false;
                                        return true;
                                    }
                                    success = SetCurrentCellAddressCore(firstVisibleColumnIndex, lastVisibleRowIndex, true, false, false);
                                    if (!success)
                                    {
                                        moved = false;
                                    }
                                }
                                else
                                {
                                    if (nextVisibleRowIndex == -1)
                                    {
                                        moved = false;
                                        return true;
                                    }
                                    if (!ScrollIntoView(ptCurrentCell.X, nextVisibleRowIndex, true /*forCurrentCellChange*/))
                                    {
                                        return true;
                                    }
                                    if (ptCurrentCell.X == -1 || IsRowOutOfBounds(nextVisibleRowIndex))
                                    {
                                        moved = false;
                                        return true;
                                    }
                                    ClearSelection();
                                    SetSelectedCellCore(ptCurrentCell.X, nextVisibleRowIndex, true);
                                    success = SetCurrentCellAddressCore(ptCurrentCell.X,
                                        nextVisibleRowIndex,
                                        true  /*setAnchorCellAddress*/,
                                        false /*validateCurrentCell*/,
                                        false /*throughMouseClick*/);
                                    if (!success)
                                    {
                                        moved = false;
                                    }
                                }
                            }
                        }
                        return true;

                    case DataGridViewSelectionMode.FullRowSelect:
                        if ((keyData & Keys.Control) == Keys.Control)
                        {
                            if ((keyData & Keys.Shift) == Keys.Shift)
                            {
                                if (ptCurrentCell.X == -1)
                                {
                                    ClearSelection();
                                    SetSelectedRowCore(lastVisibleRowIndex, true);
                                    success = ScrollIntoView(firstVisibleColumnIndex, lastVisibleRowIndex, false);
                                    Debug.Assert(success);
                                    if (IsInnerCellOutOfBounds(firstVisibleColumnIndex, lastVisibleRowIndex))
                                    {
                                        moved = false;
                                        return true;
                                    }
                                    success = SetCurrentCellAddressCore(firstVisibleColumnIndex, lastVisibleRowIndex, true, false, false);
                                    if (!success)
                                    {
                                        moved = false;
                                    }
                                }
                                else
                                {
                                    if (MultiSelect)
                                    {
                                        if (!ScrollIntoView(ptCurrentCell.X, lastVisibleRowIndex, true))
                                        {
                                            return true;
                                        }
                                        if (ptAnchorCell.Y == -1 || ptCurrentCell.X == -1 ||
                                            IsRowOutOfBounds(lastVisibleRowIndex))
                                        {
                                            moved = false;
                                            return true;
                                        }
                                        ClearSelection();
                                        Debug.Assert(ptAnchorCell.Y >= 0);
                                        SelectRowRange(ptAnchorCell.Y, lastVisibleRowIndex, true);
                                        success = 