// Code generated by Speakeasy (https://speakeasy.com). DO NOT EDIT.

package utils

import (
	"context"
	"errors"
	"fmt"
	"maps"
	"slices"
	"strings"
	"sync"
	"time"

	"golang.org/x/sync/singleflight"
)

// Describes a session token for OAuth2 authentication.
type oAuth2Session struct {
	// OAuth2 access token.
	AccessToken string

	// Authorization grant which created the session.
	authorizationGrantor OAuth2AuthorizationGrantor

	// Access token expiration time in UTC seconds since epoch. Uses int64 over
	// time.Time to reduce storage size.
	expiresAt *int64

	// OAuth2 refresh token.
	refreshToken *string

	// OAuth2 access token scopes.
	scopes map[string]struct{}
}

// Returns true if the session was authenticated with at least the given OAuth2
// scopes.
func (s oAuth2Session) containsScopes(scopes []string) bool {
	if len(scopes) > len(s.scopes) {
		return false
	}

	for _, scope := range scopes {
		if _, ok := s.scopes[scope]; !ok {
			return false
		}
	}

	return true
}

// Returns true if the session is expired or within one minute of expiring.
func (s oAuth2Session) isExpired() bool {
	if s.expiresAt == nil {
		return false
	}

	return time.Now().Unix()+60 >= *s.expiresAt
}

// Returns the sorted, space-separated scope for the set of scopes.
func (s oAuth2Session) scope() string {
	return strings.Join(slices.Sorted(maps.Keys(s.scopes)), " ")
}

// Describes a collection of OAuth2 sessions.
type oauth2Sessions struct {
	// Authorization grant for creating sessions.
	authorizationGrantor OAuth2AuthorizationGrantor

	// Mapping of access tokens to sessions.
	sessions map[string]*oAuth2Session

	// Singleflight group for session requests, keyed by requested scopes.
	// Used to prevent multiple requests for an equivalent access token.
	sessionsGroup *singleflight.Group

	// Mutex for synchronizing access to the sessions map.
	sessionsMutex *sync.Mutex
}

// Returns a new OAuth2 session collection with an empty session map.
func newOAuth2Sessions(authorizationGrantor OAuth2AuthorizationGrantor) oauth2Sessions {
	return oauth2Sessions{
		authorizationGrantor: authorizationGrantor,
		sessions:             make(map[string]*oAuth2Session),
		sessionsGroup:        new(singleflight.Group),
		sessionsMutex:        new(sync.Mutex),
	}
}

// Concurrency safe session creation.
func (s oauth2Sessions) create(ctx context.Context, httpClient HTTPClient, scopes []string) (*oAuth2Session, error) {
	scope := s.scope(scopes)

	rawSession, err, _ := s.sessionsGroup.Do(scope, func() (any, error) {
		session, err := s.authorizationGrantor.NewSession(ctx, httpClient, scopes)

		if err != nil {
			return nil, fmt.Errorf("failed to get access token: %w", err)
		}

		s.sessionsMutex.Lock()
		s.sessions[session.AccessToken] = session
		s.sessionsMutex.Unlock()

		return session, nil
	})

	if err != nil {
		return nil, err
	}

	return rawSession.(*oAuth2Session), nil
}

// Concurrency safe session deletion.
func (s oauth2Sessions) delete(accessToken string) {
	s.sessionsMutex.Lock()
	if session, ok := s.sessions[accessToken]; ok {
		s.sessionsGroup.Forget(session.scope())
	}
	delete(s.sessions, accessToken)
	s.sessionsMutex.Unlock()
}

// Returns a non-expired session that contains the given scopes, if found, or
// creates a new session.
func (s oauth2Sessions) getSessionContainingScopes(ctx context.Context, httpClient HTTPClient, scopes []string) (*oAuth2Session, error) {
	for accessToken, session := range s.sessions {
		if session == nil {
			continue
		}

		if session.isExpired() {
			go s.delete(accessToken)

			continue
		}

		if session.containsScopes(scopes) {
			return session, nil
		}
	}

	return s.create(ctx, httpClient, scopes)
}

// Returns a consistent key for a set of scopes.
func (s oauth2Sessions) scope(scopes []string) string {
	if len(scopes) == 0 {
		return ""
	}

	// Sort the scopes to ensure consistent ordering.
	slices.Sort(scopes)

	return strings.Join(scopes, " ")
}

// Describes an OAuth2 token endpoint, including credential-based sessions.
type oAuth2TokenEndpoint struct {
	// Mapping of credential hash keys to sessions.
	credentialSessions map[string]oauth2Sessions

	// Mutex for synchronizing access to the credential sessions map.
	credentialSessionsMutex *sync.Mutex
}

// Returns a new OAuth2 token endpoint with an empty credential session map.
func newOAuth2TokenEndpoint() *oAuth2TokenEndpoint {
	return &oAuth2TokenEndpoint{
		credentialSessions:      make(map[string]oauth2Sessions),
		credentialSessionsMutex: new(sync.Mutex),
	}
}

// Empties a stored credentials session.
func (e oAuth2TokenEndpoint) deleteCredentialsSession(authorizationGrantor OAuth2AuthorizationGrantor) {
	if authorizationGrantor == nil {
		return
	}

	e.credentialSessionsMutex.Lock()
	e.credentialSessions[authorizationGrantor.Hash()] = newOAuth2Sessions(authorizationGrantor)
	e.credentialSessionsMutex.Unlock()
}

// Returns a stored session, or creates a new session, for the given scopes.
func (e oAuth2TokenEndpoint) getSessionContainingScopes(ctx context.Context, httpClient HTTPClient, authorizationGrantor OAuth2AuthorizationGrantor, scopes []string) (*oAuth2Session, error) {
	if authorizationGrantor == nil {
		return nil, errors.New("missing authorization grantor")
	}

	e.credentialSessionsMutex.Lock()

	sessions, ok := e.credentialSessions[authorizationGrantor.Hash()]

	if !ok {
		sessions = newOAuth2Sessions(authorizationGrantor)
		e.credentialSessions[authorizationGrantor.Hash()] = sessions
	}

	e.credentialSessionsMutex.Unlock()

	return sessions.getSessionContainingScopes(ctx, httpClient, scopes)
}

// Describes storage for OAuth2Session.
type OAuth2SessionStore struct {
	// Mapping of token endpoints to credential sessions.
	tokenEndpoints map[string]*oAuth2TokenEndpoint

	// Mutex for synchronizing access to the token endpoints map.
	tokenEndpointsMutex *sync.Mutex
}

// Creates an initialized OAuth2 session store.
func NewOAuth2SessionStore() *OAuth2SessionStore {
	return &OAuth2SessionStore{
		tokenEndpoints:      make(map[string]*oAuth2TokenEndpoint),
		tokenEndpointsMutex: new(sync.Mutex),
	}
}

// Deletes a stored credentials session.
func (s OAuth2SessionStore) DeleteCredentialsSession(authorizationGrantor OAuth2AuthorizationGrantor) {
	if authorizationGrantor == nil {
		return
	}

	endpoint, ok := s.tokenEndpoints[authorizationGrantor.TokenEndpoint()]

	if !ok {
		return
	}

	endpoint.deleteCredentialsSession(authorizationGrantor)
}

// Returns a stored session, or creates a new session, for the given scopes.
func (s OAuth2SessionStore) GetSession(ctx context.Context, httpClient HTTPClient, authorizationGrantor OAuth2AuthorizationGrantor, scopes []string) (*oAuth2Session, error) {
	if httpClient == nil {
		return nil, errors.New("missing HTTP client")
	}

	if authorizationGrantor == nil {
		return nil, errors.New("missing authorization grantor")
	}

	tokenEndpoint := authorizationGrantor.TokenEndpoint()

	s.tokenEndpointsMutex.Lock()

	endpoint, ok := s.tokenEndpoints[tokenEndpoint]

	if !ok {
		endpoint = newOAuth2TokenEndpoint()
		s.tokenEndpoints[tokenEndpoint] = endpoint
	}

	s.tokenEndpointsMutex.Unlock()

	return endpoint.getSessionContainingScopes(ctx, httpClient, authorizationGrantor, scopes)
}
