// Code generated by Speakeasy (https://speakeasy.com). DO NOT EDIT.

package utils

import (
	"bytes"
	"context"
	"crypto/md5"
	"encoding/hex"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"strings"
	"time"
)

// HTTPClient provides an interface for supplying the SDK with a custom HTTP client
type HTTPClient interface {
	Do(req *http.Request) (*http.Response, error)
}

// Describes an interface for OAuth2 authorization grants, such as
// pre-authorized access token, client credentials, or resource owner password
// credentials.
type OAuth2AuthorizationGrantor interface {
	// Returns the authorization grant type, such as "client_credentials" or
	// "password".
	GrantType() string

	// Returns the authorization grant as a hash suitable for mapping keys.
	Hash() string

	// Returns a new session for the authorization grant.
	NewSession(ctx context.Context, httpClient HTTPClient, scopes []string) (*oAuth2Session, error)

	// Returns the token endpoint for the authorization grant.
	TokenEndpoint() string
}

var _ OAuth2AuthorizationGrantor = (*OAuth2AccessTokenCredentials)(nil)

// Describes an OAuth2 authorization grant implementation that contains a
// pre-authorized OAuth2 access token. Using this grant implementation will
// bypass the OAuth2 session store, as there is no need to call the OAuth2
// token endpoint.
type OAuth2AccessTokenCredentials struct {
	// Pre-authorized OAuth2 access token.
	AccessToken string
}

// Creates a new OAuth2 authorization grant with pre-authorized access token.
func NewOAuth2AccessTokenCredentials(accessToken string) OAuth2AccessTokenCredentials {
	return OAuth2AccessTokenCredentials{
		AccessToken: accessToken,
	}
}

// Returns the authorization grant type. This implementation is custom and
// returns an unspecified "access_token" value rather than conflating this
// grant type with the differing "implicit" grant type.
func (c OAuth2AccessTokenCredentials) GrantType() string {
	return "access_token"
}

// Unused, however required to implement OAuth2AuthorizationGrantor.
func (c OAuth2AccessTokenCredentials) Hash() string {
	return ""
}

// Creates a new OAuth2 Session.
func (c OAuth2AccessTokenCredentials) NewSession(_ context.Context, _ HTTPClient, _ []string) (*oAuth2Session, error) {
	session := &oAuth2Session{
		AccessToken:          c.AccessToken,
		authorizationGrantor: c,
	}

	return session, nil
}

// Unused, however required to implement OAuth2AuthorizationGrantor.
func (c OAuth2AccessTokenCredentials) TokenEndpoint() string {
	return ""
}

var _ OAuth2AuthorizationGrantor = (*oAuth2ClientCredentials)(nil)

// Describes OAuth2 Client Credentials Grant.
// Reference: https://datatracker.ietf.org/doc/html/rfc6749#section-4.4
type oAuth2ClientCredentials struct {
	// OAuth2 client identifier.
	clientID string

	// OAuth2 client secret.
	clientSecret string

	// Cached credentials mapping hash.
	hash *string

	// OAuth2 token endpoint.
	tokenEndpoint string
}

// Creates a new OAuth2 Client Credentials grant.
func NewOAuth2ClientCredentials(tokenEndpoint string, clientID string, clientSecret string) oAuth2ClientCredentials {
	return oAuth2ClientCredentials{
		clientID:      clientID,
		clientSecret:  clientSecret,
		tokenEndpoint: tokenEndpoint,
	}
}

// Returns the authorization grant type of "client_credentials".
func (c oAuth2ClientCredentials) GrantType() string {
	return "client_credentials"
}

// Returns the authorization grant as a hash suitable for mapping keys.
func (c oAuth2ClientCredentials) Hash() string {
	if c.hash != nil {
		return *c.hash
	}

	md5Checksum := md5.Sum([]byte(c.clientID + ":" + c.clientSecret))
	md5ChecksumStr := hex.EncodeToString(md5Checksum[:])

	c.hash = &md5ChecksumStr

	return *c.hash
}

// Creates a new OAuth2 Session.
func (c oAuth2ClientCredentials) NewSession(ctx context.Context, httpClient HTTPClient, scopes []string) (*oAuth2Session, error) {
	grant := url.Values{}
	grant.Set("grant_type", c.GrantType())
	grant.Set("client_id", c.clientID)
	grant.Set("client_secret", c.clientSecret)

	if len(scopes) > 0 {
		grant.Set("scope", strings.Join(scopes, " "))
	}

	httpRequestBody := bytes.NewBufferString(grant.Encode())

	httpRequest, err := http.NewRequestWithContext(ctx, http.MethodPost, c.tokenEndpoint, httpRequestBody)

	if err != nil {
		return nil, fmt.Errorf("failed to create token request: %w", err)
	}

	httpRequest.Header.Set("Content-Type", "application/x-www-form-urlencoded")

	httpResponse, err := httpClient.Do(httpRequest)

	if err != nil {
		return nil, fmt.Errorf("failed to send token request: %w", err)
	}

	defer httpResponse.Body.Close()

	if httpResponse.StatusCode < 200 || httpResponse.StatusCode >= 300 {
		httpResponseBody, _ := io.ReadAll(httpResponse.Body)
		return nil, fmt.Errorf("unexpected status code: %d: %s", httpResponse.StatusCode, httpResponseBody)
	}

	var response oAuth2ClientCredentialsResponse

	if err := json.NewDecoder(httpResponse.Body).Decode(&response); err != nil {
		return nil, fmt.Errorf("failed to decode token response: %w", err)
	}

	if response.TokenType != "Bearer" {
		return nil, fmt.Errorf("unexpected token type: %s", response.TokenType)
	}

	return response.newSession(c, scopes), nil
}

// Returns the token endpoint for the authorization grant.
func (c oAuth2ClientCredentials) TokenEndpoint() string {
	return c.tokenEndpoint
}

// OAuth2 Client Credentials token endpoint response.
// Reference: https://datatracker.ietf.org/doc/html/rfc6749#section-4.4.3
type oAuth2ClientCredentialsResponse struct {
	// Access token issued by the authorization server.
	AccessToken string `json:"access_token"`

	// Lifetime in seconds of the access token.
	ExpiresIn *int64 `json:"expires_in,omitempty"`

	// Refresh token, which can be used to obtain new access tokens using the
	// same authorization grant.
	RefreshToken *string `json:"refresh_token,omitempty"`

	// Scope of the access token, if not identical to the scope requested.
	Scope *string `json:"scope,omitempty"`

	// Type of the token.
	TokenType string `json:"token_type"`
}

func (r oAuth2ClientCredentialsResponse) newSession(clientCredentials oAuth2ClientCredentials, requestScopes []string) *oAuth2Session {
	scopes := make(map[string]struct{})

	if r.Scope != nil {
		for _, scope := range strings.Split(*r.Scope, " ") {
			scopes[scope] = struct{}{}
		}
	} else {
		for _, scope := range requestScopes {
			scopes[scope] = struct{}{}
		}
	}

	session := &oAuth2Session{
		AccessToken:          r.AccessToken,
		authorizationGrantor: clientCredentials,
		refreshToken:         r.RefreshToken,
		scopes:               scopes,
	}

	if r.ExpiresIn != nil {
		expiresAt := time.Now().Unix() + *r.ExpiresIn
		session.expiresAt = &expiresAt
	}

	return session
}
