package devbox

import (
	"fmt"
	"github.com/formancehq/ledger/deployments/pulumi/pkg/api"
	"github.com/formancehq/ledger/deployments/pulumi/pkg/common"
	"github.com/formancehq/ledger/deployments/pulumi/pkg/exporters"
	"github.com/formancehq/ledger/deployments/pulumi/pkg/storage"
	appsv1 "github.com/pulumi/pulumi-kubernetes/sdk/v4/go/kubernetes/apps/v1"
	corev1 "github.com/pulumi/pulumi-kubernetes/sdk/v4/go/kubernetes/core/v1"
	metav1 "github.com/pulumi/pulumi-kubernetes/sdk/v4/go/kubernetes/meta/v1"
	"github.com/pulumi/pulumi/sdk/v3/go/pulumi"
)

type Component struct {
	pulumi.ResourceState
	Deployment *appsv1.Deployment
}

type ComponentArgs struct {
	common.CommonArgs
	Storage    *storage.Component
	API       *api.Component
	Exporters *exporters.Component
}

func NewComponent(ctx *pulumi.Context, name string, args ComponentArgs, opts ...pulumi.ResourceOption) (*Component, error) {
	cmp := &Component{}
	err := ctx.RegisterComponentResource("Formance:Ledger:DevBox", name, cmp, opts...)
	if err != nil {
		return nil, err
	}

	containers := corev1.ContainerArray{
		args.Storage.GetDevBoxContainer(ctx.Context()),
		args.API.GetDevBoxContainer(ctx.Context()),
	}

	if args.Exporters != nil {
		for _, exporter := range args.Exporters.Exporters {
			containers = append(containers, exporter.Component.GetDevBoxContainer(ctx.Context()))
		}
	}

	cmp.Deployment, err = appsv1.NewDeployment(ctx, "ledger-devbox", &appsv1.DeploymentArgs{
		Metadata: &metav1.ObjectMetaArgs{
			Namespace: args.Namespace.ToOutput(ctx.Context()).Untyped().(pulumi.StringOutput),
			Labels: pulumi.StringMap{
				"com.formance.stack/app": pulumi.String("ledger-devbox"),
			},
		},
		Spec: appsv1.DeploymentSpecArgs{
			Selector: &metav1.LabelSelectorArgs{
				MatchLabels: pulumi.StringMap{
					"com.formance.stack/app": pulumi.String("ledger-devbox"),
				},
			},
			Template: &corev1.PodTemplateSpecArgs{
				Metadata: &metav1.ObjectMetaArgs{
					Labels: pulumi.StringMap{
						"com.formance.stack/app": pulumi.String("ledger-devbox"),
					},
				},
				Spec: corev1.PodSpecArgs{
					TerminationGracePeriodSeconds: pulumi.IntPtr(0),
					Containers:                    containers,
				},
			},
		},
	}, pulumi.Parent(cmp))
	if err != nil {
		return nil, fmt.Errorf("creating deployment: %w", err)
	}

	if err := ctx.RegisterResourceOutputs(cmp, pulumi.Map{}); err != nil {
		return nil, fmt.Errorf("registering outputs: %w", err)
	}

	return cmp, nil
}
