// Copyright 2020 The Flutter Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file or at https://developers.google.com/open-source/licenses/bsd.

part of 'server.dart';

Future<List<String>> requestAndroidBuildVariants(String path) async {
  if (isDevToolsServerAvailable) {
    final uri = Uri(
      path: DeeplinkApi.androidBuildVariants,
      queryParameters: {DeeplinkApi.deeplinkRootPathPropertyName: path},
    );
    final resp = await request(uri.toString());
    if (resp?.statusOk ?? false) {
      return (jsonDecode(resp!.body) as List).cast<String>();
    } else {
      logWarning(resp, DeeplinkApi.androidBuildVariants);
    }
  }
  return const <String>[];
}

Future<AppLinkSettings> requestAndroidAppLinkSettings(
  String path, {
  required String buildVariant,
}) async {
  if (isDevToolsServerAvailable) {
    final uri = Uri(
      path: DeeplinkApi.androidAppLinkSettings,
      queryParameters: {
        DeeplinkApi.deeplinkRootPathPropertyName: path,
        DeeplinkApi.androidBuildVariantPropertyName: buildVariant,
      },
    );
    final resp = await request(uri.toString());
    if (resp != null) {
      if (resp.statusOk) {
        return AppLinkSettings.fromJson(resp.body);
      } else {
        logWarning(resp, DeeplinkApi.androidAppLinkSettings);
        return AppLinkSettings.fromErrorJson(resp.body.toString());
      }
    }
  }
  return AppLinkSettings.error('DevTools server is not available');
}

Future<XcodeBuildOptions> requestIosBuildOptions(String path) async {
  if (isDevToolsServerAvailable) {
    final uri = Uri(
      path: DeeplinkApi.iosBuildOptions,
      queryParameters: {DeeplinkApi.deeplinkRootPathPropertyName: path},
    );
    final resp = await request(uri.toString());
    if (resp?.statusOk ?? false) {
      return XcodeBuildOptions.fromJson(resp!.body);
    } else {
      logWarning(resp, DeeplinkApi.iosBuildOptions);
    }
  }
  return XcodeBuildOptions.empty;
}

Future<UniversalLinkSettings> requestIosUniversalLinkSettings(
  String path, {
  required String configuration,
  required String target,
}) async {
  if (isDevToolsServerAvailable) {
    final uri = Uri(
      path: DeeplinkApi.iosUniversalLinkSettings,
      queryParameters: {
        DeeplinkApi.deeplinkRootPathPropertyName: path,
        DeeplinkApi.xcodeConfigurationPropertyName: configuration,
        DeeplinkApi.xcodeTargetPropertyName: target,
      },
    );
    final resp = await request(uri.toString());
    if (resp?.statusOk ?? false) {
      return UniversalLinkSettings.fromJson(resp!.body);
    } else {
      logWarning(resp, DeeplinkApi.iosUniversalLinkSettings);
    }
  }
  return UniversalLinkSettings.empty;
}
