/*
 * Copyright 2024 EPAM Systems, Inc
 *
 * See the NOTICE file distributed with this work for additional information
 * regarding copyright ownership. Licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.epam.deltix.qsrv.hf.tickdb.lang.compiler.cg;

import com.epam.deltix.qsrv.hf.pub.NullValueException;
import com.epam.deltix.qsrv.hf.pub.md.*;
import com.epam.deltix.qsrv.hf.pub.values.*;
import com.epam.deltix.util.collections.generated.*;
import com.epam.deltix.util.jcg.JContext;
import com.epam.deltix.util.jcg.JExpr;
import com.epam.deltix.util.jcg.JStatement;
import com.epam.deltix.util.jcg.scg.JavaSrcGenContext;


/**
 *
 */
public class QCGHelpers {
    public static final boolean        DEBUG_DUMP_CODE = Boolean.getBoolean ("deltix.qql.dump");
       
    public static final JContext       CTXT =  new JavaSrcGenContext();
    
    static JStatement   throwNVX () {
        return (CTXT.staticVarRef (NullValueException.class, "INSTANCE").throwStmt ());
    }

    public static JExpr        objtoex (Object obj) {
        if (obj instanceof JExpr)
            return ((JExpr) obj);

        if (obj instanceof Boolean)
            return (CTXT.booleanLiteral ((Boolean) obj));

        if (obj instanceof Integer)
            return (CTXT.intLiteral ((Integer) obj));

        if (obj instanceof Long)
            return (CTXT.longLiteral ((Long) obj));

        if (obj instanceof Short)
            return (CTXT.intLiteral ((Short) obj));

        if (obj instanceof Byte)
            return (CTXT.intLiteral ((Byte) obj));

        if (obj instanceof Double)
            return (CTXT.doubleLiteral ((Double) obj));
        
        if (obj instanceof Float)
            return (CTXT.floatLiteral ((Float) obj));

        return (CTXT.stringLiteral (obj.toString ()));
    }

    public static Class<?>        primitiveWrapper (Class<?> clazz) {

        if (clazz.equals(boolean.class))
            return Boolean.class;
        else if (clazz.equals(char.class))
            return Character.class;
        else if (clazz.equals(byte.class))
            return Byte.class;
        else if (clazz.equals(short.class))
            return Short.class;
        else if (clazz.equals(int.class))
            return Integer.class;
        else if (clazz.equals(float.class))
            return Float.class;
        else if (clazz.equals(double.class))
            return Double.class;
        else if (clazz.equals(long.class))
            return Long.class;

        return Object.class;
    }

    public static Class<?> primitiveArrayList(Class<?> clazz) {
        if (clazz.equals(boolean.class))
            return BooleanArrayList.class;
        else if (clazz.equals(char.class))
            return CharacterArrayList.class;
        else if (clazz.equals(byte.class))
            return ByteArrayList.class;
        else if (clazz.equals(short.class))
            return ShortArrayList.class;
        else if (clazz.equals(int.class))
            return IntegerArrayList.class;
        else if (clazz.equals(float.class))
            return FloatArrayList.class;
        else if (clazz.equals(double.class))
            return DoubleArrayList.class;
        else if (clazz.equals(long.class))
            return LongArrayList.class;

        return ObjectArrayList.class;
    }

    public static Class<?> primitiveHashSet(Class<?> clazz) {
        if (clazz.equals(char.class))
            return CharacterHashSet.class;
        else if (clazz.equals(short.class))
            return ShortHashSet.class;
        else if (clazz.equals(int.class))
            return IntegerHashSet.class;
        else if (clazz.equals(float.class))
            return FloatHashSet.class;
        else if (clazz.equals(double.class))
            return DoubleHashSet.class;
        else if (clazz.equals(long.class))
            return LongHashSet.class;

        return ObjectHashSet.class;
    }

    public static boolean isHashSetSupported(Class<?> clazz) {
        return !clazz.equals(boolean.class) && !clazz.equals(byte.class);
    }

    static JExpr []     objtoex (Object ... args) {
        int         n = args.length;
        JExpr []    ret = new JExpr [n];

        for (int ii = 0; ii < n; ii++)
            ret [ii] = objtoex (args [ii]);

        return (ret);
    }

    static JStatement   throwRX (Object ... args) {
        return (
            CTXT.newExpr (
                RuntimeException.class,
                CTXT.sum (objtoex (args))
            ).throwStmt ()
        );
    }

    static JStatement   throwIAX (String msg) {
        return throwIAX(objtoex(msg));
    }

    static JStatement   throwIAX (JExpr msgExpr) {
        return (
            CTXT.newExpr(
                IllegalArgumentException.class,
                msgExpr
            ).throwStmt()
        );
    }

    static JStatement   throwISX (Object ... args) {
        return (
            CTXT.newExpr (
                IllegalStateException.class,
                CTXT.sum (objtoex (args))
            ).throwStmt ()
        );
    }
    
    public static Class <? extends ValueBean>   getValueBeanClass (DataType type) {
        if (type instanceof IntegerDataType)
            return (IntegerValueBean.class);

        if (type instanceof FloatDataType)
            return (((FloatDataType) type).isFloat () ? FloatValueBean.class : DoubleValueBean.class);

        if (type instanceof BooleanDataType)
            return (BooleanValueBean.class);

        if (type instanceof VarcharDataType)
            return (StringValueBean.class);

        throw new UnsupportedOperationException (type.getBaseName ());
    }
}
