/*
 * Copyright 2024 EPAM Systems, Inc
 *
 * See the NOTICE file distributed with this work for additional information
 * regarding copyright ownership. Licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.epam.deltix.qsrv.hf.tickdb.comm.server;

import com.epam.deltix.qsrv.hf.pub.TimeSource;
import com.epam.deltix.qsrv.hf.tickdb.comm.server.aeron.AeronThreadTracker;
import com.epam.deltix.qsrv.hf.tickdb.comm.server.aeron.DXServerAeronContext;
import com.epam.deltix.qsrv.hf.tickdb.impl.topic.CopyTopicToStreamTaskManager;
import com.epam.deltix.qsrv.hf.tickdb.impl.topic.TopicRegistryFactory;
import com.epam.deltix.qsrv.hf.tickdb.impl.topic.TopicSupportWrapper;
import com.epam.deltix.qsrv.hf.tickdb.impl.topic.topicregistry.DirectTopicRegistry;
import com.epam.deltix.qsrv.hf.tickdb.comm.TDBProtocol;
import com.epam.deltix.qsrv.hf.tickdb.pub.*;
import com.epam.deltix.util.concurrent.QuickExecutor;
import com.epam.deltix.util.time.DefaultTimeSourceProvider;
import com.epam.deltix.util.vsocket.TLSContext;
import com.epam.deltix.util.vsocket.TransportProperties;
import com.epam.deltix.util.vsocket.VSServer;

import java.io.*;
import java.net.InetAddress;
import java.util.logging.*;

/**
 * 
 */
public class TickDBServer {
    public static final Logger  LOGGER = Logger.getLogger ("deltix.tickdb.server");

    //private final QuickExecutor         executor;
    private volatile int                port = TDBProtocol.DEFAULT_PORT;
    private VSServer                    server;
    private InetAddress                 address;
    private final DXTickDB              db;
    private final TLSContext            ssl;
    private TransportProperties         transportProperties;

    private final DXServerAeronContext aeronContext;
    private final AeronThreadTracker aeronThreadTracker = new AeronThreadTracker();
    private final DirectTopicRegistry topicRegistry;

    public TickDBServer (int port, DXTickDB db) {
        this(port, db, null, null);
    }

    public TickDBServer (int port, DXTickDB db, TLSContext ssl, TransportProperties transportProperties) {
        this(port, db, ssl, transportProperties, false);
    }

    public TickDBServer (int port, DXTickDB db, TLSContext ssl, TransportProperties transportProperties, boolean aeronEnabled) {
        this.port = port;
        this.ssl = ssl;
        this.transportProperties = transportProperties;

        this.aeronContext = DXServerAeronContext.createSimple(aeronEnabled, port);

        this.topicRegistry = TopicRegistryFactory.initRegistryAtQSHome(aeronContext);

        AeronThreadTracker aeronThreadTracker = new AeronThreadTracker();

        // Wrap the DB to provide topics support for local instances
        TimeSource timeSource = DefaultTimeSourceProvider.getTimeSourceForApp("TickDBServer");
        this.db = TopicSupportWrapper.wrap(db, this.aeronContext, this.topicRegistry, aeronThreadTracker, timeSource);

        if (this.aeronContext.isAeronEnabled()) {
            CopyTopicToStreamTaskManager copyTopicToStreamManager = new CopyTopicToStreamTaskManager(db, aeronContext, aeronThreadTracker, topicRegistry);
            copyTopicToStreamManager.startCopyToStreamThreadsForAllTopics();
        }
    }

    public int                  getPort () {
        return port;
    }

    public void                 setPort (int port) {
        this.port = port;
    }

    public InetAddress getAddress() {
        return address;
    }

    public void setAddress(InetAddress address) {
        this.address = address;
    }

    /**
     * @param waitForCompleteShutdown if false this method will only signal shutdown but will not wait for it to finish
     * @throws InterruptedException if method times out while waiting for complete shutdown
     */
    public synchronized void    shutdown (boolean waitForCompleteShutdown)
        throws InterruptedException
    {
        if (server != null) {
            server.close ();

            if (waitForCompleteShutdown)
                server.join ();

            aeronContext.stop();
        }
    }   

    public synchronized void    start () {
        start(new VSConnectionHandler (db, null, aeronContext, aeronThreadTracker, topicRegistry));
    }

    public synchronized void    start (VSConnectionHandler handler) {
        this.aeronContext.start();

        if (server != null)
            throw new IllegalStateException (this + " already started.");

        try {
            server = new VSServer(port, address, ssl, transportProperties);
        } catch (IOException iox) {
            LOGGER.log (Level.SEVERE, "Failed to start", iox);
            return;
        }

        if (port == 0)
            port = server.getLocalPort ();

        server.setConnectionListener (handler);
        server.start ();
    }

    public DXTickDB               getDB () {
        return db;
    }

    @Override
    public String               toString () {
        return ("TickDBServer (" + db.getId () + ")");
    }
}