/*
 * Copyright 2024 EPAM Systems, Inc
 *
 * See the NOTICE file distributed with this work for additional information
 * regarding copyright ownership. Licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.epam.deltix.qsrv.hf.tickdb.lang.compiler.sem.functions;

import com.epam.deltix.dfp.Decimal;
import com.epam.deltix.gflog.api.Log;
import com.epam.deltix.gflog.api.LogFactory;
import com.epam.deltix.computations.api.annotations.*;
import com.epam.deltix.qsrv.hf.pub.md.*;
import com.epam.deltix.computations.functions.DB;
import com.epam.deltix.util.annotations.Bool;
import com.epam.deltix.util.annotations.TimeOfDay;
import com.epam.deltix.util.annotations.TimestampMs;
import com.epam.deltix.util.collections.generated.*;
import com.epam.deltix.util.lang.StringUtils;
import com.epam.deltix.computations.api.annotations.Type;

import javax.annotation.Nonnull;
import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Parameter;
import java.util.*;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;


public class ReflectionUtils {

    private static final Log LOG = LogFactory.getLog(ReflectionUtils.class);

    private static final String GENERIC_OBJECT_PATTERN = "%s(?<nullable>\\?)?";
    private static final String GENERIC_ARRAY_PATTERN = "ARRAY\\(T(?<nullable>\\?)?\\)(?<arrayNullable>\\?)?";

    private static final Pattern OBJECT_PATTERN = Pattern.compile("OBJECT\\((?<classes>[\\.\\w,]+)\\)(?<nullable>\\?)?");
    private static final Pattern OBJECT_ARRAY_PATTERN = Pattern.compile("ARRAY\\(OBJECT\\((?<classes>[\\.\\w,]+)\\)(?<nullable>\\?)?\\)(?<arrayNullable>\\?)?");

    private static Class<? extends Annotation>[] IGNORE_IN_METHODS = new Class[]{
            BuiltInStartTimestampMs.class, BuiltInStartNanoTime.class,
            BuiltInTimestampMs.class, BuiltInNanoTime.class,
            Pool.class, Result.class, DB.class
    };

    //private static final ParameterNameDiscoverer PARAMETER_NAME_DISCOVERER = new DefaultParameterNameDiscoverer();

    public static List<Method> findInitMethods(@Nonnull Class<?> cls) throws IllegalArgumentException {
        return Arrays.stream(getUniqueDeclaredMethods(cls))
                .filter(((Predicate<Method>) ReflectionUtils::isPublicNonStatic)
                        .and(ReflectionUtils::isInitAnnotationPresent)
                        .and(ReflectionUtils::isVoid))
                .collect(Collectors.toList());
    }

    public static Method findComputeMethod(@Nonnull Class<?> cls) throws IllegalArgumentException {
        List<Method> methods = Arrays.stream(getUniqueDeclaredMethods(cls))
                .filter(((Predicate<Method>) ReflectionUtils::isPublicNonStatic)
                        .and(ReflectionUtils::isComputeAnnotationPresent)
                        .and(ReflectionUtils::isVoid))
                .collect(Collectors.toList());
        if (methods.isEmpty()) {
            throw new IllegalArgumentException(
                    String.format("Class %s does not contain any public void methods annotated with %s annotation.",
                            cls.getName(), Compute.class.getName())
            );
        } else if (methods.size() > 1) {
            throw new IllegalArgumentException(
                    String.format("Class %s contains multiple public void methods annotated with %s annotation.",
                            cls.getName(), Compute.class.getName())
            );
        }
        return methods.get(0);
    }

    public static Method findResultMethod(@Nonnull Class<?> cls) throws IllegalArgumentException {
        List<Method> methods = Arrays.stream(getUniqueDeclaredMethods(cls))
                .filter(((Predicate<Method>) ReflectionUtils::isPublicNonStatic)
                        .and(ReflectionUtils::isResultAnnotationPresent))
                .collect(Collectors.toList());
        if (methods.isEmpty()) {
            throw new IllegalArgumentException(
                    String.format("Class %s does not contain any public void methods annotated with %s annotation.",
                            cls.getName(), Result.class.getName())
            );
        } else if (methods.size() > 1) {
            throw new IllegalArgumentException(
                    String.format("Class %s contains multiple public void methods annotated with %s annotation.",
                            cls.getName(), Result.class.getName())
            );
        }
        return methods.get(0);
    }

    public static Method[] getUniqueDeclaredMethods(Class<?> leafClass) {
        Set<Method> methods = org.reflections.ReflectionUtils.getAllMethods(leafClass);

        Set<Method> toRemove = new HashSet<>();

        for (Method method : methods) {

            if (toRemove.contains(method))
                continue;

            for (Method existingMethod : methods) {
                if (existingMethod.equals(method) || toRemove.contains(method))
                    continue;

                if (method.getName().equals(existingMethod.getName()) &&
                        Arrays.equals(method.getParameterTypes(), existingMethod.getParameterTypes())) {
                    // Is this a covariant return type situation?
                    if (existingMethod.getReturnType() != method.getReturnType()) {
                        if (existingMethod.getReturnType().isAssignableFrom(method.getReturnType()))
                            toRemove.add(existingMethod);
                    } else {
                        // remove methods from parent
                        if (existingMethod.getDeclaringClass().isAssignableFrom(method.getDeclaringClass()))
                            toRemove.add(existingMethod);
                    }
                }
            }
        }

        toRemove.forEach(methods::remove);

        return methods.toArray(new Method[0]);
    }


    private static final String CGLIB_RENAMED_METHOD_PREFIX = "CGLIB$";

    /**
     * Determine whether the given method is a CGLIB 'renamed' method,
     * following the pattern "CGLIB$methodName$0".
     * @param renamedMethod the method to check
     */
    public static boolean isCglibRenamedMethod(Method renamedMethod) {
        String name = renamedMethod.getName();
        if (name.startsWith(CGLIB_RENAMED_METHOD_PREFIX)) {
            int i = name.length() - 1;
            while (i >= 0 && Character.isDigit(name.charAt(i))) {
                i--;
            }
            return ((i > CGLIB_RENAMED_METHOD_PREFIX.length()) &&
                    (i < name.length() - 1) && name.charAt(i) == '$');
        }
        return false;
    }

    public static Method findResetMethod(@Nonnull Class<?> cls) throws IllegalArgumentException {
        List<Method> methods = Arrays.stream(getUniqueDeclaredMethods(cls))
                .filter(((Predicate<Method>) ReflectionUtils::isPublicNonStatic)
                        .and(ReflectionUtils::isResetAnnotationPresent)
                        .and(ReflectionUtils::isVoid))
                .collect(Collectors.toList());
        if (methods.isEmpty()) {
            throw new IllegalArgumentException(
                    String.format("Class %s does not contain any public void methods annotated with %s annotation.",
                            cls.getName(), Reset.class.getName())
            );
        } else if (methods.size() > 1) {
            throw new IllegalArgumentException(
                    String.format("Class %s contains multiple public void methods annotated with %s annotation.",
                            cls.getName(), Reset.class.getName())
            );
        }
        return methods.get(0);
    }

    public static int startTimeIndex(@Nonnull Method method) {
        return annotatedParameterIndex(method, BuiltInStartTimestampMs.class);
    }

    public static int startNanoTimeIndex(@Nonnull Method method) {
        return annotatedParameterIndex(method, BuiltInStartNanoTime.class);
    }

    public static int timestampIndex(@Nonnull Method method) {
        return annotatedParameterIndex(method, BuiltInTimestampMs.class);
    }

    public static int nanoTimeIndex(@Nonnull Method method) {
        return annotatedParameterIndex(method, BuiltInNanoTime.class);
    }

    public static int resultIndex(@Nonnull Method method) {
        return annotatedParameterIndex(method, Result.class);
    }

    public static int poolIndex(@Nonnull Method method) {
        return annotatedParameterIndex(method, Pool.class);
    }

    public static int dbIndex(@Nonnull Method method) {
        return annotatedParameterIndex(method, DB.class);
    }

    private static int annotatedParameterIndex(@Nonnull Method method, Class<? extends Annotation> annotation) {
        for (int i = 0; i < method.getParameters().length; i++) {
            if (method.getParameters()[i].isAnnotationPresent(annotation)) {
                return i;
            }
        }
        return -1;
    }

    public static boolean isVoid(Method method) {
        return method.getReturnType() == void.class;
    }

    public static boolean isBoolean(Method method) {
        return method.getReturnType() == boolean.class;
    }

    public static boolean isPublicNonStatic(Method method) {
        int modifiers = method.getModifiers();
        return Modifier.isPublic(modifiers) && !Modifier.isStatic(modifiers) && !isCglibRenamedMethod(method);
    }

    public static boolean isInitAnnotationPresent(Method method) {
        return method.isAnnotationPresent(Init.class);
    }

    public static boolean isComputeAnnotationPresent(Method method) {
        return method.isAnnotationPresent(Compute.class);
    }

    public static boolean isResultAnnotationPresent(Method method) {
        return method.isAnnotationPresent(Result.class);
    }

    public static boolean isResetAnnotationPresent(Method method) {
        return method.isAnnotationPresent(Reset.class);
    }

    public static List<Argument> introspectComputeMethod(Method method, List<GenericType> genericTypes) {
        if (method.isBridge())
            throw new IllegalArgumentException("Bridge methods is not supported: " + method);

        List<Argument> result = new ArrayList<>(method.getParameters().length);

        //String[] names = PARAMETER_NAME_DISCOVERER.getParameterNames(method);

        for (int i = 0, length = method.getParameters().length; i < length; i++) {
            Parameter parameter = method.getParameters()[i];
            if (isValidParameter(parameter))
                result.add(introspectComputeParameter(parameter, parameter.getName(), genericTypes));
        }
        return result;
    }

    // Check that parameter does not contain 'ignore' annotations
    private static boolean isValidParameter(Parameter parameter) {
            for (Class<? extends Annotation> annotationToIgnore : IGNORE_IN_METHODS) {
                if (parameter.isAnnotationPresent(annotationToIgnore))
                return false;
            }

        return true;
        }

    public static List<InitArgument> introspectInitMethod(Method method, List<GenericType> genericTypes) {
        if (method.isBridge())
            throw new IllegalArgumentException("Bridge methods is not supported: " + method);

        List<InitArgument> result = new ArrayList<>(method.getParameters().length);
        //String[] names = PARAMETER_NAME_DISCOVERER.getParameterNames(method);

        for (int i = 0, length = method.getParameters().length; i < length; i++) {
            Parameter parameter = method.getParameters()[i];
            if (parameter.isAnnotationPresent(BuiltInStartTimestampMs.class))
                continue;
            //String name = names[i];
            result.add(introspectInitParameter(parameter, parameter.getName(), genericTypes));
        }
        return result;
    }

    public static Argument introspectResultMethod(Method method, List<GenericType> genericTypes) {
        Class<?> type = method.getReturnType();
        Type typeAnnotation = method.getAnnotation(Type.class);
        DataType dt = null;
        GenericType genericType = null;

        if (typeAnnotation == null) {
            dt = extractType(method);
        } else {
            for (GenericType gType : genericTypes) {
                Pattern objectPattern = genericObjectPattern(gType);
                Matcher objectMatcher = objectPattern.matcher(typeAnnotation.value());
                if (objectMatcher.matches()) {
                    genericType = gType;
                    dt = new ClassDataType(objectMatcher.group("nullable") != null);
                    break;
                }
                Pattern arrayPattern = genericArrayPattern(gType);
                Matcher arrayMatcher = arrayPattern.matcher(typeAnnotation.value());
                if (arrayMatcher.matches()) {
                    genericType = gType;
                    dt = new ArrayDataType(arrayMatcher.group("arrayNullable") != null,
                            new ClassDataType(arrayMatcher.group("nullable") != null));
                    break;
                }
            }
            if (dt == null) {
                dt = extractType(method);
            }
        }
        return new Argument("return", dt, type, genericType);
    }

    public static Argument introspectComputeParameter(Parameter parameter, String parameterName, List<GenericType> genericTypes) {
        Class<?> type = parameter.getType();
        Arg arg = parameter.getAnnotation(Arg.class);
        Type typeAnnotation = parameter.getAnnotation(Type.class);
        DataType dt = null;
        GenericType genericType = null;
        if (typeAnnotation == null) {
            dt = extractType(parameter, false, false);
            genericType = null;
        } else {
            for (GenericType gType : genericTypes) {
                Pattern objectPattern = genericObjectPattern(gType);
                Matcher objectMatcher = objectPattern.matcher(typeAnnotation.value());
                if (objectMatcher.matches()) {
                    genericType = gType;
                    dt = new ClassDataType(objectMatcher.group("nullable") != null);
                    break;
                }
                Pattern arrayPattern = genericArrayPattern(gType);
                Matcher arrayMatcher = arrayPattern.matcher(typeAnnotation.value());
                if (arrayMatcher.matches()) {
                    genericType = gType;
                    dt = new ArrayDataType(arrayMatcher.group("arrayNullable") != null,
                            new ClassDataType(arrayMatcher.group("nullable") != null));
                    break;
                }
            }
            if (dt == null) {
                dt = extractType(parameter, typeAnnotation.value());
            }
        }
        String name;
        if (arg == null) {
            name = parameterName.toUpperCase();
        } else {
            name = (StringUtils.isEmpty(arg.name()) ? parameterName : arg.name()).toUpperCase();
        }
        return new Argument(name, dt, type, genericType);
    }

    public static InitArgument introspectInitParameter(Parameter parameter, String parameterName, List<GenericType> genericTypes) {
        Class<?> type = parameter.getType();
        Arg arg = parameter.getAnnotation(Arg.class);
        Type typeAnnotation = parameter.getAnnotation(Type.class);
        DataType dt = null;
        GenericType genericType = null;
        if (typeAnnotation == null) {
            dt = extractType(parameter, false, false);
            genericType = null;
        } else {
            for (GenericType gType : genericTypes) {
                Pattern objectPattern = genericObjectPattern(gType);
                Matcher objectMatcher = objectPattern.matcher(typeAnnotation.value());
                if (objectMatcher.matches()) {
                    genericType = gType;
                    dt = new ClassDataType(objectMatcher.group("nullable") != null);
                    break;
                }
                Pattern arrayPattern = genericArrayPattern(gType);
                Matcher arrayMapper = arrayPattern.matcher(typeAnnotation.value());
                if (arrayMapper.matches()) {
                    genericType = gType;
                    dt = new ArrayDataType(objectMatcher.group("arrayNullable") != null,
                            new ClassDataType(objectMatcher.group("nullable") != null));
                    break;
                }
            }
            if (dt == null) {
                dt = extractType(parameter, typeAnnotation.value(), false, false);
            }
        }
        String name;
        String defaultValue;
        if (arg == null) {
            name = parameterName.toUpperCase();
            defaultValue = null;
        } else {
            name = (StringUtils.isEmpty(arg.name()) ? parameterName : arg.name()).toUpperCase();
            defaultValue = StringUtils.isEmpty(arg.defaultValue()) ? null : arg.defaultValue();
        }
        return new InitArgument(name, dt, type, genericType, defaultValue);
    }

    private static DataType extractType(Parameter parameter, String typeString, boolean nullable, boolean elementNullable) {
        DataType dt = forName(typeString);
        if (dt != null)
            return dt;
        return extractType(parameter, nullable, elementNullable);
    }

    private static DataType extractType(Parameter parameter, String typeString) {
        return extractType(parameter, typeString, true, true);
    }

    private static DataType extractType(Parameter parameter, boolean nullable, boolean elementNullable) {
        return extractType(parameter.getType(), parameter, nullable, elementNullable);
    }

    private static DataType extractType(Parameter parameter) {
        return extractType(parameter.getType(), parameter, true, true);
    }

    private static DataType extractType(Class<?> type, AnnotatedElement annotatedElement, boolean nullable, boolean elementNullable) {
        if (annotatedElement.isAnnotationPresent(Bool.class)) {
            if (type == byte.class) {
                return TimebaseTypes.getBooleanType(nullable);
            } else if (type == ByteArrayList.class) {
                return TimebaseTypes.getBooleanArrayType(nullable, elementNullable);
            }
        }
        if (annotatedElement.isAnnotationPresent(TimeOfDay.class)) {
            if (type == int.class) {
                return TimebaseTypes.TIME_OF_DAY_CONTAINER.getType(nullable);
            } else if (type == IntegerArrayList.class) {
                return TimebaseTypes.TIME_OF_DAY_CONTAINER.getArrayType(nullable, elementNullable);
            }
        }
        if (type == long.class) {
            if (annotatedElement.isAnnotationPresent(TimestampMs.class)) {
                return TimebaseTypes.DATE_TIME_CONTAINER.getType(nullable);
            } else if (annotatedElement.isAnnotationPresent(Decimal.class)) {
                return TimebaseTypes.DECIMAL64_CONTAINER.getType(nullable);
            }
        }
        if (type == LongArrayList.class) {
            if (annotatedElement.isAnnotationPresent(TimestampMs.class)) {
                return TimebaseTypes.DATE_TIME_CONTAINER.getArrayType(nullable, elementNullable);
            } else if (annotatedElement.isAnnotationPresent(Decimal.class)) {
                return TimebaseTypes.DECIMAL64_CONTAINER.getArrayType(nullable, elementNullable);
            }
        }
        return extractType(type, nullable, elementNullable);
    }

    private static DataType extractType(Class<?> cls, boolean nullable, boolean elementNullable) {
        if (cls == boolean.class) {
            return TimebaseTypes.getBooleanType(false);
        } else if (cls == byte.class) {
            return TimebaseTypes.getIntegerDataType(1, nullable);
        } else if (cls == short.class) {
            return TimebaseTypes.getIntegerDataType(2, nullable);
        } else if (cls == int.class) {
            return TimebaseTypes.getIntegerDataType(4, nullable);
        } else if (cls == long.class) {
            return TimebaseTypes.getIntegerDataType(8, nullable);
        } else if (cls == float.class) {
            return TimebaseTypes.FLOAT32_CONTAINER.getType(nullable);
        } else if (cls == double.class) {
            return TimebaseTypes.FLOAT64_CONTAINER.getType(nullable);
        } else if (cls == char.class) {
            return TimebaseTypes.CHAR_CONTAINER.getType(nullable);
        } else if (cls == CharSequence.class || cls == StringBuilder.class) {
            return TimebaseTypes.UTF8_CONTAINER.getType(true);
        } else if (cls == ByteArrayList.class) {
            return TimebaseTypes.getIntegerArrayDataType(1, nullable, elementNullable);
        } else if (cls == ShortArrayList.class) {
            return TimebaseTypes.getIntegerArrayDataType(2, nullable, elementNullable);
        } else if (cls == IntegerArrayList.class) {
            return TimebaseTypes.getIntegerArrayDataType(4, nullable, elementNullable);
        } else if (cls == LongArrayList.class) {
            return TimebaseTypes.getIntegerArrayDataType(8, nullable, elementNullable);
        } else if (cls == FloatArrayList.class) {
            return TimebaseTypes.FLOAT32_CONTAINER.getArrayType(nullable, elementNullable);
        } else if (cls == DoubleArrayList.class) {
            return TimebaseTypes.FLOAT64_CONTAINER.getArrayType(nullable, elementNullable);
        } else if (cls == CharacterArrayList.class) {
            return TimebaseTypes.CHAR_CONTAINER.getArrayType(nullable, elementNullable);
        } else {
            throw new IllegalArgumentException("Cannot extract DataType from " + cls.getName());
        }
    }

    public static String extractId(Class<?> cls) {
        String name;
        if (cls.isAnnotationPresent(Function.class)) {
            name = cls.getAnnotation(Function.class).value();
        } else {
            throw new IllegalArgumentException(String.format("%s is not annotated nor with %s", cls.getName(), Function.class));
        }
        return StringUtils.isEmpty(name) ? cls.getSimpleName() : name;
    }

    public static String extractId(Method method) {
        String name;
        if (method.isAnnotationPresent(Function.class)) {
            name = method.getAnnotation(Function.class).value();
        } else {
            throw new IllegalArgumentException(String.format("%s is not annotated nor with %s", method.getName(),
                    Function.class));
        }
        return StringUtils.isEmpty(name) ? method.getName() : name;
    }

    public static List<GenericType> extractGenericTypes(Class<?> cls) {
        if (cls.isAnnotationPresent(Generic.class)) {
            return Arrays.stream(cls.getAnnotation(Generic.class).value())
                    .map(p -> new GenericType(p.name()))
                    .collect(Collectors.toList());
        } else if (cls.isAnnotationPresent(GenericParameter.class)) {
            return Collections.singletonList(new GenericType(cls.getAnnotation(GenericParameter.class).name()));
        } else {
            return Collections.emptyList();
        }
    }

    public static DataType extractType(Type typeAnnotation) {
        String type = typeAnnotation.value();
        DataType dt = forName(type);
        if (dt == null)
            throw new IllegalArgumentException("Unrecognized type: " + type);
        return dt;
    }

    public static String extractTypeName(Type typeAnnotation) {
        String type = typeAnnotation.value();
        Matcher matcher = OBJECT_ARRAY_PATTERN.matcher(type);
        if (matcher.matches()) {
            String className = matcher.group("classes");
            try {
                Class<?> cls = Class.forName(className);
                return ObjectArrayList.class.getName() + "<" + cls.getName() + ">";
            } catch (ClassNotFoundException exc) {
                LOG.info().append("Class ").append(className).append(" not found, so type '")
                        .append(type).appendLast("' couldn't be parsed and created.");
            }
        }
        return null;
    }

    public static String extractTypeName(Method method) {
        if (method.isAnnotationPresent(Type.class)) {
            return extractTypeName(method.getAnnotation(Type.class));
        } else if (method.getReturnType() == boolean.class) {
            for (Parameter parameter : method.getParameters()) {
                if (parameter.isAnnotationPresent(Result.class)) {
                    if (parameter.isAnnotationPresent(Type.class)) {
                        return extractTypeName(parameter.getAnnotation(Type.class));
                    } else {
                        return null;
                    }
                }
            }
            throw new IllegalArgumentException(String.format("Method %s returns boolean, however no arguments " +
                    "annotated with %s annotation were found.", method, Result.class.getName()));
        } else {
            return null;
        }
    }

    public static DataType extractType(Method method) {
        if (method.isAnnotationPresent(Type.class)) {
            return extractType(method.getAnnotation(Type.class));
        } else if (method.getReturnType() == boolean.class) {
            for (Parameter parameter : method.getParameters()) {
                if (parameter.isAnnotationPresent(Result.class)) {
                    if (parameter.isAnnotationPresent(Type.class)) {
                        return extractType(parameter.getAnnotation(Type.class));
                    } else {
                        return extractType(parameter);
                    }
                }
            }
            throw new IllegalArgumentException(String.format("Method %s returns boolean, however no arguments " +
                    "annotated with %s annotation were found.", method, Result.class.getName()));
        } else {
            return extractType(method.getReturnType(), method, true, true);
        }
    }

    public static DataType forName(String name) {
        name = name.trim();

        DataType result = TimebaseTypes.TYPES_MAP.get(name);
        if (result != null)
            return result;

        return parseObjectType(name);
    }

    private static DataType parseObjectType(String s) {
        Matcher matcher = OBJECT_PATTERN.matcher(s);
        if (matcher.matches()) {
            boolean nullable = matcher.group("nullable") != null;
            String classes = matcher.group("classes");
            RecordClassDescriptor[] descriptors = introspectClasses(s, classes);
            if (descriptors != null) {
                return new ClassDataType(nullable, descriptors);
            }
        }
        matcher = OBJECT_ARRAY_PATTERN.matcher(s);
        if (matcher.matches()) {
            boolean arrayNullable = matcher.group("arrayNullable") != null;
            boolean nullable = matcher.group("nullable") != null;
            String classes = matcher.group("classes");
            RecordClassDescriptor[] descriptors = introspectClasses(s, classes);
            if (descriptors != null) {
                return new ArrayDataType(arrayNullable, new ClassDataType(nullable, descriptors));
            }
        }
        return null;
    }

    private static RecordClassDescriptor[] introspectClasses(String type, String classes) {
        String[] classNamesStr = classes.split(",");
        List<RecordClassDescriptor> descriptors = new ArrayList<>();

        String currentClass = null;
        try {
        for (String className : classNamesStr) {
                currentClass = className;
            if (className != null && !className.trim().isEmpty()) {
                Class<?> cls = Class.forName(className.trim());
                descriptors.add(
                    Introspector.createEmptyMessageIntrospector().introspectMemberClass("", cls)
                );
            }
        }
        } catch (ClassNotFoundException exc) {
            LOG.info().append("Class ").append(currentClass.trim()).append(" not found, so type '")
                .append(type).appendLast("' couldn't be parsed and created.");
            return null;
        } catch (Introspector.IntrospectionException e) {
            LOG.info().append("Error while introspecting class ").append(currentClass)
                .append(". Message: ").appendLast(e.getMessage());
            return null;
        }

        if (descriptors.size() == 0) {
            LOG.info().append("Error while introspecting type ").append(type)
                .append(". Classes not found.").commit();
            return null;
        }

        return descriptors.toArray(new RecordClassDescriptor[0]);
    }

    private static Pattern genericObjectPattern(GenericType genericType) {
        return Pattern.compile(String.format(GENERIC_OBJECT_PATTERN, genericType.getId()));
    }

    private static Pattern genericArrayPattern(GenericType genericType) {
        return Pattern.compile(String.format(GENERIC_ARRAY_PATTERN, genericType.getId()));
    }

}
