/*
 * Copyright 2023 EPAM Systems, Inc
 *
 * See the NOTICE file distributed with this work for additional information
 * regarding copyright ownership. Licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.epam.deltix.snmp.parser;

import java_cup.runtime.*;
import com.epam.deltix.util.parsers.*;

/**
 * ASN.1 Lexer.
 */

final class Lexer implements java_cup.runtime.Scanner {

  /** This character denotes the end of file */
  public static final int YYEOF = -1;

  /** initial size of the lookahead buffer */
  private static final int ZZ_BUFFERSIZE = 16384;

  /** lexical states */
  public static final int YYINITIAL = 0;

  /**
   * ZZ_LEXSTATE[l] is the state in the DFA for the lexical state l
   * ZZ_LEXSTATE[l+1] is the state in the DFA for the lexical state l
   *                  at the beginning of a line
   * l is of the form l = 2*k, k a non negative integer
   */
  private static final int ZZ_LEXSTATE[] = { 
     0, 0
  };

  /** 
   * Translates characters to character classes
   */
  private static final String ZZ_CMAP_PACKED = 
    "\11\0\1\4\1\2\1\0\1\3\1\1\22\0\1\4\1\41\1\22"+
    "\4\0\1\16\1\34\1\35\1\12\1\0\1\33\1\10\1\14\1\11"+
    "\1\13\1\7\10\7\1\23\1\36\1\31\1\24\1\32\1\0\1\37"+
    "\1\43\1\17\1\47\1\20\1\15\1\60\1\50\1\21\1\52\1\65"+
    "\1\5\1\53\1\57\1\45\1\55\1\51\1\62\1\61\1\44\1\46"+
    "\1\56\1\63\1\67\1\64\1\54\1\66\1\25\1\0\1\26\1\42"+
    "\2\0\4\6\1\6\25\6\1\27\1\40\1\30\uff82\0";

  /** 
   * Translates characters to character classes
   */
  private static final char [] ZZ_CMAP = zzUnpackCMap(ZZ_CMAP_PACKED);

  /** 
   * Translates DFA states to action switch labels.
   */
  private static final int [] ZZ_ACTION = zzUnpackAction();

  private static final String ZZ_ACTION_PACKED_0 =
    "\1\0\1\1\2\2\1\3\1\4\1\5\1\6\1\1"+
    "\1\5\1\7\1\3\1\10\2\3\1\11\1\12\1\13"+
    "\1\14\1\15\1\16\1\17\1\20\1\21\1\22\1\23"+
    "\1\24\1\25\1\26\1\27\1\30\1\31\20\3\4\0"+
    "\1\32\4\3\1\33\3\3\1\0\1\34\1\0\1\35"+
    "\1\36\35\3\1\37\13\3\2\0\1\40\1\41\5\3"+
    "\1\42\4\3\1\43\1\3\1\44\4\3\1\45\42\3"+
    "\1\46\1\47\11\3\1\2\6\3\1\50\16\3\1\51"+
    "\3\3\1\52\1\3\1\53\1\3\1\54\30\3\2\0"+
    "\3\3\1\55\1\56\3\3\1\57\6\3\1\60\22\3"+
    "\1\0\4\3\1\61\1\0\4\3\1\62\4\3\1\0"+
    "\1\3\1\63\2\3\1\64\2\3\1\65\4\3\1\66"+
    "\16\3\1\67\1\3\1\70\1\71\1\0\3\3\1\72"+
    "\1\73\1\74\4\3\1\75\16\3\1\76\6\3\1\0"+
    "\1\77\4\3\1\0\4\3\1\100\1\101\1\3\1\102"+
    "\1\103\16\3\1\104\1\105\2\3\1\106\1\3\1\107"+
    "\1\3\1\110\1\3\1\0\1\111\7\3\1\0\7\3"+
    "\1\112\1\113\2\3\1\0\2\3\1\114\1\115\1\116"+
    "\1\3\1\0\1\3\1\0\3\3\1\0\1\3\1\117"+
    "\1\120\4\3\1\121\11\3\1\122\14\3\1\123\1\3"+
    "\1\124\1\0\10\3\1\125\6\3\1\126\1\127\1\130"+
    "\1\3\1\131\1\132\10\3\1\133\3\3\1\134\5\3"+
    "\1\0\1\135\1\136\4\3\1\137\1\140\2\3\1\141"+
    "\7\3\1\142\10\3\1\143\3\3\1\144\3\3\1\145"+
    "\1\146\1\3\1\147\4\3\1\150\1\3\1\0\1\151"+
    "\2\3\1\152\16\3\1\153\7\3\1\154\1\155\2\3"+
    "\1\156\5\3\1\157\2\3\1\160\2\3\1\161\1\162"+
    "\1\163\1\164\1\165";

  private static int [] zzUnpackAction() {
    int [] result = new int[588];
    int offset = 0;
    offset = zzUnpackAction(ZZ_ACTION_PACKED_0, offset, result);
    return result;
  }

  private static int zzUnpackAction(String packed, int offset, int [] result) {
    int i = 0;       /* index in packed string  */
    int j = offset;  /* index in unpacked array */
    int l = packed.length();
    while (i < l) {
      int count = packed.charAt(i++);
      int value = packed.charAt(i++);
      do result[j++] = value; while (--count > 0);
    }
    return j;
  }


  /** 
   * Translates a state to a row index in the transition table
   */
  private static final int [] ZZ_ROWMAP = zzUnpackRowMap();

  private static final String ZZ_ROWMAP_PACKED_0 =
    "\0\0\0\70\0\160\0\70\0\250\0\340\0\u0118\0\u0150"+
    "\0\u0188\0\70\0\u01c0\0\u01f8\0\70\0\u0230\0\u0268\0\u02a0"+
    "\0\u02d8\0\70\0\u0310\0\u0348\0\70\0\70\0\70\0\70"+
    "\0\70\0\70\0\70\0\70\0\70\0\70\0\70\0\70"+
    "\0\u0380\0\u03b8\0\u03f0\0\u0428\0\u0460\0\u0498\0\u04d0\0\u0508"+
    "\0\u0540\0\u0578\0\u05b0\0\u05e8\0\u0620\0\u0658\0\u0690\0\u06c8"+
    "\0\u0700\0\u0738\0\u0770\0\u07a8\0\u07e0\0\u0818\0\u0850\0\u0888"+
    "\0\u08c0\0\250\0\u08f8\0\u0930\0\u0968\0\u02a0\0\u09a0\0\u09d8"+
    "\0\70\0\70\0\u0a10\0\u0a48\0\u0a80\0\u0ab8\0\u0af0\0\u0b28"+
    "\0\u0b60\0\u0b98\0\u0bd0\0\u0c08\0\u0c40\0\u0c78\0\u0cb0\0\u0ce8"+
    "\0\u0d20\0\u0d58\0\u0d90\0\u0dc8\0\u0e00\0\u0e38\0\u0e70\0\u0ea8"+
    "\0\u0ee0\0\u0f18\0\u0f50\0\u0f88\0\u0fc0\0\u0ff8\0\u1030\0\250"+
    "\0\u1068\0\u10a0\0\u10d8\0\u1110\0\u1148\0\u1180\0\u11b8\0\u11f0"+
    "\0\u1228\0\u1260\0\u1298\0\u12d0\0\u1308\0\70\0\250\0\u1340"+
    "\0\u1378\0\u13b0\0\u13e8\0\u1420\0\u1458\0\u1490\0\u14c8\0\u1500"+
    "\0\u1538\0\70\0\u1570\0\250\0\u15a8\0\u15e0\0\u1618\0\u1650"+
    "\0\250\0\u1688\0\u16c0\0\u16f8\0\u1730\0\u1768\0\u17a0\0\u17d8"+
    "\0\u1810\0\u1848\0\u1880\0\u18b8\0\u18f0\0\u1928\0\u1960\0\u1998"+
    "\0\u19d0\0\u1a08\0\u1a40\0\u1a78\0\u1ab0\0\u1ae8\0\u1b20\0\u1b58"+
    "\0\u1b90\0\u1bc8\0\u1c00\0\u1c38\0\u1c70\0\u1ca8\0\u1ce0\0\u1d18"+
    "\0\u1d50\0\u1d88\0\u1dc0\0\u1df8\0\u1e30\0\u1e68\0\u1ea0\0\u1ed8"+
    "\0\u1f10\0\u1f48\0\u1f80\0\u1fb8\0\u1ff0\0\u2028\0\u2060\0\u2098"+
    "\0\u20d0\0\u2108\0\u2140\0\u2178\0\u21b0\0\250\0\u21e8\0\u2220"+
    "\0\u2258\0\u2290\0\u22c8\0\u2300\0\u2338\0\u2370\0\u23a8\0\u23e0"+
    "\0\u2418\0\u2450\0\u2488\0\u24c0\0\250\0\u24f8\0\u2530\0\u2568"+
    "\0\250\0\u25a0\0\250\0\u25d8\0\250\0\u2610\0\u2648\0\u2680"+
    "\0\u26b8\0\u26f0\0\u2728\0\u2760\0\u2798\0\u27d0\0\u2808\0\u2840"+
    "\0\u2878\0\u28b0\0\u28e8\0\u2920\0\u2958\0\u2990\0\u29c8\0\u2a00"+
    "\0\u2a38\0\u2a70\0\u2aa8\0\u2ae0\0\u2b18\0\u2b50\0\u2b88\0\u2bc0"+
    "\0\u2bf8\0\u2c30\0\250\0\250\0\u2c68\0\u2ca0\0\u2cd8\0\250"+
    "\0\u2d10\0\u2d48\0\u2d80\0\u2db8\0\u2df0\0\u2e28\0\250\0\u2e60"+
    "\0\u2e98\0\u2ed0\0\u2f08\0\u2f40\0\u2f78\0\u2fb0\0\u2fe8\0\u3020"+
    "\0\u3058\0\u3090\0\u30c8\0\u3100\0\u3138\0\u3170\0\u31a8\0\u31e0"+
    "\0\u3218\0\u3250\0\u3288\0\u32c0\0\u32f8\0\u3330\0\250\0\u3368"+
    "\0\u33a0\0\u33d8\0\u3410\0\u3448\0\250\0\u3480\0\u34b8\0\u34f0"+
    "\0\u3528\0\u3560\0\u3598\0\250\0\u35d0\0\u3608\0\250\0\u3640"+
    "\0\u3678\0\250\0\u36b0\0\u36e8\0\u3720\0\u3758\0\250\0\u3790"+
    "\0\u37c8\0\u3800\0\u3838\0\u3870\0\u38a8\0\u38e0\0\u3918\0\u3950"+
    "\0\u3988\0\u39c0\0\u39f8\0\u3a30\0\u3a68\0\250\0\u3aa0\0\250"+
    "\0\250\0\u3ad8\0\u3b10\0\u3b48\0\u3b80\0\250\0\250\0\250"+
    "\0\u3bb8\0\u3bf0\0\u3c28\0\u3c60\0\250\0\u3c98\0\u3cd0\0\u3d08"+
    "\0\u3d40\0\u3d78\0\u3db0\0\u3de8\0\u3e20\0\u3e58\0\u3e90\0\u3ec8"+
    "\0\u3f00\0\u3f38\0\u3f70\0\u3fa8\0\u3fe0\0\u4018\0\u4050\0\u4088"+
    "\0\u40c0\0\u40f8\0\u4130\0\u4168\0\u41a0\0\u41d8\0\u4210\0\u4248"+
    "\0\u4280\0\u42b8\0\u42f0\0\u4328\0\u4360\0\250\0\250\0\u4398"+
    "\0\250\0\250\0\u43d0\0\u4408\0\u4440\0\u4478\0\u44b0\0\u44e8"+
    "\0\u4520\0\u4558\0\u4590\0\u45c8\0\u4600\0\u4638\0\u4670\0\u46a8"+
    "\0\250\0\250\0\u46e0\0\u4718\0\250\0\u4750\0\250\0\u4788"+
    "\0\250\0\u47c0\0\u47f8\0\250\0\u4830\0\u4868\0\u48a0\0\u48d8"+
    "\0\u4910\0\u4948\0\u4980\0\u49b8\0\u49f0\0\u4a28\0\u4a60\0\u4a98"+
    "\0\u4ad0\0\u4b08\0\u4b40\0\250\0\250\0\u4b78\0\u4bb0\0\u4be8"+
    "\0\u4c20\0\u4c58\0\250\0\250\0\250\0\u4c90\0\u4cc8\0\u4d00"+
    "\0\u4d38\0\u4d70\0\u4da8\0\u4de0\0\u4e18\0\u4e50\0\250\0\250"+
    "\0\u4e88\0\u4ec0\0\u4ef8\0\u4f30\0\250\0\u4f68\0\u4fa0\0\u4fd8"+
    "\0\u5010\0\u5048\0\u5080\0\u50b8\0\u50f0\0\u5128\0\250\0\u5160"+
    "\0\u5198\0\u51d0\0\u5208\0\u5240\0\u5278\0\u52b0\0\u52e8\0\u5320"+
    "\0\u5358\0\u5390\0\u53c8\0\250\0\u5400\0\u5438\0\u5470\0\u54a8"+
    "\0\u54e0\0\u5518\0\u5550\0\u5588\0\u55c0\0\u55f8\0\u5630\0\250"+
    "\0\u5668\0\u56a0\0\u56d8\0\u5710\0\u5748\0\u5780\0\250\0\250"+
    "\0\250\0\u57b8\0\250\0\250\0\u57f0\0\u5828\0\u5860\0\u5898"+
    "\0\u58d0\0\u5908\0\u5940\0\u5978\0\250\0\u59b0\0\u59e8\0\u5a20"+
    "\0\250\0\u5a58\0\u5a90\0\u5ac8\0\u5b00\0\u5b38\0\u5b70\0\250"+
    "\0\250\0\u5ba8\0\u5be0\0\u5c18\0\u5c50\0\250\0\250\0\u5c88"+
    "\0\u5cc0\0\250\0\u5cf8\0\u5d30\0\u5d68\0\u5da0\0\u5dd8\0\u5e10"+
    "\0\u5e48\0\250\0\u5e80\0\u5eb8\0\u5ef0\0\u5f28\0\u5f60\0\u5f98"+
    "\0\u5fd0\0\u6008\0\250\0\u6040\0\u6078\0\u60b0\0\250\0\u60e8"+
    "\0\u6120\0\u6158\0\250\0\250\0\u6190\0\250\0\u61c8\0\u6200"+
    "\0\u6238\0\u6270\0\250\0\u62a8\0\u62e0\0\250\0\u6318\0\u6350"+
    "\0\250\0\u6388\0\u63c0\0\u63f8\0\u6430\0\u6468\0\u64a0\0\u64d8"+
    "\0\u6510\0\u6548\0\u6580\0\u65b8\0\u65f0\0\u6628\0\u6660\0\250"+
    "\0\u6698\0\u66d0\0\u6708\0\u6740\0\u6778\0\u67b0\0\u67e8\0\250"+
    "\0\250\0\u6820\0\u6858\0\250\0\u6890\0\u68c8\0\u6900\0\u6938"+
    "\0\u6970\0\250\0\u69a8\0\u69e0\0\250\0\u6a18\0\u6a50\0\250"+
    "\0\250\0\250\0\250\0\250";

  private static int [] zzUnpackRowMap() {
    int [] result = new int[588];
    int offset = 0;
    offset = zzUnpackRowMap(ZZ_ROWMAP_PACKED_0, offset, result);
    return result;
  }

  private static int zzUnpackRowMap(String packed, int offset, int [] result) {
    int i = 0;  /* index in packed string  */
    int j = offset;  /* index in unpacked array */
    int l = packed.length();
    while (i < l) {
      int high = packed.charAt(i++) << 16;
      result[j++] = high | packed.charAt(i++);
    }
    return j;
  }

  /** 
   * The transition table of the DFA
   */
  private static final int [] ZZ_TRANS = zzUnpackTrans();

  private static final String ZZ_TRANS_PACKED_0 =
    "\1\2\1\3\3\4\1\5\1\6\1\7\1\10\1\11"+
    "\1\2\1\12\1\13\1\14\1\15\1\16\1\17\1\5"+
    "\1\20\1\21\1\22\1\23\1\24\1\25\1\26\1\27"+
    "\1\30\1\31\1\32\1\33\1\34\1\35\1\36\1\37"+
    "\1\40\1\41\1\42\1\43\1\44\1\45\1\46\1\47"+
    "\1\50\1\51\1\5\1\52\1\53\1\54\1\55\1\56"+
    "\1\5\1\57\3\5\1\60\72\0\1\4\72\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\25\5\5\0\3\6\1\62\2\0\1\6\1\0\1\6"+
    "\1\0\3\6\21\0\25\6\7\0\1\7\3\0\1\7"+
    "\64\0\1\63\71\0\1\64\71\0\1\65\60\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\2\5\1\66\16\5\1\67\3\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\70\1\0\3\5\21\0\7\5"+
    "\1\71\1\5\1\72\1\73\12\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\74\1\0\3\5\21\0\7\5"+
    "\1\75\15\5\22\76\1\77\45\76\23\0\1\100\71\0"+
    "\1\101\70\0\1\102\46\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\1\103\2\5\21\0\2\5\1\104"+
    "\1\5\1\105\1\106\1\107\4\5\1\110\11\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\111\1\0\3\5"+
    "\21\0\3\5\1\112\3\5\1\113\1\5\1\114\1\5"+
    "\1\115\11\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\12\5\1\116\1\117\11\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\120\1\0"+
    "\3\5\21\0\1\121\15\5\1\122\6\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\2\5\1\123"+
    "\21\0\12\5\1\124\3\5\1\125\6\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\16\5\1\126\6\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\10\5\1\127\5\5"+
    "\1\130\6\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\1\5\1\131\1\5\21\0\2\5\1\132"+
    "\11\5\1\133\10\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\1\134\24\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\1\135"+
    "\2\5\21\0\4\5\1\136\1\5\1\137\6\5\1\140"+
    "\1\141\6\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\2\5\1\142\22\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\1\143\6\5\1\144\2\5\1\145\12\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\1\146\15\5\1\147\6\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\150\1\0\3\5\21\0\25\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\1\151\24\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\7\5\1\152"+
    "\6\5\1\153\6\5\5\0\3\5\3\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\25\5\5\0\3\6\3\0"+
    "\1\6\1\0\1\6\1\0\3\6\21\0\25\6\1\63"+
    "\1\3\2\4\4\63\1\154\57\63\12\64\1\155\55\64"+
    "\14\0\1\156\60\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\1\5\1\157\1\5\21\0\3\5\1\160"+
    "\7\5\1\161\11\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\3\5\1\162\2\5"+
    "\1\163\16\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\5\5\1\164\17\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\3\5\1\165\21\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\12\5\1\166"+
    "\12\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\1\5\1\167\13\5\1\170\7\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\1\5\1\171\23\5\22\0\1\76\71\0"+
    "\1\172\50\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\1\5\1\173\23\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\11\5\1\174\13\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\4\5\1\175\20\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\176\1\0"+
    "\3\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\6\5\1\177\16\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\5\5\1\200\17\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\3\5"+
    "\1\201\13\5\1\202\5\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\1\203\15\5"+
    "\1\204\6\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\23\5\1\205\1\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\2\5\1\206\22\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\6\5\1\207"+
    "\16\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\3\5\1\210\21\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\10\5\1\211\14\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\21\5\1\212\3\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\5\5\1\213\17\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\1\214"+
    "\12\5\1\215\11\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\12\5\1\216\12\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\2\5\1\217\11\5\1\220\10\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\221\1\0\3\5"+
    "\21\0\25\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\12\5\1\222\12\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\13\5\1\223\11\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\224\1\0\3\5\21\0\7\5\1\225"+
    "\2\5\1\226\12\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\227\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\1\5\1\230"+
    "\1\5\21\0\3\5\1\231\1\232\20\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\6\5\1\233\16\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\1\5\1\234\23\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\22\5\1\235\2\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\3\5"+
    "\1\236\21\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\237\21\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\5\5\1\240\17\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\7\5\1\241"+
    "\15\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\2\5\1\242\1\5\1\243\14\5"+
    "\1\244\3\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\2\5\1\245\22\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\1\5"+
    "\1\246\1\5\21\0\25\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\10\5\1\247"+
    "\14\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\12\5\1\250\12\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\251\14\5\1\252\2\5\1\253\4\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\16\5\1\254\6\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\3\5\1\255\21\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\7\5\1\256\15\5\1\63\1\3\2\4"+
    "\4\63\1\257\57\63\11\64\1\4\1\155\55\64\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\260\1\0\3\5"+
    "\21\0\25\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\14\5\1\261\10\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\262\1\0\3\5"+
    "\21\0\25\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\10\5\1\263\1\5\1\264"+
    "\12\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\7\5\1\265\15\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\5\1\266\23\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\10\5\1\267\14\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\4\5\1\270\20\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\1\271"+
    "\6\5\1\272\10\5\1\273\4\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\6\5"+
    "\1\274\16\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\275\1\0\3\5\21\0\25\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\276\1\0\3\5\21\0\25\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\2\5\1\277\22\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\10\5"+
    "\1\300\14\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\14\5\1\301\10\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\13\5\1\302\11\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\3\5\1\303"+
    "\21\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\7\5\1\304\15\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\305\1\0\3\5\21\0"+
    "\25\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\3\5\1\306\21\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\6\5\1\307\16\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\7\5\1\310\15\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\10\5\1\311\14\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\3\5"+
    "\1\312\21\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\5\1\313\23\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\6\5\1\314\16\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\315\1\0\3\5\21\0\25\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\7\5\1\316\15\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\3\5\1\317"+
    "\21\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\6\5\1\320\16\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\321\24\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\13\5\1\322\11\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\1\5\1\323\23\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\1\5\1\324"+
    "\23\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\20\5\1\325\4\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\1\5\1\326"+
    "\1\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\2\5\1\327\22\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\330\1\0"+
    "\3\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\331\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\10\5\1\332\14\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\10\5\1\333\1\5"+
    "\1\334\12\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\335\21\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\336\1\0\3\5"+
    "\21\0\25\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\337\1\0\3\5\21\0\25\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\7\5"+
    "\1\340\15\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\341\24\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\3\5\1\342\14\5\1\343\4\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\1\5\1\344\1\5"+
    "\21\0\25\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\16\5\1\345\6\5\5\0"+
    "\3\5\1\346\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\25\5\5\0\3\5\1\347\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\13\5\1\350\11\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\13\5\1\351\11\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\1\5\1\352"+
    "\23\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\14\5\1\353\10\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\10\5\1\354\14\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\355\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\7\5\1\356\15\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\7\5\1\357\15\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\2\5\1\360\21\0\25\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\3\5\1\361"+
    "\21\5\10\0\1\4\64\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\16\5\1\362\6\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\363\1\0"+
    "\3\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\16\5\1\364\6\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\7\5\1\365\15\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\16\5"+
    "\1\366\6\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\2\5\1\367\22\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\370\1\0\3\5"+
    "\21\0\25\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\16\5\1\371\6\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\13\5\1\372\11\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\2\5\1\373"+
    "\22\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\1\374\24\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\10\5"+
    "\1\375\14\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\2\5\1\376\22\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\1\5\1\377\23\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\3\5\1\u0100"+
    "\21\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\7\5\1\u0101\15\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\u0102\1\0\3\5\21\0"+
    "\25\5\5\0\3\5\1\61\2\0\1\5\1\0\1\u0103"+
    "\1\0\3\5\21\0\25\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\13\5\1\u0104"+
    "\11\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\2\5\1\u0105\22\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\u0106\24\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\12\5\1\u0107\12\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\15\5\1\u0108\7\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\13\5\1\u0109"+
    "\11\5\5\0\3\5\1\u010a\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\25\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\4\5\1\u010b"+
    "\20\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\1\u010c\24\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\12\5"+
    "\1\u010d\12\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\u010e\21\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\6\5\1\u010f\16\5\5\0\3\5\1\u0110\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\25\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\u0111\1\0\3\5"+
    "\21\0\25\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\u0112\24\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\13\5\1\u0113\11\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\3\5\1\u0114\21\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\21\5\1\u0115\3\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\5\5"+
    "\1\u0116\17\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\13\5\1\u0117\11\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\7\5\1\u0118\15\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\16\5\1\u0119"+
    "\6\5\5\0\3\5\1\u011a\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\25\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\4\5\1\u011b"+
    "\20\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\3\5\1\u011c\21\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\12\5\1\u011d\12\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\2\5\1\u011e\22\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\1\5\1\u011f\23\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\u0120\1\0\3\5\21\0\25\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\1\u0121\24\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\12\5\1\u0122"+
    "\12\5\5\0\3\5\3\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\1\u0123\24\5\5\0\3\5\3\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\1\u0124\24\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\1\5\1\u0125\23\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\10\5\1\u0126"+
    "\14\5\5\0\3\5\1\61\2\0\1\5\1\0\1\u0127"+
    "\1\0\3\5\21\0\25\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\16\5\1\u0128"+
    "\6\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\1\5\1\u0129\23\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\u012a\24\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\u012b\1\0\3\5\21\0\25\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\6\5"+
    "\1\u012c\16\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\16\5\1\u012d\6\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\2\5\1\u012e\22\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\4\5\1\u012f"+
    "\20\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\3\5\1\u0130\21\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\u0131\24\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\7\5\1\u0132\15\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\10\5\1\u0133\14\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\u0134\1\0\3\5\21\0\7\5\1\u0135"+
    "\15\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\10\5\1\u0136\14\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\u0137\24\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\u0138\21\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\1\5\1\u0139\23\5\5\0\3\5\1\u013a\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\25\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\4\5\1\u013b\20\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\2\5\1\u013c"+
    "\22\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\2\5\1\u013d\22\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\5\1\u013e\23\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\5\5\1\u013f\17\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\21\5\1\u0140\3\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\16\5"+
    "\1\u0141\6\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\7\5\1\u0142\15\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\1\u0143\24\5\5\0\3\5\3\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\u0144\21\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\u0145\1\0\3\5"+
    "\21\0\25\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\4\5\1\u0146\20\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\2\5\1\u0147\22\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\7\5\1\u0148"+
    "\15\5\5\0\3\5\3\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\7\5\1\u0149\15\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\2\5"+
    "\1\u014a\22\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\u014b\21\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\4\5\1\u014c\20\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\7\5\1\u014d"+
    "\15\5\5\0\3\5\1\61\2\0\1\5\1\0\1\u014e"+
    "\1\0\3\5\21\0\25\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\1\5\1\u014f\1\5\21\0"+
    "\25\5\5\0\3\5\1\61\2\0\1\5\1\0\1\u0150"+
    "\1\0\3\5\21\0\4\5\1\u0151\20\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\3\5\1\u0152\21\5\5\0\3\5\3\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\13\5\1\u0153\11\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\3\5\1\u0154\21\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\2\5\1\u0155"+
    "\22\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\7\5\1\u0156\15\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\16\5\1\u0157\6\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\3\5\1\u0158\21\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\4\5\1\u0159\20\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\4\5"+
    "\1\u015a\20\5\5\0\3\5\1\u015b\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\25\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\u015c\1\0\3\5\21\0\25\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\u015d\1\0"+
    "\3\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\7\5\1\u015e\15\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\1\u015f\2\5\21\0\3\5\1\u0160\21\5\5\0\3\5"+
    "\1\u0161\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\25\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\16\5\1\u0162\6\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\u0163\24\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\u0164\24\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\7\5\1\u0165\15\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\1\5\1\u0166\23\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\2\5\1\u0167\22\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\6\5"+
    "\1\u0168\16\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\u0169\21\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\1\5"+
    "\1\u016a\1\5\21\0\25\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\3\5\1\u016b"+
    "\21\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\11\5\1\u016c\13\5\5\0\3\5"+
    "\3\0\1\5\1\0\1\5\1\0\3\5\21\0\4\5"+
    "\1\u016d\20\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\u016e\24\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\3\5\1\u016f\21\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\4\5\1\u0170\20\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\3\5\1\u0171\21\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\4\5"+
    "\1\u0172\20\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\10\5\1\u0173\14\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\11\5\1\u0174\13\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\3\5\1\u0175"+
    "\21\5\5\0\3\5\1\61\2\0\1\5\1\0\1\u0176"+
    "\1\0\3\5\21\0\25\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\12\5\1\u0177"+
    "\12\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\2\5\1\u0178\22\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\3\5\1\u0179\21\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\u017a\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\3\5\1\u017b\21\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\15\5\1\u017c\7\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\16\5\1\u017d\6\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\u017e\1\0\3\5\21\0\25\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\1\5\1\u017f\1\5\21\0\25\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\7\5"+
    "\1\u0180\15\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\5\1\u0181\23\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\6\5\1\u0182\16\5\5\0\3\5\1\u0183\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\1\5\1\u0184"+
    "\23\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\1\u0185\24\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\23\5"+
    "\1\u0186\1\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\5\1\u0187\23\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\12\5\1\u0188\12\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\4\5\1\u0189"+
    "\20\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\4\5\1\u018a\20\5\5\0\3\5"+
    "\3\0\1\5\1\0\1\5\1\0\3\5\21\0\7\5"+
    "\1\u018b\15\5\5\0\3\5\1\u018c\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\25\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\2\5"+
    "\1\u018d\22\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\12\5\1\u018e\12\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\10\5\1\u018f\14\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\7\5\1\u0190"+
    "\15\5\5\0\3\5\3\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\1\5\1\u0191\23\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\7\5"+
    "\1\u0192\15\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\u0193\21\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\10\5\1\u0194\14\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\3\5\1\u0195"+
    "\21\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\3\5\1\u0196\21\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\7\5\1\u0197\15\5\5\0\3\5\1\u0198\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\u0199\24\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\u019a\21\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\1\5\1\u019b\23\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\u019c\1\0\3\5\21\0\25\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\1\5\1\u019d\23\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\1\u019e\24\5"+
    "\5\0\3\5\1\u019f\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\6\5\1\u01a0\16\5"+
    "\5\0\3\5\1\u01a1\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\2\5\1\u01a2\22\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\2\5\1\u01a3\22\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\15\5"+
    "\1\u01a4\7\5\5\0\3\5\1\u01a5\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\25\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\7\5"+
    "\1\u01a6\15\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\5\1\u01a7\23\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\3\5\1\u01a8\21\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\1\5\1\u01a9\1\5\21\0"+
    "\25\5\5\0\3\5\3\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\3\5\1\u01aa\1\5\1\u01ab\1\5\1\u01ac"+
    "\15\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\10\5\1\u01ad\14\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\u01ae\24\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\u01af\24\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\16\5\1\u01b0\6\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\u01b1\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\u01b2\1\0\3\5\21\0"+
    "\25\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\2\5\1\u01b3\22\5\5\0\3\5"+
    "\3\0\1\5\1\0\1\5\1\0\3\5\21\0\4\5"+
    "\1\u01b4\2\5\1\u01b5\15\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\4\5\1\u01b6"+
    "\20\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\2\5\1\u01b7\22\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\u01b8\1\0\3\5\21\0"+
    "\25\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\12\5\1\u01b9\12\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\11\5\1\u01ba\13\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\1\5\1\u01bb\23\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\u01bc\1\0"+
    "\3\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\7\5\1\u01bd\15\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\12\5\1\u01be\12\5\5\0\3\5\3\0"+
    "\1\5\1\0\1\5\1\0\2\5\1\u01bf\21\0\25\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\6\5\1\u01c0\16\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\7\5"+
    "\1\u01c1\15\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\u01c2\21\5\5\0"+
    "\3\5\3\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\4\5\1\u01c3\20\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\u01c4\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\3\0\1\5\1\0\1\5\1\0\3\5\21\0\7\5"+
    "\1\u01c5\15\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\u01c6\21\5\5\0"+
    "\3\5\1\u01c7\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\25\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\7\5\1\u01c8\15\5\5\0"+
    "\3\5\3\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\16\5\1\u01c9\6\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\u01ca\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\u01cb\24\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\11\5\1\u01cc\13\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\16\5\1\u01cd\6\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\1\5\1\u01ce\1\5\21\0"+
    "\25\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\3\5\1\u01cf\21\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\10\5\1\u01d0\14\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\11\5\1\u01d1\13\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\1\5\1\u01d2\23\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\1\5"+
    "\1\u01d3\23\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\15\5\1\u01d4\7\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\12\5\1\u01d5\12\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\1\5\1\u01d6\1\5\21\0"+
    "\25\5\5\0\3\5\1\61\2\0\1\5\1\0\1\u01d7"+
    "\1\0\3\5\21\0\25\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\1\5\1\u01d8"+
    "\23\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\2\5\1\u01d9\22\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\2\5\1\u01da\22\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\u01db\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\1\5\1\u01dc"+
    "\1\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\12\5\1\u01dd\12\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\2\5\1\u01de\22\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\7\5"+
    "\1\u01df\15\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\u01e0\24\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\12\5\1\u01e1\12\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\7\5\1\u01e2\15\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\12\5\1\u01e3\12\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\2\5"+
    "\1\u01e4\22\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\5\1\u01e5\23\5\5\0"+
    "\3\5\3\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\16\5\1\u01e6\6\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\2\5\1\u01e7\22\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\u01e8\1\0"+
    "\3\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\16\5\1\u01e9\6\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\3\5\1\u01ea\21\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\6\5"+
    "\1\u01eb\16\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\12\5\1\u01ec\12\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\u01ed\1\0\3\5"+
    "\21\0\25\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\7\5\1\u01ee\15\5\5\0"+
    "\3\5\1\u01ef\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\25\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\5\1\u01f0\23\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\1\5\1\u01f1\23\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\7\5\1\u01f2"+
    "\15\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\14\5\1\u01f3\10\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\u01f4\1\0\3\5\21\0"+
    "\25\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\3\5\1\u01f5\21\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\2\5\1\u01f6\22\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\1\5\1\u01f7\23\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\2\5\1\u01f8\22\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\1\u01f9\2\5\21\0"+
    "\25\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\2\5\1\u01fa\22\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\12\5\1\u01fb\12\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\2\5\1\u01fc\22\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\15\5\1\u01fd\7\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\u01fe\1\0\3\5\21\0\25\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\7\5\1\u01ff\15\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\10\5"+
    "\1\u0200\14\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\u0201\1\0\3\5\21\0\25\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\u0202\1\0\3\5\21\0\25\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\13\5\1\u0203\11\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\2\5"+
    "\1\u0204\22\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\12\5\1\u0205\12\5\5\0"+
    "\3\5\3\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\5\5\1\u0206\17\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\2\5\1\u0207\22\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\6\5\1\u0208\16\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\2\5"+
    "\1\u0209\22\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\u020a\24\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\3\5\1\u020b\21\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\7\5\1\u020c\15\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\2\5\1\u020d\22\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\20\5"+
    "\1\u020e\4\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\12\5\1\u020f\12\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\17\5\1\u0210\5\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\3\5\1\u0211"+
    "\21\5\5\0\3\5\1\61\2\0\1\5\1\0\1\u0212"+
    "\1\0\3\5\21\0\25\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\1\5\1\u0213\1\5\21\0"+
    "\25\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\6\5\1\u0214\16\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\3\5\1\u0215\21\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\2\5\1\u0216\22\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\16\5\1\u0217\6\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\7\5"+
    "\1\u0218\15\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\10\5\1\u0219\14\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\3\5\1\u021a\21\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\21\5\1\u021b"+
    "\3\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\10\5\1\u021c\14\5\5\0\3\5"+
    "\1\u021d\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\5\1\u021e\23\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\u021f\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\13\5\1\u0220\11\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\11\5\1\u0221\13\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\1\u0222\24\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\7\5\1\u0223"+
    "\15\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\12\5\1\u0224\12\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\3\5\1\u0225\21\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\7\5\1\u0226\15\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\7\5\1\u0227\15\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\7\5"+
    "\1\u0228\15\5\5\0\3\5\3\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\3\5\1\u0229\1\5\1\u022a\17\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\2\5\1\u022b\22\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\7\5"+
    "\1\u022c\15\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\1\5\1\u022d\23\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\3\5\1\u022e\21\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\13\5\1\u022f"+
    "\11\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\11\5\1\u0230\13\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\u0231\24\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\3\5\1\u0232\21\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\3\5\1\u0233\21\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\11\5\1\u0234"+
    "\13\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\16\5\1\u0235\6\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\3\5\1\u0236\21\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\16\5\1\u0237\6\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\u0238\1\0"+
    "\3\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\11\5\1\u0239\13\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\6\5\1\u023a\16\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\2\5"+
    "\1\u023b\22\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\11\5\1\u023c\13\5\5\0"+
    "\3\5\1\61\2\0\1\5\1\0\1\5\1\0\3\5"+
    "\21\0\7\5\1\u023d\15\5\5\0\3\5\1\61\2\0"+
    "\1\5\1\0\1\5\1\0\3\5\21\0\6\5\1\u023e"+
    "\16\5\5\0\3\5\1\61\2\0\1\5\1\0\1\5"+
    "\1\0\3\5\21\0\12\5\1\u023f\12\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\7\5\1\u0240\15\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\u0241\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\1\5\1\u0242\23\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\4\5\1\u0243\20\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\u0244\1\0"+
    "\3\5\21\0\25\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\u0245\1\0\3\5\21\0\25\5\5\0\3\5"+
    "\1\61\2\0\1\5\1\0\1\5\1\0\3\5\21\0"+
    "\13\5\1\u0246\11\5\5\0\3\5\1\61\2\0\1\5"+
    "\1\0\1\5\1\0\3\5\21\0\12\5\1\u0247\12\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\1\5\1\u0248\23\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\u0249\1\0\3\5\21\0\25\5"+
    "\5\0\3\5\1\61\2\0\1\5\1\0\1\5\1\0"+
    "\3\5\21\0\1\5\1\u024a\23\5\5\0\3\5\1\61"+
    "\2\0\1\5\1\0\1\5\1\0\3\5\21\0\6\5"+
    "\1\u024b\16\5\5\0\3\5\1\61\2\0\1\5\1\0"+
    "\1\5\1\0\3\5\21\0\2\5\1\u024c\22\5";

  private static int [] zzUnpackTrans() {
    int [] result = new int[27272];
    int offset = 0;
    offset = zzUnpackTrans(ZZ_TRANS_PACKED_0, offset, result);
    return result;
  }

  private static int zzUnpackTrans(String packed, int offset, int [] result) {
    int i = 0;       /* index in packed string  */
    int j = offset;  /* index in unpacked array */
    int l = packed.length();
    while (i < l) {
      int count = packed.charAt(i++);
      int value = packed.charAt(i++);
      value--;
      do result[j++] = value; while (--count > 0);
    }
    return j;
  }


  /* error codes */
  private static final int ZZ_UNKNOWN_ERROR = 0;
  private static final int ZZ_NO_MATCH = 1;
  private static final int ZZ_PUSHBACK_2BIG = 2;

  /* error messages for the codes above */
  private static final String ZZ_ERROR_MSG[] = {
    "Unkown internal scanner error",
    "Error: could not match input",
    "Error: pushback value was too large"
  };

  /**
   * ZZ_ATTRIBUTE[aState] contains the attributes of state <code>aState</code>
   */
  private static final int [] ZZ_ATTRIBUTE = zzUnpackAttribute();

  private static final String ZZ_ATTRIBUTE_PACKED_0 =
    "\1\0\1\11\1\1\1\11\5\1\1\11\2\1\1\11"+
    "\4\1\1\11\2\1\14\11\20\1\4\0\11\1\1\0"+
    "\1\1\1\0\2\11\51\1\2\0\1\11\13\1\1\11"+
    "\153\1\2\0\42\1\1\0\5\1\1\0\11\1\1\0"+
    "\37\1\1\0\40\1\1\0\5\1\1\0\41\1\1\0"+
    "\10\1\1\0\13\1\1\0\6\1\1\0\1\1\1\0"+
    "\3\1\1\0\41\1\1\0\47\1\1\0\55\1\1\0"+
    "\57\1";

  private static int [] zzUnpackAttribute() {
    int [] result = new int[588];
    int offset = 0;
    offset = zzUnpackAttribute(ZZ_ATTRIBUTE_PACKED_0, offset, result);
    return result;
  }

  private static int zzUnpackAttribute(String packed, int offset, int [] result) {
    int i = 0;       /* index in packed string  */
    int j = offset;  /* index in unpacked array */
    int l = packed.length();
    while (i < l) {
      int count = packed.charAt(i++);
      int value = packed.charAt(i++);
      do result[j++] = value; while (--count > 0);
    }
    return j;
  }

  /** the input device */
  private java.io.Reader zzReader;

  /** the current state of the DFA */
  private int zzState;

  /** the current lexical state */
  private int zzLexicalState = YYINITIAL;

  /** this buffer contains the current text to be matched and is
      the source of the yytext() string */
  private char zzBuffer[] = new char[ZZ_BUFFERSIZE];

  /** the textposition at the last accepting state */
  private int zzMarkedPos;

  /** the current text position in the buffer */
  private int zzCurrentPos;

  /** startRead marks the beginning of the yytext() string in the buffer */
  private int zzStartRead;

  /** endRead marks the last character in the buffer, that has been read
      from input */
  private int zzEndRead;

  /** number of newlines encountered up to the start of the matched text */
  private int yyline;

  /** the number of characters up to the start of the matched text */
  private int yychar;

  /**
   * the number of characters from the last newline up to the start of the 
   * matched text
   */
  private int yycolumn;

  /** 
   * zzAtBOL == true <=> the scanner is currently at the beginning of a line
   */
  private boolean zzAtBOL = true;

  /** zzAtEOF == true <=> the scanner is at the EOF */
  private boolean zzAtEOF;

  /** denotes if the user-EOF-code has already been executed */
  private boolean zzEOFDone;

  /* user code: */
    private int             pos () {
        return ((yyline << 16) | yycolumn);
    }

    private Symbol charString () {
        StringBuilder       sb = new StringBuilder ();
        String              quoted = yytext ();
        int                 len = quoted.length ();

        if (len < 2)
            throw new IllegalArgumentException ("len == " + len + " < 2");
        
        len--;

        if (quoted.charAt (0) != '"' ||
            quoted.charAt (len) != '"')
            throw new IllegalArgumentException (quoted);

        for (int ii = 1; ii < len; ii++) {
            char            c = quoted.charAt (ii);

            if (c == '"') {
                ii++;
                c = quoted.charAt (ii);

                if (c != '"')
                    throw new IllegalArgumentException ("Illegal quote in " + quoted);
            }

            sb.append (c);
        }
            
        return symbol (Symbols.CharString, sb.toString ());
    }

    private Symbol symbol (int type) {
        return symbol (type, yytext ());
    }

    private Symbol symbol (int type, Object value) {
        int         n = pos ();

        return new Symbol (type, n, n + yylength (), value);
    }   


  /**
   * Creates a new scanner
   * There is also a java.io.InputStream version of this constructor.
   *
   * @param   in  the java.io.Reader to read input from.
   */
  Lexer(java.io.Reader in) {
    this.zzReader = in;
  }

  /**
   * Creates a new scanner.
   * There is also java.io.Reader version of this constructor.
   *
   * @param   in  the java.io.Inputstream to read input from.
   */
  Lexer(java.io.InputStream in) {
    this(new java.io.InputStreamReader(in));
  }

  /** 
   * Unpacks the compressed character translation table.
   *
   * @param packed   the packed character translation table
   * @return         the unpacked character translation table
   */
  private static char [] zzUnpackCMap(String packed) {
    char [] map = new char[0x10000];
    int i = 0;  /* index in packed string  */
    int j = 0;  /* index in unpacked array */
    while (i < 136) {
      int  count = packed.charAt(i++);
      char value = packed.charAt(i++);
      do map[j++] = value; while (--count > 0);
    }
    return map;
  }


  /**
   * Refills the input buffer.
   *
   * @return      <code>false</code>, iff there was new input.
   * 
   * @exception   java.io.IOException  if any I/O-Error occurs
   */
  private boolean zzRefill() throws java.io.IOException {

    /* first: make room (if you can) */
    if (zzStartRead > 0) {
      System.arraycopy(zzBuffer, zzStartRead,
                       zzBuffer, 0,
                       zzEndRead-zzStartRead);

      /* translate stored positions */
      zzEndRead-= zzStartRead;
      zzCurrentPos-= zzStartRead;
      zzMarkedPos-= zzStartRead;
      zzStartRead = 0;
    }

    /* is the buffer big enough? */
    if (zzCurrentPos >= zzBuffer.length) {
      /* if not: blow it up */
      char newBuffer[] = new char[zzCurrentPos*2];
      System.arraycopy(zzBuffer, 0, newBuffer, 0, zzBuffer.length);
      zzBuffer = newBuffer;
    }

    /* finally: fill the buffer with new input */
    int numRead = zzReader.read(zzBuffer, zzEndRead,
                                            zzBuffer.length-zzEndRead);

    if (numRead > 0) {
      zzEndRead+= numRead;
      return false;
    }
    // unlikely but not impossible: read 0 characters, but not at end of stream    
    if (numRead == 0) {
      int c = zzReader.read();
      if (c == -1) {
        return true;
      } else {
        zzBuffer[zzEndRead++] = (char) c;
        return false;
      }     
    }

	// numRead < 0
    return true;
  }

    
  /**
   * Closes the input stream.
   */
  public final void yyclose() throws java.io.IOException {
    zzAtEOF = true;            /* indicate end of file */
    zzEndRead = zzStartRead;  /* invalidate buffer    */

    if (zzReader != null)
      zzReader.close();
  }


  /**
   * Resets the scanner to read from a new input stream.
   * Does not close the old reader.
   *
   * All internal variables are reset, the old input stream 
   * <b>cannot</b> be reused (internal buffer is discarded and lost).
   * Lexical state is set to <code>ZZ_INITIAL</code>.
   *
   * @param reader   the new input stream 
   */
  public final void yyreset(java.io.Reader reader) {
    zzReader = reader;
    zzAtBOL  = true;
    zzAtEOF  = false;
    zzEOFDone = false;
    zzEndRead = zzStartRead = 0;
    zzCurrentPos = zzMarkedPos = 0;
    yyline = yychar = yycolumn = 0;
    zzLexicalState = YYINITIAL;
  }


  /**
   * Returns the current lexical state.
   */
  public final int yystate() {
    return zzLexicalState;
  }


  /**
   * Enters a new lexical state
   *
   * @param newState the new lexical state
   */
  public final void yybegin(int newState) {
    zzLexicalState = newState;
  }


  /**
   * Returns the text matched by the current regular expression.
   */
  public final String yytext() {
    return new String( zzBuffer, zzStartRead, zzMarkedPos-zzStartRead );
  }


  /**
   * Returns the character at position <code>pos</code> from the
   * matched text. 
   * 
   * It is equivalent to yytext().charAt(pos), but faster
   *
   * @param pos the position of the character to fetch. 
   *            A value from 0 to yylength()-1.
   *
   * @return the character at position pos
   */
  public final char yycharat(int pos) {
    return zzBuffer[zzStartRead+pos];
  }


  /**
   * Returns the length of the matched text region.
   */
  public final int yylength() {
    return zzMarkedPos-zzStartRead;
  }


  /**
   * Reports an error that occured while scanning.
   *
   * In a wellformed scanner (no or only correct usage of 
   * yypushback(int) and a match-all fallback rule) this method 
   * will only be called with things that "Can't Possibly Happen".
   * If this method is called, something is seriously wrong
   * (e.g. a JFlex bug producing a faulty scanner etc.).
   *
   * Usual syntax/scanner level error handling should be done
   * in error fallback rules.
   *
   * @param   errorCode  the code of the errormessage to display
   */
  private void zzScanError(int errorCode) {
    String message;
    try {
      message = ZZ_ERROR_MSG[errorCode];
    }
    catch (ArrayIndexOutOfBoundsException e) {
      message = ZZ_ERROR_MSG[ZZ_UNKNOWN_ERROR];
    }

    throw new Error(message);
  } 


  /**
   * Pushes the specified amount of characters back into the input stream.
   *
   * They will be read again by then next call of the scanning method
   *
   * @param number  the number of characters to be read again.
   *                This number must not be greater than yylength()!
   */
  public void yypushback(int number)  {
    if ( number > yylength() )
      zzScanError(ZZ_PUSHBACK_2BIG);

    zzMarkedPos -= number;
  }


  /**
   * Contains user EOF-code, which will be executed exactly once,
   * when the end of file is reached
   */
  private void zzDoEOF() throws java.io.IOException {
    if (!zzEOFDone) {
      zzEOFDone = true;
      yyclose();
    }
  }


  /**
   * Resumes scanning until the next regular expression is matched,
   * the end of input is encountered or an I/O-Error occurs.
   *
   * @return      the next token
   * @exception   java.io.IOException  if any I/O-Error occurs
   */
  public java_cup.runtime.Symbol next_token() throws java.io.IOException {
    int zzInput;
    int zzAction;

    // cached fields:
    int zzCurrentPosL;
    int zzMarkedPosL;
    int zzEndReadL = zzEndRead;
    char [] zzBufferL = zzBuffer;
    char [] zzCMapL = ZZ_CMAP;

    int [] zzTransL = ZZ_TRANS;
    int [] zzRowMapL = ZZ_ROWMAP;
    int [] zzAttrL = ZZ_ATTRIBUTE;

    while (true) {
      zzMarkedPosL = zzMarkedPos;

      boolean zzR = false;
      for (zzCurrentPosL = zzStartRead; zzCurrentPosL < zzMarkedPosL;
                                                             zzCurrentPosL++) {
        switch (zzBufferL[zzCurrentPosL]) {
        case '\u000B':
        case '\u000C':
        case '\u0085':
        case '\u2028':
        case '\u2029':
          yyline++;
          yycolumn = 0;
          zzR = false;
          break;
        case '\r':
          yyline++;
          yycolumn = 0;
          zzR = true;
          break;
        case '\n':
          if (zzR)
            zzR = false;
          else {
            yyline++;
            yycolumn = 0;
          }
          break;
        default:
          zzR = false;
          yycolumn++;
        }
      }

      if (zzR) {
        // peek one character ahead if it is \n (if we have counted one line too much)
        boolean zzPeek;
        if (zzMarkedPosL < zzEndReadL)
          zzPeek = zzBufferL[zzMarkedPosL] == '\n';
        else if (zzAtEOF)
          zzPeek = false;
        else {
          boolean eof = zzRefill();
          zzEndReadL = zzEndRead;
          zzMarkedPosL = zzMarkedPos;
          zzBufferL = zzBuffer;
          if (eof) 
            zzPeek = false;
          else 
            zzPeek = zzBufferL[zzMarkedPosL] == '\n';
        }
        if (zzPeek) yyline--;
      }
      zzAction = -1;

      zzCurrentPosL = zzCurrentPos = zzStartRead = zzMarkedPosL;
  
      zzState = ZZ_LEXSTATE[zzLexicalState];


      zzForAction: {
        while (true) {
    
          if (zzCurrentPosL < zzEndReadL)
            zzInput = zzBufferL[zzCurrentPosL++];
          else if (zzAtEOF) {
            zzInput = YYEOF;
            break zzForAction;
          }
          else {
            // store back cached positions
            zzCurrentPos  = zzCurrentPosL;
            zzMarkedPos   = zzMarkedPosL;
            boolean eof = zzRefill();
            // get translated positions and possibly new buffer
            zzCurrentPosL  = zzCurrentPos;
            zzMarkedPosL   = zzMarkedPos;
            zzBufferL      = zzBuffer;
            zzEndReadL     = zzEndRead;
            if (eof) {
              zzInput = YYEOF;
              break zzForAction;
            }
            else {
              zzInput = zzBufferL[zzCurrentPosL++];
            }
          }
          int zzNext = zzTransL[ zzRowMapL[zzState] + zzCMapL[zzInput] ];
          if (zzNext == -1) break zzForAction;
          zzState = zzNext;

          int zzAttributes = zzAttrL[zzState];
          if ( (zzAttributes & 1) == 1 ) {
            zzAction = zzState;
            zzMarkedPosL = zzCurrentPosL;
            if ( (zzAttributes & 8) == 8 ) break zzForAction;
          }

        }
      }

      // store back cached position
      zzMarkedPos = zzMarkedPosL;

      switch (zzAction < 0 ? zzAction : ZZ_ACTION[zzAction]) {
        case 52: 
          { return symbol (Symbols.UNITS);
          }
        case 118: break;
        case 4: 
          { return symbol (Symbols.ValueId);
          }
        case 119: break;
        case 37: 
          { return symbol (Symbols.SET);
          }
        case 120: break;
        case 74: 
          { return symbol (Symbols.EXTERNAL);
          }
        case 121: break;
        case 23: 
          { return symbol (Symbols.PIPE);
          }
        case 122: break;
        case 101: 
          { return symbol (Symbols.LAST_UPDATED);
          }
        case 123: break;
        case 95: 
          { return symbol (Symbols.DESCRIPTION);
          }
        case 124: break;
        case 16: 
          { return symbol (Symbols.LT);
          }
        case 125: break;
        case 63: 
          { return symbol (Symbols.MODULE);
          }
        case 126: break;
        case 105: 
          { return symbol (Symbols.NOTIFICATIONS);
          }
        case 127: break;
        case 13: 
          { return symbol (Symbols.RBKT);
          }
        case 128: break;
        case 1: 
          { int  n = pos ();

    throw new SyntaxErrorException (
        "Illegal character: '"+ yytext ()+ "'",
        Location.combine (n, n + yylength ())
    );
          }
        case 129: break;
        case 100: 
          { return symbol (Symbols.CONTACT_INFO);
          }
        case 130: break;
        case 99: 
          { return symbol (Symbols.DISPLAY_HINT);
          }
        case 131: break;
        case 9: 
          { return symbol (Symbols.DQUOT);
          }
        case 132: break;
        case 85: 
          { return symbol (Symbols.UNIVERSAL);
          }
        case 133: break;
        case 44: 
          { return symbol (Symbols.TRUE);
          }
        case 134: break;
        case 73: 
          { return symbol (Symbols.OBJECTS);
          }
        case 135: break;
        case 7: 
          { return symbol (Symbols.DOT);
          }
        case 136: break;
        case 25: 
          { return symbol (Symbols.CARET);
          }
        case 137: break;
        case 12: 
          { return symbol (Symbols.LBKT);
          }
        case 138: break;
        case 42: 
          { return symbol (Symbols.NULL);
          }
        case 139: break;
        case 108: 
          { return symbol (Symbols.PRODUCT_RELEASE);
          }
        case 140: break;
        case 47: 
          { return symbol (Symbols.WITH);
          }
        case 141: break;
        case 49: 
          { return symbol (Symbols.GROUP);
          }
        case 142: break;
        case 98: 
          { return symbol (Symbols.OBJECT_TYPE);
          }
        case 143: break;
        case 72: 
          { return symbol (Symbols.IMPORTS);
          }
        case 144: break;
        case 107: 
          { return symbol (Symbols.MINUS_INFINITY);
          }
        case 145: break;
        case 19: 
          { return symbol (Symbols.LP);
          }
        case 146: break;
        case 30: 
          { return symbol (Symbols.RVB);
          }
        case 147: break;
        case 5: 
          { return symbol (Symbols.Number, new java.math.BigInteger (yytext (), 10));
          }
        case 148: break;
        case 87: 
          { return symbol (Symbols.VARIABLES);
          }
        case 149: break;
        case 57: 
          { return symbol (Symbols.ACCESS);
          }
        case 150: break;
        case 48: 
          { return symbol (Symbols.BEGIN);
          }
        case 151: break;
        case 110: 
          { return symbol (Symbols.MODULE_IDENTITY);
          }
        case 152: break;
        case 94: 
          { return symbol (Symbols.MIN_ACCESS);
          }
        case 153: break;
        case 78: 
          { return symbol (Symbols.SUPPORTS);
          }
        case 154: break;
        case 76: 
          { return symbol (Symbols.AUGMENTS);
          }
        case 155: break;
        case 55: 
          { return symbol (Symbols.DEFVAL);
          }
        case 156: break;
        case 64: 
          { return symbol (Symbols.EXPORTS);
          }
        case 157: break;
        case 20: 
          { return symbol (Symbols.RP);
          }
        case 158: break;
        case 34: 
          { return symbol (Symbols.BIT);
          }
        case 159: break;
        case 29: 
          { return symbol (Symbols.LVB);
          }
        case 160: break;
        case 61: 
          { return symbol (Symbols.CHOICE);
          }
        case 161: break;
        case 113: 
          { return symbol (Symbols.CREATION_REQUIRES);
          }
        case 162: break;
        case 93: 
          { return symbol (Symbols.MAX_ACCESS);
          }
        case 163: break;
        case 33: 
          { return symbol (Symbols.END);
          }
        case 164: break;
        case 59: 
          { return symbol (Symbols.STRING);
          }
        case 165: break;
        case 28: 
          { return charString ();
          }
        case 166: break;
        case 83: 
          { return symbol (Symbols.TRAP_TYPE);
          }
        case 167: break;
        case 21: 
          { return symbol (Symbols.SEMIC);
          }
        case 168: break;
        case 8: 
          { return symbol (Symbols.SQUOT);
          }
        case 169: break;
        case 81: 
          { return symbol (Symbols.OPTIONAL);
          }
        case 170: break;
        case 39: 
          { return symbol (Symbols.MIN);
          }
        case 171: break;
        case 71: 
          { return symbol (Symbols.IMPLIED);
          }
        case 172: break;
        case 54: 
          { return symbol (Symbols.FALSE);
          }
        case 173: break;
        case 58: 
          { return symbol (Symbols.STATUS);
          }
        case 174: break;
        case 46: 
          { return symbol (Symbols.REAL);
          }
        case 175: break;
        case 45: 
          { return symbol (Symbols.FROM);
          }
        case 176: break;
        case 43: 
          { return symbol (Symbols.TAGS);
          }
        case 177: break;
        case 96: 
          { return symbol (Symbols.DEFINITIONS);
          }
        case 178: break;
        case 36: 
          { return symbol (Symbols.ANY);
          }
        case 179: break;
        case 40: 
          { return symbol (Symbols.BITS);
          }
        case 180: break;
        case 79: 
          { return symbol (Symbols.INCLUDES);
          }
        case 181: break;
        case 50: 
          { return symbol (Symbols.INDEX);
          }
        case 182: break;
        case 80: 
          { return symbol (Symbols.IMPLICIT);
          }
        case 183: break;
        case 116: 
          { return symbol (Symbols.NOTIFICATION_GROUP);
          }
        case 184: break;
        case 53: 
          { return symbol (Symbols.MACRO);
          }
        case 185: break;
        case 31: 
          { return symbol (Symbols.OF);
          }
        case 186: break;
        case 68: 
          { return symbol (Symbols.PRESENT);
          }
        case 187: break;
        case 104: 
          { return symbol (Symbols.WRITE_SYNTAX);
          }
        case 188: break;
        case 62: 
          { return symbol (Symbols.OBJECT);
          }
        case 189: break;
        case 77: 
          { return symbol (Symbols.SEQUENCE);
          }
        case 190: break;
        case 75: 
          { return symbol (Symbols.EXPLICIT);
          }
        case 191: break;
        case 106: 
          { return symbol (Symbols.PLUS_INFINITY);
          }
        case 192: break;
        case 82: 
          { return symbol (Symbols.REVISION);
          }
        case 193: break;
        case 27: 
          { return symbol (Symbols.BY);
          }
        case 194: break;
        case 65: 
          { return symbol (Symbols.BOOLEAN);
          }
        case 195: break;
        case 117: 
          { return symbol (Symbols.TEXTUAL_CONVENTION);
          }
        case 196: break;
        case 69: 
          { return symbol (Symbols.PRIVATE);
          }
        case 197: break;
        case 3: 
          { return symbol (Symbols.TypeId);
          }
        case 198: break;
        case 102: 
          { return symbol (Symbols.OBJECT_GROUP);
          }
        case 199: break;
        case 24: 
          { return symbol (Symbols.EXC);
          }
        case 200: break;
        case 114: 
          { return symbol (Symbols.MODULE_COMPLIANCE);
          }
        case 201: break;
        case 97: 
          { return symbol (Symbols.APPLICATION);
          }
        case 202: break;
        case 67: 
          { return symbol (Symbols.DEFINED);
          }
        case 203: break;
        case 86: 
          { return symbol (Symbols.REFERENCE);
          }
        case 204: break;
        case 111: 
          { return symbol (Symbols.MANDATORY_GROUPS);
          }
        case 205: break;
        case 56: 
          { return symbol (Symbols.ABSENT);
          }
        case 206: break;
        case 32: 
          { return symbol (Symbols.ELLIPSIS);
          }
        case 207: break;
        case 15: 
          { return symbol (Symbols.RCB);
          }
        case 208: break;
        case 112: 
          { return symbol (Symbols.NOTIFICATION_TYPE);
          }
        case 209: break;
        case 103: 
          { return symbol (Symbols.ORGANIZATION);
          }
        case 210: break;
        case 41: 
          { return symbol (Symbols.SIZE);
          }
        case 211: break;
        case 22: 
          { return symbol (Symbols.AT);
          }
        case 212: break;
        case 10: 
          { return symbol (Symbols.COLON);
          }
        case 213: break;
        case 109: 
          { return symbol (Symbols.OBJECT_IDENTITY);
          }
        case 214: break;
        case 66: 
          { return symbol (Symbols.DEFAULT);
          }
        case 215: break;
        case 90: 
          { return symbol (Symbols.ENUMERATED);
          }
        case 216: break;
        case 92: 
          { return symbol (Symbols.IDENTIFIER);
          }
        case 217: break;
        case 6: 
          { return symbol (Symbols.HYPHEN);
          }
        case 218: break;
        case 2: 
          { /* ignore */
          }
        case 219: break;
        case 115: 
          { return symbol (Symbols.AGENT_CAPABILITIES);
          }
        case 220: break;
        case 14: 
          { return symbol (Symbols.LCB);
          }
        case 221: break;
        case 84: 
          { return symbol (Symbols.COMPONENT);
          }
        case 222: break;
        case 51: 
          { return symbol (Symbols.OCTET);
          }
        case 223: break;
        case 17: 
          { return symbol (Symbols.GT);
          }
        case 224: break;
        case 38: 
          { return symbol (Symbols.MAX);
          }
        case 225: break;
        case 91: 
          { return symbol (Symbols.COMPONENTS);
          }
        case 226: break;
        case 88: 
          { return symbol (Symbols.VARIATION);
          }
        case 227: break;
        case 35: 
          { return symbol (Symbols.ASSIGN);
          }
        case 228: break;
        case 60: 
          { return symbol (Symbols.SYNTAX);
          }
        case 229: break;
        case 18: 
          { return symbol (Symbols.COMMA);
          }
        case 230: break;
        case 11: 
          { return symbol (Symbols.EQ);
          }
        case 231: break;
        case 26: 
          { return symbol (Symbols.RANGESEP);
          }
        case 232: break;
        case 89: 
          { return symbol (Symbols.ENTERPRISE);
          }
        case 233: break;
        case 70: 
          { return symbol (Symbols.INTEGER);
          }
        case 234: break;
        default: 
          if (zzInput == YYEOF && zzStartRead == zzCurrentPos) {
            zzAtEOF = true;
            zzDoEOF();
              { return new java_cup.runtime.Symbol(Symbols.EOF); }
          } 
          else {
            zzScanError(ZZ_NO_MATCH);
          }
      }
    }
  }


}