/*
 * Copyright 2024 EPAM Systems, Inc
 *
 * See the NOTICE file distributed with this work for additional information
 * regarding copyright ownership. Licensed under the Apache License,
 * Version 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.epam.deltix.qsrv.hf.topic.consumer;

import com.epam.deltix.gflog.api.Log;
import com.epam.deltix.gflog.api.LogFactory;
import com.epam.deltix.qsrv.hf.pub.TypeLoader;
import com.epam.deltix.qsrv.hf.pub.codec.CodecFactory;
import com.epam.deltix.qsrv.hf.pub.md.RecordClassDescriptor;
import com.epam.deltix.qsrv.hf.tickdb.pub.topic.MessagePoller;
import com.epam.deltix.qsrv.hf.tickdb.pub.topic.MessageProcessor;
import com.epam.deltix.qsrv.hf.tickdb.pub.topic.TopicDataLossHandler;
import com.epam.deltix.qsrv.hf.topic.consumer.annotation.AeronClientThread;
import com.epam.deltix.qsrv.hf.topic.consumer.annotation.AnyThread;
import com.epam.deltix.qsrv.hf.topic.consumer.annotation.ReaderThreadOnly;
import com.epam.deltix.util.concurrent.CursorIsClosedException;
import io.aeron.Aeron;
import io.aeron.ControlledFragmentAssembler;
import io.aeron.Image;
import io.aeron.Subscription;
import net.jcip.annotations.NotThreadSafe;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.nio.ByteOrder;
import java.util.List;

/**
 * @author Alexei Osipov
 */
@ParametersAreNonnullByDefault
@NotThreadSafe
class DirectMessageNonblockingPoller implements MessagePoller {
    private static final Log LOG = LogFactory.getLog(DirectMessageNonblockingPoller.class);

    private final Subscription subscription;
    private final ControlledFragmentAssembler fragmentAssembler;
    //private final SubscriptionPublicationLimitCounterCache counterCache;
    //private final CountersReader countersReader;
    private final MessageFragmentHandler decodingFragmentHandler;
    private final IpcFillPercentageChecker fillChecker;
    private final TopicDataLossHandler topicDataLossHandler;


    // Indicates that poller should be stopped OR already stopped
    private volatile boolean stopFlag = false;
    // Indicates that poller is stopped
    private boolean stopped = false;
    // Indicates that poller detected a data loss before graceful stop
    private volatile boolean dataLoss = false;

    DirectMessageNonblockingPoller(Aeron aeron, boolean raw, String channel, int dataStreamId,
                                   List<RecordClassDescriptor> types, CodecFactory codecFactory, TypeLoader typeLoader,
                                   @Nullable TopicDataLossHandler topicDataLossHandler) {
        // TODO: Implement loading of temp indexes from server
        if (!ByteOrder.nativeOrder().equals(ByteOrder.LITTLE_ENDIAN)) {
            throw new IllegalArgumentException("Only LITTLE_ENDIAN byte order supported");
        }

        this.topicDataLossHandler = topicDataLossHandler;
        this.fillChecker = new IpcFillPercentageChecker();

        DoubleUnavailableImageHandler unavailableImageHandler = new DoubleUnavailableImageHandler(fillChecker, DirectMessageNonblockingPoller.this::onUnavailableImage);

        // The main caveat here that we first create a subscription (so we start to collect metadata)
        // and THEN get the mapping from the server using mappingProvider.
        // Otherwise it would be possible to get situation when we get stale data in the server response
        this.subscription = aeron.addSubscription(channel, dataStreamId, fillChecker, unavailableImageHandler);
        LOG.debug().append("Subscribed to dataStreamId=").appendLast(dataStreamId);
        // Load mapping: this can be slow and may involve network interaction

        this.decodingFragmentHandler = new MessageFragmentHandler(raw, codecFactory, typeLoader, types);
        this.fragmentAssembler = new ControlledFragmentAssembler(decodingFragmentHandler);
    }

    @ReaderThreadOnly
    @Override
    public int processMessages(int messageCountLimit, MessageProcessor messageProcessor) throws CursorIsClosedException {
        if (stopFlag) {
            handleStop();
        }
        if (messageProcessor == null) {
            throw new IllegalArgumentException("messageProcessor can't be null");
        }

        MessageFragmentHandler handler = this.decodingFragmentHandler;
        handler.setProcessor(messageProcessor);
        int result = subscription.controlledPoll(fragmentAssembler, messageCountLimit);
        handler.clearProcessor();
        decodingFragmentHandler.checkException();
        return result;
    }

    @ReaderThreadOnly
    private void handleStop() {
        if (!stopped) {
            close();
        }
        if (dataLoss) {
            throw new ClosedDueToDataLossException();
        } else {
            throw new CursorIsClosedException();
        }
    }

    /**
     * Closes allocated resources.
     * Please note that this method must be called from the same thread that polls messages.
     *
     * <p>It's permitted to call this method from another thread only if
     * there are no concurrent calls to {@link #processMessages}
     * AND it's guaranteed that all previous calls of {@link #processMessages}
     * have "happens before" relationship with this {@link #close()} call
     * AND this call has "happens before" relationship with any subsequent calls to {@link #processMessages}.
     */
    @ReaderThreadOnly
    @Override
    public void close() {
        stopped = true;
        subscription.close();
        fillChecker.releaseResources();
        stopFlag = true;
    }

    @AnyThread
    @Override
    public byte getBufferFillPercentage() {
        return fillChecker.getBufferFillPercentage();
    }

    @AeronClientThread
    // That will be executed from an Aeron's thread
    private void onUnavailableImage(Image image) {
        int sessionId = image.sessionId();
        // Note: decodingFragmentHandler can be null during the initialization process
        if (!stopped && !stopFlag && decodingFragmentHandler != null && !decodingFragmentHandler.checkIfSessionGracefullyClosed(sessionId)) {
            // Not a graceful close
            onDataLossDetected();
        }
    }

    @AeronClientThread
    // That will be executed from an Aeron's thread
    private void onDataLossDetected() {
        LOG.debug("Data loss detected for subscriber with dataStreamId=%s").with(subscription.streamId());
        if (topicDataLossHandler != null) {
            boolean continuePolling = topicDataLossHandler.handleDataLoss();
            if (continuePolling) {
                return;
            }
        }
        dataLoss = true;
        stopFlag = true;
    }
}